if (AjxPackage.define("Startup1_2")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */
/*
 * Package: Startup1_2
 * 
 * Together with Startup1_1, contains everything needed to support displaying
 * the results of the initial mail search. Startup1 was split into two smaller
 * packages becausing JS parsing cost increases exponentially, so it is best to
 * keep the files under 100K or so.
 */
if (AjxPackage.define("zimbraMail.share.model.ZmObjectHandler")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file contains the zimlet object handler base class.
 * 
 */

/**
 * Creates the object handler.
 * @class
 * This class defines the default implementation for a zimlet object handler.
 * <br />
 * <br />
 * To write a zimlet, see {@link ZmZimletBase}. 
 * 
 * @param	{string}	typeName	the type name
 * @param	{string}	className	the class name
 */
ZmObjectHandler = function(typeName, className) {
	if (arguments.length > 0) {
		this.init(typeName, className);
	}
}

ZmObjectHandler.prototype.constructor = ZmObjectHandler;
ZmObjectHandler.prototype.isZmObjectHandler = true;

/**
 * This method is called by the Zimlet framework to initialize the object.
 * 
 * @param	{string}	typeName	the type name
 * @param	{string}	className	the class name; if <code>null</code>, "Object" will be used
 */
ZmObjectHandler.prototype.init =
function(typeName, className) {
	this._typeName = typeName;
	this._className = className ? className : "Object";
	this.name = this.toString();
};

/**
 * Returns a string representation of the object.
 * 
 * @return		{string}		a string representation of the object
 */
ZmObjectHandler.prototype.toString = 
function() {
	// If you can find a cleaner way to get the name of 
	// a sub-class without hard coding each instance
	// in a toString() method feel free to change.
	if(!this._toString) {
		var ctor = "" + this.constructor;
		ctor = ctor.substring(0,ctor.indexOf("("));
		this._toString = ctor.substring("function ".length);
	}
	return this._toString;
};

ZmObjectHandler.prototype.getEnabled =	function() {
	return true;
};


/**
 * Gets the type name.
 * 
 * @return	{string}		the type name
 */
ZmObjectHandler.prototype.getTypeName =
function() {
	return this._typeName;
};

/**
 * Gets the class name for a given object.
 * 
 * @param	{Object}		obj			the object
 * @param	{Object}		context		the content
 * @param	{string}		spanId		ID of the SPAN
 *
 * @return	{string}		the class name
 */
ZmObjectHandler.prototype.getClassName =
function(obj, context, spanId) {
	return this._className;
};

/**
 * Gets the hovered class name for the given object.
 * 
 * @param	{Object}		obj			the object
 * @param	{Object}		context		the content
 * @param	{string}		spanId		ID of hovered SPAN
 *
 * @return	{string}		the hovered class name
 */
ZmObjectHandler.prototype.getHoveredClassName =
function(obj, context, spanId) {
	var cname = this.getClassName(obj, context, spanId);
	if (this._cachedClassNameForHovered !== cname) {
		this._cachedClassNameForHovered = cname;
		this._classNameHovered = cname + "-" + DwtCssStyle.HOVER;
	}
	return this._classNameHovered;
};

/**
 * Gets the active class name for a given object.
 * 
 * @param	{Object}		obj		the object
 * @param	{Object}		context		the content
 * @param	{string}		spanId		ID of the SPAN
 *
 * @return	{string}		the active class name
 */
ZmObjectHandler.prototype.getActiveClassName =
function(obj, context, spanId) {
	var cname = this.getClassName(obj, context, spanId);
	if (this._cachedClassNameForActive !== cname) {
		this._cachedClassNameForActive = cname;
		this._classNameActive = cname + "-" + DwtCssStyle.ACTIVE;
	}
	return this._classNameActive;
};

/**
 * @private
 */
ZmObjectHandler.prototype.findObject =
function(content, startIndex, objectMgr) {
	if (startIndex === 0) {
		this._lastMatch = null;
		this._noMatch = false;
	}
	if (this._noMatch) {return null;}
	if (this._lastMatch && this._lastMatch.index >= startIndex) {
		return this._lastMatch;
	}
	this._lastMatch = this.match(content, startIndex, objectMgr);
	this._noMatch = (this._lastMatch === null);
	return this._lastMatch;
};


/**
 * This method is used to match content for a zimlet. Zimlet implementations should
 * override this method. Usage should return a non-null result in the format of
 * <code>String.match</code> if text on the line matched the handler regular expression.
 * 
 * <pre>
 * var result = handler.match(line);
 * result[0] // should be matched string
 * result.index // should be location within line match occurred
 * </pre>
 * 
 * Handlers can also set result.context which will be passed back to
 * them during the various method calls ({@link #getToolTipText}, etc). Handlers should set
 * regex.lastIndex to startIndex and then use <code>regex.exec(content)</code>. Handlers should
 * also use the "g" option when constructing their regex.
 */
ZmObjectHandler.prototype.match =
function(content, startIndex) {
	return null;
};

/**
 * Generates content inside the <code>&lt;span&gt;</code> tag.
 * 
 * @return	{number}	the content index
 * @private
 * */
ZmObjectHandler.prototype._getHtmlContent =
function(html, idx, obj, context, spanId) {
	html[idx++] = AjxStringUtil.htmlEncode(obj, true);
	return idx;
};

/**
 * Generates the <code>&lt;span&gt;</code> tag.
 * 
 * @return	{number}	the content index
 * @private
 */
ZmObjectHandler.prototype.generateSpan = 
function(html, idx, obj, spanId, context, options) {
	html[idx++] = "<span class='";
	html[idx++] = this.getClassName(obj);
	html[idx++] = "' role='link' id='";
	html[idx++] = spanId;
	html[idx++] = "'>";
	idx = this._getHtmlContent(html, idx, obj, context, spanId, options);
	html[idx++] = "</span>";
	return idx;
};

/**
 * Checks if the handler has tool tip text.
 * 
 * @param		{Object}	obj			the object
 * @param		{Object}	context		the context
 * @return		<code>true</code> if the handler has tool tip text; <code>false</code> otherwise
 */
ZmObjectHandler.prototype.hasToolTipText =
function(obj, context) {
	return true;
};

/**
 * Gets the handler tool tip text.
 * 
 * @param		{Object}	obj			the object
 * @param		{Object}	context		the context
 * @return		{string}	the handler has tool tip text
 */
ZmObjectHandler.prototype.getToolTipText =
function(obj, context) {
	return AjxStringUtil.htmlEncode(obj);
};

/**
 * Populates the handler tool tip text.
 * 
 * @param		{Object}	obj			the object
 * @param		{Object}	context		the context
 */
ZmObjectHandler.prototype.populateToolTip =
function(obj, context) {
};

/**
 * Gets the action menu.
 * 
 * @param		{Object}	obj			the object
 * @param		{string}	span		the span element
 * @param		{Object}	context		the context
 * @return		{ZmActionMenu}	the action menu
 * 
 * @private
 */
ZmObjectHandler.prototype.getActionMenu =
function(obj, span, context) {
	return null;
};

/**
 * This method is called by the Zimlet framework when the object is selected.
 * 
 * @param		{Object}	obj			the object
 * @param		{string}	span		the span element
 * @param		{Object}	ev			the event
 * @param		{Object}	context		the context
 * @see		#clicked
 */
ZmObjectHandler.prototype.selected =
function(obj, span, ev, context) {
	return this.clicked(span, obj, context, ev);
};

/**
 * This method is called by the Zimlet framework when the object is clicked.
 * 
 * @param		{Object}	obj			the object
 * @param		{string}	span		the span element
 * @param		{Object}	ev			the event
 * @param		{Object}	context		the context
 */
ZmObjectHandler.prototype.clicked =
function(span, obj, context, ev) {
};

/**
 * This method is called when the object is hovered-over.
 * 
 * @private
 */
ZmObjectHandler.prototype.hoverOver = function(object, context, x, y) {

	var tooltip = this.getToolTipText(object, context) || '',
		content, callback;

	if (typeof(tooltip) === "string") {
		content = tooltip;
	}
	else if (tooltip.isAjxCallback || AjxUtil.isFunction(tooltip)) {
		callback = tooltip;
	}
	else if (typeof(tooltip) === "object") {
		content = tooltip.content;
		callback = tooltip.callback;
	}

	if (!content && callback && tooltip.loading) {
		content = AjxMsg.loading;
	}

	if (content) {
		this._showTooltip(object, context, x, y, content);
	}

	if (callback) {
		var callback1 = new AjxCallback(this, this._showTooltip, [ object, context, x, y ]);
		AjxTimedAction.scheduleAction(new AjxTimedAction(null, function() {
			callback.run(callback1);
		}), 0);
	}
};

ZmObjectHandler.prototype._showTooltip = function(object, context, x, y, content) {
	var shell = DwtShell.getShell(window);
	var tooltip = shell.getToolTip();
	tooltip.setContent(content);
	tooltip.popup(x, y);
	// TODO: call below is odd; not sure if it's used much, appears to be for two-step tooltips (eg a map)
	this.populateToolTip(object, context);
};

/**
 * This method is called when the handler is hovered-out.
 * 
 * @private
 */
ZmObjectHandler.prototype.hoverOut = function(object, context) {
	var shell = DwtShell.getShell(window);
	var tooltip = shell.getToolTip();
	tooltip.popdown();
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmObjectManager")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the object manager.
 * 
 */

/**
 * Creates an object manager.
 * @class
 * This class is used to high-light objects within a given view.
 * 
 * @author Kevin Henrikson
 *
 * @param {DwtComposite}	view			the view this manager is going to high-light for
 * @param {AjxCallback}	selectCallback  the callback triggered when user clicks on high-light object (provide if you want to do something before the clicked on object opens its corresponding view)
 * @param {Boolean}	skipHandlers 	<code>true</code> to avoid adding the standard handlers
 */
ZmObjectManager = function(view, selectCallback, skipHandlers) {

	this._selectCallback = selectCallback;
	this._uuid = Dwt.getNextId();
	this._objectIdPrefix = "OBJ_PREFIX_";
	this._objectHandlers = {};

	// don't include when looking for objects. only used to provide tool tips for images
	if (appCtxt.get(ZmSetting.MAIL_ENABLED) && window["ZmImageAttachmentObjectHandler"]) {
		this._imageAttachmentHandler = new ZmImageAttachmentObjectHandler();
	}

	// create handlers (see registerHandler below)
	if (!skipHandlers) {
        this.initialized = false;
        this._addAutoHandlers();
	} else {
        this.initialized = true;
    }

    this.sortHandlers();
	this.reset();
	this.setView(view);
};

ZmObjectManager._TOOLTIP_DELAY = 275;

// Define common types for quicker object matching.
ZmObjectManager.EMAIL = "email";
ZmObjectManager.URL = "url";
ZmObjectManager.PHONE = "phone";
ZmObjectManager.DATE = "date";
ZmObjectManager.ADDRESS = "address";
ZmObjectManager.TITLE = "title";

// Allows callers to pass in a current date
ZmObjectManager.ATTR_CURRENT_DATE = "currentDate";

ZmObjectManager._autohandlers = [];

/**
 * Registers the handler.
 * 
 * @param	{Object}	obj		the object
 * @param	{constant}	type	the type
 * @param	{constant}	priority	the priority
 */
ZmObjectManager.registerHandler =
function(obj, type, priority) {
	if (typeof obj == "string") {
		obj = eval(obj);
	}
	var c = ZmObjectManager._autohandlers;
	if (!obj.__registered) {
		var id = c.push(obj);
		var i = id - 1;
		if(type) {
			c[i].useType = type;
		}
		if(priority) {
			c[i].usePrio = priority;
		}
		obj.__registered = true;
	}
};

/**
 * @private
 */
ZmObjectManager.unregisterHandler =
function(obj) {
	if (typeof obj == "string") {
		obj = eval(obj);
	}
 	var c = ZmObjectManager._autohandlers, i;
	for (i = c.length; --i >= 0;) {
		if (c[i] === obj) {
			c.splice(i, 1);
			break;
		}
	}
};

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmObjectManager.prototype.toString =
function() {
	return "ZmObjectManager";
};

/**
 * Gets the handlers.
 * 
 * @return	{Array}	an array of {@link ZmObjectHandler} objects
 */
ZmObjectManager.prototype.getHandlers =
function() {
	if (!this.initialized) {
		var zimletMgr = appCtxt.getZimletMgr();
		if (zimletMgr.isLoaded()) {
			this.initialized = true;
			var zimlets = zimletMgr.getContentZimlets();
			for (var i = 0; i < zimlets.length; i++) {
				this.addHandler(zimlets[i], zimlets[i].type, zimlets[i].prio);
			}
		}
	}
	return this._objectHandlers;
};

/**
 * Adds the handler.
 * 
 * @param	{ZmObjectHandler}	h		the handler
 * @param	{constant}			type	the type
 * @param	{constant}		priority	the priority
 */
ZmObjectManager.prototype.addHandler =
function(h, type, priority) {
	type = type || (h.getTypeName() ? h.getTypeName() : "none");
	priority = priority ? priority : -1;
	h._prio = priority;
	//DBG.println(AjxDebug.DBG3, "addHandler " + h + " type: " + type + " prio: " + priority);
	var oh = this.getHandlers();
	if (!oh[type]) {oh[type] = [];}
	oh[type].push(h);
};

/**
 * Removes the handler.
 * 
 * @param	{ZmObjectHandler}	h		the handler
 * @param	{constant}			type	the type
 */
ZmObjectManager.prototype.removeHandler =
function(h, type) {
	type = type || (h.getTypeName() ? h.getTypeName() : "none");
	var oh = this.getHandlers();
	if (oh[type]) {
		for (var i = 0, count = oh[type].length; i < count; i++) {
			if (oh[type][i] == h) {
				oh[type].splice(i, 1);
				break;
			}
		}
	}
};

/**
 * Sorts the handlers.
 * 
 */
ZmObjectManager.prototype.sortHandlers =
function() {
	this._allObjectHandlers = [];
    var objectHandlers = this.getHandlers();
    for (var type in objectHandlers) {
		// Object handlers grouped by Type
		objectHandlers[type].sort(ZmObjectManager.__byPriority);

		// Copy each array to a single array of all Object Handlers
		for (var k = 0; k < objectHandlers[type].length; k++) {
			this._allObjectHandlers.push(objectHandlers[type][k]);
		}
	}
	this._allObjectHandlers.sort(ZmObjectManager.__byPriority);
};

/**
 * @private
 */
ZmObjectManager.prototype._addAutoHandlers =
function() {
	var c = ZmObjectManager._autohandlers, i, obj, prio;
	for (i = 0; i < c.length; ++i) {
		obj = c[i];
		var	handler = obj;
		var type = obj.TYPE;
		if (!(obj.isZmObjectHandler)) {
			handler = new obj();
		}
		if (obj.useType) {
			type = obj.useType;
		}
		if (obj.usePrio) {
			prio = obj.usePrio;
		}
		this.addHandler(handler, type, prio);
	}
};

/**
 * Resets the objects.
 * 
 */
ZmObjectManager.prototype.reset =
function() {
	this._objects = {};
};

/**
 * Sets the view.
 * 
 * @param	{DwtComposite}		view		the view
 */
ZmObjectManager.prototype.setView =
function(view) {
	if (view != null && appCtxt.getZimletMgr().isLoaded()) {
	    view.addListener(DwtEvent.ONMOUSEOVER, new AjxListener(this, this._mouseOverListener));
	    view.addListener(DwtEvent.ONMOUSEOUT, new AjxListener(this, this._mouseOutListener));
	    view.addListener(DwtEvent.ONMOUSEDOWN, new AjxListener(this, this._mouseDownListener));
	    view.addListener(DwtEvent.ONMOUSEUP, new AjxListener(this, this._mouseUpListener));
	    view.addListener(DwtEvent.ONMOUSEMOVE, new AjxListener(this, this._mouseMoveListener));
		view.addListener(DwtEvent.ONCONTEXTMENU, new AjxListener(this, this._rightClickListener));
		this._hoverOverListener = new AjxListener(this, this._handleHoverOver);
	    this._hoverOutListener = new AjxListener(this, this._handleHoverOut);
	}
	this._view = view;
};

ZmObjectManager.prototype.getView = function() {
	return this._view;
};

/**
 * Gets the count of objects.
 * 
 * @return	{int}	the count
 */
ZmObjectManager.prototype.objectsCount =
function() {
	return (appCtxt.zimletsPresent()) ? appCtxt.getZimletMgr().getContentZimlets().length : 0;
};

/**
 * Gets the image attachment handler.
 * 
 * @return	{ZmImageAttachmentObjectHandler}	the handler
 */
ZmObjectManager.prototype.getImageAttachmentHandler =
function() {
	return this._imageAttachmentHandler;
};

/**
 * @private
 */
ZmObjectManager.prototype._getAjxEmailAddress =
function(obj){
    if(appCtxt.isChildWindow && obj.isAjxEmailAddress){ //Making sure child window knows its type AjxEmailAddress
        obj = AjxEmailAddress.copy(obj);
    }
    return obj;
};

/**
 * Finds objects.
 * 
 * @param	{String}	content		the content
 * @param	{Boolean}	htmlEncode	<code>true</code> to HTML encode the content
 * @param	{constant}	type		the type
 * @param	{Boolean}	isTextMsg	<code>true</code> if is text msg
 * @param	{hash}		options		arbitrary options to pass to handler
 *
 * @return	{String}	the object
 */
ZmObjectManager.prototype.findObjects =
function(content, htmlEncode, type, isTextMsg, options) {
	if  (!content) {return "";}
	var html = [];
	var idx = 0;

	var maxIndex = content.length;
	var lastIndex = 0;

    var objectHandlers = this.getHandlers();
    while (true) {
		var lowestResult = null;
		var lowestIndex = maxIndex;
		var lowestHandler = null;

		// if given a type, just go thru the handler defined for that type.
		// otherwise, go thru every handler we have. Regardless, ask each handler
		// to find us a match >= to lastIndex. Handlers that didn't match last
		// time will simply return, handlers that matched last time that we didn't
		// use (because we found a closer match) will simply return that match again.
		//
		// when we are done, we take the handler with the lowest index.
		var i;
		var handlers;
		var chunk;
		var result = null;
		if (type) {
			//DBG.println(AjxDebug.DBG3, "findObjects type [" + type + "]");
			handlers = objectHandlers[type];
			if (handlers) {
				for (i = 0; i < handlers.length; i++) {
					//DBG.println(AjxDebug.DBG3, "findObjects by TYPE (" + handlers[i] + ")");
					result = handlers[i].findObject(content, lastIndex, this);
					// No match keep trying.
					if(!result) {continue;}
					// Got a match let's handle it.
					if (result.index >= lowestIndex) {break;}
					lowestResult = result;
					lowestIndex = result.index;
					lowestHandler = handlers[i];
				}
			}
			// If it's an email address just handle it and return the result.
			if (type == "email" || content instanceof AjxEmailAddress) {
				if (lowestHandler) {
                    content = this._getAjxEmailAddress(content);
					this.generateSpan(lowestHandler, html, idx, content, lowestResult, options);
				} else {
					html[idx++] = AjxStringUtil.htmlEncode(content.toString());
				}
				return html.join("");
			}
		} else {
			for (var j = 0; j < this._allObjectHandlers.length; j++) {
				var handler = this._allObjectHandlers[j];
				//DBG.println(AjxDebug.DBG3, "findObjects trying (" + handler + ")");
				result = handler.findObject(content, lastIndex, this);
				if (result && result.index < lowestIndex) {
					lowestResult = result;
					lowestIndex = result.index;
					lowestHandler = handler;
				}
			}
		}

		if (!lowestResult) {
			// all done
			// do last chunk
			chunk = content.substring(lastIndex, maxIndex);
			if (htmlEncode) {
				html[idx++] = AjxStringUtil.htmlEncode(chunk, !!isTextMsg);
			} else {
				html[idx++] = chunk;
			}
			break;
		}

		//  add anything before the match
		if (lowestIndex > lastIndex) {
			chunk = content.substring(lastIndex, lowestIndex);
			if (htmlEncode) {
				html[idx++] = AjxStringUtil.htmlEncode(chunk, !!isTextMsg);
			} else {
				html[idx++] = chunk;
			}
		}

		// add the match
		if(lowestHandler) {
			idx = this.generateSpan(lowestHandler, html, idx, lowestResult[0], lowestResult.context);
		} else {
			html[idx++] = lowestResult[0];
		}

		// update the index
		lastIndex = lowestResult.index + (lowestResult.matchLength || lowestResult[0].length);
	}

	return html.join("");
};


/**
 * Added this customized method for the sake of ZmMailMsgView performance.
 * 
 * TODO: Integrate this method to findObjectsInNode()
 * 
 * @private
 */
ZmObjectManager.prototype.processObjectsInNode = function(doc, node){

    var objectManager = this;
	doc = doc || node.ownerDocument;
    var tmpdiv = doc.createElement("div");

    var recurse = function(node, handlers) {
		var tmp, i, val, next;
		switch (node.nodeType) {
		    case 1:	// ELEMENT_NODE
			node.normalize();
			tmp = node.tagName.toLowerCase();

			if (next == null) {
				if (/^(img|a)$/.test(tmp)) {
                    var href;
                    try {
                        // IE can throw an "Invalid Argument" error depending on value of href
                        // e.g: http://0:0:0:0:0:0:0:1%0:7070/service/soap/ContactActionRequest:1331608015326:9c4f5868c5b0b4f2
                        href = node.href;
                    }
                    catch(e) {
                        //do nothing
                    }

                    var isMailToLink = tmp === "a" && ZmMailMsgView._MAILTO_RE.test(href),
                        isUrlLink = tmp === "a" && ZmMailMsgView._URL_RE.test(href);

                    if ((isMailToLink || isUrlLink) && node.target){
						// tricky.
						var txt = isMailToLink ? href :RegExp.$1 ;
						tmp = doc.createElement("div");
						tmp.innerHTML = objectManager.findObjects(AjxStringUtil.trim(txt));
						tmp = tmp.firstChild;
						if (tmp.nodeType == 3 /* Node.TEXT_NODE */) {
							// probably no objects were found.  A warning would be OK here
							// since the regexps guarantee that objects _should_ be found.
							return tmp.nextSibling;
						}
						// here, tmp is an object span, but it
						// contains the URL (href) instead of
						// the original link text.
						node.parentNode.insertBefore(tmp, node); // add it to DOM
						tmp.innerHTML = "";
						tmp.appendChild(node); // we have the original link now
						return tmp.nextSibling;	// move on
					}
					handlers = false;
				}
			} else {
				// consider processed
				node = next;
			}

			// bug 28264: the only workaround possible seems to be
			// to remove textIndent styles that have a negative value:
			if (parseFloat(node.style.textIndent) < 0) {
				node.style.textIndent = "";
			}
            for (i = node.firstChild; i; i = recurse(i, handlers)) {}
			return node.nextSibling;

		    case 3:	// TEXT_NODE
		    case 4:	// CDATA_SECTION_NODE (just in case)
			// generate ObjectHandler-s
			if (handlers && /[^\s\xA0]/.test(node.data)) try {
 				var a = null, b = null;

				if (!AjxEnv.isIE) {
					// this block of code is supposed to free the object handlers from
					// dealing with whitespace.  However, IE sometimes crashes here, for
					// reasons that weren't possible to determine--hence we avoid this
					// step for IE.  (bug #5345)
					var results = /^[\s\xA0]+/.exec(node.data);
					if (results) {
						a = node;
						node = node.splitText(results[0].length);
					}
					results = /[\s\xA0]+$/.exec(node.data);
					if (results)
						b = node.splitText(node.data.length - results[0].length);
				}

				tmp = tmpdiv;
				var code = objectManager.findObjects(node.data, true, null, false);
				var disembowel = false;
				if (AjxEnv.isIE) {
					// Bug #6481, #4498: innerHTML in IE massacrates whitespace
					//            unless it sees a <pre> in the code.
					tmp.innerHTML = [ "<pre>", code, "</pre>" ].join("");
					disembowel = true;
				} else {
					tmp.innerHTML = code;
				}

				if (a)
					tmp.insertBefore(a, tmp.firstChild);
				if (b)
					tmp.appendChild(b);

				a = node.parentNode;
				if (disembowel)
					tmp = tmp.firstChild;
				while (tmp.firstChild)
					a.insertBefore(tmp.firstChild, node);
				tmp = node.nextSibling;
				a.removeChild(node);
				return tmp;
			} catch(ex) {};
		}
		return node.nextSibling;
	};

    // Parse through the DOM directly and find objects.
	if (node && node.childNodes && node.childNodes.length) {
		for (var i = 0; i < node.childNodes.length; i++){
			recurse(node.childNodes[i], true);
		}
	}
};

/**
 * @private
 */
ZmObjectManager.prototype.findObjectsInNode =
function(node, re_discard, re_allow, callbacks) {
	var objectManager = this, doc = node.ownerDocument, tmpdiv = doc.createElement("div");

	if (!re_discard)
		re_discard = /^(script|link|object|iframe|applet)$/i;

	// This inner function does the actual work.  BEWARE that it return-s
	// in various places, not only at the end.
	var recurse = function(node, handlers) {
		var tmp, i, val, next;
		switch (node.nodeType) {
		    case 1:	// ELEMENT_NODE
			node.normalize();
			tmp = node.tagName.toLowerCase();
			if (callbacks && callbacks.foreachElement) {
				next = callbacks.foreachElement(node, tmp, re_discard, re_allow);
			}
			if (next == null) {
				if (/^(img|a)$/.test(tmp)) {
					if (tmp == "a" && node.target
					    && (ZmMailMsgView._URL_RE.test(node.href)
						|| ZmMailMsgView._MAILTO_RE.test(node.href)))
					{
						// tricky.
						var txt = RegExp.$1;
						tmp = doc.createElement("div");
						tmp.innerHTML = objectManager.findObjects(AjxStringUtil.trim(RegExp.$1));
						tmp = tmp.firstChild;
						if (tmp.nodeType == 3 /* Node.TEXT_NODE */) {
							// probably no objects were found.  A warning would be OK here
							// since the regexps guarantee that objects _should_ be found.
							return tmp.nextSibling;
						}
						// here, tmp is an object span, but it
						// contains the URL (href) instead of
						// the original link text.
						node.parentNode.insertBefore(tmp, node); // add it to DOM
						tmp.innerHTML = "";
						tmp.appendChild(node); // we have the original link now
						return tmp.nextSibling;	// move on
					}
					handlers = false;
				} else if (re_discard.test(tmp) || (re_allow && !re_allow.test(tmp))) {
					tmp = node.nextSibling;
					node.parentNode.removeChild(node);
					return tmp;
				}
			} else {
				// consider processed
				node = next;
			}

			if (AjxEnv.isIE) {
				// strips expression()-s, bwuahahaha!
				// granted, they get lost on the server-side anyway, but assuming some get through...
				// the line below exterminates them.
				node.style.cssText = node.style.cssText;
			}

			for (i = node.firstChild; i; i = recurse(i, handlers)) {}
			return node.nextSibling;

		    case 3:	// TEXT_NODE
		    case 4:	// CDATA_SECTION_NODE (just in case)
			// generate ObjectHandler-s
			if (handlers && /[^\s\xA0]/.test(node.data)) try {
 				var a = null, b = null;

				if (!AjxEnv.isIE) {
					// this block of code is supposed to free the object handlers from
					// dealing with whitespace.  However, IE sometimes crashes here, for
					// reasons that weren't possible to determine--hence we avoid this
					// step for IE.  (bug #5345)
					var results = /^[\s\xA0]+/.exec(node.data);
					if (results) {
						a = node;
						node = node.splitText(results[0].length);
					}
					results = /[\s\xA0]+$/.exec(node.data);
					if (results)
						b = node.splitText(node.data.length - results[0].length);
				}

				tmp = tmpdiv;
				var code = objectManager.findObjects(node.data, true, null, false);
				var disembowel = false;
				if (AjxEnv.isIE) {
					// Bug #6481, #4498: innerHTML in IE massacrates whitespace
					//            unless it sees a <pre> in the code.
					tmp.innerHTML = [ "<pre>", code, "</pre>" ].join("");
					disembowel = true;
				} else {
					tmp.innerHTML = code;
				}

				if (a)
					tmp.insertBefore(a, tmp.firstChild);
				if (b)
					tmp.appendChild(b);

				a = node.parentNode;
				if (disembowel)
					tmp = tmp.firstChild;
				while (tmp.firstChild)
					a.insertBefore(tmp.firstChild, node);
				tmp = node.nextSibling;
				a.removeChild(node);
				return tmp;
			} catch(ex) {};
		}
		return node.nextSibling;
	};
	var df = doc.createDocumentFragment();
	while (node.firstChild) {
		df.appendChild(node.firstChild); // NODE now out of the displayable DOM
		recurse(df.lastChild, true, this);	 // parse tree and findObjects()
	}
	node.appendChild(df);	// put nodes back in the document
};

/**
 * Sets handler attribute.
 * 
 * @param	{String}	type		the type
 * @param	{String}	name		the attribute name
 * @param	{Object}	value		the value
 */
ZmObjectManager.prototype.setHandlerAttr =
function(type, name, value) {
    var handlers = this.getHandlers()[type];
	if (handlers) {
		for (var i = 0; i < handlers.length; i++) {
			handlers[i][name] = value;
		}
	}
};

/**
 * Generates the span.
 * 
 * @private
 */
ZmObjectManager.prototype.generateSpan =
function(handler, html, idx, obj, context, options) {
	var id = this._objectIdPrefix + Dwt.getNextId();
    if (handler && handler.name) {
        id = id + "_" + handler.name;
    }
	this._objects[id] = {object: obj, handler: handler, id: id, context: context };
	return handler.generateSpan(html, idx, obj, id, context, options);
};

/**
 * @private
 */
ZmObjectManager.prototype._findObjectSpan =
function(e) {
	while (e && (!e.id || e.id.indexOf(this._objectIdPrefix) !== 0)) {
		e = e.parentNode;
	}
	return e;
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseOverListener =
function(ev) {
	var span = this._findObjectSpan(ev.target);
	if (!span) {return false;}
	var object = this._objects[span.id];
	if (!object) {return false;}

	span.className = object.handler.getHoveredClassName(object.object, object.context, span.id);
	if (object.handler.hasToolTipText()) {
		var shell = DwtShell.getShell(window);
		var manager = shell.getHoverMgr();
		if ((!manager.isHovering() || manager.getHoverObject() != object) && !DwtMenu.menuShowing()) {
			manager.reset();
			manager.setHoverOverDelay(ZmObjectManager._TOOLTIP_DELAY);
			manager.setHoverObject(object);
			manager.setHoverOverData(object);
			manager.setHoverOverListener(this._hoverOverListener);
			manager.hoverOver(ev.docX, ev.docY);
			ev.hoverStarted = true;
		}
	}

	ev._returnValue = true;
	ev._dontCallPreventDefault = true;
	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseOutListener =
function(ev) {
	var span = this._findObjectSpan(ev.target);
	var object = span ? this._objects[span.id] : null;

	if (object) {
		span.className = object.handler.getClassName(object.object, object.context, span.id);
		var shell = DwtShell.getShell(window);
		var manager = shell.getHoverMgr();
		manager.setHoverOutDelay(150);
		manager.setHoverOutData(object);
		manager.setHoverOutListener(this._hoverOutListener);
		manager.hoverOut();
	}

	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseMoveListener =
function(ev) {
	ev._returnValue = true;
	ev._dontCallPreventDefault = true;
	ev._stopPropagation = true;
	var span = this._findObjectSpan(ev.target);
	var object = span ? this._objects[span.id] : null;

	if (object) {
		var shell = DwtShell.getShell(window);
		var manager = shell.getHoverMgr();
		if (!manager.isHovering()) {
			// NOTE: mouseOver already init'd hover settings
			manager.hoverOver(ev.docX, ev.docY);
		}
	}

	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._rightClickListener =
function(ev) {
	ev.button = DwtMouseEvent.RIGHT;
	return this._mouseDownListener(ev);
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseDownListener =
function(ev) {

	// "authoritative" means a previous listener doesn't want propagation to get reset
	if (!ev._authoritative) {
		ev._dontCallPreventDefault = true;
		ev._returnValue = true;
		ev._stopPropagation = false;
	}

	var span = this._findObjectSpan(ev.target);
	if (!span) {
		return true;
	}
	var object = this._objects[span.id];
	if (!object) {
		return true;
	}

	ev._stopPropagation = true;

	var shell = DwtShell.getShell(window);
	var manager = shell.getHoverMgr();
	manager.setHoverOutDelay(0);
	manager.setHoverOutData(object);
	manager.setHoverOutListener(this._hoverOutListener);
	manager.hoverOut();

	span.className = object.handler.getActiveClassName(object.object, object.context, span.id);
	if (ev.button == DwtMouseEvent.RIGHT) {
		var menu = object.handler.getActionMenu(object.object, span, object.context, ev);
		if (menu) {
			menu.popup(0, ev.docX, ev.docY);
			// if we have an action menu, don't let the browser show its context menu too
			ev._dontCallPreventDefault = false;
			ev._returnValue = false;
			ev._stopPropagation = true;
			return true;
		}
	} else if (ev.button == DwtMouseEvent.LEFT) {
		if (this._selectCallback) {
			this._selectCallback.run();
		}
		object.handler.selected(object.object, span, ev, object.context);
		return true;
	}
	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseUpListener =
function(ev) {
	ev._returnValue = true;
	ev._dontCallPreventDefault = true;
	ev._stopPropagation = true;
	var span = this._findObjectSpan(ev.target);
	if (!span) {return false;}
	var object = this._objects[span.id];
	if (!object) {return false;}

	span.className = object.handler.getHoveredClassName(object.object, object.context, span.id);
	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._handleHoverOver =
function(event) {
	if (!(event && event.object)) { return; }

	var span = this._findObjectSpan(event.target);
	var handler = event.object.handler;
	var object = event.object.object;
	var context = event.object.context;
	var id = event.object.id;
	var x = event.x;
	var y = event.y;

	handler.hoverOver(object, context, x, y, span, id);
};

/**
 * @private
 */
ZmObjectManager.prototype._handleHoverOut =
function(event) {
	if (!(event && event.object)) { return; }

	var span = this._findObjectSpan(event.target);
	var handler = event.object.handler;
	var object = event.object.object;
	var context = event.object.context;
	var id = event.object.id;

	handler.hoverOut(object, context, span, id);
};

// Private static functions

/**
 * @private
 */
ZmObjectManager.__byPriority =
function(a, b) {
	return (b._prio < a._prio) - (a._prio < b._prio);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmSettings")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines the settings class.
 */

/**
 * Creates a collection of settings with default values. If no app context is given,
 * then this is a skeletal, non-live version of settings which can provide default
 * settings and parse SOAP settings.
 * @class
 * This class is a collection of various sorts of settings: config values, preferences,
 * and COS features. Each setting has an ID which can be used to retrieve it.
 *
 * @author Conrad Damon
 *
 * @param {Boolean}	noInit	if <code>true</code>, skip initialization
 *
 * @extends		ZmModel
 */
ZmSettings = function(noInit) {

	ZmModel.call(this, ZmEvent.S_SETTINGS);

	this._settings = {};	// settings by ID
	this._nameToId = {};	// map to get from server setting name to setting ID

	this.getInfoResponse = null; // Cached GetInfoResponse for lazy creation of identities, etc.
	this._handleImplicitChange = new AjxListener(this, this._implicitChangeListener);

	if (!noInit) {
        this.initialize();
	}
};

ZmSettings.prototype = new ZmModel;
ZmSettings.prototype.constructor = ZmSettings;

ZmSettings.BASE64_TO_NORMAL_RATIO = 1.34;

/**
 * Creates a new setting and adds it to the settings.
 *
 * @param {String}		id			the unique ID of the setting
 * @param {Hash}	params		a hash of parameters
 * @param {String}	params.name			the name of the pref or attr on the server
 * @param {constant}	params.type		config, pref, or COS
 * @param {constant}	params.dataType	string, int, or boolean (defaults to string)
 * @param {Object}	params.defaultValue	the default value
 */
ZmSettings.prototype.registerSetting =
function(id, params) {
	ZmSetting[id] = id;
	var setting = this._settings[id] = new ZmSetting(id, params);
	if (params.name) {
		this._nameToId[params.name] = id;
	}
	if (params.isImplicit) {
		setting.addChangeListener(this._handleImplicitChange);
	}
	return setting;
};

/**
 * Returns a string representation of the object.
 *
 * @return		{String}		a string representation of the object
 */
ZmSettings.prototype.toString =
function() {
	return "ZmSettings";
};

/**
 * Initializes the settings.
 *
 */
ZmSettings.prototype.initialize =
function() {
	this._initialize();
	this._setDefaults();
	this.userSettingsLoaded = false;

	// set listeners for settings
	var listener = new AjxListener(this, this._changeListener);
	if (!appCtxt.multiAccounts) {
		this.getSetting(ZmSetting.QUOTA_USED).addChangeListener(listener);
	}
	this.getSetting(ZmSetting.POLLING_INTERVAL).addChangeListener(listener);
	this.getSetting(ZmSetting.SKIN_NAME).addChangeListener(listener);
	this.getSetting(ZmSetting.SHOW_SELECTION_CHECKBOX).addChangeListener(listener);
	this.getSetting(ZmSetting.LOCALE_NAME).addChangeListener(listener);
	this.getSetting(ZmSetting.FONT_NAME).addChangeListener(listener);
	this.getSetting(ZmSetting.FONT_SIZE).addChangeListener(listener);
	this.getSetting(ZmSetting.SHORTCUTS).addChangeListener(listener);
	this.getSetting(ZmSetting.CHILD_ACCTS_VISIBLE).addChangeListener(listener);
	this.getSetting(ZmSetting.ATTACHMENTS_BLOCKED).addChangeListener(listener);
	this.getSetting(ZmSetting.DISPLAY_TIME_IN_MAIL_LIST).addChangeListener(listener);


	if (appCtxt.isOffline) {
		this.getSetting(ZmSetting.OFFLINE_IS_MAILTO_HANDLER).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_BACKUP_ACCOUNT_ID).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_BACKUP_INTERVAL).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_BACKUP_PATH).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_BACKUP_KEEP).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_UPDATE_NOTIFY).addChangeListener(listener);
	}
};

/**
 * Gets the value of the given setting.
 *
 * @param {String}	id		the ID of the setting
 * @param	{String}	key		the key
 * @return	{Object}	the value or <code>null</code> for none
 */
ZmSettings.prototype.get =
function(id, key) {
	return (id && this._settings[id]) ? this._settings[id].getValue(key) : null;
};

/**
 * Sets the value of the given setting.
 *
 * @param {String}	id		the ID of the setting
 * @param {Object}	value			the new value for the setting
 * @param {String}	key 			optional key for use by hash table data type
 * @param {Boolean}	setDefault		if <code>true</code>, also set the default value
 * @param {Boolean}	skipNotify		if <code>true</code>, do not notify listeners
 * @param {Boolean}	skipImplicit		if <code>true</code>, do not check for change to implicit pref
 */
ZmSettings.prototype.set = function(id, value, key, setDefault, skipNotify, skipImplicit) {
	if (id && this._settings[id]) {
		this._settings[id].setValue(value, key, setDefault, skipNotify, skipImplicit);
	}
	else {
		DBG.println(AjxDebug.DBG1, "ZmSettings.set: ID missing (value = " + value);
	}
};

/**
 * Gets the setting.
 *
 * @param {String}	id		the ID of the setting
 * @return	{ZmSetting}	the setting
 */
ZmSettings.prototype.getSetting =
function(id) {
	return this._settings[id];
};

/**
 * Populates settings values.
 *
 * @param {Hash}	list		a hash of preference or attribute values
 */
ZmSettings.prototype.createFromJs =
function(list, setDefault, skipNotify, skipImplicit) {
    // default skipImplicit value is true
    skipImplicit = skipImplicit == null || skipImplicit; 

	for (var i in list) {
		var val = list[i];
		var setting = this._settings[this._nameToId[i]];
		if (setting) {
			if (setting.dataType == ZmSetting.D_HASH) {
				var pairs = val.split(",");
				var value = {};
				for (var j = 0; j < pairs.length; j++) {
					var fields = pairs[j].split(":");
					
					// In case of shared folder there can be more than two element in
					// fields array as shared folder id can be "859095ad-90d5-4db3-919d-630ac30de5a9:465"
					// so consider all the elements as key except last item of the fields array.
					if(setting.name === "zimbraPrefFoldersExpanded" && fields.length > 2) {
						var fieldValue = fields.pop();
						var fieldKey = fields.join(':');
						value[fieldKey] = fieldValue;
					} else {
						value[fields[0]] = fields[1];
					}
				}
				val = value;
			}
			setting.setValue(val, null, setDefault, skipNotify, skipImplicit);
			if (ZmSetting.IS_IMPLICIT[setting.id]) {
				setting.origValue = setting.copyValue();
			}
		} else {
			DBG.println(AjxDebug.DBG3, "*** Unrecognized setting: " + i);
		}
	}
};

/**
 * Gets the setting that is associated with the given server-side setting, if any.
 *
 * @param {String}	name	the server-side setting name (for example, "zimbraFeatureContactsEnabled")
 * @return	{String}	the setting id
 */
ZmSettings.prototype.getSettingByName =
function(name) {
	return this._nameToId[name];
};

/**
 * Checks if the given ID was received from the server. Use this method
 * to determine whether this ID is supported by a ZCS server. Currently used by
 * ZDesktop since it can "talk" to both v5 and v6 ZCS.
 *
 * @param {String}	id	the setting ID
 * @return	{Boolean}	<code>true</code> if the attribute is supported
 */
ZmSettings.prototype.attrExists =
function(id) {
	var name = this.getSetting(id).name;
	return (this.getInfoResponse.prefs._attrs[name] ||
			this.getInfoResponse.attrs._attrs[name]);
};

/**
 * Retrieves the preferences, COS settings, and metadata for the current user.
 * All the data gets stored into the settings collection.
 *
 * @param {AjxCallback}	callback 			the callback to run after response is received
 * @param {AjxCallback}	errorCallback 	the callback to run error is received
 * @param {String}	accountName		the name of account to load settings for
 * @param {Object}	response			the pre-determined JSON response object
 * @param {ZmBatchCommand}	batchCommand		set if part of a batch request
 */
ZmSettings.prototype.loadUserSettings =
function(callback, errorCallback, accountName, response, batchCommand) {
	var args = [callback, accountName];

	var soapDoc = AjxSoapDoc.create("GetInfoRequest", "urn:zimbraAccount");
	soapDoc.setMethodAttribute("rights", "createDistList"); //not sure when this is called, but let's add it anyway. (on login it's called from within launchZCS.JSP calling GetInfoJSONTag.java
	var respCallback = this._handleResponseLoadUserSettings.bind(this, args);
	if (batchCommand) {
		batchCommand.addNewRequestParams(soapDoc, respCallback);
	}
	else {
		var params = {
			soapDoc: (response ? null : soapDoc),
			accountName: accountName,
			asyncMode: true,
			callback: respCallback,
			errorCallback: errorCallback,
			response: response
		};
		appCtxt.getAppController().sendRequest(params);
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseLoadUserSettings =
function(callback, accountName, result) {
    this.setUserSettings(result.getResponse().GetInfoResponse, accountName);
    this.userSettingsLoaded = true;
    if (callback) {
        callback.run(result);
    }
};

/**
 * Sets the user settings.
 *
 * @param {hash}    params
 * @param {object}  params.info             The GetInfoResponse object.
 * @param {string}  [params.accountName]    The name of the account.
 * @param {boolean} [params.setDefault]     Set default value
 * @param {boolean} [params.skipNotify]     Skip change notification
 * @param {boolean} [params.skipImplicit]   Skip implicit changes
 * @param {boolean} [params.preInit]        Only init base settings for startup
 */
ZmSettings.prototype.setUserSettings = function(params) {

    params = Dwt.getParams(arguments, ["info", "accountName", "setDefault", "skipNotify", "skipImplicit", "preInit"]);

    var info = this.getInfoResponse = params.info;

	appCtxt.createDistListAllowed = false;
	appCtxt.createDistListAllowedDomains = [];
	appCtxt.createDistListAllowedDomainsMap = {};
	var rightTargets = info.rights && info.rights.targets;
	if (rightTargets) {
		for (var i = 0; i < rightTargets.length; i++) {
			var target = rightTargets[i];
			if (target.right == "createDistList") {
				if (target.target[0].type == "domain") {
					appCtxt.createDistListAllowed = true;
					appCtxt.createDistListAllowedDomains.push(target.target[0].name);
					appCtxt.createDistListAllowedDomainsMap[target.target[0].name] = true;
					break;
				}
			}

		}
	}

	// For delegated admin who can see prefs but not mail, we still need to register the mail settings so
	// they can be created and set below in createFromJs().
	if (this.get(ZmSetting.ADMIN_DELEGATED)) {
		if (!this.get(ZmSetting.ADMIN_MAIL_ENABLED) && this.get(ZmSetting.ADMIN_PREFERENCES_ENABLED)) {
			this.getMailAppForDelegatedAdmin().enableMailPrefs();
		}
	}

	// Voice feature
    this.set(ZmSetting.VOICE_ENABLED, this._hasVoiceFeature(), null, false, true);

    var accountName = params.accountName;
    var setDefault = params.preInit ? false : params.setDefault;
    var skipNotify = params.preInit ? true : params.skipNotify;
    var skipImplicit = params.preInit ? true : params.skipImplicit;

    var settings = [
        ZmSetting.ADMIN_DELEGATED,          info.adminDelegated,
        ZmSetting.MESSAGE_SIZE_LIMIT,    this._base64toNormalSize(info.attSizeLimit),
        ZmSetting.CHANGE_PASSWORD_URL,      info.changePasswordURL,
        ZmSetting.DOCUMENT_SIZE_LIMIT,      this._base64toNormalSize(info.docSizeLimit),
        ZmSetting.LAST_ACCESS,              info.accessed,
        ZmSetting.LICENSE_STATUS,           info.license && info.license.status,
        ZmSetting.PREVIOUS_SESSION,         info.prevSession,
        ZmSetting.PUBLIC_URL,               info.publicURL,
		ZmSetting.ADMIN_URL,                info.adminURL,
        ZmSetting.QUOTA_USED,               info.used,
        ZmSetting.RECENT_MESSAGES,          info.recent,
        ZmSetting.REST_URL,                 info.rest,
        ZmSetting.USERNAME,                 info.name,
		ZmSetting.EMAIL_VALIDATION_REGEX, 	info.zimbraMailAddressValidationRegex,
		ZmSetting.HAB_ROOT,                 (info.habRoots && info.habRoots.hab ? info.habRoots.hab : false),
		ZmSetting.SPELL_CHECK_ENABLED, 		info.isSpellCheckAvailable,
		ZmSetting.DISABLE_SENSITIVE_ZIMLETS_IN_MIXED_MODE, 	(info.domainSettings && info.domainSettings.zimbraZimletDataSensitiveInMixedModeDisabled ? info.domainSettings.zimbraZimletDataSensitiveInMixedModeDisabled : "FALSE")
    ];
    for (var i = 0; i < settings.length; i += 2) {
        var value = settings[i+1];
        if (value != null) {
            this.set(settings[i], value, null, setDefault, skipNotify, skipImplicit);
        }
    }

    // features and other settings
    if (info.attrs && info.attrs._attrs) {
        this.createFromJs(info.attrs._attrs, setDefault, skipNotify, skipImplicit);
    }

	// By default, everything but mail is enabled for delegated admin. Require an additional setting to allow admin to view mail.
	if (this.get(ZmSetting.ADMIN_DELEGATED)) {
		this.set(ZmSetting.MAIL_ENABLED, this.get(ZmSetting.ADMIN_MAIL_ENABLED), setDefault, skipNotify, skipImplicit);
		var enableMailPrefs = this.get(ZmSetting.MAIL_ENABLED) || (this.get(ZmSetting.ADMIN_DELEGATED) && this.get(ZmSetting.ADMIN_PREFERENCES_ENABLED));
		this.set(ZmSetting.MAIL_PREFERENCES_ENABLED, enableMailPrefs, setDefault, skipNotify, skipImplicit);
		// Disable other areas where mail could be exposed to a prefs-only admin
		if (this.get(ZmSetting.MAIL_PREFERENCES_ENABLED) && !this.get(ZmSetting.MAIL_ENABLED)) {
			var result = { handled: false };
			appCtxt.notifyZimlets("onZmSettings_setUserSettings", [this, setDefault, skipNotify, skipImplicit, result]);
			if (!result.handled) {
				this.set(ZmSetting.MAIL_FORWARDING_ENABLED, false, setDefault, skipNotify, skipImplicit);
				this.set(ZmSetting.FILTERS_MAIL_FORWARDING_ENABLED, false, setDefault, skipNotify, skipImplicit);
				this.set(ZmSetting.NOTIF_FEATURE_ENABLED, false, setDefault, skipNotify, skipImplicit);
			}
		}
	}

	// Server may provide us with SSO-enabled URL for Community integration (integration URL with OAuth signature)
	if (info.communityURL) {
		this.set(ZmSetting.SOCIAL_EXTERNAL_URL, info.communityURL, null, setDefault, skipNotify);
	}

	if (params.preInit) {
	    return;
    }

    // preferences
    if (info.prefs && info.prefs._attrs) {
        this.createFromJs(info.prefs._attrs, setDefault, skipNotify, skipImplicit);
    }

    // accounts
	var setting;
	if (!accountName) {
		// NOTE: only the main account can have children
		appCtxt.accountList.createAccounts(this, info);

		// for offline, find out whether this client supports prism-specific features
		if (appCtxt.isOffline) {
			if (AjxEnv.isPrism && window.platform) {
				this.set(ZmSetting.OFFLINE_SUPPORTS_MAILTO, true, null, setDefault, skipNotify, skipImplicit);
				this.set(ZmSetting.OFFLINE_SUPPORTS_DOCK_UPDATE, true, null, setDefault, skipNotify, skipImplicit);
			}

			// bug #45804 - sharing always enabled for offline
			appCtxt.set(ZmSetting.SHARING_ENABLED, true, null, setDefault, skipNotify);
		}
	}

	// handle settings whose values may depend on other settings
	setting = this._settings[ZmSetting.REPLY_TO_ADDRESS];
	if (setting) {
		setting.defaultValue = this.get(ZmSetting.USERNAME);
	}
	if (this.get(ZmSetting.FORCE_CAL_OFF)) {
		this.set(ZmSetting.CALENDAR_ENABLED, false, null, setDefault, skipNotify, skipImplicit);
	}

	if (!this.get(ZmSetting.OPTIONS_ENABLED)) {
		this.set(ZmSetting.FILTERS_ENABLED, false, null, setDefault, skipNotify, skipImplicit);
	}

	// load zimlets *only* for the main account
	if (!accountName) {
		if (info.zimlets && info.zimlets.zimlet) {
            if (this.get(ZmSetting.ZIMLETS_SYNCHRONOUS)) {
                var action = new AjxTimedAction(this, this._beginLoadZimlets, [info.zimlets.zimlet, info.props.prop, true]);
                AjxTimedAction.scheduleAction(action, 0);
            } else {
                var listener = new AjxListener(this, this._beginLoadZimlets, [info.zimlets.zimlet, info.props.prop, false]);
                appCtxt.getAppController().addListener(ZmAppEvent.POST_STARTUP, listener);
            }
		} else {
			appCtxt.allZimletsLoaded();
		}
	}

	var value = appCtxt.get(ZmSetting.REPLY_INCLUDE_ORIG);
	if (value) {
		var list = ZmMailApp.INC_MAP[value];
		appCtxt.set(ZmSetting.REPLY_INCLUDE_WHAT, list[0], null, setDefault, skipNotify);
		appCtxt.set(ZmSetting.REPLY_USE_PREFIX, list[1], null, setDefault, skipNotify);
		appCtxt.set(ZmSetting.REPLY_INCLUDE_HEADERS, list[2], null, setDefault, skipNotify);
	}

	var value = appCtxt.get(ZmSetting.FORWARD_INCLUDE_ORIG);
	if (value) {
		var list = ZmMailApp.INC_MAP[value];
		appCtxt.set(ZmSetting.FORWARD_INCLUDE_WHAT, list[0], null, setDefault, skipNotify);
		appCtxt.set(ZmSetting.FORWARD_USE_PREFIX, list[1], null, setDefault, skipNotify);
		appCtxt.set(ZmSetting.FORWARD_INCLUDE_HEADERS, list[2], null, setDefault, skipNotify);
	}

    // Populate Sort Order Defaults
    var sortPref =  ZmSettings.DEFAULT_SORT_PREF;
    sortPref[ZmId.VIEW_CONVLIST]			= ZmSearch.DATE_DESC;
    sortPref[ZmId.VIEW_CONV]				= ZmSearch.DATE_DESC;
    sortPref[ZmId.VIEW_TRAD]				= ZmSearch.DATE_DESC;
    sortPref[ZmId.VIEW_CONTACT_SRC]			= ZmSearch.NAME_ASC;
    sortPref[ZmId.VIEW_CONTACT_TGT]			= ZmSearch.NAME_ASC;
    sortPref[ZmId.VIEW_CONTACT_SIMPLE]		= ZmSearch.NAME_ASC;
    sortPref[ZmId.VIEW_CAL]					= ZmSearch.DATE_ASC;
    sortPref[ZmId.VIEW_TASKLIST]			= ZmSearch.DUE_DATE_ASC;
    sortPref[ZmId.VIEW_BRIEFCASE_DETAIL]	= ZmSearch.SUBJ_ASC;

    var sortOrderSetting = this._settings[ZmSetting.SORTING_PREF];
    if (sortOrderSetting) {
        // Populate empty sort pref's with defaultValues
        for (var pref in sortPref){
            if (!sortOrderSetting.getValue(pref)){
                sortOrderSetting.setValue(sortPref[pref], pref, false, true);
            }
        }

        // Explicitly Set defaultValue
        sortOrderSetting.defaultValue = AjxUtil.hashCopy(sortPref);
    }
	
	DwtControl.useBrowserTooltips = this.get(ZmSetting.BROWSER_TOOLTIPS_ENABLED);

	this._updateUserFontPrefsRule();
};


ZmSettings.prototype._base64toNormalSize =
function(base64) {
	if (!base64 || base64 === -1) { //-1 is unlimited
		return base64;
	}
	return base64 / ZmSettings.BASE64_TO_NORMAL_RATIO;
};


/**
 * @private
 */
ZmSettings.prototype._beginLoadZimlets =
function(zimlet, prop, sync) {
	var zimletsCallback = new AjxCallback(this, this._loadZimletPackage, [zimlet, prop, sync]);
	AjxDispatcher.require(["Startup2"], false, zimletsCallback);
};

ZmSettings.prototype._loadZimletPackage =
function(zimlet, prop, sync) {
	var zimletsCallback = new AjxCallback(this, this._loadZimlets, [zimlet, prop, sync]);
	AjxDispatcher.require("Zimlet", false, zimletsCallback);
};

/**
 * @private
 */
ZmSettings.prototype._loadZimlets =
function(allZimlets, props, sync) {

	allZimlets = allZimlets || [];
	this.registerSetting("ZIMLETS",		{type:ZmSetting.T_CONFIG, defaultValue:allZimlets, isGlobal:true});
	this.registerSetting("USER_PROPS",	{type:ZmSetting.T_CONFIG, defaultValue:props});

	var zimlets = this._getCheckedZimlets(allZimlets);

	DBG.println(AjxDebug.DBG1, "Zimlets - Loading " + zimlets.length + " Zimlets");
	var zimletMgr = appCtxt.getZimletMgr();
	zimletMgr.loadZimlets(zimlets, props, null, null, sync);

	if (zimlets && zimlets.length) {
		var activeApp = appCtxt.getCurrentApp();
		if (activeApp) {
			var overview;
			if (appCtxt.multiAccounts) {
				var containerId = activeApp.getOverviewContainer().containerId;
				var zimletLabel = ZmOrganizer.LABEL[ZmOrganizer.ZIMLET];
				var overviewId = [containerId, zimletLabel].join("_");
				overview = appCtxt.getOverviewController().getOverview(overviewId);
			} else {
				overview = activeApp.getOverview();
			}
		}

		// update overview tree
		if (overview) {
			if (overview.appName != ZmApp.PREFERENCES) {
				overview.setTreeView(ZmOrganizer.ZIMLET);
			}

			// HACK: for multi-account, hide the zimlet section if no panel zimlets
			if (appCtxt.multiAccounts && zimletMgr.getPanelZimlets().length == 0) {
				activeApp.getOverviewContainer().removeZimletSection();
			}
		}

		// create global portlets
		if (appCtxt.get(ZmSetting.PORTAL_ENABLED)) {
			var portletMgr = appCtxt.getApp(ZmApp.PORTAL).getPortletMgr();
			var portletIds = portletMgr.createPortlets(true);
		}
	}
};

/**
 * Filters a list of zimlets, returned ones that are checked.
 *
 * @param zimlets			[array]		list of zimlet objects
 *
 * @private
 */
ZmSettings.prototype._getCheckedZimlets =
function(allZimlets) {

	var zimlets = [];
	for (var i = 0; i < allZimlets.length; i++) {
		var zimletObj = allZimlets[i];
		if (zimletObj.zimletContext[0].presence != "disabled") {
			zimlets.push(zimletObj);
		}
	}

	return zimlets;
};

/**
 * Loads the preference data.
 *
 * @param	{AjxCallback}	callback		the callback
 */
ZmSettings.prototype.loadPreferenceData =
function(callback) {
	// force main account (in case multi-account) since locale/skins are global
	var command = new ZmBatchCommand(null, appCtxt.accountList.mainAccount.name);

	var skinDoc = AjxSoapDoc.create("GetAvailableSkinsRequest", "urn:zimbraAccount");
	var skinCallback = new AjxCallback(this, this._handleResponseLoadAvailableSkins);
	command.addNewRequestParams(skinDoc, skinCallback);

	var localeDoc = AjxSoapDoc.create("GetAvailableLocalesRequest", "urn:zimbraAccount");
	var localeCallback = new AjxCallback(this, this._handleResponseGetAllLocales);
	command.addNewRequestParams(localeDoc, localeCallback);

	var csvFormatsDoc = AjxSoapDoc.create("GetAvailableCsvFormatsRequest", "urn:zimbraAccount");
	var csvFormatsCallback = new AjxCallback(this, this._handleResponseGetAvailableCsvFormats);
	command.addNewRequestParams(csvFormatsDoc, csvFormatsCallback);

	command.run(callback);
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseLoadAvailableSkins =
function(result) {
	var resp = result.getResponse().GetAvailableSkinsResponse;
	var skins = resp.skin;
	if (skins && skins.length) {
		var setting = appCtxt.accountList.mainAccount.settings.getSetting(ZmSetting.AVAILABLE_SKINS);
		for (var i = 0; i < skins.length; i++) {
			// always save available skins on the main account (in case multi-account)
			setting.setValue(skins[i].name);
		}
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseGetAllLocales =
function(response) {
	var locales = response._data.GetAvailableLocalesResponse.locale;
	if (locales && locales.length) {
		for (var i = 0, count = locales.length; i < count; i++) {
			var locale = locales[i];
			// bug: 38038
			locale.id = locale.id.replace(/^in/,"id");
			ZmLocale.create(locale.id, locale.name, ZmMsg["localeName_" + locale.id] || locale.localName);
		}
        if (locales.length === 1) {
            //Fix for bug# 80762 - Set the value to always true in case of only one language/locale present
            this.set(ZmSetting.LOCALE_CHANGE_ENABLED, true);
        }
        else {
            this.set(ZmSetting.LOCALE_CHANGE_ENABLED, ZmLocale.hasChoices());
        }
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseGetAvailableCsvFormats =
function(result){
	var formats = result.getResponse().GetAvailableCsvFormatsResponse.csv;
	if (formats && formats.length) {
		var setting = appCtxt.accountList.mainAccount.settings.getSetting(ZmSetting.AVAILABLE_CSVFORMATS);
		for (var i = 0; i < formats.length; i++) {
			setting.setValue(formats[i].name);
		}
	}
};

/**
 * Saves one or more settings.
 *
 * @param {Array}		list			a list of {ZmSetting} objects
 * @param {AjxCallback}	callback		the callback to run after response is received
 * @param {ZmBatchCommand}	batchCommand	the batch command
 * @param {ZmZimbraAccount}	account		the account to save under
 * @param {boolean}			isImplicit	if true, we are saving implicit settings
 */
ZmSettings.prototype.save =
function(list, callback, batchCommand, account, isImplicit) {
	if (!(list && list.length)) { return; }

	var acct = account || appCtxt.getActiveAccount();
	var soapDoc = AjxSoapDoc.create("ModifyPrefsRequest", "urn:zimbraAccount");
	var gotOne = false;
	var metaData = [], done = {}, setting;
	for (var i = 0; i < list.length; i++) {
		setting = list[i];
        if (done[setting.id]) { continue; }
		if (setting.type == ZmSetting.T_METADATA) {
			metaData.push(setting);
			// update the local meta data
			acct.metaData.update(setting.section, setting.name, setting.getValue());
			continue;
		} else if (setting.type != ZmSetting.T_PREF) {
			DBG.println(AjxDebug.DBG1, "*** Attempt to modify non-pref: " + setting.id + " / " + setting.name);
			continue;
		}
		if (!setting.name) {
			DBG.println(AjxDebug.DBG2, "Modify internal pref: " + setting.id);
			continue;
		}
		if (setting.dataType == ZmSetting.D_LIST) {
			// LDAP supports multi-valued attrs, so don't serialize list
			var value = setting.getValue();
			if (value && value.length) {
				for (var j = 0; j < value.length; j++) {
					var node = soapDoc.set("pref", value[j]);
					node.setAttribute("name", setting.name);
				}
			} else {
				var node = soapDoc.set("pref", "");
				node.setAttribute("name", setting.name);
			}
		} else {
			var value = setting.getValue(null, true);
			var node = soapDoc.set("pref", value);
			node.setAttribute("name", setting.name);
		}

        done[setting.id] = true;
		gotOne = true;
	}

    // bug: 50668 if the setting is implicit and global, use main Account
    if(appCtxt.isOffline && ZmSetting.IS_IMPLICIT[setting.id] && ZmSetting.IS_GLOBAL[setting.id]) {
        acct = appCtxt.accountList.mainAccount;
    }

	if (metaData.length > 0) {
		var metaDataCallback = new AjxCallback(this, this._handleResponseSaveMetaData, [metaData]);
		var sections = [ZmSetting.M_IMPLICIT, ZmSetting.M_OFFLINE];
		acct.metaData.save(sections, metaDataCallback);
	}

	if (gotOne) {
		var respCallback;
		if (callback || batchCommand) {
			respCallback = new AjxCallback(this, this._handleResponseSave, [list, callback]);
		}
		if (batchCommand) {
			batchCommand.addNewRequestParams(soapDoc, respCallback);
		} else {
			appCtxt.getAppController().sendRequest({soapDoc:soapDoc, asyncMode:true, callback:respCallback,
			 										accountName:acct.name, noBusyOverlay:isImplicit});
		}
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseSaveMetaData =
function(list, result) {
	for (var i = 0; i < list.length; i++) {
		var setting = list[i];
		if (!ZmSetting.IS_IMPLICIT[setting.id]) {
			setting.origValue = setting.copyValue();
			setting._notify(ZmEvent.E_MODIFY);
		}
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseSave =
function(list, callback, result) {
	var resp = result.getResponse();
	if (resp.ModifyPrefsResponse != null) {
		// notify each changed setting's listeners
		for (var i = 0; i < list.length; i++) {
			var setting = list[i];
			setting.origValue = setting.copyValue();
			if (!ZmSetting.IS_IMPLICIT[setting.id]) {
				setting._notify(ZmEvent.E_MODIFY);
			}
		}
		// notify any listeners on the settings as a whole
		this._notify(ZmEvent.E_MODIFY, {settings:list});
	}

	if (callback) {
		callback.run(result);
	}
};

ZmSettings.DEFAULT_SORT_PREF = {};

/**
 * Set defaults which are determined dynamically (which can't be set in static code).
 *
 * @private
 */
ZmSettings.prototype._setDefaults =
function() {

	var value = AjxUtil.formatUrl({host:location.hostname, path:"/service/soap/", qsReset:true});
	this.set(ZmSetting.CSFE_SERVER_URI, value, null, false, true);

	// CSFE_MSG_FETCHER_URI
	value = AjxUtil.formatUrl({host:location.hostname, path:"/service/home/~/", qsReset:true, qsArgs:{auth:"co"}});
	this.set(ZmSetting.CSFE_MSG_FETCHER_URI, value, null, false, true);

	// CSFE_UPLOAD_URI
	value = AjxUtil.formatUrl({host:location.hostname, path:"/service/upload", qsReset:true, qsArgs:{lbfums:""}});
	this.set(ZmSetting.CSFE_UPLOAD_URI, value, null, false, true);

	// CSFE_ATTACHMENT_UPLOAD_URI
	value = AjxUtil.formatUrl({host:location.hostname, path:"/service/upload", qsReset:true});
	this.set(ZmSetting.CSFE_ATTACHMENT_UPLOAD_URI, value, null, false, true);

	// CSFE EXPORT URI
	value = AjxUtil.formatUrl({host:location.hostname, path:"/service/home/~/", qsReset:true, qsArgs:{auth:"co", id:"{0}", fmt:"csv"}});
	this.set(ZmSetting.CSFE_EXPORT_URI, value, null, false, true);

	var h = location.hostname;
	var isDev = ((h.indexOf(".zimbra.com") != -1) || (window.appDevMode && (/\.local$/.test(h) || (!appCtxt.isOffline && h == "localhost"))));
	this.set(ZmSetting.IS_DEV_SERVER, isDev);
	if (isDev || window.isScriptErrorOn) {
		this.set(ZmSetting.SHOW_SCRIPT_ERRORS, true, null, false, true);
	}

	this.setReportScriptErrorsSettings(AjxException, ZmController.handleScriptError);
};

ZmSettings.prototype.persistImplicitSortPrefs =
function(id){
    return ZmSettings.DEFAULT_SORT_PREF[id];
};

/**
 * sets AjxException static attributes. This is extracted so it can be called from ZmNewwindow as well.
 * this is since the child window gets its own AjxException variable.
 *
 * @param AjxExceptionClassVar
 * @param handler
 */
ZmSettings.prototype.setReportScriptErrorsSettings =
function(AjxExceptionClassVar, handler) {
	// script error reporting
	var rse = AjxExceptionClassVar.reportScriptErrors = this._settings[ZmSetting.SHOW_SCRIPT_ERRORS].getValue();
	if (rse) {
		AjxExceptionClassVar.setScriptErrorHandler(handler);
	}

};

/**
 * Loads the standard settings and their default values.
 *
 * @private
 */
ZmSettings.prototype._initialize =
function() {
	// CONFIG SETTINGS
    this.registerSetting("ADMIN_DELEGATED",                 {type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("AC_TIMER_INTERVAL",				{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_INT, defaultValue:300});
	this.registerSetting("ASYNC_MODE",						{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("BRANCH",							{type:ZmSetting.T_CONFIG, defaultValue:"JUDASPRIEST"});

	// next 3 are replaced during deployment
	this.registerSetting("CLIENT_DATETIME",					{type:ZmSetting.T_CONFIG, defaultValue:"20250528-1953"});
	this.registerSetting("CLIENT_RELEASE",					{type:ZmSetting.T_CONFIG, defaultValue:"20250528194648"});
	this.registerSetting("CLIENT_VERSION",					{type:ZmSetting.T_CONFIG, defaultValue:"10.0.14_GA_0225"});
	this.registerSetting("CONFIG_PATH",						{type:ZmSetting.T_CONFIG, defaultValue:appContextPath + "/js/zimbraMail/config"});
	this.registerSetting("CSFE_EXPORT_URI",					{type:ZmSetting.T_CONFIG});
	this.registerSetting("CSFE_MSG_FETCHER_URI",			{type:ZmSetting.T_CONFIG});
	this.registerSetting("CSFE_SERVER_URI",					{type:ZmSetting.T_CONFIG});
	this.registerSetting("CSFE_UPLOAD_URI",					{type:ZmSetting.T_CONFIG});
	this.registerSetting("CSFE_ATTACHMENT_UPLOAD_URI",		{type:ZmSetting.T_CONFIG});
	this.registerSetting("DEV",								{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("FORCE_CAL_OFF",					{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("HELP_URI",						{name:"zimbraHelpAdvancedURL", type:ZmSetting.T_CONFIG, defaultValue:appContextPath + ZmMsg.helpURI});
	this.registerSetting("HTTP_PORT",						{type:ZmSetting.T_CONFIG, defaultValue:ZmSetting.HTTP_DEFAULT_PORT});
	this.registerSetting("HTTPS_PORT",						{type:ZmSetting.T_CONFIG, defaultValue:ZmSetting.HTTPS_DEFAULT_PORT});
	this.registerSetting("INSTANT_NOTIFY_INTERVAL",			{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_INT, defaultValue:500}); // milliseconds
	this.registerSetting("INSTANT_NOTIFY_TIMEOUT",			{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_INT, defaultValue:300}); // seconds
	this.registerSetting("IS_DEV_SERVER",					{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("LOG_REQUEST",						{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("LOGO_URI",						{type:ZmSetting.T_CONFIG, defaultValue:null});
	this.registerSetting("PROTOCOL_MODE",					{type:ZmSetting.T_CONFIG, defaultValue:ZmSetting.PROTO_HTTP});
	this.registerSetting("SERVER_VERSION",					{type:ZmSetting.T_CONFIG});
	this.registerSetting("SHOW_SCRIPT_ERRORS",				{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("TIMEOUT",							{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_INT, defaultValue:30}); // seconds
	this.registerSetting("USE_XML",							{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SMIME_HELP_URI",						{type:ZmSetting.T_CONFIG, defaultValue:appContextPath + ZmMsg.smimeHelpURI});

	// DOMAIN SETTINGS
	this.registerSetting("CHANGE_PASSWORD_URL",				{type:ZmSetting.T_CONFIG});
	this.registerSetting("PUBLIC_URL",						{type:ZmSetting.T_CONFIG});
	this.registerSetting("ADMIN_URL",						{type:ZmSetting.T_CONFIG});
	this.registerSetting("DISABLE_SENSITIVE_ZIMLETS_IN_MIXED_MODE",		{type:ZmSetting.T_CONFIG});
	this.registerSetting("DISABLE_MODERN_CLIENT",			{name:"zimbraModernWebClientDisabled", type:ZmSetting.T_DOMAIN, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});


	// COS SETTINGS - APPS
	this.registerSetting("BRIEFCASE_ENABLED",				{name:"zimbraFeatureBriefcasesEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("DOCUMENT_EDITOR_ENABLED",			{name:"zimbraFeatureDocumentEditingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("ATTACHMENTS_BLOCKED",				{name:"zimbraAttachmentsBlocked", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CALENDAR_ENABLED",				{name:"zimbraFeatureCalendarEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CALENDAR_UPSELL_ENABLED",			{name:"zimbraFeatureCalendarUpsellEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CALENDAR_UPSELL_URL",				{name:"zimbraFeatureCalendarUpsellURL", type:ZmSetting.T_COS});
	this.registerSetting("CONTACTS_ENABLED",				{name:"zimbraFeatureContactsEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CONTACTS_UPSELL_ENABLED",			{name:"zimbraFeatureContactsUpsellEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CONTACTS_UPSELL_URL",				{name:"zimbraFeatureContactsUpsellURL", type:ZmSetting.T_COS});
	this.registerSetting("IMPORT_ENABLED",					{name:"zimbraFeatureImportFolderEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("EXPORT_ENABLED",					{name:"zimbraFeatureExportFolderEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
    this.registerSetting("MAIL_ENABLED",					{name:"zimbraFeatureMailEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
    this.registerSetting("EXTERNAL_USER_MAIL_ADDRESS",		{name:"zimbraExternalUserMailAddress", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
    this.registerSetting("ADMIN_MAIL_ENABLED",				{name:"zimbraFeatureAdminMailEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
    this.registerSetting("ADMIN_PREFERENCES_ENABLED",		{name:"zimbraFeatureAdminPreferencesEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("MAIL_UPSELL_ENABLED",				{name:"zimbraFeatureMailUpsellEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("MAIL_UPSELL_URL",					{name:"zimbraFeatureMailUpsellURL", type:ZmSetting.T_COS});
	this.registerSetting("OPTIONS_ENABLED",					{name:"zimbraFeatureOptionsEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("PORTAL_ENABLED",					{name:"zimbraFeaturePortalEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SOCIAL_ENABLED",					{name:"zimbraFeatureSocialEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SOCIAL_EXTERNAL_ENABLED",			{name:"zimbraFeatureSocialExternalEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SOCIAL_EXTERNAL_URL",				{name:"zimbraFeatureSocialExternalURL", type:ZmSetting.T_COS});
	this.registerSetting("SOCIAL_NAME",				        {name:"zimbraFeatureSocialName", type:ZmSetting.T_COS, defaultValue:ZmMsg.communityName});
	this.registerSetting("TASKS_ENABLED",					{name:"zimbraFeatureTasksEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("VOICE_ENABLED",					{name:"zimbraFeatureVoiceEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("VOICE_UPSELL_ENABLED",			{name:"zimbraFeatureVoiceUpsellEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("VOICE_UPSELL_URL",				{name:"zimbraFeatureVoiceUpsellURL", type:ZmSetting.T_COS});
	this.registerSetting("DLS_FOLDER_ENABLED",				{name:"zimbraFeatureDistributionListFolderEnabled", type: ZmSetting.T_COS, dataType: ZmSetting.D_BOOLEAN, defaultValue: true});

	// COS SETTINGS
    this.registerSetting("ATTACHMENTS_VIEW_IN_HTML_ONLY",	{name:"zimbraAttachmentsViewInHtmlOnly", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("AVAILABLE_SKINS",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST, isGlobal:true});
	this.registerSetting("AVAILABLE_CSVFORMATS",			{type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST, isGlobal:true});
	this.registerSetting("BLOCK_SEND_FROM_IMAP_POP",		{name:"zimbraBlockEmailSendFromImapPop", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CHANGE_PASSWORD_ENABLED",			{name:"zimbraFeatureChangePasswordEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("DISPLAY_NAME",					{name:"displayName", type:ZmSetting.T_COS});
	this.registerSetting("DUMPSTER_ENABLED",				{name:"zimbraDumpsterEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("ERROR_REPORT_URL",				{name:"zimbraErrorReportUrl", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("EXPORT_MAX_DAYS",					{name:"zimbraExportMaxDays", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:0});
	this.registerSetting("FILTER_BATCH_SIZE",               {name:"zimbraFilterBatchSize", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue: 10000});
    this.registerSetting("FLAGGING_ENABLED",				{name:"zimbraFeatureFlaggingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("ENABLE_RETENTION_POLICY",			{name:"zimbraFeatureRetentionPolicyEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("FOLDERS_EXPANDED",				{name:"zimbraPrefFoldersExpanded", type:ZmSetting.T_METADATA, dataType: ZmSetting.D_HASH, isImplicit:true, section:ZmSetting.M_IMPLICIT});
	this.registerSetting("FOLDER_TREE_OPEN",				{name:"zimbraPrefFolderTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isImplicit:true});
	this.registerSetting("FOLDER_TREE_SASH_WIDTH",          {name:"zimbraPrefFolderTreeSash", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_INT, isImplicit:true, section:ZmSetting.M_IMPLICIT});
	this.registerSetting("GAL_AUTOCOMPLETE_ENABLED",		{name:"zimbraFeatureGalAutoCompleteEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN,	defaultValue:false});
	this.registerSetting("GAL_ENABLED",						{name:"zimbraFeatureGalEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN,	defaultValue:true});
	this.registerSetting("HAB_ROOT",						{type:ZmSetting.T_CONFIG});
	this.registerSetting("GROUP_CALENDAR_ENABLED",			{name:"zimbraFeatureGroupCalendarEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("HTML_COMPOSE_ENABLED",			{name:"zimbraFeatureHtmlComposeEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("IDLE_SESSION_TIMEOUT",			{name:"zimbraMailIdleSessionTimeout", type:ZmSetting.T_COS, dataType:ZmSetting.D_LDAP_TIME, defaultValue:0});
	this.registerSetting("IMAP_ACCOUNTS_ENABLED",			{name:"zimbraFeatureImapDataSourceEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("IMPORT_ON_LOGIN_ENABLED",			{name:"zimbraDataSourceImportOnLogin", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("INSTANT_NOTIFY",					{name:"zimbraFeatureInstantNotify", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("LOCALE_CHANGE_ENABLED",			{name:"zimbraFeatureLocaleChangeEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("LOCALES",							{type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST});
	this.registerSetting("LOGIN_URL",						{name:"zimbraWebClientLoginURL", type:ZmSetting.T_COS});
	this.registerSetting("LOGOUT_URL",						{name:"zimbraWebClientLogoutURL", type:ZmSetting.T_COS});
	this.registerSetting("MIN_POLLING_INTERVAL",			{name:"zimbraMailMinPollingInterval", type:ZmSetting.T_COS, dataType:ZmSetting.D_LDAP_TIME, defaultValue:120});
	this.registerSetting("MOBILE_SYNC_ENABLED",				{name:"zimbraFeatureMobileSyncEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("MOBILE_POLICY_ENABLED",			{name:"zimbraFeatureMobilePolicyEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("POP_ACCOUNTS_ENABLED",			{name:"zimbraFeaturePop3DataSourceEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("PORTAL_NAME",						{name:"zimbraPortalName", type:ZmSetting.T_COS, defaultValue:"example"});
	this.registerSetting("PRIORITY_INBOX_ENABLED",          {name:"zimbraFeaturePriorityInboxEnabled", type:ZmSetting.T_COS, dataType: ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("PWD_MAX_LENGTH",					{name:"zimbraPasswordMaxLength", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:64});
	this.registerSetting("PWD_MIN_LENGTH",					{name:"zimbraPasswordMinLength", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:6});
	this.registerSetting("QUOTA",							{name:"zimbraMailQuota", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:0});
	this.registerSetting("SAVED_SEARCHES_ENABLED",			{name:"zimbraFeatureSavedSearchesEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SEARCH_TREE_OPEN",				{name:"zimbraPrefSearchTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isImplicit:true});
	this.registerSetting("SHARING_ENABLED",					{name:"zimbraFeatureSharingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHARING_PUBLIC_ENABLED",			{name:"zimbraPublicSharingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHARING_EXTERNAL_ENABLED",		{name:"zimbraExternalSharingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHORTCUT_ALIASES_ENABLED",		{name:"zimbraFeatureShortcutAliasesEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHOW_OFFLINE_LINK",				{name:"zimbraWebClientShowOfflineLink", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SIGNATURES_ENABLED",				{name:"zimbraFeatureSignaturesEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SKIN_CHANGE_ENABLED",				{name:"zimbraFeatureSkinChangeEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
    this.registerSetting("SOCIAL_FILTERS_ENABLED",          {name:"zimbraFeatureSocialFiltersEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST, isImplicit: true, section:ZmSetting.M_IMPLICIT});
	this.registerSetting("SPAM_ENABLED",					{name:"zimbraFeatureAntispamEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("TAG_TREE_OPEN",					{name:"zimbraPrefTagTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isImplicit:true});
	this.registerSetting("TAGGING_ENABLED",					{name:"zimbraFeatureTaggingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("VIEW_ATTACHMENT_AS_HTML",			{name:"zimbraFeatureViewInHtmlEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("EXPAND_DL_ENABLED",				{name:"zimbraFeatureDistributionListExpandMembersEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("FORCE_CLEAR_COOKIES",				{name:"zimbraForceClearCookies", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SPELL_DICTIONARY",                                {name:"zimbraPrefSpellDictionary", type:ZmSetting.T_COS, defaultValue:""});
	this.registerSetting("TWO_FACTOR_AUTH_AVAILABLE",	    {name:"zimbraFeatureTwoFactorAuthAvailable", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("TWO_FACTOR_AUTH_REQUIRED",	    {name:"zimbraFeatureTwoFactorAuthRequired", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("TWO_FACTOR_AUTH_METHOD_ALLOWED",  {name:"zimbraTwoFactorAuthMethodAllowed", type:ZmSetting.T_COS, defaultValue:["app"]});
	this.registerSetting("TRUSTED_DEVICES_ENABLED",         {name:"zimbraFeatureTrustedDevicesEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("APP_PASSWORDS_ENABLED",	        {name:"zimbraFeatureAppSpecificPasswordsEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("DISABLE_VERSION_CHANGE_DIALOG",   {name:"zimbraServerVersionChangeNotificationDisabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});

	this.registerSetting("RESET_PASSWORD_STATUS",				{name:"zimbraFeatureResetPasswordStatus", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("PASSWORD_RECOVERY_EMAIL",				{name:"zimbraPrefPasswordRecoveryAddress", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("PASSWORD_RECOVERY_EMAIL_STATUS",			{name:"zimbraPrefPasswordRecoveryAddressStatus", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("RESET_PASSWORD_RECOVERY_CODE_EXPIRY",				{name:"zimbraResetPasswordRecoveryCodeExpiry", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("PASSWORD_RECOVERY_CODE_VALIDITY",				{name:"zimbraRecoveryAccountCodeValidity", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("PASSWORD_RECOVERY_SUSPENSION_TIME",				{name:"zimbraFeatureResetPasswordSuspensionTime", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("SHARED_FOLDER_MOBILE_SYNC_ENABLED",				{name:"zimbraFeatureSharedFolderMobileSyncEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN});

	// user metadata (included with COS since the user can't change them)
	this.registerSetting("LICENSE_STATUS",					{type:ZmSetting.T_COS, defaultValue:ZmSetting.LICENSE_GOOD});
	this.registerSetting("QUOTA_USED",						{type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});    
	this.registerSetting("USERID",							{name:"zimbraId", type:ZmSetting.T_COS});
	this.registerSetting("USERNAME",						{type:ZmSetting.T_COS});
	this.registerSetting("CN",								{name:"cn", type:ZmSetting.T_COS});
	this.registerSetting("LAST_ACCESS",						{type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});
	this.registerSetting("PREVIOUS_SESSION",				{type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});
	this.registerSetting("RECENT_MESSAGES",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});
	this.registerSetting("REST_URL",						{name:"rest" , type:ZmSetting.T_COS});
	this.registerSetting("IS_ADMIN",						{name:"zimbraIsAdminAccount", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue: false});
	this.registerSetting("IS_EXTERNAL",						{name:"zimbraIsExternalVirtualAccount", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue: false});
	this.registerSetting("IS_DELEGATED_ADMIN",				{name:"zimbraIsDelegatedAdminAccount", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue: false});
    this.registerSetting("MESSAGE_SIZE_LIMIT",              {type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});
    this.registerSetting("DOCUMENT_SIZE_LIMIT",             {type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});

	// CLIENT SIDE FEATURE SUPPORT
	this.registerSetting("ATTACHMENT_ENABLED",				{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("ATT_VIEW_ENABLED",				{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("BROWSER_TOOLTIPS_ENABLED",		{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("EVAL_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("FEED_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("HELP_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("HISTORY_SUPPORT_ENABLED",			{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:("onhashchange" in window)});
	this.registerSetting("NOTES_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("PRINT_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SEARCH_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHORTCUT_LIST_ENABLED",			{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("OFFLINE_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:appCtxt.isOffline});
	this.registerSetting("SPELL_CHECK_ENABLED",				{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN});
	this.registerSetting("SPELL_CHECK_ADD_WORD_ENABLED",	{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:!AjxEnv.isSafari || AjxEnv.isSafari3up || AjxEnv.isChrome});

	//SETTINGS SET AT DOMAIN LEVEL
	this.registerSetting("EMAIL_VALIDATION_REGEX",			{name:"zimbraMailAddressValidationRegex", type:ZmSetting.T_DOMAIN, dataType:ZmSetting.D_LIST});
	this.registerSetting("SUPPORTED_HELPS",					{name:"zimbraWebClientSupportedHelps", type:ZmSetting.T_DOMAIN, dataType:ZmSetting.D_LIST});

	// USER PREFERENCES (mutable)

	// general preferences
	this.registerSetting("ACCOUNTS",						{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("TWO_FACTOR_AUTH_ENABLED",	        {name:"zimbraTwoFactorAuthEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("TWO_FACTOR_AUTH_METHOD_ENABLED",  {name:"zimbraTwoFactorAuthMethodEnabled", type:ZmSetting.T_COS});
	this.registerSetting("TWO_FACTOR_AUTH_PRIMARY_METHOD",  {name:"zimbraPrefPrimaryTwoFactorAuthMethod", type:ZmSetting.T_PREF});
	this.registerSetting("ACCOUNT_TREE_OPEN",				{name:"zimbraPrefAccountTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isImplicit:true});
	this.registerSetting("CHILD_ACCTS_VISIBLE",				{name:"zimbraPrefChildVisibleAccount", type:ZmSetting.T_PREF, dataType:ZmSetting.D_LIST});
	this.registerSetting("CLIENT_TYPE",						{name:"zimbraPrefClientType", type:ZmSetting.T_PREF, defaultValue:ZmSetting.CLIENT_ADVANCED});
	this.registerSetting("COMPOSE_AS_FORMAT",				{name:"zimbraPrefComposeFormat", type:ZmSetting.T_PREF, defaultValue:ZmSetting.COMPOSE_HTML, isGlobal:true});
	this.registerSetting("COMPOSE_INIT_FONT_COLOR",			{name:"zimbraPrefHtmlEditorDefaultFontColor", type:ZmSetting.T_PREF, defaultValue:ZmSetting.COMPOSE_FONT_COLOR, isGlobal:true});
	this.registerSetting("COMPOSE_INIT_FONT_FAMILY",		{name:"zimbraPrefHtmlEditorDefaultFontFamily", type:ZmSetting.T_PREF, defaultValue:ZmSetting.COMPOSE_FONT_FAM, isGlobal:true});
	this.registerSetting("COMPOSE_INIT_FONT_SIZE",			{name:"zimbraPrefHtmlEditorDefaultFontSize", type:ZmSetting.T_PREF, defaultValue:ZmSetting.COMPOSE_FONT_SIZE, isGlobal:true});
    this.registerSetting("COMPOSE_INIT_DIRECTION",			{name:"zimbraPrefComposeDirection", type:ZmSetting.T_PREF, defaultValue:ZmSetting.LTR, isGlobal:true});
    this.registerSetting("SHOW_COMPOSE_DIRECTION_BUTTONS",	{name:"zimbraPrefShowComposeDirection", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("DEFAULT_TIMEZONE",				{name:"zimbraPrefTimeZoneId", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, defaultValue:AjxTimezone.getServerId(AjxTimezone.DEFAULT), isGlobal:true});
    this.registerSetting("WEBCLIENT_OFFLINE_BROWSER_KEY",	{name:"zimbraPrefWebClientOfflineBrowserKey", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, isImplicit:true});
    this.registerSetting("DEFAULT_PRINTFONTSIZE",	    	{name:"zimbraPrefDefaultPrintFontSize", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, defaultValue:ZmSetting.PRINT_FONT_SIZE, isGlobal:true});
	this.registerSetting("GROUPBY_HASH",                    {type: ZmSetting.T_PREF, dataType:ZmSetting.D_HASH});
	this.registerSetting("GROUPBY_LIST",                    {name:"zimbraPrefGroupByList", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_HASH, isImplicit:true, section:ZmSetting.M_IMPLICIT});
    this.registerSetting("FILTERS",							{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("FONT_NAME",						{name:"zimbraPrefFont", type:ZmSetting.T_PREF, defaultValue: ZmSetting.FONT_SYSTEM, isGlobal:true});
	this.registerSetting("FONT_SIZE",						{name:"zimbraPrefFontSize", type:ZmSetting.T_PREF, defaultValue: ZmSetting.FONT_SIZE_NORMAL, isGlobal:true});
	this.registerSetting("IDENTITIES",						{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("INITIALLY_SEARCH_GAL",			{name:"zimbraPrefGalSearchEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("LIST_VIEW_COLUMNS",				{name:"zimbraPrefListViewColumns", type:ZmSetting.T_PREF, dataType:ZmSetting.D_HASH, isImplicit:true});
	this.registerSetting("LOCALE_NAME",						{name:"zimbraPrefLocale", type:ZmSetting.T_PREF, defaultValue:appRequestLocaleId, isGlobal:true});
	this.registerSetting("SHOW_SELECTION_CHECKBOX",			{name:"zimbraPrefShowSelectionCheckbox", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isGlobal:true});
	// PAGE_SIZE: number of items to fetch for virtual paging; also used for number of msgs in one page of a conv
	this.registerSetting("PAGE_SIZE",						{name: "zimbraPrefItemsPerVirtualPage", type:ZmSetting.T_PREF, dataType:ZmSetting.D_INT, defaultValue:50, isGlobal:true});
	this.registerSetting("PASSWORD",						{type:ZmSetting.T_PREF, dataType:ZmSetting.D_NONE});
	this.registerSetting("POLLING_INTERVAL",				{name:"zimbraPrefMailPollingInterval", type:ZmSetting.T_PREF, dataType:ZmSetting.D_LDAP_TIME, defaultValue:300});
	this.registerSetting("POLLING_INTERVAL_ENABLED",		{name:"zimbraFeatureMailPollingIntervalPreferenceEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SEARCH_INCLUDES_SHARED",			{name:"zimbraPrefIncludeSharedItemsInSearch", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("SEARCH_INCLUDES_SPAM",			{name:"zimbraPrefIncludeSpamInSearch", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("SEARCH_INCLUDES_TRASH",			{name:"zimbraPrefIncludeTrashInSearch", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("SHORT_ADDRESS",					{name:"zimbraPrefShortEmailAddress", type:ZmSetting.T_PREF, dataType: ZmSetting.D_BOOLEAN, defaultValue: true});
	this.registerSetting("SHORTCUTS",						{name:"zimbraPrefShortcuts", type:ZmSetting.T_PREF});
	this.registerSetting("SHOW_SEARCH_STRING",				{name:"zimbraPrefShowSearchString", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("SIGNATURES",						{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("SIGNATURES_MAX",					{name:"zimbraSignatureMaxNumEntries", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:20});
	this.registerSetting("SIGNATURES_MIN",					{name:"zimbraSignatureMinNumEntries", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:1});
	this.registerSetting("SKIN_NAME",						{name:"zimbraPrefSkin", type:ZmSetting.T_PREF, defaultValue:"skin", isGlobal:true});
	this.registerSetting("SORTING_PREF",					{name:"zimbraPrefSortOrder", type:ZmSetting.T_PREF, dataType:ZmSetting.D_HASH, isImplicit:true, isGlobal:true, dontSaveDefault: true});
	this.registerSetting("USE_KEYBOARD_SHORTCUTS",			{name:"zimbraPrefUseKeyboardShortcuts", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("VIEW_AS_HTML",					{name:"zimbraPrefMessageViewHtmlPreferred", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("VOICE_ACCOUNTS",					{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("WARN_ON_EXIT",					{name:"zimbraPrefWarnOnExit", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("DISPLAY_TIME_IN_MAIL_LIST",		{name:"zimbraPrefDisplayTimeInMailList", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});

	this._registerOfflineSettings();
	this._registerZimletsSettings();

	// need to do this before loadUserSettings(), and zimlet settings are not tied to an app where it would normally be done
	this.registerSetting("ZIMLET_TREE_OPEN",				{name:"zimbraPrefZimletTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isImplicit:true});
	
	//shared settings
	this.registerSetting("MAIL_ALIASES",					{name:"zimbraMailAlias", type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST});
	this.registerSetting("ALLOW_FROM_ADDRESSES",			{name:"zimbraAllowFromAddress", type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST});

	// Internal pref to control display of mail-related preferences. The only time it will be false is for a delegated admin with zimbraFeatureAdminMailEnabled and
	// zimbraFeatureAdminPreferencesEnabled set to FALSE.
	this.registerSetting("MAIL_PREFERENCES_ENABLED",	    {type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});

	appCtxt.notifyZimlets("onZmSettings_initialize", [this]);

	ZmApp.runAppFunction("registerSettings", this);
};

/**
 * @private
 */
ZmSettings.prototype._registerZimletsSettings =
function() {
	// zimlet-specific
	this.registerSetting("CHECKED_ZIMLETS_ENABLED",			{name:"zimbraFeatureManageZimlets", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isGlobal:true});
	this.registerSetting("CHECKED_ZIMLETS",					{name:"zimbraPrefZimlets", type:ZmSetting.T_PREF, dataType:ZmSetting.D_LIST, isGlobal:true});
    this.registerSetting("MANDATORY_ZIMLETS",		        {name:"zimbraZimletMandatoryZimlets", type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST});
    this.registerSetting("ZIMLETS_SYNCHRONOUS",		        {name:"zimbraZimletLoadSynchronously", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});

};

/**
 * @private
 */
ZmSettings.prototype._registerOfflineSettings =
function() {
	if (!appCtxt.isOffline) { return; }

	// offline-specific
	this.registerSetting("OFFLINE_ACCOUNT_FLAVOR",			{name:"offlineAccountFlavor", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING});
	this.registerSetting("OFFLINE_COMPOSE_ENABLED",			{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("OFFLINE_DEBUG_TRACE",				{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("OFFLINE_IS_MAILTO_HANDLER",		{name:"zimbraPrefMailtoHandlerEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("OFFLINE_REMOTE_SERVER_URI",		{name:"offlineRemoteServerUri", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING});
	this.registerSetting("OFFLINE_REPORT_EMAIL",			{type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, defaultValue:"zdesktop-report@zimbra.com", isGlobal:true});
	this.registerSetting("OFFLINE_SHOW_ALL_MAILBOXES",		{name:"offlineShowAllMailboxes", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, section:ZmSetting.M_OFFLINE, isGlobal:true});
	this.registerSetting("OFFLINE_ALL_MAILBOXES_TREE_OPEN",	{name:"offlineAllMailboxesTreeOpen", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, section:ZmSetting.M_OFFLINE, isGlobal:true, isImplicit:true});
	this.registerSetting("OFFLINE_NOTIFY_NEWMAIL_ON_INBOX",	{name:"offlineNotifyNewMailOnInbox", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, section:ZmSetting.M_OFFLINE, isGlobal:true});
	this.registerSetting("OFFLINE_SAVED_SEARCHES_TREE_OPEN",{name:"offlineSavedSearchesTreeOpen", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, section:ZmSetting.M_OFFLINE, isGlobal:true, isImplicit:true});
	this.registerSetting("OFFLINE_SMTP_ENABLED",			{name:"zimbraDataSourceSmtpEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("OFFLINE_SUPPORTS_MAILTO",			{type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("OFFLINE_SUPPORTS_DOCK_UPDATE",	{type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("OFFLINE_WEBAPP_URI",				{name:"offlineWebappUri", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING});
    this.registerSetting("OFFLINE_BACKUP_INTERVAL",	        {name:"zimbraPrefOfflineBackupInterval", type:ZmSetting.T_PREF, dataType:ZmSetting.D_INT, defaultValue:0, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_PATH",	            {name:"zimbraPrefOfflineBackupPath", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_KEEP",	            {name:"zimbraPrefOfflineBackupKeep", type:ZmSetting.T_PREF, dataType:ZmSetting.D_INT, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_ACCOUNT_ID",       {name:"zimbraPrefOfflineBackupAccountId", type:ZmSetting.T_PREF, dataType:ZmSetting.D_INT, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_RESTORE",          {name:"zimbraPrefOfflineBackupRestore", dataType:ZmSetting.D_INT, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_NOW_BUTTON",       {name:"zimbraPrefOfflineBackupAccount", dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
    this.registerSetting("OFFLINE_ZIMLET_SYNC_ACCOUNT_ID",  {name:"zimbraPrefOfflineZimletSyncAccountId", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, isGlobal:true});
	this.registerSetting("OFFLINE_WEBAPP_URI",				{name:"offlineWebappUri", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING});

	// reset the help URI to zimbra.com for offline
	this.registerSetting("HELP_URI",						{type:ZmSetting.T_CONFIG, defaultValue:"https://www.zimbra.com/desktop7/"});
//	// make default false for DUMPSTER_ENABLED. shouldn't be necessary since GetInfoResponse includes zimbraDumpsterEnabled:"FALSE", but can't find why settings is not read correctly
	this.registerSetting("DUMPSTER_ENABLED",				{name:"zimbraDumpsterEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
    this.registerSetting("OFFLINE_UPDATE_NOTIFY",			{name:"zimbraPrefOfflineUpdateChannel", type: ZmSetting.T_PREF, dataType: ZmSetting.D_STRING, isGlobal:true});
};

/**
 * @private
 */
ZmSettings.prototype._changeListener =
function(ev) {

	if (ev.type !== ZmEvent.S_SETTING) {
		return;
	}

	var id = ev.source.id,
		value = ev.source.getValue();

	if (id === ZmSetting.QUOTA_USED) {
		appCtxt.getAppController().setUserInfo();
	}
	else if (id === ZmSetting.POLLING_INTERVAL) {
		appCtxt.getAppController().setPollInterval();
	}
	else if (id === ZmSetting.SKIN_NAME) {
		this._showConfirmDialog(ZmMsg.skinChangeRestart, this._refreshBrowserCallback.bind(this));
	}
	else if (id === ZmSetting.SHOW_SELECTION_CHECKBOX) {
		this._showConfirmDialog(value ? ZmMsg.checkboxChangeRestartShow : ZmMsg.checkboxChangeRestartHide, this._refreshBrowserCallback.bind(this));
	}
	else if (id === ZmSetting.FONT_NAME || id === ZmSetting.FONT_SIZE) {
		this._showConfirmDialog(ZmMsg.fontChangeRestart, this._refreshBrowserCallback.bind(this));
	}
	else if (id === ZmSetting.LOCALE_NAME) {
		// bug: 29786
		if (appCtxt.isOffline && AjxEnv.isPrism) {
			try {
				netscape.security.PrivilegeManager.enablePrivilege('UniversalXPConnect');
				var prefs = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefBranch);
				if (prefs) {
					var newLocale = appCtxt.get(ZmSetting.LOCALE_NAME).replace("_", "-");
					prefs.setCharPref("general.useragent.locale", newLocale);
					prefs.setCharPref("intl.accept_languages", newLocale);
				}
			} catch (ex) {
				// do nothing for now
			}
		}
		this._showConfirmDialog(ZmMsg.localeChangeRestart, this._refreshBrowserCallback.bind(this));
	}
	else if (id === ZmSetting.CHILD_ACCTS_VISIBLE) {
		this._showConfirmDialog(ZmMsg.accountChangeRestart, this._refreshBrowserCallback.bind(this));
	}
	else if (appCtxt.isOffline && id === ZmSetting.OFFLINE_IS_MAILTO_HANDLER) {
		appCtxt.getAppController().registerMailtoHandler(true, ev.source.getValue());
	}
	else if (appCtxt.isOffline && id === ZmSetting.OFFLINE_UPDATE_NOTIFY) {
		appCtxt.getAppController()._offlineUpdateChannelPref(ev.source.getValue());
	}
	else if (id === ZmSetting.DISPLAY_TIME_IN_MAIL_LIST) {
		this._showConfirmDialog(value ? ZmMsg.timeInMailListChangeRestartShow : ZmMsg.timeInMailListChangeRestartHide, this._refreshBrowserCallback.bind(this));
	}
};

// Shows a confirm dialog that asks the user if they want to reload ZCS to show the change they just made
ZmSettings.prototype._showConfirmDialog = function(text, callback, style) {

	var confirmDialog = appCtxt.getYesNoMsgDialog();
	confirmDialog.reset();
	confirmDialog.registerCallback(DwtDialog.YES_BUTTON, callback);
	confirmDialog.setMessage(text, style || DwtMessageDialog.WARNING_STYLE);
	confirmDialog.popup();
};

ZmSettings.prototype._implicitChangeListener =
function(ev) {
	var id = ev.source.id;
	var sourceType = ev.source.type;

	// Allow to perform implicit settings save (e.g folder expand) if the source type is META even when zimbraFeatureOptionsEnabled is false.
	if (!appCtxt.get(ZmSetting.OPTIONS_ENABLED) && ZmSetting.T_METADATA !== sourceType) {
		return;
	}
	if (ev.type != ZmEvent.S_SETTING) { return; }
	var setting = this.getSetting(id);
	if (id == ZmSetting.FOLDERS_EXPANDED && window.duringExpandAll) {
		if (!window.afterExpandAllCallback) {
			window.afterExpandAllCallback = this.save.bind(this, [setting], null, null, appCtxt.getActiveAccount(), true);
		}
		return;
	}
	if (ZmSetting.IS_IMPLICIT[id] && setting) {
        if (id === ZmSetting.WEBCLIENT_OFFLINE_BROWSER_KEY) {
            var callback = this._offlineSettingsSaveCallback.bind(this, setting);
        }
		else {
			//Once implicit preference is saved, reload the application cache to get the latest changes
			var callback = appCtxt.reloadAppCache.bind(appCtxt, false);
		}
		this.save([setting], callback, null, appCtxt.getActiveAccount(), true);
	}
};

/**
 * @private
 */
ZmSettings.prototype._refreshBrowserCallback =
function(args) {
	appCtxt.getYesNoMsgDialog().popdown();
	window.onbeforeunload = ZmZimbraMail.getConfirmExitMethod();
	var url = AjxUtil.formatUrl({qsArgs : args});
	window.location.replace(url);
};

// Adds/replaces a CSS rule that comprises user font prefs
ZmSettings.prototype._updateUserFontPrefsRule =
function() {
	if (this._userFontPrefsRuleIndex != null) {
		DwtCssStyle.removeRule(document.styleSheets[0], this._userFontPrefsRuleIndex);
	}
	var selector = "." + ZmSetting.USER_FONT_CLASS;
	var declaration = "font-family:" + appCtxt.get(ZmSetting.COMPOSE_INIT_FONT_FAMILY) + ";" +
					  "font-size:" + appCtxt.get(ZmSetting.COMPOSE_INIT_FONT_SIZE) + ";" +
					  "color:" + appCtxt.get(ZmSetting.COMPOSE_INIT_FONT_COLOR) + ";";
	this._userFontPrefsRuleIndex = DwtCssStyle.addRule(document.styleSheets[0], selector, declaration);
};

// Check license to see if voice feature is allowed
// License block format:
//
//  <license status="OK">
//    <attr name="SMIME">FALSE</attr>
//    <attr name="VOICE">TRUE</attr>
//  </license>

ZmSettings.prototype._hasVoiceFeature = function() {

    var info = this.getInfoResponse;
    var license = info && info.license;
    var status = license && license.status;

    if (!license || !license.attr) {
        return false;
    }

    // License not installed or not activated or expired
    if (ZmSetting.LICENSE_MSG[status]) {
        return false;
    }

    // check for VOICE license attribute

    for (var i = 0; license && i < license.attr.length; i++) {
        var attr = license.attr[i];

        if (attr.name == "VOICE") {
            return attr._content == "TRUE";
        }
    }

    return false;
};

/**
 * @private
 */
ZmSettings.prototype._offlineSettingsSaveCallback =
function(setting) {
	var offlineBrowserKey = setting.getValue();
	var localOfflineBrowserKey = localStorage.getItem(ZmSetting.WEBCLIENT_OFFLINE_BROWSER_KEY);
	if (offlineBrowserKey && offlineBrowserKey.indexOf(localOfflineBrowserKey) !== -1) {
		this._showConfirmDialog(ZmMsg.offlineChangeRestart, this._refreshBrowserCallback.bind(this));
    }
    else {
        ZmOffline.deleteOfflineData();
        appCtxt.initWebOffline();// To reset the property isWebClientOfflineSupported
	    //delete outbox folder
	    var outboxFolder = appCtxt.getById(ZmFolder.ID_OUTBOX);
	    if (outboxFolder) {
		    outboxFolder.notifyDelete();
	    }
    }
	//Always reload appcache whenever offline setting is enabled/disabled. Appcache will be updated/emptied depending upon the setting.
	appCtxt.reloadAppCache(true);
};

ZmSettings.prototype.getMailAppForDelegatedAdmin =
function() {
    if (!this.mailApp) {
        this.mailApp = new ZmMailApp();
    }
    return this.mailApp;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmMetaData")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines classes for the storage/retrieval of generic user data.
 */

/**
 * Stores generic data to the server via <code>&lt;Set[Mailbox|Custom]MetadataRequest&gt;</code> and
 * <code>&lt;Get[Mailbox|Custom]MetadataRequest&gt;</code>.
 * @class
 * This class provides a general way to store per-user data using arbitrary
 * key/value pairs. NOTE: the server does not support modifying data so if there
 * are any changes, *all* the data must be re-set per section. Data can be
 * written on the mailbox or on an individual mailbox item. If the constructor
 * receives an itemId, then data will be retrieved/written via [Get|Set]CustomMetadataRequest.
 * <br/>
 * <br/>
 * The section data is mapped into {@link ZmSettings} (based on the key name) to allow
 * for easy access. When creating/setting a *new* key/value, naming conventions
 * should be followed as defined by prefs in {@link ZmSettings}. For example, if adding
 * a new key called "foo", the name for the key should be "zimbraPrefFoo" and
 * should be added to the list of settings in {@link ZmSettings} with type set to
 * ZmSetting.T_METADATA
 *
 * @param {ZmAccount}	account		Optional. The account this meta data belongs to
 * @param {String}	itemId		Optional.  If specified, (custom) meta data will be saved on the item
 *
 * @author Parag Shah
 */
ZmMetaData = function(account, itemId) {

	this._sections = {};
	this._account = account;
	this._itemId = itemId;
};

ZmMetaData.prototype.constructor = ZmMetaData;


// Consts

ZmMetaData.NAMESPACE		= "zwc";


// Public Methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmMetaData.prototype.toString =
function() {
	return "ZmMetaData";
};

/**
 * Saves the given section and corresponding key/value pair to the server.
 *
 * @param {String}	section		the name of the section to save
 * @param {Object}	data			the list of key/value pairs
 * @param {ZmBatchCommand}	batchCommand	if part of a batch command
 * @param {AjxCallback}	callback		the callback to trigger on successful save
 * @param {AjxCallback}	errorCallback	the error callback to trigger on error
 * @param {Boolean} sensitive if <code>true</code>, attempt to use secure conn to protect data
 */
ZmMetaData.prototype.set =
function(section, data, batchCommand, callback, errorCallback, sensitive) {
	var soapDoc;
	if (this._itemId) {
		soapDoc = AjxSoapDoc.create("SetCustomMetadataRequest", "urn:zimbraMail");
		soapDoc.getMethod().setAttribute("id", this._itemId);
	} else {
		soapDoc = AjxSoapDoc.create("SetMailboxMetadataRequest", "urn:zimbraMail");
	}
	var metaNode = soapDoc.set("meta");
	metaNode.setAttribute("section", [ZmMetaData.NAMESPACE, section].join(":"));

	for (var i in data) {
		var d = data[i];

		// serialize if we're dealing with an object
		if (AjxUtil.isObject(d)) {
			d = ZmSetting.serialize(d, ZmSetting.D_HASH);
		}

		// If we want to remove any data from SetMailboxMetadataRequest then don't send it as part of request,
		// server will automatically remove it
		if (d !== null && d !== "") {
			var a = soapDoc.set("a", d, metaNode);
			a.setAttribute("n", i);
		}
	}

	if (batchCommand) {
		batchCommand.addNewRequestParams(soapDoc, callback, errorCallback);
	}
	else {
		var params = {
			soapDoc: soapDoc,
			asyncMode: true,
			callback: callback,
			errorCallback: errorCallback,
			accountName: (this._account ? this._account.name : null),
			sensitive: sensitive
		};

		appCtxt.getAppController().sendRequest(params);
	}
};

/**
 * Fetches the given section name from the server unless its already been
 * fetched (and therefore cached)
 *
 * @param {String}	section		section of meta data to fetch
 * @param {ZmBatchCommand}	batchCommand	if part of a separate batch command
 * @param {AjxCallback}	callback		the callback to trigger once meta data is fetched
 * @param {AjxCallback}	errorCallback	the error callback to trigger on error
 */
ZmMetaData.prototype.get =
function(section, batchCommand, callback, errorCallback) {
	var command = batchCommand || (new ZmBatchCommand());
	var sectionName = [ZmMetaData.NAMESPACE, section].join(":");

	var cachedSection = this._sections[sectionName];

	// if not yet cached, go fetch it
	if (!cachedSection) {
		var soapDoc;
		if (this._itemId) {
			soapDoc = AjxSoapDoc.create("GetCustomMetadataRequest", "urn:zimbraMail");
			soapDoc.getMethod().setAttribute("id", this._itemId);
		} else {
			soapDoc = AjxSoapDoc.create("GetMailboxMetadataRequest", "urn:zimbraMail");
		}
		var metaNode = soapDoc.set("meta");
		metaNode.setAttribute("section", sectionName);

		command.addNewRequestParams(soapDoc);

		if (!batchCommand) {
			command.run(callback, errorCallback);
		}
	}
	else {
		if (callback) {
			callback.run(cachedSection);
		} else {
			return cachedSection;
		}
	}
};

/**
 * Loads meta data from the server
 *
 * @param {Array}		sections	the sections to load
 * @param {AjxCallback}	callback	the callback
 * @param {ZmBatchCommand}		batchCommand if part of batch command
 */
ZmMetaData.prototype.load =
function(sections, callback, batchCommand) {
	if (!sections) { return; }
	if (!(sections instanceof Array)) { sections = [sections]; }

	var command = batchCommand || (new ZmBatchCommand());
	for (var i = 0; i < sections.length; i++) {
		if (sections[i] == ZmSetting.M_OFFLINE && !appCtxt.isOffline) { continue; }
		this.get(sections[i], command);
	}

	if (!batchCommand) {
		if (command.size() > 0) {
			var respCallback = new AjxCallback(this, this._handleLoad, [callback]);
			var offlineCallback = this._handleOfflineLoad.bind(this, respCallback);
			command.run(respCallback, null, offlineCallback);
		}
	} else {
		if (callback) {
			callback.run(this._sections);
		}
	}
};

/**
 * @private
 */
ZmMetaData.prototype._handleLoad =
function(callback, result) {
	this._sections = {};

	var br = result.getResponse().BatchResponse;
	if (br) {
		var metaDataResp = (this._itemId != null) ? br.GetCustomMetadataResponse : br.GetMailboxMetadataResponse;
		if (metaDataResp && metaDataResp.length) {
			if (ZmOffline.isOnlineMode()) {
				localStorage.setItem("MetadataResponse", JSON.stringify(br));
			}
			for (var i = 0; i < metaDataResp.length; i++) {
				var data = metaDataResp[i].meta[0];
				this._sections[data.section] = data._attrs;
			}
		}
	}

	if (callback) {
		callback.run(this._sections);
	}
};

/**
 * @private
 */
ZmMetaData.prototype._handleOfflineLoad =
function(callback) {
	var result = localStorage.getItem("MetadataResponse");
	if (result) {
		var csfeResult = new ZmCsfeResult({BatchResponse : JSON.parse(result)});
		callback.run(csfeResult);
	}
};

/**
 * Modifies the given section with new key/value pairs
 *
 * @param {Array}			section			the section to modify
 * @param {Object}			data			the list of key/value pairs
 * @param {ZmBatchCommand}	batchCommand	Optional. the batch command the request should be a part of
 * @param {AjxCallback}		callback		the callback called on successful modify
 * @param {AjxCallback}		errorCallback	the error callback to trigger on error
 */
ZmMetaData.prototype.modify =
function(section, data, batchCommand, callback, errorCallback) {
	var soapDoc = AjxSoapDoc.create("ModifyMailboxMetadataRequest", "urn:zimbraMail");
	var metaNode = soapDoc.set("meta");
	metaNode.setAttribute("section", [ZmMetaData.NAMESPACE, section].join(":"));

	for (var i in data) {
		var a = soapDoc.set("a", data[i], metaNode);
		a.setAttribute("n", i);
	}

	if (batchCommand) {
		batchCommand.addNewRequestParams(soapDoc, callback, errorCallback);
	}
	else {
		var params = {
			soapDoc: soapDoc,
			asyncMode: true,
			callback: callback,
			errorCallback: errorCallback,
			accountName: (this._account ? this._account.name : null)
		};

		appCtxt.getAppController().sendRequest(params);
	}
};

/**
 * Saves all data within the given section out to the server. If section is not
 * provided, all sections are saved.
 *
 * @param {Array}		sections		the sections to save
 * @param {AjxCallback}	callback		the callback called on successful save
 * @param {ZmBatchCommand}	batchCommand		the batch command the request should be a part of
 */
ZmMetaData.prototype.save =
function(sections, callback, batchCommand) {
	if (!sections) { return; }
	if (!(sections instanceof Array)) { sections = [sections]; }

	var acct = this._account ? this._account.name : null;
	var command = batchCommand || (new ZmBatchCommand(null, acct));

	for (var i = 0; i < sections.length; i++) {
		var s = sections[i];
		var sectionName = [ZmMetaData.NAMESPACE, s].join(":");
		var sectionData = this._sections[sectionName];
		if (sectionData) {
			this.set(s, sectionData, command);
		}
	}

	if (!batchCommand) {
		if (command.size() > 0) {
			command.run(callback);
		}
	} else {
		if (callback) {
			callback.run();
		}
	}
};

/**
 * Updates the local section cache with the given key/value pair.
 *
 * @param {String}	section	the section to update
 * @param {String}	key		the key to update
 * @param {String}	value		the new value
 */
ZmMetaData.prototype.update =
function(section, key, value) {
	var sectionName = [ZmMetaData.NAMESPACE, section].join(":");
	var sectionObj = this._sections[sectionName];
	if (!sectionObj) {
		sectionObj = this._sections[sectionName] = {};
	}
	sectionObj[key] = value;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmKeyMap")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines the key map.
 */

/**
 * Creates a key map for the ZCS application.
 * @class
 * This class maps keys to actions for the ZCS application. There is a global key map
 * with bindings that apply to any key not handled by the current controller; these
 * global bindings apply across applications (mail, contacts, etc). Key bindings that
 * are context-dependent are tied to a particular controller. If that controller has
 * control, then those bindings will be used.
 * <br/>
 * <br/>
 * Bindings are passed in via the <code>ZmKeys</code> object, which is populated from a properties
 * file. The identifiers used in the properties file must match those used here.
 * 
 * @author Ross Dargahi
 * @author Conrad Damon
 * 
 * @extends		DwtKeyMap
 */
ZmKeyMap = function(subclassInit) {
	if (subclassInit) { return };

	ZmKeyMap._setPreconditions();
	DwtKeyMap.call(this);
	this._load(this._map, ZmKeys);
};

ZmKeyMap.prototype = new DwtKeyMap(true);
ZmKeyMap.prototype.constructor = ZmKeyMap;

ZmKeyMap.prototype.isZmKeyMap = true;
ZmKeyMap.prototype.toString = function() { return "ZmKeyMap"; };


// Map names (must match those in the key properties file ZmKeys.properties)
ZmKeyMap.MAP_ADDRESS			= "address";
ZmKeyMap.MAP_BRIEFCASE			= "briefcase";
ZmKeyMap.MAP_CALENDAR			= "calendar";
ZmKeyMap.MAP_CALL				= "call";
ZmKeyMap.MAP_COMPOSE			= "compose";
ZmKeyMap.MAP_CONTACTS			= "contacts";
ZmKeyMap.MAP_CONVERSATION		= "conversation";
ZmKeyMap.MAP_CONVERSATION_LIST	= "conversationList";
ZmKeyMap.MAP_DL_ADDRESS_LIST	= "dlAddressList";
ZmKeyMap.MAP_EDIT_APPOINTMENT	= "editAppointment";
ZmKeyMap.MAP_EDIT_CONTACT		= "editContact";
ZmKeyMap.MAP_EDIT_TASK			= "editTask";
ZmKeyMap.MAP_GLOBAL				= "global";
ZmKeyMap.MAP_MAIL				= "mail";
ZmKeyMap.MAP_MESSAGE			= "message";
ZmKeyMap.MAP_QUICK_REPLY		= "quickReply";
ZmKeyMap.MAP_OPTIONS			= "options";
ZmKeyMap.MAP_TASKS				= "tasks";
ZmKeyMap.MAP_VIEW_APPOINTMENT	= "viewAppointment";
ZmKeyMap.MAP_VOICEMAIL			= "voicemail";

// Action codes
ZmKeyMap.ADDRESS_PICKER			= "AddressPicker";
ZmKeyMap.ADD_EXTERNAL_CALENDAR	= "AddExternalCalendar";
ZmKeyMap.ATTACHMENT				= "Attachment";
ZmKeyMap.CAL_DAY_VIEW			= "DayView";
ZmKeyMap.CAL_FB_VIEW			= "FBView";
ZmKeyMap.CAL_LIST_VIEW			= "CalListView";
ZmKeyMap.CAL_MONTH_VIEW			= "MonthView";
ZmKeyMap.CAL_WEEK_VIEW			= "WeekView";
ZmKeyMap.CAL_WORK_WEEK_VIEW		= "WorkWeekView";
ZmKeyMap.CALL_MANAGER       	= "CallManager";
ZmKeyMap.CANCEL					= "Cancel";
ZmKeyMap.COLLAPSE				= "Collapse";
ZmKeyMap.COLLAPSE_ALL			= "CollapseAll";
ZmKeyMap.DEL					= "Delete";
ZmKeyMap.SHIFT_DEL				= "ShiftDelete";
ZmKeyMap.DOWNLOAD           	= "Download";
ZmKeyMap.EDIT					= "Edit";
ZmKeyMap.EXPAND					= "Expand";
ZmKeyMap.EXPAND_ALL				= "ExpandAll";
ZmKeyMap.FIRST_UNREAD			= "FirstUnread";
ZmKeyMap.FIRST_UNREAD_MSG		= "FirstUnreadMsg";
ZmKeyMap.FLAG					= "Flag";
ZmKeyMap.FOCUS_CONTENT_PANE		= "FocusContentPane";
ZmKeyMap.FOCUS_SEARCH_BOX		= "FocusSearchBox";
ZmKeyMap.FOCUS_TOOLBAR			= "FocusToolbar";
ZmKeyMap.FORWARD				= "Forward";
ZmKeyMap.GET_MAIL				= "GetMail";
ZmKeyMap.GOTO_BRIEFCASE			= "GoToBriefcase";
ZmKeyMap.GOTO_CALENDAR			= "GoToCalendar";
ZmKeyMap.GOTO_CONTACTS			= "GoToContacts";
ZmKeyMap.GOTO_DRAFTS			= "GoToDrafts";
ZmKeyMap.GOTO_JUNK				= "GoToJunk";
ZmKeyMap.GOTO_INBOX				= "GoToInbox";
ZmKeyMap.GOTO_MAIL				= "GoToMail";
ZmKeyMap.GOTO_OPTIONS			= "GoToOptions";
ZmKeyMap.GOTO_SENT				= "GoToSent";
ZmKeyMap.GOTO_TASKS				= "GoToTasks";
ZmKeyMap.GOTO_TRASH				= "GoToTrash";
ZmKeyMap.GOTO_VOICE				= "GoToVoice";
ZmKeyMap.HTML_FORMAT			= "HtmlFormat";
ZmKeyMap.KEEP_READING			= "KeepReading";
ZmKeyMap.LAST_UNREAD			= "LastUnread";
ZmKeyMap.LAST_UNREAD_MSG		= "LastUnreadMsg";
ZmKeyMap.MARK_COMPLETE			= "MarkComplete";
ZmKeyMap.MARK_HEARD				= "MarkHeard";
ZmKeyMap.MARK_READ				= "MarkRead";
ZmKeyMap.MARK_UNCOMPLETE		= "MarkUncomplete";
ZmKeyMap.MARK_UNHEARD			= "MarkUnheard";
ZmKeyMap.MARK_UNREAD			= "MarkUnread";
ZmKeyMap.MOVE					= "Move";
ZmKeyMap.MOVE_TO_INBOX			= "MoveToInbox";
ZmKeyMap.MOVE_TO_JUNK			= "MoveToJunk";
ZmKeyMap.MOVE_TO_TRASH			= "MoveToTrash";
ZmKeyMap.MUTE_UNMUTE_CONV	    = "MuteUnmuteConv";
ZmKeyMap.NEW					= "New";
ZmKeyMap.NEW_APPT				= "NewAppointment";
ZmKeyMap.NEW_BRIEFCASE			= "NewBriefcase";
ZmKeyMap.NEW_CALENDAR			= "NewCalendar";
ZmKeyMap.NEW_CONTACT			= "NewContact";
ZmKeyMap.NEW_DOC    			= "NewDocument";
ZmKeyMap.NEW_FILE				= "NewFile";
ZmKeyMap.NEW_FOLDER				= "NewFolder";
ZmKeyMap.NEW_MESSAGE			= "NewMessage";
ZmKeyMap.NEW_MESSAGE_WIN		= "NewMessageWindow";
ZmKeyMap.NEW_SEARCH				= "NewSearch";
ZmKeyMap.NEW_TAG				= "NewTag";
ZmKeyMap.NEW_TASK				= "NewTask";
ZmKeyMap.NEW_WINDOW				= "NewWindow";
ZmKeyMap.NEXT_APPT				= "NextAppointment";
ZmKeyMap.NEXT_CONV				= "NextConversation";
ZmKeyMap.NEXT_DAY				= "NextDay";
ZmKeyMap.NEXT_MSG				= "NextMessage";
ZmKeyMap.NEXT_PAGE				= "NextPage";
ZmKeyMap.NEXT_UNREAD			= "NextUnread";
ZmKeyMap.NEXT_UNREAD_MSG		= "NextUnreadMsg";
ZmKeyMap.PLAY					= "Play";
ZmKeyMap.PRESENCE_MENU			= "PresenceMenu";
ZmKeyMap.PREV_APPT				= "PreviousAppointment";
ZmKeyMap.PREV_CONV				= "PreviousConversation";
ZmKeyMap.PREV_DAY				= "PreviousDay";
ZmKeyMap.PREV_MSG				= "PreviousMessage";
ZmKeyMap.PREV_PAGE				= "PreviousPage";
ZmKeyMap.PREV_UNREAD			= "PreviousUnread";
ZmKeyMap.PREV_UNREAD_MSG		= "PreviousUnreadMsg";
ZmKeyMap.PRINT					= "Print";
ZmKeyMap.PRINT_ALL				= "PrintAll";
ZmKeyMap.QUICK_ADD				= "QuickAdd";
ZmKeyMap.QUICK_REMINDER 	    = "QuickReminder";
ZmKeyMap.READING_PANE_BOTTOM	= "ReadingPaneAtBottom";
ZmKeyMap.READING_PANE_OFF		= "ReadingPaneOff";
ZmKeyMap.READING_PANE_RIGHT		= "ReadingPaneOnRight";
ZmKeyMap.REFRESH				= "Refresh";
ZmKeyMap.REPLY					= "Reply";
ZmKeyMap.REPLY_ALL				= "ReplyAll";
ZmKeyMap.SAVE					= "Save";
ZmKeyMap.SAVED_SEARCH			= "SavedSearch";
ZmKeyMap.SELECT_ALL				= "SelectAll";
ZmKeyMap.SEND					= "Send";
ZmKeyMap.SHORTCUTS				= "Shortcuts";
ZmKeyMap.SHOW_FRAGMENT			= "ShowFragment";
ZmKeyMap.SPAM					= "Spam";
ZmKeyMap.SPELLCHECK				= "Spellcheck";
ZmKeyMap.TAG					= "Tag";
ZmKeyMap.TODAY					= "Today";
ZmKeyMap.TOGGLE					= "Toggle";
ZmKeyMap.UNTAG					= "Untag";
ZmKeyMap.VIEW_BY_CONV			= "ViewByConversation";
ZmKeyMap.VIEW_BY_MSG			= "ViewByMessage";
ZmKeyMap.VISIT					= "Visit";
ZmKeyMap.VISIT_TAG				= "VisitTag";

// HTML entities (used to display keys)
ZmKeyMap.ENTITY = {};
ZmKeyMap.ENTITY[DwtKeyMap.ARROW_LEFT]	= "&larr;"
ZmKeyMap.ENTITY[DwtKeyMap.ARROW_RIGHT]	= "&rarr;"
ZmKeyMap.ENTITY[DwtKeyMap.ARROW_UP]		= "&uarr;"
ZmKeyMap.ENTITY[DwtKeyMap.ARROW_DOWN]	= "&darr;"
ZmKeyMap.ENTITY['"'] = "&quot;"
ZmKeyMap.ENTITY['&'] = "&amp;"
ZmKeyMap.ENTITY['<'] = "&lt;"
ZmKeyMap.ENTITY['>'] = "&gt;"
ZmKeyMap.ENTITY[DwtKeyMap.COMMA]		= ",";
ZmKeyMap.ENTITY[DwtKeyMap.SEMICOLON]	= ";";
ZmKeyMap.ENTITY[DwtKeyMap.BACKSLASH] 	= "\\";

// preconditions for maps
ZmKeyMap.MAP_PRECONDITION = {};

// preconditions for specific shortcuts
ZmKeyMap.ACTION_PRECONDITION = {};

ZmKeyMap._setPreconditions =
function() {
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_COMPOSE]				= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_MAIL]				= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CONVERSATION_LIST]	= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CONVERSATION]		= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_DL_ADDRESS_LIST]		= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_MESSAGE]				= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CONTACTS]			= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_EDIT_CONTACT]		= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CALENDAR]			= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_EDIT_APPOINTMENT]	= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_OPTIONS]				= ZmSetting.OPTIONS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_BRIEFCASE]			= ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_TASKS]				= ZmSetting.TASKS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_EDIT_TASK]			= ZmSetting.TASKS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_VOICEMAIL]			= ZmSetting.VOICE_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CALL]				= ZmSetting.VOICE_ENABLED;
	
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL] = {};
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.FOCUS_SEARCH_BOX]	= ZmSetting.SEARCH_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_BRIEFCASE]		= ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_CALENDAR]		= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_CONTACTS]		= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_MAIL]			= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_OPTIONS]		= ZmSetting.OPTIONS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_TASKS]			= ZmSetting.TASKS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_VOICE]			= ZmSetting.VOICE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_APPT]			= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_BRIEFCASEITEM]	= ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_CALENDAR]		= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_CONTACT]			= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_FILE]			= ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_DOC]				= ZmSetting.DOCS_ENABLED && ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_FOLDER]			= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_MESSAGE]			= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_MESSAGE_WIN]		= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_TAG]				= ZmSetting.TAGGING_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_TASK]			= ZmSetting.TASKS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.SAVED_SEARCH]		= ZmSetting.SAVED_SEARCHES_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.TAG]					= ZmSetting.TAGGING_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.UNTAG]				= ZmSetting.TAGGING_ENABLED;

	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE] = {};
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE][ZmKeyMap.ADDRESS_PICKER]		= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE][ZmKeyMap.HTML_FORMAT]		= ZmSetting.HTML_COMPOSE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE][ZmKeyMap.NEW_WINDOW]			= ZmSetting.NEW_WINDOW_COMPOSE;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE][ZmKeyMap.SAVE]				= ZmSetting.SAVE_DRAFT_ENABLED;

	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_EDIT_APPOINTMENT] = {};
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_EDIT_APPOINTMENT][ZmKeyMap.HTML_FORMAT]	= ZmSetting.HTML_COMPOSE_ENABLED;

    ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_CALENDAR] = {};
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_CALENDAR][ZmKeyMap.CAL_FB_VIEW]		= ZmSetting.FREE_BUSY_VIEW_ENABLED;
};

/**
 * Checks if this map is valid. A map may have a precondition,
 * which is either a setting that must be true, or a function that returns
 * true.
 *
 * @param {String}	mapName	the name of map
 * @return	{Boolean}	<code>true</code> if the map is valid
 * 
 * @private
 */
ZmKeyMap.prototype._checkMap = function(mapName) {

	var result = this._checkedMap[mapName] = appCtxt.checkPrecondition(ZmKeyMap.MAP_PRECONDITION[mapName]);
	return result;
};

/**
 * Checks if this action is valid. A map or an action may have a precondition,
 * which is either a setting that must be true, or a function that returns
 * true.
 *
 * @param {String} mapName	the name of map
 * @param {String} action	the action to check
 * @return	{Boolean}	<code>true</code> if the action is valid
 * 
 * @private
 */
ZmKeyMap.prototype._checkAction = function(mapName, action) {

	if (this._checkedMap[mapName] === false || (!this._checkedMap[mapName] && !this._checkMap(mapName))) {
		return false;
	}

	var mapPre = ZmKeyMap.ACTION_PRECONDITION[mapName];
	return appCtxt.checkPrecondition(mapPre && mapPre[action]);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmTimezone")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the timezone class.
 */

/**
 * Creates a timezone
 * @class
 * This class represents a timezone.
 * 
 */
ZmTimezone = function() {}

// Static methods

/**
 * Gets the default timezone.
 * 
 * @return	{AjxTimezone}	the timezone
 */
ZmTimezone.getDefault =
function() {
	var shell = DwtShell.getShell(window);
	var serverId = appCtxt.get(ZmSetting.DEFAULT_TIMEZONE);
	return (serverId) ? AjxTimezone.getClientId(serverId) : AjxTimezone.DEFAULT;
};

/**
 * Gets the default rule.
 * 
 * @return	{String}	the rule
 */
ZmTimezone.getDefaultRule =
function() {
	return AjxTimezone.getRule(ZmTimezone.getDefault());
};

/**
 * This function mirrors the <code>AjxSoapDoc#set</code> method
 * to add a timezone element at the specific place within the
 * given SOAP document. The added element takes the form of the
 * <code>&lt;tz></code> element as defined for <code>&lt;SearchRequest&gt;</code>.
 *
 * @param {object|AjxSoapDoc}	request			the JSON request object or SOAP document
 * @param {String}	timezoneClientId	the client identifier
 * @param {Node}	parentNode		(optional) the parent node at which to add
 * @param {Boolean}	skipKnownTimezone	(optional) if <code>true</code>, does not add the "tz" element if it's one of the known set
 */
ZmTimezone.set =
function(request, timezoneClientId, parentNode, skipKnownTimezone) {
	var timezone = AjxTimezone.getRule(timezoneClientId);
	if (!timezone) { return; }

	if (timezone.autoDetected || !skipKnownTimezone) {
		if (request instanceof AjxSoapDoc) {
			ZmTimezone._setSoap(request, timezoneClientId, parentNode, timezone);
		} else {
			ZmTimezone._setJson(request, timezoneClientId, timezone);
		}
	}
};

/**
 * @private
 */
ZmTimezone._setSoap =
function(soapDoc, timezoneClientId, parentNode, timezone) {
	var tz = soapDoc.set("tz", null, parentNode);
	var id = AjxTimezone.getServerId(timezoneClientId);
	tz.setAttribute("id", id);
	if (timezone.autoDetected) {
		tz.setAttribute("stdoff", timezone.standard.offset);
		if (timezone.daylight) {
			tz.setAttribute("dayoff", timezone.daylight.offset);
            var enames = [ "standard", "daylight" ];
            var pnames = [ "mon", "mday", "week", "wkday", "hour", "min", "sec" ];
            for (var i = 0; i < enames.length; i++) {
                var ename = enames[i];
                var onset = timezone[ename];
                
                var el = soapDoc.set(ename, null, tz);
                for (var j = 0; j < pnames.length; j++) {
                    var pname = pnames[j];
                    if (pname in onset) {
                        el.setAttribute(pname, onset[pname]);
                    }
                }
            }
        }
	}
};

/**
 * @private
 */
ZmTimezone._setJson =
function(request, timezoneClientId, timezone) {
	var id = AjxTimezone.getServerId(timezoneClientId);
	var tz = request.tz = {id:id};
	if (timezone.autoDetected) {
		tz.stdoff = timezone.standard.offset;
		if (timezone.daylight) {
			tz.dayoff = timezone.daylight.offset;
            var enames = [ "standard", "daylight" ];
            var pnames = [ "mon", "mday", "week", "wkday", "hour", "min", "sec" ];
            for (var i = 0; i < enames.length; i++) {
                var ename = enames[i];
                var onset = timezone[ename];
                tz[ename] = {};
                for (var j = 0; j < pnames.length; j++) {
                    var pname = pnames[j];
                    if (pname in onset) {
                    	tz[ename][pname] = onset[pname];
                    }
                }
            }
        }
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmItem")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines an item.
 */

/**
 * Creates an item.
 * @class
 * An item is a piece of data that may contain user content. Most items are taggable. Currently,
 * the following things are items: conversation, message, attachment, appointment, and contact.
 * <br/>
 * <br/>
 * An item typically appears in the context of a containing list. Its event handling
 * is generally handled by the list so we avoid having the same listeners on each item. If we
 * create a context where an item stands alone outside a list context, then the item will have
 * its own listeners and do its own notification handling.
 *
 * @author Conrad Damon
 * 
 * @param {constant}	type		type of object (conv, msg, etc)
 * @param {int}			id			the unique id
 * @param {ZmList}		list		a list that contains this item
 * @param {Boolean}		noCache		if <code>true</code>, do not cache this item
 * 
 * @extends		ZmModel
 */
ZmItem = function(type, id, list, noCache) {

	if (arguments.length == 0) { return; }
	ZmModel.call(this, type);

	this.type = type;
	this.id = id;
	this.list = list;
	this._list = {};

    // number of views using this item
    this.refCount = 0;

	this.tags = [];
	this.tagHash = {};
	this.folderId = 0;

	// make sure the cached item knows which lists it is in, even if those other lists
	// have separate instances of this item - propagate view IDs from currently cached item
	var curItem = appCtxt.getById(id);
	if (curItem) {
		this._list = AjxUtil.hashCopy(curItem._list);
        if (!list) {
            // No list specified, preserve the previous list
            this.list = curItem.list;
        }
	}
	if (list) {
		this._list[list.id] = true;
	}
	
	if (id && !noCache) {
		appCtxt.cacheSet(id, this);
	}
};

ZmItem.prototype = new ZmModel;
ZmItem.prototype.constructor = ZmItem;

ZmItem.prototype.isZmItem = true;
ZmItem.prototype.toString = function() { return "ZmItem"; };


ZmItem.APP 				= {};	// App responsible for item
ZmItem.MSG_KEY 			= {};	// Type names
ZmItem.ICON 			= {};	// Representative icons
ZmItem.RESULTS_LIST 	= {};	// Function for creating search results list

// fields that can be part of a displayed item
ZmItem.F_ACCOUNT		= ZmId.FLD_ACCOUNT;
ZmItem.F_ATTACHMENT		= ZmId.FLD_ATTACHMENT;
ZmItem.F_CAPACITY		= ZmId.FLD_CAPACITY;
ZmItem.F_COMPANY		= ZmId.FLD_COMPANY;
ZmItem.F_DATE			= ZmId.FLD_DATE;
ZmItem.F_DEPARTMENT		= ZmId.FLD_DEPARTMENT;
ZmItem.F_EMAIL			= ZmId.FLD_EMAIL;
ZmItem.F_EXPAND			= ZmId.FLD_EXPAND;
ZmItem.F_FILE_TYPE		= ZmId.FLD_FILE_TYPE;
ZmItem.F_FLAG			= ZmId.FLD_FLAG;
ZmItem.F_FOLDER			= ZmId.FLD_FOLDER;
ZmItem.F_FRAGMENT		= ZmId.FLD_FRAGMENT;
ZmItem.F_FROM			= ZmId.FLD_FROM;
ZmItem.F_HOME_PHONE		= ZmId.FLD_HOME_PHONE;
ZmItem.F_ID				= ZmId.FLD_ID;
ZmItem.F_INDEX			= ZmId.FLD_INDEX;
ZmItem.F_ITEM_ROW		= ZmId.FLD_ITEM_ROW;
ZmItem.F_ITEM_ROW_3PANE	= ZmId.FLD_ITEM_ROW_3PANE;
ZmItem.F_LOCATION		= ZmId.FLD_LOCATION;
ZmItem.F_NAME			= ZmId.FLD_NAME;
ZmItem.F_NOTES			= ZmId.FLD_NOTES;
ZmItem.F_PARTICIPANT	= ZmId.FLD_PARTICIPANT;
ZmItem.F_PCOMPLETE		= ZmId.FLD_PCOMPLETE;
ZmItem.F_PRIORITY		= ZmId.FLD_PRIORITY;
ZmItem.F_RECURRENCE		= ZmId.FLD_RECURRENCE;
ZmItem.F_SELECTION		= ZmId.FLD_SELECTION;
ZmItem.F_SELECTION_CELL	= ZmId.FLD_SELECTION_CELL;
ZmItem.F_SIZE			= ZmId.FLD_SIZE;
ZmItem.F_SORTED_BY		= ZmId.FLD_SORTED_BY;	// placeholder for 3-pane view
ZmItem.F_STATUS			= ZmId.FLD_STATUS;
ZmItem.F_READ			= ZmId.FLD_READ;
ZmItem.F_MUTE			= ZmId.FLD_MUTE;
ZmItem.F_SUBJECT		= ZmId.FLD_SUBJECT;
ZmItem.F_TAG			= ZmId.FLD_TAG;
ZmItem.F_TAG_CELL		= ZmId.FLD_TAG_CELL;
ZmItem.F_TO             = ZmId.FLD_TO;
ZmItem.F_TYPE			= ZmId.FLD_TYPE;
ZmItem.F_VERSION        = ZmId.FLD_VERSION;
ZmItem.F_WORK_PHONE		= ZmId.FLD_WORK_PHONE;
ZmItem.F_LOCK           = ZmId.FLD_LOCK;
ZmItem.F_SHARES			= ZmId.FLD_SHARES;
ZmItem.F_MSG_PRIORITY   = ZmId.FLD_MSG_PRIORITY;
ZmItem.F_APP_PASSCODE_CREATED = ZmId.FLD_CREATED;
ZmItem.F_APP_PASSCODE_LAST_USED = ZmId.FLD_LAST_USED;

// Action requests for different items
ZmItem.SOAP_CMD = {};

// Item fields (for modify events)
ZmItem.TAGS_FIELD = 1;

// Item flags
ZmItem.FLAG_ATTACH				= "a";
ZmItem.FLAG_FLAGGED				= "f";
ZmItem.FLAG_FORWARDED			= "w";
ZmItem.FLAG_ISDRAFT 			= "d";
ZmItem.FLAG_ISSCHEDULED 		= "c";
ZmItem.FLAG_ISSENT				= "s";
ZmItem.FLAG_READ_RECEIPT_SENT	= "n";
ZmItem.FLAG_REPLIED				= "r";
ZmItem.FLAG_UNREAD				= "u";
ZmItem.FLAG_MUTE				= "(";
ZmItem.FLAG_LOW_PRIORITY		= "?";
ZmItem.FLAG_HIGH_PRIORITY		= "!";
ZmItem.FLAG_PRIORITY            = "+"; //msg prioritization
ZmItem.FLAG_NOTE                = "t"; //specially for notes
ZmItem.FLAG_OFFLINE_CREATED     = "o";

ZmItem.ALL_FLAGS = [
	ZmItem.FLAG_FLAGGED,
	ZmItem.FLAG_ATTACH,
	ZmItem.FLAG_UNREAD,
	ZmItem.FLAG_MUTE,
	ZmItem.FLAG_REPLIED,
	ZmItem.FLAG_FORWARDED,
	ZmItem.FLAG_ISSENT,
	ZmItem.FLAG_READ_RECEIPT_SENT,
	ZmItem.FLAG_ISDRAFT,
	ZmItem.FLAG_ISSCHEDULED,
	ZmItem.FLAG_HIGH_PRIORITY,
	ZmItem.FLAG_LOW_PRIORITY,
	ZmItem.FLAG_PRIORITY,
    ZmItem.FLAG_NOTE,
    ZmItem.FLAG_OFFLINE_CREATED
];

// Map flag to item property
ZmItem.FLAG_PROP = {};
ZmItem.FLAG_PROP[ZmItem.FLAG_ATTACH]			= "hasAttach";
ZmItem.FLAG_PROP[ZmItem.FLAG_FLAGGED]			= "isFlagged";
ZmItem.FLAG_PROP[ZmItem.FLAG_FORWARDED]			= "isForwarded";
ZmItem.FLAG_PROP[ZmItem.FLAG_ISDRAFT] 			= "isDraft";
ZmItem.FLAG_PROP[ZmItem.FLAG_ISSCHEDULED] 		= "isScheduled";
ZmItem.FLAG_PROP[ZmItem.FLAG_ISSENT]			= "isSent";
ZmItem.FLAG_PROP[ZmItem.FLAG_READ_RECEIPT_SENT]	= "readReceiptSent";
ZmItem.FLAG_PROP[ZmItem.FLAG_REPLIED]			= "isReplied";
ZmItem.FLAG_PROP[ZmItem.FLAG_UNREAD]			= "isUnread";
ZmItem.FLAG_PROP[ZmItem.FLAG_MUTE]			    = "isMute";
ZmItem.FLAG_PROP[ZmItem.FLAG_LOW_PRIORITY]		= "isLowPriority";
ZmItem.FLAG_PROP[ZmItem.FLAG_HIGH_PRIORITY]		= "isHighPriority";
ZmItem.FLAG_PROP[ZmItem.FLAG_PRIORITY]          = "isPriority";
ZmItem.FLAG_PROP[ZmItem.FLAG_NOTE]              = "isNote";
ZmItem.FLAG_PROP[ZmItem.FLAG_OFFLINE_CREATED]   = "isOfflineCreated";

// DnD actions this item is allowed

/**
 * Defines the "move" action.
 * 
 * @see		#getDefaultDndAction
 */
ZmItem.DND_ACTION_MOVE = 1 << 0;
/**
 * Defines the "copy" action.
 * 
 * @see		#getDefaultDndAction
 */
ZmItem.DND_ACTION_COPY = 1 << 1;
/**
 * Defines the "move & copy" action.
 * 
 * @see		#getDefaultDndAction
 */
ZmItem.DND_ACTION_BOTH = ZmItem.DND_ACTION_MOVE | ZmItem.DND_ACTION_COPY;

/**
 * Defines the notes separator which is used by items
 * (such as calendar or share invites) that have notes.
 * 
 */
ZmItem.NOTES_SEPARATOR			= "*~*~*~*~*~*~*~*~*~*";

/**
 * Registers an item and stores information about the given item type.
 *
 * @param {constant}	item		the item type
 * @param	{Hash}	params			a hash of parameters
 * @param {constant}	params.app			the app that handles this item type
 * @param {String}		params.nameKey		the message key for item name
 * @param {String}		params.icon			the name of item icon class
 * @param {String}		params.soapCmd		the SOAP command for acting on this item
 * @param {String}		params.itemClass	the name of class that represents this item
 * @param {String}		params.node			the SOAP response node for this item
 * @param {constant}	params.organizer	the associated organizer
 * @param {String}		params.searchType	the associated type in SearchRequest
 * @param {function}	params.resultsList	the function that returns a {@link ZmList} for holding search results of this type
 */
ZmItem.registerItem =
function(item, params) {
	if (params.app)				{ ZmItem.APP[item]					= params.app; }
	if (params.nameKey)			{ ZmItem.MSG_KEY[item]				= params.nameKey; }
	if (params.icon)			{ ZmItem.ICON[item]					= params.icon; }
	if (params.soapCmd)			{ ZmItem.SOAP_CMD[item]				= params.soapCmd; }
	if (params.itemClass)		{ ZmList.ITEM_CLASS[item]			= params.itemClass; }
	if (params.node)			{ ZmList.NODE[item]					= params.node; }
	if (params.organizer)		{ ZmOrganizer.ITEM_ORGANIZER[item]	= params.organizer; }
	if (params.searchType)		{ ZmSearch.TYPE[item]				= params.searchType; }
	if (params.resultsList)		{ ZmItem.RESULTS_LIST[item]			= params.resultsList; }

	if (params.node) {
		ZmList.ITEM_TYPE[params.node] = item;
	}

	if (params.dropTargets) {
		if (!ZmApp.DROP_TARGETS[params.app]) {
			ZmApp.DROP_TARGETS[params.app] = {};
		}
		ZmApp.DROP_TARGETS[params.app][item] = params.dropTargets;
	}
};

/**
* Gets an item id by taking a normalized id (or an item id) and returning the item id.
* 
* @param	{String}	id		the normalized id
* @return	{String}	the item id
*/
ZmItem.getItemId =
function(id) {
	if (!id) {
		return id;
	}
	if (!ZmItem.SHORT_ID_RE) {
		var shell = DwtShell.getShell(window);
		ZmItem.SHORT_ID_RE = new RegExp(appCtxt.get(ZmSetting.USERID) + ':', "gi");
	}
	return id.replace(ZmItem.SHORT_ID_RE, '');
};

// abstract methods
/**
 * Creates an item.
 * 
 * @param	{Hash}	args		the arguments
 */
ZmItem.prototype.create = function(args) {};
/**
 * Modifies an item.
 * 
 * @param	{Hash}	mods		the arguments
 */
ZmItem.prototype.modify = function(mods) {};

/**
 * Gets the item by id.
 *
 * @param {String}	id		an item id
 * @return	{ZmItem}	the item
 */
ZmItem.prototype.getById =
function(id) {
	if (id == this.id) {
		return this;
	}
};

ZmItem.prototype.getAccount =
function() {
	if (!this.account) {
		var account;

		if (this.folderId) {
			var ac = window.parentAppCtxt || window.appCtxt;
			var folder = ac.getById(this.folderId);
			account = folder && folder.getAccount();
		}

		if (!account) {
			var parsed = ZmOrganizer.parseId(this.id);
			account = parsed && parsed.account;
		}
		this.account = account;
	}
	return this.account;
};

/**
 * Clears the item.
 * 
 */
ZmItem.prototype.clear = function() {

    // only clear data if no views are using this item
    if (this.refCount <= 1) {
        this._evtMgr.removeAll(ZmEvent.L_MODIFY);
        if (this.tags.length) {
            for (var i = 0; i < this.tags.length; i++) {
                this.tags[i] = null;
            }
            this.tags = [];
        }
        for (var i in this.tagHash) {
            this.tagHash[i] = null;
        }
        this.tagHash = {};
    }

    this.refCount--;
};

/**
 * Caches the item.
 * 
 * @return	{Boolean}	<code>true</code> if the item is placed into cache; <code>false</code> otherwise
 */
ZmItem.prototype.cache =
function(){
  if (this.id) {
      appCtxt.cacheSet(this.id, this);
      return true;
  }
  return false;  
};

/**
 * Checks if the item has a given tag.
 * 
 * @param {String}		tagName		tag name
 * @return	{Boolean}	<code>true</code> is this item has the given tag.
 */
ZmItem.prototype.hasTag =
function(tagName) {
	return (this.tagHash[tagName] == true);
};

/**
 * is it possible to add a tag to this item?
 * @param tagName
 * @returns {boolean}
 */
ZmItem.prototype.canAddTag =
function(tagName) {
	return !this.hasTag(tagName);
};


/**
* Gets the folder id that contains this item, if available.
* 
* @return	{String}	the folder id or <code>null</code> for none
*/
ZmItem.prototype.getFolderId =
function() {
	return this.folderId;
};

/**
 * @deprecated
 * Use getRestUrl
 * 
 * @private
 * @see		#getRestUrl
 */
ZmItem.prototype.getUrl =
function() {
	return this.getRestUrl();
};

/**
 * Gets the rest url for this item.
 * 
 * @return	{String}	the url
 */
ZmItem.prototype.getRestUrl =
function() {
	// return REST URL as seen by server
	if (this.restUrl) {
		return this.restUrl;
	}

	// if server doesn't tell us what URL to use, do our best to generate
	var organizerType = ZmOrganizer.ITEM_ORGANIZER[this.type];
	var organizer = appCtxt.getById(this.folderId);
	var url = organizer
		? ([organizer.getRestUrl(), "/", AjxStringUtil.urlComponentEncode(this.name)].join(""))
		: null;

	if (url && this.folderId == ZmFolder.ID_FILE_SHARED_WITH_ME) {
		if (this.sfid) {
			url = [url, "?", "id=", this.sfid].join("");
		} else {
			url = [url, "?", "id=", this.id].join("");
		}
	}

	DBG.println(AjxDebug.DBG3, "NO REST URL FROM SERVER. GENERATED URL: " + url);

	return url;
};

/**
* Gets the appropriate tag image info for this item.
* 
* @return	{String}	the tag image info
*/
ZmItem.prototype.getTagImageInfo =
function() {
	return this.getTagImageFromNames(this.getVisibleTags());
};

/**
 * @deprecated
 * */
ZmItem.prototype.getTagImageFromIds =
function(tagIds) {
	var tagImageInfo;

	if (!tagIds || tagIds.length == 0) {
		tagImageInfo = "Blank_16";
	} else if (tagIds.length == 1) {
        tagImageInfo = this.getTagImage(tagIds[0]);
	} else {
		tagImageInfo = "TagStack";
	}

	return tagImageInfo;
};

ZmItem.prototype.getVisibleTags =
function() {
    if(!appCtxt.get(ZmSetting.TAGGING_ENABLED)){
        return [];
    }
    return this.tags;
	//todo - do we need anything from this?
//    var searchAll = appCtxt.getSearchController().searchAllAccounts;
//    if (!searchAll && this.isShared()) {
//        return [];
//    } else {
//        return this.tags;
//    }
};

ZmItem.prototype.getTagImageFromNames =
function(tags) {

	if (!tags || tags.length == 0) {
		return "Blank_16";
	}
	if (tags.length == 1) {
        return this.getTagImage(tags[0]);
	} 

	return "TagStack";
};


ZmItem.prototype.getTagImage =
function(tagName) {
	//todo - I don't think we need the qualified/normalized/whatever id anymore.
//	var tagFullId = (!this.getAccount().isMain)
//		? ([this.getAccount().id, tagName].join(":"))
//		: (ZmOrganizer.getSystemId(tagName));
	var tagList = appCtxt.getAccountTagList(this);

	var tag = tagList.getByNameOrRemote(tagName);
    return tag ? tag.getIconWithColor() : "Blank_16";
};

/**
* Gets the default action to use when dragging this item. This method
* is meant to be overloaded for items that are read-only and can only be copied.
*
* @param {Boolean}		forceCopy		If set, default DnD action is a copy
* @return	{Object}	the action
*/
ZmItem.prototype.getDefaultDndAction =
function(forceCopy) {
	return (this.isReadOnly() || forceCopy)
		? ZmItem.DND_ACTION_COPY
		: ZmItem.DND_ACTION_MOVE;
};

/**
* Checks if this item is read-only. This method should be
* overloaded by the derived object to determine what "read-only" means.
* 
* @return	{Boolean}	the read-only status
*/
ZmItem.prototype.isReadOnly =
function() {
	return false;
};

/**
 * Checks if this item is shared.
 * 
 * @return	{Boolean}	<code>true</code> if this item is shared (remote)
 */
ZmItem.prototype.isShared =
function() {
	if (this._isShared == null) {
		if (this.id === -1) {
			this._isShared = false;
		} else {
			this._isShared = appCtxt.isRemoteId(this.id);
		}
	}
	return this._isShared;
};

// Notification handling

// For delete and modify notifications, we first apply the notification to this item. Then we
// see if the item is a member of any other lists. If so, we have those other copies of this
// item handle the notification as well. Each will notify through the list that created it.

ZmItem.prototype.notifyDelete =
function() {
	this._notifyDelete();
	for (var listId in this._list) {
		var list = appCtxt.getById(listId);
		if (!list || (this.list && listId == this.list.id)) { continue; }
		var ctlr = list.controller;
		if (!ctlr || ctlr.inactive || (ctlr.getList().id != listId)) { continue; }
		var doppleganger = list.getById(this.id);
		if (doppleganger) {
			doppleganger._notifyDelete();
		}
	}
};

ZmItem.prototype._notifyDelete =
function() {
	this.deleteLocal();
	if (this.list) {
		this.list.deleteLocal([this]);
	}
	this._notify(ZmEvent.E_DELETE);
};

ZmItem.prototype.notifyModify =
function(obj, batchMode) {
	this._notifyModify(obj, batchMode);
	for (var listId in this._list) {
		var list = listId ? appCtxt.getById(listId) : null;
		if (!list || (this.list && (listId == this.list.id))) { continue; }
		var ctlr = list.controller;
		if (!ctlr || ctlr.inactive || (ctlr.getList().id != listId)) { continue; }
		var doppleganger = list.getById(this.id);
		if (doppleganger) {
			doppleganger._notifyModify(obj, batchMode);
		}
	}
};

/**
 * Handles a modification notification.
 *
 * @param {Object}	obj			the item with the changed attributes/content
 * @param {boolean}	batchMode	if true, return event type and don't notify
 */
ZmItem.prototype._notifyModify =
function(obj, batchMode) {
	// empty string is meaningful here, it means no tags
	if (obj.tn != null) {
		this._parseTagNames(obj.tn);
		this._notify(ZmEvent.E_TAGS);
	}
	// empty string is meaningful here, it means no flags
	if (obj.f != null) {
		var flags = this._getFlags();
		var origFlags = {};
		for (var i = 0; i < flags.length; i++) {
			origFlags[flags[i]] = this[ZmItem.FLAG_PROP[flags[i]]];
		}
		this._parseFlags(obj.f);
		var changedFlags = [];
		for (var i = 0; i < flags.length; i++) {
			var on = this[ZmItem.FLAG_PROP[flags[i]]];
			if (origFlags[flags[i]] != on) {
				changedFlags.push(flags[i]);
			}
		}
		if (changedFlags.length) {
			this._notify(ZmEvent.E_FLAGS, {flags: changedFlags});
		}
	}
	if (obj.l != null && obj.l != this.folderId) {
		var details = {oldFolderId:this.folderId};
		this.moveLocal(obj.l);
		if (this.list) {
			this.list.moveLocal([this], obj.l);
		}
		if (batchMode) {
			delete obj.l;			// folder has been handled
			return ZmEvent.E_MOVE;
		} else {
			this._notify(ZmEvent.E_MOVE, details);
		}
	}
};

// Local change handling

/**
 * Applies the given flag change to this item by setting a boolean property.
 *
 * @param {constant}	flag	the flag that changed
 * @param {Boolean}	on		<code>true</code> if the flag is now set
 */
ZmItem.prototype.flagLocal =
function(flag, on) {
	this[ZmItem.FLAG_PROP[flag]] = on;
};

/**
 * Sets the given flag change to this item. Both the flags string and the
 * flag properties are affected.
 *
 * @param {constant}	flag	the flag that changed
 * @param {Boolean}	on	<code>true</code> if the flag is now set
 *
 * @return	{String}		the new flags string
 */
ZmItem.prototype.setFlag =
function(flag, on) {
	this.flagLocal(flag, on);
	var flags = this.flags || "";
	if (on && flags.indexOf(flag) == -1) {
		flags = flags + flag;
	} else if (!on && flags.indexOf(flag) != -1) {
		flags = flags.replace(flag, "");
	}
	this.flags = flags;

	return flags;
};

/**
 * Adds or removes the given tag for this item.
 *
 * @param {Object}		tag		tag name
 * @param {Boolean}		doTag		<code>true</code> if tag is being added; <code>false</code> if it is being removed
 * @return	{Boolean}	<code>true</code> to notify
 */
ZmItem.prototype.tagLocal =
function(tag, doTag) {
	var bNotify = false;
	if (doTag) {
		if (!this.tagHash[tag]) {
			bNotify = true;
			this.tags.push(tag);
			this.tagHash[tag] = true;
		}
	} else {
		for (var i = 0; i < this.tags.length; i++) {
			if (this.tags[i] == tag) {
				this.tags.splice(i, 1);
				delete this.tagHash[tag];
				bNotify = true;
				break;
			}
		}
	}
	
	return bNotify;
};

/**
 * Removes all tags.
 * 
 */
ZmItem.prototype.removeAllTagsLocal =
function() {
	this.tags = [];
	for (var i in this.tagHash) {
		delete this.tagHash[i];
	}
};

/**
 * Deletes local, in case an item wants to do something while being deleted.
 */
ZmItem.prototype.deleteLocal = function() {};

/**
 * Moves the item.
 * 
 * @param	{String}	folderId
 * @param	{AjxCallback}	callback		the callback
 * @param	{AjxCallback}	errorCallback	the callback on error
 * @return	{Object}		the result of the move
 */
ZmItem.prototype.move =
function(folderId, callback, errorCallback) {
	return ZmItem.move(this.id, folderId, callback, errorCallback);
};

/**
 * Moves the item.
 * 
 * @return	{Object}		the result of the move
 */
ZmItem.move =
function(itemId, folderId, callback, errorCallback, accountName) {
	var json = {
		ItemActionRequest: {
			_jsns: "urn:zimbraMail",
			action: {
				id:	itemId instanceof Array ? itemId.join() : itemId,
				op:	"move",
				l:	folderId
			}
		}
	};

	var params = {
		jsonObj:		json,
		asyncMode:		Boolean(callback),
		callback:		callback,
		errorCallback:	errorCallback,
		accountName:	accountName
	};
	return appCtxt.getAppController().sendRequest(params);
};

/**
 * Updates the folder for this item.
 *
 * @param {String}		folderId		the new folder ID
 */
ZmItem.prototype.moveLocal =
function(folderId) {
	this.folderId = folderId;
};

/**
 * Takes a comma-separated list of tag IDs and applies the tags to this item.
 * 
 * @private
 */
ZmItem.prototype._parseTags =
function(str) {	
	this.tags = [];
	this.tagHash = {};
	if (str && str.length) {
		var tags = str.split(",");
		for (var i = 0; i < tags.length; i++) {
			var tagId = Number(tags[i]);
			if (tagId >= ZmOrganizer.FIRST_USER_ID[ZmOrganizer.TAG])
				this.tagLocal(tagId, true);
		}
	}
};

/**
 * Takes a comma-separated list of tag names and applies the tags to this item.
 *
 * @private
 */
ZmItem.prototype._parseTagNames =
function(str) {
	this.tags = [];
	this.tagHash = {};
	if (!str || !str.length) {
		return;
	}
	
	// server escapes comma with backslash
	str = str.replace(/\\,/g, "\u001D");
	var tags = str.split(",");
	
	for (var i = 0; i < tags.length; i++) {
		var tagName = tags[i].replace("\u001D", ",");
		this.tagLocal(tagName, true);
	}
};

/**
 * Takes a string of flag chars and applies them to this item.
 * 
 * @private
 */
ZmItem.prototype._parseFlags =
function(str) {
	this.flags = str;
	for (var i = 0; i < ZmItem.ALL_FLAGS.length; i++) {
		var flag = ZmItem.ALL_FLAGS[i];
		var on = (str && (str.indexOf(flag) != -1)) ? true : false;
		this.flagLocal(flag, on);
	}
};

// Listener notification

/**
 * Notify the list as well as this item.
 * 
 * @private
 */
ZmItem.prototype._notify =
function(event, details) {
	this._doNotify(event, details);
};

ZmItem.prototype._setupNotify =
function() {
    this._doNotify();
}

ZmItem.prototype._doNotify =
function(event, details) {
	if (this._evt) {
		this._evt.item = this;
		if (event != null) {
			ZmModel.prototype._notify.call(this, event, details);
		}
	} else {
		var idText = "";
		if (this.type && this.id) {
			idText = ": item = " + this.type + "(" + this.id + ")";
		}
		DBG.println(AjxDebug.DBG1, "ZmItem._doNotify, missing _evt" + idText);
	}
    if (this.list) {
        this.list._evt.item = this;
        this.list._evt.items = [this];
        if (event != null) {
            if (details) {
                details.items = [this];
            } else {
                details = {items: [this]};
            }
            this.list._notify(event, details);
        }
    }
};

/**
 * Returns a list of flags that apply to this type of item.
 * 
 * @private
 */
ZmItem.prototype._getFlags =
function() {
	return [ZmItem.FLAG_FLAGGED, ZmItem.FLAG_ATTACH];
};

/**
 * Rename the item.
 *
 * @param	{String}	newName
 * @param	{AjxCallback}	callback		the callback
 * @param	{AjxCallback}	errorCallback	the callback on error
 * @return	{Object}		the result of the move
 */
ZmItem.prototype.rename =
function(newName, callback, errorCallback) {
	return ZmItem.rename(this.id, newName, callback, errorCallback);
};

/**
 * Rename the item.
 *
 * @return	{Object}		the result of the move
 */
ZmItem.rename =
function(itemId, newName, callback, errorCallback, accountName) {
    var json = {
		ItemActionRequest: {
			_jsns: "urn:zimbraMail",
			action: {
				id:	itemId instanceof Array ? itemId[0] : itemId,
				op:	"rename",
				name:	newName
			}
		}
	};	

	var params = {
		jsonObj:		json,
		asyncMode:		Boolean(callback),
		callback:		callback,
		errorCallback:	errorCallback,
		accountName:	accountName
	};
	return appCtxt.getAppController().sendRequest(params);
};

ZmItem.prototype.getSortedTags =
function() {
	var numTags = this.tags && this.tags.length;
	if (numTags) {
		var tagList = appCtxt.getAccountTagList(this);
		var ta = [];
		for (var i = 0; i < numTags; i++) {
			var tag = tagList.getByNameOrRemote(this.tags[i]);
			//tag could be missing if this was called when deleting a whole tag (not just untagging one message). So this makes sure we don't have a null item.
			if (!tag) {
				continue;
			}
			ta.push(tag);
		}
		ta.sort(ZmTag.sortCompare);
		return ta;
	}
	return null;
};

}
if (AjxPackage.define("zimbraMail.share.model.ZmActionStack")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2010, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2010, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * @class
 * Creates a stack of undoable actions (ZmAction objects)
 *
 * @param	{int}	[maxLength]		The maximum size of the stack. Defaults to 0, meaning no limit
 *
 * Adding actions to a full stack will pop the oldest actions off
 */
ZmActionStack = function(maxLength) {
	this._stack = [];
	this._pointer = -1;
	this._maxLength = maxLength || 0; // 0 means no limit
};

ZmEvent.S_ACTION			= "ACTION";

ZmActionStack.validTypes	= [ZmId.ORG_FOLDER, ZmId.ITEM_MSG, ZmId.ITEM_CONV,
                               ZmId.ITEM_CONTACT, ZmId.ITEM_GROUP,
                               ZmId.ITEM_BRIEFCASE, ZmId.ORG_BRIEFCASE,
                               ZmId.ITEM_TASK, ZmId.ORG_TASKS
                              ]; // Set ZmActionStack.validTypes to false to allow all item types

ZmActionStack.prototype.toString = function() {
	return "ZmActionStack";
};

/**
 * Logs a raw action, interpreting the params and creates a ZmAction object that is pushed onto the stack and returned
 * 
 * @param {String}	op			operation to perform. Currently supported are "move", "trash", "spam" and "!spam"
 * @param {Hash}	[attrs] 	attributes for the operation. Pretty much the same as what the backend expects, e.g. "l" for the destination folderId of a move

 * @param {String}	[items] 	array of items to perform the action for. Valid types are specified in ZmActionStack.validTypes. Only one of [items],[item],[ids] or [id] should be specified; the first one found is used, ignoring the rest.
 * @param {String}	[item] 		item to perform the action for, if there is only one item. Accomplishes the same as putting the item in an array and giving it as [items]
 * @param {String}	[ids] 		array of ids of items to perform the action for.
 * @param {String}	[id] 		id of item to perform the action for, if there is only one. Accomplishes the same as putting the id in an array and giving it as [ids].
 */
ZmActionStack.prototype.logAction = function(params) {

	var op = params.op,
	    items = [];

	if (params.items) {
		for (var i = 0; i < params.items.length; i++) {
			var item = params.items[i];
			if (item && this._isValidType(item.type)) {
				items.push(item);
			}
		}
	}
	else if (params.item) {
		if (params.item && this._isValidType(params.item.type)) {
			items.push(params.item);
		}
	}
	else if (params.ids) {
		for (var i = 0; i < params.ids.length; i++) {
			var item = appCtxt.getById(params.ids[i]);
			if (item && this._isValidType(item.type)) {
				items.push(item);
			}
		}
	}
	else if (params.id) {
		var item = appCtxt.getById(params.id);
		if (item && this._isValidType(item.type)) {
			items.push(item);
		}
	}

	var attrs = params.attrs;

	// for a conv, create a list of undoable msg moves so msgs can be restored to their disparate original folders
	for (var i = 0; i < items.length; i++) {
		var item = items[i];
		if (item.type === ZmItem.CONV) {
			var tcon = attrs && attrs.tcon;
			for (var msgId in item.msgFolder) {
				var folderId = item.msgFolder[msgId],
					tconCode = ZmFolder.TCON_CODE[folderId];

				// if tcon kept us from moving a msg, no need to undo it
				if (!tcon || tcon.indexOf(tconCode) === -1) {
					items.push({
						isConvMsg:  true,
						id:         msgId,
						type:       ZmItem.MSG,
						folderId:   folderId,
						list:       { type: ZmItem.MSG }    // hack to expose item.list.type
					});
				}
			}
		}
	}

	var multi = items.length > 1;

	var action = null;
	var folderId;
	switch (op) {
		case "trash":
			folderId = ZmFolder.ID_TRASH;
			break;
		case "spam":
			folderId = ZmFolder.ID_SPAM;
			break;
		case "move":
		case "!spam":
			folderId = attrs.l;
			break;
	}

	var folder = appCtxt.getById(folderId);
	if (folder && !folder.isRemote()) { // Enable undo only when destination folder exists (it should!!) and is not remote (bug #51656)
		switch (op) {
			case "trash":
			case "move":
			case "spam":
			case "!spam":
				for (var i = 0; i < items.length; i++) {
					var item = items[i];
					var moveAction;
				
					if (item instanceof ZmItem) {
						if (!item.isShared()) { // Moving shared items is not undoable
							moveAction = new ZmItemMoveAction(item, item.getFolderId(), folderId, op);
						}
					}
					else if (item instanceof ZmOrganizer) {
						if (!item.isRemote()) { // Moving remote organizers is not undoable
							moveAction = new ZmOrganizerMoveAction(item, item.parent.id, folderId, op);
						}
					}
					else if (item.isConvMsg) {
						if (!appCtxt.isRemoteId(item.id)) {
							moveAction = new ZmItemMoveAction(item, item.folderId, folderId, op);
						}
					}
					if (moveAction) {
						if (multi) {
							if (!action) action = new ZmCompositeAction(folderId);
							action.addAction(moveAction);
						} else {
							action = moveAction;
						}
					}
				}
				break;
		}
		if (action) {
			this._push(action);
		}
	}

	return action;
};

/**
 * Returns whether there are actions that can be undone
 */
ZmActionStack.prototype.canUndo = function() {
	return this._pointer >= 0;
};

/**
 * Returns whether there are actions that can be redone
 */
ZmActionStack.prototype.canRedo = function() {
	return this._pointer < this._stack.length - 1;
};


/**
 * Returns whether the next undo action has completed
 */
ZmActionStack.prototype.actionIsComplete = function() {
	return this.canUndo() && this._current().getComplete();
};

/**
 * Attaches a completion callback to the current action
 */
ZmActionStack.prototype.onComplete = function(callback) {
	var action = this._current();
	if (action) {
		action.onComplete(callback);
	}
};

/**
 * Undoes the current action (if applicable) and moves the internal pointer
 */
ZmActionStack.prototype.undo = function() {
	if (this.canUndo()) {
		var action = this._pop();
		action.undo();
	}
};

/**
 * Redoes the current action (if applicable) and moves the internal pointer
 */
ZmActionStack.prototype.redo = function() {
	if (this.canRedo()) {
		var action = this._stack[++this._pointer];
		action.redo();
	}
};

/**
 * Puts an action into the stack at the current position
 * If we're not at the top of the stack (ie. undoes have been performed), we kill all later actions (so redoing the undone actions is no longer possible)
 */
ZmActionStack.prototype._push = function(action) {
	if (action && action instanceof ZmAction) {
		var next = this._pointer + 1;
		while (this._maxLength && next>=this._maxLength) {
			// Stack size is reached, shift off actions until we're under the limit
			this._stack.shift();
			next--;
		}
		this._stack[next] = action;
		this._stack.length = next+1; // Kill all actions after pointer
		this._pointer = next;
	}
};

/**
 * Returns the action at the current position and moves the pointer
 */
ZmActionStack.prototype._pop = function() {
	return this.canUndo() ? this._stack[this._pointer--] : null;
};

/**
 * Returns the action at the current position, does not move the pointer
 */
ZmActionStack.prototype._current = function() {
	return this.canUndo() ? this._stack[this._pointer] : null;
};

/**
 * Returns true if the given type is valid.
 */
ZmActionStack.prototype._isValidType = function(type) {
	return !ZmActionStack.validTypes || AjxUtil.indexOf(ZmActionStack.validTypes, type) !== -1;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmAction")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * @class
 * Represents an undoable action (e.g. move an item)
 * This class is a generic superclass that does very little on its own; the real work is being done in subclasses
 * 
 * @extends		ZmModel
 */

ZmAction = function() {
	ZmModel.call(this, ZmEvent.S_ACTION);
	this._complete = false;
};

ZmAction.prototype = new ZmModel;
ZmAction.prototype.constructor = ZmAction;

ZmAction.ACTION_ZMACTION = "ZmAction";
ZmAction.ACTION_ZMITEMACTION = "ZmItemAction";
ZmAction.ACTION_ZMITEMMOVEACTION = "ZmItemMoveAction";
ZmAction.ACTION_ZMITEMTRASHACTION = "ZmItemTrashAction";
ZmAction.ACTION_ZMORGANIZERACTION = "ZmOrganizerAction";
ZmAction.ACTION_ZMORGANIZERMOVEACTION = "ZmOrganizerMoveAction";
ZmAction.ACTION_ZMCOMPOSITEACTION = "ZmCompositeAction";

ZmAction.prototype.type = ZmAction.ACTION_ZMITEMACTION;

ZmAction.prototype.toString = function() {
	return "ZmAction";
};

ZmAction.prototype.undo = function() {
	//override me
};

ZmAction.prototype.redo = function() {
	//override me
};

ZmAction.prototype.setComplete = function() {
	if (!this._complete) {
		this._complete = true;
		this._notify(ZmEvent.E_COMPLETE);
	}
};

ZmAction.prototype.getComplete = function() {
	return this._complete;
};

ZmAction.prototype.onComplete = function(callback) {
	if (this._complete) {
		callback.run(this);
	} else {
		this.addChangeListener(new AjxListener(this, this._handleComplete, [callback]));
	}
};

ZmAction.prototype._handleComplete = function(callback, event) {
	if (event.event===ZmEvent.E_COMPLETE) {
		callback.run(this);
	}
};

/**
 * @class
 * Represents an undoable action on an item
 * This class is a generic superclass that does very little on its own; the real work is being done in subclasses
 * 
 * @extends		ZmAction
 *
 * @param {ZmItem}	item	The item to perform the action on
 * @param {String}	op		The operation to perform (e.g. "move" or "trash")	
 */

ZmItemAction = function(item, op) {
	if (!arguments.length) return;
	ZmAction.call(this);
	this._item = item;
	this._op = op;
};

ZmItemAction.prototype = new ZmAction;
ZmItemAction.prototype.constructor = ZmItemAction;
ZmItemAction.prototype.type = ZmAction.ACTION_ZMITEMACTION;

ZmItemAction.prototype.toString = function() {
	return "ZmItemAction";
};

ZmItemAction.prototype.getItem = function() {
	return this._item;
};

ZmItemAction.prototype.getOp = function() {
	return this._op;
};

/**
 * @class
 * Represents an undoable action on an organizer
 * This class is a generic superclass that does very little on its own; the real work is being done in subclasses
 * 
 * @extends		ZmAction
 *
 * @param {ZmOrganizer}		organizer	The organizer to perform the action on
 * @param {String}			op			The operation to perform (e.g. "move")
 */

ZmOrganizerAction = function(organizer, op) {
	if (!arguments.length) return;
	ZmAction.call(this);
	this._organizer = organizer;
	this._op = op;
};

ZmOrganizerAction.prototype = new ZmAction;
ZmOrganizerAction.prototype.constructor = ZmOrganizerAction;
ZmOrganizerAction.prototype.type = ZmAction.ACTION_ZMORGANIZERACTION;

ZmOrganizerAction.prototype.toString = function() {
	return "ZmOrganizerAction";
};

ZmOrganizerAction.prototype.getOrganizer = function() {
	return this._organizer;
};

ZmOrganizerAction.prototype.getOp = function() {
	return this._op;
};

/**
 * @class
 * Represents an undoable move action on an item
 * 
 * @extends		ZmItemAction
 *
 * @param {ZmItem}	item			Item to perform the move on
 * @param {int}		fromFolderId	Original folder id of the item
 * @param {int}		toFolderId		Destination folder id of the item
 * @param {String}	op				The operation to perform (e.g. "move")
 */

ZmItemMoveAction = function(item, fromFolderId, toFolderId, op) {
	ZmItemAction.call(this, item, op);
	this._fromFolderId = fromFolderId;
	this._toFolderId = toFolderId;
};

ZmItemMoveAction.prototype = new ZmItemAction;
ZmItemMoveAction.prototype.constructor = ZmItemMoveAction;

ZmItemMoveAction.prototype.type = ZmAction.ACTION_ZMITEMMOVEACTION;

ZmItemMoveAction.prototype.toString = function() {
	return "ZmItemMoveAction";
};

ZmItemMoveAction.UNDO_MSG = {
	"move" : ZmMsg.actionUndoMove,
	"trash": ZmMsg.actionUndoTrash,
	"spam": ZmMsg.actionUndoMarkAsJunk,
	"!spam": ZmMsg.actionUndoMarkAsNotJunk
};

ZmItemMoveAction.prototype.getFromFolderId = function() {
	return this._fromFolderId;
};

ZmItemMoveAction.prototype.getToFolderId = function() {
	return this._toFolderId;
};

ZmItemMoveAction.prototype._doMove = function(callback, errorCallback, folderId) {

	var items = ZmItemMoveAction._realizeItems(this._item), // probably unnecessary since conv forces multipleUndo
		list = items[0] && items[0].list;

	list.moveItems({
		items:			items,
		folder:			appCtxt.getById(folderId),
		noUndo:			true,
		finalCallback:	this._handleDoMove.bind(this, this._item.folderId, folderId),
		fromFolderId:   this._toFolderId
	});
};

ZmItemMoveAction.prototype._handleDoMove = function(oldFolderId, newFolderId, params) {
	var lists = [];
	for (var id in params.idHash) {
		var item = params.idHash[id];
		if (item instanceof ZmConv)
			item.folderId = newFolderId;
		var list = item && item.list;
		if (AjxUtil.indexOf(lists, list)==-1)
			lists.push(list);
	}
	for (var i=0; i<lists.length; i++) {
		lists[i]._notify(ZmEvent.E_MOVE, {oldFolderId:oldFolderId});
	}
	ZmListController.handleProgress({state:ZmListController.PROGRESS_DIALOG_CLOSE});
	ZmBaseController.showSummary(params.actionSummary);
};

ZmItemMoveAction.prototype.undo = function(callback, errorCallback) {
	this._doMove(callback, errorCallback, this._fromFolderId);
};

ZmItemMoveAction.prototype.redo = function(callback, errorCallback) {
	this._doMove(callback, errorCallback, this._toFolderId);
};

ZmItemMoveAction.multipleUndo = function(actions, redo, fromFolderId) {

	var sortingTable = {};
	for (var i = 0; i < actions.length; i++) {
		var action = actions[i];
		if (action instanceof ZmItemMoveAction) {
			var from = action.getFromFolderId();
			var to = action.getToFolderId();
			var item = action.getItem();
			var type = (item && item.list && item.list.type) || 0;
			if (!sortingTable[from]) sortingTable[from] = {};
			if (!sortingTable[from][to]) sortingTable[from][to] = {};
			if (!sortingTable[from][to][type]) sortingTable[from][to][type] = [];
			sortingTable[from][to][type].push(action);
		}
	}

	for (var from in sortingTable) {
		for (var to in sortingTable[from]) {
			for (var type in sortingTable[from][to]) {
				var subset = sortingTable[from][to][type];
				var items = [];
				var list = null;
				for (var i = 0; i < subset.length; i++) {
					var action = subset[i];
					var item = action.getItem();
					items.push(item);
				}
				items = ZmItemMoveAction._realizeItems(items);
				list = items[0] && items[0].list;
				if (list) {
					list.moveItems({
						items:          items,
						folder:         appCtxt.getById(redo ? to : from),
						noUndo:         true,
						fromFolderId:   fromFolderId
					});
				}
			}
		}
	}
};

ZmItemMoveAction.multipleRedo = function(actions) {
	ZmItemMoveAction.multipleUndo(actions, true);
};

// Creates ZmMailMsg out of anonymous msg-like objects
ZmItemMoveAction._realizeItems = function(items) {

	var list, msg;
	return AjxUtil.map(AjxUtil.toArray(items), function(item) {
		if (item.isConvMsg) {
			list = list || new ZmMailList(ZmItem.MSG);
			msg = new ZmMailMsg(item.id, list, true);
			msg.folderId = item.folderId;
			return msg;
		}
		else {
			return item;
		}
	});
};

/**
 * @class
 * Represents an undoable move action on an organizer
 * 
 * @extends		ZmOrganizerAction
 *
 * @param {ZmOrganizer}	organizer		Organizer to perform the move on
 * @param {int}			fromFolderId	Original parent folder id of the organizer
 * @param {int}			toFolderId		Destination parent folder id of the organizer
 * @param {String}		op				The operation to perform (e.g. "move")
 */

ZmOrganizerMoveAction = function(organizer, fromFolderId, toFolderId, op) {
	ZmOrganizerAction.call(this, organizer, op);
	this._fromFolderId = fromFolderId;
	this._toFolderId = toFolderId;
};

ZmOrganizerMoveAction.prototype = new ZmOrganizerAction;
ZmOrganizerMoveAction.prototype.constructor = ZmOrganizerMoveAction;

ZmOrganizerMoveAction.prototype.type = ZmAction.ACTION_ZMORGANIZERMOVEACTION;

ZmOrganizerMoveAction.prototype.toString = function() {
	return "ZmOrganizerMoveAction";
};

ZmOrganizerMoveAction.prototype.getFromFolderId = function() {
	return this._fromFolderId;
};

ZmOrganizerMoveAction.prototype.getToFolderId = function() {
	return this._toFolderId;
};

ZmOrganizerMoveAction.prototype._doMove = function(callback, errorCallback, folderId) {
	var folder = appCtxt.getById(folderId);
	if (folder) {
		this._organizer.move(folder, true);
	}
};

ZmOrganizerMoveAction.prototype.undo = function(callback, errorCallback) {
	this._doMove(callback, errorCallback, this._fromFolderId);
};

ZmOrganizerMoveAction.prototype.redo = function(callback, errorCallback) {
	this._doMove(callback, errorCallback, this._toFolderId);
};

ZmOrganizerMoveAction.multipleUndo = function(actions, redo) {
	for (var i=0; i<actions.length; i++) {
		var action = actions[i];
		if (action instanceof ZmOrganizerMoveAction) {
			action._doMove(null, null, redo ? this._toFolderId : this._fromFolderId);
		}
	}
};

ZmOrganizerMoveAction.multipleRedo = function(actions) {
	ZmItemMoveAction.multipleUndo(actions, true);
};

/**
 * @class
 * Represents a collection of undoable actions that will be performed as a whole
 * 
 * @extends		ZmAction
 *
 */

ZmCompositeAction = function(toFolderId) {
	ZmAction.call(this);
	this._actions = {};
	this._toFolderId = toFolderId;
};

ZmCompositeAction.prototype = new ZmAction;
ZmCompositeAction.prototype.constructor = ZmCompositeAction;
ZmCompositeAction.prototype.type = ZmAction.ACTION_ZMCOMPOSITEACTION;

ZmCompositeAction.prototype.toString = function() {
	return "ZmCompositeAction";
};

/**
 * Add an action the the collection
 *
 * @param	{ZmAction}	action	An action to add
 */
ZmCompositeAction.prototype.addAction = function(action) {
	if (action && action!=this && action instanceof ZmAction) {
		var type = action.type;
		if (!this._actions[type])
			this._actions[type] = [];
		this._actions[type].push(action);
	}
};

ZmCompositeAction.prototype.getActions = function(type) {
	return this._actions[type] || [];
};

ZmCompositeAction.prototype.hasActions = function(type) {
	return this._actions[type] && this._actions[type].length>0;
};

ZmCompositeAction.prototype.undo = function(callback, errorCallback) {

	if (this.hasActions(ZmAction.ACTION_ZMITEMMOVEACTION)) {
		ZmItemMoveAction.multipleUndo(this.getActions(ZmAction.ACTION_ZMITEMMOVEACTION), null, this._toFolderId);
	}

	if (this.hasActions(ZmAction.ACTION_ZMORGANIZERMOVEACTION)) {
		ZmOrganizerMoveAction.multipleUndo(this.getActions(ZmAction.ACTION_ZMORGANIZERMOVEACTION));
	}

	if (this.hasActions(ZmAction.ACTION_ZMCOMPOSITEACTION) || this.hasActions(ZmAction.ACTION_ZMITEMACTION)) {
		var actions = this.getActions(ZmAction.ACTION_ZMCOMPOSITEACTION).concat(this.getActions(ZmAction.ACTION_ZMITEMACTION));
		for (var i=0; i<actions.length; i++) {
			actions[i].undo();
		}
	}
};

ZmCompositeAction.prototype.redo = function(callback, errorCallback) {

	if (this.hasActions(ZmAction.ACTION_ZMITEMMOVEACTION)) {
		ZmItemMoveAction.multipleRedo(this.getActions(ZmAction.ACTION_ZMITEMMOVEACTION));
	}

	if (this.hasActions(ZmAction.ACTION_ZMORGANIZERMOVEACTION)) {
		ZmOrganizerMoveAction.multipleRedo(this.getActions(ZmAction.ACTION_ZMORGANIZERMOVEACTION));
	}

	if (this.hasActions(ZmAction.ACTION_ZMCOMPOSITEACTION) || this.hasActions(ZmAction.ACTION_ZMITEMACTION) || this.hasActions(ZmAction.ACTION_ZMORGANIZERACTION)) {
		var actions = this.getActions(ZmAction.ACTION_ZMCOMPOSITEACTION).concat(this.getActions(ZmAction.ACTION_ZMITEMACTION)).concat(this.getActions(ZmAction.ACTION_ZMORGANIZERACTION));
		for (var i=0; i<actions.length; i++) {
			actions[i].redo();
		}
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmOrganizer")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines an organizer.
 */

/**
* Creates an empty organizer.
* @class
* This class represents an "organizer", which is something used to classify or contain
* items. So far, that's either a tag or a folder. Tags and folders are represented as
* a tree structure, though tags are flat and have only one level below the root. Folders
* can be nested.
*
* @author Conrad Damon
*
* @param	{Hash}	params		a hash of parameters
* @param {constant}	params.type		the organizer type
* @param {int}		params.id			the numeric ID
* @param {String}	params.name		the name
* @param {ZmOrganizer}	params.parent		the parent organizer
* @param {ZmTree}	params.tree		the tree model that contains this organizer
* @param {constant}	params.color		the color for this organizer
* @param {String}	params.rgb		the color for this organizer, as HTML RGB value
* @param {Boolean}	params.link		<code>true</code> if this organizer is shared
* @param {Boolean}	params.broken		<code>true</code> if this link is broken
* @param {int}	params.numUnread	the number of unread items for this organizer
* @param {int}	params.numTotal	the number of items for this organizer
* @param {Boolean}	params.noTooltip	do not show tooltip 
* @param {int}	params.sizeTotal	the total size of organizer's items
* @param {String}	params.url		the URL for this organizer's feed
* @param {String}	params.owner		the owner for this organizer
* @param {String}	params.oname		the owner's name for this organizer
* @param {String}	params.zid		the Zimbra ID of owner, if remote folder
* @param {String}	params.rid		the remote ID of organizer, if remote folder
* @param {String}	params.restUrl	the REST URL of this organizer.
* @param {String}	params.newOp		the name of operation run by button in overview header
* @param {ZmZimbraAccount}	params.account	the account this organizer belongs to
*/
ZmOrganizer = function(params) {

	if (arguments.length == 0) { return; }

	this.type = params.type;
	var id = this.id = params.id;
	this.nId = ZmOrganizer.normalizeId(id);
	this.name = ZmFolder.MSG_KEY[this.nId] ? ZmMsg[ZmFolder.MSG_KEY[this.nId]] : params.name;
	this.mail = params.mail;
	this._systemName = this.nId < 256 && params.name;
	this.parent = params.parent;
	this.tree = params.tree;
	this.numUnread = params.numUnread || 0;
	this.numTotal = params.numTotal || 0;
	this.noTooltip = params.noTooltip;
	this.sizeTotal = params.sizeTotal || 0;
	this.url = params.url;
	this.owner = params.owner;
	this.oname = params.oname;
	this.link = params.link || (Boolean(params.zid)) || (this.parent && this.parent.link);
	this.isMountpoint = params.link;
	this.zid = params.zid;
	this.rid = params.rid;
	this.restUrl = params.restUrl;
	this.account = params.account;
    this.perm = params.perm;
	this.noSuchFolder = params.broken; // Is this a link to some folder that ain't there.
	this._isAdmin = this._isReadOnly = this._hasPrivateAccess = null;
    this.retentionPolicy = params.retentionPolicy;
	this.webOfflineSyncDays = params.webOfflineSyncDays;

	this.color =
        params.color ||
        (this.parent && this.parent.color) ||
        ZmOrganizer.ORG_COLOR[id] ||
        ZmOrganizer.ORG_COLOR[this.nId] ||
        ZmOrganizer.DEFAULT_COLOR[this.type] ||
        ZmOrganizer.C_NONE
    ;
	this.isColorCustom = params.rgb != null; //set so we know if the user chose a custom color (to distiguish from basic color or none
	this.rgb =
        params.rgb ||
        ZmOrganizer.COLOR_VALUES[this.color] ||
        ZmOrganizer.COLOR_VALUES[ZmOrganizer.ORG_DEFAULT_COLOR]
    ;

	if (appCtxt.isOffline && !this.account && this.id == this.nId) {
		this.account = appCtxt.accountList.mainAccount;
	}

	if (id && params.tree) {
		appCtxt.cacheSet(id, this);
		if (this.link) {
			// also store under ID that items use for parent folder ("l" attribute in node)
			appCtxt.cacheSet([this.zid, this.rid].join(":"), this);
		}
	}

	this.children = new AjxVector();
};

ZmOrganizer.prototype.isZmOrganizer = true;
ZmOrganizer.prototype.toString = function() { return "ZmOrganizer"; };

// global organizer types
ZmOrganizer.TAG					= ZmEvent.S_TAG;
ZmOrganizer.SEARCH				= ZmEvent.S_SEARCH;
ZmOrganizer.SHARE               = ZmEvent.S_SHARE;
ZmOrganizer.MOUNTPOINT			= ZmEvent.S_MOUNTPOINT;
ZmOrganizer.ZIMLET				= ZmEvent.S_ZIMLET;
ZmOrganizer.HAB					= ZmEvent.S_HAB;
ZmOrganizer.SHARING				= ZmEvent.S_SHARING;

// folder IDs defined in com.zimbra.cs.mailbox.Mailbox
// Note: since these are defined as Numbers, and IDs come into our system as Strings,
// we need to use == for comparisons (instead of ===, which will fail)
ZmOrganizer.ID_ROOT				= 1;
ZmOrganizer.ID_INBOX			= 2;
ZmOrganizer.ID_TRASH			= 3;
ZmOrganizer.ID_SPAM				= 4;
ZmOrganizer.ID_ADDRBOOK			= 7;
ZmOrganizer.ID_CALENDAR			= 10;
ZmOrganizer.ID_AUTO_ADDED 		= 13;
ZmOrganizer.ID_CHATS			= 14;
ZmOrganizer.ID_TASKS			= 15;
ZmOrganizer.ID_BRIEFCASE		= 16;
ZmOrganizer.ID_FILE_SHARED_WITH_ME	= 20;
ZmOrganizer.ID_ALL_MAILBOXES	= 249; 
ZmOrganizer.ID_NOTIFICATION_MP	= 250;
ZmOrganizer.ID_SYNC_FAILURES	= 252;		// offline only
ZmOrganizer.ID_OUTBOX    		= 254;		// offline only
ZmOrganizer.ID_ZIMLET			= -1000;	// zimlets need a range.  start from -1000 incrementing up.
ZmOrganizer.ID_ATTACHMENTS		= -17;		// Attachments View
ZmOrganizer.ID_DLS				= -18;

// fields that can be part of a displayed organizer
ZmOrganizer.F_NAME				= "name";
ZmOrganizer.F_UNREAD			= "unread";
ZmOrganizer.F_TOTAL				= "total";
ZmOrganizer.F_SIZE				= "size";
ZmOrganizer.F_COLOR				= "color";
ZmOrganizer.F_RGB				= "rgb";
ZmOrganizer.F_QUERY				= "query";
ZmOrganizer.F_SHARES			= "shares";
ZmOrganizer.F_FLAGS				= "flags";
ZmOrganizer.F_REST_URL			= "rest";
ZmOrganizer.F_PERMS				= "perms";
ZmOrganizer.F_RNAME				= "rname";	// remote name

// server representation of org flags
ZmOrganizer.FLAG_CHECKED			= "#";
ZmOrganizer.FLAG_DISALLOW_SUBFOLDER	= "o";
ZmOrganizer.FLAG_EXCLUDE_FREE_BUSY	= "b";
ZmOrganizer.FLAG_IMAP_SUBSCRIBED	= "*";
ZmOrganizer.FLAG_OFFLINE_GLOBAL		= "g";
ZmOrganizer.FLAG_OFFLINE_SYNCABLE	= "y";
ZmOrganizer.FLAG_OFFLINE_SYNCING	= "~";
ZmOrganizer.ALL_FLAGS = [
	ZmOrganizer.FLAG_CHECKED,
	ZmOrganizer.FLAG_IMAP_SUBSCRIBED,
	ZmOrganizer.FLAG_EXCLUDE_FREE_BUSY,
	ZmOrganizer.FLAG_DISALLOW_SUBFOLDER,
	ZmOrganizer.FLAG_OFFLINE_GLOBAL,
	ZmOrganizer.FLAG_OFFLINE_SYNCABLE,
	ZmOrganizer.FLAG_OFFLINE_SYNCING
];

// org property for each flag
ZmOrganizer.FLAG_PROP = {};
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_CHECKED]				= "isChecked";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_IMAP_SUBSCRIBED]		= "imapSubscribed";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_EXCLUDE_FREE_BUSY]	= "excludeFreeBusy";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_DISALLOW_SUBFOLDER]	= "disallowSubFolder";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_OFFLINE_GLOBAL]		= "isOfflineGlobalSearch";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_OFFLINE_SYNCABLE]	= "isOfflineSyncable";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_OFFLINE_SYNCING]		= "isOfflineSyncing";

// Following chars invalid in organizer names: " : / [anything less than " "]
ZmOrganizer.VALID_NAME_CHARS = "[^\\x00-\\x1F\\x7F:\\/\\\"]";
ZmOrganizer.VALID_PATH_CHARS = "[^\\x00-\\x1F\\x7F:\\\"]"; // forward slash is OK in path
ZmOrganizer.VALID_NAME_RE = new RegExp('^' + ZmOrganizer.VALID_NAME_CHARS + '+$');

ZmOrganizer.MAX_NAME_LENGTH			= 128;	// max allowed by server
ZmOrganizer.MAX_DISPLAY_NAME_LENGTH	= 30;	// max we will show

// color constants (server stores a number)
ZmOrganizer.C_NONE				= 0;
ZmOrganizer.C_BLUE				= 1;
ZmOrganizer.C_CYAN				= 2;
ZmOrganizer.C_GREEN				= 3;
ZmOrganizer.C_PURPLE			= 4;
ZmOrganizer.C_RED				= 5;
ZmOrganizer.C_YELLOW			= 6;
ZmOrganizer.C_PINK				= 7;
ZmOrganizer.C_GRAY				= 8;
ZmOrganizer.C_ORANGE			= 9;
ZmOrganizer.MAX_COLOR			= ZmOrganizer.C_ORANGE;
ZmOrganizer.ORG_DEFAULT_COLOR 	= ZmOrganizer.C_GRAY;

ZmOrganizer.COLOR_VALUES = [
	null,
	ZmMsg.colorBlue,
	ZmMsg.colorCyan,
	ZmMsg.colorGreen,
	ZmMsg.colorPurple,
	ZmMsg.colorRed,
	ZmMsg.colorYellow,
	ZmMsg.colorPink,
	ZmMsg.colorGray,
	ZmMsg.colorOrange
];

// color names
ZmOrganizer.COLOR_TEXT = {};
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_NONE]		= ZmMsg.none;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_ORANGE]	= ZmMsg.orange;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_BLUE]		= ZmMsg.blue;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_CYAN]		= ZmMsg.cyan;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_GREEN]		= ZmMsg.green;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_PURPLE]	= ZmMsg.purple;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_RED]		= ZmMsg.red;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_YELLOW]	= ZmMsg.yellow;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_PINK]		= ZmMsg.pink;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_GRAY]		= ZmMsg.gray;

// list of colors and text for populating a color select menu
ZmOrganizer.COLORS = [];
ZmOrganizer.COLOR_CHOICES = [];
(function() {
	for (var i = 0; i <= ZmOrganizer.MAX_COLOR; i++) {
		var color = ZmOrganizer.COLOR_TEXT[i];
		ZmOrganizer.COLORS.push(color);
		ZmOrganizer.COLOR_CHOICES.push( { value:i, label:color } );
	}
})();


ZmOrganizer.MSG_KEY 		= {};		// keys for org names
ZmOrganizer.ROOT_MSG_KEY	= {};		// key for name of root (used as tree header)
ZmOrganizer.ITEM_ORGANIZER 	= {};		// primary organizer for item types
ZmOrganizer.DEFAULT_FOLDER 	= {};		// default folder for org type
ZmOrganizer.SOAP_CMD 		= {};		// SOAP command for modifying an org
ZmOrganizer.FIRST_USER_ID 	= {};		// lowest valid user ID for an org type
ZmOrganizer.PRECONDITION 	= {};		// setting that this org type depends on
ZmOrganizer.HAS_COLOR 		= {};		// whether an org uses colors
ZmOrganizer.DEFAULT_COLOR 	= {};		// default color for each org type
ZmOrganizer.ORG_COLOR 		= {};		// color overrides by ID
ZmOrganizer.APP 			= {};		// App responsible for organizer
ZmOrganizer.ORG_CLASS 		= {};		// constructor for organizer
ZmOrganizer.ORG_PACKAGE 	= {};		// package required to construct organizer
ZmOrganizer.CREATE_FUNC 	= {};		// function that creates this organizer
ZmOrganizer.LABEL 			= {};		// msg key for text for tree view header item
ZmOrganizer.ITEMS_KEY 		= {};		// msg key for text describing contents
ZmOrganizer.TREE_TYPE 		= {};		// type of server data tree that contains this type of organizer
ZmOrganizer.VIEWS 			= {};		// views by org type
ZmOrganizer.VIEW_HASH		= {};		// view hash by org type
ZmOrganizer.TYPE 			= {};		// types by view (reverse map of above)
ZmOrganizer.FOLDER_KEY 		= {};		// keys for label "[org] folder"
ZmOrganizer.MOUNT_KEY 		= {};		// keys for label "mount [org]"
ZmOrganizer.DEFERRABLE 		= {};		// creation can be deferred to app launch
ZmOrganizer.PATH_IN_NAME	= {};		// if true, provide full path when asked for name
ZmOrganizer.OPEN_SETTING	= {};		// setting that controls whether the tree view is open
ZmOrganizer.NEW_OP			= {};		// name of operation for new button in tree header (optional)
ZmOrganizer.DISPLAY_ORDER	= {};		// sort number to determine order of tree view (optional)
ZmOrganizer.HIDE_EMPTY		= {};		// if true, hide tree header if tree is empty
ZmOrganizer.SHAREABLE 		= {};		// allow share or not

ZmOrganizer.APP2ORGANIZER	= {};		// organizer types, keyed by app name
ZmOrganizer.APP2ORGANIZER_R = {};		// app names, keyed by app organizer type

// allowed permission bits
ZmOrganizer.PERM_READ		= "r";
ZmOrganizer.PERM_WRITE		= "w";
ZmOrganizer.PERM_INSERT		= "i";
ZmOrganizer.PERM_DELETE		= "d";
ZmOrganizer.PERM_ADMIN		= "a";
ZmOrganizer.PERM_WORKFLOW	= "x";
ZmOrganizer.PERM_PRIVATE	= "p";

// Retention Policy Elements - Keep or Purge
ZmOrganizer.RETENTION_KEEP  = "keep";
ZmOrganizer.RETENTION_PURGE = "purge";

// Abstract methods

/**
 * Stores information about the given organizer type.
 * 
 * @param {constant}	org				the organizer type
 * @param {Hash}	params			a hash of parameters
 * @param	{constant}	app				the app that handles this org type
 * @param	{String}	nameKey			the msg key for org name
 * @param	{constant}	precondition		the setting that this org type depends on
 * @param	{int}	defaultFolder		the folder ID of default folder for this org
 * @param	{String}	soapCmd			the SOAP command for acting on this org
 * @param	{int}	firstUserId		the minimum ID for a user instance of this org
 * @param	{String}	orgClass			the name of constructor for this org
 * @param	{String}	orgPackage		the name of smallest package with org class
 * @param	{String}	treeController	the name of associated tree controller
 * @param	{String}	labelKey			the msg key for label in overview
 * @param	{String}	itemsKey			the msg key for text describing contents
 * @param	{Boolean}	hasColor			<code>true</code> if org has color associated with it
 * @param	{constant}	defaultColor		the default color for org in overview
 * @param	{Array}	orgColor			the color override by ID (in pairs)
 * @param	{constant}	treeType			the type of data tree (from server) that contains this org
 * @param	{String}	views				the associated folder views (JSON)
 * @param	{String}	folderKey			the msg key for folder props dialog
 * @param	{String}	mountKey			the msg key for folder mount dialog
 * @param	{String}	createFunc		the name of function for creating this org
 * @param	{String}	compareFunc		the name of function for comparing instances of this org
 * @param	{Boolean}	deferrable		if <code>true</code>, creation can be deferred to app launch
 * @param	{Boolean}	pathInName		if <code>true</code>, provide full path when asked for name
 * @param	{constant}	openSetting		the setting that controls whether the tree view is open
 * @param	{int}	displayOrder		the number that is used when sorting the display of trees. (Lower number means higher display.)
 * @param	{Boolean}	hideEmpty			if <code>true</code>, hide tree header if tree is empty
 */
ZmOrganizer.registerOrg =
function(org, params) {
	if (params.nameKey)			{ ZmOrganizer.MSG_KEY[org]				= params.nameKey; }
	if (params.app)				{
		ZmOrganizer.APP[org] = params.app;
		if (!ZmOrganizer.APP2ORGANIZER[params.app]) {
			ZmOrganizer.APP2ORGANIZER[params.app] = [];
		}
		ZmOrganizer.APP2ORGANIZER[params.app].push(org);
        ZmOrganizer.APP2ORGANIZER_R[org] = params.app;
	}
	if (params.defaultFolder)	{ ZmOrganizer.DEFAULT_FOLDER[org]		= params.defaultFolder; }
	if (params.precondition)	{ ZmOrganizer.PRECONDITION[org]			= params.precondition; }
	if (params.soapCmd)			{ ZmOrganizer.SOAP_CMD[org]				= params.soapCmd; }
	if (params.firstUserId)		{ ZmOrganizer.FIRST_USER_ID[org]		= params.firstUserId; }
	if (params.orgClass)		{ ZmOrganizer.ORG_CLASS[org]			= params.orgClass; }
	if (params.orgPackage)		{ ZmOrganizer.ORG_PACKAGE[org]			= params.orgPackage; }
	if (params.labelKey)		{ ZmOrganizer.LABEL[org]				= params.labelKey; }
	if (params.itemsKey)		{ ZmOrganizer.ITEMS_KEY[org]			= params.itemsKey; }
	if (params.hasColor)		{ ZmOrganizer.HAS_COLOR[org]			= params.hasColor; }
	if (params.views)			{ ZmOrganizer.VIEWS[org]				= params.views; }
	if (params.folderKey)		{ ZmOrganizer.FOLDER_KEY[org]			= params.folderKey; }
	if (params.mountKey)		{ ZmOrganizer.MOUNT_KEY[org]			= params.mountKey; }
	if (params.deferrable)		{ ZmOrganizer.DEFERRABLE[org]			= params.deferrable; }
	if (params.pathInName)		{ ZmOrganizer.PATH_IN_NAME[org]			= params.pathInName; }
	if (params.openSetting)		{ ZmOrganizer.OPEN_SETTING[org]			= params.openSetting; }
	if (params.newOp)			{ ZmOrganizer.NEW_OP[org]				= params.newOp; }
	if (params.displayOrder)	{ ZmOrganizer.DISPLAY_ORDER[org]		= params.displayOrder; }
	if (params.hideEmpty)		{ ZmOrganizer.HIDE_EMPTY[org]			= params.hideEmpty; }
	ZmOrganizer.SHAREABLE[org]	= !params.disableShare; 

	if (!appCtxt.isChildWindow || params.childWindow ) {
		if (params.compareFunc)		{ ZmTreeView.COMPARE_FUNC[org]			= params.compareFunc; }
		if (params.treeController)	{ ZmOverviewController.CONTROLLER[org]	= params.treeController; }
	}

	ZmOrganizer.TREE_TYPE[org] = params.treeType || org; // default to own type
	ZmOrganizer.CREATE_FUNC[org] = params.createFunc || "ZmOrganizer.create";

	if (params.views) {
		ZmOrganizer.VIEW_HASH[org] = AjxUtil.arrayAsHash(ZmOrganizer.VIEWS[org]);
	}

	if (params.hasColor) {
		ZmOrganizer.DEFAULT_COLOR[org] = (params.defaultColor != null)
			? params.defaultColor
			: ZmOrganizer.ORG_DEFAULT_COLOR;
	}

	if (params.orgColor) {
		for (var id in params.orgColor) {
			ZmOrganizer.ORG_COLOR[id] = params.orgColor[id];
		}
	}

	if (params.dropTargets) {
		if (!ZmApp.DROP_TARGETS[params.app]) {
			ZmApp.DROP_TARGETS[params.app] = {};
		}
		ZmApp.DROP_TARGETS[params.app][org] = params.dropTargets;
	}
};

ZmOrganizer.sortCompare = function(organizerA, organizerB) {};

/**
 * nulls value that is the default color for the type.
 * @param value
 */
ZmOrganizer.getColorValue =
function(value, type) {
	// no need to save color if missing or default
	if (value == ZmOrganizer.DEFAULT_COLOR[type]) {
		return null;
	}

	return value;
};

/**
 * Creates an organizer via <code>&lt;CreateFolderRequest&gt;</code>. Attribute pairs can
 * be passed in and will become attributes of the folder node in the request.
 * 
 * @param {Hash}	params	a hash of parameters
 */
ZmOrganizer.create =
function(params) {
	var jsonObj = {CreateFolderRequest:{_jsns:"urn:zimbraMail"}};
	var folder = jsonObj.CreateFolderRequest.folder = {};
	var errorCallback = params.errorCallback || new AjxCallback(null, ZmOrganizer._handleErrorCreate, params);
	var type = params.type;

	// set attributes
	params.view = params.view || ZmOrganizer.VIEWS[type] ? ZmOrganizer.VIEWS[type][0] : null;
	for (var i in params) {
		if (i == "type" || i == "errorCallback" || i == "account") { continue; }

		var value = params[i];
		if (value) {
			folder[i] = value;
		}
	}
	//adding support to asyncMode == false didn't eventually help me, but why not keep it?
	var asyncMode = params.asyncMode === undefined ? true : params.asyncMode; //default is true

	return appCtxt.getAppController().sendRequest({
		jsonObj: jsonObj,
		asyncMode: asyncMode,
		accountName: (params.account && params.account.name),
		callback: params.callback,
		callbackAfterNotifications: params.callbackAfterNotifications, 
		errorCallback: errorCallback
	});
};

/**
 * @private
 */
ZmOrganizer._handleErrorCreate =
function(params, ex) {
	if (!params.url && !params.name) { return false; }
	
	var msg;
	if (params.name && (ex.code == ZmCsfeException.MAIL_ALREADY_EXISTS)) {
		var type = appCtxt.getFolderTree(appCtxt.getActiveAccount()).getFolderTypeByName(params.name);
        msg = AjxMessageFormat.format(ZmMsg.errorAlreadyExists, [params.name, type.toLowerCase()]);
	} else if (params.url) {
		var errorMsg = (ex.code == ZmCsfeException.SVC_RESOURCE_UNREACHABLE) ? ZmMsg.feedUnreachable : ZmMsg.feedInvalid;
		msg = AjxMessageFormat.format(errorMsg, params.url);
	}

	if (msg) {
		ZmOrganizer._showErrorMsg(msg);
		return true;
	}

	return false;
};

/**
 * @private
 */
ZmOrganizer._showErrorMsg =
function(msg) {
	var msgDialog = appCtxt.getMsgDialog();
	msgDialog.reset();
	msgDialog.setMessage(AjxStringUtil.htmlEncode(msg), DwtMessageDialog.CRITICAL_STYLE);
	msgDialog.popup();
};

/**
 * Gets the folder.
 * 
 * @param	{String}	id		the folder id
 * @param	{AjxCallback}	callback	the callback
 * @param	{ZmBatchCommand}	batchCmd	the batch command or <code>null</code> for none
 */
ZmOrganizer.getFolder =
function(id, callback, batchCmd) {
	var jsonObj = {GetFolderRequest:{_jsns:"urn:zimbraMail"}};
	var request = jsonObj.GetFolderRequest;
	request.folder = {l:id};
	var respCallback = new AjxCallback(null, ZmOrganizer._handleResponseGetFolder, [callback]);
	if (batchCmd) {
		batchCmd.addRequestParams(jsonObj, respCallback);
	} else {
		appCtxt.getRequestMgr().sendRequest({jsonObj:jsonObj, asyncMode:true, callback:respCallback});
	}
};

/**
 * @private
 */
ZmOrganizer._handleResponseGetFolder =
function(callback, result) {
	var resp = result.getResponse().GetFolderResponse;
	var folderObj = (resp && resp.folder && resp.folder[0]) ||
					(resp && resp.link && resp.link[0]);
	var folder;
	if (folderObj) {
		folder = appCtxt.getById(folderObj.id);
		if (folder) {
			folder.clearShares();
			folder._setSharesFromJs(folderObj);
		} else {
			var parent = appCtxt.getById(folderObj.l);
			folder = ZmFolderTree.createFromJs(parent, folderObj, appCtxt.getFolderTree(), "folder");
		}
	}
	if (callback) {
		callback.run(folder);
	}
};

/**
 * Gets the folder.
 * 
 * @param	{AjxCallback}	callback	the callback
 * @param	{ZmBatchCommand}	batchCmd	the batch command or <code>null</code> for none
 */
ZmOrganizer.prototype.getFolder =
function(callback, batchCmd) {
	ZmOrganizer.getFolder(this.id, callback, batchCmd);
};


// Static methods

/**
 * Gets the view name by organizer type.
 * 
 * @param	{String}	organizerType		the organizer type
 * @return	{String}	the view
 */
ZmOrganizer.getViewName =
function(organizerType) {
	if (organizerType === ZmItem.BRIEFCASE_ITEM) {
		return ZmItem.BRIEFCASE_ITEM;
	}
	return ZmOrganizer.VIEWS[organizerType][0];
};

/**
 * Checks an organizer (folder or tag) offlineSyncInterval for validity.
 *
 * @param {String}	value		offlineSyncInterval
 * @return	{String}	<code>null</code> if the offlineSyncInterval is valid or an error message if the name is invalid
 */
ZmOrganizer.checkWebOfflineSyncDays =
function(value) {
    if (isNaN(value)) {	return ZmMsg.invalidFolderSyncInterval; }
    var interval = parseInt(value);
	if (interval < 0 ||  interval > 30) {
		return ZmMsg.invalidFolderSyncInterval;
	}
	return null;
};

/**
 * Checks an organizer (folder or tag) name for validity.
 *
 * @param {String}	name		an organizer name
 * @return	{String}	<code>null</code> if the name is valid or an error message if the name is invalid
 */
ZmOrganizer.checkName =
function(name) {
	if (name.length == 0) {	return ZmMsg.nameEmpty; }

	if (name.length > ZmOrganizer.MAX_NAME_LENGTH) {
		return AjxMessageFormat.format(ZmMsg.nameTooLong, ZmOrganizer.MAX_NAME_LENGTH);
	}

	if (!ZmOrganizer.VALID_NAME_RE.test(name)) {
		return AjxMessageFormat.format(ZmMsg.errorInvalidName, AjxStringUtil.htmlEncode(name));
	}

	return null;
};

/**
 * Checks a URL (a folder or calendar feed, for example) for validity.
 *
 * @param {String}	url	a URL
 * @return	{String}	<code>null</code> if valid or an error message
 */
ZmOrganizer.checkUrl =
function(url) {
	// TODO: be friendly and prepend "http://" when it's missing
	if (!url.match(/^[a-zA-Z]+:\/\/.*$/i)) {
		return ZmMsg.errorUrlMissing;
	}

	return null;
};

/**
 * @private
 */
ZmOrganizer.checkSortArgs =
function(orgA, orgB) {
	if (!orgA && !orgB) return 0;
	if (orgA && !orgB) return 1;
	if (!orgA && orgB) return -1;
	return null;
};

/**
 * @private
 */
ZmOrganizer.checkColor =
function(color) {
	return ((color != null) && (color >= 0 && color <= ZmOrganizer.MAX_COLOR))
		? color : ZmOrganizer.ORG_DEFAULT_COLOR;
};

/**
 * Gets the system ID for the given system ID and account. Unless this
 * is a child account, the system ID is returned unchanged. For child
 * accounts, the ID consists of the account ID and the local ID.
 * 
 * @param {int}	id		the ID of a system organizer
 * @param {ZmZimbraAccount}	account	the account
 * @param {Boolean}		force		<code>true</code> to generate the fully qualified ID even if this is the main account
 * @return	{String}	the ID
 */
ZmOrganizer.getSystemId =
function(id, account, force) {
	account = account || appCtxt.getActiveAccount();
	if ((account && !account.isMain) || force) {
		return ((typeof(id) == "string") && (id.indexOf(":") != -1) || !id)
			? id : ([account.id, id].join(":"));
	}
	return id;
};

/**
 * Normalizes the id by stripping the account ID portion from a system ID for a child account, which
 * can then be used to check against known system IDs. Any non-system ID is
 * returned unchanged (if type is provided).
 *
 * @param {String}	id	ID of an organizer
 * @param {constant}	type	the type of organizer
 * @return	{String}	the resulting id
 */
ZmOrganizer.normalizeId =
function(id, type) {
	if (typeof(id) != "string") { return id; }
	var idx = id.indexOf(":");
	var localId = (idx == -1) ? id : id.substr(idx + 1);
	return (type && (localId >= ZmOrganizer.FIRST_USER_ID[type])) ? id : localId;
};

/**
 * Parses an id into an object with fields for account and normalized id
 *
 * @param {String}	id		the ID of an organizer
 * @param {Object}	result	an optional object in which the result is stored
 * @return	{Object}	the resulting ID
 */
ZmOrganizer.parseId =
function(id, result) {
	var ac = window.parentAppCtxt || window.appCtxt;

	result = result || {};
	if (id == null) { return result; }
	var idx = (typeof id == "string") ? id.indexOf(":") : -1;
	if (idx == -1) {
		result.account = ac.accountList.mainAccount;
		result.id = id;
	} else {
		result.acctId = id.substring(0, idx);
		result.account = ac.accountList.getAccount(result.acctId);
		result.id = id.substr(idx + 1);
	}
	return result;
};

// Public methods

/**
* Gets the name of this organizer.
*
* @param {Boolean}	showUnread		<code>true</code> to display the number of unread items (in parens)
* @param {int}	maxLength		the length (in chars) to truncate the name to
* @param {Boolean}	noMarkup		if <code>true</code>, don't return any HTML
* @param {Boolean}	useSystemName	if <code>true</code>, don't use translated version of name
* @return	{String}	the name
*/
ZmOrganizer.prototype.getName = 
function(showUnread, maxLength, noMarkup, useSystemName, useOwnerName, defaultRootType) {
	if (this.nId == ZmFolder.ID_ROOT) {
		var type = defaultRootType || this.type;
		return (ZmOrganizer.LABEL[type])
			? ZmMsg[ZmOrganizer.LABEL[type]] : "";
	}
	var name = (useSystemName && this._systemName) || (useOwnerName && this.oname) || this.name || "";
	if (ZmOrganizer.PATH_IN_NAME[this.type] && this.path) {
		name = [this.path, name].join("/");
	}
	name = (maxLength && name.length > maxLength)
		? name.substring(0, maxLength - 3) + "..." : name;
	return this._markupName(name, showUnread, noMarkup);
};

/**
* Gets the full path as a string.
*
* @param {Boolean}	includeRoot		<code>true</code> to include root name at the beginning of the path
* @param {Boolean}	showUnread		<code>true</code> to display the number of unread items (in parens)
* @param {int}	maxLength		the length (in chars) to truncate the name to
* @param {Boolean}	noMarkup		if <code>true</code>, do not return any HTML
* @param {Boolean}	useSystemName	if <code>true</code>, use untranslated version of system folder names
* @return	{String}	the path
*/
ZmOrganizer.prototype.getPath = 
function(includeRoot, showUnread, maxLength, noMarkup, useSystemName, useOwnerName) {
	var parent = this.parent;
	var path = this.getName(showUnread, maxLength, noMarkup, useSystemName, useOwnerName);
	while (parent && ((parent.nId != ZmOrganizer.ID_ROOT) || includeRoot)) {
		path = parent.getName(showUnread, maxLength, noMarkup, useSystemName, useOwnerName) + ZmFolder.SEP + path;
		parent = parent.parent;
	}

	return path;
};

/**
 * Gets the tooltip. The tooltip shows number of unread items, total messages and the total size.
 *
 * @param {Boolean}	force		if <code>true</code>, don't use cached tooltip
 * @return	{String}	the tooltip
 */
ZmOrganizer.prototype.getToolTip =
function(force) {
	if (this.noTooltip) {
		return null;
	}
    if (!this._tooltip || force) {
		var itemText = this._getItemsText();
		var unreadLabel = this._getUnreadLabel();
		var subs = {name:this.name, itemText:itemText, numTotal:this.numTotal, sizeTotal:this.sizeTotal, numUnread:this.numUnread, unreadLabel:unreadLabel};
		this._tooltip = AjxTemplate.expand("share.App#FolderTooltip", subs);
	}
	return this._tooltip;
};

/**
 * Gets the full path, suitable for use in search expressions.
 *
 * @return	{String}	the path
 */
ZmOrganizer.prototype.getSearchPath =
function(useOwnerName) {
	return (this.nId != ZmOrganizer.ID_ROOT)
		? this.getPath(null, null, null, true, true, useOwnerName) : "/";
};

/**
 * Gets the URL.
 * 
 * @return	{String}	the URL
 * 
 * @deprecated use {@link getRestUrl}
 */
ZmOrganizer.prototype.getUrl =
function() {
	return this.getRestUrl();
};

/**
 * Gets the sync URL.
 * 
 * @return		{String}	the URL
 */
ZmOrganizer.prototype.getSyncUrl =
function() {
	return url;
};

/**
 * Gets the remote ID.
 * 
 * @return	{String}	the ID
 */
ZmOrganizer.prototype.getRemoteId =
function() {
	if (!this._remoteId) {
		this._remoteId = (this.isRemote() && this.zid && this.rid)
			? (this.zid + ":" + this.rid)
			: this.id;
	}
	return this._remoteId;
};


/**
 * Gets the REST URL.
 * 
 * @return	{String}	the URL
 */
ZmOrganizer.prototype.getRestUrl =
function(noRemote) {

	var restUrl = appCtxt.get(ZmSetting.REST_URL);
	if (restUrl && (!this.isRemote() || noRemote || this.id == ZmFolder.ID_FILE_SHARED_WITH_ME)) { //for remote - this does not work. either use this.restUrl (if set, which is for shared folder, but not for sub-folders) or call _generateRestUrl which seems to work for subfodlers of shared as well.
		var path = AjxStringUtil.urlEncode(this.getSearchPath()).replace("#","%23").replace(";", "%3B"); // User may type in a # in a folder name, but that's not ok for our urls
		// return REST URL as seen by the GetInfoResponse
		return ([restUrl, "/", path].join(""));
	}

	// return REST URL as seen by server - this is the remote (isRemote() true) case - shared folder.
	if (this.restUrl) {
		return this.restUrl;
	}

	// if server doesn't tell us what URL to use, do our best to generate
	var url = this._generateRestUrl();
	DBG.println(AjxDebug.DBG3, "NO REST URL FROM SERVER. GENERATED URL: " + url);

	return url;
};

/**
 * Gets the OWNER'S REST URL,used to fetch resturl of shared folders.
 *
 * @return	{String}	the URL
 */
ZmOrganizer.prototype.getOwnerRestUrl =
function(){
	var node = this;
	var restUrl;
	var path = "";

	while (node && !node.restUrl) {
		path = node.name + "/" + path;
		node = node.parent;
	}
	restUrl = node.restUrl;
	path = node.oname + "/" + path;

	path = AjxStringUtil.urlEncode(path).replace("#", "%23").replace(new RegExp("/$"), "");

	// return REST URL as seen by the GetInfoResponse
	return ([restUrl, "/", path].join(""));
};

ZmOrganizer.prototype._generateRestUrl =
function() {
	var loc = document.location;
	var uname = appCtxt.get(ZmSetting.USERNAME);
	var host = loc.host;
	var m = uname.match(/^(.*)@(.*)$/);

	host = (m && m[2]) || host;

	// REVISIT: What about port? For now assume other host uses same port
	if (loc.port && loc.port != 80) {
		host = host + ":" + loc.port;
	}

	var path = AjxStringUtil.urlEncode(this.getSearchPath()).replace("#","%23"); // User may type in a # in a folder name, but that's not ok for our urls
		
	return [
		loc.protocol, "//", host, "/service/user/", uname, "/",	path
	].join("");
};

/**
 * Gets the account.
 * 
 * @return	{ZmZimbraAccount}	the account
 */
ZmOrganizer.prototype.getAccount =
function() {
	if (appCtxt.multiAccounts) {
		if (!this.account) {
			this.account = ZmOrganizer.parseId(this.id).account;
		}
		// bug 46364:
		// still no account?! Must be remote organizer, keep checking parent
		if (!this.account) {
			var parent = this.parent;
			while (parent && !this.account) {
				this.account = parent.getAccount();
				parent = parent.parent;
			}
		}
		return this.account;
	}

	return (this.account || appCtxt.accountList.mainAccount);
};

/**
 * Gets the shares.
 * 
 * @return	{Array}	an array of shares
 */
ZmOrganizer.prototype.getShares =
function() {
	return this.shares;
};

/**
 * Adds the share.
 * 
 * @param	{Object}	share		the share to add
 */
ZmOrganizer.prototype.addShare =
function(share) {
	this.shares = this.shares || [];
	this.shares.push(share);

	var curAcct = appCtxt.getActiveAccount();
	var curZid = curAcct && curAcct.id;
	var shareId = share.grantee && share.grantee.id;
	if (shareId && (shareId == curZid)) {
		this._mainShare = share;
	}
};

/**
 * Clears all shares.
 * 
 */
ZmOrganizer.prototype.clearShares =
function() {
	if (this.shares && this.shares.length) {
		for (var i = 0; i < this.shares.length; i++) {
			this.shares[i] = null;
		}
	}
	this.shares = null;
	this._mainShare = null;
};

/**
 * Gets the share granted to the current user.
 * 
 * @return	{String}	the main share
 */
ZmOrganizer.prototype.getMainShare =
function() {
	if (!this._mainShare) {
		var curAcct = appCtxt.getActiveAccount();
		var curZid = curAcct && curAcct.id;
		if (curZid && this.shares && this.shares.length) {
			for (var i = 0; i < this.shares.length; i++) {
				var share = this.shares[i];
				var id = share && share.grantee && share.grantee.id;
				if (id && id == curZid) {
					this._mainShare = share;
					break;
				}
			}
		}
	}
	return this._mainShare;
};

/**
 * Checks if the organizer supports sharing.
 * 
 * @return	{Boolean}	<code>true</code> if the organizer supports sharing
 */
ZmOrganizer.prototype.supportsSharing =
function() {
	// overload per organizer type
	return true;
};

/**
 * Checks if the organizer supports pulbic access.
 * 
 * @return	{Boolean}	<code>true</code> if the organizer supports public access
 */
ZmOrganizer.prototype.supportsPublicAccess =
function() {
	// overload per organizer type
	return true;
};

/**
 * Checks if the organizer supports private permission.
 * 
 * @return	{Boolean}	<code>true</code> if the organizer supports private permission
 */
ZmOrganizer.prototype.supportsPrivatePermission =
function() {
	// overload per organizer type
	return false;
};

/**
 * Gets the icon.
 * 
 * @return	{String}	the icon
 */
ZmOrganizer.prototype.getIcon = function() {};

/**
 * Gets the color of the organizer
 *
 * @return	{String}	the color
 */
ZmOrganizer.prototype.getColor =
function() {
    return this.rgb || ZmOrganizer.COLOR_VALUES[this.color];
}


/**
 * Gets the icon with color
 * 
 * @return	{String}	the icon
 */
ZmOrganizer.prototype.getIconWithColor =
function() {
	var icon = this.getIcon() || "";
	var color = this.getColor();
	return color ? [icon,color].join(",color=") : icon;
};

/**
 * Gets the icon alter text.
 * 
 * @return	{String}	the icon alter text
 */

ZmOrganizer.prototype.getIconAltText = function() {
	return this.name + " " + ZmMsg.icon;
};

// Actions

/**
 * Renames the organizer.
 * 
 * @param	{String}	name		the name
 * @param	{AjxCallback}	callback		the callback
 * @param	{AjxCallback}	errorCallback		the error callback
 * @param	{ZmBatchCommand}	batchCmd		the batch command
 */
ZmOrganizer.prototype.rename =
function(name, callback, errorCallback, batchCmd) {
	if (name == this.name) { return; }
	var params = {
		action: "rename",
		attrs: {name: name},
		callback: callback,
		errorCallback: errorCallback,
		batchCmd: batchCmd
	};
	this._organizerAction(params);
};

/**
 * Sets the web offline sync interval.
 *
 * @param	{String}	        interval		the web offline sync interval
 * @param	{AjxCallback}	    callback		the callback
 * @param	{AjxCallback}	    errorCallback   the error callback
 * @param   {ZmBatchCommand}    batchCmd        optional batch command
 */
ZmOrganizer.prototype.setOfflineSyncInterval =
function(interval, callback, errorCallback, batchCmd) {
	if (this.webOfflineSyncDays == interval) { return; }

	this._organizerAction({action: "webofflinesyncdays", attrs: {numDays: interval}, callback: callback,
                           errorCallback: errorCallback, batchCmd: batchCmd});
};

/**
 * Sets the color.
 * 
 * @param	{String}	        color		    the color
 * @param	{AjxCallback}	    callback		the callback
 * @param	{AjxCallback}	    errorCallback   the error callback
 * @param   {ZmBatchCommand}    batchCmd        optional batch command
 */
ZmOrganizer.prototype.setColor =
function(color, callback, errorCallback, batchCmd) {
	var color = ZmOrganizer.checkColor(color);
	if (!this.isColorChanged(color)) { return; }

	this._organizerAction({action: "color", attrs: {color: color}, callback: callback,
                           errorCallback: errorCallback, batchCmd: batchCmd});
};

/**
 * Sets the RGB color.
 * 
 * @param	{Object}	        rgb		        the rgb
 * @param	{AjxCallback}	    callback		the callback
 * @param	{AjxCallback}	    errorCallback	the error callback
 * @param   {ZmBatchCommand}    batchCmd        optional batch command
 */
ZmOrganizer.prototype.setRGB = function(rgb, callback, errorCallback, batchCmd) {
	if (!this.isColorChanged(rgb)) { return; }
	this._organizerAction({action: "color", attrs: {rgb: rgb}, callback: callback,
                           errorCallback: errorCallback, batchCmd: batchCmd});
};


ZmOrganizer.prototype.getRetentionPolicy =
function(policyElement) {
    var policy = null;
    if (this.retentionPolicy && this.retentionPolicy[0] && this.retentionPolicy[0][policyElement] &&
        this.retentionPolicy[0][policyElement][0]       && this.retentionPolicy[0][policyElement][0].policy &&
        this.retentionPolicy[0][policyElement][0].policy[0]) {
        policy = this.retentionPolicy[0][policyElement][0].policy[0];
    }
    return policy;
}

ZmOrganizer.prototype.getRetentionPolicyLifetimeMsec =
function(policy) {
    if (policy) {
        // Apply the keep (retention) period
        var lifetime = policy.lifetime;
        var amount = parseInt(lifetime);
        // Intervals taken from DateUtil.java.
        var interval = lifetime.slice(lifetime.length-1);
        var lifetimeMsec = 0;
        switch (interval) {
            case  "d": lifetimeMsec = amount * AjxDateUtil.MSEC_PER_DAY;    break;
            case  "h": lifetimeMsec = amount * AjxDateUtil.MSEC_PER_HOUR;   break;
            case  "m": lifetimeMsec = amount * AjxDateUtil.MSEC_PER_MINUTE; break;
            case  "s": lifetimeMsec = amount * 1000; break;
            case "ms": lifetimeMsec = amount;  break;
            default  : lifetimeMsec = amount * 1000; break;
        }
    }
    return lifetimeMsec;
}

/**
 * Sets the Retention Policy.
 *
 * @param	{Object}	        retentionPolicy     the new retention policy
 * @param	{AjxCallback}	    callback		    the callback
 * @param	{AjxCallback}	    errorCallback	    the error callback
 * @param   {ZmBatchCommand}    batchCmd            optional batch command
 */
ZmOrganizer.prototype.setRetentionPolicy = function(newRetentionPolicy, callback, errorCallback, batchCmd) {
    var keepPolicy  = this.getRetentionPolicy(ZmOrganizer.RETENTION_KEEP);
    var purgePolicy = this.getRetentionPolicy(ZmOrganizer.RETENTION_PURGE);
    if (!this.policiesDiffer(keepPolicy,  newRetentionPolicy.keep) &&
        !this.policiesDiffer(purgePolicy, newRetentionPolicy.purge)) {
        // No updated policy specified or no changes.
        return;
    }

	var cmd = ZmOrganizer.SOAP_CMD[this.type] + "Request";
	var request = {
		_jsns: "urn:zimbraMail",
		action : {
			op: "retentionpolicy",
			id: this.id,
			retentionPolicy: {
				keep: {},
				purge: {}
			}
		}
	};
	var jsonObj = {};
	jsonObj[cmd] = request;

	var retentionNode = request.action.retentionPolicy;

    if (newRetentionPolicy.keep) {
        this._addPolicy(retentionNode.keep, newRetentionPolicy.keep);
    }
    if (newRetentionPolicy.purge) {
        this._addPolicy(retentionNode.purge, newRetentionPolicy.purge);
    }

	if (batchCmd) {
        batchCmd.addRequestParams(jsonObj, callback, errorCallback);
 	}
	else {
		var accountName;
		if (appCtxt.multiAccounts) {
			accountName = (this.account)
				? this.account.name : appCtxt.accountList.mainAccount.name;
		}
		appCtxt.getAppController().sendRequest({
			jsonObj:       jsonObj,
			asyncMode:     true,
			accountName:   accountName,
			callback:      callback,
			errorCallback: errorCallback
		});
	}

};

ZmOrganizer.prototype.policiesDiffer =
function(policyA, policyB) {
    var differ = false;
    if ((policyA && !policyB) || (!policyA && policyB)) {
        differ = true;
    } else if (policyA) {
        // Old and new specified
        if (policyA.type != policyB.type) {
            differ = true;
        } else {
            if (policyA.type == "user") {
                differ = policyA.lifetime != policyB.lifetime;
            } else {
                // System policy
                differ = policyA.id != policyB.id;
            }
        }
    }
    return differ;
}

ZmOrganizer.prototype._addPolicy =
function(node, policy) {
	var policyNode = node.policy = {};
	for (var attr in policy) {
		if (AjxEnv.isIE) {
			policy[attr] += ""; //To string
		}

		policyNode[attr] = policy[attr];
	}
};

/**
 * Returns color number b/w 0-9 for a given color code
 *
 * @param	{String}	color	The color (usually in #43eded format
 * @return {int} Returns 0-9 for a standard color and returns -1 for custom color
 */
ZmOrganizer.getStandardColorNumber =
function(color) {
	if (String(color).match(/^#/)) {
		var len = ZmOrganizer.COLOR_VALUES.length;
		for(var i =0; i < len; i++) {
			var currentVal = ZmOrganizer.COLOR_VALUES[i];
			if(currentVal && currentVal == color) {
				return i;
			}
		}
	} else if(color <= 9 && color >= 0) {
		return color;
	}
	return -1;
};

/**
 * Returns true if the color is changed
 *
 * @param	{String/int}	color	The color (usually in #rgb format or numeric color code
 * @return {Boolean} Returns true if the color is changed
 */
ZmOrganizer.prototype.isColorChanged =
function(color) {
    var isNewColorCustom = ZmOrganizer.getStandardColorNumber(color) === -1,
        isPrevColorCustom = this.isColorCustom;
    if ((isNewColorCustom && !isPrevColorCustom) ||
        (!isNewColorCustom && isPrevColorCustom) ) {
        //Custom changed to standard or standard changed to custom
        return true;
    }
    else if (isNewColorCustom && isPrevColorCustom) {
        //If both are custom colors check the rgb codes
        return color != this.rgb;
    }
    else if (!isNewColorCustom && !isPrevColorCustom){
        //If both are standard check the numeric color codes
        return color != this.color;
    }
    //default fallback
    return false;
};

/**
 * Updates the folder. Although it is possible to use this method to change just about any folder
 * attribute, it should only be used to set multiple attributes at once since it
 * has extra overhead on the server.
 *
 * @param {Hash}	attrs		the attributes
 */
ZmOrganizer.prototype.update =
function(attrs) {
	this._organizerAction({action: "update", attrs: attrs});
};

/**
 * Assigns the organizer a new parent, moving it within its tree.
 *
 * @param {ZmOrganizer}	newParent		the new parent of this organizer
 * @param {boolean}		noUndo			if true, action is not undoable
 */
ZmOrganizer.prototype.move =
function(newParent, noUndo, batchCmd) {

	var newId = (newParent.nId > 0)
		? newParent.id
		: ZmOrganizer.getSystemId(ZmOrganizer.ID_ROOT);

	if ((newId == this.id || newId == this.parent.id) ||
		(this.type == ZmOrganizer.FOLDER && (ZmOrganizer.normalizeId(newId, this.type) == ZmFolder.ID_SPAM)) ||
		(newParent.isChildOf(this)))
	{
		return;
	}
	var params = {};
	params.batchCmd = batchCmd;
	params.actionTextKey = 'actionMoveOrganizer';
	params.orgName = this.getName(false, false, true, false, false, this.type);
	if (newId == ZmOrganizer.ID_TRASH) {
		params.actionArg = ZmMsg.trash;
		params.action = "trash";
		params.noUndo = noUndo;
	}
	else {
		if (newParent.account && newParent.account.isLocal()) {
			newId = [ZmAccount.LOCAL_ACCOUNT_ID, newId].join(":");
		}
		params.actionArg = newParent.getName(false, false, true, false, false, this.type);
		params.action = "move";
		params.attrs = {l: newId};
		params.noUndo = noUndo;
	}
	this._organizerAction(params);
};

/**
 * Deletes an organizer. If it's a folder, the server deletes any contents and/or
 * subfolders. If the organizer is "Trash" or "Spam", the server deletes and re-creates the
 * folder. In that case, we do not bother to remove it from the UI (and we ignore
 * creates on system folders).
 *
 */
ZmOrganizer.prototype._delete =
function(batchCmd) {
	DBG.println(AjxDebug.DBG1, "deleting: " + AjxStringUtil.htmlEncode(this.name) + ", ID: " + this.id);
	var isEmptyOp = ((this.type == ZmOrganizer.FOLDER || this.type == ZmOrganizer.ADDRBOOK || this.type == ZmOrganizer.BRIEFCASE) &&
					 (this.nId == ZmFolder.ID_SPAM || this.nId == ZmFolder.ID_TRASH));
	// make sure we're not deleting a system object (unless we're emptying SPAM or TRASH)
	if (this.isSystem() && !isEmptyOp) return;

	var action = isEmptyOp ? "empty" : "delete";
	this._organizerAction({action: action, batchCmd: batchCmd});
};

/**
 * Empties the organizer.
 *
 * @param	{Boolean}	doRecursive		<code>true</code> to recursively empty the organizer
 * @param	{ZmBatchCommand}	batchCmd	the batch command
 * @param	{Object}	callback
 * @param	{number}	timeout		the timeout(in seconds)
 * @param	{AjxCallback}	errorCallback		the callback to run after timeout
 * @param	{Boolean}	noBusyOverlay		if <code>true</code>, do not show busy overlay
 */
ZmOrganizer.prototype.empty =
function(doRecursive, batchCmd, callback, timeout, errorCallback, noBusyOverlay) {
	doRecursive = doRecursive || false;

	var isEmptyOp = ((this.type == ZmOrganizer.FOLDER || this.type == ZmOrganizer.ADDRBOOK) &&
					 (this.nId == ZmFolder.ID_SPAM ||
					  this.nId == ZmFolder.ID_TRASH ||
					  this.nId == ZmOrganizer.ID_SYNC_FAILURES));

	// make sure we're not emptying a system object (unless it's SPAM/TRASH/SYNCFAILURES)
	if (this.isSystem() && !isEmptyOp) { return; }

	var params = {
		action: "empty",
		batchCmd: batchCmd,
		callback: callback,
		timeout: timeout,
		errorCallback: errorCallback,
		noBusyOverlay: noBusyOverlay
	};
	params.attrs = (this.nId == ZmFolder.ID_TRASH)
		? {recursive:true}
		: {recursive:doRecursive};

	if (this.isRemote()) {
		params.id = this.getRemoteId();
	}

	this._organizerAction(params);
};

/**
 * Marks all items as "read".
 *
 * @param	{ZmBatchCommand}	batchCmd	the batch command
 */
ZmOrganizer.prototype.markAllRead =
function(batchCmd) {
	var id = this.isRemote() ? this.getRemoteId() : null;
	this._organizerAction({action: "read", id: id, attrs: {l: this.id}, batchCmd:batchCmd});
};

/**
 * Synchronizes the organizer.
 *
 */
ZmOrganizer.prototype.sync =
function() {
	this._organizerAction({action: "sync"});
};

// Notification handling

/**
 * Handles delete notification.
 *
 */
ZmOrganizer.prototype.notifyDelete =
function() {
	// select next reasonable organizer if the currently selected organizer is
	// the one being deleted or is a descendent of the one being deleted
	var tc = appCtxt.getOverviewController().getTreeController(this.type);
	var treeView = tc && tc.getTreeView(appCtxt.getCurrentApp().getOverviewId());

	// treeview returns array of organizers for checkbox style trees
	var organizers = treeView && treeView.getSelected();
	if (organizers) {
		if (!(organizers instanceof Array)) organizers = [organizers];
		for (var i = 0; i <  organizers.length; i++) {
			var organizer = organizers[i];
			if (organizer && (organizer == this || organizer.isChildOf(this))) {
				var folderId = this.parent.id;
				if (this.parent.nId == ZmOrganizer.ID_ROOT) {
					folderId = ZmOrganizer.getSystemId(this.getDefaultFolderId());
				}
				var skipNotify = false;
				treeView.setSelected(folderId, skipNotify);
			}
		}
	}

	// perform actual delete
	this.deleteLocal();
	this._notify(ZmEvent.E_DELETE);
};

/**
 * Handles create notification.
 */
ZmOrganizer.prototype.notifyCreate = function() {};

/**
* Handles modifications to fields that organizers have in general. Note that
* the notification object may contain multiple notifications.
*
* @param {Object}	obj		a "modified" notification
* @param {Hash}	details	the event details
*/
ZmOrganizer.prototype.notifyModify =
function(obj, details) {
	var doNotify = false;
	var details = details || {};
	var fields = {};
	if (obj.name != null && (this.name != obj.name || this.id != obj.id)) {
		if (obj.id == this.id) {
			details.oldName = this.name;
			this.name = obj.name;
			fields[ZmOrganizer.F_NAME] = true;
			this.parent.children.sort(eval(ZmTreeView.COMPARE_FUNC[this.type]));
		} else {
			// rename of a remote folder
			details.newName = obj.name;
			fields[ZmOrganizer.F_RNAME] = true;
		}
		doNotify = true;
	}
	if (obj.u != null && this.numUnread != obj.u) {
		this.numUnread = obj.u;
		fields[ZmOrganizer.F_UNREAD] = true;
		doNotify = true;
	}
	if (obj.n != null && this.numTotal != obj.n) {
		this.numTotal = obj.n;
		fields[ZmOrganizer.F_TOTAL] = true;
		doNotify = true;
	}
	if (obj.s != null && this.sizeTotal != obj.s) {
		this.sizeTotal = obj.s;
		fields[ZmOrganizer.F_SIZE] = true;
		doNotify = true;
	}
	if ((obj.rgb != null || obj.color != null) && !obj._isRemote) {
        var color = obj.color || obj.rgb;
		if (this.isColorChanged(color)) {
			this.isColorCustom = obj.rgb != null;
			this.color = obj.color;
            this.rgb = obj.rgb || ZmOrganizer.COLOR_VALUES[color];
			fields[ZmOrganizer.F_COLOR] = true;
            fields[ZmOrganizer.F_RGB] = true;
		}
		doNotify = true;
	}
	if (obj.f != null && !obj._isRemote) {
		var oflags = this._setFlags().split("").sort().join("");
		var nflags = obj.f.split("").sort().join("");
		if (oflags != nflags) {
			this._parseFlags(obj.f);
			fields[ZmOrganizer.F_FLAGS] = true;
			doNotify = true;
		}
	}
	if (obj.rest != null && this.restUrl != obj.rest && !obj._isRemote) {
		this.restUrl = obj.rest;
		fields[ZmOrganizer.F_REST_URL] = true;
		doNotify = true;
	}
	// if shares changed, do wholesale replace
	if (obj.acl) {
		this.clearShares();
		if (obj.acl.grant && obj.acl.grant.length) {
			AjxDispatcher.require("Share");
			for (var i = 0; i < obj.acl.grant.length; i++) {
				share = ZmShare.createFromJs(this, obj.acl.grant[i]);
				this.addShare(share);
			}
		}
		fields[ZmOrganizer.F_SHARES] = true;
		doNotify = true;
	}
	if (obj.perm && obj._isRemote) {
		fields[ZmOrganizer.F_PERMS] = true;
		doNotify = true;

		// clear acl-related flags so they are recalculated
		this._isAdmin = this._isReadOnly = this._hasPrivateAccess = null;
	}
    if (obj.retentionPolicy) {
        // Only displayed in a modal dialog - no need to doNotify
        if (obj.retentionPolicy[0].keep || obj.retentionPolicy[0].purge) {
            this.retentionPolicy = obj.retentionPolicy;
        } else {
            this.retentionPolicy = null;
        }
    }
	if (obj.hasOwnProperty("webOfflineSyncDays")) {
		this.webOfflineSyncDays = obj.webOfflineSyncDays;
	}

	// Send out composite MODIFY change event
	if (doNotify) {
		details.fields = fields;
		this._notify(ZmEvent.E_MODIFY, details);
	}

	if (this.parent && obj.l != null && obj.l != this.parent.id) {
		var newParent = this._getNewParent(obj.l);
		if (newParent) {
			this.reparent(newParent);
			this._notify(ZmEvent.E_MOVE);
			// could be moving search between Folders and Searches - make sure
			// it has the correct tree
			this.tree = newParent.tree;
		}
	}
};

// Local change handling

/**
 * Deletes the organizer (local). Cleans up a deleted organizer:
 *
 * <ul>
 * <li>remove from parent's list of children</li>
 * <li>remove from item cache</li>
 * <li>perform above two steps for each child</li>
 * <li>clear list of children</li>
 * </ul>
 *
 */
ZmOrganizer.prototype.deleteLocal =
function() {
	this.parent.children.remove(this);
	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		var org = a[i];
		if (org) { org.deleteLocal(); }
	}
	this.children.removeAll();
};

/**
 * Checks if the organizer has a child with the given name.
 *
 * @param {String}	name		the name of the organizer to look for
 * @return	{Boolean}	<code>true</code> if the organizer has a child
 */
ZmOrganizer.prototype.hasChild =
function(name) {
	return (this.getChild(name) != null);
};

/**
* Gets the child with the given name
*
* @param {String}	name		the name of the organizer to look for
* @return	{String}	the name of the child or <code>null</code> if no child has the name
*/
ZmOrganizer.prototype.getChild =
function(name) {
	name = name ? name.toLowerCase() : "";
	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (a[i] && a[i].name && (a[i].name.toLowerCase() == name)) {
			return a[i];
		}
	}

	return null;
};

/**
* Gets the child with the given path
*
* @param {String}	path		the path of the organizer to look for
* @return	{String}	the child or <code>null</code> if no child has the path
*/
ZmOrganizer.prototype.getChildByPath =
function(path) {
	// get starting organizer
	var organizer = this;
	if (path.match(/^\//)) {
		while (organizer.nId != ZmOrganizer.ID_ROOT) {
			organizer = organizer.parent;
		}
		path = path.substr(1);
	}

	// if no path, return current organizer
	if (path.length == 0) return organizer;

	// walk descendent axis to find organizer specified by path
	var parts = path.split('/');
	var i = 0;
	while (i < parts.length) {
		var part = parts[i++];
		var child = organizer.getChild(part);
		if (child == null) {
			return null;
		}
		organizer = child;
	}
	return organizer;
};

/**
 * Changes the parent of this organizer. Note that the new parent passed
 * in may be <code>null</code>, which makes this organizer an orphan.
 *
 * @param {ZmOrganizer}	newParent		the new parent
 */
ZmOrganizer.prototype.reparent =
function(newParent) {
	if (this.parent) {
		this.parent.children.remove(this);
	}
	if (newParent) {
		newParent.children.add(this);
	}
	this.parent = newParent;
};

/**
 * Gets the organizer with the given ID, searching recursively through
 * child organizers. The preferred method for getting an organizer by ID
 * is to use <code>appCtxt.getById()</code>.
 *
 * @param {String}	id		the ID to search for
 * @return	{ZmOrganizer}	the organizer or <code>null</code> if not found
 */
ZmOrganizer.prototype.getById =
function(id) {
	if (this.link && id && typeof(id) == "string") {
		var ids = id.split(":");
		if (this.zid == ids[0] && this.rid == ids[1])
			return this;
	}

	if (this.id == id || this.nId == id) {
		return this;
	}

	var organizer;
	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (organizer = a[i].getById(id)) {
			return organizer;
		}
	}
	return null;
};

/**
 * Gets the first organizer found with the given name, starting from the root.
 *
 * @param {String}	name		the name to search for
 * @return	{ZmOrganizer}	the organizer
 */
ZmOrganizer.prototype.getByName =
function(name, skipImap) {
	return this._getByName(name.toLowerCase(), skipImap);
};

/**
 * Gets a list of organizers with the given type.
 *
 * @param {constant}	type			the desired organizer type
 * @return	{Array}	an array of {ZmOrganizer} objects
 */
ZmOrganizer.prototype.getByType =
function(type) {
	var list = [];
	this._getByType(type, list);
	return list;
};

/**
 * @private
 */
ZmOrganizer.prototype._getByType =
function(type, list) {
	if (this.type == type) {
		list.push(this);
	}
	var a = this.children.getArray();
	for (var i = 0; i < a.length; i++) {
		if (a[i]) {
			a[i]._getByType(type, list);
		}
	}
};

/**
 * Gets the organizer with the given path.
 *
 * @param {String}	path			the path to search for
 * @param {Boolean}	useSystemName	if <code>true</code>, use untranslated version of system folder names
 * @return	{ZmOrganizer}	the organizer
 */
ZmOrganizer.prototype.getByPath =
function(path, useSystemName) {
	return this._getByPath(path.toLowerCase(), useSystemName);
};

/**
 * Test the path of this folder and then descendants against the given path, case insensitively.
 *
 * @private
 */
ZmOrganizer.prototype._getByPath =
function(path, useSystemName) {
	if (this.nId == ZmFolder.ID_TAGS) { return null; }

	if (path == this.getPath(false, false, null, true, useSystemName).toLowerCase()) {
		return this;
	}

	var a = this.children.getArray();
	for (var i = 0; i < a.length; i++) {
		var organizer = a[i]._getByPath(path, useSystemName);
		if (organizer) {
			return organizer;
		}
	}
	return null;
};

/**
 * Gets the number of children of this organizer.
 *
 * @return	{int}	the size
 */
ZmOrganizer.prototype.size =
function() {
	return this.children.size();
};

/**
 * Checks if the given organizer is a descendant of this one.
 *
 * @param {ZmOrganizer}	organizer		a possible descendant of ours
 * @return	{Boolean}	<code>if the given organizer is a descendant; <code>false</code> otherwise
 */
ZmOrganizer.prototype.isChildOf =
function (organizer) {
	var parent = this.parent;
	while (parent) {
		if (parent == organizer) {
			return true;
		}
		parent = parent.parent;
	}
	return false;
};

/**
 * Gets the organizer with the given ID (looks in this organizer tree).
 *
 * @param {int}	parentId	the ID of the organizer to find
 * @return	{ZmOrganizer}	the organizer
 *
 * @private
 */
ZmOrganizer.prototype._getNewParent =
function(parentId) {
	return appCtxt.getById(parentId);
};

/**
 * Checks if the organizer with the given ID is under this organizer.
 *
 * @param	{String}	id		the ID
 * @return	{Boolean}	<code>true</code> if the organizer is under this organizer
 */
ZmOrganizer.prototype.isUnder =
function(id) {
	id = id.toString();
	if (this.nId == id || (this.isRemote() && this.rid == id)) { return true; }

	var parent = this.parent;
	while (parent && parent.nId != ZmOrganizer.ID_ROOT) {
		if (parent.nId == id) {
			return true;
		}
		parent = parent.parent;
	}
	return false;
};

/**
 * Checks if this organizer is in "Trash".
 *
 * @return	{Boolean}	<code>true</code> if in "Trash"
 */
ZmOrganizer.prototype.isInTrash =
function() {
	return this.isUnder(ZmOrganizer.ID_TRASH);
};

/**
 * Checks if permissions are allowed.
 *
 * @return	{Boolean}	<code>true</code> if permissions are allowed
 */
ZmOrganizer.prototype.isPermAllowed =
function(perm) {
	if (this.perm) {
		var positivePerms = this.perm.replace(/-./g, "");
		return (positivePerms.indexOf(perm) != -1);
	}
	return false;
};

/**
 * Checks if the organizer is read-only.
 *
 * @return	{Boolean}	<code>true</code> if read-only
 */
ZmOrganizer.prototype.isReadOnly =
function() {
	if (this._isReadOnly == null) {
		var share = this.getMainShare();
		this._isReadOnly = (share != null)
			? (this.isRemote() && !share.isWrite())
			: (this.isRemote() && this.isPermAllowed(ZmOrganizer.PERM_READ) && !this.isPermAllowed(ZmOrganizer.PERM_WRITE));
	}
	return this._isReadOnly;
};

/**
 * Checks if admin.
 *
 * @return	{Boolean}	<code>true</code> if this organizer is admin
 */
ZmOrganizer.prototype.isAdmin =
function() {
	if (this._isAdmin == null) {
		var share = this.getMainShare();
		this._isAdmin = (share != null)
			? (this.isRemote() && share.isAdmin())
			: (this.isRemote() && this.isPermAllowed(ZmOrganizer.PERM_ADMIN));
	}
	return this._isAdmin;
};

/**
 * Checks if the organizer has private access.
 *
 * @return	{Boolean}	<code>true</code> if has private access
 */
ZmOrganizer.prototype.hasPrivateAccess =
function() {
	if (this._hasPrivateAccess == null) {
		var share = this.getMainShare();
		this._hasPrivateAccess = (share != null)
			? (this.isRemote() && share.hasPrivateAccess())
			: (this.isRemote() && this.isPermAllowed(ZmOrganizer.PERM_PRIVATE));
	}
	return this._hasPrivateAccess;
};

/**
 * Checks if the organizer is "remote". That applies to mountpoints (links),
 * the folders they represent, and any subfolders we know about.
 *
 * @return	{Boolean}	<code>true</code> if the organizer is "remote"
 */
ZmOrganizer.prototype.isRemote =
function() {
	if (this._isRemote == null) {
		if (this.zid != null) {
			this._isRemote = true;
		} else {
			if (appCtxt.multiAccounts) {
				var account = this.account;
				var parsed = ZmOrganizer.parseId(this.id);

				if (!account) {
					if (parsed.account && parsed.account.isMain) {
						this._isRemote = false;
						return this._isRemote;
					} else {
						account = appCtxt.getActiveAccount();
					}
				}
				this._isRemote = Boolean(!parsed.account || (parsed.account && (parsed.account != account)));
			} else {
				var id = String(this.id);
				this._isRemote = ((id.indexOf(":") != -1) && (id.indexOf(appCtxt.getActiveAccount().id) != 0));
			}
		}
	}
	return this._isRemote;
};

ZmOrganizer.prototype.isRemoteRoot = function() {
	return this.isRemote() && (this.rid == ZmOrganizer.ID_ROOT);
}

/**
 * Checks if the organizer is a system tag or folder.
 *
 * @return	{Boolean}	<code>true</code> if system tag or folder
 */
ZmOrganizer.prototype.isSystem =
function () {
	return (this.nId < ZmOrganizer.FIRST_USER_ID[this.type]);
};

ZmOrganizer.prototype.isDefault =
function () {
	return this.nId == this.getDefaultFolderId();
};

ZmOrganizer.prototype.getDefaultFolderId =
function() {
	return ZmOrganizer.DEFAULT_FOLDER[this.type];
};

ZmOrganizer.prototype.isTrash =
function () {
	return this.nId == ZmFolder.ID_TRASH;
};


/**
 * Checks if the organizer gets its contents from an external feed.
 *
 * @return	{Boolean}	<code>true</code>  if from external feed
 */
ZmOrganizer.prototype.isFeed =
function () {
	return Boolean(this.url);
};

/** Returns true if organizer has feeds. */
ZmOrganizer.prototype.hasFeeds = function() { return false; };

/**
 * Checks if this folder maps to a datasource. If type is given, returns
 * true if folder maps to a datasource *and* is of the given type.
 *
 * @param	{constant}	type			the type (see {@link ZmAccount.TYPE_POP} or {@link ZmAccount.TYPE_IMAP})
 * @param	{Boolean}	checkParent		if <code>true</code>, walk-up the parent chain
 * @return	{Boolean}	<code>true</code> if this folder maps to a datasource
 */
ZmOrganizer.prototype.isDataSource =
function(type, checkParent) {
	var dss = this.getDataSources(type, checkParent);
	return (dss && dss.length > 0);
};

/**
 * Gets the data sources this folder maps to. If type is given,
 * returns non-null result only if folder maps to datasource(s) *and* is of the
 * given type.
 *
 * @param	{constant}	type			the type (see {@link ZmAccount.TYPE_POP} or {@link ZmAccount.TYPE_IMAP})
 * @param	{Boolean}	checkParent		if <code>true</code>, walk-up the parent chain
 * @return	{Array}	the data sources this folder maps to or <code>null</code> for none
 */
ZmOrganizer.prototype.getDataSources =
function(type, checkParent) {
	if (!appCtxt.get(ZmSetting.MAIL_ENABLED)) { return null; }

	var dsc = appCtxt.getDataSourceCollection();
	var dataSources = dsc.getByFolderId(this.nId, type);

	if (dataSources.length == 0) {
		return (checkParent && this.parent)
			? this.parent.getDataSources(type, checkParent)
			: null;
	}

	return dataSources;
};

/**
 * Gets the owner.
 *
 * @return	{String}	the owner
 */
ZmOrganizer.prototype.getOwner =
function() {
	return this.owner || (this.parent && this.parent.getOwner()) || appCtxt.get(ZmSetting.USERNAME);
};

/**
 * Gets the sort index.
 *
 * @return	{int}	the sort index
 */
ZmOrganizer.getSortIndex =
function(child, sortFunction) {
	if (!(child && child.parent && sortFunction)) { return null; }
	var children = child.parent.children.getArray();
	for (var i = 0; i < children.length; i++) {
		var test = sortFunction(child, children[i]);
		if (test == -1) {
			return i;
		}
	}
	return i;
};

/**
 * Sends a request to the server. Note that it's done asynchronously, but
 * there is no callback given. Hence, an organizer action is the last thing
 * done before returning to the event loop. The result of the action is
 * handled via notifications.
 *
 * @param {String}	action		the operation to perform
 * @param {Hash}	attrs		a hash of additional attributes to set in the request
 * @param {ZmBatchCommand}	batchCmd	the batch command that contains this request
 *
 * @private
 */
ZmOrganizer.prototype._organizerAction =
function(params) {

	var cmd = ZmOrganizer.SOAP_CMD[this.type] + "Request";
	var request = {
		_jsns: "urn:zimbraMail",
		action : {
			op: params.action,
			id: params.id || this.id
		}
	};
	var jsonObj = {};
	jsonObj[cmd] = request;

	for (var attr in params.attrs) {
		if (AjxEnv.isIE) {
			params.attrs[attr] += ""; //To string
		}
		request.action[attr] = params.attrs[attr];
	}
	var actionController = appCtxt.getActionController();
	actionController.dismiss();
	var actionLogItem = (!params.noUndo && actionController && actionController.actionPerformed({op: params.action, id: params.id || this.id, attrs: params.attrs})) || null;
	var respCallback = new AjxCallback(this, this._handleResponseOrganizerAction, [params, actionLogItem]);
	if (params.batchCmd) {
        params.batchCmd.addRequestParams(jsonObj, respCallback, params.errorCallback);
 	}
	else {
		var accountName;
		if (appCtxt.multiAccounts) {
			accountName = (this.account)
				? this.account.name : appCtxt.accountList.mainAccount.name;
		}
		appCtxt.getAppController().sendRequest({
			jsonObj: jsonObj,
			asyncMode: true,
			accountName: accountName,
			callback: respCallback,
			errorCallback: params.errorCallback,
			timeout: params.timeout,
			noBusyOverlay: params.noBusyOverlay
		});
	}
};

/**
 * @private
 */
ZmOrganizer.prototype._handleResponseOrganizerAction =
function(params, actionLogItem, result) {

	if (actionLogItem) {
		actionLogItem.setComplete();
	}
	if (params.callback) {
		params.callback.run(result);
	}
	if (params.actionTextKey) {
		var actionController = appCtxt.getActionController();
		var summary = ZmOrganizer.getActionSummary({
			actionTextKey:  params.actionTextKey,
			numItems:       params.numItems || 1,
			type:           this.type,
			orgName:        params.orgName,
			actionArg:      params.actionArg
		});
		var undoLink = actionLogItem && actionController && actionController.getUndoLink(actionLogItem);
		if (undoLink && actionController) {
			actionController.onPopup();
			appCtxt.setStatusMsg({msg: summary+undoLink, transitions: actionController.getStatusTransitions()});
		} else {
			appCtxt.setStatusMsg(summary);
		}
	}
};

/**
 * Returns a string describing an action, intended for display as toast to tell the
 * user what they just did.
 *
 * @param   {Object}        params          hash of params:
 *          {String}        type            organizer type (ZmOrganizer.*)
 *          {String}        actionTextKey   ZmMsg key for text string describing action
 *          {String}        orgName         name of the organizer that was affected
 *          {String}        actionArg       (optional) additional argument
 *
 * @return {String}     action summary
 */
ZmOrganizer.getActionSummary =
function(params) {

	var type = params.type,
		typeKey = ZmOrganizer.FOLDER_KEY[type],
		typeText = ZmMsg[typeKey],
		capKey = AjxStringUtil.capitalize(typeKey),
		alternateKey = params.actionTextKey + capKey,
		text = ZmMsg[alternateKey] || ZmMsg[params.actionTextKey],
		orgName = AjxStringUtil.htmlEncode(params.orgName),
		arg = AjxStringUtil.htmlEncode(params.actionArg);

	return AjxMessageFormat.format(text, [ typeText, orgName, arg ]);
};

/**
 * Test the name of this organizer and then descendants against the given name, case insensitively.
 * 
 * @private
 */
ZmOrganizer.prototype._getByName =
function(name, skipImap) {
	if (this.name && name == this.name.toLowerCase()) {
		return this;
	}

	var organizer;
	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (organizer = a[i]._getByName(name, skipImap)) {
			if (skipImap && organizer.isDataSource(ZmAccount.TYPE_IMAP, true)) {
				continue;
			}
			return organizer;
		}
	}
	return null;
};

/**
 * Takes a string of flag chars and applies them to this organizer.
 * 
 * @private
 */
ZmOrganizer.prototype._parseFlags =
function(str) {
	for (var i = 0; i < ZmOrganizer.ALL_FLAGS.length; i++) {
		var flag = ZmOrganizer.ALL_FLAGS[i];
		this[ZmOrganizer.FLAG_PROP[flag]] = (Boolean(str && (str.indexOf(flag) != -1)));
	}
};

/**
 * Converts this organizer's flag-related props into a string of flag chars.
 * 
 * @private
 */
ZmOrganizer.prototype._setFlags =
function() {
	var flags = "";
	for (var i = 0; i < ZmOrganizer.ALL_FLAGS.length; i++) {
		var flag = ZmOrganizer.ALL_FLAGS[i];
		var prop = ZmOrganizer.FLAG_PROP[flag];
		if (this[prop]) {
			flags = flags + flag;
		}
	}
	return flags;
};

/**
 * Adds a change listener.
 * 
 * @param	{AjxListener}	the listener
 */
ZmOrganizer.prototype.addChangeListener =
function(listener) {
	this.tree.addChangeListener(listener);
};

/**
 * Removes a change listener.
 * 
 * @param	{AjxListener}	the listener
 */
ZmOrganizer.prototype.removeChangeListener =
function(listener) {
	this.tree.removeChangeListener(listener);
};

/**
 * @private
 */
ZmOrganizer.prototype._setSharesFromJs =
function(obj) {

	// a folder object will have an acl with grants if this user has
	// shared it, or if it has been shared to this user with admin rights
	if (obj.acl && obj.acl.grant && obj.acl.grant.length > 0) {
		AjxDispatcher.require("Share");
		for (var i = 0; i < obj.acl.grant.length; i++) {
			var grant = obj.acl.grant[i];
			this.addShare(ZmShare.createFromJs(this, grant));
		}
	}
};

/**
 * Handle notifications through the tree.
 * 
 * @private
 */
ZmOrganizer.prototype._notify =
function(event, details) {

	if (details) {
		details.organizers = [this];
	} else {
		details = {organizers: [this]};
	}
	this.tree._evt.type = this.type;	// all folder types are in a single tree
	this.tree._notify(event, details);
};

/**
 * Gets a marked-up version of the name.
 *
 * @param {String}	name			the name to mark up
 * @param {Boolean}	showUnread		if <code>true</code>, display the number of unread items (in parens)
 * @param {Boolean}	noMarkup		if <code>true</code>, do not return any HTML
 * 
 * @private
 */
ZmOrganizer.prototype._markupName = 
function(name, showUnread, noMarkup) {
	if (!noMarkup) {
		name = AjxStringUtil.htmlEncode(name, true);
	}
	if (showUnread && this.hasUnreadDescendent()) {
		if (this.numUnread > 0) {
            name = AjxMessageFormat.format(ZmMsg.folderUnread, [name, this.numUnread]);
		}
		if (!noMarkup) {
			name = ["<span style='font-weight:bold'>", name, "</span>"].join("");
		}
	}
	if (this.noSuchFolder && !noMarkup) {
		name = ["<del>", name, "</del>"].join("");
	}
	return name;
};

/**
 * @private
 */
ZmOrganizer.prototype._getItemsText =
function() {
	var result = ZmMsg[ZmOrganizer.ITEMS_KEY[this.type]];
	if (!result || this.isTrash()) {
		result = ZmMsg.items;
	}
	return result;
};

ZmOrganizer.prototype._getUnreadLabel = 
function() {
	return ZmMsg.unread;	
};

/**
 * Returns true if any descendent folders have unread messages.
 *
 * @returns {boolean}   true if any descendent folders have unread messages
 */
ZmOrganizer.prototype.hasUnreadDescendent = function() {

	if (this.numUnread > 0) {
		return true;
	}

	var a = this.children.getArray(),
		sz = this.children.size();

	for (var i = 0; i < sz; i++) {
		if (a[i].hasUnreadDescendent()) {
			return true;
		}
	}

	return false;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmFolder")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines a folder.
 *
 */

/**
 * Creates a folder.
 * @class
 * This class represents a folder, which may contain mail. At some point, folders may be
 * able to contain contacts and/or appointments.
 *
 * @author Conrad Damon
 *
 * @param	{Hash}	params		a hash of parameters
 * @param {int}	params.id		the numeric ID
 * @param {String}	params.name		the name
 * @param {ZmOrganizer}	params.parent	the parent folder
 * @param {ZmTree}	params.tree		the tree model that contains this folder
 * @param {int}	params.numUnread	the number of unread items for this folder
 * @param {int}	params.numTotal		the number of items for this folder
 * @param {int}	params.sizeTotal	the total size of folder's items
 * @param {String}	params.url		the URL for this folder's feed
 * @param {String}	params.owner	the Owner for this organizer
 * @param {String}	params.oname	the Owner's name for this organizer, if remote folder
 * @param {String}	params.zid		the Zimbra ID of owner, if remote folder
 * @param {String}	params.rid		the Remote ID of organizer, if remote folder
 * @param {String}	params.restUrl	the REST URL of this organizer
 * 
 * @extends		ZmOrganizer
 */
ZmFolder = function(params) {
	if (arguments.length == 0) { return; }
	params.type = params.type || ZmOrganizer.FOLDER;
	ZmOrganizer.call(this, params);
};

ZmFolder.prototype = new ZmOrganizer;
ZmFolder.prototype.constructor = ZmFolder;

ZmFolder.prototype.isZmFolder = true;
ZmFolder.prototype.toString = function() { return "ZmFolder"; };


// needed to construct USER_ROOT if mail disabled
ZmOrganizer.ORG_CLASS[ZmId.ORG_FOLDER] = "ZmFolder";

ZmFolder.SEP 									= "/";							// path separator

// system folders (see Mailbox.java in ZimbraServer for positive int consts)
// Note: since these are defined as Numbers, and IDs come into our system as Strings,
// we need to use == for comparisons (instead of ===, which will fail)
ZmFolder.ID_LOAD_FOLDERS						= -3;							// special "Load remaining folders" placeholder
ZmFolder.ID_OTHER								= -2;							// used for tcon value (see below)
ZmFolder.ID_SEP									= -1;							// separator
ZmFolder.ID_ROOT								= ZmOrganizer.ID_ROOT;
ZmFolder.ID_INBOX								= ZmOrganizer.ID_INBOX;
ZmFolder.ID_TRASH								= ZmOrganizer.ID_TRASH;
ZmFolder.ID_SPAM								= ZmOrganizer.ID_SPAM;
ZmFolder.ID_SENT								= 5;
ZmFolder.ID_DRAFTS								= 6;
ZmFolder.ID_CONTACTS							= ZmOrganizer.ID_ADDRBOOK;
ZmFolder.ID_AUTO_ADDED							= ZmOrganizer.ID_AUTO_ADDED;
ZmFolder.ID_TAGS	 							= 8;
ZmFolder.ID_TASKS								= ZmOrganizer.ID_TASKS;
ZmFolder.ID_SYNC_FAILURES						= ZmOrganizer.ID_SYNC_FAILURES;
ZmFolder.ID_OUTBOX	 							= ZmOrganizer.ID_OUTBOX;
ZmFolder.ID_ATTACHMENTS                         = ZmOrganizer.ID_ATTACHMENTS;
ZmFolder.ID_DLS									= ZmOrganizer.ID_DLS;
ZmFolder.ID_FILE_SHARED_WITH_ME					= ZmOrganizer.ID_FILE_SHARED_WITH_ME;

// system folder names
ZmFolder.MSG_KEY = {};
ZmFolder.MSG_KEY[ZmFolder.ID_INBOX]				= "inbox";
ZmFolder.MSG_KEY[ZmFolder.ID_TRASH]				= "trash";
ZmFolder.MSG_KEY[ZmFolder.ID_SPAM]				= "junk";
ZmFolder.MSG_KEY[ZmFolder.ID_SENT]				= "sent";
ZmFolder.MSG_KEY[ZmFolder.ID_DRAFTS]			= "drafts";
ZmFolder.MSG_KEY[ZmFolder.ID_CONTACTS]			= "contacts";
ZmFolder.MSG_KEY[ZmFolder.ID_AUTO_ADDED]		= "emailedContacts";
ZmFolder.MSG_KEY[ZmFolder.ID_TASKS]				= "tasks";
ZmFolder.MSG_KEY[ZmFolder.ID_TAGS]				= "tags";
ZmFolder.MSG_KEY[ZmOrganizer.ID_CALENDAR]		= "calendar";
ZmFolder.MSG_KEY[ZmOrganizer.ID_BRIEFCASE]		= "briefcase";
ZmFolder.MSG_KEY[ZmOrganizer.ID_ALL_MAILBOXES]	= "allMailboxes";
ZmFolder.MSG_KEY[ZmFolder.ID_OUTBOX]			= "outbox";
ZmFolder.MSG_KEY[ZmFolder.ID_SYNC_FAILURES]		= "errorReports";
ZmFolder.MSG_KEY[ZmFolder.ID_ATTACHMENTS]       = "attachments";
ZmFolder.MSG_KEY[ZmFolder.ID_FILE_SHARED_WITH_ME]	= "filesSharedWithMe";

// system folder icons
ZmFolder.ICON = {};
ZmFolder.ICON[ZmFolder.ID_INBOX]				= "Inbox";
ZmFolder.ICON[ZmFolder.ID_TRASH]				= "Trash";
ZmFolder.ICON[ZmFolder.ID_SPAM]					= "SpamFolder";
ZmFolder.ICON[ZmFolder.ID_SENT]					= "SentFolder";
ZmFolder.ICON[ZmFolder.ID_SYNC_FAILURES]		= "SendReceive";
ZmFolder.ICON[ZmFolder.ID_OUTBOX]				= "Outbox";
ZmFolder.ICON[ZmFolder.ID_DRAFTS]				= "DraftFolder";
ZmFolder.ICON[ZmFolder.ID_LOAD_FOLDERS]			= "Plus";
ZmFolder.ICON[ZmFolder.ID_ATTACHMENTS]          = "Attachment";

// name to use within the query language
ZmFolder.QUERY_NAME = {};
ZmFolder.QUERY_NAME[ZmFolder.ID_INBOX]			= "inbox";
ZmFolder.QUERY_NAME[ZmFolder.ID_TRASH]			= "trash";
ZmFolder.QUERY_NAME[ZmFolder.ID_SPAM]			= "junk";
ZmFolder.QUERY_NAME[ZmFolder.ID_SENT]			= "sent";
ZmFolder.QUERY_NAME[ZmFolder.ID_OUTBOX]			= "outbox";
ZmFolder.QUERY_NAME[ZmFolder.ID_DRAFTS]			= "drafts";
ZmFolder.QUERY_NAME[ZmOrganizer.ID_CALENDAR]	= "calendar";
ZmFolder.QUERY_NAME[ZmFolder.ID_CONTACTS]		= "contacts";
ZmFolder.QUERY_NAME[ZmFolder.ID_TASKS]			= "tasks";
ZmFolder.QUERY_NAME[ZmFolder.ID_AUTO_ADDED]		= "Emailed Contacts";
ZmFolder.QUERY_NAME[ZmOrganizer.ID_BRIEFCASE]	= "briefcase";
ZmFolder.QUERY_NAME[ZmFolder.ID_SYNC_FAILURES]	= "Error Reports";
ZmFolder.QUERY_NAME[ZmFolder.ID_FILE_SHARED_WITH_ME]	= "Files shared with me";

ZmFolder.QUERY_ID = AjxUtil.valueHash(ZmFolder.QUERY_NAME);

// order within the overview panel
ZmFolder.SORT_ORDER = {};
ZmFolder.SORT_ORDER[ZmFolder.ID_INBOX]			= 1;
ZmFolder.SORT_ORDER[ZmFolder.ID_SENT]			= 2;
ZmFolder.SORT_ORDER[ZmFolder.ID_DRAFTS]			= 3;
ZmFolder.SORT_ORDER[ZmFolder.ID_SPAM]			= 4;
ZmFolder.SORT_ORDER[ZmFolder.ID_OUTBOX]			= 5;
ZmFolder.SORT_ORDER[ZmFolder.ID_TRASH]			= 6;
ZmFolder.SORT_ORDER[ZmFolder.ID_SYNC_FAILURES]	= 7;
ZmFolder.SORT_ORDER[ZmFolder.ID_SEP]			= 8;
ZmFolder.SORT_ORDER[ZmFolder.ID_ATTACHMENTS]    = 99; // Last

// character codes for "tcon" attribute in conv action request, which controls
// which folders are affected
ZmFolder.TCON_CODE = {};
ZmFolder.TCON_CODE[ZmFolder.ID_TRASH]			= "t";
ZmFolder.TCON_CODE[ZmFolder.ID_SYNC_FAILURES]	= "o";
ZmFolder.TCON_CODE[ZmFolder.ID_SPAM]			= "j";
ZmFolder.TCON_CODE[ZmFolder.ID_SENT]			= "s";
ZmFolder.TCON_CODE[ZmFolder.ID_DRAFTS]			= "d";
ZmFolder.TCON_CODE[ZmFolder.ID_OTHER]			= "o";

// folders that look like mail folders that we don't want to show
ZmFolder.HIDE_ID = {};
ZmFolder.HIDE_ID[ZmOrganizer.ID_CHATS]				= true;
ZmFolder.HIDE_ID[ZmOrganizer.ID_NOTIFICATION_MP]	= true;

// Hide folders migrated from Outlook mailbox
ZmFolder.HIDE_NAME = {};
//ZmFolder.HIDE_NAME["Journal"]		= true;
//ZmFolder.HIDE_NAME["Notes"]		= true;
//ZmFolder.HIDE_NAME["Outbox"]		= true;
//ZmFolder.HIDE_NAME["Tasks"]		= true;

// folders that contain mail from me instead of to me
ZmFolder.OUTBOUND = [ZmFolder.ID_SENT, ZmFolder.ID_OUTBOX, ZmFolder.ID_DRAFTS];

// The extra-special, visible but untouchable outlook folder
ZmFolder.SYNC_ISSUES 							= "Sync Issues";

// map name to ID
ZmFolder.QUERY_ID = {};
(function() {
	for (var i in ZmFolder.QUERY_NAME) {
		ZmFolder.QUERY_ID[ZmFolder.QUERY_NAME[i]] = i;
	}
})();

/**
 * Comparison function for folders. Intended for use on a list of user folders
 * through a call to <code>Array.sort()</code>.
 *
 * @param {ZmFolder}	folderA		a folder
 * @param {ZmFolder}	folderB		a folder
 * @param {Boolean}		nonMail		this is sorting non mail tree.
 * @return	{int} 0 if the folders match
 */
ZmFolder.sortCompare =
function(folderA, folderB, nonMail) {
	var check = ZmOrganizer.checkSortArgs(folderA, folderB);
	if (check != null) { return check; }

	// offline client wants POP folders above all else *unless* we are POP'ing into Inbox
	if (appCtxt.isOffline) {
		if (folderA.isDataSource(ZmAccount.TYPE_POP)) {
			if (folderA.id == ZmFolder.ID_INBOX) return -1;
			if (folderB.isDataSource(ZmAccount.TYPE_POP)) {
				if (folderA.name.toLowerCase() > folderB.name.toLowerCase()) { return 1; }
				if (folderA.name.toLowerCase() < folderB.name.toLowerCase()) { return -1; }
				return 0;
			}
			return -1;
		} else if (folderB.isDataSource(ZmAccount.TYPE_POP)) {
			return 1;
		}
	}

	if (ZmFolder.SORT_ORDER[folderA.nId] && ZmFolder.SORT_ORDER[folderB.nId]) {
		return (ZmFolder.SORT_ORDER[folderA.nId] - ZmFolder.SORT_ORDER[folderB.nId]);
	}

	// links (shared folders or mailboxes) appear after personal folders
	if (folderA.link !== folderB.link) {
		return folderA.link ? 1 : -1;
	}

	if (nonMail) {
		//for nonp-mail apps, trash last of all things
		if (folderA.isTrash()) {
			return 1;
		}
		if (folderB.isTrash()) {
			return -1;
		}
		//system before non-system (except for trash)
		if (folderA.isSystem() && !folderB.isSystem()) {
			return -1;
		}
		if (!folderA.isSystem() && folderB.isSystem()) {
			return 1;
		}
		if (folderA.isSystem() && folderB.isSystem()) {
			//for 2 system folders, the default one is first, and the rest ordered alphabetically (again except for trash that appears after the user folders)
			if (folderA.isDefault()) {
				return -1;
			}
			if (folderB.isDefault()) {
				return 1;
			}
			//the other cases will be sorted by name below. Either 2 system or 2 user folders.
		}
	}
	else {
		if (!ZmFolder.SORT_ORDER[folderA.nId] && ZmFolder.SORT_ORDER[folderB.nId]) { return 1; }
		if (ZmFolder.SORT_ORDER[folderA.nId] && !ZmFolder.SORT_ORDER[folderB.nId]) { return -1; }
	}

	if (folderA.name.toLowerCase() > folderB.name.toLowerCase()) { return 1; }
	if (folderA.name.toLowerCase() < folderB.name.toLowerCase()) { return -1; }
	return 0;
};


ZmFolder.sortCompareNonMail =
function(folderA, folderB) {
	return ZmFolder.sortCompare(folderA, folderB, true);
};

/**
 * Compares the folders by path.
 * 
 * @param {ZmFolder}	folderA		a folder
 * @param {ZmFolder}	folderB		a folder
 * @return	{int} 0 if the folders match
 */
ZmFolder.sortComparePath =
function(folderA, folderB) {

	var pathA = folderA && folderA.getPath(false, false, null, true, true);
	var pathB = folderB && folderB.getPath(false, false, null, true, true);
	var check = ZmOrganizer.checkSortArgs(pathA, pathB);
	if (check != null) { return check; }

	if (ZmFolder.SORT_ORDER[folderA.nId] && ZmFolder.SORT_ORDER[folderB.nId]) {
		return (ZmFolder.SORT_ORDER[folderA.nId] - ZmFolder.SORT_ORDER[folderB.nId]);
	}
	if (!ZmFolder.SORT_ORDER[folderA.nId] && ZmFolder.SORT_ORDER[folderB.nId]) { return 1; }
	if (ZmFolder.SORT_ORDER[folderA.nId] && !ZmFolder.SORT_ORDER[folderB.nId]) { return -1; }
	if (pathA.toLowerCase() > pathB.toLowerCase()) { return 1; }
	if (pathA.toLowerCase() < pathB.toLowerCase()) { return -1; }
	return 0;
};

/**
 * Checks a folder name for validity. Note: that a name, rather than a path, is checked.
 *
 * @param {String}	name		the folder name
 * @param {ZmFolder}	parent		the parent folder
 * @return	{String} an error message if the name is invalid; <code>null</code>if the name is valid. 
 */
ZmFolder.checkName =
function(name, parent) {
	var error = ZmOrganizer.checkName(name);
	if (error) { return error; }

	// make sure path isn't same as a system folder
	parent = parent || appCtxt.getFolderTree().root;
	if (parent && (parent.id == ZmFolder.ID_ROOT)) {
		var lname = name.toLowerCase();
		for (var id in ZmFolder.MSG_KEY) {
			var sysname = ZmMsg[ZmFolder.MSG_KEY[id]];
			if (sysname && (lname == sysname.toLowerCase())) {
				return ZmMsg.folderNameReserved;
			}
		}
		/*if (lname == ZmFolder.SYNC_ISSUES.toLowerCase()) {
			return ZmMsg.folderNameReserved;
		}*/
	}

	return null;
};

/**
 * Gets the "well-known" ID for a given folder name.
 * 
 * @param	{String}	folderName	the folder name
 * @return	{String}	the id or <code>null</code> if not found
 */
ZmFolder.getIdForName =
function(folderName) {
	var name = folderName.toLowerCase();
	for (var i in ZmFolder.MSG_KEY) {
		if (ZmFolder.MSG_KEY[i] == name) {
			return i;
		}
	}
	return null;
};

/**
 * Moves a folder. A user can move a folder to "Trash" even if there is already a folder in "Trash" with the
 * same name. A new name will be generated for this folder and a rename is performed before the move.
 * 
 * @param	{ZmFolder}	newParent		the new parent
 * @param	{boolean}	noUndo			true if the action should not be undoable
 * @param	{String}	actionText		optional custom action text to display as summary
 */
ZmFolder.prototype.move =
function(newParent, noUndo, actionText, batchCmd) {
	var origName = this.name;
	var name = this.name;
	while (newParent.hasChild(name)) {
		name = name + "_";
	}
	if (origName != name) {
		this.rename(name);
	}
	ZmOrganizer.prototype.move.call(this, newParent, noUndo, batchCmd);
};

/**
 * Sends <code>&lt;FolderActionRequest&gt;</code> to turn sync'ing on/off for IMAP folders. Currently,
 * this is only used by Offline/ZDesktop client
 *
 * @param {Boolean}	syncIt		the flag indicating whether to sync this folder
 * @param {AjxCallback}	callback		the callback to call once server request is successful
 * @param {AjxCallback}	errorCallback	the callback to call if server returns error
 */
ZmFolder.prototype.toggleSyncOffline =
function(callback, errorCallback) {
	if (!this.isOfflineSyncable) { return; }

	var op = this.isOfflineSyncing ? "!syncon" : "syncon";
	var soapDoc = AjxSoapDoc.create("FolderActionRequest", "urn:zimbraMail");
	var actionNode = soapDoc.set("action");
	actionNode.setAttribute("op", op);
	actionNode.setAttribute("id", this.id);

	var params = {
		soapDoc: soapDoc,
		asyncMode: true,
		callback: callback,
		errorCallback: errorCallback
	};
	appCtxt.getAppController().sendRequest(params);
};

/**
 * Checks folders recursively for feeds.
 * 
 * @return	{Boolean}	<code>true</code> for feeds
 */
ZmFolder.prototype.hasFeeds =
function() {
	if (this.type != ZmOrganizer.FOLDER) { return false; }

	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (a[i].isFeed()) {
			return true;
		}
		if (a[i].children && a[i].children.size() > 0) {
			return (a[i].hasFeeds && a[i].hasFeeds());
		}
	}
	return false;
};

/**
 * Checks if the folder has search.
 * 
 * @param	{String}	id	not used
 * @return	{Boolean}	<code>true</code> if has search
 */
ZmFolder.prototype.hasSearch =
function(id) {
	if (this.type == ZmOrganizer.SEARCH) { return true; }

	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (a[i].hasSearch()) {
			return true;
		}
	}

	return false;
};

/**
 * Checks if the folder supports public access. Override this method if you dont want a folder to be accessed publicly
 * 
 * @return	{Boolean}	always returns <code>true</code>
 */
ZmFolder.prototype.supportsPublicAccess =
function() {
	return true;
};

/**
 * Handles the creation of a folder or search folder. This folder is the parent
 * of the newly created folder. A folder may hold a folder or search folder,
 * and a search folder may hold another search folder.
 *
 * @param {Object}	obj				a JS folder object from the notification
 * @param {String}	elementType		the type of containing JSON element
 * @param {Boolean}	skipNotify		<code>true</code> if notifying client should be ignored
 */
ZmFolder.prototype.notifyCreate =
function(obj, elementType, skipNotify) {
	// ignore creates of system folders
	var nId = ZmOrganizer.normalizeId(obj.id);
	if (this.isSystem() && nId < ZmOrganizer.FIRST_USER_ID[this.type]) { return; }

	var account = ZmOrganizer.parseId(obj.id).account;
	var folder = ZmFolderTree.createFromJs(this, obj, this.tree, elementType, null, account);
	if (folder) {
		var index = ZmOrganizer.getSortIndex(folder, eval(ZmTreeView.COMPARE_FUNC[this.type]));
		this.children.add(folder, index);

		if (!skipNotify) {
			folder._notify(ZmEvent.E_CREATE);
		}
	}
};

/**
 * Provide some extra info in the change event about the former state
 * of the folder. Note that we null out the field after setting up the
 * change event, so the notification isn't also sent when the parent
 * class's method is called.
 *
 * @param {Object}	obj	a "modified" notification
 */
ZmFolder.prototype.notifyModify =
function(obj) {
	var details = {};
	var fields = {};
	var doNotify = false;
	if (obj.name != null && this.name != obj.name && obj.id == this.id) {
		details.oldPath = this.getPath();
		this.name = obj.name;
		fields[ZmOrganizer.F_NAME] = true;
		this.parent.children.sort(eval(ZmTreeView.COMPARE_FUNC[this.type]));
		doNotify = true;
		obj.name = null;
	}
	if (doNotify) {
		details.fields = fields;
		this._notify(ZmEvent.E_MODIFY, details);
	}

	if (obj.l != null && (!this.parent || (obj.l != this.parent.id))) {
		var newParent = this._getNewParent(obj.l);
		if (newParent) {
			details.oldPath = this.getPath();
			this.reparent(newParent);
			this._notify(ZmEvent.E_MOVE, details);
			obj.l = null;
		}
	}

	ZmOrganizer.prototype.notifyModify.apply(this, [obj]);
};

/**
 * Creates a query.
 * 
 * @param	{Boolean}	pathOnly	<code>true</code> if to use the path only
 * @return	{String}	the query
 */
ZmFolder.prototype.createQuery =
function(pathOnly) {
	if (!this.isRemote() && this.isSystem()) {
		var qName = ZmFolder.QUERY_NAME[this.nId] || this.getName(false, null, true, true) || this.name;
		// put quotes around folder names that consist of multiple words or have special characters.
		var quote = /^\w+$/.test(qName) ? "" : "\"";
		return pathOnly
			? qName
			: ("in:" + quote + qName + quote);
	}

	var path = this.isSystem() ? ZmFolder.QUERY_NAME[this.nId] : this.name;
	var f = this.parent;
	while (f && (f.nId != ZmFolder.ID_ROOT) && f.name.length) {
		var name = (f.isSystem() && ZmFolder.QUERY_NAME[f.nId]) || f.name;
		path = name + "/" + path;
		f = f.parent;
	}
	path = '"' + path + '"';
	return pathOnly ? path : ("in:" + path);
};

/**
 * Gets the name.
 * 
 * @param	{Boolean}	 showUnread		<code>true</code> to show unread
 * @param	{int}		maxLength		the max length
 * @param	{Boolean}	noMarkup		<code>true</code> to not include markup
 * @param	{Boolean}	useSystemName	<code>true</code> to use the system name
 * 
 * @return	{String}	the name
 */
ZmFolder.prototype.getName =
function(showUnread, maxLength, noMarkup, useSystemName) {
	if (this.nId == ZmFolder.ID_DRAFTS ||
		this.nId == ZmFolder.ID_OUTBOX ||
		this.rid == ZmFolder.ID_DRAFTS)
	{
		var name = (useSystemName && this._systemName) ? this._systemName : this.name;
		if (showUnread && this.numTotal > 0) {
			name = AjxMessageFormat.format(ZmMsg.folderUnread, [name, this.numTotal]);
			if (!noMarkup) {
				name = ["<span style='font-weight:bold'>", name, "</span>"].join("");
			}
		}
		return name;
	}
	else {
		return ZmOrganizer.prototype.getName.apply(this, arguments);
	}
};

/**
 * Gets the icon.
 * 
 * @return	{String}	the icon
 */
ZmFolder.prototype.getIcon =
function() {
	if (this.nId == ZmOrganizer.ID_ROOT)			{ return null; }
	if (ZmFolder.ICON[this.nId])					{ return ZmFolder.ICON[this.nId]; }
	if (this.isFeed())								{ return "RSS"; }
	if (this.isRemote())							{ return "SharedMailFolder"; }
	if (this.isDataSource(ZmAccount.TYPE_POP))		{ return "POPAccount"; }

	// make a "best-effort" to map imap folders to a well-known icon
	// (parent will be the root imap folder)
	var mappedId = this.getSystemEquivalentFolderId();
	if (mappedId) {
		return ZmFolder.ICON[mappedId] || "Folder";
	}

	return "Folder";
};

ZmFolder.prototype.getSystemEquivalentFolderId =
function() {
	if (this.parent && this.parent.isDataSource(ZmAccount.TYPE_IMAP)) {
		return ZmFolder.getIdForName(this.name);
	}
	return null;
};

ZmFolder.prototype.isSystemEquivalent =
function() {
	return this.getSystemEquivalentFolderId() != null;
};

ZmFolder.prototype.mayContainFolderFromAccount =
function(otherAccount) {
	var thisAccount = this.getAccount();
	if (thisAccount == otherAccount) {
		return true;
	}
	return thisAccount.isLocal(); // can only move to local

};

/**
 * Returns true if the given object(s) may be placed in this folder.
 *
 * If the object is a folder, check that:
 * <ul>
 * <li>We are not the immediate parent of the folder</li>
 * <li>We are not a child of the folder</li>
 * <li>We are not Spam or Drafts</li>
 * <li>We don't already have a child with the folder's name (unless we are in Trash)</li>
 * <li>We are not moving it into a folder of a different type</li>
 * <li>We are not moving a folder into itself</li>
 * </ul>
 *
 * If the object is an item or a list or items, check that:
 * <ul>
 * <li>We are not the Folders container</li>
 * <li>We are not a search folder</li>
 * <li>The items aren't already in this folder</li>
 * <li>A contact can only be moved to Trash</li>
 * <li> A draft can be moved to Trash or Drafts</li>
 * <li>Non-drafts cannot be moved to Drafts</li>
 * </ul>
 *
 * @param {Object}	what		the object(s) to possibly move into this folder (item or organizer)
 * @param {constant}	folderType	the contextual folder type (for tree view root items)
 * @param {boolean}	ignoreExisting  Set to true if checks for item presence in the folder should be skipped (e.g. when recovering deleted items)
 */
ZmFolder.prototype.mayContain =
function(what, folderType, ignoreExisting) {

	if (!what) {
		return true;
	}
	if (this.isFeed() /*|| this.isSyncIssuesFolder()*/) {
		return false;
	}
	// placeholder for showing a large number of folders
	if (this.id == ZmFolder.ID_LOAD_FOLDERS) {
		return false;
	}

	var thisType = folderType || this.type;
	var invalid = false;
	if (what instanceof ZmFolder) {
        invalid = ((what.parent === this && !ignoreExisting) || this.isChildOf(what) || this.nId == ZmFolder.ID_DRAFTS || this.nId == ZmFolder.ID_SPAM ||
				   (!this.isInTrash() && this.hasChild(what.name) && !ignoreExisting) ||
	               (what.type !== thisType && this.nId != ZmFolder.ID_TRASH) ||
				   (what.id === this.id) ||
				   (this.disallowSubFolder) ||
				   (appCtxt.multiAccounts && !this.mayContainFolderFromAccount(what.getAccount())) || // cannot move folders across accounts, unless the target is local
                   (this.isRemote() && !this._remoteMoveOk(what)) ||
				   (what.isRemote() && !this._remoteMoveOk(what)));				// a remote folder can be DnD but not its children
    } else {
		// An item or an array of items is being moved
		var items = AjxUtil.toArray(what);
		var item = items[0];

        // container can only have folders/searches or calendars
		if ((this.nId == ZmOrganizer.ID_ROOT && (what.type !== ZmOrganizer.CALENDAR)) ||
             // nothing can be moved to outbox/sync failures folders
			 this.nId == ZmOrganizer.ID_OUTBOX ||
			 this.nId == ZmOrganizer.ID_SYNC_FAILURES)
		{
			invalid = true;
		} else if (thisType === ZmOrganizer.SEARCH) {
			invalid = true;														// can't drop items into saved searches
		} else if (item && (item.type === ZmItem.CONTACT) && item.isGal) {
			invalid = true;
		} else if (item && (item.type === ZmItem.CONV) && item.list && item.list.search && (item.list.search.folderId === this.id)) {
			invalid = true;														// convs which are a result of a search for this folder
		} else {																// checks that need to be done for each item
			for (var i = 0; i < items.length; i++) {
				var childItem = items[i];
				if (!childItem) {
					invalid = true;
					break;
				}
				if (Dwt.instanceOf(childItem, "ZmBriefcaseFolderItem")) {
                     if (childItem.folder && childItem.folder.isRemote() && !childItem.folder.rid) {
                        invalid = true;
                        break;
                     }
                } else if (item.type === ZmItem.MSG && childItem.isDraft && (this.nId != ZmFolder.ID_TRASH && this.nId != ZmFolder.ID_DRAFTS && this.rid != ZmFolder.ID_DRAFTS)) {
					// can move drafts only into Trash or Drafts
					invalid = true;
					break;
				} else if ((this.nId == ZmFolder.ID_DRAFTS || this.rid == ZmFolder.ID_DRAFTS) && !childItem.isDraft)	{
					// only drafts can be moved into Drafts
					invalid = true;
					break;
				}
			}
			// items in the "Sync Failures" folder cannot be dragged out
			if (appCtxt.isOffline && !invalid) {
				// bug: 41531 - don't allow items to be moved into exchange
				// account when moving across accounts
				var acct = this.getAccount();
				if (acct && item.getAccount() != acct &&
					(acct.type === ZmAccount.TYPE_MSE ||
					 acct.type === ZmAccount.TYPE_EXCHANGE))
				{
					invalid = true;
				}
				else {
					var cs = appCtxt.getCurrentSearch();
					var folder = cs && appCtxt.getById(cs.folderId);
					if (folder && folder.nId == ZmOrganizer.ID_SYNC_FAILURES) {
						invalid = true;
					}
				}
			}

			// bug #42890 - disable moving to shared folders across accounts
			// until server bug is fixed
			if (appCtxt.multiAccounts && this.isRemote() &&
				what.getAccount && this.getAccount().id != what.getAccount().id)
			{
				invalid = true;
			}

			// can't move items to folder they're already in; we're okay if we
			// have one item from another folder
			if (!invalid && !ignoreExisting) {
				if (item && item.folderId) {
					invalid = true;
					for (var i = 0; i < items.length; i++) {
						if (items[i].folderId != this.id) {
							invalid = false;
							break;
						}
					}
				}
			}
		}
		if (!invalid && this.link) {
			invalid = this.isReadOnly();										// cannot drop anything onto a read-only item
		}
	}
	return !invalid;
};

/**
 * Checks if this is the sync issues folder.
 * 
 * @return	{Boolean}	<code>true</code> if the folder is the one dealing with Outlook sync issues
 */

//Bug#68799 Removing special handling of the folder named "Sync Issues"

/*ZmFolder.prototype.isSyncIssuesFolder =
function() {
	return (this.name == ZmFolder.SYNC_ISSUES);
};*/

/**
 * Checks if this folder required hard delete.
 * 
 * @return	{Boolean}	<code>true</code> if deleting items w/in this folder should be hard deleted.
 */
ZmFolder.prototype.isHardDelete =
function() {
	return (this.isInTrash() || this.isInSpam() || (appCtxt.isOffline && this.isUnder(ZmOrganizer.ID_SYNC_FAILURES)));
};

/**
 * Checks if this folder is in spam folder.
 * 
 * @return	{Boolean}	<code>true</code> if in spam
 */
ZmFolder.prototype.isInSpam =
function(){
	return this.isUnder(ZmFolder.ID_SPAM);
};

/**
 *
 * @param {ZmFolder}	folder  the source folder
 * 
 * @return {Boolean}	<code>true/code> if the given remote folder can be moved into this remote folder.
 * The source and the target folder must belong to the same account. The source
 * must have delete permission and the target must have insert permission.
 * 
 * @private
 */
ZmFolder.prototype._remoteMoveOk =
function(folder) {
	if (!this.isRemote() && folder.isMountpoint && folder.rid) { return true; }
	if (!this.link || !folder.link || this.getOwner() !== folder.getOwner()) { return false; }
	if (!this._folderActionOk(this, "isInsert")) {
		return false;
	}
	return this._folderActionOk(folder, "isDelete");
};

ZmFolder.prototype._folderActionOk =
function(folder, func) {
	var share = folder.shares && folder.shares[0];
	if (!share) {
		//if shares is not set, default to readOnly.
		return !folder.isReadOnly();
	}
	return share[func]();
};
/**
 * Returns true if this folder is for outbound mail.
 */
ZmFolder.prototype.isOutbound =
function() {
	for (var i = 0; i < ZmFolder.OUTBOUND.length; i++) {
		if (this.isUnder(ZmFolder.OUTBOUND[i])) {
			return true;
		}
	}
	return false;
};


/**
 * Sets the Global Mark Read flag.  When the user sets this flag, read flags are global for all
 * shared instances of the folder. When not set, each user accessing the shared folder will maintain
 * their own read/unread flag.
 *
 * @param	{Object}	        globalMarkRead		the globalMarkRead boolean flag
 * @param	{AjxCallback}	    callback		    the callback
 * @param	{AjxCallback}	    errorCallback		the error callback
 * @param   {ZmBatchCommand}    batchCmd            optional batch command
 */
ZmOrganizer.prototype.setGlobalMarkRead = function(globalMarkRead, callback, errorCallback, batchCmd) {
	if (this.globalMarkRead == globalMarkRead) { return; }
    // TODO: Bug 59559, awaiting server side implementation (Bug 24567)
    // TODO: - For ZmFolderPropsDialog and ZmSharePropsDialog:
    // TODO:     Make sure that the attrName is indeed globalMarkRead - used in the dialogs
    // TODO:     Make the globalMarkRead labels and controls visible.
    // TODO: - Uncomment this once the server call is ready, make sure action/attrs are correct
	//this._organizerAction({action: "globalMarkRead", attrs: {globalMarkRead: globalMarkRead}, callback: callback,
    //                       errorCallback: errorCallback, batchCmd: batchCmd});
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmSearchFolder")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines a search folder class.
 */

/**
 * Creates the search folder.
 * @class
 * This class represents a search folder.
 * 
 * @param	{Hash}	params		a hash of parameters
 * 
 * @extends	ZmFolder
 */
ZmSearchFolder = function(params) {
	params.type = ZmOrganizer.SEARCH;
	ZmFolder.call(this, params);
	
	if (params.query) {
		var searchParams = {
			query:			params.query,
			types:			params.types,
			checkTypes:		true,
			sortBy:			params.sortBy,
			searchId:		params.id,
			accountName:	(params.account && params.account.name)
		};
		this.search = new ZmSearch(searchParams);
	}
};

ZmSearchFolder.prototype = new ZmFolder;
ZmSearchFolder.prototype.constructor = ZmSearchFolder;

/**
 * Returns a string representation of the object.
 *
 * @return		{String}		a string representation of the object
 */
ZmSearchFolder.prototype.toString =	function() {
	return "ZmSearchFolder";
};

ZmSearchFolder.ID_ROOT = ZmOrganizer.ID_ROOT;

/**
 * Creates a search folder.
 * 
 * @param	{Hash}	params		a hash of parameters
 */
ZmSearchFolder.create = function(params) {

	params = params || {};

	var search = params.search,
		jsonObj = { CreateSearchFolderRequest: { _jsns:"urn:zimbraMail" } },
		searchNode = jsonObj.CreateSearchFolderRequest.search = {};

	searchNode.name = params.name;
	searchNode.query = search.query;
	searchNode.l = params.l;
	if (params.sortBy) {
		searchNode.sortBy = params.sortBy;
	}

	searchNode.types = ZmSearchFolder._getSearchTypes(search);

	if (params.rgb) {
		searchNode.rgb = params.rgb;
	}
	else if (params.color) {
		var color = ZmOrganizer.getColorValue(params.color, params.type);
		if (color) {
			searchNode.color = color;
		}
	}

	var accountName;
	if (params.isGlobal) {
		searchNode.f = 'g';
		accountName = appCtxt.accountList.mainAccount.name;
	}

	return appCtxt.getAppController().sendRequest({
		jsonObj:        jsonObj,
		asyncMode:      params.asyncMode !== false,
		accountName:    accountName,
		callback:       ZmSearchFolder._handleCreate,
		errorCallback:  params.errorCallback || ZmOrganizer._handleErrorCreate.bind(null)
	});
};

// converts a vector of types to a string the server can understand
ZmSearchFolder._getSearchTypes = function(search) {

	var typeStr = "";
	if (search && search.types) {
		var a = search.types.getArray();
		if (a.length) {
			var typeStr = [];
			for (var i = 0; i < a.length; i++) {
				typeStr.push(ZmSearch.TYPE[a[i]]);
			}
			typeStr = typeStr.join(",");
		}
	}
	return typeStr;
};

ZmSearchFolder._handleCreate =
function(params) {
	appCtxt.setStatusMsg(ZmMsg.searchSaved);
};

/**
 * Sets the underlying search query.
 *
 * @param	{String}	    query		    search query
 * @param	{AjxCallback}	callback		the callback
 * @param	{AjxCallback}	errorCallback		the error callback
 * @param	{ZmBatchCommand}	batchCmd		the batch command
 */
ZmSearchFolder.prototype.setQuery = function(query, callback, errorCallback, batchCmd) {

	if (query === this.search.query) {
		return;
	}

	var params = {
		callback:       callback,
		errorCallback:  errorCallback,
		batchCmd:       batchCmd
	};

	var cmd = "ModifySearchFolderRequest";
	var request = {
		_jsns: "urn:zimbraMail",
		search: {
			query:  query,
			id:     params.id || this.id,
			types:  ZmSearchFolder._getSearchTypes(this.search)
		}
	};
	var jsonObj = {};
	jsonObj[cmd] = request;

	var respCallback = this._handleResponseOrganizerAction.bind(this, params);
	if (params.batchCmd) {
		params.batchCmd.addRequestParams(jsonObj, respCallback, params.errorCallback);
	}
	else {
		var accountName;
		if (appCtxt.multiAccounts) {
			accountName = this.account ? this.account.name : appCtxt.accountList.mainAccount.name;
		}
		appCtxt.getAppController().sendRequest({
			jsonObj:        jsonObj,
			asyncMode:      true,
			accountName:    accountName,
			callback:       respCallback,
			errorCallback:  params.errorCallback
		});
	}
};

/**
 * Gets the icon.
 * 
 * @return	{String}	the icon
 */
ZmSearchFolder.prototype.getIcon = 
function() {
	return (this.nId == ZmOrganizer.ID_ROOT)
		? null
		: (this.isOfflineGlobalSearch ? "GlobalSearchFolder" : "SearchFolder");
};

/**
 * Gets the tool tip.
 * 
 */
ZmSearchFolder.prototype.getToolTip = function() {};

/**
 * Returns the organizer with the given ID. Looks in this organizer's tree first.
 * Since a search folder may have either a regular folder or another search folder
 * as its parent, we may need to get the parent folder from another type of tree.
 *
 * @param {int}	parentId	the ID of the organizer to find
 * 
 * @private
 */
ZmSearchFolder.prototype._getNewParent =
function(parentId) {
	var parent = appCtxt.getById(parentId);
	if (parent) {
		return parent;
	}
	
	return appCtxt.getById(parentId);
};

// Handle a change to the underlying search query
ZmSearchFolder.prototype.notifyModify =	function(obj) {

	if (obj.query && obj.query !== this.search.query && obj.id === this.id) {
		this.search.query = obj.query;
		var fields = {};
		fields[ZmOrganizer.F_QUERY] = true;
		this._notify(ZmEvent.E_MODIFY, {
			fields: fields
		});
		obj.query = null;
	}
	ZmFolder.prototype.notifyModify.apply(this, [obj]);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmSearch")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * The file defines a search class.
 * 
 */

/**
 * Creates a new search with the given properties.
 * @class
 * This class represents a search to be performed on the server. It has properties for
 * the different search parameters that may be used. It can be used for a regular search,
 * or to search within a conversation. The results are returned via a callback.
 *
 * @param {Hash}		params		a hash of parameters
 * @param   {String}	params.query					the query string
 * @param	{String}	params.queryHint				the query string that gets appended to the query but not something the user needs to know about
 * @param	{AjxVector}	params.types					the item types to search for
 * @param	{Boolean}	params.forceTypes				use the types we pass, do not override (in case of mail) to the current user's view pref (MSG vs. CONV).
 * @param	{constant}	params.sortBy					the sort order
 * @param	{int}		params.offset					the starting point within result set
 * @param	{int}		params.limit					the number of results to return
 * @param	{Boolean}	params.getHtml					if <code>true</code>, return HTML part for inlined msg
 * @param	{constant}	params.contactSource			where to search for contacts (GAL or personal)
 * @param	{Boolean}	params.isGalAutocompleteSearch	if <code>true</code>, autocomplete against GAL
 * @param	{constant}	params.galType					the type of GAL autocomplete (account or resource)
 * @param	{constant}	params.autocompleteType			the type of autocomplete (account or resource or all)
 * @param	{int}		params.lastId					the ID of last item displayed (for pagination)
 * @param	{String}	params.lastSortVal				the value of sort field for above item
 * @param	{Boolean}	params.fetch					if <code>true</code>, fetch first hit message
 * @param	{int}		params.searchId					the ID of owning search folder (if any)
 * @param	{Array}		params.conds					the list of search conditions (<code><SearchCalendarResourcesRequest></code>)
 * @param	{Array}		params.attrs					the list of attributes to return (<code><SearchCalendarResourcesRequest></code>)
 * @param	{String}	params.field					the field to search within (instead of default)
 * @param	{Object}	params.soapInfo					the object with method, namespace, response, and additional attribute fields for creating soap doc
 * @param	{Object}	params.response					the canned JSON response (no request will be made)
 * @param	{Array}		params.folders					the list of folders for autocomplete
 * @param	{Array}		params.allowableTaskStatus		the list of task status types to return (assuming one of the values for "types" is "task")
 * @param	{String}	params.accountName				the account name to run this search against
 * @param	{Boolean}	params.idsOnly					if <code>true</code>, response returns item IDs only
 * @param   {Boolean}   params.inDumpster               if <code>true</code>, search in the dumpster
 * @param	{string}	params.origin					indicates what initiated the search
 * @param	{boolean}	params.isEmpty					if true, return empty response without sending a request
 */
ZmSearch = function(params) {

	params = params || {};
	for (var p in params) {
		this[p] = params[p];
	}
	this.galType					= this.galType || ZmSearch.GAL_ACCOUNT;
	this.join						= this.join || ZmSearch.JOIN_AND;

	if (this.query || this.queryHint) {
		// only parse regular searches
		if (!this.isGalSearch && !this.isAutocompleteSearch &&
			!this.isGalAutocompleteSearch && !this.isCalResSearch) {
			
			var pq = this.parsedQuery = new ZmParsedQuery(this.query || this.queryHint);
			this._setProperties();
			var sortTerm = pq.getTerm("sort");
			if (sortTerm) {
				this.sortBy = sortTerm.arg;
			}
		}
	}

	this.isGalSearch = false;
	this.isCalResSearch = false;

	if (ZmSearch._mailEnabled == null) {
		ZmSearch._mailEnabled = appCtxt.get(ZmSetting.MAIL_ENABLED);
		if (ZmSearch._mailEnabled) {
			AjxDispatcher.require("MailCore");
		}
	}

	if (params.checkTypes) {
		var types = AjxUtil.toArray(this.types);
		var enabledTypes = [];
		for (var i = 0; i < types.length; i++) {
			var type = types[i];
			var app = ZmItem.APP[type];
			if (appCtxt.get(ZmApp.SETTING[app])) {
				enabledTypes.push(type);
			}
		}
		this.types = AjxVector.fromArray(enabledTypes);
	}
};

ZmSearch.prototype.isZmSearch = true;
ZmSearch.prototype.toString = function() { return "ZmSearch"; };

// Search types
ZmSearch.TYPE = {};
ZmSearch.TYPE_ANY = "any";

ZmSearch.GAL_ACCOUNT	= "account";
ZmSearch.GAL_RESOURCE	= "resource";
ZmSearch.GAL_ALL		= "";

ZmSearch.JOIN_AND	= 1;
ZmSearch.JOIN_OR	= 2;

ZmSearch.TYPE_MAP = {};

ZmSearch.DEFAULT_LIMIT = DwtListView.DEFAULT_LIMIT;

// Sort By
ZmSearch.DATE_DESC 		= "dateDesc";
ZmSearch.DATE_ASC 		= "dateAsc";
ZmSearch.SUBJ_DESC 		= "subjDesc";
ZmSearch.SUBJ_ASC 		= "subjAsc";
ZmSearch.NAME_DESC 		= "nameDesc";
ZmSearch.NAME_ASC 		= "nameAsc";
ZmSearch.SIZE_DESC 		= "sizeDesc";
ZmSearch.SIZE_ASC 		= "sizeAsc";
ZmSearch.RCPT_ASC       = "rcptAsc";
ZmSearch.RCPT_DESC      = "rcptDesc";
ZmSearch.ATTACH_ASC     = "attachAsc"
ZmSearch.ATTACH_DESC    = "attachDesc"
ZmSearch.FLAG_ASC       = "flagAsc";
ZmSearch.FLAG_DESC      = "flagDesc";
ZmSearch.MUTE_ASC       = "muteAsc";
ZmSearch.MUTE_DESC      = "muteDesc";
ZmSearch.READ_ASC       = "readAsc";
ZmSearch.READ_DESC      = "readDesc";
ZmSearch.PRIORITY_ASC   = "priorityAsc";
ZmSearch.PRIORITY_DESC  = "priorityDesc";
ZmSearch.SCORE_DESC 	= "scoreDesc";
ZmSearch.DURATION_DESC	= "durDesc";
ZmSearch.DURATION_ASC	= "durAsc";
ZmSearch.STATUS_DESC	= "taskStatusDesc";
ZmSearch.STATUS_ASC		= "taskStatusAsc";
ZmSearch.PCOMPLETE_DESC	= "taskPercCompletedDesc";
ZmSearch.PCOMPLETE_ASC	= "taskPercCompletedAsc";
ZmSearch.DUE_DATE_DESC	= "taskDueDesc";
ZmSearch.DUE_DATE_ASC	= "taskDueAsc";



ZmSearch.prototype.execute =
function(params) {
	if (params.batchCmd || this.soapInfo) {
		return this._executeSoap(params);
	} else {
		return this._executeJson(params);
	}
};

/**
 * Creates a SOAP request that represents this search and sends it to the server.
 *
 * @param {Hash}	params		a hash of parameters
 * @param {AjxCallback}	params.callback		the callback to run when response is received
 * @param {AjxCallback}	params.errorCallback	the callback to run if there is an exception
 * @param {ZmBatchCommand}	params.batchCmd		the batch command that contains this request
 * @param {int}	params.timeout		the timeout value (in seconds)
 * @param {Boolean}	params.noBusyOverlay	if <code>true</code>, don't use the busy overlay
 * 
 * @private
 */
ZmSearch.prototype._executeSoap =
function(params) {

	this.isGalSearch = (this.contactSource && (this.contactSource == ZmId.SEARCH_GAL));
	this.isCalResSearch = (!this.contactSource && this.conds != null);
    if (appCtxt.isOffline && this.isCalResSearch) {
        this.isCalResSearch =  appCtxt.isZDOnline();
    }
	if (this.isEmpty) {
		this._handleResponseExecute(params.callback);
		return null;
	}

	var soapDoc;
	if (!this.response) {
		if (this.isGalSearch) {
			// DEPRECATED: Use JSON version
			soapDoc = AjxSoapDoc.create("SearchGalRequest", "urn:zimbraAccount");
			var method = soapDoc.getMethod();
			if (this.galType) {
				method.setAttribute("type", this.galType);
			}
			soapDoc.set("name", this.query);
			var searchFilterEl = soapDoc.set("searchFilter");
			if (this.conds && this.conds.length) {
				var condsEl = soapDoc.set("conds", null, searchFilterEl);
				this._applySoapCond(this.conds, soapDoc, condsEl);
			}
		} else if (this.isAutocompleteSearch) {
			soapDoc = AjxSoapDoc.create("AutoCompleteRequest", "urn:zimbraMail");
			var method = soapDoc.getMethod();
			if (this.limit) {
				method.setAttribute("limit", this.limit);
			}
			soapDoc.set("name", this.query);
		} else if (this.isGalAutocompleteSearch) {
			soapDoc = AjxSoapDoc.create("AutoCompleteGalRequest", "urn:zimbraAccount");
			var method = soapDoc.getMethod();
			method.setAttribute("limit", this._getLimit());
			if (this.galType) {
				method.setAttribute("type", this.galType);
			}
			soapDoc.set("name", this.query);
		} else if (this.isCalResSearch) {
			soapDoc = AjxSoapDoc.create("SearchCalendarResourcesRequest", "urn:zimbraAccount");
			var method = soapDoc.getMethod();
			if (this.attrs) {
				var attrs = [].concat(this.attrs);
				AjxUtil.arrayRemove(attrs, "fullName");
				method.setAttribute("attrs", attrs.join(","));
			}
			var searchFilterEl = soapDoc.set("searchFilter");
			if (this.conds && this.conds.length) {
				var condsEl = soapDoc.set("conds", null, searchFilterEl);
				this._applySoapCond(this.conds, soapDoc, condsEl);
			}
		} else {
			if (this.soapInfo) {
				soapDoc = AjxSoapDoc.create(this.soapInfo.method, this.soapInfo.namespace);
				// Pass along any extra soap data. (Voice searches use this to pass user identification.)
				for (var nodeName in this.soapInfo.additional) {
					var node = soapDoc.set(nodeName);
					var attrs = this.soapInfo.additional[nodeName];
					for (var attr in attrs) {
						node.setAttribute(attr, attrs[attr]);
					}
				}
			} else {
				soapDoc = AjxSoapDoc.create("SearchRequest", "urn:zimbraMail");
			}
			var method = this._getStandardMethod(soapDoc);
			if (this.types) {
				var a = this.types.getArray();
				if (a.length) {
					var typeStr = [];
					for (var i = 0; i < a.length; i++) {
						typeStr.push(ZmSearch.TYPE[a[i]]);
					}
					method.setAttribute("types", typeStr.join(","));
					if (this.types.contains(ZmItem.MSG) || this.types.contains(ZmItem.CONV)) {
						// special handling for showing participants ("To" instead of "From")
						var folder = this.folderId && appCtxt.getById(this.folderId);
						method.setAttribute("recip", (folder && folder.isOutbound()) ? "1" : "0");
					}
					if (this.types.contains(ZmItem.CONV)) {
						// get ID/folder for every msg in each conv result
						method.setAttribute("fullConversation", 1);
					}
					// if we're prefetching the first hit message, also mark it as read
					if (this.fetch) {

						method.setAttribute("fetch", ( this.fetch == "all" ) ? "all" : "1");
						// and set the html flag if we want the html version
						if (this.getHtml) {
							method.setAttribute("html", "1");
						}
					}
					if (this.markRead) {
						method.setAttribute("read", "1");
					}
				}
			}
			if (this.inDumpster) {
				method.setAttribute("inDumpster", "1");
			}
		}
	}

	var soapMethod = this._getStandardMethod(soapDoc);
	soapMethod.setAttribute("needExp", 1);

	var respCallback = this._handleResponseExecute.bind(this, params.callback);

	if (params.batchCmd) {
		params.batchCmd.addRequestParams(soapDoc, respCallback);
	} else {
		return appCtxt.getAppController().sendRequest({soapDoc:soapDoc, asyncMode:true, callback:respCallback,
													   errorCallback:params.errorCallback,
													   timeout:params.timeout, noBusyOverlay:params.noBusyOverlay,
													   response:this.response});
	}
};

/**
 * Creates a JSON request that represents this search and sends it to the server.
 *
 * @param {Hash}	params		a hash of parameters
 * @param {AjxCallback}	params.callback		the callback to run when response is received
 * @param {AjxCallback}	params.errorCallback	the callback to run if there is an exception
 * @param {ZmBatchCommand}	params.batchCmd		the batch command that contains this request
 * @param {int}	params.timeout		the timeout value (in seconds)
 * @param {Boolean}	params.noBusyOverlay	if <code>true</code>, don't use the busy overlay
 * 
 * @private
 */
ZmSearch.prototype._executeJson =
function(params) {

	this.isGalSearch = (this.contactSource && (this.contactSource == ZmId.SEARCH_GAL));
	this.isCalResSearch = (!this.contactSource && this.conds != null);
    if (appCtxt.isOffline && this.isCalResSearch) {
        this.isCalResSearch = appCtxt.isZDOnline();
    }
	if (this.isEmpty) {
		this._handleResponseExecute(params.callback);
		return null;
	}

	var jsonObj, request, soapDoc;
	if (!this.response) {
		if (this.isGalSearch) {
			request = {
				_jsns:"urn:zimbraAccount",
				needIsOwner: "1",
				needIsMember: "directOnly"
			};
			jsonObj = {SearchGalRequest: request};
			if (this.galType) {
				request.type = this.galType;
			}
			request.name = this.query;

			// bug #36188 - add offset/limit for paging support
			request.offset = this.offset = (this.offset || 0);
			request.limit = this._getLimit();

			// bug 15878: see same in ZmSearch.prototype._getStandardMethodJson
			request.locale = { _content: AjxEnv.DEFAULT_LOCALE };

			if (this.lastId) { // add lastSortVal and lastId for cursor-based paging
				request.cursor = {id:this.lastId, sortVal:(this.lastSortVal || "")};
			}
			if (this.sortBy) {
				request.sortBy = this.sortBy;
			}
			if (this.conds && this.conds.length) {
				request.searchFilter = {conds:{}};
				request.searchFilter.conds = ZmSearch.prototype._applyJsonCond(this.conds, request);
			}
		} else if (this.isAutocompleteSearch) {
			jsonObj = {AutoCompleteRequest:{_jsns:"urn:zimbraMail"}};
			request = jsonObj.AutoCompleteRequest;
			if (this.limit) {
				request.limit = this.limit;
			}
			request.name = {_content:this.query};
			if (params.autocompleteType) {
				request.t = params.autocompleteType;
			}
		} else if (this.isGalAutocompleteSearch) {
			jsonObj = {AutoCompleteGalRequest:{_jsns:"urn:zimbraAccount"}};
			request = jsonObj.AutoCompleteGalRequest;
			request.limit = this._getLimit();
			request.name = this.query;
			if (this.galType) {
				request.type = this.galType;
			}
		} else if (this.isCalResSearch) {
			jsonObj = {SearchCalendarResourcesRequest:{_jsns:"urn:zimbraAccount"}};
			request = jsonObj.SearchCalendarResourcesRequest;
			if (this.attrs) {
				var attrs = [].concat(this.attrs);
				request.attrs = attrs.join(",");
			}
            request.offset = this.offset = (this.offset || 0);
            request.limit = this._getLimit();
			if (this.conds && this.conds.length) {
				request.searchFilter = {conds:{}};
				request.searchFilter.conds = ZmSearch.prototype._applyJsonCond(this.conds, request);
			}
		} else {
			if (this.soapInfo) {
				soapDoc = AjxSoapDoc.create(this.soapInfo.method, this.soapInfo.namespace);
				// Pass along any extra soap data. (Voice searches use this to pass user identification.)
				for (var nodeName in this.soapInfo.additional) {
					var node = soapDoc.set(nodeName);
					var attrs = this.soapInfo.additional[nodeName];
					for (var attr in attrs) {
						node.setAttribute(attr, attrs[attr]);
					}
				}
			} else {
				jsonObj = {SearchRequest:{_jsns:"urn:zimbraMail"}};
				request = jsonObj.SearchRequest;
			}
			this._getStandardMethodJson(request);
			if (this.types) {
				var a = this.types.getArray();
				if (a.length) {
					var typeStr = [];
					for (var i = 0; i < a.length; i++) {
						typeStr.push(ZmSearch.TYPE[a[i]]);
					}
					request.types = typeStr.join(",");

					if (this.types.contains(ZmItem.MSG) || this.types.contains(ZmItem.CONV)) {
						// special handling for showing participants ("To" instead of "From")
						var folder = this.folderId && appCtxt.getById(this.folderId);
						request.recip = (folder && folder.isOutbound()) ? "2" : "0";
					}

					if (this.types.contains(ZmItem.CONV)) {
						// get ID/folder for every msg in each conv result
						request.fullConversation = 1;
					}

					// if we're prefetching the first hit message, also mark it as read
					if (this.fetch) {
                        request.fetch = ( this.fetch == "all" ) ? "all" : 1;
						// and set the html flag if we want the html version
						if (this.getHtml) {
							request.html = 1;
						}
					}

					if (this.markRead) {
						request.read = 1;
					}

                    if (this.headers) {
                        for (var hdr in this.headers) {
                            if (!request.header) { request.header = []; }
                            request.header.push({n: this.headers[hdr]});
                        }
                    }

					if (a.length == 1 && a[0] == ZmItem.TASK && this.allowableTaskStatus) {
						request.allowableTaskStatus = this.allowableTaskStatus;
					}
                }
            }
			if (this.inDumpster) {
				request.inDumpster = 1;
			}
        }
    }

	if (request) {
		request.needExp = 1;
	}


	var respCallback = this._handleResponseExecute.bind(this, params.callback);

	if (params.batchCmd) {
		params.batchCmd.addRequestParams(soapDoc, respCallback);
	} else {
		var searchParams = {
			jsonObj:jsonObj,
			soapDoc:soapDoc,
			asyncMode:true,
			callback:respCallback,
			errorCallback:params.errorCallback,
            offlineCallback:params.offlineCallback,
			timeout:params.timeout,
            offlineCache:params.offlineCache,
			noBusyOverlay:params.noBusyOverlay,
			response:this.response,
			accountName:this.accountName,
            offlineRequest:params.offlineRequest
		};
		return appCtxt.getAppController().sendRequest(searchParams);
	}
};

ZmSearch.prototype._applySoapCond =
function(inConds, soapDoc, condsEl, or) {
	if (or || this.join == ZmSearch.JOIN_OR) {
		condsEl.setAttribute("or", 1);
	}
	for (var i = 0; i < inConds.length; i++) {
		var c = inConds[i];
		if (AjxUtil.isArray(c)) {
			var subCondsEl = soapDoc.set("conds", null, condsEl);
			this._applySoapCond(c, soapDoc, subCondsEl, true);
		} else if (c.attr=="fullName" && c.op=="has") {
			var nameEl = soapDoc.set("name", c.value);
		} else {
			var condEl = soapDoc.set("cond", null, condsEl);
			condEl.setAttribute("attr", c.attr);
			condEl.setAttribute("op", c.op);
			condEl.setAttribute("value", c.value);
		}
	}
};

ZmSearch.prototype._applyJsonCond =
function(inConds, request, or) {
	var outConds = {};
	if (or || this.join == ZmSearch.JOIN_OR) {
		outConds.or = 1;
	}

	for (var i = 0; i < inConds.length; i++) {
		var c = inConds[i];
		if (AjxUtil.isArray(c)) {
			if (!outConds.conds)
				outConds.conds = [];
			outConds.conds.push(this._applyJsonCond(c, request, true));
		} else if (c.attr=="fullName" && c.op=="has") {
			request.name = {_content: c.value};
		} else {
			if (!outConds.cond)
				outConds.cond = [];
			outConds.cond.push({attr:c.attr, op:c.op, value:c.value});
		}
	}
	return outConds;
};

/**
 * Converts the response into a {ZmSearchResult} and passes it along.
 * 
 * @private
 */
ZmSearch.prototype._handleResponseExecute =
function(callback, result) {
	
	if (result) {
		var response = result.getResponse();
	
		if      (this.isGalSearch)				{ response = response.SearchGalResponse; }
		else if (this.isCalResSearch)			{ response = response.SearchCalendarResourcesResponse; }
		else if (this.isAutocompleteSearch)		{ response = response.AutoCompleteResponse; }
		else if (this.isGalAutocompleteSearch)	{ response = response.AutoCompleteGalResponse; }
		else if (this.soapInfo)					{ response = response[this.soapInfo.response]; }
		else									{ response = response.SearchResponse; }
	}
	else {
		response = { _jsns: "urn:zimbraMail", more: false };
	}
	var searchResult = new ZmSearchResult(this);
	searchResult.set(response);
	result = result || new ZmCsfeResult();
	result.set(searchResult);

	if (callback) {
		callback.run(result);
	}
};

/**
 * Fetches a conversation from the server.
 *
 * @param {Hash}		params				a hash of parameters:
 * @param {String}		params.cid			the conv ID
 * @param {AjxCallback}	params.callback		the callback to run with result
 * @param {String}		params.fetch		which msg bodies to load (see soap.txt)
 * @param {Boolean}		params.markRead		if <code>true</code>, mark msg read
 * @param {Boolean}		params.noTruncate	if <code>true</code>, do not limit size of msg
 * @param {boolean}		params.needExp		if not <code>false</code>, have server check if addresses are DLs
 */
ZmSearch.prototype.getConv =
function(params) {
	if ((!this.query && !this.queryHint) || !params.cid) { return; }

	var jsonObj = {SearchConvRequest:{_jsns:"urn:zimbraMail"}};
	var request = jsonObj.SearchConvRequest;
	this._getStandardMethodJson(request);
	request.cid = params.cid;
	if (params.fetch) {
		request.fetch = params.fetch;
		if (params.markRead) {
			request.read = 1;			// mark that msg read
		}
		if (this.getHtml) {
			request.html = 1;			// get it as HTML
		}
		if (params.needExp !== false) {
			request.needExp = 1;
		}
	}

	if (!params.noTruncate) {
		request.max = appCtxt.get(ZmSetting.MAX_MESSAGE_SIZE);
	}

	//get both TO and From
	request.recip =  "2";

	var searchParams = {
		jsonObj:		jsonObj,
		asyncMode:		true,
		callback:		this._handleResponseGetConv.bind(this, params.callback),
		accountName:	this.accountName
	};
	appCtxt.getAppController().sendRequest(searchParams);
};

/**
 * @private
 */
ZmSearch.prototype._handleResponseGetConv =
function(callback, result) {
	var response = result.getResponse().SearchConvResponse;
	var searchResult = new ZmSearchResult(this);
	searchResult.set(response, null, true);
	result.set(searchResult);

	if (callback) {
		callback.run(result);
	}
};

/**
 * Clears cursor-related fields from this search so that it will not create a cursor.
 */
ZmSearch.prototype.clearCursor =
function() {
	this.lastId = this.lastSortVal = this.endSortVal = null;
};

/**
 * Gets a title that summarizes this search.
 * 
 * @return	{String}	the title
 */
ZmSearch.prototype.getTitle =
function() {
	var where;
	var pq = this.parsedQuery;
	// if this is a saved search, show its name, otherwise show folder or tag name if it's the only term
	var orgId = this.searchId || ((pq && (pq.getNumTokens() == 1)) ? this.folderId || this.tagId : null);
	if (orgId) {
		var org = appCtxt.getById(ZmOrganizer.getSystemId(orgId));
		if (org) {
			where = org.getName(true, ZmOrganizer.MAX_DISPLAY_NAME_LENGTH, true);
		}
	}
	return where ? ([ZmMsg.zimbraTitle, where].join(": ")) : ([ZmMsg.zimbraTitle, ZmMsg.searchResults].join(": "));
};

/**
 * Checks if this search is multi-account.
 * 
 * @return	{Boolean}	<code>true</code> if multi-account
 */
ZmSearch.prototype.isMultiAccount =
function() {
	if (!this._isMultiAccount) {
		this._isMultiAccount = (this.queryHint && this.queryHint.length > 0 &&
								(this.queryHint.split("inid:").length > 1 ||
								 this.queryHint.split("underid:").length > 1));
	}
	return this._isMultiAccount;
};

/**
 * @private
 */
ZmSearch.prototype._getStandardMethod =
function(soapDoc) {

	var method = soapDoc.getMethod();

	if (this.sortBy) {
		method.setAttribute("sortBy", this.sortBy);
	}

	if (this.types.contains(ZmItem.MSG) || this.types.contains(ZmItem.CONV)) {
		ZmMailMsg.addRequestHeaders(soapDoc);
	}

	// bug 5771: add timezone and locale info
	ZmTimezone.set(soapDoc, AjxTimezone.DEFAULT, null);
	soapDoc.set("locale", appCtxt.get(ZmSetting.LOCALE_NAME), null);

	if (this.lastId != null && this.lastSortVal) {
		// cursor is used for paginated searches
		var cursor = soapDoc.set("cursor");
		cursor.setAttribute("id", this.lastId);
		cursor.setAttribute("sortVal", this.lastSortVal);
		if (this.endSortVal) {
			cursor.setAttribute("endSortVal", this.endSortVal);
		}
	}

	this.offset = this.offset || 0;
	method.setAttribute("offset", this.offset);

	// always set limit
	method.setAttribute("limit", this._getLimit());

	var query = this._getQuery();

	soapDoc.set("query", query);

	// set search field if provided
	if (this.field) {
		method.setAttribute("field", this.field);
	}

	return method;
};

/**
 * @private
 */
ZmSearch.prototype._getStandardMethodJson = 
function(req) {

	if (this.sortBy) {
		req.sortBy = this.sortBy;
	}

	if (this.types.contains(ZmItem.MSG) || this.types.contains(ZmItem.CONV)) {
		ZmMailMsg.addRequestHeaders(req);
	}

	// bug 5771: add timezone and locale info
	ZmTimezone.set(req, AjxTimezone.DEFAULT);
	// bug 15878: We can't use appCtxt.get(ZmSetting.LOCALE) because that
	//            will return the server's default locale if it is not set
	//            set for the user or their COS. But AjxEnv.DEFAULT_LOCALE
	//            is set to the browser's locale setting in the case when
	//            the user's (or their COS) locale is not set.
	req.locale = { _content: AjxEnv.DEFAULT_LOCALE };

	if (this.lastId != null && this.lastSortVal) {
		// cursor is used for paginated searches
		req.cursor = {id:this.lastId, sortVal:this.lastSortVal};
		if (this.endSortVal) {
			req.cursor.endSortVal = this.endSortVal;
		}
	}

	req.offset = this.offset = this.offset || 0;

	// always set limit
	req.limit = this._getLimit();

	if (this.idsOnly) {
		req.resultMode = "IDS";
	}

	req.query = this._getQuery();

	// set search field if provided
	if (this.field) {
		req.field = this.field;
	}
};

/**
 * @private
 */
ZmSearch.prototype._getQuery =
function() {
	// and of course, always set the query and append the query hint if applicable
	// only use query hint if this is not a "simple" search
	if (this.queryHint) {
		var query = this.query ? ["(", this.query, ") "].join("") : "";
		return [query, "(", this.queryHint, ")"].join("");
	}
	return this.query;
};

/**
 * @private
 */
ZmSearch.prototype._getLimit =
function() {

	if (this.limit) { return this.limit; }

	var limit;
	if (this.isGalAutocompleteSearch) {
		limit = appCtxt.get(ZmSetting.AUTOCOMPLETE_LIMIT);
	} else {
		var type = this.types && this.types.get(0);
		var app = appCtxt.getApp(ZmItem.APP[type]) || appCtxt.getCurrentApp();
		if (app && app.getLimit) {
			limit = app.getLimit(this.offset);
		} else {
			limit = appCtxt.get(ZmSetting.PAGE_SIZE) || ZmSearch.DEFAULT_LIMIT;
		}
	}

	this.limit = limit;
	return limit;
};

/**
 * Tests the given item against a matching function generated from the query.
 * 
 * @param {ZmItem}	item		an item
 * @return	true if the item matches, false if it doesn't, and null if a matching function could not be generated
 */
ZmSearch.prototype.matches =
function(item) {

	if (!this.parsedQuery) {
		return null;
	}

	// if search is constrained to a folder, we can return false if item is not in that folder
	if (this.folderId && !this.parsedQuery.hasOrTerm) {
		if (item.type === ZmItem.CONV) {
			if (item.folders && !item.folders[this.folderId]) {
				return false;
			}
		}
		else if (item.folderId && item.folderId !== this.folderId) {
			return false;
		}
	}

	var matchFunc = this.parsedQuery.getMatchFunction();
	return matchFunc ? matchFunc(item) : null;
};

/**
 * Returns true if the query has a folder-related term with the given value.
 * 
 * @param 	{string}	path		a folder path (optional)
 */
ZmSearch.prototype.hasFolderTerm =
function(path) {
	return this.parsedQuery && this.parsedQuery.hasTerm(["in", "under"], path);
};

/**
 * Replaces the old folder path with the new folder path in the query string, if found.
 * 
 * @param	{string}	oldPath		the old folder path
 * @param	{string}	newPath		the new folder path
 * 
 * @return	{boolean}	true if replacement was performed
 */
ZmSearch.prototype.replaceFolderTerm =
function(oldPath, newPath) {
	if (!this.parsedQuery) {
		return this.query;
	}
	var newQuery = this.parsedQuery.replaceTerm(["in", "under"], oldPath, newPath);
	if (newQuery) {
		this.query = newQuery;
	}
	return Boolean(newQuery);
};

/**
 * Returns true if the query has a tag term with the given value.
 * 
 * @param 	{string}	tagName		a tag name (optional)
 */
ZmSearch.prototype.hasTagTerm =
function(tagName) {
	return this.parsedQuery && this.parsedQuery.hasTerm("tag", tagName);
};

/**
 * Replaces the old tag name with the new tag name in the query string, if found.
 * 
 * @param	{string}	oldName		the old tag name
 * @param	{string}	newName		the new tag name
 * 
 * @return	{boolean}	true if replacement was performed
 */
ZmSearch.prototype.replaceTagTerm =
function(oldName, newName) {
	if (!this.parsedQuery) {
		return this.query;
	}
	var newQuery = this.parsedQuery.replaceTerm("tag", oldName, newName);
	if (newQuery) {
		this.query = newQuery;
	}
	return Boolean(newQuery);
};

/**
 * Returns true if the query has a term related to unread status.
 */
ZmSearch.prototype.hasUnreadTerm =
function() {
	return (this.parsedQuery && (this.parsedQuery.hasTerm("is", "read") ||
								 this.parsedQuery.hasTerm("is", "unread")));
};

/**
 * Returns true if the query has the term "is:anywhere".
 */
ZmSearch.prototype.isAnywhere =
function() {
	return (this.parsedQuery && this.parsedQuery.hasTerm("is", "anywhere"));
};

/**
 * Returns true if the query has a "content" term.
 */
ZmSearch.prototype.hasContentTerm =
function() {
	return (this.parsedQuery && this.parsedQuery.hasTerm("content"));
};

/**
 * Returns true if the query has just one term, and it's a folder or tag term.
 */
ZmSearch.prototype.isSimple =
function() {
	var pq = this.parsedQuery;
	if (pq && (pq.getNumTokens() == 1)) {
		return pq.hasTerm(["in", "inid", "tag"]);
	}
	return false;
};

ZmSearch.prototype.getTokens =
function() {
	return this.parsedQuery && this.parsedQuery.getTokens();
};

ZmSearch.prototype._setProperties =
function() {
	var props = this.parsedQuery && this.parsedQuery.getProperties();
	for (var key in props) {
		this[key] = props[key];
	}
};





/**
 * This class is a parsed representation of a query string. It parses the string into tokens.
 * A token is a paren, a conditional operator, or a search term (which has an operator and an
 * argument). The query string is assumed to be valid.
 * 
 * Compound terms such as "in:(inbox or sent)" will be exploded into multiple terms.
 * 
 * @param	{string}	query		a query string
 * 
 * TODO: handle "field[lastName]" and "#lastName"
 */
ZmParsedQuery = function(query) {

	this.hasOrTerm = false;
	this._tokens = this._parse(AjxStringUtil.trim(query, true));

	// preconditions for flags
	if (!ZmParsedQuery.IS_VALUE_PRECONDITION) {
		ZmParsedQuery.IS_VALUE_PRECONDITION = {};
		ZmParsedQuery.IS_VALUE_PRECONDITION['flagged']      = ZmSetting.FLAGGING_ENABLED;
		ZmParsedQuery.IS_VALUE_PRECONDITION['unflagged']    = ZmSetting.FLAGGING_ENABLED;
	}
};

ZmParsedQuery.prototype.isZmParsedQuery = true;
ZmParsedQuery.prototype.toString = function() { return "ZmParsedQuery"; };

ZmParsedQuery.TERM	= "TERM";	// search operator such as "in"
ZmParsedQuery.COND	= "COND";	// AND OR NOT
ZmParsedQuery.GROUP	= "GROUP";	// ( or )

ZmParsedQuery.OP_CONTENT	= "content";

ZmParsedQuery.OP_LIST = [
	"content", "subject", "msgid", "envto", "envfrom", "contact", "to", "from", "cc", "tofrom", 
	"tocc", "fromcc", "tofromcc", "in", "under", "inid", "underid", "has", "filename", "type", 
	"attachment", "is", "date", "mdate", "day", "week", "month", "year", "after", "before", 
	"size", "bigger", "larger", "smaller", "tag", "priority", "message", "my", "modseq", "conv", 
	"conv-count", "conv-minm", "conv-maxm", "conv-start", "conv-end", "appt-start", "appt-end", "author", "title", "keywords", 
	"company", "metadata", "item", "sort"
];
ZmParsedQuery.IS_OP		= AjxUtil.arrayAsHash(ZmParsedQuery.OP_LIST);

// valid arguments for the search term "is:"
ZmParsedQuery.IS_VALUES = [	"unread", "read", "flagged", "unflagged",
							"draft", "sent", "received", "replied", "unreplied", "forwarded", "unforwarded",
							"invite",
							"solo",
							"tome", "fromme", "ccme", "tofromme", "toccme", "fromccme", "tofromccme",
							"local", "remote", "anywhere" ];

// ops that can appear more than once in a query
ZmParsedQuery.MULTIPLE = {};
ZmParsedQuery.MULTIPLE["to"]			= true;
ZmParsedQuery.MULTIPLE["is"]			= true;
ZmParsedQuery.MULTIPLE["has"]			= true;
ZmParsedQuery.MULTIPLE["tag"]			= true;
ZmParsedQuery.MULTIPLE["appt-start"]	= true;
ZmParsedQuery.MULTIPLE["appt-end"]		= true;
ZmParsedQuery.MULTIPLE["type"]			= true;

ZmParsedQuery.isMultiple =
function(term) {
	return Boolean(term && ZmParsedQuery.MULTIPLE[term.op]);
};

// ops that are mutually exclusive
ZmParsedQuery.EXCLUDE = {};
ZmParsedQuery.EXCLUDE["before"]	= ["date"];
ZmParsedQuery.EXCLUDE["after"]	= ["date"];

// values that are mutually exclusive - list value implies full multi-way exclusivity
ZmParsedQuery.EXCLUDE["is"]					= {};
ZmParsedQuery.EXCLUDE["is"]["read"]			= ["unread"];
ZmParsedQuery.EXCLUDE["is"]["flagged"]		= ["unflagged"];
ZmParsedQuery.EXCLUDE["is"]["sent"]			= ["received"];
ZmParsedQuery.EXCLUDE["is"]["replied"]		= ["unreplied"];
ZmParsedQuery.EXCLUDE["is"]["forwarded"]	= ["unforwarded"];
ZmParsedQuery.EXCLUDE["is"]["local"]		= ["remote", "anywhere"];
ZmParsedQuery.EXCLUDE["is"]["tome"]			= ["tofromme", "toccme", "tofromccme"];
ZmParsedQuery.EXCLUDE["is"]["fromme"]		= ["tofromme", "fromccme", "tofromccme"];
ZmParsedQuery.EXCLUDE["is"]["ccme"]			= ["toccme", "fromccme", "tofromccme"];

ZmParsedQuery._createExcludeMap =
function(excludes) {

	var excludeMap = {};
	for (var key in excludes) {
		var value = excludes[key];
		if (AjxUtil.isArray1(value)) {
			value.push(key);
			ZmParsedQuery._permuteExcludeMap(excludeMap, value);
		}
		else {
			for (var key1 in value) {
				var value1 = excludes[key][key1];
				value1.push(key1);
				ZmParsedQuery._permuteExcludeMap(excludeMap, AjxUtil.map(value1,
						function(val) {
							return new ZmSearchToken(key, val).toString();
						}));
			}
		}
	}
	return excludeMap;
};

// makes each possible pair in the list exclusive
ZmParsedQuery._permuteExcludeMap =
function(excludeMap, list) {
	if (list.length < 2) { return; }
	for (var i = 0; i < list.length - 1; i++) {
		var a = list[i];
		for (var j = i + 1; j < list.length; j++) {
			var b = list[j];
			excludeMap[a] = excludeMap[a] || {};
			excludeMap[b] = excludeMap[b] || {};
			excludeMap[a][b] = true;
			excludeMap[b][a] = true;
		}
	}
};

/**
 * Returns true if the given search terms should not appear in the same query.
 * 
 * @param {ZmSearchToken}	termA	search term
 * @param {ZmSearchToken}	termB	search term
 */
ZmParsedQuery.areExclusive =
function(termA, termB) {
	if (!termA || !termB) { return false; }
	var map = ZmParsedQuery.EXCLUDE_MAP;
	if (!map) {
		map = ZmParsedQuery.EXCLUDE_MAP = ZmParsedQuery._createExcludeMap(ZmParsedQuery.EXCLUDE);
	}
	var opA = termA.op, opB = termB.op;
	var strA = termA.toString(), strB = termB.toString();
	return Boolean((map[opA] && map[opA][opB]) || (map[opB] && map[opB][opA]) ||
				   (map[strA] && map[strA][strB]) || (map[strB] && map[strB][strA]));
};

// conditional ops
ZmParsedQuery.COND_AND		= "and"
ZmParsedQuery.COND_OR		= "or";
ZmParsedQuery.COND_NOT		= "not";
ZmParsedQuery.GROUP_OPEN	= "(";
ZmParsedQuery.GROUP_CLOSE	= ")";

// JS version of conditional
ZmParsedQuery.COND_OP = {};
ZmParsedQuery.COND_OP[ZmParsedQuery.COND_AND]	= " && ";
ZmParsedQuery.COND_OP[ZmParsedQuery.COND_OR]	= " || ";
ZmParsedQuery.COND_OP[ZmParsedQuery.COND_NOT]	= " !";

// word separators
ZmParsedQuery.EOW_LIST	= [" ", ":", ZmParsedQuery.GROUP_OPEN, ZmParsedQuery.GROUP_CLOSE];
ZmParsedQuery.IS_EOW	= AjxUtil.arrayAsHash(ZmParsedQuery.EOW_LIST);

// map is:xxx to item properties
ZmParsedQuery.FLAG = {};
ZmParsedQuery.FLAG["unread"]		= "item.isUnread";
ZmParsedQuery.FLAG["read"]			= "!item.isUnread";
ZmParsedQuery.FLAG["flagged"]		= "item.isFlagged";
ZmParsedQuery.FLAG["unflagged"]		= "!item.isFlagged";
ZmParsedQuery.FLAG["forwarded"]		= "item.isForwarded";
ZmParsedQuery.FLAG["unforwarded"]	= "!item.isForwarded";
ZmParsedQuery.FLAG["sent"]			= "item.isSent";
ZmParsedQuery.FLAG["draft"]			= "item.isDraft";
ZmParsedQuery.FLAG["replied"]		= "item.isReplied";
ZmParsedQuery.FLAG["unreplied"]		= "!item.isReplied";

ZmParsedQuery.prototype._parse =
function(query) {

	function getQuotedStr(str, pos, q) {
		var q = q || str.charAt(pos);
		pos++;
		var done = false, ch, quoted = "";
		while (pos < str.length && !done) {
			ch = str.charAt(pos);
			if (ch == q) {
				done = true;
			} else {
				quoted += ch;
				pos++;
			}
		}

		return done ? {str:quoted, pos:pos + 1} : null;
	}
	
	function skipSpace(str, pos) {
		while (pos < str.length && str.charAt(pos) == " ") {
			pos++;
		}
		return pos;
	}
	
	function fail(reason, query) {
		DBG.println(AjxDebug.DBG1, "ZmParsedQuery failure: " + reason + "; query: [" + query + "]");
		this.parseFailed = reason;
		return tokens;		
	}

	var len = query.length;
	var tokens = [], ch, lastCh, op, word = "", isEow = false, endOk = true, compound = 0, numParens = 0;
	var pos = skipSpace(query, 0);
	while (pos < len) {
		lastCh = (ch != " ") ? ch : lastCh;
		ch = query.charAt(pos);
		isEow = ZmParsedQuery.IS_EOW[ch];

		if (ch == ":") {
			if (ZmParsedQuery.IS_OP[word]) {
				op = word;
			} else {
				return fail("unrecognized op '" + word + "'", query);
			}
			word = "";
			pos = skipSpace(query, pos + 1);
			continue;
		}

		if (isEow) {
			var lcWord = word.toLowerCase();
			var isCondOp = !!ZmParsedQuery.COND_OP[lcWord];
			if (op && word && !(isCondOp && compound > 0)) {
				tokens.push(new ZmSearchToken(op, lcWord));
				if (compound == 0) {
					op = "";
				}
				word = "";
				endOk = true;
			} else if (!op || (op && compound > 0)) {
				if (isCondOp) {
					tokens.push(new ZmSearchToken(lcWord));
					endOk = false;
					if (lcWord == ZmParsedQuery.COND_OR) {
						this.hasOrTerm = true;
					}
				} else if (word) {
					tokens.push(new ZmSearchToken(ZmParsedQuery.OP_CONTENT, word));
				}
				word = "";
			}
		}

		if (ch == '"') {
			var results = getQuotedStr(query, pos);
			if (results) {
				word = results.str;
				pos = results.pos;
			} else {
				return fail("improper use of quotes", query);
			}
		} else if (ch == ZmParsedQuery.GROUP_OPEN) {
			var done = false;
			if (compound > 0) {
				compound++;
			}
			else if (lastCh == ":") {
				compound = 1;
				// see if parens are being used as secondary quoting mechanism by looking for and/or
				var inside = query.substr(pos, query.indexOf(ZmParsedQuery.GROUP_CLOSE, pos + 1));
				inside = inside && inside.toLowerCase();
				if (inside && (inside.indexOf(" " + ZmParsedQuery.COND_OR + " ") == -1) &&
							  (inside.indexOf(" " + ZmParsedQuery.COND_AND + " ") == -1)) {
					var results = getQuotedStr(query, pos, ZmParsedQuery.GROUP_CLOSE);
					if (results) {
						word = results.str;
						pos = results.pos;
						compound = 0;
					} else {
						return fail("improper use of paren-based quoting", query);
					}
					done = true;
				}
			}
			if (!done) {
				tokens.push(new ZmSearchToken(ch));
				numParens++;
			}
			pos = skipSpace(query, pos + 1);
		} else if (ch == ZmParsedQuery.GROUP_CLOSE) {
			if (compound > 0) {
				compound--;
			}
			if (compound == 0) {
				op = "";
			}
			tokens.push(new ZmSearchToken(ch));
			pos = skipSpace(query, pos + 1);
		} else if (ch == "-" && !word && !op) {
			tokens.push(new ZmSearchToken(ZmParsedQuery.COND_NOT));
			pos = skipSpace(query, pos + 1);
			endOk = false;
		} else {
			if (ch != " ") {
				word += ch;
			}
			pos++;
		}
	}

	// check for term at end
	if ((pos >= query.length) && op && word) {
		tokens.push(new ZmSearchToken(op, word));
		endOk = true;
	} else if (!op && word) {
		tokens.push(new ZmSearchToken(word));
	}
	
	// remove unnecessary enclosing parens from when a single compound term is expanded, for example when
	// "subject:(foo bar)" is expanded into "(subject:foo subject:bar)"
	if (tokens.length >= 3 && numParens == 1 && tokens[0].op == ZmParsedQuery.GROUP_OPEN &&
			tokens[tokens.length - 1].op == ZmParsedQuery.GROUP_CLOSE) {
		tokens.shift();
		tokens.pop();
	}

	if (!endOk) {
		return fail("unexpected end of query", query);
	}
	
	return tokens;
};

ZmParsedQuery.prototype.getTokens =
function() {
	return this._tokens;
};

ZmParsedQuery.prototype.getNumTokens =
function() {
	return this._tokens ? this._tokens.length : 0;
};

ZmParsedQuery.prototype.getProperties =
function() {
	
	var props = {};
	for (var i = 0, len = this._tokens.length; i < len; i++) {
		var t = this._tokens[i];
		if (t.type == ZmParsedQuery.TERM) {
			var prev = i > 0 ? this._tokens[i-1] : null;
			if (!((prev && prev.op == ZmParsedQuery.COND_NOT) || this.hasOrTerm)) {
				if ((t.op == "in" || t.op == "inid") ) {
					this.folderId = props.folderId = (t.op == "in") ? this._getFolderId(t.arg) : t.arg;
				} else if (t.op == "tag") {
					// TODO: make sure there's only one tag term?
					this.tagId = props.tagId = this._getTagId(t.arg, true);
				}
			}
		}
	}
	return props;
};

/**
 * Returns a function based on the parsed query. The function is passed an item (msg or conv) and returns
 * true if the item matches the search.
 * 
 * @return {Function}	the match function
 */
ZmParsedQuery.prototype.getMatchFunction =
function() {
	
	if (this._matchFunction) {
		return this._matchFunction;
	}
	if (this.parseFailed || this.hasTerm(ZmParsedQuery.OP_CONTENT)) {
		return null;
	}
	
	var folderId, tagId;
	var func = ["return Boolean("];
	for (var i = 0, len = this._tokens.length; i < len; i++) {
		var t = this._tokens[i];
		if (t.type === ZmParsedQuery.TERM) {
			if (t.op === "in" || t.op === "inid") {
				folderId = (t.op === "in") ? this._getFolderId(t.arg) : t.arg;
				if (folderId) {
					func.push("((item.type === ZmItem.CONV) ? item.folders && item.folders['" + folderId +"'] : item.folderId === '" + folderId + "')");
				}
			}
			else if (t.op === "tag") {
				tagId = this._getTagId(t.arg, true);
				if (tagId) {
					func.push("item.hasTag('" + t.arg + "')");
				}
			}
			else if (t.op === "is") {
				var test = ZmParsedQuery.FLAG[t.arg];
				if (test) {
					func.push(test);
				}
			}
			else if (t.op === 'has' && t.arg === 'attachment') {
				func.push("item.hasAttach");
			}
			else {
				// search had a term we don't know how to match
				return null;
			}
			var next = this._tokens[i + 1];
			if (next && (next.type == ZmParsedQuery.TERM || next == ZmParsedQuery.COND_OP[ZmParsedQuery.COND_NOT] || next == ZmParsedQuery.GROUP_CLOSE)) {
				func.push(ZmParsedQuery.COND_OP[ZmParsedQuery.COND_AND]);
			}
		}
		else if (t.type === ZmParsedQuery.COND) {
			func.push(ZmParsedQuery.COND_OP[t.op]);
		}
		else if (t.type === ZmParsedQuery.GROUP) {
			func.push(t.op);
		}
	}
	func.push(")");

	// the way multi-account searches are done, we set the queryHint *only* so
	// set the folderId if it exists for simple multi-account searches
	// TODO: multi-acct part seems wrong; search with many folders joined by OR would incorrectly set folderId to last folder
	var isMultiAccountSearch = (appCtxt.multiAccounts && this.isMultiAccount() && !this.query && this.queryHint);
	if (!this.hasOrTerm || isMultiAccountSearch) {
		this.folderId = folderId;
		this.tagId = tagId;
	}
	
	try {
		this._matchFunction = new Function("item", func.join(""));
	} catch(ex) {}
	
	return this._matchFunction;
};

/**
 * Returns a query string that should be logically equivalent to the original query.
 */
ZmParsedQuery.prototype.createQuery =
function() {
	var terms = [];
	for (var i = 0, len = this._tokens.length; i < len; i++) {
		terms.push(this._tokens[i].toString());
	}
	return terms.join(" ");
};

// Returns the fully-qualified ID for the given folder path.
ZmParsedQuery.prototype._getFolderId =
function(path) {
	// first check if it's a system folder (name in query string may not match actual name)
	var folderId = ZmFolder.QUERY_ID[path];

	var accountName = this.accountName;
	if (!accountName) {
		var active = appCtxt.getActiveAccount();
		accountName = active ? active.name : appCtxt.accountList.mainAccount;
	}

	// now check all folders by name
	if (!folderId) {
		var account = accountName && appCtxt.accountList.getAccountByName(accountName);
		var folders = appCtxt.getFolderTree(account);
		var folder = folders ? folders.getByPath(path, true) : null;
		if (folder) {
			folderId = folder.id;
		}
	}

	if (accountName) {
		folderId = ZmOrganizer.getSystemId(folderId, appCtxt.accountList.getAccountByName(accountName));
	}

	return folderId;
};

// Returns the ID for the given tag name.
ZmParsedQuery.prototype._getTagId =
function(name, normalized) {
	var tagTree = appCtxt.getTagTree();
	if (tagTree) {
		var tag = tagTree.getByName(name.toLowerCase());
		if (tag) {
			return normalized ? tag.nId : tag.id;
		}
	}
	return null;
};

/**
 * Gets the given term with the given argument. Case-insensitive. Returns the first term found.
 * 
 * @param	{array}		opList		list of ops 
 * @param	{string}	value		argument value (optional)
 * 
 * @return	{object}	a token object, or null
 */
ZmParsedQuery.prototype.getTerm =
function(opList, value) {
	var opHash = AjxUtil.arrayAsHash(opList);
	var lcValue = value && value.toLowerCase();
	for (var i = 0, len = this._tokens.length; i < len; i++) {
		var t = this._tokens[i];
		var lcArg = t.arg && t.arg.toLowerCase();
		if (t.type == ZmParsedQuery.TERM && opHash[t.op] && (!value || lcArg == lcValue)) {
			return t;
		}
	}
	return null;
};

/**
 * Returns true if the query contains the given term with the given argument. Case-insensitive.
 * 
 * @param	{array}		opList		list of ops 
 * @param	{string}	value		argument value (optional)
 * 
 * @return	{boolean}	true if the query contains the given term with the given argument
 */
ZmParsedQuery.prototype.hasTerm =
function(opList, value) {
	return Boolean(this.getTerm(opList, value));
};

/**
 * Replaces the argument within the query for the given ops, if found. Case-insensitive. Replaces
 * only the first match.
 * 
 * @param	{array}		opList		list of ops 
 * @param	{string}	oldValue	the old argument
 * @param	{string}	newValue	the new argument
 * 
 * @return	{string}	a new query string (if the old argument was found and replaced), or the empty string
 */
ZmParsedQuery.prototype.replaceTerm =
function(opList, oldValue, newValue) {
	var lcValue = oldValue && oldValue.toLowerCase();
	var opHash = AjxUtil.arrayAsHash(opList);
	if (oldValue && newValue) {
		for (var i = 0, len = this._tokens.length; i < len; i++) {
			var t = this._tokens[i];
			var lcArg = t.arg && t.arg.toLowerCase();
			if (t.type == ZmParsedQuery.TERM && opHash[t.op] && (lcArg == lcValue)) {
				t.arg = newValue;
				return this.createQuery();
			}
		}
	}
	return "";
};

/**
 * This class represents one unit of a search query. That may be a search term ("is:unread"),
 * and conditional operator (AND, OR, NOT), or a grouping operator (left or right paren).
 * 
 * @param {string}	op		operator
 * @param {string}	arg		argument part of search term
 */
ZmSearchToken = function(op, arg) {
	
	if (op && arguments.length == 1) {
		var parts = op.split(":");
		op = parts[0];
		arg = parts[1];
	}
	
	this.op = op;
	this.arg = arg;
	if (ZmParsedQuery.IS_OP[op] && arg) {
		this.type = ZmParsedQuery.TERM;
	}
	else if (op && ZmParsedQuery.COND_OP[op.toLowerCase()]) {
		this.type = ZmParsedQuery.COND;
		this.op = op.toLowerCase();
	}
	else if (op == ZmParsedQuery.GROUP_OPEN || op == ZmParsedQuery.GROUP_CLOSE) {
		this.type = ZmParsedQuery.GROUP;
	} else if (op) {
		this.type = ZmParsedQuery.TERM;
		this.op = ZmParsedQuery.OP_CONTENT;
		this.arg = op;
	}
};

ZmSearchToken.prototype.isZmSearchToken = true;

/**
 * Returns the string version of this token.
 * 
 * @param {boolean}		force		if true, return "and" instead of an empty string ("and" is implied)
 */
ZmSearchToken.prototype.toString =
function(force) {
	if (this.type == ZmParsedQuery.TERM) {
		var arg = this.arg;
		if (this.op == ZmParsedQuery.OP_CONTENT) {
			return /\W/.test(arg) ? '"' + arg.replace(/"/g, '\\"') + '"' : arg;
		}
		else {
			// quote arg if it has any spaces and is not already quoted
			arg = (arg && (arg.indexOf('"') !== 0) && arg.indexOf(" ") != -1) ? '"' + arg + '"' : arg;
			return [this.op, arg].join(":");
		}
	}
	else {
		return (!force && this.op == ZmParsedQuery.COND_AND) ? "" : this.op;
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmSearchResult")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the search result class.
 */

/**
 * Creates the search result
 * @class
 * This class represents a search result.
 * 
 * @param	{ZmSearch}	search		the search
 */
ZmSearchResult = function(search) {
	if (!search) { return; }
	this._results = {};
	this.search = search;
	this.type = search.searchFor;
};

ZmSearchResult.prototype.isZmSearchResult = true;
ZmSearchResult.prototype.toString = function() { return "ZmSearchResult"; };

/**
 * Gets the results.
 * 
 * @param	{constant}	type	the type
 * @return	{Array}	an array of results
 */
ZmSearchResult.prototype.getResults =
function(type) {

	type = type || this.type;
	if (!this._results) {
		// probably got an exception - return an empty list
		return this._getResultsList(type);
	} else if (this.search.idsOnly) {
		return this._results;
	} else {
		// if we don't have results for the requested type, the search was probably for the wrong type
		return this._results[type] ? this._results[type] : type && this._getResultsList(type);
	}
};

/**
 * Gets the attribute.
 * 
 * @param	{String}	name		the attribute name
 * @return	{Object}	the attribute
 */
ZmSearchResult.prototype.getAttribute = 
function(name) {
	return this._respEl ? this._respEl[name] : null;
};

/**
 * Sets the response.
 * 
 * @private
 */
ZmSearchResult.prototype.set =
function(respEl) {

	if (!this.search) { return; }

	this._respEl = respEl;

	// <match> objects are returned for autocomplete search, not items; let caller handle them
	if (this.search.isAutocompleteSearch) { return; }

	var foundType = {};
	var numTypes = 0;
	var currentType, defaultType;
	var isGalSearch = this.search.isGalSearch;
	
	var _st = new Date();
	var count = 0;
	if (isGalSearch || this.search.isCalResSearch) {
		// process JS eval result for SearchGalRequest
		currentType = defaultType = isGalSearch ? ZmItem.CONTACT : ZmItem.RESOURCE;
		var data = isGalSearch ? respEl.cn : respEl.calresource;
		if (data) {
			if (!this._results[currentType]) {
				// create list as needed - may invoke package load
				this._results[currentType] =  this._getResultsList(currentType);
			}
			for (var j = 0; j < data.length; j++) {
				this._results[currentType].addFromDom(data[j]);
			}

			// manually sort gal results since server won't do it for us :(
			if (isGalSearch) {
				this._results[currentType].getArray().sort(ZmSearchResult._sortGalResults)
			}
			count = data.length;
		}
	} else if (this.search.idsOnly) {
		this._results = respEl.hit || [];
		return;
	} else {
		// process JS eval result for SearchResponse
		var types = this.search.types.getArray();
		defaultType = types[0];

		// bug fix #44232 - resolve default type if none provided
		if (!defaultType) {
			var allTypes = AjxUtil.values(ZmList.NODE);
			for (var i = 0; i < allTypes.length; i++) {
				var t = allTypes[i];
				if (respEl[t]) {
					defaultType = ZmList.ITEM_TYPE[t];
					if (types && types.length == 0) {
						types = [defaultType];
					}
					break;
				}
			}
		}

		if (!defaultType) {
			var curApp = appCtxt.getCurrentAppName();
			var types = ZmApp.SEARCH_TYPES[curApp];
			defaultType = types && types.length && types[0];
		}

		for (var i = 0; i < types.length; i++) {
			var type = types[i];
			var data = respEl[ZmList.NODE[type]];

			// A Resource is a Contact. Futz with the types to deal with this.
			if (!data && (type == ZmItem.RESOURCE)) {
				data = respEl[ZmList.NODE[ZmItem.CONTACT]];
			}

			// do a bunch of sanity checks
			if (data && data.length) {
				count += data.length;
				if (!this._results[type]) {
					// create list as needed - may invoke package load
					this._results[type] = this._getResultsList(type);
				}
				for (var j = 0; j < data.length; j++) {
					var item = data[j];
					item._type = type;
					this._results[type].addFromDom(item);
				}

				if (!foundType[type]) {
					foundType[type] = true;
					numTypes++;
					currentType = type;
				}
			}
		}
	}
	if (!count && defaultType) {
		this._results[defaultType] = this._getResultsList(defaultType);
	}
	if ((isGalSearch || this.search.isGalAutocompleteSearch) && this._results[ZmItem.CONTACT]) {
		this._results[ZmItem.CONTACT].setIsGal(true);
	}
	if (this.search.isGalAutocompleteSearch) {
		this.isTokenized = (this._respEl.tokenizeKey != null);
	}
	
	var _en = new Date();
	DBG.println(AjxDebug.DBG1, "TOTAL PARSE TIME for " + count + " NODES: " + (_en.getTime() - _st.getTime()));

	currentType = currentType || defaultType;
	if (numTypes <= 1) {
		this.type = currentType;
	}

	return this.type;
};

/**
 * @private
 */
ZmSearchResult._sortGalResults =
function(a, b) {
	var af = a.getFileAs && a.getFileAs().toLowerCase();
	var bf = b.getFileAs && b.getFileAs().toLowerCase();
	return af < bf ? -1 : (af > bf ? 1 : 0);
};

ZmSearchResult.prototype._getResultsList =
function(type) {

	if (type && typeof(ZmItem.RESULTS_LIST[type]) === "function") {
		return ZmItem.RESULTS_LIST[type](this.search);
	} else {
		DBG.println(
			AjxDebug.DBG1,
			AjxMessageFormat.format(
				"!type || ZmItem.RESULTS_LIST[type] !== function. Active app: {0}, type: {1}, searchFor: {2}.",
				[appCtxt.getCurrentAppName(), type, this.search.searchFor]
			)
		);
		return new ZmList(type, this.search);
	}

};

}
if (AjxPackage.define("zimbraMail.share.model.ZmTag")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines the tag class.
 */

/**
 * Creates a tag
 * @class
 * This class represents a tag.
 * 
 * @param	{Hash}	params		a hash of parameters
 * @extends	ZmOrganizer
 */
ZmTag = function(params) {
	params.type = ZmOrganizer.TAG;
	ZmOrganizer.call(this, params);
	this.notLocal = params.notLocal;
};

ZmTag.prototype = new ZmOrganizer;
ZmTag.prototype.constructor = ZmTag;
ZmTag.prototype.isZmTag = true;

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmTag.prototype.toString = 
function() {
	return "ZmTag";
};

// color icons
ZmTag.COLOR_ICON = new Object();
ZmTag.COLOR_ICON[ZmOrganizer.C_ORANGE]	= "TagOrange";
ZmTag.COLOR_ICON[ZmOrganizer.C_BLUE]	= "TagBlue";
ZmTag.COLOR_ICON[ZmOrganizer.C_CYAN]	= "TagCyan";
ZmTag.COLOR_ICON[ZmOrganizer.C_GREEN]	= "TagGreen";
ZmTag.COLOR_ICON[ZmOrganizer.C_PURPLE]	= "TagPurple";
ZmTag.COLOR_ICON[ZmOrganizer.C_RED]		= "TagRed";
ZmTag.COLOR_ICON[ZmOrganizer.C_YELLOW]	= "TagYellow";


// system tags
ZmTag.ID_ROOT = ZmOrganizer.ID_ROOT;
ZmTag.ID_UNREAD		= 32;
ZmTag.ID_FLAGGED	= 33;
ZmTag.ID_FROM_ME	= 34;
ZmTag.ID_REPLIED	= 35;
ZmTag.ID_FORWARDED	= 36;
ZmTag.ID_ATTACHED	= 37;

/**
 * Tags come from back end as a flat list, and we manually create a root tag, so all tags
 * have the root as parent. If tags ever have a tree structure, then this should do what
 * ZmFolder does (recursively create children).
 * 
 * @private
 */
ZmTag.createFromJs =
function(parent, obj, tree, sorted, account) {
	var tag;
	var nId = ZmOrganizer.normalizeId(obj.id);
	if (nId < ZmOrganizer.FIRST_USER_ID[ZmOrganizer.TAG]) { return; }
	tag = tree.getById(obj.id);
	if (tag) { return tag; }

	var params = {
		id: obj.id,
		name: obj.name,
		color: ZmTag.checkColor(obj.color),
		rgb: obj.rgb,
		parent: parent,
		tree: tree,
		numUnread: obj.u,
		account: account
	};
	tag = new ZmTag(params);
	var index = sorted ? ZmOrganizer.getSortIndex(tag, ZmTag.sortCompare) : null;
	parent.children.add(tag, index);

	var tagNameMap = parent.getTagNameMap();
	tagNameMap[obj.name] = tag;

	return tag;
};

ZmTag.createNotLocalTag =
function(name) {
	//cache so we don't create many objects in case many items are tagged by non local tags.
	var cache = ZmTag.notLocalCache = ZmTag.notLocalCache || [];
	var tag = cache[name];
	if (tag) {
		return tag;
	}
	tag = new ZmTag({notLocal: true, id: "notLocal_" + name, name: name});
	cache[name] = tag;
	return tag;
};

/**
 * Compares the tags by name.
 * 
 * @param	{ZmTag}	tagA		the first tag
 * @param	{ZmTag}	tagB		the second tag
 * @return	{int}	0 if the tag names match (case-insensitive); 1 if "a" is before "b"; -1 if "b" is before "a"
 */
ZmTag.sortCompare = 
function(tagA, tagB) {
	var check = ZmOrganizer.checkSortArgs(tagA, tagB);
	if (check != null) return check;

	if (tagA.name.toLowerCase() > tagB.name.toLowerCase()) return 1;
	if (tagA.name.toLowerCase() < tagB.name.toLowerCase()) return -1;
	return 0;
};

/**
 * Checks the tag name.
 * 
 * @param	{String}	name		the name
 * @return	{String}	<code>null</code> if the name is valid or a error message
 */
ZmTag.checkName =
function(name) {
	var msg = ZmOrganizer.checkName(name);
	if (msg) { return msg; }

	if (name.indexOf('\\') == 0) {
		return AjxMessageFormat.format(ZmMsg.errorInvalidName, AjxStringUtil.htmlEncode(name));
	}

	return null;
};

/**
 * Checks the color.
 * 
 * @param	{String}	color	the color
 * @return	{Number}	the valid color
 */
ZmTag.checkColor =
function(color) {
	color = Number(color);
	return ((color != null) && (color >= 0 && color <= ZmOrganizer.MAX_COLOR)) ? color : ZmOrganizer.DEFAULT_COLOR[ZmOrganizer.TAG];
};

ZmTag.getIcon = function(color) {
    var object = { getIcon:ZmTag.prototype.getIcon, getColor:ZmTag.prototype.getColor, color:color };
    if (String(color).match(/^#/)) {
        object.rgb = color;
        object.color = null;
    }
    return ZmTag.prototype.getIconWithColor.call(object);
}

/**
 * Creates a tag.
 * 
 * @param	{Hash}	params	a hash of parameters
 */
ZmTag.create =
function(params) {
	var request = {_jsns: "urn:zimbraMail"};
	var jsonObj = {CreateTagRequest: request};
	request.tag = {name: params.name}

    if (params.rgb) {
        request.tag.rgb = params.rgb;
    }
    else {
        request.tag.color = ZmOrganizer.checkColor(params.color) || ZmOrganizer.DEFAULT_COLOR[ZmOrganizer.TAG];
    }
	var errorCallback = new AjxCallback(null, ZmTag._handleErrorCreate, params);
	appCtxt.getAppController().sendRequest({
			jsonObj: jsonObj,
			asyncMode: true,
			errorCallback: errorCallback,
			accountName: params.accountName
	});
};

/**
 * @private
 */
ZmTag._handleErrorCreate =
function(params, ex) {
	if (ex.code == ZmCsfeException.MAIL_INVALID_NAME) {
		var msg = AjxMessageFormat.format(ZmMsg.errorInvalidName, AjxStringUtil.htmlEncode(params.name));
		var msgDialog = appCtxt.getMsgDialog();
		msgDialog.setMessage(msg, DwtMessageDialog.CRITICAL_STYLE);
		msgDialog.popup();
		return true;
	}
	return false;
};

/**
 * Gets the icon.
 * 
 * @return	{String}	the icon or <code>null</code> for no icon
 */
ZmTag.prototype.getIcon = 
function() {
	if (this.notLocal) {
		return "TagShared";
	}
	
	return (this.id == ZmOrganizer.ID_ROOT) ? null : "Tag";
};

/**
 * map from tag names to tags. used by getByNameOrRemote
 */
ZmTag.prototype.getTagNameMap =
function() {
	if (!this.tagNameMap) {
		this.tagNameMap = {};
	}
	return this.tagNameMap;
};

/**
 * Creates a query for this tag.
 * 
 * @return	{String}	the tag query
 */
ZmTag.prototype.createQuery =
function() {
	return ['tag:"', this.name, '"'].join("");
};

/**
 * Gets the tool tip.
 * 
 * @return	{String}	the tool tip
 */
ZmTag.prototype.getToolTip = function() {};

/**
 * @private
 */
ZmTag.prototype.notifyCreate =
function(obj) {
	var child = ZmTag.createFromJs(this, obj, this.tree, true);
	child._notify(ZmEvent.E_CREATE);
};


ZmTag.prototype.notifyModify =
function(obj) {
	if (obj.name) {
		//this is a rename - update the tagNameMap
		var oldName = this.name;
		var nameMap = this.parent.getTagNameMap();
		delete nameMap[oldName];
		nameMap[obj.name] = this;
		//we don't change the name on this ZmTag object here, it is done in ZmOrganizer.prototype.notifyModify
	}
	ZmOrganizer.prototype.notifyModify.call(this, obj);
};


ZmTag.prototype.notifyDelete =
function() {
	var nameMap = this.parent.getTagNameMap();
	delete nameMap[this.name];  //remove from name map
	
	ZmOrganizer.prototype.notifyDelete.call(this);
};

/**
 * Checks if the tag supports sharing.
 * 
 * @return	{Boolean}	always returns <code>false</code>. Tags cannot be shared.
 */
ZmTag.prototype.supportsSharing =
function() {
	// tags cannot be shared
	return false;
};

ZmTag.prototype.getByNameOrRemote =
function(name) {
	var tagNameMap = this.getTagNameMap();
	var tag = tagNameMap[name];
	if (tag) {
		return tag;
	}
	return ZmTag.createNotLocalTag(name);
};



}
if (AjxPackage.define("zimbraMail.share.model.ZmTree")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the tree class
 */

/**
 * Creates the tree
 * @class
 * This class represents a tree.
 * 
 * @param	{constant}	type		the type
 * @extends	ZmModel
 */
ZmTree = function(type) {

	if (arguments.length == 0) { return; }
	ZmModel.call(this, type);

	this.type = type;
	this.root = null;
};

ZmTree.prototype = new ZmModel;
ZmTree.prototype.constructor = ZmTree;

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmTree.prototype.toString = 
function() {
	return "ZmTree";
};

/**
 * Gets this tree as a string.
 * 
 * @return	{String}	the tree
 */
ZmTree.prototype.asString = 
function() {
	return this.root ? this._asString(this.root, "") : "";
};

/**
 * Gets the item by id.
 * 
 * @param	{String}	id		the id
 * @return	{Object}	the item
 */
ZmTree.prototype.getById =
function(id) {
	return this.root ? this.root.getById(id) : null;
};

/**
 * Gets the item by name.
 * 
 * @param	{String}	name		the name
 * @return	{Object}	the item
 */
ZmTree.prototype.getByName =
function(name) {
	return this.root ? this.root.getByName(name) : null;
};

/**
 * Gets the item type by name.
 *
 * @param	{String}	name		the name
 * @return	{String}    type of folder
 */
//Bug:47848: new method that returns type of the item given its name
ZmTree.prototype.getFolderTypeByName =
function(name){

    //As folder names are case-insensitive
    var formattedName = name.toLowerCase();

    //Iterate through folders of loaded apps
    var folderList = appCtxt.getTree(ZmOrganizer.FOLDER).asList();
    var type;
    var i;
    for(i=0 ; i < folderList.length ; i ++){
        var currentName = folderList[i].name;
        if(formattedName == currentName.toLowerCase()){
            return folderList[i].type;
        }
    }

    // check for _deferredFolders in the apps that have not been loaded
    var apps = ZmApp.APPS;

    for(i=0 ; i<apps.length; i++){
       var currentApp = appCtxt.getApp(apps[i]);
       var deferredFolders = currentApp && currentApp._deferredFolders;
       if(!deferredFolders){
           continue;
       }
       var j;
       for(j=0 ; j < deferredFolders.length ; j++){
           var currentFolder = deferredFolders[j];
           var currentName = currentFolder.obj && currentFolder.obj.name;
            if(formattedName == currentName.toLowerCase()){
                return currentFolder.type;
            }
       }

    }
    // if still not found return type as "Folder"
    type = ZmOrganizer.FOLDER;
    return type;
}

/**
 * Gets the item by type.
 * 
 * @param	{String}	name		the type name
 * @return	{Object}	the item
 */
ZmTree.prototype.getByType =
function(name) {
	return this.root ? this.root.getByType(name) : null;
};

/**
 * Gets the size of the tree.
 * 
 * @return	{int}	the size
 */
ZmTree.prototype.size =
function() {
	return this.root ? this.root.size() : 0;
};

/**
 * Resets the tree.
 */
ZmTree.prototype.reset =
function() {
	this.root = null;
};

/**
 * Gets the tree as a list.
 * 
 * @return	{Array}	an array
 */
ZmTree.prototype.asList =
function(options) {
	var list = [];
	return this.root ? this._addToList(this.root, list, options) : list;
};

/**
 * Gets the unread hash.
 * 
 * @param	{Hash}	unread		the unread hash
 * @return	{Hash} the unread tree as a hash
 */
ZmTree.prototype.getUnreadHash =
function(unread) {
	if (!unread) {
		unread = {};
	}
	return this.root ? this._getUnreadHash(this.root, unread) : unread;
};

/**
 * @private
 */
ZmTree.prototype._addToList =
function(organizer, list, options) {
	var incRemote = options && options.includeRemote;
	var remoteOnly = options && options.remoteOnly;
	var isRemote = organizer.isRemote();
	if ((!isRemote && !remoteOnly) || (isRemote && (remoteOnly || incRemote))) {
		list.push(organizer);
	}
	var children = organizer.children.getArray();
    for (var i = 0; i < children.length; i++) {
        this._addToList(children[i], list, options);
    }
	return list;
};

/**
 * @private
 */
ZmTree.prototype._asString =
function(organizer, str) {
	if (organizer.id) {
		str = str + organizer.id;
	}
	var children = organizer.children.clone().getArray();
	if (children.length) {
		children.sort(function(a,b){return a.id - b.id;});
		str = str + "[";
		for (var i = 0; i < children.length; i++) {
			if (children[i].id == ZmFolder.ID_TAGS) { // Tags "folder" added when view is set
				continue;
			}
			if (i > 0) {
				str = str + ",";
			}
			str = this._asString(children[i], str);
		}
		str = str + "]";
	}
	return str;
};

/**
 * @private
 */
ZmTree.prototype._getUnreadHash =
function(organizer, unread) {
	unread[organizer.id] = organizer.numUnread;
	var children = organizer.children.getArray();
	for (var i = 0; i < children.length; i++) {
		this._getUnreadHash(children[i], unread);
	}

	return unread;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmTagTree")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the tag tree class.
 */

/**
 * Creates the tag tree
 * @class
 * This class represents the tag tree.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @extends	ZmTree
 */
ZmTagTree = function(account) {
	ZmTree.call(this, ZmOrganizer.TAG);
	var id = (account)
		? ([account.id, ZmTag.ID_ROOT].join(":"))
		: ZmTag.ID_ROOT;
	this.root = new ZmTag({ id:id, tree:this });
};

ZmTagTree.prototype = new ZmTree;
ZmTagTree.prototype.constructor = ZmTagTree;

// ordered list of colors
ZmTagTree.COLOR_LIST = [
    ZmOrganizer.C_BLUE,
    ZmOrganizer.C_CYAN,
    ZmOrganizer.C_GREEN,
    ZmOrganizer.C_PURPLE,
    ZmOrganizer.C_RED,
    ZmOrganizer.C_YELLOW,
    ZmOrganizer.C_PINK,
    ZmOrganizer.C_GRAY,
    ZmOrganizer.C_ORANGE
];

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmTagTree.prototype.toString = 
function() {
	return "ZmTagTree";
};

/**
 * @private
 */
ZmTagTree.prototype.loadFromJs =
function(tagsObj, type, account) {
	if (!tagsObj || !tagsObj.tag || !tagsObj.tag.length) { return; }

	for (var i = 0; i < tagsObj.tag.length; i++) {
		ZmTag.createFromJs(this.root, tagsObj.tag[i], this, null, account);
	}
	var children = this.root.children.getArray();
	if (children.length) {
		children.sort(ZmTag.sortCompare);
	}
};

/**
 * Gets the tag by index.
 * 
 * @param	{int}	idx		the index
 * @return	{ZmTag}	the tag
 */
ZmTagTree.prototype.getByIndex =
function(idx) {
	var list = this.asList();	// tag at index 0 is root
	if (list && list.length && (idx < list.length))	{
		return list[idx];
	}
};

/**
 * Resets the tree.
 */
ZmTagTree.prototype.reset =
function() {
	this.root = new ZmTag({id: ZmTag.ID_ROOT, tree: this});
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmFolderTree")) {
	/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines a folder tree.
 *
 */

/**
 * Creates an empty folder tree.
 * @class
 * This class represents a tree of folders. It may be typed, in which case
 * the folders are all of that type, or untyped.
 * 
 * @author Conrad Damon
 * 
 * @param {constant}	type		the organizer type
 * 
 * @extends	ZmTree
 */
ZmFolderTree = function(type) {
	ZmTree.call(this, type);
};

ZmFolderTree.prototype = new ZmTree;
ZmFolderTree.prototype.constructor = ZmFolderTree;


// Consts
ZmFolderTree.IS_PARSED = {};


// Public Methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmFolderTree.prototype.toString =
function() {
	return "ZmFolderTree";
};

/**
 * Loads the folder or the zimlet tree.
 * 
 * @param	{Object}		rootObj		the root object
 * @param	{String}		elementType		the element type
 * @param	{ZmZimbraAccount}		account		the account
 */
ZmFolderTree.prototype.loadFromJs =
function(rootObj, elementType, account) {
	this.root = (elementType == "zimlet")
		? ZmZimlet.createFromJs(null, rootObj, this)
		: ZmFolderTree.createFromJs(null, rootObj, this, elementType, null, account);
};

/**
 * Generic function for creating a folder. Handles any organizer type that comes
 * in the folder list.
 * 
 * @param {ZmFolder}	parent		the parent folder
 * @param {Object}	obj			the JSON with folder data
 * @param {ZmFolderTree}	tree			the containing tree
 * @param {String}	elementType		the type of containing JSON element
 * @param {Array}	path			the list of path elements
 * @param {ZmZimbraAccount}	account		the account this folder belongs to
 */
ZmFolderTree.createFromJs =
function(parent, obj, tree, elementType, path, account) {
	if (!(obj && obj.id)) { return; }
	if (obj && (obj.owner || obj.id == ZmFolder.ID_FILE_SHARED_WITH_ME) && (!appCtxt.isExternalAccount() && !appCtxt.get(ZmSetting.SHARING_ENABLED))) { return; }

	var folder;
	if (elementType == "search") {
		var types;
		var idParts = obj.id.split(":");
		// Suppress display of searches for the shared mailbox (See Bug 96090) - it will have an id
		// of the form 'uuid:id'.  Local searches will just have 'id'
		if (!idParts || (idParts.length <= 1)) {
			if (obj.types) {
				var t = obj.types.split(",");
				types = [];
				var mailEnabled = appCtxt.get(ZmSetting.MAIL_ENABLED);
				for (var i = 0; i < t.length; i++) {
					var type = ZmSearch.TYPE_MAP[t[i]];
					if (!type || (!mailEnabled && (type == ZmItem.CONV || type == ZmItem.MSG))) {
						continue;
					}
					types.push(type);
				}
				if (types.length == 0) {
					return null;
				}
			}
			DBG.println(AjxDebug.DBG2, "Creating SEARCH with id " + obj.id + " and name " + obj.name);
			var params = {
				id: obj.id,
				name: obj.name,
				parent: parent,
				tree: tree,
				numUnread: obj.u,
				query: obj.query,
				types: types,
				sortBy: obj.sortBy,
				account: account,
				color: obj.color,
				rgb: obj.rgb
			};
			folder = new ZmSearchFolder(params);
			ZmFolderTree._fillInFolder(folder, obj, path);
			ZmFolderTree._traverse(folder, obj, tree, (path || []), elementType, account);
		}
	} else {
		var type = obj.view
			? (ZmOrganizer.TYPE[obj.view])
			: (parent ? parent.type : ZmOrganizer.FOLDER);

		if (!type) {
			DBG.println(AjxDebug.DBG1, "No known type for view " + obj.view);
			return;
		}
		// let's avoid deferring folders for offline since multi-account folder deferring is hairy
		var hasGrants = (obj.acl && obj.acl.grant && obj.acl.grant.length > 0);
		if (appCtxt.inStartup && ZmOrganizer.DEFERRABLE[type] && !appCtxt.isOffline) {
			var app = appCtxt.getApp(ZmOrganizer.APP[type]);
			var defParams = {
				type:			type,
				parent:			parent,
				obj:			obj,
				tree:			tree,
				path:			path,
				elementType:	elementType,
				account:		account
			};
			app.addDeferredFolder(defParams);
		} else {
			var pkg = ZmOrganizer.ORG_PACKAGE[type];
			if (pkg) {
				AjxDispatcher.require(pkg);
			}
			folder = ZmFolderTree.createFolder(type, parent, obj, tree, path, elementType, account);
            if (appCtxt.isExternalAccount() && folder.isSystem() && folder.id != ZmOrganizer.ID_ROOT) { return; }
			ZmFolderTree._traverse(folder, obj, tree, (path || []), elementType, account);
		}
	}

	return folder;
};

ZmFolderTree.createAllDeferredFolders =
function() {
	var ac = appCtxt.getAppController();
	for (var appId in ZmApp.ORGANIZER) {
		var app = ac.getApp(appId);
		app.createDeferred();
	}
};

/**
 * @private
 */
ZmFolderTree._traverse =
function(folder, obj, tree, path, elementType, account) {

	var isRoot = (folder.nId == ZmOrganizer.ID_ROOT);

	if (elementType === "hab") {
		obj.folder = obj.habGroup
	}

	if (obj.folder && obj.folder.length) {
		if (!isRoot) {
			path.push(obj.name);
		}
		for (var i = 0; i < obj.folder.length; i++) {
			var folderObj = obj.folder[i];
			var childFolder = ZmFolderTree.createFromJs(folder, folderObj, tree, (elementType || "folder"), path, account);
			if (folder && childFolder) {
				folder.children.add(childFolder);
			}
		}
		if (!isRoot) {
			path.pop();
		}
	}
	
	if (obj.search && obj.search.length) {
		if (!isRoot) {
			path.push(obj.name);
		}
		for (var i = 0; i < obj.search.length; i++) {
			var searchObj = obj.search[i];
			var childSearch = ZmFolderTree.createFromJs(folder, searchObj, tree, "search", path, account);
			if (childSearch) {
				folder.children.add(childSearch);
			}
		}
		if (!isRoot) {
			path.pop();
		}
	}

	if (obj.link && obj.link.length) {
		for (var i = 0; i < obj.link.length; i++) {
			var link = obj.link[i];
			var childFolder = ZmFolderTree.createFromJs(folder, link, tree, "link", path, account);
			if (childFolder) {
				folder.children.add(childFolder);
			}
		}
	}
};

/**
 * Creates the folder.
 * 
 * @param {String}	type		the folder type
 * @param {ZmFolder}	parent		the parent folder
 * @param {Object}	obj			the JSON with folder data
 * @param {ZmFolderTree}	tree			the containing tree
 * @param {Array}	path			the list of path elements
 * @param {String}	elementType		the type of containing JSON element
 * @param {ZmZimbraAccount}	account		the account this folder belongs to
 */
ZmFolderTree.createFolder =
function(type, parent, obj, tree, path, elementType, account) {
	var orgClass = eval(ZmOrganizer.ORG_CLASS[type]);
	if (!orgClass) { return null; }

	DBG.println(AjxDebug.DBG2, "Creating " + type + " with id " + obj.id + " and name " + obj.name);

	var params = {
		id: 		obj.id,
		name: 		obj.name,
		parent: 	parent,
		tree: 		tree,
		color: 		obj.color,
		rgb:		obj.rgb,
		owner: 		obj.owner,
		oname: 		obj.oname,
		zid: 		obj.zid,
		rid: 		obj.rid,
		restUrl: 	obj.rest,
		url: 		obj.url,
		numUnread: 	obj.u,
		numTotal: 	obj.n,
		sizeTotal: 	obj.s,
		perm: 		obj.perm,
		link: 		elementType == "link",
		broken: 	obj.broken,
		account:	account,
        webOfflineSyncDays : obj.webOfflineSyncDays,
        retentionPolicy: obj.retentionPolicy
	};

	if (elementType == "hab") {
		params.name = obj._attrs ? obj._attrs.displayName : obj.name;
		params.mail = obj._attrs ? obj._attrs.mail : undefined;
	}

	var folder = new orgClass(params);
	ZmFolderTree._fillInFolder(folder, obj, path);
	ZmFolderTree.IS_PARSED[type] = true;

	return folder;
};

/**
 * @private
 */
ZmFolderTree._fillInFolder =
function(folder, obj, path) {
	if (path && path.length) {
		folder.path = path.join("/");
	}

	if (obj.f && folder._parseFlags) {
		folder._parseFlags(obj.f);
	}

	folder._setSharesFromJs(obj);
};

/**
 * Gets the folder by type.
 * 
 * @param	{String}	type	the type
 * @return	{ZmFolder}	the folder or <code>null</code> if not found
 */
ZmFolderTree.prototype.getByType =
function(type) {
	return this.root ? this.root.getByType(type) : null;
};

/**
 * Gets the folder by path.
 * 
 * @param	{String}	path	the path
 * @param	{Boolean}	useSystemName		<code>true</code> to use the system name
 * @return	{ZmFolder}	the folder or <code>null</code> if not found
 */
ZmFolderTree.prototype.getByPath =
function(path, useSystemName) {
	return this.root ? this.root.getByPath(path, useSystemName) : null;
};

/**
 * Handles a missing link by marking its organizer as not there, redrawing it in
 * any tree views, and asking to delete it.
 *
 * @param {int}	organizerType		the type of organizer (constants defined in {@link ZmOrganizer})
 * @param {String}	zid			the zid of the missing folder
 * @param {String}	rid			the rid of the missing folder
 * @return	{Boolean}	<code>true</code> if the error is handled
 */
ZmFolderTree.prototype.handleNoSuchFolderError =
function(organizerType, zid, rid) {
	var items = this.getByType(organizerType);

	var treeView;
	var handled = false;
	if (items) {
		for (var i = 0; i < items.length; i++) {
			if ((items[i].zid == zid) && (items[i].rid == rid)) {
				// Mark that the item is not there any more.
				items[i].noSuchFolder = true;

				// Change its appearance in the tree.
				if (!treeView) {
					var overviewId = appCtxt.getAppController().getOverviewId();
					treeView = appCtxt.getOverviewController().getTreeView(overviewId, organizerType);
				}
				var node = treeView.getTreeItemById(items[i].id);
				node.setText(items[i].getName(true));

				// Ask if it should be deleted now.
				this.handleDeleteNoSuchFolder(items[i]);
				handled = true;
			}
		}
	}
	return handled;
};

/**
 * Handles no such folder. The user will be notified that a linked organizer generated a "no such folder",
 * error, giving the user a chance to delete the folder.
 *
 * @param {ZmOrganizer}	organizer	the organizer
 */
ZmFolderTree.prototype.handleDeleteNoSuchFolder =
function(organizer) {
	var ds = appCtxt.getYesNoMsgDialog();
	ds.reset();
	ds.registerCallback(DwtDialog.YES_BUTTON, this._deleteOrganizerYesCallback, this, [organizer, ds]);
	ds.registerCallback(DwtDialog.NO_BUTTON, appCtxt.getAppController()._clearDialog, this, ds);
	var msg = AjxMessageFormat.format(ZmMsg.confirmDeleteMissingFolder, AjxStringUtil.htmlEncode(organizer.getName(false, 0, true)));
	ds.setMessage(msg, DwtMessageDialog.WARNING_STYLE);
	ds.popup();
};

/**
 * Handles the "Yes" button in the delete organizer dialog.
 * 
 * @param	{ZmOrganizer}	organizer		the organizer
 * @param	{ZmDialog}		dialog		the dialog
 */
ZmFolderTree.prototype._deleteOrganizerYesCallback =
function(organizer, dialog) {
	organizer._delete();
	appCtxt.getAppController()._clearDialog(dialog);
};

/**
 * Issues a <code>&lt;BatchRequest&gt;</code> of <code>&lt;GetFolderRequest&gt;</code>s for existing
 * mountpoints that do not have permissions set.
 *
 * @param	{Hash}	params	a hash of parameters
 * @param {int}	params.type			the {@link ZmItem} type constant
 * @param {AjxCallback}	params.callback			the callback to trigger after fetching permissions
 * @param {Boolean}	params.skipNotify		<code>true</code> to skip notify after fetching permissions
 * @param {Array}	params.folderIds			the list of folder Id's to fetch permissions for
 * @param {Boolean}	params.noBusyOverlay		<code>true</code> to not block the UI while fetching permissions
 * @param {String}	params.accountName		the account to issue request under
 */
ZmFolderTree.prototype.getPermissions =
function(params) {
	var needPerms = params.folderIds || this._getItemsWithoutPerms(params.type);

	// build batch request to get all permissions at once
	if (needPerms.length > 0) {
		var soapDoc = AjxSoapDoc.create("BatchRequest", "urn:zimbra");
		soapDoc.setMethodAttribute("onerror", "continue");

		var doc = soapDoc.getDoc();
		for (var j = 0; j < needPerms.length; j++) {
			var folderRequest = soapDoc.set("GetFolderRequest", null, null, "urn:zimbraMail");
			var folderNode = doc.createElement("folder");
			folderNode.setAttribute("l", needPerms[j]);
			folderRequest.appendChild(folderNode);
		}

		var respCallback = new AjxCallback(this, this._handleResponseGetShares, [params.callback, params.skipNotify]);
		appCtxt.getRequestMgr().sendRequest({
			soapDoc: soapDoc, 
			asyncMode: true,
			callback: respCallback,
			noBusyOverlay: params.noBusyOverlay,
			accountName: params.accountName
		});
	} else {
		if (params.callback) {
			params.callback.run();
		}
	}
};

/**
 * @private
 */
ZmFolderTree.prototype._getItemsWithoutPerms =
function(type) {
	var needPerms = [];
	var orgs = type ? [type] : [ZmOrganizer.FOLDER, ZmOrganizer.CALENDAR, ZmOrganizer.TASKS, ZmOrganizer.BRIEFCASE, ZmOrganizer.ADDRBOOK];

	for (var j = 0; j < orgs.length; j++) {
		var org = orgs[j];
		if (!ZmFolderTree.IS_PARSED[org]) { continue; }

		var items = this.getByType(org);

		for (var i = 0; i < items.length; i++) {
			if (items[i].link && items[i].shares == null) {
				needPerms.push(items[i].id);
			}
		}
	}

	return needPerms;
};

/**
 * @private
 */
ZmFolderTree.prototype._handleResponseGetShares =
function(callback, skipNotify, result) {
	var batchResp = result.getResponse().BatchResponse;
	this._handleErrorGetShares(batchResp);

	var resp = batchResp.GetFolderResponse;
	if (resp) {
		for (var i = 0; i < resp.length; i++) {
			var link = resp[i].link ? resp[i].link[0] : null;
			if (link) {
				var mtpt = appCtxt.getById(link.id);
				if (mtpt) {
					// update the mtpt perms with the updated link perms
					mtpt.perm = link.perm;
                    if (link.n) mtpt.numTotal=link.n;
                    if (link.u) mtpt.numUnread=link.u;
					mtpt._setSharesFromJs(link);
				}

				if (link.folder && link.folder.length > 0) {
					var parent = appCtxt.getById(link.id);
					if (parent) {
						// TODO: only goes one level deep - should we recurse?
						for (var j = 0; j < link.folder.length; j++) {
							if (appCtxt.getById(link.folder[j].id)) { continue; }
							parent.notifyCreate(link.folder[j], "link", skipNotify);
						}
					}
				}
			}
		}
	}

	if (callback) {
		callback.run();
	}
};

/**
 * Handles errors that come back from the GetShares batch request.
 *
 * @param {Array}	organizerTypes	the types of organizer (constants defined in {@link ZmOrganizer})
 * @param {Object}	batchResp			the response
 *
 */
ZmFolderTree.prototype._handleErrorGetShares =
function(batchResp) {
	var faults = batchResp.Fault;
	if (faults) {
		var rids = [];
		var zids = [];
		for (var i = 0, length = faults.length; i < length; i++) {
			var ex = ZmCsfeCommand.faultToEx(faults[i]);
			if (ex.code == ZmCsfeException.MAIL_NO_SUCH_FOLDER) {
				var itemId = ex.data.itemId[0];
				var index = itemId.lastIndexOf(':');
				zids.push(itemId.substring(0, index));
				rids.push(itemId.substring(index + 1, itemId.length));
			}
		}
		if (zids.length) {
			this._markNoSuchFolder(zids, rids);
		}
	}
};

/**
 * Handles missing links by marking the organizers as not there
 *
 * @param {Array}	zids		the zids of the missing folders
 * @param {Array}	rids		the rids of the missing folders. rids and zids must have the same length
 *
 */
ZmFolderTree.prototype._markNoSuchFolder =
function(zids, rids) {
	var treeData = appCtxt.getFolderTree();
	var items = treeData && treeData.root
		? treeData.root.children.getArray()
		: null;

	for (var i = 0; i < items.length; i++) {
		for (var j = 0; j < rids.length; j++) {
			if ((items[i].zid == zids[j]) && (items[i].rid == rids[j])) {
				items[i].noSuchFolder = true;
			}
		}
	}
};

/**
 * @private
 */
ZmFolderTree.prototype._sortFolder =
function(folder) {
	var children = folder.children;
	if (children && children.length) {
		children.sort(ZmFolder.sortCompare);
		for (var i = 0; i < children.length; i++)
			this._sortFolder(children[i]);
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmList")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines a list of items.
 */

/**
 * Creates an empty list of items of the given type.
 * @class
 * This class represents a list of items ({@link ZmItem} objects). Any SOAP method that can be
 * applied to a list of item IDs is represented here, so that we can perform an action
 * on multiple items with just one CSFE call. For the sake of convenience, a hash 
 * matching item IDs to items is maintained. Items are assumed to have an 'id'
 * property.
 * <br/>
 * <br/>
 * The calls are made asynchronously. We are assuming that any action taken will result
 * in a notification, so the action methods generally do not have an async callback 
 * chain and thus are leaf nodes. An exception is moving conversations. We don't
 * know enough from the ensuing notifications (which only indicate that messages have
 * moved), we need to update the UI based on the response.
 *
 * @author Conrad Damon
 * 
 * @param {constant}	type		the item type
 * @param {ZmSearch}	search	the search that generated this list
 * 
 * @extends	ZmModel
 */
ZmList = function(type, search) {

	if (arguments.length == 0) return;
	ZmModel.call(this, type);

	this.type = type;
	this.search = search;
	
	this._vector = new AjxVector();
	this._hasMore = false;
	this._idHash = {};

	var tagList = appCtxt.getTagTree();
	if (tagList) {
		this._tagChangeListener = new AjxListener(this, this._tagTreeChangeListener);
		tagList.addChangeListener(this._tagChangeListener);
	}
	
	this.id = "LIST" + ZmList.NEXT++;
	appCtxt.cacheSet(this.id, this);
};

ZmList.prototype = new ZmModel;
ZmList.prototype.constructor = ZmList;

ZmList.prototype.isZmList = true;
ZmList.prototype.toString = function() { return "ZmList"; };


ZmList.NEXT = 1;

// for item creation
ZmList.ITEM_CLASS = {};

// node names for item types
ZmList.NODE = {};

// item types based on node name (reverse map of above)
ZmList.ITEM_TYPE = {};

ZmList.CHUNK_SIZE	= 100;	// how many items to act on at a time via a server request
ZmList.CHUNK_PAUSE	= 500;	// how long to pause to allow UI to catch up


/**
 * Gets the item.
 * 
 * @param	{int}	index		the index
 * @return	{ZmItem}	the index
 */
ZmList.prototype.get =
function(index) {
	return this._vector.get(index);
};

/**
 * Adds an item to the list.
 *
 * @param {ZmItem}	item	the item to add
 * @param {int}	index	the index at which to add the item (defaults to end of list)
 */
ZmList.prototype.add = 
function(item, index) {
	this._vector.add(item, index);
	if (item.id) {
		this._idHash[item.id] = item;
	}
};

/**
 * Removes an item from the list.
 *
 * @param {ZmItem}	item	the item to remove
 */
ZmList.prototype.remove = 
function(item) {
	this._vector.remove(item);
	if (item.id) {
		delete this._idHash[item.id];
	}
};

/**
 * Creates an item from the given arguments. A subclass may override
 * <code>sortIndex()</code> to add it to a particular point in the list. By default, it
 * will be added at the end.
 *
 * <p>
 * The item will invoke a SOAP call, which generates a create notification from the
 * server. That will be handled by notifyCreate(), which will call _notify()
 * so that views can be updated.
 * </p>
 *
 * @param {Hash}	args	a hash of arugments to pass along to the item constructor
 * @return	{ZmItem}	the newly created item
 */
ZmList.prototype.create =
function(args) {
	var item;
	var obj = eval(ZmList.ITEM_CLASS[this.type]);
	if (obj) {
		item = new obj(this);
		item.create(args);
	}

	return item;
};

/**
 * Returns the number of items in the list.
 * 
 * @return	{int}	the number of items
 */
ZmList.prototype.size = 
function() {
	return this._vector.size();
};

/**
 * Returns the index of the given item in the list.
 * 
 * @param	{ZmItem}	item		the item
 * @return	{int}	the index
 */
ZmList.prototype.indexOf = 
function(item) {
	return this._vector.indexOf(item);
};

/**
 * Gets if there are more items for this search.
 * 
 * @return	{Boolean}	<code>true</code> if there are more items
 */
ZmList.prototype.hasMore = 
function() {
	return this._hasMore;
};

/**
 * Sets the "more" flag for this list.
 *
 * @param {Boolean}	bHasMore	<code>true</code> if there are more items
 */
ZmList.prototype.setHasMore = 
function(bHasMore) {
	this._hasMore = bHasMore;
};

/**
 * Returns the list as an array.
 * 
 * @return	{Array}	an array of {ZmItem} objects
 */
ZmList.prototype.getArray =
function() {
	return this._vector.getArray();
};

/**
 * Returns the list as a vector.
 * 
 * @return	{AjxVector}	a vector of {ZmItem} objects
 */
ZmList.prototype.getVector =
function() {
	return this._vector;
};

/**
 * Gets the item with the given id.
 *
 * @param {String}	id		an item id
 * 
 * @return	{ZmItem}	the item
 */
ZmList.prototype.getById =
function(id) {
	return this._idHash[id];
};

/**
 * Clears the list, including the id hash.
 * 
 */
ZmList.prototype.clear =
function() {
	// First, let each item run its clear() method
	var a = this.getArray();
	for (var i = 0; i < a.length; i++) {
		a[i].clear && a[i].clear();
	}

	this._evtMgr.removeAll(ZmEvent.L_MODIFY);
	this._vector.removeAll();
	for (var id in this._idHash) {
		this._idHash[id] = null;
	}
	this._idHash = {};
};

/**
 * Populates the list with elements created from the response to a SOAP command. Each
 * node in the response should represent an item of the list's type. Items are added
 * in the order they are received; no sorting is done.
 *
 * @param {Object}	respNode	an XML node whose children are item nodes
 */
ZmList.prototype.set = 
function(respNode) {
	this.clear();
	var nodes = respNode.childNodes;
	var args = {list:this};
	for (var i = 0; i < nodes.length; i++) {
		var node = nodes[i];
		if (node.nodeName == ZmList.NODE[this.type]) {
			/// TODO: take this out, let view decide whether to show items in Trash
			if (parseInt(node.getAttribute("l")) == ZmFolder.ID_TRASH && (this.type != ZmItem.CONTACT))	{ continue; }
			var obj = eval(ZmList.ITEM_CLASS[this.type]);
			if (obj) {
				this.add(obj.createFromDom(node, args));
			}
		}
	}
};

/**
 * Adds an item to the list from the given XML node.
 *
 * @param {Object}	node	an XML node
 * @param {Hash}	args	an optional list of arguments to pass to the item contructor
 */
ZmList.prototype.addFromDom = 
function(node, args) {
	args = args || {};
	args.list = this;
	var obj = eval(ZmList.ITEM_CLASS[this.type]);
	if (obj) {
		this.add(obj.createFromDom(node, args));
	}
};

/**
 * Gets a vector containing a subset of items of this list.
 *
 * @param {int}		offset		the starting index
 * @param {int}		limit		the size of sublist
 * @return	{AjxVector}	the vector
 */
ZmList.prototype.getSubList = 
function(offset, limit) {
	var subVector = null;
	var end = (offset + limit > this.size()) ? this.size() : offset + limit;
	var subList = this.getArray();
	if (offset < end) {
		subVector = AjxVector.fromArray(subList.slice(offset, end));
	}
	return subVector;
};

/**
 * Caches the list.
 * 
 * @param	{int}	offset	the index
 * @param	{AjxVector}	newList		the new list
 */
ZmList.prototype.cache = 
function(offset, newList) {
	this.getVector().merge(offset, newList);
	// reparent each item within new list, and add it to ID hash
	var list = newList.getArray();
	for (var i = 0; i < list.length; i++) {
		var item = list[i];
		item.list = this;
		if (item.id) {
			this._idHash[item.id] = item;
		}
	}
};

// Actions

/**
 * Sets and unsets a flag for each of a list of items.
 *
 * @param 	{Hash}				params					a hash of parameters
 * @param	{Array}     		params.items			a list of items to set/unset a flag for
 * @param	{String}			params.op				the name of the flag operation ("flag" or "read")
 * @param	{Boolean|String}	params.value			whether to set the flag, or for "update" the flags string
 * @param	{AjxCallback}		params.callback			the callback to run after each sub-request
 * @param	{closure}			params.finalCallback	the callback to run after all items have been processed
 * @param	{int}				params.count			the starting count for number of items processed
 * @param   {String}    		params.actionTextKey   	pattern for generating action summarykey to action summary message
 */
ZmList.prototype.flagItems =
function(params) {

	params = Dwt.getParams(arguments, ["items", "op", "value", "callback"]);

	params.items = AjxUtil.toArray(params.items);

	if (params.op == "update") {
		params.action = params.op;
		params.attrs = {f:params.value};
	} else {
		params.action = params.value ? params.op : "!" + params.op;
	}

    if (appCtxt.multiAccounts) {
		// check if we're flagging item from remote folder, in which case, always send
		// request on-behalf-of the account the item originally belongs to.
        var folderId = this.search.folderId;
        var fromFolder = folderId && appCtxt.getById(folderId);
        if (fromFolder && fromFolder.isRemote()) {
                params.accountName = params.items[0].getAccount().name;
        }
	}

	this._itemAction(params);
};

/**
 * Tags or untags a list of items. A sanity check is done first, so that items
 * aren't tagged redundantly, and so we don't try to remove a nonexistent tag.
 *
 * @param {Hash}		params					a hash of parameters
 * @param {Array}		params.items			a list of items to tag/untag
 * @param {String}  	params.tagId            ID of tag to add/remove
 * @param {String}		params.tag  			the tag to add/remove from each item (optional)
 * @param {Boolean}		params.doTag			<code>true</code> if adding the tag, <code>false</code> if removing it
 * @param {AjxCallback}	params.callback			the callback to run after each sub-request
 * @param {closure}		params.finalCallback	the callback to run after all items have been processed
 * @param {int}			params.count			the starting count for number of items processed
 */
ZmList.prototype.tagItems =
function(params) {

	params = Dwt.getParams(arguments, ["items", "tagId", "doTag"]);

    var tagName = params.tagName || (params.tag && params.tag.name);

	//todo - i hope this is no longer needed. I think the item we apply the tag to should determine the tag id on the server side.
//	// for multi-account mbox, normalize tagId
//	if (appCtxt.multiAccounts && !appCtxt.getActiveAccount().isMain) {
//		tagId = ZmOrganizer.normalizeId(tagId);
//	}

	// only tag items that don't have the tag, and untag ones that do
	// always tag a conv, because we don't know if all items in the conv have the tag yet
	var items = AjxUtil.toArray(params.items);
	var items1 = [], doTag = params.doTag;
	if (items[0] && items[0] instanceof ZmItem) {
		for (var i = 0; i < items.length; i++) {
			var item = items[i];
			if ((doTag && (!item.hasTag(tagName) || item.type == ZmItem.CONV)) ||	(!doTag && item.hasTag(tagName))) {
				items1.push(item);
			}
		}
	} else {
		items1 = items;
	}
	params.items = items1;
	params.attrs = {tn: tagName};
	params.action = doTag ? "tag" : "!tag";
    params.actionTextKey = doTag ? 'actionTag' : 'actionUntag';
	params.actionArg = params.tag && params.tag.name;

	this._itemAction(params);
};

/**
 * Removes all tags from a list of items.
 *
 * @param	{Hash}			params					a hash of parameters
 * @param	{Array}			params.items			a list of items to tag/untag
 * @param	{AjxCallback}	params.callback			the callback to run after each sub-request
 * @param	{closure}		params.finalCallback	the callback to run after all items have been processed
 * @param	{int}			params.count			the starting count for number of items processed
 */
ZmList.prototype.removeAllTags = 
function(params) {

	params = (params && params.items) ? params : {items:params};

	var items = AjxUtil.toArray(params.items);
	var items1 = [];
	if (items[0] && items[0] instanceof ZmItem) {
		for (var i = 0; i < items.length; i++) {
			var item = items[i];
			if (item.tags && item.tags.length) {
				items1.push(item);
			}
		}
	} else {
		items1 = items;
	}

	params.items = items1;
	params.action = "update";
	params.attrs = {t: ""};
    params.actionTextKey = 'actionRemoveTags';

	this._itemAction(params);
};

/**
 * Moves a list of items to the given folder.
 * <p>
 * Search results are treated as though they're in a temporary folder, so that they behave as
 * they would if they were in any other folder such as Inbox. When items that are part of search
 * results are moved, they will disappear from the view, even though they may still satisfy the
 * search.
 * </p>
 *
 * @param	{Hash}			params					a hash of parameters
 * @param	{Array}			params.items			a list of items to move
 * @param	{ZmFolder}		params.folder			the destination folder
 * @param	{Hash}			params.attrs			the additional attrs for SOAP command
 * @param	{AjxCallback}	params.callback			the callback to run after each sub-request
 * @param	{closure}		params.finalCallback	the callback to run after all items have been processed
 * @param	{int}			params.count			the starting count for number of items processed
 * @param	{boolean}		params.noUndo			true if the action is not undoable (e.g. performed as an undo)
 */
ZmList.prototype.moveItems =
function(params) {
	
	params = Dwt.getParams(arguments, ["items", "folder", "attrs", "callback", "errorCallback" ,"finalCallback", "noUndo"]);

	var params1 = AjxUtil.hashCopy(params);
	params1.items = AjxUtil.toArray(params.items);
	params1.attrs = params.attrs || {};
	params1.childWin = params.childWin;
	params1.closeChildWin = params.closeChildWin;
	
	if (params1.folder.id == ZmFolder.ID_TRASH) {
		params1.actionTextKey = 'actionTrash';
		params1.action = "trash";
	} else {
		params1.actionTextKey = 'actionMove';
		params1.actionArg = params.folder.getName(false, false, true);
		params1.action = "move";
		params1.attrs.l = params.folder.id;
	}
	params1.callback = new AjxCallback(this, this._handleResponseMoveItems, [params]);
	if (params.noToast) {
		params1.actionTextKey = null;
	}

    if (appCtxt.multiAccounts) {
		// Reset accountName for multi-account to be the respective account if we're
		// moving a draft out of Trash.
		// OR,
		// check if we're moving to or from a shared folder, in which case, always send
		// request on-behalf-of the account the item originally belongs to.

        var folderId = params.items[0].getFolderId && params.items[0].getFolderId();

        // on bulk delete, when the second chunk loads try to get folderId from the item id.
        if (!folderId) {
            var itemId = params.items[0] && params.items[0].id;
            folderId = itemId && appCtxt.getById(itemId) && appCtxt.getById(itemId).folderId;
        }
        var fromFolder = appCtxt.getById(folderId);
		if ((params.items[0].isDraft && params.folder.id == ZmFolder.ID_DRAFTS) ||
			(params.folder.isRemote()) || (fromFolder && fromFolder.isRemote()))
		{
			params1.accountName = params.items[0].getAccount().name;
		}
	}
	//Error Callback
	params1.errorCallback = params.errorCallback;

	if (this._handleDeleteFromSharedFolder(params, params1)) {
		return;
	}
    
	this._itemAction(params1);
};

ZmList.prototype._handleDeleteFromSharedFolder =
function(params, params1) {

	// Bug 26103: when deleting an item in a folder shared to us, save a copy in our own trash
	if (params.folder && params.folder.id == ZmFolder.ID_TRASH) {
		var fromFolder;
		var toCopy = [];
		for (var i = 0; i < params.items.length; i++) {
			var item = params.items[i];
			var index = item.id.indexOf(":");
			if (index != -1) { //might be shared
				var acctId = item.id.substring(0, index);
				if (!appCtxt.accountList.getAccount(acctId)) {
					fromFolder = appCtxt.getById(item.folderId);
					// Don't do the copy if the source folder is shared with view only rights
					if (fromFolder && !fromFolder.isReadOnly()) {
						toCopy.push(item);
					}
				}
			}
		}
		if (toCopy.length) {
			var params2 = {
				items:			toCopy,
				folder:			params.folder, // Should refer to our own trash folder
				finalCallback:	this._itemAction.bind(this, params1, null),
				actionTextKey:	null
			};
			this.copyItems(params2);
			return true;
		}
	}
};

/**
 * @private
 */
ZmList.prototype._handleResponseMoveItems =
function(params, result) {

	var movedItems = result.getResponse();
	if (movedItems && movedItems.length && (movedItems[0] instanceof ZmItem)) {
		this.moveLocal(movedItems, params.folder.id);
		for (var i = 0; i < movedItems.length; i++) {
			var item = movedItems[i];
			var details = {oldFolderId:item.folderId};
			item.moveLocal(params.folder.id);
			//ZmModel.prototype._notify.call(item, ZmEvent.E_MOVE, details);
		}
		// batched change notification
		//todo - it's probably possible that different items have different _lists they are in
		// thus getting the lists just from the first item is not enough. But hopefully good
		// enough for the most common cases. Prior to this fix it was only taking the current list
		// the first item is in, so this is already better. :)
		var item = movedItems[0];
		// Bug:104027 Adds the missing current list to _lists
		if(!item._list[item.list.id]) {
			item._list[item.list.id] = true;
		}
		for (var listId in item._list) {
			var ac = window.parentAppCtxt || appCtxt; //always get the list in the parent window. The child might be closed or closing, causing bugs.
			var list = ac.getById(listId);
			if (!list) {
				continue;
			}
            list._evt.batchMode = true;
            list._evt.item = item;	// placeholder
            list._evt.items = movedItems;
            list._notify(ZmEvent.E_MOVE, details);
        }
	}

	if (params.callback) {
		params.callback.run(result);
	}
};

/**
 * Copies a list of items to the given folder.
 *
 * @param {Hash}		params					the hash of parameters
 * @param {Array}		params.items			a list of items to move
 * @param {ZmFolder}	params.folder			the destination folder
 * @param {Hash}		params.attrs			the additional attrs for SOAP command
 * @param {closure}		params.finalCallback	the callback to run after all items have been processed
 * @param {int}			params.count			the starting count for number of items processed
 * @param {String}		params.actionTextKey	key to optional text to display in the confirmation toast instead of the default summary. May be set explicitly to null to disable the confirmation toast
 */
ZmList.prototype.copyItems =
function(params) {

	params = Dwt.getParams(arguments, ["items", "folder", "attrs", "actionTextKey"]);

	params.items = AjxUtil.toArray(params.items);
	params.attrs = params.attrs || {};
    if (!appCtxt.isExternalAccount()) {
        params.attrs.l = params.folder.id;
        params.action = "copy";
        params.actionTextKey = 'itemCopied';
    }
    else {
        params.action = 'trash';
    }
	params.actionArg = params.folder.getName(false, false, true);
	params.callback = new AjxCallback(this, this._handleResponseCopyItems, params);

	if (appCtxt.multiAccounts && params.folder.isRemote()) {
		params.accountName = params.items[0].getAccount().name;
	}

	this._itemAction(params);
};

/**
 * @private
 */
ZmList.prototype._handleResponseCopyItems =
function(params, result) {
	var resp = result.getResponse();
	if (resp.length > 0) {
		if (params.actionTextKey) {
			var msg = AjxMessageFormat.format(ZmMsg[params.actionTextKey], resp.length);
			appCtxt.getAppController().setStatusMsg(msg);
		}
	}
};

/**
 * Deletes one or more items from the list. Normally, deleting an item just
 * moves it to the Trash (soft delete). However, if it's already in the Trash,
 * it will be removed from the data store (hard delete).
 *
 * @param {Hash}	params		a hash of parameters
 * @param	{Array}		params.items			list of items to delete
 * @param	{Boolean}	params.hardDelete		<code>true</code> to force physical removal of items
 * @param	{Object}	params.attrs			additional attrs for SOAP command
 * @param	{window}	params.childWin			the child window this action is happening in
 * @param	{closure}	params.finalCallback	the callback to run after all items have been processed
 * @param	{int}		params.count			the starting count for number of items processed
 * @param	{Boolean}	params.confirmDelete		the user confirmed hard delete
 */
ZmList.prototype.deleteItems =
function(params) {

	params = Dwt.getParams(arguments, ["items", "hardDelete", "attrs", "childWin"]);

	var items = params.items = AjxUtil.toArray(params.items);

	// figure out which items should be moved to Trash, and which should actually be deleted
	var toMove = [];
	var toDelete = [];
	if (params.hardDelete) {
		toDelete = items;
	} else if (items[0] && items[0] instanceof ZmItem) {
		for (var i = 0; i < items.length; i++) {
			var item = items[i];
			var folderId = item.getFolderId();
			var folder = appCtxt.getById(folderId);
			if (folder && folder.isHardDelete()) {
				toDelete.push(item);
			} else {
				toMove.push(item);
			}
		}
	} else {
		toMove = items;
	}

	if (toDelete.length && !params.confirmDelete) {
		if (appCtxt.getAppViewMgr().isBlockItemDeletionWarningDialogPoppedUp()) {
			return;
		}
		if (this.isDeletingItemOpen(toDelete)) {
			if (appCtxt.getOkCancelMsgDialog().isPoppedUp()){
				appCtxt.getOkCancelMsgDialog().popdown();
			}
			appCtxt.getAppViewMgr().popupBlockItemDeletionWarningDialog();
			return;
		}
		params.confirmDelete = true;
		var callback = ZmList.prototype.deleteItems.bind(this, params);
		this._popupDeleteWarningDialog(callback, toMove.length, toDelete.length);
		return;
	}

	params.callback = params.childWin && new AjxCallback(this._handleDeleteNewWindowResponse, params.childWin);

	// soft delete - items moved to Trash
	if (toMove.length) {
		if (appCtxt.multiAccounts) {
			var accounts = this._filterItemsByAccount(toMove);
			if (!params.callback) {
				params.callback = new AjxCallback(this, this._deleteAccountItems, [accounts, params]);
			}
			this._deleteAccountItems(accounts, params);
		}
		else {
			params.items = toMove;
			params.folder = appCtxt.getById(ZmFolder.ID_TRASH);
			this.moveItems(params);
		}
	}

	// hard delete - items actually deleted from data store
	if (toDelete.length) {
		params.items = toDelete;
		params.action = "delete";
        params.actionTextKey = 'actionDelete';
		this._itemAction(params);
	}
};

/**
 * Check if deleting item(s) is opened
 *
 * @param {Array}   toDelete   list of items to delete
 */
ZmList.prototype.isDeletingItemOpen =
function(toDelete) {
	var deletingItemIds = [];
	for (var i = 0; i < toDelete.length; i++) {
		deletingItemIds.push(toDelete[i].id);
		if (toDelete[i].msgIds) {
			for (var j = 0; j < toDelete[i].msgIds.length; j++) {
				deletingItemIds.push(toDelete[i].msgIds[j]);
			}
		}
	}

	var viewIdToIgnore;
	if (
		appCtxt.getCurrentView() instanceof ZmMailMsgView ||
		appCtxt.getCurrentView() instanceof ZmConvView2 ||
		(typeof ZmEditContactView !== 'undefined' && appCtxt.getCurrentView() instanceof ZmEditContactView)
	) {
		viewIdToIgnore = appCtxt.getCurrentView().getController().tabId;
	}
	return appCtxt.getAppViewMgr().isDeletingItemOpen(deletingItemIds, null, viewIdToIgnore);
};

ZmList.prototype._popupDeleteWarningDialog =
function(callback, onlySome, count) {
	var dialog = appCtxt.getOkCancelMsgDialog();
	dialog.reset();
	dialog.setMessage(AjxMessageFormat.format(ZmMsg[onlySome ? "confirmDeleteSomeForever" : "confirmDeleteForever"], [count]), DwtMessageDialog.WARNING_STYLE); 
	dialog.registerCallback(DwtDialog.OK_BUTTON, this._deleteWarningDialogListener.bind(this, callback, dialog));
	dialog.associateEnterWithButton(DwtDialog.OK_BUTTON);
	dialog.popup(null, DwtDialog.OK_BUTTON);
};

ZmList.prototype._deleteWarningDialogListener =
function(callback, dialog) {
	dialog.popdown();
	callback();
};


/**
 * @private
 */
ZmList.prototype._deleteAccountItems =
function(accounts, params) {
	var items;
	for (var i in accounts) {
		items = accounts[i];
		break;
	}

	if (items) {
		delete accounts[i];

        var ac = window.parentAppCtxt || window.appCtxt;
        params.accountName = ac.accountList.getAccount(i).name;
		params.items = items;
		params.folder = appCtxt.getById(ZmFolder.ID_TRASH);

		this.moveItems(params);
	}
};

/**
 * @private
 */
ZmList.prototype._filterItemsByAccount =
function(items) {
	// separate out the items based on which account they belong to
	var accounts = {};
	if (items[0] && items[0] instanceof ZmItem) {
		for (var i = 0; i < items.length; i++) {
			var item = items[i];
			var acctId = item.getAccount().id;
			if (!accounts[acctId]) {
				accounts[acctId] = [];
			}
			accounts[acctId].push(item);
		}
	} else {
		var id = appCtxt.accountList.mainAccount.id;
		accounts[id] = items;
	}

	return accounts;
};

/**
 * @private
 */
ZmList.prototype._handleDeleteNewWindowResponse =
function(childWin, result) {
	if (childWin) {
		childWin.close();
	}
};

/**
 * Applies the given list of modifications to the item.
 *
 * @param {ZmItem}	item			the item to modify
 * @param {Hash}	mods			hash of new properties
 * @param	{AjxCallback}	callback	the callback
 */
ZmList.prototype.modifyItem =
function(item, mods, callback) {
	item.modify(mods, callback);
};

// Notification handling

/**
 * Create notification.
 * 
 * @param	{Object}	node		not used
 */
ZmList.prototype.notifyCreate =
function(node) {
	var obj = eval(ZmList.ITEM_CLASS[this.type]);
	if (obj) {
		var item = obj.createFromDom(node, {list:this});
		this.add(item, this._sortIndex(item));
		this.createLocal(item);
		this._notify(ZmEvent.E_CREATE, {items: [item]});
	}
};

// Local change handling

// These generic methods allow a derived class to perform the appropriate internal changes

/**
 * Modifies the items (local).
 * 
 * @param	{Array}	items		an array of items
 * @param	{Object}	mods	a hash of properties to modify
 */
ZmList.prototype.modifyLocal 		= function(items, mods) {};

/**
 * Creates the item (local).
 * 
 * @param	{ZmItem}	item	the item to create
 */
ZmList.prototype.createLocal 		= function(item) {};

// These are not currently used; will need support in ZmItem if they are.
ZmList.prototype.flagLocal 			= function(items, flag, state) {};
ZmList.prototype.tagLocal 			= function(items, tag, state) {};
ZmList.prototype.removeAllTagsLocal = function(items) {};

// default action is to remove each deleted item from this list
/**
 * Deletes the items (local).
 * 
 * @param	{Array}	items		an array of items
 */
ZmList.prototype.deleteLocal =
function(items) {
	for (var i = 0; i < items.length; i++) {
		this.remove(items[i]);
	}
};

// default action is to remove each moved item from this list
/**
 * Moves the items (local).
 * 
 * @param	{Array}	items		an array of items
 * @param	{String}	folderId	the folder id
 */
ZmList.prototype.moveLocal = 
function(items, folderId) {
	for (var i = 0; i < items.length; i++) {
		this.remove(items[i]);
	}
};

/**
 * Performs an action on items via a SOAP request.
 *
 * @param {Hash}				params				a hash of parameters
 * @param	{Array}				params.items			a list of items to act upon
 * @param	{String}			params.action			the SOAP operation
 * @param	{Object}			params.attrs			a hash of additional attrs for SOAP request
 * @param	{AjxCallback}		params.callback			the async callback
 * @param	{closure}			params.finalCallback	the callback to run after all items have been processed
 * @param	{AjxCallback}		params.errorCallback	the async error callback
 * @param	{String}			params.accountName		the account to send request on behalf of
 * @param	{int}				params.count			the starting count for number of items processed
 * @param	{ZmBatchCommand}	batchCmd				if set, request data is added to batch request
 * @param	{boolean}			params.noUndo			true if the action is performed as an undo (not undoable)
 * @param	{boolean}			params.safeMove			true if the action wants to resolve any conflicts before completion
 */
ZmList.prototype._itemAction =
function(params, batchCmd) {

	var result = this._getIds(params.items);
	var idHash = result.hash;
	var idList = result.list;
	if (!(idList && idList.length)) {
		if (params.callback) {
			params.callback.run(new ZmCsfeResult([]));
		}
		if (params.finalCallback) {
			params.finalCallback(params);
		}
		return;
	}

	DBG.println("sa", "ITEM ACTION: " + idList.length + " items");
	var type;
	if (params.items.length == 1 && params.items[0] && params.items[0].type) {
		type = params.items[0].type;
	} else {
		type = this.type;
	}
	if (!type) { return; }

	// set accountName for multi-account to be the main "local" account since we
	// assume actioned ID's will always be fully qualified
	if (!params.accountName && appCtxt.multiAccounts) {
		params.accountName = appCtxt.accountList.mainAccount.name;
	}

	var soapCmd = ZmItem.SOAP_CMD[type] + "Request";
	var useJson = batchCmd ? batchCmd._useJson : true ;
	var request, action;
	if (useJson) {
		request = {};
		var urn = this._getActionNamespace();
		request[soapCmd] = {_jsns:urn};
		var action = request[soapCmd].action = {};
		action.op = params.action;
		for (var attr in params.attrs) {
			action[attr] = params.attrs[attr];
		}
	} else {
		request = AjxSoapDoc.create(soapCmd, this._getActionNamespace());
		action = request.set("action");
		action.setAttribute("op", params.action);
		for (var attr in params.attrs) {
			action.setAttribute(attr, params.attrs[attr]);
		}
	}
    var ac =  window.parentAppCtxt || appCtxt;
	var actionController = ac.getActionController();
	var undoPossible = !params.noUndo && (this.type != ZmItem.CONV || this.search && this.search.folderId); //bug 74169 - since the convs might not be fully loaded we might not know where the messages are moved from at all. so no undo.
	var actionLogItem = (undoPossible && actionController && actionController.actionPerformed({op: params.action, ids: idList, attrs: params.attrs})) || null;
	var respCallback = new AjxCallback(this, this._handleResponseItemAction, [params.callback, actionLogItem]);

	var params1 = {
		ids:			idList,
		idHash:			idHash,
		accountName:	params.accountName,
		request:		request,
		action:			action,
		type:			type,
		callback:		respCallback,
		finalCallback:	params.finalCallback,
		errorCallback:	params.errorCallback,
		batchCmd:		batchCmd,
		numItems:		params.count || 0,
		actionTextKey:	params.actionTextKey,
		actionArg:		params.actionArg,
		actionLogItem:	actionLogItem,
		childWin:		params.childWin,
		closeChildWin: 	params.closeChildWin,
		safeMove:		params.safeMove
	};

	if (idList.length >= ZmList.CHUNK_SIZE) {
		var pdParams = {
			state:		ZmListController.PROGRESS_DIALOG_INIT,
			callback:	new AjxCallback(this, this._cancelAction, [params1])
		}
		ZmListController.handleProgress(pdParams);
	}
	
	this._doAction(params1);
};

/**
 * @private
 */
ZmList.prototype._handleResponseItemAction =
function(callback, actionLogItem, items, result) {
	if (actionLogItem) {
		actionLogItem.setComplete();
	}
	
	if (callback) {
		result.set(items);
		callback.run(result);
	}
};

/**
 * @private
 */
ZmList.prototype._doAction =
function(params) {

	var list = params.ids.splice(0, ZmList.CHUNK_SIZE);
	var idStr = list.join(",");
	var useJson = true;
	if (params.action.setAttribute) {
		params.action.setAttribute("id", idStr);
		useJson = false;
	} else {
		params.action.id = idStr;
	}
	var more = Boolean(params.ids.length && !params.cancelled);

	var respCallback = new AjxCallback(this, this._handleResponseDoAction, [params]);
    var isOutboxFolder = this.controller && this.controller.isOutboxFolder();
    var offlineCallback = this._handleOfflineResponseDoAction.bind(this, params, isOutboxFolder);

	if (params.batchCmd) {
		params.batchCmd.addRequestParams(params.request, respCallback, params.errorCallback);
	} else {
		var reqParams = {asyncMode:true, callback:respCallback, errorCallback: params.errorCallback, offlineCallback: offlineCallback, accountName:params.accountName, more:more};
		if (useJson) {
			reqParams.jsonObj = params.request;
		} else {
			reqParams.soapDoc = params.request;
		}
		if (params.safeMove) {
			reqParams.useChangeToken = true;
		}
        if (isOutboxFolder) {
            reqParams.offlineRequest = true;
        }
		DBG.println("sa", "*** do action: " + list.length + " items");
		params.reqId = appCtxt.getAppController().sendRequest(reqParams);
	}
};

/**
 * @private
 */
ZmList.prototype._handleResponseDoAction =
function(params, result) {

	var summary;
	var response = result.getResponse();
	var resp = response[ZmItem.SOAP_CMD[params.type] + "Response"];
	if (resp && resp.action) {
		var ids = resp.action.id.split(",");
		if (ids) {
			var items = [];
			for (var i = 0; i < ids.length; i++) {
				var item = params.idHash[ids[i]];
				if (item) {
					items.push(item);
				}
			}
			params.numItems += items.length;
			if (params.callback) {
				params.callback.run(items, result);
			}

			if (params.actionTextKey) {
				summary = ZmList.getActionSummary(params);
				var pdParams = {
					state:		ZmListController.PROGRESS_DIALOG_UPDATE,
					summary:	summary
				}
				ZmListController.handleProgress(pdParams);
			}
		}
	}

	if (params.ids.length && !params.cancelled) {
		DBG.println("sa", "item action setting up next chunk, remaining: " + params.ids.length);
		AjxTimedAction.scheduleAction(new AjxTimedAction(this, this._doAction, [params]), ZmItem.CHUNK_PAUSE);
	} else {
		params.reqId = null;
		params.actionSummary = summary;
		if (params.finalCallback) {
			// finalCallback is responsible for showing status or clearing dialog
			DBG.println("sa", "item action running finalCallback");
			params.finalCallback(params);
		} else {
			DBG.println("sa", "no final callback");
			ZmListController.handleProgress({state:ZmListController.PROGRESS_DIALOG_CLOSE});
			ZmBaseController.showSummary(params.actionSummary, params.actionLogItem, params.closeChildWin);
		}
	}
};

/**
 * @private
 */
ZmList.prototype._handleOfflineResponseDoAction =
function(params, isOutboxFolder, requestParams) {

    var action = params.action,
        callback = this._handleOfflineResponseDoActionCallback.bind(this, params, isOutboxFolder, requestParams.callback);

    if (isOutboxFolder && action.op === "trash") {
        var key = {
            methodName : "SendMsgRequest", //Outbox folder only contains offline sent emails
			id : action.id.split(",")
        };
        ZmOfflineDB.deleteItemInRequestQueue(key, callback);
    }
    else {
        var obj = requestParams.jsonObj;
        obj.methodName = ZmItem.SOAP_CMD[params.type] + "Request";
        obj.id = action.id;
        ZmOfflineDB.setItem(obj, ZmOffline.REQUESTQUEUE, callback);
    }
};

/**
 * @private
 */
ZmList.prototype._handleOfflineResponseDoActionCallback =
function(params, isOutboxFolder, callback) {

    var data = {},
        header = this._generateOfflineHeader(params),
        result,
        hdr,
        notify;

    data[ZmItem.SOAP_CMD[params.type] + "Response"] = params.request[ZmItem.SOAP_CMD[params.type] + "Request"];
    result = new ZmCsfeResult(data, false, header);
    hdr = result.getHeader();
    if (callback) {
        callback.run(result);
    }
    if (hdr) {
        notify = hdr.context.notify[0];
        if (notify) {
            appCtxt._requestMgr._notifyHandler(notify);
            this._updateOfflineData(params, isOutboxFolder, notify);
        }
    }
};

/**
 * @private
 */
ZmList.prototype._generateOfflineHeader =
function(params) {

    var action = params.action,
        op = action.op,
        ids = action.id.split(","),
        idsLength = ids.length,
        id,
        msg,
        flags,
        folderId,
        folder,
        targetFolder,
        mObj,
        cObj,
        folderObj,
        m = [],
        c = [],
        folderArray = [],
        header;

    for (var i = 0; i < idsLength; i++) {

        id = ids[i];
        msg = this.getById(id);
        flags =  msg.flags || "";
        folderId = msg.getFolderId();
        folder = appCtxt.getById(folderId);
        mObj = {
            id : id
        };
        cObj = {
            id : "-" + mObj.id
        };
        folderObj = {
            id : folderId
        };

        switch (op)
        {
            case "flag":
                mObj.f = flags + "f";
                break;
            case "!flag":
                mObj.f = flags.replace("f", "");
                break;
            case "read":
                mObj.f = flags.replace("u", "");
                folderObj.u = folder.numUnread - 1;
                break;
            case "!read":
                mObj.f = flags + "u";
                folderObj.u = folder.numUnread + 1;
                break;
            case "trash":
                mObj.l = ZmFolder.ID_TRASH;
                break;
            case "spam":
                mObj.l = ZmFolder.ID_SPAM;
                break;
            case "!spam":
                mObj.l = ZmFolder.ID_INBOX;// Have to set the old folder id. Currently point to inbox
                break;
            case "move":
                if (action.l) {
                    mObj.l = action.l;
                }
                folderObj.n = folder.numTotal - 1;
                if (msg.isUnread && folder.numUnread > 1) {
                    folderObj.u = folder.numUnread - 1;
                }
                targetFolder = appCtxt.getById(mObj.l);
                folderArray.push({
                    id : targetFolder.id,
                    n : targetFolder.numTotal + 1,
                    u : (msg.isUnread ? targetFolder.numUnread + 1 : targetFolder.numUnread)
                });
                break;
            case "tag":
                msg.tags.push(action.tn);
                mObj.tn = msg.tags.join();
                break;
            case "!tag":
                AjxUtil.arrayRemove(msg.tags, action.tn);
                mObj.tn = msg.tags.join();
                break;
            case "update":
                if (action.t === "") {//Removing all tag names for a msg
                    mObj.tn = "";
                    mObj.t = "";
                }
                break;
        }
        m.push(mObj);
        c.push(cObj);
        folderArray.push(folderObj);
    }

    header = {
        context : {
            notify : [{
                modified : {
                    m : m,
                    c : c,
                    folder : folderArray
                }
            }]
        }
    };

    return header;
};

ZmList.prototype._updateOfflineData =
function(params, isOutboxFolder, notify) {

    var modified = notify.modified;
    if (!modified) {
        return;
    }

    var m = modified.m;
    if (!m) {
        return;
    }

    var callback = this._updateOfflineDataCallback.bind(this, params, m);
    ZmOfflineDB.getItem(params.action.id.split(","), ZmApp.MAIL, callback);
};

ZmList.prototype._updateOfflineDataCallback =
function(params, msgArray, result) {
    result = ZmOffline.recreateMsg(result);
    var newMsgArray = [];
    result.forEach(function(res) {
        msgArray.forEach(function(msg) {
            if (msg.id === res.id) {
                newMsgArray.push($.extend(res, msg));
            }
        });
    });
    ZmOfflineDB.setItem(newMsgArray, ZmApp.MAIL);
};

/**
 * Returns a string describing an action, intended for display as toast to tell the
 * user what they just did.
 *
 * @param   {Object}        params          hash of params:
 *          {String}        type            item type (ZmItem.*)
 *          {Number}        numItems        number of items affected
 *          {String}        actionTextKey   ZmMsg key for text string describing action
 *          {String}        actionArg       (optional) additional argument
 *
 * @return {String}     action summary
 */
ZmList.getActionSummary =
function(params) {

	var type = params.type,
		typeKey = ZmItem.MSG_KEY[type],
		typeText = ZmMsg[typeKey],
		capKey = AjxStringUtil.capitalizeFirstLetter(typeKey),
		countKey = 'type' + capKey,
		num = params.numItems,
		alternateKey = params.actionTextKey + capKey,
		text = ZmMsg[alternateKey] || ZmMsg[params.actionTextKey],
		countText = ZmMsg[countKey],
		arg = AjxStringUtil.htmlEncode(params.actionArg),
		textAuto = countText ? AjxMessageFormat.format(countText, num) : typeText,
		textSingular = countText ? AjxMessageFormat.format(ZmMsg[countKey], 1) : typeText;

	return AjxMessageFormat.format(text, [ num, textAuto, arg, textSingular ]);
};

/**
 * Cancel current server request if there is one, and set flag to
 * stop cascade of requests.
 *
 * @param {Hash}	params	a hash of parameters
 * 
 * @private
 */
ZmList.prototype._cancelAction =
function(params) {
	params.cancelled = true;
	if (params.reqId) {
		appCtxt.getRequestMgr().cancelRequest(params.reqId);
	}
	if (params.finalCallback) {
		params.finalCallback(params);
	}
	ZmListController.handleProgress({state:ZmListController.PROGRESS_DIALOG_CLOSE});
};

/**
 * @private
 */
ZmList.prototype._getTypedItems =
function(items) {
	var typedItems = {};
	for (var i = 0; i < items.length; i++) {
		var type = items[i].type;
		if (!typedItems[type]) {
			typedItems[type] = [];
		}
		typedItems[type].push(items[i]);
	}
	return typedItems;
};

/**
 * Grab the IDs out of a list of items, and return them as both a string and a hash.
 * 
 * @private
 */
ZmList.prototype._getIds =
function(list) {

	var idHash = {};
	if (list instanceof ZmItem) {
		list = [list];
	}
	
	var ids = [];
	if ((list && list.length)) {
		for (var i = 0; i < list.length; i++) {
			var item = list[i];
			var id = item.id;
			if (id) {
				ids.push(id);
				idHash[id] = item;
			}
		}
	}

	return {hash:idHash, list:ids};
};

/**
 * Returns the index at which the given item should be inserted into this list.
 * Subclasses should override to return a meaningful value.
 * 
 * @private
 */
ZmList.prototype._sortIndex = 
function(item) {
	return 0;
};

/**
 * @private
 */
ZmList.prototype._redoSearch = 
function(ctlr) {
	var sc = appCtxt.getSearchController();
	sc.redoSearch(ctlr._currentSearch);
};

/**
 * @private
 */
ZmList.prototype._getActionNamespace =
function() {
	return "urn:zimbraMail";
};

/**
 * @private
 */
ZmList.prototype._folderTreeChangeListener = 
function(ev) {
	if (ev.type != ZmEvent.S_FOLDER) return;

	var folder = ev.getDetail("organizers")[0];
	var fields = ev.getDetail("fields");
	var ctlr = appCtxt.getCurrentController();
	var isCurrentList = (appCtxt.getCurrentList() == this);

	if (ev.event == ZmEvent.E_DELETE &&
		(ev.source instanceof ZmFolder) &&
		ev.source.id == ZmFolder.ID_TRASH)
	{
		// user emptied trash - reset a bunch of stuff w/o having to redo the search
		var curView = ctlr.getListView && ctlr.getListView();
		if (curView) {
			curView.offset = 0;
		}
		ctlr._resetNavToolBarButtons(view);
	}
	else if (isCurrentList && ctlr && ctlr._currentSearch &&
			 (ev.event == ZmEvent.E_MOVE || (ev.event == ZmEvent.E_MODIFY) && fields && fields[ZmOrganizer.F_NAME]))
	{
		// on folder rename or move, update current query if folder is part of query
		if (ctlr._currentSearch.replaceFolderTerm(ev.getDetail("oldPath"), folder.getPath())) {
			appCtxt.getSearchController().setSearchField(ctlr._currentSearch.query);
		}
	}
};

/**
 * this method is for handling changes in the tag tree itself (tag rename, delete). In some places it is named _tagChangeListener.
 * the ZmListView equivalent is actually called ZmListView.prototype._tagChangeListener 
 * @private
 */
ZmList.prototype._tagTreeChangeListener =
function(ev) {
	if (ev.type != ZmEvent.S_TAG) { return; }

	var tag = ev.getDetail("organizers")[0];
	var fields = ev.getDetail("fields");
	var ctlr = appCtxt.getCurrentController();
	if (!ctlr) { return; }

	var a = this.getArray();

	if ((ev.event == ZmEvent.E_MODIFY) && fields && fields[ZmOrganizer.F_NAME]) {
		// on tag rename, update current query if tag is part of query
		var oldName = ev.getDetail("oldName");
		if (ctlr._currentSearch && ctlr._currentSearch.hasTagTerm(oldName)) {
			ctlr._currentSearch.replaceTagTerm(oldName, tag.getName());
			appCtxt.getSearchController().setSearchField(ctlr._currentSearch.query);
		}

		//since we tag (and map the tags) by name, replace the tag name in the list and hash of tags.
		var newName = tag.name;
		for (var i = 0; i < a.length; i++) {
			var item = a[i]; //not using the following here as it didn't seem to work for contacts, the list is !isCanonical and null is returned, even though a[i] is fine ==> this.getById(a[i].id); // make sure item is realized (contact may not be)
			if (!item || !item.isZmItem || !item.hasTag(oldName)) {
				continue; //nothing to do if item does not have tag
			}
			if (item.isShared()) {
				continue; //overview tag rename does not affect remote items tags
			}
			var tagHash = item.tagHash;
			var tags = item.tags;
			delete tagHash[oldName];
			tagHash[newName] = true;
			for (var j = 0 ; j < tags.length; j++) {
				if (tags[j] == oldName) {
					tags[j] = newName;
					break;
				}
			}
		}


	} else if (ev.event == ZmEvent.E_DELETE) {
		// Remove tag from any items that have it
		var hasTagListener = this._evtMgr.isListenerRegistered(ZmEvent.L_MODIFY);
		for (var i = 0; i < a.length; i++) {
			var item = this.getById(a[i].id);	// make sure item is realized (contact may not be)
            if (item) {
                if (item.isShared()) {
                    continue; //overview tag delete does not affect remote items tags
                }
                if (item.hasTag(tag.name)) {
                    item.tagLocal(tag.name, false);
                    if (hasTagListener) {
                        this._notify(ZmEvent.E_TAGS, {items:[item]});
                    }
                }
            }
		}

		// If search results are based on this tag, keep them around so that user can still
		// view msgs or open convs, but disable pagination and sorting since they're based
		// on the current query.
		if (ctlr._currentSearch && ctlr._currentSearch.hasTagTerm(tag.getName())) {
			var viewId = appCtxt.getCurrentViewId();
			var viewType = appCtxt.getCurrentViewType();
			ctlr.enablePagination(false, viewId);
			var view = ctlr.getListView && ctlr.getListView();
			if (view && view.sortingEnabled) {
				view.sortingEnabled = false;
			}
			if (viewType == ZmId.VIEW_CONVLIST) {
				ctlr._currentSearch.query = "is:read is:unread";
			}
			ctlr._currentSearch.tagId = null;
			appCtxt.getSearchController().setSearchField("");
		}
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmAccountList")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines a list of accounts.
 *
 */

/**
 * Creates the account list.
 * @class
 * This class is used to store and manage a list of accounts for a mailbox.
 *
 * @author Parag Shah
 */
ZmAccountList = function() {
	this._accounts = {};
	this._count = 0;
	this.visibleAccounts = [];
	this.mainAccount = null;
	this.activeAccount = null;
	this.defaultAccount = null; // the first non-main account.

	this._evtMgr = new AjxEventMgr();
};

ZmAccountList.prototype.constructor = ZmAccountList;


// Consts

ZmAccountList.DEFAULT_ID = "main";


// Public methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAccountList.prototype.toString =
function() {
	return "ZmAccountList";
};

/**
 * Gets the number of visible accounts for this mailbox.
 *
 * @param	{Boolean}	includeInvisible	if <code>true</code>, include the number of invisible accounts for this mailbox
 * @return	{int}							the number of accounts for this mailbox
 */
ZmAccountList.prototype.size =
function(includeInvisible) {
	return (includeInvisible) ? this._count : this.visibleAccounts.length;
};

/**
 * Adds the account.
 * 
 * @param	{ZmAccount}	account		the account
 */
ZmAccountList.prototype.add =
function(account) {
	this._accounts[account.id] = account;
	this._count++;

	if (account.visible || account.id == ZmAccountList.DEFAULT_ID) {
		this.visibleAccounts.push(account);
	}

	if (account.id == ZmAccountList.DEFAULT_ID) {
		this.mainAccount = account;
	}
};

/**
 * Gets the accounts.
 * 
 * @return	{Array}	an array of {ZmAccount} objects
 */
ZmAccountList.prototype.getAccounts =
function() {
	return this._accounts;
};

/**
 * Gets the account by id.
 * 
 * @param	{String}	id		the id
 * @return	{ZmAccount}	the account
 */
ZmAccountList.prototype.getAccount =
function(id) {
	return id ? this._accounts[id] : this.mainAccount;
};

/**
 * Gets the account by name.
 * 
 * @param	{String}	name	the name
 * @return	{ZmAccount}	the account
 */
ZmAccountList.prototype.getAccountByName =
function(name) {
	for (var i in this._accounts) {
		if (this._accounts[i].name == name) {
			return this._accounts[i];
		}
	}
	return null;
};

/**
 * Gets the account by email.
 * 
 * @param	{String}	email	the email
 * @return	{ZmAccount}	the account
 */
ZmAccountList.prototype.getAccountByEmail =
function(email) {
	for (var i in this._accounts) {
		if (this._accounts[i].getEmail() == email) {
			return this._accounts[i];
		}
	}
	return null;
};

/**
 * Gets the cumulative item count of all accounts for the given folder ID.
 *
 * @param {String}	folderId		the folder id
 * @param {Boolean}	checkUnread		if <code>true</code>, checks the unread count instead of item count
 * @return	{int}	the item count
 */
ZmAccountList.prototype.getItemCount =
function(folderId, checkUnread) {
	var count = 0;
	for (var i = 0; i < this.visibleAccounts.length; i++) {
		var acct = this.visibleAccounts[i];
		if (acct.isMain) { continue; } // local account should never have drafts

		var fid = ZmOrganizer.getSystemId(folderId, acct);
		var folder = appCtxt.getById(fid);
		if (folder) {
			count += (checkUnread ? folder.numUnread : folder.numTotal);
		}
	}

	return count;
};

/**
 * Generates a query.
 * 
 * @param {String}	folderId		the folder id
 * @param	{Array}	types		the types
 * @return	{String}	the query
 */
ZmAccountList.prototype.generateQuery =
function(folderId, types) {
	// XXX: for now, let's just search for *one* type at a time
	var type = types && types.get(0);
	var query = [];
	var list = this.visibleAccounts;
	var fid = folderId || ZmOrganizer.ID_ROOT;
	var syntax = folderId ? "inid" : "underid";
	for (var i = 0; i < list.length; i++) {
		var acct = list[i];

		// dont add any apps not supported by this account
		if ((type && !acct.isAppEnabled(ZmItem.APP[type])) || acct.isMain) { continue; }

		var part = [syntax, ':"', ZmOrganizer.getSystemId(fid, acct, true), '"'];
		query.push(part.join(""));
	}
    if(fid == ZmOrganizer.ID_ROOT) {
        query.push([syntax, ':"', appCtxt.accountList.mainAccount.id, ':', fid, '"'].join(""));
    }
	DBG.println(AjxDebug.DBG2, "query = " + query.join(" OR "));
	return (query.join(" OR "));
};

/**
 * Loads each visible account serially by requesting the following requests from
 * the server in a batch request:
 * 
 * <ul>
 * <li><code><GetInfoRequest></code></li>
 * <li><code><GetTafReqyuest></code></li>
 * <li><code><GetFolderRequest></code></li>
 * </ul>
 * 
 * @param {AjxCallback}	callback		the callback to trigger once all accounts have been loaded
 */
ZmAccountList.prototype.loadAccounts =
function(callback) {
	var list = (new Array()).concat(this.visibleAccounts);
	this._loadAccount(list, callback);
};

/**
 * @private
 */
ZmAccountList.prototype._loadAccount =
function(accounts, callback) {
	var acct = accounts.shift();
	if (acct) {
		var respCallback = new AjxCallback(this, this._loadAccount, [accounts, callback]);
		acct.load(respCallback);
	} else {
		// do any post account load initialization
		ZmOrganizer.HIDE_EMPTY[ZmOrganizer.TAG] = true;
		ZmOrganizer.HIDE_EMPTY[ZmOrganizer.SEARCH] = true;

		// enable compose based on whether at least one account supports smtp
		for (var i = 0; i < this.visibleAccounts.length; i++) {
			if (appCtxt.get(ZmSetting.OFFLINE_SMTP_ENABLED, null, this.visibleAccounts[i])) {
				appCtxt.set(ZmSetting.OFFLINE_COMPOSE_ENABLED, true, null, null, true);
				break;
			}
		}

		if (callback) {
			callback.run();
		}
	}
};

/**
 * Sets the given account as the active one, which will then be used when fetching
 * any account-specific data such as settings or folder tree.
 *
 * @param {ZmZimbraAccount}	account		the account to make active
 * @param {Boolean}	skipNotify		if <code>true</code>, skip notify
 */
ZmAccountList.prototype.setActiveAccount =
function(account, skipNotify) {
	this.activeAccount = account;

	this._evt = this._evt || new ZmEvent();
	this._evt.account = account;

	if (!skipNotify) {
		this._evtMgr.notifyListeners("ACCOUNT", this._evt);
	}
};

/**
 * Adds an active account listener.
 * 
 * @param	{AjxListener}	listener		the listener
 * @param	{int}	index		the index where to insert the listener
 */
ZmAccountList.prototype.addActiveAcountListener =
function(listener, index) {
	return this._evtMgr.addListener("ACCOUNT", listener, index);
};

/**
 * Checks if any of the non-main, visible accounts is currently doing an initial sync.
 * 
 * @return	{Boolean}	<code>true</code> if any of the non-main accounts are doing initial sync
 */
ZmAccountList.prototype.isInitialSyncing =
function() {
	for (var i = 0; i < this.visibleAccounts.length; i++) {
		var acct = this.visibleAccounts[i];
		if (acct.isMain) { continue; }

		if (acct.isOfflineInitialSync()) {
			return true;
		}
	}

	return false;
};

/**
 * Returns true if any of the visible accounts have the given status
 *
 * @param 	{String}		status 		Status to check for
 */
ZmAccountList.prototype.isSyncStatus =
function(status) {
	for (var i = 0; i < this.visibleAccounts.length; i++) {
		var acct = this.visibleAccounts[i];
		if (acct.isMain) { continue; }

		if (acct.status == status) {
			return true;
		}
	}

	return false;
};

/**
 * Checks if there is at least one of the given account types in the
 * account list. Note: if the given account type is ZCS, the local parent
 * account is NOT included when searching the account list.
 *
 * @param {String}	type	the type of account to check
 * @return	{Boolean}	<code>true</code> if the account exists
 */
ZmAccountList.prototype.accountTypeExists =
function(type) {
	for (var i = 0; i < this.visibleAccounts.length; i++) {
		var acct = this.visibleAccounts[i];
		if (type == ZmAccount.TYPE_ZIMBRA && acct.isMain) { continue; }
		if (acct.type == type) { return true; }
	}

	return false;
};

/**
 * Syncs all visible accounts.
 * 
 * @param	{AjxCallback}	callback		the callback
 */
ZmAccountList.prototype.syncAll =
function(callback) {
	var list = (new Array()).concat(this.visibleAccounts);
	this._sendSync(list, callback);
};

/**
 * @private
 */
ZmAccountList.prototype._sendSync =
function(accounts, callback) {
	var acct = accounts.shift();
	if (acct) {
		if (!acct.isMain) { // skip the main account
			acct.sync();
		}
		AjxTimedAction.scheduleAction(new AjxTimedAction(this, this._sendSync, [accounts, callback]), 500);
	} else {
		if (callback) {
			callback.run();
		}
	}
};

/**
 * Creates the main account and all its children. In the normal case, the "main"
 * account is the only account, and represents the user who logged in. If family
 * mailbox is enabled, that account is a parent account with dominion over child
 * accounts. If offline, the main account is the "local" account.
 *
 * @param {ZmSettings}	settings	the settings for the main account
 * @param {Object}	obj		the JSON obj containing meta info about the main account and its children
 */
ZmAccountList.prototype.createAccounts =
function(settings, obj) {
	// first, replace the dummy main account with real information
	var account = appCtxt.accountList.mainAccount;
	account.id = obj.id;
	account.name = obj.name;
	account.isMain = true;
	account.isZimbraAccount = true;
	account.loaded = true;
	account.visible = true;
	account.settings = settings;
	account.type = ZmAccount.TYPE_ZIMBRA;
	account.icon = "AccountZimbra";
	account.active = true; // always set active for main/parent account

	this._accounts[account.id] = account;
	delete this._accounts[ZmAccountList.DEFAULT_ID];

	this.setActiveAccount(account);

	if (appCtxt.isOffline) {
		account.displayName = ZmMsg.localFolders;
	}

	// second, create all child accounts if applicable
	var childAccounts = obj.childAccounts && obj.childAccounts.childAccount;
	if (childAccounts) {
		for (var i = 0; i < childAccounts.length; i++) {
			this.add(ZmZimbraAccount.createFromDom(childAccounts[i]));
		}

		// set global vars per number of child accounts
		appCtxt.multiAccounts = this.size() > 1;
		appCtxt.isFamilyMbox = appCtxt.multiAccounts && !appCtxt.isOffline;

		this.defaultAccount = appCtxt.isFamilyMbox ? this.mainAccount : this.visibleAccounts[1];
	}
};

/**
 * Resets the trees.
 * 
 */
ZmAccountList.prototype.resetTrees =
function() {
	for (var i = 0; i < this.visibleAccounts.length; i++) {
		for (var type in trees) {
			var tree = trees[type];
			if (tree && tree.reset) {
				tree.reset();
			}
		}
	}
};

/**
 * Saves the implicit preferences on the visible accounts.
 * 
 */
ZmAccountList.prototype.saveImplicitPrefs =
function() {
	for (var i = 0; i < this.visibleAccounts.length; i++) {
		this.visibleAccounts[i].saveImplicitPrefs();
	}
};

/**
 * Gets the tool tip for the folder.
 * 
 * @param	{String}	folderId	the folder id
 * @return	{String}	the tool tip
 */
ZmAccountList.prototype.getTooltipForVirtualFolder =
function(folderId) {
	var numTotal = 0;
	var sizeTotal = 0;

	for (var i = 0; i < this.visibleAccounts.length; i++) {
		var acct = this.visibleAccounts[i];
		var fid = ZmOrganizer.getSystemId(folderId, acct);
		var folder = appCtxt.getById(fid);
		if (folder) {
			numTotal += folder.numTotal;
			sizeTotal += folder.sizeTotal;
		}
	}

	var subs = {
		itemText: ZmMsg.messages,
		numTotal: numTotal,
		sizeTotal: sizeTotal
	};

	return AjxTemplate.expand("share.App#FolderTooltip", subs);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmAccount")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2007, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2007, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines the account base class.
 *
 */

/**
 * Creates the account.
 * @class
 * This class represents an account.
 * 
 * @param	{constant}	type	the account type (see <code>ZmAccount.TYPE_</code> constants)
 * @param	{String}	id		the account id
 * @param	{String}	name	the account name
 * @see		ZmAccount
 */
ZmAccount = function(type, id, name) {
	if (arguments.length == 0) { return; }

	this.id = id;
	this.name = name;
	this.type = type || ZmAccount.TYPE_ZIMBRA;
};


//
// Consts
//

/**
 * Defines the "AOL" account type.
 */
ZmAccount.TYPE_AOL		= "AOL";
/**
 * Defines the "Gmail" account type.
 */
ZmAccount.TYPE_GMAIL	= "Gmail";
/**
 * Defines the "IMAP" account type.
 */
ZmAccount.TYPE_IMAP		= "Imap";
/**
 * Defines the "Microsoft Live" or "Hotmail" account type.
 */
ZmAccount.TYPE_LIVE		= "Live";   // MS Live / hotmail
/**
 * Defines the "Microsoft Exchange IMAP" account type.
 */
ZmAccount.TYPE_MSE		= "MSE";    // exchange IMAP
/**
 * Defines the "Microsoft Exchange Mobile Sync" account type.
 */
ZmAccount.TYPE_EXCHANGE = "Xsync";  // exchange (using mobile sync protocol)
/**
 * Defines the "persona" account type.
 */
ZmAccount.TYPE_PERSONA	= "PERSONA";
/**
 * Defines the "POP" account type.
 */
ZmAccount.TYPE_POP		= "Pop";
/**
 * Defines the "Y! Mail" account type.
 */
ZmAccount.TYPE_YMP		= "YMP";    // Y! mail
/**
 * Defines the "Zimbra" account type.
 */
ZmAccount.TYPE_ZIMBRA	= "Zimbra";
/**
 * Defines the "Zimbra" account type.
 */
ZmAccount.TYPE_CALDAV	= "CalDAV";


ZmAccount.LOCAL_ACCOUNT_ID = "ffffffff-ffff-ffff-ffff-ffffffffffff";


//
// Public static methods
//

/**
 * Gets the name of the specified type.
 * 
 * @param	{constant}	type		the type (see <code>ZmAccount.TYPE_</code> constants)
 * @return	{String}	the name or unknown
 * 
 * @see		ZmAccount
 */
ZmAccount.getTypeName =
function(type) {
	switch (type) {
		case ZmAccount.TYPE_AOL:		return ZmMsg.aol;
		case ZmAccount.TYPE_GMAIL:		return ZmMsg.gmail;
		case ZmAccount.TYPE_IMAP:		return ZmMsg.accountTypeImap;
		case ZmAccount.TYPE_LIVE:		return ZmMsg.msLive;
		case ZmAccount.TYPE_MSE:		return ZmMsg.msExchange;
		case ZmAccount.TYPE_EXCHANGE:	return ZmMsg.msExchange;
		case ZmAccount.TYPE_PERSONA:	return ZmMsg.accountTypePersona;
		case ZmAccount.TYPE_POP:		return ZmMsg.accountTypePop;
		case ZmAccount.TYPE_YMP:		return ZmMsg.yahooMail;
		case ZmAccount.TYPE_ZIMBRA:		return ZmMsg.zimbraTitle;
	}
	return ZmMsg.unknown;
};


//
// Public methods
//

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAccount.prototype.toString =
function() {
	return "ZmAccount";
};

/**
 * Sets the name of the account.
 * 
 * @param		{String}	name		the account name
 */
ZmAccount.prototype.setName =
function(name) {
	this.name = name;
};

/**
 * Gets the name of the account.
 * 
 * @return		{String}		the account name
 */
ZmAccount.prototype.getName =
function() {
	return this.name;
};

// sub-classes MUST override these methods

/**
 * Sets the email address for this account. Subclasses should override this method.
 * 
 * @param	{String}	email 	the email address
 */
ZmAccount.prototype.setEmail =
function(email) {
	throw this.toString()+"#setEmail";
};

/**
 * Gets the email address for this account. Subclasses should override this method.
 * 
 * @return	{String}	the email address
 */
ZmAccount.prototype.getEmail =
function() {
	throw this.toString()+"#getEmail";
};

/**
 * Gets the identity for this account. Subclasses should override this method.
 * 
 * @return	{String}	the identity
 */
ZmAccount.prototype.getIdentity =
function() {
	throw this.toString()+"#getIdentity";
};

ZmAccount.prototype.isLocal =
function() {
	return this.id == ZmAccount.LOCAL_ACCOUNT_ID;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmZimbraAccount")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the zimbra account class.
 */

/**
 * Creates an account object containing meta info about the account.
 * @class
 * This class represents an account. This object is created primarily if a user has added sub-accounts
 * to manage (i.e. a family mailbox).
 *
 * @author Parag Shah
 *
 * @param {String}		id			the unique ID for this account
 * @param {String}		name		the email address
 * @param {Boolean}		visible		if <code>true</code>, make this account available in the overview (i.e. child accounts)
 *
 * @extends	ZmAccount
 */
ZmZimbraAccount = function(id, name, visible) {

	ZmAccount.call(this, null, id, name);

	this.visible = (visible !== false);
	/**
	 * The account settings.
	 * @type	ZmSettings
	 */
	this.settings = null;
	this.trees = {};
	this.loaded = false;
	/**
	 * The account Access Control List.
	 * @type	ZmAccessControlList
	 */
	this.acl = new ZmAccessControlList();
	this.metaData = new ZmMetaData(this);
};

ZmZimbraAccount.prototype = new ZmAccount;
ZmZimbraAccount.prototype.constructor = ZmZimbraAccount;

ZmZimbraAccount.prototype.isZmZimbraAccount = true;
ZmZimbraAccount.prototype.toString = function() { return "ZmZimbraAccount"; };


//
// Constants
//

/**
 * Defines the "unknown" status.
 */
ZmZimbraAccount.STATUS_UNKNOWN	= "unknown";
/**
 * Defines the "offline" status.
 */
ZmZimbraAccount.STATUS_OFFLINE	= "offline";
/**
 * Defines the "online" status.
 */
ZmZimbraAccount.STATUS_ONLINE	= "online";
/**
 * Defines the "running" status.
 */
ZmZimbraAccount.STATUS_RUNNING	= "running";
/**
 * Defines the "authentication fail" status.
 */
ZmZimbraAccount.STATUS_AUTHFAIL	= "authfail";
/**
 * Defines the "error" status.
 */
ZmZimbraAccount.STATUS_ERROR	= "error";

//
// Public methods
//

/**
 * Sets the name of the account.
 * 
 * @param		{String}	name	the account name
 */
ZmZimbraAccount.prototype.setName =
function(name) {
	var identity = this.getIdentity();
	// TODO: If no identity and name is set, should create one!
	if (!identity) return;
	identity.name = name;
};

/**
 * Gets the name of the account.
 * 
 * @return		{String}		the account name
 */
ZmZimbraAccount.prototype.getName =
function() {
	var identity = this.getIdentity();
	var name = (!identity)
		? this.settings.get(ZmSetting.DISPLAY_NAME)
		: identity.name;

	if (!name) {
		name = this.getDisplayName();
	}
	return identity.isDefault && name == ZmIdentity.DEFAULT_NAME ? ZmMsg.accountDefault : name;
};

/**
 * Sets the email address for this account. This method does nothing. The email address is set
 * when the object is created.
 * 
 * @param	{String}	email 	the email address (ignored)
 */
ZmZimbraAccount.prototype.setEmail =
function(email) {}; // IGNORE

/**
 * Gets the email address for this account.
 * 
 * @return	{String}	the email address
 */
ZmZimbraAccount.prototype.getEmail =
function() {
	return this.name;
};

/**
 * Gets the display name.
 * 
 * @return	{String}	the display name
 */
ZmZimbraAccount.prototype.getDisplayName =
function() {
	if (!this.displayName) {
		var dispName = this.isMain
			? this.settings.get(ZmSetting.DISPLAY_NAME)
			: this._displayName;
		this.displayName = (this._accountName || dispName || this.name);
	}
	return this.displayName;
};

/**
 * Gets the identity.
 * 
 * @return	{ZmIdentity}	the identity
 */
ZmZimbraAccount.prototype.getIdentity =
function() {
	var defaultIdentity = appCtxt.getIdentityCollection(this).defaultIdentity;
	if (!appCtxt.isFamilyMbox || this.isMain) {
		return defaultIdentity;
	}

	// for family mbox, create dummy identities for child accounts
	if (!this.dummyIdentity) {
		this.dummyIdentity = new ZmIdentity(this.name);
		AjxUtil.hashUpdate(this.dummyIdentity, defaultIdentity, true, ["name","isDefault"]);
	}
	return this.dummyIdentity;
};

/**
 * Gets the tool tip.
 * 
 * @return	{String}		the tool tip
 */
ZmZimbraAccount.prototype.getToolTip =
function() {
	if (this.status || this.lastSync || this.isMain) {
		var lastSyncDate = (this.lastSync && this.lastSync != 0)
			? (new Date(parseInt(this.lastSync))) : null;

		var quota = appCtxt.get(ZmSetting.QUOTA_USED, null, this);
		var lastSync;
		if (!lastSyncDate) {
			// this means, we've synced but server lost the last sync timestamp
			if (quota > 0 && !this.isMain) {
				lastSync = ZmMsg.unknown;
			}
		} else {
			lastSync = AjxDateUtil.computeWordyDateStr(new Date(), lastSyncDate);
		}

		var params = {
			lastSync: lastSync,
			hasNotSynced: this.hasNotSynced(),
			status: this.getStatusMessage(),
			quota: AjxUtil.formatSize(quota, false, 1)
		};

		return AjxTemplate.expand("share.App#ZimbraAccountTooltip", params);
	}
	return "";
};

/**
 * Gets the default color.
 * 
 * @return	{String}		the default color
 * @see		ZmOrganizer
 */
ZmZimbraAccount.prototype.getDefaultColor =
function() {
	if (this.isMain) {
		return ZmOrganizer.C_GRAY;
	}

	switch (this.type) {
		case ZmAccount.TYPE_GMAIL:		return ZmOrganizer.C_RED;
		case ZmAccount.TYPE_MSE:		return ZmOrganizer.C_GREEN;
		case ZmAccount.TYPE_EXCHANGE:	return ZmOrganizer.C_GREEN;
		case ZmAccount.TYPE_YMP:		return ZmOrganizer.C_PURPLE;
	}

	return null;
};

/**
 * Checks if the account has sync'd.
 * 
 * @return	{Boolean}	if <code>true</code>, this account has never been sync'd
 */
ZmZimbraAccount.prototype.hasNotSynced =
function() {
	return (this.isOfflineInitialSync() && 
			this.status == ZmZimbraAccount.STATUS_UNKNOWN &&
			appCtxt.get(ZmSetting.QUOTA_USED, null, this) == 0);
};

/**
 * Check is this account is currently sync'ing for the first time.
 * 
 * @return	{Boolean}	if <code>true</code>, this account is currently sync'ing for the first time
 */
ZmZimbraAccount.prototype.isOfflineInitialSync =
function() {
	return (appCtxt.isOffline && (!this.lastSync || (this.lastSync && this.lastSync == 0)));
};

/**
 * Checks if this account is CalDAV based.
 * 
 * @return	{Boolean}	if <code>true</code>, account is CalDAV based
 */
ZmZimbraAccount.prototype.isCalDavBased =
function() {
	return (this.type == ZmAccount.TYPE_GMAIL ||
			this.type == ZmAccount.TYPE_YMP);
};

/**
 * Gets the default calendar. For CalDAV based accounts, the default calendar is hidden;
 * therefore, this method returns the first non-default calendar.
 * 
 * @return	{Object}		the calendar
 * @see		ZmZimbraAccount.isCalDavBased
 */
ZmZimbraAccount.prototype.getDefaultCalendar =
function() {
	var tree = appCtxt.getFolderTree(this);
	if (this.isCalDavBased()) {
		var calendars = tree.getByType(ZmOrganizer.CALENDAR);
		for (var i = 0; i < calendars.length; i++) {
			if (calendars[i].nId == ZmOrganizer.ID_CALENDAR) { continue; }
			return calendars[i];
		}
	}
	return tree.getById(ZmOrganizer.ID_CALENDAR);
};

/**
 * Updates the account status.
 * 
 * @private
 */
ZmZimbraAccount.prototype.updateState =
function(acctInfo) {
	if (this.isMain) { return; } // main account doesn't sync

	// update last sync timestamp
	var updateTooltip = false;
	if (this.lastSync != acctInfo.lastsync) {
		this.lastSync = acctInfo.lastsync;
		if (this.visible) {
			updateTooltip = true;
		}
	}

	// set to update account (offline) status if changed
	var updateStatus = false;
	if (this.status != acctInfo.status) {
		this.status = acctInfo.status;
		if (this.visible) {
			updateStatus = true;
		}
	}

	// for all overview containers, update status/tooltip
	var container = appCtxt.getOverviewController()._overviewContainer;
	for (var i in container) {
		var c = container[i];
		if (updateStatus || updateTooltip) {
			c.updateAccountInfo(this, updateStatus, updateTooltip);
		}
	}

	if (this.visible && acctInfo.unread != this.unread) {
		this.unread = acctInfo.unread;
	}

	this.code = acctInfo.code;
	if (acctInfo.error) {
		var error = acctInfo.error[0];
		this.errorDetail = error.exception[0]._content;
		this.errorMessage = error.message;
	}
};

/**
 * Gets the status icon.
 * 
 * @return	{String}	the status icon
 */
ZmZimbraAccount.prototype.getStatusIcon =
function() {
	if (this.inNewMailMode) {
		return "NewMailAlert";
	}

	switch (this.status) {
//		case ZmZimbraAccount.STATUS_UNKNOWN:	return "Offline"; 				// bug: 42403 - remove
		case ZmZimbraAccount.STATUS_OFFLINE:	return "ImAway";
//		case ZmZimbraAccount.STATUS_ONLINE:		return "";						// no icon for "online"
//		case ZmZimbraAccount.STATUS_RUNNING:	// animated, so cannot be set using AjxImg
		case ZmZimbraAccount.STATUS_AUTHFAIL:	return "ImDnd";
		case ZmZimbraAccount.STATUS_ERROR:		return "Critical";
	}
	return null;
};

/**
 * Checks if this account is in error status.
 * 
 * @return	{Boolean}	if <code>true</code>, the account is in error status
 */
ZmZimbraAccount.prototype.isError =
function() {
	return (this.status == ZmZimbraAccount.STATUS_AUTHFAIL ||
			this.status == ZmZimbraAccount.STATUS_ERROR);
};

/**
 * Gets the icon.
 * 
 * @return	{String}	the icon
 */
ZmZimbraAccount.prototype.getIcon =
function() {
	return (this.isMain && appCtxt.isOffline) ? "LocalFolders" : this.icon;
};

/**
 * Gets the Zd message.
 * 
 * @private
 */
ZmZimbraAccount.prototype.getZdMsg =
function(code) {
	var msg = ((ZdMsg["client." + code]) || (ZdMsg["exception." + code]));
	if (!msg && code) {
		msg = ZdMsg["exception.offline.UNEXPECTED"];
	}
	return msg;
};

/**
 * Gets the status message.
 * 
 * @return	{String}		the status message
 */
ZmZimbraAccount.prototype.getStatusMessage =
function() {
	if (this.inNewMailMode) {
		return AjxMessageFormat.format(ZmMsg.unreadCount, this.unread);
	}

	switch (this.status) {
//		case ZmZimbraAccount.STATUS_UNKNOWN:	return ZmMsg.unknown;
		case ZmZimbraAccount.STATUS_OFFLINE:	return ZmMsg.netStatusOffline;
		case ZmZimbraAccount.STATUS_ONLINE:		return ZmMsg.netStatusOnline;
		case ZmZimbraAccount.STATUS_RUNNING:	return ZmMsg.running;
		case ZmZimbraAccount.STATUS_AUTHFAIL:	return this.code ? this.getZdMsg(this.code) : AjxMessageFormat.format(ZmMsg.authFailure, this.getEmail());
		case ZmZimbraAccount.STATUS_ERROR:		return this.code ? this.getZdMsg(this.code) : ZmMsg.error;
	}
	return "";
};

/**
 * Shows an error message.
 * 
 * Offline use only.
 * 
 * @private
 */
ZmZimbraAccount.prototype.showErrorMessage =
function() {
	if (!this.isError()) { return; }

	var dialog = (this.status == ZmZimbraAccount.STATUS_ERROR)
		? appCtxt.getErrorDialog() : appCtxt.getMsgDialog();

	// short message
	var msg = this.getZdMsg(this.code);
	if (msg == "") {
		msg = this.getStatusMessage();
	}
	dialog.setMessage(msg);

	if (this.status == ZmZimbraAccount.STATUS_ERROR) {
		// detailed message
		var html = [];
		var i = 0;
		if (this.errorMessage) {
			html[i++] = "<p><b>";
			html[i++] = ZdMsg.DebugMsg;
			html[i++] = "</b>: ";
			html[i++] = this.errorMessage;
			html[i++] = "</p>";
		}

		if (this.errorDetail) {
			html[i++] = "<p><b>";
			html[i++] = ZdMsg.DebugStack;
			html[i++] = "</b>:</p><p><pre>";
			html[i++] = this.errorDetail;
			html[i++] = "</pre></p>";
		}

		html[i++] = "<p><b>";
		html[i++] = ZdMsg.DebugActionNote;
		html[i++] = "</b></p>";

		dialog.setDetailString(html.join(""));
	}

	dialog.popup(null, true);
};

/**
 * @private
 */
ZmZimbraAccount.createFromDom =
function(node) {
	var acct = new ZmZimbraAccount();
	acct._loadFromDom(node);
	return acct;
};

/**
 * Loads the account.
 * 
 * @param	{AjxCallback}	callback		the callback
 */
ZmZimbraAccount.prototype.load =
function(callback) {
	if (!this.loaded) {
		// create new ZmSetting for this account
		this.settings = new ZmSettings();

		// check "{APP}_ENABLED" state against main account's settings
		var mainAcct = appCtxt.accountList.mainAccount;

		// for all *loaded* apps, add their app-specific settings
		for (var i = 0; i < ZmApp.APPS.length; i++) {
			var appName = ZmApp.APPS[i];
			var setting = ZmApp.SETTING[appName];
			if (setting && appCtxt.get(setting, null, mainAcct)) {
				var app = appCtxt.getApp(appName);
				if (app) {
					app._registerSettings(this.settings);
				}
			}
		}

		var command = new ZmBatchCommand(null, this.name);

		// load user settings retrieved from server now
		var loadCallback = new AjxCallback(this, this._handleLoadSettings);
		this.settings.loadUserSettings(loadCallback, null, this.name, null, command);

		// get tag info for this account *FIRST* - otherwise, root ID get overridden
		var tagDoc = AjxSoapDoc.create("GetTagRequest", "urn:zimbraMail");
		var tagCallback = new AjxCallback(this, this._handleLoadTags);
		command.addNewRequestParams(tagDoc, tagCallback);

		// get meta data for this account
		this.loadMetaData(null, command);

		// get folder info for this account
		var folderDoc = AjxSoapDoc.create("GetFolderRequest", "urn:zimbraMail");
		folderDoc.getMethod().setAttribute("visible", "1");
		var folderCallback = new AjxCallback(this, this._handleLoadFolders);
		command.addNewRequestParams(folderDoc, folderCallback);

		var respCallback = new AjxCallback(this, this._handleLoadUserInfo, callback);
		var errCallback = new AjxCallback(this, this._handleErrorLoad, callback);
		command.run(respCallback, errCallback);
	}
	else if (callback) {
		callback.run();
	}
};

ZmZimbraAccount.prototype.loadMetaData =
function(callback, batchCommand) {
	var metaDataCallback = new AjxCallback(this, this._handleLoadMetaData, [callback]);
	var sections = [ZmSetting.M_IMPLICIT, ZmSetting.M_OFFLINE];
	this.metaData.load(sections, metaDataCallback, batchCommand);
};

/**
 * Unloads the account and removes any account-specific data stored globally.
 * 
 */
ZmZimbraAccount.prototype.unload =
function() {
	if (!appCtxt.inStartup) {
		// unset account-specific shortcuts
		this.settings.loadShortcuts(true);
	}
};

/**
 * Sync the account.
 * 
 * @param	{AjxCallback}	callback		the callback
 */
ZmZimbraAccount.prototype.sync =
function(callback) {
	var soapDoc = AjxSoapDoc.create("SyncRequest", "urn:zimbraOffline");
	if (appCtxt.get(ZmSetting.OFFLINE_DEBUG_TRACE)) {
		var method = soapDoc.getMethod();
		method.setAttribute("debug", 1);
	}
	appCtxt.getAppController().sendRequest({
		soapDoc:soapDoc,
		asyncMode:true,
		noBusyOverlay:true,
		callback:callback,
		accountName:this.name
	});
};

/**
 * Saves the account.
 * 
 * @param	{AjxCallback}	callback		the callback
 * @param	{AjxCallback}	errorCallback		the error callback
 * @param	{Object}	batchCmd		the batch command
 */
ZmZimbraAccount.prototype.save =
function(callback, errorCallback, batchCmd) {
	return (this.getIdentity().save(callback, errorCallback, batchCmd));
};

/**
 * Saves implicit prefs. Because it's done onunload, the save is sync.
 * 
 * @private
 */
ZmZimbraAccount.prototype.saveImplicitPrefs =
function() {
    var isExternal = this.settings ? this.settings.get(ZmSetting.IS_EXTERNAL) : false;
    if (isExternal) {
        return;
    }
	var list = [];
	for (var id in ZmSetting.CHANGED_IMPLICIT) {
		var setting = this.settings ? this.settings.getSetting(id) : null;
		if (ZmSetting.IS_GLOBAL[setting.id] && !this.isMain) { continue; }
		if (setting && (setting.getValue(null, true) != setting.getOrigValue(null, true))) {
			list.push(setting);
		}
	}

	if (list.length > 0) {
		this.settings.save(list, null, null, this);
	}
};

/**
 * Checks if this account supports the given application name
 *
 * @param {String}		appName		the name of the application
 * @return	{Boolean}	<code>true</code> if account supports the application
 */
ZmZimbraAccount.prototype.isAppEnabled =
function(appName) {
	switch (appName) {
		case ZmApp.BRIEFCASE: 	return appCtxt.get(ZmSetting.BRIEFCASE_ENABLED, null, this);
		case ZmApp.CALENDAR:	return appCtxt.get(ZmSetting.CALENDAR_ENABLED, 	null, this);
		case ZmApp.CONTACTS:	return appCtxt.get(ZmSetting.CONTACTS_ENABLED, 	null, this);
		case ZmApp.MAIL:		return appCtxt.get(ZmSetting.MAIL_ENABLED, 		null, this);
		case ZmApp.PREFERENCES:	return appCtxt.get(ZmSetting.OPTIONS_ENABLED, 	null, this);
		case ZmApp.TASKS:		return appCtxt.get(ZmSetting.TASKS_ENABLED, 	null, this);
	}
	return false;
};


//
// Protected methods
//

/**
 * @private
 */
ZmZimbraAccount.prototype._handleLoadSettings =
function(result) {
	DBG.println(AjxDebug.DBG1, "Account settings successfully loaded for " + this.name);

	// set account type
	this.type = appCtxt.isOffline
		? appCtxt.get(ZmSetting.OFFLINE_ACCOUNT_FLAVOR, null, this)
		: ZmAccount.TYPE_ZIMBRA;

	this.isZimbraAccount = this.type == ZmAccount.TYPE_ZIMBRA;

	// set icon now that we know the type
	switch (this.type) {
		case ZmAccount.TYPE_AOL:		this.icon = "AccountAOL"; break;
		case ZmAccount.TYPE_GMAIL:		this.icon = "AccountGmail"; break;
		case ZmAccount.TYPE_IMAP:		this.icon = "AccountIMAP"; break;
		case ZmAccount.TYPE_LIVE:		this.icon = "AccountMSN"; break;
		case ZmAccount.TYPE_MSE:		this.icon = "AccountExchange"; break;
		case ZmAccount.TYPE_EXCHANGE:	this.icon = "AccountExchange"; break;
		case ZmAccount.TYPE_POP:		this.icon = "AccountPOP"; break;
		case ZmAccount.TYPE_YMP:		this.icon = "AccountYahoo"; break;
		case ZmAccount.TYPE_ZIMBRA:		this.icon = "AccountZimbra"; break;
	}

	// initialize identities/data-sources/signatures for this account
	var obj = result.getResponse().GetInfoResponse;
	appCtxt.getIdentityCollection(this).initialize(obj.identities);
	appCtxt.getDataSourceCollection(this).initialize(obj.dataSources);
	appCtxt.getSignatureCollection(this).initialize(obj.signatures);

};

/**
 * @private
 */
ZmZimbraAccount.prototype._handleLoadFolders =
function(result) {
	var resp = result.getResponse().GetFolderResponse;
	var folders = resp ? resp.folder[0] : null;
	if (folders) {
		appCtxt.getRequestMgr()._loadTree(ZmOrganizer.FOLDER, null, resp.folder[0], "folder", this);
	}
};

/**
 * @private
 */
ZmZimbraAccount.prototype._handleLoadTags =
function(result) {
	var resp = result.getResponse().GetTagResponse;
	appCtxt.getRequestMgr()._loadTree(ZmOrganizer.TAG, null, resp, null, this);
};

/**
 * @private
 */
ZmZimbraAccount.prototype._handleLoadUserInfo =
function(callback) {
	this.loaded = true;

	// bug fix #33168 - get perms for all mountpoints in account
	var folderTree = appCtxt.getFolderTree(this);
	if (folderTree) {
		folderTree.getPermissions({noBusyOverlay:true, accountName:this.name});
	}

	if (callback) {
		callback.run();
	}
};

/**
 * @private
 */
ZmZimbraAccount.prototype._handleLoadMetaData =
function(callback, sections) {
	for (var i in sections) {
		this.settings.createFromJs(sections[i]);
	}

	if (callback) {
		callback.run();
	}
};

/**
 * @private
 */
ZmZimbraAccount.prototype._handleErrorLoad =
function(callback, ev) {
	DBG.println(AjxDebug.DBG1, "------- ERROR loading account settings for " + this.name);
	if (callback) {
		callback.run();
	}
};

/**
 * @private
 */
ZmZimbraAccount.prototype._loadFromDom =
function(node) {
	this.id = node.id;
	this.name = node.name;
	this.visible = node.visible;
	this.active = node.active;

	var data = node.attrs && node.attrs._attrs;
	this._displayName = data ? data.displayName : this.email;
	this._accountName = data && data.zimbraPrefLabel;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmAccessControlList")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines an access control list and associated classes.
 *
 */

/**
 * Creates an empty access control list (ACL).
 * @class
 * An access control list is a collection of access control entries (ACEs). Each entry contains
 * information about a certain permission applied by the current user to another user or users
 * for a particular type of action. So far, there are two types of rights that are managed in
 * this way:
 * 
 * <ul>
 * <li><b>viewFreeBusy</b> - governs whether other users may view this user's free/busy information</li>
 * <li><b>invite</b> - determines whether an invite from other users will automatically create a tentative appointment on this user's calendar</li>
 * </ul>
 * 
 * Note: that shared organizers ({@link ZmShare}) manage rights (read/write/manage) in their own way.
 * 
 * @author Conrad Damon
 * 
 * @param {Array}	aces		the list of {@link ZmAccessControlEntry} objects
 */
ZmAccessControlList = function(aces) {
	this._aces = {};
}

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAccessControlList.prototype.toString =
function() {
	return "ZmAccessControlList";
};

/**
 * Loads the list.
 * 
 * @param	{AjxCallback}	callback	the function to callback after the loaded
 * 
 * @private
 */
ZmAccessControlList.prototype.load =
function(callback) {
	var jsonObj = {GetRightsRequest:{_jsns:"urn:zimbraAccount"}};
	var respCallback = new AjxCallback(this, this._handleResponseLoad, [callback]);
	appCtxt.getAppController().sendRequest({jsonObj:jsonObj, asyncMode:true, callback:respCallback});
};

/**
 * @private
 */
ZmAccessControlList.prototype._handleResponseLoad =
function(callback, result) {
	var response = result.getResponse();
	var aces = response.GetRightsResponse.ace;
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			this.add(ZmAccessControlEntry.createFromDom(aces[i]));
		}
	}
	if (callback) {
		callback.run();
	}
};

/**
 * Gets the access control entry by right.
 * 
 * @param	{String}	right		the right
 * @return	{ZmAccessControlEntry}	the entry
 */
ZmAccessControlList.prototype.getACLByRight =
function(right) {
	return this._aces[right];
};

/**
 * Gets the grantee type.
 * 
 * @param	{String}	right		the right
 * @return	{constant}	the grantee type (see <code>ZmSetting.ACL_</code> constants)
 * 
 * @see		ZmSetting
 */
ZmAccessControlList.prototype.getGranteeType =
function(right) {
	var aces = this._aces[right];
	var gt = ZmSetting.ACL_PUBLIC;
	
	var gtMap = {};
	if(aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = aces[i];
			DBG.println("<font color=red>ace:</font>" + (ace.negative?"-":"") + ace.granteeType +"," +  ace.grantee );
			var aceGranteeType =  (ace.granteeType == ZmSetting.ACL_USER || ace.granteeType == ZmSetting.ACL_GROUP)  ? ZmSetting.ACL_USER : ace.granteeType;
			gtMap[aceGranteeType] = ace.negative ? -1 : 1;
		}
	}
	
	var allowPublic = (gtMap[ZmSetting.ACL_PUBLIC] == 1);
	var denyPublic  = (gtMap[ZmSetting.ACL_PUBLIC] == -1);
	var allowLocal  = (gtMap[ZmSetting.ACL_AUTH] == 1);
	var denyLocal   = (gtMap[ZmSetting.ACL_AUTH] == -1);
    var allowDomainOnly   = (gtMap[ZmSetting.ACL_DOMAIN] == 1);
	
	var allowUser = (gtMap[ZmSetting.ACL_USER] == 1);
	var allowNone = (denyPublic || denyLocal) && (gtMap[ZmSetting.ACL_USER] == null);
				
	if(allowPublic) {
		return ZmSetting.ACL_PUBLIC;
	}
	
	if(allowLocal) {
		return ZmSetting.ACL_AUTH;
	}
	
	if(denyPublic) {
		if(allowLocal) {
			return ZmSetting.ACL_AUTH;
		}
	}
	
	if(allowUser) {
		return ZmSetting.ACL_USER;
	}

    if(allowDomainOnly) {
		return ZmSetting.ACL_DOMAIN;
	}
	
	if(allowNone) {
		return ZmSetting.ACL_NONE;
	}
	return gt;
};

/**
 * Gets the access control entry by grantee type.
 * 
 * @param	{String}	right	the right
 * @param	{constant}	gt		the grantee type (see <code>ZmSetting.ACL_</code> constants)
 * @return	{Array}	an array of {@link ZmAccessControlEntry} objects
 */
ZmAccessControlList.prototype.getACLByGranteeType =
function(right, gt) {
	var aces = this._aces[right];
	var list = [];
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = aces[i];
			if (ace.granteeType == gt) {
				list.push(ace);
			}
		}
	}
	list.sort();
	return list;
};

/**
 * Gets the grantees.
 * 
 * @param	{String}	right	the right
 * @return	{Array}		an array of grantees
 */
ZmAccessControlList.prototype.getGrantees =
function(right) {
	var aces = this._aces[right];
	var list = [];
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = aces[i];
			if (ace.granteeType == ZmSetting.ACL_USER || ace.granteeType == ZmSetting.ACL_GROUP) {
				list.push(ace.grantee);
			}
		}
	}
	list.sort();
	return list;
};

/**
 * Gets the grantees info.
 * 
 * @param	{String}	right		the right
 * @return	{Array}	an array of grantree info objects (obj.grantee, obj.zid)
 */
ZmAccessControlList.prototype.getGranteesInfo =
function(right) {
	var aces = this._aces[right];
	var list = [];
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = aces[i];
			if (ace.granteeType == ZmSetting.ACL_USER || ace.granteeType == ZmSetting.ACL_GROUP) {
				list.push({grantee: ace.grantee, zid: ace.zid});
			}
		}
	}
	list.sort(ZmAccessControlList.sortByGrantee);
	return list;
};

/**
 * Grants permissions on the access control entries.
 * 
 * @param	{Array}	aces		an array of {@link ZmAccessControlEntry} objects
 * @param	{AjxCallback}	callback	the callback
 * @param	{Boolean}	batchCmd	<code>true</code> to submit as a batch command
 */
ZmAccessControlList.prototype.grant =
function(aces, callback, batchCmd) {
	this._setPerms(aces, false, callback, batchCmd);
};

/**
 * Revokes and denies permissions the access control entries.
 * 
 * @param	{Array}	aces		an array of {@link ZmAccessControlEntry} objects
 * @param	{AjxCallback}	callback	the callback
 * @param	{Boolean}	batchCmd	<code>true</code> to submit as a batch command
 */
ZmAccessControlList.prototype.revoke =
function(aces, callback, batchCmd) {
	this._setPerms(aces, true, callback, batchCmd);
};

/**
 * Sets the permissions.
 * 
 * @param	{Array}	aces		an array of {@link ZmAccessControlEntry} objects
 * @param	{Boolean}	revoke	<code>true</code> to deny; <code>false</code> to grant
 * @param	{AjxCallback}	callback	the callback
 * @param	{Boolean}	batchCmd	<code>true</code> to submit as a batch command
 *
 * @private
 */
ZmAccessControlList.prototype._setPerms =
function(aces, revoke, callback, batchCmd) {
	var reqName = revoke ? "RevokeRightsRequest" : "GrantRightsRequest";
	var soapDoc = AjxSoapDoc.create(reqName, "urn:zimbraAccount");
	for (var i = 0; i < aces.length; i++) {
		var ace = aces[i];
		var aceNode = soapDoc.set("ace");
		aceNode.setAttribute("right", ace.right);
		aceNode.setAttribute("gt", ace.granteeType);
		if(ace.grantee) {
			aceNode.setAttribute("d", ace.grantee);
		}
		if (ace.zid) {
			aceNode.setAttribute("zid", ace.zid);
		}
		if (ace.negative) {
			aceNode.setAttribute("deny", 1);
		}
	}
	var respCallback = new AjxCallback(this, this._handleResponseSetPerms, [revoke, callback]);
	if (batchCmd) {
		batchCmd.addNewRequestParams(soapDoc, respCallback);
	} else {
		appCtxt.getAppController().sendRequest({soapDoc:soapDoc, asyncMode:true, callback:respCallback});
	}
};

/**
 * @private
 */
ZmAccessControlList.prototype._handleResponseSetPerms =
function(revoke, callback, result) {
	var response = result.getResponse();
	var resp = revoke ? response.RevokeRightsResponse : response.GrantRightsResponse;
	var aces = resp && resp.ace;
	var aceList = [];
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = ZmAccessControlEntry.createFromDom(aces[i]);
			aceList.push(ace);
			if (revoke) {
				this.remove(ace);
			} else {
				this.update(ace);
			}
		}
	}

	if (callback) {
		callback.run(aceList);
	}
};

/**
 * Adds the entry to the ACL.
 * 
 * @param	{ZmAccessControlEntry}	ace	the entry to add
 */
ZmAccessControlList.prototype.add =
function(ace) {
	if (!ace) { return; }
	var right = ace.right;
	if (!this._aces[right]) {
		this._aces[right] = [];
	}
	this._aces[right].push(ace);
};

/**
 * Removes the entry to the ACL.
 * 
 * @param	{ZmAccessControlEntry}	ace	the entry to remove
 */
ZmAccessControlList.prototype.remove =
function(ace) {
	if (!ace) { return; }
	var list = this._aces[ace.right];
	var newList = [];
	if (list && list.length) {
		for (var i = 0; i < list.length; i++) {
			if (list[i].grantee != ace.grantee) {
				newList.push(list[i]);
			}
		}
	}
	this._aces[ace.right] = newList;
};

/**
 * Updates the entry to the ACL.
 * 
 * @param	{ZmAccessControlEntry}	ace	the entry to update
 * @param	{Boolean}	removeEnty	not used
 */
ZmAccessControlList.prototype.update =
function(ace, removeEntry) {
	if (!ace || !ace.right) { return; }
	var found = false;
	
	if(!this._aces[ace.right]) {
		this._aces[ace.right] = [];
	}

	var list = this._aces[ace.right];	
	if (list.length) {
		//search for ace to update
		for (var i = 0; i < list.length; i++) {
			if ((list[i].grantee == ace.grantee) && (list[i].granteeType == ace.granteeType)) {
				this._aces[ace.right][i] = ace;
				found = true;
			}
		}
	}
	if(!found) {
		//adding new entry to ace list
		this._aces[ace.right].push(ace);
	}
};

/**
 * Cleans up the ACL.
 * 
 */
ZmAccessControlList.prototype.cleanup =
function() {
	this._aces = {};
};

/**
 * Sorts the ACL by grantee.
 * 
 * @param	{Hash}	a		grantee "a"
 * @param	{String}	a.grantee	the grantee
 * @param	{Hash}	b		grantee "b"
 * @param	{Hash}	b.grantee		grantee "b"
 * @return	{int}	0 if "a" and "b" are the same; 1 if "a" is before "b"; -1 if "b" is before "a"
 */
ZmAccessControlList.sortByGrantee =
function(a, b) {

    var granteeA = a.grantee || "";
    var granteeB = b.grantee || "";

    if (granteeA.toLowerCase() > granteeB.toLowerCase()) { return 1; }
    if (granteeA.toLowerCase() < granteeB.toLowerCase()) { return -1; }
    
	return 0;
};


/**
 * Creates an access control entry.
 * @class
 * An access control entry encapsulates the permission information pertaining to a user or users
 * regarding a certain right.
 * 
 * @param {Hash}	params		a hash of parameters
 * @param	{String}	params.right		the action governed by this ACE
 * @param	{String}	params.grantee		the account name of user or group permission applies to
 * @param	{String}	params.zid			the ZID of grantee
 * @param	{constant}	params.granteeType	type of grantee (see <code>ZmSetting.ACL_</code> constants)
 * @param	{Boolean}	params.negative		if <code>true</code>, permission is denied by this ACE
 * @see		ZmSetting
 */
ZmAccessControlEntry =
function(params) {
	this.grantee = params.grantee;
	this.zid = params.zid;
	this.granteeType = params.granteeType;
	this.right = params.right;
	this.negative = params.negative;
}

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAccessControlEntry.prototype.toString =
function() {
	return "ZmAccessControlEntry";
};

/**
 * Creates an entry from the DOM object.
 * 
 * @param	{Hash}	obj		the DOM object
 * @param	{String}	obj.right		the action governed by this ACE
 * @param	{String}	obj.d		the account name of user or group permission applies to
 * @param	{String}	obj.zid			the ZID of grantee
 * @param	{constant}	obj.gt		the type of grantee (see <code>ZmSetting.ACL_</code> constants)
 * @param	{Boolean}	obj.deny		if <code>1</code>, permission is denied by this ACE
 * @return	{ZmAccessControlEntry}	the newly created entry
 */
ZmAccessControlEntry.createFromDom =
function(obj) {
	var params = {};
	params.grantee = obj.d;
	params.granteeType = obj.gt;
	params.zid = obj.zid;
	params.right = obj.right;
	params.negative = (obj.deny == "1");
	
	return new ZmAccessControlEntry(params);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmDomainList")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @class
 * This class represents a list of domains that have shown up in email addresses.
 * 
 * @extends		ZmModel
 */
ZmDomainList = function() {
	ZmModel.call(this);
};

ZmDomainList.prototype = new ZmModel;
ZmDomainList.prototype.constructor = ZmDomainList;

ZmDomainList.prototype.isZmDomainList = true;
ZmDomainList.prototype.toString = function() { return "ZmDomainList"; };

ZmDomainList.DOMAIN_RE = new RegExp("\\.\\w{2,3}$");

/**
 * Gets a sorted list of domains matching the given string (if any).
 * 
 * @param {String}		str			the string to search for
 * @param {int}			limit		the max number of domains to return
 * @param {AjxCallback}	callback	the callback to run when response is received
 */
ZmDomainList.prototype.search =
function(str, limit, callback) {

	var jsonObj = {BrowseRequest:{_jsns:"urn:zimbraMail"}};
	var request = jsonObj.BrowseRequest;
	request.browseBy = "domains";
	if (str && (/[a-z]/i.test(str))) {
		if (ZmDomainList.DOMAIN_RE.test(str)) {
			request.regex = [".*", AjxStringUtil.regExEscape(str), "$"].join("");
		} else {
			request.regex = ["^", str, ".*"].join("");
		}
	}
	if (limit) {
		request.maxToReturn = limit;
	}
	var respCallback = ZmDomainList._handleResponseSearch.bind(null, str, callback);
	appCtxt.getAppController().sendRequest({jsonObj:jsonObj, asyncMode:true, callback:respCallback});
};

ZmDomainList._handleResponseSearch =
function(str, callback, result) {
	var domains = result.getResponse().BrowseResponse.bd;
	var list = [];
	if (domains) {
		for (var i = 0; i < domains.length; i++) {
			var domain = domains[i];
			list[i] = new ZmDomain(domain._content, domain.h);
		}
	}
	list.sort(ZmDomain.sortCompare);
	callback.run(list);
};




/**
 * @class
 * This class represents a domain.
 * 
 * @param	{String}	name			the name
 * @param	{String}	headerFlags		header flags (where domain was found)
 * 
 * @extends	ZmModel
 */
ZmDomain = function(name, headerFlags) {
	
	ZmModel.call(this);

	this.name = name.toLowerCase();
	this._headerFlags = headerFlags;
};

ZmDomain.prototype = new ZmModel;
ZmDomain.prototype.constructor = ZmDomain;

ZmDomain.prototype.isZmDomain = true;
ZmDomain.prototype.toString = function() { return "ZmDomain"; };


ZmDomain.ADDR_FLAG = {};
ZmDomain.ADDR_FLAG[AjxEmailAddress.FROM]	= "f";
ZmDomain.ADDR_FLAG[AjxEmailAddress.TO]		= "t";
ZmDomain.ADDR_FLAG[AjxEmailAddress.CC]		= "c";


/**
 * Compares two domains by name.
 * 
 * @param	{ZmDomain}	a		the first domain
 * @param	{ZmDomain}	b		the second domain
 * @return	{int}	0 if the domains match; 1 if "a" is before "b"; -1 if "b" is before "a"
 */
ZmDomain.sortCompare = 
function(a, b) {
	var check = ZmOrganizer.checkSortArgs(a, b);
	if (check != null) { return check; }

	if (a.name < b.name) { return -1; }
	if (a.name > b.name) { return 1; }
	return 0;
};

ZmDomain.prototype.hasAddress =
function(addressType) {
	var flag = ZmDomain.ADDR_FLAG[addressType];
	return flag && this._headerFlags && (this._headerFlags.indexOf(flag) != -1);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmAttachmentTypeList")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines a list of attachment types.
 *
 */

/**
 * Creates an attachment type list
 * @class
 * This class represents attachment types.
 * 
 * @extends	ZmModel
 */
ZmAttachmentTypeList = function() {
	ZmModel.call(this, ZmEvent.S_ATT);
};

ZmAttachmentTypeList.prototype = new ZmModel;
ZmAttachmentTypeList.prototype.constructor = ZmAttachmentTypeList;

ZmAttachmentTypeList.prototype.isZmAttachmentTypeList = true;
ZmAttachmentTypeList.prototype.toString = function() { return "ZmAttachmentTypeList"; };

/**
 * Gets the attachments.
 * 
 * @return	{Array}	an array of attachments
 */
ZmAttachmentTypeList.prototype.getAttachments =
function() {
	return this._attachments;
};

/**
 * Compares attachment type lists by description.
 * 
 * @param	{ZmAttachmentTypeList}	a			the first entry
 * @param	{ZmAttachmentTypeList}	b			the first entry
 * @return	{int}	0 if the entries match; 1 if "a" is before "b"; -1 if "b" is before "a"
 */
ZmAttachmentTypeList.compareEntry = 
function(a,b) {
	if (a.desc.toLowerCase() < b.desc.toLowerCase())	{ return -1; }
	if (a.desc.toLowerCase() > b.desc.toLowerCase())	{ return 1; }
	return 0;
};

/**
 * Loads the attachments.
 * 
 * @param	{AjxCallback}	callback		the callback to call after load
 */
ZmAttachmentTypeList.prototype.load =
function(callback) {

	this._attachments = [];

	var jsonObj = {BrowseRequest:{_jsns:"urn:zimbraMail"}};
	var request = jsonObj.BrowseRequest;
	request.browseBy = "attachments";

	var respCallback = this._handleResponseLoad.bind(this, callback);
	appCtxt.getAppController().sendRequest({jsonObj: jsonObj, asyncMode: true, callback: respCallback});
};

/**
 * @private
 */
ZmAttachmentTypeList.prototype._handleResponseLoad =
function(callback, result) {
	var att = this._organizeTypes(result.getResponse().BrowseResponse.bd);
    var isZipFileIncluded = false;
	if (att) {
		for (var i = 0; i < att.length; i++) {
			var type = att[i]._content;
			if (!ZmMimeTable.isIgnored(type)) {
				this._attachments.push(ZmMimeTable.getInfo(type, true));
			}
		}
		this._attachments.sort(ZmAttachmentTypeList.compareEntry);
	}

	if (callback) {
		callback.run(this._attachments);
	}
};

/**
 * Check whether type is from the following list
 * Adobe PDF
 * Microsoft Word (doc, docx)
 * Microsoft Powerpoint
 * Microsoft Excel
 * Email Message
 * HTML
 * Calendar (ical)
 *
 *
 * @param	{String}			attachment type
 * @return	{Boolean}           true if the type in the above list, otherwise false
 *
 * @private
 */
ZmAttachmentTypeList.prototype._isSupportedType  =
function(type){
	var supportedTypes =  [ZmMimeTable.APP_ADOBE_PDF, ZmMimeTable.APP_MS_WORD,ZmMimeTable.APP_MS_EXCEL,
    	                   ZmMimeTable.APP_MS_PPT, ZmMimeTable.APP_ZIP,ZmMimeTable.APP_ZIP2, ZmMimeTable.MSG_RFC822,
        	               ZmMimeTable.TEXT_HTML, ZmMimeTable.TEXT_CAL];

    return AjxUtil.arrayContains(supportedTypes, type);
};

/**
 * Returns group type if type belongs to following group:
 *  Text (vcard, csv)
 *  Video (mpeg, mov)
 *  Audio (wav, mp3, etc)
 *  Archive (zip, etc)
 *  Application (any)
 *  Image (bmp, png, gif, tiff, jpg, psd, ai, jpeg)
 *
 * @param	{String}	    attachment type
 * @return	{String}	    attachment group if it exits in the above list, otherwise null
 *
 * @private
 */
ZmAttachmentTypeList.prototype._isSupportedGroup =
function(type){
    var supportedGroups = [ZmMimeTable.APP,ZmMimeTable.AUDIO,ZmMimeTable.IMG,ZmMimeTable.TEXT,ZmMimeTable.VIDEO ];
    var regExp =new RegExp("^" + supportedGroups.join("|^") ,"ig");
    var groupType = type.match(regExp)
    return groupType && groupType[0];

};

/**
 * Returns set of supported type/group of attachments
 *
 * @param	{Array} list of attachment types
 * @return	{Array} Set of types which is an intersection of att and supported types/groups
 * @private
 */
ZmAttachmentTypeList.prototype._organizeTypes =
function(att) {

	var res = [];
	if (!(att && att.length)) {
		return res;
	}

	for (var i = 0; i < att.length; i++) {
		var type = att[i]._content;
		var freq = att[i].freq;
		var skip = true;
		var groupType = null;
		if (this._isSupportedType(type)) {
			skip = false;
		} else if (type = this._isSupportedGroup(type)) {
			skip = false;
			// Check if group is already in result
			for (var j=0; j < res.length; j++) {
				if (res[j]. _content === type ) {
					res[j].freq += freq;
					skip = true;
					break;
				}
			}
		}
	
		if (!skip) {
			res.push({_content: type, freq: freq});
		}

	}
	return res;
};
}

if (AjxPackage.define("zimbraMail.core.ZmApp")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 *
 * This file defines a Zimbra Application class.
 *
 */

/**
 * Creates the application.
 * @class
 * This object represents a Zimbra Application. This class is a base class for application classes.
 * "App" is a useful abstraction for a set of related functionality, such as mail,
 * address book, or calendar. Looked at another way, an app is a collection of one or more controllers.
 * 
 * @param	{String}	name		the application name
 * @param	{DwtControl}	container	the control that contains components
 * @param	{ZmController}	parentController	the parent window controller (set by the child window)
 *
 */
ZmApp = function(name, container, parentController) {

	if (arguments.length == 0) return;
	
	this._name = name;
	this._container = container;
	this._parentController = parentController;
	this._active = false;
	this.currentSearch = null;
    this._defaultFolderId = null;   //reqd in case of external user

	this._deferredFolders = [];
	this._deferredFolderHash = {};
	this._deferredNotifications = [];

	this._sessionController		= {};
	this._nextSessionId			= {};
	this._curSessionId			= {};

	ZmApp.DROP_TARGETS[name] = {};

	this._defineAPI();
	if (!parentController) {
		this._registerSettings();
	}
	this._registerOperations();
	this._registerItems();
	this._registerOrganizers();
	if (!parentController) {
		this._setupSearchToolbar();
	}
	this._registerApp();

};

// app information ("_R" means "reverse map")

// these are needed statically (before we get user settings)
ZmApp.CLASS					= {};	// constructor for app class
ZmApp.SETTING				= {};	// ID of setting that's true when app is enabled
ZmApp.UPSELL_SETTING		= {};	// ID of setting that's true when app upsell is enabled
ZmApp.LOAD_SORT				= {};	// controls order in which apps are instantiated
ZmApp.BUTTON_ID				= {};	// ID for app button on app chooser toolbar

// these are set via registerApp() in app constructor
ZmApp.MAIN_PKG				= {};	// main package that composes the app
ZmApp.NAME					= {};	// msg key for app name
ZmApp.ICON					= {};	// name of app icon class
ZmApp.TEXT_PRECEDENCE		= {};	// order for removing button text
ZmApp.IMAGE_PRECEDENCE		= {};	// order for removing button image
ZmApp.QS_ARG				= {};	// arg for 'app' var in QS to jump to app
ZmApp.QS_ARG_R				= {};
ZmApp.CHOOSER_TOOLTIP		= {};	// msg key for app view menu tooltip
ZmApp.VIEW_TOOLTIP			= {};	// msg key for app tooltip
ZmApp.DEFAULT_SEARCH		= {};	// type of item to search for in the app
ZmApp.ORGANIZER				= {};	// main organizer for this app
ZmApp.OVERVIEW_TREES		= {};	// list of tree IDs to show in overview
ZmApp.HIDE_ZIMLETS			= {};	// whether to show Zimlet tree in overview
ZmApp.SEARCH_TYPES			= {};	// list of types of saved searches to show in overview
ZmApp.SEARCH_TYPES_R		= {};
ZmApp.GOTO_ACTION_CODE		= {};	// key action for jumping to this app
ZmApp.GOTO_ACTION_CODE_R	= {};
ZmApp.NEW_ACTION_CODE		= {};	// default "new" key action
ZmApp.ACTION_CODES			= {};	// key actions that map to ops
ZmApp.ACTION_CODES_R		= {};
ZmApp.OPS					= {};	// IDs of operations for the app
ZmApp.OPS_R					= {};	// map of operation ID to app
ZmApp.QS_VIEWS				= {};	// list of views to handle in query string
ZmApp.TRASH_VIEW_OP			= {};	// menu choice for "Show Only ..." in Trash view
ZmApp.UPSELL_URL			= {};	// URL for content of upsell
//ZmApp.QUICK_COMMAND_TYPE	= {};
ZmApp.DROP_TARGETS			= {};	// drop targets (organizers) by item/organizer type
ZmApp.SEARCH_RESULTS_TAB	= {};	// whether to show search results in a tab

// indexes to control order of appearance/action
ZmApp.CHOOSER_SORT			= {};	// controls order of apps in app chooser toolbar
ZmApp.DEFAULT_SORT			= {};	// controls order in which app is chosen as default start app

ZmApp.ENABLED_APPS			= {};	// hash for quick detection if app is enabled

// ordered lists of apps
ZmApp.APPS					= [];	// ordered list
ZmApp.DEFAULT_APPS			= [];	// ordered list

ZmApp.OVERVIEW_ID			= "main";	// ID for main overview

ZmApp.BATCH_NOTIF_LIMIT = 25;	// threshold for doing batched change notifications

ZmApp.MAIN_SESSION			= "main";
ZmApp.HIDDEN_SESSION		= "hidden";

/**
 * Initializes the application.
 *
 * @private
 */
ZmApp.initialize =
function() {
	if (appCtxt.get(ZmSetting.USE_KEYBOARD_SHORTCUTS)) {
		ZmApp.ACTION_CODES[ZmKeyMap.NEW_FOLDER]	= ZmOperation.NEW_FOLDER;
		ZmApp.ACTION_CODES[ZmKeyMap.NEW_TAG]	= ZmOperation.NEW_TAG;
	}
};

/**
 * Registers and stores information about an app. Note: Setting a value that evaluates to
 * false (such as 0 or an empty string) will not do anything.
 *
 * @param {constant}	app				the app ID
 * @param {Hash}	params			a hash of parameters
 * @param params.mainPkg			[string]	main package that contains the app
 * @param params.nameKey			[string]	msg key for app name
 * @param params.icon				[string]	name of app icon class
 * @param params.textPrecedence		[int]		order for removing button text
 * @param params.imagePrecedence	[int]		order for removing button image
 * @param params.chooserTooltipKey	[string]	msg key for app tooltip
 * @param params.viewTooltipKey		[string]	msg key for app view menu tooltip
 * @param params.defaultSearch		[constant]	type of item to search for in the app
 * @param params.organizer			[constant]	main organizer for this app
 * @param params.overviewTrees		[array]		list of tree IDs to show in overview
 * @param params.hideZimlets		[boolean]	if true, hide Zimlet tree in overview
 * @param params.searchTypes		[array]		list of types of saved searches to show in overview
 * @param params.gotoActionCode		[constant]	key action for jumping to this app
 * @param params.newActionCode		[constant]	default "new" action code
 * @param params.actionCodes		[hash]		keyboard actions mapped to operations
 * @param params.newItemOps			[hash]		IDs of operations that create a new item, and their text keys
 * @param params.newOrgOps			[hash]		IDs of operations that create a new organizer, and their text keys
 * @param params.qsViews			[array]		list of views to handle in query string
 * @param params.chooserSort		[int]		controls order of apps in app chooser toolbar
 * @param params.defaultSort		[int]		controls order in which app is chosen as default start app
 * @param params.trashViewOp		[constant]	menu choice for "Show Only ..." in Trash view
 * @param params.upsellUrl			[string]	URL for content of upsell
 * @param params.searchResultsTab	[string]	if true, show search results in a tab
 *
 * @private
 */
ZmApp.registerApp =
function(app, params) {

	// TODO: why the ifs? this should only be called once per app
	if (params.mainPkg)				{ ZmApp.MAIN_PKG[app]			= params.mainPkg; }
	if (params.nameKey)				{ ZmApp.NAME[app]				= params.nameKey; }
	if (params.icon)				{ ZmApp.ICON[app]				= params.icon; }
	if (params.textPrecedence)		{ ZmApp.TEXT_PRECEDENCE[app]	= params.textPrecedence; }
	if (params.imagePrecedence)		{ ZmApp.IMAGE_PRECEDENCE[app]	= params.imagePrecedence; }
	if (params.chooserTooltipKey)	{ ZmApp.CHOOSER_TOOLTIP[app]	= params.chooserTooltipKey; }
	if (params.viewTooltipKey)		{ ZmApp.VIEW_TOOLTIP[app]		= params.viewTooltipKey; }
	if (params.defaultSearch)		{ ZmApp.DEFAULT_SEARCH[app]		= params.defaultSearch; }
	if (params.organizer)			{ ZmApp.ORGANIZER[app]			= params.organizer; }
	if (params.overviewTrees)		{ ZmApp.OVERVIEW_TREES[app]		= params.overviewTrees; }
	if (params.hideZimlets)			{ ZmApp.HIDE_ZIMLETS[app]		= params.hideZimlets; }
	if (params.searchTypes) 		{ ZmApp.SEARCH_TYPES[app]		= params.searchTypes; }
	if (params.gotoActionCode)		{ ZmApp.GOTO_ACTION_CODE[app]	= params.gotoActionCode; }
	if (params.newActionCode)		{ ZmApp.NEW_ACTION_CODE[app]	= params.newActionCode; }
	if (params.qsViews)				{ ZmApp.QS_VIEWS[app]			= params.qsViews; }
	if (params.chooserSort)			{ ZmApp.CHOOSER_SORT[app]		= params.chooserSort; }
	if (params.defaultSort)			{ ZmApp.DEFAULT_SORT[app]		= params.defaultSort; }
	if (params.trashViewOp)			{ ZmApp.TRASH_VIEW_OP[app]		= params.trashViewOp; }
	if (params.upsellUrl)			{ ZmApp.UPSELL_URL[app]			= params.upsellUrl; }
	//if (params.quickCommandType)	{ ZmApp.QUICK_COMMAND_TYPE[app]	= params.quickCommandType; }
	if (params.searchResultsTab)	{ ZmApp.SEARCH_RESULTS_TAB[app]	= params.searchResultsTab; }

	if (params.searchTypes) {
		ZmApp.SEARCH_TYPES_R[app] = {};
		for (var i = 0; i < params.searchTypes.length; i++) {
			ZmApp.SEARCH_TYPES_R[app][params.searchTypes[i]] = true;
		}
	}

	if (params.gotoActionCode) {
		ZmApp.GOTO_ACTION_CODE_R[params.gotoActionCode] = app;
	}

	if (params.actionCodes) {
		for (var ac in params.actionCodes) {
			if (!ac) { continue; }
			ZmApp.ACTION_CODES_R[ac] = app;
			ZmApp.ACTION_CODES[ac] = params.actionCodes[ac];
		}
	}

    var appEnabled = appCtxt.get(ZmApp.SETTING[app]);
	if (params.newItemOps && appEnabled) {
		for (var op in params.newItemOps) {
			if (!op) { continue; }
			ZmApp.OPS_R[op] = app;
			ZmOperation.NEW_ITEM_OPS.push(op);
			ZmOperation.NEW_ITEM_KEY[op] = params.newItemOps[op];
		}
	}
	if (params.newOrgOps && appEnabled) {
		for (var op in params.newOrgOps) {
			if (!op) { continue; }
			ZmApp.OPS_R[op] = app;
			ZmOperation.NEW_ORG_OPS.push(op);
			ZmOperation.NEW_ORG_KEY[op] = params.newOrgOps[op];
		}
	}

	if (params.qsViews) {
		for (var i = 0; i < params.qsViews.length; i++) {
			ZmApp.QS_VIEWS[params.qsViews[i]] = app;
		}
	}

    /* if (params.quickCommandType) {
        ZmQuickCommand.itemTypes.push(params.quickCommandType);
    } */
};


/**
 * Runs the given function for all known (e.g. part of ZmApp.CLASS)
 * app classes, passing args.
 * NOTE: This runs class functions only, not instance (prototype) functions.
 * @static
 * @param funcName {String} The name of the function we will run on each
 * application.
 * @param mixed {mixed} 0 to n additional arguments are passed to funcName
 * via apply.
 */
ZmApp.runAppFunction =
function(funcName) {
    var args;

    for (var appName in ZmApp.CLASS) {
        var app = window[ZmApp.CLASS[appName]];
        var func = app && app[funcName];
        if (func && (typeof(func) == "function")) {
            args = args || Array.prototype.slice.call(arguments, 1);
            func.apply(app, args);
        }
    }
};


// Public instance methods

/**
 * Returns a string representation of the object.
 *
 * @return		{String}		a string representation of the object
 */
ZmApp.prototype.toString =
function() {
	return "ZmApp";
};

// Functions called during construction
ZmApp.prototype._defineAPI				= function() {};
ZmApp.prototype._registerSettings		= function() {};
ZmApp.prototype._registerOperations		= function() {};
ZmApp.prototype._registerItems			= function() {};
ZmApp.prototype._registerOrganizers		= function() {};
ZmApp.prototype._setupSearchToolbar		= function() {};
ZmApp.prototype._registerApp			= function() {};
ZmApp.prototype._registerPrefs			= function() {};						// called when Preferences pkg is loaded

// Functions that apps can override in response to certain events
ZmApp.prototype.startup					= function(result) {};					// run during startup
ZmApp.prototype.preNotify				= function(notify) {};					// run before handling notifications
ZmApp.prototype.deleteNotify			= function(ids) {};						// run on delete notifications
ZmApp.prototype.createNotify			= function(list) {};					// run on create notifications
ZmApp.prototype.modifyNotify			= function(list) {};					// run on modify notifications
ZmApp.prototype.postNotify				= function(notify) {};					// run after handling notifications
ZmApp.prototype.refresh					= function(refresh) {};					// run when a <refresh> block arrives
ZmApp.prototype.handleOp				= function(op, params) {};				// handle an operation

/**
 * Gets the application name.
 *
 * @return	{String}	the name
 */
ZmApp.prototype.getName =
function() {
	return this._name;
};

/**
 * Gets the application display name.
 *
 * @return	{String}	the display name
 */
ZmApp.prototype.getDisplayName =
function() {
	return ZmMsg[ZmApp.NAME[this._name]] || ZmApp.NAME[this._name];
};

/**
 * Gets the initial search type.
 *
 * @return	{Object}	<code>null</code> since only set if different from the default
 */
ZmApp.prototype.getInitialSearchType =
function() {
	return null;
};

/**
 * Gets the limit for the search triggered by the application launch or an overview click.
 *
 * @return	{int}	the limit
 */
ZmApp.prototype.getLimit =
function(offset) {
	return appCtxt.get(ZmSetting.PAGE_SIZE);
};

/**
 * Sets the application view.
 *
 * @param		{String}	view		the view
 * @see		ZmAppViewMgr
 */
ZmApp.prototype.setAppView =
function(view) {
	appCtxt.getAppViewMgr().setAppView(this._name, view);
};

/**
 * Creates the application view.
 *
 * @param		{Hash}	params		a hash of parameters
 * @see		ZmAppViewMgr
 * @see		ZmAppViewMgr#createView
 */
ZmApp.prototype.createView =
function(params) {
	params.appName = this._name;
	return appCtxt.getAppViewMgr().createView(params);
};

/**
 * Pushes the application view.
 *
 * @param	{String}	name	the view name
 * @param	{Boolean}	force	<code>true</code> to force the view onto the stack
 * @see		ZmAppViewMgr#pushView
 */
ZmApp.prototype.pushView =
function(name, force) {
	return appCtxt.getAppViewMgr().pushView(name, force);
};

/**
 * Pops the application view.
 *
 * @param	{Boolean}	force	<code>true</code> to force the view off the stack
 * @see		ZmAppViewMgr#popView
 */
ZmApp.prototype.popView =
function(force, viewId, skipHistory) {
	return appCtxt.getAppViewMgr().popView(force, viewId, skipHistory);
};

/**
 * Sets the application view.
 *
 * @param	{String}	name	the view name
 * @param	{Boolean}	force	<code>true</code> to force the view
 * @see		ZmAppViewMgr#setView
 */
ZmApp.prototype.setView =
function(name, force) {
	return appCtxt.getAppViewMgr().setView(name, force);
};

/**
 * Stages the application view.
 *
 * @param	{String}	name	the view name
 * @see		ZmAppViewMgr#stageView
 */
ZmApp.prototype.stageView =
function(name) {
	return appCtxt.getAppViewMgr().stageView(name);
};

/**
 * Adds a deferred folder.
 *
 * @param	{Hash}	params		a hash of parameters
 */
ZmApp.prototype.addDeferredFolder =
function(params) {
	var id = params.obj && params.obj.id;
	if (id && !this._deferredFolderHash[id]) {
		this._deferredFolders.push(params);
		this._deferredFolderHash[id] = true;
		appCtxt.cacheSetDeferred(id, this._name);
	}
};

/**
 * Gets the remote folder ids.
 *
 * @param	{Object}	account		the account
 * @return	{Array}		an array of {String} ids
 */
ZmApp.prototype.getRemoteFolderIds =
function(account) {
	// XXX: optimize by caching this list? It would have to be cleared anytime
	// folder structure changes
	var list = [];
	if (appCtxt.getOverviewController(true)) {
		var type = ZmApp.ORGANIZER[this.getName()];

		// first, make sure there aren't any deferred folders that need to be created
		if (this._deferredFolders.length) {
			this._createDeferredFolders(type);
		}

		var tree = appCtxt.getFolderTree(account);
		var folders = tree ? tree.getByType(type) : [];
		for (var i = 0; i < folders.length; i++) {
			var folder = folders[i];
			if (folder.isRemote()) {
				list.push(folder.id);
			}
		}
	}
	return list;
};

/**
 * Creates the overview content for this app. The default implementation creates
 * a {@link ZmOverview} with standard options. Other apps may want to use different
 * options, or create a {@link DwtComposite} instead.
 *
 * @return	{String}	the content
 */
ZmApp.prototype.getOverviewPanelContent =
function() {
	if (!this._overviewPanelContent) {
		var params = this._getOverviewParams();
		params.overviewId = this.getOverviewId();
		var ov = this._overviewPanelContent = appCtxt.getOverviewController().createOverview(params);
		ov.set(this._getOverviewTrees());
	}

	return this._overviewPanelContent;
};



/**
 * Gets the overview container.
 *
 * @return	{ZmOverview}		the overview container
 */
ZmApp.prototype.getOverviewContainer =
function(dontCreate) {
	if (!this._overviewContainer && !dontCreate) {
		var containerParams = {
			appName: this._name,
			containerId: ([ZmApp.OVERVIEW_ID, this._name].join("_")),
			posStyle: Dwt.ABSOLUTE_STYLE
		};
		var overviewParams = this._getOverviewParams();
		overviewParams.overviewTrees = this._getOverviewTrees();

		this._overviewContainer = appCtxt.getOverviewController().createOverviewContainer(containerParams, overviewParams);
	}

	return this._overviewContainer;
};

/**
 * Sets the overview tree to display overview content for this application.
 *
 * @param {Boolean}	reset		if <code>true</code>, clear the content first
 */
ZmApp.prototype.setOverviewPanelContent =
function(reset) {
	if (reset) {
		this._overviewPanelContent = null;
		this._overviewContainer = null;
	}

	// only set overview panel content if not in full screen mode
	var avm = appCtxt.getAppViewMgr();
	if (!avm.isFullScreen()) {
		Dwt.setLoadingTime(this.toString() + "-overviewPanel");
		var ov = ((appCtxt.multiAccounts && appCtxt.accountList.size() > 1) || this.getName() == ZmApp.VOICE)
			? this.getOverviewContainer()
			: this.getOverviewPanelContent();
		var components = {};
		components[ZmAppViewMgr.C_TREE] = ov;
		avm.setViewComponents(ZmAppViewMgr.APP, components, true, this.getName());
		Dwt.setLoadedTime(this.toString() + "-overviewPanel");
	}
};

/**
 * Gets the current overview, if any. Subclasses should ensure that a {@link ZmOverview} is returned.
 *
 * @return	{ZmOverview}	the overview
 */
ZmApp.prototype.getOverview =
function() {
	var opc = appCtxt.getOverviewController();
	return opc && opc.getOverview(this.getOverviewId());
};

/**
 * Resets the current overview, preserving expansion.
 *
 * @param {String}		overviewId	the id of overview to reset
 */
ZmApp.prototype.resetOverview =
function(overviewId) {
	var overview = overviewId ? appCtxt.getOverviewController().getOverview(overviewId) : this.getOverview();
	if (overview) {
		var expIds = [];
		var treeIds = overview.getTreeViews(), len = treeIds.length;
		for (var i = 0; i < len; i++) {
			var treeId = treeIds[i];
			var treeView = overview.getTreeView(treeId);
			if (treeView) {
				var items = treeView.getTreeItemList();
				var len1 = items.length;
				for (var j = 0; j < len1; j++) {
					var treeItem = items[j];
					if (treeItem._expanded) {
						expIds.push(treeItem._htmlElId);
					}
				}
			}
		}
		overview.clear();
		overview.set(this._getOverviewTrees());
		len = expIds.length;
		for (var i = 0; i < len; i++) {
			var treeItem = DwtControl.fromElementId(expIds[i]);
			if (treeItem && !treeItem._expanded) {
				treeItem.setExpanded(true);
			}
		}
	}
};

/**
 * Gets the overview id of the current {@link ZmOverview}, if any.
 *
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{String}	the id
 */
ZmApp.prototype.getOverviewId =
function(account) {
	return appCtxt.getOverviewId([ZmApp.OVERVIEW_ID, this._name], account);
};

/**
 * Returns a hash of params with the standard overview options.
 *
 * @private
 */
ZmApp.prototype._getOverviewParams =
function() {
	// Get the sorted list of overview trees.
	var treeIds = [];
	for (var id in ZmOverviewController.CONTROLLER) {
		treeIds.push(id);
	}
	var sortFunc = function(a, b) {
		return (ZmOrganizer.DISPLAY_ORDER[a] || 9999) - (ZmOrganizer.DISPLAY_ORDER[b] || 9999);
	};
	treeIds.sort(sortFunc);

	return {
		posStyle:			Dwt.ABSOLUTE_STYLE,
		selectionSupported:	true,
		actionSupported:	true,
		dndSupported:		true,
		showUnread:			true,
		showNewButtons:		true,
		isAppOverview:		true,
		treeIds:			treeIds,
		appName:			this._name,
		account:			appCtxt.getActiveAccount()
	};
};

/**
 * Returns the list of trees to show in the overview for this app. Don't show
 * Folders unless mail is enabled. Other organizer types won't be created unless
 * their apps are enabled, so we don't need to check for them.
 *
 * @private
 */
ZmApp.prototype._getOverviewTrees =
function() {
	var list = ZmApp.OVERVIEW_TREES[this._name] || [];
	var newList = [];
	for (var i = 0, count = list.length; i < count; i++) {
		if ((list[i] == ZmOrganizer.FOLDER && !appCtxt.get(ZmSetting.MAIL_ENABLED))) {
			continue;
		}
		newList.push(list[i]);
	}

	if (!appCtxt.multiAccounts &&
		window[ZmOverviewController.CONTROLLER[ZmOrganizer.ZIMLET]] &&
		!ZmApp.HIDE_ZIMLETS[this._name])
	{
		newList.push(ZmOrganizer.ZIMLET);
	}
	return newList;
};

/**
 * Gets the number of active session controllers
 *
 * @return	{number} the number of active session controllers
 */
ZmApp.prototype.getNumSessionControllers =
function(type) {
    var controllers = this._sessionController[type] || [];
    var activeCount = 0;
    for (var id in controllers) {
        if (!controllers[id].inactive) {
            activeCount++;
        }
    }
    return activeCount;
};

/**
 * Evaluates the controller class and returns the default view type from that controller.
 *
 * @param	{hash}							params						hash of params:
 * @param	{string}						controllerClass				string name of controller class
 * @param	{string}						sessionId					unique identifier for this controller
 * @param 	{ZmSearchResultsController}		searchResultsController		containing controller
 *
 * @returns	{string}													default view type
 */
ZmApp.prototype.getTypeFromController =
function(controllerClass) {
	var controller = eval(controllerClass);
	if (!controller.getDefaultViewType) {
		throw new AjxException("Session controller " + controllerClass + " must implement getDefaultViewType()");
	}
	return controller.getDefaultViewType();
};

/**
 * Returns a controller of the given type and class. If no sessionId is provided, then
 * the controller's session ID will be an incremental number. If a sessionId is given,
 * then a check is made for an existing controller with that session ID. If none is
 * found, one is created and given that session ID.
 * 
 * @param	{hash}							params						hash of params:
 * @param	{string}						controllerClass				string name of controller class
 * @param	{string}						sessionId					unique identifier for this controller
 * @param 	{ZmSearchResultsController}		searchResultsController		containing controller
 */
ZmApp.prototype.getSessionController =
function(params) {

	var type = this.getTypeFromController(params.controllerClass);

	// track controllers of this type
	if (!this._sessionController[type]) {
		this._sessionController[type] = {};
		this._nextSessionId[type] = 1;
	}

	// check if we've already created a session controller with the given ID
	var sessionId = params.sessionId;
	if (sessionId && this._sessionController[type][sessionId]) {
		return this._sessionController[type][sessionId];
	}

	// re-use an inactive controller if possible
	var controller;
	if (!sessionId) {
		var controllers = this._sessionController[type];
		for (var id in controllers) {
			if (controllers[id].inactive && !controllers[id].isPinned && !controllers[id].isHidden) {
				controller = controllers[id];
				break;
			}
		}
	}

	sessionId = (controller && controller.getSessionId()) || sessionId || String(this._nextSessionId[type]++);

	if (!controller) {
		var ctlrClass = eval(params.controllerClass);
		controller = this._sessionController[type][sessionId] =
			new ctlrClass(this._container, this, type, sessionId, params.searchResultsController);
	}
	this._curSessionId[type] = sessionId;
	controller.inactive = false;

	return controller;
};

/**
 * Deletes a controller of the given type, class, and sessionId.
 *
 * @param	{hash}							params						hash of params:
 * @param	{string}						controllerClass				string name of controller class
 * @param	{string}						sessionId					unique identifier for this controller
 * @param 	{ZmSearchResultsController}		searchResultsController		containing controller
 */
ZmApp.prototype.deleteSessionController =
function(params) {
	var type		= this.getTypeFromController(params.controllerClass);
	var sessionId	= params.sessionId;

	if (!this._sessionController[type]) {
		return;
	}
	delete this._sessionController[type][sessionId];
};

// returns the session ID of the most recently retrieved controller of the given type
ZmApp.prototype.getCurrentSessionId =
function(type) {
	return this._curSessionId[type];
};

// returns a list of this app's controllers
ZmApp.prototype.getAllControllers =
function() {

	var controllers = [];
	for (var viewType in this._sessionController) {
		var viewHash = this._sessionController[viewType];
		if (viewHash) {
			for (var viewId in viewHash) {
				var ctlr = viewHash[viewId];
				if (ctlr) {
					controllers.push(ctlr);
				}
			}
		}
	}
	
	return controllers;
};

/**
 * @private
 */
ZmApp.prototype._addSettingsChangeListeners =
function() {
	if (!this._settingListener) {
		this._settingListener = new AjxListener(this, this._settingChangeListener);
	}
};

/**
 * @private
 */
ZmApp.prototype._settingChangeListener =
function(ev) {

};

// Returns a hash of properties for the New Button
ZmApp.prototype.getNewButtonProps =
function() {
	return {};
};

/**
 * Gets the search parameters.
 *
 * @param {Hash}	params	a hash of arguments for the search
 * @see		ZmSearchController
 */
ZmApp.prototype.getSearchParams =
function(params) {
	return (params || {});
};

/**
 * Default function to run after an app's core package has been loaded. It assumes that the
 * classes that define items and organizers for this app are in the core package.
 *
 * @private
 */
ZmApp.prototype._postLoadCore =
function() {
	if (!appCtxt.isChildWindow) {
		this._setupDropTargets();
	}
};

/**
 * Default function to run after an app's main package has been loaded.
 *
 * @private
 */
ZmApp.prototype._postLoad =
function(type) {
	if (type) {
		this._createDeferredFolders(type);
	}
	this._handleDeferredNotifications();
    if(appCtxt.isExternalAccount()) {
        this._handleExternalAccountSettings(type);
    }
};


ZmApp.prototype.containsWritableFolder =
function() {
    return appCtxt.isExternalAccount() ? (this._containsWritableFolder ? true : false) : true;
};

ZmApp.prototype.getDefaultFolderId =
function() {
    return this._defaultFolderId;
};

/**
 * @private
 */
ZmApp.prototype._handleExternalAccountSettings =
function(type) {
    //Handle the external account settings
    var dataTree = appCtxt.getTree(type, appCtxt.getActiveAccount()),
        folders = dataTree ? dataTree.getByType(type) : [],
        len = folders.length,
        folder,
        i;
    this._containsWritableFolder = false;
    for (i=0; i<len; i++) {
        folder = folders[i];
        if(!this._defaultFolderId) { this._defaultFolderId = folder.id; }
        if (folder.isPermAllowed(ZmOrganizer.PERM_WRITE)) {
            this._containsWritableFolder = true;
        }
    }
};

/**
 * @private
 */
ZmApp.prototype._setupDropTargets =
function() {
	var appTargets = ZmApp.DROP_TARGETS[this._name];
	for (var type in appTargets) {
		var targets = appTargets[type];
		for (var i = 0; i < targets.length; i++) {
			var orgType = targets[i];
			var ctlr = appCtxt.getOverviewController().getTreeController(orgType, true);
			var className = ZmList.ITEM_CLASS[type] || ZmOrganizer.ORG_CLASS[type];
			if (ctlr) {
				ctlr._dropTgt.addTransferType(className);
			} else {
				if (!ZmTreeController.DROP_SOURCES[orgType]) {
					ZmTreeController.DROP_SOURCES[orgType] = [];
				}
				ZmTreeController.DROP_SOURCES[orgType].push(className);
			}
		}
	}
};

/**
 * Disposes of the tree controllers (right now mainly gets rid of change listeners.
 */
ZmApp.prototype.disposeTreeControllers =
function() {

	var overviewController = appCtxt.getOverviewController(true); //see if overview controller was created (false param means it won't create it if not created)
	//this is created lazily in case of child window. There's nothing to do if it was not created.
	if (!overviewController) {
		return;
	}

	var appTargets = ZmApp.DROP_TARGETS[this._name];
	for (var type in appTargets) {
		var targets = appTargets[type];
		for (var i = 0; i < targets.length; i++) {
			var orgType = targets[i];
			var treeController = overviewController.getTreeController(orgType, true);
			if (!treeController) {
				continue;
			}
			treeController.dispose();
		}
	}
};


/**
 * @private
 */
ZmApp.prototype.createDeferred = function() {
	var types = ZmOrganizer.APP2ORGANIZER[this._name] || [];
	for (var i = 0; i < types.length; i++) {
		var type = types[i];
		var packageName = ZmOrganizer.ORG_PACKAGE[type];
		AjxDispatcher.require(packageName);
		this._createDeferredFolders(type);
	}
};

/**
 * Lazily create folders received in the initial <refresh> block.
 *
 * @private
 */
ZmApp.prototype._createDeferredFolders =
function(type) {
	for (var i = 0; i < this._deferredFolders.length; i++) {
		var params = this._deferredFolders[i];
		var folder = ZmFolderTree.createFolder(params.type, params.parent, params.obj, params.tree, params.path, params.elementType);
        if (appCtxt.isExternalAccount() && folder.isSystem()) {
            continue;
        }
		params.parent.children.add(folder); // necessary?
		folder.parent = params.parent;
		ZmFolderTree._traverse(folder, params.obj, params.tree, params.path || []);
	}
	this._clearDeferredFolders();
};

/**
 * @private
 */
ZmApp.prototype._clearDeferredFolders =
function() {
	this._deferredFolders = [];
	this._deferredFolderHash = {};
};

/**
 * Defer notifications if this app's main package has not been loaded.
 * Returns true if notifications were deferred.
 *
 * @param type	[string]	type of notification (delete, create, or modify)
 * @param data	[array]		list of notifications
 *
 * TODO: revisit use of MAIN_PKG, it's hokey
 *
 * @private
 */
ZmApp.prototype._deferNotifications =
function(type, data) {
	var pkg = ZmApp.MAIN_PKG[this._name];
	if (pkg && !AjxDispatcher.loaded(pkg)) {
		this._deferredNotifications.push({type:type, data:data});
		return true;
	} else {
		this._noDefer = true;
		return false;
	}
};

/**
 * @private
 */
ZmApp.prototype._handleDeferredNotifications =
function() {
	var dns = this._deferredNotifications;
	for (var i = 0; i < dns.length; i++) {
		var dn = dns[i];
		if (dn.type == "delete") {
			this.deleteNotify(dn.data, true);
		} else if (dn.type == "create") {
			this.createNotify(dn.data, true);
		} else if (dn.type == "modify") {
			this.modifyNotify(dn.data, true);
		}
	}
};

/**
 * Notify change listeners with a list of notifications, rather than a single
 * item, so that they can optimize. For example, a list view can wait to
 * fix its alternation of dark and light rows until after all the moved ones
 * have been taken out, rather than after the removal of each row.
 *
 * @param mods	{Array}		list of notification objects
 */
ZmApp.prototype._batchNotify =
function(mods) {

	if (!(mods && mods.length >= ZmApp.BATCH_NOTIF_LIMIT)) { return; }

	var notifs = {}, item, gotOne = false;
	for (var i = 0, len = mods.length; i < len; i++) {
		var mod = mods[i];
		item = appCtxt.cacheGet(mod.id);
		if (item) {
			var ev = item.notifyModify(mod, true);
			if (ev) {
				if (!notifs[ev]) {
					notifs[ev] = [];
				}
				mod.item = item;
				notifs[ev].push(mod);
				gotOne = true;
			}
		}
	}

	if (!gotOne || !item) { return; }

	var list = item.list;
	if (!list) { return; }
	list._evt.batchMode = true;
	list._evt.item = item;	// placeholder - change listeners like it to be there
	list._evt.items = null;
	for (var ev in notifs) {
		var details = {notifs:notifs[ev]};
		list._notify(ev, details);
	}
};

/**
 * Depending on "Always in New Window" option and whether Shift key is pressed,
 * determine whether action should be in new window or not.
 *
 * @private
 */
ZmApp.prototype._inNewWindow =
function(ev) {
	if (appCtxt.isWebClientOffline()) {
		return false;
	}  else {
		var setting = appCtxt.get(ZmSetting.NEW_WINDOW_COMPOSE);
		return !ev ? setting : ((!setting && ev && ev.shiftKey) || (setting && ev && !ev.shiftKey));
	}
};

/**
 * @private
 */
ZmApp.prototype._handleCreateFolder =
function(create, org) {
	var parent = appCtxt.getById(create.l);
	if (parent && (ZmOrganizer.VIEW_HASH[org][create.view])) {
		parent.notifyCreate(create, "folder");
		create._handled = true;
	}
};

/**
 * @private
 */
ZmApp.prototype._handleCreateLink =
function(create, org) {
	var parent = appCtxt.getById(create.l);
	var view = create.view || "message";
	if (parent && parent.supportsSharing() && (ZmOrganizer.VIEW_HASH[org][view])) {
		parent.notifyCreate(create, "link");
		create._handled = true;
	}
};

// Abstract/protected methods

/**
 * Launches the application.
 *
 * @param	{Hash}	params		a hash of parameters
 * @param	{AjxCallback}	callback		the callback
 */
ZmApp.prototype.launch =
function(params, callback) {
	this.createDeferred();
    if (callback) {
        callback.run();
    }
};

/**
 * Activates the application.
 *
 * @param	{Boolean}	active	<code>true</code> if the application is active
 * @param	{string}	viewId	ID of view becoming active
 */
ZmApp.prototype.activate =
function(active, viewId) {
	this._active = active;
	if (active) {
		appCtxt.getAppController().setNewButtonProps(this.getNewButtonProps());
		this.setOverviewPanelContent();
		this.stopAlert();
		if (appCtxt.isWebClientOfflineSupported) {
			this.resetWebClientOfflineOperations();
		}
		this._setRefreshButtonTooltip();
	}
};

/**
 * Handle the common aspects of a transition from online to offline and offline to online, and also do so
 * when an app is activated
 */
ZmApp.prototype.resetWebClientOfflineOperations =
function() {
	var isWebClientOnline = !appCtxt.isWebClientOffline();
	var overview = this.getOverview();
	if (overview) {
		var zimletTreeView = overview.getTreeView(ZmOrganizer.ZIMLET);
		if (zimletTreeView) {
			zimletTreeView.setVisible(isWebClientOnline);
		}
		// enable/disable right click
		overview.actionSupported = isWebClientOnline;
		// enable/disable drag and drop
		overview.dndSupported = isWebClientOnline;
	}
	// new button enable/disable
	var newButton = appCtxt.getAppController().getNewButton();
	if (newButton) {
		if (ZmController._defaultNewId === ZmOperation.NEW_MESSAGE) {
			newButton._setDropDownCellMouseHandlers(isWebClientOnline);
		}
		else {
			newButton.setEnabled(isWebClientOnline);
		}
	}
};

/**
 * Checks if the application is active.
 *
 * @return	{Boolean}	<code>true</code> if the application is active
 */
ZmApp.prototype.isActive =
function() {
	return this._active;
};

/**
 * Resets the application state.
 *
 * @return	{Boolean}	<code>true</code> if the application is active
 */
ZmApp.prototype.reset =
function(active) {
};

/**
 * Starts an alert on the application tab.
 * 
 */
ZmApp.prototype.startAlert =
function() {
	AjxDispatcher.require("Alert");
	this._alert = this._alert || new ZmAppAlert(this);
	this._alert.start();
};

/**
 * Stops an alert on the application tab.
 */
ZmApp.prototype.stopAlert =
function() {
	if (this._alert) {
		this._alert.stop();
	}
};

ZmApp.prototype._setRefreshButtonTooltip =
function() {
	if (appCtxt.refreshButton) {
		appCtxt.refreshButton.setToolTipContent(this._getRefreshButtonTooltip());
	}
};

/**
 * this is the default refresh button tooltip. overridden in Calendar. (see bug 85965)
 * @private
 */
ZmApp.prototype._getRefreshButtonTooltip =
function() {
	 return ZmMsg.checkMailPrefUpdate;
};

/**
 * @private
 */
ZmApp.prototype._notifyRendered =
function() {
	if (!this._hasRendered) {
		appCtxt.getAppController().appRendered(this._name);
		this._hasRendered = true;
	}
	this.stopAlert();
};

/**
 * @private
 */
ZmApp.prototype._getExternalAccount =
function() {

	// bug #43464 - get the first non-local account that supports this app
	var defaultAcct;
	if (appCtxt.multiAccounts) {
		var accounts = appCtxt.accountList.visibleAccounts;
		for (var i = 0; i < accounts.length; i++) {
			var acct = accounts[i];
			if (acct.isMain) { continue; }

			if (appCtxt.get(ZmApp.SETTING[this.name], null, acct)) {
				defaultAcct = acct;
				break;
			}
		}
	}
	return defaultAcct;
};

/**
 * Sets a hidden div for performance metrics.  Marks the time an app has been launched
 * @param appName {String}
 * @param date {Date}
 * @private
 */
ZmApp.prototype._setLaunchTime = 
function(appName, date) {
	if (!window.isPerfMetric) {
		return;
	}
	var id = appName + "_launched";
	if (!date) {
		date = new Date();
	}
	if (!document.getElementById(id)) {
		var div = document.createElement("DIV");
		div.id = id;
		div.innerHTML = date.getTime();
		div.style.display = "none";
		document.body.appendChild(div);
	}
	if (window.appDevMode) {
		console.profile(id);
	}
};

/**
 * Sets a hidden div for performance metrics.  Marks the time an app has completed loading
 * @param appName {String}
 * @param date {Date}
 * @private
 */
ZmApp.prototype._setLoadedTime = 
function(appName, date) {
	if (!window.isPerfMetric) {
		return;
	}
	var id = appName + "_loaded";
	if (!date) {
		date = new Date();
	}
	if (!document.getElementById(id)) {
		var div = document.createElement("DIV");
		div.id = id;
		div.innerHTML = date.getTime();
		div.style.display = "none";
		document.body.appendChild(div);
	}
	if (window.appDevMode) {
		console.profileEnd();
	}
};
}
if (AjxPackage.define("zimbraMail.share.ZmSearchApp")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the search application class.
 */

/**
 * Creates and initializes the search application.
 * @constructor
 * @class
 * The search app manages user-initiated searches.
 *
 * @param	{DwtControl}	container		the container
 * @param	{ZmController}	parentController	the parent window controller (set by the child window)
 * 
 * @author Conrad Damon
 * 
 * @extends		ZmApp
 */
ZmSearchApp = function(container, parentController) {

	ZmApp.call(this, ZmApp.SEARCH, container, parentController);

	this._groupBy = appCtxt.get(ZmSetting.GROUP_MAIL_BY);
};

ZmSearchApp.prototype = new ZmApp;
ZmSearchApp.prototype.constructor = ZmSearchApp;

ZmSearchApp.prototype.isZmSearchApp = true;
ZmSearchApp.prototype.toString = function() {	return "ZmSearchApp"; };

ZmApp.SEARCH					= ZmId.APP_SEARCH;
ZmApp.CLASS[ZmApp.SEARCH]		= "ZmSearchApp";
ZmApp.SETTING[ZmApp.SEARCH]		= ZmSetting.SEARCH_ENABLED;

ZmSearchApp.CONTROLLER_CLASS = "ZmSearchResultsController";

ZmSearchApp.prototype.getSearchResultsController =
function(sessionId, appName) {
	return this.getSessionController({
				controllerClass:	ZmSearchApp.CONTROLLER_CLASS,
				sessionId:			sessionId,
				appName:			appName
			});
};

// override so we don't try to set overview panel content
ZmSearchApp.prototype.activate =
function(active) {
	this._active = active;
};

// Not hooked up for activate, but it will be called after displaying the search results
ZmSearchApp.prototype.resetWebClientOfflineOperations =
function(searchResultsController) {
	ZmApp.prototype.resetWebClientOfflineOperations.apply(this);
	if (!searchResultsController) {
		var controllerType = this.getTypeFromController(ZmSearchApp.CONTROLLER_CLASS);
		var sessionId = this.getCurrentSessionId(controllerType);
		searchResultsController = this.getSearchResultsController(sessionId);
	}
	// Only Save affected currently
	var searchResultsToolBar = searchResultsController && searchResultsController._toolbar;
	var saveButton = searchResultsToolBar && searchResultsToolBar.getButton(ZmSearchToolBar.SAVE_BUTTON);
	if (saveButton) {
		saveButton.setEnabled(!appCtxt.isWebClientOffline());
	}
};

// search app maintains its own "group mail by" setting
ZmSearchApp.prototype.getGroupMailBy = function() {
	return ZmMailApp.prototype.getGroupMailBy.call(this);
};

ZmSearchApp.prototype.setGroupMailBy = 	function(groupBy) {
	this._groupBy = groupBy;
};
}
if (AjxPackage.define("zimbraMail.share.ZmSocialApp")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Creates the social application.
 * @class
 * This class represents the social application.
 *
 * @param	{DwtControl}	container		the container
 *
 * @extends		ZmApp
 */
ZmSocialApp = function(container) {
	ZmApp.call(this, ZmApp.SOCIAL, container);
}

ZmSocialApp.prototype = new ZmApp;
ZmSocialApp.prototype.constructor = ZmSocialApp;

ZmSocialApp.prototype.isZmSocialApp = true;
ZmSocialApp.prototype.toString = function() {	return "ZmSocialApp"; };

//
// Constants
//

ZmApp.SOCIAL                        = ZmId.APP_SOCIAL;
ZmApp.CLASS[ZmApp.SOCIAL]		    = "ZmSocialApp";
ZmApp.SETTING[ZmApp.SOCIAL]		    = ZmSetting.SOCIAL_ENABLED;
ZmApp.UPSELL_SETTING[ZmApp.SOCIAL]	= ZmSetting.SOCIAL_EXTERNAL_ENABLED;
ZmApp.LOAD_SORT[ZmApp.SOCIAL]	    = 100;

ZmSocialApp.prototype._registerApp = function() {
	ZmApp.registerApp(ZmApp.SOCIAL, {
		nameKey:            "communityName",
		icon:               "Globe",
		chooserTooltipKey:  "goToSocial",
		chooserSort:        32,
		defaultSort:        100,
		upsellUrl:			ZmSetting.SOCIAL_EXTERNAL_URL
	});

	// overwrite community name with value from settings, if any
	var appName = appCtxt.get(ZmSetting.SOCIAL_NAME);
	if (appName) {
		ZmMsg[ZmApp.NAME[this._name]] = appName;
	}
};

// User has clicked refresh button
ZmSocialApp.prototype.runRefresh = function() {

	var mainCtlr = appCtxt.getAppController(),
		communityView = mainCtlr._appIframeView[ZmApp.SOCIAL];

	if (communityView) {
		communityView.runRefresh();
	}
};
}

if (AjxPackage.define("zimbraMail.share.view.ZmPopupMenu")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * Creates a popup menu.
 * @class
 * This class represents a basic popup menu which can add menu items, manage listeners, and
 * enable/disabled its menu items.
 *
 * @author Conrad Damon
 *
 * @param {DwtComposite}	parent		the containing widget
 * @param {string}	className		the CSS class
 * @param {string}	id			an explicit ID to use for the control's HTML element
 * @param {ZmController}	controller	the owning controller
 * 
 * @extends		DwtMenu
 */
ZmPopupMenu = function(parent, className, id, controller) {

	if (arguments.length == 0) return;
	var params = Dwt.getParams(arguments, ZmPopupMenu.PARAMS);
	params.className = params.className ? params.className : "ActionMenu";
	params.style = params.style || DwtMenu.POPUP_STYLE;
    params.id = params.id || "POPUP_" + Dwt.getNextId();
	DwtMenu.call(this, params);

	controller = controller || appCtxt.getCurrentController();
	if (controller) {
		this._controller = controller;
		this._keyMap = this._controller.getKeyMapName();
	}

	this._menuItems = {};
};

ZmPopupMenu.PARAMS = ["parent", "className", "id", "controller"];

ZmPopupMenu.prototype = new DwtMenu;
ZmPopupMenu.prototype.constructor = ZmPopupMenu;

ZmPopupMenu.prototype.isZmPopupMenu = true;
ZmPopupMenu.prototype.toString = function() { return "ZmPopupMenu"; };

/**
 * Adds a section listener.
 * 
 * @param	{string}		menuItemId		the menu item id
 * @param	{AjxListener}	listener		the selection listener
 * @param	{number}		index				the index where to insert the listener
 */
ZmPopupMenu.prototype.addSelectionListener =
function(menuItemId, listener, index) {
	var menuItem = this._menuItems[menuItemId];
	if (menuItem) {
		menuItem.addSelectionListener(listener, index);
	}
};

/**
 * Removes a section listener.
 * 
 * @param	{string}		menuItemId		the menu item id
 * @param	{AjxListener}	listener		the selection listener
 */
ZmPopupMenu.prototype.removeSelectionListener =
function(menuItemId, listener) {
	var menuItem = this._menuItems[menuItemId];
	if (menuItem) {
		menuItem.removeSelectionListener(listener);
	}
};

ZmPopupMenu.prototype.popup =
function(delay, x, y, kbGenerated) {
	delay = delay ? delay : 0;
	x = (x != null) ? x : Dwt.DEFAULT;
	y = (y != null) ? y : Dwt.DEFAULT;
	DwtMenu.prototype.popup.call(this, delay, x, y, kbGenerated);
};

/**
 * Enables/disables menu items.
 *
 * @param {array}	ids		a list of menu item IDs
 * @param {boolean}		enabled	if <code>true</code>, enable the menu items
 */
ZmPopupMenu.prototype.enable =
function(ids, enabled) {
	ids = (ids instanceof Array) ? ids : [ids];
	for (var i = 0; i < ids.length; i++) {
		if (this._menuItems[ids[i]]) {
			this._menuItems[ids[i]].setEnabled(enabled);
		}
	}
};

/**
 * Enables/disables all menu items.
 *
 * @param {boolean}		enabled	if <code>true</code>, enable the menu items
 */
ZmPopupMenu.prototype.enableAll =
function(enabled) {
	for (var i in this._menuItems) {
		this._menuItems[i].setEnabled(enabled);
	}
};

/**
 * Creates a menu item and adds the item to this menu.
 *
 * @param {string}	id			the menu item ID
 * @param {hash}	params		a hash of parameters
 * @param {string}	params.text		the menu item text
 * @param {string}	params.image		the icon class for the or menu item
 * @param {string}	params.disImage	disabled version of icon
 * @param {boolean}	params.enabled		if <code>true</code>, menu item is enabled
 * @param {constant}	params.style			the menu item style
 * @param {string}	params.radioGroupId	the ID of radio group for this menu item
 * @param {constant}	params.shortcut		the shortcut ID (from {@link ZmKeyMap}) for showing hint
 * 
 * @see		DwtMenuItem
 */
ZmPopupMenu.prototype.createMenuItem =
function(id, params, htmlElId) {
	var mi = this._menuItems[id] = new DwtMenuItem({parent:this, style:params.style, radioGroupId:params.radioGroupId,
													id: (htmlElId || params.id || id), index: params.index});
	if (params.image) {
		mi.setImage(params.image);
	}
	if (params.text) {
		mi.setText(AjxStringUtil.htmlEncode(params.text));
	}
	if (params.shortcut) {
		mi.setShortcut(appCtxt.getShortcutHint(this._keyMap, params.shortcut));
	}

	mi.setEnabled(params.enabled !== false);
	mi.setData(ZmOperation.MENUITEM_ID, id);

    //set context menu tr id
    var row = mi.getRowElement();
	if (row) {
		row.setAttribute("id", "POPUP" + "_" + mi.getHTMLElId().toString().replace(/\s/g, ""));
    }

	return mi;
};

/**
 * Gets the menu item with the given ID.
 *
 * @param {string}	id		an operation ID
 * @return	{DwtMenuItem}		the menu item
 */
ZmPopupMenu.prototype.getMenuItem =
function(id) {
	return this._menuItems[id];
};

/**
 * sets an item visibility. finds the menu item by id. 
 *
 * @param	{String}	id  the operation id
 * @param	{Boolean}	visible
 */
ZmPopupMenu.prototype.setItemVisible =
function(id, visible) {
	var item = this.getMenuItem(id);
	if (!item) {
		return;
	}
	item.setVisible(visible);
};

/**
 * Gets the menu items.
 *
 * @return	{array}		an array of {@link DwtMenuItem} objects
 */
ZmPopupMenu.prototype.getMenuItems =
function() {
	return this._menuItems;
};

/**
 * Gets the menu search sub-menu (if any).
 *
 * @return {DwtMenu}        the menu
 */
ZmPopupMenu.prototype.getSearchMenu =
function() {
    var menuItem = this.getMenuItem(ZmOperation.SEARCH_MENU);
    if (menuItem) {
        return menuItem.getMenu();
    }
};

ZmPopupMenu.prototype.getContactGroupMenu =
function() {
	var menuItem = this.getMenuItem(ZmOperation.CONTACTGROUP_MENU);
	if (menuItem) {
		return menuItem.getMenu();
	}
};


/**
 * Creates a menu item separator.
 * 
 * @return	{DwtMenuItem}	the separator menu item
 */
ZmPopupMenu.prototype.createSeparator =
function(params) {
	new DwtMenuItem({parent:this, style:DwtMenuItem.SEPARATOR_STYLE, index:params && params.index});
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmActionMenu")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * Creates an action menu with the given menu items.
 * @class
 * This class represents an action menu, which is a popup menu with a few added features.
 * It can be easily created using a set of standard operations, and/or custom menu items
 * can be provided. This class is designed for use with items ({@link ZmItem}), so it can for
 * example contain a tab submenu. See also {@link ZmButtonToolBar}.
 *
 * @author Conrad Damon
 *
 * @param {hash}	params		a hash of parameters
 * @param {DwtComposite}	params.parent		the containing widget
 * @param {ZmController}	params.controller	the owning controller
 * @param {array}	params.menuItems	a list of operation IDs
 * @param {hash}	params.overrides	a hash of overrides by op ID
 * @param {string}	params.context		the context (used to create ID)
 * @param {constant}	params.menuType		the menu type (used to generate menu item IDs)
 * 
 * @extends		ZmPopupMenu
 */
ZmActionMenu = function(params) {

    var id = params.id || (params.context ? ZmId.getMenuId(params.context, params.menuType) : null);
	ZmPopupMenu.call(this, params.parent, null, id, params.controller);

	// standard menu items default to Tag/Print/Delete
	var menuItems = params.menuItems;
	if (!menuItems) {
		menuItems = [ZmOperation.TAG_MENU, ZmOperation.PRINT, ZmOperation.DELETE];
	} else if (menuItems == ZmOperation.NONE) {
		menuItems = null;
	}
	// weed out disabled ops, save list of ones that make it
	this.opList = ZmOperation.filterOperations(menuItems);
	this._context = params.context;
	this._menuType = params.menuType;

	this._menuItems = ZmOperation.createOperations(this, this.opList, params.overrides);
};

ZmActionMenu.prototype = new ZmPopupMenu;
ZmActionMenu.prototype.constructor = ZmActionMenu;

ZmActionMenu.prototype.isZmActionMenu = true;
ZmActionMenu.prototype.toString = function() { return "ZmActionMenu"; };


// Public methods


/**
 * Creates a menu item and adds its operation ID as data.
 * 
 * @param {String}	id			the name of the operation
 * @param	{hash}	params		a hash of parameters
 * @param  {string}	params.text			the menu item text
 * @param {string}	params.image			the icon class for the menu item
 * @param {string}	params.disImage		the disabled version of icon
 * @param {boolean}	params.enabled		if <code>true</code>, menu item is enabled
 * @param {constant}	params.style			the menu item style
 * @param {string}	params.radioGroupId	the ID of radio group for this menu item
 * @param {constant}	params.shortcut		the shortcut ID (from {@link ZmKeyMap}) for showing hint
 * 
 * @private
 */
ZmActionMenu.prototype.createOp =
function(id, params, elementId) {
	params.id = this._context ? ZmId.getMenuItemId(this._context, id, this._menuType) : null;
	var mi = this.createMenuItem(id, params, elementId);
	mi.setData(ZmOperation.KEY_ID, id);

	return mi;
};

ZmActionMenu.prototype.addOp =
function(id) {
	ZmOperation.addOperation(this, id, this._menuItems);
};

ZmActionMenu.prototype.removeOp =
function(id) {
	ZmOperation.removeOperation(this, id, this._menuItems);
};

/**
 * Gets the menu item with the given ID.
 *
 * @param {constant}	id		an operation ID
 * @return	{DwtMenuItem}	the menu item
 * @see		ZmOperation
 */
ZmActionMenu.prototype.getOp =
function(id) {
	return this.getMenuItem(id);
};

/**
 * Gets the menu tag sub-menu (if any).
 * 
 * @return	{DwtMenu}		the menu
 */
ZmActionMenu.prototype.getTagMenu =
function() {
	var menuItem = this.getMenuItem(ZmOperation.TAG_MENU);
	if (menuItem) {
		return menuItem.getMenu();
	}
};


// Private methods

// Returns the ID for the given menu item.
ZmActionMenu.prototype._menuItemId =
function(menuItem) {
	return menuItem.getData(ZmOperation.KEY_ID);
};

ZmActionMenu.prototype.removeMenuItemById =
function(menuItemId) {
    var mi = this.getMenuItem(menuItemId);
    this.removeMenuItem(mi);
};

ZmActionMenu.prototype.removeMenuItem =
function(menuItem) {
    if (!menuItem) {return};
    this.removeChild(menuItem);
    menuItem.dispose();
};

}
if (AjxPackage.define("zimbraMail.share.view.ZmToolBar")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines a toolbar.
 */

/**
 * Creates a toolbar.
 * @class
 * This class represents a basic toolbar which can add buttons, manage listeners, and
 * enable/disabled its buttons.
 *
 * @author Conrad Damon
 *
 * @param {Hash}	params		a hash of parameters
 * @param	{DwtComposite}	params.parent		the containing widget
 * @param	{String}	params.className	the CSS class
 * @param	{constant}	params.posStyle		the positioning style
 * @param	{String}	params.id			an explicit ID to use for the control's HTML element
 * @param	{ZmController}	params.controller	the owning controller
 * @param	{String}	params.refElementId	the id of element that contains toolbar
 *        
 * @extends	DwtToolBar
 */
ZmToolBar = function(params) {
	if (arguments.length == 0) return;

	params.posStyle = params.posStyle || DwtControl.ABSOLUTE_STYLE;
	DwtToolBar.call(this, params);

	var controller = params.controller || appCtxt.getCurrentController();
	if (controller) {
		this._controller = controller;
		this._keyMap = this._controller.getKeyMapName();
	}

	this._refElementId = params.refElementId;
	this._buttons = {};
};

ZmToolBar.prototype = new DwtToolBar;
ZmToolBar.prototype.constructor = ZmToolBar;

ZmToolBar.prototype.isZmToolBar = true;
ZmToolBar.prototype.toString = function() { return "ZmToolBar"; };

/**
 * Adds a selection listener.
 * 
 * @param	{String}	buttonId	the button id
 * @param	{AjxListener}	listener	the listener
 */
ZmToolBar.prototype.addSelectionListener =
function(buttonId, listener) {
	var button = this._buttons[buttonId];
	if (button) {
		button.addSelectionListener(listener);
	}
};

/**
 * Removes a selection listener.
 * 
 * @param	{String}	buttonId	the button id
 * @param	{AjxListener}	listener	the listener
 */
ZmToolBar.prototype.removeSelectionListener =
function(buttonId, listener) {
	var button = this._buttons[buttonId];
	if (button) {
		button.removeSelectionListener(listener);
	}
};

/**
 * Gets the button.
 * 
 * @param	{String}	buttonId	the button id
 * @return	{ZmAppButton}	the button
 */
ZmToolBar.prototype.getButton =
function(buttonId) {
	return this._buttons[buttonId];
};

/**
 * sets an item visibility. finds the button by id. 
 *
 * @param	{String}	buttonId	the button id
 * @param	{Boolean}	visible
 */
ZmToolBar.prototype.setItemVisible =
function(buttonId, visible) {
	var button = this.getButton(buttonId);
	if (!button) {
		return;
	}
	button.setVisible(visible);
};


/**
 * Sets the data.
 * 
 * @param	{String}	buttonId	the button id
 * @param	{String}	key		the data key
 * @param	{Object}	data	the data
 */
ZmToolBar.prototype.setData = 
function(buttonId, key, data) {
	this._buttons[buttonId].setData(key, data);
};

/**
 * Enables or disables the specified buttons.
 *
 * @param {Array}	ids		a list of button ids
 * @param {Boolean}	enabled	if <code>true</code>, enable the buttons
 */
ZmToolBar.prototype.enable =
function(ids, enabled) {
	ids = (ids instanceof Array) ? ids : [ids];
	for (var i = 0; i < ids.length; i++) {
		if (this._buttons[ids[i]]) {
			this._buttons[ids[i]].setEnabled(enabled);
		}
	}
};

ZmToolBar.prototype.setSelected =
function(id) {
    var oldButton = this._selectedId ? this._buttons[this._selectedId] : null;
    var newButton = id ? this._buttons[id] : null;
    if (oldButton) {
        oldButton.setSelected(false);
    }
    if (newButton) {
        newButton.setSelected(true);
        this._selectedId = id;
    }
};

/**
 * Enables or disables all buttons.
 *
 * @param {Boolean}	enabled			if <code>true</code>, enable the buttons
 */
ZmToolBar.prototype.enableAll =
function(enabled) {
	for (var i in this._buttons) {
		this._buttons[i].setEnabled(enabled);
	}
};

/**
 * Creates a button and adds the button to this toolbar.
 *
 * @param {String}	id			the button id
 * @param {Hash}	params		a hash of parameters:
 * @param {function}	params.constructor	the constructor for button object (default is {@link DwtToolBarButton})
 * @param {String}	params.template		the button template
 * @param {String}	params.text			the button text
 * @param {String}	params.tooltip		the button tooltip text
 * @param {String}	params.image		the icon class for the button
 * @param {String}	params.disImage	the disabled version of icon
 * @param {Boolean}	params.enabled		if <code>true</code>, button is enabled
 * @param {String}	params.className	the CSS class name
 * @param {String}	params.style		the button style
 * @param {int}	params.index			the position at which to add the button
 * @param {constant}	params.shortcut		the shortcut id (from {@link ZmKeyMap}) for showing hint
 * @param {AjxCallback|DwtMenu}	params.menu				the menu creation callback (recommended) or menu
 * @param {Boolean}	params.menuAbove	if <code>true</code>, popup menu above the button.
 *
 * @param {Object}	params.whatToShow		if exists, determines what to show as follows: (for usage, see ZmToolBar.prototype._createButton and DwtButton.prototype.setImage and DwtButton.prototype.setText
 * @param {Boolean}	params.whatToShow.showImage		if <code>true</code>, display image
 * @param {Boolean}	params.whatToShow.showText		if <code>true</code>, display text
 *
 */
ZmToolBar.prototype.createButton =
function(id, params) {
	var b = this._buttons[id] = this._createButton(params);
	if (params.image) {
		b.setImage(params.image);
	}
	if (params.text) {
		b.setText(params.text);
	}
	if (params.tooltip) {
		b.setToolTipContent(ZmOperation.getToolTip(id, this._keyMap) || params.tooltip, true);
	}
	b.setEnabled(params.enabled !== false);
	b.setData("_buttonId", id);
	if (params.menu) {
		b.setMenu(params.menu, false, null, params.menuAbove);
	}

	return b;
};

//
// Data
//

ZmToolBar.prototype.SEPARATOR_TEMPLATE = "share.Widgets#ZmToolBarSeparator";

//
// Protected methods
//

/**
 * @private
 */
ZmToolBar.prototype._createButton =
function(params, className) {
	var ctor = params.ctor || DwtToolBarButton;
    var button = new ctor({
		parent:this,
		style:params.style,
		className:className,
		index:params.index,
		id:params.id,
		template: params.template,
		role: params.role,
		ariaLabel: params.ariaLabel
	});
	button.textPrecedence = params.textPrecedence;
	button.imagePrecedence = params.imagePrecedence;
	button.whatToShow = params.whatToShow;

	return button;
};

/**
 * @private
 */
ZmToolBar.prototype._buttonId =
function(button) {
	return button.getData("_buttonId");
};

/**
 * Creates an ordered list of which bits of text or images get removed when we need space.
 * 
 * @private
 */
ZmToolBar.prototype._createPrecedenceList =
function() {
	this._precedenceList = [];
	for (var id in this._buttons) {
		if (ZmOperation.isSep(id)) { continue; }
		var b = this._buttons[id];
		var tp = b.textPrecedence;
		if (tp) {
			this._precedenceList.push({id:id, type:"text", precedence:tp});
		}
		var ip = b.imagePrecedence;
		if (ip) {
			this._precedenceList.push({id:id, type:"image", precedence:ip});
		}
	}
	this._precedenceList.sort(function(a, b) {
		return (a.precedence > b.precedence) ? 1 : (a.precedence < b.precedence) ? -1 : 0;
	});
};

// The following overrides are so that we check our width after a call to a function that
// may affect our size.

/**
 * Sets the size. This method is called by the application view manager <code>fitToContainer()</code>,
 * which happens during initial layout as well as in response to the user changing the browser size.
 * 
 * @param	{int}	width	the width (in pixels)
 * @param	{int}	height	the height (in pixels)
 */
ZmToolBar.prototype.setSize =
function(width, height) {
	DBG.println("tb", "------ setSize " + width + " x " + height);
	var sz = this.getSize();
	if (sz && (width != sz.x || height != sz.y)) {
		DwtToolBar.prototype.setSize.apply(this, arguments);
	}
};

ZmToolBar.prototype.adjustSize =
function() {
	if (!this._refElementId || !this._inited) { return; }
    if (!this._refElement) {
        this._refElement = document.getElementById(this._refElementId);
    }
    var container = this._refElement && this._refElement.parentNode;
    var offsetWidth;
    if (container && ((offsetWidth = container.offsetWidth) >= 30)) {
        var style = this._refElement.style;
		style.maxWidth = style.width =  (offsetWidth - 30) + "px";
        style.overflow = "hidden";
    }
}

/**
 * Adds a button to the element with the given ID. Designed to handle non-ZmToolBar toolbars.
 * 
 * @param params	[hash]			hash of params:
 * 		  parent	[DwtControl]	parent control
 *        setting	[const]			setting that must be true for this button to be added
 *        tdId		[string]		ID of TD that is to contain this button
 *        buttonId	[string]*		ID of the button
 *        style		[const]*		button style
 *        type		[string]*		used to differentiate between regular and toolbar buttons
 *        lbl		[string]*		button text
 *        icon		[string]*		button icon
 *        tooltip	[string]*		button tooltip
 */
ZmToolBar.addButton =
function(params) {

	if (params.setting && !appCtxt.get(params.setting)) { return; }

	var button;
	var tdId = params.parent._htmlElId + (params.tdId || params.buttonId);
	var buttonEl = document.getElementById(tdId);
	if (buttonEl) {
		var btnParams = {parent:params.parent, index: params.index, style:params.style, id:params.buttonId, template: params.template, className: params.className};
		button = (params.type && params.type == "toolbar") ? (new DwtToolBarButton(btnParams)) : (new DwtButton(btnParams));
		var hint = Dwt.getAttr(buttonEl, "hint");
		ZmToolBar._setButtonStyle(button, hint, params.lbl, params.icon);
		if (params.tooltip) {
			button.setToolTipContent(params.tooltip, true);
		}
		button.reparentHtmlElement(tdId);
	}

	return button;
};

ZmToolBar._setButtonStyle =
function(button, hint, text, image) {
	if (hint == "text") {
		button.whatToShow = { showText: true };
	} else if (hint == "icon") {
		button.whatToShow = { showImage: true };
	} else { // add icon and text if no hint (or unsupported hint) provided
		button.whatToShow = { showImage: true, showText: true };
	}

	button.setText(text);
	button.setImage(image);
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmButtonToolBar")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * Creates a toolbar with the given buttons.
 * @class
 * This class represents a toolbar that contains buttons.
 * It can be easily created using a set of standard operations, and/or custom buttons
 * can be provided. This class is designed for use with items ({@link ZmItem}), so it can for
 * example contain a button with a tab submenu. See also {@link ZmActionMenu}.
 *
 * @author Conrad Damon
 *
 * @param {Hash}	params			a hash of parameters
 * @param	       {DwtComposite}	params.parent		the containing widget
 * @param	{Array}	params.buttons			a list of operation IDs
 * @param	{constant}	params.posStyle			the positioning style
 * @param	{String}	params.className			the CSS class name
 * @param	{Stirng}	params.buttonClassName	the CSS class name for buttons
 * @param	{Hash}	params.overrides			a hash of overrides by op ID
 * @param	{Array}	params.secondaryButtons		a list of operation IDs
 * @param	{Array}	params.rightSideButtons		a list of operation IDs
 * @param	{constant}	params.context			the vcontextID (used to generate button IDs)
 * @param	{constant}	params.toolbarType		the toolbar type (used to generate button IDs)
 * @param	{Boolean}	params.addTextElement		if true, add a text "button" (element) at the end (but before the view button, if any). This can be used for message counts etc
 * @param	{ZmController}	params.controller		the owning controller
 *
 * @extends		ZmToolBar
 */
ZmButtonToolBar = function(params) {
	if (arguments.length == 0) return;

	if (!params.className && (params.controller && params.controller._elementsToHide == ZmAppViewMgr.LEFT_NAV)) {
		params.className = "ZToolbar itemToolbar";
	}
    params.className = params.className || "ZToolbar";
    params.id = params.context ? ZmId.getToolbarId(params.context, params.toolbarType) : null;
    ZmToolBar.call(this, params);
	
	this._context = params.context;
	this._toolbarType = params.toolbarType;
	this._buttonStyle = params.buttonClassName;

	// standard buttons default to New/Tag/Print/Delete
	var buttonOps = params.buttons;
	if (!buttonOps) {
		buttonOps = [ZmOperation.NEW_MENU, ZmOperation.TAG_MENU, ZmOperation.PRINT, ZmOperation.DELETE];
	} else if (buttonOps == ZmOperation.NONE) {
		buttonOps = null;
	}
	// weed out disabled ops, save list of ones that make it
	/**
	 * The operation list property.
	 * @type Array
	 */
	this.opList = ZmOperation.filterOperations(buttonOps);

	this._zimletButtonLocation = this.opList.length;

	var secondaryOpList = ZmOperation.filterOperations(params.secondaryButtons);

	if (secondaryOpList && secondaryOpList.length) {
		this.opList.push(ZmOperation.SEP, ZmOperation.ACTIONS_MENU);
	}

	var rightSideOpList = ZmOperation.filterOperations(params.rightSideButtons);

	if (rightSideOpList.length > 0 || params.addTextElement) {
		this.opList.push(ZmOperation.FILLER);
	}
	
	if (params.addTextElement) {
		this.opList.push(ZmOperation.TEXT);
	}

	if (rightSideOpList.length > 0) {
		this.opList = this.opList.concat(rightSideOpList);
	}

	this._buttons = ZmOperation.createOperations(this, this.opList, params.overrides);

	if (secondaryOpList && secondaryOpList.length) {
		var actionsButton =  this._secondaryButton = this.getButton(ZmOperation.ACTIONS_MENU);

		actionsButton.noMenuBar = true;

		var secondaryMenu = this._secondaryButtonMenu = new ZmActionMenu({parent: actionsButton, menuItems: ZmOperation.NONE, context: this._context, controller: params.controller});
		var secondaryButtons  = ZmOperation.createOperations(secondaryMenu, secondaryOpList, params.overrides);
		actionsButton.setMenu(secondaryMenu);

		//add secondary buttons to buttons list as I believe from now on it shouldn't matter if they are primary or under the secondary "actions" menu.
		//that way we don't need to operate on 2 different collections when enabling/disabling, adding listeners, etc.
		//var secondaryButtons = secondaryMenu._menuItems;
		for (var id in secondaryButtons) {
			this._buttons[id] = secondaryButtons[id];
		}
		//same as buttons, with opList.
		this.opList = this.opList.concat(secondaryOpList);

	}

	//todo - I guess in the new UI a button (primary) will have either text or image. not both. Think of whether this precedence is still required then.
	this._createPrecedenceList(); //this is only done to the primary, not the secondary buttons (since the secondary are in a drop-down so removing one's image or text won't make sense.)
	
	this._inited = true;
};

ZmButtonToolBar.prototype = new ZmToolBar;
ZmButtonToolBar.prototype.constructor = ZmButtonToolBar;

ZmButtonToolBar.prototype.isZmButtonToolBar = true;
ZmButtonToolBar.prototype.toString = function() { return "ZmButtonToolBar"; };


// Public methods


/**
 * Creates a button and adds its operation ID as data.
 * 
 * @param {String}	id			the name of the operation
 * @param {Hash}	params		a hash of parameters
 * @param {String}	params.text			a button text
 * @param {String}	params.tooltip		a button tooltip text
 * @param {String}	params.image			a icon class for the button
 * @param {String}	params.disImage		a disabled version of icon
 * @param {Boolean}	params.enabled		if <code>true</code>, button is enabled
 * @param {String}	params.className		the CSS class name
 * @param {String}	params.style			thebutton style
 * @param {int} params.index			the position at which to add the button
 * @param {Boolean}	params.showImageInToolbar	if <code>true</code>, the button should show image (default is false)
 * @param {Boolean}	params.showTextInToolbar	if <code>true</code>, the button should show text (default is !params.showImageInToolbar)
 */
ZmButtonToolBar.prototype.createOp =
function(id, params) {
	params.className = this._buttonStyle;
	var b;
	if (id == ZmOperation.TEXT) {
		var id;
		if (this._context) {
			var context = this._toolbarType ? [this._context, this._toolbarType].join("_") : this._context;
			id = [ZmId.WIDGET, AjxStringUtil.toMixed(context, "_", true), AjxStringUtil.toMixed(id, "_")].join("");
		}
		params.textClassName = params.textClassName || "DwtText ZWidgetTitle";
		b = new DwtText({parent:this, className:params.textClassName, id:id});
	} else {
		params.id = params.domId || (this._context ? ZmId.getButtonId(this._context, id, this._toolbarType) : null);
		params.textPrecedence = ZmOperation.getProp(id, "textPrecedence");
		params.iconPrecedence = ZmOperation.getProp(id, "iconPrecedence");
		var showImage = params.showImageInToolbar || false; //default to false;
		var showText = !showImage || params.showTextInToolbar;
		showImage = showImage || !params.text; //no text? gotta show image
		showText = showText || !params.image; //no image? gotta show text
		params.image = showImage && params.image;
		params.whatToShow = {showImage: showImage, showText: showText}
		b = this.createButton(id, params);
	}
	b.setData(ZmOperation.KEY_ID, id);

	return b;
};

/**
 * Creates a zimlet button and adds its operation ID as data. This method selects the best location for the zimlet, so zimlets don't have to do it and it's consistent.
 *
 * for parameters see createOp
 */
ZmButtonToolBar.prototype.createZimletOp =
function(id, params) {
	params.index = this._zimletButtonLocation;
	return this.createOp(id, params);
};


/**
 * Adds the operation.
 * 
 * @param	{String}	id		the id
 * @param	{int}		index	the index
 */
ZmButtonToolBar.prototype.addOp =
function(id, index) {
	if(this.getOp(id)) {
		return;
	}
	ZmOperation.addOperation(this, id, this._buttons, index);
	AjxUtil.arrayAdd(this.opList, id, index);
};

/**
 * Removes the operation.
 * 
 * @param	{String}	id		the id
 * 
 * @see ZmOperation
 */
ZmButtonToolBar.prototype.removeOp =
function(id) {
	ZmOperation.removeOperation(this, id, this._buttons);
	AjxUtil.arrayRemove(this.opList, id);
};

/**
 * Gets the button.
 *
 * @param {constant}	id		the button
 * @return	{DwtButton}	the button
 * 
 * @see ZmOperation
 */
ZmButtonToolBar.prototype.getOp =
function(id) {
	return this.getButton(id);
};

/**
 * Gets the menu tag sub-menu (if any).
 * 
 * @return	{ZmTagMenu}		the menu
 */
ZmButtonToolBar.prototype.getTagMenu =
function() {
	var button = this.getButton(ZmOperation.TAG_MENU);
	if (button) {
		return button.getMenu();
	}
};

/**
 * gets the secondary menu (the "Actions" menu in the toolbar)
 */
ZmButtonToolBar.prototype.getActionsMenu =
function() {
	return this._secondaryButtonMenu;
};

/**
 * gets the secondary button (the "Actions" button in the toolbar)
 */
ZmButtonToolBar.prototype.getActionsButton =
function() {
	return this._secondaryButton;
};


//
// Private methods
//

// Returns the ID for the given button.
ZmButtonToolBar.prototype._buttonId =
function(button) {
	return button.getData(ZmOperation.KEY_ID);
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmNavToolBar")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Creates a navigation tool bar.
 * @class
 * Navigation toolbar for the client. This toolbar is affected by every 
 * push/pop of a view and must be context sensitive since it can custom apply 
 * to any view. A new class was created since nav toolbar may be expanded in 
 * the future (i.e. to incl. a text input indicating current page, etc)
 *
 * @param {Hash}	params			a hash of parameters
 * @param {DwtComposite}	params.parent			the containing widget
 * @param {constant}	params.posStyle			the positioning style
 * @param {String}	params.className			the CSS class name
 * @param {Boolean}	params.hasText			if <code>true</code>, this toolbar includes text in the middle
 * @param {constant}	params.context			the view ID (used to generate button IDs)
 * 
 * @extends	ZmButtonToolBar
 */
ZmNavToolBar = function(params) {

	params.className = params.className || "ZmNavToolBar";
	var hasText = (params.hasText !== false);
	params.buttons = this._getButtons(hasText);
	params.toolbarType = ZmId.TB_NAV;
	params.posStyle = params.posStyle || DwtControl.STATIC_STYLE;
	ZmButtonToolBar.call(this, params);
	
	this._prevButton = params.prevButton;
	this._nextButton = params.nextButton;

	if (hasText) {
		this._textButton = this.getButton(ZmOperation.TEXT);
	}
};

ZmNavToolBar.prototype = new ZmButtonToolBar;
ZmNavToolBar.prototype.constructor = ZmNavToolBar;

ZmNavToolBar.prototype.toString = 
function() {
	return "ZmNavToolBar";
};

ZmNavToolBar.prototype.handleKeyAction =
function (actionCode, ev) {
	
	var currentFocuseItem = this._getCurrentFocusItem();
	switch(actionCode) {
		case DwtKeyMap.PREV:
			if (this._prevButton && currentFocuseItem === this.leftMostButton()) {
				this.blur();
				this.parent.focus(this._prevButton);
				return true;
			}
			break;
		case DwtKeyMap.NEXT:
			if (this._nextButton && currentFocuseItem === this.rightMostButton()) {
				this.blur();
				this.parent.focus(this._nextButton);
				return true;
			}
			break;
	}
	return DwtToolBar.prototype.handleKeyAction.call(this, actionCode, ev);
};

/**
 * Enables/disables buttons.
 *
 * @param {Array}	ids		a list of button IDs
 * @param {Boolean}	enabled	if <code>true</code>, enable the buttons
 * 
 */
ZmNavToolBar.prototype.enable =
function(ids, enabled) {
	ZmButtonToolBar.prototype.enable.call(this, ids, enabled);

	// 	also kill the tooltips if buttons are disabled
	if (!enabled) {
		if (!(ids instanceof Array))
			ids = [ids];
		for (var i = 0; i < ids.length; i++) {
			var button = this.getButton(ids[i]);
			if (button)
				button.setToolTipContent(null);
		}
	}
};

ZmNavToolBar.prototype.leftMostButton =
function() {
	return this.nextSibling(true);
};

ZmNavToolBar.prototype.rightMostButton =
function() {
	return this.nextSibling(false);
};

ZmNavToolBar.prototype.nextSibling =
function(next) {

	var childCount = this.getItemCount();
	var sibling = next ? this.getChild(0) : this.getChild(childCount - 1);

	for (var i = 0; i < childCount; i++) {
		var childIndex = next ? i : childCount - 1 - i;
		var child = this.getChild(childIndex);
		if (child.isDwtText) {
			if (child._enabled && child.getText()) {
				sibling = child;
				break;
			}
		} else if (child._enabled) {
			sibling = child;
			break;
		}
	}
	return sibling;
};

/**
 * Sets the tool tip for the button.
 * 
 * @param	{String}	buttonId		the button id
 * @param	{String}	tooltip			the tool tip
 */
ZmNavToolBar.prototype.setToolTip = 
function(buttonId, tooltip) {
	var button = this.getButton(buttonId);
	if (button) {
		button.setToolTipContent(tooltip);
		button.setAriaLabel(tooltip);
	}
};

/**
 * Sets the text.
 * 
 * @param	{String}	text		the text
 */
ZmNavToolBar.prototype.setText =
function(text, ariaLive) {
	if (!this._textButton) return;
	this._textButton.setText(text, ariaLive);
};

ZmNavToolBar.prototype._getButtons = 
function(hasText) {

	var buttons = [];
	buttons.push(ZmOperation.PAGE_BACK);
	if (hasText) {
		buttons.push(ZmOperation.TEXT);
	}
	buttons.push(ZmOperation.PAGE_FORWARD);

	return buttons;
};

ZmNavToolBar.prototype.createOp =
function(id, params) {
	params.textClassName = "ZWidgetTitle ZmNavToolBarTitle";
	return ZmButtonToolBar.prototype.createOp.apply(this, arguments);
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmSearchToolBar")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @class
 * This class represent a search toolbar. The components are some set of: an input field,
 * a search button, a save button, and a button to choose what type of item to search for.
 * 
 * @param {hash}			params		a hash of parameters:
 * @param {DwtComposite}	parent		the parent widget
 * @param {string}			id			an explicit ID to use for the control's HTML element
 * 
 * @extends		DwtComposite
 */
ZmSearchToolBar = function(params) {

	if (arguments.length == 0) { return; }

	params.className = params.className || "ZmSearchToolbar";
	this._button = {};
	DwtToolBar.apply(this, arguments);

	this._origin = ZmId.SEARCH;
	this._searchMenu = null;
};

ZmSearchToolBar.prototype = new DwtToolBar;
ZmSearchToolBar.prototype.constructor = ZmSearchToolBar;

ZmSearchToolBar.prototype.isZmSearchToolBar = true;
ZmSearchToolBar.prototype.toString = function() { return "ZmSearchToolBar"; };
ZmSearchToolBar.prototype.role = 'toolbar';

// Consts


ZmSearchToolBar.TYPES_BUTTON		= "TYPES";
ZmSearchToolBar.SEARCH_BUTTON 		= "SEARCH";
ZmSearchToolBar.SAVE_BUTTON 		= "SAVE";
ZmSearchToolBar.SEARCH_MENU_BUTTON	= ZmSearchToolBar.TYPES_BUTTON;	// back-compatibility

ZmSearchToolBar.MENU_ITEMS 			= [];		// list of menu items
ZmSearchToolBar.SETTING 			= {};		// required setting for menu item to appear
ZmSearchToolBar.MSG_KEY 			= {};		// text for menu item
ZmSearchToolBar.TT_MSG_KEY 			= {};		// tooltip text for menu item
ZmSearchToolBar.ICON 				= {};		// icon for menu item
ZmSearchToolBar.SHARE_ICON			= {};		// icon for shared menu item
ZmSearchToolBar.ID 					= {};		// ID for menu item
ZmSearchToolBar.DISABLE_OFFLINE     = {};       // Disable when offline detected


// Public static methods

/**
 * Defines a menu item to add when the types menu is created. Static so that it can be called before the
 * toolbar has been created.
 * 
 * @param {string}	id			ID of menu item
 * @param {hash}	params		menu item properties
 */
ZmSearchToolBar.addMenuItem =
function(id, params) {

	if (params.msgKey)		   { ZmSearchToolBar.MSG_KEY[id]		 = params.msgKey; }
	if (params.tooltipKey)	   { ZmSearchToolBar.TT_MSG_KEY[id]	 	 = params.tooltipKey; }
	if (params.icon)		   { ZmSearchToolBar.ICON[id]			 = params.icon; }
	if (params.shareIcon)	   { ZmSearchToolBar.SHARE_ICON[id]	 	 = params.shareIcon; }
	if (params.setting)		   { ZmSearchToolBar.SETTING[id]		 = params.setting; }
	if (params.id)			   { ZmSearchToolBar.ID[id]				 = params.id; }
	if (params.disableOffline) { ZmSearchToolBar.DISABLE_OFFLINE[id] = params.disableOffline; }

	if (params.index == null || params.index < 0 || params.index >= ZmSearchToolBar.MENU_ITEMS.length) {
		ZmSearchToolBar.MENU_ITEMS.push(id);
	} else {
		ZmSearchToolBar.MENU_ITEMS.splice(params.index, 0, id);
	}
};


// Public methods

/**
 * Removes a menu item from the types menu.
 * 
 * @param {string}	id			ID of menu item
 */
ZmSearchToolBar.prototype.removeMenuItem =
function(id) {
	
	var menu = this._searchMenu;
	if (menu) {
		var mi = menu.getItemById(ZmOperation.MENUITEM_ID, id);
		if (mi) {
			menu.removeChild(mi);
			mi.dispose();
		}
		this._cleanupSeparators(menu);
	}
};

// Remove unneeded separators
ZmSearchToolBar.prototype._cleanupSeparators =
function(menu) {

	var button = this._button[ZmSearchToolBar.TYPES_BUTTON];
	menu = menu || (button && button.getMenu());
	if (!menu) { return; }

	var items = menu.getItems();
	var toRemove = [];
	for (var i = 0; i < items.length; i++) {
		var mi = items[i];
		if (mi.isSeparator() && (i == 0 || i == items.length - 1 || items[i - 1].isSeparator())) {
			toRemove.push(mi);
		}
	}
	for (var i = 0; i < toRemove.length; i++) {
		var mi = toRemove[i];
		menu.removeChild(mi);
		mi.dispose();
	}
};

ZmSearchToolBar.prototype.getSearchField =
function() {
	return this._searchField.getInputElement();
};

// sets up a function to call when Enter has been pressed
ZmSearchToolBar.prototype.registerEnterCallback =
function(callback) {
	this._enterCallback = callback;
};

ZmSearchToolBar.prototype.addSelectionListener =
function(buttonId, listener) {
	var button = this._button[buttonId];
	if (button) {
		button.addSelectionListener(listener);
	}
};

ZmSearchToolBar.prototype.getButton =
function(buttonId) {
	return this._button[buttonId];
};

ZmSearchToolBar.prototype.getButtons =
function() {
	return AjxUtil.values(this._button);
};

ZmSearchToolBar.prototype.focus = function(item) {

    if (item) {
        // focus is being moved via a shortcut (arrow key)
        return DwtToolBar.prototype.focus.apply(this, arguments);
    }
	else if (this._searchField) {
		this._searchField.focus();
		this._searchField.moveCursorToEnd();
        return this._searchField;
	}
};

ZmSearchToolBar.prototype.blur =
function() {
	if (this._searchField) {
		this._searchField.blur();
	}
};

ZmSearchToolBar.prototype.setEnabled =
function(enable) {
	if (this._searchField) {
		this._searchField.setEnabled(enable);
	}
	for (var buttonId in this._button) {
		this._button[buttonId].setEnabled(enable);
	}
};
ZmSearchToolBar.prototype.setSearchFieldValue =
function(value) {
	if (this._searchField && value != this.getSearchFieldValue()) {
		this._searchField.setValue(value);
	}
};

ZmSearchToolBar.prototype.getSearchFieldValue =
function() {
	return this._searchField ? this._searchField.getValue() : null;
};



// Private methods

ZmSearchToolBar.prototype._handleKeyDown =
function(ev) {
	var key = DwtKeyEvent.getCharCode(ev);
	if (DwtKeyEvent.IS_RETURN[key]) {
		return this._handleEnterKeyPress(ev);
	}
	return true;
};

ZmSearchToolBar.prototype._handleEnterKeyPress =
function(ev) {
	if (this._enterCallback) {
		this._enterCallback.run({
					ev:				ev,
					zimletEvent:	"onKeyPressSearchField",
					origin:			this._origin
				});
	}
	return false;
};

/**
 * Initializes search autocomplete.
 */
ZmSearchToolBar.prototype.initAutocomplete =
function() {
	if (!this._acList) {
		this._acList = new ZmAutocompleteListView(this._getAutocompleteParams());
		this._acList.handle(this.getSearchField());
	}
};

ZmSearchToolBar.prototype._getAutocompleteParams =
function() {
	var params = {
		dataClass:			new ZmSearchAutocomplete(),
		matchValue:			"matchText",
		delims:				[" ", "\t"],
		delimCodes:			[3, 13, 9],
		separator:			" ",
		keyDownCallback:	this._handleKeyDown.bind(this),
		contextId:			this.toString(),
		locationCallback:	this._getAcLocation.bind(this)
	};
	return params;
};

ZmSearchToolBar.prototype.getAutocompleteListView =
function() {
	return this._acList;
};

ZmSearchToolBar.prototype._getAcLocation =
function() {
	var el = this._searchField.getInputElement();
	if (!el) { return {}; }
	
	var elLoc = Dwt.getLocation(el);
	var elSize = Dwt.getSize(el);
	var strWidth = AjxStringUtil.getWidth(el.value);
	if (AjxEnv.isWindows && (AjxEnv.isFirefox || AjxEnv.isSafari || AjxEnv.isChrome) ){
		// FF/Win: fudge factor since string is longer in INPUT than when measured in SPAN
		strWidth = strWidth * 1.2;
	}
	var x = elLoc.x + strWidth;
	var y = elLoc.y + elSize.y;
	DwtPoint.tmp.set(x, y);
	return DwtPoint.tmp;
};




/**
 * Adds a types button and support for a custom search menu item to a search toolbar
 * 
 * @param params
 */
ZmMainSearchToolBar = function(params) {

	if (arguments.length == 0) { return; }

	ZmSearchToolBar.apply(this, arguments);

    this._initialize();

	// setup "include shared" menu item
	var miParams = {
		msgKey:			"searchShared",
		tooltipKey:		"searchShared",
		icon:			"Group",
		setting:		ZmSetting.SHARING_ENABLED,
		id:				ZmId.getMenuItemId(ZmId.SEARCH, ZmId.SEARCH_SHARED),
		disableOffline: true
	};
	ZmSearchToolBar.addMenuItem(ZmId.SEARCH_SHARED, miParams);

	// setup "all accounts" menu item for multi account
	if (appCtxt.multiAccounts) {
		var miParams = {
			msgKey:	"searchAllAccounts",
			icon:	"Globe",
			id:		ZmId.getMenuItemId(ZmId.SEARCH, ZmId.SEARCH_ALL_ACCOUNTS)
		};
		ZmSearchToolBar.addMenuItem(ZmId.SEARCH_ALL_ACCOUNTS, miParams);
	}
};

ZmMainSearchToolBar.prototype = new ZmSearchToolBar;
ZmMainSearchToolBar.prototype.constructor = ZmMainSearchToolBar;

ZmMainSearchToolBar.prototype.isZmMainSearchToolBar = true;
ZmMainSearchToolBar.prototype.toString = function() { return "ZmMainSearchToolBar"; };

ZmMainSearchToolBar.CUSTOM_ITEM_ID		= "CustomSearchItem";	// custom search menu item key
ZmMainSearchToolBar.CUSTOM_BUTTON 		= "CUSTOM";				// button ID

ZmMainSearchToolBar.prototype._initialize = function() {

    var isExternalAccount = appCtxt.isExternalAccount();

	// add "search types" menu
    var firstItem = ZmSearchToolBar.MENU_ITEMS[0];
    var buttonId = ZmId.getButtonId(ZmId.SEARCH, ZmId.SEARCH_MENU);
    var button = this._button[ZmSearchToolBar.TYPES_BUTTON] = new DwtButton({
        parent:		this,
        index:		0,
        id:         buttonId
    });
    button.setImage(ZmSearchToolBar.ICON[firstItem]);
    button.setToolTipContent(ZmMsg[ZmSearchToolBar.TT_MSG_KEY[firstItem]], true);

    var menu = new AjxCallback(this, this._createSearchMenu);
    button.setMenu(menu, false, DwtMenuItem.RADIO_STYLE);
    if (isExternalAccount) {
        button.setEnabled(false);
    }

    // add search box
    var searchBox = this._searchField = new DwtInputField({
        parent:     this,
        hint:       ZmMsg.searchInput,
        label:      ZmMsg.searchInput,
        inputId:    ZmId.SEARCH_INPUTFIELD
    });
    var inputEl = searchBox.getInputElement();
    inputEl.className = "search_input";
    this._searchField._showHint();
    this._searchField.addListener(DwtEvent.ONFOCUS, this._onInputFocus.bind(this));
    this._searchField.addListener(DwtEvent.ONBLUR, this._onInputBlur.bind(this));
    if (isExternalAccount) {
        this._searchField.setEnabled(false);
    }
    searchBox.addListener(DwtEvent.ONFOCUS, this._childFocusListener.bind(this));

    // add search button
    button = this._button[ZmSearchToolBar.SEARCH_BUTTON] = new DwtButton({
        parent:		this,
        className: 	"ZmSearchButton",
        id:         ZmId.getButtonId(ZmId.SEARCH, ZmId.SEARCH_SEARCH)
    });
    button.setImage("Search2");
    button.setAriaLabel(ZmMsg.searchTooltip);
    button.setToolTipContent(ZmMsg.searchTooltip, true);

    // add save search button if saved-searches enabled
    if (isExternalAccount) {
        if (this._button[ZmSearchToolBar.SEARCH_BUTTON]) {
            this._button[ZmSearchToolBar.SEARCH_BUTTON].setEnabled(false);
        }
    }
};

ZmMainSearchToolBar.prototype._createSearchMenu =
function() {

	var menu = this._searchMenu = new DwtMenu({
				parent:		this._button[ZmSearchToolBar.TYPES_BUTTON],
				className:	"ActionMenu",
				id:			ZmId.getMenuId(ZmId.SEARCH)
			});
	var mi;
	if (this._customSearchMenuItems) {
		for (var i = 0; i < this._customSearchMenuItems.length; i++) {
			var csmi = this._customSearchMenuItems[i];
			this._createCustomSearchMenuItem(menu, csmi.icon, csmi.text, csmi.listener);
		}
	}
	var params = {
		parent:         menu,
		enabled:        true,
		radioGroupId:   0,
		style:          DwtMenuItem.RADIO_STYLE
	};
	for (var i = 0; i < ZmSearchToolBar.MENU_ITEMS.length; i++) {
		var id = ZmSearchToolBar.MENU_ITEMS[i];

		// add separator *before* "shared" menu item
		if (id == ZmId.SEARCH_SHARED) {
			if (ZmSearchToolBar.MENU_ITEMS.length <= 1) { continue; }
			mi = new DwtMenuItem({parent:menu, style:DwtMenuItem.SEPARATOR_STYLE});
		}

		var setting = ZmSearchToolBar.SETTING[id];
		if (setting && !appCtxt.get(setting)) { continue; }

		var isCheckStyle = (id == ZmId.SEARCH_SHARED || id == ZmId.SEARCH_ALL_ACCOUNTS);
		if (isCheckStyle) {
			params.style = DwtMenuItem.CHECK_STYLE;
		}
		params.style = (id == ZmId.SEARCH_SHARED || id == ZmId.SEARCH_ALL_ACCOUNTS)
			? DwtMenuItem.CHECK_STYLE : DwtMenuItem.RADIO_STYLE;
		params.imageInfo = ZmSearchToolBar.ICON[id];
		params.text = ZmMsg[ZmSearchToolBar.MSG_KEY[id]];
		params.id = ZmSearchToolBar.ID[id];
		mi = DwtMenuItem.create(params);
		mi.setData(ZmOperation.MENUITEM_ID, id);
		if (!isCheckStyle) {
			mi.setAttribute('aria-label', ZmMsg[ZmSearchToolBar.TT_MSG_KEY[id]]);
		}
	}
	
	this._checkSharedMenuItem();
	appCtxt.getSettings().getSetting(ZmSetting.SEARCH_INCLUDES_SHARED).addChangeListener(this._checkSharedMenuItem.bind(this));

	appCtxt.getSearchController()._addMenuListeners(menu);
	this._searchMenuCreated = true;

	return menu;
};

ZmMainSearchToolBar.prototype.setOfflineState = function(offline) {
	var button   = this._button[ZmSearchToolBar.TYPES_BUTTON];
	var menu     = button && button.getMenu();
	var numItems = menu.getItemCount();
	for (var i = 0; i < numItems; i++) {
	    var item = menu.getItem(i);
		if (item) {
			var id = item.getData(ZmOperation.MENUITEM_ID);
			if (id && ZmSearchToolBar.DISABLE_OFFLINE[id])  {
				item.setEnabled(!offline);
			}
		}
	}
}

ZmMainSearchToolBar.prototype.getSearchType =
function() {
	var button = this._button[ZmSearchToolBar.TYPES_BUTTON];
	var menu = button && button.getMenu();
    var item = menu ? menu.getSelectedItem() || menu.getItems()[0] : null;
	var data = item ? item.getData(ZmMainSearchToolBar.CUSTOM_ITEM_ID) || item.getData(ZmOperation.MENUITEM_ID) :
					  ZmSearchToolBar.MENU_ITEMS[0];
	return data;
};

ZmMainSearchToolBar.prototype.createCustomSearchBtn =
function(icon, text, listener, id) {

	if (!this._customSearchListener) {
		this._customSearchListener = this._customSearchBtnListener.bind(this);
	}

	// check if custom search should be a button by checking for the Id against the template
	var customSearchBtn = document.getElementById(this._htmlElId + "_customSearchButton");
	if (customSearchBtn) {
		var data = { icon:icon, text:text, listener:listener };
		var button = this._button[ZmSearchToolBar.CUSTOM_BUTTON]
		if (!button) {
			button = this._button[ZmSearchToolBar.CUSTOM_BUTTON] = ZmToolBar.addButton({
						parent:		this,
						tdId:		"_customSearchButton",
						buttonId:	ZmId.getButtonId(ZmId.SEARCH, ZmId.SEARCH_CUSTOM),
						lbl:		text,
						icon:		icon
					});
			button.setData(ZmMainSearchToolBar.CUSTOM_ITEM_ID, data);
			button.addSelectionListener(this._customSearchListener);

			// show the separator now that we've added a custom search button
			var sep = document.getElementById(this._htmlElId + "_customSearchButtonSep");
			if (sep) {
				Dwt.setVisible(sep, true);
			}
		} else {
			var menu = button && button.getMenu();
			var item;
			var params = {
				parent:			menu,
				enabled:		true,
				style:			DwtMenuItem.RADIO_STYLE,
				radioGroupId:	0,
				id:				id
			};
			if (!menu) {
				var btnData = button.getData(ZmMainSearchToolBar.CUSTOM_ITEM_ID);
				menu = new DwtMenu({
							parent:		button,
							className:	"ActionMenu",
							id:			ZmId.getMenuId(ZmId.SEARCH, ZmId.SEARCH_CUSTOM)
						});
				button.setMenu(menu, false, DwtMenuItem.RADIO_STYLE);
				params.imageInfo = btnData.icon;
				params.text = btnData.text;
				item = DwtMenuItem.create(params);
				item.setData(ZmMainSearchToolBar.CUSTOM_ITEM_ID, btnData);
				item.setData(ZmOperation.MENUITEM_ID, ZmId.SEARCH_CUSTOM);
				item.setChecked(true, true);
				item.addSelectionListener(this._customSearchListener);
			}
			params.imageInfo = icon;
			params.text = text;
			item = DwtMenuItem.create(params);
			item.setData(ZmMainSearchToolBar.CUSTOM_ITEM_ID, data);
			item.addSelectionListener(this._customSearchListener);
		}
	} else {
		if (this._searchMenuCreated) {
			var menu = this._button[ZmSearchToolBar.TYPES_BUTTON].getMenu();
			this._createCustomSearchMenuItem(menu, icon, text, listener, id);
		} else {
			if (!this._customSearchMenuItems) {
				this._customSearchMenuItems = [];
			}
			this._customSearchMenuItems.push({icon:icon, text:text, listener:listener, id:id});
		}
	}
};

ZmMainSearchToolBar.prototype._createCustomSearchMenuItem =
function(menu, icon, text, listener, id) {
	var mi = menu.getItem(0);
	var params = {
		parent: menu,
		imageInfo: icon,
		text: text,
		enabled: true,
		style: DwtMenuItem.RADIO_STYLE,
		radioGroupId: 0,
		index: 0,
		id: id
	};
	mi = DwtMenuItem.create(params);
	var data = { icon:icon, text:text, listener:listener };
	mi.setData(ZmMainSearchToolBar.CUSTOM_ITEM_ID, data);
	mi.setData(ZmOperation.MENUITEM_ID, ZmId.SEARCH_CUSTOM);
	mi.addSelectionListener(this._customSearchListener);

	// only add separator if this is the first custom search menu item
	if (!(mi && mi.getData(ZmMainSearchToolBar.CUSTOM_ITEM_ID))) {
		mi = new DwtMenuItem({parent:menu, style:DwtMenuItem.SEPARATOR_STYLE, index:1});
	}
};

ZmMainSearchToolBar.prototype._customSearchBtnListener = 
function(ev) {
	var item = ev.item;
	if (!item) { return; }
	var data = item.getData(ZmMainSearchToolBar.CUSTOM_ITEM_ID);
	if (!data) { return; }
	if (this._customSearchBtn) {
		if (item.isDwtMenuItem) {
			if (ev.detail != DwtMenuItem.CHECKED) { return; }
			this._customSearchBtn.setToolTipContent(data[1]);
			this._customSearchBtn.setData(ZmMainSearchToolBar.CUSTOM_ITEM_ID, data);
		}
		data.listener.run(ev); // call original listener
	} else {
		var button = this._button[ZmSearchToolBar.TYPES_BUTTON];
		button.setToolTipContent(data.text);

		var menu = item.parent;
		var shareMenuItem = menu ? menu.getItemById(ZmOperation.MENUITEM_ID, ZmId.SEARCH_SHARED) : null;
		if (shareMenuItem) {
			shareMenuItem.setChecked(false, true);
			shareMenuItem.setEnabled(false);
		}

		button.setImage(data.icon);
		button.setText(data.text);
	}
};

// Expand INPUT when it gets focus
ZmMainSearchToolBar.prototype._onInputFocus = function(ev) {

	this._setInputExpanded(true);
};

// Collapse INPUT when it loses focus (unless that was due to a click on a search toolbar button)
ZmMainSearchToolBar.prototype._onInputBlur = function(ev) {

	var focusObj = appCtxt.getKeyboardMgr().getFocusObj();
	if (focusObj !== this._button[ZmSearchToolBar.TYPES_BUTTON] && focusObj !== this._button[ZmSearchToolBar.SEARCH_BUTTON] && !this._movingFocus) {
		this._setInputExpanded(false);
	}
    this._movingFocus = false;  // done here since blur handler may be called on a timer
};

// note when we're moving focus within the toolbar so we don't collapse the INPUT
ZmMainSearchToolBar.prototype._moveFocus = function(back) {

    this._movingFocus = true;
    ZmSearchToolBar.prototype._moveFocus.apply(this, arguments);
};

ZmMainSearchToolBar.prototype._setInputExpanded = function(expanded) {

    // Don't collapse input if user just popped up menu (which causes blur on input)
    if (!expanded && this._searchMenu && this._searchMenu.isPoppedUp()) {
        return;
    }

	var input = this._searchField.getInputElement();
	var cls = expanded ? "search_input-expanded" : "search_input";

	if (AjxEnv.isIE9) {
		// bug 83493: IE9 gets the layout wrong on the first try
		setTimeout(function() {
			input.className = cls;
		}, 0);
	} else {
		input.className = cls;
	}
};

ZmMainSearchToolBar.prototype._checkSharedMenuItem =
function() {
	var mi = this._searchMenu && this._searchMenu.getItemById(ZmOperation.MENUITEM_ID, ZmId.SEARCH_SHARED);
	if (mi) {
		mi.setChecked(appCtxt.get(ZmSetting.SEARCH_INCLUDES_SHARED));
	}
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmSearchResultsToolBar")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @class
 * This class represents a search toolbar that shows up above search results. It can be
 * used to refine the search results. Each search term is contained within a bubble that
 * can easily be removed.
 * 
 * @param {hash}			params		a hash of parameters:
 * @param {DwtComposite}	parent		the parent widget
 * @param {string}			id			an explicit ID to use for the control's HTML element
 * 
 * @extends		ZmSearchToolBar
 * 
 * @author Conrad Damon
 */
ZmSearchResultsToolBar = function(params) {

	params.posStyle = Dwt.ABSOLUTE_STYLE;
	params.className = "ZmSearchResultsToolBar";
	this._controller = params.controller;

	ZmSearchToolBar.apply(this, arguments);
	
	this._bubbleId = {};
	this._origin = ZmId.SEARCHRESULTS;
	
	this.initAutocomplete();
};

ZmSearchResultsToolBar.prototype = new ZmSearchToolBar;
ZmSearchResultsToolBar.prototype.constructor = ZmSearchResultsToolBar;

ZmSearchResultsToolBar.prototype.isZmSearchResultsToolBar = true;
ZmSearchResultsToolBar.prototype.toString = function() { return "ZmSearchResultsToolBar"; };


ZmSearchResultsToolBar.prototype.TEMPLATE = "share.Widgets#ZmSearchResultsToolBar";

ZmSearchResultsToolBar.prototype.PENDING_BUBBLE_CONTAINER_CLASS = "addrBubbleContainerPending";
ZmSearchResultsToolBar.prototype.PENDING_SEARCH_DELAY = 2000; // 2 seconds

ZmSearchResultsToolBar.prototype._createHtml =
function() {

	this.getHtmlElement().innerHTML = AjxTemplate.expand(this.TEMPLATE, {id:this._htmlElId});

	var idContext = this._controller.getCurrentViewId();
	
	this._label = document.getElementById(this._htmlElId + "_label");
	this._iconDiv = document.getElementById(this._htmlElId + "_icon");
	
	// add search input field
	var inputFieldCellId = this._htmlElId + "_inputFieldCell";
	var inputFieldCell = document.getElementById(inputFieldCellId);
	if (inputFieldCell) {
		var aifParams = {
			parent:					this,
			strictMode:				false,
			id:						DwtId.makeId(ZmId.WIDGET_INPUT, idContext),
			bubbleAddedCallback:	this._bubbleChange.bind(this),
			bubbleRemovedCallback:	this._bubbleChange.bind(this),
			noOutsideListening:		true,
			type:					ZmId.SEARCH
		}
		var aif = this._searchField = new ZmAddressInputField(aifParams);
		aif.reparentHtmlElement(inputFieldCell);
		
		var inputEl = this._searchField.getInputElement();
		inputEl.className = "search_results_input";
		inputEl.setAttribute('aria-label', ZmMsg.search);
	}
	
	// add search button
	this._button[ZmSearchToolBar.SEARCH_BUTTON] = ZmToolBar.addButton({
				parent:		this, 
				tdId:		"_searchButton",
				buttonId:	ZmId.getButtonId(idContext, ZmId.SEARCHRESULTS_SEARCH),
				lbl:		ZmMsg.search,
				tooltip:	ZmMsg.searchTooltip
			});

	// add save search button if saved-searches enabled
	this._button[ZmSearchToolBar.SAVE_BUTTON] = ZmToolBar.addButton({
				parent:		this, 
				setting:	ZmSetting.SAVED_SEARCHES_ENABLED,
				tdId:		"_saveButton",
				buttonId:	ZmId.getButtonId(idContext, ZmId.SEARCHRESULTS_SAVE),
				lbl:		ZmMsg.save,
				tooltip:	ZmMsg.saveSearchTooltip
			});
};

// TODO: use the main search toolbar's autocomplete list - need to manage location callback
ZmSearchResultsToolBar.prototype.initAutocomplete =
function() {
	if (!this._acList) {
		this._acList = new ZmAutocompleteListView(this._getAutocompleteParams());
		this._acList.handle(this.getSearchField(), this._searchField._htmlElId);
	}
	this._searchField.setAutocompleteListView(this._acList);
};

ZmSearchResultsToolBar.prototype._getAutocompleteParams =
function() {
	var params = ZmSearchToolBar.prototype._getAutocompleteParams.apply(this, arguments);
	params.options = { noBubbleParse: true };
	return params;
};

ZmSearchResultsToolBar.prototype.setSearch =
function(search) {
	this._settingSearch = true;
	this._searchField.clear(true);
	var tokens = search.getTokens();
	if (search.query && tokens && tokens.length) {
		for (var i = 0, len = tokens.length; i < len; i++) {
			var token = tokens[i], prevToken = tokens[i - 1], nextToken = tokens[i + 1];
			var showAnd = (prevToken && prevToken.op == ZmParsedQuery.GROUP_CLOSE) || (nextToken && nextToken.op == ZmParsedQuery.GROUP_OPEN);
			var text = token.toString(showAnd);
			if (text) {
				var bubble = this._searchField.addBubble({address:text, noParse:true});
				this._bubbleId[text] = bubble.id;
			}
		}
	}
	this._settingSearch = false;
};

ZmSearchResultsToolBar.prototype.setLabel =
function(text, showError) {
	this._label.innerHTML = text;
	this._iconDiv.style.display = showError ? "inline-block" : "none";
};

// Don't let the removal or addition of a bubble when we're setting up trigger a search loop.
ZmSearchResultsToolBar.prototype._bubbleChange =
function(bubble, added) {

	//cancel existing timeout since we restart the 2 seconds wait.
	this._clearPendingSearchTimeout();
	if (this._settingSearch) {
		return;
	}
	var pq = new ZmParsedQuery(bubble.address);
	var tokens = pq.getTokens();
	// don't run search if a conditional was added or removed
	if (tokens && tokens[0] && tokens[0].type == ZmParsedQuery.COND) {
		return;
	}
	//add class to make the search bar yellow to indicate it's a "pending" state (a.k.a. dirty or edit).
	Dwt.addClass(this._searchField._elRef, this.PENDING_BUBBLE_CONTAINER_CLASS);
	// wait 2 seconds before running the search, to see if the user continues to create bubbles (or delete them), so we don't send requests to the server annoying and blocking the user.
	// Interestingly, I piggy-back on an existing timeout (but that was 10 miliseconds only), This is an old comment as to why there was already a very short timeout ==>
	// use timer to let content of search bar get set - if a search term is autocompleted,
	// the bubble is added before the text it replaces is removed
	this._pendingSearchTimeout = setTimeout(this._handleEnterKeyPress.bind(this), this.PENDING_SEARCH_DELAY);
};

ZmSearchResultsToolBar.prototype._clearPendingSearchTimeout =
function() {
	if (!this._pendingSearchTimeout) {
		return;
	}
	clearTimeout(this._pendingSearchTimeout);
	this._pendingSearchTimeout = null;
};

ZmSearchResultsToolBar.prototype._handleKeyDown =
function(ev) {
	//cancel existing timeout since the user is still typing (new bubble, but not completed yet).
	this._clearPendingSearchTimeout();
	ZmSearchToolBar.prototype._handleKeyDown.apply(this, arguments);
};

ZmSearchResultsToolBar.prototype._handleEnterKeyPress =
function(ev) {
	//cancel existing timeout in case we explicitly clicked the enter key, so we don't want this to be called twice.
	this._clearPendingSearchTimeout();
	Dwt.delClass(this._searchField._elRef, this.PENDING_BUBBLE_CONTAINER_CLASS);
	this.setLabel(ZmMsg.searching);
	ZmSearchToolBar.prototype._handleEnterKeyPress.apply(this, arguments);
};

/**
 * Adds a bubble for the given search term.
 * 
 * @param {ZmSearchToken}	term		search term
 * @param {boolean}			skipNotify	if true, don't trigger a search
 * @param {boolean}			addingCond	if true, user clicked to add a conditional term
 */
ZmSearchResultsToolBar.prototype.addSearchTerm =
function(term, skipNotify, addingCond) {

	var text = term.toString(addingCond);
	var index;
	if (addingCond) {
		var bubbleList = this._searchField._getBubbleList();
		var bubbles = bubbleList.getArray();
		for (var i = 0; i < bubbles.length; i++) {
			if (bubbleList.isSelected(bubbles[i])) {
				index = i;
				break;
			}
		}
	}

	var bubble = this._searchField.addBubble({
				address:	text,
				addClass:	term.type,
				skipNotify:	skipNotify,
				index:		index
			});
	if (bubble) {
		this._bubbleId[text] = bubble.id;
		return bubble.id;
	}
};

/**
 * Removes the bubble with the given search term.
 * 
 * @param {ZmSearchToken|string}	term		search term or bubble ID
 * @param {boolean}					skipNotify	if true, don't trigger a search
 */
ZmSearchResultsToolBar.prototype.removeSearchTerm =
function(term, skipNotify) {
	if (!term) { return; }
	var text = term.toString();
	var id = term.isZmSearchToken ? this._bubbleId[text] : term;
	if (id) {
		this._searchField.removeBubble(id, skipNotify);
		delete this._bubbleId[text];
	}
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmSearchResultsFilterPanel")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @class
 * This class represents a panel used to modify the current search.
 * 
 * @param {hash}			params		a hash of parameters:
 * @param {DwtComposite}	parent		parent widget
 * @param {ZmController}	controller	search results controller
 * @param {constant}		resultsApp	name of app corresponding to type of results
 * 
 * TODO: Add change listeners to update filters as necessary, eg folders and tags.
 */
ZmSearchResultsFilterPanel = function(params) {

	params.className = params.className || "ZmSearchResultsFilterPanel";
	params.posStyle = Dwt.ABSOLUTE_STYLE;
	DwtComposite.apply(this, arguments);
	
	// Need to wait for ZmApp.* constants to have been defined
	if (!ZmSearchResultsFilterPanel.BASIC_FILTER) {
		ZmSearchResultsFilterPanel._initConstants();
	}

	this._controller = params.controller;
	this._resultsApp = params.resultsApp;
	this._viewId = this._controller.getCurrentViewId();
	
	// basic filters
	this._checkbox = {};
	
	// advanced filters
	this._menu		= {};
	this._advancedFilterHandlers = {};
	
	this._createHtml();
	this._addFilters();
	this._addConditionals();
};

ZmSearchResultsFilterPanel.prototype = new DwtComposite;
ZmSearchResultsFilterPanel.prototype.constructor = ZmSearchResultsFilterPanel;

ZmSearchResultsFilterPanel.prototype.isZmSearchResultsFilterPanel = true;
ZmSearchResultsFilterPanel.prototype.toString = function() { return "ZmSearchResultsFilterPanel"; };

ZmSearchResultsFilterPanel.prototype.TEMPLATE = "share.Widgets#ZmSearchResultsFilterPanel";

// used for element IDs
ZmSearchResultsFilterPanel.BASIC	= "BasicFilter";
ZmSearchResultsFilterPanel.ADVANCED	= "AdvancedFilter";

// filter types
ZmSearchResultsFilterPanel.ID_ATTACHMENT	= "ATTACHMENT";
ZmSearchResultsFilterPanel.ID_FLAGGED		= "FLAGGED";
ZmSearchResultsFilterPanel.ID_UNREAD		= "UNREAD";
ZmSearchResultsFilterPanel.ID_TO			= "TO";
ZmSearchResultsFilterPanel.ID_FROM			= "FROM";
ZmSearchResultsFilterPanel.ID_DATE			= "DATE";
ZmSearchResultsFilterPanel.ID_DATE_BRIEFCASE = "DATE_BRIEFCASE";
ZmSearchResultsFilterPanel.ID_DATE_SENT		= "DATE_SENT";
ZmSearchResultsFilterPanel.ID_SIZE			= "SIZE";
ZmSearchResultsFilterPanel.ID_STATUS		= "STATUS";
ZmSearchResultsFilterPanel.ID_TAG			= "TAG";
ZmSearchResultsFilterPanel.ID_FOLDER		= "FOLDER";

// filter can be used within any app
ZmSearchResultsFilterPanel.ALL_APPS = "ALL";

// ordered list of basic filters
ZmSearchResultsFilterPanel.BASIC_FILTER_LIST = [
	ZmSearchResultsFilterPanel.ID_ATTACHMENT,
	ZmSearchResultsFilterPanel.ID_FLAGGED,
	ZmSearchResultsFilterPanel.ID_UNREAD
];

// ordered list of advanced filters
ZmSearchResultsFilterPanel.ADVANCED_FILTER_LIST = [
	ZmSearchResultsFilterPanel.ID_FROM,
	ZmSearchResultsFilterPanel.ID_TO,
	ZmSearchResultsFilterPanel.ID_DATE,
	ZmSearchResultsFilterPanel.ID_DATE_BRIEFCASE,
	ZmSearchResultsFilterPanel.ID_DATE_SENT,
	ZmSearchResultsFilterPanel.ID_ATTACHMENT,
	ZmSearchResultsFilterPanel.ID_SIZE,
	ZmSearchResultsFilterPanel.ID_STATUS,
	ZmSearchResultsFilterPanel.ID_TAG,
	ZmSearchResultsFilterPanel.ID_FOLDER
];

ZmSearchResultsFilterPanel._initConstants =
function() {

	// basic filters
	ZmSearchResultsFilterPanel.BASIC_FILTER = {};
	ZmSearchResultsFilterPanel.BASIC_FILTER[ZmSearchResultsFilterPanel.ID_ATTACHMENT] = {
		text:	ZmMsg.filterHasAttachment,
		term:	new ZmSearchToken("has", "attachment"),
		apps:	[ZmApp.MAIL, ZmApp.CALENDAR, ZmApp.TASKS]
	};
	ZmSearchResultsFilterPanel.BASIC_FILTER[ZmSearchResultsFilterPanel.ID_FLAGGED] = {
		text:	        ZmMsg.filterIsFlagged,
		term:	        new ZmSearchToken("is", "flagged"),
		precondition:   ZmSetting.FLAGGING_ENABLED
	};
	ZmSearchResultsFilterPanel.BASIC_FILTER[ZmSearchResultsFilterPanel.ID_UNREAD] = {
		text:	ZmMsg.filterisUnread,
		term:	new ZmSearchToken("is", "unread")
	};
	
	// advanced filters
	ZmSearchResultsFilterPanel.ADVANCED_FILTER = {};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_FROM] = {
		text: 		ZmMsg.filterReceivedFrom,
		handler:	"ZmAddressSearchFilter",
		searchOp:	"from"
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_TO] = {
		text: 		ZmMsg.filterSentTo,
		handler:	"ZmAddressSearchFilter",
		searchOp:	"to"
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_DATE] = {
		text: 		ZmMsg.filterDate,
		handler:	"ZmApptDateSearchFilter",
		apps:		[ZmApp.CALENDAR, ZmApp.TASKS]
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_DATE_SENT] = {
		text: 		ZmMsg.filterDateSent,
		handler:	"ZmDateSearchFilter"
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_DATE_BRIEFCASE] = {
		text: 		ZmMsg.filterDate,
		handler:	"ZmDateSearchFilter",
		apps:       [ZmApp.BRIEFCASE]
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_ATTACHMENT] = {
		text: 		ZmMsg.filterAttachments,
		handler:	"ZmAttachmentSearchFilter",
		searchOp:	"attachment",
		apps:		[ZmApp.MAIL, ZmApp.CALENDAR, ZmApp.TASKS]
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_SIZE] = {
		text: 		ZmMsg.filterSize,
		handler:	"ZmSizeSearchFilter",
		apps:       [ZmApp.MAIL, ZmApp.BRIEFCASE]
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_STATUS] = {
		text: 		ZmMsg.filterStatus,
		handler:	"ZmStatusSearchFilter",
		searchOp:	"is"
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_TAG] = {
		text: 			ZmMsg.filterTag,
		handler:		"ZmTagSearchFilter",
		searchOp:		"tag",
		apps:			ZmSearchResultsFilterPanel.ALL_APPS,
		precondition:	appCtxt.getTagTree() && appCtxt.getTagTree().size() > 0
	};
	ZmSearchResultsFilterPanel.ADVANCED_FILTER[ZmSearchResultsFilterPanel.ID_FOLDER] = {
		text: 		ZmMsg.filterFolder,
		handler:	"ZmFolderSearchFilter",
		searchOp:	"in",
		noMenu:		true,						// has own menu to add to button
		apps:		ZmSearchResultsFilterPanel.ALL_APPS
	};
};

ZmSearchResultsFilterPanel.CONDITIONALS = [
	ZmParsedQuery.COND_AND,
	ZmParsedQuery.COND_OR,
	ZmParsedQuery.COND_NOT,
	ZmParsedQuery.GROUP_OPEN,
	ZmParsedQuery.GROUP_CLOSE
];




ZmSearchResultsFilterPanel.prototype._createHtml =
function() {
	this.getHtmlElement().innerHTML = AjxTemplate.expand(this.TEMPLATE, {id:this._htmlElId});
	this._basicPanel			= document.getElementById(this._htmlElId + "_basicPanel");
	this._basicContainer		= document.getElementById(this._htmlElId + "_basic");
	this._advancedPanel			= document.getElementById(this._htmlElId + "_advancedPanel");
	this._advancedContainer		= document.getElementById(this._htmlElId + "_advanced");
	this._conditionalsContainer	= document.getElementById(this._htmlElId + "_conditionals");
};

// returns a list of filters that apply for the results' app
ZmSearchResultsFilterPanel.prototype._getApplicableFilters = function(filterIds, filterHash) {
	
	var filters = [];
	for (var i = 0; i < filterIds.length; i++) {
		var id = filterIds[i];
		var filter = filterHash[id];
		filter.index = i;
		if (!appCtxt.checkPrecondition(filter.precondition)) {
			continue;
		}
		var apps = (filter.apps == ZmSearchResultsFilterPanel.ALL_APPS) ? filter.apps : AjxUtil.arrayAsHash(filter.apps || [ZmApp.MAIL]);
		if ((filter.apps == ZmSearchResultsFilterPanel.ALL_APPS) || apps[this._resultsApp]) {
			filters.push({id:id, filter:filter});
		}
	}
	return filters;
};

ZmSearchResultsFilterPanel.prototype._addFilters =
function() {
	var results = this._getApplicableFilters(ZmSearchResultsFilterPanel.BASIC_FILTER_LIST, ZmSearchResultsFilterPanel.BASIC_FILTER);
	Dwt.setVisible(this._basicPanel, (results.length > 0));
	for (var i = 0; i < results.length; i++) {
		var result = results[i];
		this._addBasicFilter(result.id, result.filter.text);
	}
	
	var results = this._getApplicableFilters(ZmSearchResultsFilterPanel.ADVANCED_FILTER_LIST, ZmSearchResultsFilterPanel.ADVANCED_FILTER);
	Dwt.setVisible(this._advancedPanel, (results.length > 0));
	this._addAdvancedFilters();
};

ZmSearchResultsFilterPanel.prototype._addAdvancedFilters =
function(attTypes) {
	ZmSearchResultsFilterPanel.attTypes = attTypes;
	var results = this._getApplicableFilters(ZmSearchResultsFilterPanel.ADVANCED_FILTER_LIST, ZmSearchResultsFilterPanel.ADVANCED_FILTER);
	Dwt.setVisible(this._advancedPanel, (results.length > 0));
	for (var i = 0; i < results.length; i++) {
		var result = results[i];
		this._addAdvancedFilter(result.id, result.filter.text);
	}
};

// basic filter is just an on/off checkbox
ZmSearchResultsFilterPanel.prototype._addBasicFilter =
function(id, text) {
	var cb = this._checkbox[id] = new DwtCheckbox({
				parent:			this,
				className:		"filter",
				parentElement:	this._basicContainer,
				id:				DwtId.makeId(ZmId.WIDGET_CHECKBOX, this._viewId, ZmSearchResultsFilterPanel.BASIC, id)
			});
	cb.setText(text);
	cb.addSelectionListener(this._checkboxListener.bind(this, id));
};

ZmSearchResultsFilterPanel.prototype._checkboxListener =
function(id, ev) {
	var cb = this._checkbox[id];
	if (!cb) { return; }
	var filter = ZmSearchResultsFilterPanel.BASIC_FILTER[id];
	var term = filter && filter.term;
	if (term) {
		if (cb.isSelected()) {
			this._controller.addSearchTerm(term);
		}
		else {
			this._controller.removeSearchTerm(term);
		}
	}
};

ZmSearchResultsFilterPanel.prototype._addAdvancedFilter =
function(id, text) {

	var button, menu
	// button is what shows up in search panel
	button = new DwtButton({
				parent:			this,
				parentElement:	this._advancedContainer,
				id:				ZmId.getButtonId(this._viewId, id)
			});
	button.setText(text);
	
	// we want a wide button with dropdown on far right
	var buttonEl = button.getHtmlElement();
	var table = buttonEl && buttonEl.firstChild;
	if (table && table.tagName && (table.tagName.toLowerCase() == "table")) {
		table.style.width = "100%";
	}

	var filter = ZmSearchResultsFilterPanel.ADVANCED_FILTER[id];
	// most filters start with a generic menu
	if (!filter.noMenu) {
		var params = {
			parent:	button,
			id:		ZmId.getMenuId(this._viewId, id),
			style:	DwtMenu.POPUP_STYLE
		};
		menu = new AjxCallback(this, this._createMenu, [params, id, filter]);
		button.setMenu({menu: menu, menuPopupStyle: DwtButton.MENU_POPUP_STYLE_CASCADE});
	}
	else {
		this._createFilter(button, id, filter);
	}
};

ZmSearchResultsFilterPanel.prototype._createMenu =
function(params, id, filter, button) {

	var menu = this._menu[id] = new DwtMenu(params);
	this._createFilter(menu, id, filter);
	return menu;
};

ZmSearchResultsFilterPanel.prototype._createFilter =
function(parent, id, filter) {
	var handler = filter && filter.handler;
	var updateCallback = this.update.bind(this, id);
	var params = {
		parent:			parent,
		id:				id,
		viewId:			this._viewId,
		searchOp:		filter.searchOp,
		updateCallback:	updateCallback,
		resultsApp:		this._resultsApp
	}
	var filterClass = eval(handler);
	this._advancedFilterHandlers[id] = new filterClass(params);
};

/**
 * Updates the current search with the given search term. A check is done to see if any of the current
 * search terms should be removed first. Some search operators should only appear once in a query (eg "in"),
 * and some conflict with others (eg "is:read" and "is:unread").
 * 
 * @param {string}			id			filter ID
 * @param {ZmSearchToken}	newTerms	search term(s)
 * @param {boolean}			noPopdown	if true, don't popdown menu after update
 */
ZmSearchResultsFilterPanel.prototype.update =
function(id, newTerms, noPopdown) {
	
	if (!id || !newTerms) { return; }
	
	newTerms = AjxUtil.toArray(newTerms);
	
	var curTerms = this._controller.getSearchTerms();
	if (curTerms && curTerms.length) {
		var ops = AjxUtil.arrayAsHash(AjxUtil.map(curTerms, function(a) { return a.op; }));
		var hasOr = ops[ZmParsedQuery.COND_OR];
		for (var i = 0; i < curTerms.length; i++) {
			var curTerm = curTerms[i];
			for (var j = 0; j < newTerms.length; j++) {
				var newTerm = newTerms[j];
				if (this._areExclusiveTerms(curTerm, newTerm, hasOr)) {
					this._controller.removeSearchTerm(curTerm, true);
				}
			}
		}
	}
	
	for (var i = 0; i < newTerms.length; i++) {
		this._controller.addSearchTerm(newTerms[i]);
	}
	
	if (this._menu[id] && !noPopdown) {
		this._menu[id].popdown();
	}
};

/**
 * Resets the filter panel by unchecking the basic filter checkboxes.
 */
ZmSearchResultsFilterPanel.prototype.reset =
function() {
	for (var id in this._checkbox) {
		var cb = this._checkbox[id];
		if (cb) {
			cb.setSelected(false);
		}
	}
	//reset all the advanced filters.
	for (var i in this._advancedFilterHandlers) {
		var handler = this._advancedFilterHandlers[i];
		if (handler && handler.reset) {
			handler.reset();
		}
	}
};

ZmSearchResultsFilterPanel.prototype.resetBasicFiltersToQuery =
function(query) {
	var filtersIds = ZmSearchResultsFilterPanel.BASIC_FILTER_LIST;
	for (var i = 0; i < filtersIds.length; i++) {
		var id = filtersIds[i];
		var cb = this._checkbox[id];
		if (!cb) {
			continue;
		}
		var filter = ZmSearchResultsFilterPanel.BASIC_FILTER[id];
		//checked if query has the filter term in it.
		cb.setSelected(query.indexOf(filter.term.toString()) !== -1);
	}
};

ZmSearchResultsFilterPanel.prototype._areExclusiveTerms =
function(termA, termB, hasOr) {
	termA = this._translateTerm(termA);
	termB = this._translateTerm(termB);
	return (ZmParsedQuery.areExclusive(termA, termB) || (!hasOr && (termA.op == termB.op) && !ZmParsedQuery.isMultiple(termA)));
};

// Treat "appt-start" like "before", "after", or "date" depending on its argument.
ZmSearchResultsFilterPanel.prototype._translateTerm =
function(term) {
	var newOp;
	if (term.op == "appt-start") {
		var first = term.arg.substr(0, 1);
		newOp = (first == "<") ? "before" : (first == ">") ? "after" : "date";
	}
	return newOp ? new ZmSearchToken(newOp, term.arg) : term;
};

ZmSearchResultsFilterPanel.prototype._addConditionals =
function() {
	var conds = ZmSearchResultsFilterPanel.CONDITIONALS;
	for (var i = 0; i < conds.length; i++) {
		var cond = conds[i];
		var bubbleParams = {
			parent:			appCtxt.getShell(),
			parentElement:	this._conditionalsContainer,
			address:		cond,
			addClass:		ZmParsedQuery.COND_OP[cond] ? ZmParsedQuery.COND : ZmParsedQuery.GROUP
		};
		var bubble = new ZmAddressBubble(bubbleParams);
		bubble.addSelectionListener(this._conditionalSelectionListener.bind(this));
	}
};

ZmSearchResultsFilterPanel.prototype._conditionalSelectionListener =
function(ev) {
	var bubble = ev.item;
	this._controller.addSearchTerm(new ZmSearchToken(bubble.address), true, true);
};

/**
 * Base class for widget that adds a term to the current search.
 * 
 * @param {hash}		params			hash of params:
 * @param {DwtControl}	parent			usually a DwtMenu
 * @param {string}		id				ID of filter
 * @param {string}		searchOp		search operator for this filter (optional)
 * @param {function}	updateCallback	called when value of filter (as a search term) has changed
 * @param {constant}	resultsApp		name of app corresponding to type of results
 */
ZmSearchFilter = function(params) {
	
	if (arguments.length == 0) { return; }
	
	this.parent = params.parent;
	this.id = params.id;
	this._viewId = params.viewId;
	this._searchOp = params.searchOp;
	this._updateCallback = params.updateCallback;
	this._resultsApp = params.resultsApp;
	
	this._setUi(params.parent);
};

ZmSearchFilter.prototype.isZmSearchFilter = true;
ZmSearchFilter.prototype.toString = function() { return "ZmSearchFilter"; };


// used to store data with a menu item
ZmSearchFilter.DATA_KEY = "DATA";

ZmSearchFilter.prototype._setUi = function(menu) {};

// Default listener for click on menu item. Constructs a search term from the value
// of that item and the search op for the filter.
ZmSearchFilter.prototype._selectionListener =
function(ev) {
	var data = ev && ev.dwtObj && ev.dwtObj.getData(ZmSearchFilter.DATA_KEY);
	if (data && this._searchOp) {
		var term = new ZmSearchToken(this._searchOp, data);
		this._updateCallback(term);
	}
};


/**
 * Allows the user to search by address or domain.
 * 
 * @param params
 */
ZmAddressSearchFilter = function(params) {
	ZmSearchFilter.apply(this, arguments);
};

ZmAddressSearchFilter.prototype = new ZmSearchFilter;
ZmAddressSearchFilter.prototype.constructor = ZmAddressSearchFilter;

ZmAddressSearchFilter.prototype.isZmAddressSearchFilter = true;
ZmAddressSearchFilter.prototype.toString = function() { return "ZmAddressSearchFilter"; };

// used for element IDs
ZmAddressSearchFilter.ADDRESS	= "address";
ZmAddressSearchFilter.DOMAIN	= "domain";

// map search op to address type
ZmAddressSearchFilter.ADDR = {};
ZmAddressSearchFilter.ADDR["from"]	= AjxEmailAddress.FROM;
ZmAddressSearchFilter.ADDR["to"]	= AjxEmailAddress.TO;

ZmAddressSearchFilter.INPUT_WIDTH = 25;
ZmAddressSearchFilter.NUM_DOMAINS_TO_FETCH = 100;
ZmAddressSearchFilter.NUM_DOMAINS_TO_SHOW = 10;

ZmAddressSearchFilter.prototype._addInput =
function(menu, text, width) {
	var menuItem = new DwtMenuItem({
				parent:	menu,
				id:		ZmId.getMenuItemId(this._viewId, this.id, ZmAddressSearchFilter.ADDRESS)
			});
	menuItem.setText(text);
	var subMenu = new DwtMenu({
				parent:	menuItem,
				id:		ZmId.getMenuId(this._viewId, this.id, ZmAddressSearchFilter.ADDRESS),
				style:	DwtMenu.GENERIC_WIDGET_STYLE
			});
	menuItem.setMenu({menu: subMenu, menuPopupStyle: DwtButton.MENU_POPUP_STYLE_CASCADE});
	var input = new DwtInputField({
				parent:	subMenu,
				id:		DwtId.makeId(ZmId.WIDGET_INPUT, this._viewId, this.id, ZmAddressSearchFilter.ADDRESS),
				size:	width
			});
	return input;
};

ZmAddressSearchFilter.prototype._addComboBox =
function(menu, text, width) {
	var menuItem = new DwtMenuItem({
				parent:	menu,
				id:		ZmId.getMenuItemId(this._viewId, this.id, ZmAddressSearchFilter.DOMAIN)
			});
	menuItem.setText(text);
	var subMenu = new DwtMenu({
				parent:	menuItem,
				id:		ZmId.getMenuId(this._viewId, this.id, ZmAddressSearchFilter.DOMAIN),
				style:	DwtMenu.GENERIC_WIDGET_STYLE
			});
	menuItem.setMenu({menu: subMenu, menuPopupStyle: DwtButton.MENU_POPUP_STYLE_CASCADE});
	var comboBox = new DwtComboBox({
				parent:			subMenu,
				id:		DwtId.makeId(ZmId.WIDGET_COMBOBOX, this._viewId, this.id, ZmAddressSearchFilter.ADDRESS),
				inputParams:	{size: width},
				maxRows: ZmAddressSearchFilter.NUM_DOMAINS_TO_SHOW,
				layout:DwtMenu.LAYOUT_SCROLL,
				autoScroll:true
			});
	comboBox.addChangeListener(this._domainChangeListener.bind(this));
	comboBox.input.addListener(DwtEvent.ONKEYUP, this._keyUpListener.bind(this));
	subMenu.addPopdownListener(comboBox.popdown.bind(comboBox));
	return comboBox;
};

ZmAddressSearchFilter.prototype._initAutocomplete =
function() {
	if (appCtxt.get(ZmSetting.CONTACTS_ENABLED) || appCtxt.get(ZmSetting.GAL_ENABLED) || appCtxt.isOffline) {
		var params = {
			dataClass:			appCtxt.getAutocompleter(),
			matchValue:			ZmAutocomplete.AC_VALUE_EMAIL,
			compCallback:		this._acCompHandler.bind(this),
			separator:			"",
			contextId:			[this._viewId, this.id].join("-")
		};
		var aclv = new ZmAutocompleteListView(params);
		aclv.handle(this._addressBox.getInputElement());
	}
};

// Process the filter if an address is autocompleted.
ZmAddressSearchFilter.prototype._acCompHandler =
function() {
	this._doUpdate(this._addressBox.getValue());
	this._addressBox.clear();
};


// Handles click on domain in the menu. Key events from the input will also
// come here. They are ignored.
ZmAddressSearchFilter.prototype._domainChangeListener =
function(ev) {
	// a menu item mouseup event will have dwtObj set
	if (ev && ev.dwtObj) {
		this._doUpdate(ev._args.newValue);
	}
};

// Process the filter if Enter is pressed.
ZmAddressSearchFilter.prototype._keyUpListener =
function(ev) {
	var keyCode = DwtKeyEvent.getCharCode(ev);
	if (keyCode == 13 || keyCode == 3) {
		this._doUpdate(this._domainBox.getText());
	}
};

ZmAddressSearchFilter.prototype._doUpdate =
function(address) {
	if (address) {
		var term = new ZmSearchToken(this._searchOp, address);
		this._updateCallback(term);
	}
};

ZmAddressSearchFilter.prototype._setUi =
function(menu) {
	
	this._addressBox = this._addInput(menu, ZmMsg.address, ZmAddressSearchFilter.INPUT_WIDTH);
	this._initAutocomplete();
	
	if (!ZmSearchResultsFilterPanel.domains) {
		var domainList = new ZmDomainList();
		domainList.search("", ZmAddressSearchFilter.NUM_DOMAINS_TO_FETCH, this._addDomains.bind(this, menu));
	}
	else {
		this._addDomains(menu, ZmSearchResultsFilterPanel.domains);
	}
};

ZmAddressSearchFilter.prototype._addDomains =
function(menu, domains) {

	ZmSearchResultsFilterPanel.domains = domains;
	this._domainBox = this._addComboBox(menu, ZmMsg.domain, ZmAddressSearchFilter.INPUT_WIDTH);
	if (domains && domains.length) {
		for (var i = 0; i < domains.length; i++) {
			var domain = domains[i];
			var addrType = ZmAddressSearchFilter.ADDR[this._searchOp];
			if (domain.hasAddress(addrType)) {
				this._domainBox.add(domain.name, domain.name);
			}
		}
	}
};

ZmAddressSearchFilter.prototype.reset =
function() {
	if (this._domainBox) {
		this._domainBox.setText('');
	}
	if (this._addressBox) {
		this._addressBox.setValue('');
	}
}

/**
 * Allows the user to search by date (before, after, or on a particular date).
 * 
 * @param params
 */
ZmDateSearchFilter = function(params) {

	this._calendar = {};	// calendar widgets

	ZmSearchFilter.apply(this, arguments);
	
	this._formatter = AjxDateFormat.getDateInstance(AjxDateFormat.SHORT);
};

ZmDateSearchFilter.prototype = new ZmSearchFilter;
ZmDateSearchFilter.prototype.constructor = ZmDateSearchFilter;

ZmDateSearchFilter.prototype.isZmDateSearchFilter = true;
ZmDateSearchFilter.prototype.toString = function() { return "ZmDateSearchFilter"; };

ZmDateSearchFilter.BEFORE	= "BEFORE";
ZmDateSearchFilter.AFTER	= "AFTER";
ZmDateSearchFilter.ON		= "ON";

ZmDateSearchFilter.TEXT_KEY = {};
ZmDateSearchFilter.TEXT_KEY[ZmDateSearchFilter.BEFORE]	= "filterBefore";
ZmDateSearchFilter.TEXT_KEY[ZmDateSearchFilter.AFTER]	= "filterAfter";
ZmDateSearchFilter.TEXT_KEY[ZmDateSearchFilter.ON]		= "filterOn";

ZmDateSearchFilter.OP = {};
ZmDateSearchFilter.OP[ZmDateSearchFilter.BEFORE]	= "before";
ZmDateSearchFilter.OP[ZmDateSearchFilter.AFTER]		= "after";
ZmDateSearchFilter.OP[ZmDateSearchFilter.ON]		= "date";

ZmDateSearchFilter.prototype._createCalendar =
function(menu, type) {
	var menuItem = new DwtMenuItem({
				parent:	menu,
				id:		ZmId.getMenuItemId(this._viewId, this.id, type)
			}); 
	menuItem.setText(ZmMsg[ZmDateSearchFilter.TEXT_KEY[type]]);
	var subMenu = new DwtMenu({
				parent:	menuItem,
				id:		ZmId.getMenuId(this._viewId, this.id, type),
				style:	DwtMenu.CALENDAR_PICKER_STYLE
			});
	menuItem.setMenu({menu: subMenu, menuPopupStyle: DwtButton.MENU_POPUP_STYLE_CASCADE});
	var calendar = new DwtCalendar({
				parent:	subMenu,
				id:		DwtId.makeId(ZmId.WIDGET_CALENDAR, this._viewId, this.id, type)
			});
	calendar.addSelectionListener(this._doUpdate.bind(this, type));
	return calendar;
};

ZmDateSearchFilter.prototype._doUpdate =
function(type) {
	var cal = this._calendar[type];
	var date = this._formatter.format(cal.getDate());
	var term = this._getSearchTerm(type, date);
	this._updateCallback(term, true);
};

ZmDateSearchFilter.prototype._getTypes =
function() {
	return [
		ZmDateSearchFilter.BEFORE,
		ZmDateSearchFilter.AFTER,
		ZmDateSearchFilter.ON
	];
};

ZmDateSearchFilter.prototype._getSearchTerm =
function(type, date) {
	return new ZmSearchToken(ZmDateSearchFilter.OP[type], date);
};

ZmDateSearchFilter.prototype._setUi =
function(menu) {
	var calTypes = this._getTypes();
	for (var i = 0; i < calTypes.length; i++) {
		var calType = calTypes[i];
		this._calendar[calType] = this._createCalendar(menu, calType);
	}
};



/**
 * Allows the user to search for appts by date (before, after, or on a particular date).
 * 
 * @param params
 */
ZmApptDateSearchFilter = function(params) {
	ZmDateSearchFilter.apply(this, arguments);
};

ZmApptDateSearchFilter.prototype = new ZmDateSearchFilter;
ZmApptDateSearchFilter.prototype.constructor = ZmApptDateSearchFilter;

ZmApptDateSearchFilter.prototype.isZmApptDateSearchFilter = true;
ZmApptDateSearchFilter.prototype.toString = function() { return "ZmApptDateSearchFilter"; };

ZmApptDateSearchFilter.prototype._getSearchTerm =
function(type, date) {
	if (type == ZmDateSearchFilter.BEFORE) {
		return new ZmSearchToken("appt-start", "<" + date);
	}
	else if (type == ZmDateSearchFilter.AFTER) {
		return new ZmSearchToken("appt-end", ">" + date);
	}
	else if (type == ZmDateSearchFilter.ON) {
		return [new ZmSearchToken("appt-start", "<=" + date),
				new ZmSearchToken("appt-end", ">=" + date)];
	}
};


/**
 * Allows the user to search by attachment type (MIME type).
 * 
 * @param params
 */
ZmAttachmentSearchFilter = function(params) {
	ZmSearchFilter.apply(this, arguments);
};

ZmAttachmentSearchFilter.prototype = new ZmSearchFilter;
ZmAttachmentSearchFilter.prototype.constructor = ZmAttachmentSearchFilter;

ZmAttachmentSearchFilter.prototype.isZmAttachmentSearchFilter = true;
ZmAttachmentSearchFilter.prototype.toString = function() { return "ZmAttachmentSearchFilter"; };


ZmAttachmentSearchFilter.prototype._setUi =
function(menu) {

	if (!ZmSearchResultsFilterPanel.attTypes) {
		var attTypeList = new ZmAttachmentTypeList();
		attTypeList.load(this._addAttachmentTypes.bind(this, menu));
	}
	else {
		this._addAttachmentTypes(menu, ZmSearchResultsFilterPanel.attTypes);
	}
};
	
ZmAttachmentSearchFilter.prototype._addAttachmentTypes =
function(menu, attTypes) {

	ZmSearchResultsFilterPanel.attTypes = attTypes;
	var added = {};
	if (attTypes && attTypes.length) {
		for (var i = 0; i < attTypes.length; i++) {
			var attType = attTypes[i];
			if (added[attType.desc]) { continue; }
			var menuItem = new DwtMenuItem({
						parent:	menu,
						id:		ZmId.getMenuItemId(this._viewId, this.id, attType.type.replace("/", ":"))
					}); 
			menuItem.setText(attType.desc);
			menuItem.setImage(attType.image);
			menuItem.setData(ZmSearchFilter.DATA_KEY, attType.query || attType.type);
			added[attType.desc] = true;
		}
	}
	else {
		var menuItem = new DwtMenuItem({parent:	menu}); 
		menuItem.setText(ZmMsg.noAtt);
	}
	menu.addSelectionListener(this._selectionListener.bind(this));
};

ZmAttachmentSearchFilter.prototype._selectionListener =
function(ev) {
	var data = ev && ev.dwtObj && ev.dwtObj.getData(ZmSearchFilter.DATA_KEY);
	if (data && this._searchOp) {
		var terms = [];
		if (data == ZmMimeTable.APP_ZIP  || data == ZmMimeTable.APP_ZIP2) {
			var other = (data == ZmMimeTable.APP_ZIP) ? ZmMimeTable.APP_ZIP2 : ZmMimeTable.APP_ZIP;
			terms.push(new ZmSearchToken(this._searchOp, data));
			terms.push(new ZmSearchToken(ZmParsedQuery.COND_OR));
			terms.push(new ZmSearchToken(this._searchOp, other));
		}
		else {
			terms.push(new ZmSearchToken(this._searchOp, data));
		}
		this._updateCallback(terms);
	}
};


/**
 * Allows the user to search by size (larger or smaller than a particular size).
 * 
 * @param params
 */
ZmSizeSearchFilter = function(params) {
	this._input = {};
	ZmSearchFilter.apply(this, arguments);
};

ZmSizeSearchFilter.prototype = new ZmSearchFilter;
ZmSizeSearchFilter.prototype.constructor = ZmSizeSearchFilter;

ZmSizeSearchFilter.prototype.isZmSizeSearchFilter = true;
ZmSizeSearchFilter.prototype.toString = function() { return "ZmSizeSearchFilter"; };

ZmSizeSearchFilter.LARGER	= "LARGER";
ZmSizeSearchFilter.SMALLER	= "SMALLER";

// used for element IDs
ZmSizeSearchFilter.UNIT	= "unit";

ZmSizeSearchFilter.COMBO_INPUT_WIDTH = 2;

ZmSizeSearchFilter.TYPES = [
		ZmSizeSearchFilter.LARGER,
		ZmSizeSearchFilter.SMALLER
];

ZmSizeSearchFilter.TEXT_KEY = {};
ZmSizeSearchFilter.TEXT_KEY[ZmSizeSearchFilter.LARGER]	= "filterLarger";
ZmSizeSearchFilter.TEXT_KEY[ZmSizeSearchFilter.SMALLER]	= "filterSmaller";

ZmSizeSearchFilter.OP = {};
ZmSizeSearchFilter.OP[ZmSizeSearchFilter.LARGER]	= "larger";
ZmSizeSearchFilter.OP[ZmSizeSearchFilter.SMALLER]	= "smaller";

ZmSizeSearchFilter.prototype._setUi =
function(menu) {

	var types = ZmSizeSearchFilter.TYPES;
	for (var i = 0; i < types.length; i++) {
		var type = types[i];
		var menuItem = new DwtMenuItem({
					parent:	menu,
					id:		ZmId.getMenuItemId(this._viewId, this.id, type)
				});
		menuItem.setText(ZmMsg[ZmSizeSearchFilter.TEXT_KEY[type]]);
		var subMenu = new DwtMenu({
					parent:	menuItem,
					id:		ZmId.getMenuId(this._viewId, this.id, type),
					style:	DwtMenu.GENERIC_WIDGET_STYLE
				});
		subMenu.addClassName(this.toString() + "SubMenu");
		menuItem.setMenu({menu: subMenu, menuPopupStyle: DwtButton.MENU_POPUP_STYLE_CASCADE});
		var input = this._input[type] = new DwtInputField({
			parent:          subMenu,
			type:            DwtInputField.FLOAT,
			errorIconStyle:  DwtInputField.ERROR_ICON_LEFT,
			validationStyle: DwtInputField.CONTINUAL_VALIDATION,
			size:            5
		});
		input.setValidNumberRange(0, 1e6);
		var comboBox = new DwtComboBox({
			parent:			input,
			parentElement: input.getInputElement().parentNode,
			id:		DwtId.makeId(ZmId.WIDGET_COMBOBOX, this._viewId, this.id, type+ZmSizeSearchFilter.UNIT),
			inputParams:	{size: ZmSizeSearchFilter.COMBO_INPUT_WIDTH},
			useLabel: true,
			posStyle:DwtControl.ABSOLUTE_STYLE,
			className: this.toString() + "Combobox"
		});
		input.addListener(DwtEvent.ONKEYUP, this._keyUpListener.bind(this, type, comboBox));
		comboBox.addChangeListener(this._unitChangeListener.bind(this, type, comboBox));
		comboBox.add(ZmMsg.kb,"KB",true); //select kb as default value
		comboBox.add(ZmMsg.mb,"MB");
	}
};

ZmSizeSearchFilter.prototype._unitChangeListener =
function(type, comboBox, ev) {
	var value = this._input[type].getValue();
	if (value && value != "") {
		var term = new ZmSearchToken(ZmSizeSearchFilter.OP[type], value + comboBox.getValue());
		this._updateCallback(term);
	}
};

ZmSizeSearchFilter.prototype._keyUpListener =
function(type, comboBox, ev) {
	var keyCode = DwtKeyEvent.getCharCode(ev);
	var input = this._input[type];
	if (keyCode == 13 || keyCode == 3) {
		var errorMsg = input.getValidationError();
		if (errorMsg) {
			appCtxt.setStatusMsg(errorMsg, ZmStatusView.LEVEL_WARNING);
		} else {
			var term = new ZmSearchToken(ZmSizeSearchFilter.OP[type],
			                             input.getValue() + comboBox.getValue());
			this._updateCallback(term);
		}
	}
};



/**
 * Allows the user to search by various message statuses (unread, flagged, etc).
 * 
 * @param params
 */
ZmStatusSearchFilter = function(params) {
	ZmSearchFilter.apply(this, arguments);
};

ZmStatusSearchFilter.prototype = new ZmSearchFilter;
ZmStatusSearchFilter.prototype.constructor = ZmStatusSearchFilter;

ZmStatusSearchFilter.prototype.isZmStatusSearchFilter = true;
ZmStatusSearchFilter.prototype.toString = function() { return "ZmStatusSearchFilter"; };


ZmStatusSearchFilter.prototype._setUi =
function(menu) {
	var values = ZmParsedQuery.IS_VALUES;
	for (var i = 0; i < values.length; i++) {
		var value = values[i];
		var menuItem = new DwtMenuItem({
					parent:	menu,
					id:		ZmId.getMenuItemId(this._viewId, this.id, value)
				}); 
		menuItem.setText(ZmMsg["QUERY_IS_" + value]);
		menuItem.setData(ZmSearchFilter.DATA_KEY, value);
	}
	menu.addSelectionListener(this._selectionListener.bind(this));
};



/**
 * Allows the user to search by tags.
 * 
 * @param params
 * TODO: filter should not show up if no tags
 */
ZmTagSearchFilter = function(params) {
	ZmSearchFilter.apply(this, arguments);

	this._tagList = appCtxt.getTagTree();
	if (this._tagList) {
		this._tagList.addChangeListener(this._tagChangeListener.bind(this));
	}
};

ZmTagSearchFilter.prototype = new ZmSearchFilter;
ZmTagSearchFilter.prototype.constructor = ZmTagSearchFilter;

ZmTagSearchFilter.prototype.isZmTagSearchFilter = true;
ZmTagSearchFilter.prototype.toString = function() { return "ZmTagSearchFilter"; };

ZmTagSearchFilter.prototype._setUi =
function(menu) {

	this._menu = menu;
	var tags = appCtxt.getTagTree().asList();
	if (tags && tags.length) {
		for (var i = 0; i < tags.length; i++) {
			var tag = tags[i];
			if (tag.id == ZmOrganizer.ID_ROOT) { continue; }
			var menuItem = new DwtMenuItem({
						parent:	menu,
						id:		ZmId.getMenuItemId(this._viewId, this.id, tag.id)
					}); 
			menuItem.setText(tag.getName());
			menuItem.setImage(tag.getIconWithColor());
			menuItem.setData(ZmSearchFilter.DATA_KEY, tag.getName(false, null, true));
		}
	}
	menu.addSelectionListener(this._selectionListener.bind(this));
};

// for any change to tags, just re-render
ZmTagSearchFilter.prototype._tagChangeListener =
function(ev) {
	if (this._menu) {
		this._menu.removeChildren();
		this._setUi(this._menu);
	}
};

/**
 * Allows the user to search by folder.
 * 
 * @param params
 */
ZmFolderSearchFilter = function(params) {
	ZmSearchFilter.apply(this, arguments);
	
	// set button title to appropriate organizer type
	if (!ZmFolderSearchFilter.TEXT_KEY) {
		ZmFolderSearchFilter._initConstants();
	}
	var title = ZmMsg[ZmFolderSearchFilter.TEXT_KEY[this._resultsApp]] || ZmMsg.filterFolder;
	params.parent.setText(title);
};

ZmFolderSearchFilter.prototype = new ZmSearchFilter;
ZmFolderSearchFilter.prototype.constructor = ZmFolderSearchFilter;

ZmFolderSearchFilter.prototype.isZmFolderSearchFilter = true;
ZmFolderSearchFilter.prototype.toString = function() { return "ZmFolderSearchFilter"; };

ZmFolderSearchFilter._initConstants =
function(button) {
	ZmFolderSearchFilter.TEXT_KEY = {};
	ZmFolderSearchFilter.TEXT_KEY[ZmApp.MAIL]		= "filterFolder";
	ZmFolderSearchFilter.TEXT_KEY[ZmApp.CALENDAR]	= "filterCalendar";
	ZmFolderSearchFilter.TEXT_KEY[ZmApp.CONTACTS]	= "filterAddressBook";
	ZmFolderSearchFilter.TEXT_KEY[ZmApp.TASKS]		= "filterTasksFolder";
	ZmFolderSearchFilter.TEXT_KEY[ZmApp.BRIEFCASE]	= "filterBriefcase";
};

ZmFolderSearchFilter.prototype._setUi =
function(button) {

	// create menu for button
	var moveMenu = this._moveMenu = new DwtMenu({
				parent: button,
				id:		ZmId.getMenuId(this._viewId, this.id)
			});
	moveMenu.getHtmlElement().style.width = "auto";
	button.setMenu({menu: moveMenu, menuPopupStyle: DwtButton.MENU_POPUP_STYLE_CASCADE});

	var chooser = this._folderChooser = new ZmFolderChooser({
				parent:			moveMenu,
				id:				DwtId.makeId(ZmId.WIDGET_CHOOSER, this._viewId, this.id),
				hideNewButton:	true
			});
	var moveParams = this._getMoveParams(chooser);
	chooser.setupFolderChooser(moveParams, this._selectionListener.bind(this));
};

ZmFolderSearchFilter.prototype._getMoveParams =
function(dlg) {
	return {
		overviewId:		dlg.getOverviewId([this.toString(), this._resultsApp, this._viewId].join("_")),
		treeIds:		[ZmApp.ORGANIZER[this._resultsApp]],
		noRootSelect: 	true,
		treeStyle:		DwtTree.SINGLE_STYLE
	};
};

ZmFolderSearchFilter.prototype._selectionListener =
function(folder) {
	if (folder) {
		var term = new ZmSearchToken("in", folder.createQuery(true));
		this._updateCallback(term);
		this._moveMenu.popdown();
	}
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmTreeView")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * Creates an empty tree view.
 * @class
 * This class displays data in a tree structure.
 *
 * @author Conrad Damon
 * 
 * @param {Hash}	params				the hash of parameters
 * @param {DwtControl}	params.parent				the tree's parent widget
 * @param {constant}	params.type				the organizer type
 * @param {String}	params.className				the CSS class
 * @param {constant}	params.posStyle				the positioning style
 * @param {constant}	params.overviewId			theoverview ID
 * @param {String}	params.headerClass			the CSS class for header item
 * @param {DwtDragSource}	params.dragSrc				the drag source
 * @param {DwtDropTarget}	params.dropTgt				the drop target
 * @param {constant}	params.treeStyle				tree style (see {@link DwtTree})
 * @param {Boolean}	params.isCheckedByDefault	sets the default state of "checked" tree style
 * @param {Hash}	params.allowedTypes			a hash of org types this tree may display
 * @param {Hash}	params.allowedSubTypes		a hash of org types this tree may display below top level
 * @param {boolean}    params.actionSupported     (default to value from Overview if not passed)
 *
 * @extends		DwtTree
 */
ZmTreeView = function(params) {

	if (arguments.length == 0) { return; }

	DwtTree.call(this, {
		parent: params.parent,
		parentElement: params.parentElement,
		style: params.treeStyle,
		isCheckedByDefault: params.isCheckedByDefault,
		className: (params.className || "OverviewTree"),
		posStyle: params.posStyle,
		id: params.id
	});

	this._headerClass = params.headerClass || "overviewHeader";
	this.overviewId = params.overviewId;
	this.type = params.type;
	this.allowedTypes = params.allowedTypes;
	this.allowedSubTypes = params.allowedSubTypes;

	this._overview = appCtxt.getOverviewController().getOverview(this.overviewId);
	
	this._dragSrc = params.dragSrc;
	this._dropTgt = params.dropTgt;

	this.actionSupported = params.actionSupported !== undefined
							? params.actionSupported
							: this._overview.actionSupported;

	this.dynamicWidth = this._overview.dynamicWidth;

	this._dataTree = null;
	this._treeItemHash  = {};	// map organizer to its corresponding tree item by ID
	this._idToOrganizer = {};	// map DwtControl htmlElId to the organizer for external Drag and Drop

};

ZmTreeView.KEY_TYPE	= "_type_";
ZmTreeView.KEY_ID	= "_treeId_";

// compare functions for each type
ZmTreeView.COMPARE_FUNC = {};

// add space after the following items
ZmTreeView.ADD_SEP = {};
ZmTreeView.ADD_SEP[ZmFolder.ID_TRASH] = true;

ZmTreeView.MAX_ITEMS = 50;

// Static methods

/**
 * Finds the correct position for an organizer within a node, given
 * a sort function.
 *
 * @param {DwtTreeItem}	node			the node under which organizer is to be added
 * @param {ZmOrganizer}	organizer		the organizer
 * @param {function}	sortFunction	the function for comparing two organizers
 * @return	{int}	the index
 */
ZmTreeView.getSortIndex =
function(node, organizer, sortFunction) {
	if (!sortFunction) return null;
	var cnt = node.getItemCount();
	var children = node.getItems();
	for (var i = 0; i < children.length; i++) {
		if (children[i]._isSeparator) continue;
		var child = children[i].getData(Dwt.KEY_OBJECT);
		if (!child) continue;
		var test = sortFunction(organizer, child);
		if (test == -1) {
			return i;
		}
	}
	return i;
};

ZmTreeView.prototype = new DwtTree;
ZmTreeView.prototype.constructor = ZmTreeView;

// Public methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmTreeView.prototype.toString = 
function() {
	return "ZmTreeView";
};


/**
 * Populates the tree view with the given data and displays it.
 *
 * @param {Hash}	params		a hash of parameters
 * @param   {ZmTree}	params.dataTree		data in tree form
 * @param	{Boolean}	params.showUnread	if <code>true</code>, show unread counts
 * @param	{Hash}	params.omit			a hash of organizer IDs to ignore
 * @param	{Hash}	params.include		a hash of organizer IDs to include
 * @param	{Boolean}	params.omitParents	if <code>true</code>, do NOT insert parent nodes as needed
 * @param	{Hash}	params.searchTypes	the types of saved searches to show
 * @param	{Boolean}	params.noTooltips	if <code>true</code>, don't show tooltips for tree items
 * @param	{Boolean}	params.collapsed		if <code>true</code>, initially leave the root collapsed 
 * @param 	{Hash}          params.optButton        a hash of data for showing a options button in the item: image, tooltip, callback
 */
ZmTreeView.prototype.set =
function(params) {
	this._showUnread = params.showUnread;
	this._dataTree = params.dataTree;
	this._optButton = params.optButton;

	this.clearItems();

	// create header item
	var root = this._dataTree.root;
	var isMultiAcctSubHeader = (appCtxt.multiAccounts && (this.type == ZmOrganizer.SEARCH || this.type == ZmOrganizer.TAG));
	var imageInfo = this._getHeaderTreeItemImage();
	var ti = this._headerItem = new DwtHeaderTreeItem({
		parent:				this,
		className:			isMultiAcctSubHeader ? "DwtTreeItem" : this._headerClass,
		imageInfo:			imageInfo,
		id:					ZmId.getTreeItemId(this.overviewId, null, this.type),
		optButton:			params.optButton,
		dndScrollCallback:	this._overview && this._overview._dndScrollCallback,
		dndScrollId:		this._overview && this._overview._scrollableContainerId
	});
	ti._isHeader = true;
	var name = ZmMsg[ZmOrganizer.LABEL[this.type]] || root.name;
	if (name) {
		ti.setText(name);
	}
	ti.setData(Dwt.KEY_ID, root.id);
	ti.setData(Dwt.KEY_OBJECT, root);
	ti.setData(ZmTreeView.KEY_ID, this.overviewId);
	ti.setData(ZmTreeView.KEY_TYPE, this.type);
	if (this._dropTgt) {
		ti.setDropTarget(this._dropTgt);
	}
	this._treeItemHash[root.id] = ti;
	ti.getHtmlElement().style.overflow = "hidden";
	// render the root item's children (ie everything else)
	params.treeNode = ti;
	params.organizer = root;
	this._render(params);
	ti.setExpanded(!params.collapsed, null, true);

	if (!appCtxt.multiAccounts) {
		this.addSeparator();
	}


	if (appCtxt.getSkinHint("noOverviewHeaders") ||
		this._hideHeaderTreeItem())
	{
		ti.setVisible(false, true);
	}
};

/**
 * Gets the tree item that represents the organizer with the given ID.
 *
 * @param {int}		id		an organizer ID
 * @return	{DwtTreeItem}		the item
 */
ZmTreeView.prototype.getTreeItemById =
function(id) {
	return this._treeItemHash[id];
};

/**
 * Gets the tree view's header node.
 * 
 * @return	{DwtHeaderTreeItem}		the item
 */
ZmTreeView.prototype.getHeaderItem =
function() {
	return this._headerItem;
};

/**
 * Gets the currently selected organizer(s). If tree view is checkbox style,
 * return value is an {Array} otherwise, a single {DwtTreeItem} object is returned.
 * 
 * @return	{Array|DwtTreeItem}		the selected item(s)
 */
ZmTreeView.prototype.getSelected =
function() {
	if (this.isCheckedStyle) {
		var selected = [];
		// bug #44805 - iterate thru the entire tree item hash in case there are
		// more than one header items in the tree view (e.g. Imap accounts)
		for (var i in this._treeItemHash) {
			var ti = this._treeItemHash[i];
			if (ti && ti.getChecked()) {
				selected.push(ti.getData(Dwt.KEY_OBJECT));
			}
		}
		return selected;
	} else {
		return (this.getSelectionCount() != 1)
			? null : this.getSelection()[0].getData(Dwt.KEY_OBJECT);
	}
};

/**
 * Selects the tree item for the given organizer.
 *
 * @param {ZmOrganizer}	organizer		the organizer to select, or its ID
 * @param {Boolean}	skipNotify	if <code>true</code>, skip notifications
 * @param {Boolean}	noFocus		if <code>true</code>, select item but don't set focus to it
 */
ZmTreeView.prototype.setSelected =
function(organizer, skipNotify, noFocus) {
	var id = ZmOrganizer.getSystemId((organizer instanceof ZmOrganizer) ? organizer.id : organizer);
	if (!id || !this._treeItemHash[id]) { return; }
	this.setSelection(this._treeItemHash[id], skipNotify, false, noFocus);
};


// Private and protected methods

/**
 * Draws the children of the given node.
 *
 * @param params		[hash]			hash of params:
 *        treeNode		[DwtTreeItem]	current node
 *        organizer		[ZmOrganizer]	its organizer
 *        omit			[Object]*		hash of system folder IDs to ignore	
 *        include		[object]*		hash of system folder IDs to include
 *        showOrphans	[boolean]*		if true, show parent chain of any
 * 										folder of this type, as well as the folder
 *        searchTypes	[hash]*			types of saved searches to show
 *        noTooltips	[boolean]*		if true, don't show tooltips for tree items
 *        startPos		[int]*			start rendering this far into list of children
 * 
 * TODO: Add logic to support display of folders that are not normally allowed in
 * 		this tree, but that have children (orphans) of an allowed type
 * TODO: Only sort folders we're showing (requires two passes).
 * 
 * @private
 */
ZmTreeView.prototype._render =
function(params) {

	params.omit = params.omit || {};
	this._setOmit(params.omit, params.dataTree);

	var org = params.organizer;
	var children = org.children.getArray();
	if (org.isDataSource(ZmAccount.TYPE_IMAP)) {
		children.sort(ZmImapAccount.sortCompare);
	} else if (ZmTreeView.COMPARE_FUNC[this.type]) {
		if (appCtxt.isOffline && this.type == ZmOrganizer.SEARCH) {
			var local = [];
			for (var j = 0; j < children.length; j++) {
				var child = children[j];
				if (child && child.type == ZmOrganizer.SEARCH && !child.isOfflineGlobalSearch) {
					local.push(child);
				}
			}
			children = local;
		}
		// IE loses type info on the children array - the props are there and it can be iterated,
		// but a function call like sort() blows up. So create an array local to child win.
		if (appCtxt.isChildWindow && AjxEnv.isIE) {
			var children1 = [];
			for (var i = 0, len = children.length; i < len; i++) {
				children1.push(children[i]);
			}
			children = children1;
		}
		children.sort(eval(ZmTreeView.COMPARE_FUNC[this.type]));
	}
	DBG.println(AjxDebug.DBG3, "Render: " + org.name + ": " + children.length);
	var addSep = true;
	var numItems = 0;
	var len = children.length;
    if (params.startPos === undefined && params.lastRenderedFolder ){
        for (var i = 0, len = children.length; i < len; i++) {
            if (params.lastRenderedFolder == children[i] ){
               params.startPos = i + 1; // Next to lastRenderedFolder
               break;
            }
        }
        DBG.println(AjxDebug.DBG1, "load remaining folders: " + params.startPos);
    }
	for (var i = params.startPos || 0; i < len; i++) {
		var child = children[i];
		if (!child || (params.omit && params.omit[child.nId])) { continue; }
		if (!(params.include && params.include[child.nId])) {
			if (!this._isAllowed(org, child)) {
				if (params.omitParents) continue;
				var proxy = AjxUtil.createProxy(params);
				proxy.treeNode = null;
				proxy.organizer = child;
				this._render(proxy);
				continue;
			}
		}

		if (child.numTotal == 0 && (child.nId == ZmFolder.ID_SYNC_FAILURES)) {
			continue;
		}

		var parentNode = params.treeNode;
		var account = appCtxt.multiAccounts && child.getAccount();

		// bug: 43067 - reparent calendars for caldav-based accounts
		if (account && account.isCalDavBased() &&
			child.parent.nId == ZmOrganizer.ID_CALENDAR)
		{
			parentNode = parentNode.parent;
		}

		// if there's a large number of folders to display, make user click on special placeholder
		// to display remainder; we then display them MAX_ITEMS at a time
		if (numItems >= ZmTreeView.MAX_ITEMS) {
			if (params.startPos) {
				// render next chunk
				params.startPos = i;
				params.len = (params.startPos + ZmTreeView.MAX_ITEMS >= len) ? len : 0;	// hint that we're done
				this._showRemainingFolders(params);
				return;
			} else if (numItems >= ZmTreeView.MAX_ITEMS * 2) {
				// add placeholder tree item "Show remaining folders"
				var orgs = ZmMsg[ZmOrganizer.LABEL[this.type]].toLowerCase();
				var name = AjxMessageFormat.format(ZmMsg.showRemainingFolders, orgs);
				child = new ZmFolder({id:ZmFolder.ID_LOAD_FOLDERS, name:name, parent:org});
				child._tooltip = AjxMessageFormat.format(ZmMsg.showRemainingFoldersTooltip, [(children.length - i), orgs]);
				var ti = this._addNew(parentNode, child);
				ti.enableSelection(true);
				if (this.isCheckedStyle) {
					ti.showCheckBox(false);
				}
                params.lastRenderedFolder  = children[i - 1];
				params.showRemainingFoldersNode = ti;
				child._showFoldersCallback = new AjxCallback(this, this._showRemainingFolders, [params]);
				if (this._dragSrc) {
					// Bug 55763 - expand placeholder on hover; replacing the _dragHover function is the easiest way, if a bit hacky
					ti._dragHover = this._showRemainingFolders.bind(this, params);
				}

				return;
			}
		}

		// NOTE: Separates public and shared folders
		if ((org.nId == ZmOrganizer.ID_ROOT) && child.link && addSep) {
			params.treeNode.addSeparator();
			addSep = false;
		}
		this._addNew(parentNode, child, null, params.noTooltips, params.omit);
		numItems++;
	}
};

ZmTreeView.prototype._setOmit =
function(omit, dataTree) {
	for (var id in ZmFolder.HIDE_ID) {
		omit[id] = true;
	}
	//note - the dataTree thing was in the previous code so I keep it, but seems all the ZmFolder.HIDE_NAME code is commented out, so
	//not sure it's still needed.
	dataTree = this.type !== ZmOrganizer.VOICE && dataTree;
	if (!dataTree) {
		return;
	}
	for (var name in ZmFolder.HIDE_NAME) {
		var folder = dataTree.getByName(name);
		if (folder) {
			omit[folder.id] = true;
		}
	}
};

/**
 * a bit complicated and hard to explain - We should only allow (render on this view)
 * a child of an "allowedSubTypes", if all its ancestors are allowed all the way to the root ("Folders"), meaning
 * it has an ancestor that is of the allowedTypes (but is not the root)
 * e.g.
 * allowed:
 * Folders-->folder1--->searchFolder1
 * Folders--->folder1--->folder2--->folder3--->searchFolder1
 *
 * not allowed:
 * Folders-->searchFolder1
 * Folders-->searchFolder1--->searchFolder2
 *
 * @param org
 * @param child
 * @returns {*}
 * @private
 */
ZmTreeView.prototype._isAllowed =
function(org, child) {

	if (!org) { //could happen, for example the Zimlets root doesn't have a parent.
		return true; //seems returning true in this case works... what a mess.
	}

	// Within the Searches tree, only show saved searches that return a type that belongs to this app
	if (this.type === ZmOrganizer.SEARCH && child.type === ZmOrganizer.SEARCH && this._overview.appName) {
		var searchTypes = child.search.types && child.search.types.getArray();
		if (!searchTypes || searchTypes.length === 0) {
			searchTypes = [ ZmItem.MSG ];   // search with no types defaults to "message"
		}
		var common = AjxUtil.intersection(searchTypes,
			ZmApp.SEARCH_TYPES[this._overview.appName] ||  ZmApp.SEARCH_TYPES[appCtxt.getCurrentAppName()]);
		if (common.length === 0) {
			return false;
		}
	}

	if (org.nId == ZmOrganizer.ID_ROOT) {
		return this.allowedTypes[child.type];
	}

	//org is not root
	if (this.allowedTypes[child.type]) {
		return true; //optimization, end the recursion if we find a non root allowed ancestor.
	}

	if (this.allowedSubTypes[child.type]) {
		return this._isAllowed(org.parent, org); //go up parent to see if eventually it's allowed.
	}

	return false;
};

/**
 * Adds a tree item node for the given organizer to the tree, and then adds its children.
 *
 * @param parentNode	[DwtTreeItem]	node under which to add the new one
 * @param organizer		[ZmOrganizer]	organizer for the new node
 * @param index			[int]*			position at which to add the new node
 * @param noTooltips	[boolean]*		if true, don't show tooltips for tree items
 * @param omit			[Object]*		hash of system folder IDs to ignore
 * 
 * @private
 */
ZmTreeView.prototype._addNew =
function(parentNode, organizer, index, noTooltips, omit) {
	var ti;
	var parentControlId;
	// check if we're adding a datasource folder
	var dsColl = (organizer.type == ZmOrganizer.FOLDER) && appCtxt.getDataSourceCollection();
	var dss = dsColl && dsColl.getByFolderId(organizer.nId);
	var ds = (dss && dss.length > 0) ? dss[0] : null;

	if (ds && ds.type == ZmAccount.TYPE_IMAP) {
		var cname = appCtxt.isFamilyMbox ? null : this._headerClass;
		ti = new DwtHeaderTreeItem({
			parent:this,
			text:organizer.getName(),
			className:cname
		});
	} else {
		// create parent chain
		if (!parentNode) {
			var stack = [];
			var parentOrganizer = organizer.parent;
			if (parentOrganizer) {
				while ((parentNode = this.getTreeItemById(parentOrganizer.id)) == null) {
					stack.push(parentOrganizer);
					parentOrganizer = parentOrganizer.parent;
				}
			}
			while (parentOrganizer = stack.pop()) {
				parentNode = this.getTreeItemById(parentOrganizer.parent.id);
				parentControlId = ZmId.getTreeItemId(this.overviewId, parentOrganizer.id);
				parentNode = new DwtTreeItem({
					parent:					parentNode,
					text:					parentOrganizer.getName(),
					imageInfo:				parentOrganizer.getIconWithColor(),
					forceNotifySelection:	true,
					arrowDisabled:			!this.actionSupported,
					dynamicWidth:			this.dynamicWidth,
					dndScrollCallback:		this._overview && this._overview._dndScrollCallback,
					dndScrollId:			this._overview && this._overview._scrollableContainerId,
					id:						parentControlId
				});
				parentNode.setData(Dwt.KEY_ID, parentOrganizer.id);
				parentNode.setData(Dwt.KEY_OBJECT, parentOrganizer);
				parentNode.setData(ZmTreeView.KEY_ID, this.overviewId);
				parentNode.setData(ZmTreeView.KEY_TYPE, parentOrganizer.type);
				this._treeItemHash[parentOrganizer.id] = parentNode;
				this._idToOrganizer[parentControlId] = parentOrganizer.id;
			}
		}
		var params = {
			parent:				parentNode,
			index:				index,
			text:				organizer.getName(this._showUnread),
			arrowDisabled:		!this.actionSupported,
			dynamicWidth:		this.dynamicWidth,
			dndScrollCallback:	this._overview && this._overview._dndScrollCallback,
			dndScrollId:		this._overview && this._overview._scrollableContainerId,
			imageInfo:			organizer.getIconWithColor(),
			imageAltText:		organizer.getIconAltText(),
			id:					ZmId.getTreeItemId(this.overviewId, organizer.id)
		};
		// now add item
		ti = new DwtTreeItem(params);
		this._idToOrganizer[params.id] = organizer.id;
	}

	if (appCtxt.multiAccounts &&
		(organizer.type == ZmOrganizer.SEARCH ||
		 organizer.type == ZmOrganizer.TAG))
	{
		ti.addClassName("DwtTreeItemChildDiv");
	}

	ti.setDndText(organizer.getName());
	ti.setData(Dwt.KEY_ID, organizer.id);
	ti.setData(Dwt.KEY_OBJECT, organizer);
	ti.setData(ZmTreeView.KEY_ID, this.overviewId);
	ti.setData(ZmTreeView.KEY_TYPE, organizer.type);
	if (!noTooltips) {
		var tooltip = organizer.getToolTip();
		if (tooltip) {
			ti.setToolTipContent(tooltip);
		}
	}
	if (this._dragSrc) {
		ti.setDragSource(this._dragSrc);
	}
	if (this._dropTgt) {
		ti.setDropTarget(this._dropTgt);
	}
	this._treeItemHash[organizer.id] = ti;

	if (ZmTreeView.ADD_SEP[organizer.nId]) {
		parentNode.addSeparator();
	}

	// recursively add children
	if (organizer.children && organizer.children.size()) {
		this._render({treeNode:ti, organizer:organizer, omit:omit});
	}

	if (ds && ds.type == ZmAccount.TYPE_IMAP) {
		ti.setExpanded(!appCtxt.get(ZmSetting.COLLAPSE_IMAP_TREES));
	}

	return ti;
};


/**
 * Gets the data (an organizer) from the tree item nearest the one
 * associated with the given ID.
 *
 * @param {int}	id	an organizer ID
 * @return	{Object}	the data or <code>null</code> for none
 */
ZmTreeView.prototype.getNextData =
function(id) {
	var treeItem = this.getTreeItemById(id);
	if(!treeItem || !treeItem.parent) { return null; }

	while (treeItem && treeItem.parent) {
		var parentN = treeItem.parent;
		if (!(parentN instanceof DwtTreeItem)) {
			return null;
		}
		var treeItems = parentN.getItems();
		var result = null;
		if (treeItems && treeItems.length > 1) {
			for(var i = 0; i < treeItems.length; i++) { 
				var tmp = treeItems[i];
				if (tmp == treeItem) {
					var nextData = this.findNext(treeItem, treeItems, i);
					if (nextData) { return nextData; }
					var prevData = this.findPrev(treeItem, treeItems, i);
					if (prevData) {	return prevData; }
				}
			}
		}
		treeItem = treeItem.parent;
	}
	return null;
};

ZmTreeView.prototype.findNext =
function(treeItem, treeItems, i) {
	for (var j = i + 1; j < treeItems.length; j++) {
		var next = treeItems[j];
		if (next && next.getData) {
			return next.getData(Dwt.KEY_OBJECT);
		}
	}
	return null;
};

ZmTreeView.prototype.findPrev =
function(treeItem, treeItems, i) {
	for (var j = i - 1; j >= 0; j--) {
		var prev = treeItems[j];
		if (prev && prev.getData) {
			return prev.getData(Dwt.KEY_OBJECT);
		}
	}
	return null;
};

/**
 * Renders a chunk of tree items, using a timer so that the browser doesn't get overloaded.
 * 
 * @param params	[hash]		hash of params (see _render)
 * 
 * @private
 */
ZmTreeView.prototype._showRemainingFolders =
function(params) {

	if (params.showRemainingFoldersNode){
		params.showRemainingFoldersNode.dispose();
	}

	AjxTimedAction.scheduleAction(new AjxTimedAction(this,
		function() {
			this._render(params);
			if (params.len) {
				var orgs = ZmMsg[ZmOrganizer.LABEL[this.type]].toLowerCase();
				appCtxt.setStatusMsg(AjxMessageFormat.format(ZmMsg.foldersShown, [params.len, orgs]));
				params.len = 0;
			}
		}), 100);
};

ZmTreeView.prototype._getNextTreeItem =
function(next) {
	var nextItem = DwtTree.prototype._getNextTreeItem.apply(this, arguments);
	return nextItem || (this._overview && this._overview._getNextTreeItem(next, this));
};

ZmTreeView.prototype._getFirstTreeItem =
function() {
	if (!this._overview) {
		return DwtTree.prototype._getFirstTreeItem.call(tree);
	}

	var treeids = this._overview.getTreeViews();
	var tree = this._overview.getTreeView(treeids[0]);
	return tree && DwtTree.prototype._getFirstTreeItem.call(tree);
};

ZmTreeView.prototype._getLastTreeItem =
function() {
	if (!this._overview) {
		return DwtTree.prototype._getLastTreeItem.call(tree);
	}

	var treeids = this._overview.getTreeViews();
	var tree = this._overview.getTreeView(treeids[treeids.length - 1]);
	return tree && DwtTree.prototype._getLastTreeItem.call(tree);
};

ZmTreeView.prototype._hideHeaderTreeItem =
function() {
	return (appCtxt.multiAccounts && appCtxt.accountList.size() > 1 &&
			(this.type == ZmOrganizer.FOLDER ||
			 this.type == ZmOrganizer.ADDRBOOK ||
			 this.type == ZmOrganizer.CALENDAR ||
			 this.type == ZmOrganizer.TASKS ||
			 this.type == ZmOrganizer.BRIEFCASE ||
			 this.type == ZmOrganizer.PREF_PAGE ||
			 this.type == ZmOrganizer.ZIMLET));
};

ZmTreeView.prototype._getHeaderTreeItemImage =
function() {
	if (appCtxt.multiAccounts) {
		if (this.type == ZmOrganizer.SEARCH)	{ return "SearchFolder"; }
		if (this.type == ZmOrganizer.TAG)		{ return "TagStack"; }
	}
	return null;
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmTagMenu")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * Creates an empty tag menu.
 * @class
 * This class represents a menu structure of tags that can be added to or removed
 * from item(s). Based on the items passed in when it renders, it presents a 
 * list of tags that can be added (any tag all the items don't already have), and a
 * list of tags that can be removed (tags that any of the items have).
 * <p>
 * Since the content is set every time it is displayed, the tag menu doesn't need
 * a change listener.</p>
 *
 * @param {DwtControl}	parent		the parent widget
 * @param {ZmController}	controller	the owning controller
 * 
 * @extends		ZmPopupMenu
 */
ZmTagMenu = function(parent, controller) {

	// create a menu (though we don't put anything in it yet) so that parent widget shows it has one
	ZmPopupMenu.call(this, parent, null, parent.getHTMLElId() + "|MENU", controller);

	parent.setMenu(this);
	this._addHash = {};
	this._removeHash = {};
	this._evtMgr = new AjxEventMgr();
	this._desiredState = true;
	this._items = null;
	this._dirty = true;

	// Use a delay to make sure our slow popup operation isn't called when someone
	// is just rolling over a menu item to get somewhere else.
	if (parent instanceof DwtMenuItem) {
		parent.setHoverDelay(ZmTagMenu._HOVER_TIME);
	}
};

ZmTagMenu.prototype = new ZmPopupMenu;
ZmTagMenu.prototype.constructor = ZmTagMenu;

ZmTagMenu.KEY_TAG_EVENT		= "_tagEvent_";
ZmTagMenu.KEY_TAG_ADDED		= "_tagAdded_";
ZmTagMenu.MENU_ITEM_ADD_ID	= "tag_add";
ZmTagMenu.MENU_ITEM_REM_ID	= "tag_remove";

ZmTagMenu._HOVER_TIME = 200;

ZmTagMenu.prototype.toString =
function() {
	return "ZmTagMenu";
};

ZmTagMenu.prototype.addSelectionListener = 
function(listener) {
	this._evtMgr.addListener(DwtEvent.SELECTION, listener);
};

ZmTagMenu.prototype.removeSelectionListener = 
function(listener) {
	this._evtMgr.removeListener(DwtEvent.SELECTION, listener);    	
};

ZmTagMenu.prototype.setEnabled =
function(enabled) {
	// If there are no tags, then enable later
	this._desiredState = enabled;
	if (enabled && !this._tagList) { return; }

	this.parent.setEnabled(enabled);
};

// Dynamically set the list of tags that can be added/removed based on the given list of items.
ZmTagMenu.prototype.set =
function(items, tagList) {
	DBG.println(AjxDebug.DBG3, "set tag menu");
	this._tagList = tagList;
	this._items = items;
	this._dirty = true;

	//commented out since in ZmMailMsgCapsuleView.prototype._resetOperations we call resetOperations of the ctrlr before this set. And I don't think this should enable the button anyway - this should be done elsewhere like it is.
	//another option would have been to reorder but I think this one is the safer one.
	//this.parent.setEnabled(true);

	// Turn on the hover delay.
	if (this.parent instanceof DwtMenuItem) {
		this.parent.setHoverDelay(ZmTagMenu._HOVER_TIME);
	}
};

ZmTagMenu.prototype._doPopup =
function(x, y, kbGenerated) {
	if (this._dirty) {
		// reset the menu
		this.removeChildren();

		if (this._tagList) {
			var rootTag = this._tagList.root;
			var addRemove = this._getAddRemove(this._items, rootTag);
			this._render(rootTag, addRemove);
		}
		this._dirty = false;

		// Remove the hover delay to prevent flicker when mousing around.
		if (this.parent instanceof DwtMenuItem) {
			this.parent.setHoverDelay(0);
		}
	}
	ZmPopupMenu.prototype._doPopup.call(this, x, y, kbGenerated);
};


// Given a list of items, produce two lists: one of tags that could be added (any tag
// that the entire list doesn't have), and one of tags that could be removed (any tag
// that any item has).
ZmTagMenu.prototype._getAddRemove = 
function(items, tagList) {
	// find out how many times each tag shows up in the items
	var tagCount = {};
	var tagRemoveHash = {};
	for (var i = 0; i < items.length; i++) {
		var item = items[i];
		if (!item.tags) {
			continue;
		}
		for (var j = 0; j < item.tags.length; j++) {
			var tagName = item.tags[j];
			tagCount[tagName] = tagCount[tagName] || 0;
			tagRemoveHash[tagName]  = true;
			//NOTE hasTag and canAddTag are not interchangeable - for Conv it's possible you can both add the tag and remove (if only some messages are tagged)
			if (!item.canAddTag(tagName)) {
				tagCount[tagName] += 1;
			}
		}
	}
	var remove = AjxUtil.keys(tagRemoveHash);

	var add = [];
	// any tag held by fewer than all the items can be added
	var a = tagList.children.getArray();
	for (i = 0; i < a.length; i++) {
		var tag = a[i];
		tagName = tag.name;
		if (!tagCount[tagName] || (tagCount[tagName] < items.length)) {
			add.push(tagName);
		}
	}

	return {add: add, remove: remove};
};

// Create the list of tags that can be added, and the submenu with the list of
// tags that can be removed.
ZmTagMenu.prototype._render =
function(tagList, addRemove) {

	for (var i = 0; i < addRemove.add.length; i++) {
		var tagName = addRemove.add[i];
		this._addNewTag(this, tagName, tagList, true, null, this._addHash);
	}

	if (addRemove.add.length) {
		new DwtMenuItem({parent:this, style:DwtMenuItem.SEPARATOR_STYLE});
	}

	// add static "New Tag" menu item
	var map = appCtxt.getCurrentController() && appCtxt.getCurrentController().getKeyMapName();
	var addid = map ? (map + "_newtag"):this._htmlElId + "|NEWTAG";
	var removeid = map ? (map + "_removetag"):this._htmlElId + "|REMOVETAG";

	var miNew = this._menuItems[ZmTagMenu.MENU_ITEM_ADD_ID] = new DwtMenuItem({parent:this, id: addid});
	miNew.setText(ZmMsg.newTag);
	miNew.setImage("NewTag");
	miNew.setShortcut(appCtxt.getShortcutHint(this._keyMap, ZmKeyMap.NEW_TAG));
	miNew.setData(ZmTagMenu.KEY_TAG_EVENT, ZmEvent.E_CREATE);
	miNew.addSelectionListener(new AjxListener(this, this._menuItemSelectionListener), 0);
	miNew.setEnabled(!appCtxt.isWebClientOffline());

	// add static "Remove Tag" menu item
	var miRemove = this._menuItems[ZmTagMenu.MENU_ITEM_REM_ID] = new DwtMenuItem({parent:this, id: removeid});
	miRemove.setEnabled(false);
	miRemove.setText(ZmMsg.removeTag);
	miRemove.setImage("DeleteTag");

	var removeList = addRemove.remove;
	if (removeList.length > 0) {
		miRemove.setEnabled(true);
		var removeMenu = null;
		if (removeList.length > 1) {
			for (i = 0; i < removeList.length; i++) {
				if (!removeMenu) {
					removeMenu = new DwtMenu({parent:miRemove, className:this._className});
					miRemove.setMenu(removeMenu);
                    removeMenu.setHtmlElementId('REMOVE_TAG_MENU_' + this.getHTMLElId());
				}
				var tagName = removeList[i];
                var tagHtmlId = 'Remove_tag_' + i;
				this._addNewTag(removeMenu, tagName, tagList, false, null, this._removeHash, tagHtmlId);
			}
			// if multiple removable tags, offer "Remove All"
			new DwtMenuItem({parent:removeMenu, style:DwtMenuItem.SEPARATOR_STYLE});
			var mi = new DwtMenuItem({parent:removeMenu, id:"REMOVE_ALL_TAGS"});
			mi.setText(ZmMsg.allTags);
			mi.setImage("TagStack");
			mi.setShortcut(appCtxt.getShortcutHint(this._keyMap, ZmKeyMap.UNTAG));
			mi.setData(ZmTagMenu.KEY_TAG_EVENT, ZmEvent.E_REMOVE_ALL);
			mi.setData(Dwt.KEY_OBJECT, removeList);
			mi.addSelectionListener(new AjxListener(this, this._menuItemSelectionListener), 0);
		}
		else {
			var tag = tagList.getByNameOrRemote(removeList[0]);
			miRemove.setData(ZmTagMenu.KEY_TAG_EVENT, ZmEvent.E_TAGS);
			miRemove.setData(ZmTagMenu.KEY_TAG_ADDED, false);
			miRemove.setData(Dwt.KEY_OBJECT, tag);
			miRemove.addSelectionListener(new AjxListener(this, this._menuItemSelectionListener), 0);
		}		

	}
};

ZmTagMenu.tagNameLength = 20;
ZmTagMenu.prototype._addNewTag =
function(menu, newTagName, tagList, add, index, tagHash, tagHtmlId) {
	var newTag = tagList.getByNameOrRemote(newTagName);
	var mi = new DwtMenuItem({parent:menu, index:index, id:tagHtmlId});
    var tagName = AjxStringUtil.clipByLength(newTag.getName(false),ZmTagMenu.tagNameLength);
	var nameText = newTag.notLocal ? AjxMessageFormat.format(ZmMsg.tagNotLocal, tagName) : tagName;
    mi.setText(nameText);
    mi.setImage(newTag.getIconWithColor());
	mi.setData(ZmTagMenu.KEY_TAG_EVENT, ZmEvent.E_TAGS);
	mi.setData(ZmTagMenu.KEY_TAG_ADDED, add);
	mi.setData(Dwt.KEY_OBJECT, newTag);
	mi.addSelectionListener(new AjxListener(this, this._menuItemSelectionListener), 0);
//	mi.setShortcut(appCtxt.getShortcutHint(null, ZmKeyMap.TAG));
	tagHash[newTag.id] = mi;
};

ZmTagMenu.prototype._menuItemSelectionListener =
function(ev) {
	// Only notify if the node is one of our nodes
	if (ev.item.getData(ZmTagMenu.KEY_TAG_EVENT)) {
		this._evtMgr.notifyListeners(DwtEvent.SELECTION, ev);
	}
};

}
if (AjxPackage.define("zimbraMail.share.view.ZmListView")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 */

/**
 * Creates a list view.
 * @class
 * A list view presents a list of items as rows with fields (columns).
 *
 * @author Parag Shah
 * @author Conrad Damon
 *
 * @param {Hash}	params		a hash of parameters
 * @param {DwtComposite}	params.parent		the parent widget
 * @param {String}	params.className		the CSS class
 * @param {constant}	params.posStyle		the positioning style
 * @param {String}	params.id			the HTML ID for element
 * @param {Array}	params.headerList	the list of IDs for columns
 * @param {Boolean}	params.noMaximize	if <code>true</code>, all columns are fixed-width (otherwise, one will expand to fill available space)
 * @param {constant}	params.view			the ID of view
 * @param {constant}	params.type			the type of item displayed
 * @param {ZmListController}	params.controller	the owning controller
 * @param {DwtDropTarget}	params.dropTgt		the drop target
 * @param {Boolean}	params.pageless		if <code>true</code>, enlarge page via scroll rather than pagination
 *        
 * @extends		DwtListView
 */
ZmListView = function(params) {

	if (arguments.length == 0) { return; }
	
	params.id = params.id || ZmId.getViewId(params.view);
	DwtListView.call(this, params);

	this.view = params.view;
	this.type = params.type;
	this._controller = params.controller;
	this.setDropTarget(params.dropTgt);

	// create listeners for changes to the list model, folder tree, and tag list
	this._listChangeListener = new AjxListener(this, this._changeListener);
	this._tagListChangeListener = new AjxListener(this, this._tagChangeListener);
	var tagList = appCtxt.getTagTree();
	if (tagList) {
		tagList.addChangeListener(this._tagListChangeListener);
	}
	var folderTree = appCtxt.getFolderTree();
	if (folderTree) {
		this._boundFolderChangeListener =  this._folderChangeListener.bind(this);
		folderTree.addChangeListener(this._boundFolderChangeListener);
	}

	this._handleEventType = {};
	this._handleEventType[this.type] = true;
	this._disallowSelection = {};
	this._disallowSelection[ZmItem.F_FLAG] = true;
	this._disallowSelection[ZmItem.F_MSG_PRIORITY] = true;
	this._selectAllEnabled = false;

	if (params.dropTgt) {
		var args = {container:this._parentEl, threshold:15, amount:5, interval:10, id:params.id};
		this._dndScrollCallback = new AjxCallback(null, DwtControl._dndScrollCallback, [args]);
		this._dndScrollId = params.id;
	}

	this._isPageless = params.pageless;
	if (this._isPageless) {
		Dwt.setHandler(this._getScrollDiv(), DwtEvent.ONSCROLL, ZmListView.handleScroll);
	}
	this._state = {};
};

ZmListView.prototype = new DwtListView;
ZmListView.prototype.constructor = ZmListView;
ZmListView.prototype.isZmListView = true;

ZmListView.prototype.toString =
function() {
	return "ZmListView";
};


// Consts

ZmListView.KEY_ID							= "_keyId";

// column widths
ZmListView.COL_WIDTH_ICON 					= 19;
ZmListView.COL_WIDTH_NARROW_ICON			= 11;

// TD class for fields
ZmListView.FIELD_CLASS = {};
ZmListView.FIELD_CLASS[ZmItem.F_TYPE]		= "ListViewIcon";
ZmListView.FIELD_CLASS[ZmItem.F_FLAG]		= "Flag";
ZmListView.FIELD_CLASS[ZmItem.F_TAG]		= "Tag";
ZmListView.FIELD_CLASS[ZmItem.F_ATTACHMENT]	= "Attach";

ZmListView.ITEM_FLAG_CLICKED 				= DwtListView._LAST_REASON + 1;
ZmListView.DEFAULT_REPLENISH_THRESHOLD		= 0;

ZmListView.COL_JOIN = "|";

ZmListView.CHECKED_IMAGE = "CheckboxChecked";
ZmListView.UNCHECKED_IMAGE = "CheckboxUnchecked";
ZmListView.CHECKED_CLASS = "ImgCheckboxChecked";
ZmListView.UNCHECKED_CLASS = "ImgCheckboxUnchecked";
ZmListView.ITEM_CHECKED_ATT_NAME = "itemChecked";


ZmListView.prototype._getHeaderList = function() {};

/**
 * Gets the controller.
 * 
 * @return	{ZmListController}		the list controller
 */
ZmListView.prototype.getController =
function() {
	return this._controller;
};

ZmListView.prototype.set =
function(list, sortField) {

	this._sortByString = this._controller._currentSearch && this._controller._currentSearch.sortBy;
    //TODO: We need a longer term fix but this is to prevent a sort by that doesn't match our ZmSearch
	//constants and lead to notification issues.
	if (this._sortByString) {
    	this._sortByString = this._sortByString.replace("asc", "Asc").replace("desc", "Desc");// bug 75687
	}

	var settings = appCtxt.getSettings();
	if (!appCtxt.isExternalAccount() && this.view) {
		appCtxt.set(ZmSetting.SORTING_PREF,
					this._sortByString,
					this.view,
					false, //setDefault
					false, //skipNotify
					null, //account
					settings && !settings.persistImplicitSortPrefs(this.view)); //skipImplicit - do not persist
	}

	this.setSelectionHdrCbox(false);

	// bug fix #28595 - in multi-account, reset tag list change listeners
	if (appCtxt.multiAccounts) {
		var tagList = appCtxt.getTagTree();
		if (tagList) {
			tagList.addChangeListener(this._tagListChangeListener);
		}
	}

	if (this._isPageless) {
		if (this._itemsToAdd) {
			if (this._itemsToAdd.length) {
				this.addItems(this._itemsToAdd);
				this._itemsToAdd = null;
			}
		} else {
			var lvList = list;
			if (list && list.isZmList) {
				list.addChangeListener(this._listChangeListener);
				lvList = list.getSubList(0, list.size());
			}
			DwtListView.prototype.set.call(this, lvList, sortField);
		}
		this._setRowHeight();
	} else {
		var subList;
		if (list && list.isZmList) {
			list.addChangeListener(this._listChangeListener);
			subList = list.getSubList(this.offset, this.getLimit());
		} else {
			subList = list;
		}
		DwtListView.prototype.set.call(this, subList, sortField);
	}
	this._rendered = true;

	// check in case there are more items but no scrollbar
	if (this._isPageless) {
		AjxTimedAction.scheduleAction(new AjxTimedAction(this, this._checkItemCount), 1000);
	}
};

ZmListView.prototype.reset =
function() {
	this._rendered = false;
};

ZmListView.prototype.setUI =
function(defaultColumnSort) {
	DwtListView.prototype.setUI.call(this, defaultColumnSort);
	this._resetColWidth();	// reset column width in case scrollbar is set
};

/**
 * Gets the limit value.
 * 
 * @param	{Boolean}	offset		if <code>true</code>, offset
 * @return	{int}	the limit page size
 */
ZmListView.prototype.getLimit =
function(offset) {
	if (this._isPageless) {
		var limit = appCtxt.get(ZmSetting.PAGE_SIZE);
		return offset ? limit : 2 * limit;
	} else {
		return appCtxt.get(ZmSetting.PAGE_SIZE);
	}
};

/**
 * Gets the pageless threshold.
 * 
 * @return	{int}		the pageless threshold
 */
ZmListView.prototype.getPagelessThreshold =
function() {
	return Math.ceil(this.getLimit() / 5);
};

/**
 * Gets the replenish threshold.
 * 
 * @return	{int}	the replenish threshold
 */
ZmListView.prototype.getReplenishThreshold =
function() {
	return ZmListView.DEFAULT_REPLENISH_THRESHOLD;
};

/**
 * Returns the underlying ZmList.
 */
ZmListView.prototype.getItemList =
function() {
	return this._controller && this._controller._list;
};

ZmListView.prototype._changeListener =
function(ev) {

	var item = this._getItemFromEvent(ev);
	if (!item || ev.handled || !this._handleEventType[item.type]) {
		return;
	}

	if (ev.event === ZmEvent.E_TAGS || ev.event === ZmEvent.E_REMOVE_ALL) {
		this._replaceTagImage(item, ZmItem.F_TAG, this._getClasses(ZmItem.F_TAG));
	}

	if (ev.event === ZmEvent.E_FLAGS) {
		var flags = ev.getDetail("flags");
		for (var j = 0; j < flags.length; j++) {
			var flag = flags[j];
			var on = item[ZmItem.FLAG_PROP[flag]];
			if (flag === ZmItem.FLAG_FLAGGED) {
				this._setImage(item, ZmItem.F_FLAG, on ? "FlagRed" : "FlagDis", this._getClasses(ZmItem.F_FLAG));
			}
			else if (flag === ZmItem.FLAG_ATTACH) {
				this._setImage(item, ZmItem.F_ATTACHMENT, on ? "Attachment" : null, this._getClasses(ZmItem.F_ATTACHMENT));
			}
			else if (flag === ZmItem.FLAG_PRIORITY) {
				this._setImage(item, ZmItem.F_MSG_PRIORITY, on ? "Priority" : "PriorityDis", this._getClasses(ZmItem.F_MSG_PRIORITY));
			}
		}
	}

	// Note: move and delete support batch notification mode
	if (ev.event === ZmEvent.E_DELETE || ev.event === ZmEvent.E_MOVE) {
		var items = ev.batchMode ? this._getItemsFromBatchEvent(ev) : [item];
		var needsSort = false;
		for (var i = 0, len = items.length; i < len; i++) {
			var item = items[i];
            var movedHere = (item.type === ZmId.ITEM_CONV) ? item.folders[this._folderId] : item.folderId === this._folderId;
			if (movedHere && ev.event === ZmEvent.E_MOVE) {
				// We've moved the item into this folder
				if (this._getRowIndex(item) === null) { // Not already here
					this.addItem(item);
					// TODO: couldn't we just find the sort index and insert it?
					needsSort = true;
				}
			}
			else {
				// remove the item if the user is working in this view,
				// if we know the item no longer matches the search, or if the item was hard-deleted
				if (ev.event === ZmEvent.E_DELETE || this.view == appCtxt.getCurrentViewId() || this._controller._currentSearch.matches(item) === false) {
					this.removeItem(item, true, ev.batchMode);
					// if we've removed it from the view, we should remove it from the reference
					// list as well so it doesn't get resurrected via replenishment *unless*
					// we're dealing with a canonical list (i.e. contacts)
					var itemList = this.getItemList();
					if (ev.event !== ZmEvent.E_MOVE || !itemList.isCanonical) {
						itemList.remove(item);
					}
				}
			}
		}
		if (needsSort) {
			this._saveState({scroll: true, selection:true, focus: true});
			this._redoSearch(this._restoreState.bind(this, this._state));
		}
		if (ev.batchMode) {
			this._fixAlternation(0);
		}
		this._checkReplenishOnTimer();
		this._controller._resetToolbarOperations();
	}

	this._updateLabelForItem(item);
};

ZmListView.prototype._getItemFromEvent =
function(ev) {
	var item = ev.item;
	if (!item) {
		var items = ev.getDetail("items");
		item = (items && items.length) ? items[0] : null;
	}
	return item;
};

ZmListView.prototype._getItemsFromBatchEvent =
function(ev) {

	if (!ev.batchMode) { return []; }

	var items = ev.items;
	if (!items) {
		items = [];
		var notifs = ev.getDetail("notifs");
		if (notifs && notifs.length) {
			for (var i = 0, len = notifs.length; i < len; i++) {
				var mod = notifs[i];
				items.push(mod.item || appCtxt.cacheGet(mod.id));
			}
		}
	}

	return items;
};

// refreshes the content of the given field for the given item
ZmListView.prototype._updateField =
function(item, field) {
	var fieldId = this._getFieldId(item, field);
	var el = document.getElementById(fieldId);
	if (el) {
		var html = [];
		var colIdx = this._headerHash[field] && this._headerHash[field]._index;
		this._getCellContents(html, 0, item, field, colIdx, new Date());
		//replace the old inner html with the new updated data
		el.innerHTML = $(html.join("")).html();
	}

	this._updateLabelForItem(item);
};

ZmListView.prototype._checkReplenishOnTimer =
function(ev) {
	if (!this.allSelected) {
		if (!this._isPageless) {
			this._controller._app._checkReplenishListView = this;
		} else {
			// Many rows may be removed quickly, so skip unnecessary replenishes
			if (!this._replenishTimedAction) {
				this._replenishTimedAction = new AjxTimedAction(this, this._handleResponseCheckReplenish);
			}
			AjxTimedAction.scheduleAction(this._replenishTimedAction, 10);
		}
	}
};

ZmListView.prototype._checkReplenish =
function(item, forceSelection) {
	var respCallback = new AjxCallback(this, this._handleResponseCheckReplenish, [false, item, forceSelection]);
	this._controller._checkReplenish(respCallback);
};

ZmListView.prototype._handleResponseCheckReplenish =
function(skipSelection, item, forceSelection) {
	if (this.size() == 0) {
		this._controller._handleEmptyList(this);
	} else {
		this._controller._resetNavToolBarButtons();
	}
	if (!skipSelection) {
		this._setNextSelection(item, forceSelection);
	}
};

ZmListView.prototype._folderChangeListener =
function(ev) {
	// make sure this is current list view
	if (appCtxt.getCurrentController() != this._controller) { return; }
	// see if it will be handled by app's postNotify()
	if (this._controller._app._checkReplenishListView == this) { return; }

	var organizers = ev.getDetail("organizers");
	var organizer = (organizers && organizers.length) ? organizers[0] : ev.source;

	var id = organizer.id;
	var fields = ev.getDetail("fields");
	if (ev.event == ZmEvent.E_MODIFY) {
		if (!fields) { return; }
		if (fields[ZmOrganizer.F_TOTAL]) {
			this._controller._resetNavToolBarButtons();
		}
	}
};

ZmListView.prototype._tagChangeListener =
function(ev) {
	if (ev.type != ZmEvent.S_TAG) return;

	var fields = ev.getDetail("fields");

	var divs = this._getChildren();
	var tag = ev.getDetail("organizers")[0];
	for (var i = 0; i < divs.length; i++) {
		var item = this.getItemFromElement(divs[i]);
		if (!item || !item.tags || !item.hasTag(tag.name)) {
			continue;
		}
		var updateRequired = false;
		if (ev.event == ZmEvent.E_MODIFY && (fields && (fields[ZmOrganizer.F_COLOR] || fields[ZmOrganizer.F_NAME]))) {
			//rename could change the color (for remote shared items, from the remote gray icon to local color and vice versa)
			updateRequired = item.tags.length == 1;
		}
		else if (ev.event == ZmEvent.E_DELETE) {
			updateRequired = true;
		}
		else if (ev.event == ZmEvent.E_CREATE) {
			//this could affect item if it had a tag not on tag list (remotely created on shared item, either shared by this user or shared to this user)
			updateRequired = true;
		}
		if (updateRequired) {
			this._replaceTagImage(item, ZmItem.F_TAG, this._getClasses(ZmItem.F_TAG));
		}
	}
};

// returns all child divs for this list view
ZmListView.prototype._getChildren =
function() {
	return this._parentEl.childNodes;
};

// Common routines for createItemHtml()

ZmListView.prototype._getRowId =
function(item) {
	return DwtId.getListViewItemId(DwtId.WIDGET_ITEM_FIELD, this._view, item ? item.id : Dwt.getNextId(), ZmItem.F_ITEM_ROW);
};

// Note that images typically get IDs in _getCellContents().
ZmListView.prototype._getCellId =
function(item, field) {
	if (field == ZmItem.F_DATE) {
		return this._getFieldId(item, field);
	} else if (field == ZmItem.F_SELECTION) {
		return this._getFieldId(item, ZmItem.F_SELECTION_CELL);

	} else {
		return DwtListView.prototype._getCellId.apply(this, arguments);
	}
};

ZmListView.prototype._getCellClass =
function(item, field, params) {
	return ZmListView.FIELD_CLASS[field];
};

ZmListView.prototype._getCellContents =
function(htmlArr, idx, item, field, colIdx, params, classes) {
	if (field == ZmItem.F_SELECTION) {
		idx = this._getImageHtml(htmlArr, idx, "CheckboxUnchecked", this._getFieldId(item, field), classes);
	} else if (field == ZmItem.F_TYPE) {
		idx = this._getImageHtml(htmlArr, idx, ZmItem.ICON[item.type], this._getFieldId(item, field), classes);
	} else if (field == ZmItem.F_FLAG) {
		idx = this._getImageHtml(htmlArr, idx, this._getFlagIcon(item.isFlagged), this._getFieldId(item, field), classes);
	} else if (field == ZmItem.F_TAG) {
		idx = this._getImageHtml(htmlArr, idx, item.getTagImageInfo(), this._getFieldId(item, field), classes);
	} else if (field == ZmItem.F_ATTACHMENT) {
		idx = this._getImageHtml(htmlArr, idx, item.hasAttach ? "Attachment" : null, this._getFieldId(item, field), classes);
	} else if (field == ZmItem.F_DATE) {
		htmlArr[idx++] = AjxDateUtil.computeDateStr(params.now || new Date(), item.date);
	} else if (field == ZmItem.F_PRIORITY) {
        var priorityImage = null;
        if (item.isHighPriority) {
            priorityImage = "PriorityHigh_list";
        } else if (item.isLowPriority) {
			priorityImage = "PriorityLow_list";
		}
		if (priorityImage) {
        	idx = this._getImageHtml(htmlArr, idx, priorityImage, this._getFieldId(item, field), classes);
		} else {
			htmlArr[idx++] = "<div id='" + this._getFieldId(item, field) + "' " + AjxUtil.getClassAttr(classes) + "></div>";
		}
	} else {
		idx = DwtListView.prototype._getCellContents.apply(this, arguments);
	}
	return idx;
};

ZmListView.prototype._getImageHtml =
function(htmlArr, idx, imageInfo, id, classes) {
	htmlArr[idx++] = "<div";
	if (id) {
		htmlArr[idx++] = [" id='", id, "' "].join("");
	}
	htmlArr[idx++] = AjxUtil.getClassAttr(classes);
	htmlArr[idx++] = ">";
	htmlArr[idx++] = AjxImg.getImageHtml(imageInfo || "Blank_16");
	htmlArr[idx++] = "</div>";
	return idx;
};

ZmListView.prototype._getClasses =
function(field, classes) {
	if (this.isMultiColumn && this.isMultiColumn() && this._headerHash[field]) {
		classes = classes || [];
		classes = [this._headerHash[field]._cssClass];
	}
	return classes;
};

ZmListView.prototype._setImage =
function(item, field, imageInfo, classes) {
	var cell = this._getElement(item, field);
	if (cell) {
		if (classes) {
			cell.className = AjxUtil.uniq(classes).join(" ");
		}
		cell.innerHTML = AjxImg.getImageHtml(imageInfo || "Blank_16");
	}
};

ZmListView.prototype._replaceTagImage =
function(item, field, classes) {
	this._setImage(item, field, item.getTagImageInfo(), classes);
};

ZmListView.prototype._getFragmentSpan =
function(item) {
	return ["<span class='ZmConvListFragment' aria-hidden='true' id='",
			this._getFieldId(item, ZmItem.F_FRAGMENT),
			"'>", this._getFragmentHtml(item), "</span>"].join("");
};

ZmListView.prototype._getFragmentHtml =
function(item) {
	return [" - ", AjxStringUtil.htmlEncode(item.fragment, true)].join("");
};

ZmListView.prototype._getFlagIcon =
function(isFlagged, isMouseover, disabled) {
	if (!isFlagged && !isMouseover) {
		return "Blank_16";
	} else if (disabled) {
		return "FlagDis";
	} else {
		return "FlagRed";
	}
};

/**
 * Parse the DOM ID to figure out what got clicked. IDs consist of three to five parts
 * joined by the "|" character.
 *
 *		type		type of ID (zli, zlir, zlic, zlif) - see DwtId.WIDGET_ITEM*)
 * 		view		view identifier (eg "TV")
 * 		item ID		usually numeric
 * 		field		field identifier (eg "fg") - see ZmId.FLG_*
 * 		participant	index of participant
 */
ZmListView.prototype._parseId =
function(id) {
	var parts = id.split(DwtId.SEP);
	if (parts && parts.length) {
		return {view:parts[1], item:parts[2], field:parts[3], participant:parts[4]};
	} else {
		return null;
	}
};

ZmListView.prototype._mouseDownAction =
function(ev, div) {
	return !Dwt.ffScrollbarCheck(ev);
};

ZmListView.prototype._mouseUpAction =
function(ev, div) {
	return !Dwt.ffScrollbarCheck(ev);
};

ZmListView.prototype._getField =
function(ev, div) {

	var target = this._getEventTarget(ev);

	var id = target && target.id || div.id;
	if (!id) {
		return null;
	}

	var data = this._data[div.id];
	var type = data.type;
	if (!type || type != DwtListView.TYPE_LIST_ITEM) {
		return null;
	}

	var m = this._parseId(id);
	if (!m || !m.field) {
		return null;
	}
	return m.field;

};


ZmListView.prototype._mouseOutAction =
function(ev, div) {
	DwtListView.prototype._mouseOutAction.call(this, ev, div);

	var field = this._getField(ev, div);
	if (!field) {
		return true;
	}

	if (field == ZmItem.F_FLAG) {
		var item = this.getItemFromElement(div);
		if (!item.isFlagged) {
			var target = this._getEventTarget(ev);
			AjxImg.setImage(target, this._getFlagIcon(item.isFlagged, false), false, false);
			target.className = this._getClasses(field);
		}
	}
	return true;
};


ZmListView.prototype._mouseOverAction =
function(ev, div) {
	DwtListView.prototype._mouseOverAction.call(this, ev, div);

	var field = this._getField(ev, div);
	if (!field) {
		return true;
	}

	if (field === ZmItem.F_FLAG) {
		var item = this.getItemFromElement(div);
		if (!item.isReadOnly() && !item.isFlagged) {
			var target = this._getEventTarget(ev);
			AjxImg.setDisabledImage(target, this._getFlagIcon(item.isFlagged, true), false);
			target.className = this._getClasses(field);
		}
	}
	return true;
};



ZmListView.prototype._doubleClickAction =
function(ev, div) {
	var target = this._getEventTarget(ev);
	var id = target && target.id || div.id;
	if (!id) { return true; }

	var m = this._parseId(id);
	return (!(m && (m.field == ZmItem.F_FLAG)));
};

ZmListView.prototype._itemClicked =
function(clickedEl, ev) {
	if (appCtxt.get(ZmSetting.SHOW_SELECTION_CHECKBOX) && ev.button == DwtMouseEvent.LEFT) {
		if (!ev.shiftKey && !ev.ctrlKey) {
			// get the field being clicked
			var target = this._getEventTarget(ev);
			var id = (target && target.id && target.id.indexOf("AjxImg") == -1) ? target.id : clickedEl.id;
			var m = id ? this._parseId(id) : null;
			if (m && (m.field == ZmItem.F_SELECTION || m.field == ZmItem.F_SELECTION_CELL)) {
				//user clicked on a checkbox
				if (this._selectedItems.size() == 1) {
					var sel = this._selectedItems.get(0);
					var item = this.getItemFromElement(sel);
					var selFieldId = item ? this._getFieldId(item, ZmItem.F_SELECTION) : null;
					var selField = selFieldId ? document.getElementById(selFieldId) : null;
					if (selField && sel == clickedEl) {
						var isChecked = this._getItemData(sel, ZmListView.ITEM_CHECKED_ATT_NAME);
						this._setImage(item, ZmItem.F_SELECTION, isChecked ? ZmListView.UNCHECKED_IMAGE : ZmListView.CHECKED_IMAGE);
						this._setItemData(sel, ZmListView.ITEM_CHECKED_ATT_NAME, !isChecked);
						if (!isChecked) {
							return; //nothing else to do. It's already selected, and was the only selected one. Nothing to remove
						}
					} else {
						if (selField && !this._getItemData(sel, ZmListView.ITEM_CHECKED_ATT_NAME)) {
							this.deselectAll();
							this._markUnselectedViewedItem(true);
						}
					}
				}
				var bContained = this._selectedItems.contains(clickedEl);
				this.setMultiSelection(clickedEl, bContained);
				this._controller._setItemSelectionCountText();
				return;	// do not call base class if "selection" field was clicked
			}
		} else if (ev.shiftKey) {
			// uncheck all selected items first
			this._checkSelectedItems(false);

			// run base class first so we get the finalized list of selected items
			DwtListView.prototype._itemClicked.call(this, clickedEl, ev);

			// recheck new list of selected items
			this._checkSelectedItems(true);

			return;
		}
	}

	DwtListView.prototype._itemClicked.call(this, clickedEl, ev);
};

ZmListView.prototype._columnClicked =
function(clickedCol, ev) {
	DwtListView.prototype._columnClicked.call(this, clickedCol, ev);
	this._checkSelectionColumnClicked(clickedCol, ev);
};

ZmListView.prototype._checkSelectionColumnClicked =
function(clickedCol, ev) {

	if (!appCtxt.get(ZmSetting.SHOW_SELECTION_CHECKBOX)) { return; }

	var list = this.getList();
	var size = list ? list.size() : null;
	if (size > 0) {
		var idx = this._data[clickedCol.id].index;
		var item = this._headerList[idx];
		if (item && (item._field == ZmItem.F_SELECTION)) {
			var hdrId = DwtId.getListViewHdrId(DwtId.WIDGET_HDR_ICON, this._view, item._field);
			var hdrDiv = document.getElementById(hdrId);
			if (hdrDiv) {
				if (hdrDiv.className == ZmListView.CHECKED_CLASS) {
					if (ev.shiftKey && !this.allSelected) {
						this.selectAll(ev.shiftKey);
					} else {
						this.deselectAll();
						hdrDiv.className = ZmListView.UNCHECKED_CLASS;
					}
				} else {
					this.allSelected = false;
					hdrDiv.className = ZmListView.CHECKED_CLASS;
					this.selectAll(ev.shiftKey);
				}
			}
		}
		this._controller._resetToolbarOperations();
	}
};

ZmListView.prototype.handleKeyAction =
function(actionCode, ev) {
	var rv = DwtListView.prototype.handleKeyAction.call(this, actionCode, ev);

	if (actionCode == DwtKeyMap.SELECT_ALL) {
		this._controller._resetToolbarOperations();
	}

	return rv;
};

ZmListView.prototype.setMultiSelection =
function(clickedEl, bContained, ev) {
	if (ev && ev.ctrlKey && this._selectedItems.size() == 1) {
		this._checkSelectedItems(true);
	}

	// call base class
	DwtListView.prototype.setMultiSelection.call(this, clickedEl, bContained);

	this.setSelectionCbox(clickedEl, bContained);
	this.setSelectionHdrCbox(this._isAllChecked());

	// reset toolbar operations LAST
	this._controller._resetToolbarOperations();
};

/**
 * check whether all items in the list are checked
 * @return {Boolean} true if all items are checked
 */
ZmListView.prototype._isAllChecked = 
function() {
	var list = this.getList();
	return (list && (this.getSelection().length == list.size()));
};


/**
 * Sets the selection checkbox.
 * 
 * @param	{Element}	obj		the item element object
 * @param	{Boolean}	bContained		(not used)
 * 
 */
ZmListView.prototype.setSelectionCbox =
function(obj, bContained) {
	if (!obj) { return; }

	var item = obj.tagName ? this.getItemFromElement(obj) : obj;
	var selFieldId = item ? this._getFieldId(item, ZmItem.F_SELECTION) : null;
	var selField = selFieldId ? document.getElementById(selFieldId) : null;
	if (selField) {
		this._setImage(item, ZmItem.F_SELECTION, bContained ? ZmListView.UNCHECKED_IMAGE : ZmListView.CHECKED_IMAGE);
		this._setItemData(this._getElFromItem(item), ZmListView.ITEM_CHECKED_ATT_NAME, !bContained);
		this._updateLabelForItem(item);
	}
};

/**
 * Sets the selection header checkbox.
 * 
 * @param	{Boolean}	check		if <code>true</code>, check the header checkbox
 */
ZmListView.prototype.setSelectionHdrCbox =
function(check) {
	var col = this._headerHash ? this._headerHash[ZmItem.F_SELECTION] : null;
	var hdrId = col ? DwtId.getListViewHdrId(DwtId.WIDGET_HDR_ICON, this._view, col._field) : null;
	var hdrDiv = hdrId ? document.getElementById(hdrId) : null;
	if (hdrDiv) {
		hdrDiv.className = check
			? ZmListView.CHECKED_CLASS
			: ZmListView.UNCHECKED_CLASS;
	}
};

/**
 * Sets the selected items.
 * 
 * @param	{Array}	selectedArray		an array of {Element} objects to select
 * @param	{boolean}	dontCheck		do not check the selected item. (special case. see ZmListView.prototype._restoreState)
 */
ZmListView.prototype.setSelectedItems =
function(selectedArray, dontCheck) {
	DwtListView.prototype.setSelectedItems.call(this, selectedArray);

	if (!dontCheck && appCtxt.get(ZmSetting.SHOW_SELECTION_CHECKBOX)) {
		this._checkSelectedItems(true, true);
	}
};

/**
 * Selects all items.
 * 
 * @param	{Boolean}	allResults		if <code>true</code>, set all search selected
 */
ZmListView.prototype.selectAll =
function(allResults) {

	DwtListView.prototype.selectAll.apply(this, arguments);

	if (this._selectAllEnabled) {
		var curResult = this._controller._activeSearch;
		if (curResult && curResult.getAttribute("more")) {

			var list = this.getList(),
				type = this.type,
				countKey = 'type' + AjxStringUtil.capitalize(ZmItem.MSG_KEY[type]),
				typeText = AjxMessageFormat.format(ZmMsg[countKey], list ? list.size() : 2),
				shortcut = appCtxt.getShortcutHint(null, ZmKeyMap.SELECT_ALL),
				args = [list ? list.size() : ZmMsg.all, typeText, shortcut, "ZmListView.selectAllResults()"],
				toastMsg = AjxMessageFormat.format(ZmMsg.allPageSelected, args);

			if (allResults) {
				this.allSelected = true;
				toastMsg = ZmMsg.allSearchSelected;
			}
			appCtxt.setStatusMsg(toastMsg);
		}

		var sel = this._selectedItems.getArray();
		for (var i = 0; i < sel.length; i++) {
			this.setSelectionCbox(sel[i], false);
		}
	}
};

// Handle click of link in toast
ZmListView.selectAllResults =
function() {
	var ctlr = appCtxt.getCurrentController();
	var view = ctlr && ctlr.getListView();
	if (view && view.selectAll) {
		view.selectAll(true);
	}
};

/**
 * Deselects all items.
 * 
 */
ZmListView.prototype.deselectAll =
function() {

	this.allSelected = false;
	if (appCtxt.get(ZmSetting.SHOW_SELECTION_CHECKBOX)) {
		this._checkSelectedItems(false);
		var hdrId = DwtId.getListViewHdrId(DwtId.WIDGET_HDR_ICON, this._view, ZmItem.F_SELECTION);
		var hdrDiv = document.getElementById(hdrId);
		if (hdrDiv) {
			hdrDiv.className = ZmListView.UNCHECKED_CLASS;
		}
		var sel = this._selectedItems.getArray();
		for (var i=0; i<sel.length; i++) {
			this.setSelectionCbox(sel[i], true);
		}
	}

	DwtListView.prototype.deselectAll.call(this);
};

ZmListView.prototype._checkSelectedItems =
function(check) {
	var sel = this.getSelection();
	for (var i = 0; i < sel.length; i++) {
		this.setSelectionCbox(sel[i], !check);
	}

	var list = this.getList();
	var size = list && list.size();
	this.setSelectionHdrCbox(size && sel.length == size);
};

ZmListView.prototype._setNoResultsHtml =
function() {
	DwtListView.prototype._setNoResultsHtml.call(this);
	this.setSelectionHdrCbox(false);
	this._rendered = true;
};

/**
 * override to call _resetToolbarOperations since we change the selection.
 * @private
 */
ZmListView.prototype._clearRightSel =
function() {
	DwtListView.prototype._clearRightSel.call(this);
	this._controller._resetToolbarOperations();
};


/*
 get sort menu for views that provide a right-click sort by menu in single-column view (currently mail and briefcase)
 */
ZmListView.prototype._getSortMenu = function (sortFields, defaultSortField, parent) {

	// create an action menu for the header list
	var menu = new ZmPopupMenu(parent || this, null, Dwt.getNextId("SORT_MENU_"));
	var actionListener = this._sortMenuListener.bind(this);

	for (var i = 0; i < sortFields.length; i++) {
		var column = sortFields[i];
		var fieldName = ZmMsg[column.msg];
		var mi = menu.createMenuItem(column.field, {
			text:   parent && parent.isDwtMenuItem ? fieldName : AjxMessageFormat.format(ZmMsg.arrangeBy, fieldName),
			style:  DwtMenuItem.RADIO_STYLE
		});
		if (column.field == defaultSortField) {
			mi.setChecked(true, true);
		}
		mi.setData(ZmListView.KEY_ID, column.field);
		menu.addSelectionListener(column.field, actionListener);
	}

	return menu;
};

/*
listener used by views that provide a right-click sort by menu in single-column view (currently mail and briefcase)
 */
ZmListView.prototype._sortMenuListener =
function(ev) {
	var column;
	if (this.isMultiColumn()) { //this can happen when called from the view menu, that now, for accessibility reasons, includes the sort, for both reading pane on right and at the bottom.
		var sortField = ev && ev.item && ev.item.getData(ZmOperation.MENUITEM_ID);
		column = this._headerHash[sortField];
	}
	else {
		column = this._headerHash[ZmItem.F_SORTED_BY];
		var cell = document.getElementById(DwtId.getListViewHdrId(DwtId.WIDGET_HDR_LABEL, this._view, column._field));
		if (cell) {
	        var text = ev.item.getText();
	        cell.innerHTML = text && text.replace(ZmMsg.sortBy, ZmMsg.sortedBy);
		}
		column._sortable = ev.item.getData(ZmListView.KEY_ID);
	}
	this._bSortAsc = (column._sortable === this._currentSortColId) ? !this._bSortAsc : this._isDefaultSortAscending(column);
	this._sortColumn(column, this._bSortAsc);
};

ZmListView.prototype._getActionMenuForColHeader = function(force, parent, context) {

	var menu;
	if (!this._colHeaderActionMenu || force) {
		// create an action menu for the header list
		menu = new ZmPopupMenu(parent || this);
		var actionListener = this._colHeaderActionListener.bind(this);
		for (var i = 0; i < this._headerList.length; i++) {
			var hCol = this._headerList[i];
			// lets not allow columns w/ relative width to be removed (for now) - it messes stuff up
			if (hCol._width) {
				var id = ZmId.getMenuItemId([ this._view, context ].join("_"), hCol._field);
				var mi = menu.createMenuItem(id, {text:hCol._name, style:DwtMenuItem.CHECK_STYLE});
				mi.setData(ZmListView.KEY_ID, hCol._id);
				mi.setChecked(hCol._visible, true);
                if (hCol._noRemove) {
					mi.setEnabled(false);
				}
				menu.addSelectionListener(id, actionListener);
			}
		}
	}

	return menu;
};

ZmListView.prototype._colHeaderActionListener =
function(ev) {

	var menuItemId = ev.item.getData(ZmListView.KEY_ID);

	for (var i = 0; i < this._headerList.length; i++) {
		var col = this._headerList[i];
		if (col._id == menuItemId) {
			col._visible = !col._visible;
			break;
		}
	}

	this._relayout();

	if (this._headerFocusElementIndex) {
		var headerElement = document.getElementById(this._headerItemsId[this._headerFocusElementIndex]);
		this._compositeTabGroup.replaceMember(this._headerFocusElement, headerElement);
		this._compositeTabGroup.setFocusMember(headerElement);
		this._headerFocusElement = headerElement;
	}
};

ZmListView.prototype.addHeaderItemInTagGroup =
function () {
	Dwt.setHandler(this._listColDiv, DwtEvent.ONKEYDOWN, ZmListView._headerKeyDown.bind(this));

	var headerFirstItem = document.getElementById(this._headerItemsId[0]);
	if (headerFirstItem) {
		headerFirstItem.setAttribute('tabindex', 0);
		if (this._headerFocusElement) {
			this._compositeTabGroup.replaceMember(this._headerFocusElement, headerFirstItem);
		} else {
			this._compositeTabGroup.removeAllMembers();
			this._compositeTabGroup.addMember(headerFirstItem);
		}
		this._headerFocusElement = headerFirstItem;
		this._headerFocusElementIndex = 0;
	}
};

ZmListView._headerKeyDown =
function (ev) {
	var keyCode = DwtKeyEvent.getCharCode(ev);
	if(keyCode === DwtKeyEvent.KEY_ARROW_RIGHT || keyCode === DwtKeyEvent.KEY_ARROW_LEFT) {
		var nextElementIndex = keyCode === DwtKeyEvent.KEY_ARROW_RIGHT ? this._headerFocusElementIndex + 1 : this._headerFocusElementIndex - 1;
		var nextElement = ZmListView._nextFocusableHeaderElement.call(this, keyCode, nextElementIndex);

		if (nextElement) {
			nextElement.setAttribute('tabindex', 0);
			this._compositeTabGroup.replaceMember(this._headerFocusElement, nextElement);
			this._compositeTabGroup.setFocusMember(nextElement);
			this._headerFocusElement = nextElement;
		}
		ev.stopPropagation();
	}

	if (keyCode === DwtKeyEvent.KEY_SPACE) {
		var currentHeaderItem = this._headerList[this._headerFocusElementIndex];
		clickEl = document.getElementById(currentHeaderItem._id);
		this._columnClicked(clickEl,ev);
		ev.stopPropagation();
	}

	if (keyCode === DwtKeyEvent.KEY_ENTER && ev.ctrlKey) {
		var actionMenu = this._getActionMenuForColHeader(true);
		if (actionMenu && actionMenu instanceof DwtMenu) {
			var rect = ev.target.getBoundingClientRect();
			actionMenu.popup(0, rect.x, rect.y);
			ev.stopPropagation();
		}
	}
};

ZmListView._nextFocusableHeaderElement =
function (keyCode, nextIndex) {
	var nextElement = null;
	var startIndex = nextIndex;

	if (keyCode === DwtKeyEvent.KEY_ARROW_LEFT) {
		this._headerItemsId.reverse();
		startIndex = this._headerItemsId.length - 1 - nextIndex;
	}

	for (var i = startIndex; i < this._headerItemsId.length; i++) {
		var element = document.getElementById(this._headerItemsId[i]);
		if (element) {
			this._headerFocusElementIndex = keyCode === DwtKeyEvent.KEY_ARROW_RIGHT ? i : (this._headerItemsId.length - 1 - i);
			nextElement = element;
			break;
		}
	}

	if (keyCode === DwtKeyEvent.KEY_ARROW_LEFT) {
		this._headerItemsId.reverse();
	}
	return nextElement;
};

/**
 * Gets the tool tip content.
 * 
 * @param	{Object}	ev		the hover event
 * @return	{String}	the tool tip content
 */
ZmListView.prototype.getToolTipContent = function(ev) {

	var div = this.getTargetItemDiv(ev);
	if (!div) {
        return "";
    }
	var target = Dwt.findAncestor(this._getEventTarget(ev), "id"),
	    id = (target && target.id) || div.id;

	if (!id) {
        return "";
    }

	// check if we're hovering over a column header
	var data = this._data[div.id];
	var type = data.type;
	var tooltip;
	if (type && type == DwtListView.TYPE_HEADER_ITEM) {
		var itemIdx = data.index;
		var field = this._headerList[itemIdx]._field;
		tooltip = this._getHeaderToolTip(field, itemIdx);
	}
    else {
		var match = this._parseId(id);
		if (match && match.field) {
			var item = this.getItemFromElement(div);
			var params = {field:match.field, item:item, ev:ev, div:div, match:match};
			tooltip = this._getToolTip(params);
		}
	}

	return tooltip;
};

ZmListView.prototype.getTooltipBase =
function(hoverEv) {
	return hoverEv ? DwtUiEvent.getTargetWithProp(hoverEv.object, "id") : DwtListView.prototype.getTooltipBase.apply(this, arguments);
};

ZmListView.prototype._getHeaderToolTip =
function(field, itemIdx, isOutboundFolder) {

	var tooltip = null;
	var sortable = this._headerList[itemIdx]._sortable;
	if (field == ZmItem.F_SELECTION) {
		tooltip = ZmMsg.selectionColumn;
	} else if (field == ZmItem.F_FLAG) {
        tooltip = ZmMsg.flagHeaderToolTip;
    } else if (field == ZmItem.F_PRIORITY){
        tooltip = ZmMsg.priorityHeaderTooltip;
    } else if (field == ZmItem.F_TAG) {
        tooltip = ZmMsg.tag;
    } else if (field == ZmItem.F_ATTACHMENT) {
        tooltip = ZmMsg.attachmentHeaderToolTip;
    } else if (field == ZmItem.F_SUBJECT) {
        tooltip = sortable ? ZmMsg.sortBySubject : ZmMsg.subject;
    } else if (field == ZmItem.F_DATE) {
		if (sortable) {
			if (isOutboundFolder) {
				tooltip = (this._folderId == ZmFolder.ID_DRAFTS) ? ZmMsg.sortByLastSaved : ZmMsg.sortBySent;
			} else {
				tooltip = ZmMsg.sortByReceived;
			}
		} else {
			tooltip = ZmMsg.date;
		}
    } else if (field == ZmItem.F_FROM) {
        tooltip = sortable ? isOutboundFolder ? ZmMsg.sortByTo : ZmMsg.sortByFrom : isOutboundFolder ? ZmMsg.to : ZmMsg.from ;
    } else if (field == ZmItem.F_SIZE){
        tooltip = sortable ? ZmMsg.sortBySize : ZmMsg.sizeToolTip;
	} else if (field == ZmItem.F_ACCOUNT) {
		tooltip = ZmMsg.account;
    } else if (field == ZmItem.F_FOLDER) {
        tooltip = ZmMsg.folder;
    } else if (field == ZmItem.F_MSG_PRIORITY) {
		tooltip = ZmMsg.messagePriority
	} 
    
    return tooltip;
};

/**
 * @param params		[hash]			hash of params:
 *        field			[constant]		column ID
 *        item			[ZmItem]*		underlying item
 *        ev			[DwtEvent]*		mouseover event
 *        div			[Element]*		row div
 *        match			[hash]*			fields from div ID
 *        callback		[AjxCallback]*	callback (in case tooltip content retrieval is async)
 *        
 * @private
 */
ZmListView.prototype._getToolTip =
function(params) {
    var tooltip, field = params.field, item = params.item, div = params.div;
	if (field == ZmItem.F_FLAG) {
		return null; //no tooltip for the flag
    } else if (field == ZmItem.F_PRIORITY) {
        if (item.isHighPriority) {
            tooltip = ZmMsg.highPriorityTooltip;
        } else if (item.isLowPriority) {
            tooltip = ZmMsg.lowPriorityTooltip;
        }
    } else if (field == ZmItem.F_TAG) {
        tooltip = this._getTagToolTip(item);
    } else if (field == ZmItem.F_ATTACHMENT) {
        // disable att tooltip for now, we only get att info once msg is loaded
        // tooltip = this._getAttachmentToolTip(item);
    } else if (div && (field == ZmItem.F_DATE)) {
        tooltip = this._getDateToolTip(item, div);
    }
    return tooltip;
};

/*
 * Get the list of fields for the accessibility label. Normally, this
 * corresponds to the header columns.
 *
 * @protected
 */
ZmListView.prototype._getLabelFieldList =
function() {
	var headers = this._getHeaderList();

	if (headers) {
		return AjxUtil.map(headers, function(header) {
			return header._field;
		});
	}
};

/*
 * Get the accessibility label corresponding to the given field.
 *
 * @protected
 */
ZmListView.prototype._getLabelForField =
function(item, field) {

	if (field === ZmItem.F_SELECTION) {
		var itemHtmlEl = this._getElFromItem(item);
		if (itemHtmlEl && itemHtmlEl.getAttribute('aria-selected') === 'true') {
			return ZmMsg.selected;
		}
		return "";
	} 

	var tooltip = this._getToolTip({ item: item, field: field });
	// TODO: fix for tooltips that are callbacks (such as for appts)
	return AjxStringUtil.stripTags(tooltip);
};

ZmListView.prototype._updateLabelForItem =
function(item) {
	var fields = this._getLabelFieldList();
	var itemel = this._getElFromItem(item);

	if (!item || !fields || !itemel) {
		return;
	}

	var buf = [];

	for (var i = 0; i < fields.length; i++) {
		var label = this._getLabelForField(item, fields[i]);

		if (label) {
			buf.push(label);
		}
	}

	if (buf.length > 0) {
		itemel.setAttribute('aria-label', buf.join(', '));
	} else {
		itemel.removeAttribute('aria-label');
	}
};

ZmListView.prototype._getTagToolTip =
function(item) {
	if (!item) { return; }
	var numTags = item.tags && item.tags.length;
	if (!numTags) { return; }
	var tagList = appCtxt.getAccountTagList(item);
	var tags = item.tags;
	var html = [];
	var idx = 0;
    for (var i = 0; i < numTags; i++) {
		var tag = tagList.getByNameOrRemote(tags[i]);
        if (!tag) { continue; }        
		var nameText = tag.notLocal ? AjxMessageFormat.format(ZmMsg.tagNotLocal, tag.name) : tag.name;
        html[idx++] = "<table role='presentation'><tr><td>";
		html[idx++] = AjxImg.getImageHtml(tag.getIconWithColor());
		html[idx++] = "</td><td valign='middle'>";
		html[idx++] = AjxStringUtil.htmlEncode(nameText);
		html[idx++] = "</td></tr></table>";
	}
	return html.join("");
};

ZmListView.prototype._getAttachmentToolTip =
function(item) {
	var tooltip = null;
	var atts = item && item.attachments ? item.attachments : [];
	if (atts.length == 1) {
		var info = ZmMimeTable.getInfo(atts[0].ct);
		tooltip = info ? info.desc : null;
	} else if (atts.length > 1) {
		tooltip = AjxMessageFormat.format(ZmMsg.multipleAttachmentsTooltip, [atts.length]);
	}
	return tooltip;
};

ZmListView.prototype._getDateToolTip =
function(item, div) {
	div._dateStr = div._dateStr || this._getDateToolTipText(item.date);
	return div._dateStr;
};

ZmListView.prototype._getDateToolTipText =
function(date, prefix) {
	if (!date) { return ""; }
	var dateStr = [];
	var i = 0;
	dateStr[i++] = prefix;
	var dateFormatter = AjxDateFormat.getDateTimeInstance(AjxDateFormat.FULL, AjxDateFormat.MEDIUM);
	dateStr[i++] = dateFormatter.format(new Date(date));
	var delta = AjxDateUtil.computeDateDelta(date);
	if (delta) {
		dateStr[i++] = "<br><center><span style='white-space:nowrap'>(";
		dateStr[i++] = delta;
		dateStr[i++] = ")</span></center>";
	}
	return dateStr.join("");
};

/*
* Add a few properties to the list event for the listener to pick up.
*/
ZmListView.prototype._setListEvent =
function (ev, listEv, clickedEl) {
	DwtListView.prototype._setListEvent.call(this, ev, listEv, clickedEl);
	var target = this._getEventTarget(ev);
	var id = (target && target.id && target.id.indexOf("AjxImg") == -1) ? target.id : clickedEl.id;
	if (!id) return false; // don't notify listeners

	var m = this._parseId(id);
	if (ev.button == DwtMouseEvent.LEFT) {
		this._selEv.field = m ? m.field : null;
	} else if (ev.button == DwtMouseEvent.RIGHT) {
		this._actionEv.field = m ? m.field : null;
		if (m && m.field) {
			if (m.field == ZmItem.F_PARTICIPANT) {
				var item = this.getItemFromElement(clickedEl);
				this._actionEv.detail = item.participants ? item.participants.get(m.participant) : null;
			}
		}
	}
	return true;
};

ZmListView.prototype._allowLeftSelection =
function(clickedEl, ev, button) {
	// We only care about mouse events
	if (!(ev instanceof DwtMouseEvent)) { return true; }
	var target = this._getEventTarget(ev);
	var id = (target && target.id && target.id.indexOf("AjxImg") == -1) ? target.id : clickedEl.id;
	var data = this._data[clickedEl.id];
	var type = data.type;
	if (id && type && type == DwtListView.TYPE_LIST_ITEM) {
		var m = this._parseId(id);
		if (m && m.field) {
			return this._allowFieldSelection(m.item, m.field);
		}
	}
	return true;
};

ZmListView.prototype._allowFieldSelection =
function(id, field) {
	return (!this._disallowSelection[field]);
};

ZmListView.prototype._redoSearch =
function(callback) {
	var search = this._controller._currentSearch;
	if (!search) {
		return;
	}
	var sel = this.getSelection();
	var selItem = sel && sel[0];
	var changes = {
		isRedo: true,
		selectedItem: selItem
	};
	appCtxt.getSearchController().redoSearch(search, false, changes, callback);
};

ZmListView.prototype._sortColumn = function(columnItem, bSortAsc, callback) {

	// change the sort preference for this view in the settings
	var sortBy;
	switch (columnItem._sortable) {
		case ZmItem.F_FROM:		    sortBy = bSortAsc ? ZmSearch.NAME_ASC : ZmSearch.NAME_DESC; break;
        case ZmItem.F_TO:           sortBy = bSortAsc ? ZmSearch.RCPT_ASC : ZmSearch.RCPT_DESC; break;
		case ZmItem.F_NAME:		    sortBy = bSortAsc ? ZmSearch.SUBJ_ASC : ZmSearch.SUBJ_DESC; break; //used for Briefcase only now. SUBJ is mappaed to the filename of the document on the server side
		case ZmItem.F_SUBJECT:	    sortBy = bSortAsc ? ZmSearch.SUBJ_ASC : ZmSearch.SUBJ_DESC;	break;
		case ZmItem.F_DATE:		    sortBy = bSortAsc ? ZmSearch.DATE_ASC : ZmSearch.DATE_DESC;	break;
		case ZmItem.F_SIZE:		    sortBy = bSortAsc ? ZmSearch.SIZE_ASC : ZmSearch.SIZE_DESC;	break;
        case ZmItem.F_FLAG:		    sortBy = bSortAsc ? ZmSearch.FLAG_ASC : ZmSearch.FLAG_DESC;	break;
        case ZmItem.F_ATTACHMENT:   sortBy = bSortAsc ? ZmSearch.ATTACH_ASC : ZmSearch.ATTACH_DESC; break;
		case ZmItem.F_READ:		    sortBy = bSortAsc ? ZmSearch.READ_ASC : ZmSearch.READ_DESC;	break;
        case ZmItem.F_PRIORITY:     sortBy = bSortAsc ? ZmSearch.PRIORITY_ASC : ZmSearch.PRIORITY_DESC; break;
		case ZmItem.F_SORTED_BY:    sortBy = bSortAsc ? ZmSearch.DATE_ASC : ZmSearch.DATE_DESC;	break;
	}

	if (sortBy) {
		this._currentSortColId = columnItem._sortable;
		//special case - switching from read/unread to another sort column - remove it from the query, so users are not confused that they still see only unread messages after clicking on another sort column.
		if (columnItem._sortable != ZmItem.F_READ && (this._sortByString == ZmSearch.READ_ASC || this._sortByString == ZmSearch.READ_DESC)) {
			var controller = this._controller;
			var query = controller.getSearchString();
			if (query) {
				 controller.setSearchString(AjxStringUtil.trim(query.replace("is:unread", "")));
			}
		}
		this._sortByString = sortBy;
		var skipFirstNotify = this._folderId ? true : false; //just making it explicit boolean
        if (!appCtxt.isExternalAccount()) {
			var settings = appCtxt.getSettings();
           	appCtxt.set(ZmSetting.SORTING_PREF,
					   sortBy,
					   this.view,
					   false, //setDefault
					   skipFirstNotify, //skipNotify
					   null, //account
					   settings && !settings.persistImplicitSortPrefs(this.view)); //skipImplicit
            if (this._folderId) {
                appCtxt.set(ZmSetting.SORTING_PREF, sortBy, this._folderId);
            }
        }
		if (!this._isMultiColumn) {
			this._setSortedColStyle(columnItem._id);
		}
	}
	if (callback) {
		callback.run();
	}
};

ZmListView.prototype._setNextSelection =
function(item, forceSelection) {
	// set the next appropriate selected item
	if (this.firstSelIndex < 0) {
		this.firstSelIndex = 0;
	}
	if (this._list && !item) {
		item = this._list.get(this.firstSelIndex) || this._list.getLast();
    }
	if (item) {
		this.setSelection(item, false, forceSelection);
	}
};

ZmListView.prototype._relayout =
function() {
	DwtListView.prototype._relayout.call(this);
	this._checkColumns();
};

ZmListView.prototype._checkColumns =
function() {
	var numCols = this._headerList.length;
	var fields = [];
	for (var i = 0; i < numCols; i++) {
		var headerCol = this._headerList[i];
		// bug 43540: always skip account header since its a multi-account only
		// column and we don't want it to sync
		if (headerCol && headerCol._field != ZmItem.F_ACCOUNT) {
			fields.push(headerCol._field + (headerCol._visible ? "" : "*"));
		}
	}
	var value = fields.join(ZmListView.COL_JOIN);
	value = (value == this._defaultCols) ? "" : value;
    if (!appCtxt.isExternalAccount() && !this._controller.isSearchResults) {
	    appCtxt.set(ZmSetting.LIST_VIEW_COLUMNS, value, appCtxt.getViewTypeFromId(this.view));
    }

	this._colHeaderActionMenu = this._getActionMenuForColHeader(true); // re-create action menu so order is correct
};

/**
 * Scroll-based paging. Make sure we have at least one page of items below the visible list.
 * 
 * @param ev
 */
ZmListView.handleScroll =
function(ev) {
	var target = DwtUiEvent.getTarget(ev);
	var lv = DwtControl.findControl(target);
	if (lv) {
		lv._checkItemCount();
	}
};

/**
 * Figure out if we should fetch some more items, based on where the scroll is. Our goal is to have
 * a certain number available below the bottom of the visible view.
 */
ZmListView.prototype._checkItemCount =
function() {
	var itemsNeeded = this._getItemsNeeded();
	if (itemsNeeded) {
		this._controller._paginate(this._view, true, null, itemsNeeded);
	}
};

/**
 * Figure out how many items we need to fetch to maintain a decent number
 * below the fold. Nonstandard list views may override.
 */
ZmListView.prototype._getItemsNeeded =
function(skipMoreCheck) {

	if (!skipMoreCheck) {
		var itemList = this.getItemList();
		if (!(itemList && itemList.hasMore()) || !this._list) { return 0; }
	}
	if (!this._rendered || !this._rowHeight) { return 0; }

	DBG.println(AjxDebug.DBG2, "List view: checking item count");

	var sbCallback = new AjxCallback(null, AjxTimedAction.scheduleAction, [new AjxTimedAction(this, this._resetColWidth), 100]);
	var params = {scrollDiv:	this._getScrollDiv(),
				  rowHeight:	this._rowHeight,
				  threshold:	this.getPagelessThreshold(),
				  limit:		this.getLimit(1),
				  listSize:		this._list.size(),
				  sbCallback:	sbCallback};
	return ZmListView.getRowsNeeded(params);
};

ZmListView.prototype._getScrollDiv =
function() {
	return this._parentEl;
};

ZmListView.getRowsNeeded =
function(params) {

	var div = params.scrollDiv;
	var sh = div.scrollHeight, st = div.scrollTop, rh = params.rowHeight;

	// view (porthole) height - everything measured relative to its top
	// prefer clientHeight since (like scrollHeight) it doesn't include borders
	var h = div.clientHeight || Dwt.getSize(div).y;

	// where we'd like bottom of list view to be (with extra hidden items at bottom)
	var target = h + (params.threshold * rh);

	// where bottom of list view is (including hidden items)
	var bottom = sh - st;

	if (bottom == h) {
		// handle cases where there's no scrollbar, but we have more items (eg tall browser, or replenishment)
		bottom = (params.listSize * rh) - st;
		if (st == 0 && params.sbCallback) {
			// give list view a chance to fix width since it may be getting a scrollbar
			params.sbCallback.run();
		}
	}

	var rowsNeeded = 0;
	if (bottom < target) {
		// buffer below visible bottom of list view is not full
		rowsNeeded = Math.max(Math.floor((target - bottom) / rh), params.limit);
	}
	return rowsNeeded;
};

ZmListView.prototype._sizeChildren =
function(height) {
	if (DwtListView.prototype._sizeChildren.apply(this, arguments)) {
		this._checkItemCount();
	}
};

// Allow list view classes to override type used in nav text. Return null to say "items".
ZmListView.prototype._getItemCountType =
function() {
	return this.type;
};

/**
 * Checks if the given item is in this view's list. Note that the view's list may
 * be only part of the controller's list (the currently visible page).
 *
 * @param {String|ZmItem}	item		the item ID, or item to check for
 * @return	{Boolean}	<code>true</code> if the item is in the list
 */
ZmListView.prototype.hasItem =
function(item) {

	var id = (typeof item == "string") ? item : item && item.id;
	if (id && this._list) {
		var a = this._list.getArray();
		for (var i = 0, len = a.length; i < len; i++) {
			var item = a[i];
			if (item && item.id == id) {
				return true;
			}
		}
	}
	return false;
};

/**
 * The following methods allow a list view to maintain state after it has
 * been rerendered. State may include such elements as: which items are selected,
 * focus, scroll position, etc.
 *
 * @private
 * @param {hash}		params		hash of parameters:
 * @param {boolean}		selection	if true, preserve selection
 * @param {boolean}		focus		if true, preserve focus
 * @param {boolean}		scroll		if true, preserve scroll position
 */
ZmListView.prototype._saveState =
function(params) {

	var s = this._state = {};
	params = params || {};
	if (params.selection) {
		s.selected = this.getSelection();
		if (s.selected.length == 1) {
			//still a special case for now till we rewrite this thing.
			var el = this._getElFromItem(s.selected[0]); //terribly ugly, get back to the html element so i can have access to the item data
			s.singleItemChecked = this._getItemData(el, ZmListView.ITEM_CHECKED_ATT_NAME);
		}
	}
	if (params.focus) {
		s.focused = this.hasFocus();
		s.anchorItem = this._kbAnchor && this.getItemFromElement(this._kbAnchor);
	}
	if (params.scroll) {
		s.rowHeight = this._rowHeight;
		s.scrollTop = this._listDiv.scrollTop;
	}
};

ZmListView.prototype._restoreState =
function(state) {

	var s = state || this._state;
	if (s.selected && s.selected.length) {
		var dontCheck = s.selected.length == 1 && !s.singleItemChecked;
		this.setSelectedItems(s.selected, dontCheck);
	}
	if (s.anchorItem) {
		var el = this._getElFromItem(s.anchorItem);
		if (el) {
			this._setKbFocusElement(el);
		}
	}
	if (s.focused) {
		this.focus();
	}
	// restore scroll position based on row height ratio
	if (s.rowHeight) {
		this._listDiv.scrollTop = s.scrollTop * (this._rowHeight / s.rowHeight);
	}
	this._state = {};
};

ZmListView.prototype._renderList =
function(list, noResultsOk, doAdd) {
    var group = this._group;
    if (!group) {
        return DwtListView.prototype._renderList.call(this, list, noResultsOk, doAdd);
    }
	if (list instanceof AjxVector && list.size()) {
		var now = new Date();
		var size = list.size();
		var htmlArr = [];
        var section;
        var headerDiv;
		for (var i = 0; i < size; i++) {
			var item = list.get(i);
			var div = this._createItemHtml(item, {now:now}, !doAdd, i);
			if (div) {
				if (div instanceof Array) {
					for (var j = 0; j < div.length; j++){
                        section = group.addMsgToSection(item, div[j]);
                        if (group.getSectionSize(section) == 1){
                            headerDiv = this._getSectionHeaderDiv(group, section);
                            this._addRow(headerDiv);
                        }
						this._addRow(div[j]);
					}
				} else if (div.tagName || doAdd) {
                    section = group.addMsgToSection(item, div);
                    if (group.getSectionSize(section) == 1){
                        headerDiv = this._getSectionHeaderDiv(group, section);
                        this._addRow(headerDiv);
                    }
                    this._addRow(div);
				} else {
                    group.addMsgToSection(item, div);
				}
			}
		}
		if (group && !doAdd) {
			group.resetSectionHeaders();
			htmlArr.push(group.getAllSections(this._bSortAsc));
		}

		if (htmlArr.length && !doAdd) {
			this._parentEl.innerHTML = htmlArr.join("");
		}
	} else if (!noResultsOk) {
		this._setNoResultsHtml();
	}

};

ZmListView.prototype._addRow =
function(row, index) {
	DwtListView.prototype._addRow.apply(this, arguments);

	this._updateLabelForItem(this.getItemFromElement(row));
};

ZmListView.prototype._itemAdded = function(item) {
    item.refCount++;
};

ZmListView.prototype._getSectionHeaderDiv =
function(group, section) {
    if (group && section) {
        var headerDiv = document.createElement("div");
        var sectionTitle = group.getSectionTitle(section);
        var html = group.getSectionHeader(sectionTitle);
        headerDiv.innerHTML = html;
        return headerDiv.firstChild;
    }
};

ZmListView.prototype.deactivate =
function() {
	this._controller.inactive = true;
};

ZmListView.prototype._getEventTarget =
function(ev) {
	var target = ev && ev.target;
	if (target && (target.nodeName === "IMG" || (target.className && target.className.match(/\bImg/)))) {
		return target.parentNode;
	}
	return target;
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmAppChooser")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines the zimbra application chooser.
 *
 */

/**
 * @class
 * This class represents a zimbra application chooser. The chooser is the "tab application" toolbar shown
 * in the Zimbra Web Client. The toolbar buttons are represented as "tabs".
 * 
 * @param	{Hash}	params		a hash of parameters
 * 
 * @extends	ZmToolBar
 */
ZmAppChooser = function(params) {

	params.className = params.className || "ZmAppChooser";
	params.width = appCtxt.getSkinHint("appChooser", "fullWidth") ? "100%" : null;

	ZmToolBar.call(this, params);

    Dwt.setLocation(this.getHtmlElement(), Dwt.LOC_NOWHERE, Dwt.LOC_NOWHERE);

	this.setScrollStyle(Dwt.CLIP);

	this._buttonListener = new AjxListener(this, this._handleButton);
    this._initOverFlowTabs();
	var buttons = params.buttons;
	for (var i = 0; i < buttons.length; i++) {
		var id = buttons[i];
		if (id == ZmAppChooser.SPACER) {
			this.addSpacer(ZmAppChooser.SPACER_HEIGHT);
		} else {
			this._createButton(id);
		}
	}

	this._createPrecedenceList();
	this._inited = true;
};

ZmAppChooser.prototype = new ZmToolBar;
ZmAppChooser.prototype.constructor = ZmAppChooser;
ZmAppChooser.prototype.role = "tablist";

ZmAppChooser.prototype._initOverFlowTabs =
function(){
    this._leftOverflow = document.getElementById("moreTabsLeftContainer");
    this._rightOverflow = document.getElementById("moreTabsRightContainer");
    if (this._leftOverflow) {
		this._leftOverflow.onclick = this._showLeftTab.bind(this);
	}
	if (this._rightOverflow) {
    	this._rightOverflow.onclick = this._showRightTab.bind(this);
	}
    this._leftBtnIndex = -1;
    this._deletedButtons = [];
};

ZmAppChooser.prototype._showLeftTab =
function(){
    var items = this.getItems();
    if(this._leftBtnIndex > -1){
        items[this._leftBtnIndex].setVisible(true);
        this._leftBtnIndex--;
    }
    this._checkArrowVisibility();
};

ZmAppChooser.prototype._showRightTab =
function(){
    var items = this.getItems();
    this._leftBtnIndex++;
    items[this._leftBtnIndex].setVisible(false);
    this._checkArrowVisibility();
};

ZmAppChooser.prototype._showTab =
function(id, ev){
    var button = this._buttons[id];
    this._moreTabsBtn.getMenu().popdown();
    if (!button) return;
    if (!button.getVisible()){ // Left side
        var found = false;
        for (var index in this._buttons){
            if (!found && this._buttons[index].getHTMLElId() == button.getHTMLElId())
              found = true;
            else if (this._buttons[index].getVisible())
                break;
            if (found){
                this._buttons[index].setVisible(true)
                this._leftBtnIndex--;
            }
        }
    }else { // Right side
        while(this._isTabOverflow(button.getHtmlElement())){
            this._showRightTab();
        }
    }
    this._checkArrowVisibility();
    appCtxt.getAppController()._appButtonListener(ev);


};

ZmAppChooser.prototype._attachMoreTabMenuItems =
function(menu){

    for (var deletedIndex=0; deletedIndex < this._deletedButtons.length; deletedIndex++){
        var mi = menu.getItemById(ZmOperation.MENUITEM_ID, this._deletedButtons[deletedIndex] + "_menu");
        if (mi) {
            menu.removeChild(mi);
            mi.dispose();
        }
    }

    this._deletedButtons = [];
    for(var index in this._buttons){
        var item = menu.getItemById(ZmOperation.MENUITEM_ID, index + "_menu");
        if (item){
            if (item.getText() != this._buttons[index].getText()){
                item.setText(this._buttons[index].getText());
            }
        } else {
            var mi = new DwtMenuItem({parent:menu, style:DwtMenuItem.CASCADE_STYLE, id: index + "_menu"});
            mi.setData(ZmOperation.MENUITEM_ID, index + "_menu" );
            mi.setData(Dwt.KEY_ID, index);
            mi.addSelectionListener(this._showTab.bind(this, index));
            mi.setText(this._buttons[index].getText());
        }
    }

    if(menu.getHtmlElement().style.width == "0px"){
        this._moreTabsBtn.popup();
    }
};

ZmAppChooser.prototype._showOverflowTabsMenu =
function(){
    var menu = new DwtMenu({parent:this._moreTabsBtn});
    menu.addPopupListener(new AjxListener(this, this._attachMoreTabMenuItems,[menu]));
    return menu;
};


ZmAppChooser.prototype._checkArrowVisibility =
function(){
    var items = this.getItems();
    if (this._leftBtnIndex < 0)
        this._setArrowVisibility(this._leftOverflow, "none");
    else
        this._setArrowVisibility(this._leftOverflow, "");

    if (!this._isTabOverflow(items[items.length -1].getHtmlElement())){
        this._setArrowVisibility(this._rightOverflow, "none");
    }else{
        this._setArrowVisibility(this._rightOverflow, "");
    }
    this._adjustWidth();
};



ZmAppChooser.prototype._setArrowVisibility =
function(element, option){
	if (!element) {
		return;
	}
    element.style.display = option|| "";
    var display = ((this._leftOverflow && this._leftOverflow.style.display == "none") && 
					(this._rightOverflow && this._rightOverflow.style.display == "none")) ? "none" : "";
	var moreTabsMenu = document.getElementById("moreTabsMenu");
	if (moreTabsMenu) {
    	moreTabsMenu.style.display = display;
	}
    if (display != "none" && !this._moreTabsBtn && moreTabsMenu ){
        var containerEl = moreTabsMenu;
        var button = new DwtToolBarButton({parent:DwtShell.getShell(window), id: "moreTabsMenuBtn", style:"background:none no-repeat scroll 0 0 transparent; border: none"});
        button.setToolTipContent(ZmMsg.more, true);
        button.setText("");
        button.reparentHtmlElement(moreTabsMenu);
        button.setMenu(new AjxListener(this, this._showOverflowTabsMenu));
        this._moreTabsBtn =  button;
    }
};

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAppChooser.prototype.toString =
function() {
	return "ZmAppChooser";
};

//
// Constants
//

ZmAppChooser.SPACER								= "spacer";
ZmAppChooser.B_HELP								= "Help";
ZmAppChooser.B_LOGOUT							= "Logout";

ZmApp.CHOOSER_SORT[ZmAppChooser.SPACER]			= 160;
ZmApp.CHOOSER_SORT[ZmAppChooser.B_HELP]			= 170;
ZmApp.CHOOSER_SORT[ZmAppChooser.B_LOGOUT]		= 190;

// hard code help/logout since they are not real "apps"
ZmApp.ICON[ZmAppChooser.B_HELP]					= "Help";
ZmApp.ICON[ZmAppChooser.B_LOGOUT]				= "Logoff";
ZmApp.CHOOSER_TOOLTIP[ZmAppChooser.B_HELP]		= "goToHelp";
ZmApp.CHOOSER_TOOLTIP[ZmAppChooser.B_LOGOUT]	= "logOff";

ZmAppChooser.SPACER_HEIGHT = 10;

//
// Data
//

ZmAppChooser.prototype.TEMPLATE = "share.Widgets#ZmAppChooser";
ZmAppChooser.prototype.ITEM_TEMPLATE = "share.Widgets#ZmAppChooserItem";
ZmAppChooser.prototype.SPACER_TEMPLATE = "dwt.Widgets#ZmAppChooserSpacer";

//
// Public methods
//
/**
 * Adds a selection listener.
 * 
 * @param	{AjxListener}	listener	the listener
 */
ZmAppChooser.prototype.addSelectionListener =
function(listener) {
	this.addListener(DwtEvent.SELECTION, listener);
};

ZmAppChooser.prototype._checkTabOverflowAdd =
function(button) {
    var display = "none";
    if (this._isTabOverflow(button)){
        display = "";
    }
    this._setArrowVisibility(this._rightOverflow, display);
    this._adjustWidth();
};

ZmAppChooser.prototype._isTabOverflow =
function(tab){
        var tabPos = tab.offsetLeft + tab.clientWidth + 30;
        if (!this._refElement){
            this._refElement = document.getElementById(this._refElementId);
        }
        var container = this._refElement && this._refElement.parentNode;

        if (!container) return false;
        var offsetWidth = container.offsetWidth;
        return (offsetWidth < tabPos);
};

ZmAppChooser.prototype._adjustWidth =
function(){
    var container = this._refElement && this._refElement.parentNode;
	if (container && container.offsetWidth >= 30) {
    	this._refElement.style.maxWidth = this._refElement.style.width = container.offsetWidth - 30 + "px";
    	this._refElement.style.overflow = "hidden";
	}

};

ZmAppChooser.prototype._checkTabOverflowDelete =
function(index){
    var items = this.getItems();
    if(this._isTabOverflow(items[items.length - 1])){
      return;
    }
    this._showLeftTab();
}


/**
 * Adds a button to the toolbar.
 * 
 * @param	{String}	id		the button id
 * @param	{Hash}		params		a hash of parameters
 * @param	{String}	params.text			the text
 * @param	{String}	params.image		the image
 * @param	{int}	params.index		the index
 * @param	{String}	params.tooltip		the tool top
 * @param	{String}	params.textPrecedence	the image precedence
 * @param	{String}	params.imagePrecedence		the image precedence
 * 
 * @return	{ZmAppButton}		the newly created button
 */
ZmAppChooser.prototype.addButton =
function(id, params) {

	var buttonParams = {
		parent:		this,
		id:			ZmId.getButtonId(ZmId.APP, id),
		text:		params.text,
		image:		params.image,
		leftImage:	params.leftImage,
		rightImage:	params.rightImage,
		index:		params.index,
		ariaControls:	'skin_outer_td_main'
	};
    buttonParams.style = params.style ? params.style : DwtLabel.IMAGE_LEFT;
    var button = new ZmAppButton(buttonParams);
	button.setToolTipContent(params.tooltip, true);
	button.textPrecedence = params.textPrecedence;
	button.imagePrecedence = params.imagePrecedence;
	button.setData(Dwt.KEY_ID, id);
	button.addSelectionListener(this._buttonListener);
	this._buttons[id] = button;
    this._checkTabOverflowAdd(button.getHtmlElement());
    var parentEl = button.getHtmlElement().parentElement;
    if (parentEl) {
        parentEl.setAttribute('role', 'heading');
        parentEl.setAttribute('aria-level', '2');
    }
	return button;
};

/**
 * Removes a button.
 * 
 * @param	{String}	id		the id of the button to remove
 */
ZmAppChooser.prototype.removeButton =
function(id) {
	var button = this._buttons[id];
	if (button) {
        var index = this.__getButtonIndex(id);
		button.dispose();
		this._buttons[id] = null;
		delete this._buttons[id];
        if (this._moreTabsBtn &&
            this._moreTabsBtn.getMenu() &&
            this._moreTabsBtn.getMenu().getItemCount() > 0){
            this._deletedButtons.push(id);
        }
        this._checkTabOverflowDelete(index);
	}
};

/**
 * Replaces a button.
 * 
 * @param	{String}	oldId		the old button id
 * @param	{String}	newId		the new button id
 * @param	{Hash}		params		a hash of parameters
 * @param	{String}	params.text		the text
 * @param	{String}	params.image	the image
 * @param	{int}	params.index		the index
 * @param	{String}	params.tooltip			the tool tip
 * @param	{String}	params.textPrecedence	the text display precedence
 * @param	{String}	params.imagePrecedence	the image display precedence
 * 
 * @return	{ZmAppButton}		the newly created button
 */
ZmAppChooser.prototype.replaceButton =
function(oldId, newId, params) {
	if (!this._buttons[oldId]) { return null; }
	params.index = this.__getButtonIndex(oldId);
	this.removeButton(oldId);
	return this.addButton(newId, params);
};

ZmAppChooser.prototype.getButton =
function(id) {
	return this._buttons[id];
};

/**
 * Sets the specified button as selected.
 * 
 * @param	{String}	id		the button id
 */
ZmAppChooser.prototype.setSelected =
function(id) {
	var oldBtn = this._buttons[this._selectedId];
	if (this._selectedId && oldBtn) {
        this.__markPrevNext(this._selectedId, false);
		oldBtn.setSelected(false);
    }

	var newBtn = this._buttons[id];
	if (newBtn) {
		newBtn.setSelected(true);
		this.setAttribute('aria-activedescendant', newBtn.getHTMLElId());

		if (newBtn._toggleText != null && newBtn._toggleText != "") {
			// hide text for previously selected button first
			if (oldBtn) {
				oldBtn._toggleText = (oldBtn._toggleText != null && oldBtn._toggleText != "")
					? oldBtn._toggleText : oldBtn.getText();
				oldBtn.setText("");
			}

			// reset original text for  newly selected button
			newBtn.setText(newBtn._toggleText);
			newBtn._toggleText = null;
		}
	}

	this._selectedId = id;
};

/**
 * @private
 */
ZmAppChooser.prototype._createButton =
function(id) {
	this.addButton(id, {text:ZmMsg[ZmApp.NAME[id]] || ZmApp.NAME[id], image:ZmApp.ICON[id], tooltip:ZmMsg[ZmApp.CHOOSER_TOOLTIP[id]],
						textPrecedence:ZmApp.TEXT_PRECEDENCE[id], imagePrecedence:ZmApp.IMAGE_PRECEDENCE[id]});
};

/**
 * @private
 */
ZmAppChooser.prototype._handleButton =
function(evt) {
	this.notifyListeners(DwtEvent.SELECTION, evt);
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmAppButton")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines the tab application button.
 *
 */

/**
 * @class
 * This class represents a button that behaves like a "tab" button, designed specifically for the row of
 * applications buttons at the top of the Zimbra Web Client interface.
 * <p>
 * Limitations:
 * <ul>
 * <li>cannot have a menu</li>
 * <li>does not support enabled/disabled</li>
 * </ul>
 * </p>
 * 
 * @author Conrad Damon
 * 
 * @param	{Hash}		params		a hash of parameters
 * 
 * @extends		DwtButton
 */
ZmAppButton = function(params) {

	if (arguments.length == 0) {
		return;
	}

    params.style = params.style ? params.style : DwtLabel.IMAGE_LEFT;
	params.posStyle = DwtControl.RELATIVE_STYLE;
    DwtButton.call(this, params);

	if (params.image) {
		this.setImage(params.image);
	}
	else {
		if (params.leftImage) {
			this.setImage(	params.leftImage, DwtLabel.LEFT);
		}
		if (params.rightImage) {
			this.setImage(params.rightImage, DwtLabel.RIGHT);
		}
	}

	if (params.hoverImage) {
		this.setHoverImage(params.hoverImage);
	}
	else {
		if (params.leftHoverImage) {
			this.setHoverImage(params.leftHoverImage, DwtLabel.LEFT);
		}
		if (params.rightHoverImage) {
			this.setHoverImage(params.rightHoverImage, DwtLabel.RIGHT);
		}
	}
    this.setText(params.text);
};

ZmAppButton.prototype = new DwtButton;
ZmAppButton.prototype.constructor = ZmAppButton;
ZmAppButton.prototype.role = "tab";

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAppButton.prototype.toString =
function() {
	return "ZmAppButton";
};

//
// Data
//

ZmAppButton.prototype.TEMPLATE = "share.Widgets#ZmAppChooserButton";

//
// Public methods
//
ZmAppButton.prototype.setSelected =
function(selected) {
    this.isSelected = selected;
    this.setDisplayState(selected ? DwtControl.SELECTED : DwtControl.NORMAL);
};

/**
 * Sets the display state.
 * 
 * @param	{String}	state		the display state
 * @see		DwtControl
 */
ZmAppButton.prototype.setDisplayState =
function(state) {
    if (this.isSelected && state != DwtControl.SELECTED) {
        state = [DwtControl.SELECTED, state].join(" ");
    }
    DwtButton.prototype.setDisplayState.call(this, state);
};

ZmAppButton.prototype.handleKeyAction =
function(actionCode, ev) {

	switch (actionCode) {

		case DwtKeyMap.SELECT:
			if (this.isListenerRegistered(DwtEvent.SELECTION)) {
				var selEv = DwtShell.selectionEvent;
				selEv.item = this;
				this.notifyListeners(DwtEvent.SELECTION, selEv);
			}
			break;

		default:
			return false;
	}
	return true;
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmStatusView")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * Creates the status view.
 * @class
 * This class represents the status view.
 * 
 * @param    {DwtControl}    parent        the parent
 * @param    {String}        className     the class name
 * @param    {constant}      posStyle      the position style
 * @param    {String}        id            the id
 * 
 * @extends		DwtControl
 */
ZmStatusView = function(parent, className, posStyle, id) {

    DwtControl.call(this, {parent:parent, className:(className || "ZmStatus"), posStyle:posStyle, id:id});

    this._toast = this._standardToast = new ZmToast(this, ZmId.TOAST);
    this._statusQueue = [];
};

ZmStatusView.prototype = new DwtControl;
ZmStatusView.prototype.constructor = ZmStatusView;


// Constants
/**
 * Defines the "informational" status level.
 */
ZmStatusView.LEVEL_INFO             = 1;    // informational
/**
 * Defines the "warning" status level.
 */
ZmStatusView.LEVEL_WARNING          = 2;    // warning
/**
 * Defines the "critical" status level.
 */
ZmStatusView.LEVEL_CRITICAL         = 3;    // critical

ZmStatusView.MSG_PARAMS = ["msg", "level", "detail", "transitions", "toast", "force", "dismissCallback", "finishCallback"];

// Public methods

ZmStatusView.prototype.toString =
function() {
    return "ZmStatusView";
};

/**
 * Displays a status message.
 * 
 * @param {String}    msg the message
 * @param {constant}    [level]         the level (see {@link ZmStatusView}<code>.LEVEL_</code> constants) 
 * @param {String}    [detail]         the details
 * @param {String}    [transitions] the transitions (see {@link ZmToast})
 * @param {String}    [toast]     the toast control
 * @param {boolean}    [force]        force any displayed toasts out of the way
 * @param {AjxCallback}    [dismissCallback]    callback to run when the toast is dismissed (by another message using [force], or explicitly calling ZmStatusView.prototype.dismiss())
 * @param {AjxCallback}    [finishCallback]     callback to run when the toast finishes its transitions by itself (not when dismissed)
 */
ZmStatusView.prototype.setStatusMsg =
function(params) {
    params = Dwt.getParams(arguments, ZmStatusView.MSG_PARAMS);
    if (typeof params == "string") {
        params = { msg: params };
    }
    var work = {
        msg: params.msg,
        level: params.level || ZmStatusView.LEVEL_INFO,
        detail: params.detail,
        date: new Date(),
        transitions: params.transitions,
        toast: params.toast || this._standardToast,
        dismissCallback: (params.dismissCallback instanceof AjxCallback) ? params.dismissCallback : null,
        finishCallback: (params.finishCallback instanceof AjxCallback) ? params.finishCallback : null,
		dismissed: false
    };

	if (params.force) { // We want to dismiss ALL messages in the queue and display the new message
		for (var i=0; i<this._statusQueue.length; i++) {
			this._statusQueue[i].dismissed = true; // Dismiss all messages in the queue in turn, calling their dismissCallbacks along the way
		}
	}
    // always push so we know one is active
    this._statusQueue.push(work);
    if (!this._toast.isPoppedUp()) {
        this._updateStatusMsg();
    } else if (params.force) {
        this.dismissStatusMsg();
    }
};

ZmStatusView.prototype.nextStatus =
function() {
    if (this._statusQueue.length > 0) {
        this._updateStatusMsg();
        return true;
    }
    return false;
};

ZmStatusView.prototype.dismissStatusMsg =
function(all) {
	if (all) {
		for (var i=0; i<this._statusQueue.length; i++) {
			this._statusQueue[i].dismissed = true; // Dismiss all messages in the queue in turn, calling their dismissCallbacks along the way
		}
	}
    this._toast.dismiss();
};

// Protected methods

ZmStatusView.prototype._updateStatusMsg =
function() {
    var work = this._statusQueue.shift();
    if (!work) { return; }
	if (work.dismissed) { // If preemptively dismissed, just run the callback and proceed to the next msg
		if (work.dismissCallback)
			work.dismissCallback.run();
		this.nextStatus();
	} else {
		this._toast = work.toast;
		this._toast.popup(work);
	}
};


//
// ZmToast
//

/**
 * Creates the "toaster".
 * @class
 * This class represents the "toaster".
 * 
 * @extends	DwtComposite
 */
ZmToast = function(parent, id) {
    if (arguments.length == 0) { return; }

    DwtComposite.call(this, {parent:parent.shell, className:"ZToast", posStyle:Dwt.ABSOLUTE_STYLE, id:id});
    this._statusView = parent;
    this._createHtml();

    this._funcs = {};
    this._funcs["position"] = AjxCallback.simpleClosure(this.__position, this);
    this._funcs["show"] = AjxCallback.simpleClosure(this.__show, this);
    this._funcs["hide"] = AjxCallback.simpleClosure(this.__hide, this);
    this._funcs["pause"] = AjxCallback.simpleClosure(this.__pause, this);
    this._funcs["hold"] = AjxCallback.simpleClosure(this.__hold, this);
    this._funcs["idle"] = AjxCallback.simpleClosure(this.__idle, this);
    this._funcs["fade"] = AjxCallback.simpleClosure(this.__fade, this);
    this._funcs["fade-in"] = this._funcs["fade"];
    this._funcs["fade-out"] = this._funcs["fade"];
    this._funcs["slide"] = AjxCallback.simpleClosure(this.__slide, this);
    this._funcs["slide-in"] = this._funcs["slide"];
    this._funcs["slide-out"] = this._funcs["slide"];
    this._funcs["next"] = AjxCallback.simpleClosure(this.transition, this);
}
ZmToast.prototype = new DwtComposite;

ZmToast.prototype.constructor = ZmToast;
ZmToast.prototype.toString =
function() {
    return "ZmToast";
};

ZmToast.prototype.role = 'alert';
ZmToast.prototype.isFocusable = true;

// Constants
/**
 * Defines the "fade" transition.
 */
ZmToast.FADE = { type: "fade" };
/**
 * Defines the "fade-in" transition.
 */
ZmToast.FADE_IN = { type: "fade-in" };
/**
 * Defines the "fade-out" transition.
 */
ZmToast.FADE_OUT = { type: "fade-out" };
/**
 * Defines the "slide" transition.
 */
ZmToast.SLIDE = { type: "slide" };
/**
 * Defines the "slide-in" transition.
 */
ZmToast.SLIDE_IN = { type: "slide-in" };
/**
 * Defines the "slide-out" transition.
 */
ZmToast.SLIDE_OUT = { type: "slide-out" };
/**
 * Defines the "pause" transition.
 */
ZmToast.PAUSE = { type: "pause" };
/**
 * Defines the "hold" transition.
 */
ZmToast.HOLD = { type: "hold" };
/**
 * Defines the "idle" transition.
 */
ZmToast.IDLE = {type: "idle" };
/**
 * Defines the "show" transition.
 */
ZmToast.SHOW = {type: "show" };

//ZmToast.DEFAULT_TRANSITIONS = [ ZmToast.FADE_IN, ZmToast.PAUSE, ZmToast.FADE_OUT ];
ZmToast.DEFAULT_TRANSITIONS = [ ZmToast.SLIDE_IN, ZmToast.PAUSE, ZmToast.SLIDE_OUT ];

ZmToast.DEFAULT_STATE = {};
ZmToast.DEFAULT_STATE["position"] = { location: "C" }; // center
ZmToast.DEFAULT_STATE["pause"] = { duration: 1200 };
ZmToast.DEFAULT_STATE["hold"] = {};
ZmToast.DEFAULT_STATE["fade"] = { duration: 100, multiplier: 1 };
ZmToast.DEFAULT_STATE["fade-in"] = { start: 0, end: 99, step: 10, duration: 200, multiplier: 1 };
ZmToast.DEFAULT_STATE["fade-out"] = { start: 99, end: 0, step: -10, duration: 200, multiplier: 1 };
ZmToast.DEFAULT_STATE["slide"] = { duration: 100, multiplier: 1 };
ZmToast.DEFAULT_STATE["slide-in"] = { start: -40, end: 0, step: 1, duration: 100, multiplier: 1 };
ZmToast.DEFAULT_STATE["slide-out"] = { start: 0, end: -40, step: -1, duration: 100, multiplier: 1 };

ZmToast.LEVEL_RE = /\b(ZToastCrit|ZToastWarn|ZToastInfo)\b/g;
ZmToast.DISMISSABLE_STATES = [ZmToast.HOLD];

// Data

ZmToast.prototype.TEMPLATE = "share.Widgets#ZToast";


// Public methods

ZmToast.prototype.dispose =
function() {
    this._textEl = null;
    this._iconEl = null;
    this._detailEl = null;
    DwtComposite.prototype.dispose.call(this);
};

ZmToast.prototype.popup =
function(work) {
    this.__clear();
    this._poppedUp = true;
    this._dismissed = false;
    this._dismissCallback = work.dismissCallback;
    this._finishCallback = work.finishCallback;

    var icon, className, label;

    switch (work.level) {
    case ZmStatusView.LEVEL_CRITICAL:
        className = "ZToastCrit";
        icon = "Critical";
        label = AjxMsg.criticalMsg;
        break;

    case ZmStatusView.LEVEL_WARNING:
        className = "ZToastWarn";
        icon = "Warning";
        label = AjxMsg.warningMsg;
        break;

    case ZmStatusView.LEVEL_INFO:
    default:
        className = "ZToastInfo";
        icon = "Success";
        label = AjxMsg.infoMsg;
        break;
    }

    // setup display
    var el = this.getHtmlElement();
    Dwt.delClass(el, ZmToast.LEVEL_RE, className);

    if (this._iconEl) {
        this._iconEl.innerHTML = AjxImg.getImageHtml({
            imageName: icon, altText: label
        });
    }

    if (this._textEl) {
        // we use and add a dedicated SPAN to make sure that we trigger all
        // screen readers
        var span = document.createElement('SPAN');
        span.innerHTML = work.msg || "";

        Dwt.removeChildren(this._textEl);
        this._textEl.appendChild(span);
    }

    // get transitions
    var location = appCtxt.getSkinHint("toast", "location");
    var transitions =
        (work.transitions || appCtxt.getSkinHint("toast", "transitions") ||
         ZmToast.DEFAULT_TRANSITIONS);

    transitions = [].concat( {type:"position", location:location}, transitions, {type:"hide"} );

    // start animation
    this._transitions = transitions;
    this.transition();
};

ZmToast.prototype.popdown =
function() {
    this.__clear();
    this.setLocation(Dwt.LOC_NOWHERE, Dwt.LOC_NOWHERE);
    this._poppedUp = false;

    if (!this._dismissed) {
        if (this._finishCallback)
            this._finishCallback.run();
    }

    this._dismissed = false;
};

ZmToast.prototype.isPoppedUp =
function() {
    return this._poppedUp;
};

ZmToast.prototype.transition =
function() {

    if (this._pauseTimer) {
        clearTimeout(this._pauseTimer);
        this._pauseTimer = null;
    }
    if (this._held) {
        this._held = false;
    }

    var transition = this._transitions && this._transitions.shift();
    if (!transition) {
        this._poppedUp = false;
        if (!this._statusView.nextStatus()) {
            this.popdown();
        }
        return;
    }

    var state = this._state = this._createState(transition);

    this.setLocation(state.x, state.y);

    this._funcs[transition.type || "next"]();
};

// Protected methods

ZmToast.prototype._createHtml =
function(templateId) {
    var data = { id: this._htmlElId };
    this._createHtmlFromTemplate(templateId || this.TEMPLATE, data);
    this.setZIndex(Dwt.Z_TOAST);
    var el = this.getHtmlElement();

    el.setAttribute('aria-live', 'assertive');
    el.setAttribute('aria-relevant', 'additions');
    el.setAttribute('aria-atomic', true);
};

ZmToast.prototype._createHtmlFromTemplate =
function(templateId, data) {
    DwtComposite.prototype._createHtmlFromTemplate.call(this, templateId, data);
    this._textEl = document.getElementById(data.id+"_text");
    this._iconEl = document.getElementById(data.id+"_icon");
    this._detailEl = document.getElementById(data.id+"_detail");
};

ZmToast.prototype._createState =
function(transition) {
    var state = AjxUtil.createProxy(transition);
    var defaults = ZmToast.DEFAULT_STATE[state.type];
    for (var name in defaults) {
        if (!state[name]) {
            state[name] = defaults[name];
        }
    }

    switch (state.type) {
        case "fade-in":
            this.setOpacity(0);
            this.setLocation(null, 0);
            state.value = state.start;
            break;
        case "fade-out":
        case "fade":
            this.setLocation(null, 0);
            state.value = state.start;
            break;
        case "slide-in":
        case "slide-out":
        case "slide":{
            this.setLocation(null, -36);
            this.setOpacity(100);
            state.value = state.start;
            break;
        }
    }
    return state;
};

// Private methods

ZmToast.prototype.__clear =
function() {
    clearTimeout(this._actionId);
    clearInterval(this._actionId);
    this._actionId = -1;
};

// transition handlers

ZmToast.prototype.__position =
function() {
    var location = this._state.location || "C";
    var containerId = "skin_container_toast"; // Skins may specify an optional element with this id. Toasts will then be placed relative to this element, rather than to the the zshell

    var container = Dwt.byId(containerId) || this.shell.getHtmlElement();
    
    var bsize = Dwt.getSize(container);
    var tsize = this.getSize();

    var x = (bsize.x - tsize.x) / 2;
    var y = (bsize.y - tsize.y) / 2;

    switch (location.toUpperCase()) {
        case 'N': y = 0-tsize.y; break;
        case 'S': y = bsize.y - tsize.y; break;
        case 'E': x = bsize.x - tsize.x; break;
        case 'W': x = 0; break;
        case 'NE': x = bsize.x - tsize.x; y = 0; break;
        case 'NW': x = 0; y = 0; break;
        case 'SE': x = bsize.x - tsize.x; y = bsize.y - tsize.y; break;
        case 'SW': x = 0; y = bsize.y - tsize.y; break;
        case 'C': default: /* nothing to do */ break;
    }

    var offset = Dwt.toWindow(container);
    x += offset.x;
    y += offset.y;
    this.setLocation(x, y);

    this._funcs["next"]();
};

ZmToast.prototype.__show =
function() {
    this.setVisible(true);
    this.setVisibility(true);
    this._funcs["next"]();
};

ZmToast.prototype.__hide =
function() {
    this.setLocation(Dwt.LOC_NOWHERE, Dwt.LOC_NOWHERE);
    if (this._textEl) {
		Dwt.removeChildren(this._textEl);
    }
    if (this._iconEl) {
		Dwt.removeChildren(this._iconEl);
    }
    this._funcs["next"]();
};

ZmToast.prototype.__pause =
function() {
    if (this._dismissed && ZmToast.__mayDismiss(ZmToast.PAUSE)) {
        this._funcs["next"]();
    } else {
        this._pauseTimer = setTimeout(this._funcs["next"], this._state.duration);
    }
};


/**
 * Hold the toast in place until dismiss() is called. If dismiss() was already called before this function (ie. during fade/slide in), continue immediately
 */
ZmToast.prototype.__hold =
function() {
    if (this._dismissed && ZmToast.__mayDismiss(ZmToast.HOLD)!=-1) {
        this._funcs["next"]();
    } else {
        this._held = true;
    }
};

ZmToast.__mayDismiss =
function(state) {
    return AjxUtil.indexOf(ZmToast.DISMISSABLE_STATES, state)!=-1;
};

/**
 * Dismiss (continue) a held or paused toast (Given that ZmToast.DISMISSABLE_STATES agrees). If not yet held or paused, those states will be skipped when they occur
 */
ZmToast.prototype.dismiss =
function() {
    if (!this._dismissed && this._poppedUp) {
        var doDismiss = (this._pauseTimer && ZmToast.__mayDismiss(ZmToast.PAUSE)) || 
            (this._held && ZmToast.__mayDismiss(ZmToast.HOLD));
        if (doDismiss) {
            this._funcs["next"]();
        }
        this._dismissed = true;
        if (this._dismissCallback instanceof AjxCallback) {
            this._dismissCallback.run();
        }
    }
};

ZmToast.prototype.__idle =
function() {
    if (!this._idleTimer) {
        this._idleTimer = new DwtIdleTimer(0, new AjxCallback(this, this.__idleCallback));
    } else {
        this._idleTimer.resurrect(0);
    }
};

ZmToast.prototype.__idleCallback =
function(idle) {
    if (!idle) {
        this.transition();
        this._idleTimer.kill();
    }
};

ZmToast.prototype.__move =
function() {
    // TODO
    this._funcs["next"]();
};

ZmToast.prototype.__fade =
function() {
    var opacity = this._state.value;
    var step = this._state.step;

    var isOver = step > 0 ? opacity >= this._state.end : opacity <= this._state.end;
    if (isOver) {
        opacity = this._state.end;
    }

    this.setOpacity(opacity);

    if (isOver) {
        this.__clear();
        setTimeout(this._funcs["next"], 0);
        return;
    }

    if (this._actionId == -1) {
        var duration = this._state.duration;
        var delta = duration / Math.abs(step);
        this._actionId = setInterval(this._funcs["fade"], delta);
    }

    this._state.value += step;
    this._state.step *= this._state.multiplier;
};

ZmToast.prototype.__slide =
function() {
    var top = this._state.value;
    var step = this._state.step;

    var isOver = step > 0 ? top >= this._state.end : top <= this._state.end;
    if (isOver) {
        top = this._state.end;
    }

    //this.setOpacity(opacity);
    this.setLocation(null, top);
    //el.style.top = top+'px';


    if (isOver) {
        this.__clear();
        setTimeout(this._funcs["next"], 0);
        return;
    }

    if (this._actionId == -1) {
        var duration = this._state.duration;
        var delta = duration / Math.abs(step);
        this._actionId = setInterval(this._funcs["slide"], delta);
    }

    this._state.value += step;
    this._state.step *= this._state.multiplier;
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmOverviewContainer")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * Creates an overview container.
 * @constructor
 * @class
 * Creates a header tree item for an account if mailbox has multiple accounts
 * configured. For each account header, a {@link ZmOverview} is added a child. If mailbox
 * only has one account configured, no account header is created and the
 * {@link ZmOverview} is added as a child of the container.
 *
 * @param	{Hash}	params		a hash of parameters
 * @author Parag Shah
 */
ZmOverviewContainer = function(params) {
	if (arguments.length == 0) { return; }

	params.className = params.className || "ZmOverviewContainer";
	params.id = params.id || ZmId.getOverviewContainerId(params.containerId);
	DwtTree.call(this, params);

	this.setScrollStyle(params.scroll || Dwt.SCROLL_Y);

	this.containerId = params.containerId;
	this._appName = params.appName;
	this._controller = params.controller;
	this._headerItems = {};
	this._overview = {};

	// add listeners
	this.addSelectionListener(new AjxListener(this, this._treeViewListener));
	this.addTreeListener(new AjxListener(this, this._treeListener));
};

ZmOverviewContainer.prototype = new DwtTree;
ZmOverviewContainer.prototype.constructor = ZmOverviewContainer;

ZmOverviewContainer.prototype.toString =
function() {
	return "ZmOverviewContainer";
};

/**
 * Initializes the overview container.
 * 
 * @param	{Hash}	params		a hash of parameters
 */
ZmOverviewContainer.prototype.initialize =
function(params) {
	// overload
};

/**
 * Gets the overview.
 * 
 * @param	{String}	overviewId		the overview id
 * @return	{ZmOverview}		the overview
 */
ZmOverviewContainer.prototype.getOverview =
function(overviewId) {
	return this._overview[overviewId];
};

/**
 * Gets the overviews.
 * 
 * @return	{Array}	an array of {ZmOverview} objects
 */
ZmOverviewContainer.prototype.getOverviews =
function() {
	return this._overview;
};

/**
 * Gets the header item.
 * 
 * 
 */
ZmOverviewContainer.prototype.getHeaderItem =
function() {
	// overload
};

/**
 * Gets the selected overview.
 * 
 * @return	{ZmOverview}	the overview
 */
ZmOverviewContainer.prototype.getSelected =
function() {
	var selected;
	for (var i in this._overview) {
		selected = this._overview[i].getSelected();
		if (selected) {
			return selected;
		}
	}
};

/**
 * Deselects all tree views for each overview in this container
 *
 * @param exception		[ZmOverview]*	If set, this overview is skipped during deselection
 */
ZmOverviewContainer.prototype.deselectAll =
function(exception) {
	DwtTree.prototype.deselectAll.call(this);
	this._deselectAllTreeViews(exception);
};

/**
 * Sets the overview trees.
 * 
 * @param	{Array}		treeIds		an array of tree ids
 */
ZmOverviewContainer.prototype.setOverviewTrees =
function(treeIds) {
	for (var i in this._overview) {
		this._overview[i].set(treeIds);
	}
};

/**
 * Resets the operations.
 * 
 * 
 */
ZmOverviewContainer.prototype.resetOperations =
function(parent, acctId) {
	// overload me
};

ZmOverviewContainer.prototype._treeViewListener =
function(ev) {
	// overload
};

ZmOverviewContainer.prototype._treeListener =
function(ev) {
	// overload
};

ZmOverviewContainer.prototype._initializeActionMenu =
function(account) {
	// overload
};

ZmOverviewContainer.prototype._getActionMenu =
function(ev) {
	if (this._actionMenu instanceof AjxCallback) {
		var callback = this._actionMenu;
		this._actionMenu = callback.run();
	}
	return this._actionMenu;
};

ZmOverviewContainer.prototype._createActionMenu =
function(parent, menuItems, account) {
	// overload
};

ZmOverviewContainer.prototype._actionMenuListener =
function(ev) {
	// overload
};

/**
 * Deselects any selection for each overview this container is managing.
 *
 * @param exception		[ZmOverview]*	If set, deselects all overviews except this one.
 */
ZmOverviewContainer.prototype._deselectAllTreeViews =
function(exception) {
	// make sure none of the other items in the other overviews are selected
	for (var i in this._overview) {
		var overview = this._overview[i];
		if (exception && exception == overview ) { continue; }

		var trees = overview._treeHash;
		for (var j in trees) {
			if (trees[j].getSelectionCount() > 0) {
				trees[j].deselectAll();
				break;
			}
		}
	}
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmAccountOverviewContainer")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 */

/**
 * Creates an overview container for a multi-account mailbox.
 * @class
 * Creates a header tree item for an account if mailbox has multiple accounts
 * configured. For each account header, a {@link ZmOverview} is added a child. If mailbox
 * only has one account configured, no account header is created and the
 * {@link ZmOverview} is added as a child of the container.
 *
 * @param	{Hash}	params		a hash of parameters
 * 
 * @author Parag Shah
 */
ZmAccountOverviewContainer = function(params) {
	if (arguments.length == 0) { return; }

	ZmOverviewContainer.call(this, params);

	this._vFolderTreeItemMap = {};
	this._settingChangeListener = new AjxListener(this, this._handleSettingChange);

	var mouseListener = new AjxListener(this, this._mouseListener);
	this.addListener(DwtEvent.ONMOUSEDOWN, mouseListener);
	this.addListener(DwtEvent.ONMOUSEUP, mouseListener);
};

ZmAccountOverviewContainer.prototype = new ZmOverviewContainer;
ZmAccountOverviewContainer.prototype.constructor = ZmAccountOverviewContainer;


// Consts
ZmAccountOverviewContainer.VIRTUAL_FOLDERS = [
	ZmFolder.ID_INBOX,
	ZmFolder.ID_SENT,
	ZmFolder.ID_DRAFTS,
	ZmFolder.ID_SPAM,
	ZmFolder.ID_OUTBOX,
	ZmFolder.ID_TRASH
];


// Public methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAccountOverviewContainer.prototype.toString =
function() {
	return "ZmAccountOverviewContainer";
};

ZmAccountOverviewContainer.prototype.getHeaderItem =
function(account) {
	return account && this._headerItems[account.id];
};

/**
 * Expands the given account only (collapses all other accounts).
 *
 * @param {ZmAccount}	account		the account to expand
 */
ZmAccountOverviewContainer.prototype.expandAccountOnly =
function(account) {
	if (!account) {
		account = appCtxt.getActiveAccount();
	}

	for (var i in this._headerItems) {
		this._headerItems[i].setExpanded((i == account.id), false, false);
	}
};

ZmAccountOverviewContainer.prototype.getSelected =
function() {
	var selected = ZmOverviewContainer.prototype.getSelected.call(this);

	if (!selected) {
		selected = this.getSelection()[0];
		var account = selected && appCtxt.accountList.getAccount(selected.getData(Dwt.KEY_ID));
		var tree = account && appCtxt.getFolderTree(account);
		return tree && tree.root;
	}

	return selected;
};

/**
 * Initializes the account overview.
 * 
 * @param	{Hash}	params		a hash of parameters
 * 
 */
ZmAccountOverviewContainer.prototype.initialize =
function(params) {

	var header, acct;
	var accounts = appCtxt.accountList.visibleAccounts;
	var showAllMailboxes = (appCtxt.isOffline && this._appName == ZmApp.MAIL && (accounts.length > 2));
	var showBackgroundColor = showAllMailboxes;
	var mainAcct = appCtxt.accountList.mainAccount;
	var origOmit = params.omit;

	for (var i = 0; i < accounts.length; i++) {
		acct = accounts[i];
		// skip the main account in offline mode since we'll add it at the end
		if (appCtxt.isOffline && acct.isMain && this._appName != ZmApp.PREFERENCES) { continue; }
		if (!acct.active) { continue; }

		 params.omit = {};

		if (acct.type == ZmAccount.TYPE_POP) {
			params.omit[ZmFolder.ID_SPAM]   = true;
			params.omit[ZmFolder.ID_OUTBOX] = true;
		}

        if(params.overviewId && !params.isAppOverview) {
            var overviewId = params.overviewId.split(":")[1];
            if(overviewId && (overviewId == "ZmNewOrganizerDialog")){
                params.omit[ZmFolder.ID_DRAFTS] = true;
            }
        }

		this._addAccount(params, acct, showBackgroundColor, null, "account" + i);

		header = this.getHeaderItem(acct);
		if (header) {
			this._setupHeader(header, acct);
		}

		this.updateAccountInfo(acct, true, true);

		showBackgroundColor = !showBackgroundColor;
	}

	// add "All Mailboxes"
	skip = origOmit && origOmit[ZmOrganizer.ID_ALL_MAILBOXES];
	if (showAllMailboxes && !skip) {
		var text = ZmMsg[ZmFolder.MSG_KEY[ZmOrganizer.ID_ALL_MAILBOXES]];
		var hdrText = appCtxt.get(ZmSetting.OFFLINE_ALL_MAILBOXES_TREE_OPEN)
			? text : this._getFolderLabel(ZmOrganizer.ID_INBOX, text);
		var params1 = {
			parent: this,
			text: hdrText,
			imageInfo: "AccountAll"
		};
		var showAllMboxes = appCtxt.get(ZmSetting.OFFLINE_SHOW_ALL_MAILBOXES);
		var allTi = this._allMailboxesTreeHeader = new DwtTreeItem(params1);
		allTi.setData(Dwt.KEY_ID, ZmOrganizer.ID_ALL_MAILBOXES);
		allTi.addClassName("ZmOverviewGlobalInbox");
		allTi._initialize(0, true);
		allTi.setVisible(showAllMboxes);
		allTi.__origText = text;
		if (showAllMboxes) {
			this.highlightAllMboxes();
		}

		var setting = appCtxt.getSettings(mainAcct).getSetting(ZmSetting.OFFLINE_SHOW_ALL_MAILBOXES);
		setting.addChangeListener(this._settingChangeListener);

		var folders = ZmAccountOverviewContainer.VIRTUAL_FOLDERS;
		for (var i = 0; i < folders.length; i++) {
			var folderId = folders[i];

			// add virtual system folders
			params1 = {
				parent: allTi,
				text: this._getFolderLabel(folderId, ZmMsg[ZmFolder.MSG_KEY[folderId]]),
				imageInfo: ZmFolder.ICON[folderId]
			};
			var ti = this._vFolderTreeItemMap[folderId] = new DwtTreeItem(params1);
			ti.setData(Dwt.KEY_ID, folderId);
			ti.addClassName("DwtTreeItemChildDiv");
			ti._initialize(null, true);
			ti.setToolTipContent(appCtxt.accountList.getTooltipForVirtualFolder(folderId));
		}

		// add global searches
		params1 = {
			parent: allTi,
			text: ZmMsg.globalSearches,
			imageInfo: "SearchFolder"
		};
		var searchTi = this._searchTreeHeader = new DwtTreeItem(params1);
		searchTi.addClassName("DwtTreeItemChildDiv");
		searchTi._initialize(null, true);
		searchTi.__isSearch = true;

		var root = appCtxt.getById(ZmOrganizer.ID_ROOT, mainAcct);
		var searchFolders = root.getByType(ZmOrganizer.SEARCH);
		for (var i = 0; i < searchFolders.length; i++) {
			var folder = searchFolders[i];
			if (folder.id != ZmOrganizer.ID_ALL_MAILBOXES &&
				folder.isOfflineGlobalSearch)
			{
				this.addSearchFolder(folder);
			}
		}
		searchTi.setVisible(searchTi.getItemCount() > 0);

		// bug #43734 - set expand/collapse state based on user's pref
		if (appCtxt.get(ZmSetting.OFFLINE_SAVED_SEARCHES_TREE_OPEN)) {
			searchTi.setExpanded(true, null, true);
		}
		if (appCtxt.get(ZmSetting.OFFLINE_ALL_MAILBOXES_TREE_OPEN)) {
			allTi.setExpanded(true, null, true);
		}
	}

	// add the "local" account last
	if (appCtxt.isOffline) {
		var params2 = AjxUtil.hashCopy(params);
		params2.omit = {};

		if (this._appName != ZmApp.PREFERENCES) {
			this._addAccount(params2, mainAcct, showBackgroundColor, "ZmOverviewLocalHeader", "LocalFolders");

			header = this.getHeaderItem(mainAcct);
			header.setExpanded(appCtxt.get(ZmSetting.ACCOUNT_TREE_OPEN, null, mainAcct));

			this.updateAccountInfo(mainAcct, false, true);
		}
		else {
			var params3 = {
				parent: this,
				text: mainAcct.getDisplayName(),
				imageInfo: mainAcct.getIcon()
			};
			var localPrefTi = new DwtTreeItem(params3);
			localPrefTi._initialize(null, true);

			var globalPrefOverviewId = appCtxt.getOverviewId(this.containerId, mainAcct);
			var tv = this._overview[globalPrefOverviewId].getTreeView(ZmOrganizer.PREF_PAGE);
			var importExportTi = tv.getTreeItemById("PREF_PAGE_IMPORT_EXPORT");

			tv.getHeaderItem().removeChild(importExportTi);
			localPrefTi._addItem(importExportTi);
			importExportTi.addClassName("DwtTreeItemChildDiv");
			localPrefTi.setExpanded(true, null, true);
		}
	}

	// add zimlets at the end of all overviews
	var skip = params.omit && params.omit[ZmOrganizer.ID_ZIMLET];

	if (!appCtxt.inStartup && !skip && appCtxt.getZimletMgr().getPanelZimlets().length == 0) {
		skip = true;
	}

	if (!skip) {
		AjxDispatcher.require("Zimlet");
	}

	if (!skip && window[ZmOverviewController.CONTROLLER[ZmOrganizer.ZIMLET]] &&
		this._appName != ZmApp.PREFERENCES)
	{
		var headerLabel = ZmOrganizer.LABEL[ZmOrganizer.ZIMLET];
		var headerDataId = params.overviewId = appCtxt.getOverviewId([this.containerId, headerLabel], null);
		var headerParams = {
			label: ZmMsg[headerLabel],
			icon: "Zimlet",
			dataId: headerDataId,
			className: "ZmOverviewZimletHeader"
		};
		params.overviewTrees = [ZmOrganizer.ZIMLET];

		this._addSection(headerParams, null, params);

		var header = this._headerItems[headerDataId];
		if (header) {
			header.__isZimlet = true;
			header.setExpanded(appCtxt.get(ZmSetting.ZIMLET_TREE_OPEN, null, mainAcct));
		}
	}

	this._initializeActionMenu();
};

/**
 * Adds a search folder.
 * 
 * @param	{ZmFolder}		folder		the folder
 */
ZmAccountOverviewContainer.prototype.addSearchFolder =
function(folder) {
	if (!this._searchTreeHeader) { return; }

	var params = {
		parent: this._searchTreeHeader,
		text: folder.getName(),
		imageInfo: folder.getIcon()
	};
	var ti = new DwtTreeItem(params);
	ti.setData(Dwt.KEY_ID, folder);
	ti._initialize(null, true);

	if (!this._searchTreeHeader.getVisible()) {
		this._searchTreeHeader.setVisible(true);
	}
};

/**
 * Sets/updates the account-level status icon next to account name tree header.
 * This only applies to app-based overview containers (i.e. not dialogs). Also resets the
 * tooltip for the account header tree item.
 *
 * @param {ZmZimbraAccount}	account		the account to update status icon for
 * @param	{Boolean}	updateStatus	if <code>true</code>, update the status
 * @param	{Boolean}	updateTooltip	if <code>true</code>, update the tool tip
 */
ZmAccountOverviewContainer.prototype.updateAccountInfo =
function(account, updateStatus, updateTooltip) {
	// check if appName is a real app (and not a dialog) before setting the account status
	var hi = appCtxt.getApp(this._appName) && this.getHeaderItem(account);
	if (hi) {
		if (updateStatus) {
			var html = (account.status == ZmZimbraAccount.STATUS_RUNNING)
				? ("<img src='/img/animated/ImgSpinner.gif' width=16 height=16 border=0>")
				: (AjxImg.getImageHtml(account.getStatusIcon()));

			if (hi._extraCell) {
				hi._extraCell.innerHTML = (html || "");
			}
            if (appCtxt.isOffline && account.status == ZmZimbraAccount.STATUS_AUTHFAIL) {
                var dialog = appCtxt.getPasswordChangeDialog();
                dialog.popup(account);
            }
		}

		if (updateTooltip || updateStatus) {
			hi.setToolTipContent(account.getToolTip());
		}
	}
};

/**
 * Updates the label.
 * 
 * @param	{ZmOrganizer}	organizer		the organizer
 */
ZmAccountOverviewContainer.prototype.updateLabel =
function(organizer) {
	// update account header if necessary
	if (organizer.nId == ZmOrganizer.ID_INBOX) {
		var acct = organizer.getAccount();
		var header = this.getHeaderItem(acct);
		if (header && !header.getExpanded()) {
			header.setText(this._getAccountHeaderLabel(acct));
		}
	}

	// update virtual folder label
	var ti = this._vFolderTreeItemMap[organizer.nId];
	if (ti) {
		ti.setText(this._getFolderLabel(organizer.nId, organizer.name));
		if (organizer.nId == ZmOrganizer.ID_INBOX &&
			!this._allMailboxesTreeHeader.getExpanded())
		{
			var text = this._getFolderLabel(organizer.nId, this._allMailboxesTreeHeader.__origText);
			this._allMailboxesTreeHeader.setText(text);
		}
	}
};

/**
 * Updates the tool tip.
 * 
 * @param	{String}	folderId		the folder id
 * 
 */
ZmAccountOverviewContainer.prototype.updateTooltip =
function(folderId) {
	var ti = this._allMailboxesTreeHeader && this._vFolderTreeItemMap[folderId];
	if (ti) {
		ti.setToolTipContent(appCtxt.accountList.getTooltipForVirtualFolder(folderId));
	}
};

ZmAccountOverviewContainer.prototype.resetOperations =
function(parent, data) {

	var emptyFolderOp = parent.getOp(ZmOperation.EMPTY_FOLDER);

	if (data instanceof ZmSearchFolder) {
		parent.getOp(ZmOperation.MARK_ALL_READ).setVisible(false);
		emptyFolderOp.setVisible(false);
		parent.getOp(this._newOp).setVisible(false);
		if (appCtxt.isOffline) {
			parent.getOp(ZmOperation.SYNC).setVisible(false);
		}
		parent.getOp(ZmOperation.DELETE).setVisible(true);
		return;
	}

	var acct = appCtxt.accountList.getAccount(data);
	var isAcctType = (acct || data == ZmOrganizer.ID_ALL_MAILBOXES);

	parent.getOp(ZmOperation.MARK_ALL_READ).setVisible(!isAcctType);
	emptyFolderOp.setVisible(false);
	parent.getOp(this._newOp).setVisible(isAcctType && data != ZmOrganizer.ID_ALL_MAILBOXES);
	if (appCtxt.isOffline) {
		parent.getOp(ZmOperation.SYNC).setVisible(isAcctType && (!acct || (acct && !acct.isMain)));
	}
	parent.getOp(ZmOperation.DELETE).setVisible(false);

	if (isAcctType) {
		parent.enable(this._newOp, true); 
		parent.enable(ZmOperation.SYNC, (!acct || (acct && !acct.isMain)));
	} else {
		// reset mark all based on a "friendly" hack ;)
		var markAllEnabled = false;
		if (data != ZmOrganizer.ID_OUTBOX && data != ZmFolder.ID_DRAFTS &&
			this._actionedHeaderItem.getText().indexOf("bold") != -1)
		{
			markAllEnabled = true;
		}
		parent.enable(ZmOperation.MARK_ALL_READ, markAllEnabled);

		// reset empty "folder" based on Trash/Junk
		if (data == ZmOrganizer.ID_TRASH || data == ZmOrganizer.ID_SPAM) {
			var text = (data == ZmOrganizer.ID_TRASH) ? ZmMsg.emptyTrash : ZmMsg.emptyJunk;
			emptyFolderOp.setText(text);
			emptyFolderOp.setVisible(true);
			parent.enable(ZmOperation.EMPTY_FOLDER, !this._isFolderEmpty(data));
		}
	}
};

// HACK - when the overview container for mail is initially created, zimlet
// data has yet to be parsed so we remove the zimlet section after zimlet load
// if there are no panel zimlets.
ZmAccountOverviewContainer.prototype.removeZimletSection =
function() {
	var headerLabel = ZmOrganizer.LABEL[ZmOrganizer.ZIMLET];
	var headerDataId = appCtxt.getOverviewId([this.containerId, headerLabel], null);
	var header = this._headerItems[headerDataId];
	if (header) {
		this.removeChild(header);
	}
};

ZmAccountOverviewContainer.prototype.highlightAllMboxes =
function() {
	this.deselectAll();
	this.setSelection(this._allMailboxesTreeHeader, true);
};

ZmAccountOverviewContainer.prototype._addAccount =
function(params, account, showBackgroundColor, headerClassName, predictableId) {
	params.overviewId = appCtxt.getOverviewId(this.containerId, account);
	params.account = account;	// tree controller might need reference to account

	// only show sections for apps that are supported unless this is prefs app
	var app = appCtxt.getApp(this._appName);
	var isSupported = (!app || (app && appCtxt.get(ZmApp.SETTING[this._appName], null, account)));

	if (this._appName == ZmApp.PREFERENCES || isSupported) {
		var omit = params.omitPerAcct
			? params.omitPerAcct[account.id] : params.omit;

		var headerLabel, headerIcon;
		if (this._appName == ZmApp.PREFERENCES && account.isMain && appCtxt.isOffline) {
			headerLabel = ZmMsg.allAccounts;
			headerIcon = "AccountAll";
		} else {
			headerLabel = account.getDisplayName();
			if (!appCtxt.isFamilyMbox) {
				headerIcon = account.getIcon()
			}
		}

		var headerParams = {
			label: headerLabel,
			icon: headerIcon,
			dataId: account.id,
			className: headerClassName,
			predictableId: predictableId
		};

		this._addSection(headerParams, omit, params, showBackgroundColor);
	}

	var setting = appCtxt.getSettings(account).getSetting(ZmSetting.QUOTA_USED);
	setting.addChangeListener(this._settingChangeListener);
};

ZmAccountOverviewContainer.prototype._addSection =
function(headerParams, omit, overviewParams, showBackgroundColor) {
	// create a top-level section header
	var params = {
		parent: this,
		text:			headerParams.label,
		imageInfo:		headerParams.icon,
		selectable:		overviewParams.selectable,
		className:		headerParams.className,
		id:				this.getHTMLElId() + "__" + (headerParams.predictableId || headerParams.dataId) + "__SECTION"
	};
	var header = this._headerItems[headerParams.dataId] = new DwtTreeItem(params);
	header.setData(Dwt.KEY_ID, headerParams.dataId);
	header.setScrollStyle(Dwt.CLIP);
	header._initialize(null, true, true);
	header.addClassName(showBackgroundColor ? "ZmOverviewSectionHilite" : "ZmOverviewSection");

	// reset some params for child overviews
	overviewParams.id = ZmId.getOverviewId(overviewParams.overviewId);
	overviewParams.parent = header;
	overviewParams.scroll = Dwt.CLIP;
	overviewParams.posStyle = Dwt.STATIC_STYLE;

	// next, create an overview for this account and add it to the account header
	var ov = this._controller._overview[overviewParams.overviewId] = this._overview[overviewParams.overviewId] = new ZmOverview(overviewParams, this._controller);
	header._dndScrollCallback = this._overview._dndScrollCallback,
	header._dndScrollId = this._overview._scrollableContainerId,
	header._addItem(ov, null, true);

	// finally set treeviews for this overview
	var treeIds = overviewParams.overviewTrees || overviewParams.treeIds;
	ov.set(treeIds, omit);
};

ZmAccountOverviewContainer.prototype._setupHeader =
function(header, acct) {
	// always expand header in prefs app, otherwise follow implicit user pref
	if (this._appName == ZmApp.PREFERENCES) {
		header.setExpanded(true, false, true);
		header.enableSelection(false);
	} else {
		var isExpanded = appCtxt.get(ZmSetting.ACCOUNT_TREE_OPEN, null, acct);
		header.setExpanded(isExpanded);
		if (!isExpanded) {
			header.setText(this._getAccountHeaderLabel(acct));
		}
	}

	// add onclick support
	if (header._extraCell) {
		header._extraCell.onclick = AjxCallback.simpleClosure(this._handleStatusClick, this, acct);
	}

	// add DnD support
	var dropTgt = this._controller.getTreeController(ZmOrganizer.FOLDER).getDropTarget();
	var root = ZmOrganizer.getSystemId(ZmOrganizer.ID_ROOT, acct);
	header.setDropTarget(dropTgt);
	header.setData(Dwt.KEY_OBJECT, appCtxt.getById(root));
};

/**
 * Iterates all visible account and checks whether folder for each account is
 * empty. Used by "All Mailboxes" to determine whether a virtual folder needs
 * to be disabled or not when right-clicked.
 *
 * @param folderId		[String]		Normalized folder ID (should not be fully qualified)
 */
ZmAccountOverviewContainer.prototype._isFolderEmpty =
function(folderId) {
	var accounts = appCtxt.accountList.visibleAccounts;
	for (var i = 0; i < accounts.length; i++) {
		var folder = appCtxt.getById(ZmOrganizer.getSystemId(folderId, accounts[i]));
		if (folder && folder.numTotal > 0) {
			return false;
		}
	}

	return true;
};

ZmAccountOverviewContainer.prototype._syncAccount =
function(dialog, account) {
	dialog.popdown();
	account.sync();
};

ZmAccountOverviewContainer.prototype._treeViewListener =
function(ev) {
	if (ev.detail != DwtTree.ITEM_ACTIONED &&
		ev.detail != DwtTree.ITEM_SELECTED &&
		ev.detail != DwtTree.ITEM_DBL_CLICKED)
	{
		return;
	}

	var item = this._actionedHeaderItem = ev.item;

	// do nothing if zimlet/search is clicked
	if (item && (item.__isZimlet || item.__isSearch)) { return; }

	var data = item && item.getData(Dwt.KEY_ID);

	if (ev.detail == DwtTree.ITEM_ACTIONED && appCtxt.getApp(this._appName)) {	// right click
		var actionMenu = this._getActionMenu(data);
		if (actionMenu) {
			this.resetOperations(actionMenu, data);
			actionMenu.popup(0, ev.docX, ev.docY);
		}
	}
	else if ((ev.detail == DwtTree.ITEM_SELECTED) && item) {					// left click
		// if calendar/prefs app, do nothing
		if (this._appName == ZmApp.CALENDAR ||
			this._appName == ZmApp.PREFERENCES)
		{
			return;
		}

		this._deselectAllTreeViews();

		// this avoids processing clicks in dialogs etc.
		if (!ZmApp.NAME[this._appName]) { return; }

		// if an account header item was clicked, run the default search for it
		if (data) {
			var sc = appCtxt.getSearchController();

			var account = appCtxt.accountList.getAccount(data);
			if (account) {
				// bug 41196 - turn off new mail notifier if inactive account header clicked
				if (appCtxt.isOffline && account.inNewMailMode) {
					account.inNewMailMode = false;
					var allContainers = appCtxt.getOverviewController()._overviewContainer;
					for (var i in allContainers) {
						allContainers[i].updateAccountInfo(account, true, true);
					}
				}

				// don't process click if user clicked on error status icon
				if ((ev.target.parentNode == ev.item._extraCell) && account.isError()) {
					return;
				}

				sc.searchAllAccounts = false;
				appCtxt.accountList.setActiveAccount(account);

				if (appCtxt.isOffline && account.hasNotSynced() && !account.__syncAsked) {
					account.__syncAsked = true;

					var dialog = appCtxt.getYesNoMsgDialog();
					dialog.registerCallback(DwtDialog.YES_BUTTON, this._syncAccount, this, [dialog, account]);
					dialog.setMessage(ZmMsg.neverSyncedAsk, DwtMessageDialog.INFO_STYLE);
					dialog.popup();
				}

				var fid = ZmOrganizer.DEFAULT_FOLDER[ZmApp.ORGANIZER[this._appName]];
				var folder = appCtxt.getById(ZmOrganizer.getSystemId(fid, account));

				// briefcase is not a ZmFolder so let's skip for now
				if (!(folder instanceof ZmFolder)) { return; }

				var defaultSortBy = (this._appName == ZmApp.TASKS)
					? ZmSearch.DUE_DATE_DESC : ZmSearch.DATE_DESC;

				var params = {
					query: folder.createQuery(),
					getHtml: appCtxt.get(ZmSetting.VIEW_AS_HTML),
					searchFor: (ZmApp.DEFAULT_SEARCH[this._appName]),
					sortBy: ((sc.currentSearch && folder.nId == sc.currentSearch.folderId) ? null : defaultSortBy),
					accountName: (account && account.name),
					noUpdateOverview: true
				};
			} else {
				var main = appCtxt.accountList.mainAccount;
				sc.resetSearchAllAccounts();
				sc.searchAllAccounts = true;

				if (data instanceof ZmSearchFolder) {
					params = {
						searchAllAccounts: true,
						accountName: main.name,
						getHtml: appCtxt.get(ZmSetting.VIEW_AS_HTML),
						noUpdateOverview: true
					};
					sc.redoSearch(data.search, false, params);
					return;
				}
				if (data == ZmOrganizer.ID_ALL_MAILBOXES) {
					data = ZmFolder.ID_INBOX;
				}

				params = {
					queryHint: appCtxt.accountList.generateQuery(data),
					folderId: null,
					getHtml: appCtxt.get(ZmSetting.VIEW_AS_HTML, null, main),
					searchFor: (ZmApp.DEFAULT_SEARCH[this._appName]),
					sortBy: ZmSearch.DATE_DESC,
					accountName: main.name,
					noUpdateOverview: true
				};
			}

			sc.search(params);
		}
	} else {																	// double click
		// handle double click?
	}
};

ZmAccountOverviewContainer.prototype._treeListener =
function(ev) {
	if (ev.detail != DwtTree.ITEM_COLLAPSED &&
		ev.detail != DwtTree.ITEM_EXPANDED)
	{
		return;
	}

	var header = ev.item;
	var expanded = ev.detail == DwtTree.ITEM_EXPANDED;

	var acct;
	if (header) {
		if (header.__isSearch) {
			appCtxt.set(ZmSetting.OFFLINE_SAVED_SEARCHES_TREE_OPEN, expanded);
			return;
		}

		var data = header.getData(Dwt.KEY_ID);
		if (data == ZmOrganizer.ID_ALL_MAILBOXES) {
			var text = expanded
				? header.__origText
				: this._getFolderLabel(ZmOrganizer.ID_INBOX, header.__origText);
			header.setText(text);

			// bug #43734 - remember expand/collapse state
			appCtxt.set(ZmSetting.OFFLINE_ALL_MAILBOXES_TREE_OPEN, expanded);

			return;
		}

		acct = header.__isZimlet
			? appCtxt.accountList.mainAccount
			: appCtxt.accountList.getAccount(data);
	}

	if (acct && appCtxt.getCurrentAppName() != ZmApp.PREFERENCES) {
		if (!appCtxt.inStartup) {
            // set account as active when account tree is opened/closed
            appCtxt.accountList.setActiveAccount(acct);
			appCtxt.set(ZmSetting.ACCOUNT_TREE_OPEN, expanded, null, null, null, acct);
		}

		if (!header.__isZimlet) {
			var text = expanded
				? acct.getDisplayName()
				: this._getAccountHeaderLabel(acct);
			header.setText(text);
		}
	}
};

ZmAccountOverviewContainer.prototype._mouseListener =
function(ev) {
	return !Dwt.ffScrollbarCheck(ev);
};

ZmAccountOverviewContainer.prototype._handleSettingChange =
function(ev) {
	if (ev.type != ZmEvent.S_SETTING) { return; }

	var setting = ev.source;

	if (setting.id == ZmSetting.OFFLINE_SHOW_ALL_MAILBOXES) {
		var isVisible = setting.getValue();
		this._allMailboxesTreeHeader.setVisible(isVisible);
		if (!isVisible) {
			if (appCtxt.getActiveAccount().isMain) {
				appCtxt.accountList.setActiveAccount(appCtxt.accountList.defaultAccount);
			}
			appCtxt.getSearchController().searchAllAccounts = false;
			appCtxt.getApp(ZmApp.MAIL).mailSearch();
		} else {
			this._deselect(this._allMailboxesTreeHeader);
		}
	}
	else if (setting.id == ZmSetting.QUOTA_USED) {
		this.updateAccountInfo(ev.getDetails().account, false, true);
	}
};

ZmAccountOverviewContainer.prototype._getAccountHeaderLabel =
function(acct, header) {
	var inboxId = (this._appName == ZmApp.MAIL)
		? ZmOrganizer.getSystemId(ZmOrganizer.ID_INBOX, acct, true) : null;
	var inbox = inboxId && appCtxt.getById(inboxId);
	if (inbox && inbox.numUnread > 0) {
		var name = AjxMessageFormat.format(ZmMsg.folderUnread, [acct.getDisplayName(), inbox.numUnread]);
		return (["<span style='font-weight:bold;'>", name, "</span>"].join(""));
	}

	return acct.getDisplayName();
};

ZmAccountOverviewContainer.prototype._getFolderLabel =
function(folderId, label) {
	var checkUnread = (folderId != ZmFolder.ID_DRAFTS && folderId != ZmFolder.ID_OUTBOX);
	var count = appCtxt.accountList.getItemCount(folderId, checkUnread);
	if (count > 0) {
		var name = AjxMessageFormat.format(ZmMsg.folderUnread, [label, count]);
		return (["<span style='font-weight:bold;'>", name, "</span>"].join(""));
	}

	return label;
};

ZmAccountOverviewContainer.prototype._initializeActionMenu =
function() {
	if (!this._actionMenu) {
		var orgType = ZmApp.ORGANIZER[this._appName];
		this._newOp = ZmOrganizer.NEW_OP[orgType];

		var ops = [this._newOp];
		if (appCtxt.isOffline) {
			ops.push(ZmOperation.SYNC);
		}
		ops.push(ZmOperation.MARK_ALL_READ,
				ZmOperation.EMPTY_FOLDER,
				ZmOperation.DELETE);

		this._actionMenu = new AjxCallback(this, this._createActionMenu, [ops]);
	}
};

ZmAccountOverviewContainer.prototype._createActionMenu =
function(menuItems) {
	var listener = new AjxListener(this, this._actionMenuListener);
	var actionMenu = new ZmActionMenu({parent:appCtxt.getShell(), menuItems:menuItems});
	menuItems = actionMenu.opList;
	for (var i = 0; i < menuItems.length; i++) {
		var mi = actionMenu.getItem(i);
		var op = menuItems[i];
		if (op == ZmOperation.SYNC) {
			mi.setText(ZmMsg.sendReceive);
		}
		actionMenu.addSelectionListener(op, listener);
	}

	actionMenu.addPopdownListener(new AjxListener(this, this._menuPopdownActionListener));

	return actionMenu;
};

ZmAccountOverviewContainer.prototype._menuPopdownActionListener =
function() {
	this._actionedHeaderItem._setActioned(false);
};

ZmAccountOverviewContainer.prototype._actionMenuListener =
function(ev) {
	var opId = ev.item.getData(ZmOperation.KEY_ID);
	var data = this._actionedHeaderItem.getData(Dwt.KEY_ID);

	if (opId == this._newOp) {
		var treeId = ZmApp.ORGANIZER[this._appName];
		var tc = this._controller.getTreeController(treeId, true);
		if (tc) {
			tc._actionedOrganizer = null;
			var account = appCtxt.accountList.getAccount(data);
			tc._actionedOrganizer = appCtxt.getFolderTree(account).root;
			tc._newListener(ev, account);
		}
	}
	else if (opId == ZmOperation.SYNC) {
		if (data == ZmOrganizer.ID_ALL_MAILBOXES) {
			appCtxt.accountList.syncAll();
		} else {
			var account = appCtxt.accountList.getAccount(data);
			if (account) {
				account.sync();
			}
		}
	}
	else if (opId == ZmOperation.MARK_ALL_READ) {
		this._doAction(data, opId);
	}
	else if (opId == ZmOperation.EMPTY_FOLDER) {
		this._confirmEmptyAction(data, opId);
	}
	else if (opId == ZmOperation.DELETE) {
		data._delete();
		var parent = this._actionedHeaderItem.parent;
		parent.removeChild(this._actionedHeaderItem); // HACK: just nuke it
		parent.setVisible(parent.getItemCount() > 0);
	}
};

ZmAccountOverviewContainer.prototype._confirmEmptyAction =
function(data, opId) {
	var dialog = appCtxt.getOkCancelMsgDialog();
	dialog.reset();
	dialog.registerCallback(DwtDialog.OK_BUTTON, this._emptyCallback, this, [dialog, data, opId]);

	var msg = (data == ZmFolder.ID_TRASH)
		? ZmMsg.confirmEmptyTrashFolder
		: (AjxMessageFormat.format(ZmMsg.confirmEmptyFolder, ZmMsg[ZmFolder.MSG_KEY[data]]));

	dialog.setMessage(msg, DwtMessageDialog.WARNING_STYLE);
	dialog.popup();
};

ZmAccountOverviewContainer.prototype._emptyCallback =
function(dialog, folderId, opId) {
	dialog.popdown();
	this._doAction(folderId, opId);
};

ZmAccountOverviewContainer.prototype._doAction =
function(folderId, opId) {
	var bc = new ZmBatchCommand(true, appCtxt.accountList.mainAccount.name);

	var accounts = appCtxt.accountList.visibleAccounts;
	for (var i = 0; i < accounts.length; i++) {
		var account = accounts[i];
		if (account.isMain) { continue; }

		var fid = ZmOrganizer.getSystemId(folderId, account);
		var folder = appCtxt.getById(fid);
		if (folder) {
			if (opId == ZmOperation.MARK_ALL_READ) {
				folder.markAllRead(bc);
			} else {
				folder.empty(null, bc);
			}
			bc.curId++;
		}
	}

	bc.run();
	if (appCtxt.isOffline) {
		appCtxt.getApp(ZmApp.MAIL).clearNewMailBadge();
	}
};

ZmAccountOverviewContainer.prototype._handleStatusClick =
function(account, ev) {

    if(!account.isError()) {
        return;
    } else if (appCtxt.isOffline && (account.status == ZmZimbraAccount.STATUS_AUTHFAIL)) {
        var dialog = appCtxt.getPasswordChangeDialog();
        dialog.popup(account);
    } else {
        account.showErrorMessage();
    }
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmOverview")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines an overview, which holds tree views.
 *
 */

/**
 * @class
 * Creates an overview. An overview is a {@link DwtComposite} that holds tree views.
 * 
 * @author Conrad Damon
 *
 * @param {Hash}	params 				a hash of parameters
 * @param	{String}	params.id 	the id for the HTML element
 * @param	{String}	params.overviewId 	the overview id
 * @param	{String}	params.containerId 	the overview container id (multi-account)
 * @param	{Array}	params.treeIds an array of organizer types that may be displayed in this overview
 * @param	{ZmZimbraAccount}	params.account		the account this overview belongs to
 * @param	{DwtControl}	params.parent			the containing widget
 * @param	{String}	params.overviewClass		the class name for overview DIV
 * @param	{constant}	params.posStyle				the positioning style for overview DIV
 * @param	{constant}	params.scroll				the scrolling style for overview DIV
 * @param	{Boolean}	params.selectionSupported <code>true</code> left-click selection is supported
 * @param	{Boolean}	params.actionSupported		<code>true</code> if right-click action menus are supported
 * @param	{Boolean}	params.dndSupported			<code>true</code> if drag-and-drop is supported
 * @param	{String}	params.headerClass			the class name for header item
 * @param	{Boolean}	params.showUnread			if <code>true</code>, unread counts will be shown
 * @param	{Boolean}	params.showNewButtons		if <code>true</code>, tree headers may have buttons for creating new organizers
 * @param	{constant}	params.treeStyle			the default display style for tree views
 * @param	{Boolean}	params.isCheckedByDefault	the default state for "checked" display style
 * @param	{Boolean}	params.noTooltips			if <code>true</code>, do not show toolt ips for tree items
 * @param	{Boolean}	params.skipImplicit			if <code>true</code>, do not save implicit prefs of expanded/collapsed node status for this overview (see ZmDialog.prototype._setOverview)
 * @param	{Boolean}	params.dynamicWidth			if <code>true</code>, the width is dynamic, i.e. the width is auto instead of fixed. Used for ZmDolderChooser so far.
 * @param {ZmOverviewController}	controller			the overview controller
 * 
 * @extends	DwtComposite
 */
ZmOverview = function(params, controller) {

	var overviewClass = params.overviewClass ? params.overviewClass : "ZmOverview";
	params.id = params.id || ZmId.getOverviewId(params.overviewId);
	DwtComposite.call(this, {parent:params.parent, className:overviewClass, posStyle:params.posStyle, id:params.id});

	this._controller = controller;

	this.setScrollStyle(params.scroll || Dwt.SCROLL_Y);

	this.overviewId			= params.overviewId;
	this.containerId		= params.containerId;
	this.account			= params.account;
	this.selectionSupported	= params.selectionSupported;
	this.actionSupported	= params.actionSupported;
	this.dynamicWidth		= params.dynamicWidth;
	this.dndSupported		= params.dndSupported;
	this.headerClass		= params.headerClass;
	this.showUnread			= params.showUnread;
	this.showNewButtons		= params.showNewButtons;
	this.treeStyle			= params.treeStyle;
	this.isCheckedByDefault = params.isCheckedByDefault;
	this.noTooltips			= params.noTooltips;
	this.isAppOverview		= params.isAppOverview;
	this.skipImplicit 		= params.skipImplicit;
	this.appName            = params.appName;

	this._treeIds			= [];
	this._treeHash			= {};
	this._treeParents		= {};

	// Create a parent div for each overview tree.
	var doc = document;
	var element = this.getHtmlElement();
	if (params.treeIds) {
		for (var i = 0, count = params.treeIds.length; i < count; i++) {
			var div = doc.createElement("DIV");
			var treeId = params.treeIds[i];
			this._treeParents[treeId] = div.id = [this.overviewId, treeId].join("-parent-");
			element.appendChild(div);
		}
	}

	if (this.dndSupported) {
		this._scrollableContainerId = this.containerId || this.overviewId;
		var container = this.containerId ? document.getElementById(this.containerId) : this.getHtmlElement();
		var params = {container:container, threshold:15, amount:5, interval:10, id:this._scrollableContainerId};
		this._dndScrollCallback = new AjxCallback(null, DwtControl._dndScrollCallback, [params]);
	}
	if (this.overviewId === 'main_Options') {
		this.setAttribute('aria-label', ZmMsg.preferences);
	} else {
		this.setAttribute('aria-label', ZmMsg.overviewLabel);
	}

    // Let overview be a single tab stop, then manage focus among items using arrow keys
    this.tabGroupMember = this;
};

ZmOverview.prototype = new DwtComposite;
ZmOverview.prototype.constructor = ZmOverview;

ZmOverview.prototype.isZmOverview = true;
ZmOverview.prototype.toString = function() { return "ZmOverview"; };

ZmOverview.prototype.role = "navigation";


/**
 * Gets the parent element for the given tree id.
 * 
 * @param	{String}	treeId		the tree id
 * @return	{Object}	the tree parent element
 */
ZmOverview.prototype.getTreeParent =
function(treeId) {
	return this._treeParents[treeId];
};

/**
 * Displays the given list of tree views in this overview.
 *
 * @param {Array}	treeIds		an array of organizer ids
 * @param {Hash}	omit		the hash of organizer ids to ignore
 */
ZmOverview.prototype.set =
function(treeIds, omit) {
	if (treeIds && treeIds.length) {
		for (var i = 0; i < treeIds.length; i++) {
			this.setTreeView(treeIds[i], omit);
		}
	}
};

/**
 * Sets the given tree view. Its tree controller is responsible for using the appropriate
 * data tree to populate the tree view. The tree controller will be lazily created if
 * necessary. The tree view is cleared before it is set. The tree view inherits options
 * from this overview.
 * 
 * @param {String}	treeId	the organizer ID
 * @param {Hash}	omit	a hash of organizer ids to ignore
 */
ZmOverview.prototype.setTreeView = function(treeId, omit) {

	if (!appCtxt.checkPrecondition(ZmOrganizer.PRECONDITION[treeId])) {
		return;
	}

	AjxDispatcher.require(ZmOrganizer.ORG_PACKAGE[treeId]);
	var treeController = this._controller.getTreeController(treeId);
	if (!treeController) { return; }
	if (this._treeHash[treeId]) {
		treeController.clearTreeView(this.overviewId);
	} else {
		this._treeIds.push(treeId);
	}
	var params = {
		overviewId:		this.overviewId,
		omit:			omit,
		showUnread:		this.showUnread,
		account:		this.account
	};
	this._treeHash[treeId] = treeController.show(params); // render tree view
};

ZmOverview.prototype.clearChangeListener = function(treeIds) {
	// Added for the attachMail zimlet, operating in a child window.  This clears the listeners added to
	// the parent window trees (which causes problems in IE when the child window closes).  See Bugs
	// 99453 and 99913
	for (var i = 0; i < treeIds.length; i++) {
		var treeController = this._controller.getTreeController(treeIds[i]);
		var changeListener = treeController._getTreeChangeListener();
		if (changeListener) {
			var folderTree = appCtxt.getFolderTree();
			if (folderTree) {
				folderTree.removeChangeListener(changeListener);
			}
		}
	}
}

/**
 * Gets the tree view.
 * 
 * @param	{String}	treeId		the tree id
 * @return	{Object}	the tree view
 */
ZmOverview.prototype.getTreeView =
function(treeId) {
	return this._treeHash[treeId];
};

/**
 * Gets the tree views.
 * 
 * @return	{Array}	an array of tree ids
 */
ZmOverview.prototype.getTreeViews =
function() {
	return this._treeIds;
};

/**
 * Searches the tree views for the tree item whose data object has the given ID and type.
 * 
 * @param {int}	id			the id to look for
 * @param {constant}	type			the item must also have this type
 * @return	{Object}	the item or <code>null</code> if not found
 */
ZmOverview.prototype.getTreeItemById =
function(id, type) {
	if (!id) { return null; }
	for (var i = 0; i < this._treeIds.length; i++) {
		var treeView = this._treeHash[this._treeIds[i]];
		if (treeView) {
			var item = treeView.getTreeItemById && treeView.getTreeItemById(id);
			if (item && (!type || (this._treeIds[i] == type))) {
				return item;
			}
		}
	}
	return null;
};

/**
* Returns the first selected item within this overview.
* 
* @param	{Boolean}	typeOnly	if <code>true</code>, return the type only
* @return	{Object}	the item (or type if <code>typeOnly</code>) or <code>null</code> if not found
*/
ZmOverview.prototype.getSelected =
function(typeOnly) {
	for (var i = 0; i < this._treeIds.length; i++) {
		var treeView = this._treeHash[this._treeIds[i]];
		if (treeView) {
			var item = treeView.getSelected();
			if (item) {
				return typeOnly ? treeView.type : item;
			} // otherwise continue with other treeviews to look for selected item
		}
	}
	return null;
};

ZmOverview.prototype.deselectAllTreeViews =
function() {
	for (var i = 0; i < this._treeIds.length; i++) {
		var treeView = this._treeHash[this._treeIds[i]];
		if (treeView) {
			treeView.deselectAll();
		}
	}
};


/**
 * Selects the item with the given ID within the given tree in this overview.
 *
 * @param {String}	id	the item id
 * @param {constant}	type	the tree type
 */
ZmOverview.prototype.setSelected =
function(id, type) {
	var ti, treeView;
	if (type) {
		treeView = this._treeHash[type];
		ti = treeView && treeView.getTreeItemById(id);
	} else {
		for (var type in this._treeHash) {
			treeView = this._treeHash[type];
			ti = treeView && treeView.getTreeItemById(id);
			if (ti) { break; }
		}
	}

	if (ti && (this._selectedTreeItem != ti)) {
		treeView.setSelected(id, true, true);
	}
	this.itemSelected(ti);
};

/**
 * Given a tree item, de-selects all items in the overview's
 * other tree views, enforcing single selection within the overview.
 * Passing a null argument will clear selection in all tree views.
 *
 * @param {DwtTreeItem}	treeItem		the tree item
 */
ZmOverview.prototype.itemSelected =
function(treeItem) {
	if (appCtxt.multiAccounts && treeItem) {
		var name = this.overviewId.substring(0, this.overviewId.indexOf(":"));
		var container = this._controller.getOverviewContainer(name);
		if (container) {
			container.deselectAll(this);
		}
	}

	if (this._selectedTreeItem && (this._selectedTreeItem._tree != (treeItem && treeItem._tree))) {
		this._selectedTreeItem._tree.deselectAll();
	}

	this._selectedTreeItem = treeItem;
};

/**
 * Clears the tree views.
 */
ZmOverview.prototype.clear =
function() {
	for (var i = 0; i < this._treeIds.length; i++) {
		var treeId = this._treeIds[i];
		if (this._treeHash[treeId]) {
			var treeController = this._controller.getTreeController(treeId);
			treeController.clearTreeView(this.overviewId);
			delete this._treeHash[treeId];
		}
	}
};

ZmOverview.prototype.clearSelection =
function() {
	if (this._selectedTreeItem) {
		this._selectedTreeItem._tree.deselectAll();
	}
};

/**
 * @private
 */
ZmOverview.prototype._initialize =
function() {
	// do nothing. 
	// - called by DwtTreeItem b/c it thinks its adding another tree item
};

/**
 * @private
 */
ZmOverview.prototype.focus = function() {

	var item = this._selectedTreeItem;
	if (!item) {
		var tree = this._treeHash[this._treeIds[0]];
		if (tree) {
			item = tree._getNextTreeItem(true);
		}
	}

    if (item) {
        item.focus();
        item._tree.setSelection(item, false, true);
        return item;
    }
};

/**
 * @private
 */
ZmOverview.prototype.blur = function() {

	var item = this._selectedTreeItem;
	if (item) {
		item._blur();
	}
};

/**
 * Returns the next/previous selectable tree item within this overview, starting with the
 * tree immediately after/before the given one. Used to handle tree item selection that
 * spans trees.
 *
 * @param {Boolean}	next		if <code>true</code>, look for next item as opposed to previous item
 * @param {ZmTreeView}	tree		the tree that we are just leaving
 *
 * @private
 */
ZmOverview.prototype._getNextTreeItem =
function(next, tree) {

	for (var i = 0; i < this._treeIds.length; i++) {
		if (this._treeHash[this._treeIds[i]] == tree) {
			break;
		}
	}

	var nextItem = null;
	var idx = next ? i + 1 : i - 1;
	tree = this._treeHash[this._treeIds[idx]];
	while (tree) {
		nextItem = DwtTree.prototype._getNextTreeItem.call(tree, next);
		if (nextItem) {
			break;
		}
		idx = next ? idx + 1 : idx - 1;
		tree = this._treeHash[this._treeIds[idx]];
	}

	return nextItem;
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmAppIframeView")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * A generic iframe view that can be associated with an app tab. One use is to display upsell content from an external
 * URL if the user does not currently have the app enabled.
 * @class
 * This class displays an external URL in an iframe.
 *
 * @extends		DwtControl
 *
 * @author Conrad Damon
 */
ZmAppIframeView = function(params) {

	if (arguments.length === 0) {
		return;
	}

	DwtControl.call(this, {
		parent:     appCtxt.getShell(),
		posStyle:   Dwt.ABSOLUTE_STYLE,
		className:  'ZmAppIframeView'
	});

	this._createFrame(params);
};

ZmAppIframeView.prototype = new DwtControl;
ZmAppIframeView.prototype.constructor = ZmAppIframeView;

ZmAppIframeView.prototype.isZmAppIframeView = true;
ZmAppIframeView.prototype.toString = function() { return "ZmAppIframeView"; };

ZmAppIframeView.prototype._createFrame = function(params) {

	params = params || {};

	var app = this._appName = params.appName,
		iframeUrl = appCtxt.get(ZmApp.UPSELL_URL[app]),
		htmlArr = [],
		idx = 0;

	var	iframeId = this._iframeId = this._getIframeId();

	htmlArr[idx++] = "<iframe id='" + iframeId + "' width='100%' height='100%' frameborder='0' src='";
	htmlArr[idx++] = iframeUrl;
	htmlArr[idx++] = "'>";
	this.setContent(htmlArr.join(""));
};

ZmAppIframeView.prototype._getIframeId = function() {
	return 'iframe_' + this.getHTMLElId();
};

ZmAppIframeView.prototype.activate = function(active) {};

ZmAppIframeView.prototype.runRefresh = function() {};

ZmAppIframeView.prototype.setBounds =
function(x, y, width, height, showToolbar) {
    var deltaHeight = 0;
    if(!showToolbar) {
        deltaHeight = this._getToolbarHeight();
    }
	DwtControl.prototype.setBounds.call(this, x, y - deltaHeight, width, height + deltaHeight);
	var id = "iframe_" + this.getHTMLElId();
	var iframe = document.getElementById(id);
	if(iframe) {
    	iframe.width = width;
    	iframe.height = height + deltaHeight;
	}
};

ZmAppIframeView.prototype._getToolbarHeight =
function() {
    var topToolbar = appCtxt.getAppViewMgr().getViewComponent(ZmAppViewMgr.C_TOOLBAR_TOP);
	if (topToolbar) {
		var sz = topToolbar.getSize();
		var height = sz.y ? sz.y : topToolbar.getHtmlElement().clientHeight;
		return height;
	}
	return 0;
};

ZmAppIframeView.prototype.getTitle = function() {
	return [ ZmMsg.zimbraTitle, appCtxt.getApp(this._appName).getDisplayName() ].join(": ");
};
}
if (AjxPackage.define("zimbraMail.share.view.ZmCommunityView")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Sets up an iframe which displays content from Community (activity stream, notifications, chat).
 * @class
 * This class displays Community content in an iframe.
 *
 * @extends		ZmAppIframeView
 *
 * @author Conrad Damon
 */
ZmCommunityView = function(params) {
	ZmAppIframeView.apply(this, arguments);
	this._createFrame(params);
	this._setupMessageHandling();
};

ZmCommunityView.prototype = new ZmAppIframeView;
ZmCommunityView.prototype.constructor = ZmCommunityView;

ZmCommunityView.prototype.isZmCommunityView = true;
ZmCommunityView.prototype.toString = function() { return "ZmCommunityView"; };

ZmCommunityView.prototype._setupMessageHandling = function(params) {

	// Set up to handle messages sent to us via postMessage()
	var iframe = document.getElementById(this._iframeId);
	if (iframe) {
		var callback = ZmCommunityView.handleMessage.bind(null, this);
		if (window.addEventListener) {
			window.addEventListener('message', callback, false);
		}
		else if (window.attachEvent) {
			window.attachEvent('onmessage', callback);
		}
	}
};

ZmCommunityView.prototype._getIframeId = function() {
	return 'fragment-41812_iframe';     // this is what Community is expecting
};

/**
 * If Community tells us there is new content, turn the tab orange if it's not the current tab,
 * and refresh the content.
 *
 * @param view
 * @param event
 */
ZmCommunityView.handleMessage = function(view, event) {

	var iframe = document.getElementById(view._iframeId);
	if (iframe && event.source === iframe.contentWindow) {
		var data = AjxStringUtil.parseQueryString(event.data || '');
		var isUnread = (data.unread && data.unread.toLowerCase() === 'true');
		if (data.type === 'community-notification' && isUnread) {
			appCtxt.getApp(view._appName).startAlert();
			view.getUpdates();
		}
	}
};

/**
 * Sends a message to Community to refresh the content.
 */
ZmCommunityView.prototype.getUpdates = function() {

	var iframe = document.getElementById(this._iframeId)
	if (iframe) {
		iframe.contentWindow.postMessage('type=community-update', '*');
	}
};

// Called when user switches to this tab.
ZmCommunityView.prototype.activate = function(active) {
	if (active) {
		this.getUpdates();
	}
};

// Code to run when the user clicks the refresh (circle-arrow) button. Shouldn't really be
// needed, but doesn't hurt to have it.
ZmCommunityView.prototype.runRefresh = function() {
	this.getUpdates();
};
}

if (AjxPackage.define("zimbraMail.share.controller.ZmController")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines an application controller.
 *
 */

/**
 * Creates a controller. 
 * @class
 * This class represents an application controller.
 * 
 * @param	{DwtShell}		container		the application container
 * @param	{ZmApp}			app				the application
 * @param	{constant}		type			type of controller (typically a view type)				
 * @param	{string}		sessionId		the session id
 */
ZmController = function(container, app, type, sessionId) {

	if (arguments.length == 0) { return; }

	this.setCurrentViewType(this.getDefaultViewType());
	this.setCurrentViewId(this.getDefaultViewType());
	if (sessionId) {
		this.setSessionId(sessionId, type);
	}
	
	this._container = container;
	this._app = app;
		
	this._shell = appCtxt.getShell();
	this._appViews = {};
	
	this._authenticating = false;
	this.isHidden = (sessionId == ZmApp.HIDDEN_SESSION);
	this._elementsToHide = null;
};

ZmController.prototype.isZmController = true;
ZmController.prototype.toString = function() { return "ZmController"; };


ZmController.SESSION_ID_SEP = "-";

// Abstract methods

ZmController.prototype._setView = function() {};

/**
 * Returns the default view type
 */
ZmController.getDefaultViewType = function() {};	// needed by ZmApp::getSessionController
ZmController.prototype.getDefaultViewType = function() {};

// _defaultView is DEPRECATED in 8.0
ZmController.prototype._defaultView = ZmController.prototype.getDefaultViewType;



// Public methods

/**
 * Gets the session ID.
 * 
 * @return	{string}	the session ID
 */
ZmController.prototype.getSessionId =
function() {
	return this._sessionId;
};

/**
 * Sets the session id, view id, and tab id (using the type and session id).
 * Controller for a view that shows up in a tab within the app chooser bar.
 * Examples include compose, send confirmation, and msg view.
 *
 * @param {string}						sessionId					the session id
 * @param {string}						type						the type
 * @param {ZmSearchResultsController}	searchResultsController		owning controller
 */
ZmController.prototype.setSessionId =
function(sessionId, type) {

	this._sessionId = sessionId;
	if (type) {
		this.setCurrentViewType(type);
		this.setCurrentViewId(sessionId ? [type, sessionId].join(ZmController.SESSION_ID_SEP) : type);
		this.tabId = sessionId ? ["tab", this.getCurrentViewId()].join("_") : "";
	}
	
	// this.sessionId and this.viewId are DEPRECATED in 8.0;
	// use getSessionId() and getCurrentViewId() instead
	this.sessionId = this._sessionId;
	this.viewId = this.getCurrentViewId();
};

/**
 * Gets the current view type.
 * 
 * @return	{constant}			the view type
 */
ZmController.prototype.getCurrentViewType =
function(viewType) {
	return this._currentViewType;
};
// _getViewType is DEPRECATED in 8.0
ZmController.prototype._getViewType = ZmController.prototype.getCurrentViewType;

/**
 * Sets the current view type.
 * 
 * @param	{constant}	viewType		the view type
 */
ZmController.prototype.setCurrentViewType =
function(viewType) {
	this._currentViewType = viewType;
};

/**
 * Gets the current view ID.
 * 
 * @return	{DwtComposite}	the view Id
 */
ZmController.prototype.getCurrentViewId =
function() {
	return this._currentViewIdOverride || this._currentViewId;
};

/**
 * Sets the current view ID.
 * 
 * @param	{string}	viewId		the view ID
 */
ZmController.prototype.setCurrentViewId =
function(viewId) {
	this._currentViewId = viewId;
	
	// this._currentView is DEPRECATED in 8.0; use getCurrentViewId() instead
	this._currentView = this._currentViewId;
};

/**
 * Gets the application.
 * 
 * @return	{ZmApp}		the application
 */
ZmController.prototype.getApp = function() {
	return this._app;
};

/**
 * return the view elements. Currently a toolbar, app content, and "new" button.
 * 
 * @param view (optional if provided toolbar)
 * @param appContentView
 * @param toolbar (used only if view param is null)
 *
 */
ZmController.prototype.getViewElements =
function(view, appContentView, toolbar) {
	var elements = {};
	toolbar = toolbar || this._toolbar[view];
	elements[ZmAppViewMgr.C_TOOLBAR_TOP] = toolbar;
	elements[ZmAppViewMgr.C_APP_CONTENT] = appContentView;

	return elements;
};

/**
 * Pops-up the error dialog.
 * 
 * @param	{String}	msg		the error msg
 * @param	{ZmCsfeException}	ex		the exception
 * @param	{Boolean}	noExecReset		(not used)
 * @param	{Boolean}	hideReportButton		if <code>true</code>, hide the "Send error report" button
 * @param	{Boolean}	expanded		if <code>true</code>, contents are expanded by default
 */
ZmController.prototype.popupErrorDialog = 
function(msg, ex, noExecReset, hideReportButton, expanded, noEncoding) {
	// popup alert
	var errorDialog = appCtxt.getErrorDialog();
	var detailStr = "";
	if (typeof ex == "string") {
		// in case an Error makes it here
		detailStr = ex;
	} else if (ex instanceof Object) {
		ex.msg = ex.msg || msg;
		var fields = ["method", "msg", "code", "detail", "trace", "request",
					"fileName", "lineNumber", "message", "name", "stack" ];
		var html = [], i = 0;
		html[i++] = "<table role='presentation'>";
		for (var j = 0; j < fields.length; j++) {
			var fld = fields[j];
			var value = AjxStringUtil.htmlEncode(ex[fld]);
			if (value) {
				if (fld == "request") {
					value = ["<pre>", value, "</pre>"].join("");
					var msgDiv = document.getElementById(errorDialog._msgCellId);
					if (msgDiv) {
						msgDiv.className = "DwtMsgDialog-wide";
					}
				}
				html[i++] = ["<tr><td valign='top'>", fields[j], ":</td><td valign='top'>", value, "</td></tr>"].join("");
			}
		}
		html[i++] = "</table>";
		detailStr = html.join("");
	}
	errorDialog.registerCallback(DwtDialog.OK_BUTTON, this._errorDialogCallback, this);
	if (!noEncoding) {
		msg = AjxStringUtil.htmlEncode(msg);
	}
	errorDialog.setMessage(msg, detailStr, DwtMessageDialog.CRITICAL_STYLE, ZmMsg.zimbraTitle);
	errorDialog.popup(null, hideReportButton);
	if (expanded)
		errorDialog.showDetail();
};

/**
 * Pops-up an error dialog describing an upload error.
 *
 * @param	{constant}	type		the type of the uploaded item, e.g. <code>ZmItem.MSG</code>.
 * @param	{Number}	respCode		the HTTP reponse status code
 * @param	{String}	extraMsg		optional message to append to the status
 */
ZmController.prototype.popupUploadErrorDialog =
function(type, respCode, extraMsg) {
    var warngDlg = appCtxt.getMsgDialog();
    var style = DwtMessageDialog.CRITICAL_STYLE;
    var msg = this.createErrorMessage(type, respCode, extraMsg);
	if (msg.length > 0) {
		warngDlg.setMessage(msg, style);
		warngDlg.popup();
	}
};

ZmController.prototype.createErrorMessage = function(type, respCode, extraMsg) {
	var msg = "";

	switch (respCode) {
		case AjxPost.SC_OK:
			break;

		case AjxPost.SC_REQUEST_ENTITY_TOO_LARGE:
			var basemsg =
				type && ZmMsg['attachmentSizeError_' + type] ||
					ZmMsg.attachmentSizeError;
			var sizelimit =
				AjxUtil.formatSize(appCtxt.get(ZmSetting.MESSAGE_SIZE_LIMIT));
			msg = AjxMessageFormat.format(basemsg, sizelimit);
			break;

		default:
			var basemsg =
				type && ZmMsg['errorAttachment_' + type] ||
					ZmMsg.errorAttachment;
			msg = AjxMessageFormat.format(basemsg, respCode || AjxPost.SC_NO_CONTENT);
			break;
	}

	if ((msg.length > 0) && extraMsg) {
		msg += '<br /><br />';
		msg += extraMsg;
	}
	return msg;
};

ZmController.handleScriptError =
function(ex, debugWindowOnly) {

	var text = [];
	var eol = "<br/>";
	if (ex) {
		var msg = ZmMsg.scriptError + ": " + ex.message;
		var m = ex.fileName && ex.fileName.match(/(\w+\.js)/);
		if (m && m.length) {
			msg += " - " + m[1] + ":" + ex.lineNumber;
		}
		if (ex.fileName)	{ text.push("File: " + ex.fileName); }
		if (ex.lineNumber)	{ text.push("Line: " + ex.lineNumber); }
		if (ex.name)		{ text.push("Error: " + ex.name); }
		if (ex.stack)		{ text.push("Stack: " + ex.stack.replace("\n", eol, "g")); }
	}
	var content = text.join(eol);
	var errorMsg = [msg, content].join(eol + eol);
	if (debugWindowOnly) {
		// Display the error in the debug window
		DBG.println(AjxDebug.DBG1, errorMsg);
	} else {
		// Record the error in a log buffer and display a script error popup
		AjxDebug.println(AjxDebug.EXCEPTION, errorMsg);
		appCtxt.getAppController().popupErrorDialog(msg, content, null, false, true);
	}
};

/**
 * Gets the key map name.
 * 
 * @return	{String}	the key map name
 */
ZmController.prototype.getKeyMapName =
function() {
	return ZmKeyMap.MAP_GLOBAL;
};

/**
 * Handles the key action.
 * 
 * @param	{constant}		actionCode		the action code
 * @return	{Boolean}	<code>true</code> if the key action is handled
 * 
 * @see		ZmApp.ACTION_CODES_R
 * @see		ZmKeyMap
 */
ZmController.prototype.handleKeyAction =
function(actionCode, ev) {
	DBG.println(AjxDebug.DBG3, "ZmController.handleKeyAction");
	
	// tab navigation shortcut
	var tabView = this.getTabView ? this.getTabView() : null;
	if (tabView && tabView.handleKeyAction(actionCode)) {
		return true;
	}

	// shortcuts tied directly to operations
    var isExternalAccount = appCtxt.isExternalAccount();
	var app = ZmApp.ACTION_CODES_R[actionCode];
	if (app) {
		var op = ZmApp.ACTION_CODES[actionCode];
		if (op) {
            if (isExternalAccount) { return true; }
			appCtxt.getApp(app).handleOp(op);
			return true;
		}
	}

    switch (actionCode) {

		case ZmKeyMap.NEW: {
            if (isExternalAccount) { break; }
			// find default "New" action code for current app
			app = appCtxt.getCurrentAppName();
			var newActionCode = ZmApp.NEW_ACTION_CODE[app];
			if (newActionCode) {
				var op = ZmApp.ACTION_CODES[newActionCode];
				if (op) {
					appCtxt.getApp(app).handleOp(op);
					return true;
				}
			}
			break;
		}

		case ZmKeyMap.NEW_FOLDER:
		case ZmKeyMap.NEW_TAG:
            if (isExternalAccount || appCtxt.isWebClientOffline()) { break; }
			var op = ZmApp.ACTION_CODES[actionCode];
			if (op) {
				this._newListener(null, op);
			}
			break;

	    case ZmKeyMap.NEW_SEARCH: {
		    appCtxt.getSearchController().openNewSearchTab();
		    break;
	    }

		case ZmKeyMap.SAVED_SEARCH:
            if (isExternalAccount) { break; }
			var searches = appCtxt.getFolderTree().getByType(ZmOrganizer.SEARCH);
			if (searches && searches.length > 0) {
				var dlg = appCtxt.getChooseFolderDialog();
				// include app name in ID so we have one overview per app to show only its saved searches
				var params = {treeIds:		[ZmOrganizer.SEARCH],
							  overviewId:	dlg.getOverviewId(ZmOrganizer.SEARCH, this._app._name),
							  appName:      this._app._name,
							  title:		ZmMsg.selectSearch};
				ZmController.showDialog(dlg, new AjxCallback(null, ZmController._searchSelectionCallback, [dlg]), params);
			}
			break;

		case ZmKeyMap.VISIT:
			var dlg = appCtxt.getChooseFolderDialog();
			var orgType = ZmApp.ORGANIZER[this._app._name] || ZmOrganizer.FOLDER;
			var params = {treeIds:		[orgType],
						  overviewId:	dlg.getOverviewId(ZmOrganizer.APP[orgType]),
						  appName:		this._app._name,
						  noRootSelect: true,
						  title:		AjxMessageFormat.format(ZmMsg.goToFolder, ZmMsg[ZmOrganizer.MSG_KEY[orgType]])};
			ZmController.showDialog(dlg, new AjxCallback(null, ZmController._visitOrgCallback, [dlg, orgType]), params);
			break;

		case ZmKeyMap.VISIT_TAG:
			if (appCtxt.getTagTree().size() > 0) {
				var dlg = appCtxt.getPickTagDialog();
				ZmController.showDialog(dlg, new AjxCallback(null, ZmController._visitOrgCallback, [dlg, ZmOrganizer.TAG]));
			}
			break;

		default:
			return false;
	}
	return true;
};

/**
 * @private
 */
ZmController._searchSelectionCallback =
function(dialog, searchFolder) {
	if (searchFolder) {
		appCtxt.getSearchController().redoSearch(searchFolder.search);
	}
	dialog.popdown();
};

/**
 * @private
 */
ZmController._visitOrgCallback =
function(dialog, orgType, org) {
	if (org) {
		var tc = appCtxt.getOverviewController().getTreeController(orgType);
		if (tc && tc._itemClicked) {
			tc._itemClicked(org);
		}
	}
	dialog.popdown();
};

/**
 * Checks if shortcuts for the given map are supported for this view. For example, given the map
 * "tabView", a controller that creates a tab view would return <code>true</code>.
 *
 * @param {String}	map		the name of a map (see {@link DwtKeyMap})
 * @return	{Boolean}		<code>true</code> if shortcuts are supported
 */
ZmController.prototype.mapSupported =
function(map) {
	return false;
};

/**
 * @private
 */
ZmController.prototype._newListener =
function(ev, op) {
	switch (op) {
		// new organizers
		case ZmOperation.NEW_FOLDER: {
			// note that this shortcut only happens if mail app is around - it means "new mail folder"
			ZmController.showDialog(appCtxt.getNewFolderDialog(), this.getNewFolderCallback());
			break;
		}
		case ZmOperation.NEW_TAG: {
			if (!this._newTagCb) {
				this._newTagCb = new AjxCallback(this, this._newTagCallback);
			}
			ZmController.showDialog(appCtxt.getNewTagDialog(), this._newTagCb);
			break;
		}
	}
};

/**
 * @private
 */
ZmController.prototype._newFolderCallback =
function(parent, name, color, url) {
	// REVISIT: Do we really want to close the dialog before we
	//          know if the create succeeds or fails?
	var dialog = appCtxt.getNewFolderDialog();
	dialog.popdown();

	var oc = appCtxt.getOverviewController();
	oc.getTreeController(ZmOrganizer.FOLDER)._doCreate(parent, name, color, url);
};

/**
 * @private
 */
ZmController.prototype._newTagCallback =
function(params) {
	appCtxt.getNewTagDialog().popdown();
	var oc = appCtxt.getOverviewController();
	oc.getTreeController(ZmOrganizer.TAG)._doCreate(params);
};

/**
 * @private
 */
ZmController.prototype._createTabGroup =
function(name) {
	name = name ? name : this.toString();
	this._tabGroup = new DwtTabGroup(name);
	return this._tabGroup;
};

/**
 * @private
 */
ZmController.prototype._setTabGroup =
function(tabGroup) {
	this._tabGroup = tabGroup;
};

/**
 * Gets the tab group.
 * 
 * @return	{Object}	the tab group
 */
ZmController.prototype.getTabGroup =
function() {
	return this._tabGroup;
};

/**
 * Gets the new folder callback.
 * 
 * @return	{AjxCallback}	the callback
 */
ZmController.prototype.getNewFolderCallback =
function() {
	if (!this._newFolderCb) {
		this._newFolderCb = new AjxCallback(this, this._newFolderCallback);
	}
	return this._newFolderCb;
};

/**
 * Remember the currently focused item before this view is hidden. Typically called by a preHideCallback.
 * 
 * @private
 */
ZmController.prototype._saveFocus = 
function() {
	var currentFocusMember = appCtxt.getRootTabGroup().getFocusMember();
	var myTg = this.getTabGroup();
	this._savedFocusMember = (currentFocusMember && myTg && myTg.contains(currentFocusMember)) ? currentFocusMember : null;
	return this._savedFocusMember;
};

/**
 * Make our tab group the current app view tab group, and restore focus to
 * whatever had it last time we were visible. Typically called by a postShowCallback.
 * 
 * @private
 */
ZmController.prototype._restoreFocus = 
function(focusItem, noFocus) {

	var rootTg = appCtxt.getRootTabGroup();

	var curApp = appCtxt.getCurrentApp();
	var ovId = curApp && curApp.getOverviewId();
	var overview = ovId && appCtxt.getOverviewController().getOverview(ovId);
	if (rootTg && overview && (overview != ZmController._currentOverview)) {
		var currTg = ZmController._currentOverview &&
			ZmController._currentOverview.getTabGroupMember();
		rootTg.replaceMember(currTg, overview.getTabGroupMember(),
		                     false, false, null, true);
		ZmController._currentOverview = overview;
	}

	var myTg = this.getTabGroup();
	focusItem = focusItem || this._savedFocusMember || this._getDefaultFocusItem() || rootTg.getFocusMember();
	noFocus = noFocus || ZmController.noFocus;
	ZmController.noFocus = false;
	if (rootTg && myTg && (myTg != ZmController._currentAppViewTabGroup)) {
		rootTg.replaceMember(ZmController._currentAppViewTabGroup, myTg, false, false, focusItem, noFocus);
		ZmController._currentAppViewTabGroup = myTg;
	} else if (focusItem && !noFocus) {
		appCtxt.getKeyboardMgr().grabFocus(focusItem);
	}
};

/**
 * @private
 */
ZmController.prototype._getDefaultFocusItem = 
function() {
	var myTg = this.getTabGroup();
	return myTg ? myTg.getFirstMember(true) : null;
};

// Callbacks to run on changes in view state
ZmController.prototype._preUnloadCallback	= function() { return true; };
ZmController.prototype._postHideCallback	= function() { return true; };
ZmController.prototype._postRemoveCallback	= function() { return true; };
ZmController.prototype._preShowCallback		= function() { return true; };

// preserve focus state
ZmController.prototype._preHideCallback = 
function() {
	DBG.println(AjxDebug.DBG2, "ZmController.prototype._preHideCallback");
	this._saveFocus();
	return true;
};

// restore focus state
ZmController.prototype._postShowCallback = 
function() {
	DBG.println(AjxDebug.DBG2, "ZmController.prototype._postShowCallback");
	this._restoreFocus();
	return true;
};

/**
 * Common exception handling entry point for sync and async commands.
 * 
 * @private
 */
ZmController.prototype._handleError =
function(ex, continuation) {
	this._handleException(ex, continuation);
};

/**
 * Handles exceptions. There is special handling for auth-related exceptions.
 * Other exceptions generally result in the display of an error dialog. An
 * auth-expired exception results in the display of a login dialog. After the
 * user logs in, we use the continuation to re-run the request that failed.
 * 
 * @param {AjxException}	ex				the exception
 * @param {Hash}	continuation		the original request params
 * 
 * @private
 */
ZmController.prototype._handleException = function(ex, continuation) {

	if (ex.code == AjxSoapException.INVALID_PDU) {
		ex.code = ZmCsfeException.SVC_FAILURE;
		ex.detail = ["contact your administrator (", ex.msg, ")"].join("");
		ex.msg = "Service failure";
	}
	
	if (ex.code == ZmCsfeException.SVC_AUTH_EXPIRED || ex.code == ZmCsfeException.SVC_AUTH_REQUIRED || ex.code == ZmCsfeException.NO_AUTH_TOKEN) {
		ZmCsfeCommand.noAuth = true;
        DBG.println(AjxDebug.DBG1, "ZmController.prototype._handleException ex.code : " + ex.code + ". Invoking logout.");
		ZmZimbraMail.logOff(null, true);
		return;
	}

	// If we get this error, user is probably looking at a stale list. Let's
	// refetch user's search results. This is more likely to happen in zdesktop.
	// See bug 33760.
	if (ex.code == ZmCsfeException.MAIL_NO_SUCH_MSG) {
		var vid = appCtxt.getCurrentViewId();
		// only process if we're in one of these views otherwise, do the default
		if (vid == ZmId.VIEW_CONVLIST || vid == ZmId.VIEW_TRAD) {
			var mailApp = appCtxt.getApp(ZmApp.MAIL);
			var callback = appCtxt.isOffline ? new AjxCallback(this, this._handleMailSearch, mailApp) : null;
			mailApp.mailSearch(null, callback);
			return;
		}
	}

	// silently ignore polling exceptions
	if (ex.method !== "NoOpRequest") {
		var args;
		if (ex.code === ZmCsfeException.MAIL_NO_SUCH_ITEM) {
			args = ex.data.itemId;
		}
        else if (ex.code === ZmCsfeException.MAIL_SEND_FAILURE) {
			args = ex.code; // bug fix #5603 - error msg for mail.SEND_FAILURE takes an argument
		}
        else if (ex.code === ZmCsfeException.MAIL_INVALID_NAME) {
			args = ex.data.name;
		}
        else if (ex.code === ZmCsfeException.SVC_UNKNOWN_DOCUMENT) {
            args = ex.msg.split(': ')[1];
        }

		if (ex.lineNumber && !ex.detail) {
			// JS error that was caught before our JS-specific handler got it
			ZmController.handleScriptError(ex);
		}
        else {
            var msg = ex.getErrorMsg ? ex.getErrorMsg(args) : ex.msg || ex.message;

			this.popupErrorDialog(msg, ex, true, this._hideSendReportBtn(ex));
		}
	}
};

ZmController.prototype._handleMailSearch =
function(app) {
	if (appCtxt.get(ZmSetting.OFFLINE_SHOW_ALL_MAILBOXES)) {
		app.getOverviewContainer().highlightAllMboxes();
	}
};

/**
 * @private
 */
ZmController.prototype._hideSendReportBtn =
function(ex) {
	return (ex.code == ZmCsfeException.MAIL_TOO_MANY_TERMS ||
		  	ex.code == ZmCsfeException.MAIL_MAINTENANCE_MODE ||
			ex.code == ZmCsfeException.MAIL_MESSAGE_TOO_BIG ||
			ex.code == ZmCsfeException.NETWORK_ERROR ||
		   	ex.code == ZmCsfeException.EMPTY_RESPONSE ||
		   	ex.code == ZmCsfeException.BAD_JSON_RESPONSE ||
		   	ex.code == ZmCsfeException.TOO_MANY_TAGS ||
			ex.code == ZmCsfeException.OFFLINE_ONLINE_ONLY_OP);
};

//
// Msg dialog Callbacks
//

/**
 * @private
 */
ZmController.prototype._errorDialogCallback =
function() {
	appCtxt.getErrorDialog().popdown();
};

/**
 * Shows a dialog. Since the dialog is a shared resource, a dialog reset is performed.
 * 
 * @param	{DwtDialog}		dialog		the dialog
 * @param	{AjxCallback}	callback	the callback
 * @param	{Hash}		params		a hash of parameters
 * @param	{ZmAccount}	account		the account
 * 
 * @see DwtDialog#reset
 * @see DwtDialog#popup
 */
ZmController.showDialog = 
function(dialog, callback, params, account) {
	dialog.reset(account);
	dialog.registerCallback(DwtDialog.OK_BUTTON, callback);
	dialog.popup(params, account);
};

/**
 * Pop down the dialog and clear any pending actions (initiated from an action menu).
 * 
 * @private
 */
ZmController.prototype._clearDialog =
function(dialog) {
	dialog.popdown();
	this._pendingActionData = null;
};

/**
 * @private
 */
ZmController.prototype._menuPopdownActionListener = function() {};

/**
 * Checks if the view is transient.
 * 
 * @param	{Object}	oldView		the old view
 * @param	{Object}	newView		the new view
 * @return	{Boolean}		<code>true</code> if the controller is transient.
 */
ZmController.prototype.isTransient =
function(oldView, newView) {
	return false;
};

}
if (AjxPackage.define("zimbraMail.share.controller.ZmBaseController")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines a base controller class.
 *
 */

/**
 * This class is a base class for any controller that manages items such as messages, contacts,
 * appointments, tasks, etc. It handles operations that can be performed on those items such as
 * move, delete, tag, print, etc.
 *
 * @author Conrad Damon
 *
 * @param {DwtControl}					container					the containing shell
 * @param {ZmApp}						app							the containing application
 * @param {constant}					type						type of controller (typically a view type)				
 * @param {string}						sessionId					the session id
 * @param {ZmSearchResultsController}	searchResultsController		containing controller
 * 
 * @extends		ZmController
 */
ZmBaseController = function(container, app, type, sessionId, searchResultsController) {

	if (arguments.length == 0) { return; }
	ZmController.apply(this, arguments);

	this.setSessionId(sessionId, type || this.getDefaultViewType(), searchResultsController);
	
    //this._refreshQuickCommandsClosure = this._refreshQuickCommands.bind(this);
    //this._quickCommandMenuHandlerClosure = this._quickCommandMenuHandler.bind(this);

	// hashes keyed by view type
	this._view		= {};
	this._toolbar	= {};	// ZmButtonToolbar
	this._tabGroups = {};	// DwtTabGroup

	this._tagList = appCtxt.getTagTree();
	if (this._tagList) {
		this._boundTagChangeListener = this._tagChangeListener.bind(this);
		this._tagList.addChangeListener(this._boundTagChangeListener);
	}

	// create a listener for each operation
	this._listeners = {};
	this._listeners[ZmOperation.NEW_MENU]		= this._newListener.bind(this);
	this._listeners[ZmOperation.TAG_MENU]		= this._tagButtonListener.bind(this);
	this._listeners[ZmOperation.MOVE_MENU]		= this._moveButtonListener.bind(this);
	this._listeners[ZmOperation.ACTIONS_MENU]	= this._actionsButtonListener.bind(this);
	this._listeners[ZmOperation.TAG]			= this._tagListener.bind(this);
	this._listeners[ZmOperation.PRINT]			= this._printListener.bind(this);
	this._listeners[ZmOperation.DELETE]			= this._deleteListener.bind(this);
	this._listeners[ZmOperation.DELETE_WITHOUT_SHORTCUT]			= this._deleteListener.bind(this);
	this._listeners[ZmOperation.CLOSE]			= this._backListener.bind(this);
	this._listeners[ZmOperation.MOVE]			= this._moveListener.bind(this);
	this._listeners[ZmOperation.SEARCH]			= this._searchListener.bind(this);
	this._listeners[ZmOperation.NEW_MESSAGE]	= this._composeListener.bind(this);
	this._listeners[ZmOperation.CONTACT]		= this._contactListener.bind(this);
	this._listeners[ZmOperation.VIEW]			= this._viewMenuItemListener.bind(this);

	// TODO: do this better - avoid referencing specific apps
	if (window.ZmImApp) {
		this._listeners[ZmOperation.IM] = ZmImApp.getImMenuItemListener();
	}

	/**
	 * List of toolbar operations to enable on Zero/no selection
	 * - Default is only enable ZmOperation.NEW_MENU
	 */
	this.operationsToEnableOnZeroSelection = [ZmOperation.NEW_MENU];

	/**
	 * List of toolbar operations to enable when multiple items are selected
	 * - Default is to enable: ZmOperation.NEW_MENU, ZmOperation.TAG_MENU, ZmOperation.DELETE, ZmOperation.MOVE,
	 * 						ZmOperation.MOVE_MENU, ZmOperation.FORWARD & ZmOperation.ACTIONS_MENU
	 */
	this.operationsToEnableOnMultiSelection = [ZmOperation.NEW_MENU, ZmOperation.TAG_MENU, ZmOperation.DELETE,
												ZmOperation.MOVE, ZmOperation.MOVE_MENU, ZmOperation.FORWARD,
												ZmOperation.ACTIONS_MENU];
	/**
	 * List of toolbar operations to *disable*
	 * Default is to enable-all
	 */
	this.operationsToDisableOnSingleSelection = [];
};

ZmBaseController.prototype = new ZmController;
ZmBaseController.prototype.constructor = ZmBaseController;

ZmBaseController.prototype.isZmBaseController = true;
ZmBaseController.prototype.toString = function() { return "ZmBaseController"; };



// public methods

/**
 * Sets the session id, view id, and tab id. Notes whether this controller is being
 * used to display search results.
 *
 * @param {string}						sessionId					the session id
 * @param {string}						type						the type
 * @param {ZmSearchResultsController}	searchResultsController		owning controller
 */
ZmBaseController.prototype.setSessionId =
function(sessionId, type, searchResultsController) {

	ZmController.prototype.setSessionId.apply(this, arguments);
	this.searchResultsController = searchResultsController;
	this.isSearchResults = Boolean(searchResultsController);
};

/**
 * Gets the current view object.
 * 
 * @return	{DwtComposite}	the view object
 */
ZmBaseController.prototype.getCurrentView =
function() {
	return this._view[this._currentViewId];
};

/**
 * Returns the view used to display a single item, if any.
 */
ZmBaseController.prototype.getItemView = function() {
	return null;
};

/**
 * Gets the current tool bar.
 * 
 * @return	{ZmButtonToolbar}		the toolbar
 */
ZmBaseController.prototype.getCurrentToolbar =
function() {
	return this._toolbar[this._currentViewId];
};

/**
 * Returns the list of items to be acted upon.
 */
ZmBaseController.prototype.getItems = function() {};

/**
 * Returns the number of items to be acted upon.
 */
ZmBaseController.prototype.getItemCount = function() {};

/**
 * Handles a shortcut.
 * 
 * @param	{constant}	actionCode		the action code
 * @return	{Boolean}	<code>true</code> if the action is handled
 */
ZmBaseController.prototype.handleKeyAction =
function(actionCode, ev) {

	DBG.println(AjxDebug.DBG3, "ZmBaseController.handleKeyAction");
    var isExternalAccount = appCtxt.isExternalAccount();

	switch (actionCode) {

		case ZmKeyMap.MOVE:
            if (isExternalAccount) { break; }
			var items = this.getItems();
			if (items && items.length) {
				this._moveListener();
			}
			break;

		case ZmKeyMap.PRINT:
			if (appCtxt.get(ZmSetting.PRINT_ENABLED) && !appCtxt.isWebClientOffline()) {
				this._printListener();
			}
			break;

		case ZmKeyMap.TAG:
            if (isExternalAccount) { break; }
			var items = this.getItems();
			if (items && items.length && (appCtxt.getTagTree().size() > 0)) {
				var dlg = appCtxt.getPickTagDialog();
				ZmController.showDialog(dlg, new AjxCallback(this, this._tagSelectionCallback, [items, dlg]));
			}
			break;

		case ZmKeyMap.UNTAG:
            if (isExternalAccount) { break; }
			if (appCtxt.get(ZmSetting.TAGGING_ENABLED)) {
				var items = this.getItems();
				if (items && items.length) {
					this._doRemoveAllTags(items);
				}
			}
			break;

		default:
			return ZmController.prototype.handleKeyAction.apply(this, arguments);
	}
	return true;
};

/**
 * Returns true if this controller's view is currently being displayed (possibly within a search results tab)
 */
ZmBaseController.prototype.isCurrent =
function() {
	return (this._currentViewId == appCtxt.getCurrentViewId());
};

ZmBaseController.prototype.supportsDnD =
function() {
	return !appCtxt.isExternalAccount();
};

// abstract protected methods

// Creates the view element
ZmBaseController.prototype._createNewView	 		= function() {};

// Populates the view with data
ZmBaseController.prototype._setViewContents			= function(view) {};

// Returns text for the tag operation
ZmBaseController.prototype._getTagMenuMsg 			= function(num) {};

// Returns text for the move dialog
ZmBaseController.prototype._getMoveDialogTitle		= function(num) {};

// Returns a list of desired toolbar operations
ZmBaseController.prototype._getToolBarOps 			= function() {};

// Returns a list of secondary (non primary) toolbar operations
ZmBaseController.prototype._getSecondaryToolBarOps 	= function() {};

// Returns a list of buttons that align to the right, like view and detach
ZmBaseController.prototype._getRightSideToolBarOps 	= function() {};


// private and protected methods

/**
 * Creates basic elements and sets the toolbar and action menu.
 * 
 * @private
 */
ZmBaseController.prototype._setup =
function(view) {
	this._initialize(view);
	this._resetOperations(this._toolbar[view], 0);
};

/**
 * Creates the basic elements: toolbar, list view, and action menu.
 *
 * @private
 */
ZmBaseController.prototype._initialize =
function(view) {
	this._initializeToolBar(view);
	this._initializeView(view);
	this._initializeTabGroup(view);
};

// Below are functions that return various groups of operations, for cafeteria-style
// operation selection.

/**
 * @private
 */
ZmBaseController.prototype._standardToolBarOps =
function() {
	return [ZmOperation.DELETE, ZmOperation.MOVE_MENU, ZmOperation.PRINT];
};

/**
 * Initializes the toolbar buttons and listeners.
 * 
 * @private
 */
ZmBaseController.prototype._initializeToolBar =
function(view, className) {

	if (this._toolbar[view]) { return; }

	var buttons = this._getToolBarOps();
	var secondaryButtons = this._getSecondaryToolBarOps() || [];
	var rightSideButtons = this._getRightSideToolBarOps() || [];
	if (!(buttons || secondaryButtons)) { return; }

	var tbParams = {
		parent:				this._container,
		buttons:			buttons,
		secondaryButtons:	secondaryButtons,
		rightSideButtons: 	rightSideButtons,
		overrides:          this._getButtonOverrides(buttons.concat(secondaryButtons).concat(rightSideButtons)),
		context:			view,
		controller:			this,
		refElementId:		ZmId.SKIN_APP_TOP_TOOLBAR,
		addTextElement:		true,
		className:			className
	};
	var tb = this._toolbar[view] = new ZmButtonToolBar(tbParams);

	var text = tb.getButton(ZmOperation.TEXT);
	if (text) {
		text.addClassName("itemCountText");
	}

	var button;
	for (var i = 0; i < tb.opList.length; i++) {
		button = tb.opList[i];
		if (this._listeners[button]) {
			tb.addSelectionListener(button, this._listeners[button]);
		}
	}

	button = tb.getButton(ZmOperation.TAG_MENU);
	if (button) {
		button.noMenuBar = true;
		this._setupTagMenu(tb);
	}

	button = tb.getButton(ZmOperation.MOVE_MENU);
	if (button) {
		button.noMenuBar = true;
		this._setupMoveMenu(tb);
	}


	// add the selection listener for when user clicks on the little drop-down arrow (unfortunately we have to do that here separately) It is done for the main button area in a generic way to all toolbar buttons elsewhere
	var actionsButton = tb.getActionsButton();
	if (actionsButton) {
		actionsButton.addDropDownSelectionListener(this._listeners[ZmOperation.ACTIONS_MENU]);
	}

	var actionsMenu = tb.getActionsMenu();
	if (actionsMenu) {
		this._setSearchMenu(actionsMenu, true);
	}	

	var children = tb.getHtmlElement().children;
	var child = children ? children[0] : null;
	if(child) {
		child.setAttribute('role', 'region');
		child.setAttribute('aria-label', ZmMsg.toolbar);
	}

	appCtxt.notifyZimlets("initializeToolbar", [this._app, tb, this, view], {waitUntilLoaded:true});
};

ZmBaseController.prototype._getButtonOverrides = function(buttons) {};

/**
 * Initializes the view and its listeners.
 * 
 * @private
 */
ZmBaseController.prototype._initializeView =
function(view) {

	if (this._view[view]) { return; }

	this._view[view] = this._createNewView(view);
	this._view[view].addSelectionListener(this._listSelectionListener.bind(this));
	this._view[view].addActionListener(this._listActionListener.bind(this));
};

// back-compatibility (bug 60073)
ZmBaseController.prototype._initializeListView = ZmBaseController.prototype._initializeView;

/**
 * Sets up tab groups (focus ring).
 * 
 * @private
 */
ZmBaseController.prototype._initializeTabGroup = function(view) {

	if (this._tabGroups[view]) {
        return;
    }

	this._tabGroups[view] = this._createTabGroup();
	this._tabGroups[view].newParent(appCtxt.getRootTabGroup());
	this._tabGroups[view].addMember(this._toolbar[view].getTabGroupMember());
    this._tabGroups[view].addMember(this._view[view].getTabGroupMember());
};

/**
 * Creates the desired application view.
 *
 * @param params		[hash]			hash of params:
 *        view			[constant]		view ID
 *        elements		[array]			array of view components
 *        controller	[ZmController]	controller responsible for this view
 *        isAppView		[boolean]*		this view is a top-level app view
 *        clear			[boolean]*		if true, clear the hidden stack of views
 *        pushOnly		[boolean]*		if true, don't reset the view's data, just swap the view in
 *        noPush		[boolean]*		if true, don't push the view, just set its contents
 *        isTransient	[boolean]*		this view doesn't go on the hidden stack
 *        stageView		[boolean]*		stage the view rather than push it
 *        tabParams		[hash]*			button params; view is opened in app tab instead of being stacked
 *        
 * @private
 */
ZmBaseController.prototype._setView =
function(params) {

	var view = params.view;
	
	// create the view (if we haven't yet)
	if (!this._appViews[view]) {
		// view management callbacks
		var callbacks = {};
		callbacks[ZmAppViewMgr.CB_PRE_HIDE]		= this._preHideCallback.bind(this);
		callbacks[ZmAppViewMgr.CB_PRE_UNLOAD]	= this._preUnloadCallback.bind(this);
		callbacks[ZmAppViewMgr.CB_POST_HIDE]	= this._postHideCallback.bind(this);
		callbacks[ZmAppViewMgr.CB_POST_REMOVE]	= this._postRemoveCallback.bind(this);
		callbacks[ZmAppViewMgr.CB_PRE_SHOW]		= this._preShowCallback.bind(this);
		callbacks[ZmAppViewMgr.CB_POST_SHOW]	= this._postShowCallback.bind(this);

		params.callbacks = callbacks;
		params.viewId = view;
		params.controller = this;
		this._app.createView(params);
		this._appViews[view] = true;
	}

	// populate the view
	if (!params.pushOnly) {
		this._setViewContents(view);
	}

	// push the view
	if (params.stageView) {
		this._app.stageView(view);
	} else if (!params.noPush) {
		return (params.clear ? this._app.setView(view) : this._app.pushView(view));
	}
};



// Operation listeners

/**
 * Tag button has been pressed. We don't tag anything (since no tag has been selected),
 * we just show the dynamic tag menu.
 * 
 * @private
 */
ZmBaseController.prototype._tagButtonListener =
function(ev) {
	var toolbar = this._toolbar[this._currentViewId];
	if (ev.item.parent == toolbar) {
		this._setTagMenu(toolbar);
	}
};

/**
 * Move button has been pressed. We don't move anything (since no folder has been selected),
 * we just show the dynamic move menu.
 *
 * @private
 */
ZmBaseController.prototype._moveButtonListener =
function(ev, list) {
	this._pendingActionData = list || this.getItems();

	var toolbar = this._toolbar[this._currentViewId];

	var moveButton = toolbar.getOp(ZmOperation.MOVE_MENU);
	if (!moveButton) {
		return;
	}
	if (!this._moveButtonInitialized) {
		this._moveButtonInitialized = true;
		appCtxt.getShell().setBusy(true);
		this._setMoveButton(moveButton);
		appCtxt.getShell().setBusy(false);
	}
	else {
		//need to update this._data so the chooser knows from which folder we are trying to move.
		this._folderChooser.updateData(this._getMoveParams(this._folderChooser).data);
	}
	var newButton = this._folderChooser._getNewButton();
	if (newButton) {
		newButton.setVisible(!appCtxt.isWebClientOffline());
	}
	moveButton.popup();
	moveButton.getMenu().getHtmlElement().style.width = "auto"; //reset the width so it's dynamic. without this it is set to 0, and in any case even if it was set to some other > 0 value, it needs to be dynamic due to collapse/expand (width changes)
	this._folderChooser.focus();
};

/**
 * Actions button has been pressed.
 * @private
 */
ZmBaseController.prototype._actionsButtonListener =
function(ev) {
	var menu = this.getCurrentToolbar().getActionsMenu();
	menu.parent.popup();	
};


/**
 * Tag/untag items.
 * 
 * @private
 */
ZmBaseController.prototype._tagListener =
function(ev, items) {

	if (this.isCurrent()) {
		var menuItem = ev.item;
		var tagEvent = menuItem.getData(ZmTagMenu.KEY_TAG_EVENT);
		var tagAdded = menuItem.getData(ZmTagMenu.KEY_TAG_ADDED);
		items = items || this.getItems();

		if (tagEvent == ZmEvent.E_TAGS && tagAdded) {
			this._doTag(items, menuItem.getData(Dwt.KEY_OBJECT), true);
		} else if (tagEvent == ZmEvent.E_CREATE) {
			this._pendingActionData = items;
			var newTagDialog = appCtxt.getNewTagDialog();
			if (!this._newTagCb) {
				this._newTagCb = new AjxCallback(this, this._newTagCallback);
			}
			ZmController.showDialog(newTagDialog, this._newTagCb);
			newTagDialog.registerCallback(DwtDialog.CANCEL_BUTTON, this._clearDialog, this, newTagDialog);
		} else if (tagEvent == ZmEvent.E_TAGS && !tagAdded) {
			//remove tag
			this._doTag(items, menuItem.getData(Dwt.KEY_OBJECT), false);
		} else if (tagEvent == ZmEvent.E_REMOVE_ALL) {
			// bug fix #607
			this._doRemoveAllTags(items);
		}
	}
};

/**
 * Called after tag selection via dialog.
 * 
 * @private
 */
ZmBaseController.prototype._tagSelectionCallback =
function(items, dialog, tag) {
	if (tag) {
		this._doTag(items, tag, true);
	}
	dialog.popdown();
};

/**
 * overload if you want to print in a different way.
 * 
 * @private
 */
ZmBaseController.prototype._printListener =
function(ev) {
	var items = this.getItems();
    if (items && items[0]) {
	    window.open(items[0].getRestUrl(), "_blank");
	}
};

ZmBaseController.prototype._backListener =
function(ev) {
	this._app.popView();
};

/**
 * Delete one or more items.
 * 
 * @private
 */
ZmBaseController.prototype._deleteListener =
function(ev) {
	this._doDelete(this.getItems(), ev.shiftKey);
};

/**
 * Move button has been pressed, show the dialog.
 * 
 * @private
 */
ZmBaseController.prototype._moveListener =
function(ev, list) {

	this._pendingActionData = list || this.getItems();
	var moveToDialog = appCtxt.getChooseFolderDialog();
	if (!this._moveCb) {
		this._moveCb = new AjxCallback(this, this._moveCallback);
	}
	ZmController.showDialog(moveToDialog, this._moveCb, this._getMoveParams(moveToDialog));
	moveToDialog.registerCallback(DwtDialog.CANCEL_BUTTON, this._clearDialog, this, moveToDialog);
};

/**
 * @protected
 */
ZmBaseController.prototype._getMoveParams =
function(dlg) {

	var org = ZmApp.ORGANIZER[this._app._name] || ZmOrganizer.FOLDER;
	return {
		overviewId:		dlg.getOverviewId(this._app._name),
		data:			this._pendingActionData,
		treeIds:		[org],
		title:			this._getMoveDialogTitle(this._pendingActionData.length, this._pendingActionData),
		description:	ZmMsg.targetFolder,
		treeStyle:		DwtTree.SINGLE_STYLE,
		noRootSelect: 	true, //I don't think you can ever use the "move" dialog to move anything to a root folder... am I wrong?
		appName:		this._app._name
	};
};

/**
 * Switch to selected view.
 * 
 * @private
 */
ZmBaseController.prototype._viewMenuItemListener =
function(ev) {
	if (ev.detail == DwtMenuItem.CHECKED || ev.detail == DwtMenuItem.UNCHECKED) {
		this.switchView(ev.item.getData(ZmOperation.MENUITEM_ID));
	}
};


// new organizer callbacks

/**
 * Created a new tag, now apply it.
 * 
 * @private
 */
ZmBaseController.prototype._tagChangeListener =
function(ev) {

	// only process if current view is this view!
	if (this.isCurrent()) {
		if (ev.type == ZmEvent.S_TAG && ev.event == ZmEvent.E_CREATE && this._pendingActionData) {
			var tag = ev.getDetail("organizers")[0];
			this._doTag(this._pendingActionData, tag, true);
			this._pendingActionData = null;
			this._menuPopdownActionListener();
		}
	}
};

/**
 * Move stuff to a new folder.
 * 
 * @private
 */
ZmBaseController.prototype._moveCallback =
function(folder) {
	this._doMove(this._pendingActionData, folder);
	this._clearDialog(appCtxt.getChooseFolderDialog());
	this._pendingActionData = null;
};

/**
 * Move stuff to a new folder. 
 *
 * @private
 */
ZmBaseController.prototype._moveMenuCallback =
function(moveButton, folder) {
	this._doMove(this._pendingActionData, folder);
	moveButton.getMenu().popdown();
	this._pendingActionData = null;
};

// Data handling

// Actions on items are performed through their containing list
ZmBaseController.prototype._getList =
function(items) {

	items = AjxUtil.toArray(items);
	var item = items[0];
	return item && item.list;
};

// callback (closure) to run when an action has completely finished
ZmBaseController.prototype._getAllDoneCallback = function() {};

/**
 * Shows the given summary as status toast.
 *
 * @param {String}		summary						the text that summarizes the recent action
 * @param {ZmAction}	actionLogItem				the logged action for possible undoing
 * @param {boolean}		showToastOnParentWindow		the toast message should be on the parent window (since the child window is being closed)
 */
ZmBaseController.showSummary =
function(summary, actionLogItem, showToastOnParentWindow) {
	
	if (!summary) {
		return;
	}
	var ctxt = showToastOnParentWindow ? parentAppCtxt : appCtxt;
	var actionController = ctxt.getActionController();
	var undoLink = actionLogItem && actionController && actionController.getUndoLink(actionLogItem);
	if (undoLink && actionController) {
		actionController.onPopup();
		ctxt.setStatusMsg({msg: summary + undoLink, transitions: actionController.getStatusTransitions()});
	} else {
		ctxt.setStatusMsg(summary);
	}
};

/**
 * Flag/unflag an item
 * 
 * @private
 */
ZmBaseController.prototype._doFlag =
function(items, on) {

	items = AjxUtil.toArray(items);
	if (!items.length) { return; }

	if (items[0].isZmItem) {
		if (on !== true && on !== false) {
			on = !items[0].isFlagged;
		}
		var items1 = [];
		for (var i = 0; i < items.length; i++) {
			if (items[i].isFlagged != on) {
				items1.push(items[i]);
			}
		}
	} else {
		items1 = items;
	}

	var params = {items:items1, op:"flag", value:on};
    params.actionTextKey = on ? 'actionFlag' : 'actionUnflag';
	var list = params.list = this._getList(params.items);
	this._setupContinuation(this._doFlag, [on], params);
	list.flagItems(params);
};

// TODO: shouldn't this be in ZmMailItemController?
ZmBaseController.prototype._doMsgPriority = 
function(items, on) {
	items = AjxUtil.toArray(items);
	if (!items.length) { return; }

	if (items[0].isZmItem) {
		if (on !== true && on !== false) {
			on = !items[0].isPriority;
		}
		var items1 = [];
		for (var i = 0; i < items.length; i++) {
			if (items[i].isPriority != on) {
				items1.push(items[i]);
			}
		}
	} else {
		items1 = items;
	}

	var params = {items:items1, op:"priority", value:on};
    params.actionTextKey = on ? 'actionMsgPriority' : 'actionUnMsgPriority';
	var list = params.list = this._getList(params.items);
	this._setupContinuation(this._doMsgPriority, [on], params);
	list.flagItems(params);	
};

/**
 * Tag/untag items
 * 
 * @private
 */
ZmBaseController.prototype._doTag =
function(items, tag, doTag) {

	items = AjxUtil.toArray(items);
	if (!items.length) { return; }

	//see bug 79756 as well as this bug, bug 98316.
	for (var i = 0; i < items.length; i++) {
		if (items[i].cloneOf) {
			items[i] = items[i].cloneOf;
		}
	}

	var params = {items:items, tag:tag, doTag:doTag};
	var list = params.list = this._getList(params.items);
	this._setupContinuation(this._doTag, [tag, doTag], params);
	list.tagItems(params);
};

/**
 * Remove all tags for given items
 * 
 * @private
 */
ZmBaseController.prototype._doRemoveAllTags =
function(items) {

	items = AjxUtil.toArray(items);
	if (!items.length) { return; }

	//see bug 79756 as well as this bug.
	for (var i = 0; i < items.length; i++) {
		if (items[i].cloneOf) {
			items[i] = items[i].cloneOf;
		}
	}
	var params = {items:items};
	var list = params.list = this._getList(params.items);
	this._setupContinuation(this._doRemoveAllTags, null, params);
	list.removeAllTags(params);
};

/**
* Deletes one or more items from the list.
*
* @param items			[Array]			list of items to delete
* @param hardDelete		[boolean]*		if true, physically delete items
* @param attrs			[Object]*		additional attrs for SOAP command
* @param confirmDelete  [Boolean]       user already confirmed hard delete (see ZmBriefcaseController.prototype._doDelete and ZmBriefcaseController.prototype._doDelete2) 
* 
* @private
*/
ZmBaseController.prototype._doDelete =
function(items, hardDelete, attrs, confirmDelete) {

	items = AjxUtil.toArray(items);
	if (!items.length) { return; }

	// If the initial set of deletion items is incomplete (we will be using continuation) then if its deletion
	// from the trash folder mark it as a hardDelete.  Otherwise, upon continuation the items will be moved
	// (Trash to Trash) instead of deleted.
	var folder = this._getSearchFolder();
	var inTrashFolder = (folder && folder.nId == ZmFolder.ID_TRASH);
	if (inTrashFolder) {
		hardDelete = true;
	}

	var params = {
		items:			items,
		hardDelete:		hardDelete,
		attrs:			attrs,
		childWin:		appCtxt.isChildWindow && window,
		closeChildWin:	appCtxt.isChildWindow,
		confirmDelete:	confirmDelete
	};
	var allDoneCallback = this._getAllDoneCallback();
	var list = params.list = this._getList(params.items);
	this._setupContinuation(this._doDelete, [hardDelete, attrs, true], params, allDoneCallback);
	
	if (!hardDelete) {
		var anyScheduled = false;
		for (var i=0, cnt=items.length; i<cnt; i++) {
			if (items[i] && items[i].isScheduled) {
				anyScheduled = true;
				break;
			}
		}
		if (anyScheduled) {
			params.noUndo = true;
			this._popupScheduledWarningDialog(list.deleteItems.bind(list, params));
		} else {
			list.deleteItems(params);
		}
	} else {
		list.deleteItems(params);
	}
};

/**
 * Moves a list of items to the given folder. Any item already in that folder is excluded.
 *
 * @param {Array}	items		a list of items to move
 * @param {ZmFolder}	folder		the destination folder
 * @param {Object}	attrs		the additional attrs for SOAP command
 * @param {Boolean}		isShiftKey	<code>true</code> if forcing a copy action
 * @param {Boolean}		noUndo	<code>true</code> undo not allowed
 * @private
 */
ZmBaseController.prototype._doMove =
function(items, folder, attrs, isShiftKey, noUndo) {

	items = AjxUtil.toArray(items);
	if (!items.length) { return; }

	var move = [];
	var copy = [];
	if (items[0].isZmItem) {
		for (var i = 0; i < items.length; i++) {
			var item = items[i];
			if (!item.folderId || (item.folderId != folder.id || (attrs && attrs.op == "recover"))) {
				if (!this._isItemMovable(item, isShiftKey, folder)) {
					copy.push(item);
				} else {
					move.push(item);
				}
			}
		}
	} else {
		move = items;
	}

	var params = {folder:folder, attrs:attrs, noUndo: noUndo};
    params.errorCallback = this._actionErrorCallback.bind(this);

	var allDoneCallback = this._getAllDoneCallback();
	if (move.length) {
		params.items = move;
		var list = params.list = this._getList(params.items);
		this._setupContinuation(this._doMove, [folder, attrs, isShiftKey], params, allDoneCallback);

		if (folder.isInTrash()) {
			var anyScheduled = false;
			var mItems = AjxUtil.toArray(move);
			for (var i=0, cnt=mItems.length; i<cnt; i++) {
				if (mItems[i] && mItems[i].isScheduled) {
					anyScheduled = true;
					break;
				}
			}
			if (anyScheduled) {
				params.noUndo = true;
				this._popupScheduledWarningDialog(list.moveItems.bind(list, params));
			} else {
				list.moveItems(params);
			}
		}
		else if (folder.id == appCtxt.get(ZmSetting.MAIL_ACTIVITYSTREAM_FOLDER) && items.length == 1) { 
			list.moveItems(params);
			var activityStreamDialog = appCtxt.getActivityStreamFilterDialog();
			activityStreamDialog.setFields(items[0]);
			activityStreamDialog.popup();
		}
		else if (items.length == 1 && folder.id == ZmFolder.ID_INBOX) {
			list.moveItems(params);
			var fromFolder = appCtxt.getById(items[0].folderId);
			if (fromFolder && fromFolder.id == appCtxt.get(ZmSetting.MAIL_ACTIVITYSTREAM_FOLDER)) { 
				var activityStreamDialog = appCtxt.getActivityToInboxFilterDialog();
				activityStreamDialog.setFields(items[0]);
				activityStreamDialog.popup();
			}
		}
		else {
			list.moveItems(params);
		}
	}

	if (copy.length) {
		params.items = copy;
		var list = params.list = this._getList(params.items);
		this._setupContinuation(this._doMove, [folder, attrs, isShiftKey], params, allDoneCallback, true);
		list.copyItems(params);
	}
};

ZmBaseController.prototype._actionErrorCallback =
function(ex){
    return false;
};

ZmBaseController.prototype._popupScheduledWarningDialog =
function(callback) {
	var dialog = appCtxt.getOkCancelMsgDialog();
	dialog.reset();
	dialog.setMessage(ZmMsg.moveScheduledMessageWarning, DwtMessageDialog.WARNING_STYLE);
	dialog.registerCallback(DwtDialog.OK_BUTTON, this._scheduledWarningDialogListener.bind(this, callback, dialog));
	dialog.associateEnterWithButton(DwtDialog.OK_BUTTON);
	dialog.popup(null, DwtDialog.OK_BUTTON);
};

ZmBaseController.prototype._scheduledWarningDialogListener =
function(callback, dialog) {
	dialog.popdown()
	callback();
};

/**
 * Decides whether an item is movable
 *
 * @param {Object}	item			the item to be checked
 * @param {Boolean}		isShiftKey	<code>true</code> if forcing a copy (not a move)
 * @param {ZmFolder}	folder		the folder this item belongs under
 * 
 * @private
 */
ZmBaseController.prototype._isItemMovable =
function(item, isShiftKey, folder) {
	return (!isShiftKey && !item.isReadOnly() && !folder.isReadOnly());
};

/**
 * Modify an item.
 * 
 * @private
 */
ZmBaseController.prototype._doModify =
function(item, mods) {
	var list = this._getList(item);
	list.modifyItem(item, mods);
};

/**
 * Create an item. We need to be passed a list since we may not have one.
 * 
 * @private
 */
ZmBaseController.prototype._doCreate =
function(list, args) {
	list.create(args);
};

// Miscellaneous


/**
 * Add listener to tag menu
 * 
 * @private
 */
ZmBaseController.prototype._setupTagMenu =
function(parent, listener) {
	if (!parent) return;
	var tagMenu = parent.getTagMenu();
	listener = listener || this._listeners[ZmOperation.TAG];
	if (tagMenu) {
		tagMenu.addSelectionListener(listener);
	}
	if (parent.isZmButtonToolBar) {
		var tagButton = parent.getOp(ZmOperation.TAG_MENU);
		if (tagButton) {
			tagButton.addDropDownSelectionListener(this._listeners[ZmOperation.TAG_MENU]);
		}
	}
};

/**
 * setup the move menu
 *
 * @private
 */
ZmBaseController.prototype._setupMoveMenu =
function(parent) {
	if (!parent) {
		return;
	}
	if (!parent.isZmButtonToolBar) {
		return;
	}
	var moveButton = parent.getOp(ZmOperation.MOVE_MENU);
	if (moveButton) {
		moveButton.addDropDownSelectionListener(this._listeners[ZmOperation.MOVE_MENU]);
	}
};

/**
 * Dynamically build the tag menu based on selected items and their tags.
 * 
 * @private
 */
ZmBaseController.prototype._setTagMenu =
function(parent, items) {

	if (!parent) { return; }

	var tagOp = parent.getOp(ZmOperation.TAG_MENU);
	if (tagOp) {
		var tagMenu = parent.getTagMenu();
		if (!tagMenu) { return; }

		// dynamically build tag menu add/remove lists
		items = items || AjxUtil.toArray(this.getItems());

		for (var i=0; i<items.length; i++) {
			if (items[i].cloneOf) {
				items[i] = items[i].cloneOf;
			}
		}

		var account = (appCtxt.multiAccounts && items.length == 1) ? items[0].getAccount() : null;

		// fetch tag tree from appctxt (not cache) for multi-account case
		tagMenu.set(items, appCtxt.getTagTree(account));
		if (parent.isZmActionMenu) {
			tagOp.setText(this._getTagMenuMsg(items.length, items));
		}
		else {
			tagMenu.parent.popup();

			// bug #17584 - we currently don't support creating new tags in new window
			if (appCtxt.isChildWindow || appCtxt.isWebClientOffline()) {
				var mi = tagMenu.getMenuItem(ZmTagMenu.MENU_ITEM_ADD_ID);
				if (mi) {
					mi.setVisible(false);
				}
			}
		}
	}
};

/**
 * copied some from ZmCalendarApp.createMiniCalButton
 * initializes the move button with {@link ZmFolderChooser} as the menu.
 *
 * @param	{DwtButton}	the button
 */
ZmBaseController.prototype._setMoveButton =
function(moveButton) {

	// create menu for button
	var moveMenu = new DwtMenu({parent: moveButton, style:DwtMenu.CALENDAR_PICKER_STYLE, id: "ZmMoveButton_" + this.getCurrentViewId()});
	moveMenu.getHtmlElement().style.width = "auto"; //make it dynamic  (so expanding long named sub-folders would expand width. (plus right now it sets it to 0 due to some styles)
	moveButton.setMenu(moveMenu, true);

	var chooser = this._folderChooser = new ZmFolderChooser({parent:moveMenu});
	var moveParams = this._getMoveParams(chooser);
	moveParams.overviewId += this._currentViewId; //so it works when switching views (cuz the tree has a listener and the tree is shared unless it's different ID). maybe there's a different way to solve this.
	chooser.setupFolderChooser(moveParams, this._moveMenuCallback.bind(this, moveButton));

	return moveButton;
};

/**
 * Resets the available operations on a toolbar or action menu.
 * 
 * @param {DwtControl}	parent		toolbar or action menu
 * @param {number}		num			number of items selected currently
 * @private
 */
ZmBaseController.prototype._resetOperations =
function(parent, num) {

	if (!parent) { return; }

	if (num == 0) {
		parent.enableAll(false);
		parent.enable(this.operationsToEnableOnZeroSelection, true);
	} else if (num == 1) {
		parent.enableAll(true);
		parent.enable(this.operationsToDisableOnSingleSelection, false);
	} else if (num > 1) {
		parent.enableAll(false);
		parent.enable(this.operationsToEnableOnMultiSelection, true);
    }

	// bug: 41758 - don't allow shared items to be tagged
	var folder = (num > 0) && this._getSearchFolder();
	if (folder && folder.isReadOnly()) {
		parent.enable(ZmOperation.TAG_MENU, false);
	}
    //this._resetQuickCommandOperations(parent);
};

/**
 * Resets a single operation on a toolbar or action menu.
 * 
 * @param {DwtControl}	parent		toolbar or action menu
 * @param {number}		num			number of items selected currently
 * @param {constant}	op			operation
 * @private
 */
ZmBaseController.prototype._resetOperation = function(parent, num, op) {};

/**
 * Resets the available options on the toolbar.
 * 
 * @private
 */
ZmBaseController.prototype._resetToolbarOperations =
function() {
	this._resetOperations(this._toolbar[this._currentViewId], this.getItemCount());
};


/**
 * @private
 */
ZmBaseController.prototype._getDefaultFocusItem =
function() {
	return this.getCurrentView();
};

/**
 * Sets a callback that shows a summary of what was done. The first three arguments are
 * provided for overriding classes that want to apply an action to an extended list of
 * items (retrieved via successive search, for example).
 *
 * @param {function}	actionMethod		the controller action method
 * @param {Array}		args				an arg list for above (except for items arg)
 * @param {Hash}		params				the params that will be passed to list action method
 * @param {closure}		allDoneCallback		the callback to run after all items processed
 * 
 * @private
 */
ZmBaseController.prototype._setupContinuation =
function(actionMethod, args, params, allDoneCallback) {
	params.finalCallback = this._continueAction.bind(this, {allDoneCallback:allDoneCallback});
};

/**
 * Runs the "all done" callback and shows a summary of what was done.
 *
 * @param {Hash}		params				a hash of parameters
 * @param {closure}	 	allDoneCallback		the callback to run when we're all done
 * 
 * @private
 */
ZmBaseController.prototype._continueAction =
function(params) {

	if (params.allDoneCallback) {
		params.allDoneCallback();
	}
	ZmBaseController.showSummary(params.actionSummary, params.actionLogItem, params.closeChildWin);
};



ZmBaseController.prototype._bubbleSelectionListener = function(ev) {

	this._actionEv = ev;
	var bubble = ev.item;
	if (ev.detail === DwtEvent.ONDBLCLICK) {
		this._actionEv.bubble = bubble;
		this._actionEv.address = bubble.addrObj || bubble.address;
		this._composeListener(ev);
	}
	else {
		var view = this.getItemView(),
			bubbleList = view && view._bubbleList;

		if (bubbleList && bubbleList.selectAddressText) {
			bubbleList.selectAddressText();
		}
	}
};

ZmBaseController.prototype._bubbleActionListener = function(ev, addr) {

	this._actionEv = ev;
	var bubble = this._actionEv.bubble = ev.item,
		address = this._actionEv.address = addr || bubble.addrObj || bubble.address,
		menu = this._getBubbleActionMenu();

	if (menu) {
		menu.enable(
			[
				ZmOperation.CONTACT,
				ZmOperation.ADD_TO_FILTER_RULE
			],
			!appCtxt.isWebClientOffline()
		);
		this._loadContactForMenu(menu, address, ev);
	}
};

ZmBaseController.prototype._getBubbleActionMenu = function() {

	if (this._bubbleActionMenu) {
		return this._bubbleActionMenu;
	}

	var menuItems = this._getBubbleActionMenuOps();
	var menu = this._bubbleActionMenu = new ZmActionMenu({
		parent:     this._shell,
		menuItems:  menuItems,
		controller: this,
		id:         ZmId.create({
			componentType:  ZmId.WIDGET_MENU,
			componentName:  this._currentViewId,
			app:            this._app
		})
	});

	if (appCtxt.get(ZmSetting.SEARCH_ENABLED)) {
		this._setSearchMenu(menu, false);
	}

	if (appCtxt.get(ZmSetting.FILTERS_ENABLED) && this._setAddToFilterMenu) {
		this._setAddToFilterMenu(menu);
	}

	menu.addPopdownListener(this._bubbleMenuPopdownListener.bind(this));

	for (var i = 0; i < menuItems.length; i++) {
		var menuItem = menuItems[i];
		if (this._listeners[menuItem]) {
			menu.addSelectionListener(menuItem, this._listeners[menuItem]);
		}
	}

	menu.setVisible(true);
	var clipboard = appCtxt.getClipboard();
	if (clipboard) {
		clipboard.init(menu.getOp(ZmOperation.COPY), {
			onMouseDown:    this._clipCopy.bind(this),
			onComplete:     this._clipCopyComplete.bind(this)
		});
	}

	appCtxt.notifyZimlets("onBubbleActionMenu", [this, menuItems, menu]);
	return menu;
};

ZmBaseController.prototype._getBubbleActionMenuOps = function() {

	var ops = [];
	if (AjxClipboard.isSupported()) {
		// we use Zero Clipboard (a Flash hack) to copy address
		ops.push(ZmOperation.COPY);
	}
	ops.push(ZmOperation.SEARCH_MENU);
	ops.push(ZmOperation.NEW_MESSAGE);
	if (appCtxt.get(ZmSetting.CONTACTS_ENABLED)) {
		ops.push(ZmOperation.CONTACT);
	}

	if (appCtxt.get(ZmSetting.FILTERS_ENABLED) && this._filterListener) {
		ops.push(ZmOperation.ADD_TO_FILTER_RULE);
	}

	appCtxt.notifyZimlets("onBubbleActionMenuOps", [this, ops]);
	return ops;
};

// Copies address text from the active bubble to the clipboard.
ZmBaseController.prototype._clipCopy = function(clip) {
	clip.setText(this._actionEv.address + AjxEmailAddress.SEPARATOR);
};

ZmBaseController.prototype._clipCopyComplete = function(clip) {
	this._bubbleActionMenu.popdown();
};

// This will get called before the menu item listener. If that causes issues,
// we can run this function on a timer.
ZmBaseController.prototype._bubbleMenuPopdownListener = function() {

	var itemView = this.getItemView(),
		bubbleList = itemView && itemView._bubbleList,
		bubble = this._actionEv && this._actionEv.bubble;

	if (bubbleList) {
		bubbleList.clearRightSelection();
		if (bubble) {
			bubble.setClassName(bubbleList._normalClass);
		}
	}
	this._actionEv.bubble = null;
};

// handle click on an address (or "Select All") in popup DL expansion list
ZmBaseController.prototype._dlAddrSelected = function(match, ev) {
	this._actionEv.address = match;
	this._composeListener(ev);
};

ZmBaseController.prototype._loadContactForMenu = function(menu, address, ev, imItem) {

	var ac = window.parentAppCtxt || appCtxt;
	var contactsApp = ac.getApp(ZmApp.CONTACTS),
		address = address.isAjxEmailAddress ? address : new AjxEmailAddress(address),
		email = address.getAddress();

	if (!email) {
		return;
	}

	// first check if contact is cached, and no server call is needed
	var contact = contactsApp && contactsApp.getContactByEmail(email);
	if (contact) {
		this._handleResponseGetContact(menu, address, ev, imItem, contact);
		return;
	}

	var op = menu.getOp(ZmOperation.CONTACT);
	if (op) {
		op.setText(ZmMsg.loading);
	}
	if (imItem) {
		if (ZmImApp.updateImMenuItemByAddress(imItem, address, false)) {
			imItem.setText(ZmMsg.loading);
		}
		else {
			imItem = null;	// done updating item, didn't need server call
		}
	}
	menu.popup(0, ev.docX || ev.item.getXW(), ev.docY || ev.item.getYH());
	var respCallback = this._handleResponseGetContact.bind(this, menu, address, ev, imItem);
	contactsApp && contactsApp.getContactByEmail(email, respCallback);
};

ZmBaseController.prototype._handleResponseGetContact = function(menu, address, ev, imItem, contact) {

	this._actionEv.contact = contact;
	this._setContactText(contact, menu);

	if (imItem) {
		if (contact) {
			ZmImApp.updateImMenuItemByContact(imItem, contact, address);
		}
		else {
			ZmImApp.handleResponseGetContact(imItem, address, true);
		}
	}
	menu.popup(0, ev.docX || ev.item.getXW(), ev.docY || ev.item.getYH());
};

/**
 * Sets text to "add" or "edit" based on whether a participant is a contact or not.
 * contact - the contact (or null)
 * extraMenu - see ZmMailListController.prototype._setContactText
 *
 * @private
 */
ZmBaseController.prototype._setContactText = function(contact, menu) {
	ZmBaseController.setContactTextOnMenu(contact, menu || this._actionMenu);
};

/**
 * Sets text to "add" or "edit" based on whether a participant is a contact or not.
 * contact - the contact (or null)
 * menus - array of one or more menus
 *
 * @private
 */
ZmBaseController.setContactTextOnMenu = function(contact, menu) {

	if (!menu) {
		return;
	}

	var newOp = ZmOperation.EDIT_CONTACT;
	var newText = null; //no change ("edit contact")

	if (contact && contact.isDistributionList()) {
		newText = ZmMsg.AB_EDIT_DL;
	}
	else if (contact && contact.isGroup()) {
		newText = ZmMsg.AB_EDIT_GROUP;
	}
	else if (!contact || contact.isGal) {
		// if there's no contact, or it's a GAL contact - there's no "edit" - just "add".
		newText = ZmMsg.AB_ADD_CONTACT;
		newOp = ZmOperation.NEW_CONTACT;
	}

	ZmOperation.setOperation(menu, ZmOperation.CONTACT, newOp, newText);

	if (appCtxt.isWebClientOffline()) {
		menu.enable(ZmOperation.CONTACT, false);
	}
};

/**
 * Add listener to search menu
 *
 * @param parent
 */
ZmBaseController.prototype._setSearchMenu = function(parent, isToolbar) {

	var searchMenu = parent && parent.getSearchMenu && parent.getSearchMenu();
	if (!searchMenu) {
		return;
	}
	searchMenu.addSelectionListener(ZmOperation.SEARCH, this._searchListener.bind(this, AjxEmailAddress.FROM, isToolbar));
	searchMenu.addSelectionListener(ZmOperation.SEARCH_TO, this._searchListener.bind(this, AjxEmailAddress.TO, isToolbar));

	if (this.getSearchFromText()) {
		searchMenu.getMenuItem(ZmOperation.SEARCH).setText(this.getSearchFromText());
	}
	if (this.getSearchToText()) {
		searchMenu.getMenuItem(ZmOperation.SEARCH_TO).setText(this.getSearchToText());
	}
};

/**
 * From Search based on email address.
 *
 * @private
 */
ZmBaseController.prototype._searchListener = function(addrType, isToolbar, ev) {

	var folder = this._getSearchFolder(),
		item = this._actionEv.item,
		address = this._actionEv.address,
		name;

	if (item && item.isZmMailMsg && folder && folder.isOutbound()) {
		/* sent/drafts search from all recipients */
		var toAddrs = item.getAddresses(AjxEmailAddress.TO).getArray(),
			ccAddrs = item.getAddresses(AjxEmailAddress.CC).getArray();

		name = toAddrs.concat(ccAddrs);
	}
	else if (address) {
		name = address.isAjxEmailAddress ? address.getAddress() : address;
	}

	if (name) {
        var ac = window.parentAppCtxt || window.appCtxt;
		var srchCtlr = ac.getSearchController();
		if (addrType === AjxEmailAddress.FROM) {
			srchCtlr.fromSearch(name);
		}
		else if (addrType === AjxEmailAddress.TO) {
			srchCtlr.toSearch(name);
		}
	}
};

/**
 * Compose message to participant.
 *
 * @private
 */
ZmBaseController.prototype._composeListener = function(ev, addr) {

	var addr = addr || (this._actionEv && this._actionEv.address),
		email = addr && addr.toString();

	if (email) {
		AjxDispatcher.run("Compose", {
			action:         ZmOperation.NEW_MESSAGE,
			inNewWindow:    this._app._inNewWindow(ev),
			toOverride:     email + AjxEmailAddress.SEPARATOR
		});
	}
};

/**
 * If there's a contact for the participant, edit it, otherwise add it.
 *
 * @private
 */
ZmBaseController.prototype._contactListener = function(ev) {
	var loadCallback = this._handleLoadContactListener.bind(this);
	AjxDispatcher.require(["ContactsCore", "Contacts"], false, loadCallback, null, true);
};

/**
 * @private
 */
ZmBaseController.prototype._handleLoadContactListener = function() {

	var cc = appCtxt.isChildWindow ? window.parentAppCtxt.getApp(ZmApp.CONTACTS).getContactController() :
									AjxDispatcher.run("GetContactController");
	var contact = this._actionEv.contact;
	if (contact) {
		if (contact.isDistributionList()) {
			this._editListener(this._actionEv, contact);
			return;
		}
		if (contact.isLoaded) {
			var isDirty = contact.isGal;
			cc.show(contact, isDirty);
		} else {
			var callback = this._loadContactCallback.bind(this);
			contact.load(callback);
		}
	} else {
		var contact = cc._createNewContact(this._actionEv);
		cc.show(contact, true);
	}
	if (appCtxt.isChildWindow) {
		window.close();
	}
};

ZmBaseController.prototype.getSearchFromText = function() {
	return null;
};

ZmBaseController.prototype.getSearchToText = function() {
	return null;
};

ZmBaseController.prototype._createNewContact = function(ev) {
	var contact = new ZmContact(null);
	contact.initFromEmail(ev.address);
	return contact;
};

ZmBaseController.prototype._loadContactCallback = function(resp, contact) {
	AjxDispatcher.run("GetContactController").show(contact);
};

ZmBaseController.prototype._getSearchFolder = function() {
	var id = this._getSearchFolderId();
	return id && appCtxt.getById(id);
};

/**
 * This method gets overridden if folder id is retrieved another way
 *
 * @param {boolean}		allowComplex	if true, search can have other terms aside from the folder term
 * @private
 */
ZmBaseController.prototype._getSearchFolderId = function(allowComplex) {
	var s = this._activeSearch && this._activeSearch.search;
	return s && (allowComplex || s.isSimple()) && s.folderId;
};
}
if (AjxPackage.define("zimbraMail.share.controller.ZmListController")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines a list controller class.
 *
 */

/**
 * This class is a base class for any controller that manages a list of items such as mail messages
 * or contacts. It can handle alternative views of the same list.
 *
 * @author Conrad Damon
 *
 * @param {DwtControl}					container					the containing shell
 * @param {ZmApp}						app							the containing application
 * @param {constant}					type						type of controller
 * @param {string}						sessionId					the session id
 * @param {ZmSearchResultsController}	searchResultsController		containing controller
 * 
 * @extends		ZmBaseController
 */
ZmListController = function(container, app, type, sessionId, searchResultsController) {

	if (arguments.length == 0) { return; }
	ZmBaseController.apply(this, arguments);

	// hashes keyed by view type
	this._navToolBar = {};			// ZmNavToolBar
	this._listView = this._view;	// ZmListView (back-compatibility for bug 60073)

	this._list = null;				// ZmList
	this._activeSearch = null;
	this._newButton = null;
	this._actionMenu = null;		// ZmActionMenu
	this._actionEv = null;
	this._lastSelectedListItem = null; // For Bug: 106342 [Selection of list item is lost after context-menu is destroyed]
	
	if (this.supportsDnD()) {
		this._dropTgt = new DwtDropTarget("ZmTag");
		this._dropTgt.markAsMultiple();
		this._dropTgt.addDropListener(this._dropListener.bind(this));
	}

	this._menuPopdownListener = this._menuPopdownActionListener.bind(this);
	
	this._itemCountText = {};
	this._continuation = {count:0, totalItems:0};
};

ZmListController.prototype = new ZmBaseController;
ZmListController.prototype.constructor = ZmListController;

ZmListController.prototype.isZmListController = true;
ZmListController.prototype.toString = function() { return "ZmListController"; };

// When performing a search action (bug 10317) on all items (including those not loaded),
// number of items to load on each search to work through all results. Should be a multiple
// of ZmList.CHUNK_SIZE. Make sure to test if you change these.
ZmListController.CONTINUATION_SEARCH_ITEMS = 500;

// states of the progress dialog
ZmListController.PROGRESS_DIALOG_INIT	= "INIT";
ZmListController.PROGRESS_DIALOG_UPDATE	= "UPDATE";
ZmListController.PROGRESS_DIALOG_CLOSE	= "CLOSE";


/**
 * Performs some setup for displaying the given search results in a list view. Subclasses will need
 * to do the actual display work, typically by calling the list view's {@link #set} method.
 *
 * @param {ZmSearchResult}	searchResults		the search results
 */
ZmListController.prototype.show	=
function(searchResults) {
	
	this._activeSearch = searchResults;
	// save current search for use by replenishment
	if (searchResults) {
		this._currentSearch = searchResults.search;
		this._activeSearch.viewId = this._currentSearch.viewId = this._currentViewId;
	}
	this.currentPage = 1;
	this.maxPage = 1;
};

/**
 * Returns the current list view.
 * 
 * @return {ZmListView}	the list view
 */
ZmListController.prototype.getListView =
function() {
	return this._view[this._currentViewId];
};

/**
 * Gets the current search results.
 * 
 * @return	{ZmSearchResults}	current search results
 */
ZmListController.prototype.getCurrentSearchResults =
function() {
	return this._activeSearch;
};

/**
 * Gets the search string.
 * 
 * @return	{String}	the search string
 */
ZmListController.prototype.getSearchString =
function() {
	return this._currentSearch ? this._currentSearch.query : "";
};


ZmListController.prototype.setSearchString =
function(query) {
	this._currentSearch.query = query;
};

/**
 * Gets the search string hint.
 * 
 * @return	{String}	the search string hint
 */
ZmListController.prototype.getSearchStringHint =
function() {
	return this._currentSearch ? this._currentSearch.queryHint : "";
};

ZmListController.prototype.getSelection =
function(view) {
    view = view || this.getListView();
    return view ? view.getSelection() : [];
};

ZmListController.prototype.getSelectionCount =
function(view) {
    view = view || this.getListView();
    return view ? view.getSelectionCount() : 0;
};

/**
 * Gets the list.
 * 
 * @return	{ZmList}		the list
 */
ZmListController.prototype.getList =
function() {
	return this._list;
};

/**
 * Sets the list.
 * 
 * @param	{ZmList}	newList		the new list
 */
ZmListController.prototype.setList =
function(newList) {
	if (newList != this._list && newList.isZmList) {
		if (this._list) {
			this._list.clear();	// also removes change listeners
		}
		this._list = newList;
		this._list.controller = this;
	}
};

/**
 * Sets the "has more" state.
 * 
 * @param	{Boolean}	hasMore		<code>true</code> if has more
 */
ZmListController.prototype.setHasMore =
function(hasMore) {
	// Note: This is a bit of a HACK that is an attempt to overcome an
	// offline issue. The problem is during initial sync when more
	// messages come in: the forward navigation arrow doesn't get enabled.
	
	if (hasMore && this._list) {
		// bug: 30546
		this._list.setHasMore(hasMore);
		this._resetNavToolBarButtons();
	}
};

/**
 * Returns a list of the selected items.
 */
ZmListController.prototype.getItems =
function() {
	return this.getSelection();
};

/**
 * Returns the number of selected items.
 */
ZmListController.prototype.getItemCount =
function() {
	return this.getSelectionCount();
};

/**
 * Handles the key action.
 * 
 * @param	{constant}	actionCode		the action code
 * @return	{Boolean}	<code>true</code> if the action is handled
 */
ZmListController.prototype.handleKeyAction =
function(actionCode, ev) {

	DBG.println(AjxDebug.DBG3, "ZmListController.handleKeyAction");
	var listView = this._view[this._currentViewId];
	var result = false;
    var activeEl = document.activeElement;

	switch (actionCode) {

		case DwtKeyMap.DBLCLICK:
            if (activeEl && activeEl.nodeName && activeEl.nodeName.toLowerCase() === 'a') {
                return false;
            }
			return listView.handleKeyAction(actionCode);

		case ZmKeyMap.SHIFT_DEL:
		case ZmKeyMap.DEL:
			var tb = this.getCurrentToolbar();
			var button = tb && (tb.getButton(ZmOperation.DELETE) || tb.getButton(ZmOperation.DELETE_MENU));
			if (button && button.getEnabled()) {
				this._doDelete(this.getSelection(), (actionCode == ZmKeyMap.SHIFT_DEL));
				result = true;
			}
			break;

		case ZmKeyMap.NEXT_PAGE:
			var ntb = this._navToolBar[this._currentViewId];
			var button = ntb ? ntb.getButton(ZmOperation.PAGE_FORWARD) : null;
			if (button && button.getEnabled()) {
				this._paginate(this._currentViewId, true);
				result = true;
			}
			break;

		case ZmKeyMap.PREV_PAGE:
			var ntb = this._navToolBar[this._currentViewId];
			var button = ntb ? ntb.getButton(ZmOperation.PAGE_BACK) : null;
			if (button && button.getEnabled()) {
				this._paginate(this._currentViewId, false);
				result = true;
			}
			break;

		// Esc pops search results tab
		case ZmKeyMap.CANCEL:
			var ctlr = this.isSearchResults && this.searchResultsController;
			if (ctlr) {
				ctlr._closeListener();
			}
			break;

		default:
			return ZmBaseController.prototype.handleKeyAction.apply(this, arguments);
	}
	return result;
};

// Returns a list of desired action menu operations
ZmListController.prototype._getActionMenuOps = function() {};

/**
 * @private
 */
ZmListController.prototype._standardActionMenuOps =
function() {
	return [ZmOperation.TAG_MENU, ZmOperation.MOVE, ZmOperation.PRINT];
};

/**
 * @private
 */
ZmListController.prototype._participantOps =
function() {
	var ops = [ZmOperation.SEARCH_MENU];

	if (appCtxt.get(ZmSetting.MAIL_ENABLED)) {
		ops.push(ZmOperation.NEW_MESSAGE);
	}

	if (appCtxt.get(ZmSetting.CONTACTS_ENABLED)) {
		ops.push(ZmOperation.CONTACT);
	}

	return ops;
};

/**
 * Initializes action menu: menu items and listeners
 * 
 * @private
 */
ZmListController.prototype._initializeActionMenu =
function() {

	if (this._actionMenu) { return; }

	var menuItems = this._getActionMenuOps();
	if (!menuItems) { return; }

	var menuParams = {parent:this._shell,
		menuItems:	menuItems,
		context:	this._getMenuContext(),
		controller:	this
	};
	this._actionMenu = new ZmActionMenu(menuParams);
	this._addMenuListeners(this._actionMenu);
	if (appCtxt.get(ZmSetting.TAGGING_ENABLED)) {
		this._setupTagMenu(this._actionMenu);
	}
};

/**
 * Sets up tab groups (focus ring).
 *
 * @private
 */
ZmListController.prototype._initializeTabGroup =
function(view) {
	if (this._tabGroups[view]) { return; }

	ZmBaseController.prototype._initializeTabGroup.apply(this, arguments);

	if (this._view[view]._compositeTabGroup) {
		var tabSize = this._tabGroups[view].size();
		this._tabGroups[view].addMember(this._view[view]._compositeTabGroup, tabSize - 1);
	}
};

/**
 * Gets the tab group.
 * 
 * @return	{Object}	the tab group
 */
ZmListController.prototype.getTabGroup =
function() {
	return this._tabGroups[this._currentViewId];
};

/**
 * @private
 */
ZmListController.prototype._addMenuListeners =
function(menu) {

	var menuItems = menu.opList;
	for (var i = 0; i < menuItems.length; i++) {
		var menuItem = menuItems[i];
		if (this._listeners[menuItem]) {
			menu.addSelectionListener(menuItem, this._listeners[menuItem], 0);
		}
	}
	menu.addPopdownListener(this._menuPopdownListener);
};

ZmListController.prototype._menuPopdownActionListener =
function(ev) {

	var view = this.getListView();
	if (!this._pendingActionData) {
		if (view && view.handleActionPopdown) {
			view.handleActionPopdown(ev);
		}
	}
	// Reset back to item count unless there is multiple selection
	var selCount = view ? view.getSelectionCount() : -1;
	if (selCount <= 1) {
		this._setItemCountText();
	}
};



// List listeners

/**
 * List selection event - handle flagging if a flag icon was clicked, otherwise
 * reset the toolbar based on how many items are selected.
 * 
 * @private
 */
ZmListController.prototype._listSelectionListener =
function(ev) {

	if (ev.field == ZmItem.F_FLAG) {
		this._doFlag([ev.item]);
		return true;
	} 
	else {
		var lv = this._listView[this._currentViewId];
		if (lv) {
			if (appCtxt.get(ZmSetting.SHOW_SELECTION_CHECKBOX) && !ev.ctrlKey) {
				if (lv.setSelectionHdrCbox) {
					lv.setSelectionHdrCbox(false);
				}
			}
			this._resetOperations(this.getCurrentToolbar(), lv.getSelectionCount());
			if (ev.shiftKey) {
				this._setItemSelectionCountText();
			}
			else {
				this._setItemCountText();
			}
		}
	}
	return false;
};

/**
 * List action event - set the dynamic tag menu, and enable operations in the
 * action menu based on the number of selected items. Note that the menu is not
 * actually popped up here; that's left up to the subclass, which should
 * override this function.
 * 
 * @private
 */
ZmListController.prototype._listActionListener =
function(ev) {

	this._actionEv = ev;
	var actionMenu = this.getActionMenu();
	if (appCtxt.get(ZmSetting.TAGGING_ENABLED)) {
		this._setTagMenu(actionMenu);
	}

    if (appCtxt.get(ZmSetting.SEARCH_ENABLED)) {
        this._setSearchMenu(actionMenu);
    }
	this._resetOperations(actionMenu, this.getSelectionCount());
	this._setItemSelectionCountText();
};


// Navbar listeners

/**
 * @private
 */
ZmListController.prototype._navBarListener =
function(ev) {

	// skip listener for non-current views
	if (!this.isCurrent()) { return; }

	var op = ev.item.getData(ZmOperation.KEY_ID);

	if (op == ZmOperation.PAGE_BACK || op == ZmOperation.PAGE_FORWARD) {
		this._paginate(this._currentViewId, (op == ZmOperation.PAGE_FORWARD));
	}
};

// Drag and drop listeners

/**
 * @private
 */
ZmListController.prototype._dragListener =
function(ev) {

	if (this.isSearchResults && ev.action == DwtDragEvent.DRAG_START) {
		this.searchResultsController.showOverview(true);
	}
	else if (ev.action == DwtDragEvent.SET_DATA) {
		ev.srcData = {data: ev.srcControl.getDnDSelection(), controller: this};
	}
	else if (this.isSearchResults && (ev.action == DwtDragEvent.DRAG_END || ev.action == DwtDragEvent.DRAG_CANCEL)) {
		this.searchResultsController.showOverview(false);
	}
};

/**
 * The list view as a whole is the drop target, since it's the lowest-level widget. Still, we
 * need to find out which item got dropped onto, so we get that from the original UI event
 * (a mouseup). The header is within the list view, but not an item, so it's not a valid drop
 * target. One drawback of having the list view be the drop target is that we can't exercise
 * fine-grained control on what's a valid drop target. If you enter via an item and then drag to
 * the header, it will appear to be valid.
 * 
 * @protected
 */
ZmListController.prototype._dropListener =
function(ev) {

	var view = this._view[this._currentViewId];
	var div = view.getTargetItemDiv(ev.uiEvent);
	var item = view.getItemFromElement(div);

	// only tags can be dropped on us
	var data = ev.srcData.data;
	if (ev.action == DwtDropEvent.DRAG_ENTER) {
		ev.doIt = (item && (item instanceof ZmItem) && !item.isReadOnly() && this._dropTgt.isValidTarget(data));
        // Bug: 44488 - Don't allow dropping tag of one account to other account's item
        if (appCtxt.multiAccounts) {
           var listAcctId = item ? item.getAccount().id : null;
           var tagAcctId = (data.account && data.account.id) || data[0].account.id;
           if (listAcctId != tagAcctId) {
               ev.doIt = false;
           }
        }
		DBG.println(AjxDebug.DBG3, "DRAG_ENTER: doIt = " + ev.doIt);
		if (ev.doIt) {
			view.dragSelect(div);
		}
	} else if (ev.action == DwtDropEvent.DRAG_DROP) {
		view.dragDeselect(div);
		var items = [item];
		var sel = this.getSelection();
		if (sel.length) {
			var vec = AjxVector.fromArray(sel);
			if (vec.contains(item)) {
				items = sel;
			}
		}
		this._doTag(items, data, true);
	} else if (ev.action == DwtDropEvent.DRAG_LEAVE) {
		view.dragDeselect(div);
	} else if (ev.action == DwtDropEvent.DRAG_OP_CHANGED) {
		// nothing
	}
};

/**
 * @private
 */

/**
 * returns true if the search folder is drafts
 */
ZmListController.prototype.isDraftsFolder =
function() {
	var folder = this._getSearchFolder();
	if (!folder) {
		return false;
	}
	return folder.nId ==  ZmFolder.ID_DRAFTS;
};

/**
 * returns true if the search folder is drafts
 */
ZmListController.prototype.isOutboxFolder =
function() {
    var folder = this._getSearchFolder();
    if (!folder) {
        return false;
    }
    return folder.nId == ZmFolder.ID_OUTBOX;
};

/**
 * returns true if the search folder is sync failures
 */
ZmListController.prototype.isSyncFailuresFolder =
function() {
	var folder = this._getSearchFolder();
	if (!folder) {
		return false;
	}
	return folder.nId ==  ZmFolder.ID_SYNC_FAILURES;
};


// Actions on items are performed through their containing list
ZmListController.prototype._getList =
function(items) {

	var list = ZmBaseController.prototype._getList.apply(this, arguments);
	if (!list) {
		list = this._list;
	}

	return list;
};

// if items were removed, see if we need to fetch more
ZmListController.prototype._getAllDoneCallback =
function() {
	return this._checkItemCount.bind(this);
};

/**
 * Manages the progress dialog that appears when an action is performed on a large number of items.
 * The arguments include a state and any arguments relative to that state. The state is one of:
 * 
 * 			ZmListController.PROGRESS_DIALOG_INIT
 *			ZmListController.PROGRESS_DIALOG_UPDATE
 *			ZmListController.PROGRESS_DIALOG_CLOSE
 *  
 * @param {hash}		params		a hash of params:
 * @param {constant}	state		state of the dialog
 * @param {AjxCallback}	callback	cancel callback (INIT)
 * @param {string}		summary		summary text (UPDATE)
 */
ZmListController.handleProgress =
function(params) {

	var dialog = appCtxt.getCancelMsgDialog();
	if (params.state == ZmListController.PROGRESS_DIALOG_INIT) {
		dialog.reset();
		dialog.registerCallback(DwtDialog.CANCEL_BUTTON, params.callback);
		ZmListController.progressDialogReady = true;
	}
	else if (params.state == ZmListController.PROGRESS_DIALOG_UPDATE && ZmListController.progressDialogReady) {
		dialog.setMessage(params.summary, DwtMessageDialog.INFO_STYLE, AjxMessageFormat.format(ZmMsg.inProgress));
		if (!dialog.isPoppedUp()) {
			dialog.popup();
		}
	}
	else if (params.state == ZmListController.PROGRESS_DIALOG_CLOSE) {
		dialog.unregisterCallback(DwtDialog.CANCEL_BUTTON);
		dialog.popdown();
		ZmListController.progressDialogReady = false;
	}
};


// Pagination

/**
 * @private
 */
ZmListController.prototype._cacheList =
function(search, offset) {

	if (this._list) {
		var newList = search.getResults().getVector();
		offset = offset ? offset : parseInt(search.getAttribute("offset"));
		this._list.cache(offset, newList);
	} else {
		this._list = search.getResults(type);
	}
};

/**
 * @private
 */
ZmListController.prototype._search =
function(view, offset, limit, callback, isCurrent, lastId, lastSortVal) {
	var originalSearch = this._activeSearch && this._activeSearch.search;
	var params = {
		query:			this.getSearchString(),
		queryHint:		this.getSearchStringHint(),
		types:			originalSearch && originalSearch.types || [], // use types from original search
		userInitiated:	originalSearch && originalSearch.userInitiated,
		sortBy:			appCtxt.get(ZmSetting.SORTING_PREF, view),
		offset:			offset,
		limit:			limit,
		lastId:			lastId,
		lastSortVal:	lastSortVal
	};
	// add any additional params...
	this._getMoreSearchParams(params);

	var search = new ZmSearch(params);
	if (isCurrent) {
		this._currentSearch = search;
	}

	appCtxt.getSearchController().redoSearch(search, true, null, callback);
};

/**
 * Gets next or previous page of items. The set of items may come from the
 * cached list, or from the server (using the current search as a base).
 * <p>
 * The loadIndex is the index'd item w/in the list that needs to be loaded -
 * initiated only when user is in CV and pages a conversation that has not
 * been loaded yet.</p>
 * <p>
 * Note that this method returns a value even though it may make an
 * asynchronous SOAP request. That's possible as long as no caller
 * depends on the results of that request. Currently, the only caller that
 * looks at the return value acts on it only if no request was made.</p>
 *
 * @param {constant}	view		the current view
 * @param {Boolean}	forward		if <code>true</code>, get next page rather than previous
 * @param {int}		loadIndex	the index of item to show
 * @param {int}	limit		the number of items to fetch
 * 
 * @private
 */
ZmListController.prototype._paginate =
function(view, forward, loadIndex, limit) {

	var needMore = false;
	var lv = this._view[view];
	if (!lv) { return; }
	var offset, max;

    limit = limit || lv.getLimit(offset);

	if (lv._isPageless) {
		offset = this._list.size();
		needMore = true;
	} else {
		offset = lv.getNewOffset(forward);
		needMore = (offset + limit > this._list.size());
		this.currentPage = this.currentPage + (forward ? 1 : -1);
		this.maxPage = Math.max(this.maxPage, this.currentPage);
	}

	// see if we're out of items and the server has more
	if (needMore && this._list.hasMore()) {
		lv.offset = offset; // cache new offset
		if (lv._isPageless) {
			max = limit;
		} else {
			// figure out how many items we need to fetch
			var delta = (offset + limit) - this._list.size();
			max = delta < limit && delta > 0 ? delta : limit;
			if (max < limit) {
				offset = ((offset + limit) - max) + 1;
			}
		}

		// handle race condition - user has paged quickly and we don't want
		// to do second fetch while one is pending
		if (this._searchPending) { return false;	}

		// figure out if this requires cursor-based paging
		var list = lv.getList();
		var lastItem = list && list.getLast();
		var lastSortVal = (lastItem && lastItem.id) ? lastItem.sf : null;
		var lastId = lastSortVal ? lastItem.id : null;

		this._setItemCountText(ZmMsg.loading);

		// get next page of items from server; note that callback may be overridden
		this._searchPending = true;
		var respCallback = this._handleResponsePaginate.bind(this, view, false, loadIndex, offset);
		this._search(view, offset, max, respCallback, true, lastId, lastSortVal);
		return false;
	} else if (!lv._isPageless) {
		lv.offset = offset; // cache new offset
		this._resetOperations(this._toolbar[view], 0);
		this._resetNavToolBarButtons(view);
		this._setViewContents(view);
		this._resetSelection();
		return true;
	}
};

/**
 * Updates the list and the view after a new page of items has been retrieved.
 *
 * @param {constant}	view				the current view
 * @param {Boolean}	saveSelection			if <code>true</code>, maintain current selection
 * @param {int}	loadIndex				the index of item to show
 * @param {ZmCsfeResult}	result			the result of SOAP request
 * @param {Boolean}	ignoreResetSelection	if <code>true</code>, do not reset selection
 * 
 * @private
 */
ZmListController.prototype._handleResponsePaginate =
function(view, saveSelection, loadIndex, offset, result, ignoreResetSelection) {

	var searchResult = result.getResponse();

	// update "more" flag
	this._list.setHasMore(searchResult.getAttribute("more"));

	this._cacheList(searchResult, offset);

	var lv = this._view[this._currentViewId];
	var num = lv._isPageless ? this.getSelectionCount() : 0;
	this._resetOperations(this._toolbar[view], num);

	// remember selected index if told to
	var selItem = saveSelection ? this.getSelection()[0] : null;
	var selectedIdx = selItem ? lv.getItemIndex(selItem) : -1;

	var items = searchResult && searchResult.getResults().getArray();
	if (lv._isPageless && items && items.length) {
		lv._itemsToAdd = items;
	} else {
		lv._itemsToAdd = null;
	}
	var wasEmpty = (lv._isPageless && (lv.size() == 0));

	this._setViewContents(view);

	// add new items to selection if all results selected, in a way that doesn't call deselectAll()
	if (lv.allSelected) {
		for (var i = 0, len = items.length; i < len; i++) {
			lv.selectItem(items[i], true);
			lv.setSelectionCbox(items[i], false);
		}
		lv.setSelectionHdrCbox(true);
		DBG.println("scr", "pagination - selected more items: " + items.length);
		DBG.println("scr", "items selected: " + this.getSelectionCount());
	}
	this._resetNavToolBarButtons(view);

	// bug fix #5134 - some views may not want to reset the current selection
	if (!ignoreResetSelection && !lv._isPageless) {
		this._resetSelection(selectedIdx);
	} else if (wasEmpty) {
		lv._setNextSelection();
	}

	this._searchPending = false;
};

/**
 * @private
 */
ZmListController.prototype._getMoreSearchParams =
function(params) {
	// overload me if more params are needed for SearchRequest
};

/**
 * @private
 */
ZmListController.prototype._checkReplenish =
function(callback) {

	var view = this.getListView();
	var list = view.getList();
	// don't bother if the view doesn't really have a list
	var replenishmentDone = false;
	if (list) {
		var replCount = view.getLimit() - view.size();
		if (replCount > view.getReplenishThreshold()) {
			this._replenishList(this._currentViewId, replCount, callback);
			replenishmentDone = true;
		}
	}
	if (callback && !replenishmentDone) {
		callback.run();
	}
};

/**
 * All items in the list view are gone - show "No Results".
 * 
 * @private
 */
ZmListController.prototype._handleEmptyList =
function(listView) {
	if (this.currentPage > 1) {
		this._paginate(this._currentViewId, false, 0);
	} else {
		listView.removeAll(true);
		listView._setNoResultsHtml();
		this._resetNavToolBarButtons();
		listView._checkItemCount();
	}
};

/**
 * @private
 */
ZmListController.prototype._replenishList =
function(view, replCount, callback) {

	// determine if there are any more items to replenish with
	var idxStart = this._view[view].offset + this._view[view].size();
	var totalCount = this._list.size();

	if (idxStart < totalCount) {
		// replenish from cache
		var idxEnd = (idxEnd > totalCount) ? totalCount : (idxStart + replCount);
		var list = this._list.getVector().getArray();
		var sublist = list.slice(idxStart, idxEnd);
		var subVector = AjxVector.fromArray(sublist);
		this._view[view].replenish(subVector);
		if (callback) {
			callback.run();
		}
	} else {
		// replenish from server request
		this._getMoreToReplenish(view, replCount, callback);
	}
};

/**
 * @private
 */
ZmListController.prototype._resetSelection =
function(idx) {
	var list = this.getListView().getList();
	if (list) {
		var selIdx = idx >= 0 ? idx : 0;
		var first = list.get(selIdx);
		this._view[this._currentViewId].setSelection(first, false);
	}
};

/**
 * Requests replCount items from the server to replenish current listview.
 *
 * @param {constant}	view		the current view to replenish
 * @param {int}	replCount 	the number of items to replenish
 * @param {AjxCallback}	callback	the async callback
 * 
 * @private
 */
ZmListController.prototype._getMoreToReplenish =
function(view, replCount, callback) {

	if (this._list.hasMore()) {
		// use a cursor if we can
		var list = this._view[view].getList();
		var lastItem = list.getLast();
		var lastSortVal = (lastItem && lastItem.id) ? lastItem.sf : null;
		var lastId = lastSortVal ? lastItem.id : null;
		var respCallback = this._handleResponseGetMoreToReplenish.bind(this, view, callback);
		this._search(view, this._list.size(), replCount, respCallback, false, lastId, lastSortVal);
	} else {
		if (callback) {
			callback.run();
		}
	}
};

/**
 * @private
 */
ZmListController.prototype._handleResponseGetMoreToReplenish =
function(view, callback, result) {

	var searchResult = result.getResponse();

	// set updated has more flag
	var more = searchResult.getAttribute("more");
	this._list.setHasMore(more);

	// cache search results into internal list
	this._cacheList(searchResult);

	// update view w/ replenished items
	var list = searchResult.getResults().getVector();
	this._view[view].replenish(list);

	// reset forward pagination button only
	this._toolbar[view].enable(ZmOperation.PAGE_FORWARD, more);

	if (callback) {
		callback.run(result);
	}
};

ZmListController.prototype._initializeNavToolBar =
function(view) {
	var prevButton, nextButton;

	if (view.includes('TKL')) {
		prevButton = this._toolbar[view].getButton(ZmOperation.VIEW_MENU);
	}

	if (view.includes('MSG')) {
		prevButton = this._toolbar[view].getButton(ZmOperation.DETACH);
	}

	var tb = new ZmNavToolBar({parent:this._toolbar[view], context:view, prevButton: prevButton, nextButton: nextButton});
	this._setNavToolBar(tb, view);
};

ZmListController.prototype._setNavToolBar =
function(toolbar, view) {
	this._navToolBar[view] = toolbar;
	if (this._navToolBar[view]) {
		var navBarListener = this._navBarListener.bind(this);
		this._navToolBar[view].addSelectionListener(ZmOperation.PAGE_BACK, navBarListener);
		this._navToolBar[view].addSelectionListener(ZmOperation.PAGE_FORWARD, navBarListener);
	}
};

/**
 * @private
 */
ZmListController.prototype._resetNavToolBarButtons =
function(view) {

	var lv;
    if (view) {
        lv = this._view[view];
    } else {
        lv = this.getListView();
        view = this._currentViewId;
    }
	if (!lv) { return; }

	if (lv._isPageless) {
		this._setItemCountText();
	}

	if (!this._navToolBar[view]) { return; }

	this._navToolBar[view].enable(ZmOperation.PAGE_BACK, lv.offset > 0);

	// determine if we have more cached items to show (in case hasMore is wrong)
	var hasMore = false;
	if (this._list) {
		hasMore = this._list.hasMore();
		if (!hasMore && ((lv.offset + lv.getLimit()) < this._list.size())) {
			hasMore = true;
		}
	}

	this._navToolBar[view].enable(ZmOperation.PAGE_FORWARD, hasMore);

	this._navToolBar[view].setText(this._getNavText(view));
};

/**
 * @private
 */
ZmListController.prototype.enablePagination =
function(enabled, view) {

	if (!this._navToolBar[view]) { return; }

	if (enabled) {
		this._resetNavToolBarButtons(view);
	} else {
		this._navToolBar[view].enable([ZmOperation.PAGE_BACK, ZmOperation.PAGE_FORWARD], false);
	}
};

/**
 * @private
 */
ZmListController.prototype._getNavText =
function(view) {

	var se = this._getNavStartEnd(view);
	if (!se) { return ""; }

    var size  = se.size;
    var msg   = "";
    if (size === 0) {
        msg = AjxMessageFormat.format(ZmMsg.navTextNoItems, ZmMsg[ZmApp.NAME[ZmApp.TASKS]]);
    } else if (size === 1) {
        msg = AjxMessageFormat.format(ZmMsg.navTextOneItem, ZmMsg[ZmItem.MSG_KEY[ZmItem.TASK]]);
    } else {
        // Multiple items
        var lv    = this._view[view];
        var limit = se.limit;
        if (size < limit) {
            // We have the exact size of the filtered items
            msg = AjxMessageFormat.format(ZmMsg.navTextWithTotal, [se.start, se.end, size]);
        } else {
            // If it's more than the limit, we don't have an exact count
            // available from the server
            var sizeText = this._getUpperLimitSizeText(size);
            var msgText = sizeText ? ZmMsg.navTextWithTotal : ZmMsg.navTextRange;
            msg = AjxMessageFormat.format(msgText, [se.start, se.end, sizeText]);
        }
    }
    return msg;
};

/**
 * @private
 */
ZmListController.prototype._getNavStartEnd =
function(view) {

	var lv = this._view[view];
	var limit = lv.getLimit();
	var size = this._list ? this._list.size() : 0;

	var start, end;
	if (size > 0) {
		start = lv.offset + 1;
		end = Math.min(lv.offset + limit, size);
	}

	return (start && end) ? {start:start, end:end, size:size, limit:limit} : null;
};

/**
 * @private
 */
ZmListController.prototype._getNumTotal =
function() {

	var folderId = this._getSearchFolderId();
	if (folderId && (folderId != ZmFolder.ID_TRASH)) {
		var folder = appCtxt.getById(folderId);
		if (folder) {
			return folder.numTotal;
		}
	}
	return null;
};

/**
 * @private
 */
ZmListController.prototype.getActionMenu =
function() {
	if (!this._actionMenu) {
		this._initializeActionMenu();
		//DBG.timePt("_initializeActionMenu");
		this._resetOperations(this._actionMenu, 0);
		//DBG.timePt("this._resetOperation(actionMenu)");
	}
	return this._actionMenu;
};

/**
 * Returns the context for the action menu created by this controller (used to create
 * an ID for the menu).
 * 
 * @private
 */
ZmListController.prototype._getMenuContext =
function() {
	return this._app && this._app._name;
};

/**
 * @private
 */
ZmListController.prototype._getItemCountText =
function() {

	var size = this._getItemCount();
	// Size can be null or a number
	if (!size) { return ""; }

	var lv = this._view[this._currentViewId],
		list = lv && lv._list,
		type = lv._getItemCountType(),
		total = this._getNumTotal(),
		num = total || size,
		countKey = 'type' + AjxStringUtil.capitalizeFirstLetter(ZmItem.MSG_KEY[type]),
        typeText = type ? AjxMessageFormat.format(ZmMsg[countKey], num) : "";

	if (total && (size != total)) {
		return AjxMessageFormat.format(ZmMsg.itemCount1, [size, total, typeText]);
	} else {
		var sizeText = this._getUpperLimitSizeText(size);
		return AjxMessageFormat.format(ZmMsg.itemCount, [sizeText, typeText]);
	}
};

ZmListController.prototype._getUpperLimitSizeText =
function(size) {
    var sizeText = size;
    if (this._list.hasMore()) {
        //show 4+, 5+, 10+, 20+, 100+, 200+
        var granularity = size < 10 ? 1	: size < 100 ? 10 : 100;
        sizeText = (Math.floor(size / granularity)) * granularity + "+"; //round down to the chosen granularity
    }
    return sizeText;

}



ZmListController.prototype._getItemCount =
function() {
	var lv = this.getListView();
	var list = lv && lv._list;
	if (!list) {
        return 0;
    }
	return list.size();
};

/**
 * Sets the text that shows the number of items, if we are pageless.
 * 
 * @private
 */
ZmListController.prototype._setItemCountText =
function(text) {

	text = text || this._getItemCountText();
	var field = this._itemCountText[this._currentViewId];
	if (field) {
		field.setText(text);
	}
};

// Returns text that describes how many items are selected for action
ZmListController.prototype._getItemSelectionCountText = function() {

	var lv = this._view[this._currentViewId],
		list = lv && lv._list,
		type = lv._getItemCountType(),
		num = lv.getSelectionCount(),
		countKey = 'type' + AjxStringUtil.capitalizeFirstLetter(ZmItem.MSG_KEY[type]),
		typeText = type ? AjxMessageFormat.format(ZmMsg[countKey], num) : "";

	return num > 0 ? AjxMessageFormat.format(ZmMsg.itemSelectionCount, [num, typeText]) : '';
};

ZmListController.prototype._setItemSelectionCountText = function() {
	this._setItemCountText(this._getItemSelectionCountText());
};

/**
 * Records total items and last item before we do any more searches. Adds a couple
 * params to the args for the list action method.
 *
 * @param {function}	actionMethod		the controller action method
 * @param {Array}		args				an arg list for above (except for items arg)
 * @param {Hash}		params				the params that will be passed to list action method
 * @param {closure}		allDoneCallback		the callback to run after all items processed
 * 
 * @private
 */
ZmListController.prototype._setupContinuation =
function(actionMethod, args, params, allDoneCallback, notIdsOnly) {

	// need to use AjxCallback here so we can prepend items arg when calling it
	var actionCallback = new AjxCallback(this, actionMethod, args);
	params.finalCallback = this._continueAction.bind(this, {actionCallback:actionCallback, allDoneCallback:allDoneCallback, notIdsOnly: notIdsOnly});
	
	params.count = this._continuation.count;
	params.idsOnly = !notIdsOnly;

	if (!this._continuation.lastItem) {
		this._continuation.lastItem = params.list.getVector().getLast();
		this._continuation.totalItems = params.list.size();
	}
};

/**
 * See if we are performing an action on all items, including ones that match the current search
 * but have not yet been retrieved. If so, keep doing searches and performing the action on the
 * results, until there are no more results.
 *
 * The arguments in the action callback should be those after the initial 'items' argument. The
 * array of items retrieved by the search is prepended to the callback's argument list before it
 * is run.
 *
 * @param {Hash}		params				a hash of parameters
 * @param {AjxCallback}	actionCallback		the callback with action to be performed on search results
 * @param {closure} 	allDoneCallback		the callback to run when we're all done
 * @param {Hash}		actionParams		the params from <code>ZmList._itemAction</code>, added when this is called
 * 
 * @private
 */
ZmListController.prototype._continueAction =
function(params, actionParams) {

	var lv = this._view[this._currentViewId];
	var cancelled = actionParams && actionParams.cancelled;
	var contResult = this._continuation.result;
	var hasMore = contResult ? contResult.getAttribute("more") : (this._list ? this._list.hasMore() : false);
	DBG.println("sa", "lv.allSelected: " + lv.allSelected + ", hasMore: " + hasMore);
	if (lv.allSelected && hasMore && !cancelled) {
		var cs = this._currentSearch;
		var limit = ZmListController.CONTINUATION_SEARCH_ITEMS;
		var searchParams = {
			query:		this.getSearchString(),
			queryHint:	this.getSearchStringHint(),
			types:		cs.types,
			sortBy:		cs.sortBy,
			limit:		limit,
			idsOnly:	!params.notIdsOnly
		};

		var list = contResult ? contResult.getResults() : this._list.getArray();
		var lastItem = this._continuation.lastItem;
		if (!lastItem) {
			lastItem = list && list[list.length - 1];
		}
		if (lastItem) {
			searchParams.lastId = lastItem.id;
			searchParams.lastSortVal = lastItem.sf;
			DBG.println("sa", "***** continuation search: " + searchParams.query + " --- " + [lastItem.id, lastItem.sf].join("/"));
		} else {
			searchParams.offset = limit + (this._continuation.search ? this._continuation.search.offset : 0);
		}

		this._continuation.count = actionParams.numItems;
		if (!this._continuation.totalItems) {
			this._continuation.totalItems = list.length;
		}

		this._continuation.search = new ZmSearch(searchParams);
		var respCallback = this._handleResponseContinueAction.bind(this, params.actionCallback);
		appCtxt.getSearchController().redoSearch(this._continuation.search, true, null, respCallback);
	} else {
		DBG.println("sa", "end of continuation");
		if (contResult) {
			if (lv.allSelected) {
				// items beyond page were acted on, give user a total count
				if (actionParams.actionTextKey) {
					var type = contResult.type;
					if (type === ZmId.SEARCH_MAIL) {
						type = this._list.type; //get the specific CONV/MSG type instead of the "searchFor" "MAIL".
					}
					actionParams.actionSummary = ZmList.getActionSummary({
						actionTextKey:  actionParams.actionTextKey,
						numItems:       this._continuation.totalItems,
						type:           type,
						actionArg:      actionParams.actionArg
					});
				}
				lv.deselectAll();
			}
			this._continuation = {count:0, totalItems:0};
		}
		if (params.allDoneCallback) {
			params.allDoneCallback();
		}

		ZmListController.handleProgress({state:ZmListController.PROGRESS_DIALOG_CLOSE});
		ZmBaseController.showSummary(actionParams.actionSummary, actionParams.actionLogItem, actionParams.closeChildWin);
	}
};

/**
 * @private
 */
ZmListController.prototype._handleResponseContinueAction =
function(actionCallback, result) {

	this._continuation.result = result.getResponse();
	var items = this._continuation.result.getResults();
	DBG.println("sa", "continuation search results: " + items.length);
	if (items.isZmMailList) { //no idsOnly case
		items = items.getArray();
	}
	if (items.length) {
		this._continuation.lastItem = items[items.length - 1];
		this._continuation.totalItems += items.length;
		DBG.println("sa", "continuation last item: " + this._continuation.lastItem.id);
		actionCallback.args = actionCallback.args || [];
		actionCallback.args.unshift(items);
		DBG.println("sa", "calling continuation action on search results");
		actionCallback.run();
	} else {
		DBG.println(AjxDebug.DBG1, "Continuation with empty search results!");
	}
};

/**
 * @private
 */
ZmListController.prototype._checkItemCount =
function() {
	var lv = this._view[this._currentViewId];
	lv._checkItemCount();
	lv._handleResponseCheckReplenish(true);
};

// Returns true if this controller supports sorting its items
ZmListController.prototype.supportsSorting = function() {
    return true;
};

// Returns true if this controller supports alternatively grouped list views
ZmListController.prototype.supportsGrouping = function() {
    return false;
};
}
if (AjxPackage.define("zimbraMail.share.controller.ZmTreeController")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines a tree controller.
 *
 */

/**
 * Creates a tree controller.
 * @class
 * This class is a base class for controllers for organizers. Those are
 * represented by trees, both as data and visually. This class uses the support provided by
 * {@link ZmOperation}. Each type of organizer has a singleton tree controller which manages all 
 * the tree views of that type.
 *
 * @author Conrad Damon
 * 
 * @param {constant}	type		the type of organizer we are displaying/controlling
 * 
 * @extends	ZmController
 */
ZmTreeController = function(type) {

	if (arguments.length == 0) { return; }

	ZmController.call(this, null);

	this.type = type;
	this._opc = appCtxt.getOverviewController();
	
	// common listeners
	this._listeners = {};
	this._listeners[ZmOperation.DELETE]			            = this._deleteListener.bind(this);
	this._listeners[ZmOperation.DELETE_WITHOUT_SHORTCUT]    = this._deleteListener.bind(this);
	this._listeners[ZmOperation.MOVE]			            = this._moveListener.bind(this);
	this._listeners[ZmOperation.EXPAND_ALL]		            = this._expandAllListener.bind(this);
	this._listeners[ZmOperation.MARK_ALL_READ]	            = this._markAllReadListener.bind(this);
	this._listeners[ZmOperation.SYNC]			            = this._syncListener.bind(this);
	this._listeners[ZmOperation.SYNC_ALL]		            = this._syncAllListener.bind(this);
	this._listeners[ZmOperation.EDIT_PROPS]		            = this._editPropsListener.bind(this);
	this._listeners[ZmOperation.EMPTY_FOLDER]               = this._emptyListener.bind(this);
	this._listeners[ZmOperation.FIND_SHARES]	            = this._findSharesListener.bind(this);
	this._listeners[ZmOperation.OPEN_IN_TAB]                = this._openInTabListener.bind(this);

	// drag-and-drop
	this._dragSrc = new DwtDragSource(Dwt.DND_DROP_MOVE);
	this._dragSrc.addDragListener(this._dragListener.bind(this));
	this._dropTgt = new DwtDropTarget(ZmTreeController.DROP_SOURCES[type]);
	this._dropTgt.addDropListener(this._dropListener.bind(this));

	this._treeView = {};	// hash of tree views of this type, by overview ID
	this._hideEmpty = {};	// which tree views to hide if they have no data
	this._dataTree = {};	// data tree per account

	this._treeSelectionShortcutDelay = ZmTreeController.TREE_SELECTION_SHORTCUT_DELAY;
};

ZmTreeController.prototype = new ZmController;
ZmTreeController.prototype.constructor = ZmTreeController;

ZmTreeController.COLOR_CLASS = {};
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_ORANGE]	= "OrangeBg";
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_BLUE]	= "BlueBg";
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_CYAN]	= "CyanBg";
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_GREEN]	= "GreenBg";
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_PURPLE]	= "PurpleBg";
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_RED]		= "RedBg";
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_YELLOW]	= "YellowBg";
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_PINK]	= "PinkBg";
ZmTreeController.COLOR_CLASS[ZmOrganizer.C_GRAY]	= "Gray";	// not GrayBg so it doesn't blend in

// time that selection via up/down arrow must remain on an item to trigger a search
ZmTreeController.TREE_SELECTION_SHORTCUT_DELAY = 750;

// valid sources for drop target for different tree controllers
ZmTreeController.DROP_SOURCES = {};

// interval of retrying empty folder (seconds)
ZmTreeController.EMPTY_FOLDER_RETRY_INTERVAL = 5;

// the maximum number of trials of empty folder
ZmTreeController.EMPTY_FOLDER_MAX_TRIALS = 6;

// Abstract protected methods

// Enables/disables operations based on the given organizer ID
ZmTreeController.prototype.resetOperations = function() {};

// Returns a list of desired header action menu operations
ZmTreeController.prototype._getHeaderActionMenuOps = function() {};

// Returns a list of desired action menu operations
ZmTreeController.prototype._getActionMenuOps = function() {};

// Returns the dialog for organizer creation
ZmTreeController.prototype._getNewDialog = function() {};

// Returns the dialog for renaming an organizer
ZmTreeController.prototype._getRenameDialog = function() {};

// Method that is run when a tree item is left-clicked
ZmTreeController.prototype._itemClicked = function() {};

// Method that is run when a tree item is dbl-clicked
ZmTreeController.prototype._itemDblClicked = function() {};

// Handles a drop event
ZmTreeController.prototype._dropListener = function() {};

// Returns an appropriate title for the "Move To" dialog
ZmTreeController.prototype._getMoveDialogTitle = function() {};

/**
 * @private
 */
ZmTreeController.prototype._resetOperation =
function(parent, id, text, image, enabled, visible) {
	var op = parent && parent.getOp(id);
	if (!op) return;

	if (text) op.setText(text);
	if (image) op.setImage(image);
	if (enabled != null) op.setEnabled(enabled);
	if (visible != null) op.setVisible(visible);
};

/**
 * @private
 */
ZmTreeController.prototype._resetButtonPerSetting =
function(parent, op, isSupported) {
	var button = parent.getOp(op);
	if (button) {
		if (isSupported) {
			button.setVisible(true);
			if (appCtxt.isOffline && !appCtxt.getActiveAccount().isZimbraAccount) {
				button.setEnabled(false);
			}
		} else {
			button.setVisible(false);
		}
	}
};

ZmTreeController.prototype._enableRecoverDeleted =
function (parent, isTrash) {
	op = parent.getOp(ZmOperation.RECOVER_DELETED_ITEMS);
	if (!op) {
		return;
	}
	var featureEnabled = appCtxt.get(ZmSetting.DUMPSTER_ENABLED);
	op.setVisible(featureEnabled && isTrash);
	op.setEnabled(isTrash);
};

ZmTreeController.prototype._findSharesListener =
function(ev) {
	var folder = this._getActionedOrganizer(ev);
	var account = folder.getAccount();

	if (appCtxt.multiAccounts && account && account.isZimbraAccount) {
		appCtxt.accountList.setActiveAccount(account);
	}
	var dialog = appCtxt.getShareSearchDialog();
	var addCallback = this._handleAddShare.bind(this);
	dialog.popup(folder.type, addCallback);
};

ZmTreeController.prototype._handleAddShare = function () {
	var dialog = appCtxt.getShareSearchDialog();
	var shares = dialog.getShares();
	dialog.popdown();
	if (shares.length === 0) {
		return;
	}

	AjxDispatcher.require("Share");
	var requests = [];
	for (var i = 0; i < shares.length; i++) {
		var share = shares[i];
		requests.push({
			_jsns: "urn:zimbraMail",
			link: {
				l: ZmOrganizer.ID_ROOT,
				name: share.defaultMountpointName,
				view: share.view,
				zid: share.ownerId,
				rid: share.folderId
			}
		});
	}

	var params = {
		jsonObj: {
			BatchRequest: {
				_jsns: "urn:zimbra",
				CreateMountpointRequest: requests
			}
		},
		asyncMode: true
	};
	appCtxt.getAppController().sendRequest(params);
};

// Opens a view of the given organizer in a search tab
ZmTreeController.prototype._openInTabListener = function(ev) {
	this._itemClicked(this._getActionedOrganizer(ev), true);
};





// Public methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmTreeController.prototype.toString =
function() {
	return "ZmTreeController";
};

/**
 * Displays the tree of this type.
 *
 * @param {Hash}	params		a hash of parameters
 * @param	{constant}	params.overviewId		the overview ID
 * @param	{Boolean}	params.showUnread		if <code>true</code>, unread counts will be shown
 * @param	{Object}	params.omit				a hash of organizer IDs to ignore
 * @param	{Object}	params.include			a hash of organizer IDs to include
 * @param	{Boolean}	params.forceCreate		if <code>true</code>, tree view will be created
 * @param	{String}	params.app				the app that owns the overview
 * @param	{Boolean}	params.hideEmpty		if <code>true</code>, don't show header if there is no data
 * @param	{Boolean}	params.noTooltips	if <code>true</code>, don't show tooltips for tree items
 */
ZmTreeController.prototype.show =
function(params) {
	var id = params.overviewId;
	this._hideEmpty[id] = params.hideEmpty;

	if (!this._treeView[id] || params.forceCreate) {
		this._treeViewCreated = false;
		this._treeView[id] = null;
		this._treeView[id] = this.getTreeView(id, true);
	}

	// bug fix #24241 - for offline, zimlet tree is re-used across accounts
	var isMultiAccountZimlet = (appCtxt.multiAccounts && this.type == ZmOrganizer.ZIMLET);
	var account = isMultiAccountZimlet
		? appCtxt.accountList.mainAccount
		: (this.type == ZmOrganizer.VOICE ? id : params.account); // HACK for voice app
	var dataTree = this.getDataTree(account);

	if (dataTree) {
		params.dataTree = dataTree;
		var setting = ZmOrganizer.OPEN_SETTING[this.type];
		params.collapsed = (!isMultiAccountZimlet && (!(!setting || (appCtxt.get(setting, null, account) !== false)))); // yikes!

		var overview = this._opc.getOverview(id);

		if (overview && overview.showNewButtons && this.type != ZmOrganizer.ZIMLET && this.type != ZmId.ORG_PREF_PAGE ) { 
			this._setupOptButton(params);
		}

		this._treeView[id].set(params);
		this._checkTreeView(id);
	}

	if (!this._treeViewCreated) {
		this._treeViewCreated = true;
		this._postSetup(id, params.account);
	}
	return this._treeView[id];
};

/**
 * Gets the tree view for the given overview.
 *
 * @param {constant}	overviewId	the overview ID
 * @param {Boolean}	force			if <code>true</code>, force tree view creation
 * @return	{ZmTreeView}		the tree view
 */
ZmTreeController.prototype.getTreeView =
function(overviewId, force) {
	// TODO: What side-effects will this have in terms of the _postSetup???
	if (force && !this._treeView[overviewId]) {
		this._treeView[overviewId] = this._setup(overviewId);
	}
	return this._treeView[overviewId];
};

/**
 * Clears the tree view for the given overview.
 *
 * @param {constant}		overviewId		the overview ID
 *
 */
ZmTreeController.prototype.clearTreeView =
function(overviewId) {
	// TODO: remove change listener if last tree view cleared
	if (this._treeView[overviewId]) {
		this._treeView[overviewId].dispose();
		delete this._treeView[overviewId];
	}
};

/**
 * Gets the controller drop target.
 * 
 * @return	{DwtDropTarget}	the drop target
 */
ZmTreeController.prototype.getDropTarget =
function() {
	return this._dropTgt;
};

/**
 * Gets the data tree.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{Object}	the data tree
 */
ZmTreeController.prototype.getDataTree =
function(account) {
	account = account || appCtxt.getActiveAccount();
	var dataTree = this._dataTree[account.id];
	if (!dataTree) {
		dataTree = this._dataTree[account.id] = appCtxt.getTree(this.type, account);
		if (dataTree) {
			dataTree.addChangeListener(this._getTreeChangeListener());
		}
	}
	return dataTree;
};

/**
 * Dispose of this controller. Removes the tree change listener.
 * called when ZmComposeController is disposed (new window).
 * If the change listener stayed we would get exceptions since this window will no longer exist.
 *
 */
ZmTreeController.prototype.dispose =
function() {
	var account = appCtxt.getActiveAccount();
	var dataTree = this._dataTree[account.id];
	if (!dataTree) {
		return;
	}
	dataTree.removeChangeListener(this._getTreeChangeListener());
};



ZmTreeController.prototype.setVisibleIfExists =
function(parent, opId, visible) {
	var op = parent.getOp(opId);
	if (!op) {
		return;
	}
	op.setVisible(visible);
};

// Private and protected methods

/**
 * Sets up the params for the new button in the header item
 *
 * @param {Hash}	params		a hash of parameters
 * 
 * @private
 */
ZmTreeController.prototype._setupOptButton =
function(params) {
	var tooltipKey = ZmOperation.getProp(ZmOperation.OPTIONS, "tooltipKey");
	params.optButton = {
		image: ZmOperation.getProp(ZmOperation.OPTIONS, "image"),
		tooltip: tooltipKey ? ZmMsg[tooltipKey] : null,
		callback: new AjxCallback(this, this._dispOpts)
	};
};

/**
 * Shows options for header item
 *
 * @param {Hash}	params		a hash of parameters
 * 
 * @private
 */

ZmTreeController.prototype._dispOpts =
function(ev){

	var treeItem = ev.dwtObj;

       var type = treeItem && treeItem.getData(ZmTreeView.KEY_TYPE);
       if (!type) { return; }

       var actionMenu = this._getHeaderActionMenu(ev);
       if (actionMenu) {
		actionMenu.popup(0, ev.docX, ev.docY);
	}
};

ZmTreeController.prototype._getTreeChangeListener =
function() {
	if (!this._dataChangeListener) {
		this._dataChangeListener = appCtxt.isChildWindow ? AjxCallback.simpleClosure(this._treeChangeListener, this) : new AjxListener(this, this._treeChangeListener);
	}
	return this._dataChangeListener;
};

/**
 * Performs initialization.
 *
 * @param overviewId		[constant]	overview ID
 */
ZmTreeController.prototype._setup =
function(overviewId) {
	var treeView = this._initializeTreeView(overviewId);
	if (this._opc.getOverview(overviewId).actionSupported) {
		this._initializeActionMenus();
	}
	return treeView;
};

/**
 * Performs any little fixups after the tree view is first created
 * and shown.
 *
 * @param {constant}	overviewId		the overview ID
 * @param {ZmZimbraAccount}	account			the current account
 * 
 * @private
 */
ZmTreeController.prototype._postSetup =
function(overviewId, account) {

	var treeView = this.getTreeView(overviewId);
	if (!treeView.isCheckedStyle && !ZmOrganizer.HAS_COLOR[this.type]) { return; }

	var rootId = ZmOrganizer.getSystemId(ZmOrganizer.ID_ROOT, account);
	var rootTreeItem = treeView.getTreeItemById(rootId);
	if (!rootTreeItem) { return; }
	if (treeView.isCheckedStyle) {
		rootTreeItem.showCheckBox(false);
	}
	var treeItems = rootTreeItem.getItems();
	for (var i = 0; i < treeItems.length; i++) {
		this._fixupTreeNode(treeItems[i], null, treeView, true);
	}
};

/**
 * Takes care of the tree item's color and/or checkbox.
 *
 * @param {DwtTreeItem}	treeItem	the tree item
 * @param {ZmOrganizer}	organizer	the organizer it represents
 * @param {ZmTreeView}	treeView	the tree view this organizer belongs to
 * 
 * @private
 */
ZmTreeController.prototype._fixupTreeNode =
function(treeItem, organizer, treeView, skipNotify) {
	if (treeItem._isSeparator) { return; }
	organizer = organizer || treeItem.getData(Dwt.KEY_OBJECT);
	if (organizer) {
		if (ZmOrganizer.HAS_COLOR[this.type]) {
			this._setTreeItemColor(treeItem, organizer);
		}
		if (treeView.isCheckedStyle) {
			if ((organizer.type == this.type && treeView.isCheckedStyle) ||
                organizer.nId == ZmOrganizer.ID_TRASH || organizer.nId == ZmOrganizer.ID_DRAFTS) {
				treeItem.setChecked(organizer.isChecked, true);
			} else {
				treeItem.showCheckBox(false);
				treeItem.enableSelection(true);
			}
		}

		// set expand state per user's prefs
		this._expandTreeItem(treeItem, skipNotify);
	}
    var treeItems = treeItem.getItems();
    for (var i = 0; i < treeItems.length; i++) {
        this._fixupTreeNode(treeItems[i], null, treeView, skipNotify);
    }
};

ZmTreeController.prototype._expandTreeItem =
function(treeItem, skipNotify) {
    var expanded = appCtxt.get(ZmSetting.FOLDERS_EXPANDED);
	var folderId = treeItem.getData(Dwt.KEY_ID);
	var parentTi = treeItem.parent;

	// only expand if the parent is also expanded
	if (expanded[folderId] &&
		parentTi && (parentTi instanceof DwtTreeItem) && parentTi.getExpanded())
	{
		treeItem.setExpanded(true, null, skipNotify);
	}
};

ZmTreeController.prototype._expandTreeItems =
function(treeItem) {
	if (treeItem._isSeparator) { return; }

	this._expandTreeItem(treeItem);

	// recurse!
	var treeItems = treeItem.getItems();
	for (var i = 0; i < treeItems.length; i++) {
		this._expandTreeItems(treeItems[i]);
	}
};

/**
 * Sets the background color of the tree item.
 *
 * @param treeItem	[DwtTreeItem]		tree item
 * @param organizer	[ZmOrganizer]		organizer it represents
 */
ZmTreeController.prototype._setTreeItemColor =
function(treeItem, organizer) {
	treeItem.setImage(organizer.getIconWithColor(), organizer.getIconAltText());
};

ZmTreeController.prototype._getTreeItemColorClassName =
function(treeItem, organizer) {
	if (!treeItem || !organizer) { return null; }
	if (organizer.isInTrash()) { return null; }

	// a color value of 0 means DEFAULT
	var color = organizer.color
		? organizer.color
		: ZmOrganizer.DEFAULT_COLOR[organizer.type];

	return (color && (color != ZmOrganizer.C_NONE))
		? ZmTreeController.COLOR_CLASS[color] : "";
};

/**
 * Lazily creates a tree view of this type, using options from the overview.
 *
 * @param {constant}	overviewId		the overview ID
 * 
 * @private
 */
ZmTreeController.prototype._initializeTreeView =
function(overviewId) {
	var overview = this._opc.getOverview(overviewId);
	var params = {
		parent: overview,
		parentElement: overview.getTreeParent(this.type),
		overviewId: overviewId,
		type: this.type,
		headerClass: overview.headerClass,
		dragSrc: (overview.dndSupported ? this._dragSrc : null),
		dropTgt: (overview.dndSupported ? this._dropTgt : null),
		treeStyle: overview.treeStyle,
		isCheckedByDefault: overview.isCheckedByDefault,
		allowedTypes: this._getAllowedTypes(),
		allowedSubTypes: this._getAllowedSubTypes()
	};
	params.id = ZmId.getTreeId(overviewId, params.type);
	if (params.type && params.type.match(/TASK|ADDRBOOK|FOLDER|BRIEFCASE|CALENDAR|PREF_PAGE/) && 
			(!params.headerClass || params.headerClass == "overviewHeader")){
		params.headerClass = "FirstOverviewHeader overviewHeader";
	}
	var treeView = this._createTreeView(params);
	treeView.addSelectionListener(new AjxListener(this, this._treeViewListener));
	treeView.addTreeListener(new AjxListener(this, this._treeListener));

	return treeView;
};

/**
 * @private
 */
ZmTreeController.prototype._createTreeView =
function(params) {
	return new ZmTreeView(params);
};

/**
 * Creates up to two action menus, one for the tree view's header item, and
 * one for the rest of the items. Note that each of these two menus is a
 * singleton, shared among the tree views of this type.
 * 
 * @private
 */
ZmTreeController.prototype._initializeActionMenus =
function() {
	var obj = this;
	var func = this._createActionMenu;

	var ops = this._getHeaderActionMenuOps();
	if (!this._headerActionMenu && ops) {
		var args = [this._shell, ops];
		this._headerActionMenu = new AjxCallback(obj, func, args);
	}
	var ops = this._getActionMenuOps();
	if (!this._actionMenu && ops) {
		var args = [this._shell, ops];
		this._actionMenu = new AjxCallback(obj, func, args);
	}
};

/**
 * Instantiates the header action menu if necessary.
 * 
 * @private
 */
ZmTreeController.prototype._getHeaderActionMenu =
function(ev) {
	if (this._headerActionMenu instanceof AjxCallback) {
		var callback = this._headerActionMenu;
		this._headerActionMenu = callback.run();
	}
	return this._headerActionMenu;
};

/**
 * Instantiates the action menu if necessary.
 * 
 * @private
 */
ZmTreeController.prototype._getActionMenu =
function(ev, item) {
    var controller = this;

    // special case - search folder. might have moved under a regular folder  
    if (item && item.type == ZmOrganizer.SEARCH) {
        controller = this._opc.getTreeController(ZmOrganizer.SEARCH);
    }

	if (controller._actionMenu instanceof AjxCallback) {
		var callback = controller._actionMenu;
		controller._actionMenu = callback.run();
	}
	return controller._actionMenu;
};

/**
 * Creates and returns an action menu, and sets its listeners.
 *
 * @param {DwtControl}	parent		the menu parent widget
 * @param {Array}	menuItems		the list of menu items
 * 
 * @private
 */
ZmTreeController.prototype._createActionMenu =
function(parent, menuItems) {
	if (!menuItems) return;

	var map = appCtxt.getCurrentController() && appCtxt.getCurrentController().getKeyMapName();
	var id = map ? ("ZmActionMenu_" + map):Dwt.getNextId("ZmActionMenu_")
	id = (map && this.type) ? id + "_" + this.type : id;
	var actionMenu = new ZmActionMenu({parent:parent, menuItems:menuItems, id: id});

	menuItems = actionMenu.opList;
	for (var i = 0; i < menuItems.length; i++) {
		var menuItem = menuItems[i];
		if (this._listeners[menuItem]) {
			actionMenu.addSelectionListener(menuItem, this._listeners[menuItem]);
		}
	}
	actionMenu.addPopdownListener(new AjxListener(this, this._menuPopdownActionListener));

	return actionMenu;
};

/**
 * Determines which types of organizer may be displayed at the top level. By default,
 * the tree shows its own type.
 * 
 * @private
 */
ZmTreeController.prototype._getAllowedTypes =
function() {
	var types = {};
	types[this.type] = true;
	return types;
};

/**
 * Determines which types of organizer may be displayed below the top level. By default,
 * the tree shows its own type.
 * 
 * @private
 */
ZmTreeController.prototype._getAllowedSubTypes =
function() {
	var types = {};
	types[this.type] = true;
	return types;
};

// Actions

/**
 * Creates a new organizer and adds it to the tree of that type.
 *
 * @param {Hash}	params	a hash of parameters
 * @param {constant}	params.type		the type of organizer
 * @param {ZmOrganizer}	params.parent	parent of the new organizer
 * @param {String}	params.name		the name of the new organizer
 *        
 * @private
 */
ZmTreeController.prototype._doCreate =
function(params) {
	params.type = this.type;
	var funcName = ZmOrganizer.CREATE_FUNC[this.type];
	if (funcName) {
		var func = eval(funcName);
		return func(params);
	}
};

/**
 * Deletes an organizer and removes it from the tree.
 *
 * @param {ZmOrganizer}	organizer		the organizer to delete
 */
ZmTreeController.prototype._doDelete =
function(organizer) {
	organizer._delete();
};

/**
 * 
 * @param {ZmOrganizer}	organizer		the organizer
 * @param {int}	trialCounter		the number of trials of empty folder
 * @param {AjxException}	ex		the exception
 *
 * @private
 */
ZmTreeController.prototype._doEmpty =
function(organizer, trialCounter, ex) {
	var recursive = false;
	var timeout = ZmTreeController.EMPTY_FOLDER_RETRY_INTERVAL;
	var noBusyOverlay = true;
	if (!trialCounter) {
		trialCounter = 1;
	}
	var errorCallback = this._doEmptyErrorHandler.bind(this, organizer, trialCounter);
	organizer.empty(recursive, null, this._doEmptyHandler.bind(this, organizer), timeout, errorCallback, noBusyOverlay);
};

/**
 *
 * @param {ZmOrganizer}	organizer		the organizer
 * @param {int}	trialCounter		the number of trials of empty folder
 * @param {AjxException}	ex		the exception
 *
 * @private
 */
ZmTreeController.prototype._doEmptyErrorHandler =
function(organizer, trialCounter, ex) {
	if (ex) {
		if (ex.code == ZmCsfeException.SVC_ALREADY_IN_PROGRESS) {
			appCtxt.setStatusMsg(ZmMsg.emptyFolderAlreadyInProgress);
			return true;
		} else if(ex.code != AjxException.CANCELED) {
			return false;
		}
	}

	if (trialCounter > ZmTreeController.EMPTY_FOLDER_MAX_TRIALS -1){
		appCtxt.setStatusMsg(ZmMsg.emptyFolderNoResponse, ZmStatusView.LEVEL_CRITICAL);
		return true;
	}
	trialCounter++;
	this._doEmpty(organizer, trialCounter);
};

ZmTreeController.prototype._doEmptyHandler =
function(organizer) {
	appCtxt.setStatusMsg({msg: AjxMessageFormat.format(ZmMsg.folderEmptied, organizer.getName())});
	var ctlr = appCtxt.getCurrentController();
	if (!ctlr || !ctlr._getSearchFolderId || !ctlr.getListView) {
		return;
	}
	var folderId = ctlr._getSearchFolderId();
	if (folderId !== organizer.id) {
		return;
	}
	var view = ctlr.getListView();
	view._resetList();
	view._setNoResultsHtml();
};

/**
 * Renames an organizer.
 *
 * @param {ZmOrganizer}	organizer	the organizer to rename
 * @param {String}	name		the new name of the organizer
 * 
 * @private
 */
ZmTreeController.prototype._doRename =
function(organizer, name) {
	organizer.rename(name);
};

/**
 * Moves an organizer to a new folder.
 *
 * @param {ZmOrganizer}	organizer	the organizer to move
 * @param {ZmFolder}	folder		the target folder
 * 
 * @private
 */
ZmTreeController.prototype._doMove =
function(organizer, folder) {
	organizer.move(folder);
};

/**
 * Marks an organizer's items as read.
 *
 * @param {ZmOrganizer}	organizer	the organizer
 * 
 * @private
 */
ZmTreeController.prototype._doMarkAllRead =
function(organizer) {
	organizer.markAllRead();
};

/**
 * Syncs an organizer to its feed (URL).
 *
 *  @param {ZmOrganizer}	organizer	the organizer
 *  
 *  @private
 */
ZmTreeController.prototype._doSync =
function(organizer) {
	organizer.sync();
};

// Listeners

/**
 * Handles left and right mouse clicks. A left click generates a selection event.
 * If selection is supported for the overview, some action (typically a search)
 * will be performed. A right click generates an action event, which pops up an
 * action menu if supported.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._treeViewListener = function(ev) {

	if (ev.detail !== DwtTree.ITEM_ACTIONED && ev.detail !== DwtTree.ITEM_SELECTED && ev.detail !== DwtTree.ITEM_DBL_CLICKED) {
		return;
	}

	var treeItem = ev.item;

	var type = treeItem.getData(ZmTreeView.KEY_TYPE);
	if (!type) {
        return;
    }

	var item = treeItem.getData(Dwt.KEY_OBJECT);
	if (item) {
		this._actionedOrganizer = item;
		if (item.noSuchFolder) {
			var folderTree = appCtxt.getFolderTree();
			if (folderTree) {
				folderTree.handleDeleteNoSuchFolder(item);
			}
			return;
		}
        if (item && item.type === ZmOrganizer.SEARCH) {
            var controller = this._opc.getTreeController(ZmOrganizer.SEARCH);
            if (controller) {
                controller._actionedOrganizer = item;
                controller._actionedOverviewId = treeItem.getData(ZmTreeView.KEY_ID);
            }
        }
	}

	var id = treeItem.getData(Dwt.KEY_ID);
	var overviewId = this._actionedOverviewId = treeItem.getData(ZmTreeView.KEY_ID);
	var overview = this._opc.getOverview(overviewId);
	if (!overview) {
        return;
    }

	if (ev.detail === DwtTree.ITEM_ACTIONED) {
		// right click
		if (overview.actionSupported) {
			var actionMenu = this.getItemActionMenu(ev, item);
			if (actionMenu) {
				this.resetOperations(actionMenu, type, id);
				actionMenu.popup(0, ev.docX, ev.docY);
			}
		}
	}
    else if ((ev.detail === DwtTree.ITEM_SELECTED) && item) {
		if (appCtxt.multiAccounts && (item instanceof ZmOrganizer)) {
			this._handleMultiAccountItemSelection(ev, overview, treeItem, item);
		}
        else {
			this._handleItemSelection(ev, overview, treeItem, item);
		}
	}
    else if ((ev.detail === DwtTree.ITEM_DBL_CLICKED) && item) {
		this._itemDblClicked(item);
	}
};

ZmTreeController.prototype.getItemActionMenu = function(ev, item) {
	var actionMenu = (item.nId == ZmOrganizer.ID_ROOT || item.isDataSource(ZmAccount.TYPE_IMAP))
		? this._getHeaderActionMenu(ev)
		: this._getActionMenu(ev, item);
	return actionMenu;
}

/**
 * @private
 */
ZmTreeController.prototype._handleItemSelection =
function(ev, overview, treeItem, item) {
	// left click or selection via shortcut
	overview.itemSelected(treeItem);

	if (ev.kbNavEvent) {
		Dwt.scrollIntoView(treeItem._itemDiv, overview.getHtmlElement());
		ZmController.noFocus = true;
	}

	if (overview._treeSelectionShortcutDelayActionId) {
		AjxTimedAction.cancelAction(overview._treeSelectionShortcutDelayActionId);
	}

	if ((overview.selectionSupported || item._showFoldersCallback) && !treeItem._isHeader) {
		if (ev.kbNavEvent) {
			// for shortcuts, process selection via Enter immediately; selection via up/down keys
			// is delayed (or can be disabled by setting the delay to 0)
			if (ev.enter || this._treeSelectionShortcutDelay) {
				var action = new AjxTimedAction(this, ZmTreeController.prototype._treeSelectionTimedAction, [item, overview]);
				overview._treeSelectionShortcutDelayActionId = AjxTimedAction.scheduleAction(action, this._treeSelectionShortcutDelay);
			}
		} else {
			if ((appCtxt.multiAccounts && (item instanceof ZmOrganizer)) ||
				(item.type == ZmOrganizer.VOICE))
			{
				appCtxt.getCurrentApp().getOverviewContainer().deselectAll(overview);

				// set the active account based on the item clicked
				var account = item.account || appCtxt.accountList.mainAccount;
				appCtxt.accountList.setActiveAccount(account);
			}

			this._itemSelected(item);
		}
	}
};

/**
 * @private
 */
ZmTreeController.prototype._itemSelected =
function(item) {
	if (item && item._showFoldersCallback) {
		item._showFoldersCallback.run();
	} else {
		this._itemClicked(item);
	}

};

/**
 * Allows subclass to overload in case something needs to be done before
 * processing tree item selection in a multi-account environment. Otherwise,
 * do the normal tree item selection.
 * 
 * @private
 */
ZmTreeController.prototype._handleMultiAccountItemSelection =
function(ev, overview, treeItem, item) {
	this._handleItemSelection(ev, overview, treeItem, item);
};

/**
 * @private
 */
ZmTreeController.prototype._treeSelectionTimedAction =
function(item, overview) {
	if (overview._treeSelectionShortcutDelayActionId) {
		AjxTimedAction.cancelAction(overview._treeSelectionShortcutDelayActionId);
	}
	this._itemSelected(item);
};

/**
 * Propagates a change in tree state to other trees of the same type in app overviews.
 * 
 * @param {ZmTreeEvent}	ev		a tree event
 * 
 * @private
 */
ZmTreeController.prototype._treeListener =
function(ev) {
	var treeItem = ev && ev.item;
	var overviewId = treeItem && treeItem._tree && treeItem._tree.overviewId;
    var overview = appCtxt.getOverviewController().getOverview(overviewId);
    var acct = overview.account;
    if (appCtxt.multiAccounts && acct) {
        appCtxt.accountList.setActiveAccount(acct);
    }

	// persist expand/collapse state for folders
	var isExpand = ev.detail == DwtTree.ITEM_EXPANDED;
	var folderId = (ev.detail == DwtTree.ITEM_COLLAPSED || isExpand)
		? treeItem.getData(Dwt.KEY_ID) : null;

	if (folderId && !treeItem._isHeader) {
		var setExpanded = appCtxt.get(ZmSetting.FOLDERS_EXPANDED, folderId) || false; //I think it's set as undefined if "false" in ZmSetting.prototype.setValue)
		if (typeof(setExpanded) == "string") {//I can't figure out why it becomes a string sometimes. That's nasty.
			setExpanded = (setExpanded === "true");
		}
		//setting in case of skipImplicit is still causing problems (the fix to bug 72590 was not good enough), since even if this "set" is not persisting,
		//future ones (collapse/expand in the mail tab) would cause it to save implicitly, which is not what we want.
		//so I simply do not call "set" in case of skipImplicit. Might want to change the name of this variable slightly, but not sure to what.
		if (!overview.skipImplicit && setExpanded !== isExpand) { //set only if changed (ZmSetting.prototype.setValue is supposed to not send a request if no change, but it might have bugs)
			appCtxt.set(ZmSetting.FOLDERS_EXPANDED, isExpand, folderId);
		}

		// check if any of this treeItem's children need to be expanded as well
		if (isExpand) {
			this._expandTreeItems(treeItem);
		}
	}

	// only handle events that come from headers in app overviews
	if (!(ev && ev.detail && overview && overview.isAppOverview && treeItem._isHeader)) { return; }

	var settings = appCtxt.getSettings(acct);
	var setting = settings.getSetting(ZmOrganizer.OPEN_SETTING[this.type]);
	if (setting) {
		setting.setValue(ev.detail == DwtTree.ITEM_EXPANDED);
	}
};

/**
 * Handles changes to the underlying model. The change is propagated to
 * all the tree views known to this controller.
 *
 * @param {ZmEvent}	ev		a change event
 * 
 * @private
 */
ZmTreeController.prototype._treeChangeListener =
function(ev) {
	this._evHandled = {};
	for (var overviewId in this._treeView) {
		this._changeListener(ev, this._treeView[overviewId], overviewId);
	}
};

/**
 * Handles a change event for one tree view.
 *
 * @param {ZmEvent}	ev				a change event
 * @param {ZmTreeView}	treeView		a tree view
 * @param {constant}	overviewId		overview ID
 * 
 * @private
 */
ZmTreeController.prototype._changeListener =
function(ev, treeView, overviewId) {
	if (this._evHandled[overviewId]) { return; }
	if (!treeView.allowedTypes[ev.type] && !treeView.allowedSubTypes[ev.type]) { return; }

	var organizers = ev.getDetail("organizers");
	if (!organizers && ev.source) {
		organizers = [ev.source];
	}

	// handle one organizer at a time
	for (var i = 0; i < organizers.length; i++) {
		var organizer = organizers[i];

		var node = treeView.getTreeItemById(organizer.id);
		// Note: source tree handles moves - it will have node
		if (!node && (ev.event != ZmEvent.E_CREATE)) { continue; }

		var fields = ev.getDetail("fields");
		if (ev.event == ZmEvent.E_DELETE) {
			if (organizer.nId == ZmFolder.ID_TRASH || organizer.nId == ZmFolder.ID_SPAM) {
				node.setText(organizer.getName(false));	// empty Trash or Junk
			} else {
				node.dispose();
			}
            this._checkTreeView(overviewId);
			this._evHandled[overviewId] = true;
		} else if (ev.event == ZmEvent.E_CREATE || ev.event == ZmEvent.E_MOVE) {
			// for multi-account, make sure this organizer applies to the given overview
			if (appCtxt.multiAccounts) {
				var overview = this._opc.getOverview(overviewId);
				if (overview && overview.account != organizer.getAccount()) {
					continue;
				}
			}
			var parentNode = this._getParentNode(organizer, ev, overviewId);
			var idx = parentNode ? ZmTreeView.getSortIndex(parentNode, organizer, eval(ZmTreeView.COMPARE_FUNC[organizer.type])) : null;
			if (parentNode && (ev.event == ZmEvent.E_CREATE)) {
				// parent's tree controller should handle creates - root is shared by all folder types
				var type = ((organizer.parent.nId == ZmOrganizer.ID_ROOT) || organizer.parent.isRemoteRoot()) ? ev.type : organizer.parent.type;
				if (type !== this.type || !treeView._isAllowed(organizer.parent, organizer)) {
					continue;
				}
				if (organizer.isOfflineGlobalSearch) {
					appCtxt.getApp(ZmApp.MAIL).getOverviewContainer().addSearchFolder(organizer);
					return;
				} else {
					node = this._addNew(treeView, parentNode, organizer, idx); // add to new parent
				}
                this.createDataSource(organizer);
			} else if (ev.event == ZmEvent.E_MOVE) {
				var selectedItem = treeView.getSelected();
				if (AjxUtil.isArray1(selectedItem)) { //make sure this tree is not a checked style one (no idea where we have that, but see the getSelected code
					selectedItem = null;
				}
				node.dispose();
				if (parentNode) {
					node = this._addNew(treeView, parentNode, organizer, idx); // add to new parent
				}
				//highlight the current chosen one again, in case it was moved, thus losing selection
				if (!treeView.getSelected() && selectedItem) { //if item was selected but now it is not
					treeView.setSelected(selectedItem.id, true, true);
				}
			}
			if (parentNode) {
				parentNode.setExpanded(true); // so that new node is visible

				this._fixupTreeNode(node, organizer, treeView);
			}
			this._checkTreeView(overviewId);
			this._evHandled[overviewId] = true;
		} else if (ev.event == ZmEvent.E_MODIFY) {
			if (!fields) { return; }
			if (fields[ZmOrganizer.F_TOTAL] || fields[ZmOrganizer.F_SIZE] || fields[ZmOrganizer.F_UNREAD] || fields[ZmOrganizer.F_NAME]) {
				node.setToolTipContent(organizer.getToolTip(true));
				if (appCtxt.multiAccounts && organizer.type == ZmOrganizer.FOLDER) {
					appCtxt.getApp(ZmApp.MAIL).getOverviewContainer().updateTooltip(organizer.nId);
				}
			}

			if (fields[ZmOrganizer.F_NAME] ||
				fields[ZmOrganizer.F_UNREAD] ||
				fields[ZmOrganizer.F_FLAGS] ||
				fields[ZmOrganizer.F_COLOR] ||
				((organizer.nId == ZmFolder.ID_DRAFTS || organizer.rid == ZmFolder.ID_DRAFTS ||
				  organizer.nId == ZmOrganizer.ID_OUTBOX) && fields[ZmOrganizer.F_TOTAL]))
			{
				this._updateOverview({
					organizer:  organizer,
					node:       node,
					fields:     fields,
					treeView:   treeView,
					overviewId: overviewId,
					ev:         ev
				});

				this._evHandled[overviewId] = true;
			}
		}
	}
};

/**
 * Handle an organizer change by updating the tree view. For example, a name change requires sorting.
 *
 * @param   params      hash            hash of params:
 *
 *          organizer   ZmOrganizer     organizer that changed
 *          node        DwtTreeItem     organizer node in tree view
 *          fields      hash            changed fields
 *          treeView    ZmTreeView      tree view for this organizer type
 *          overviewId  string          ID of containing overview
 *          ev          ZmEvent         change event
 *
 * @private
 */
ZmTreeController.prototype._updateOverview = function(params) {

	var org = params.organizer,
		node = params.node,
		parentNode = this._getParentNode(org, params.ev, params.overviewId);

	node.setText(org.getName(params.treeView._showUnread));

	// If the name changed, re-sort the containing list
	if (params.fields && params.fields[ZmOrganizer.F_NAME]) {
		if (parentNode && (parentNode.getNumChildren() > 1)) {
			var nodeSelected = node._selected;
			// remove and re-insert the node (if parent has more than one child)
			node.dispose();
			var idx = ZmTreeView.getSortIndex(parentNode, org, eval(ZmTreeView.COMPARE_FUNC[org.type]));
			node = params.treeView._addNew(parentNode, org, idx);
			if (nodeSelected) {
				//if it was selected, re-select it so it is highlighted. No need for notifications.
				params.treeView.setSelected(org, true);
			}
		} else {
			node.setDndText(org.getName());
		}
		appCtxt.getAppViewMgr().updateTitle();
	}

	// A folder aggregates unread status of its descendents, so propagate up parent chain
	if (params.fields[ZmOrganizer.F_UNREAD]) {
		var parent = org.parent;
		while (parent && parentNode && parent.nId != ZmOrganizer.ID_ROOT) {
			parentNode.setText(parent.getName(params.treeView._showUnread));
			parentNode = this._getParentNode(parent, params.ev, params.overviewId);
			parent = parent.parent;
		}
	}

	// Miscellaneous cleanup (color, selection)
	this._fixupTreeNode(node, org, params.treeView);
};

ZmTreeController.prototype._getParentNode = function(organizer, ev, overviewId) {

	if (organizer.parent) {
		// if node being moved to root, we assume new parent must be the container of its type
		var type = (organizer.parent.nId == ZmOrganizer.ID_ROOT) ? ev.type : null;
		return this._opc.getOverview(overviewId).getTreeItemById(organizer.parent.id, type);
	}
};

/**
 * Makes a request to add a new item to the tree, returning true if the item was
 * actually added, or false if it was omitted.
 *
 * @param {ZmTreeView}	treeView		the tree view
 * @param {DwtTreeItem}	parentNode	the node under which to add the new one
 * @param {ZmOrganizer}	organizer		the organizer for the new node
 * @param {int}	idx			the position at which to add the new node
 * 
 * @private
 */
ZmTreeController.prototype._addNew =
function(treeView, parentNode, organizer, idx) {
	return treeView._addNew(parentNode, organizer, idx);
};

/**
 * Pops up the appropriate "New ..." dialog.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * @param {ZmZimbraAccount}	account	used by multi-account mailbox (optional)
 * 
 * @private
 */
ZmTreeController.prototype._newListener =
function(ev, account) {
	this._pendingActionData = this._getActionedOrganizer(ev);
	var newDialog = this._getNewDialog();
	if (!this._newCb) {
		this._newCb = new AjxCallback(this, this._newCallback);
	}
	if (this._pendingActionData && !appCtxt.getById(this._pendingActionData.id)) {
		this._pendingActionData = appCtxt.getFolderTree(account).root;
	}

	if (!account && appCtxt.multiAccounts) {
		var ov = this._opc.getOverview(this._actionedOverviewId);
		account = ov && ov.account;
	}

	ZmController.showDialog(newDialog, this._newCb, this._pendingActionData, account);
	newDialog.registerCallback(DwtDialog.CANCEL_BUTTON, this._clearDialog, this, newDialog);
};

ZmTreeController.prototype.createDataSource =
function(organizer) {
    //override
};

/**
 * Pops up the appropriate "Rename ..." dialog.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._renameListener =
function(ev) {
	this._pendingActionData = this._getActionedOrganizer(ev);
	var renameDialog = this._getRenameDialog();
	if (!this._renameCb) {
		this._renameCb = new AjxCallback(this, this._renameCallback);
	}
	ZmController.showDialog(renameDialog, this._renameCb, this._pendingActionData);
	renameDialog.registerCallback(DwtDialog.CANCEL_BUTTON, this._clearDialog, this, renameDialog);
};

/**
 * Deletes an organizer.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._deleteListener =
function(ev) {
	this._doDelete(this._getActionedOrganizer(ev));
};

/**
 * @private
 */
ZmTreeController.prototype._emptyListener =
function(ev) {
	this._doEmpty(this._getActionedOrganizer(ev));
};

/**
 * Moves an organizer into another folder.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._moveListener =
function(ev) {
	this._pendingActionData = this._getActionedOrganizer(ev);
	var moveToDialog = appCtxt.getChooseFolderDialog();
	if (!this._moveCb) {
		this._moveCb = new AjxCallback(this, this._moveCallback);
	}
	ZmController.showDialog(moveToDialog, this._moveCb, this._getMoveParams(moveToDialog));
	moveToDialog.registerCallback(DwtDialog.CANCEL_BUTTON, this._clearDialog, this, moveToDialog);
};

/**
 * @private
 */
ZmTreeController.prototype._getMoveParams =
function(dlg) {
	var omit = {};
	omit[ZmFolder.ID_SPAM] = true;
	return {
		data:			this._pendingActionData,
		treeIds:		[this.type],
		overviewId:		dlg.getOverviewId(appCtxt.getCurrentAppName() + '_' + this.type),
		omit:			omit,
		title:			AjxStringUtil.htmlEncode(this._getMoveDialogTitle()),
		description:	ZmMsg.targetFolder,
		appName:		ZmOrganizer.APP[this.type]
	};
};

/**
 * Expands the tree below the action'd node.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._expandAllListener =
function(ev) {
	var organizer = this._getActionedOrganizer(ev);
	var treeView = this.getTreeView(this._actionedOverviewId);
	var ti = treeView.getTreeItemById(organizer.id);
	window.duringExpandAll = true;
	ti.setExpanded(true, true);
	window.duringExpandAll = false;
	if (window.afterExpandAllCallback) {
		window.afterExpandAllCallback(); //save the explicit setting now after all was expanded - so only one request instead of many
		window.afterExpandAllCallback  = null;
	}
};

/**
 * Mark's an organizer's contents as read.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._markAllReadListener =
function(ev) {
	this._doMarkAllRead(this._getActionedOrganizer(ev));
};

/**
 * Syncs all the organizers to its feed (URL).
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._syncAllListener =
function(ev) {
	// Loop over all the TreeViews
	for (var overviewId in this._treeView) {
		var treeView = this.getTreeView(overviewId);
		var rootId = ZmOrganizer.getSystemId(ZmOrganizer.ID_ROOT, appCtxt.getActiveAccount());
		var rootTreeItem = treeView.getTreeItemById(rootId);
		var treeItems = rootTreeItem && rootTreeItem.getItems();
		if (treeItems) {
			for (var i = 0; i < treeItems.length; i++) {
				var ti = treeItems[i];
				var folder = ti && ti.getData && ti.getData(Dwt.KEY_OBJECT);
				if (folder && (folder.isFeed() || folder.hasFeeds())) {
					this._syncFeeds(folder);
				}
			}
		}
	}
};

/**
 * Syncs an organizer to its feed (URL).
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._syncListener =
function(ev) {
	this._syncFeeds(this._getActionedOrganizer(ev));
};

/**
 * @private
 */
ZmTreeController.prototype._syncFeeds =
function(f) {
	if (f.isFeed()) {
		this._doSync(f);
	} else if (f.hasFeeds()) {
		var a = f.children.getArray();
		var sz = f.children.size();
		for (var i = 0; i < sz; i++) {
			if (a[i].isFeed() || (a[i].hasFeeds && a[i].hasFeeds())) {
				this._syncFeeds(a[i]);
			}
		}
	}
};

/**
 * Brings up a dialog for editing organizer properties.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._editPropsListener = 
function(ev) {
	var folderPropsDialog = appCtxt.getFolderPropsDialog();
	folderPropsDialog.popup(this._getActionedOrganizer(ev));
};

/**
 * Handles a drag event by setting the source data.
 *
 * @param {DwtDragEvent}	ev		a drag event
 * 
 * @private
 */
ZmTreeController.prototype._dragListener =
function(ev) {
	switch (ev.action) {
		case DwtDragEvent.SET_DATA:
			ev.srcData = {data:ev.srcControl.getData(Dwt.KEY_OBJECT), controller:this};
			break;
	}
};

/**
 * Called when a dialog we opened is closed. Sets the style of the actioned
 * tree item from "actioned" back to its normal state.
 * 
 * @private
 */
ZmTreeController.prototype._menuPopdownActionListener = 
function() {
	if (this._pendingActionData) { return; }

	var treeView = this.getTreeView(this._actionedOverviewId);
	if (this._actionedOrganizer && (treeView.getSelected() != this._actionedOrganizer)) {
		var ti = treeView.getTreeItemById(this._actionedOrganizer.id);
		if (ti) {
			ti._setActioned(false);
		}
	}
};

// Callbacks

/**
 * Called when a "New ..." dialog is submitted to create the organizer.
 *
 * @param {Hash}	params	a hash of parameters
 * @param {ZmOrganizer}	params.organizer	the parent organizer
 * @param {String}  params.name	the name of the new organizer
 * 
 * @private
 */
ZmTreeController.prototype._newCallback =
function(params) {
	this._doCreate(params);
	this._clearDialog(this._getNewDialog());
};

/**
 * Called when a "Rename ..." dialog is submitted to rename the organizer.
 *
 * @param {ZmOrganizer}	organizer		the organizer
 * @param {String}	name		the new name of the organizer
 * 
 * @private
 */
ZmTreeController.prototype._renameCallback =
function(organizer, name) {
	this._doRename(organizer, name);
	this._clearDialog(this._getRenameDialog());
};

/**
 * Called when a "Move To ..." dialog is submitted to move the organizer.
 *
 * @param {ZmFolder}	folder		the target folder
 * 
 * @private
 */
ZmTreeController.prototype._moveCallback =
function(folder) {
	this._doMove(this._pendingActionData, folder);
	this._clearDialog(appCtxt.getChooseFolderDialog());
};

/**
 * Called if a user has agreed to go ahead and delete an organizer.
 *
 * @param {ZmOrganizer}	organizer	the organizer to delete
 * 
 * @private
 */
ZmTreeController.prototype._deleteShieldYesCallback =
function(organizer) {
	this._doDelete(organizer);
	this._clearDialog(this._deleteShield);
};

/**
 * @private
 */
ZmTreeController.prototype._emptyShieldYesCallback = 
function(organizer) {
	this._doEmpty(organizer);
	this._clearDialog(this._emptyShield);
};

/**
 * Prompts user before folder is emptied.
 *
 * @param {DwtUiEvent}		ev		the UI event
 *
 * @private
 */

ZmTreeController.prototype._getEmptyShieldWarning =
function(ev) {
    var organizer = this._pendingActionData = this._getActionedOrganizer(ev);
	if (appCtxt.getAppViewMgr().isOpenedItemIncludedInFolder(organizer.nId)) {
		appCtxt.getAppViewMgr().popupBlockItemDeletionWarningDialog();
		return;
	}
	var ds = this._emptyShield = appCtxt.getOkCancelMsgDialog();
	ds.reset();
	ds.registerCallback(DwtDialog.OK_BUTTON, this._emptyShieldYesCallback, this, organizer);
	ds.registerCallback(DwtDialog.CANCEL_BUTTON, this._clearDialog, this, this._emptyShield);
	var msg = (organizer.nId != ZmFolder.ID_TRASH)
		? (AjxMessageFormat.format(ZmMsg.confirmEmptyFolder, organizer.getName()))
		: ZmMsg.confirmEmptyTrashFolder;
	ds.setMessage(msg, DwtMessageDialog.WARNING_STYLE);

	var focusButtonId = (organizer.nId == ZmFolder.ID_TRASH || organizer.nId == ZmFolder.ID_SPAM) ?  DwtDialog.OK_BUTTON : DwtDialog.CANCEL_BUTTON;
	ds.associateEnterWithButton(focusButtonId);
	ds.popup(null, focusButtonId);

	if (!(organizer.nId == ZmFolder.ID_SPAM || organizer.isInTrash())) {
		var cancelButton = ds.getButton(DwtDialog.CANCEL_BUTTON);
		cancelButton.focus();
	}
};

// Miscellaneous private methods

/**
 * Returns the organizer that's currently selected for action (via right-click).
 * Note: going up the object tree to find the actioned organizer will only work 
 * for tree item events; it won't work for action menu item events, since action
 * menus are children of the shell.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTreeController.prototype._getActionedOrganizer =
function(ev) {
	if (this._actionedOrganizer) {
		return this._actionedOrganizer;
	}
		
	var obj = ev.item;
	while (obj) {
		var data = obj.getData(Dwt.KEY_OBJECT);
		if (data instanceof ZmOrganizer) {
			this._actionedOrganizer = data;
			return this._actionedOrganizer;
		}
		obj = obj.parent;
	}
	return null;
};

/**
 * Shows or hides the tree view. It is hidden only if there is no data, and we
 * have been told to hide empty tree views of this type.
 * 
 * @param {constant}	overviewId		the overview ID
 * 
 * @private
 */
ZmTreeController.prototype._checkTreeView =
function(overviewId) {
	if (!overviewId || !this._treeView[overviewId]) { return; }

	var account = this._opc.getOverview(overviewId).account;
	var dataTree = this.getDataTree(account);
	var hide = (ZmOrganizer.HIDE_EMPTY[this.type] && dataTree && (dataTree.size() == 0));
	this._treeView[overviewId].setVisible(!hide);
};
}
if (AjxPackage.define("zimbraMail.share.controller.ZmTagTreeController")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines the tag tree controller.
 *
 */

/**
 * Creates a tag tree controller.
 * @class
 * This class controls a tree display of tags.
 *
 * @extends	ZmTreeController
 */
ZmTagTreeController = function() {

	ZmTreeController.call(this, ZmOrganizer.TAG);

	this._listeners[ZmOperation.NEW_TAG]		= this._newListener.bind(this);
	this._listeners[ZmOperation.RENAME_TAG]		= this._renameListener.bind(this);
	this._listeners[ZmOperation.TAG_COLOR_MENU]	= this._colorListener.bind(this);
};

ZmTagTreeController.prototype = new ZmTreeController;
ZmTagTreeController.prototype.constructor = ZmTagTreeController;

ZmTagTreeController.prototype.isZmTagTreeController = true;
ZmTagTreeController.prototype.toString = function() { return "ZmTagTreeController"; };

// Public methods

/**
 * Adds listeners for the color change menu items.
 * 
 * @return	{ZmActionMenu}		the action menu
 * 
 * @private
 */
ZmTagTreeController.prototype._getActionMenu =
function() {
	var menu = ZmTreeController.prototype._getActionMenu.call(this);
	if (menu && !menu._initialized) {
		var mi = menu.getMenuItem(ZmOperation.TAG_COLOR_MENU);
		if (mi) {
            mi.getMenu().addSelectionListener(this._listeners[ZmOperation.TAG_COLOR_MENU]);
		}
		menu._initialized = true;
	}
	return menu;
};

/**
* Resets and enables/disables operations based on context.
*
* @param {Object}		parent		the widget that contains the operations
* @param {String}		id			the currently selected/activated organizer
*/
ZmTagTreeController.prototype.resetOperations = 
function(parent, type, id) {
	var tag = appCtxt.getById(id);
	parent.enableAll(true);
	if (tag.isSystem()) {
		parent.enable([ZmOperation.RENAME_TAG, 
					   ZmOperation.TAG_COLOR_MENU, ZmOperation.DELETE_WITHOUT_SHORTCUT], false);
	}
	parent.enable(ZmOperation.MARK_ALL_READ, (tag && (tag.numUnread > 0)));
//	this._resetOperation(parent, ZmOperation.EXPORT_FOLDER, ZmMsg.exportTag);
};

// Private/protected methods

/**
 * Returns ops available for "Tags" container.
 * 
 * @private
 */
ZmTagTreeController.prototype._getHeaderActionMenuOps =
function() {
	return [ZmOperation.NEW_TAG];
};

/**
 * Returns ops available for tags.
 * 
 * @private
 */
ZmTagTreeController.prototype._getActionMenuOps = function() {

	return [
		ZmOperation.NEW_TAG,
		ZmOperation.MARK_ALL_READ,
		ZmOperation.DELETE_WITHOUT_SHORTCUT,
		ZmOperation.RENAME_TAG,
		ZmOperation.TAG_COLOR_MENU,
		ZmOperation.OPEN_IN_TAB
	];
};

/**
 * Returns a "New Tag" dialog.
 * 
 * @private
 */
ZmTagTreeController.prototype._getNewDialog =
function() {
	return appCtxt.getNewTagDialog();
};

/**
 * Returns a "Rename Tag" dialog.
 * 
 * @private
 */
ZmTagTreeController.prototype._getRenameDialog =
function() {
	return appCtxt.getRenameTagDialog();
};

// Actions

/**
 * Called when a left click occurs (by the tree view listener). A search for items with
 * the tag will be performed.
 *
 * @param {ZmTag}	tag		the tag that was clicked
 * 
 * @private
 */
ZmTagTreeController.prototype._itemClicked = function(tag, openInTab) {

	var searchFor;
	switch (appCtxt.getCurrentAppName()) {
		case ZmApp.CONTACTS:    searchFor = ZmItem.CONTACT; break;
		case ZmApp.CALENDAR:    searchFor = ZmItem.APPT; break;
		case ZmApp.BRIEFCASE:   searchFor = ZmItem.BRIEFCASE_ITEM; break;
		case ZmApp.TASKS:       searchFor = ZmItem.TASK; break;
		default:                searchFor = ZmId.SEARCH_MAIL; break;
	}

	var params = {
		query:              tag.createQuery(),
		searchFor:          searchFor,
		noGal:              true,
		inclSharedItems:    true,
		getHtml:            appCtxt.get(ZmSetting.VIEW_AS_HTML),
		accountName:        appCtxt.multiAccounts ? tag.getAccount().name : null,
		userInitiated:      openInTab
	};

    //Bug:45878 Don't do a multi-account search for tags
    var sc = appCtxt.getSearchController();
	sc.searchAllAccounts = false;
	sc.search(params);
};

// Listeners

/**
 * Deletes a tag. A dialog will first be displayed asking the user if they
 * are sure they want to delete the tag.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTagTreeController.prototype._deleteListener = 
function(ev) {
	var organizer = this._pendingActionData = this._getActionedOrganizer(ev);
	var ds = this._deleteShield = appCtxt.getYesNoMsgDialog();
	ds.reset();
	ds.registerCallback(DwtDialog.NO_BUTTON, this._clearDialog, this, this._deleteShield);
	ds.registerCallback(DwtDialog.YES_BUTTON, this._deleteShieldYesCallback, this, organizer);
	var msg = AjxMessageFormat.format(ZmMsg.askDeleteTag, organizer.getName(false, ZmOrganizer.MAX_DISPLAY_NAME_LENGTH));
	ds.setMessage(msg, DwtMessageDialog.WARNING_STYLE);
	ds.popup();
};

/**
 * Changes the color of a tag.
 *
 * @param {DwtUiEvent}	ev		the UI event
 * 
 * @private
 */
ZmTagTreeController.prototype._colorListener = 
function(ev) {
	var tag = this._getActionedOrganizer(ev);
	if (tag) {
        var color = ev.item.getData(ZmOperation.MENUITEM_ID);
        if (String(color).match(/^#/)) {
            tag.setRGB(color);
        }
        else {
            tag.setColor(color);
        }
	}
};

/**
 * Handles the potential drop of something onto a tag. Only items may be dropped.
 * The source data is not the items themselves, but an object with the items (data)
 * and their controller, so they can be moved appropriately. Dropping an item onto
 * a tag causes the item to be tagged.
 *
 * @param {DwtDropEvent}	ev		the drop event
 * 
 * @private
 */
ZmTagTreeController.prototype._dropListener =
function(ev) {
	var data = ev.srcData.data;
	if (ev.action == DwtDropEvent.DRAG_ENTER) {
		var sample = (data instanceof Array) ? data[0] : data;
		var tag = ev.targetControl.getData(Dwt.KEY_OBJECT);
		if (tag.id == ZmOrganizer.ID_ROOT) {
			ev.doIt = false;
		} else if (sample instanceof ZmItem && sample.isReadOnly()) {
			ev.doIt = false;
		} else if (appCtxt.multiAccounts && tag.getAccount() != sample.account) {
			ev.doIt = false;
		} else {
			ev.doIt = this._dropTgt.isValidTarget(data);
		}
	} else if (ev.action == DwtDropEvent.DRAG_DROP) {
		var ctlr = ev.srcData.controller;
		var items = (data instanceof Array) ? data : [data];
		ctlr._doTag(items, ev.targetControl.getData(Dwt.KEY_OBJECT), true);
	}
};

/**
 * Handles a color change event.
 *
 * @param {ZmEvent}		ev				the change event
 * @param {ZmTreeView}	treeView		the tree view
 * @param {constant}	overviewId		the overview ID
 * 
 * @private
 */
ZmTagTreeController.prototype._changeListener =
function(ev, treeView, overviewId) {
	var fields = ev.getDetail("fields");
	var organizers = ev.getDetail("organizers");
	for (var i = 0; i < organizers.length; i++) {
		var tag = organizers[i];
		if (ev.event == ZmEvent.E_MODIFY && ((fields && fields[ZmOrganizer.F_COLOR]))) {
			var node = treeView.getTreeItemById(tag.id);
			if (node)
				node.setImage(tag.getIconWithColor());
		} else {
			ZmTreeController.prototype._changeListener.call(this, ev, treeView, overviewId);
		}
	}
};

/**
 * @private
 */
ZmTagTreeController.prototype._setTreeItemColor =
function(treeItem, organizer) {
};
}
if (AjxPackage.define("zimbraMail.share.controller.ZmFolderTreeController")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 