#
# ***** BEGIN LICENSE BLOCK *****
# Zimbra Collaboration Suite Server
# Copyright (C) 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
#
# This program is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software Foundation,
# version 2 of the License.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# You should have received a copy of the GNU General Public License along with this program.
# If not, see <https://www.gnu.org/licenses/>.
# ***** END LICENSE BLOCK *****
#


from logmsg import *
import re
import time

class Section():
	def __init__(self, name=None):
		self.name		= name
		self.changed = False
		self.config = {
						"depends"    : {},
						"rewrites"   : {},
						"restarts"   : {},
						"ldap"       : {},
						"configkeys" : {},
						"requiredvars" : {},
						"postconf"   : {},
						"postconfd"  : {},
					}

	def depends(self, key=None, val=None):
		if val is not None:
			self.config["depends"][key] = val
		if key is not None:
			if key in self.config["depends"]:
				return self.config["depends"][key]
			return None
		return self.config["depends"]

	def rewrites(self, key=None, val=None, mode=None):
		if val is not None:
			self.config["rewrites"][key] = (val,mode)
		if key is not None:
			if key in self.config["rewrites"]:
				return self.config["rewrites"][key]
			return None
		return self.config["rewrites"]

	def restarts(self, service=None, val=None):
		if val is not None:
			self.config["restarts"][service] = val
		if service is not None:
			if service in self.config["restarts"]:
				return self.config["restarts"][service]
			return None
		return self.config["restarts"]

	def requiredvars(self, var=None, val=None):
		if val is not None:
			self.config["requiredvars"][var] = val
		if var is not None:
			if var in self.config["requiredvars"]:
				return self.config["requiredvars"][var]
			return None
		return self.config["requiredvars"]

	def postconf(self, key=None, val=None):
		if val is not None:
			self.config["postconf"][key] = val
		if key is not None:
			if key in self.config["postconf"]:
				return self.config["postconf"][key]
			return None
		return self.config["postconf"]

	def postconfd(self, key=None, val=None):
		if val is not None:
			self.config["postconfd"][key] = val
		if key is not None:
			if key in self.config["postconfd"]:
				return self.config["postconfd"][key]
			return None
		return self.config["postconfd"]

	def ldap(self, key=None, val=None):
		if val is not None:
			self.config["ldap"][key] = val
		if key is not None:
			if key in self.config["ldap"]:
				return self.config["ldap"][key]
			return None
		return self.config["ldap"]

class MtaConfig():
	def __init__(self):
		self.sections = {}
		self.sectionMap = {
				"amavis"     : "mta",
				"dhparam"    : "!dhparam",
				"sasl"       : "mta",
				"webxml"     : "mailbox",
				"nginx"      : "proxy",
				}

	def getSection(self,name):
		if (self.sections and name in self.sections):
			return self.sections[name]
		return None

	def getSections(self):
		return self.sections

	def addSection(self, section):
		self.sections[section.name] = section

	def getServiceMap(self, sname):
		if self.sectionMap.has_key(sname):
			return self.sectionMap[sname]
		return sname

	def load(self, cf, state):
		self.loaded = True
		self.config = {}

		t1 = time.clock()
		lines = open(cf,'r').readlines()
			
		if (len(lines) == 0):
			raise Exception, "Empty config file cf"

		i = 0
		while i < len(lines):
			if lines[i] == "":
				continue

			# No strip() required for zero-arg split()
			fields = lines[i].split()

			section = Section()

			if re.match(r"SECTION", lines[i]):
				section.name = fields[1]
				servicemap = self.getServiceMap(section.name)

			i += 1

			# the previous version continued to add the section to the list, with
			# no data; this resulted in the forced run of proxyconfgen even when proxy
			# was disabled.  Probably a bug, not replicating.
			if not state.checkConditional("SERVICE", servicemap):
				Log.logMsg(4, "Service %s is not enabled.  Skipping %s" % (servicemap, section.name))
				while i < len(lines) and (not re.match(r"SECTION", lines[i])):
					i += 1
				continue

			if len(fields) > 2 and fields[2] == "DEPENDS":
				for f in fields[3:]:
					Log.logMsg(5, "Adding dependency %s to section %s" % (f, section.name))
					section.depends(f, True)
				
			# Process the entire section
			while i < len(lines) and (not re.match(r"SECTION", lines[i])):
				if lines[i].strip() == "":
					i+=1
					continue

				fields = lines[i].split()
				ln = lines[i].strip()
				if re.match(r"REWRITE", ln):
					Log.logMsg(5, "Adding file rewrite %s to section %s" % (fields[1], section.name))
					if len(fields) > 3:
						section.rewrites(fields[1], fields[2], fields[4])
					else:
						section.rewrites(fields[1], fields[2])
				elif re.match(r"RESTART", ln):
					for service in fields[1:]:
						Log.logMsg(5, "Adding service %s to restarts in section %s" % (service, section.name)); 
						section.restarts(service, True)
				elif re.match(r"VAR|LOCAL|MAPLOCAL", ln):
					Log.logMsg(5, "Adding %s to required vars:  processing %s" % (fields[1], ln));
					section.requiredvars(fields[1], fields[0])
				elif re.match(r"POSTCONFD", ln):
					if len(fields) > 2:
						if (re.match(r"VAR|LOCAL|FILE", fields[2])):
							val = state.lookUpConfig(fields[2], fields[3])
							section.requiredvars(fields[3], fields[2])
							if val is not None:
								if (str(val).upper() == "TRUE"):
									val = "yes"
								if (str(val).upper() == "FALSE"):
									val = "no"
							else:
								val = ""
							Log.logMsg(5, "Adding to postconfd commands: \'%s\' %s=\'%s\'" % (ln, fields[1], val))
							section.postconfd(fields[1],val)
						else:
							value = " ".join(fields[2:len(fields)])
							section.postconfd(fields[1],value)
					else:
						section.postconfd(fields[1],"")
				elif re.match(r"POSTCONF", ln):
					if len(fields) > 2:
						if (re.match(r"VAR|LOCAL|FILE|MAPLOCAL", fields[2])):
							val = state.lookUpConfig(fields[2], fields[3])
							section.requiredvars(fields[3], fields[2])
							if val is not None:
								if (str(val).upper() == "TRUE"):
									val = "yes"
								if (str(val).upper() == "FALSE"):
									val = "no"
							else:
								val = ""
							Log.logMsg(5, "Adding to postconf commands: \'%s\' %s=\'%s\'" % (ln, fields[1], val))
							section.postconf(fields[1],val)
						else:
							value = " ".join(fields[2:len(fields)]);
							section.postconf(fields[1],value)
					else:
						section.postconf(fields[1],"")
				elif re.match(r"PROXYGEN", ln):
					# ignore this; proxygen hardcoded in the proxy section logic
					pass
				elif re.match(r"LDAP", ln):
					if (re.match(r"LOCAL|MAPLOCAL", fields[2])):
						val = state.lookUpConfig(fields[2], fields[3])
						Log.logMsg(5, "Adding to ldap commands: \'%s\' %s=\'%s\'" % (ln, fields[1], val))
						section.ldap(fields[1],val)
				elif re.match(r"if", ln):
					if state.checkConditional(fields[1], fields[2]):
						Log.logMsg(5, "checkConditional %s %s is true" % (fields[1], fields[2]));
						i += 1
						while i < len(lines) and (not re.match(r"fi", lines[i].strip())):
							if lines[i].strip() == "":
								i += 1
								continue
							fields = lines[i].split()
							ln = lines[i].strip()
							if re.match(r"POSTCONFD", ln):
								if len(fields) > 2:
									if (re.match(r"VAR|LOCAL|FILE", fields[2])):
										val = state.lookUpConfig(fields[2], fields[3])
										section.requiredvars(fields[3], fields[2])
										Log.logMsg(5, "Adding to postconfd commands: \'%s\' %s=\'%s\'" % (ln, fields[1], val))
										section.postconfd(fields[1],val)
									else:
										value = " ".join(fields[2:len(fields)]);
										section.postconfd(fields[1],value)
								else:
									section.postconfd(fields[1],"")
							elif re.match(r"POSTCONF", ln):
								if len(fields) > 2:
									if (re.match(r"VAR|LOCAL|FILE", fields[2])):
										val = state.lookUpConfig(fields[2], fields[3])
										section.requiredvars(fields[3], fields[2])
										Log.logMsg(5, "Adding to postconf commands: \'%s\' %s=\'%s\'" % (ln, fields[1], val))
										section.postconf(fields[1],val)
									else:
										value = " ".join(fields[2:len(fields)]);
										section.postconf(fields[1],value)
								else:
									section.postconf(fields[1],"")
							elif re.match(r"VAR|LOCAL", ln):
								Log.logMsg(5, "Adding %s to required vars:  processing %s" % (fields[1], ln));
								section.requiredvars(fields[1], fields[0])
							else:
								Log.logMsg(2, "Error processing line %s" % (lines[i],));
							i += 1
					else:
						Log.logMsg(5, "checkConditional %s %s is false: \'%s\'" % (fields[1], fields[2], lines[i]));
						while i < len(lines) and (not re.match(r"fi", lines[i].strip())):
							Log.logMsg(5, "Skipping line=\'%s\'" % (lines[i],));
							i += 1
				elif re.match(r"fi", ln):
					Log.logMsg(5, "endof conditional reached");
					continue
				elif re.match(r"MAPFILE", ln):
					value = state.lookUpConfig(fields[0], fields[1])
				else:
					Log.logMsg(2, "Unknown line format %s" % (lines[i],));

				i+=1

			self.addSection(section)

		dt = time.clock()-t1
		Log.logMsg(5,"zmconfigd.cf loaded in %.2f seconds" % dt)
