if (AjxPackage.define("NewWindow_2")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */
/*
 * NewWindow, part 2
 * 
 * Special package to support actions in a new window. So far, there is
 * just composing a message and displaying a RFC attachment in a new window.
 * Everything that might be needed for those is in here: HTML editor,
 * contact picker (search, timezone, list view), msg view, attachments, etc.
 * 
 * NOTE: This package is not optimized at all - it contains everythings that
 * might possibly be needed in a new window.
 */
if (AjxPackage.define("ajax.dwt.keyboard.DwtTabGroupEvent")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @constructor
 * @class
 * This class represents a the tab event. This event is used to indicate changes in
 * the state of {@link DwtTabGroup} objects (e.g. member addition and deletion). 
 * 
 * @author Ross Dargahi
 * 
 * @private
 */
DwtTabGroupEvent = function() {
	/**
	 * Tab group for which the event is being generated
	 * @type DwtTabGroup
	 */
	this.tabGroup = null;
	
	/**
	 * New focus member
	 * @type DwtControl|HTMLElement
	 */
	this.newFocusMember = null;
}

/**
 * Returns a string representation of this object.
 * 
 * @return {string}	a string representation of this object
 */
DwtTabGroupEvent.prototype.toString = 
function() {
	return "DwtTabGroupEvent";
}

/**
 * Resets the members of the event.
 * 
 */
DwtTabGroupEvent.prototype.reset =
function() {
	this.tabGroup = null;
	this.newFocusMember = null;
}
}
if (AjxPackage.define("ajax.dwt.keyboard.DwtKeyMapMgr")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */


/**
 * Creates and initializes a manager for the given keymap.
 * @constructor
 * @class
 * A keymap manager parses the keymap into a form that is easily used for 
 * translating key codes into actions. It also provides some static methods
 * that map the available keyboard to key codes, and which qualify certain
 * keys as punctuation, etc.
 * 
 * @author Ross Dargahi
 *
 * @param {DwtKeyMap}	keyMap the keymap
 *
 * @private
 */
DwtKeyMapMgr = function(keyMap) {

	var map = this._map = keyMap.getMap();
	this._args = keyMap._args;
	
	// build FSA for each mapping
	this._fsas = {};
	for (var key in map) {
		DBG.println(AjxDebug.DBG3, "building FSA for key: " + key);
		this._fsas[key] = DwtKeyMapMgr.__buildFSA({}, map[key], key);
	}
	DBG.dumpObj(AjxDebug.DBG3, this._fsas);
};

DwtKeyMapMgr.prototype.toString = function() { return "DwtKeyMapMgr"; };
DwtKeyMapMgr.prototype.isDwtKeyMapMgr = true;

DwtKeyMapMgr.NOT_A_TERMINAL = -999;
DwtKeyMapMgr.TAB_KEYCODE = DwtKeyEvent.KEY_TAB;


/**
 * This method will attempt to look up the action code for a given key sequence in
 * a given key map. 
 * 
 * @param {string}		keySeq				key sequence to lookup
 * @param {string}		mappingName			keymap name in which to search
 * @param {boolean}		forceActionCode		if <code>true</code>, then if the key sequence contains both
 * 											a submap and an action code, then return the action code.
 * 											If this parameter is false or omitted, then
 * 											{@link DwtKeyMapMgr.NOT_A_TERMINAL} will be returned for
 * 											a key sequence that contains both a submap and an action code.
 * 
 * @return {string|number}	the action code for the provided key map name, null if there is no action code
 * 		or {@link DwtKeyMapMgr.NOT_A_TERMINAL} if the key sequence is an intermediate
 * 		node in the key map (i.e. has a submap)
 * 
 */
DwtKeyMapMgr.prototype.getActionCode =
function(keySeq, mappingName, forceActionCode) {
	//DBG.println(AjxDebug.DBG3, "Getting action code for: " + keySeq + " in map: " + mappingName);
	var mapping =  this._fsas[mappingName];

	if (!mapping) {
		DBG.println(AjxDebug.DBG3, "No keymap for: " + mappingName);
		return null;
	}

	var keySeqLen = keySeq.length;
	var tmpFsa = mapping;
	var key;
	for (var j = 0; j < keySeqLen && tmpFsa; j++) {
		key = keySeq[j];

		if (!tmpFsa || !tmpFsa[key]) break;

		if (j < keySeqLen - 1) {
			tmpFsa = tmpFsa[key].subMap;
		}
	}

	if (tmpFsa && tmpFsa[key]) {
		var binding = tmpFsa[key];
		/* If the binding does not have a submap, then it must have an action code
		 * so return it. Else if the binding does not have an action code (i.e. it
		 * has a submap only) or if forceActionCode is false, then return DwtKeyMapMgr.NOT_A_TERMINAL
		 * since we are to behave like an intermediate node. Else return the action code. */
		if (!binding.subMap || forceActionCode) {
			var inherited = this.__getInheritedActionCode(keySeq, mapping, forceActionCode);
            //if keyMap not available then return the inherited keyMap.
            return inherited == DwtKeyMapMgr.NOT_A_TERMINAL ? DwtKeyMapMgr.NOT_A_TERMINAL : ( binding.actionCode || inherited );
		} else {
			return DwtKeyMapMgr.NOT_A_TERMINAL;
		}
	} else {
		return this.__getInheritedActionCode(keySeq, mapping, forceActionCode);
	}
};

/**
 * Returns the action for the given map and key sequence.
 * 
 */
DwtKeyMapMgr.prototype.getAction =
function(mapName, keySeq) {
	return this._map[mapName][keySeq];
};

/**
 * Returns the key sequences associated with the given map and action.
 */
DwtKeyMapMgr.prototype.getKeySequences =
function(mapName, action) {
	var keySeqs = [];
	for (var ks in this._map[mapName]) {
		if (this._map[mapName][ks] == action) {
			keySeqs.push(ks);
		}
	}
	return keySeqs;
};

/**
 * Allow the programmatic setting of a key sequence mapping for a given map
 * 
 * @param {string} 			mapName map name to affect
 * @param {string} 			keySeq the key sequence to set
 * @param {string|number} action the action code for the key sequence
 */
DwtKeyMapMgr.prototype.setMapping =
function(mapName, keySeq, action) {
	this._map[mapName][keySeq] = action;
};

/**
 * Allow the programatting removal of a key sequence mapping for a given map
 * 
 * @param {string} mapName map name to affect
 * @param {string} keySeq the key sequence to remove
 */
DwtKeyMapMgr.prototype.removeMapping =
function(mapName, keySeq) {
	delete this._map[mapName][keySeq];
};

/**
 * Replace the key sequence for a given action in a keymap 
 * 
 * @param {string} mapName map name to affect
 * @param {string} oldKeySeq the key sequence to replace
 * @param {string} newKeySeq the new key sequence
 */
DwtKeyMapMgr.prototype.replaceMapping =
function(mapName, oldKeySeq, newKeySeq) {
	var action = this._map[mapName][oldKeySeq];
	if (!action) return;
	this.removeMapping(mapName, oldKeySeq);
	this.setMapping(mapName, newKeySeq, action);
};

DwtKeyMapMgr.prototype.setArg =
function(mapName, action, arg) {
	if (!this._args[mapName]) {
		this._args[mapName] = {};
	}
	this._args[mapName][action] = arg;
};

DwtKeyMapMgr.prototype.removeArg =
function(mapName, action) {
	delete this._args[mapName][action];
};

DwtKeyMapMgr.prototype.getArg =
function(mapName, action) {
	return this._args[mapName] ? this._args[mapName][action] : null;
};

/**
 * Reloads a given keymap
 * 
 * @param {string} mapName Name of the keymap to reload
 */
DwtKeyMapMgr.prototype.reloadMap =
function(mapName) {
	this._fsas[mapName] = DwtKeyMapMgr.__buildFSA({}, this._map[mapName], mapName);
};

/**
 * Returns a list of maps that the given map inherits from.
 *
 * @param {string} mapName Name of the keymap to reload
 */
DwtKeyMapMgr.prototype.getAncestors =
function(mapName, list) {
    list = list || [];
    var subMap = this._fsas[mapName];
    var parents = subMap && subMap.inherit;
    if (parents && parents.length) {
        for (var i = 0; i < parents.length; i++) {
            list.push(parents[i]);
            list = this.getAncestors(parents[i], list);
        }
    }
    return list;
};

/**
 * Returns true if the given element accepts text input.
 * 
 * @param element	[Element]		DOM element
 */
DwtKeyMapMgr.isInputElement =
function(element) {
	if (!element) { return false; }
	// Check designMode in case we're in an HTML editor iframe
	var dm = element.ownerDocument ? element.ownerDocument.designMode : null;
	if (dm && (dm.toLowerCase() == "on")) { return true; }

	var tag = element.tagName.toUpperCase();
	return (tag == "INPUT" || tag == "TEXTAREA");
};

DwtKeyMapMgr.__buildFSA =
function(fsa, mapping, mapName) {
	for (var i in mapping) {
		// check for inheritance from other maps (in CSV list)
		if (i == DwtKeyMap.INHERIT) {
			fsa.inherit = mapping[i].split(/\s*,\s*/);
			continue;
		}
		 
		var keySeq = i.split(DwtKeyMap.SEP);
		var keySeqLen = keySeq.length;
		var tmpFsa = fsa;
		for (var j = 0; j < keySeqLen; j++) {
			var key = keySeq[j];
			//DBG.println(AjxDebug.DBG3, "Processing: " + key);
			
			if (!tmpFsa[key]) {
				tmpFsa[key] = {};	// first time visiting this key
			}

			if (j == keySeqLen - 1) {
				/* We are at the last key in the sequence so we can bind the
				 * action code to it */
				//DBG.println(AjxDebug.DBG3, "BINDING: " + mapping[i]);
				tmpFsa[key].actionCode = mapping[i];
			} else {
				/* We have more keys in the sequence. If our subMap is null,
				 * then we need to create it to hold the new key sequences */
				if (!tmpFsa[key].subMap) {
					tmpFsa[key].subMap = {};
					//DBG.println(AjxDebug.DBG3, "NEW SUBMAP");
				}
					
				tmpFsa = tmpFsa[key].subMap;
			}			
		}
	}
	return fsa;
};

DwtKeyMapMgr.prototype.__getInheritedActionCode =
function(keySeq, mapping, forceActionCode) {
	if (mapping.inherit && mapping.inherit.length) {
		var actionCode = null;
		var len = mapping.inherit.length;
		for (var i = 0; i < len; i++) {
			DBG.println(AjxDebug.DBG3, "checking inherited map: " + mapping.inherit[i]);
			actionCode = this.getActionCode(keySeq, mapping.inherit[i], forceActionCode);
			if (actionCode != null) {
				return actionCode;
			}
		}
	}
	return null;
};

/**
 * Returns true if the given key event has a modifier which makes it nonprintable.
 * 
 * @param ev	[Event]		key event
 */
DwtKeyMapMgr.hasModifier =
function(ev) {
	return (ev.altKey || ev.ctrlKey || ev.metaKey);
};
}
if (AjxPackage.define("ajax.dwt.keyboard.DwtKeyboardMgr")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Creates an empty keyboard manager. Intended for use as a singleton.
 * @constructor
 * @class
 * This class is responsible for managing focus and shortcuts via the keyboard. That includes dispatching
 * keyboard events (shortcuts), as well as managing tab groups. It is at the heart of the
 * Dwt keyboard navigation framework.
 * <p>
 * {@link DwtKeyboardMgr} intercepts key strokes and translates
 * them into actions which it then dispatches to the component with focus. If the key
 * stroke is a TAB (or Shift-TAB), then focus is moved based on the current tab group.
 * </p><p>
 * A {@link DwtShell} instantiates its own <i>DwtKeyboardMgr</i> at construction.
 * The keyboard manager may then be retrieved via the shell's <code>getKeyboardMgr()</code>
 * function. Once a handle to the shell's keyboard manager is retrieved, then the user is free
 * to add tab groups, and to register keymaps and handlers with the keyboard manager.
 * </p><p>
 * Focus is managed among a stack of tab groups. The TAB button will move the focus within the
 * current tab group. When a non-TAB is received, we first check if the control can handle it.
 * In general, control key events simulate something the user could do with the mouse, and change
 * the state/appearance of the control. For example, ENTER on a DwtButton simulates a button
 * press. If the control does not handle the key event, the event is handed to the application,
 * which handles it based on its current state. The application key event handler is in a sense
 * global, since it does not matter which control received the event.
 * </p><p>
 * At any given time there is a default handler, which is responsible for determining what
 * action is associated with a particular key sequence, and then taking it. A handler should support
 * the following methods:
 * 
 * <ul>
 * <li><i>getKeyMapName()</i> -- returns the name of the map that defines shortcuts for this handler</li>
 * <li><i>handleKeyAction()</i> -- performs the action associated with a shortcut</li>
 * <li><i>handleKeyEvent()</i>	-- optional override; handler solely responsible for handling event</li>
 * </ul>
 * </p>
 *
 * @author Ross Dargahi
 *
 * @param	{DwtShell}	shell		the shell
 * @see DwtShell
 * @see DwtTabGroup
 * @see DwtKeyMap
 * @see DwtKeyMapMgr
 * 
 * @private
 */
DwtKeyboardMgr = function(shell) {

	DwtKeyboardMgr.__shell = shell;

    this.__kbEventStatus = DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED;
    this.__keyTimeout = DwtKeyboardMgr.SHORTCUT_TIMEOUT;

    // focus
    this.__tabGrpStack = [];
    this.__currTabGroup = null;
    this.__tabGroupChangeListenerObj = this.__tabGrpChangeListener.bind(this);

    // shortcuts
    this.__shortcutsEnabled = false;
	this.__defaultHandlerStack = [];
	this.__currDefaultHandler = null;
    this.__killKeySeqTimedAction = new AjxTimedAction(this, this.__killKeySequenceAction);
    this.__killKeySeqTimedActionId = -1;
    this.__keySequence = [];
    this._evtMgr = new AjxEventMgr();

    Dwt.setHandler(document, DwtEvent.ONKEYDOWN, DwtKeyboardMgr.__keyDownHdlr);
    Dwt.setHandler(document, DwtEvent.ONKEYUP, DwtKeyboardMgr.__keyUpHdlr);
    Dwt.setHandler(document, DwtEvent.ONKEYPRESS, DwtKeyboardMgr.__keyPressHdlr);
};

DwtKeyboardMgr.prototype.isDwtKeyboardMgr = true;
DwtKeyboardMgr.prototype.toString = function() { return "DwtKeyboardMgr"; };

DwtKeyboardMgr.SHORTCUT_TIMEOUT = 750;

DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED	= "NOT HANDLED";
DwtKeyboardMgr.__KEYSEQ_HANDLED		= "HANDLED";
DwtKeyboardMgr.__KEYSEQ_PENDING		= "PENDING";

/**
 * Checks if the event may be a shortcut from within an input (text input or
 * textarea). Since printable characters are echoed, the shortcut must be non-printable:
 * 
 * <ul>
 * <li>Alt or Ctrl or Meta plus another key</li>
 * <li>Esc</li>
 * </ul>
 * 
 * @param {DwtKeyEvent}	ev	the key event
 * @return	{boolean}	<code>true</code> if the event may be a shortcut
 */

// Enter and all four arrows can be used as shortcuts in an INPUT
DwtKeyboardMgr.IS_INPUT_SHORTCUT_KEY = AjxUtil.arrayAsHash([
    DwtKeyEvent.KEY_END_OF_TEXT,
    DwtKeyEvent.KEY_RETURN,
    DwtKeyEvent.KEY_ARROW_LEFT,
    DwtKeyEvent.KEY_ARROW_UP,
    DwtKeyEvent.KEY_ARROW_RIGHT,
    DwtKeyEvent.KEY_ARROW_DOWN
]);

// Returns true if the key event has a keycode that could be used in an input (INPUT or TEXTAREA) as a shortcut. That
// excludes printable characters.
DwtKeyboardMgr.isPossibleInputShortcut = function(ev) {

	var target = DwtUiEvent.getTarget(ev);
    return !DwtKeyMap.IS_MODIFIER[ev.keyCode] && (ev.keyCode === DwtKeyEvent.KEY_ESCAPE || DwtKeyMapMgr.hasModifier(ev) ||
			(target && target.nodeName.toLowerCase() == "input" && DwtKeyboardMgr.IS_INPUT_SHORTCUT_KEY[ev.keyCode]));
};

/**
 * Pushes the tab group onto the stack and makes it the active tab group.
 * 
 * @param 	{DwtTabGroup}	tabGroup	the tab group to push onto the stack
 * 
 * @see		#popTabGroup
 */
DwtKeyboardMgr.prototype.pushTabGroup = function(tabGroup, preventFocus) {

    if (!(tabGroup && tabGroup.isDwtTabGroup)) {
        DBG.println(AjxDebug.DBG1, "pushTabGroup() called without a tab group: " + tabGroup);
        return;
    }

	DBG.println(AjxDebug.FOCUS, "PUSH tab group " + tabGroup.getName());
	this.__tabGrpStack.push(tabGroup);
	this.__currTabGroup = tabGroup;
	var focusMember = tabGroup.getFocusMember();
	if (!focusMember) {
		focusMember = tabGroup.resetFocusMember(true);
	}
	if (!focusMember) {
		DBG.println(AjxDebug.FOCUS, "DwtKeyboardMgr.pushTabGroup: tab group " + tabGroup.__name + " has no members!");
		return;
	}
	tabGroup.addFocusChangeListener(this.__tabGroupChangeListenerObj);
	if (!preventFocus) {
		this.grabFocus(focusMember);
	}
};

/**
 * Pops the current tab group off the top of the tab group stack. The previous 
 * tab group (if there is one) then becomes the current tab group.
 * 
 * @param {DwtTabGroup} [tabGroup]		the tab group to pop. If supplied, then the tab group
 * 		stack is searched for the tab group and it is removed. If <code>null</code>, then the
 * 		top tab group is popped.
 * 
 * @return {DwtTabGroup}	the popped tab group or <code>null</code> if there is one or less tab groups
 */
DwtKeyboardMgr.prototype.popTabGroup = function(tabGroup) {

    if (!(tabGroup && tabGroup.isDwtTabGroup)) {
        DBG.println(AjxDebug.DBG1, "popTabGroup() called without a tab group: " + tabGroup);
        return null;
    }

    DBG.println(AjxDebug.FOCUS, "POP tab group " + tabGroup.getName());
	
	// we never want an empty stack
	if (this.__tabGrpStack.length <= 1) {
		return null;
	}
	
	// If we are popping a tab group that is not on the top of the stack then
	// we need to find it and remove it.
	if (tabGroup && this.__tabGrpStack[this.__tabGrpStack.length - 1] != tabGroup) {
		var a = this.__tabGrpStack;
		var len = a.length;
		for (var i = len - 1; i >= 0; i--) {
			if (tabGroup == a[i]) {
				a[i].dump(AjxDebug.DBG1);
				break;
			}
		}
		
		/* If there is no match in the stack for tabGroup, then simply return null,
		 * else if the match is not the top item on the stack, then remove it from 
		 * the stack. Else we are dealing with the topmost item on the stack so handle it 
		 * as a simple pop. */
		if (i < 0) { // No match
			return null;
		} else if (i != len - 1) { // item is not on top
			// Remove tabGroup
			a.splice(i, 1);
			return tabGroup;
		}
	} 

	var tabGroup = this.__tabGrpStack.pop();
	tabGroup.removeFocusChangeListener(this.__tabGroupChangeListenerObj);
	
	var currTg = null;
	if (this.__tabGrpStack.length > 0) {
		currTg = this.__tabGrpStack[this.__tabGrpStack.length - 1];
		var focusMember = currTg.getFocusMember();
		if (!focusMember) {
			focusMember = currTg.resetFocusMember(true);
		}
		if (focusMember) {
			this.grabFocus(focusMember);
		}
	}
	this.__currTabGroup = currTg;

	return tabGroup;
};

/**
 * Replaces the current tab group with the given tab group.
 * 
 * @param {DwtTabGroup} tabGroup 	the tab group to use
 * @return {DwtTabGroup}	the old tab group
 */
DwtKeyboardMgr.prototype.setTabGroup = function(tabGroup) {

	var otg = this.popTabGroup();
	this.pushTabGroup(tabGroup);

	return otg;
};

/**
 * Gets the current tab group
 *
 * @return {DwtTabGroup}	current tab group
 */
DwtKeyboardMgr.prototype.getCurrentTabGroup = function() {

    return this.__currTabGroup;
};

/**
 * Adds a default handler to the stack. A handler should define a 'handleKeyAction' method.
 *
 * @param {Object}  handler     default handler
 */
DwtKeyboardMgr.prototype.pushDefaultHandler = function(handler) {

	if (!this.isEnabled() || !handler) {
        return;
    }
	DBG.println(AjxDebug.FOCUS, "PUSH default handler: " + handler);
		
	this.__defaultHandlerStack.push(handler);
	this.__currDefaultHandler = handler;
};

/**
 * Removes a default handler from the stack.
 *
 * @return {Object}  handler     a default handler
 */
DwtKeyboardMgr.prototype.popDefaultHandler = function() {

	DBG.println(AjxDebug.FOCUS, "POP default handler");
	// we never want an empty stack
	if (this.__defaultHandlerStack.length <= 1) {
        return null;
    }

	DBG.println(AjxDebug.FOCUS, "Default handler stack length: " + this.__defaultHandlerStack.length);
	var handler = this.__defaultHandlerStack.pop();
	this.__currDefaultHandler = this.__defaultHandlerStack[this.__defaultHandlerStack.length - 1];
	DBG.println(AjxDebug.FOCUS, "Default handler is now: " + this.__currDefaultHandler);

	return handler;
};

/**
 * Sets the focus to the given object.
 * 
 * @param {HTMLInputElement|DwtControl|string} focusObj		the object to which to set focus, or its ID
 */ 
DwtKeyboardMgr.prototype.grabFocus = function(focusObj) {

	if (typeof focusObj === "string") {
		focusObj = document.getElementById(focusObj);
	}
    else if (focusObj && focusObj.isDwtTabGroup) {
        focusObj = focusObj.getFocusMember() || focusObj.getFirstMember();
    }

    if (!focusObj) {
        return;
    }

	// Make sure tab group knows what's currently focused
	if (this.__currTabGroup) {
		this.__currTabGroup.setFocusMember(focusObj, false, true);
	}
		
	this.__doGrabFocus(focusObj);
};

/**
 * Tells the keyboard manager that the given control now has focus. That control will handle shortcuts and become
 * the reference point for tabbing.
 *
 * @param {DwtControl|Element}  focusObj    control (or element) that has focus
 */
DwtKeyboardMgr.prototype.updateFocus = function(focusObj, ev) {

    if (!focusObj) {
        return;
    }

    var ctg = this.__currTabGroup;
    if (ctg) {
        this.__currTabGroup.__showFocusedItem(focusObj, "updateFocus");
    }
    var control = focusObj.isDwtControl ? focusObj : DwtControl.findControl(focusObj);

    // Set the keyboard mgr's focus obj, which will be handed shortcuts. It must be a DwtControl.
    if (control) {
        this.__focusObj = control;
        DBG.println(AjxDebug.FOCUS, "DwtKeyboardMgr UPDATEFOCUS kbMgr focus obj: " + control);
    }

    // Update the current (usually root) tab group's focus member to whichever of these it contains: the focus obj,
    // its tab group member, or its control.
    var tgm = this._findTabGroupMember(ev || focusObj);
    if (tgm && ctg) {
        ctg.setFocusMember(tgm, false, true);
    }
};

// Goes up the DOM looking for something (element or control) that is in the current tab group.
DwtKeyboardMgr.prototype._findTabGroupMember = function(obj) {

    var ctg = this.__currTabGroup;
    if (!obj || !ctg) {
        return;
    }

    var htmlEl = (obj.isDwtControl && obj.getHtmlElement()) || DwtUiEvent.getTarget(obj, false) || obj;

    try {
        while (htmlEl) {
            if (ctg.contains(htmlEl)) {
                return htmlEl;
            }
            else {
                var control = DwtControl.ALL_BY_ID[htmlEl.id];
                if (control && ctg.contains(control)) {
                    return control;
                }
                else {
                    var tgm = control && control.getTabGroupMember && control.getTabGroupMember();
                    if (tgm && ctg.contains(tgm)) {
                        return tgm;
                    }
                }
            }
            htmlEl = htmlEl.parentNode;
        }
    } catch(e) {
    }

    return null;
};

/**
 * Gets the object that has focus.
 *
 * @return {HTMLInputElement|DwtControl} focusObj		the object with focus
 */
DwtKeyboardMgr.prototype.getFocusObj = function(focusObj) {

	return this.__focusObj;
};

/**
 * This method is used to register an application key handler. If registered, this
 * handler must support the following methods:
 * <ul>
 * <li><i>getKeyMapName</i>: This method returns a string representing the key map 
 * to be used for looking up actions
 * <li><i>handleKeyAction</i>: This method should handle the key action and return
 * true if it handled it else false. <i>handleKeyAction</i> has two formal parameters
 *    <ul>
 *    <li><i>actionCode</i>: The action code to be handled</li>
 *    <li><i>ev</i>: the {@link DwtKeyEvent} corresponding to the last key event in the sequence</li>
 *    </ul>
 * </ul>
 * 
 * @param 	{function}	hdlr	the handler function. This method should have the following
 * 									signature <code>Boolean hdlr(Int actionCode DwtKeyEvent event);</code>
 * 
 * @see DwtKeyEvent
 */
DwtKeyboardMgr.prototype.registerDefaultKeyActionHandler = function(hdlr) {

	if (this.isEnabled()) {
        this.__defaultKeyActionHdlr = hdlr;
    }
};

/**
 * Registers a keymap with the shell. A keymap typically
 * is a subclass of {@link DwtKeyMap} and defines the mappings from key sequences to
 * actions.
 *
 * @param {DwtKeyMap} keyMap		the key map to register
 * 
 */
DwtKeyboardMgr.prototype.registerKeyMap = function(keyMap) {

	if (this.isEnabled()) {
	    this.__keyMapMgr = new DwtKeyMapMgr(keyMap);
    }
};

/**
 * Sets the timeout (in milliseconds) between key presses for handling multi-keypress sequences.
 * 
 * @param 	{number}	timeout		the timeout (in milliseconds)
 */
DwtKeyboardMgr.prototype.setKeyTimeout = function(timeout) {
	this.__keyTimeout = timeout;
};

/**
 * Clears the key sequence. The next key event will begin a new one.
 * 
 */
DwtKeyboardMgr.prototype.clearKeySeq = function() {

	this.__killKeySeqTimedActionId = -1;
	this.__keySequence = [];
};

/**
 * Enables/disables keyboard nav (shortcuts).
 * 
 * @param 	{boolean}	enabled		if <code>true</code>, enable keyboard nav
 */
DwtKeyboardMgr.prototype.enable = function(enabled) {

	DBG.println(AjxDebug.DBG2, "keyboard shortcuts enabled: " + enabled);
	this.__shortcutsEnabled = enabled;
};

DwtKeyboardMgr.prototype.isEnabled = function() {
	return this.__shortcutsEnabled;
};

/**
 * Adds a global key event listener.
 *
 * @param {constant}	ev			key event type
 * @param {AjxListener}	listener	listener to notify
 */
DwtKeyboardMgr.prototype.addListener = function(ev, listener) {
	this._evtMgr.addListener(ev, listener);
};

/**
 * Removes a global key event listener.
 *
 * @param {constant}	ev			key event type
 * @param {AjxListener}	listener	listener to remove
 */
DwtKeyboardMgr.prototype.removeListener = function(ev, listener) {
	this._evtMgr.removeListener(ev, listener);
};

DwtKeyboardMgr.prototype.__doGrabFocus = function(focusObj) {

	if (!focusObj) {
        return;
    }

    var curFocusObj = this.getFocusObj();
    if (curFocusObj && curFocusObj.blur) {
        DBG.println(AjxDebug.FOCUS, "DwtKeyboardMgr DOGRABFOCUS cur focus obj: " + [curFocusObj, curFocusObj._htmlElId || curFocusObj.id].join(' / '));
        curFocusObj.blur();
    }

    DBG.println(AjxDebug.FOCUS, "DwtKeyboardMgr DOGRABFOCUS new focus obj: " + [focusObj, focusObj._htmlElId || focusObj.id].join(' / '));
    if (focusObj.focus) {
        // focus handler should lead to focus update, but just in case ...
        this.updateFocus(focusObj.focus() || focusObj);
    }
};

/**
 * @private
 */
DwtKeyboardMgr.__keyUpHdlr = function(ev) {

	ev = DwtUiEvent.getEvent(ev);
	DBG.println(AjxDebug.KEYBOARD, "keyup: " + ev.keyCode);

	var kbMgr = DwtKeyboardMgr.__shell.getKeyboardMgr();
	if (kbMgr._evtMgr.notifyListeners(DwtEvent.ONKEYUP, ev) === false) {
		return false;
	}

	// clear saved Gecko key
	if (AjxEnv.isMac && AjxEnv.isGeckoBased && ev.keyCode === 0) {
		return DwtKeyboardMgr.__keyDownHdlr(ev);
	}
    else {
		return DwtKeyboardMgr.__handleKeyEvent(ev);
	}
};

/**
 * @private
 */
DwtKeyboardMgr.__keyPressHdlr = function(ev) {

	ev = DwtUiEvent.getEvent(ev);
	DBG.println(AjxDebug.KEYBOARD, "keypress: " + (ev.keyCode || ev.charCode));

	var kbMgr = DwtKeyboardMgr.__shell.getKeyboardMgr();
	if (kbMgr._evtMgr.notifyListeners(DwtEvent.ONKEYPRESS, ev) === false) {
		return false;
	}

	DwtKeyEvent.geckoCheck(ev);

	return DwtKeyboardMgr.__handleKeyEvent(ev);
};

/**
 * @private
 */
DwtKeyboardMgr.__handleKeyEvent =
function(ev) {

	if (DwtKeyboardMgr.__shell._blockInput) {
        return false;
    }

	ev = DwtUiEvent.getEvent(ev, this);
	DBG.println(AjxDebug.KEYBOARD, [ev.type, ev.keyCode, ev.charCode, ev.which].join(" / "));
	var kbMgr = DwtKeyboardMgr.__shell.getKeyboardMgr();
	var kev = DwtShell.keyEvent;
	kev.setFromDhtmlEvent(ev);

	if (kbMgr.__kbEventStatus != DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED) {
		return kbMgr.__processKeyEvent(ev, kev, false);
	}
};

/**
 * @private
 */
DwtKeyboardMgr.__keyDownHdlr = function(ev) {

	try {

	ev = DwtUiEvent.getEvent(ev, this);
	var kbMgr = DwtKeyboardMgr.__shell.getKeyboardMgr();
	ev.focusObj = null;
	if (kbMgr._evtMgr.notifyListeners(DwtEvent.ONKEYDOWN, ev) === false) {
		return false;
	}

	if (DwtKeyboardMgr.__shell._blockInput) {
        return false;
    }
	DBG.println(AjxDebug.KEYBOARD, [ev.type, ev.keyCode, ev.charCode, ev.which].join(" / "));

	var kev = DwtShell.keyEvent;
	kev.setFromDhtmlEvent(ev);
	var keyCode = DwtKeyEvent.getCharCode(ev);
	DBG.println(AjxDebug.KEYBOARD, "keydown: " + keyCode + " -------- " + ev.target);

	// Popdown any tooltip
	DwtKeyboardMgr.__shell.getToolTip().popdown();

    /********* FOCUS MANAGEMENT *********/

	/* The first thing we care about is the tab key since we want to manage
	 * focus based on the tab groups. 
	 * 
	 * If the tab hit happens in the currently
	 * focused obj, the go to the next/prev element in the tab group. 
	 * 
	 * If the tab happens in an element that is in the tab group hierarchy, but that 
	 * element is not the currently focus element in the tab hierarchy (e.g. the user
	 * clicked in it and we didnt detect it) then sync the tab group's current focus 
	 * element and handle the tab
	 * 
	 * If the tab happens in an object not under the tab group hierarchy, then set
	 * focus to the current focus object in the tab hierarchy i.e. grab back control
	 */
    var ctg = kbMgr.__currTabGroup,
        member;

	if (keyCode == DwtKeyEvent.KEY_TAB) {
	    if (ctg && !DwtKeyMapMgr.hasModifier(kev)) {
			DBG.println(AjxDebug.FOCUS, "Tab");
			// If the tab hit is in an element or if the current tab group has a focus member
			if (ctg.getFocusMember()) {
                member = kev.shiftKey ? ctg.getPrevFocusMember(true) : ctg.getNextFocusMember(true);
			}
            else {
			 	DBG.println(AjxDebug.FOCUS, "DwtKeyboardMgr.__keyDownHdlr: no current focus member, resetting to first in tab group");
			 	// If there is no current focus member, then reset
                member = ctg.resetFocusMember(true);
			}
	    }
        // If we did not handle the Tab, let the browser handle it
        return kbMgr.__processKeyEvent(ev, kev, !member, member ? DwtKeyboardMgr.__KEYSEQ_HANDLED : DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED);
	}
    else if (ctg && AjxEnv.isGecko && kev.target instanceof HTMLHtmlElement) {
	 	/* With FF we focus get set to the <html> element when tabbing in
	 	 * from the address or search fields. What we want to do is capture
	 	 * this here and reset the focus to the first element in the tabgroup
	 	 * 
	 	 * TODO Verify this trick is needed/works with IE/Safari
	 	 */
        member = ctg.resetFocusMember(true);
	}
	 
    // Allow key events to propagate when keyboard manager is disabled (to avoid taking over browser shortcuts). Bugzilla #45469.
    if (!kbMgr.isEnabled()) {
        return true;
    }


    /********* SHORTCUTS *********/

	// Filter out modifier keys. If we're in an input field, filter out legitimate input.
	// (A shortcut from an input field must use a modifier key.)
	if (DwtKeyMap.IS_MODIFIER[keyCode] || (kbMgr.__killKeySeqTimedActionId === -1 &&
		kev.target && DwtKeyMapMgr.isInputElement(kev.target) && !kev.target["data-hidden"] && !DwtKeyboardMgr.isPossibleInputShortcut(kev))) {

	 	return kbMgr.__processKeyEvent(ev, kev, true, DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED);
	}
	 
	/* Cancel any pending time action to kill the keysequence */
	if (kbMgr.__killKeySeqTimedActionId != -1) {
		AjxTimedAction.cancelAction(kbMgr.__killKeySeqTimedActionId);
		kbMgr.__killKeySeqTimedActionId = -1;
	}
		
 	var parts = [];
	if (kev.altKey) 	{ parts.push(DwtKeyMap.ALT); }
	if (kev.ctrlKey) 	{ parts.push(DwtKeyMap.CTRL); }
 	if (kev.metaKey) 	{ parts.push(DwtKeyMap.META); }
	if (kev.shiftKey) 	{ parts.push(DwtKeyMap.SHIFT); }
	parts.push(keyCode);
	kbMgr.__keySequence[kbMgr.__keySequence.length] = parts.join(DwtKeyMap.JOIN);

	DBG.println(AjxDebug.KEYBOARD, "KEYCODE: " + keyCode + " - KEY SEQ: " + kbMgr.__keySequence.join(""));
	
	var handled = DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED;

	// First see if the control that currently has focus can handle the key event
	var obj = ev.focusObj || kbMgr.__focusObj;
    var hasFocus = obj && obj.hasFocus && obj.hasFocus();
    DBG.println(AjxDebug.KEYBOARD, "DwtKeyboardMgr::__keyDownHdlr - focus object " + obj + " has focus: " + hasFocus);
	if (hasFocus && obj.handleKeyAction) {
		handled = kbMgr.__dispatchKeyEvent(obj, kev);
		while ((handled === DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED) && obj.parent) {
			obj = obj.parent;
            if (obj.getKeyMapName) {
			    handled = kbMgr.__dispatchKeyEvent(obj, kev);
            }
		}
	}

	// If the currently focused control didn't handle the event, hand it to the default key event handler
	if (handled === DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED && kbMgr.__currDefaultHandler) {
		handled = kbMgr.__dispatchKeyEvent(kbMgr.__currDefaultHandler, kev);
	}

	// see if we should let browser handle the event as well; note that we need to set the 'handled' var rather than
	// just the 'propagate' one below, since the keyboard mgr isn't built for both it and the browser to handle the event.
	if (kev.forcePropagate) {
		handled = DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED;
		kev.forcePropagate = false;
	}
	
	kbMgr.__kbEventStatus = handled;
	var propagate = (handled == DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED);

	if (handled != DwtKeyboardMgr.__KEYSEQ_PENDING) {
		kbMgr.clearKeySeq();
	}

	return kbMgr.__processKeyEvent(ev, kev, propagate);

	} catch (ex) {
		AjxException.reportScriptError(ex);
	}
};

/**
 * Handles event dispatching
 * 
 * @private
 */
DwtKeyboardMgr.prototype.__dispatchKeyEvent = function(hdlr, ev, forceActionCode) {

	if (hdlr && hdlr.handleKeyEvent) {
		var handled = hdlr.handleKeyEvent(ev);
		return handled ? DwtKeyboardMgr.__KEYSEQ_HANDLED : DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED;
	}

	var mapName = (hdlr && hdlr.getKeyMapName) ? hdlr.getKeyMapName() : null;
	if (!mapName) {
		return DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED;
	}

	DBG.println(AjxDebug.KEYBOARD, "DwtKeyboardMgr.__dispatchKeyEvent: handler " + hdlr.toString() + " handling " + this.__keySequence + " for map: " + mapName);
	var actionCode = this.__keyMapMgr.getActionCode(this.__keySequence, mapName, forceActionCode);
	if (actionCode === DwtKeyMapMgr.NOT_A_TERMINAL) {
		DBG.println(AjxDebug.KEYBOARD, "scheduling action to kill key sequence");
		/* setup a timed action to redispatch/kill the key sequence in the event
		 * the user does not press another key in the allotted time */
		this.__hdlr = hdlr;
		this.__mapName = mapName;
		this.__ev = ev;
		this.__killKeySeqTimedActionId = AjxTimedAction.scheduleAction(this.__killKeySeqTimedAction, this.__keyTimeout);
		return DwtKeyboardMgr.__KEYSEQ_PENDING;	
	}
    else if (actionCode != null) {
		/* It is possible that the component may not handle a valid action
		 * particulary actions defined in the default map */
		DBG.println(AjxDebug.KEYBOARD, "DwtKeyboardMgr.__dispatchKeyEvent: handling action: " + actionCode);
		if (!hdlr.handleKeyAction) {
			return DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED;
		}
		var result = hdlr.handleKeyAction(actionCode, ev);
		return result ? DwtKeyboardMgr.__KEYSEQ_HANDLED : DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED;
	}
    else {
		DBG.println(AjxDebug.KEYBOARD, "DwtKeyboardMgr.__dispatchKeyEvent: no action code for " + this.__keySequence);
		return DwtKeyboardMgr.__KEYSEQ_NOT_HANDLED;
	}
};

/**
 * This method will reattempt to handle the event in the case that the intermediate
 * node in the keymap may have an action code associated with it.
 * 
 * @private
 */
DwtKeyboardMgr.prototype.__killKeySequenceAction = function() {

	DBG.println(AjxDebug.KEYBOARD, "DwtKeyboardMgr.__killKeySequenceAction: " + this.__mapName);
	this.__dispatchKeyEvent(this.__hdlr, this.__ev, true);
	this.clearKeySeq();
};

/**
 * @private
 */
DwtKeyboardMgr.prototype.__tabGrpChangeListener = function(ev) {
	this.__doGrabFocus(ev.newFocusMember);
};

/**
 * @private
 */
DwtKeyboardMgr.prototype.__processKeyEvent = function(ev, kev, propagate, status) {

	if (status) {
		this.__kbEventStatus = status;
	}
	kev._stopPropagation = !propagate;
	kev._returnValue = propagate;
	kev.setToDhtmlEvent(ev);
	DBG.println(AjxDebug.KEYBOARD, "key event returning: " + propagate);
	return propagate;
};
}
if (AjxPackage.define("ajax.dwt.keyboard.DwtTabGroup")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */


/**
 * Creates an empty tab group.
 * @constructor
 * @class
 * A tab group is used to manage keyboard focus among a group of related visual 
 * elements. It is a tree structure consisting of elements and other tab groups.
 * <p>
 * The root tab group is the only one without a parent tab group, and is the one
 * that the application interacts with. Focus listeners register with the root
 * tab group. The root tab group tracks where focus is.
 * 
 * @param {string}	name					the name of this tab group
 *
 * @author Ross Dargahi
 */
DwtTabGroup = function(name) {

	this.__members = new AjxVector();
	this.__parent = null;
	this.__name = name;
	this.__currFocusMember = null;
	this.__evtMgr = new AjxEventMgr();

    DwtTabGroup.BY_NAME[name] = this;
};

DwtTabGroup.prototype.isDwtTabGroup = true;
DwtTabGroup.prototype.toString = function() { return "DwtTabGroup"; };



/** 
 * Exception string that is thrown when an operation is attempted
 * on a non-root tab group.
 */
DwtTabGroup.NOT_ROOT_TABGROUP = "NOT ROOT TAB GROUP";

DwtTabGroup.__changeEvt = new DwtTabGroupEvent();

// Allow static access to any tab group by its name
DwtTabGroup.getByName = function(name) {
    return DwtTabGroup.BY_NAME[name];
};
DwtTabGroup.BY_NAME = {};

/**
 * Gets the name of this tab group.
 * 
 * @return	{string}	the tab group name
 */
DwtTabGroup.prototype.getName = function() {
	return this.__name;
};

/**
 * Adds a focus change listener to the root tab group. The listener is called
 * when the focus member changes. Note that the focus member hasn't actually
 * been focused yet - only its status within the tab group has changed. It is
 * up to the listener to implement the appropriate focus action.
 * 
 * @param {AjxListener} listener	a listener
 * 
 * @throws DwtTabGroup.NOT_ROOT_TABGROUP
 */
DwtTabGroup.prototype.addFocusChangeListener = function(listener) {

	this.__checkRoot();		
	this.__evtMgr.addListener(DwtEvent.STATE_CHANGE, listener);
};

/**
 * Removes a focus change listener from the root tab group.
 * 
 * @param {AjxListener} listener	a listener
 * 
 * @throws DwtTabGroup.NOT_ROOT_TABGROUP
 */
DwtTabGroup.prototype.removeFocusChangeListener = function(listener) {

	this.__checkRoot();		
	this.__evtMgr.removeListener(DwtEvent.STATE_CHANGE, listener);
};

/**
 * Adds a member to the tab group.
 * 
 * @param {Array|DwtControl|DwtTabGroup|HTMLElement} member	the member(s) to be added
 * @param {number} [index] 		the index at which to add the member. If omitted, the member
 * 		will be added to the end of the tab group
 */
DwtTabGroup.prototype.addMember = function(member, index) {

    index = (index != null) ? index : this.__members.size();
    var members = AjxUtil.collapseList(AjxUtil.toArray(member));

	for (var i = 0, len = members.length; i < len; i++) {
        var member = members[i];
        this.__members.add(member, index + i);
        // If adding a tab group, register me as its parent
        if (member.isDwtTabGroup) {
            member.newParent(this);
        }
	}
};

/**
 * Resets all members of the tab group to the given arguments.
 * 
 * @param {Array|DwtControl|DwtTabGroup|HTMLElement} members	the member(s) for the tab group
 */
DwtTabGroup.prototype.setMembers = function(members) {
	this.removeAllMembers();
	this.addMember(members);
};

/**
 * Adds a member to the tab group, positioned after another member.
 * 
 * @param {DwtControl|DwtTabGroup|HTMLElement} member 		the member to be added
 * @param {DwtControl|DwtTabGroup|HTMLElement} afterMember 	the member after which to add <code>member</code>
 */
DwtTabGroup.prototype.addMemberAfter = function(newMember, afterMember) {

	this.addMember(newMember, this.__indexOfMember(afterMember) + 1);
};

/**
 * Adds a member to the tab group, positioned before another member.
 * 
 * @param {DwtControl|DwtTabGroup|HTMLElement} member 		the member to be added
 * @param {DwtControl|DwtTabGroup|HTMLElement} beforeMember 	the member before which to add <code>member</code>
 */
DwtTabGroup.prototype.addMemberBefore = function(newMember, beforeMember) {

	this.addMember(newMember, this.__indexOfMember(beforeMember));
};

/**
 * This method removes a member from the tab group. If the member being removed
 * is currently the focus member, then we will try to set focus to the
 * previous member. If that fails, we will try the next member.
 * 
 * @param {DwtControl|DwtTabGroup|HTMLElement} member 	the member to be removed
 * @param {boolean} [checkEnabled] 		if <code>true</code>, then make sure that if we have a newly focused member it is enabled
 * @param {boolean} [skipNotify] 		if <code>true</code>, notification is not fired. This flag typically set by Dwt tab management framework when it is calling into this method
 * @return {DwtControl|DwtTabGroup|HTMLElement}	the removed member or <code>null</code> if <code>oldMember</code> is not in the tab groups hierarchy
 */
DwtTabGroup.prototype.removeMember = function(member, checkEnabled, skipNotify) {

	return this.replaceMember(member, null, checkEnabled, skipNotify);
};

/**
 * Removes all members.
 * 
 */
DwtTabGroup.prototype.removeAllMembers = function() {

	this.__members.removeAll();
};

/**
 * This method replaces a member in the tab group with a new member. If the member being
 * replaced is currently the focus member, then we will try to set focus to the
 * previous member. If that fails, we will try the next member.
 * 
 * @param {DwtControl|DwtTabGroup|HTMLElement} oldMember 	the member to be replaced
 * @param {DwtControl|DwtTabGroup|HTMLElement} newMember 	the replacing member
 * 		If this parameter is <code>null</code>, then this method effectively removes <code>oldMember</code>
 * @param {boolean} [checkEnabled] 	if <code>true</code>, then make sure that if we have a newly focused
 * 		member it is enabled
 * @param {boolean} [skipNotify] if <code>true</code>, notification is not fired. This flag is
 * 		typically set by the tab management framework when it is calling into this method
 * @return {DwtControl|DwtTabGroup|HTMLElement}	replaced member or <code>null></code> if <code>oldMember</code> is not in the tab group
 */
DwtTabGroup.prototype.replaceMember = function(oldMember, newMember, checkEnabled, skipNotify, focusItem, noFocus) {

	var tg = this.__getTabGroupForMember(oldMember);
	if (!tg) {
		this.addMember(newMember);
		return null;
	}

	/* If we are removing the current focus member, then we need to adjust the focus
	 * member index. If the tab group is empty as a result of the removal
	 */
	var root = this.__getRootTabGroup();
	var newFocusMember;
	if (focusItem) {
		newFocusMember = focusItem;
	}
    else if (root.__currFocusMember === oldMember || (oldMember && oldMember.isDwtTabGroup && oldMember.contains(root.__currFocusMember))) {
		if (newMember) {
			newFocusMember = (newMember.isDwtTabGroup) ? newMember.getFirstMember() : newMember;
		}
        else {
			newFocusMember = this.__getPrevMember(oldMember, checkEnabled);
			if (!newFocusMember) {
				newFocusMember =  this.__getNextMember(oldMember, checkEnabled);
			}
		}
	}

	if (newFocusMember && !noFocus) {
		root.__currFocusMember = newFocusMember;
		this.__showFocusedItem(this.__currFocusMember, "replaceMember");
		if (!skipNotify) {
			this.__notifyListeners(newFocusMember);
		}
	}

	if (newMember && newMember.isDwtTabGroup) {
		newMember.newParent(this);
	}
		
	return newMember ? this.__members.replaceObject(oldMember, newMember) : this.__members.remove(oldMember);
};

/**
 * Returns true if this tab group contains <code>member</code>.
 * 
 * @param {DwtControl|DwtTabGroup|HTMLElement} member	the member for which to search
 * 
 * @return {boolean}	<code>true</code> if the tab group contains member
 */
DwtTabGroup.prototype.contains = function(member) {

	return !!this.__getTabGroupForMember(member);
};

/**
 * Sets a new parent for this tab group.
 * 
 * @param {DwtTabGroup} newParent 	the new parent. If the parent is <code>null</code>, then this tabGroup is the root tab group.
 */
DwtTabGroup.prototype.newParent = function(newParent) {

	this.__parent = newParent;
};

/**
 * Gets the first member of the tab group.
 * 
 * @param {boolean} [checkEnabled]		if <code>true</code>, then return first enabled member
 *
 * @return {DwtControl|HTMLElement}	the first member of the tab group
 */
DwtTabGroup.prototype.getFirstMember = function(checkEnabled) {

	return this.__getLeftMostMember(checkEnabled);
};

/**
 * Gets the last member of the tab group.
 * 
 * @param {boolean} [checkEnabled]		if <code>true</code>, then return last enabled member
 *
 * @return {DwtControl|HTMLElement}	the last member of the tab group
 */
DwtTabGroup.prototype.getLastMember = function(checkEnabled) {

	return this.__getRightMostMember(checkEnabled);
};
 
/**
 * Returns the current focus member.
 * 
 * @return {DwtControl|HTMLElement}	current focus member
 * 
 * @throws DwtTabGroup.NOT_ROOT_TABGROUP
 */
DwtTabGroup.prototype.getFocusMember = function(){

	this.__checkRoot();
	return this.__currFocusMember;
};

/**
 * Sets the current focus member. 
 * 
 * @param {DwtControl|HTMLElement} member 		the member to which to set focus
 * @param {boolean} [checkEnabled] 	if <code>true</code>, then make sure the member is enabled
 * @param {boolean} [skipNotify] if <code>true</code>, notification is not fired. This flag
 * 		typically set by Dwt tab management framework when it is calling into this method
 * 
 * @return {boolean}	<code>true</code> if member was part of the tab group hierarchy, else false
 *
 * @throws DwtTabGroup.NOT_ROOT_TABGROUP
 */
DwtTabGroup.prototype.setFocusMember = function(member, checkEnabled, skipNotify) {

    if (!member) {
        return false;
    }

    if (member.isDwtTabGroup) {
        DBG.println(AjxDebug.FOCUS, "DwtTabGroup SETFOCUSMEMBER to a DwtTabGroup: " + member + " / " + member.getName());
        member = member.getFocusMember() || member.getFirstMember();
    }
	this.__checkRoot();	
	if (!this.__checkEnabled(member, checkEnabled)) {
		return false;
	}

	if (this.contains(member)) {
		this.__currFocusMember = member;
		this.__showFocusedItem(this.__currFocusMember, "setFocusMember");
		if (!skipNotify) {
			this.__notifyListeners(this.__currFocusMember);
		}
		return true;	
	}

	return false;
};

/**
 * This method sets and returns the next focus member in this tab group. If there is no next
 * member, sets and returns the first member in the tab group.
 * 
 * @param {boolean} [checkEnabled] 	if <code>true</code>, get the next enabled member
 * @param {boolean} [skipNotify] if <code>true</code>, notification is not fired. This flag
 * 		typically set by {@link Dwt} tab management framework when it is calling into this method
 * 
 * @return {DwtControl|HTMLElement}	new focus member or <code>null</code> if there is no focus member or if the focus
 * 		member has not changed (i.e. only one member in the tabgroup)
 *
 * @throws DwtTabGroup.NOT_ROOT_TABGROUP
 */
DwtTabGroup.prototype.getNextFocusMember = function(checkEnabled, skipNotify) {

	this.__checkRoot();		
	return this.__setFocusMember(true, checkEnabled, skipNotify);
};

/**
 * This method sets and returns the previous focus member in this tab group. If there is no
 * previous member, sets and returns the last member in the tab group.
 * 
 * @param {boolean} [checkEnabled] 	if <code>true</code>, get the previously enabled member
 * @param {boolean} [skipNotify] if <code>true</code>, notification is not fired. This flag
 * 		typically set by Dwt tab management framework when it is calling into this method
 * 
 * @return {DwtControl|HTMLElement}	new focus member or <code>null</code> if there is no focus member or if the focus
 * 		member has not changed (i.e. only one member in the tabgroup)
 *
 * @throws DwtTabGroup.NOT_ROOT_TABGROUP
 */
DwtTabGroup.prototype.getPrevFocusMember = function(checkEnabled, skipNotify) {

	this.__checkRoot();		
	return this.__setFocusMember(false, checkEnabled, skipNotify);
};

/**
 * Resets the the focus member to the first element in the tab group.
 * 
 * @param {boolean} [checkEnabled] 	if <code>true</code>, then pick a enabled member to which to set focus
 * @param {boolean} [skipNotify] if <code>true</code>, notification is not fired. This flag
 * 		typically set by Dwt tab management framework when it is calling into this method
 * 
 * @return {DwtControl|HTMLElement}	the new focus member
 *
 * @throws DwtTabGroup.NOT_ROOT_TABGROUP
 */
DwtTabGroup.prototype.resetFocusMember = function(checkEnabled, skipNotify) {

	this.__checkRoot();
	var focusMember = this.__getLeftMostMember(checkEnabled);
	if ((focusMember != this.__currFocusMember) && !skipNotify) {
		this.__notifyListeners(this.__currFocusMember);
	}
	this.__showFocusedItem(this.__currFocusMember, "resetFocusMember");
    DBG.println(AjxDebug.FOCUS, "DwtTabGroup RESETFOCUSMEMBER: " + focusMember);
	this.__currFocusMember = focusMember;
	
	return this.__currFocusMember;
};

/**
 * Pretty-prints the contents of the tab group to the browser console or the
 * debug window.
 *
 * @param {number} [debugLevel]     if specified, dump to the debug window
 *                                  at the given level.
 */
DwtTabGroup.prototype.dump = function(debugLevel) {

	if (debugLevel) {
		if (!window.AjxDebug || !window.DBG) {
			return;
		}

		var logger = function(s) {
			var s = AjxStringUtil.convertToHtml(s);
			DBG.println(debugLevel, s);
		}

		DwtTabGroup.__dump(this, logger, 0);
	} else if (window.console && window.console.log) {
		var r = [];
		DwtTabGroup.__dump(this, r.push.bind(r), 0);
		console.log(r.join('\n'));
	}
};

/**
 * Gets the size of the group.
 * 
 * @return	{number}	the size
 */
DwtTabGroup.prototype.size = function() {

	return this.__members.size();
};

/**
 * Returns the previous member in the tag group.
 * 
 * @private
 */
DwtTabGroup.prototype.__getPrevMember = function(member, checkEnabled) {

	var a = this.__members.getArray();

	// Start working from the member to the immediate left, then keep going left
	for (var i = this.__lastIndexOfMember(member) - 1; i > -1; i--) {
		var prevMember = a[i];
		/* if sibling is not a tab group, then it is the previous child. If the
		 * sibling is a tab group, get its rightmost member.*/
		if (!prevMember.isDwtTabGroup) {
			if (this.__checkEnabled(prevMember, checkEnabled)) {
				return prevMember;
			}
		} else {
			prevMember = prevMember.__getRightMostMember(checkEnabled);
			if (this.__checkEnabled(prevMember, checkEnabled)) {
				return prevMember;
			}
		}
	}

	/* If we have fallen through to here it is because the tab group only has 
	 * one member. So we roll up to the parent, unless we are at the root in 
	 * which case we return null. */
	return this.__parent ? this.__parent.__getPrevMember(this, checkEnabled) : null;
};

/**
 * Returns true if the given member can accept focus, or if there is no need to check.
 * If we are checking, the member must be enabled and visible if it is a control, and
 * enabled otherwise. A member may also set the "noTab" flag to take itself out of the
 * tab hierarchy.
 * 
 * @private
 */
DwtTabGroup.prototype.__checkEnabled = function(member, checkEnabled) {

	if (!checkEnabled) {
		return true;
	}

	if (!member || member.noTab) {
		return false;
	}

	if (member.isDwtControl ? !member.getEnabled() : member.disabled) {
		return false;
	}

	if (member.isDwtControl) {
		member = member.getHtmlElement();
	}

	var loc = Dwt.getLocation(member);
	if (loc.x === null || loc.y === null || loc.x === Dwt.LOC_NOWHERE || loc.y === Dwt.LOC_NOWHERE) {
		return false;
	}

	var size = Dwt.getSize(member);
	if (!size || size.x === 0 || size.y === 0) {
		return false;
	}

	if (member.nodeName && member.nodeName.toLowerCase() === "body") {
		return true;
	}
	return (Dwt.getZIndex(member, true) > Dwt.Z_HIDDEN &&
	        Dwt.getVisible(member) && Dwt.getVisibility(member));
};

DwtTabGroup.prototype.__indexOfMember = function(member) {

    return this.__members.indexOf(member);
};

DwtTabGroup.prototype.__lastIndexOfMember = function(member) {

    return this.__members.lastIndexOf(member);
};

/**
 * Sets and returns the next member in the tag group.
 * 
 * @private
 */
DwtTabGroup.prototype.__getNextMember = function(member, checkEnabled) {

	var a = this.__members.getArray();
	var sz = this.__members.size();

	// Start working from the member rightwards
	for (var i = this.__indexOfMember(member) + 1; i < sz; i++) {
		var nextMember = a[i];
		/* if sibling is not a tab group, then it is the next child. If the
		 * sibling is a tab group, get its leftmost member.*/
		if (!nextMember.isDwtTabGroup) {
			if (this.__checkEnabled(nextMember, checkEnabled)) {
				return nextMember;
			}
		}
        else {
			nextMember = nextMember.__getLeftMostMember(checkEnabled);
			if (this.__checkEnabled(nextMember, checkEnabled)) {
				return nextMember;
			}
		}
	}

	/* If we have fallen through to here it is because the tab group only has 
	 * one member or we are at the end of the list. So we roll up to the parent, 
	 * unless we are at the root in which case we return null. */
	return this.__parent ? this.__parent.__getNextMember(this, checkEnabled) : null;
};

/**
 * Finds the rightmost member of the tab group. Will recurse down
 * into contained tab groups if necessary.
 * @private
 */
DwtTabGroup.prototype.__getRightMostMember = function(checkEnabled) {

	var a = this.__members.getArray();
	var member = null;
	
	/* Work backwards from the rightmost member. If the member is a tab group, then
	 * recurse into it. If member is not a tab group, return it as it is the 
	 * rightmost element. */
	for (var i = this.__members.size() - 1; i >= 0; i--) {
		member = a[i]
		if (!member.isDwtTabGroup) {
			if (this.__checkEnabled(member, checkEnabled)) {
                break;
            }
		}
        else {
			member = member.__getRightMostMember(checkEnabled);
			if (this.__checkEnabled(member, checkEnabled)) {
                break;
            }
		}
	}

	return this.__checkEnabled(member, checkEnabled) ? member : null;
};

/**
 *  Finds the leftmost member of the tab group. Will recurse down
 * into contained tab groups if necessary.
 * @private
 */
DwtTabGroup.prototype.__getLeftMostMember = function(checkEnabled) {

	var sz = this.__members.size();
	var a = this.__members.getArray();
	var member = null;

	/* Work forwards from the leftmost member. If the member is a tabgroup, then
	 * recurse into it. If member is not a tabgroup, return it as it is the 
	 * rightmost element */
	for (var i = 0; i < sz; i++) {
		member = a[i]
		if (!member.isDwtTabGroup) {
			if  (this.__checkEnabled(member, checkEnabled)) {
                break;
            }
		}
        else {
			member = member.__getLeftMostMember(checkEnabled);
			if (this.__checkEnabled(member, checkEnabled)) {
                break;
            }
		}
	}

	return this.__checkEnabled(member, checkEnabled) ? member : null;
};

/**
 * Notifies focus change listeners.
 * @private
 */
DwtTabGroup.prototype.__notifyListeners = function(newFocusMember) {

	// Only the root tab group will issue notifications
	var rootTg = this.__getRootTabGroup();
	if (rootTg.__evtMgr) {
		var evt = DwtTabGroup.__changeEvt;
		evt.reset();
		evt.tabGroup = this;
		evt.newFocusMember = newFocusMember;
		rootTg.__evtMgr.notifyListeners(DwtEvent.STATE_CHANGE, evt);
	}
};

/**
 * @private
 */
DwtTabGroup.prototype.__getRootTabGroup = function() {

	var root = this;
	while (root.__parent) {
		root = root.__parent;
	}
	
	return root;
}

DwtTabGroup.DUMP_INDENT = '|\t';

/**
 * @private
 */
DwtTabGroup.__dump = function(tg, logger, level) {

	var myIndent = AjxStringUtil.repeat(DwtTabGroup.DUMP_INDENT, level);

	logger(myIndent + "TABGROUP: " + tg.__name);

	myIndent += DwtTabGroup.DUMP_INDENT;

	var sz = tg.__members.size();
	var a = tg.__members.getArray();
	for (var i = 0; i < sz; i++) {
        var m = a[i];
		if (m.isDwtTabGroup) {
			DwtTabGroup.__dump(m, logger, level + 1);
		}
        else {
			var desc = m.nodeName ? [ m.nodeName, m.id, m.className ].join(' ') : [ String(m), m._htmlElId ].join(' ');
			if (m.noTab) {
				desc += ' - no tab!';
			}
			logger(myIndent + desc);
		}
	}
};

/**
 * Sets the next or previous focus member.
 * @private
 */
DwtTabGroup.prototype.__setFocusMember = function(next, checkEnabled, skipNotify) {

	// If there is currently no focus member, then reset to the first member and return
	if (!this.__currFocusMember) {
		return this.resetFocusMember(checkEnabled, skipNotify);
	}
	
	var tabGroup = this.__getTabGroupForMember(this.__currFocusMember);
	if (!tabGroup) {
		DBG.println(AjxDebug.DBG1, "tab group not found for focus member: " + this.__currFocusMember);
		return null;
	}
	var m = next ? tabGroup.__getNextMember(this.__currFocusMember, checkEnabled)
				 : tabGroup.__getPrevMember(this.__currFocusMember, checkEnabled);

	if (!m) {
        // wrap around
		m = next ? this.__getLeftMostMember(checkEnabled)
				 : this.__getRightMostMember(checkEnabled);

		// Test for the case where there is only one member in the tabgroup
		if (m == this.__currFocusMember) {
			return this.__currFocusMember;
		}
	}

	this.__currFocusMember = m;
	
	this.__showFocusedItem(this.__currFocusMember, "__setFocusMember");
	if (!skipNotify) {
		this.__notifyListeners(this.__currFocusMember);
	}
	
	return this.__currFocusMember;
};

/**
 * Returns the tab group from within this tab group's hierarchy that contains the given member. Traverses the tree top-down.
 *
 * @private
 */
DwtTabGroup.prototype.__getTabGroupForMember = function(member) {

	if (!member) {
        return null;
    }

    var a = this.__members.getArray(),
        ln = a.length, i, m;

	for (i = 0; i < ln; i++) {
		m = a[i];
		if (m === member) {
			return this;
		}
        else if (m.isDwtTabGroup && (m = m.__getTabGroupForMember(member))) {
			return m;
		}
	}
	return null;
};

/**
 * Throws an exception if this is not the root tab group.
 * 
 * @private
 */
DwtTabGroup.prototype.__checkRoot = function() {

	if (this.__parent) {
        DBG.println(AjxDebug.DBG1, "DwtTabGroup NOT_ROOT_TABGROUP: " + this.getName());
//		throw DwtTabGroup.NOT_ROOT_TABGROUP;
	}
};

// Prints out a debug line describing the currently focused member
DwtTabGroup.prototype.__showFocusedItem = function(item, caller) {

	if (item && window.AjxDebug && window.DBG) {
		var callerText = caller ? "DwtTabGroup." + caller + ": " : "",
			idText = " [" + (item.isDwtControl ? item._htmlElId : item.id) + "] ",
            itemText = (item.nodeName || item) + " " + idText,
			otherText = (item.getTitle && item.getTitle()) || (item.getText && item.getText()) || "",
			fullText = itemText + otherText;

		DBG.println(AjxDebug.FOCUS, callerText + "current focus member is now " + itemText);
		DBG.println(AjxDebug.FOCUS1, "Focus: " + fullText);
	}
};
}
if (AjxPackage.define("ajax.dwt.core.DwtId")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @class
 * This class is responsible for providing unique, predictable IDs for HTML elements.
 * That way, code outside the client can locate particular elements.
 * <p>
 * Not every element that has an associated JS object will have a known ID. Those are
 * allocated only for elements it would be useful to locate: major components of the UI,
 * toolbars, buttons, views, menus, some menu items, and some selects.
 * <p>
 * There is a simple naming scheme for the IDs themselves. Each ID starts with a "z" followed
 * by one to a few letters that indicate the type of object (widget) represented by the element.
 * 
 * @author Conrad Damon
 */
 
DwtId = function() {}

// separator for parts used in constructing IDs - need to pick one that
// doesn't show up in any of the parts
DwtId.SEP = "__";

// widget types (used to prefix IDs)
/**
 * Defines the widget "list view".
 */
DwtId.WIDGET_LIST_VIEW		= "zl";			// list view
/**
 * Defines the widget "list view header".
 */
DwtId.WIDGET_HDR			= "zlh";		// list view header
/**
 * Defines the widget "list view header table".
 */
DwtId.WIDGET_HDR_TABLE		= "zlht";		// list view header table
/**
 * Defines the widget "list view header icon image".
 */
DwtId.WIDGET_HDR_ICON		= "zlhi";		// list view header image
/**
 * Defines the widget "list view header text".
 */
DwtId.WIDGET_HDR_LABEL		= "zlhl";		// list view header text
/**
 * Defines the widget "list view header dropdown arrow".
 */
DwtId.WIDGET_HDR_ARROW		= "zlha";		// list view header dropdown arrow
/**
 * Defines the widget "sash between list view headers".
 */
DwtId.WIDGET_HDR_SASH		= "zlhs";		// sash between list view headers
/**
 * Defines the widget "list view item".
 */
DwtId.WIDGET_ITEM			= "zli";		// list view item
/**
 * Defines the widget "list view item row".
 */
DwtId.WIDGET_ITEM_ROW		= "zlir";		// list view item row
/**
 * Defines the widget "list view item cell".
 */
DwtId.WIDGET_ITEM_CELL		= "zlic";		// list view item cell
/**
 * Defines the widget "list view item field".
 */
DwtId.WIDGET_ITEM_FIELD		= "zlif";		// list view item field

// list view modifiers
/**
 * Defines the list view "headers" modifier.
 */
DwtId.LIST_VIEW_HEADERS	= "headers";
/**
 * Defines the list view "rows" modifier.
 */
DwtId.LIST_VIEW_ROWS	= "rows";

DwtId.IFRAME = "iframe";

DwtId.DND_PLUS_ID		= "z__roundPlus";

/**
 * Joins the given arguments into an ID, excluding empty ones.
 * 
 * @private
 */
DwtId.makeId =
function() {
	var list = [];
	for (var i = 0; i < arguments.length; i++) {
		var arg = arguments[i];
		if (arg != null && arg != "") {
			list.push(arg);
		}
	}
	return list.join(DwtId.SEP);
};
DwtId._makeId = DwtId.makeId;	// back-compatibility

/**
 * Gets an ID for a list view.
 * 
 * @param {constant}	context		the owning view identifier
 * @param {DwtId.LIST_VIEW_HEADERS|DwtId.LIST_VIEW_ROWS}	modifier	indicates element within list view (see <code>DwtId.LIST_VIEW*</code> constants)	
 * @return	{string}	the ID
 */
DwtId.getListViewId =
function(context, modifier) {
	return DwtId.makeId(DwtId.WIDGET_LIST_VIEW, context, modifier);
};

/**
 * Gets an ID for an element within a list view header.
 * 
 * @param {constant}	type		the type of hdr element (see <code>DwtId.WIDGET_HDR*</code> constants)
 * @param {constant}	context	the the ID of owning view
 * @param {constant}	hdr		the header ID
 * @return	{string}	the ID
 */
DwtId.getListViewHdrId =
function(type, context, hdr) {
	return DwtId.makeId(type, context, hdr);
};

/**
 * Gets an ID for an element associated with the display of an item in a list view.
 * 
 * @param {constant}	type		the type of item element (see <code>DwtId.WIDGET_ITEM*</code> constants)
 * @param {constant}	context		the ID of owning view
 * @param {string}	itemId	the item ID (typically numeric)
 * @param {constant}	field		the field identifier (for example, "su" for subject)
 * @return	{string}	the ID
 */
DwtId.getListViewItemId =
function(type, context, itemId, field) {
	return DwtId.makeId(type, context, itemId, field);
};

/**
 * Gets an ID for an IFRAME.
 * 
 * @param {constant}	context	the ID of owning {@link DwtIframe}
 * @return	{string}	the ID
 */
DwtId.getIframeId =
function(context) {
	return DwtId.makeId(context, DwtId.IFRAME);
};
}
if (AjxPackage.define("ajax.dwt.dnd.DwtDragEvent")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @constructor
 * @class
 * DwtDragEvent is generated by the Drag and Drop framework when a drag operation is
 * in process. The drag event is dispatched to the registered {@link DwtDragSource} instance.
 * 
 * @author Ross Dargahi
 * 
 * @see DwtDragSource
 */
DwtDragEvent = function() {
	/**
	 * Type of drag operation. One of:
	 * <ul>
	 * <li>{@link DwtDragEvent.DRAG_START}</li>
	 * <li>{@link DwtDragEvent.SET_DATA}</li>
	 * <li>{@link DwtDragEvent.DRAG_END}</li>
	 * </ul>
	 */
	this.operation = null;
	
	/**
	 * Drag source control
	 * @type DwtControl
	 */
	this.srcControl = null;
	
	/**
	 * Action being performed. One of:
	 * <ul>
	 * <li>{@link Dwt.DND_DROP_NONE}</li>
	 * <li>{@link Dwt.DND_DROP_COPY}</li>
	 * <li>{@link Dwt.DND_DROP_MOVE}</li>
	 * </ul>
	 */
	this.action = null;
	
	/**
	 * Whether the DnD framework should perform the operation. The application is
	 * responsible for setting this value based on whatever business logic it is
	 * implementing
	 * @type boolean
	 */
	this.doIt = false;
	
	/**
	 * Drag source data. This is the application data associated with the item being dragged.
	 */
	this.srcData = null;
};

/**
 * Drag initialization.
 */
DwtDragEvent.DRAG_INIT = "INIT";

/**
 * Drag is starting.
 */
DwtDragEvent.DRAG_START = "START";

/**
 * Set the <code>srcData</code> field of the event.
 */
DwtDragEvent.SET_DATA = "SET_DATA";

/**
 * Drag movement has occurred.
 */
DwtDragEvent.DRAG_MOVE = "MOVE";

/**
 * Drag has ended.
 */
DwtDragEvent.DRAG_END = "END";

/**
 * Drag canceled (i.e. dropped on invalid target).
 */
DwtDragEvent.DRAG_CANCEL = "CANCEL";
}
if (AjxPackage.define("ajax.dwt.dnd.DwtDragSource")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @constructor
 * @class
 * A drag source is registered with a control to indicate that the control is 
 * draggable. The drag source is the mechanism by which the DnD framework provides 
 * the binding between the UI components and the application.
 * <p>
 * Application developers instantiate {@link DwtDragSource} and register it with the control
 * which is to be draggable (via {@link DwtControl.setDragSource}). The
 * application should then register a listener with the {@link DwtDragSource}. This way
 * when drag events occur the application will be notified and may act on them 
 * accordingly
 * </p>
 * 
 * @author Ross Dargahi
 * 
 * @param {number} supportedOps 	the supported operations. This is an arithmetic OR'ing of
 * 		the operations supported by the drag source. Supported values are:
 * 		<ul>
 * 			<li>{@link Dwt.DND_DROP_NONE}</li>
 * 			<li>{@link Dwt.DND_DROP_COPY}</li>
 * 			<li>{@link Dwt.DND_DROP_MOVE}</li>
 * 		</ul> 
 * 
 * @see DwtDragEvent
 * @see DwtControl
 * @see DwtControl#setDragSource
 */
DwtDragSource = function(supportedOps) {
	this.__supportedOps = supportedOps
	this.__evtMgr = new AjxEventMgr();
};

/** @private */
DwtDragSource.__DRAG_LISTENER = "DwtDragSource.__DRAG_LISTENER";

/** @private */
DwtDragSource.__dragEvent = new DwtDragEvent();

/**
 * Returns a string representation of this object.
 * 
 * @return {string}	a string representation of this object
 */
DwtDragSource.prototype.toString = 
function() {
	return "DwtDragSource";
};


/**
 * Registers a listener for <i>DwtDragEvent</i> events.
 *
 * @param {AjxListener} dragSourceListener Listener to be registered 
 * 
 * @see DwtDragEvent
 * @see AjxListener
 * @see #removeDragListener
 */
DwtDragSource.prototype.addDragListener =
function(dragSourceListener) {
	this.__evtMgr.addListener(DwtDragSource.__DRAG_LISTENER, dragSourceListener);
};

/**
 * Removes a registered event listener.
 * 
 * @param {AjxListener} dragSourceListener Listener to be removed
 * 
 * @see AjxListener
 * @see #addDragListener
 */
DwtDragSource.prototype.removeDragListener =
function(dragSourceListener) {
	this.__evtMgr.removeListener(DwtDragSource.__DRAG_LISTENER, dragSourceListener);
};

// The following methods are called by DwtControl during the drag lifecycle 

/** @private */
DwtDragSource.prototype._beginDrag =
function(operation, srcControl) {
	if (!(this.__supportedOps & operation))
		return Dwt.DND_DROP_NONE;
		
	DwtDragSource.__dragEvent.operation = operation;
	DwtDragSource.__dragEvent.srcControl = srcControl;
	DwtDragSource.__dragEvent.action = DwtDragEvent.DRAG_START;
	DwtDragSource.__dragEvent.srcData = null;
	DwtDragSource.__dragEvent.doit = true;
	this.__evtMgr.notifyListeners(DwtDragSource.__DRAG_LISTENER, DwtDragSource.__dragEvent);
	return DwtDragSource.__dragEvent.operation;
};

/** @private */
DwtDragSource.prototype._getData =
function() {
	DwtDragSource.__dragEvent.action = DwtDragEvent.SET_DATA;
	this.__evtMgr.notifyListeners(DwtDragSource.__DRAG_LISTENER, DwtDragSource.__dragEvent);
	return DwtDragSource.__dragEvent.srcData;
};

/** @private */
DwtDragSource.prototype._endDrag =
function() {
	DwtDragSource.__dragEvent.action = DwtDragEvent.DRAG_END;
	DwtDragSource.__dragEvent.doit = false;
	this.__evtMgr.notifyListeners(DwtDragSource.__DRAG_LISTENER, DwtDragSource.__dragEvent);
	return DwtDragSource.__dragEvent.doit;
};

/** @private */
DwtDragSource.prototype._cancelDrag =
function() {
	DwtDragSource.__dragEvent.action = DwtDragEvent.DRAG_CANCEL;
	DwtDragSource.__dragEvent.doit = false;
	this.__evtMgr.notifyListeners(DwtDragSource.__DRAG_LISTENER, DwtDragSource.__dragEvent);
	return DwtDragSource.__dragEvent.doit;
};
}
if (AjxPackage.define("ajax.dwt.dnd.DwtDropEvent")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @constructor
 * @class
 * DwtDropEvent is generated by the Drag and Drop framework when a drag n drop operation is
 * in process. The drop event is dispatched to the registered {@link DwtDropTarget} instance.
 * 
 * @author Ross Dargahi
 * 
 * @see DwtDropTarget
 */
DwtDropEvent = function() {
	/**
	 * Type of drag operation. One of:
	 * <ul>
	 * <li>{@link DwtDragEvent.DRAG_START}</li>
	 * <li>{@link DwtDragEvent.SET_DATA}</li>
	 * <li>{@link DwtDragEvent.DRAG_END}</li>
	 * </ul>
	 */
	this.operation = null;
	
	/**
	 * Drop target control.
	 * @type DwtControl
	 * */	
	this.targetControl = null;
	
	/**
	 * Action being performed. One of:
	 * <ul>
	 * <li>{@link Dwt.DND_DROP_NONE}</li>
	 * <li>{@link Dwt.DND_DROP_COPY}</li>
	 * <li>{@link Dwt.DND_DROP_MOVE}</li>
	 * </ul>
	 */
	this.action = null;
	
	/**
	 * Drag source data. This is the application data associated with the item being dragged.
	 */
	this.srcData = null;

	/**
	 * Whether the DnD framework should perform the operation. The application is
	 * responsible for setting this value based on whatever business logic it is
	 * implementing.
	 * @type boolean
	 */
	this.doIt = false;
}

/**
 * A draggable object has entered the drop target.
 */
DwtDropEvent.DRAG_ENTER = 1;

/**
 * A draggable object has left the drop target.
 */
DwtDropEvent.DRAG_LEAVE = 2;

/**
 * Drag operation has changed e.g. from move to copy.
 */
DwtDropEvent.DRAG_OP_CHANGED = 3;

/**
 * A draggable object has been dropped on the drag target.
 */
DwtDropEvent.DRAG_DROP = 4;
}
if (AjxPackage.define("ajax.dwt.dnd.DwtDropTarget")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @constructor
 * @class
 * A drop target is registered with a control to indicate that the control is 
 * a drop target. The drop target is the mechanism by which the DnD framework provides 
 * the binding between the UI components and the application.
 * <p>
 * Application developers instantiate {@link DwtDropTarget} and register it with the control
 * which is to be a drop target (via {@link DwtControl.setDropTarget}). The
 * application should then register a listener with the {@link DwtDropTarget}. This way
 * when drop events occur the application will be notified and may act on them 
 * accordingly
 * </p>
 * 
 * @author Ross Dargahi
 * 
 * @param {array} transferType	a list of supported object types that may be dropped onto
 * 		this drop target. Typically the items represent classes (i.e. functions) whose 
 * 		instances may be dropped on this drop target e.g. 
 * 		<code>new DwtDropTarget(MailItem, AppointmentItme)</code>
 * 
 * @see DwtDropEvent
 * @see DwtControl
 * @see DwtControl#setDropTarget
 */
DwtDropTarget = function(types) {
	/** @private */
	this._evtMgr = new AjxEventMgr();

	/** @private */
	this.__hasMultiple = false;
	
	this._types = {};
	if (typeof types == "string") {
		types = [types];
	}
	if (types && types.length) {
		for (var i = 0; i < types.length; i++) {
			this.addTransferType(types[i]);
		}
	}
}

/** @private */
DwtDropTarget.__DROP_LISTENER = "DwtDropTarget.__DROP_LISTENER";

/** @private */
DwtDropTarget.__dropEvent = new DwtDropEvent();

/**
 * Returns a string representation of this object.
 * 
 * @return {string}	a string representation of this object
 */
DwtDropTarget.prototype.toString = 
function() {
	return "DwtDropTarget";
}

/**
 * Registers a listener for {@link DwtDragEvent} events.
 *
 * @param {AjxListener} dropTargetListener Listener to be registered 
 * 
 * @see DwtDropEvent
 * @see AjxListener
 * @see #removeDropListener
 */
DwtDropTarget.prototype.addDropListener =
function(dropTargetListener) {
	this._evtMgr.addListener(DwtDropTarget.__DROP_LISTENER, dropTargetListener);
}

/**
 * Removes a registered event listener.
 * 
 * @param {AjxListener} dropTargetListener Listener to be removed
 * 
 * @see AjxListener
 * @see #addDropListener
 */
DwtDropTarget.prototype.removeDropListener =
function(dropTargetListener) {
	this._evtMgr.removeListener(DwtDropTarget.__DROP_LISTENER, dropTargetListener);
}

/**
 *  Check to see if the types in <code>items</code> can be dropped on this drop target
 *
 * @param {object|array} items an array of objects or single object whose types are
 * 		to be checked against the set of transfer types supported by this drop target
 * 
 * @return true if all of the objects in <code>items</code> may legally be dropped on 
 * 		this drop target
 * @type boolean
 */
DwtDropTarget.prototype.isValidTarget =
function(items) {
	if (items instanceof Array) {
		var len = items.length;
		for (var i = 0; i < len; i++) {
			if (!this.__checkTarget(items[i])) {
				return false;
			}
		}
		return true;
	} else {
		return this.__checkTarget(items);
	}
}

/**
 * Calling this method indicates that the UI component backing this drop target has multiple 
 * sub-components
 */
DwtDropTarget.prototype.markAsMultiple = 
function() {
	this.__hasMultiple = true;
};

/**
 * Checks if the UI component backing this drop target has multiple sub-components.
 * 
 * @return	{boolean}		<code>true</code> if the UI component has multiple sub-components
 */
DwtDropTarget.prototype.hasMultipleTargets = 
function () {
	return this.__hasMultiple;
};

/**
 * Gets the transfer types.
 * 
 * @return {array}	the list of transfer types supported by this drop target
 * 
 * @see #setTransferTypes
 */
DwtDropTarget.prototype.getTransferTypes =
function() {
	return this._types;
}

/**
 * Declares a type of object as valid for being dropped onto this target. The type is provided
 * as a string, since the corresponding class may not yet be defined. The type is eval'ed before
 * it is used for any validation, since the check is done with <code>instanceof</code>.
 * 
 * @param {string}	type		the name of class
 */
DwtDropTarget.prototype.addTransferType =
function(type) {
	this._types[type] = null;
};

// The following methods are called by DwtControl during the Drag lifecycle 

/** @private */
DwtDropTarget.prototype._dragEnter =
function(operation, targetControl, srcData, ev, dndProxy) {
	DwtDropTarget.__dropEvent.operation = operation;
	DwtDropTarget.__dropEvent.targetControl = targetControl;
	DwtDropTarget.__dropEvent.action = DwtDropEvent.DRAG_ENTER;
	DwtDropTarget.__dropEvent.srcData = srcData;
	DwtDropTarget.__dropEvent.uiEvent = ev;
	DwtDropTarget.__dropEvent.doIt = true;
	DwtDropTarget.__dropEvent.dndProxy = dndProxy;
	this._evtMgr.notifyListeners(DwtDropTarget.__DROP_LISTENER, DwtDropTarget.__dropEvent);
	return DwtDropTarget.__dropEvent.doIt;
}

/** @private */
DwtDropTarget.prototype._dragLeave =
function() {
	DwtDropTarget.__dropEvent.action = DwtDropEvent.DRAG_LEAVE;
	this._evtMgr.notifyListeners(DwtDropTarget.__DROP_LISTENER, DwtDropTarget.__dropEvent);
}

/** @private */
DwtDropTarget.prototype._dragOpChanged =
function(newOperation) {
	DwtDropTarget.__dropEvent.operation = newOperation;
	DwtDropTarget.__dropEvent.action = DwtDropEvent.DRAG_OP_CHANGED;
	this._evtMgr.notifyListeners(DwtDropTarget.__DROP_LISTENER, DwtDropTarget.__dropEvent);
	return DwtDropTarget.__dropEvent.doIt;
};

/** @private */
DwtDropTarget.prototype._drop =
function(srcData, ev) {
	DwtDropTarget.__dropEvent.action = DwtDropEvent.DRAG_DROP;
	DwtDropTarget.__dropEvent.srcData = srcData;
	DwtDropTarget.__dropEvent.uiEvent = ev;
	this._evtMgr.notifyListeners(DwtDropTarget.__DROP_LISTENER, DwtDropTarget.__dropEvent);
	return DwtDropTarget.__dropEvent.doIt;
};


// Private methods

/**@private*/
DwtDropTarget.prototype.__checkTarget =
function(item) {
	if (this._types) {
		for (var i in this._types) {
			var ctor;
			if (this._types[i]) {
				ctor = this._types[i];
			} else {
				ctor = this._types[i] = eval(i);
			}
			if (ctor && (typeof ctor == "function") && (item instanceof ctor)) {
				return true;
			}
		}
		return false;
	}
};
}
if (AjxPackage.define("ajax.dwt.dnd.DwtDragBox")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @constructor
 * @class
 * A drag box is registered with a control to indicate that the control supports the
 * presence of an elastic box created by clicking and dragging, and typically used to
 * select visual objects within a space.
 * <p>
 * Application developers instantiate {@link DwtDragBox} and register it with the control
 * which is to be draggable (via {@link DwtControl.setDragBox}). The
 * application should then register a listener with the {@link DwtDragBox}. This way
 * when drag events occur the application will be notified and may act on them 
 * accordingly.
 * </p>
 * 
 * @author Conrad Damon
 * 
 * @see DwtDragEvent
 * @see DwtControl
 * @see DwtControl#setDragBox
 */
DwtDragBox = function() {
	this.__evtMgr = new AjxEventMgr();
};

/** @private */
DwtDragBox.__DRAG_LISTENER = "DwtDragBox.__DRAG_LISTENER";

/** @private */
DwtDragBox.__dragEvent = new DwtDragEvent();

/**
 * Returns a string representation of this object.
 * 
 * @return {string}	a string representation of this object
 */
DwtDragBox.prototype.toString = 
function() {
	return "DwtDragBox";
};


/**
 * Registers a listener for <i>DwtDragEvent</i> events.
 *
 * @param {AjxListener} dragBoxListener Listener to be registered
 * 
 * @see DwtDragEvent
 * @see AjxListener
 * @see #removeDragListener
 */
DwtDragBox.prototype.addDragListener =
function(dragBoxListener) {
	this.__evtMgr.addListener(DwtDragBox.__DRAG_LISTENER, dragBoxListener);
};

/**
 * Removes a registered event listener.
 * 
 * @param {AjxListener} dragBoxListener Listener to be removed
 * 
 * @see AjxListener
 * @see #addDragListener
 */
DwtDragBox.prototype.removeDragListener =
function(dragBoxListener) {
	this.__evtMgr.removeListener(DwtDragBox.__DRAG_LISTENER, dragBoxListener);
};

// The following methods are called by DwtControl during the drag lifecycle 

DwtDragBox.prototype._setStart =
function(mouseEv, srcControl) {

	this._startX = mouseEv.docX;
	this._startY = mouseEv.docY;
	this._dragObj = DwtDragBox.__dragEvent.srcControl = srcControl;
	DwtDragBox.__dragEvent.action = DwtDragEvent.DRAG_INIT;
	DwtDragBox.__dragEvent.target = mouseEv.target;
	return (this.__evtMgr.notifyListeners(DwtDragBox.__DRAG_LISTENER, DwtDragBox.__dragEvent) !== false);
};

DwtDragBox.prototype._beginDrag =
function(srcControl) {

	srcControl._dragging = DwtControl._DRAGGING;
	DwtDragBox.__dragEvent.srcControl = srcControl;
	DwtDragBox.__dragEvent.action = DwtDragEvent.DRAG_START;
	this.__evtMgr.notifyListeners(DwtDragBox.__DRAG_LISTENER, DwtDragBox.__dragEvent);
};

DwtDragBox.prototype._dragMove =
function(mouseEv, srcControl) {

	var deltaX = mouseEv.docX - this._startX;
	var deltaY = mouseEv.docY - this._startY;
	var locX = (deltaX > 0) ? this._startX : mouseEv.docX;
	var locY = (deltaY > 0) ? this._startY : mouseEv.docY;

	var box = srcControl.getDragSelectionBox();
	Dwt.setLocation(box, locX, locY);
	Dwt.setSize(box, Math.abs(deltaX), Math.abs(deltaY));

	DwtDragBox.__dragEvent.srcControl = srcControl;
	DwtDragBox.__dragEvent.action = DwtDragEvent.DRAG_MOVE;
	this.__evtMgr.notifyListeners(DwtDragBox.__DRAG_LISTENER, DwtDragBox.__dragEvent);
};

DwtDragBox.prototype._endDrag =
function(srcControl) {

	srcControl._dragging = DwtControl._NO_DRAG;
	DwtDragBox.__dragEvent.action = DwtDragEvent.DRAG_END;
	if (!this.__evtMgr.notifyListeners(DwtDragBox.__DRAG_LISTENER, DwtDragBox.__dragEvent)) {
		srcControl.destroyDragSelectionBox();
	}
	this._dragObj = null;
};

/*
 *  return starting X position
 *  @return {int} starting X position
 */
DwtDragBox.prototype.getStartX =
function() {
    return this._startX;
};

/*  return starting Y position
 *  @return {int} starting Y position
 */
DwtDragBox.prototype.getStartY =
function() {
    return this._startY;
};
}
if (AjxPackage.define("ajax.dwt.events.DwtDisposeEvent")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * 
 * @private
 */
DwtDisposeEvent = function(init) {
	if (arguments.length == 0) return;
	DwtEvent.call(this, true);
}

DwtDisposeEvent.prototype = new DwtEvent;
DwtDisposeEvent.prototype.constructor = DwtDisposeEvent;

DwtDisposeEvent.prototype.toString = 
function() {
	return "DwtDisposeEvent";
}
}

if (AjxPackage.define("ajax.dwt.widgets.DwtTreeItem")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */


/**
 * Creates a Tree Item.
 * @constructor
 * @class
 * This class implements a tree item widget.
 *
 * @author Ross Dargahi
 * 
 * @param {hash}	params				a hash of parameters
 * @param {DwtComposite}      params.parent	the parent widget
 * @param {number}      params.index 			the index at which to add this control among parent's children
 * @param {string}      params.text 					the label text for the tree item
 * @param {string}      params.imageInfo			the icon for the left end of the tree item
 * @param {string}      params.extraInfo				the icon for the right end of the tree item
 * @param {string}      params.expandNodeImage		the icon to use for expanding tree item (instead of default)
 * @param {string}      params.collapseNodeImage     the icon to use for collapsing tree item (instead of default)
 * @param {string}      params.className				the CSS class
 * @param {constant}      params.posStyle				the positioning style (see {@link DwtControl})
 * @param {boolean}      params.deferred				if <code>true</code>, postpone initialization until needed.
 * @param {boolean}      params.selectable			if <code>true</code>, this item is selectable
 * @param {boolean}      params.forceNotifySelection	force notify selection even if checked style
 * @param {boolean}      params.forceNotifyAction		force notify action even if checked style
 * @param {boolean}      params.singleClickAction		if <code>true</code>, an action is performed in single click
 * @param {AjxCallback}      params.dndScrollCallback	the callback triggered when scrolling of a drop area for an object being dragged
 * @param {string}      params.dndScrollId			the id
 * @param {boolean}    params.arrowDisabled
 * @param {boolean}     params.dynamicWidth		if <code>true</code>, the table should be width auto instead of the default fixed
 *
 * @extends		DwtComposite		
 */
DwtTreeItem = function(params) {

    if (arguments.length == 0) { return; }    

    params = Dwt.getParams(arguments, DwtTreeItem.PARAMS);
	var parent = params.parent;
	if (parent instanceof DwtTree) {
		this._tree = parent;
	} else if (parent instanceof DwtTreeItem) {
		this._tree = parent._tree;
	} else {
		throw new DwtException("DwtTreeItem parent must be a DwtTree or DwtTreeItem", DwtException.INVALIDPARENT, "DwtTreeItem");
	}

	this._origClassName = params.className || "DwtTreeItem";
	this._textClassName = [this._origClassName, "Text"].join("-");
	this._selectedClassName = this._origClassName + ' ' + [this._origClassName, DwtCssStyle.SELECTED].join("-");
	this._selectedFocusedClassName = this._selectedClassName + ' ' + [this._origClassName, DwtCssStyle.SELECTED, DwtCssStyle.FOCUSED].join("-");
	this._actionedClassName = this._origClassName + ' ' + [this._origClassName, DwtCssStyle.ACTIONED].join("-");
	this._dragOverClassName = this._origClassName + ' ' + [this._origClassName, DwtCssStyle.DRAG_OVER].join("-");
    this._treeItemTextClass = "DwtTreeItem-Text";
    this._treeItemExtraImgClass = "DwtTreeItem-ExtraImg";

	this._dynamicWidth = params.dynamicWidth;

	params.deferred = (params.deferred !== false);
	params.className = 'DwtTreeItem-Control';
	DwtComposite.call(this, params);

	this._imageInfoParam = params.imageInfo;
	this._imageAltInfo = params.imageAltText;
	this._extraInfo = params.extraInfo;
	this._textParam = params.text;
	this._deferred = params.deferred;
	this._expandNodeImage = params.expandNodeImage || "NodeExpanded";
	this._collapseNodeImage = params.collapseNodeImage || "NodeCollapsed";
	this._itemChecked = false;
	this._initialized = false;
	this._selectionEnabled = Boolean(params.selectable !== false);
	this._forceNotifySelection = Boolean(params.forceNotifySelection);
	this._actionEnabled = true;
	this._forceNotifyAction = Boolean(params.forceNotifyAction);
	this._dndScrollCallback = params.dndScrollCallback;
	this._dndScrollId = params.dndScrollId;
	this._arrowDisabled = params.arrowDisabled;

	if (params.singleClickAction) {
		this._singleClickAction = true;
		this._selectedFocusedClassName = this._selectedClassName = this._textClassName;
		this._hoverClassName = [this._origClassName, DwtCssStyle.HOVER].join("-");
	} else {
		this._hoverClassName = this._textClassName;
	}

	// if our parent is DwtTree or our parent is initialized and is not deferred
	// type or is expanded, then initialize ourself, else wait
	if (parent instanceof DwtTree || (parent._initialized && (!parent._deferred || parent._expanded)) || !params.deferred) {
		this._initialize(params.index);
	} else {
		parent._addDeferredChild(this, params.index);
		this._index = params.index;
	}
};

DwtTreeItem.PARAMS = ["parent", "index", "text", "imageInfo", "deferred", "className", "posStyle",
					  "forceNotifySelection", "forceNotifyAction"];

DwtTreeItem.prototype = new DwtComposite;
DwtTreeItem.prototype.constructor = DwtTreeItem;

DwtTreeItem.prototype.isDwtTreeItem = true;
DwtTreeItem.prototype.toString = function() { return "DwtTreeItem"; };

DwtTreeItem.prototype.TEMPLATE = "dwt.Widgets#ZTreeItem";

DwtTreeItem.prototype.role = "treeitem";
DwtTreeItem.prototype.isFocusable = true;

DwtTreeItem.prototype._checkBoxVisible = true; // Assume it's shown, if check style

// Consts

DwtTreeItem._NODECELL_DIM = "16px";
DwtTreeItem._processedMouseDown = false;

// Public Methods

DwtTreeItem.prototype.dispose =
function() {
    DwtComposite.prototype.dispose.call(this);
	this._itemDiv = null;
	this._nodeCell = null;
	this._checkBoxCell = null;
	this._checkedImg = null;
	this._checkBox = null;
	this._imageCell = null;
	this._textCell = null;
	this._childDiv = null;
	this._initialized = false;
};

/**
 * override DwtControl.prototype.getData to take care of special case of KEY_OBJECT of type ZmOrganizer. See bug 82027
 * @param key
 * @return {*}
 */
DwtTreeItem.prototype.getData =
function(key) {
	var obj = this._data[key];
	if (key !== Dwt.KEY_OBJECT || !obj || !obj.isZmOrganizer) {
		return obj;
	}
	//special case for ZmOrganizer instance of the Dwt.KEY_OBJECT attribute.
	//bug 82027 - the folder attributes such as name could be wrong after refresh block+ rename when new instance was created but not set to the item Dwt.KEY_OBJECT attribute.
	var cachedOrganizer = obj && appCtxt.cacheGet(obj.id);
	return cachedOrganizer || obj; //just in case somehow it's no longer cached. No idea if could happen.
};

/**
 * Checks if the item is checked.
 * 
 * @return	{boolean}	<code>true</code> if the item is checked
 */
DwtTreeItem.prototype.getChecked =
function() {
	return this._itemChecked;
};

/**
 * Sets the checked flag.
 * 
 * @param	{boolean}	checked		if <code>true</code>, check the item
 * @param	{boolean}	force		if <code>true</code>, force the setting
 */
DwtTreeItem.prototype.setChecked =
function(checked, force) {
	if ((this._itemChecked != checked) || force) {
		this._itemChecked = checked;
		if (this._checkBox != null &&
			(this._checkBoxCell && Dwt.getVisible(this._checkBoxCell)))
		{
			Dwt.setVisible(this._checkedImg, checked);
		}
	}
};

DwtTreeItem._handleKeyPress =
function(event) {
	var keyCode = DwtKeyEvent.getCharCode(event);
	if (keyCode === DwtKeyEvent.KEY_SPACE) {
		this._handleCheckboxOnclick(event);
	}
}

DwtTreeItem.prototype._handleCheckboxOnclick =
function(ev) {
	this.setChecked(!Dwt.getVisible(this._checkedImg));

	ev = ev || window.event;
	ev.item = this;
	this._tree._itemChecked(this, ev);
};

DwtTreeItem.prototype.getExpanded =
function() {
	return this._expanded;
};

/**
 * Expands or collapses this tree item.
 *
 * @param {boolean}	expanded		if <code>true</code>, expands this node; otherwise collapses it
 * @param {boolean}	recurse		if <code>true</code>, expand children recursively (does not apply to collapsing)
 * @param	{boolean}	skipNotify		if <code>true</code>, do not notify the listeners
 */
DwtTreeItem.prototype.setExpanded =
function(expanded, recurse, skipNotify) {
	// Go up the chain, ensuring that parents are expanded/initialized
	if (expanded) {
		var p = this.parent;
		while (p instanceof DwtTreeItem && !p._expanded) {
			p.setExpanded(true);
			p = p.parent;
		}
		// Realize any deferred children
		this._realizeDeferredChildren();
	}
		
	// If we have children, then allow for expanding/collapsing
	if (this.getNumChildren()) {
		if (expanded && recurse) {
			if (!this._expanded) {
				this._expand(expanded, null, skipNotify);
			}
			var a = this.getChildren();
			for (var i = 0; i < a.length; i++) {
				if (a[i] instanceof DwtTreeItem) {
					a[i].setExpanded(expanded, recurse, skipNotify);
				}
			}
		} else if (this._expanded != expanded) {
			this._expand(expanded, null, skipNotify);
		}
	}
};

/**
 * Gets the child item count.
 * 
 * @return	{number}	the child item count
 */
DwtTreeItem.prototype.getItemCount =
function() {
	return this._children.size();
};

/**
 * Gets the items.
 * 
 * @return	{array}	an array of child {@link DwtTreeItem} objects
 */
DwtTreeItem.prototype.getItems =
function() {
	return this._children.getArray();
};

DwtTreeItem.prototype.getChildIndex =
function(item) {
	return this._children.indexOf(item);
};

/**
 * Get the nesting level; the toplevel tree is zero, and each lower layer
 * increases by one.
 * 
 * @return	{number}	the child item count
 */
DwtTreeItem.prototype.getNestingLevel =
function() {
	var nestingLevelCheck;
	// the toplevel tree is zero, we can have only one level(1) heading in the page which is already defined in the banner,so initializing the toplevel with level2 for all nested folders
	if (this.parent.getNestingLevel() == 0) {
		nestingLevelCheck = this.parent.getNestingLevel() + 2;
	}
	else {
		nestingLevelCheck = this.parent.getNestingLevel() + 1;
	}
	return (nestingLevelCheck > 6 ? 6 : nestingLevelCheck);
};

/**
 * Gets the image.
 * 
 * @return	{string}	the image
 */
DwtTreeItem.prototype.getImage =
function() {
	return this._imageInfo;
};

/**
 * Sets the image.
 * 
 * @param	{string}	imageInfo		the image
 * * @param	{string}	imageAlt		the image alter text
 */
DwtTreeItem.prototype.setImage =
function(imageInfo, imageAltText) {
	if (this._initialized) {
		if (this._imageCell) {
			AjxImg.setImage(this._imageCell, imageInfo, null, null, null, imageAltText);
		}
		this._imageInfo = imageInfo;
		this._imageAltInfo = imageAltText;
	} else {
		this._imageInfoParam = imageInfo;
		this._imageAltInfo = imageAltText;
	}	
};

DwtTreeItem.prototype.setDndImage =
function(imageInfo) {
	this._dndImageInfo = imageInfo;
};

DwtTreeItem.prototype.getSelected =
function() {
	return this._selected;
};

DwtTreeItem.prototype.getActioned =
function() {
	return this._actioned;
};

/**
 * Gets the text.
 * 
 * @return	{string}	the text
 */
DwtTreeItem.prototype.getText =
function() {
	return this._text;
};

/**
 * Sets the text.
 * 
 * @param	{string}	text		the text
 */
DwtTreeItem.prototype.setText =
function(text) {
	if (this._initialized && this._textCell) {
		if (!text) text = "";
		this._text = this._textCell.innerHTML = text;
	} else {
		this._textParam = text;
	}
};

/**
 * Sets the drag-and-drop text.
 * 
 * @param	{string}	text		the text
 */
DwtTreeItem.prototype.setDndText =
function(text) {
	this._dndText = text;
};

/**
 * Shows (or hides) the check box.
 * 
 * @param	{boolean}	show		if <code>true</code>, show the check box
 */
DwtTreeItem.prototype.showCheckBox =
function(show) {
	this._checkBoxVisible = show;
	if (this._checkBoxCell) {
		Dwt.setVisible(this._checkBoxCell, show);
	}
};

/**
 * Shows (or hides) the expansion icon.
 * 
 * @param	{boolean}	show		if <code>true</code>, show the expansion icon
 */
DwtTreeItem.prototype.showExpansionIcon =
function(show) {
	if (this._nodeCell) {
		Dwt.setVisible(this._nodeCell, show);
	}
};

/**
 * Enables (or disables) the selection.
 * 
 * @param	{boolean}	enable		if <code>true</code>, enable selection
 */
DwtTreeItem.prototype.enableSelection =
function(enable) {
	this._selectionEnabled = enable;
	this._selectedClassName = enable
		? this._origClassName + "-" + DwtCssStyle.SELECTED
		: this._origClassName;

};

DwtTreeItem.prototype.isSelectionEnabled =
function() {
	return this._selectionEnabled;
};


DwtTreeItem.prototype.enableAction =
function(enable) {
	this._actionEnabled = enable;
};

/**
 * Adds a separator at the given index. If no index is provided, adds it at the
 * end. A separator cannot currently be added as the first item (the child DIV will
 * not have been created).
 *
 * @param {number}	index		the position at which to add the separator
 */
DwtTreeItem.prototype.addSeparator =
function(index) {
	this._children.add((new DwtTreeItemSeparator(this)), index);
};

/**
 * Makes this tree item, or just part of it, visible or hidden.
 *
 * @param {boolean}	visible		if <code>true</code>, item (or part of it) becomes visible
 * @param {boolean}	itemOnly		if <code>true</code>, apply to this item's DIV only; child items are unaffected
 * @param {boolean}	childOnly		if <code>true</code>, apply to this item's child items only
 */
DwtTreeItem.prototype.setVisible =
function(visible, itemOnly, childOnly) {
	if (itemOnly && !childOnly) {
		Dwt.setVisible(this._itemDiv, visible);
	} else if (childOnly && !itemOnly) {
		Dwt.setVisible(this._childDiv, visible);
	} else {
		DwtComposite.prototype.setVisible.call(this, visible);
	}
};

DwtTreeItem.prototype.removeChild =
function(child) {
	if (child._initialized) {
		this._tree._deselect(child);
		if (this._childDiv) {
			this._childDiv.removeChild(child.getHtmlElement());
		}
	}
	this._children.remove(child);

	// if we have no children and we are expanded, then mark us a collapsed.
	// Also if there are no deferred children, then make sure we remove the
	// expand/collapse icon and replace it with a blank16Icon.
	if (this._children.size() == 0) {
		if (this._expanded)
			this._expanded = false;

		this._expandable = false;
		this.removeAttribute('aria-expanded')
		
		if (this._initialized && this._nodeCell) {
			AjxImg.setImage(this._nodeCell, "Blank_16");
			var imgEl = AjxImg.getImageElement(this._nodeCell);
			if (imgEl)
				Dwt.clearHandler(imgEl, DwtEvent.ONMOUSEDOWN);
		}
	}
};

DwtTreeItem.prototype.getKeyMapName =
function() {
	return DwtKeyMap.MAP_TREE;
};

DwtTreeItem.prototype.handleKeyAction =
function(actionCode, ev) {

	switch (actionCode) {
		
		case DwtKeyMap.ENTER:
			this._tree.setEnterSelection(this, true);
			break;


		case DwtKeyMap.NEXT: {
			var ti = this._tree._getNextTreeItem(true);
			if (ti) {
				ti._tree.setSelection(ti, false, true);
			}
			break;
		}

		case DwtKeyMap.PREV: {
			var ti = this._tree._getNextTreeItem(false);
			if (ti) {
				ti._tree.setSelection(ti, false, true);
			}
			break;
		}

		case DwtKeyMap.SELECT_FIRST:
		case DwtKeyMap.SELECT_LAST: {
			var ti = (actionCode === DwtKeyMap.SELECT_FIRST) ?
				this._tree._getFirstTreeItem() : this._tree._getLastTreeItem();
			if (ti) {
				ti._tree.setSelection(ti, false, true);
			}
			break;
		}

		case DwtKeyMap.EXPAND: {
			if (!this._expanded) {
				this.setExpanded(true, false, true);
			} else if (this._children.size() > 0) {
				// Select first child node
				var firstChild = this._children.get(0);
				this._tree.setSelection(firstChild, false, true);
			}
			break;
		}

		case DwtKeyMap.COLLAPSE: {
			if (this._expanded) {
				this.setExpanded(false, false, true);
			} else if (this.parent.isDwtTreeItem) {
				// select parent
				this._tree.setSelection(this.parent, false, true);
			}
			break;
		}

		case DwtKeyMap.SUBMENU: {
			var target = this.getHtmlElement();
			var p = Dwt.toWindow(target, 0, 0);
			var s = this.getSize();
			var docX = p.x + s.x / 4;
			var docY = p.y + s.y / 2;
			this._gotMouseDownRight = true;
			this._emulateSingleClick({dwtObj:this, target:target, button:DwtMouseEvent.RIGHT,
									  docX:docX, docY:docY, kbNavEvent:true});
			break;
		}

		default:
			return false;

	}

	return true;
};

DwtTreeItem.prototype.addNodeIconListeners =
function() {
	var imgEl = AjxImg.getImageElement(this._nodeCell);
	if (imgEl) {
		Dwt.setHandler(imgEl, DwtEvent.ONMOUSEDOWN, DwtTreeItem._nodeIconMouseDownHdlr);
		Dwt.setHandler(imgEl, DwtEvent.ONMOUSEUP, DwtTreeItem._nodeIconMouseUpHdlr);
	}
};

DwtTreeItem.prototype._initialize =
function(index, realizeDeferred, forceNode) {
	this._checkState();
	if (AjxEnv.isIE) {
		this._setEventHdlrs([DwtEvent.ONMOUSEENTER, DwtEvent.ONMOUSELEAVE]);
	}
	if (AjxEnv.isSafari) {	// bug fix #25016
		this._setEventHdlrs([DwtEvent.ONCONTEXTMENU]);
	}
	var data = {
		id: this._htmlElId,
		divClassName: this._origClassName,
		isCheckedStyle: this._tree.isCheckedStyle,
		textClassName: this._textClassName
	};

	this._createHtmlFromTemplate(this.TEMPLATE, data);

	// add this object's HTML element to the DOM
	this.parent._addItem(this, index, realizeDeferred);

	// cache DOM objects here
	this._itemDiv = document.getElementById(data.id + "_div");
	this._nodeCell = document.getElementById(data.id + "_nodeCell");
	this._checkBoxCell = document.getElementById(data.id + "_checkboxCell");
	this._checkBox = document.getElementById(data.id + "_checkbox");
	this._checkedImg = document.getElementById(data.id + "_checkboxImg");
	this._imageCell = document.getElementById(data.id + "_imageCell");
	this._textCell = document.getElementById(data.id + "_textCell");
	this._extraCell = document.getElementById(data.id + "_extraCell");

	/* assign the ARIA level */
	if (this._itemDiv) {
		this._itemDiv.setAttribute("aria-level", this.getNestingLevel());
		this._itemDiv.setAttribute("role", "heading");

		/* add a label for screenreaders, so that they don't read the entire element */
		if (this._textCell) {
			this._itemDiv.setAttribute("aria-labelledby", this._textCell.id);
		}
	}

	if (this._dynamicWidth){
		var tableNode = document.getElementById(data.id + "_table");
		if (tableNode) {
			tableNode.style.tableLayout = "auto";
		}
	}

	this._expandable = false;
	this.removeAttribute('aria-expanded');

	// If we have deferred children, then make sure we set up accordingly
	if (this._nodeCell) {
		this._nodeCell.style.minWidth = this._nodeCell.style.width = this._nodeCell.style.height = DwtTreeItem._NODECELL_DIM;
		if (this._children.size() > 0 || forceNode) {
			this._expandable = true;
			AjxImg.setImage(this._nodeCell, this._collapseNodeImage);
			this.addNodeIconListeners();
		}
	}

	if (this._extraCell) {
		AjxImg.setImage(this._extraCell, (this._extraInfo ||  "Blank_16"));
		this._extraCell.className = this._treeItemExtraImgClass;
	}

	// initialize checkbox
	if (this._tree.isCheckedStyle && this._checkBox) {
		Dwt.setHandler(this.getHtmlElement(), DwtEvent.ONKEYUP, DwtTreeItem._handleKeyPress.bind(this));
		this._checkBox.onclick = AjxCallback.simpleClosure(this._handleCheckboxOnclick, this);
		this.showCheckBox(this._checkBoxVisible);
		this.setChecked(this._tree.isCheckedByDefault, true);
	}

	// initialize icon
	if (this._imageCell && this._imageInfoParam) {
		AjxImg.setImage(this._imageCell, this._imageInfoParam, null, null, null, this._imageAltInfo);
		this._imageInfo = this._imageInfoParam;
	}

	// initialize text
	if (this._textCell && this._textParam) {
		this._textCell.innerHTML = this._text = this._textParam;
	}
	this._expanded = this._selected = this._actioned = false;
	this._gotMouseDownLeft = this._gotMouseDownRight = false;
	this._addMouseListeners();

	this._initialized = true;
};

/**
 * Sets the tree item color.
 * 
 * @param	{string}	className		the class name
 */
DwtTreeItem.prototype.setTreeItemColor = 
function(className) {
	var id = this._htmlElId +"_table";
	var treeItemTableEl = document.getElementById(id);
	var treeItemDivEl = document.getElementById(this._htmlElId + "_div");
	var treeItemEl = this.getHtmlElement();

	var newClassName = this._origClassName + " " + className;
	if (treeItemDivEl) {
		treeItemDivEl.className = newClassName;
	} else if (treeItemEl) {
		treeItemEl.className =  className;
	}
};

DwtTreeItem.prototype._addMouseListeners =
function() {
	var events = [DwtEvent.ONMOUSEDOWN, DwtEvent.ONMOUSEUP, DwtEvent.ONDBLCLICK];
	if (AjxEnv.isIE) {
		events.push(DwtEvent.ONMOUSEENTER, DwtEvent.ONMOUSELEAVE);
	} else {
		events.push(DwtEvent.ONMOUSEOVER, DwtEvent.ONMOUSEOUT);
	}
	if (AjxEnv.isSafari) {
		events.push(DwtEvent.ONCONTEXTMENU);
	}
	for (var i = 0; i < events.length; i++) {
		this.addListener(events[i], DwtTreeItem._listeners[events[i]]);
	}
};

DwtTreeItem.prototype._addDeferredChild =
function(child, index) {
	// If we are initialized, then we need to add a expansion node
	if (this._initialized && this._children.size() == 0) {
		if (this._nodeCell) {
			AjxImg.setImage(this._nodeCell, this._collapseNodeImage);
			var imgEl = AjxImg.getImageElement(this._nodeCell);
			if (imgEl) {
				this._expandable = true;
				this.setAttribute('aria-expanded', this._expanded);
				Dwt.setHandler(imgEl, DwtEvent.ONMOUSEDOWN, DwtTreeItem._nodeIconMouseDownHdlr);
				Dwt.setHandler(imgEl, DwtEvent.ONMOUSEUP, DwtTreeItem._nodeIconMouseUpHdlr);
			}
		}
	}
	this._children.add(child, index);
};

DwtTreeItem.prototype.addChild =
function(child) { /* do nothing since we add to the DOM our own way */ };

DwtTreeItem.prototype._addItem =
function(item, index, realizeDeferred) {
	if (!this._children.contains(item)) {
		this._children.add(item, index);
	}
	this._expandable = true;

	if (this._childDiv == null) {
		this._childDiv = document.createElement("div");
		this._childDiv.className = (this.parent != this._tree)
			? "DwtTreeItemChildDiv" : "DwtTreeItemLevel1ChildDiv";
		this._childDiv.setAttribute('role', 'group');
		this._childDiv.setAttribute('aria-labelledby', this._itemDiv.id);
		this._childDiv.setAttribute('aria-expanded', this._expanded);
		this.getHtmlElement().appendChild(this._childDiv);
		if (!this._expanded) {
			this._childDiv.style.display = "none";
		}
	}

	if (realizeDeferred && this._nodeCell) {
		if (AjxImg.getImageClass(this._nodeCell) == AjxImg.getClassForImage("Blank_16")) {
			AjxImg.setImage(this._nodeCell, this._expanded ? this._expandNodeImage : this._collapseNodeImage);
			var imgEl = AjxImg.getImageElement(this._nodeCell);
			if (imgEl) {
				Dwt.setHandler(imgEl, DwtEvent.ONMOUSEDOWN, DwtTreeItem._nodeIconMouseDownHdlr);
			}
		}
	}

	var childDiv = this._childDiv;
	var numChildren = childDiv.childNodes.length;
	if (index == null || index >= numChildren || numChildren == 0) {
		childDiv.appendChild(item.getHtmlElement());
	} else {
		childDiv.insertBefore(item.getHtmlElement(), childDiv.childNodes[index]);
	}
};

DwtTreeItem.prototype.sort =
function(cmp) {
	this._children.sort(cmp);
	if (this._childDiv) {
		this._setChildElOrder();
	} else {
		this._needsSort = true;
	}
};

DwtTreeItem.prototype._setChildElOrder =
function(cmp) {
	var df = document.createDocumentFragment();
	this._children.foreach(function(item, i) {
		df.appendChild(item.getHtmlElement());
		item._index = i;
	});
	this._childDiv.appendChild(df);
};

DwtTreeItem.prototype._getDragProxy =
function() {
	var icon = document.createElement("div");
	Dwt.setPosition(icon, Dwt.ABSOLUTE_STYLE); 
	var table = document.createElement("table");
	icon.appendChild(table);
	table.cellSpacing = table.cellPadding = 0;

	var row = table.insertRow(0);
	var i = 0;

	var c = row.insertCell(i++);
	c.noWrap = true;
	if (this._dndImageInfo) {
		AjxImg.setImage(c, this._dndImageInfo);
	} else if (this._imageInfo) {
		AjxImg.setImage(c, this._imageInfo);
	}

	c = row.insertCell(i);
	c.noWrap = true;
	c.className = this._origClassName;
	if (this._dndText) {
		c.innerHTML = this._dndText;
	} else if (this._text) {
		c.innerHTML = this._text;
	}

	this.shell.getHtmlElement().appendChild(icon);
	Dwt.setZIndex(icon, Dwt.Z_DND);
	return icon;
};

DwtTreeItem.prototype._dragEnter =
function() {
	this._preDragClassName = this._textCell.className;
	this._textCell.className = this._dragOverClassName;
	this._draghovering = true;
};

DwtTreeItem.prototype._dragHover =
function() {
	if (this.getNumChildren() > 0 && !this.getExpanded()) {
		this.setExpanded(true);
	}
};

DwtTreeItem.prototype._dragLeave =
function(ev) {
	if (this._preDragClassName) {
		this._textCell.className = this._preDragClassName;
	}
	this._draghovering = false;
};

DwtTreeItem.prototype._drop =
function() {
	if (this._preDragClassName) {
		this._textCell.className = this._preDragClassName;
	}
	this._draghovering = false;
};

/**
 *   This is for bug 45129.
 *   In the DwControl's focusByMouseDownEvent, it focuses the TreeItem 
 *   And change TreeItem's color. But sometimes when mousedown and mouseup
 *   haven't been matched on the one element. It will cause multiple selection. 
 *   For in the mouseup handle function, we has done focus if we find both mouse 
 *   down and up happened on the same element. So when the mouse is down, we just
 *   do nothing.
 */
DwtTreeItem.prototype._focusByMouseDownEvent =
function(ev) {
	
}

DwtTreeItem._nodeIconMouseDownHdlr =
function(ev) {
	var obj = DwtControl.getTargetControl(ev);
	var mouseEv = DwtShell.mouseEvent;
	mouseEv.setFromDhtmlEvent(ev, obj);
	if (mouseEv.button == DwtMouseEvent.LEFT) {
		obj._expand(!obj._expanded, mouseEv);
	} else if (mouseEv.button == DwtMouseEvent.RIGHT) {
		mouseEv.dwtObj._tree._itemActioned(mouseEv.dwtObj, mouseEv);
	}

	mouseEv._stopPropagation = true;
	mouseEv._returnValue = false;
	mouseEv.setToDhtmlEvent(ev);
	return false;
};

DwtTreeItem._nodeIconMouseUpHdlr = 
function(ev) {
	var obj = DwtControl.getTargetControl(ev);
	var mouseEv = DwtShell.mouseEvent;
	mouseEv._stopPropagation = true;
	mouseEv._returnValue = false;
	mouseEv.setToDhtmlEvent(ev);
	return false;
};

DwtTreeItem.prototype._expand =
function(expand, ev, skipNotify) {
	if (expand !== this._expanded) {
		if (!expand) {
			this._expanded = false;
			this._childDiv.style.display = "none";
			if (this._nodeCell) {
				AjxImg.setImage(this._nodeCell, this._collapseNodeImage);
			}
			this._tree._itemCollapsed(this, ev, skipNotify);
		} else {
			// The first thing we need to do is initialize any deferred children so that they
			// actually have content
			this._realizeDeferredChildren();
			this._expanded = true;
			if(this._childDiv && this._childDiv.style)
				this._childDiv.style.display = "block";
			if (this._nodeCell) {
				AjxImg.setImage(this._nodeCell, this._expandNodeImage);
			}
			this._tree._itemExpanded(this, ev, skipNotify);
		}	

		this.setAttribute('aria-expanded', expand);
		this._childDiv.setAttribute('aria-expanded', expand);
		this._childDiv.setAttribute('aria-hidden', !expand);
	}
};

DwtTreeItem.prototype._realizeDeferredChildren =
function() {
	var a = this._children.getArray();
	for (var i = 0; i < a.length; i++) {
		var treeItem = a[i];
		if (!treeItem._initialized) {
			treeItem._initialize(treeItem._index, true);
		} else if (treeItem._isSeparator && !treeItem.div && this._childDiv) {
			// Note: separators marked as initialized on construction
			var div = treeItem.div = document.createElement("div");
			div.className = "vSpace";
			this._childDiv.appendChild(div);
			treeItem._initialized = true;
		}
	}
	if (this._needsSort) {
		if (a.length) {
			this._setChildElOrder();
		}
		delete this.__needsSort;
	}
};

DwtTreeItem.prototype._isChildOf =
function(item) {
	var test = this.parent;
	while (test && test != this._tree) {
		if (test == item)
			return true;
		test = test.parent;
	}
	return false;
};

DwtTreeItem.prototype._setTreeElementStyles =
function(img, focused) {
   if (this._arrowDisabled || this._draghovering) {
        return;
   }
   var selected = focused ? "-focused" : "";
   if (this._extraCell) {
        AjxImg.setImage(this._extraCell, img);
        this._extraCell.className = this._treeItemExtraImgClass + selected;
   }
   if (this._textCell)
        this._textCell.className = this._treeItemTextClass + selected;
}

DwtTreeItem.prototype._setSelected =
function(selected, noFocus) {
	if (this._selected != selected && !this._disposed) {
		this._selected = selected;
		if (!this._initialized) {
			this._initialize();
		}
		if (!this._itemDiv) { return; }

		var didSelect;

		if (selected && (this._selectionEnabled || this._forceNotifySelection) /*&& this._origClassName == "DwtTreeItem"*/) {
			this._itemDiv.className = this._selectedClassName;
			this._setTreeElementStyles("DownArrowSmall", true);
			this._tree.setAttribute('aria-activedescendant', this.getHTMLElId());
            if (!noFocus) {
				this.focus();
			}
			didSelect = true;
		} else {
			this.blur();
			this._setTreeElementStyles("Blank_16", false);
			this._itemDiv.className = this._origClassName;;
			this._tree.removeAttribute('aria-activedescendant');
			didSelect = false;
		}

		this.getHtmlElement().setAttribute('aria-selected', selected);
		/* TODO: disable on IE? screenreaders in IE may announce items twice if
		 * we do the below, which is not strictly necessary */
		var treeEl = this._tree.getHtmlElement();

		if (selected) {
			treeEl.setAttribute('aria-activedescendant', this.getHTMLElId());
		} else {
			treeEl.removeAttribute('aria-activedescendant');
		}

		return didSelect;
	}
};

DwtTreeItem.prototype._setActioned =
function(actioned) {
	if (this._actioned != actioned) {
		this._actioned = actioned;
		if (!this._initialized) {
			this._initialize();
		}

		if (!this._itemDiv) { return; }

		if (actioned && (this._actionEnabled || this._forceNotifyAction) && !this._selected) {
			this._itemDiv.className = this._actionedClassName;
			return true;
		}

		if (!actioned) {
			if (!this._selected) {
				this._itemDiv.className = this._origClassName;
			}
			return false;
		}
	}
};

DwtTreeItem.prototype._focus =
function() {
	if (!this._itemDiv) { return; }
	// focused tree item should always be selected as well
	this._itemDiv.className = this._selectedFocusedClassName;
	this._setTreeElementStyles("DownArrowSmall", true);
};

DwtTreeItem.prototype._blur =
function() {
	if (!this._itemDiv) { return; }
	this._itemDiv.className = this._selected
		? this._selectedClassName : this._origClassName;
	this._setTreeElementStyles(this._selected ? "DownArrowSmall" : "Blank_16", this._selected);
};

DwtTreeItem._mouseDownListener =
function(ev) {
	var treeItem = ev.dwtObj;
	if (!treeItem) { return false; }
	if (ev.target == treeItem._childDiv) { return; }

	if (ev.button == DwtMouseEvent.LEFT && (treeItem._selectionEnabled || treeItem._forceNotifySelection)) {
		treeItem._gotMouseDownLeft = true;
	} else if (ev.button == DwtMouseEvent.RIGHT && (treeItem._actionEnabled || treeItem._forceNotifyAction)) {
		treeItem._gotMouseDownRight = true;
	}

};

DwtTreeItem._mouseOutListener = 
function(ev) {
	var treeItem = ev.dwtObj;
	if (!treeItem) { return false; }
	if (ev.target == treeItem._childDiv) { return; }

	treeItem._gotMouseDownLeft = false;
	treeItem._gotMouseDownRight = false;
	if (treeItem._singleClickAction && treeItem._textCell) {
		treeItem._textCell.className = treeItem._textClassName;
	}
    if(!treeItem._selected){
       treeItem._setTreeElementStyles("Blank_16", false);
    }

};

DwtTreeItem._mouseOverListener =
function(ev) {
	var treeItem = ev.dwtObj;
	if (!treeItem) { return false; }
	if (ev.target == treeItem._childDiv) { return; }

	if (treeItem._singleClickAction && treeItem._textCell) {
		treeItem._textCell.className = treeItem._hoverClassName;
	}
    if(!treeItem._selected){
       treeItem._setTreeElementStyles("ColumnDownArrow", true);
    }
};

DwtTreeItem._mouseUpListener = function(ev) {

	var treeItem = ev.dwtObj;
	if (!treeItem) {
        return false;
    }

	// Ignore any mouse events in the child div i.e. the div which 
	// holds all the items children. In the case of IE, no clicks are
	// reported when clicking in the padding area (note all children
	// are indented using padding-left style); however, mozilla
	// reports mouse events that happen in the padding area
	if (ev.target === treeItem._childDiv) {
        return;
    }

	//ignore the collapse/expand arrow. This is handled in DwtTreeItem._nodeIconMouseDownHdlr. It should only collapse/expand and never select this item, so no point in going on.
	if (treeItem._expandable && ev.target === AjxImg.getImageElement(treeItem._nodeCell)) {
		return;
	}

    var targetElement = DwtUiEvent.getTargetWithProp(ev, "id"),
        isContextCmd = (treeItem._extraCell && targetElement && treeItem._extraCell.id === targetElement.id);

    if ((ev.button === DwtMouseEvent.RIGHT && treeItem._gotMouseDownRight) || isContextCmd) {
        treeItem._tree._itemActioned(treeItem, ev);
    }
    else if (ev.button === DwtMouseEvent.LEFT && treeItem._gotMouseDownLeft) {
		treeItem._tree._itemClicked(treeItem, ev);
	}
};

DwtTreeItem._doubleClickListener =
function(ev) {
	var treeItem = ev.dwtObj;
	if (!treeItem) { return false; }
	// See comment in DwtTreeItem.prototype._mouseDownListener
	if (ev.target == treeItem._childDiv) { return; }

	var obj = DwtControl.getTargetControl(ev);
	var mouseEv = DwtShell.mouseEvent;
	mouseEv.setFromDhtmlEvent(ev, obj);
	if (mouseEv.button == DwtMouseEvent.LEFT || mouseEv.button == DwtMouseEvent.NONE) {	// NONE for IE bug
		mouseEv.dwtObj._tree._itemDblClicked(mouseEv.dwtObj, mouseEv);
	}
};

DwtTreeItem._contextListener =
function(ev) {
	// for Safari, we have to fake a right click
	if (AjxEnv.isSafari) {
		var obj = DwtControl.getTargetControl(ev);
		var prevent = obj ? obj.preventContextMenu() : true;
		if (prevent) {
			obj.notifyListeners(DwtEvent.ONMOUSEDOWN, ev);
			return obj.notifyListeners(DwtEvent.ONMOUSEUP, ev);
		}
	}
};

DwtTreeItem.prototype._emulateSingleClick =
function(params) {
	var mev = new DwtMouseEvent();
	this._setMouseEvent(mev, params);
	mev.kbNavEvent = params.kbNavEvent;
	this.notifyListeners(DwtEvent.ONMOUSEUP, mev);
};

DwtTreeItem.prototype.getTooltipBase =
function(hoverEv) {
	return this._itemDiv;
};

DwtTreeItem._listeners = {};
DwtTreeItem._listeners[DwtEvent.ONMOUSEDOWN] = new AjxListener(null, DwtTreeItem._mouseDownListener);
DwtTreeItem._listeners[DwtEvent.ONMOUSEOUT] = new AjxListener(null, DwtTreeItem._mouseOutListener);
DwtTreeItem._listeners[DwtEvent.ONMOUSELEAVE] = new AjxListener(null, DwtTreeItem._mouseOutListener);
DwtTreeItem._listeners[DwtEvent.ONMOUSEENTER] = new AjxListener(null, DwtTreeItem._mouseOverListener);
DwtTreeItem._listeners[DwtEvent.ONMOUSEOVER] = new AjxListener(null, DwtTreeItem._mouseOverListener);
DwtTreeItem._listeners[DwtEvent.ONMOUSEUP] = new AjxListener(null, DwtTreeItem._mouseUpListener);
DwtTreeItem._listeners[DwtEvent.ONDBLCLICK] = new AjxListener(null, DwtTreeItem._doubleClickListener);
DwtTreeItem._listeners[DwtEvent.ONCONTEXTMENU] = new AjxListener(null, DwtTreeItem._contextListener);


/**
 * Minimal class for a separator (some vertical space) between other tree items.
 * The functions it has are to handle a dispose() call when the containing tree
 * is disposed.
 * 
 * TODO: At some point we should just make this a DwtControl, or find some other
 * 		 way of keeping it minimal.
 * 
 * 
 * @private
 */
DwtTreeItemSeparator = function(parent) {
	this.parent = parent;
	this._isSeparator = true;
	this._initialized = true;
};

DwtTreeItemSeparator.prototype.dispose =
function() {
	DwtComposite.prototype.removeChild.call(this.parent, this);
};

DwtTreeItemSeparator.prototype.isInitialized =
function() {
	return this._initialized;
};

DwtTreeItemSeparator.prototype.getHtmlElement =
function() {
	return this.div;
};
}
if (AjxPackage.define("ajax.dwt.widgets.DwtHeaderTreeItem")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */


/**
 * Creates a Header Tree Item.
 * @constructor
 * @class
 * This class implements a tree item widget.
 *
 * @author Dave Comfort
 *
 * @param {hash}	params				a hash of parameters
 * @param {DwtComposite}      params.parent				the parent widget
 * @param {number}      params.index 				the index at which to add this control among parent's children
 * @param {string}      params.text 					the label text for the tree item
 * @param {string}      params.imageInfo				the icon for the tree item
 * @param {boolean}      params.deferred				if <code>true</code>, postpone initialization until needed.
 * @param {string}      params.className				the CSS class
 * @param  {constant}	params.posStyle				the positioning style
 * @param {boolean}      params.forceNotifySelection	force notify selection even if checked style
 * @param {boolean}      params.forceNotifyAction		force notify action even if checked style
 * @param {hash}		  params.optButton				a hash of data for showing a options button in the item: image, tooltip, callback
 * @param {boolean}      params.selectable			if <code>true</code>, this item is selectable
 *        
 * @extend		DwtTreeItem
 */
DwtHeaderTreeItem = function(params) {
	this.overview = params.overview;
	this._optButton = params.optButton;
	this._noNodeCell = params.noNodeCell;
	DwtTreeItem.call(this, params);
	this._arrowDisabled = true; //override what DwTreeItem constructor sets.
};

DwtHeaderTreeItem.prototype = new DwtTreeItem;
DwtHeaderTreeItem.prototype.constructor = DwtHeaderTreeItem;

DwtHeaderTreeItem.prototype.TEMPLATE = "dwt.Widgets#ZHeaderTreeItem";

DwtHeaderTreeItem.prototype.toString =
function() {
	return "DwtHeaderTreeItem";
};

DwtHeaderTreeItem.prototype._createHtmlFromTemplate =
function(template, data) {
	data.noNodeCell = this._noNodeCell;
	DwtTreeItem.prototype._createHtmlFromTemplate.apply(this, arguments);
};

DwtHeaderTreeItem.prototype._initialize =
function() {
	DwtTreeItem.prototype._initialize.apply(this, arguments);

	// We must label the tree root, otherwise IE will let screen readers read
	// THE ENTIRE TREE when it gets focus
	var treeEl = this._tree.getHtmlElement();
	treeEl.setAttribute("aria-labelledby", this._textCell.id);

	if (this._optButton) {
		this._optButtonId = this._htmlElId + "_optButton";
		var optButtonEl = document.getElementById(this._optButtonId);
		if (optButtonEl) {
			this._optButtonItem = new DwtBorderlessButton({parent:this, style:DwtLabel.IMAGE_LEFT});
			this._optButtonItem.setToolTipContent(this._optButton.tooltip);
			this._optButtonItem.callback = this._optButton.callback;
			this._optButtonItem.addSelectionListener(new AjxListener(this, this._onclickHandler));
			this._optButtonItem.replaceElement(this._optButtonId);
			this._optButtonItem.setImage("ContextMenu");
			this._optButtonItem.setAriaLabel(this._optButton.tooltip);
			this._optButtonItem.setIconEl(this._optButtonItem.getHtmlElement()); // image container is button
		}
	}
};

DwtHeaderTreeItem.prototype._onclickHandler =
function(ev) {
    this._tree._itemActioned(this, ev);
};


DwtHeaderTreeItem.prototype._focusByMouseUpEvent =
function(ev)  {
	var targetId = ev.target && ev.target.id;
	if (targetId && (targetId == this._headerButtonId)) { return; }
	DwtTreeItem.prototype._focusByMouseUpEvent.apply(this, arguments);
};
}
if (AjxPackage.define("ajax.dwt.widgets.DwtTree")) {
    /*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */


/**
 * Creates a Tree widget.
 * @constructor
 * @class
 * This class implements a tree widget. Tree widgets may contain one or more DwtTreeItems.
 *
 * @author Ross Dargahi
 * 
 * @param {hash}	params				a hash of parameters
 * @param  {DwtComposite}     params.parent			the parent widget
 * @param  {DwtTree.SINGLE_STYLE|DwtTree.MULTI_STYLE|DwtTree.CHECKEDITEM_STYLE}     params.style 	the tree style
 * @param  {string}     params.className				the CSS class
 * @param  {constant}     params.posStyle				the positioning style (see {@link DwtControl})
 * @param  {boolean}     params.isCheckedByDefault	default checked state if tree styles is "checked"
 * 
 * @extends		DwtComposite
 */
DwtTree = function(params) {
	if (arguments.length == 0) { return; }
	params = Dwt.getParams(arguments, DwtTree.PARAMS);
	params.className = params.className || "DwtTree";
	DwtComposite.call(this, params);

	var events = [DwtEvent.ONMOUSEDOWN, DwtEvent.ONMOUSEUP, DwtEvent.ONDBLCLICK];
	if (!AjxEnv.isIE) {
		events = events.concat([DwtEvent.ONMOUSEOVER, DwtEvent.ONMOUSEOUT]);
	}
	this._setEventHdlrs(events);

	var style = params.style;
	if (!style) {
		this._style = DwtTree.SINGLE_STYLE;
	} else {
		if (style == DwtTree.CHECKEDITEM_STYLE) {
			style |= DwtTree.SINGLE_STYLE;
		}
		this._style = style;
	}
	this.isCheckedStyle = ((this._style & DwtTree.CHECKEDITEM_STYLE) != 0);
	this.isCheckedByDefault = params.isCheckedByDefault;

	this._selectedItems = new AjxVector();
	this._selEv = new DwtSelectionEvent(true);
	this._selByClickEv = new DwtSelectionEvent(true);
	this._selByClickEv.clicked = true;
	this._selByEnterEv = new DwtSelectionEvent(true);
	this._selByEnterEv.enter = true;

    // Let tree be a single tab stop, then manage focus among items using arrow keys
    this.tabGroupMember = this;
};

DwtTree.PARAMS = ["parent", "style", "className", "posStyle"];

DwtTree.prototype = new DwtComposite;
DwtTree.prototype.constructor = DwtTree;
DwtTree.prototype.role = "tree";

DwtTree.prototype.toString = 
function() {
	return "DwtTree";
};

/**
 * Defines the "single" style.
 */
DwtTree.SINGLE_STYLE = 1;
/**
 * Defines the "multi" style.
 */
DwtTree.MULTI_STYLE = 2;
/**
 * Defines the "checked-item" style.
 */
DwtTree.CHECKEDITEM_STYLE = 4;

DwtTree.ITEM_SELECTED = 0;
DwtTree.ITEM_DESELECTED = 1;
DwtTree.ITEM_CHECKED = 2;
DwtTree.ITEM_ACTIONED = 3;
DwtTree.ITEM_DBL_CLICKED = 4;

DwtTree.ITEM_EXPANDED = 1;
DwtTree.ITEM_COLLAPSED = 2;

/**
 * Gets the style.
 * 
 * @return	{constant}	the style
 */
DwtTree.prototype.getStyle =
function() {
	return this._style;
};

/**
 * Get the nesting level; this is zero for trees.
 *
 * @return	{number}	the child item count
 */
DwtTree.prototype.getNestingLevel =
function() {
	return 0;
};

/**
 * Adds a selection listener.
 * 
 * @param	{AjxListener}	listener	the listener
 */
DwtTree.prototype.addSelectionListener = 
function(listener) {
	this.addListener(DwtEvent.SELECTION, listener);
};

/**
 * Removes a selection listener.
 * 
 * @param	{AjxListener}	listener	the listener
 */
DwtTree.prototype.removeSelectionListener = 
function(listener) {
	this.removeListener(DwtEvent.SELECTION, listener);    	
};

/**
 * Adds a tree listener.
 * 
 * @param	{AjxListener}	listener	the listener
 */
DwtTree.prototype.addTreeListener = 
function(listener) {
	this.addListener(DwtEvent.TREE, listener);
};

/**
 * Removes a selection listener.
 * 
 * @param	{AjxListener}	listener	the listener
 */
DwtTree.prototype.removeTreeListener = 
function(listener) {
	this.removeListener(DwtEvent.TREE, listener);
};

/**
 * Gets the tree item count.
 * 
 * @return	{number}	the item count
 */
DwtTree.prototype.getItemCount =
function() {
	return this.getItems().length;
};

/**
 * Gets the items.
 * 
 * @return	{array}	an array of {@link DwtTreeItem} objects
 */
DwtTree.prototype.getItems =
function() {
	return this._children.getArray();
};

/** Clears the tree items. */
DwtTree.prototype.clearItems = function() {
    var items = this.getItems();
    for (var i = 0; i < items.length; i++) {
        this.removeChild(items[i]);
    }
    this._getContainerElement().innerHTML = "";
};


/**
 * De-selects all items.
 * 
 */
DwtTree.prototype.deselectAll =
function() {
	var a = this._selectedItems.getArray();
	var sz = this._selectedItems.size();
	for (var i = 0; i < sz; i++) {
		if (a[i]) {
			a[i]._setSelected(false);
		}
	}
	if (sz > 0) {
		this._notifyListeners(DwtEvent.SELECTION, this._selectedItems.getArray(), DwtTree.ITEM_DESELECTED, null, this._selEv);
	}
	this._selectedItems.removeAll();
};

/**
 * Gets an array of selection items.
 * 
 * @return	{array}	an array of {@link DwtTreeItem} objects
 */
DwtTree.prototype.getSelection =
function() {
	return this._selectedItems.getArray();
};

DwtTree.prototype.setEnterSelection =
function(treeItem, kbNavEvent) {
	if (!treeItem) {
		return;
	}
	this._notifyListeners(DwtEvent.SELECTION, [treeItem], DwtTree.ITEM_SELECTED, null, this._selByEnterEv, kbNavEvent);
};


DwtTree.prototype.setSelection =
function(treeItem, skipNotify, kbNavEvent, noFocus) {
	if (!treeItem || !treeItem.isSelectionEnabled()) {
		return;
	}

	// Remove currently selected items from the selection list. if <treeItem> is in that list, then note it and return
	// after we are done processing the selected list
	var a = this._selectedItems.getArray();
	var sz = this._selectedItems.size();
	var da;
	var j = 0;
	var alreadySelected = false;
	for (var i = 0; i < sz; i++) {
		if (a[i] == treeItem) {
			alreadySelected = true;
		} else {
			a[i]._setSelected(false);
			this._selectedItems.remove(a[i]);
			if (da == null) {
				da = new Array();
			}
			da[j++] = a[i];
		}
	}

	if (da && !skipNotify) {
		this._notifyListeners(DwtEvent.SELECTION, da, DwtTree.ITEM_DESELECTED, null, this._selEv, kbNavEvent);
	}

	if (alreadySelected) { return; }
	this._selectedItems.add(treeItem);

	// Expand all parent nodes, and then set item selected
	this._expandUp(treeItem);
	if (treeItem._setSelected(true, noFocus) && !skipNotify) {
		this._notifyListeners(DwtEvent.SELECTION, [treeItem], DwtTree.ITEM_SELECTED, null, this._selEv, kbNavEvent);
	}
};

DwtTree.prototype.getSelectionCount =
function() {
	return this._selectedItems.size();
};

DwtTree.prototype.addChild = function(child) {

    // HACK: Tree items are added via _addItem. But we need to keep
    // HACK: the original addChild behavior for other controls that
    // HACK: may be added to the tree view.
    if (child.isDwtTreeItem) {
        return;
    }

    DwtComposite.prototype.addChild.apply(this, arguments);
};

/**
 * Adds a separator.
 * 
 */
DwtTree.prototype.addSeparator =
function() {
	var sep = document.createElement("div");
	sep.className = "vSpace";
	this._getContainerElement().appendChild(sep);
};

// Expand parent chain from given item up to root
DwtTree.prototype._expandUp =
function(item) {
	var parent = item.parent;
	while (parent instanceof DwtTreeItem) {
		parent.setExpanded(true);
		parent.setVisible(true);
		parent = parent.parent;
	}
};

DwtTree.prototype._addItem = function(item, index) {

	this._children.add(item, index);
	var thisHtmlElement = this._getContainerElement();
	var numChildren = thisHtmlElement.childNodes.length;
	if (index == null || index > numChildren) {
		thisHtmlElement.appendChild(item.getHtmlElement());
	} else {
		//IE Considers undefined as an illegal value for second argument in the insertBefore method
		thisHtmlElement.insertBefore(item.getHtmlElement(), thisHtmlElement.childNodes[index] || null);
	}
};

DwtTree.prototype._getContainerElement = DwtTree.prototype.getHtmlElement;

DwtTree.prototype.sort =
function(cmp) {
    var children = this.getItems();
    children.sort(cmp);
    var fragment = document.createDocumentFragment();
    AjxUtil.foreach(children, function(item, i){
        fragment.appendChild(item.getHtmlElement());
        item._index = i;
    });
    this._getContainerElement().appendChild(fragment);
};

DwtTree.prototype.removeChild =
function(child) {
	this._children.remove(child);
	this._selectedItems.remove(child);
    var childEl = child.getHtmlElement();
    if (childEl.parentNode) {
        childEl.parentNode.removeChild(childEl);
    }
};

/**
 * Returns the next (or previous) tree item relative to the currently selected
 * item, in top-to-bottom order as the tree appears visually. Items such as
 * separators that cannot be selected are skipped.
 * </p><p>
 * If there is no currently selected item, return the first or last item. If we go past
 * the beginning or end of the tree, return null.
 * </p><p>
 * For efficiency, a flattened list of the visible and selectable tree items is maintained.
 * It will be cleared on any change to the tree's display, then regenerated when it is
 * needed.
 *
 * @param {boolean}	next		if <code>true</code>, return next tree item; otherwise, return previous tree item
 * 
 * @private
 */
DwtTree.prototype._getNextTreeItem =
function(next) {

	var sel = this.getSelection();
	var curItem = (sel && sel.length) ? sel[0] : null;

	var nextItem = null, idx = -1;
	var list = this.getTreeItemList(true);
	if (curItem) {
		for (var i = 0, len = list.length; i < len; i++) {
			var ti = list[i];
			if (ti == curItem) {
				idx = next ? i + 1 : i - 1;
				break;
			}
		}
		nextItem = list[idx]; // if array index out of bounds, nextItem is undefined
	} else {
		// if nothing is selected yet, return the first or last item
		if (list && list.length) {
			nextItem = next ? list[0] : list[list.length - 1];
		}
	}
	return nextItem;
};

DwtTree.prototype._getFirstTreeItem =
function() {
	var a = this.getTreeItemList(true);
	if (a && a.length > 0) {
		return a[0];
	}
	return null;
};

DwtTree.prototype._getLastTreeItem =
function() {
	var a = this.getTreeItemList(true);
	if (a && a.length > 0) {
		return a[a.length - 1];
	}
	return null;
};

/**
 * Creates a flat list of this tree's items, going depth-first.
 *
 * @param {boolean}	visible		if <code>true</code>, only include visible/selectable items
 * @return	{array}	an array of {@link DwtTreeItem} objects
 */
DwtTree.prototype.getTreeItemList =
function(visible) {
	return this._addToList([], visible);
};

DwtTree.prototype._addToList =
function(list, visible, treeItem) {
	if (treeItem && !treeItem._isSeparator &&
		(!visible || (treeItem.getVisible() && treeItem._selectionEnabled))) {

		list.push(treeItem);
	}
	if (!treeItem || !visible || treeItem._expanded) {
		var parent = treeItem || this;
		var children = parent.getChildren ? parent.getChildren() : [];
		for (var i = 0; i < children.length; i++) {
			this._addToList(list, visible, children[i]);
		}
	}
	return list;
};

DwtTree.prototype._deselect =
function(item) {
	if (this._selectedItems.contains(item)) {
		this._selectedItems.remove(item);
		item._setSelected(false);
		this._notifyListeners(DwtEvent.SELECTION, [item], DwtTree.ITEM_DESELECTED, null, this._selEv);
	}
};

DwtTree.prototype._itemActioned =
function(item, ev) {
	if (this._actionedItem && !this._actionedItem.isDisposed()) {
		this._actionedItem._setActioned(false);
		this._notifyListeners(DwtEvent.SELECTION, [this._actionedItem], DwtTree.ITEM_DESELECTED, ev, this._selEv);
	}
	this._actionedItem = item;
	item._setActioned(true);
	this._notifyListeners(DwtEvent.SELECTION, [item], DwtTree.ITEM_ACTIONED, ev, this._selEv);
};

DwtTree.prototype._itemChecked =
function(item, ev) {
	this._notifyListeners(DwtEvent.SELECTION, [item], DwtTree.ITEM_CHECKED, ev, this._selEv);
};

DwtTree.prototype._itemClicked =
function(item, ev) {
	var i;
	var a = this._selectedItems.getArray();
	var numSelectedItems = this._selectedItems.size();
	if (this._style & DwtTree.SINGLE_STYLE || (!ev.shiftKey && !ev.ctrlKey)) {
		if (numSelectedItems > 0) {
			for (i = 0; i < numSelectedItems; i++) {
				a[i]._setSelected(false);
			}
			// Notify listeners of deselection
			this._notifyListeners(DwtEvent.SELECTION, this._selectedItems.getArray(), DwtTree.ITEM_DESELECTED, ev, this._selByClickEv);
			this._selectedItems.removeAll();
		}
		this._selectedItems.add(item);
		if (item._setSelected(true)) {
			this._notifyListeners(DwtEvent.SELECTION, [item], DwtTree.ITEM_SELECTED, ev, this._selByClickEv);
		}
	} else {
		if (ev.ctrlKey) {
			if (this._selectedItems.contains(item)) {
				this._selectedItems.remove(item);
				item._setSelected(false);
				this._notifyListeners(DwtEvent.SELECTION, [item], DwtTree.ITEM_DESELECTED, ev, this._selByClickEv);
			} else {
				this._selectedItems.add(item);
				if (item._setSelected(true)) {
					this._notifyListeners(DwtEvent.SELECTION, [item], DwtTree.ITEM_SELECTED, ev, this._selByClickEv);
				}
			}
		} else {
			// SHIFT KEY
		}
	}
};

DwtTree.prototype._itemDblClicked = 
function(item, ev) {
	this._notifyListeners(DwtEvent.SELECTION, [item], DwtTree.ITEM_DBL_CLICKED, ev, this._selEv);
};

DwtTree.prototype._itemExpanded =
function(item, ev, skipNotify) {
	if (!skipNotify) {
		this._notifyListeners(DwtEvent.TREE, [item], DwtTree.ITEM_EXPANDED, ev, DwtShell.treeEvent);
	}
};

DwtTree.prototype._itemCollapsed =
function(item, ev, skipNotify) {
	var i;
	if (!skipNotify) {
		this._notifyListeners(DwtEvent.TREE, [item], DwtTree.ITEM_COLLAPSED, ev, DwtShell.treeEvent);
	}
	var setSelection = false;
	var a = this._selectedItems.getArray();
	var numSelectedItems = this._selectedItems.size();
	var da;
	var j = 0;
	for (i = 0; i < numSelectedItems; i++) {
		if (a[i]._isChildOf(item)) {
			setSelection = true;
			if (da == null) {
				da = new Array();
			}
			da[j++] = a[i];
			a[i]._setSelected(false);
			this._selectedItems.remove(a[i]);
		}		
	}

	if (da) {
		this._notifyListeners(DwtEvent.SELECTION, da, DwtTree.ITEM_DESELECTED, ev, this._selEv);
	}

	if (setSelection && !this._selectedItems.contains(item)) {
		if (item._setSelected(true)) {
			this._selectedItems.add(item);
			this._notifyListeners(DwtEvent.SELECTION, [item], DwtTree.ITEM_SELECTED, ev, this._selEv);
		}
	}
};

DwtTree.prototype._notifyListeners =
function(listener, items, detail, srcEv, destEv, kbNavEvent) {
	if (this.isListenerRegistered(listener)) {
		if (srcEv) {
			DwtUiEvent.copy(destEv, srcEv);
		}
		destEv.items = items;
		if (items.length == 1) {
			destEv.item = items[0];
		}
		destEv.detail = detail;
		destEv.kbNavEvent = kbNavEvent;
		this.notifyListeners(listener, destEv);
		if (listener == DwtEvent.SELECTION) {
			this.shell.notifyGlobalSelection(destEv);
		}
	}
};
}
if (AjxPackage.define("ajax.dwt.widgets.DwtCheckbox")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Creates a checkbox.
 * @constructor
 * @class
 * This class represents a checkbox.
 * 
 * @param {hash}	params	a hash of parameters
 * @param {DwtComposite}	params.parent	the parent widget
 * @param {DwtCheckbox.TEXT_LEFT|DwtCheckbox.TEXT_RIGHT}       [params.style=DwtCheckbox.TEXT_RIGHT] 	the text style
 * @param {string}       params.name		the input control name (required for IE)
 * @param {string}       params.value     the input control value
 * @param {boolean}       params.checked	the input control checked status (required for IE)
 * @param {string}       params.className	the CSS class
 * @param {constant}       params.posStyle	the positioning style (see {@link Dwt})
 * @param {string}       params.id		an explicit ID to use for the control's HTML element
 * @param {number}       params.index 	the index at which to add this control among parent's children
 * 
 *  @extends		DwtControl
 */
DwtCheckbox = function(params) {
	if (arguments.length == 0) { return; }

	params = Dwt.getParams(arguments, DwtCheckbox.PARAMS);
	params.className = params.className || "DwtCheckbox";

	DwtControl.call(this, params);

	this._textPosition = DwtCheckbox.DEFAULT_POSITION;
	this._initName = params.name;
    this._initValue = params.value;
	this._createHtml();

	this.setSelected(params.checked);
};

DwtCheckbox.prototype = new DwtControl;
DwtCheckbox.prototype.constructor = DwtCheckbox;

DwtCheckbox.prototype.isDwtCheckbox = true;
DwtCheckbox.prototype.isInputControl = true;
DwtCheckbox.prototype.toString = function() { return "DwtCheckbox"; };

//
// Constants
//
DwtCheckbox.PARAMS = [
	"parent",
	"style",
	"name",
	"checked",
	"className",
	"posStyle",
	"id",
	"index",
    "value"
];
/**
 * Defines the "left" text style position.
 */
DwtCheckbox.TEXT_LEFT			= "left";
/**
 * Defines the "right" text style position.
 */
DwtCheckbox.TEXT_RIGHT			= "right";
/**
 * Defines the default text style position.
 */
DwtCheckbox.DEFAULT_POSITION	= DwtCheckbox.TEXT_RIGHT;

//
// Data
//
DwtCheckbox.prototype.TEMPLATE = "dwt.Widgets#DwtCheckbox";

DwtCheckbox.prototype.INPUT_TYPE = 'checkbox';

//
// Public methods
//
DwtCheckbox.prototype.getInputElement =
function() {
	return this._inputEl;
};

DwtCheckbox.prototype._focus =
function() {
	Dwt.addClass(this.getHtmlElement(), DwtControl.FOCUSED);
};

DwtCheckbox.prototype._blur =
function() {
	Dwt.delClass(this.getHtmlElement(), DwtControl.FOCUSED);
};

// listeners

/**
 * Adds a selection listener.
 * 
 * @param	{AjxListener}	listener		the listener
 */
DwtCheckbox.prototype.addSelectionListener =
function(listener) {
	this.addListener(DwtEvent.SELECTION, listener);
};

/**
 * Removes a selection listener.
 * 
 * @param	{AjxListener}	listener		the listener
 */
DwtCheckbox.prototype.removeSelectionListener =
function(listener) {
	this.removeListener(DwtEvent.SELECTION, listener);
};

// properties

/**
 * Sets the enabled state.
 * 
 * @param	{boolean}	enabled		if <code>true</code>, the checkbox is enabled
 */
DwtCheckbox.prototype.setEnabled =
function(enabled) {
	if (enabled != this._enabled) {
		DwtControl.prototype.setEnabled.call(this, enabled);
		this._inputEl.disabled = !enabled;
		var className = enabled ? "Text" : "DisabledText";
		if (this._textElLeft) this._textElLeft.className = className;
		if (this._textElRight) this._textElRight.className = className;
	}
};

/**
 * Sets the selected state.
 * 
 * @param	{boolean}	selected		if <code>true</code>, the checkbox is selected
 */
DwtCheckbox.prototype.setSelected =
function(selected) {
	if (this._inputEl && this._inputEl.checked != selected) {
		this._inputEl.checked = selected;
	}
};

/**
 * Checks if the checkbox is selected state.
 * 
 * @return	{boolean}	<code>true</code> if the checkbox is selected
 */
DwtCheckbox.prototype.isSelected =
function() {
	return this._inputEl && this._inputEl.checked;
};

/**
 * Sets the checkbox text.
 * 
 * @param		{string}	text		the text
 */
DwtCheckbox.prototype.setText =
function(text) {
	if (this._textEl && this._text != text) {
		this._text = text;
		this._textEl.innerHTML = text || "";
	}
};

/**
 * Gets the checkbox text.
 * 
 * @return	{string}	the text
 */
DwtCheckbox.prototype.getText =
function() {
	return this._text;
};

/**
 * Sets the text position.
 * 
 * @param	{DwtCheckbox.TEXT_LEFT|DwtCheckbox.TEXT_RIGHT}		position	the position
 */
DwtCheckbox.prototype.setTextPosition =
function(position) {
	this._textEl = position == DwtCheckbox.TEXT_LEFT ? this._textElLeft : this._textElRight;
	if (this._textPosition != position) {
		this._textPosition = position;
		if (this._textElLeft) this._textElLeft.innerHTML = "";
		if (this._textElRight) this._textElRight.innerHTML = "";
		this.setText(this._text);
	}
};

/**
 * Gets the text position.
 * 
 * @return	{DwtCheckbox.TEXT_LEFT|DwtCheckbox.TEXT_RIGHT}		the position
 */
DwtCheckbox.prototype.getTextPosition =
function() {
	return this._textPosition;
};

/**
 * Sets the value.
 * 
 * @param	{string}		value		the value
 */
DwtCheckbox.prototype.setValue =
function(value) {
    var object = this._inputEl || this;
	if (object.value !== value) {
        object.value = value;
    }
};

/**
 * Gets the value.
 * 
 * @return		{string}		the value
 */
DwtCheckbox.prototype.getValue =
function() {
    var object = this._inputEl || this;
	return object.value != null ? object.value : this.getText();
};

/**
 * Gets the input element.
 * 
 * @return		{Element}		the element
 */
DwtCheckbox.prototype.getInputElement =
function() {
	return this._inputEl;
};

//
// DwtControl methods
//

DwtCheckbox.prototype.setToolTipContent = function(content) {
    if (content && !this.__mouseEventsSet) {
        // NOTE: We need mouse events in order to initiate tooltips on hover.
        // TODO: This should be done transparently in DwtControl for all
        // TODO: controls with tooltips.
        this.__mouseEventsSet = true;
        this._setMouseEvents();
    }
    DwtControl.prototype.setToolTipContent.apply(this, arguments);
};

//
// Protected methods
//

/**
 * The input field inherits the id for accessibility purposes.
 * 
 * @private
 */
DwtCheckbox.prototype._replaceElementHook =
function(oel, nel, inheritClass, inheritStyle) {
	nel = this.getInputElement();
	DwtControl.prototype._replaceElementHook.call(this, oel, nel, inheritClass, inheritStyle);
	if (oel.id) {
		this.setHtmlElementId(oel.id+"_control");
		nel.id = oel.id;
		if (this._textEl) {
			this._textEl.setAttribute(AjxEnv.isIE ? "htmlFor" : "for", oel.id);
		}
	}
};

//
// Private methods
//

DwtCheckbox.prototype._createHtml =
function(templateId) {
	var data = { id: this._htmlElId };
	this._createHtmlFromTemplate(templateId || this.TEMPLATE, data);
};

DwtCheckbox.prototype._createHtmlFromTemplate =
function(templateId, data) {
	// NOTE: If  you don't set the name and checked status when
	//       creating checkboxes and radio buttons on IE, they will
	//       not take the first programmatic value. So we pass in
	//       the init values from the constructor.
	data.name = this._initName || this._htmlElId;
    data.value = this._initValue;
	data.type = this.INPUT_TYPE;
	DwtControl.prototype._createHtmlFromTemplate.call(this, templateId, data);
	this._inputEl = document.getElementById(data.id+"_input");
	if (this._inputEl) {
		var keyboardMgr = DwtShell.getShell(window).getKeyboardMgr();
		var handleFocus = AjxCallback.simpleClosure(keyboardMgr.grabFocus, keyboardMgr, this.getInputElement());
		Dwt.setHandler(this._inputEl, DwtEvent.ONFOCUS, handleFocus);
		Dwt.setHandler(this._inputEl, DwtEvent.ONCLICK, DwtCheckbox.__handleClick);
		this.setFocusElement();
	}
	this._textElLeft = document.getElementById(data.id+"_text_left");
	this._textElRight = document.getElementById(data.id+"_text_right");
	this.setTextPosition(this._textPosition);
};

//
// Private functions
//

DwtCheckbox.__handleClick =
function(evt) {
	var event = DwtUiEvent.getEvent(evt);
	var target = DwtUiEvent.getTarget(event);

	var selEv = DwtShell.selectionEvent;
	DwtUiEvent.copy(selEv, event);
	selEv.item = this;
	selEv.detail = target.checked;

	var checkbox = DwtControl.findControl(target);
	checkbox.setSelected(target.checked);
	checkbox.focus();
	checkbox.notifyListeners(DwtEvent.SELECTION, selEv);
};
}
if (AjxPackage.define("ajax.dwt.widgets.DwtRadioButton")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Creates a radio button.
 * @constructor
 * @class
 * This class implements a radio button.
 * 
 * @param {hash}	params	a hash of parameters
 * @param  {DwtComposite}     params.parent	the parent widget
 * @param  {constant}     params.style 	the text style. May be one of: {@link DwtCheckbox.TEXT_LEFT} or
 * 									{@link DwtCheckbox.TEXT_RIGHT} arithimatically or'd (|) with one of:
 * 									{@link DwtCheckbox.ALIGN_LEFT}, {@link DwtCheckbox.ALIGN_CENTER}, or
 * 									{@link DwtCheckbox.ALIGN_LEFT}.
 * 									The first determines were in the checkbox the text will appear
 * 									(if set), the second determine how the content of the text will be
 * 									aligned. The default value for this parameter is: 
 * 									{@link DwtCheckbox.TEXT_LEFT} | {@link DwtCheckbox.ALIGN_CENTER}
 * @param  {string}     params.name		the input control name (required for IE)
 * @param  {string}     params.value     the input control value.
 * @param  {boolean}     params.checked	the input control checked status (required for IE)
 * @param  {string}     params.className	the CSS class
 * @param  {constant}     params.posStyle	the positioning style (see {@link DwtControl})
 * @param  {string}     params.id		an explicit ID to use for the control's HTML element
 * @param  {number}     params.index 	the index at which to add this control among parent's children
 * 
 * @extends	DwtCheckbox
 */
DwtRadioButton = function(params) {
	if (arguments.length == 0) { return; }
	params = Dwt.getParams(arguments, DwtRadioButton.PARAMS);
	params.className = params.className || "DwtRadioButton";
	DwtCheckbox.call(this, params);
}

DwtRadioButton.PARAMS = DwtCheckbox.PARAMS;

DwtRadioButton.prototype = new DwtCheckbox;
DwtRadioButton.prototype.constructor = DwtRadioButton;

DwtRadioButton.prototype.isDwtRadioButton = true;
DwtRadioButton.prototype.isInputControl = true;
DwtRadioButton.prototype.toString = function() { return "DwtRadioButton"; };

//
// Data
//

DwtRadioButton.prototype.INPUT_TYPE = 'radio';
}
if (AjxPackage.define("ajax.dwt.widgets.DwtRadioButtonGroup")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */


/**
 * Creates a radio button group.
 * @constructor
 * @class
 * This class implements a group of radio buttons
 *
 * @param {hash} [radios] 	a hash whose keys are the ids of the radio button elements
 * 		and whose values are the values associated with those buttons
 * @param {string} [selectedId]	the id of the button to select initially
 * 
 * TODO: this really should be a DwtComposite
 * 
 * @private
 */
DwtRadioButtonGroup = function(radios, selectedId) {
	this._radios = {};
	this._radioButtons = {};
	this._values = {};
	this._value2id = {};
	this._eventMgr = new AjxEventMgr();
	
	for (var id in radios) {
		this.addRadio(id, radios[id], (id == selectedId));
	}
};

DwtRadioButtonGroup.prototype.toString =
function() {
	return "DwtRadioButtonGroup";
};

//
// Data
//

DwtRadioButtonGroup.prototype._enabled = true;
DwtRadioButtonGroup.prototype._visible = true;

//
// Public methods
//

DwtRadioButtonGroup.prototype.addSelectionListener = function(listener) {
	return this._eventMgr.addListener(DwtEvent.SELECTION, listener);
};

DwtRadioButtonGroup.prototype.removeSelectionListener = function(listener) {
	return this._eventMgr.removeListener(DwtEvent.SELECTION, listener);
};

DwtRadioButtonGroup.prototype.setEnabled = function(enabled) {
	this._enabled = enabled;
	for (var id in this._radios) {
		this._radios[id].disabled = !enabled;
	}
};
DwtRadioButtonGroup.prototype.isEnabled = function() {
	return this._enabled;
};

DwtRadioButtonGroup.prototype.setVisible = function(visible) {
	this._visible = visible;
	for (var id in this._radioButtons) {
		this._radioButtons[id].setVisible(visible);
	}
	for (var id in this._radios) {
		Dwt.setVisible(this._radios[id], visible);
	}
};
DwtRadioButtonGroup.prototype.isVisible = function() {
	return this._visible;
};

DwtRadioButtonGroup.prototype.addRadio =
function(id, radioButtonOrValue, selected) {
	var isRadioButton = radioButtonOrValue instanceof DwtRadioButton;
	var radioButton = isRadioButton ? radioButtonOrValue : null;
	var value = radioButton ? radioButton.getValue() : radioButtonOrValue;

	this._values[id] = value;
	this._value2id[value] = id;
	var element = document.getElementById(id);
	this._radios[id] = element;
	this._radioButtons[id] = radioButton;
	var handler = AjxCallback.simpleClosure(this._handleClick, this);
	Dwt.setHandler(element, DwtEvent.ONCLICK, handler);
   	element.checked = selected;
    if (selected) {
    	this._selectedId = id;
    }
};

DwtRadioButtonGroup.prototype.getRadioByValue = function(value) {
	var id = this._value2id[value];
	return this._radios[id];
};

DwtRadioButtonGroup.prototype.getRadioButtonByValue = function(value) {
	var id = this._value2id[value];
	return this._radioButtons[id];
};

DwtRadioButtonGroup.prototype.setSelectedId =
function(id, skipNotify) {
	if (id != this._selectedId) {
		var el = document.getElementById(id);
		if (!el) return;
		el.checked = true;
		this._selectedId = id;
		if (!skipNotify) {
			var selEv = DwtShell.selectionEvent;
			selEv.reset();
			this._notifySelection(selEv);
		}
	}
};

DwtRadioButtonGroup.prototype.setSelectedValue =
function(value, skipNotify) {
	var id = this._valueToId(value);
	this.setSelectedId(id, skipNotify);
};

DwtRadioButtonGroup.prototype.getSelectedId =
function() {
	return this._selectedId;
};

DwtRadioButtonGroup.prototype.getSelectedValue =
function() {
	return this._values[this._selectedId];
};

DwtRadioButtonGroup.prototype.getValue =
function() {
	return this.getSelectedValue();
};

DwtRadioButtonGroup.prototype.getData =
function(key) {
	var selectedRadio = !AjxUtil.isUndefined(this._selectedId) && this._radioButtons[this._selectedId];
	if (selectedRadio) {
		return selectedRadio.getData(key);
	}
	// return undefined;
}

//
// Protected methods
//

DwtRadioButtonGroup.prototype._valueToId =
function(value) {
	for (var id in this._values) {
		if (this._values[id] == value) {
			return id;
		}
		if (value === true && this._values[id] == "true") {
			return id;
		}
		if (value === false && (this._values[id] == "false" || this._values[id] == "")) {
			return id;
		}
	}
	return null;
};

DwtRadioButtonGroup.prototype._notifySelection = 
function(selEv) {
    selEv.item = this;
    selEv.detail = { id: this._selectedId, value: this._values[this._selectedId] };
    this._eventMgr.notifyListeners(DwtEvent.SELECTION, selEv);
};

DwtRadioButtonGroup.prototype._handleClick = 
function(event) {
	event = DwtUiEvent.getEvent(event);

	var target = DwtUiEvent.getTarget(event);
	if (target && target.nodeName.match(/label/i)) {
        target = document.getElementById(target.getAttribute(AjxEnv.isIE ? "htmlFor" : "for"));
    }

	var id = target.id;
	// NOTE: When you use the arrows on radio button groups in FF,
	//       the radio button that is being unselected is the target
	//       of the event. So we need to check to see if this target
	//       is the one that is checked.
	if (!target.checked) {
		for (id in this._radios) {
			if (this._radios[id].checked) {
				break;
			}
		}
	}
	if (id != this._selectedId) {
		this._selectedId = id;
	    var selEv = DwtShell.selectionEvent;
	    DwtUiEvent.copy(selEv, event);
		this._notifySelection(selEv);
	}
};
}
if (AjxPackage.define("ajax.dwt.widgets.DwtForm")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Creates a form.
 * @class
 * This class represents a form.
 * 
 * @param	{hash}	params		a hash of parameters
 * 
 * @extends		DwtComposite
 * 
 * @private
 */
DwtForm = function(params) {
	if (arguments.length == 0) return;
	params = Dwt.getParams(arguments, DwtForm.PARAMS);
	params.className = params.className || "DwtForm";
	DwtComposite.apply(this, arguments);
	this.setScrollStyle(DwtControl.SCROLL);

	// data
	this._tabGroup = new DwtTabGroup(this._htmlElId);

	// context
	this._context = {
		set: AjxCallback.simpleClosure(this.set, this),
		get: AjxCallback.simpleClosure(this.get, this)
	};

	// construct form
	this._dirty = {};
	this._ignore = {};
	this._invalid = {};
	this._errorMessages = {};
	this.setModel(params.model);
	this.setForm(params.form);
	this.reset();
};
DwtForm.prototype = new DwtComposite;
DwtForm.prototype.constructor = DwtForm;

DwtForm.prototype.toString = function() {
	return "DwtForm";
};

//
// Constants
//

DwtForm.PARAMS = DwtControl.PARAMS.concat("form", "model");

//
// Public methods
//

/**
 * Sets the value.
 * 
 * @param	{string}	id	the id
 * @param	{string}	value		the value
 * @param	{boolean}	force		if <code>true</code>, to force update
 */
DwtForm.prototype.setValue = function(id, value, force) {
	if (typeof id != "string") id = String(id);
	if (id.match(/\./) || id.match(/\[/)) {
		var parts = id.replace(/\[(\d+)\](\.)?/,".$1$2").split(".");
		var control = this.getControl(parts[0]);
		if (Dwt.instanceOf(control, "DwtForm")) {
			control.setValue(parts.slice(1).join("."), value, force);
		}
		return;
	}
	var item = this._items[id];
	if (!item) return;
	if (!force && value == item.value) return;
	this._setModelValue(id, value);
	this._setControlValue(id, value);
};

/**
 * Gets the value.
 * 
 * @param	{string}	id	the id
 * @param	{string}	defaultValue		the default value
 * @return	{string}	the value
 */
DwtForm.prototype.getValue = function(id, defaultValue) {
	if (typeof id !== "string") {
		id = String(id);
	}
	if (id.match(/\./) || id.match(/\[/)) {
		var parts = id.replace(/\[(\d+)\](\.)?/,".$1$2").split(".");
		var control = this.getControl(parts[0]);
		if (Dwt.instanceOf(control, "DwtForm")) {
			return control.getValue(parts.slice(1).join("."));
		}
		return null;
	}
	var item = this._items[id];
	if (!item) {
		return;
	}
	if (item.getter) {
		return this._call(item.getter) || defaultValue;
	}
	var value = this._getControlValue(id);
    if (value == null) {
		value = item.value;
	}

    //added <|| ""> because ... if value="" than it always returns defaultValue which could be undefined.
	return value || defaultValue || "";
};

/**
 * Gets the control for the item.
 * 
 * @param	{string}	id		the id
 * @return	{DwtControl}	the control
 */
DwtForm.prototype.getControl = function(id) {
	if (typeof id != "string") id = String(id);
	var item = this._items[id];
	return item && item.control;
};

/**
 * Checks if the id is relevant (meaning: is visible and is enabled).
 * 
 * @param	{string}	id 		the id
 * @return	 {boolean}	<code>true</code> if the item is relevant
 */
DwtForm.prototype.isRelevant = function(id) {
	return this.isVisible(id) && this.isEnabled(id);
};

DwtForm.prototype.getTabGroupMember = function() {
	return this._tabGroup;
};

// control methods

/**
 * Sets the label.
 * 
 * @param	{string}	id 		the id
 * @param	{string}	label 		the label
 */
DwtForm.prototype.setLabel = function(id, label) {
	var item = this._items[id];
	if (!item) return;
	if (label == this.getLabel(id)) return;
	var control = item.control;
	if (!control) return;
	if (control.setLabel) { control.setLabel(label); return; }
	if (control.setText) { control.setText(label); return; }
};

/**
 * Sets the image.
 *
 * @param	{string}	id 		the id
 * @param	{string}	image 	the image
 * @param	{string}	altText	alternate text for non-visual users
 */
DwtForm.prototype.setImage = function(id, image, altText) {
	var item = this._items[id];
	if (!item) {
		return;
	}
	var control = item.control;
	if (!control) {
		return;
	}
	control.setImage(image, null, altText);
};


/**
 * Gets the label.
 * 
 * @param	{string}	id 		the id
 * @return	{string}	the label
 */
DwtForm.prototype.getLabel = function(id) {
	var item = this._items[id];
	var control = item && item.control;
	if (control) {
		if (control.getLabel) return control.getLabel();
		if (control.getText) return control.getText();
	}
	return "";
};

DwtForm.prototype.setVisible = function(id, visible) {
	// set the form's visibility
	if (arguments.length == 1) {
		DwtComposite.prototype.setVisible.call(this, arguments[0]);
		return;
	}
	// set control's visibility
	var item = this._items[id];
	var control = item && item.control;
	if (!control) return;
	if (control.setVisible) {
		control.setVisible(visible);
	}
	else {
		Dwt.setVisible(control, visible);
	}
	// if there's a corresponding "*_row" element
	var el = document.getElementById([this._htmlElId, id, "row"].join("_"));
	if (el) {
		Dwt.setVisible(el, visible);
	}
};

DwtForm.prototype.isVisible = function(id) {
	// this form's visibility
	if (arguments.length == 0) {
		return DwtComposite.prototype.isVisible.call(this);
	}
	// control's visibility
	var item = this._items[id];
	var control = item && item.control;
	if (!control) return false;
	if (control.getVisible) return control.getVisible();
	if (control.isVisible) return control.isVisible();
	return  Dwt.getVisible(control);
};

/**
 * Sets the enabled flag.
 * 
 * @param	{string}	id 		the id
 * @param	{boolean}	enabled		if <code>true</code>, the item is enabled
 */
DwtForm.prototype.setEnabled = function(id, enabled) {
	// set the form enabled
	if (arguments.length == 1) {
		DwtComposite.prototype.setEnabled.call(this, arguments[0]);
		return;
	}
	// set the control enabled
	var item = this._items[id];
	var control = item && item.control;
	if (!control) return;
	if (control.setEnabled) {
		control.setEnabled(enabled);
	}
	else {
		control.disabled = !enabled;
	}
};

/**
 * Checks if the item is enabled.
 * 
 * @param	{string}	id 		the id
 * @return	{boolean}	<code>true</code> if the item is enabled
 */
DwtForm.prototype.isEnabled = function(id) {
	// this form enabled?
	if (arguments.length == 0) {
		return DwtComposite.prototype.isEnabled.call(this);
	}
	// the control enabled?
	var item = this._items[id];
	var control = item && item.control;
	if (!control) return false;
	if (control.isEnabled) return control.isEnabled();
	if (control.getEnabled) return control.getEnabled();
	return  !control.disabled;
};

/**
 * Sets the valid flag.
 * 
 * @param	{string}	id 		the id
 * @param	{boolean}	valid		if <code>true</code>, the item is valid
 */
DwtForm.prototype.setValid = function(id, valid) {
	if (typeof(id) == "boolean") {
		valid = arguments[0];
		for (id in this._items) {
			this.setValid(id, valid);
		}
		return;
	}
	if (valid) {
		delete this._invalid[id]; 
	}
	else {
		this._invalid[id] = true;
	}
};

/**
 * Checks if the item is valid.
 * 
 * @param	{string}	id 		the id
 * @return	{boolean}	<code>true</code> if the item is valid
 */
DwtForm.prototype.isValid = function(id) {
	if (arguments.length == 0 || AjxUtil.isUndefined(id)) {
		for (var id in this._invalid) {
			return false;
		}
		return true;
	}
	return !(id in this._invalid);
};

/**
 * Sets the error message.
 * 
 * @param	{string}	id 		the id
 * @param	{string}	message	the message
 */
DwtForm.prototype.setErrorMessage = function(id, message) {
	if (!id || id == "") {
		this._errorMessages = {};
		return;
	}
	if (!message) {
		delete this._errorMessages[id]; 
	} else {
		this._errorMessages[id] = message;
	}
};

/**
 * Gets the error message.
 * 
 * @param	{string}	id 		the id
 * @return	{string|array}	the message(s)
 */
DwtForm.prototype.getErrorMessage = function(id) {
	if (arguments.length == 0) {
		var messages = {};
		for (var id in this._invalid) {
			messages[id] = this._errorMessages[id];
		}
		return messages;
	}
	return this._errorMessages[id];
};

DwtForm.prototype.getInvalidItems = function() {
	return AjxUtil.keys(this._invalid);
};

DwtForm.prototype.setDirty = function(id, dirty, skipNotify) {
	if (typeof id == "boolean") {
		dirty = arguments[0];
		for (id in this._items) {
			this.setDirty(id, dirty, true);
		}
		if (!skipNotify && this._ondirty) {
			this._call(this._ondirty, ["*"]);
		}
		return;
	}
	if (dirty) {
		this._dirty[id] = true;
	}
	else {
		delete this._dirty[id]; 
	}
	if (!skipNotify && this._ondirty) {
		var item = this._items[id];
		if (!item.ignore || !this._call(item.ignore)) {
			this._call(this._ondirty, [id]);
		}
	}
};
DwtForm.prototype.isDirty = function(id) {
	if (arguments.length == 0) {
		for (var id in this._dirty) {
			var item = this._items[id];
			if (item.ignore && this._call(item.ignore)) {
				continue;
			}
			return true;
		}
		return false;
	}
	var item = this._items[id];
	return item.ignore && this._call(item.ignore) ? false : id in this._dirty;
};
DwtForm.prototype.getDirtyItems = function() {
	// NOTE: This avoids needing a closure
	DwtForm.__acceptDirtyItem.form = this;
	return AjxUtil.keys(this._dirty, DwtForm.__acceptDirtyItem);
};
DwtForm.__acceptDirtyItem = function(id) {
	var form = arguments.callee.form;
	var item = form._items[id];
	return !item.ignore || !form._call(item.ignore);
};

DwtForm.prototype.setIgnore = function(id, ignore) {
	if (typeof id == "boolean") {
		this._ignore = {};
		return;
	}
	if (ignore) {
		this._ignore[id] = true;
		return;
	}
	delete this._ignore[id];
};
DwtForm.prototype.isIgnore = function(id) {
	return id in this._ignore;
};

// convenience control methods

DwtForm.prototype.set = function(id, value) {
	this.setValue(id, value, true);
	this.update();
};
DwtForm.prototype.get = DwtForm.prototype.getValue;

// properties

DwtForm.prototype.setModel = function(model, reset) {
	this._context.model = this.model = model;
};

DwtForm.prototype.setForm = function(form) {
	this._context.form = this.form = form;
	this._createHtml(form.template);
};

// form maintenance

DwtForm.prototype.validate = function(id) {
	if (arguments.length == 0) {
		this.setValid(true);
		for (var id in this._items) {
			this._validateItem(id);
		}
		return this.isValid();
	}
	return this._validateItem(id);
};

DwtForm.prototype._validateItem = function(id) {
	if (!id) return true;
	var item = this._items[id];
	if (!item) return true;
	try {
		var value = this.getValue(id);
		var outcome = item.validator ? item.validator(value) : ((item.control && item.control.validator) ? item.control.validator(value) : true);
		// the validator may return false to signify that the validation failed (but preferably throw an error with a message)
		// it may return true to signify that the field validates
		// It also may return a string or hash (truthy) that we may put into the value field (for normalization of data; e.g. if 13/10/2009 is transformed to 1/10/2010 by the validator)
		this.setValid(id, Boolean(outcome) || outcome === "");
		if (AjxUtil.isString(outcome) || AjxUtil.isObject(outcome)) {
			this._setControlValue(id, outcome); // Set display value
			item.value = item.setter ? this._call(item.setter, [outcome]) : outcome; // Set model value
			var dirty = !Boolean(this._call(item.equals, [item.value,item.ovalue]));
			this.setDirty(id, dirty);
		}
	}
	catch (e) {
		this.setErrorMessage(id, AjxUtil.isString(e) ? e : e.message);
		this.setValid(id, false);
	}
	return !(id in this._invalid);
};

DwtForm.prototype.reset = function(useCurrentValues) {
	// init state
	this._dirty = {};
	this._ignore = {};
	this._invalid = {};
	for (var id in this._items) {
		var item = this._items[id];
		if (item.control instanceof DwtForm) {
			item.control.reset(useCurrentValues);
		}
		var itemDef = this._items[id].def;
		if (!itemDef) continue;
		this._initControl(itemDef, useCurrentValues);
	}
	// update values
	this.update();
	for (var id in this._items) {
		var item = this._items[id];
		item.ovalue = item.value;
	}
	// clear state
	this.validate();
    this.setDirty(false);
	// call handler
	if (this._onreset) {
		this._call(this._onreset);
	}
};

DwtForm.prototype.update = function() {
	// update all the values first
	for (var id in this._items) {
		var item = this._items[id];
		if (item.control instanceof DwtForm) {
			item.control.update();
		}
		if (item.getter) {
			this.setValue(id, this._call(item.getter));
		}
	}
	// now set visible/enabled/ignore based on values
	for (var id in this._items) {
		var item = this._items[id];
		if (item.visible) {
			this.setVisible(id, Boolean(this._call(item.visible)));
		}
		if (item.enabled) {
			this.setEnabled(id, Boolean(this._call(item.enabled)));
		}
		if (item.ignore) {
			this.setIgnore(id, Boolean(this._call(item.ignore)));
		}
	}
	// call handler
	if (this._onupdate) {
		this._call(this._onupdate);
	}
};

//
// Protected methods
//

DwtForm.prototype._setModelValue = function(id, value) {
	var item = this._items[id];
	item.value = item.setter ? this._call(item.setter, [value]) : value;
	var dirty = !Boolean(this._call(item.equals, [item.value,item.ovalue]));
	this.setDirty(id, dirty);
	this.validate(id);
	return dirty;
};

DwtForm.prototype._setControlValue = function(id, value) {
	var control = this._items[id].control;
	if (control) {
		// TODO: display value
		if (control instanceof DwtCheckbox || control instanceof DwtRadioButton) {
			control.setSelected(value);
			return;
		}
		if (control instanceof DwtMenuItem && control.isStyle(DwtMenuItem.CHECK_STYLE)) {
			control.setChecked(value, true);
			return;
		}
		if (control.setSelectedValue) { control.setSelectedValue(value); return; }
		if (control.setValue) { control.setValue(value); return; }
		if (control.setText && !(control instanceof DwtButton)) { control.setText(value); return; }
		if (!(control instanceof DwtControl)) {
			// TODO: support other native form elements like select
			if (control.type == "checkbox" || control == "radio") {
				control.checked = value;
			}
			else {
				// TODO: handle error setting form input value
				control.value = value;
			}
			return;
		}
	}
};
DwtForm.prototype._getControlValue = function(id) {
	var control = this._items[id].control;
	if (control) {
		if (control instanceof DwtCheckbox || control instanceof DwtRadioButton) {
			return control.isSelected();
		}
		if (control.getSelectedValue) {
			return control.getSelectedValue();
		}
		if (control.getValue) {
			return control.getValue();
		}
		if (control.getText && !(control instanceof DwtButton)) {
			return control.getText();
		}
		if (!(control instanceof DwtControl)) {
			if (control.type == "checkbox" || control == "radio") return control.checked;
			return control.value;
		}
	}
};

DwtForm.prototype._deleteItem = function(id) {
	delete this._items[id];
	delete this._dirty[id];
	delete this._invalid[id];
	delete this._ignore[id];
};

// utility

DwtForm.prototype._call = function(func, args) {
	if (func) {
		if (args) return func.apply(this, args);
		// NOTE: Hack for IE which barfs with null args on apply
		return func.call(this);
	}
};

// html creation

DwtForm.prototype._createHtml = function(templateId) {
	this._createHtmlFromTemplate(templateId || this.TEMPLATE, { id: this._htmlElId });
};

DwtForm.prototype._createHtmlFromTemplate = function(templateId, data) {
	DwtComposite.prototype._createHtmlFromTemplate.apply(this, arguments);

	// initialize state
	var tabIndexes = [];
	this._items = {};
	this._tabGroup.removeAllMembers();
	this._onupdate = null;
	this._onreset = null;
	this._ondirty = null;

	// create form
	var form = this.form;
	if (form && form.items) {
		// create controls
		this._registerControls(form.items, null, tabIndexes);
		// create handlers
		this._onupdate = DwtForm.__makeFunc(form.onupdate);
		this._onreset = DwtForm.__makeFunc(form.onreset);
		this._ondirty = DwtForm.__makeFunc(form.ondirty);
	}

	// add links to list of tabIndexes
	var links = this.getHtmlElement().getElementsByTagName("A");
	for (var i = 0; i < links.length; i++) {
		var link = links[i];
		if (!link.href || link.getAttribute("notab") == "true") continue;
        var controlId = link.id && link.id.substr(this.getHTMLElId().length+1);
		if (this._items[controlId]) continue;
		tabIndexes.push({
			tabindex:	link.getAttribute("tabindex") || Number.MAX_VALUE,
			control:	link
		});
	}

	// add controls to tab group
	tabIndexes.sort(DwtForm.__byTabIndex);
	for (var i = 0; i < tabIndexes.length; i++) {
		var control = tabIndexes[i].control;
		var member = (control.getTabGroupMember && control.getTabGroupMember()) || control;
		this._tabGroup.addMember(member);
	}
};

DwtForm.prototype._registerControls = function(itemDefs, parentDef,
                                               tabIndexes, params,
                                               parent, defaultType) {
	for (var i = 0; i < itemDefs.length; i++) {
		this._registerControl(itemDefs[i], parentDef, tabIndexes, params, parent, defaultType);
	}
};

DwtForm.prototype._registerControl = function(itemDef, parentDef,
                                              tabIndexes, params,
                                              parent, defaultType) {
	// create item entry
	var id = itemDef.id || [this._htmlElId, Dwt.getNextId()].join("_");
	var item = this._items[id] = {
		id:			id, // for convenience
		def:		itemDef,
		parentDef:	parentDef,
		equals:		DwtForm.__makeFunc(itemDef.equals) || DwtForm.__equals,
		getter:		DwtForm.__makeGetter(itemDef),
		setter:		DwtForm.__makeSetter(itemDef),
		value:		itemDef.value,
		visible:	DwtForm.__makeFunc(itemDef.visible),
		enabled:	DwtForm.__makeFunc(itemDef.enabled),
		validator:	DwtForm.__makeFunc(itemDef.validator),
		ignore:		DwtForm.__makeFunc(itemDef.ignore),
		control:	itemDef.control
	};
	// NOTE: This is used internally for indexing of rows
	if (itemDef.aka) {
		this._items[id].aka = itemDef.aka;
		this._items[itemDef.aka] = item;
	}

	// is control already created?
	var control = item.control;
	if (control) {
		return control;
	}

	// create control
	parent = parent || this;
	var type = itemDef.type = itemDef.type || defaultType;
	var isMenu = (parentDef && parentDef.menu == itemDef);
	var element = document.getElementById([parent._htmlElId,id].join("_"));
	if (Dwt.instanceOf(type, "DwtRadioButtonGroup")) {
		// create control
		control = new window[type]({});
		item.control = control;

		// add children
		var nparams = {
			name:  [parent._htmlElId, id].join("_"),
			value: itemDef.value
		};
		if (itemDef.items) {
			for (var i = 0; i < itemDef.items.length; i++) {
				var radioItemDef = itemDef.items[i];
				var checked = radioItemDef.checked || radioItemDef.value == itemDef.value;
				var radio = this._registerControl(radioItemDef, itemDef, tabIndexes, nparams, parent, "DwtRadioButton");
				this._items[radioItemDef.id].value = checked;
				if (radio) {
					control.addRadio(radio.getInputElement().id, radio, checked);
					// handlers
					var handler = DwtForm.__makeFunc(radioItemDef.onclick || itemDef.onclick);
					radio.addSelectionListener(new AjxListener(this, this._radio2group2model, [radioItemDef.id, id, handler]));
					// HACK: Work around fact that the DwtRadioButtonGroup overwrites
					//       the radio button input element's onclick handler.
					DwtForm.__hack_fixRadioButtonHandler(radio);
				}
			}
		}
	}
	else if (type) {
		if (Dwt.instanceOf(type, "DwtInputField")) {
			item.value = item.value || "";
		}
		if (Dwt.instanceOf(type, "DwtFormRows")) {
		    item.equals = DwtFormRows.__equals;
		}
		if (element || isMenu) {
			control = item.control = this._createControl(itemDef, parentDef, tabIndexes, params, parent, defaultType);
		}
	}
	else if (element) {
		this._attachElementHandlers(itemDef, parentDef, tabIndexes, parent, element);
		control = item.control = element;
		if (itemDef.items) {
			this._registerControls(itemDef.items, itemDef, tabIndexes, null, parent, null);
		}
	}
	if (element && control instanceof DwtControl) {
		control.replaceElement(element);
	}
	if (element && control instanceof DwtInputField) {
		control.getInputElement().id += "_input";
		control.setHandler(DwtEvent.ONPASTE, this._onPaste.bind(this, id));
	}

	// add to list of tab indexes
	if (itemDef.notab == null) {
		itemDef.notab = element && element.getAttribute("notab") == "true";
	}
	if (tabIndexes && control && !itemDef.notab && !(control instanceof DwtRadioButtonGroup)) {
		tabIndexes.push({
			tabindex:	(element && element.getAttribute("tabindex")) || Number.MAX_VALUE,
			control:	control
		});
	}

	// clean up
	if (control instanceof DwtListView) {
		item.getter = item.getter || AjxCallback.simpleClosure(this.__list_getValue, this, id);
		item.setter = item.setter || AjxCallback.simpleClosure(this.__list_setValue, this, id);
	}

	// return control
	return control;
};


DwtForm.prototype._onPaste = function(id, evt) {
	// Delay the value processing - the paste text will not be applied to the control till after this event
	AjxTimedAction.scheduleAction(new AjxTimedAction(this, this._applyPasteInput, [id]), 100);
};

DwtForm.prototype._applyPasteInput = function(id, value) {
	this._setModelValue(id, this._getControlValue(id));
}


DwtForm.prototype._attachElementHandlers = function(itemDef, parentDef, tabIndexes, parent, element) {
	var id = itemDef.id;
	var name = element.nodeName.toLowerCase();
	var type = element.type;
	if (type == "checkbox" || type == "radio") {
		var parentId;
		if (type == "radio") {
			parentId = element.name;
			if (!this._items[parentId]) this._items[parentId] = { id: parentId };
			if (element.checked) {
				this._items[parentId].value = element.value;
			}
		}
		// checked
		var onclick = element.onclick ;
		var handler = DwtForm.__makeFunc(itemDef.onclick);
		element.onclick = AjxCallback.simpleClosure(this._htmlInput_checked, this, id, parentId, handler, onclick);
	}
	else if (name == "select") {
		// map selectedIndex to value of option
		var onchange = element.onchange;
		var handler = DwtForm.__makeFunc(itemDef.onchange);
		element.onchange = AjxCallback.simpleClosure(this._htmlSelect_selectedIndex, this, id, handler, onchange);
	}
	else if (name == "button" || name == "a" || 
	         type == "button" || type == "reset" || type == "submit") {
		// checked
		var onclick = element.onclick ;
		var handler = DwtForm.__makeFunc(itemDef.onclick);
		element.onclick = AjxCallback.simpleClosure(this._htmlElement, this, id, handler, onclick);
	}
	else if (name == "textarea" || name == "input") { // type == "text" ||  || type == "file" || type == "password") {
		// value
		var onchange = element.onchange;
		var handler = DwtForm.__makeFunc(itemDef.onchange);
		element.onchange = AjxCallback.simpleClosure(this._htmlInput_value, this, id, handler, onchange);
	}
	// TODO: attach other handlers
	return element;
};

DwtForm.prototype._createControl = function(itemDef, parentDef,
                                            tabIndexes, params,
                                            parent, defaultType) {
	var id = itemDef.id || [this._htmlElId, Dwt.getNextId()].join("_");
	var type = itemDef.type = itemDef.type || defaultType;
	params = params ? AjxUtil.createProxy(params) : {};
	params.id = params.id || [this._htmlElId, id].join("_");
	params.parent = parent || this;
	params.template = itemDef.template || params.template;
	params.className = itemDef.className || params.className;

	// constructor params for radio buttons
	var isRadioButton = Dwt.instanceOf(type, "DwtRadioButton");
	var isCheckBox = Dwt.instanceOf(type, "DwtCheckbox");
	if (isRadioButton || isCheckBox) {
		params.name = itemDef.name || params.name;
        params.value = itemDef.value || params.value;
		params.checked = itemDef.checked != null ? itemDef.checked : params.checked;
	}

	// constructor params for input fields
	var isTextField = Dwt.instanceOf(type, "DwtInputField");
	if (isTextField) {
		params.type = itemDef.password ? DwtInputField.PASSWORD : null;
		params.size = itemDef.cols;
		params.rows = itemDef.rows;
	}

	var isTabPage = Dwt.instanceOf(type, "DwtTabViewPage");
	if (isTabPage) {
		params.contentTemplate = itemDef.template;
		delete itemDef.template;
	}

    var isTree = Dwt.instanceOf(type, "DwtTree");
    if (isTree) {
        params.style = itemDef.style;
    }

	// add extra params
	params.formItemDef = itemDef;
	if (itemDef.params) {
		for (var p in itemDef.params) {
			params[p] = itemDef.params[p];
		}
	}

	// create control
	var control = new window[type](params);

	// init select
	if (control instanceof DwtSelect) {
		var options = itemDef.items;
		if (options) {
			for (var i = 0; i < options.length; i++) {
				var option = options[i];
				// convert to format that DwtSelect#addOption recognizes
				option.displayValue = option.label || option.value;
				control.addOption(option);
			}
		}
		var handler = DwtForm.__makeFunc(itemDef.onchange);
		control.addChangeListener(new AjxListener(this, this._control2model, [id, handler]));
	}

	// init button, menu item
	else if (control instanceof DwtButton || control instanceof DwtMenuItem) {
		if (itemDef.label) { control.setText(itemDef.label); }
		if (itemDef.image) { control.setImage(itemDef.image, null, itemDef.imageAltText); }
		if (itemDef.menu) {
			var isMenu = Dwt.instanceOf(itemDef.menu.type || "DwtMenu", "DwtMenu");
			var menu;
			if (isMenu) {
				menu = this._registerControl(itemDef.menu, itemDef, null, null, control, "DwtMenu");
			}
			else {
				menu = new DwtMenu({parent:control});
				var style = Dwt.instanceOf(itemDef.menu.type, "DwtCalendar") ?
							DwtMenu.CALENDAR_PICKER_STYLE : DwtMenu.GENERIC_WIDGET_STYLE;
				this._registerControl(itemDef.menu, itemDef, null, { style: style }, menu);
			}
			control.setMenu(menu);
		}
		var parentId;
		if (parent instanceof DwtToolBar || parent instanceof DwtMenu) {
			parentId = parentDef.id;
		}
		if (itemDef.ariaLabel) {
			control.setAriaLabel(itemDef.ariaLabel);
		}
		// handlers
		var handler = DwtForm.__makeFunc(itemDef.onclick || (parentDef && parentDef.onclick));
		control.addSelectionListener(new AjxListener(this, this._item2parent, [id, parentId, handler]));
	}

	// init checkbox, radio button
	else if (control instanceof DwtCheckbox && !(control instanceof DwtRadioButton)) {
		var handler = DwtForm.__makeFunc(itemDef.onclick);
		control.addSelectionListener(new AjxListener(this, this._control2model, [id, handler]));
	}

	// init input field
	else if (control instanceof DwtInputField) {
		var changehandler = DwtForm.__makeFunc(itemDef.onchange);
		var onkeyup = AjxCallback.simpleClosure(this._input2model2handler, this, id, changehandler);
		control.addListener(DwtEvent.ONKEYUP, onkeyup);
        if (AjxEnv.isFirefox){
            var onkeydown = this._onkeydownhandler.bind(this, id, changehandler);
            control.addListener(DwtEvent.ONKEYDOWN, onkeydown);
        }
		var blurhandler = DwtForm.__makeFunc(itemDef.onblur);
        if (blurhandler) {
		    var onblur = AjxCallback.simpleClosure(this._input2model2handler, this, id, blurhandler);
		    control.addListener(DwtEvent.ONBLUR, onblur);
        }

		itemDef.tooltip = itemDef.tooltip || itemDef.hint;
		control.setHint(itemDef.hint);
		control.setLabel(itemDef.label || itemDef.tooltip);
	}

	// init list
	else if (control instanceof DwtListView) {
		control.addSelectionListener(new AjxListener(this, this._handleListSelection, [id]));
	}

	// init menu
	else if (control instanceof DwtMenu) {
		if (itemDef.items) {
			var menuItemDefs = itemDef.items;
			for (var i = 0; i < menuItemDefs.length; i++) {
				var menuItemDef = menuItemDefs[i];
				if (menuItemDef.type == DwtMenuItem.SEPARATOR_STYLE) {
					new DwtMenuItem({parent:control, style:DwtMenuItem.SEPARATOR_STYLE});
					continue;
				}
				this._registerControl(menuItemDef, itemDef, null, null, control, "DwtMenuItem");
			}
		}
	}

	// init tabs
	else if (control instanceof DwtTabView) {
		var pageDefs = itemDef.items;
		if (pageDefs) {
			this._registerControls(pageDefs, itemDef, null, null, control, "DwtTabViewPage");
		}
	}

	// init tab page
	else if (control instanceof DwtTabViewPage && parent instanceof DwtTabView) {
		var key = parent.addTab(itemDef.label, control);
		if (itemDef.image) {
			parent.getTabButton(key).setImage(itemDef.image, null, itemDef.imageAltText);
		}
		if (itemDef.items) {
			this._registerControls(itemDef.items, itemDef, tabIndexes, null, control);
		}
	}

	// init toolbar
	else if (control instanceof DwtToolBar) {
		var toolbarItemDefs = itemDef.items;
		if (toolbarItemDefs) {
			for (var i = 0; i < toolbarItemDefs.length; i++) {
				var toolbarItemDef = toolbarItemDefs[i];
				if (toolbarItemDef.type == DwtToolBar.SPACER) {
					control.addSpacer(toolbarItemDef.size);
					continue;
				}
				if (toolbarItemDef.type == DwtToolBar.SEPARATOR) {
					control.addSeparator(toolbarItemDef.className);
					continue;
				}
				if (toolbarItemDef.type == DwtToolBar.FILLER) {
					control.addFiller(toolbarItemDef.className);
					continue;
				}
				this._registerControl(toolbarItemDef, itemDef, null, null, control, "DwtToolBarButton");
			}
		}
	}
	else if (control instanceof DwtCalendar) {
		if (itemDef.onselect instanceof AjxListener) {
			control.addSelectionListener(itemDef.onselect);
		}
	}

	// TODO: other controls (e.g. combobox, listview, slider, spinner, tree)

	// init anonymous composites
	else if (control instanceof DwtComposite) {
		if (itemDef.items) {
			this._registerControls(itemDef.items, itemDef, tabIndexes, null, control);
		}
	}

    // size control
    if (itemDef.width || itemDef.height) {
        if (control instanceof DwtInputField) {
            Dwt.setSize(control.getInputElement(), itemDef.width, itemDef.height);
        }
        else {
            control.setSize(itemDef.width, itemDef.height);
        }
    }

	if (itemDef.tooltip) {
		control.setToolTipContent(itemDef.tooltip);
	}

	if (itemDef.ariaLabel) {
		control.setAriaLabel(itemDef.ariaLabel);
	}

	// return control
	return control;
};

DwtForm.prototype._onkeydownhandler  = function(id, changehandler){
    setTimeout(this._input2model2handler.bind(this, id, changehandler), 500);
};

DwtForm.prototype._initControl = function(itemDef, useCurrentValues) {
	var id = itemDef.id;
	if (itemDef.label) this.setLabel(id, itemDef.label);
	var item = this._items[id];
	if (useCurrentValues) {
		item.ovalue = item.value;
	}
	else if (itemDef.value) {
		if (Dwt.instanceOf(itemDef.type, "DwtRadioButton")) {
			item.ovalue = item.value = item.control && item.control.isSelected();
		}
		else {
			this.setValue(id, itemDef.value, true);
			item.ovalue = item.value;
		}
	}
	else {
		item.ovalue = null;
	}
	if (typeof itemDef.enabled == "boolean") this.setEnabled(id, itemDef.enabled);
	if (typeof itemDef.visible == "boolean") this.setVisible(id, itemDef.visible);
};

// html handlers

DwtForm.prototype._htmlElement = function(id, formHandler, elementHandler, evt) {
	if (formHandler) {
		this._call(formHandler, [id]);
	}
	if (elementHandler) {
		elementHandler(evt);
	}
};

DwtForm.prototype._htmlInput_checked = function(id, parentId, handler, onclick, evt) {
	var control = this.getControl(id);
	var checked = control.checked;
	this._setModelValue(id, checked);
	if (parentId && checked) {
		this._setModelValue(parentId, control.value);
	}
	this.update();
	this._htmlElement(id, handler, onclick, evt);
};

DwtForm.prototype._htmlInput_value = function(id, handler, onchange, evt) {
	this._setModelValue(id, this.getControl(id).value);
	this.update();
	this._htmlElement(id, handler, onchange, evt);
};

DwtForm.prototype._htmlSelect_selectedIndex = function(id, handler, onchange, evt) {
	var select = this.getControl(id);
	this._setModelValue(id, select.options[select.selectedIndex].value);
	this.update();
	this._htmlElement(id, handler, onchange, evt);
};

// dwt handlers

DwtForm.prototype._control2model = function(id, handler) {
	this._setModelValue(id, this._getControlValue(id));
	this.update();
	if (handler) {
		this._call(handler, [id]);
	}
};

DwtForm.prototype._radio2group2model = function(radioId, groupId, handler) {
	this._setModelValue(groupId, this.getControl(radioId).getValue());
	this._setModelValue(radioId, this._getControlValue(radioId));
	this.update();
	if (handler) {
		this._call(handler, [radioId]);
	}
};

DwtForm.prototype._input2model2handler = function(id, handler) {
	this._setModelValue(id, this._getControlValue(id));
	this.update();
	if (handler) {
		this._call(handler, [id]);
	}
};

DwtForm.prototype._item2parent = function(itemId, parentId, handler) {
	var control = this.getControl(itemId);
	var itemDef = this._items[itemId].def;
	if (control instanceof DwtButtonColorPicker || (itemDef.menu && !itemDef.onclick)) {
		control._toggleMenu(); // HACK: button should have public API
	}
	else if (parentId) {
		this._setModelValue(parentId, this._getControlValue(itemId) || itemId);
		this.update();
	}
	if (handler) {
		this._call(handler, [itemId]);
	}
};

DwtForm.prototype._handleListSelection = function(id, evt) {
	this.update();
};

// setters and getters

DwtForm.prototype.__list_getValue = function(id) {
	return this.getControl(id).getSelection();
};
DwtForm.prototype.__list_setValue = function(id, value) {
	this.getControl(id).setSelection(value);
};

//
// Private functions
//

// code generation

DwtForm.__makeGetter = function(item) {
	var getter = item.getter;
	if (getter) return DwtForm.__makeFunc(getter);

	var ref = item.ref;
	if (!ref) return null;

	var parts = ref.split(".");
	var body = [
		"var context = this.model;"
	];
	for (var i = 0; i < parts.length; i++) {
		var name = parts[i];
		var fname = DwtForm.__makeFuncName(name);
		if (i == parts.length - 1) break;
		body.push(
			"context = context && (context.",fname," ? context.",fname,"() : context.",name,");"
		);
	}
	body.push(
		"var value = context ? (context.",fname," ? context.",fname,"() : context.",name,") : this._items.",name,".value;",
		"return value !== undefined ? value : defaultValue;"
	);
	return new Function("defaultValue", body.join(""));
};

DwtForm.__makeSetter = function(item) {
	var setter = item.setter;
	if (setter) return DwtForm.__makeFunc(setter);

	var ref = item.ref;
	if (!ref) return null;

	var parts = ref.split(".");
	var body = [
		"var context = this.model;"
	];
	for (var i = 0; i < parts.length; i++) {
		var isLast = i == parts.length - 1;
		var name = parts[i];
		var fname = DwtForm.__makeFuncName(name, isLast ? "set" : "get");
		if (isLast) break;
		body.push(
			"context = context && (context.",fname," ? context.",fname,"() : context.",name,");"
		);
	}
	body.push(
		"if (context) {",
			"if (context.",fname,") {",
				"context.",fname,"(value);",
			"}",
			"else {",
				"context.",name," = value;",
			"}",
		"}"
	);
	return new Function("value", body.join("\n"));
};

DwtForm.__makeFuncName = function(name, prefix) {
	return [prefix||"get",name.substr(0,1).toUpperCase(),name.substr(1)].join("");
};

DwtForm.__makeFunc = function(value) {
	if (value == null) return null;
	if (typeof value == "function" && !(value instanceof RegExp)) return value;
	var body = [
		"with (this._context) {",
			"return (",value,");",
		"}"
	].join("");
	return new Function(body);
};

DwtForm.__equals = function(a, b) {
	return a == b;
};

// Array.sort

DwtForm.__byTabIndex = function(a, b) {
	return a.tabindex - b.tabindex;
};

// hacks

DwtForm.__hack_fixRadioButtonHandler = function(radio) {
	var handlers = [radio.getInputElement().onclick, DwtCheckbox.__handleClick];
	var handler = function(evt) {
		for (var i = 0; i < handlers.length; i++) {
			var func = handlers[i];
			if (func) {
				func(evt);
			}
		}
	};
	Dwt.setHandler(radio.getInputElement(), DwtEvent.ONCLICK, handler);
};

//
// Class: DwtFormRows
//

// TODO: tab-group

/**
 * 
 * @extends		DwtForm
 * 
 * @private
 */
DwtFormRows = function(params) {
	if (arguments.length == 0) return;
	this._itemDef = params.formItemDef || {};
	params.className = params.className || "DwtFormRows";
	DwtForm.call(this, {
		id:params.id, parent:params.parent,
		form:{}, template:this._itemDef.template
	});

	// init state
	this._rowsTabGroup = new DwtTabGroup(this._htmlElId);

	// save state
	this._rowDef = this._itemDef.rowitem || {};
	this._equals = DwtForm.__makeFunc(this._rowDef.equals) || DwtForm.__equals;
	this._rowCount = 0;
	this._minRows = this._itemDef.minrows || 1;
	this._maxRows = this._itemDef.maxrows || Number.MAX_VALUE;
	if (this._itemDef.rowtemplate) {
		this.ROW_TEMPLATE = this._itemDef.rowtemplate;
	}

	// add default rows
	var itemDefs = this._itemDef.items || [];
	for (var i = 0; i < itemDefs .length; i++) {
		this.addRow(itemDefs[i]);
	}

	// add empty rows to satisfy minimum row count
	for ( ; i < this._minRows; i++) {
		this.addRow();
	}

	// remember listeners
	this._onaddrow = DwtForm.__makeFunc(this._itemDef.onaddrow);
	this._onremoverow = DwtForm.__makeFunc(this._itemDef.onremoverow);
};
DwtFormRows.prototype = new DwtForm;
DwtFormRows.prototype.constructor = DwtFormRows;

DwtFormRows.prototype.toString = function() {
	return "DwtFormRows";
};

// Data

DwtFormRows.prototype.TEMPLATE = "dwt.Widgets#DwtFormRows";
DwtFormRows.prototype.ROW_TEMPLATE = "dwt.Widgets#DwtFormRow";

// Public methods

DwtFormRows.prototype.getTabGroupMember = function() {
	return this._rowsTabGroup;
};

DwtFormRows.prototype.setValue = function(array) {
	if (arguments.length > 1) {
		DwtForm.prototype.setValue.apply(this, arguments);
		return;
	}
	// adjust row count
	var min = Math.max(array.length, this._minRows);
	for (var i = this._rowCount; i > min; i--) {
		this.removeRow(i-1);
	}
	var max = Math.min(array.length, this._maxRows);
	for (var i = this._rowCount; i < max; i++) {
		this.addRow();
	}
	// initialize values
	for (var i = 0; i < max; i++) {
		this.setValue(String(i), array[i], true);
	}
	for (var i = array.length; i < this._rowCount; i++) {
		this.setValue(String(i), null, true);
	}
};

DwtFormRows.prototype.getValue = function() {
	if (arguments.length > 0) {
		return DwtForm.prototype.getValue.apply(this, arguments);
	}
	var array = new Array(this._rowCount);
	for (var i = 0; i < this._rowCount; i++) {
		array[i] = this.getValue(String(i));
	}
	return array;
};

DwtFormRows.prototype.getRowCount = function() {
	return this._rowCount;
};

DwtFormRows.prototype.addRow = function(itemDef, index) {
	if (this._rowCount >= this._maxRows) {
		return;
	}
	itemDef = itemDef || (this._rowDef && AjxUtil.createProxy(this._rowDef));
	if (!itemDef) {
		return;
	}

	if (index == null) index = this._rowCount;

	// move other rows "up"
	for (var i = this._rowCount - 1; i >= index; i--) {
		var oindex = i, nindex = i+1;
		var item = this._items[oindex];
		item.aka = String(nindex);
		delete this._items[oindex];
		this._items[item.aka] = item;
		this._setControlIds(item.id, item.aka);
	}

	// initialize definition
	itemDef.id = itemDef.id || Dwt.getNextId();
	itemDef.aka = String(index);
	this._rowCount++;

	// create row html
	var data = { id: [this.getHTMLElId(), itemDef.id].join("_") };
	var rowHtml = AjxTemplate.expand(this.ROW_TEMPLATE, data);

	var rowsEl = this._rowsEl;
	rowsEl.appendChild(Dwt.toDocumentFragment(rowHtml, data.id+"_row"));
	var rowEl = rowsEl.lastChild;
	if (index != this._rowCount - 1) {
		rowsEl.insertBefore(rowEl, rowsEl.childNodes[index]);
	}

	// create controls
	var tabIndexes = [];
	var rowControl = this._registerControl(itemDef, null, tabIndexes);

	var addDef = this._itemDef.additem ? AjxUtil.createProxy(this._itemDef.additem) : { image: "Add", tooltip: ZmMsg.addRow, ariaLabel: ZmMsg.addRow };
	addDef.id = addDef.id || itemDef.id+"_add";
	addDef.visible = "this.getRowCount() < this.getMaxRows()";
	addDef.ignore = true;
	var addButton = this._registerControl(addDef,null,tabIndexes,null,null,"DwtButton");
	if (!addDef.onclick) {
		addButton.addSelectionListener(new AjxListener(this, this._handleAddRow, [itemDef.id]));
	}

	var removeDef = this._itemDef.removeitem ? AjxUtil.createProxy(this._itemDef.removeitem) : { image: "Remove", tooltip: ZmMsg.removeRow, ariaLabel: ZmMsg.removeRow };
	removeDef.id = removeDef.id || itemDef.id+"_remove";
	removeDef.visible = "this.getRowCount() > this.getMinRows()";
	removeDef.ignore = true;
	var removeButton = this._registerControl(removeDef,null,tabIndexes,null,null,"DwtButton");
	if (!removeDef.onclick) {
		removeButton.addSelectionListener(new AjxListener(this, this._handleRemoveRow, [itemDef.id]));
	}

	// remember where we put it
	var item = this._items[itemDef.id];
	item._rowEl = rowEl;
	item._addId= addDef.id;
	item._removeId = removeDef.id;

	// set control identifiers
	this._setControlIds(item.id, index);

	// create tab group for row
	var tabGroup = new DwtTabGroup(itemDef.id);
	tabIndexes.sort(DwtForm.__byTabIndex);
	for (var i = 0; i < tabIndexes.length; i++) {
		var control = tabIndexes[i].control;
		tabGroup.addMember(control.getTabGroupMember() || control);
	}

	// add to tab group
	if (index == this._rowCount - 1) {
		this._rowsTabGroup.addMember(tabGroup);
	}
	else {
		var indexItemDef = this._items[String(index+1)];
		this._rowsTabGroup.addMemberBefore(tabGroup, DwtTabGroup.getByName(indexItemDef.id));
	}

	// update display and notify handler
	this.update();
	if (this._onaddrow) {
		this._call(this._onaddrow, [index]);
	}

	return rowControl;
};

DwtFormRows.prototype.removeRow = function(indexOrId) {
	if (this._rowCount <= this._minRows) {
		return;
	}

	var item = this._items[indexOrId];

	// this only recognizes if a properly accessible widgets (i.e. those that
	// receive browser focus) had focus
	var hadFocus = Dwt.isAncestor(item._rowEl, document.activeElement);

	// delete item at specified index
	if (item.control instanceof DwtControl) {
		this.removeChild(item.control);
	}
	delete this._items[item.aka];
	this._deleteItem(item.id);

	// delete add item
	var addItem = this._items[item._addId];
	if (addItem) {
		this.removeChild(addItem.control);
		this._deleteItem(addItem.id);
	}

	// delete remove item
	var removeItem = this._items[item._removeId];
	if (removeItem) {
		this.removeChild(removeItem.control);
		this._deleteItem(removeItem.id);
	}

	// shift everything down one, removing old last row
	var fromIndex = Number(item.aka);
	for (var i = fromIndex + 1; i < this._rowCount; i++) {
		var oindex = i, nindex = i-1;
		this._items[nindex] = this._items[oindex];
		this._items[nindex].aka = String(nindex);
		this._setControlIds(this._items[nindex].id, this._items[nindex].aka);
	}
	this._deleteItem(String(--this._rowCount));

	// remove row element
	var rowEl = item._rowEl;
	rowEl.parentNode.removeChild(rowEl);
	delete item._rowEl;

	// remove from tab group
	var tabGroup = DwtTabGroup.getByName(item.id);
	this._rowsTabGroup.removeMember(tabGroup);

	// update display and notify handler
	this.update();

	if (hadFocus) {
		var otherItem = this._items[item.aka] || this._items[this._rowCount - 1];
		var tabGroup = otherItem.control.getTabGroupMember();
		tabGroup.setFocusMember(tabGroup.getFirstMember(true), true, false);
	}

	if (this._onremoverow) {
		this._call(this._onremoverow, [Number(item.aka)]);
	}
};

DwtFormRows.prototype.getMinRows = function() {
	return this._minRows;
};
DwtFormRows.prototype.getMaxRows = function() {
	return this._maxRows;
};
DwtFormRows.prototype.getRowCount = function() {
	return this._rowCount;
};

DwtFormRows.prototype.getIndexForRowId = function(rowId) {
	var children = this._rowsEl.childNodes;
	for (var i = 0; i < children.length; i++) {
		if (children[i].id == [this._htmlElId,rowId,"row"].join("_")) {
			return i;
		}
	}
	return -1;
};

DwtFormRows.__equals = function(a,b) {
	if (a === b) return true;
	if (!a || !b || a.length != b.length) return false;
	for (var i = 0; i < a.length; i++) {
		if (!this._call(this._equals, [a[i],b[i]])) {
			return false;
		}
	}
	return true;
};

// Protected methods

/** Override to set child controls' identifiers. */
DwtFormRows.prototype._setControlIds = function(rowId, index) {
	var id = [this.getHTMLElId(), index].join("_");
	var item = this._items[rowId];
	this._setControlId(item && item.control, id);
	var addButton = this._items[item._addId];
	this._setControlId(addButton && addButton.control, id+"_add");
	var removeButton = this._items[item._removeId];
	this._setControlId(removeButton && removeButton.control, id+"_remove");
	// TODO: update parentid attribute of children
};

DwtFormRows.prototype._setControlId = function(control, id) {
	if (!control) return;
	if (control instanceof DwtControl) {
		control.setHtmlElementId(id);
	}
	else {
		control.id = id;
	}
};

DwtFormRows.prototype._handleAddRow = function(rowId) {
	if (this.getRowCount() < this.getMaxRows()) {
		var index = this.getIndexForRowId(rowId) + 1;
		this.addRow(null, index);
	}
};

DwtFormRows.prototype._handleRemoveRow = function(rowId) {
	this.removeRow(rowId);
};

// DwtForm methods

DwtFormRows.prototype._setModelValue = function(id, value) {
	if (DwtForm.prototype._setModelValue.apply(this, arguments)) {
		this.parent.setDirty(this._itemDef.id, true);
	}
};

// DwtControl methods

DwtFormRows.prototype._createHtmlFromTemplate = function(templateId, data) {
	DwtForm.prototype._createHtmlFromTemplate.apply(this, arguments);
	this._rowsEl = document.getElementById(this._htmlElId+"_rows");
};


}
if (AjxPackage.define("ajax.dwt.widgets.DwtCalendar")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Creates a calendar widget
 * @constructor
 * @class
 * This class provides a calendar view.
 *
 * @author Ross Dargahi
 * @author Roland Schemers
 *
 * @param {hash}		params			a hash of parameters
 * @param {DwtComposite}      params.parent			the parent widget
 * @param {string}      params.className			the CSS class
 * @param {constant}      params.posStyle			the positioning style (see {@link Dwt})
 * @param {constant}     [params.firstDayOfWeek=DwtCalendar.SUN]		the first day of the week
 * @param {boolean}	[params.forceRollOver=true] 	if <code>true</code>, then clicking on (or setting) the widget to a 
 *												date that is not part of the current month (i.e. one of 
 *												the grey prev or next month days) will result in the 
 *												widget rolling 	the date to that month.
 * @param {array}      params.workingDays		a list of days that are work days. This array assumes that
 * 												index 0 is Sunday. Defaults to Mon-Fri being work days.
 * @param {boolean}      params.hidePrevNextMo 	a flag indicating whether widget should hide days of the 
 *												previous/next month
 * @param {boolean}      params.readOnly 		a flag indicating that this widget is read-only (should not 
 *												process events such as mouse clicks)
 * @param {boolean}      params.showWeekNumber	a flag indicating whether widget should show week number
 *        
 * @extends		DwtComposite
 */
DwtCalendar = function(params) {
	if (arguments.length == 0) { return; }
	params = Dwt.getParams(arguments, DwtCalendar.PARAMS);
	params.className = params.className || "DwtCalendar";
	DwtComposite.call(this, params);

	this._skipNotifyOnPage = false;
	this._hidePrevNextMo = params.hidePrevNextMo;
	this._readOnly = params.readOnly;
	this._showWeekNumber = params.showWeekNumber;
	this._uuid = Dwt.getNextId();
	var cn = this._origDayClassName = params.className + "Day";
	this._todayClassName = " " + params.className + "Day-today";
	this._selectedDayClassName = " " + cn + "-" + DwtCssStyle.SELECTED;
	this._hoveredDayClassName = " " + cn + "-" + DwtCssStyle.HOVER;
	this._activeDayClassName = " " + cn + "-" + DwtCssStyle.ACTIVE;
	this._hiliteClassName = " " + cn + "-hilited";
	this._greyClassName = " " + cn + "-grey";
	
	if (!this._readOnly) {
		this._installListeners();
	}

	this._selectionMode = DwtCalendar.DAY;
	this.navigationMembers = [];
	this.currentNavigationMemberInTabGroup = null;

	this.dayMembers = [];
	this.currentDayMemberInTabGroup = null;

	var menuInstance = this.parent;
	while (typeof menuInstance !== 'undefined' && !(menuInstance instanceof DwtMenu)) {
		menuInstance = menuInstance.parent;
	}

	if (menuInstance instanceof DwtMenu) {
		this.tabgroup = menuInstance._compositeTabGroup;
	} else {
		this.tabgroup = this._compositeTabGroup;
	}

	this._init();

	this._weekDays = new Array(7);
	this._workingDays = params.workingDays || DwtCalendar._DEF_WORKING_DAYS;
    this._useISO8601WeekNo = params.useISO8601WeekNo;
	this.setFirstDayOfWeek(params.firstDayOfWeek || DwtCalendar.SUN);
	
	this._forceRollOver = (params.forceRollOver !== false);
};

DwtCalendar.PARAMS = ["parent", "className", "posStyle", "firstDayOfWeek", "forceRollOver",
					  "workingDaysArray", "hidePrevNextMo", "readOnly"];

DwtCalendar.prototype = new DwtComposite;
DwtCalendar.prototype.constructor = DwtCalendar;

DwtCalendar.KEY_UP = 'KEY_UP';
DwtCalendar.KEY_DOWN = 'KEY_DOWN';
DwtCalendar.KEY_RIGHT = 'KEY_RIGHT';
DwtCalendar.KEY_LEFT = 'KEY_LEFT';
/**
 * Sunday.
 */
DwtCalendar.SUN = 0;
/**
 * Monday.
 */
DwtCalendar.MON = 1;
/**
 * Tuesday.
 */
DwtCalendar.TUE = 2;
/**
 * Wednesday.
 */
DwtCalendar.WED = 3;
/**
 * Thursday.
 */
DwtCalendar.THU = 4;
/**
 * Friday.
 */
DwtCalendar.FRI = 5;
/**
 * Saturday.
 */
DwtCalendar.SAT = 6;

// Selection modes
/**
 * Defines the "day" selection mode.
 */
DwtCalendar.DAY = 1;
/**
 * Defines the "week" selection mode.
 */
DwtCalendar.WEEK = 2;
/**
 * Defines the "work week" selection mode.
 */
DwtCalendar.WORK_WEEK = 3;
/**
 * Defines the "month" selection mode.
 */
DwtCalendar.MONTH = 4;

DwtCalendar.RANGE_CHANGE = "DwtCalendar.RANGE_CHANGE";

DwtCalendar._FULL_WEEK = [1, 1, 1, 1, 1, 1, 1];
DwtCalendar._DEF_WORKING_DAYS = [0, 1, 1, 1, 1, 1, 0];
DwtCalendar._DAYS_IN_MONTH = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];

DwtCalendar._NO_MONTH = -2;
DwtCalendar._PREV_MONTH = -1;
DwtCalendar._THIS_MONTH = 0;
DwtCalendar._NEXT_MONTH = 1;

DwtCalendar._NORMAL = 1;
DwtCalendar._HOVERED = 2;
DwtCalendar._ACTIVE = 3;
DwtCalendar._SELECTED = 4;
DwtCalendar._DESELECTED = 5;

DwtCalendar.DATE_SELECTED 		= 1;
DwtCalendar.DATE_DESELECTED 	= 2;
DwtCalendar.DATE_DBL_CLICKED 	= 3;

DwtCalendar._LAST_DAY_CELL_IDX = 41;

DwtCalendar._BUTTON_CLASS = "DwtCalendarButton";
DwtCalendar._BUTTON_HOVERED_CLASS = DwtCalendar._BUTTON_CLASS + "-" + DwtCssStyle.HOVER;
DwtCalendar._BUTTON_ACTIVE_CLASS = DwtCalendar._BUTTON_CLASS + "-" + DwtCssStyle.ACTIVE;

DwtCalendar._TITLE_CLASS = "DwtCalendarTitle";
DwtCalendar._TITLE_HOVERED_CLASS = DwtCalendar._TITLE_CLASS + "-" + DwtCssStyle.HOVER;
DwtCalendar._TITLE_ACTIVE_CLASS = DwtCalendar._TITLE_CLASS + "-" + DwtCssStyle.ACTIVE;

/**
 * Returns a string representation of the object.
 * 
 * @return		{string}		a string representation of the object
 */
DwtCalendar.prototype.toString = 
function() {
	return "DwtCalendar";
};

/**
 * Adds a selection listener.
 * 
 * @param	{AjxListener}	listener		the listener
 */
DwtCalendar.prototype.addSelectionListener = 
function(listener) {
	this.addListener(DwtEvent.SELECTION, listener);
};

/**
 * Removes a selection listener.
 * 
 * @param	{AjxListener}	listener		the listener
 */
DwtCalendar.prototype.removeSelectionListener = 
function(listener) { 
	this.removeListener(DwtEvent.SELECTION, listener);
};

/**
 * Adds an action listener.
 * 
 * @param	{AjxListener}	listener		the listener
 */
DwtCalendar.prototype.addActionListener = 
function(listener) {
	this.addListener(DwtEvent.ACTION, listener);
};

/**
 * Removes an action listener.
 * 
 * @param	{AjxListener}	listener		the listener
 */
DwtCalendar.prototype.removeActionListener = 
function(listener) { 
	this.removeListener(DwtEvent.ACTION, listener);
};

/**
 * Adds a date range listener. Date range listeners are called whenever the date range of the calendar
 * changes (i.e. when it rolls over due to a programatic action via {@link #setDate} or
 * via user selection).
 *
 * @param 	{AjxListener}		listener		the listener
 */
DwtCalendar.prototype.addDateRangeListener = 
function(listener) {
	this.addListener(DwtEvent.DATE_RANGE, listener);
};

/**
 * Removes a date range listener.
 * 
 * @param 	{AjxListener}		listener		the listener
 */
DwtCalendar.prototype.removeDateRangeListener = 
function(listener) { 
	this.removeListener(DwtEvent.DATE_RANGE, listener);
};

/**
 * Sets the skip notify on page. This method notify (or not) selection when paging arrow buttons
 * are clicked.
 *
 * @param	{boolean}	skip		if <code>true</code>, do not notify selection
 */
DwtCalendar.prototype.setSkipNotifyOnPage = 
function(skip) {
	this._skipNotifyOnPage = skip;
};

/**
 * Gets the skip notify on page setting.
 * 
 * @return	{boolean}	<code>true</code>, do not notify selection
 */
DwtCalendar.prototype.getSkipNotifyOnPage = 
function() {
	return this._skipNotifyOnPage;
};

/**
 * Sets the date.
 * 
 * @param	{Date}	date	the date
 * @param	{boolean}	skipNotify		if <code>true</code>, do not notify selection
 * @param {boolean}	forceRollOver 	if <code>true</code>, then clicking on (or setting) the widget to a 
 *												date that is not part of the current month (i.e. one of 
 *												the grey prev or next month days) will result in the 
 *												widget rolling 	the date to that month.
 * @param	{boolean}	dblClick		if <code>true</code>, require a double click
 */
DwtCalendar.prototype.setDate =
function(date, skipNotify, forceRollOver, dblClick) {

	forceRollOver = (forceRollOver == null) ? this._forceRollOver : forceRollOver;
	
	// Check if the date is available for selection. Basically it is unless we are in
	// work week selection mode and <date> is not a working day
	//if (this._selectionMode == DwtCalendar.WORK_WEEK && !this._currWorkingDays[date.getDay()])
	//	return false;

	if(!date) {
		date = new Date();
	}
	var newDate = new Date(date.getTime());
	var oldDate = this._date;

	var layout = false;
	var notify = false;
	var cellId;

	if (this._date2CellId != null) {
		var idx = (newDate.getFullYear() * 10000) + (newDate.getMonth() * 100) + newDate.getDate();
		var cellId = this._date2CellId[idx];
		
		if (cellId) {
		 	if (cellId == this._selectedCellId)
		 		notify = true;

			var cell = document.getElementById(cellId);
			if (cell._dayType == DwtCalendar._THIS_MONTH)
				notify = true;
			else if (forceRollOver)
				notify = layout = true;
			else
				notify = true;
		} else {
			 notify = layout = true;
		}
	} else {
		notify = layout = true;
	}

	// update before layout, notify after layout
	if (notify) {
		if (this._date){
			// 5/13/2005 EMC -- I'm not sure why this was setting the hours to 0.
			// I think it should respect what the user passed in, and only change
			// the parts of the date that it is responsible for.
			//newDate.setHours(0,0,0,0);
			//handle daylight saving
			if(AjxDateUtil.isDayShifted(newDate)) {
				AjxDateUtil.rollToNextDay(newDate);
			}
			newDate.setHours(this._date.getHours(), this._date.getMinutes(), this._date.getSeconds(), 0);            
		}

		this._date = newDate;
		if (!layout && !this._readOnly) {
			this._setSelectedDate();
			this._setToday();
		}
	}

	if (layout) {
		this._layout();
	}

	if (notify && !skipNotify) {
		var type = dblClick ? DwtCalendar.DATE_DBL_CLICKED : DwtCalendar.DATE_SELECTED;
		this._notifyListeners(DwtEvent.SELECTION, type, this._date);
	}
	
	return true;
};

/**
 * Checks if the cell is selected.
 * 
 * @param	{string}	cellId			the cell id	
 * @return	{boolean}	<code>true</code> if the cell is the selected day
 */
DwtCalendar.prototype.isSelected =
function(cellId) {
	// if cellId is the selected day, then return true, else if we are NOT in
	// day selection mode (i.e. week/work week) then compute the row and index
	// of cellId and look it up in the week array to see if it is a selectable day
	if (cellId == this._selectedDayElId) {
		return true;
	} else if (this._selectionMode != DwtCalendar.DAY) {
		// If the cell is in the same row as the currently selected cell and it
		// is a selectable day (i.e. a working day in the case of work week),
		// then say it is selected
		var cellIdx = this._getDayCellIndex(cellId);
		if (Math.floor(cellIdx / 7) == Math.floor(this._getDayCellIndex(this._selectedDayElId) / 7)
			&& this._currWorkingDays[cellIdx % 7])
			return true;
	}
	return false;
};

/**
 * Gets the force roll over setting. Force roll over is occurs when a date that
 * is not part of the current month (i.e. one of the grey prev or next month
 * days) will result in the widget rolling 	the date to that month.
 * 
 * @return	{boolean}	<code>true</code> if force roll over is set
 */
DwtCalendar.prototype.getForceRollOver =
function() {
	return this._forceRollOver;
};

/**
 * Sets the force roll over setting. Force roll over is occurs when a date that
 * is not part of the current month (i.e. one of the grey prev or next month
 * days) will result in the widget rolling 	the date to that month.
 * 
 * @param	{boolean}	force		if <code>true</code>, force roll over
 */
DwtCalendar.prototype.setForceRollOver =
function(force) {
	if (force == null) { return; }
	
	if (this._forceRollOver != force) {
		this._forceRollOver = force;
		this._layout();
	}
};

/**
 * Gets the selection mode.
 * 
 * @return	{constant}		the selection mode
 */
DwtCalendar.prototype.getSelectionMode =
function() {
	return this._selectionMode;
};

/**
 * Sets the selection mode.
 * 
 * @return	{constant}		selectionMode		the selection mode
 */
DwtCalendar.prototype.setSelectionMode =
function(selectionMode) {
	if (this._selectionMode == selectionMode) { return; }

	this._selectionMode = selectionMode;
	if (selectionMode == DwtCalendar.WEEK) {
		this._currWorkingDays = DwtCalendar._FULL_WEEK;
	} else if (selectionMode == DwtCalendar.WORK_WEEK) {
		this._currWorkingDays = this._workingDays;
	}

	this._layout();
};

/**
 * Sets the working week.
 * 
 * @param	{array}	workingDaysArray		an array of days
 */
DwtCalendar.prototype.setWorkingWeek =
function(workingDaysArray) {
	// TODO Should really create a copy of workingDaysArray
	this._workingDays = this._currWorkingDays = workingDaysArray;
	
	if (this._selectionMode == DwtCalendar.WORK_WEEK) {
		DBG.println("FOO!!!");
		this._layout();
	}
};

/**
 * Enables/disables the highlight (i.e. "bolding") on the dates in <code>&lt;dates&gt;</code>.
 *
 * @param {object} dates associative array of {@link Date} objects for
 * which to enable/disable highlighting
 * @param {boolean}	enable 	if <code>true</code>, enable highlighting
 * @param {boolean}	clear 	if <code>true</code>, clear current highlighting
 */
DwtCalendar.prototype.setHilite =
function(dates, enable, clear) {
	if (this._date2CellId == null) { return; }

	var cell;
	var aDate;
	if (clear) {
		for (aDate in this._date2CellId) {
			cell = document.getElementById(this._date2CellId[aDate]);
			if (cell._isHilited) {
				cell._isHilited = false;
				this._setClassName(cell, DwtCalendar._NORMAL);
			}	
		}
	}

	var cellId;
	for (var i in dates) {
        // NOTE: Protect from prototype extensions.
        if (dates.hasOwnProperty(i)) {
            aDate = dates[i];
            cellId = this._date2CellId[aDate.getFullYear() * 10000 + aDate.getMonth() * 100 + aDate.getDate()];

            if (cellId) {
                cell = document.getElementById(cellId);
                if (cell._isHilited != enable) {
                    cell._isHilited = enable;
                    this._setClassName(cell, DwtCalendar._NORMAL);
                }
            }
        }
	}
};

/**
 * Gets the date.
 * 
 * @return	{Date}	the date
 */
DwtCalendar.prototype.getDate =
function() {
	return this._date;
};

/**
 * Sets the first date of week.
 * 
 * @param	{constant}		firstDayOfWeek		the first day of week
 */
DwtCalendar.prototype.setFirstDayOfWeek =
function(firstDayOfWeek) {
	for (var i = 0; i < 7; i++) {
		this._weekDays[i] = (i < firstDayOfWeek)
			? (6 - (firstDayOfWeek -i - 1))
			: (i - firstDayOfWeek);

		var dowCell = document.getElementById(this._getDOWCellId(i));
		dowCell.innerHTML = AjxDateUtil.WEEKDAY_SHORT[(firstDayOfWeek + i) % 7];
	}
    this._firstDayOfWeek = firstDayOfWeek
	this._layout();
};

/**
 * Gets the date range.
 * 
 * @return	{Object}		the range (<code>range.start</code> and <code>range.end</code>)
 */
DwtCalendar.prototype.getDateRange =
function () {
	return this._range;
};

DwtCalendar.prototype._getDayCellId =
function(cellId) {
	return ("c:" + cellId + ":" + this._uuid);
};

DwtCalendar.prototype._getDayCellIndex =
function(cellId) {
	return cellId.substring(2, cellId.indexOf(":", 3));
};

DwtCalendar.prototype._getDOWCellId =
function(cellId) {
	return ("w:" + cellId + ":" + this._uuid);
};

DwtCalendar.prototype._getWeekNumberCellId =
function(cellId) {
	return ("k:" + cellId + ":" + this._uuid);
};

DwtCalendar.prototype._getDaysInMonth =
function(mo, yr) {
	/* If we are not dealing with Feb, then simple lookup
	 * Leap year rules
	 *  1. Every year divisible by 4 is a leap year.
	 *  2. But every year divisible by 100 is NOT a leap year
	 *  3. Unless the year is also divisible by 400, then it is still a leap year.*/
	if (mo != 1) {
		return DwtCalendar._DAYS_IN_MONTH[mo];
	}

	if (yr % 4 != 0 || (yr % 100 == 0 && yr % 400 != 0)) {
		return 28;
	}

	return 29;
};

DwtCalendar.prototype._installListeners =
function() {
	this._setMouseEventHdlrs();
	this.addListener(DwtEvent.ONMOUSEOVER, new AjxListener(this, this._mouseOverListener));
	this.addListener(DwtEvent.ONMOUSEOUT, new AjxListener(this, this._mouseOutListener));
	this.addListener(DwtEvent.ONMOUSEDOWN, new AjxListener(this, this._mouseDownListener));
	this.addListener(DwtEvent.ONMOUSEUP, new AjxListener(this, this._mouseUpListener));
	this.addListener(DwtEvent.ONDBLCLICK, new AjxListener(this, this._doubleClickListener));
};

DwtCalendar.prototype._notifyListeners =
function(eventType, type, detail, ev) {
	if (!this.isListenerRegistered(eventType)) { return; }

	var selEv = DwtShell.selectionEvent;
	if (ev) {
		DwtUiEvent.copy(selEv, ev);
	} else {
		selEv.reset();
	}
	selEv.item = this;
	selEv.detail = detail;
	selEv.type = type;
	this.notifyListeners(eventType, selEv);
};

DwtCalendar.prototype._layout =
function() {
	if (this._date == null) { this._date = new Date(); }

	if (!this._calWidgetInited) {
		this._init();
	}

	var date = new Date(this._date.getTime());
	date.setDate(1);
	var year = date.getFullYear();
	var month  = date.getMonth();
	var firstDay = date.getDay();
	var daysInMonth = this._getDaysInMonth(month, year);
	var day = 1;
	var nextMoDay = 1;

	this._date2CellId = new Object();
	this._selectedDayElId = null;

	// Figure out how many days from the previous month we have to fill in
	// (see comment below)
	var lastMoDay, lastMoYear, lastMoMonth, nextMoMonth, nextMoYear;
	if (!this._hidePrevNextMo) {
		if (month != 0) {
			lastMoDay = this._getDaysInMonth(month - 1, year) - this._weekDays[firstDay] + 1;
			lastMoYear = year;
			lastMoMonth = month - 1;
			if (month != 11) {
				nextMoMonth = month + 1;
				nextMoYear = year;
			} else {
				nextMoMonth = 0;
				nextMoYear = year + 1;
			}
		} else {
			lastMoDay = this._getDaysInMonth(11, year - 1) - this._weekDays[firstDay] + 1;
			lastMoYear = year - 1;
			lastMoMonth = 11;
			nextMoMonth = 1;
			nextMoYear = year;
		}
	}

	for (var i = 0; i < 6; i++) {
		for (var j = 0; j < 7; j++) {
			var dayCell = document.getElementById(this._getDayCellId(i * 7 + j));

			if (dayCell._isHilited == null) {
				dayCell._isHilited = false;
			}

			if (day <= daysInMonth) {
				/* The following if statement deals with the first day of this month not being
				 * the first day of the week. In this case we must fill the preceding days with
				 * the final days of the previous month */
				if (i != 0 || j >= this._weekDays[firstDay]) {
					this._date2CellId[(year * 10000) + (month * 100) + day] = dayCell.id;
					dayCell._day = day;
					dayCell._month = month;
					dayCell._year = year;
					dayCell.innerHTML = day++;
					dayCell._dayType = DwtCalendar._THIS_MONTH;
					if (this._readOnly) {
						dayCell.style.fontFamily = "Arial";
						dayCell.style.fontSize = "10px";
					}
				} else {
					if (this._hidePrevNextMo) {
						dayCell.innerHTML = "";
					} else {
						this._date2CellId[(lastMoYear * 10000) + (lastMoMonth * 100) + lastMoDay] = dayCell.id;
						dayCell._day = lastMoDay;
						dayCell._month = lastMoMonth;
						dayCell._year = lastMoYear;
						dayCell.innerHTML = lastMoDay++;
						dayCell._dayType = DwtCalendar._PREV_MONTH;
					}
				}
			} else if (!this._hidePrevNextMo) {
				// Fill any remaining slots with days from next month
				this._date2CellId[(nextMoYear * 10000) + (nextMoMonth * 100) + nextMoDay] = dayCell.id;
				dayCell._day = nextMoDay;
				dayCell._month = nextMoMonth;
				dayCell._year = nextMoYear;
				dayCell.innerHTML = nextMoDay++;
				dayCell._dayType = DwtCalendar._NEXT_MONTH;
			}
			this._setClassName(dayCell, DwtCalendar._NORMAL);

			dayCell.setAttribute('role', 'cell');
			dayCell.setAttribute('aria-label', dayCell.innerText);
		}

		if (this._showWeekNumber) {
			var kwCellId = this._getWeekNumberCellId('kw' + i * 7);
			var kwCell = document.getElementById(kwCellId);
			if (kwCell) {
				var firstDayCell = document.getElementById(this._getDayCellId(i * 7));
				kwCell.innerHTML = AjxDateUtil.getWeekNumber(new Date(firstDayCell._year, firstDayCell._month, firstDayCell._day), this._firstDayOfWeek, null, this._useISO8601WeekNo);
			}
		}
	}

	this._setTitle(month, year);

	// Compute the currently selected day
	if (!this._readOnly) {
		this._setSelectedDate();
		this._setToday();
	}
	
	this._setRange();
};

DwtCalendar.prototype._setRange =
function() {
	var cell = document.getElementById(this._getDayCellId(0));
	var start = new Date(cell._year, cell._month, cell._day, 0, 0, 0, 0);

	cell = document.getElementById(this._getDayCellId(DwtCalendar._LAST_DAY_CELL_IDX));
	
	var daysInMo = this._getDaysInMonth(cell._month, cell._year);
	var end;
	if (cell._day < daysInMo) {
		end = new Date(cell._year, cell._month, cell._day + 1, 0, 0, 0, 0);
	} else if (cell._month < 11) {
		end = new Date(cell._year, cell._month + 1, 1, 0, 0, 0, 0);
	} else {
		end = new Date(cell._year + 1, 0, 1, 0, 0, 0, 0);
	}

	if (this._range == null) {
		this._range = {};
	} else if (this._range.start.getTime() == start.getTime() && this._range.end.getTime() == end.getTime()) {
		return false;
	}

	this._range.start = start;
	this._range.end = end;

	// Notify any listeners
	if (!this.isListenerRegistered(DwtEvent.DATE_RANGE)) { return; }

	if (!this._dateRangeEvent) {
		this._dateRangeEvent = new DwtDateRangeEvent(true);
	}

	this._dateRangeEvent.item = this;
	this._dateRangeEvent.start = start;
	this._dateRangeEvent.end = end;
	this.notifyListeners(DwtEvent.DATE_RANGE, this._dateRangeEvent);
};

DwtCalendar.prototype._setToday =
function() {
	var cell;
	var today = new Date();
	var todayDay = today.getDate();

	if (!this._todayDay || this._todayDay != todayDay) {
		if (this._todayCellId != null) {
			cell = document.getElementById(this._todayCellId);
			cell._isToday = false;
			this._setClassName(cell, DwtCalendar._NORMAL);
		}

		this._todayCellId = this._date2CellId[(today.getFullYear() * 10000) + (today.getMonth() * 100) + todayDay];
		if (this._todayCellId != null) {
			cell = document.getElementById(this._todayCellId);
			cell._isToday = true;
			this._setClassName(cell, DwtCalendar._NORMAL);
		}
	}
};

DwtCalendar.prototype._setSelectedDate =
function() {
	var day = this._date.getDate();
	var month = this._date.getMonth();
	var year = this._date.getFullYear();
	var cell;

	if (this._selectedDayElId) {
		cell = document.getElementById(this._selectedDayElId);
		this._setClassName(cell, DwtCalendar._DESELECTED);
	}

	var cellId = this._date2CellId[(year * 10000) + (month * 100) + day];
	cell = document.getElementById(cellId);
	this._selectedDayElId = cellId;
	this._setClassName(cell, DwtCalendar._SELECTED);
};

DwtCalendar.prototype._setCellClassName = 
function(cell, className, mode) {
	if (cell._dayType != DwtCalendar._THIS_MONTH) {
		className += this._greyClassName;
	}

	if (this._selectionMode == DwtCalendar.DAY &&
		cell.id == this._selectedDayElId &&
		mode != DwtCalendar._DESELECTED)
	{
		className += this._selectedDayClassName;
		this.setDayMemberInTabGroup(null, cell);
	}
	else if (this._selectionMode != DwtCalendar.DAY &&
			 mode != DwtCalendar._DESELECTED &&
			 this._selectedDayElId != null)
	{
		var idx = this._getDayCellIndex(cell.id);
		if (Math.floor(this._getDayCellIndex(this._selectedDayElId) / 7) == Math.floor(idx / 7) &&
			this._currWorkingDays[idx % 7])
		{
			className += this._selectedDayClassName;
		}
	}

	if (cell._isHilited) {
		className += this._hiliteClassName;
	}

	if (cell._isToday) {
		className += this._todayClassName;
	}

	return className;
};

DwtCalendar.prototype._setClassName = 
function(cell, mode) {
	var className = "";
	
	if (mode == DwtCalendar._NORMAL) {
		className = this._origDayClassName;
	} else if (mode == DwtCalendar._HOVERED) {
		className = this._hoveredDayClassName;
	} else if (mode == DwtCalendar._ACTIVE) {
		className = this._activeDayClassName;
	} else if (mode == DwtCalendar._DESELECTED && this._selectionMode == DwtCalendar.DAY) {
		className = this._origDayClassName;
	} else if (this._selectionMode != DwtCalendar.DAY &&
			(mode == DwtCalendar._SELECTED || mode == DwtCalendar._DESELECTED))
	{
		// If we are not in day mode, then we need to highlite multiple cells
		// e.g. the whole week if we are in week mode
		var firstCellIdx = Math.floor(this._getDayCellIndex(this._selectedDayElId) / 7) * 7;

		for (var i = 0; i < 7; i++) {
			className = this._origDayClassName;
			var aCell = document.getElementById(this._getDayCellId(firstCellIdx++));
			aCell.className = this._setCellClassName(aCell, className, mode);
		}
		return;
	}

	cell.className = this._setCellClassName(cell, className, mode);
};

DwtCalendar.prototype._setTitle =
function(month, year) {
	var cell = document.getElementById(this._monthCell);
	var formatter = DwtCalendar.getMonthFormatter();
	var date = new Date(year, month);
	cell.innerHTML = formatter.format(date);
};

DwtCalendar.prototype.getKeyMapName =
function() {
	return DwtKeyMap.MAP_MENU;
};

DwtCalendar.prototype.handleKeyAction = function(actionCode, ev) {

	switch (actionCode) {
		case DwtKeyMap.SELECT:
			if (ev.target || this.tabgroup.__currFocusMember) {
				ev.button = DwtMouseEvent.LEFT;
				this._mouseDownListener(ev);
				this._mouseUpListener(ev);
				this._mouseOutListener(ev);
			}
			break;
		case DwtKeyMap.SUBMENU: // right arrow
		case DwtKeyMap.PARENTMENU: // Left arrow
		case DwtKeyMap.SELECT_PREV: // Up arrow
		case DwtKeyMap.SELECT_NEXT: // down arrow
			this.selectElement(actionCode, ev);
			break;
		default:
			return false;
	}
	return true;
};


DwtCalendar.prototype.selectElement =
function (type, ev) {
	var currentMember = ev.target || this.tabgroup.__currFocusMember;

	// focus should not move to any element using arrow key when focus is on calendar dailog outline.
	if (currentMember instanceof DwtControl) {
		return false;
	}

	switch (type) {
		case DwtKeyMap.SUBMENU: // right arrow
			if (currentMember.id.indexOf('b:') === 0) { // Navigation focus
				this.setNavigationInTabGroup(true, true);
			}
			else if (currentMember.id.indexOf('c:') === 0) { // Day focus
				this.setDayMemberInTabGroup(DwtCalendar.KEY_RIGHT, null, true);
			}
			break;
		case DwtKeyMap.PARENTMENU: // left arrow
			if (currentMember.id.indexOf('b:') === 0) { // Navigation focus
				this.setNavigationInTabGroup(false, true);
			}
			else if (currentMember.id.indexOf('c:') === 0) { // Day focus
				this.setDayMemberInTabGroup(DwtCalendar.KEY_LEFT, null, true);
			}
			break;
		case DwtKeyMap.SELECT_PREV: // up arrow
			if (currentMember.id.indexOf('c:') === 0) { // Day focus
				this.setDayMemberInTabGroup(DwtCalendar.KEY_UP, null, true);
			}
			break;
		case DwtKeyMap.SELECT_NEXT: // down arrow
			if (currentMember.id.indexOf('c:') === 0) { // Day focus
				this.setDayMemberInTabGroup(DwtCalendar.KEY_DOWN, null, true);
			}
	}
};

DwtCalendar.prototype._init =
function() {
	var html = new Array(100);
	var idx = 0;
	this._monthCell = "t:" + this._uuid;

	// Construct the header row with the prev/next year and prev/next month
	// icons as well as the month/year title cell
	html[idx++] =	"<table role='presentation' width=100%>";
	html[idx++] =		"<tr><td class='DwtCalendarTitlebar'>";
	html[idx++] =			"<table role='presentation'>";
	html[idx++] =				"<tr>";
	html[idx++] =					"<td align='center' tabindex='0' role='cell' aria-label='" + ZmMsg.previousYear + "' class='";
	html[idx++] =						DwtCalendar._BUTTON_CLASS;
	html[idx++] =						"' id='b:py:";
	html[idx++] =						this._uuid;
	html[idx++] =						"'>";
	html[idx++] =						AjxImg.getImageHtml("FastRevArrowSmall", null, ["id='b:py:img:", this._uuid, "'"].join(""));
	html[idx++] =					"</td>";
	html[idx++] =					"<td align='center' tabindex='0' role='cell' aria-label='" + ZmMsg.previousMonth + "' class='";
	html[idx++] =						DwtCalendar._BUTTON_CLASS;
	html[idx++] =						"' id='b:pm:";
	html[idx++] =						this._uuid;
	html[idx++] =						"'>";
	html[idx++] =						AjxImg.getImageHtml("RevArrowSmall", null, ["id='b:pm:img:", this._uuid, "'"].join(""));
	html[idx++] =					"</td>";
	html[idx++] =					"<td class='DwtCalendarTitleCell' 'nowrap' style='width: 60%'><span class='";
	html[idx++] =						DwtCalendar._TITLE_CLASS;
	html[idx++] = 						"' id='";
	html[idx++] =						this._monthCell;
	html[idx++] =					"'></span></td>";
	html[idx++] =					"<td align='center' tabindex='0' role='cell' aria-label='" + ZmMsg.nextMonth + "' class='";
	html[idx++] =						DwtCalendar._BUTTON_CLASS;
	html[idx++] =						"' id='b:nm:";
	html[idx++] =						this._uuid;
	html[idx++] =						"'>";
	html[idx++] =						AjxImg.getImageHtml("FwdArrowSmall", null, ["id='b:nm:img:", this._uuid, "'"].join(""));
	html[idx++] =					"</td>";
	html[idx++] =					"<td align='center' tabindex='0' role='cell' aria-label='" + ZmMsg.nextYear + "' class='";
	html[idx++] =						DwtCalendar._BUTTON_CLASS;
	html[idx++] =						"' id='b:ny:";
	html[idx++] =						this._uuid;
	html[idx++] =						"'>";
	html[idx++] =						AjxImg.getImageHtml("FastFwdArrowSmall", null, ["id='b:ny:img:", this._uuid, "'"].join(""));
	html[idx++] =					"</td>";
	html[idx++] =				"</tr>";
	html[idx++] =			"</table>";
	html[idx++] =		"</td></tr>";
	html[idx++] =	"<tr><td class='DwtCalendarBody'>";
	html[idx++] =		"<table role='presentation' width='100%' style='border-collapse:separate;' cellspacing='0'>";
	html[idx++] =			"<tr>";

	if (this._showWeekNumber) {
		html[idx++] = "<td class='DwtCalendarWeekNoTitle' width='14%' id='";
		html[idx++] = this._getWeekNumberCellId('kw');
		html[idx++] = "'>";
		html[idx++] = AjxMsg.calendarWeekTitle;
		html[idx++] = "</td>";
	}

	for (var i = 0; i < 7; i++) {
		html[idx++] = "<td class='DwtCalendarDow' width='";
		html[idx++] = (i < 5 ? "14%" : "15%");
		html[idx++] = "' id='";
		html[idx++] = this._getDOWCellId(i);
		html[idx++] = "'>&nbsp;</td>";
	}
	html[idx++] = "</tr>";

	for (var i = 0; i < 6; i++) {
		html[idx++] = "<tr>";
		if (this._showWeekNumber) {
			html[idx++] = "<td class='DwtCalendarWeekNo' id='" + this._getWeekNumberCellId('kw' + i * 7) + "'>&nbsp;</td>";
		}
		for (var j = 0; j < 7; j++) {
			html[idx++] = "<td id='";
			html[idx++] = this._getDayCellId(i * 7 + j);
			html[idx++] = "' tabindex='0'>&nbsp;</td>";
		}
		html[idx++] ="</tr>";
	}

	html[idx++] = "</td></tr></table></table>";

	this.getHtmlElement().innerHTML = html.join("");
	this.getHtmlElement().setAttribute('role', 'complementary');
	this.getHtmlElement().setAttribute('aria-label', ZmMsg.miniCalendar);

	if (!this._readOnly) {
		document.getElementById("b:py:img:" + this._uuid)._origClassName = AjxImg.getClassForImage("FastRevArrowSmall");
		document.getElementById("b:pm:img:" + this._uuid)._origClassName = AjxImg.getClassForImage("RevArrowSmall");
		document.getElementById("b:nm:img:" + this._uuid)._origClassName = AjxImg.getClassForImage("FwdArrowSmall");
		document.getElementById("b:ny:img:" + this._uuid)._origClassName = AjxImg.getClassForImage("FastFwdArrowSmall");
	}

	this._calWidgetInited = true;

	this.navigationMembers.push(document.getElementById("b:py:" + this._uuid));
	this.navigationMembers.push(document.getElementById("b:pm:" + this._uuid));
	this.navigationMembers.push(document.getElementById("b:nm:" + this._uuid));
	this.navigationMembers.push(document.getElementById("b:ny:" + this._uuid));

	this.setNavigationInTabGroup(true);

	for (var i = 0; i < 42; i++) {
		var dayElement = document.getElementById(this._getDayCellId(i));
		if (dayElement) {
			this.dayMembers.push(dayElement);
		}
	}
};

DwtCalendar.prototype.__doBlur = function(ev) {
	DwtControl.prototype.__doBlur.call(this, ev);
	this._hasFocus = true;
	return true;
};

DwtCalendar.prototype.assignToTagGroup =
function(element) {
	this.tabgroup.addMember(element);
};

DwtCalendar.prototype.setDayMemberInTabGroup =
function(action, nextMember, setFocus) {
	if (!this.currentDayMemberInTabGroup) {
		this.assignToTagGroup(nextMember);
	}
	else if (!action) {
		this.tabgroup.replaceMember(this.currentDayMemberInTabGroup, nextMember);
	}
	else if (action){
		var index = this.dayMembers.indexOf(this.currentDayMemberInTabGroup);
		var nextIndex = index + 1;
		if (action === DwtCalendar.KEY_RIGHT) {
			nextIndex = index + 1;
			if (nextIndex > 41) {
				// call next month
				if (parseInt(this.currentDayMemberInTabGroup.textContent) > 7) {
					nextIndex = nextIndex + 14;
				} else {
					nextIndex = nextIndex + 7;
				}
				this._nextMonth();
			}
		}
		else if (action === DwtCalendar.KEY_LEFT) {
			nextIndex = (index + this.dayMembers.length - 1);
			if (nextIndex === 41) {
				// call prev month
				var currentDay = parseInt(this.currentDayMemberInTabGroup.textContent);
				this._prevMonth();

				var matchFound = false;
				for (var i = 0 ; i < 3 ; i++) {
					if (parseInt(this.dayMembers[nextIndex].textContent) + 1 === currentDay) {
						matchFound = true;
						break;
					}
					nextIndex = nextIndex - 7;
				}

				if (!matchFound) {
					nextIndex = 34;
				}
			}
		}
		else if (action === DwtCalendar.KEY_UP) {
			nextIndex = index - 7;
			if (nextIndex < 0) {
				nextIndex = nextIndex + 42;
				// call prevMonth
				var currentDay = parseInt(this.currentDayMemberInTabGroup.textContent);
				this._prevMonth();
				for (var i = 0; i < 3; i++) {
					var prevElement = this.dayMembers[nextIndex];
					var enableElement = prevElement.classList.value.indexOf('DwtCalendarDay-grey') === -1;
					var prevDay = parseInt(prevElement.textContent);
					if ( enableElement && currentDay !== prevDay) {
						break;
					}
					nextIndex = nextIndex - 7;
				}

			}
		}
		else if (action === DwtCalendar.KEY_DOWN) {
			nextIndex = index + 7;
			if (nextIndex > 41) {
				// call next month
				var currentDay = parseInt(this.currentDayMemberInTabGroup.textContent);
				this._nextMonth();
				var matchFound = false;
				var updatedIndex = nextIndex;
				for (var i = 0; i < 3; i++) {
					var nextElement = this.dayMembers[updatedIndex%42]
					var enableNextElement = nextElement.classList.value.indexOf('DwtCalendarDay-grey') === -1;
					var nextDay = parseInt(nextElement.textContent);
					if ( enableNextElement && currentDay !== nextDay && currentDay < nextDay) {
						matchFound = true;
						break;
					}
					updatedIndex = updatedIndex + 7;
				}

				nextIndex = matchFound ? updatedIndex : nextIndex + 7;
			}
		}

		var nextMember = this.dayMembers[nextIndex % 42];
		this.tabgroup.replaceMember(this.currentDayMemberInTabGroup, nextMember);
	}
	this.currentDayMemberInTabGroup = nextMember;

	if (setFocus) {
		this.tabgroup.setFocusMember(this.currentDayMemberInTabGroup);
	}
};

DwtCalendar.prototype.setNavigationInTabGroup =
function (next, setFocus) {
	var index = this.navigationMembers.indexOf(this.currentNavigationMemberInTabGroup);
	var nextIndex = next ? index + 1 : (index + this.navigationMembers.length - 1);
	var nextMember = this.navigationMembers[nextIndex%4];
	if (this.currentNavigationMemberInTabGroup) {
		this.tabgroup.replaceMember(this.currentNavigationMemberInTabGroup, nextMember);
	} else {
		this.assignToTagGroup(nextMember);
	}
	this.currentNavigationMemberInTabGroup = nextMember;
	if (setFocus) {
		this.tabgroup.setFocusMember(this.currentNavigationMemberInTabGroup);
	}
}

/**
 * Sets the mouse over day callback.
 * 
 * @param	{AjxCallback}		callback		the callback
 */
DwtCalendar.prototype.setMouseOverDayCallback =
function(callback) {
	this._mouseOverDayCB = callback;
};

/**
 * Sets the mouse out day callback.
 * 
 * @param	{AjxCallback}		callback		the callback
 */
DwtCalendar.prototype.setMouseOutDayCallback =
function(callback) {
	this._mouseOutDayCB = callback;
};

/**
 * Gets the date value for the last cell that the most recent
 * Drag-and-drop operation occurred over. Typically it will be called by a DwtDropTarget
 * listener when an item is dropped onto the mini calendar
 * 
 * @return	{Date}		the date or <code>null</code> for none
 */
DwtCalendar.prototype.getDndDate =
function() {
	var dayCell = this._lastDndCell;
	if (dayCell) {
		return new Date(dayCell._year, dayCell._month, dayCell._day);
	}

	return null;
};

// Temp date used for callback in mouseOverListener
DwtCalendar._tmpDate = new Date();
DwtCalendar._tmpDate.setHours(0, 0, 0, 0);

DwtCalendar.prototype._mouseOverListener = 
function(ev) {
	var target = ev.target;
	if (target.id.charAt(0) == 'c') {
		this._setClassName(target, DwtCalendar._HOVERED);
		// If a mouse over callback has been registered, then call it to give it
		// chance do work like setting the tooltip content
		if (this._mouseOverDayCB) {
			DwtCalendar._tmpDate.setFullYear(target._year, target._month, target._day);
			this._mouseOverDayCB.run(this, DwtCalendar._tmpDate);
		}
	} else if (target.id.charAt(0) == 't') {
		// Dont activate title for now
		return;
	} else if (target.id.charAt(0) == 'b') {
		var img;
		if (target.firstChild == null) {
			img = target;
			AjxImg.getParentElement(target).className = DwtCalendar._BUTTON_HOVERED_CLASS;
		} else {
			target.className = DwtCalendar._BUTTON_HOVERED_CLASS;
			img = AjxImg.getImageElement(target);
		}
		img.className = img._origClassName;
	}

	ev._stopPropagation = true;
};

DwtCalendar.prototype._mouseOutListener = 
function(ev) {
	this.setToolTipContent(null);
	var target = ev.target;
	if (target.id.charAt(0) == 'c') {
		this._setClassName(target, DwtCalendar._NORMAL);
		if (this._mouseOutDayCB) {
			this._mouseOutDayCB.run(this);
		}
	} else if (target.id.charAt(0) == 'b') {
		var img;
		target.className = DwtCalendar._BUTTON_CLASS;
		if (target.firstChild == null) {
			img = target;
			AjxImg.getParentElement(target).className = DwtCalendar._BUTTON_CLASS;
		} else {
			target.className = DwtCalendar._BUTTON_CLASS;
			img = AjxImg.getImageElement(target);
		}
		img.className = img._origClassName;
	}
};

DwtCalendar.prototype._mouseDownListener = 
function(ev) {
	if (ev.button == DwtMouseEvent.LEFT) {
		var target = ev.target;
		if (target.id.charAt(0) == 'c') {
			this._setClassName(target, DwtCalendar._ACTIVE);
		} else if (target.id.charAt(0) == 't') {
			target.className = DwtCalendar._TITLE_ACTIVE_CLASS;
		} else if (target.id.charAt(0) == 'b') {
			var img;
			if (target.firstChild == null) {
				img = target;
				AjxImg.getParentElement(target).className = DwtCalendar._BUTTON_ACTIVE_CLASS;
			} else {
				target.className = DwtCalendar._BUTTON_ACTIVE_CLASS;
				img = AjxImg.getImageElement(target);
			}
			img.className = img._origClassName;
		} else if (target.id.charAt(0) == 'w') {
		}
	}
};

DwtCalendar.prototype._mouseUpListener = 
function(ev) {
	var target = ev.target;
	if (ev.button == DwtMouseEvent.LEFT) {
		if (target.id.charAt(0) == 'c') {
			// If our parent is a menu then we need to have it close
			if (this.parent instanceof DwtMenu)
				DwtMenu.closeActiveMenu();

            var sDate = new Date(target._year, target._month, target._day);
            if(sDate.getDate() != target._day) {
                sDate.setDate(target._day);                 
            }
			if (this.setDate(sDate)) { return; }

			this._setClassName(target, DwtCalendar._HOVERED);
		} else if (target.id.charAt(0) == 'b') {
			var img;
			if (target.firstChild == null) {
				img = target;
				AjxImg.getParentElement(target).className = DwtCalendar._BUTTON_HOVERED_CLASS;
			} else {
				target.className = DwtCalendar._BUTTON_HOVERED_CLASS;
				img = AjxImg.getImageElement(target);
			}
			img.className = img._origClassName;
			
			if (img.id.indexOf("py") != -1) {
				this._prevYear();
			} else if (img.id.indexOf("pm") != -1) {
				this._prevMonth();
			} else if (img.id.indexOf("nm") != -1) {
				this._nextMonth();
			} else {
				this._nextYear();
			}
		} else if (target.id.charAt(0) == 't') {
			// TODO POPUP MENU
			target.className = DwtCalendar._TITLE_HOVERED_CLASS;
			this.setDate(new Date(), this._skipNotifyOnPage);
			// If our parent is a menu then we need to have it close
			if (this.parent instanceof DwtMenu) {
				DwtMenu.closeActiveMenu();
			}
		}
	} else if (ev.button == DwtMouseEvent.RIGHT && target.id.charAt(0) == 'c') {
		this._notifyListeners(DwtEvent.ACTION, 0, new Date(target._year, target._month, target._day), ev);
	}
};

DwtCalendar.prototype._doubleClickListener =
function(ev) {
	var target = ev.target;
	if (this._selectionEvent) {
		this._selectionEvent.type = DwtCalendar.DATE_DBL_CLICKED;
	}
	if (target.id.charAt(0) == 'c') {
		// If our parent is a menu then we need to have it close
		if (this.parent instanceof DwtMenu) {
			DwtMenu.closeActiveMenu();
		}
		this.setDate(new Date(target._year, target._month, target._day), false, false, true)
	}
};

DwtCalendar.prototype._prevMonth = 
function(ev) {
	var d = new Date(this._date.getTime());
	this.setDate(AjxDateUtil.roll(d, AjxDateUtil.MONTH, -1), this._skipNotifyOnPage);
};

DwtCalendar.prototype._nextMonth = 
function(ev) {
	var d = new Date(this._date.getTime());
	this.setDate(AjxDateUtil.roll(d, AjxDateUtil.MONTH, 1), this._skipNotifyOnPage);
};

DwtCalendar.prototype._prevYear = 
function(ev) {
	var d = new Date(this._date.getTime());
	this.setDate(AjxDateUtil.roll(d, AjxDateUtil.YEAR, -1), this._skipNotifyOnPage);
};

DwtCalendar.prototype._nextYear = 
function(ev) {
	var d = new Date(this._date.getTime());
	this.setDate(AjxDateUtil.roll(d, AjxDateUtil.YEAR, 1), this._skipNotifyOnPage);
};

/**
 * Gets the date formatter.
 * 
 * @return	{AjxDateFormat}		the date formatter
 * 
 * @private
 */
DwtCalendar.getDateFormatter =
function() {
	if (!DwtCalendar._dateFormatter) {
		DwtCalendar._dateFormatter = new AjxDateFormat(AjxMsg.formatCalDate);
	}
	return DwtCalendar._dateFormatter;
};

/**
 * Gets the date long formatter.
 * 
 * @return	{AjxDateFormat}		the date formatter
 * 
 * @private
 */
DwtCalendar.getDateLongFormatter =
function() {
	if (!DwtCalendar._dateLongFormatter) {
		DwtCalendar._dateLongFormatter = new AjxDateFormat(AjxMsg.formatCalDateLong);
	}
	return DwtCalendar._dateLongFormatter;
};

/**
 * Gets the date full formatter.
 * 
 * @return	{AjxDateFormat}		the date formatter
 * 
 * @private
 */
DwtCalendar.getDateFullFormatter =
function() {
	if (!DwtCalendar._dateFullFormatter) {
		DwtCalendar._dateFullFormatter = new AjxDateFormat(AjxMsg.formatCalDateFull);
	}
	return DwtCalendar._dateFullFormatter;
};

/**
 * Gets the hour formatter.
 * 
 * @return	{AjxDateFormat}		the date formatter
 * 
 * @private
 */
DwtCalendar.getHourFormatter =
function() {
	if (!DwtCalendar._hourFormatter) {
		DwtCalendar._hourFormatter = new AjxMessageFormat(AjxMsg.formatCalHour);
	}
	return DwtCalendar._hourFormatter;
};

/**
 * Gets the day formatter.
 * 
 * @return	{AjxDateFormat}		the date formatter
 * 
 * @private
 */
DwtCalendar.getDayFormatter =
function() {
	if (!DwtCalendar._dayFormatter) {
		DwtCalendar._dayFormatter = new AjxDateFormat(AjxMsg.formatCalDay);
	}
	return DwtCalendar._dayFormatter;
};

/**
 * Gets the month formatter.
 * 
 * @return	{AjxDateFormat}		the date formatter
 * 
 * @private
 */
DwtCalendar.getMonthFormatter =
function() {
	if (!DwtCalendar._monthFormatter) {
		DwtCalendar._monthFormatter = new AjxDateFormat(AjxMsg.formatCalMonth);
	}
	return DwtCalendar._monthFormatter;
};

/**
 * Gets the short month formatter.
 * 
 * @return	{AjxDateFormat}		the date formatter
 * 
 * @private
 */
DwtCalendar.getShortMonthFormatter =
function() {
	if (!DwtCalendar._shortMonthFormatter) {
		DwtCalendar._shortMonthFormatter = new AjxDateFormat(AjxMsg.formatShortCalMonth);
	}
	return DwtCalendar._shortMonthFormatter;
};

DwtCalendar.prototype._dragEnter =
function(ev) {
};

DwtCalendar.prototype._dragHover =
function(ev) {
};

DwtCalendar.prototype._dragOver =
function(ev) {
	var target = ev.target;
	if (target.id.charAt(0) == 'c') {
		this._setClassName(target, DwtCalendar._HOVERED);
		this._lastDndCell = target;
	} else {
		this._lastDndCell = null;
	}
};

DwtCalendar.prototype._dragLeave =
function(ev) {
};
}
if (AjxPackage.define("ajax.dwt.widgets.DwtMessageComposite")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */


/**
 * Creates a composite that is populated from a message pattern.
 * @constructor
 * @class
 * This class allows you to create a composite that is populated from
 * a message pattern and inserts controls at the appropriate places.
 * For example, say that the message <code>MyMsg.repeatTimes</code> is
 * defined as the following:
 * <pre>
 * MyMsg.repeatTimes = "Repeat: {0} times";
 * </pre>
 * and you want to replace "{0}" with an input field or perhaps a
 * drop-down menu that enumerates a specific list of choices as part of
 * the application. To do this, you just create a
 * {@link DwtMessageComposite} and set the message format, like so:
 * <pre>
 * var comp = new DwtMessageComposite(parent);
 * comp.setFormat(MyMsg.repeatTimes);
 * </pre>
 * <p>
 * The message composite instantiates an {@link AjxMessageFormat}
 * from the specified message pattern. Then, for each segment it creates
 * static text or a {@link DwtInputField} for replacement segments
 * such as "{0}".
 * <p>
 * To have more control over the controls that are created and inserted
 * into the resulting composite, you can pass a callback object to the
 * method. Each time that a replacement segment is found in the
 * message pattern, the callback is called with the following parameters:
 * <ul>
 * <li>a reference to this message composite object;
 * <li>a reference to the segment object.
 * <li>the index at which the segment was found in the message pattern; and
 * </ul>
 * The segment object will be an instance of
 * <code>AjxMessageFormat.MessageSegment</code> and has the following
 * methods of interest:
 * <ul>
 * <li>toSubPattern
 * <li>getIndex
 * <li>getType
 * <li>getStyle
 * <li>getSegmentFormat
 * </ul>
 * <p>
 * The callback can use this information to determine whether or not
 * a custom control should be created for the segment. If the callback
 * returns <code>null</code>, a standard {@link DwtInputField} is
 * created and inserted. Note: if the callback returns a custom control,
 * it <em>must</em> be an instance of {@link AjxControl}.
 * <p>
 * Here is an example of a message composite created with a callback
 * that generates a custom control for each replacement segment:
 * <pre>
 * function createCustomControl(parent, segment, i) {
 *     return new DwtInputField(parent);
 * }
 *
 * var compParent = ...;
 * var comp = new DwtMessageComposite(compParent);
 *
 * var message = MyMsg.repeatTimes;
 * var callback = new AjxCallback(null, createCustomControl);
 * comp.setFormat(message, callback);
 * </pre>
 *
 * @author Andy Clark
 *
 * @param {Object}		params		hash of params:
 * @param {DwtComposite}	parent    the parent widget.
 * @param {string}	className 	the CSS class
 * @param {constant}	posStyle  		the position style (see {@link DwtControl})
 * @param {DwtComposite}	parent    the parent widget.
 * @param {string}	format   the message that defines the text and controls within this composite control
 * @param {AjxCallback}	[controlCallback]   the callback to create UI components (only used with format specified)
 * @param {AjxCallback}	[hintsCallback]   the callback to provide display hints for the container element of the UI component (only used with format specified)
 * 
 * @extends		DwtComposite
 */
DwtMessageComposite = function(params) {
	if (arguments.length == 0) return;

	params = Dwt.getParams(arguments, DwtMessageComposite.PARAMS);

	if (!params.className) {
		params.className = "DwtMessageComposite";
	}

	DwtComposite.call(this, params);

	this._tabGroup = new DwtTabGroup("DwtMessageComposite");

	if (params.format) {
		this.setFormat(params.format,
		               params.controlCallback,
		               params.hintsCallback);
	}
}

DwtMessageComposite.PARAMS = ['parent', 'className', 'posStyle'];

DwtMessageComposite.prototype = new DwtComposite;
DwtMessageComposite.prototype.constructor = DwtMessageComposite;
DwtMessageComposite.prototype.isDwtMessageComposite = true;

DwtMessageComposite.prototype.toString =
function() {
	return "DwtMessageComposite";
}

// Public methods

/**
 * Sets the format.
 * 
 * @param {string}	message   the message that defines the text and controls that comprise this composite
 * @param {AjxCallback}	[callback]   the callback to create UI components
 * @param {AjxCallback}	[hintsCallback]   the callback to provide display hints for the container element of the UI component
 */
DwtMessageComposite.prototype.setFormat =
function(message, callback, hintsCallback) {
    // create formatter
    this._formatter = new AjxMessageFormat(message);
    this._controls = {};

    // create HTML
    var id = this._htmlElId;
    this.getHtmlElement().innerHTML = "<table role='presentation' class='DwtCompositeTable' border='0' cellspacing='0' cellpadding='0'><tr valign='center'></tr></table>";
    var row = this.getHtmlElement().firstChild.rows[0];

    var segments = this._formatter.getSegments();
    for (var i = 0; i < segments.length; i++) {
        var segment = segments[i];
        var isMsgSegment = segment instanceof AjxMessageFormat.MessageSegment;

        var cid = [id,i].join("_");
        var cell = document.createElement('TD');

        cell.id = cid;
        cell.className = 'DwtCompositeCell';
        row.appendChild(cell);

        if (isMsgSegment) {
            cell.className += ' MessageControl' + segment.getIndex();
            var control = callback ? callback.run(this, segment, i) : null;
            if (!control) {
                control = new DwtInputField({parent:this, parentElement: cell});
            } else {
                control.reparentHtmlElement(cell);
            }
            this._tabGroup.addMember(control.getTabGroupMember());
            if (hintsCallback) {
                var hints = hintsCallback.run(this, segment, i);

                AjxUtil.hashUpdate(control.getHtmlElement(), hints, true);
            }

            var sindex = segment.getIndex();
            this._controls[sindex] = this._controls[sindex] || control;
        }
        else {
            control = new DwtText({parent:this, parentElement: cell});
            control.setText(segment.toSubPattern());
            this._tabGroup.addMember(control);
        }
    }
};

/**
 * Gets the format.
 * 
 * @return	{string}	the format
 */
DwtMessageComposite.prototype.format = function() {
    var args = [];
    for (var sindex in this._controls) {
        args[sindex] = this._controls[sindex].getValue();
    }
    return this._formatter.format(args);
};

DwtMessageComposite.prototype.getTabGroupMember = function() {
	return this._tabGroup;
};
}

if (AjxPackage.define("ajax.util.AjxDateUtil")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * 
 * @private
 */
AjxDateUtil = function() {
};

AjxDateUtil.YEAR		= 1;
AjxDateUtil.MONTH		= 2;
AjxDateUtil.WEEK		= 3;
AjxDateUtil.DAY			= 4;
AjxDateUtil.TWO_WEEKS	= 5;

AjxDateUtil.MSEC_PER_MINUTE = 60000;
AjxDateUtil.MSEC_PER_FIFTEEN_MINUTES = 900000;
AjxDateUtil.MSEC_PER_HALF_HOUR = 1800000;
AjxDateUtil.MSEC_PER_HOUR = 3600000;
AjxDateUtil.MSEC_PER_DAY = 24 * AjxDateUtil.MSEC_PER_HOUR;

AjxDateUtil.MINUTES_PER_DAY = 60 * 24;
AjxDateUtil.SECONDS_PER_DAY = 60 * 60 * 24;

AjxDateUtil.DAYS_PER_WEEK = 7;

AjxDateUtil.WEEKDAY_SHORT = AjxDateFormat.WeekdaySegment.WEEKDAYS[AjxDateFormat.SHORT];
AjxDateUtil.WEEKDAY_MEDIUM = AjxDateFormat.WeekdaySegment.WEEKDAYS[AjxDateFormat.MEDIUM];
AjxDateUtil.WEEKDAY_LONG = AjxDateFormat.WeekdaySegment.WEEKDAYS[AjxDateFormat.LONG];

AjxDateUtil.MONTH_SHORT = AjxDateFormat.MonthSegment.MONTHS[AjxDateFormat.SHORT];
AjxDateUtil.MONTH_MEDIUM = AjxDateFormat.MonthSegment.MONTHS[AjxDateFormat.MEDIUM];
AjxDateUtil.MONTH_LONG = AjxDateFormat.MonthSegment.MONTHS[AjxDateFormat.LONG];

AjxDateUtil._daysPerMonth = {
	0:31,
	1:29,
	2:31,
	3:30,
	4:31,
	5:30,
	6:31,
	7:31,
	8:30,
	9:31,
	10:30,
	11:31
};

AjxDateUtil.MAX_DAYS_PER_MONTH = 31;

AjxDateUtil.WEEK_ONE_JAN_DATE = 1;

AjxDateUtil._init =
function() {
	AjxDateUtil._dateFormat = AjxDateFormat.getDateInstance(AjxDateFormat.SHORT).clone();
	var segments = AjxDateUtil._dateFormat.getSegments();
	for (var i = 0; i < segments.length; i++) {
		if (segments[i] instanceof AjxDateFormat.YearSegment) {
			segments[i] = new AjxDateFormat.YearSegment(AjxDateUtil._dateFormat, "yyyy");
		}
	}

	AjxDateUtil._dateTimeFormat = new AjxDateFormat(I18nMsg.formatDateTimeShort);
	segments = AjxDateUtil._dateTimeFormat.getSegments();
	for (i = 0; i < segments.length; i++) {
		if (segments[i] instanceof AjxDateFormat.YearSegment) {
			segments[i] = new AjxDateFormat.YearSegment(AjxDateUtil._dateTimeFormat, "yyyy");
		}
	}

	AjxDateUtil._dateFormatNoYear = new AjxDateFormat(AjxMsg.formatDateMediumNoYear);
	AjxDateUtil._dateTimeFormatNoYear = new AjxDateFormat(AjxMsg.formatDateTimeMediumNoYear);
};

AjxDateUtil._init();                    

/* return true if the specified date (yyyy|yy, m (0-11), d (1-31)) 
 * is valid or not.
 */
AjxDateUtil.validDate =
function(y, m, d) {
	var date = new Date(y, m, d);
	var year = y > 999 ? date.getFullYear() : date.getYear();
	return date.getMonth() == m && date.getDate() == d && year == y;
};

/* return number of days (1-31) in specified month (yyyy, mm (0-11))
 */
AjxDateUtil.daysInMonth =
function(y, m) {
	var date = new Date(y, m, 1, 12);
	date.setMonth(date.getMonth()+1);
	date.setDate(date.getDate()-1);
	return date.getDate();
};

/* return true if year is a leap year
 */
AjxDateUtil.isLeapYear =
function(y) {
	return (new Date(y, 1, 29)).getMonth() == 1;
};

/* returns true if user's locale uses 24-hour time
 */
AjxDateUtil.isLocale24Hour =
function() {
	// XXX: is there better/easier way to determine this?!
	var timeFormatter = AjxDateFormat.getTimeInstance(AjxDateFormat.SHORT);
	var len = timeFormatter._segments.length;
	for (var j = 0; j < len; j++) {
		if (timeFormatter._segments[j]._s == "a")
			return false;
	}
	return true;
};

/**
 * rolls the month/year. If the day of month in the date passed in is greater
 * then the max day in the new month, set it to the max. The date passed in is
 * modified and also returned.
 */
AjxDateUtil.roll = 
function(date, field, offset) {
	var d = date.getDate();
	 // move back to first day before rolling in case previous
	 // month/year has less days

	if (field == AjxDateUtil.MONTH) {
		date.setDate(1);	
		date.setMonth(date.getMonth() + offset);
		var max = AjxDateUtil.daysInMonth(date.getFullYear(), date.getMonth());
		date.setDate(Math.min(d, max));		
	} else if (field == AjxDateUtil.YEAR) {
		date.setDate(1);		
		date.setFullYear(date.getFullYear() + offset);
		var max = AjxDateUtil.daysInMonth(date.getFullYear(), date.getMonth());
		date.setDate(Math.min(d, max));		
	} else if (field == AjxDateUtil.WEEK) {
		date.setDate(date.getDate() + 7*offset);
	} else if (field == AjxDateUtil.DAY) {
		date.setDate(date.getDate() + offset);
	} else if (field == AjxDateUtil.TWO_WEEKS) {
		date.setDate(date.getDate() + 14*offset);
	} else {
		return date;
	}
	return date;
};

/**
 * checks whether given date is derived from DST shift
 */
AjxDateUtil.isDayShifted =
function(date) {
    var refDate = new Date(date.getTime());

    //advance it by 1 day and reset to beginning of the day
    refDate.setDate(refDate.getDate() +1)
    refDate.setHours(0,0,0,0);

    //if DST has no effect the advanced time should differ from given time
    return refDate.getTime() == date.getTime();
};

/**
 * rolls to next day. This can be used to roll to next day avoiding the daylight saving shift in time.
 */
AjxDateUtil.rollToNextDay =
function(date) {
    date.setHours(0,0,0,0);
    date.setTime(date.getTime() + AjxDateUtil.MSEC_PER_DAY);
};

// Computes the difference between now and <dateMSec>. Returns a string describing
// the difference
AjxDateUtil.computeDateDelta =
function(dateMSec) {
	var deltaMSec = (new Date()).getTime() - dateMSec;
	var durationStr = AjxDateUtil.computeDuration(deltaMSec);
	return durationStr ? (durationStr + " " + AjxMsg.ago) : null;
};

// Computes the difference between now and <dateMSec>. Returns a simplified string describing
// the difference
AjxDateUtil.agoTime =
function(dateMSec) {
	var deltaMSec = (new Date()).getTime() - dateMSec;
	var durationStr = AjxDateUtil.computeDuration(deltaMSec, false, true);
	return durationStr ? (durationStr + " " + AjxMsg.ago) : null;
};



// Returns a string describing the duration, which is in milliseconds.
AjxDateUtil.computeDuration =
function(duration, brief, simplified) {
	// bug fix #2203 - if delta is less than zero, dont bother computing
	if (duration < 0) return null;

	var years =  Math.floor(duration / (AjxDateUtil.MSEC_PER_DAY * 365));
	if (years != 0)
		duration -= years * AjxDateUtil.MSEC_PER_DAY * 365;
	var months = Math.floor(duration / (AjxDateUtil.MSEC_PER_DAY * 30.42));
	if (months > 0)
		duration -= Math.floor(months * AjxDateUtil.MSEC_PER_DAY * 30.42);
	var days = Math.floor(duration / AjxDateUtil.MSEC_PER_DAY);
	if (days > 0)
		duration -= days * AjxDateUtil.MSEC_PER_DAY;
	var hours = Math.floor(duration / AjxDateUtil.MSEC_PER_HOUR);
	if (hours > 0) 
		duration -= hours * AjxDateUtil.MSEC_PER_HOUR;
	var mins = Math.floor(duration / 60000);
	if (mins > 0)
		duration -= mins * 60000;
	var secs = Math.floor(duration / 1000);

	var formatter = brief ? AjxDurationFormatConcise : AjxDurationFormatVerbose;
	if (years > 0) {
		return simplified
            ? formatter.formatYears(years)
            : formatter.formatYears(years, months);
	} else if (months > 0) {
		return simplified
            ? formatter.formatMonths(months)
            : formatter.formatMonths(months, days);
	} else if (days > 0) {
		return simplified
            ? formatter.formatDays(days)
            : formatter.formatDays(days, hours);
	} else if (hours > 0) {
		return simplified
            ? formatter.formatHours(hours)
            : formatter.formatHours(hours, mins);
	} else if (mins > 0) {
		return simplified
            ? formatter.formatMinutes(mins)
            : formatter.formatMinutes(mins, secs);
	} else {
		return formatter.formatSeconds(secs);
	}
};

AjxDateUtil.simpleComputeDateStr = 
function(date, stringToPrepend) {
	var dateStr = AjxDateUtil._dateFormat.format(date);
	return stringToPrepend ? stringToPrepend + dateStr : dateStr;
};
AjxDateUtil.simpleParseDateStr =
function(dateStr) {
	return AjxDateUtil._dateFormat.parse(dateStr);
};

AjxDateUtil.simpleComputeDateTimeStr = 
function(date, stringToPrepend) {
	var dateTimeStr = AjxDateUtil._dateTimeFormat.format(date);
	return stringToPrepend ? stringToPrepend + dateTimeStr : dateTimeStr;
};
AjxDateUtil.simpleParseDateTimeStr =
function(dateTimeStr) {
	return AjxDateUtil._dateTimeFormat.parse(dateTimeStr);
};

AjxDateUtil.longComputeDateStr = 
function(date) {
	var formatter = AjxDateFormat.getDateInstance(AjxDateFormat.FULL);
	return formatter.format(date);
}

AjxDateUtil.computeDateStr =
function(now, dateMSec, requireTime) {
	if (dateMSec == null)
		return "";

	var result = { value: null };
	appCtxt.notifyZimlets("onAjxDateUtil_computeDateStr", [now, dateMSec, requireTime, result]);
	if (result.value) {
		return result.value;
	}

	var date = new Date(dateMSec);
	if (now.getTime() - dateMSec < AjxDateUtil.MSEC_PER_DAY &&
		now.getDay() == date.getDay()) {
		return AjxDateUtil.computeTimeString(date);
	}

	if (now.getFullYear() == date.getFullYear()) {
		if (requireTime) {
			return AjxDateUtil._dateTimeFormatNoYear.format(date);
		} else {
			return AjxDateUtil._dateFormatNoYear.format(date);
		}
	}

	if (requireTime) {
		return AjxDateUtil.simpleComputeDateTimeStr(date);
	} else {
		return AjxDateUtil.simpleComputeDateStr(date);
	}
};

AjxDateUtil.computeDateStrNoYear =
function(date) {
    return AjxDateUtil._dateFormatNoYear.format(date);
};

// Example output: "Today, 9:44 AM" "Yesterday, 12:22 PM" "Sun, 1/11/01 1:11 PM"
AjxDateUtil.computeWordyDateStr =
function(now, dateMSec) {
	if (dateMSec == null) {
		return "";
	}

	var date = new Date(dateMSec);
	if (now.getTime() - dateMSec < AjxDateUtil.MSEC_PER_DAY && now.getDay() == date.getDay()) {
		if (!AjxDateUtil._wordyDateToday) {
			AjxDateUtil._wordyDateToday = new AjxDateFormat(AjxMsg.formatWordyDateToday);
		}
		return AjxDateUtil._wordyDateToday.format(date);
	} else if ((now.getTime() - dateMSec) < (2 * AjxDateUtil.MSEC_PER_DAY) && (now.getDay() - 1) == date.getDay()) {
		if (!AjxDateUtil._wordyDateYesterday) {
			AjxDateUtil._wordyDateYesterday = new AjxDateFormat(AjxMsg.formatWordyDateYesterday);
		}
		return AjxDateUtil._wordyDateYesterday.format(date);
	} else {
		if (!AjxDateUtil._wordyDate) {
			AjxDateUtil._wordyDate = new AjxDateFormat(AjxMsg.formatWordyDate);
		}
		return AjxDateUtil._wordyDate.format(date);
	}
};

/* returns true if dateString is a valid and understandable date string
 * in compliance with the locale of the user ie. dd/mm/yy or mm/dd/yy etc.
 * Also for date strings like 1/32/2000 (that roll over to 2/1/2000), false is returned.
 */
AjxDateUtil.isValidSimpleDateStr =
function(str){
        if(!str) {return false};
        var dateValue = AjxDateUtil.getSimpleDateFormat().parse(str);
        if (!dateValue) {return false};
        var dateValueStr = AjxDateUtil.simpleComputeDateStr(dateValue);
        return (str == dateValueStr);
}

AjxDateUtil.computeTimeString =
function(date) {
	var formatter = AjxDateFormat.getTimeInstance(AjxDateFormat.SHORT);
	return formatter.format(date);
};

AjxDateUtil.computeDateTimeString =
function(date) {
	var formatter = AjxDateFormat.getDateTimeInstance(AjxDateFormat.LONG);
	return formatter.format(date);
};

AjxDateUtil._getHoursStr =
function(date, pad, useMilitary) {
	var myVal = date.getHours();
	if (!useMilitary) {
		myVal %= 12;
		if (myVal == 0) myVal = 12;
	}
	return pad ? AjxDateUtil._pad(myVal) : myVal;
};

AjxDateUtil._getMinutesStr = 
function(date) {
	return AjxDateUtil._pad(date.getMinutes());
};

AjxDateUtil._getSecondsStr = 
function(date) {
	return AjxDateUtil._pad(date.getSeconds());
};

AjxDateUtil._getAMPM = 
function (date, upper) {
	var myHour = date.getHours();
	return (myHour < 12) ? (upper ? 'AM' : 'am') : (upper ? 'PM' : 'pm');
};

AjxDateUtil._getMonthName = 
function(date, abbreviated) {
	return abbreviated
		? AjxDateUtil.MONTH_MEDIUM[date.getMonth()]
		: AjxDateUtil.MONTH_LONG[date.getMonth()];
};

AjxDateUtil._getMonth = 
function(date, pad) {
	var myMonth = date.getMonth() + 1;
	if (pad) {
		return AjxDateUtil._pad(myMonth);
	} else {
		return myMonth;
	}
};

AjxDateUtil._getDate = 
function(date, pad) {
	var myVal = date.getDate();
	return pad ? AjxDateUtil._pad(myVal) : myVal;
};

AjxDateUtil._getWeekday =
function (date) {
	var myVal = date.getDay();
	return AjxDateUtil.WEEKDAY_LONG[myVal];
};

// Returns "Mon", "Tue", etc.
AjxDateUtil._getWeekdayMedium =
function (date) {
	var myVal = date.getDay();
	return AjxDateUtil.WEEKDAY_MEDIUM[myVal];
};

AjxDateUtil._getFullYear =
function(date) {
	return date.getFullYear();
};

AjxDateUtil.getFirstDayOfWeek =
function (dt, startOfWeek) {
    startOfWeek = startOfWeek || 0;
    var dayOfWeekIndex = dt.getDay();
    var dayOfWeek = (dayOfWeekIndex - startOfWeek + 7) % 7;
    dt.setDate(dt.getDate() - dayOfWeek);
    return dt;
};

AjxDateUtil.getLastDayOfWeek =
function (dt, startOfWeek) {
    startOfWeek = startOfWeek || 0;
    var dayOfWeekIndex = dt.getDay();
    var dayOfWeek = (dayOfWeekIndex - startOfWeek + 7) % 7;
    dt.setDate(dt.getDate() - dayOfWeek + 6);
    dt.setHours(23, 59, 59, 999);
    return dt;
};

AjxDateUtil.getWeekNumber =
function(date, firstDayOfWeek, janDate, isISO8601WeekNum) {

    // Setup Defaults
    firstDayOfWeek = firstDayOfWeek || 0;
    janDate = janDate || AjxDateUtil.WEEK_ONE_JAN_DATE;
    date = date || new Date();

    date.setHours(12,0,0,0);
    var targetDate = date,
            startOfWeek,
            endOfWeek;

    if (targetDate.getDay() === firstDayOfWeek) {
        startOfWeek = targetDate;
    } else {
        startOfWeek = AjxDateUtil.getFirstDayOfWeek(targetDate, firstDayOfWeek);
    }

    var startYear = startOfWeek.getFullYear(),
            startTime = startOfWeek.getTime();

    // DST shouldn't be a problem here, math is quicker than setDate();
    endOfWeek = new Date(startOfWeek.getTime() + 6*AjxDateUtil.MSEC_PER_DAY);

    var weekNum;

    if(!isISO8601WeekNum) {
        if (startYear !== endOfWeek.getFullYear() && endOfWeek.getDate() >= janDate) {
            weekNum = 1;
        } else {
            var weekOne = (new Date(startYear, 0, janDate));
            weekOne.setHours(12,0,0,0);
            var weekOneDayOne = AjxDateUtil.getFirstDayOfWeek(weekOne, firstDayOfWeek);

            // Round days to smoothen out 1 hr DST diff
            var daysDiff  = Math.round((targetDate.getTime() - weekOneDayOne.getTime())/AjxDateUtil.MSEC_PER_DAY);

            // Calc. Full Weeks
            var rem = daysDiff % 7;
            var weeksDiff = (daysDiff - rem)/7;
            weekNum = weeksDiff + 1;
        }
        return weekNum;
    }else {

        var newYear = new Date(date.getFullYear(),0,1);
        var day = newYear.getDay() - 1;
        day = (day >= 0 ? day : day + 7);
        var dayOftheYear = Math.floor((date.getTime()-newYear.getTime() - (date.getTimezoneOffset()-newYear.getTimezoneOffset())*60000)/AjxDateUtil.MSEC_PER_DAY) + 1;

        if(day < 4)
        {
            weekNum = Math.floor((dayOftheYear+day-1)/7) + 1;
            if(weekNum > 52)
            {
                var nxtYear = new Date(date.getFullYear() + 1,0,1);
                var nxtDay = nxtYear.getDay() - 1;
                nxtDay = nxtDay >= 0 ? nxtDay : nxtDay + 7;
                weekNum = nxtDay < 4 ? 1 : 53;
            }
        }else {
            weekNum = Math.floor((dayOftheYear+day -1 )/7);
            if(weekNum == 0)
            {
                var prevYear = new Date(date.getFullYear()-1,0,1);
                var prevDay = prevYear.getDay()-1;
                prevDay = (prevDay >= 0 ? prevDay : prevDay + 7);
                weekNum = ( prevDay==3 || ( AjxDateUtil.isLeapYear(prevYear.getFullYear()) && prevDay==2 ) ) ? 53 : 52;
            }
        }
        return weekNum;
    }
};

AjxDateUtil.getTimeStr = 
function(date, format) {
	var s = format;
	s = s.replace(/%d/g, AjxDateUtil._getDate(date, true));				// zero padded day of the month
	s = s.replace(/%D/g, AjxDateUtil._getDate(date, false));			// day of the month without padding
	s = s.replace(/%w/g, AjxDateUtil._getWeekday(date));				// day of the week
	s = s.replace(/%M/g, AjxDateUtil._getMonthName(date));				// full month name
	s = s.replace(/%t/g, AjxDateUtil._getMonthName(date, true));		// abbr. month name
	s = s.replace(/%n/g, AjxDateUtil._getMonth(date, true));		    // zero padded month
	s = s.replace(/%Y/g, AjxDateUtil._getFullYear(date));				// full year
	s = s.replace(/%h/g, AjxDateUtil._getHoursStr(date, false, false));	// non-padded hours
	s = s.replace(/%H/g, AjxDateUtil._getHoursStr(date, true, false ));	// padded hours
	s = s.replace(/%m/g, AjxDateUtil._getMinutesStr(date));				// padded minutes
	s = s.replace(/%s/g, AjxDateUtil._getSecondsStr(date));				// padded seconds
	s = s.replace(/%P/g, AjxDateUtil._getAMPM(date, true));				// upper case AM PM
	s = s.replace(/%p/g, AjxDateUtil._getAMPM(date, false));			// lower case AM PM
	return s;
};

AjxDateUtil.getRoundedMins = 
function (date, roundTo) {
	var mins = date.getMinutes();
	if (mins != 0 && roundTo)
		mins = (Math.ceil( (mins/roundTo) )) * roundTo;
	return mins;
};

AjxDateUtil.roundTimeMins = 
function(date, roundTo) {
	var mins = date.getMinutes();
	var hours = date.getHours();
	if (mins != 0 && roundTo){
		mins = (Math.ceil( (mins/roundTo) )) * roundTo;
		if (mins == 60) {
			mins = 0;
			hours++;
		}
		date.setMinutes(mins);
		date.setHours(hours);
	}
	return date;
};

AjxDateUtil.isInRange = 
function(startTime1, endTime1, startTime2, endTime2) {
	return (startTime1 < endTime2 && endTime1 > startTime2);
}

AjxDateUtil.getSimpleDateFormat =
function() {
	return AjxDateUtil._dateFormat;
};

/**
 * The following are helper routines for processing server date/time which comes
 * in this format: YYYYMMDDTHHMMSSZ
*/
AjxDateUtil.getServerDate = 
function(date) {
	if (!AjxDateUtil._serverDateFormatter) {
		AjxDateUtil._serverDateFormatter = new AjxDateFormat("yyyyMMdd");
	}
	return AjxDateUtil._serverDateFormatter.format(date);
};

AjxDateUtil.getServerDateTime = 
function(date, useUTC) {
	var newDate = date;
	var formatter = null;

	if (useUTC) {
		if (!AjxDateUtil._serverDateTimeFormatterUTC) {
			AjxDateUtil._serverDateTimeFormatterUTC = new AjxDateFormat("yyyyMMdd'T'HHmmss'Z'");
		}
		formatter = AjxDateUtil._serverDateTimeFormatterUTC;
		// add timezone offset to this UTC date
		newDate = new Date(date.getTime());
		newDate.setMinutes(newDate.getMinutes() + newDate.getTimezoneOffset());
	} else {
		if (!AjxDateUtil._serverDateTimeFormatter) {
			AjxDateUtil._serverDateTimeFormatter = new AjxDateFormat("yyyyMMdd'T'HHmmss");
		}
		formatter = AjxDateUtil._serverDateTimeFormatter;
	}

	return formatter.format(newDate);
};

AjxDateUtil.parseServerTime = 
function(serverStr, date, noSpecialUtcCase) {
	if (serverStr.charAt(8) == 'T') {
		var hh = parseInt(serverStr.substr(9,2), 10);
		var mm = parseInt(serverStr.substr(11,2), 10);
		var ss = parseInt(serverStr.substr(13,2), 10);
		if (!noSpecialUtcCase && serverStr.charAt(15) == 'Z') {
			mm += AjxTimezone.getOffset(AjxTimezone.DEFAULT, date);
		}
		date.setHours(hh, mm, ss, 0);
	}
	return date;
};

AjxDateUtil.parseISO8601Date = function(s) {
    var formatters = AjxDateUtil.__ISO8601_formats;
    if (!formatters) {
        formatters = AjxDateUtil.__ISO8601_formats = [
            new AjxDateUtil.TZDFormat("yyyy-MM-dd'T'HH:mm:ss.SZ"),
            new AjxDateUtil.TZDFormat("yyyy-MM-dd'T'HH:mm:ssZ"),
            new AjxDateUtil.TZDFormat("yyyy-MM-dd'T'HH:mmZ"),
            new AjxDateFormat("yyyy-MM-dd"),
            new AjxDateFormat("yyyy-MM"),
            new AjxDateFormat("yyyy")
        ];
    }
    for (var i = 0; i < formatters.length; i++) {
        var date = formatters[i].parse(s);
        if (date) return date;
    }
    return null;
};

AjxDateUtil.TZDFormat = function(pattern) {
    if (arguments.length == 0) return;
    AjxDateFormat.apply(this, arguments);
    var segments = this._segments || [];
    for (var i = 0; i < segments.length; i++) {
        var segment = segments[i];
        if (segment instanceof AjxDateFormat.TimezoneSegment) {
            segments[i] = new AjxDateUtil.TZDSegment(segment.toSubPattern());
        }
    }
};
AjxDateUtil.TZDFormat.prototype = new AjxDateFormat;
AjxDateUtil.TZDFormat.prototype.constructor = AjxDateUtil.TZDFormat;
AjxDateUtil.TZDFormat.prototype.toString = function() { return "TZDFormat"; };

AjxDateUtil.TZDSegment = function(pattern) {
    if (arguments.length == 0) return;
    AjxDateFormat.TimezoneSegment.apply(this, arguments);
};
AjxDateUtil.TZDSegment.prototype = new AjxDateFormat.TimezoneSegment;
AjxDateUtil.TZDSegment.prototype.constructor = AjxDateUtil.TZDSegment;
AjxDateUtil.TZDSegment.prototype.toString = function() { return "TZDSegment"; };

AjxDateUtil.TZDSegment.prototype.parse = function(o, s, i) {
    var m = /^(Z)|^(\+|\-)(\d\d):(\d\d)/.exec(s.substr(i));
    if (m) {
        var offset = new Date().getTimezoneOffset();
        if (m[1]) o.timezone = offset;
        else {
            var hours = parseInt(m[3],10), mins = parseInt(m[4],10);
            o.timezone = hours * 60 + mins;
            if (m[2] != "-") o.timezone *= -1;
            o.timezone -= offset;
        }
    }
    return i + (m ? m[0].length : 0);
};

AjxDateUtil.parseServerDateTime = 
function(serverStr, noSpecialUtcCase) {
	if (serverStr == null) return null;

	var d = new Date();
	var yyyy = parseInt(serverStr.substr(0,4), 10);
	var MM = parseInt(serverStr.substr(4,2), 10);
	var dd = parseInt(serverStr.substr(6,2), 10);
	d.setFullYear(yyyy);
	d.setMonth(MM - 1);
	d.setMonth(MM - 1); // DON'T remove second call to setMonth (see bug #3839)
	d.setDate(dd);
	AjxDateUtil.parseServerTime(serverStr, d, noSpecialUtcCase);
	return d;
};

AjxDateUtil._pad = 
function(n) {
	return n < 10 ? ('0' + n) : n;
};

/**
 * Returns the year portion of the given date as a YYYY string.
 *
 * @param {Date}    date    (optional, defaults to current date) a date
 * @returns {string}    year as YYYY
 */
AjxDateUtil.getYearStr = function(date) {
    date = date || new Date();
    return date.getFullYear() + "";
};

AjxDurationFormatVerbose = function() { }

AjxDurationFormatVerbose.formatYears =
function(years, months) {
	var deltaStr =  years + " ";
	deltaStr += (years > 1) ? AjxMsg.years : AjxMsg.year;
	if (years <= 3 && months > 0) {
		deltaStr += " " + months;
		deltaStr += " " + ((months > 1) ? AjxMsg.months : AjxMsg.months);
	}
	return deltaStr;
};

AjxDurationFormatVerbose.formatMonths =
function(months, days) {
	var deltaStr =  months + " ";
	deltaStr += (months > 1) ? AjxMsg.months : AjxMsg.month;
	if (months <= 3 && days > 0) {
		deltaStr += " " + days;
		deltaStr += " " + ((days > 1) ? AjxMsg.days : AjxMsg.day);
	}
	return deltaStr;
};

AjxDurationFormatVerbose.formatDays =
function(days, hours) {
	var deltaStr = days + " ";
	deltaStr += (days > 1) ? AjxMsg.days : AjxMsg.day;
	if (days <= 2 && hours > 0) {
		deltaStr += " " + hours;
		deltaStr += " " + ((hours > 1) ? AjxMsg.hours : AjxMsg.hour);
	}
	return deltaStr;
};

AjxDurationFormatVerbose.formatHours =
function(hours, mins) {
	var deltaStr = hours + " ";
	deltaStr += (hours > 1) ? AjxMsg.hours : AjxMsg.hour;
	if (hours < 5 && mins > 0) {
		deltaStr += " " + mins;
		deltaStr += " " + ((mins > 1) ? AjxMsg.minutes : AjxMsg.minute);
	}
	return deltaStr;
};

AjxDurationFormatVerbose.formatMinutes =
function(mins, secs) {
	var deltaStr = mins + " ";
	deltaStr += ((mins > 1) ? AjxMsg.minutes : AjxMsg.minute);
	if (mins < 5 && secs > 0) {
		deltaStr += " " + secs;
		deltaStr += " " + ((secs > 1) ? AjxMsg.seconds : AjxMsg.second);
	}
	return deltaStr;
};

AjxDurationFormatVerbose.formatSeconds =
function(secs) {
	return (secs + " " + ((secs > 1) ? AjxMsg.seconds : AjxMsg.second));
};

AjxDurationFormatConcise = function() { }

AjxDurationFormatConcise.formatYears =
function(years, months) {
	return this._format(years, months);
};

AjxDurationFormatConcise.formatMonths =
function(months, days) {
	return this._format(months, days);
};

AjxDurationFormatConcise.formatDays =
function(days, hours) {
	return this._format(days, hours);
};

AjxDurationFormatConcise.formatHours =
function(hours, mins) {
	return this._format(hours, mins);
};

AjxDurationFormatConcise.formatMinutes =
function(mins, secs) {
	return this._format(mins, secs);
};

AjxDurationFormatConcise.formatSeconds =
function(secs) {
	return this._format(0, secs);
};

AjxDurationFormatConcise._format =
function(a, b) {
	var i = 0;
	var result = [];
	result[i++] = a;
	result[i++] = ':';
	if (b < 10) {
		result[i++] = '0';
	}
	result[i++] = b;
	return result.join('');
};

/**
 * Added more utility functions for date finding and navigating
 */

AjxDateUtil.SUNDAY = 0;
AjxDateUtil.MONDAY = 1;
AjxDateUtil.TUESDAY = 2;
AjxDateUtil.WEDNESDAY = 3;
AjxDateUtil.THURSDAY = 4;
AjxDateUtil.FRIDAY = 5;
AjxDateUtil.SATURDAY = 6;                                                                              

/**
 *
 * @param fromThisDate The searching starts from this date.
 * @param thisWeekday  The day to find ( eg. AjxDateUtil.SUNDAY)
 * @param count Which occurence, like first, second.. has to be always positive
 * 
 */
AjxDateUtil.getDateForNextDay =
function(fromThisDate, thisWeekday, count) {
	count = count || 1;
	var r = new Date(fromThisDate);
	for (var i = 0; i < count; i++) {
		r = AjxDateUtil._getDateForNextWeekday(r, thisWeekday);
		if (i < count-1) {
			r.setDate(r.getDate() + 1);
		}
	}
	return r;
}

/**
 *
 * @param fromThisDate The searching work week days starting from this date
 * @param count Which occurence, like first, second.. has to be always positive
 *
 */
AjxDateUtil.getDateForNextWorkWeekDay =
function(fromThisDate, count) {
	count = count?count:1;
	var r = new Date(fromThisDate);
	for (var i = 0; i < count; i++) {
		r = AjxDateUtil._getDateForNextWorkWeekday(r);
		if (i < count-1) {
			r.setDate(r.getDate() + 1);
		}
	}
	return r;
}

/**
 *
 * @param fromThisDate The starting point
 * @param thisWeekday  The day to find
 * @param count this many positions to navigate, if negative goes in reverse, if positive goes forward
 */
AjxDateUtil.getDateForThisDay =
function(fromThisDate, thisWeekday, count) {
	if (count < 0 ) {
		return AjxDateUtil.getDateForPrevDay(fromThisDate, thisWeekday, -count);//-(-)  is plus
	} else {
		return AjxDateUtil.getDateForNextDay(fromThisDate, thisWeekday, count);
	}
}

/**
 *
 * @param fromThisDate The starting point
 * @param count this many positions to navigate, if negative goes in reverse, if positive goes forward
 */
AjxDateUtil.getDateForThisWorkWeekDay =
function(fromThisDate, count) {
	if (count < 0 ) {
		return AjxDateUtil.getDateForPrevWorkWeekDay(fromThisDate, -count);		//-(-)  is plus
	}else{
		return AjxDateUtil.getDateForNextWorkWeekDay(fromThisDate, count);
	}
}

/**
 *
 * @param fromThisDate The searching starts from this date in reverse direction. 
 * @param thisWeekday  The day to find ( eg. AjxDateUtil.SUNDAY)
 * @param count Which occurence, like first, second..has to be always positive
 */
AjxDateUtil.getDateForPrevDay =
function(fromThisDate,thisWeekday,count) {
	count = count || 1;
	var r = new Date(fromThisDate);
	for (var i = 0; i < count; i++) {
		r = AjxDateUtil._getDateForPrevWeekday(r, thisWeekday);
		if (i < count-1) {
			r.setDate(r.getDate()-1);
		}
	}
	return r;
}

/**
 *
 * @param fromThisDate The searching for work week days starting from this date in reverse direction.
 * @param count Which occurence, like first, second..has to be always positive
 */

AjxDateUtil.getDateForPrevWorkWeekDay =
function(fromThisDate, count) {
	count = count || 1;
	var r = new Date(fromThisDate);
	for(var i = 0; i < count; i++) {
		r = AjxDateUtil._getDateForPrevWorkWeekday(r);
		if (i < count-1) {
			r.setDate(r.getDate()-1);
		}
	}
	return r;
}

/**
 * note - this deals with the format we save from Prefs page. Careful if using for other cases.
 * @param value
 * @return {String}
 */
AjxDateUtil.dateLocal2GMT =
function(value) {
	if (!value) { return ""; }

	var yr, mo, da, hr, mi, se; // really smart parsing.
	yr = parseInt(value.substr(0,  4), 10);
	mo = parseInt(value.substr(4,  2), 10);
	da = parseInt(value.substr(6,  2), 10);
	hr = parseInt(value.substr(8,  2), 10);
	mi = parseInt(value.substr(10, 2), 10);
	se = parseInt(value.substr(12, 2), 10);
	var date = new Date(yr, mo - 1, da, hr, mi, se, 0);
	yr = date.getUTCFullYear();
	mo = date.getUTCMonth() + 1;
	da = date.getUTCDate();
	hr = date.getUTCHours();
	mi = date.getUTCMinutes();
	se = date.getUTCSeconds();
	var a = [ yr, mo, da, hr, mi, se ];
	for (var i = a.length; --i > 0;) {
		var n = a[i];
		if (n < 10)
			a[i] = "0" + n;
	}
	return (a.join("") + "Z");
};

/**
 * note - this deals with the format we save from Prefs page. Careful if using for other cases.
 * @param value
 * @return {String}
 */
AjxDateUtil.dateGMT2Local =
function(value) {
	if (!value) { return ""; }

	var yr, mo, da, hr, mi, se; // really smart parsing.
	yr = parseInt(value.substr(0,  4), 10);
	mo = parseInt(value.substr(4,  2), 10);
	da = parseInt(value.substr(6,  2), 10);
	hr = parseInt(value.substr(8,  2), 10);
	mi = parseInt(value.substr(10, 2), 10);
	se = parseInt(value.substr(12, 2), 10);
	var date = new Date();
	date.setUTCHours(hr, mi, se, 0);
	date.setUTCFullYear(yr, mo - 1, da);
	yr = date.getFullYear();
	mo = date.getMonth() + 1;
	da = date.getDate();
	hr = date.getHours();
	mi = date.getMinutes();
	se = date.getSeconds();
	var a = [yr, mo, da, hr, mi, se];
	for (var i = a.length; --i > 0;) {
		var n = a[i];
		if (n < 10)
			a[i] = "0" + n;
	}
	return (a.join("") + "Z");
};


AjxDateUtil._getDateForNextWeekday =
function(fromThisDate,thisWeekday) {
	var newDate = new Date(fromThisDate);
	var weekDay = fromThisDate.getDay();
	if (weekDay == thisWeekday) {
		return newDate;
	}
	var diff = (thisWeekday-weekDay);
	if (diff > 0) {
		newDate.setDate(fromThisDate.getDate() + diff);
	} else {
		newDate.setDate(fromThisDate.getDate() + (7 + diff));
	}
	return newDate;
}

AjxDateUtil._getDateForNextWorkWeekday =
function(fromThisDate) {
	var newDate = new Date(fromThisDate);
	var weekDay = fromThisDate.getDay();
	if (weekDay == AjxDateUtil.SUNDAY) {
		newDate.setDate(fromThisDate.getDate()+1);
	} else if (weekDay == AjxDateUtil.SATURDAY) {
		newDate.setDate(fromThisDate.getDate()+2);
	}
	return newDate;
}

AjxDateUtil._getDateForPrevWeekday =
function(fromThisDate, thisWeekday) {
	var newDate = new Date(fromThisDate);
	var weekDay = fromThisDate.getDay();
	if (weekDay == thisWeekday) {
		return newDate;
	}
	var diff = (weekDay-thisWeekday);
	if (diff > 0) {
		newDate.setDate(fromThisDate.getDate() - diff);
	} else {
		newDate.setDate(fromThisDate.getDate() - (7 + diff));
	}
	return newDate;
}

AjxDateUtil._getDateForPrevWorkWeekday =
function(fromThisDate) {
	var newDate = new Date(fromThisDate);
	var weekDay = fromThisDate.getDay();
	if (weekDay == AjxDateUtil.SUNDAY) {
		newDate.setDate(fromThisDate.getDate() - 2);
	} else if (weekDay == AjxDateUtil.SATURDAY) {
		newDate.setDate(fromThisDate.getDate() - 1);
	}
	return newDate;
}

//
// Date calculator functions
//

AjxDateUtil.calculate =
function(rule, date) {
	// initialize
	if (!AjxDateUtil.__calculate_initialized) {
		AjxDateUtil.__calculate_initialized = true;
		AjxDateUtil.__calculate_init();
	}

	var now = date || new Date;
	rule = rule.replace(/^\s*|\s*$/, "").replace(/\s*=\s*/g,"=").replace(/\s*,\s*/g,",");
	var a = rule.split(/\s+/g);
	var s, m, plusminus, number, type, amount, weekord, daynum;
	for (var i = 0; i < a.length; i++) {
		s = a[i];
		// comment
		if (s.match(AjxDateUtil.RE_COMMENT)) {
			break;
		}
		// context date
		if (s.match(AjxDateUtil.RE_NOW)) {
			date = new Date(now.getTime());
			continue;
		}
		// add
		if (m = s.match(AjxDateUtil.RE_ADD_NUMBER)) {
			plusminus = m[1];
			number = AjxDateUtil.__calculate_parseInt(m[2]);
			type = a[++i];
			amount = plusminus == '+' ? number : number * -1;
			AjxDateUtil.__calculate_add(date, type, amount);
			continue;
		}
		// set
		if (m = s.match(AjxDateUtil.RE_SET)) {
			AjxDateUtil.__calculate_set(date, m[1], m[2]);
			continue;
		}
		// try to parse as a date
		date = AjxDateFormat.parse("yyyyy-MM-dd", s);
		if (!date && (date = AjxDateFormat.parse("yyyy-MM-dd'T'hh:mm:ss'Z'", s))) {
			date.setMinutes(date.getMinutes() - date.getTimezoneOffset());
		}
		if (!date) date = AjxDateFormat.parse("yyyy-MM-dd'T'HH:mm:ss", s);
		if (!date) throw "invalid date pattern: \""+s+"\"";
	}
	return date;
};

//
// Date calculator constants
//

AjxDateUtil.S_DAYNAME = [
	AjxMsg["calc.dayname.sunday"],
	AjxMsg["calc.dayname.monday"],
	AjxMsg["calc.dayname.tuesday"],
	AjxMsg["calc.dayname.wednesday"],
	AjxMsg["calc.dayname.thursday"],
	AjxMsg["calc.dayname.friday"],
	AjxMsg["calc.dayname.saturday"]
].join("|");

AjxDateUtil.S_MONTHNAME = [
	AjxMsg["calc.monthname.january"],
	AjxMsg["calc.monthname.february"],
	AjxMsg["calc.monthname.march"],
	AjxMsg["calc.monthname.april"],
	AjxMsg["calc.monthname.may"],
	AjxMsg["calc.monthname.june"],
	AjxMsg["calc.monthname.july"],
	AjxMsg["calc.monthname.august"],
	AjxMsg["calc.monthname.september"],
	AjxMsg["calc.monthname.october"],
	AjxMsg["calc.monthname.november"],
	AjxMsg["calc.monthname.december"]
].join("|");

AjxDateUtil.S_WEEKORD = [
	AjxMsg["calc.ordinal.first"],
	AjxMsg["calc.ordinal.second"],
	AjxMsg["calc.ordinal.third"],
	AjxMsg["calc.ordinal.fourth"],
	AjxMsg["calc.ordinal.fifth"],
	AjxMsg["calc.ordinal.last"]
].join("|");

AjxDateUtil.WEEKORD_RE = [
    new RegExp("(first|"+AjxMsg["calc.ordinal.first"]+")",  "i"),
    new RegExp("(second|"+AjxMsg["calc.ordinal.second"]+")", "i"),
    new RegExp("(third|"+AjxMsg["calc.ordinal.third"]+")",  "i"),
    new RegExp("(fourth|"+AjxMsg["calc.ordinal.fourth"]+")", "i"),
    new RegExp("(last|"+AjxMsg["calc.ordinal.last"]+")",   "i")
];

// NOTE: Originally, the keywords for the date calculation rules
//       were in the message bundle so that they could be translated.
//       But while the keywords were translated, the rules were not
//       updated to use the translated keywords. So none of the date
//       matching worked in other languages. So I am reverting that
//       decision and hard-coding all of the relevant keywords. The
//       ordinals, day names, and month names still need to be
//       translated, though.

AjxMsg["calc.now"]	= "now";
AjxMsg["calc.date"]	= "date";

AjxMsg["calc.duration.year"]		= "year|years";
AjxMsg["calc.duration.month"]		= "mons|month|months";
AjxMsg["calc.duration.day"]			= "day|days";
AjxMsg["calc.duration.hour"]		= "hour|hours";
AjxMsg["calc.duration.minute"]		= "min|mins|minute|minutes";
AjxMsg["calc.duration.week"]        = "week";
AjxMsg["calc.duration.second"]		= "sec|secs|second|seconds";
AjxMsg["calc.duration.millisecond"]	= "milli|millis|millisecond|milliseconds";

AjxDateUtil.S_DURATION = [
	AjxMsg["calc.duration.year"],
	AjxMsg["calc.duration.month"],
    AjxMsg["calc.duration.week"],
	AjxMsg["calc.duration.day"],
	AjxMsg["calc.duration.hour"],
	AjxMsg["calc.duration.minute"],
	AjxMsg["calc.duration.second"],
	AjxMsg["calc.duration.millisecond"]
].join("|");

//
// Date calculator private functions
//

AjxDateUtil.__calculate_init =
function() {
	AjxDateUtil.WEEKDAYS = {};
	var weekdays = [
		"sunday", "monday", "tuesday", "wednesday", "thursday", "friday", "saturday"
	];
	for (var i = 0; i < weekdays.length; i++) {
		var weekday = AjxMsg["calc.dayname."+weekdays[i]].split("|");
		for (var j = 0; j < weekday.length; j++) {
			AjxDateUtil.WEEKDAYS[weekday[j].toLowerCase()] = i;
		}
	}

	AjxDateUtil.MONTHNAME2MONTHNUM = {};
	var months = [
		"january", "february", "march", "april", "may", "june",
		"july", "august", "september", "october", "november", "december"
	];
	for (var i = 0; i < months.length; i++) {
		var month = AjxMsg["calc.monthname."+months[i]].split("|");
		for (var j = 0; j < month.length; j++) {
			AjxDateUtil.MONTHNAME2MONTHNUM[month[j].toLowerCase()] = i;
		}
	}

	AjxDateUtil.RE_YEAR = new RegExp("^("+AjxMsg["calc.duration.year"]+")$", "i");
	AjxDateUtil.RE_MONTH = new RegExp("^("+AjxMsg["calc.duration.month"]+")$", "i");
	AjxDateUtil.RE_WEEK = new RegExp("^("+AjxMsg["calc.duration.week"]+")$", "i");
	AjxDateUtil.RE_DAY = new RegExp("^("+AjxMsg["calc.duration.day"]+")$", "i");
	AjxDateUtil.RE_HOUR = new RegExp("^("+AjxMsg["calc.duration.hour"]+")$", "i");
	AjxDateUtil.RE_MINUTE = new RegExp("^("+AjxMsg["calc.duration.minute"]+")$", "i");
	AjxDateUtil.RE_SECOND = new RegExp("^("+AjxMsg["calc.duration.second"]+")$", "i");
	AjxDateUtil.RE_MILLISECOND = new RegExp("^("+AjxMsg["calc.duration.millisecond"]+")$", "i");

	AjxDateUtil.RE_DATE = new RegExp("^("+AjxMsg["calc.date"]+")$", "i");
	
	AjxDateUtil.RE_DAYNAME = new RegExp("^("+AjxDateUtil.S_DAYNAME+")$", "i");
	AjxDateUtil.RE_MONTHNAME = new RegExp("^("+AjxDateUtil.S_MONTHNAME+")$", "i");
	AjxDateUtil.RE_WEEKORD = new RegExp("^("+AjxDateUtil.S_WEEKORD+")$", "i");

	AjxDateUtil.RE_COMMENT = /^#/;
	AjxDateUtil.RE_NOW = new RegExp("^("+AjxMsg["calc.now"]+")$", "i");
	AjxDateUtil.RE_ADD_NUMBER = new RegExp("^([+\\-])(\\d+)$", "i");
	AjxDateUtil.RE_SET = new RegExp("^("+AjxDateUtil.S_DURATION+"|"+AjxMsg["calc.date"]+")=(.*)$", "i");
};

AjxDateUtil.__calculate_normalizeFullWidthDigit =
function(digit) {
	var charCode = "0".charCodeAt(0) + digit.charCodeAt(0) - "\uff10".charCodeAt(0);
	return String.fromCharCode(charCode);
};

/** This is needed to handle asian full-width digits. */
AjxDateUtil.__calculate_replaceFullWidthDigit =
function($0, digit) {
	return AjxDateUtil.__calculate_normalizeFullWidthDigit(digit);
};

AjxDateUtil.__calculate_parseInt =
function(s) {
	s = s.replace(/([\uFF10-\uFF19])/g, AjxDateUtil.__calculate_normalizeFullWidthDigit);
	return parseInt(s, 10);
};

AjxDateUtil.__calculate_add =
function(date, type, amount) {
	if (type.match(AjxDateUtil.RE_YEAR)) {
		date.setFullYear(date.getFullYear() + amount);
		return;
	}
	if (type.match(AjxDateUtil.RE_MONTH)) {
		var month = date.getMonth();
		date.setMonth(month + amount);
		// avoid roll
		if (Math.abs(month + amount) % 12 != date.getMonth()) {
			date.setDate(0);
		}
		return;
	}
	if (type.match(AjxDateUtil.RE_WEEK)) {
		date.setDate(date.getDate() + amount * 7);
		return;
	}
	if (type.match(AjxDateUtil.RE_DAY)) {
		date.setDate(date.getDate() + amount);
		return;
	}
	if (type.match(AjxDateUtil.RE_HOUR)) {
		date.setHours(date.getHours() + amount);
		return;
	}
	if (type.match(AjxDateUtil.RE_MINUTE)) {
		date.setMinutes(date.getMinutes() + amount);
		return;
	}
	if (type.match(AjxDateUtil.RE_SECOND)) {
		date.setSeconds(date.getSeconds() + amount);
		return;
	}
	if (type.match(AjxDateUtil.RE_MILLISECOND)) {
		date.setMilliseconds(date.getMilliseconds() + amount);
		return;
	}
	if (type.match(AjxDateUtil.RE_MONTHNAME)) {
		var monthnum = AjxDateUtil.MONTHNAME2MONTHNUM[type.toLowerCase()];
		if (monthnum < date.getMonth()) {
			amount += amount > 0 ? 0 : 1;
		}
		else if (monthnum > date.getMonth()) {
			amount += amount > 0 ? -1 : 0;
		}
		date.setFullYear(date.getFullYear() + amount, monthnum, 1);
		return;
	}
	if (type.match(AjxDateUtil.RE_DAYNAME)) {
		var daynum = AjxDateUtil.WEEKDAYS[type.toLowerCase()];
		if (daynum < date.getDay()) {
			amount += amount > 0 ? 0 : 1;
		}
		else if (daynum > date.getDay()) {
			amount += amount > 0 ? -1 : 0;
		}
		date.setDate(date.getDate() + (daynum - date.getDay()) + 7 * amount);
		return;
	}
	throw "unknown type: "+type;
};

AjxDateUtil.__calculate_add_ordinal =
function() {
	throw "TODO: not implemented";
};

AjxDateUtil.__calculate_set =
function(date, type, value) {
	var args = value.split(/,/);
	//Add support for Japanese Heisei year format represented by H{year-number}
	//The year is H23 in H23/12/31, means 2011/12/31; we get that by adding year 1988 to 23
	//For example: H23 = 23 + 1988 = 2011(English year)
	if(args[0].indexOf("H") == 0) {
		args[0] = parseInt(args[0].replace("H", "")) + 1988;
	}
	if (type.match(AjxDateUtil.RE_YEAR)) {
		args[0] = AjxDateUtil.__calculate_fullYear(args[0]); // year
		if (args[1] != null) args[1] = AjxDateUtil.__calculate_month(args[1]); // month
		if (args[2] != null) args[2] = parseInt(args[2], 10); // date
		date.setFullYear.apply(date, args);
		return;
	}
	if (type.match(AjxDateUtil.RE_MONTH)) {
		args[0] = AjxDateUtil.__calculate_month(args[0]); // month
		if (args[1] != null) args[1] = parseInt(args[1], 10); // date
		date.setMonth.apply(date, args);
		return;
	}
    if (type.match(AjxDateUtil.RE_WEEK)) {
        var ord = AjxDateUtil.__calculate_week(args[0]); // week
        var day = args[1] ? AjxDateUtil.__calculate_day(args[1]) : date.getDay(); // day

        var target;
        if (ord != -1) {
            var firstday = new Date(date.getFullYear(), date.getMonth(), 1, 12, 0, 0, 0);
            var firstdow = firstday.getDay();
            var delta = firstdow - day;

            target = new Date(firstday.getTime());
            target.setDate(1 - delta);
            if (delta > 0) {
                target.setDate(target.getDate() + 7);
            }
            target.setDate(target.getDate() + 7 * ord);
        }
        else {
            var lastday = new Date(date.getFullYear(), date.getMonth()+1, 0, 12, 0, 0, 0);

            target = new Date(lastday.getTime());
            target.setDate(target.getDate() - (target.getDay() - day));
            if (target.getMonth() != lastday.getMonth()) {
                target.setDate(target.getDate() - 7);
            }
        }

        if (target && (date.getMonth() == target.getMonth())) {
            date.setTime(target.getTime());
        }
        return;
    }
	if (type.match(AjxDateUtil.RE_DATE)) {
		args[0] = parseInt(args[0], 10); // date
		date.setDate.apply(date, args);
		return;
	}
	if (type.match(AjxDateUtil.RE_HOUR)) {
		args[0] = parseInt(args[0], 10); // hour
		if (args[1] != null) args[1] = parseInt(args[1], 10); // minutes
		if (args[2] != null) args[2] = parseInt(args[2], 10); // seconds
		if (args[3] != null) args[3] = parseInt(args[3], 10); // milliseconds
		date.setHours.apply(date, args);
		return;
	}
	if (type.match(AjxDateUtil.RE_MINUTE)) {
		args[0] = parseInt(args[0], 10); // minutes
		if (args[1] != null) args[1] = parseInt(args[1], 10); // seconds
		if (args[2] != null) args[2] = parseInt(args[2], 10); // milliseconds
		date.setMinutes.apply(date, args);
		return;
	}
	if (type.match(AjxDateUtil.RE_SECOND)) {
		args[0] = parseInt(args[0], 10); // seconds
		if (args[1] != null) args[1] = parseInt(args[1], 10); // milliseconds
		date.setSeconds.apply(date, args);
		return;
	}
	if (type.match(AjxDateUtil.RE_MILLISECOND)) {
		date.setMilliseconds.apply(date, args); // milliseconds
		return;
	}
	throw "unknown type: "+type;
};

AjxDateUtil.__calculate_fullYear =
function(value) {
	if (value.length == 2) {
		var d = new Date;
		d.setYear(parseInt(value, 10));
        var fullYear = d.getFullYear();
        if (fullYear <= AjxMsg.dateParsing2DigitStartYear) {
            value = String(fullYear + 100);
        }
        else {
            value = String(fullYear).substr(0,2) + value;
        }
	}
	return parseInt(value, 10);
};

AjxDateUtil.__calculate_month =
function(value) {
	var monthnum = AjxDateUtil.MONTHNAME2MONTHNUM[value.toLowerCase()];
	return monthnum != null ? monthnum : parseInt(value, 10) - 1;
};

AjxDateUtil.__calculate_week = function(value) {
    for (var i = 0; i < AjxDateUtil.WEEKORD_RE.length; i++) {
        if (value.match(AjxDateUtil.WEEKORD_RE[i])) {
            if (i == AjxDateUtil.WEEKORD_RE.length - 1) {
                return -1;
            }
            return i;
        }
    }
    return 0;
};

AjxDateUtil.__calculate_day =
function(value) {
	var daynum = AjxDateUtil.WEEKDAYS[value.toLowerCase()];
	return daynum != null ? daynum : parseInt(value, 10);
};
}
if (AjxPackage.define("ajax.util.AjxPluginDetector")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Does nothing (static class).
 * @constructor
 * @class
 * 
 * This class provides static methods to determine which standard plugins are
 * installed in the browser.
 *
 * @private
 */
AjxPluginDetector = function() {
}

AjxPluginDetector.canDetectPlugins =
function() {
	return AjxEnv.isIE || (navigator.plugins && navigator.plugins.length > 0);
};

AjxPluginDetector.detectFlash =
function() {
	if(AjxEnv.isIE) {
		return AjxPluginDetector.detectActiveXControl('ShockwaveFlash.ShockwaveFlash.1');
	} else {
		return AjxPluginDetector.detectPlugin('Shockwave','Flash'); 
	}
};

AjxPluginDetector.detectPDFReader =
function(){
    if(AjxEnv.isIE){
        return  ( AjxPluginDetector.detectActiveXControl('PDF.PdfCtrl.5')
                || AjxPluginDetector.detectActiveXControl('AcroExch.Document') );
    }else{
        var hasPDFReader = false;
        if(AjxEnv.isChrome){
            hasPDFReader = AjxPluginDetector.detectPlugin('Chrome PDF Viewer');
        }else if(AjxEnv.isFirefox){
            hasPDFReader = AjxPluginDetector.detectPlugin('Firefox PDF Plugin for Mac OS X');
        }
        if(!hasPDFReader){
            hasPDFReader = AjxPluginDetector.detectPlugin('Adobe Acrobat');
        }
        return hasPDFReader;
    }
};

AjxPluginDetector.detectDirector =
function() { 
	if(AjxEnv.isIE) {
		return AjxPluginDetector.detectActiveXControl('SWCtl.SWCtl.1');
	} else {
		return AjxPluginDetector.detectPlugin('Shockwave','Director');
	}
};

AjxPluginDetector.detectQuickTime =
function() {
	if(AjxEnv.isIE) {
		return AjxPluginDetector.detectQuickTimeActiveXControl();
	} else {
		return AjxPluginDetector.detectPlugin('QuickTime');
	}
};

// If quicktime is installed, returns the version as an array: [major, minor, build]
AjxPluginDetector.getQuickTimeVersion =
function() {
	if(AjxEnv.isIE) {
		var object = new ActiveXObject("QuickTimeCheckObject.QuickTimeCheck.1");
		DBG.println(AjxDebug.DBG1, "AjxPluginDetector: Quicktime is " + object.IsQuickTimeAvailable(0) ? "available" : "not available");
		if (object.IsQuickTimeAvailable(0)) {
			try {
				var version = Number(object.QuickTimeVersion).toString(16);
				var result = [];
				for(var i = 0; i < 3; i++) {
					result[i] = Number(version.charAt(i));
				}
				return result;
			} catch(e) {
				DBG.println(AjxDebug.DBG1, "AjxPluginDetector: Error while checking QuickTimeVersion: " + e);
			}
		}
		return null;
	} else {
		var match = AjxPluginDetector.matchPluginName(/QuickTime Plug-in (\d+)\.?(\d+)?\.?(\d+)?/);
		if (match) {
			DBG.println("AjxPluginDetector: able to find match for QuickTime plugin with version: " + match);
			var result = [];
			for(var i = 0; i < 3; i++) {
				result[i] = Number(match[i + 1] || 0);
			}
			return result;
		} else {
			DBG.println("AjxPluginDetector: unable to find match for QuickTime plugin with version");
			return null;
		}
	}
};

/**
 * This code is part of JQuery's Flash plugin.
 * http://jquery.lukelutman.com/plugins/flash/
 *
 * @return Flash plugin version
 */
AjxPluginDetector.getFlashVersion =
function() {
    var flashVersion = "0,0,0";
    // ie
    try {
        try {
            // avoid fp6 minor version lookup issues
            // see: http://blog.deconcept.com/2006/01/11/getvariable-setvariable-crash-internet-explorer-flash-6/
            var axo = new ActiveXObject('ShockwaveFlash.ShockwaveFlash.6');
            try {
                axo.AllowScriptAccess = 'always';
            }
            catch(e) {
                return '6,0,0';
            }
        } catch(e) {
            }
        flashVersion = new ActiveXObject('ShockwaveFlash.ShockwaveFlash').GetVariable('$version').replace(/\D+/g, ',').match(/^,?(.+),?$/)[1];
        // other browsers
    } catch(e) {
        try {
            if (navigator.mimeTypes["application/x-shockwave-flash"].enabledPlugin) {
                flashVersion = (navigator.plugins["Shockwave Flash 2.0"] || navigator.plugins["Shockwave Flash"]).description.replace(/\D+/g, ",").match(/^,?(.+),?$/)[1];
            }
        } catch(e) {
        }
    }
	return flashVersion;
};

AjxPluginDetector.detectReal =
function() {
	if(AjxEnv.isIE) {
		return AjxPluginDetector.detectActiveXControl('rmocx.RealPlayer G2 Control') ||
		       AjxPluginDetector.detectActiveXControl('RealPlayer.RealPlayer(tm) ActiveX Control (32-bit)') ||
		       AjxPluginDetector.detectActiveXControl('RealVideo.RealVideo(tm) ActiveX Control (32-bit)');
	} else {
		return AjxPluginDetector.detectPlugin('RealPlayer');
	}
};

AjxPluginDetector.detectWindowsMedia =
function() {
	if(AjxEnv.isIE) {
		return AjxPluginDetector.detectActiveXControl('MediaPlayer.MediaPlayer.1');
	} else {
		return AjxPluginDetector.detectPlugin('Windows Media');
	}
};

AjxPluginDetector.detectPlugin =
function() {
	DBG.println(AjxDebug.DBG1, "-----------------------<br>AjxPluginDetector: Looking for plugin: [" + AjxPluginDetector._argumentsToString(AjxPluginDetector.detectPlugin.arguments) + "]");
	var names = AjxPluginDetector.detectPlugin.arguments;
	var allPlugins = navigator.plugins;
	var pluginsArrayLength = allPlugins.length;
	for (var pluginsArrayCounter=0; pluginsArrayCounter < pluginsArrayLength; pluginsArrayCounter++ ) {
	    // loop through all desired names and check each against the current plugin name
	    var numFound = 0;
	    for(var namesCounter=0; namesCounter < names.length; namesCounter++) {
			// if desired plugin name is found in either plugin name or description
			if (allPlugins[pluginsArrayCounter]) {
				if( (allPlugins[pluginsArrayCounter].name.indexOf(names[namesCounter]) >= 0)) {
					// this name was found
					DBG.println(AjxDebug.DBG1, "AjxPluginDetector: found name match '" + allPlugins[pluginsArrayCounter].name + "'");
					numFound++;
				} else if (allPlugins[pluginsArrayCounter].description.indexOf(names[namesCounter]) >= 0) {
					// this name was found
					DBG.println(AjxDebug.DBG1, "AjxPluginDetector: found description match '" + allPlugins[pluginsArrayCounter].description + "'");
					numFound++;
				}
			}
	    }
	    // now that we have checked all the required names against this one plugin,
	    // if the number we found matches the total number provided then we were successful
	    if(numFound == names.length) {
			DBG.println(AjxDebug.DBG1, "AjxPluginDetector: Found plugin!<br>-----------------------");
			return true;
	    } else if (numFound) {
			DBG.println(AjxDebug.DBG1, "AjxPluginDetector: Found partial plugin match, numFound=" + numFound);
		}
	}
	DBG.println(AjxDebug.DBG1, "AjxPluginDetector: Failed to find plugin.<br>-----------------------");
	return false;
};

AjxPluginDetector.matchPluginName =
function(regExp) {
	var allPlugins = navigator.plugins;
	var pluginsArrayLength = allPlugins.length;
	for (var pluginsArrayCounter=0; pluginsArrayCounter < pluginsArrayLength; pluginsArrayCounter++ ) {
		var match = allPlugins[pluginsArrayCounter].name.match(regExp);
		if (match) {
			return match;
		}
	}
	return null;
};

AjxPluginDetector.detectActiveXControl =
function(progId) {
	try {
		new ActiveXObject(progId);
		DBG.println(AjxDebug.DBG1, "AjxPluginDetector: found ActiveXObject '" + progId + "'");
		return true;
	} catch (e) {
		DBG.println(AjxDebug.DBG1, "AjxPluginDetector: unable to find ActiveXObject '" + progId + "'");
		return false;
	}
};

AjxPluginDetector.detectQuickTimeActiveXControl =
function(progId) {
	try {
		var object = new ActiveXObject("QuickTimeCheckObject.QuickTimeCheck.1");
		return object.IsQuickTimeAvailable(0);
	} catch (e) {
		return false;
	}
};

// Util method to log arguments, which to my surprise are not actually an array.
AjxPluginDetector._argumentsToString =
function(args) {
	var array = [];
	for (var i = 0, count = args.length; i < count; i++) {
		array[i] = args[i];
	}
	return array.join(',')
};
}
if (AjxPackage.define("ajax.util.AjxClipboard")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * Clipboard access. Current implementation is built on clipboard.js
 *
 * @class
 * @constructor
 */
AjxClipboard = function() {
	AjxDispatcher.require("Clipboard");
};

/**
 * Returns true if clipboard access is supported.
 * @returns {Boolean}   true if clipboard access is supported
 */
AjxClipboard.isSupported = function() {
	// clipboard.js works on all browsers except IE8 and Safari
	return !AjxEnv.isIE8 && !(AjxEnv.isSafari && !AjxEnv.isChrome);
};

/**
 * Initialize clipboard action
 *
 * @param {DwtControl}          op          widget that initiates copy (eg button or menu item)
 * @param {Object}              listeners   hash of events
 */
AjxClipboard.prototype.init = function(op, listeners) {
	if (listeners.onComplete) {
		this._completionListener = listeners.onComplete.bind(null, this);
	}
	if (op && listeners.onMouseDown) {
		op.addSelectionListener(listeners.onMouseDown.bind(null, this));
	}
};

AjxClipboard.prototype.setText = function(text) {
	if (window.clipboard) {
		clipboard.copy(text).then(this._completionListener, this._onError);
	}
};

AjxClipboard.prototype._onError = function(error) {
	appCtxt.setStatusMsg(error && error.message, ZmStatusView.LEVEL_WARNING);
};
}
if (AjxPackage.define("ajax.util.AjxSHA1")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */
/*
 * Based on code by Paul Johnston:
 *
 * A JavaScript implementation of the Secure Hash Algorithm, SHA-1, as defined
 * in FIPS PUB 180-1
 * Version 2.1a Copyright Paul Johnston 2000 - 2002.
 * Other contributors: Greg Holt, Andrew Kepert, Ydnar, Lostinet
 * Distributed under the BSD License
 * See http://pajhome.org.uk/crypt/md5 for details.
 */

var AjxSHA1 = function() {

	/*
	 * Configurable variables. You may need to tweak these to be compatible with
	 * the server-side, but the defaults work in most cases.
	 */
	var hexcase = 0;  /* hex output format. 0 - lowercase; 1 - uppercase        */
	var b64pad  = ""; /* base-64 pad character. "=" for strict RFC compliance   */
	var chrsz   = 8;  /* bits per input character. 8 - ASCII; 16 - Unicode      */

	/*
	 * These are the functions you'll usually want to call
	 * They take string arguments and return either hex or base-64 encoded strings
	 */
	function hex_sha1(s){return binb2hex(core_sha1(str2binb(s),s.length * chrsz));};
	function b64_sha1(s){return binb2b64(core_sha1(str2binb(s),s.length * chrsz));};
	function str_sha1(s){return binb2str(core_sha1(str2binb(s),s.length * chrsz));};
	function hex_hmac_sha1(key, data){ return binb2hex(core_hmac_sha1(key, data));};
	function b64_hmac_sha1(key, data){ return binb2b64(core_hmac_sha1(key, data));};
	function str_hmac_sha1(key, data){ return binb2str(core_hmac_sha1(key, data));};

	/*
	 * Perform a simple self-test to see if the VM is working
	 */
	function sha1_vm_test()
	{
		return hex_sha1("abc") == "a9993e364706816aba3e25717850c26c9cd0d89d";
	};

	/*
	 * Calculate the SHA-1 of an array of big-endian words, and a bit length
	 */
	function core_sha1(x, len)
	{
		/* append padding */
		x[len >> 5] |= 0x80 << (24 - len % 32);
		x[((len + 64 >> 9) << 4) + 15] = len;

		var w = Array(80);
		var a =  1732584193;
		var b = -271733879;
		var c = -1732584194;
		var d =  271733878;
		var e = -1009589776;

		for(var i = 0; i < x.length; i += 16) {
			var olda = a;
			var oldb = b;
			var oldc = c;
			var oldd = d;
			var olde = e;

			for(var j = 0; j < 80; j++) {
				if(j < 16) w[j] = x[i + j];
				else w[j] = rol(w[j-3] ^ w[j-8] ^ w[j-14] ^ w[j-16], 1);
				var t = safe_add(safe_add(rol(a, 5), sha1_ft(j, b, c, d)),
						 safe_add(safe_add(e, w[j]), sha1_kt(j)));
				e = d;
				d = c;
				c = rol(b, 30);
				b = a;
				a = t;
			}

			a = safe_add(a, olda);
			b = safe_add(b, oldb);
			c = safe_add(c, oldc);
			d = safe_add(d, oldd);
			e = safe_add(e, olde);
		}
		return Array(a, b, c, d, e);

	};

	/*
	 * Perform the appropriate triplet combination function for the current
	 * iteration
	 */
	function sha1_ft(t, b, c, d)
	{
		if(t < 20) return (b & c) | ((~b) & d);
		if(t < 40) return b ^ c ^ d;
		if(t < 60) return (b & c) | (b & d) | (c & d);
		return b ^ c ^ d;
	};

	/*
	 * Determine the appropriate additive constant for the current iteration
	 */
	function sha1_kt(t)
	{
		return (t < 20) ?  1518500249 : (t < 40) ?  1859775393 :
			(t < 60) ? -1894007588 : -899497514;
	};

	/*
	 * Calculate the HMAC-SHA1 of a key and some data
	 */
	function core_hmac_sha1(key, data)
	{
		var bkey = str2binb(key);
		if(bkey.length > 16) bkey = core_sha1(bkey, key.length * chrsz);

		var ipad = Array(16), opad = Array(16);
		for(var i = 0; i < 16; i++) {
			ipad[i] = bkey[i] ^ 0x36363636;
			opad[i] = bkey[i] ^ 0x5C5C5C5C;
		}

		var hash = core_sha1(ipad.concat(str2binb(data)), 512 + data.length * chrsz);
		return core_sha1(opad.concat(hash), 512 + 160);
	};

	/*
	 * Add integers, wrapping at 2^32. This uses 16-bit operations internally
	 * to work around bugs in some JS interpreters.
	 */
	function safe_add(x, y)
	{
		var lsw = (x & 0xFFFF) + (y & 0xFFFF);
		var msw = (x >> 16) + (y >> 16) + (lsw >> 16);
		return (msw << 16) | (lsw & 0xFFFF);
	};

	/*
	 * Bitwise rotate a 32-bit number to the left.
	 */
	function rol(num, cnt)
	{
		return (num << cnt) | (num >>> (32 - cnt));
	};

	/*
	 * Convert an 8-bit or 16-bit string to an array of big-endian words
	 * In 8-bit function, characters >255 have their hi-byte silently ignored.
	 */
	function str2binb(str)
	{
		var bin = Array();
		var mask = (1 << chrsz) - 1;
		for(var i = 0; i < str.length * chrsz; i += chrsz)
			bin[i>>5] |= (str.charCodeAt(i / chrsz) & mask) << (32 - chrsz - i%32);
		return bin;
	};

	/*
	 * Convert an array of big-endian words to a string
	 */
	function binb2str(bin)
	{
		var str = "";
		var mask = (1 << chrsz) - 1;
		for(var i = 0; i < bin.length * 32; i += chrsz)
			str += String.fromCharCode((bin[i>>5] >>> (32 - chrsz - i%32)) & mask);
		return str;
	};

	/*
	 * Convert an array of big-endian words to a hex string.
	 */
	function binb2hex(binarray)
	{
		var hex_tab = hexcase ? "0123456789ABCDEF" : "0123456789abcdef";
		var str = "";
		for(var i = 0; i < binarray.length * 4; i++) {
			str += hex_tab.charAt((binarray[i>>2] >> ((3 - i%4)*8+4)) & 0xF) +
				hex_tab.charAt((binarray[i>>2] >> ((3 - i%4)*8  )) & 0xF);
		}
		return str;
	};

	/*
	 * Convert an array of big-endian words to a base-64 string
	 */
	function binb2b64(binarray)
	{
		var tab = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
		var str = "";
		for(var i = 0; i < binarray.length * 4; i += 3) {
			var triplet = (((binarray[i   >> 2] >> 8 * (3 -  i   %4)) & 0xFF) << 16)
				| (((binarray[i+1 >> 2] >> 8 * (3 - (i+1)%4)) & 0xFF) << 8 )
				|  ((binarray[i+2 >> 2] >> 8 * (3 - (i+2)%4)) & 0xFF);
			for(var j = 0; j < 4; j++) {
				if(i * 8 + j * 6 > binarray.length * 32) str += b64pad;
				else str += tab.charAt((triplet >> 6*(3-j)) & 0x3F);
			}
		}
		return str;
	};

	// export functions
	this.hex_sha1 = hex_sha1;
	this.b64_sha1 = b64_sha1;
	this.str_sha1 = str_sha1;
	this.hex_hmac_sha1 = hex_hmac_sha1;
	this.b64_hmac_sha1 = b64_hmac_sha1;
	this.str_hmac_sha1 = str_hmac_sha1;

	this.sha1_vm_test = sha1_vm_test;

};

AjxSHA1 = new AjxSHA1();
}

if (AjxPackage.define("zimbra.csfe.ZmBatchCommand")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the batch command class.
 */

/**
 * Creates an empty batch command. Use the {@link #add} method to add commands to it,
 * and {@link #run} to invoke it.
 * @class
 * This class represent a batch command, which is a collection of separate 
 * requests. Each command is a callback with a method, arguments, and (usually) an
 * object on which to call the method. Normally, when the command is run, it creates
 * a SOAP document or JSON object which it hands to the app controller's <code>sendRequest()</code>
 * method. It may also pass a response callback and/or an error callback.
 * <p>
 * Instead of calling sendRequest(), the command should hand the batch command its SOAP
 * document or JSON object, response callback, and error callback. The last argument that
 * the command receives is a reference to the batch command; that's how it knows it's in batch mode.
 * </p>
 * <p>
 * After all commands have been added to the batch command, call its run() method. That will
 * create a BatchRequest out of the individual commands' requests and send it to the
 * server. Each subrequest gets an ID. When the BatchResponse comes back, it is broken into
 * individual responses. If a response indicates success (it is a <code>*Response</code>), the corresponding
 * response callback is called with the result. If the response is a fault, the corresponding
 * error callback is called with the exception.
 * </p>
 * <p>
 * A command does not have to be the method that generates a SOAP document or JSON object.
 * It can be higher-level. Just make sure that the reference to the batch command gets passed down to it.
 * </p>
 * @author Conrad Damon
 * 
 * @param {Boolean}	continueOnError	if <code>true</code>, the batch request continues processing when a subrequest fails (defaults to <code>true</code>)
 * @param {String}	accountName		the account name to run this batch command as.
 * @param {Boolean}	useJson			if <code>true</code>, send JSON rather than XML
 */
ZmBatchCommand = function(continueOnError, accountName, useJson) {
	
	this._onError = (continueOnError === false) ? ZmBatchCommand.STOP : ZmBatchCommand.CONTINUE;
	this._accountName = accountName;
	this._useJson = useJson;
    this._requestBody = null;

	this.curId = 0;
    this._cmds = [];
	this._requests = [];
	this._respCallbacks = [];
	this._errorCallbacks = [];
};

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmBatchCommand.prototype.toString =
function() {
	return "ZmBatchCommand";
};

//
// Data
//

ZmBatchCommand.prototype._sensitive = false;
ZmBatchCommand.prototype._noAuthToken = false;

//
// Constants
//
ZmBatchCommand.STOP = "stop";
ZmBatchCommand.CONTINUE = "continue";

//
// Public methods
//

/**
 * Sets the sensitive flag. This indicates that this batch command
 * contains a request with sensitive data. Note: There is no way to unset
 * this value for the batch command.
 * 
 * @param	{Boolean}	sensitive		<code>true</code> to set command as sensitive
 */
ZmBatchCommand.prototype.setSensitive = function(sensitive) {
	this._sensitive = this._sensitive || sensitive;
};

/**
 * Sets the noAuthToken flag.
 *
 * @param	{Boolean}	noAuthToken		<code>true</code> to send command with noAuthToken
 */
ZmBatchCommand.prototype.setNoAuthToken = function(noAuthToken) {
	this._noAuthToken = noAuthToken;
};

/**
 * Checks if the command is sensitive.
 * 
 * @return	{Boolean}	<code>true</code> if the command is sensitive
 */
ZmBatchCommand.prototype.isSensitive = function() {
	return this._sensitive;
};

/**
 * Adds a command to the list of commands to run as part of this batch request.
 * 
 * @param {AjxCallback}	cmd		the command
 */
ZmBatchCommand.prototype.add =
function(cmd) {
	this._cmds.push(cmd);
};

/**
 * Gets the number of commands that are part of this batch request.
 * 
 * @return	{int}	the size
 */
ZmBatchCommand.prototype.size =
function() {
	return this.curId || this._cmds.length;
};

/**
 * Runs the batch request. For each individual request, either a response or an
 * error callback will be called.
 * 
 * @param {AjxCallback}		callback		the callback to run after entire batch request has completed
 * @param {AjxCallback}		errorCallback	the error callback called if anything fails.
 *										The error callbacks arguments are all
 *										of the exceptions that occurred. Note:
 *										only the first exception is passed if
 *										this batch command's onError is set to
 *										stop.
 */
ZmBatchCommand.prototype.run =
function(callback, errorCallback, offlineCallback) {

	// Invoke each command so that it hands us its SOAP doc, response callback,
	// and error callback
	for (var i = 0; i < this._cmds.length; i++) {
		var cmd = this._cmds[i];
		cmd.run(this);
		this.curId++;
	}

	var params = {
		sensitive:		this._sensitive,
        noAuthToken:	this._noAuthToken,
		asyncMode:		true,
		callback:		new AjxCallback(this, this._handleResponseRun, [callback, errorCallback]),
		errorCallback:	errorCallback,
		offlineCallback: offlineCallback,
		accountName:	this._accountName
	};

	// Create the BatchRequest
	if (this._useJson) {
		var jsonObj = {BatchRequest:{_jsns:"urn:zimbra", onerror:this._onError}};
		var batchRequest = jsonObj.BatchRequest;
		var size = this.size();
		if (size && this._requests.length) {
			for (var i = 0; i < size; i++) {
				var request = this._requests[i];
                //Bug fix # 67110 the request object is sometimes undefined
                if(request) {
                    request.requestId = i;
                    var methodName = ZmCsfeCommand.getMethodName(request);
                    if (!batchRequest[methodName]) {
                        batchRequest[methodName] = [];
                    }
				    request[methodName].requestId = i;
				    batchRequest[methodName].push(request[methodName]);
                }
			}
			params.jsonObj = jsonObj;
            this._requestBody = jsonObj;
		}
	}
	else {
		var batchSoapDoc = AjxSoapDoc.create("BatchRequest", "urn:zimbra");
		batchSoapDoc.setMethodAttribute("onerror", this._onError);
		// Add each command's request element to the BatchRequest, and set its ID
		var size = this.size();
		if (size > 0) {
			for (var i = 0; i < size; i++) {
				var soapDoc = this._requests[i];
				var reqEl = soapDoc.getMethod();
				reqEl.setAttribute("requestId", i);
				var node = batchSoapDoc.adoptNode(reqEl);
				batchSoapDoc.getMethod().appendChild(node);
			}
			params.soapDoc = batchSoapDoc;
            this._requestBody = batchSoapDoc;
		}
	}

	// Issue the BatchRequest *but* only when there's something to request
	if (params.jsonObj || params.soapDoc) {
		appCtxt.getAppController().sendRequest(params);
	}
	else if (callback) {
		callback.run();
	}
};

ZmBatchCommand.prototype.getRequestBody =
function() {
    return this._requestBody;
}

/**
 * @private
 */
ZmBatchCommand.prototype._handleResponseRun =
function(callback, errorCallback, result) {
	var batchResponse = result.getResponse();
	if (!batchResponse.BatchResponse) {
		DBG.println(AjxDebug.DBG1, "Missing batch response!");
		return;
	}
	// NOTE: In case the order of the requests is significant, we process
	//       the responses in the same order.
	var responses = [];
	for (var method in batchResponse.BatchResponse) {
		if (method.match(/^_/)) continue;

		var methodResponses = batchResponse.BatchResponse[method];
		for (var i = 0; i < methodResponses.length; i++) {
			responses[methodResponses[i].requestId] = { method: method, resp: methodResponses[i] };
		}
	}
	var exceptions = [];
	for (var i = 0; i < responses.length; i++) {
		var response = responses[i];
		try {
			this._processResponse(response.method, response.resp);
		}
		catch (ex) {
			exceptions.push(ex);
			if (this._onError == ZmBatchCommand.STOP) {
				break;
			}
		}
	}
	if (exceptions.length > 0 && errorCallback) {
		errorCallback.run.apply(errorCallback, exceptions);
	}
	else if (callback) {
		callback.run(result);
	}
};

/**
 * Adds the given command parameters to the batch command, as part of a command's
 * invocation. Should be called by a function that was added via {@link #add} earlier; that
 * function should pass the request object.
 * 
 * @param {AjxSoapDoc|Object}	request		a SOAP document or JSON object with the command's request
 * @param {AjxCallback}	respCallback	the next callback in chain for async request
 * @param {AjxCallback}		errorCallback	the callback to run if there is an exception
 * 
 * @see		#add
 */
ZmBatchCommand.prototype.addRequestParams =
function(request, respCallback, errorCallback) {
	this._requests[this.curId] = request;
	this._respCallbacks[this.curId] = respCallback;
	this._errorCallbacks[this.curId] = errorCallback;
};

/**
 * Adds the given command parameters to the batch command, as part of a command's
 * invocation. Should be called without a previous {@link #add} command, when the request
 * object can immediately generate its request object.
 * 
 * @param {AjxSoapDoc|object}	request		a SOAP document or JSON object with the command's request
 * @param {AjxCallback}	respCallback	the next callback in chain for async request
 * @param {AjxCallback}	errorCallback	the callback to run if there is an exception
 * 
 * @see		#add
 */
ZmBatchCommand.prototype.addNewRequestParams =
function(request, respCallback, errorCallback) {
    this.addRequestParams(request, respCallback, errorCallback);
    this.curId++;
};

/**
 * Each type of request will return an array of <code>*Response</code> elements. There may also be
 * an array of Fault elements. Each element has an ID, so we can match it to its
 * response or error callback, and run whichever is appropriate.
 * 
 * @private
 */
ZmBatchCommand.prototype._processResponse =
function(method, resp) {
	var id = resp.requestId;

	// handle error
	if (method == "Fault") {
		var ex = ZmCsfeCommand.faultToEx(resp, "ZmBatchCommand.prototype.run");
		if (this._errorCallbacks[id]) {
			var handled = this._errorCallbacks[id].run(ex);
			if (!handled) {
				appCtxt.getAppController()._handleException(ex);
			}
		}
		throw ex;
	}

	// process response callback
	if (this._respCallbacks[id]) {
		var data = {};
		data[method] = resp;
		var result = new ZmCsfeResult(data);
		this._respCallbacks[id].run(result, resp);
	}
};
}
if (AjxPackage.define("zimbra.csfe.ZmCsfeCommand")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the command class.
 */

/**
 * Creates a command.
 * @class
 * This class represents a command.
 * 
 */
ZmCsfeCommand = function() {
};

ZmCsfeCommand.prototype.isZmCsfeCommand = true;
ZmCsfeCommand.prototype.toString = function() { return "ZmCsfeCommand"; };

// Static properties

// Global settings for each CSFE command
ZmCsfeCommand._COOKIE_NAME = "ZM_AUTH_TOKEN";
ZmCsfeCommand.serverUri = null;

ZmCsfeCommand._sessionId = null;	// current session ID
ZmCsfeCommand._staleSession = {};	// old sessions

// Reasons for re-sending a request
ZmCsfeCommand.REAUTH	= "reauth";
ZmCsfeCommand.RETRY		= "retry";

// Static methods

/**
 * Gets the auth token cookie.
 * 
 * @return	{String}	the auth token
 * @deprecated this method does not work as the ZM_AUTH_TOKEN cookie is a http only cookie
 */
ZmCsfeCommand.getAuthToken =
function() {
	return AjxCookie.getCookie(document, ZmCsfeCommand._COOKIE_NAME);
};

/**
 * Sets the auth token cookie name.
 * 
 * @param	{String}	cookieName		the cookie name to user
 */
ZmCsfeCommand.setCookieName =
function(cookieName) {
	ZmCsfeCommand._COOKIE_NAME = cookieName;
};

/**
 * Sets the server URI.
 * 
 * @param	{String}	uri		the URI
 */
ZmCsfeCommand.setServerUri =
function(uri) {
	ZmCsfeCommand.serverUri = uri;
};

/**
 * Sets the auth token.
 * 
 * @param	{String}	authToken		the auth token
 * @param	{int}		lifetimeMs		the token lifetime in milliseconds
 * @param	{String}	sessionId		the session id
 * @param	{Boolean}	secure		<code>true</code> for secure
 * 
 */
ZmCsfeCommand.setAuthToken =
function(authToken, lifetimeMs, sessionId, secure) {
	ZmCsfeCommand._curAuthToken = authToken;
	if (lifetimeMs != null) {
		var exp = null;
		if(lifetimeMs > 0) {
			exp = new Date();
			var lifetime = parseInt(lifetimeMs);
			exp.setTime(exp.getTime() + lifetime);
		}
		AjxCookie.setCookie(document, ZmCsfeCommand._COOKIE_NAME, authToken, exp, "/", null, secure);
	} else {
		AjxCookie.deleteCookie(document, ZmCsfeCommand._COOKIE_NAME, "/");
	}
	if (sessionId) {
		ZmCsfeCommand.setSessionId(sessionId);
	}
};

/**
 * Clears the auth token cookie.
 * 
 */
ZmCsfeCommand.clearAuthToken =
function() {
	AjxCookie.deleteCookie(document, ZmCsfeCommand._COOKIE_NAME, "/");
};

/**
 * Gets the session id.
 * 
 * @return	{String}	the session id
 */
ZmCsfeCommand.getSessionId =
function() {
	return ZmCsfeCommand._sessionId;
};

/**
 * Sets the session id and, if the session id is new, designates the previous
 * session id as stale.
 * 
 * @param	{String}	sessionId		the session id
 * 
 */
ZmCsfeCommand.setSessionId =
function(sessionId) {
    var sid = ZmCsfeCommand.extractSessionId(sessionId);
    if (sid) {
        if (sid && !ZmCsfeCommand._staleSession[sid]) {
            if (sid != ZmCsfeCommand._sessionId) {
                if (ZmCsfeCommand._sessionId) {
                    // Mark the old session as stale...
                    ZmCsfeCommand._staleSession[ZmCsfeCommand._sessionId] = true;
                }
                // ...before accepting the new session.
                ZmCsfeCommand._sessionId = sid;
            }
        }
    }
};

ZmCsfeCommand.clearSessionId =
function() {
	ZmCsfeCommand._sessionId = null;
};

/**
 * Isolates the parsing of the various forms of session types that we
 * might have to handle.
 *
 * @param {mixed} session Any valid session object: string, number, object,
 * or array.
 * @return {Number|Null} If the input contained a valid session object, the
 * session number will be returned. If the input is not valid, null will
 * be returned.
 */
ZmCsfeCommand.extractSessionId =
function(session) {
    var id;

    if (session instanceof Array) {
        // Array form
	    session = session[0].id;
    }
    else if (session && session.id) {
        // Object form
        session = session.id;
    }

    // We either have extracted the id or were given some primitive form.
    // Whatever we have at this point, attempt conversion and clean up response.
    id = parseInt(session, 10);
    // Normalize response
    if (isNaN(id)) {
        id = null;
    }

	return id;
};

/**
 * Converts a fault to an exception.
 * 
 * @param	{Hash}	fault		the fault
 * @param	{Hash}	params		a hash of parameters
 * @return	{ZmCsfeException}	the exception
 */
ZmCsfeCommand.faultToEx =
function(fault, params) {
	var newParams = {
		msg: AjxStringUtil.getAsString(fault.Reason.Text),
		code: AjxStringUtil.getAsString(fault.Detail.Error.Code),
		method: (params ? params.methodNameStr : null),
		detail: AjxStringUtil.getAsString(fault.Code.Value),
		data: fault.Detail.Error.a,
		trace: (fault.Detail.Error.Trace || "")
	};

	var request;
	if (params) {
		if (params.soapDoc) {
			// note that we don't pretty-print XML if we get a soapDoc
			newParams.request = params.soapDoc.getXml();
		} else if (params.jsonRequestObj) {
			if (params.jsonRequestObj && params.jsonRequestObj.Header && params.jsonRequestObj.Header.context) {
				params.jsonRequestObj.Header.context.authToken = "(removed)";
			}
			newParams.request = AjxStringUtil.prettyPrint(params.jsonRequestObj, true);
		}
	}

	return new ZmCsfeException(newParams);
};

/**
 * Gets the method name of the given request or response.
 *
 * @param {AjxSoapDoc|Object}	request	the request
 * @return	{String}			the method name or "[unknown]"
 */
ZmCsfeCommand.getMethodName =
function(request) {

	// SOAP request
	var methodName = (request && request._methodEl && request._methodEl.tagName)
		? request._methodEl.tagName : null;

	if (!methodName) {
		for (var prop in request) {
			if (/Request|Response$/.test(prop)) {
				methodName = prop;
				break;
			}
		}
	}
	return (methodName || "[unknown]");
};

/**
 * Sends a SOAP request to the server and processes the response. The request can be in the form
 * of a SOAP document, or a JSON object.
 *
 * @param	{Hash}			params				a hash of parameters:
 * @param	{AjxSoapDoc}	soapDoc				the SOAP document that represents the request
 * @param	{Object}		jsonObj				the JSON object that represents the request (alternative to soapDoc)
 * @param	{Boolean}		noAuthToken			if <code>true</code>, the check for an auth token is skipped
 * @param	{Boolean}		authToken			authToken to use instead of the local one
 * @param	{String}		serverUri			the URI to send the request to
 * @param	{String}		targetServer		the host that services the request
 * @param	{Boolean}		useXml				if <code>true</code>, an XML response is requested
 * @param	{Boolean}		noSession			if <code>true</code>, no session info is included
 * @param	{String}		changeToken			the current change token
 * @param	{int}			highestNotifySeen 	the sequence # of the highest notification we have processed
 * @param	{Boolean}		asyncMode			if <code>true</code>, request sent asynchronously
 * @param	{AjxCallback}	callback			the callback to run when response is received (async mode)
 * @param	{Boolean}		logRequest			if <code>true</code>, SOAP command name is appended to server URL
 * @param	{String}		accountId			the ID of account to execute on behalf of
 * @param	{String}		accountName			the name of account to execute on behalf of
 * @param	{Boolean}		skipAuthCheck		if <code>true</code> to skip auth check (i.e. do not check if auth token has changed)
 * @param	{constant}		resend				the reason for resending request
 * @param	{boolean}		useStringify1		use JSON.stringify1 (gets around IE child win issue with Array)
 * @param	{boolean}		emptyResponseOkay	if true, empty or no response from server is not an erro
 */
ZmCsfeCommand.prototype.invoke =
function(params) {
	this.cancelled = false;
	if (!(params && (params.soapDoc || params.jsonObj))) { return; }

	var requestStr = ZmCsfeCommand.getRequestStr(params);

	var rpcCallback;
	try {
		var uri = (params.serverUri || ZmCsfeCommand.serverUri) + params.methodNameStr;
		this._st = new Date();
		
		var requestHeaders = {"Content-Type": "application/soap+xml; charset=utf-8"};
			
		if (params.asyncMode) {
			//DBG.println(AjxDebug.DBG1, "set callback for asynchronous response");
			rpcCallback = new AjxCallback(this, this._runCallback, [params]);
			this._rpcId = AjxRpc.invoke(requestStr, uri, requestHeaders, rpcCallback);
		} else {
			//DBG.println(AjxDebug.DBG1, "parse response synchronously");
			var response = AjxRpc.invoke(requestStr, uri, requestHeaders);
			return (!params.returnXml) ? (this._getResponseData(response, params)) : response;
		}
	} catch (ex) {
		this._handleException(ex, params, rpcCallback);
	}
};

/**
 * Sends a REST request to the server via GET and returns the response.
 *
 * @param {Hash}	params			a hash of parameters
 * @param	{String}       params.restUri			the REST URI to send the request to
 * @param	{Boolean}       params.asyncMode			if <code>true</code> request sent asynchronously
 * @param	{AjxCallback}	params.callback			the callback to run when response is received (async mode)
 */
ZmCsfeCommand.prototype.invokeRest =
function(params) {

	if (!(params && params.restUri)) { return; }

	var rpcCallback;
	try {
		this._st = new Date();
		if (params.asyncMode) {
			rpcCallback = new AjxCallback(this, this._runCallback, [params]);
			this._rpcId = AjxRpc.invoke(null, params.restUri, null, rpcCallback, true);
		} else {
			var response = AjxRpc.invoke(null, params.restUri, null, null, true);
			return response.text;
		}
	} catch (ex) {
		this._handleException(ex, params, rpcCallback);
	}
};

/**
 * Cancels this request (which must be async).
 * 
 */
ZmCsfeCommand.prototype.cancel =
function() {
	DBG.println("req", "CSFE cancel: " + this._rpcId);
	if (!this._rpcId) { return; }
	this.cancelled = true;
	var req = AjxRpc.getRpcRequestById(this._rpcId);
	if (req) {
		req.cancel();
		if (AjxEnv.isFirefox3_5up) {
			AjxRpc.removeRpcCtxt(req);
		}
	}
};

/**
 * Gets the request string.
 * 
 * @param	{Hash}	params		a hash of parameters
 * @return	{String}	the request string
 */
ZmCsfeCommand.getRequestStr =
function(params) {
	return 	params.soapDoc ? ZmCsfeCommand._getSoapRequestStr(params) : ZmCsfeCommand._getJsonRequestStr(params);
};

/**
 * @private
 */
ZmCsfeCommand._getJsonRequestStr =
function(params) {

	var obj = {Header:{}, Body:params.jsonObj};

	var context = obj.Header.context = {_jsns:"urn:zimbra"};
	var ua_name = ["ZimbraWebClient - ", AjxEnv.browser, " (", AjxEnv.platform, ")"].join("");
	context.userAgent = {name:ua_name};
	if (ZmCsfeCommand.clientVersion) {
		context.userAgent.version = ZmCsfeCommand.clientVersion;
	}
	if (params.noSession) {
		context.nosession = {};
	} else {
		var sessionId = ZmCsfeCommand.getSessionId();
		if (sessionId) {
			context.session = {_content:sessionId, id:sessionId};
		} else {
			context.session = {};
		}
	}
	if (params.targetServer) {
		context.targetServer = {_content:params.targetServer};
	}
	if (params.highestNotifySeen) {
		context.notify = {seq:params.highestNotifySeen};
	}
	if (params.changeToken) {
		context.change = {token:params.changeToken, type:"new"};
	}

	// if we're not checking auth token, we don't want token/acct mismatch	
	if (!params.skipAuthCheck) {
		if (params.accountId) {
			context.account = {_content:params.accountId, by:"id"}
		} else if (params.accountName) {
			context.account = {_content:params.accountName, by:"name"}
		}
	}
	
	// Tell server what kind of response we want
	if (params.useXml) {
		context.format = {type:"xml"};
	}

	params.methodNameStr = ZmCsfeCommand.getMethodName(params.jsonObj);

	// Get auth token from cookie if required
	if (!params.noAuthToken) {
		var authToken = params.authToken || ZmCsfeCommand.getAuthToken();
		if (!authToken) {
			throw new ZmCsfeException(ZMsg.authTokenRequired, ZmCsfeException.NO_AUTH_TOKEN, params.methodNameStr);
		}
		if (ZmCsfeCommand._curAuthToken && !params.skipAuthCheck && 
			(params.resend != ZmCsfeCommand.REAUTH) && (authToken != ZmCsfeCommand._curAuthToken)) {
			throw new ZmCsfeException(ZMsg.authTokenChanged, ZmCsfeException.AUTH_TOKEN_CHANGED, params.methodNameStr);
		}
		context.authToken = ZmCsfeCommand._curAuthToken = authToken;
	}
	else if (ZmCsfeCommand.noAuth) {
		throw new ZmCsfeException(ZMsg.authRequired, ZmCsfeException.NO_AUTH_TOKEN, params.methodNameStr);
	}

	if (window.csrfToken) {
		context.csrfToken = window.csrfToken;
	}

	AjxDebug.logSoapMessage(params);
	DBG.dumpObj(AjxDebug.DBG1, obj);

	params.jsonRequestObj = obj;
	
	var requestStr = (params.useStringify1 ?
	                  JSON.stringify1(obj) : JSON.stringify(obj));

	// bug 74240: escape non-ASCII characters to prevent the browser from
	// combining decomposed characters in paths
	return AjxStringUtil.jsEncode(requestStr)
};

/**
 * @private
 */
ZmCsfeCommand._getSoapRequestStr =
function(params) {

	var soapDoc = params.soapDoc;

	if (!params.resend) {

		// Add the SOAP header and context
		var hdr = soapDoc.createHeaderElement();
		var context = soapDoc.set("context", null, hdr, "urn:zimbra");
	
		var ua = soapDoc.set("userAgent", null, context);
		var name = ["ZimbraWebClient - ", AjxEnv.browser, " (", AjxEnv.platform, ")"].join("");
		ua.setAttribute("name", name);
		if (ZmCsfeCommand.clientVersion) {
			ua.setAttribute("version", ZmCsfeCommand.clientVersion);
		}
	
		if (params.noSession) {
			soapDoc.set("nosession", null, context);
		} else {
			var sessionId = ZmCsfeCommand.getSessionId();
			var si = soapDoc.set("session", null, context);
			if (sessionId) {
				si.setAttribute("id", sessionId);
			}
		}
		if (params.targetServer) {
			soapDoc.set("targetServer", params.targetServer, context);
		}
		if (params.highestNotifySeen) {
		  	var notify = soapDoc.set("notify", null, context);
		  	notify.setAttribute("seq", params.highestNotifySeen);
		}
		if (params.changeToken) {
			var ct = soapDoc.set("change", null, context);
			ct.setAttribute("token", params.changeToken);
			ct.setAttribute("type", "new");
		}
	
		// if we're not checking auth token, we don't want token/acct mismatch	
		if (!params.skipAuthCheck) {
			if (params.accountId) {
				var acc = soapDoc.set("account", params.accountId, context);
				acc.setAttribute("by", "id");
			} else if (params.accountName) {
				var acc = soapDoc.set("account", params.accountName, context);
				acc.setAttribute("by", "name");
			}
		}
	
		if (params.skipExpiredToken) {
			var tokenControl = soapDoc.set("authTokenControl", null, context);
			tokenControl.setAttribute("voidOnExpired", "1");
		}	
		// Tell server what kind of response we want
		if (!params.useXml) {
			var js = soapDoc.set("format", null, context);
			js.setAttribute("type", "js");
		}
	}

	params.methodNameStr = ZmCsfeCommand.getMethodName(soapDoc);

	// Get auth token from cookie if required
	if (!params.noAuthToken) {
		var authToken = params.authToken || ZmCsfeCommand.getAuthToken();
		if (!authToken) {
			throw new ZmCsfeException(ZMsg.authTokenRequired, ZmCsfeException.NO_AUTH_TOKEN, params.methodNameStr);
		}
		if (ZmCsfeCommand._curAuthToken && !params.skipAuthCheck && 
			(params.resend != ZmCsfeCommand.REAUTH) && (authToken != ZmCsfeCommand._curAuthToken)) {
			throw new ZmCsfeException(ZMsg.authTokenChanged, ZmCsfeException.AUTH_TOKEN_CHANGED, params.methodNameStr);
		}
		ZmCsfeCommand._curAuthToken = authToken;
		if (params.resend == ZmCsfeCommand.REAUTH) {
			// replace old auth token with current one
			var nodes = soapDoc.getDoc().getElementsByTagName("authToken");
			if (nodes && nodes.length == 1) {
				DBG.println(AjxDebug.DBG1, "Re-auth: replacing auth token");
				nodes[0].firstChild.data = authToken;
			} else {
				// can't find auth token, just add it to context element
				nodes = soapDoc.getDoc().getElementsByTagName("context");
				if (nodes && nodes.length == 1) {
					DBG.println(AjxDebug.DBG1, "Re-auth: re-adding auth token");
					soapDoc.set("authToken", authToken, nodes[0]);
				} else {
					DBG.println(AjxDebug.DBG1, "Re-auth: could not find context!");
				}
			}
		} else if (!params.resend){
			soapDoc.set("authToken", authToken, context);
		}
	}
	else if (ZmCsfeCommand.noAuth && !params.ignoreAuthToken) {
		throw new ZmCsfeException(ZMsg.authRequired, ZmCsfeException.NO_AUTH_TOKEN, params.methodNameStr);
	}

	if (window.csrfToken) {
		soapDoc.set("csrfToken", window.csrfToken, context);
	}

	AjxDebug.logSoapMessage(params);
	DBG.printXML(AjxDebug.DBG1, soapDoc.getXml());

	return soapDoc.getXml();
};

/**
 * Runs the callback that was passed to invoke() for an async command.
 *
 * @param {AjxCallback}	callback	the callback to run with response data
 * @param {Hash}	params	a hash of parameters (see method invoke())
 * 
 * @private
 */
ZmCsfeCommand.prototype._runCallback =
function(params, result) {
	if (!result) { return; }
	if (this.cancelled && params.skipCallbackIfCancelled) {	return; }

	var response;
	if (result instanceof ZmCsfeResult) {
		response = result; // we already got an exception and packaged it
	} else {
		response = this._getResponseData(result, params);
	}
	this._en = new Date();

	if (params.callback && response) {
		params.callback.run(response);
	} else if (!params.emptyResponseOkay) {
		DBG.println(AjxDebug.DBG1, "ZmCsfeCommand.prototype._runCallback: Missing callback!");
	}
};

/**
 * Takes the response to an RPC request and returns a JS object with the response data.
 *
 * @param {Object}	response	the RPC response with properties "text" and "xml"
 * @param {Hash}	params	a hash of parameters (see method invoke())
 */
ZmCsfeCommand.prototype._getResponseData =
function(response, params) {
	this._en = new Date();
	DBG.println(AjxDebug.DBG1, "ROUND TRIP TIME: " + (this._en.getTime() - this._st.getTime()));

	var result = new ZmCsfeResult();
	var xmlResponse = false;
	var restResponse = Boolean(params.restUri);
	var respDoc = null;

	// check for un-parseable HTML error response from server
	if (!response.success && !response.xml && (/<html/i.test(response.text))) {
		// bad XML or JS response that had no fault
		var ex = new ZmCsfeException(null, ZmCsfeException.CSFE_SVC_ERROR, params.methodNameStr, "HTTP response status " + response.status);
		if (params.asyncMode) {
			result.set(ex, true);
			return result;
		} else {
			throw ex;
		}
	}

	if (typeof(response.text) == "string" && response.text.indexOf("{") == 0) {
		respDoc = response.text;
	} else if (!restResponse) {
		// an XML response if we requested one, or a fault
		try {
			xmlResponse = true;
			if (!(response.text || (response.xml && (typeof response.xml) == "string"))) {
				if (params.emptyResponseOkay) {
					return null;
				}
				else {
					// If we can't reach the server, req returns immediately with an empty response rather than waiting and timing out
					throw new ZmCsfeException(null, ZmCsfeException.EMPTY_RESPONSE, params.methodNameStr);
				}
			}
			// responseXML is empty under IE
			respDoc = (AjxEnv.isIE || response.xml == null) ? AjxSoapDoc.createFromXml(response.text) :
															  AjxSoapDoc.createFromDom(response.xml);
		} catch (ex) {
			DBG.dumpObj(AjxDebug.DBG1, ex);
			if (params.asyncMode) {
				result.set(ex, true);
				return result;
			} else {
				throw ex;
			}
		}
		if (!respDoc) {
			var ex = new ZmCsfeException(null, ZmCsfeException.SOAP_ERROR, params.methodNameStr, "Bad XML response doc");
			DBG.dumpObj(AjxDebug.DBG1, ex);
			if (params.asyncMode) {
				result.set(ex, true);
				return result;
			} else {
				throw ex;
			}
		}
	}

	var obj = restResponse ? response.text : {};

	if (xmlResponse) {
		DBG.printXML(AjxDebug.DBG1, respDoc.getXml());
		obj = respDoc._xmlDoc.toJSObject(true, false, true);
	} else if (!restResponse) {
		try {
			obj = JSON.parse(respDoc);
		} catch (ex) {
			if (ex.name == "SyntaxError") {
				ex = new ZmCsfeException(null, ZmCsfeException.BAD_JSON_RESPONSE, params.methodNameStr, respDoc);
				AjxDebug.println(AjxDebug.BAD_JSON, "bad json. respDoc=" + respDoc);
			}
			DBG.dumpObj(AjxDebug.DBG1, ex);
			if (params.asyncMode) {
				result.set(ex, true);
				return result;
			} else {
				throw ex;
			}
		}

	}

	params.methodNameStr = ZmCsfeCommand.getMethodName(obj.Body);
	AjxDebug.logSoapMessage(params);
	DBG.dumpObj(AjxDebug.DBG1, obj, -1);

	var fault = obj && obj.Body && obj.Body.Fault;
	if (fault) {
		// JS response with fault
		if (AjxUtil.isString(fault) && fault.indexOf("<")==0) { // We got an xml string
			fault = AjxXmlDoc.createFromXml(fault).toJSObject(true, false, true);
		}
		var ex = ZmCsfeCommand.faultToEx(fault, params);
		if (params.asyncMode) {
			result.set(ex, true, obj.Header);
			return result;
		} else {
			throw ex;
		}
	} else if (!response.success) {
		// bad XML or JS response that had no fault
		var ex = new ZmCsfeException(null, ZmCsfeException.CSFE_SVC_ERROR, params.methodNameStr, "HTTP response status " + response.status);
		if (params.asyncMode) {
			result.set(ex, true);
			return result;
		} else {
			throw ex;
		}
	} else {
		// good response
		if (params.asyncMode) {
			result.set(obj);
		}
	}

	// check for new session ID
	var session = obj.Header && obj.Header.context && obj.Header.context.session;
    ZmCsfeCommand.setSessionId(session);

	return params.asyncMode ? result : obj;
};

/**
 * @private
 */
ZmCsfeCommand.prototype._handleException =
function(ex, params, callback) {
	if (!(ex && (ex instanceof ZmCsfeException || ex instanceof AjxSoapException || ex instanceof AjxException))) {
		var newEx = new ZmCsfeException();
		newEx.method = params.methodNameStr || params.restUri;
		newEx.detail = ex ? ex.toString() : "undefined exception";
		newEx.code = ZmCsfeException.UNKNOWN_ERROR;
		newEx.msg = "Unknown Error";
		ex = newEx;
	}
	if (params.asyncMode) {
		callback.run(new ZmCsfeResult(ex, true));
	} else {
		throw ex;
	}
};
}
if (AjxPackage.define("zimbra.csfe.ZmCsfeException")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2023 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the exception class.
 */
/**
 * Creates an exception.
 * @class
 * This class represents an exception returned by the server as a response, generally as a fault. The fault
 * data is converted to properties of the exception.
 *
 * @param {Hash}	params	a hash of parameters
 * @param {String}      params.msg		the explanation (Fault.Reason.Text)
 * @param {String}      params.code		the error code (Fault.Detail.Error.Code)
 * @param {String}      params.method	the request name
 * @param {String}      params.detail	the Fault.Code.Value
 * @param {Object}      [params.data]		an optional structured fault data (Fault.Detail.Error.a)
 * @param {String}      params.trace		the trace info (Fault.Detail.Error.Trace)
 * @param {String}       params.request	the SOAP or JSON that represents the request
 * 
 * @extends		AjxException
 */
ZmCsfeException = function(params) {

	params = Dwt.getParams(arguments, ZmCsfeException.PARAMS);

	AjxException.call(this, params.msg, params.code, params.method, params.detail);
	
	if (params.data) {
		this.data = {};
		for (var i = 0; i < params.data.length; i++) {
			var item = params.data[i];
			var key = item.n;
			if (!this.data[key]) {
				this.data[key] = [];
			}
			this.data[key].push(item._content);
		}
	}
	
	this.trace = params.trace;
	this.request = params.request;
};

ZmCsfeException.PARAMS = ["msg", "code", "method", "detail", "data", "trace"];

ZmCsfeException.prototype = new AjxException;
ZmCsfeException.prototype.constructor = ZmCsfeException;
ZmCsfeException.prototype.isZmCsfeException = true;

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmCsfeException.prototype.toString =
function() {
	return "ZmCsfeException";
};

//
// Constants
//

// structured data keys
ZmCsfeException.MAIL_SEND_ADDRESS_FAILURE_INVALID = "invalid";
ZmCsfeException.MAIL_SEND_ADDRESS_FAILURE_UNSENT = "unsent";

//
// Static functions
//

/**
 * Gets the error messages.
 * 
 * @param	{String}	code	the code
 * @param	{Array}	args		the message format args
 * 
 * @return	{String}	the message
 */
ZmCsfeException.getErrorMsg =
function(code, args) {
	var msg = ZMsg[code];
	if (!msg) {
		ZmCsfeException._unknownFormat = ZmCsfeException._unknownFormat || new AjxMessageFormat(ZMsg.unknownError);
		return ZmCsfeException._unknownFormat.format(code);
	}
	this.msg = this.msg || msg;
	return args ? AjxMessageFormat.format(msg, args) : msg;
};

//
// Public methods
//

/**
 * Gets the error message.
 * 
 * @param	{Array}	args		the message format args
 * @return	{String}	the message
 */
ZmCsfeException.prototype.getErrorMsg =
function(args) {
	return ZmCsfeException.getErrorMsg(this.code, args);
};

/**
 * Gets the data.
 * 
 * @param	{Object}	key		the key
 * 
 * @return	{Object}	the data
 */
ZmCsfeException.prototype.getData =
function(key) {
	return this.data && this.data[key];
};

//
// Constants for server exceptions
//

ZmCsfeException.AUTH_TOKEN_CHANGED					= "AUTH_TOKEN_CHANGED";
ZmCsfeException.BAD_JSON_RESPONSE					= "BAD_JSON_RESPONSE";
ZmCsfeException.CSFE_SVC_ERROR						= "CSFE_SVC_ERROR";
ZmCsfeException.EMPTY_RESPONSE						= "EMPTY_RESPONSE";
ZmCsfeException.NETWORK_ERROR						= "NETWORK_ERROR";
ZmCsfeException.NO_AUTH_TOKEN						= "NO_AUTH_TOKEN";
ZmCsfeException.SOAP_ERROR							= "SOAP_ERROR";

ZmCsfeException.LICENSE_ERROR						= "service.LICENSE_ERROR";
ZmCsfeException.SVC_ALREADY_IN_PROGRESS				= "service.ALREADY_IN_PROGRESS";
ZmCsfeException.SVC_AUTH_EXPIRED					= "service.AUTH_EXPIRED";
ZmCsfeException.SVC_AUTH_REQUIRED					= "service.AUTH_REQUIRED";
ZmCsfeException.SVC_FAILURE							= "service.FAILURE";
ZmCsfeException.SVC_INVALID_REQUEST					= "service.INVALID_REQUEST";
ZmCsfeException.SVC_PARSE_ERROR						= "service.PARSE_ERROR";
ZmCsfeException.SVC_PERM_DENIED						= "service.PERM_DENIED";
ZmCsfeException.SVC_RESOURCE_UNREACHABLE			= "service.RESOURCE_UNREACHABLE";
ZmCsfeException.SVC_UNKNOWN_DOCUMENT				= "service.UNKNOWN_DOCUMENT";
ZmCsfeException.SVC_TEMPORARILY_UNAVAILABLE			= "service.TEMPORARILY_UNAVAILABLE";
ZmCsfeException.SVC_WRONG_HOST						= "service.WRONG_HOST";
ZmCsfeException.SIEVE_SCRIPT_MAX_SIZE_EXCEPTION		= "service.SIEVE_SCRIPT_MAX_SIZE_EXCEPTION";
ZmCsfeException.RECOVERY_EMAIL_SAME_AS_PRIMARY_OR_ALIAS	= "service.RECOVERY_EMAIL_SAME_AS_PRIMARY_OR_ALIAS";

ZmCsfeException.ACCT_AUTH_FAILED					= "account.AUTH_FAILED";
ZmCsfeException.ACCT_CHANGE_PASSWORD				= "account.CHANGE_PASSWORD";
ZmCsfeException.ACCT_EXISTS							= "account.ACCOUNT_EXISTS";
ZmCsfeException.ACCT_TOO_MANY_ACCOUNTS      		= "account.TOO_MANY_ACCOUNTS" ;
ZmCsfeException.ACCT_INVALID_ATTR_VALUE				= "account.INVALID_ATTR_VALUE";
ZmCsfeException.ACCT_INVALID_PASSWORD				= "account.INVALID_PASSWORD";
ZmCsfeException.ACCT_INVALID_PREF_NAME				= "account.INVALID_PREF_NAME";
ZmCsfeException.ACCT_INVALID_PREF_VALUE				= "account.INVALID_PREF_VALUE";
ZmCsfeException.ACCT_MAINTENANCE_MODE				= "account.MAINTENANCE_MODE";
ZmCsfeException.ACCT_NO_SUCH_ACCOUNT				= "account.NO_SUCH_ACCOUNT";
ZmCsfeException.ACCT_NO_SUCH_SAVED_SEARCH			= "account.NO_SUCH_SAVED_SEARCH";
ZmCsfeException.ACCT_NO_SUCH_TAG					= "account.ACCT_NO_SUCH_TAG";
ZmCsfeException.ACCT_PASS_CHANGE_TOO_SOON			= "account.PASSWORD_CHANGE_TOO_SOON";
ZmCsfeException.ACCT_PASS_LOCKED					= "account.PASSWORD_LOCKED";
ZmCsfeException.ACCT_PASS_RECENTLY_USED				= "account.PASSWORD_RECENTLY_USED";
ZmCsfeException.ACCT_CANNOT_DISABLE_TWO_FACTOR_AUTH	= "account.CANNOT_DISABLE_TWO_FACTOR_AUTH";
ZmCsfeException.ACCT_TWO_FACTOR_AUTH_INVALID_CONFIG	= "account.TWO_FACTOR_AUTH_INVALID_CONFIG";
ZmCsfeException.ACCT_TWO_FACTOR_AUTH_METHOD_NOT_ALLOWED	= "account.TWO_FACTOR_AUTH_METHOD_NOT_ALLOWED";
ZmCsfeException.ACCT_CANNOT_ENABLE_TWO_FACTOR_AUTH	= "account.CANNOT_ENABLE_TWO_FACTOR_AUTH";
ZmCsfeException.COS_EXISTS							= "account.COS_EXISTS";
ZmCsfeException.DISTRIBUTION_LIST_EXISTS			= "account.DISTRIBUTION_LIST_EXISTS";
ZmCsfeException.DOMAIN_EXISTS						= "account.DOMAIN_EXISTS";
ZmCsfeException.DOMAIN_NOT_EMPTY					= "account.DOMAIN_NOT_EMPTY";
ZmCsfeException.IDENTITY_EXISTS						= "account.IDENTITY_EXISTS";
ZmCsfeException.NO_SUCH_DISTRIBUTION_LIST			= "account.NO_SUCH_DISTRIBUTION_LIST";
ZmCsfeException.NO_SUCH_DOMAIN						= "account.NO_SUCH_DOMAIN";
ZmCsfeException.MAINTENANCE_MODE					= "account.MAINTENANCE_MODE";
ZmCsfeException.TOO_MANY_IDENTITIES					= "account.TOO_MANY_IDENTITIES";
ZmCsfeException.TOO_MANY_SEARCH_RESULTS				= "account.TOO_MANY_SEARCH_RESULTS";
ZmCsfeException.NO_SUCH_COS 						= "account.NO_SUCH_COS";
ZmCsfeException.SIGNATURE_EXISTS                    = "account.SIGNATURE_EXISTS";

ZmCsfeException.CANNOT_CHANGE_VOLUME = "volume.CANNOT_CHANGE_TYPE_OF_CURRVOL";
ZmCsfeException.CANNOT_DELETE_VOLUME_IN_USE = "volume.CANNOT_DELETE_VOLUME_IN_USE";
ZmCsfeException.NO_SUCH_VOLUME						= "volume.NO_SUCH_VOLUME";
ZmCsfeException.ALREADY_EXISTS						= "volume.ALREADY_EXISTS";
ZmCsfeException.VOLUME_NO_SUCH_PATH					= "volume.NO_SUCH_PATH";

ZmCsfeException.MAIL_ALREADY_EXISTS					= "mail.ALREADY_EXISTS";
ZmCsfeException.MAIL_IMMUTABLE						= "mail.IMMUTABLE_OBJECT";
ZmCsfeException.MAIL_INVALID_NAME					= "mail.INVALID_NAME";
ZmCsfeException.MAIL_INVITE_OUT_OF_DATE				= "mail.INVITE_OUT_OF_DATE";
ZmCsfeException.MAIL_MAINTENANCE_MODE				= "mail.MAINTENANCE";
ZmCsfeException.MAIL_MESSAGE_TOO_BIG				= "mail.MESSAGE_TOO_BIG";
ZmCsfeException.MAIL_MUST_RESYNC					= "mail.MUST_RESYNC";
ZmCsfeException.MAIL_NO_SUCH_CALITEM				= "mail.NO_SUCH_CALITEM";
ZmCsfeException.MAIL_NO_SUCH_CONV					= "mail.NO_SUCH_CONV";
ZmCsfeException.MAIL_NO_SUCH_CONTACT				= "mail.NO_SUCH_CONTACT";
ZmCsfeException.MAIL_NO_SUCH_FOLDER					= "mail.NO_SUCH_FOLDER";
ZmCsfeException.MAIL_NO_SUCH_ITEM					= "mail.NO_SUCH_ITEM";
ZmCsfeException.MAIL_NO_SUCH_MOUNTPOINT				= "mail.NO_SUCH_MOUNTPOINT";
ZmCsfeException.MAIL_NO_SUCH_MSG					= "mail.NO_SUCH_MSG";
ZmCsfeException.MAIL_NO_SUCH_PART					= "mail.NO_SUCH_PART";
ZmCsfeException.MAIL_NO_SUCH_TAG					= "mail.NO_SUCH_TAG";
ZmCsfeException.MAIL_QUERY_PARSE_ERROR				= "mail.QUERY_PARSE_ERROR";
ZmCsfeException.MAIL_QUOTA_EXCEEDED					= "mail.QUOTA_EXCEEDED";
ZmCsfeException.MAIL_SEND_ABORTED_ADDRESS_FAILURE	= "mail.SEND_ABORTED_ADDRESS_FAILURE";
ZmCsfeException.MAIL_SEND_FAILURE					= "mail.SEND_FAILURE";
ZmCsfeException.MAIL_TOO_MANY_CONTACTS				= "mail.TOO_MANY_CONTACTS";
ZmCsfeException.MAIL_TOO_MANY_TERMS					= "mail.TOO_MANY_QUERY_TERMS_EXPANDED";
ZmCsfeException.MAIL_UNABLE_TO_IMPORT_APPOINTMENTS	= "mail.MAIL_UNABLE_TO_IMPORT_APPOINTMENTS";
ZmCsfeException.MAIL_UNABLE_TO_IMPORT_CONTACTS		= "mail.UNABLE_TO_IMPORT_CONTACTS";
ZmCsfeException.MODIFY_CONFLICT						= "mail.MODIFY_CONFLICT";
ZmCsfeException.TOO_MANY_TAGS						= "mail.TOO_MANY_TAGS";
ZmCsfeException.CANNOT_RENAME                       = "mail.CANNOT_RENAME";
ZmCsfeException.CANNOT_UNLOCK                       = "mail.CANNOT_UNLOCK";
ZmCsfeException.CANNOT_LOCK                         = "mail.CANNOT_LOCK";
ZmCsfeException.LOCKED                              = "mail.LOCKED";
ZmCsfeException.UPLOAD_REJECTED						= "mail.UPLOAD_REJECTED";

ZmCsfeException.MUST_BE_ORGANIZER					= "mail.MUST_BE_ORGANIZER";


ZmCsfeException.OFFLINE_ONLINE_ONLY_OP				= "offline.ONLINE_ONLY_OP";
}
if (AjxPackage.define("zimbra.csfe.ZmCsfeResult")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the result class.
 */

/**
 * Creates a CSFE result object.
 * @class
 * This class represents the result of a CSFE request. The data is either the 
 * response that was received, or an exception. If the request resulted in a 
 * SOAP fault from the server, there will also be a SOAP header present.
 *
 * @author Conrad Damon
 * 
 * @param {Object}	data			the response data
 * @param {Boolean}	isException	if <code>true</code>, the data is an exception object
 * @param {Object}	header			the SOAP header
 * 
 */
ZmCsfeResult = function(data, isException, header) {
	this.set(data, isException, header);
};

ZmCsfeResult.prototype.isZmCsfeResult = true;
ZmCsfeResult.prototype.toString = function() { return "ZmCsfeResult"; };

/**
 * Sets the content of the result.
 *
 * @param {Object}	data			the response data
 * @param {Boolean}	isException	if <code>true</code>, the data is an exception object
 * @param {Object}	header			the SOAP header
 */
ZmCsfeResult.prototype.set =
function(data, isException, header) {
	this._data = data;
	this._isException = (isException === true);
	this._header = header;
};

/**
 * Gets the response data. If there was an exception, throws the exception.
 * 
 * @return	{Object}	the data
 */
ZmCsfeResult.prototype.getResponse =
function() {
	if (this._isException) {
		throw this._data;
	} else {
		return this._data;
	}
};

/**
 * Gets the exception object, if any.
 * 
 * @return	{ZmCsfeException}	the exception or <code>null</code> for none
 */
ZmCsfeResult.prototype.getException =
function() {
	return this._isException ? this._data : null;
};

/**
 * Checks if this result is an exception.
 * 
 * @return	{Boolean}	<code>true</code> if an exception
 */
ZmCsfeResult.prototype.isException = 
function() {
	return this._isException;
};

/**
 * Gets the SOAP header that came with a SOAP fault.
 * 
 * @return	{String}	the header
 */
ZmCsfeResult.prototype.getHeader =
function() {
	return this._header;
};
}

if (AjxPackage.define("zimbraMail.core.ZmId")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains ids.
 * 
 */

/**
 * Constructor
 * @class
 * This class is responsible for providing unique, predictable IDs for HTML elements.
 * That way, code outside the client can easily locate particular elements.
 * <br/>
 * <br/>
 * Not every element that has an associated JS object will have a known ID. Those are
 * allocated only for elements it would be useful to locate: major components of the UI,
 * toolbars, buttons, views, menus, some menu items, some selects, and miscellaneous other
 * components.
 * <br/>
 * <br/>
 * Unless noted otherwise, a getElementById() on any of the non-skin IDs will return a DIV. One exception
 * is input fields. The ID is given to the DwtInputField's actual INPUT, rather than to the
 * DIV that contains it. Most other exceptions are table-related: TABLE, TR, and TD.
 * <br/>
 * <br/>
 * There is a simple naming scheme for the IDs themselves. Each ID starts with a "z" followed
 * by one to a few letters that indicate the type of object (widget) represented by the element:
 * 
 * <ul>
 * <li><b>z</b> a component that is not a special-purpose widget listed below</li>
 * <li><b>ztb</b> 		a toolbar (see {@link ZmId.WIDGET})</li>
 * <li><b>zb</b> 		a button (see {@link ZmId.WIDGET_TOOLBAR})</li>
 * <li><b>zi</b> 		an input field (see {@link ZmId.WIDGET_INPUT})</li>
 * <li><b>zm</b> 		a menu (see {@link ZmId.WIDGET_MENU})</li>
 * <li><b>zmi</b> 		a menu item (see {@link ZmId.WIDGET_MENU_ITEM})</li>
 * <li><b>zs</b> 		a select (see {@link ZmId.WIDGET_SELECT})</li>
 * <li><b>zov</b> 		an overview (see {@link ZmId.WIDGET_OVERVIEW})</li>
 * <li><b>zt</b> 		a tree (see {@link ZmId.WIDGET_TREE})</li>
 * <li><b>zti</b> 		a tree item (see {@link ZmId.WIDGET_TREE_ITEM})</li>
 * <li><b>ztih</b> 	a tree item header (see {@link ZmId.WIDGET_TREE_ITEM_HDR})</li>
 *</ul>
 *
 * The skin defines its own container IDs starting with "skin_", which we provide constants for here.
 * 
 * @author Conrad Damon
 */
ZmId = function() {};

//
// Element IDs, and functions to generate them
//

// widget types (used to prefix IDs)
// TODO: move most of these to DwtId
ZmId.WIDGET					= "z";			// generic element
ZmId.WIDGET_VIEW			= "zv";			// view within content area
ZmId.WIDGET_TOOLBAR			= "ztb";		// toolbar
ZmId.WIDGET_BUTTON			= "zb";			// button
ZmId.WIDGET_INPUT			= "zi";			// text input or textarea
ZmId.WIDGET_MENU			= "zm";			// menu
ZmId.WIDGET_MENU_ITEM		= "zmi";		// menu item
ZmId.WIDGET_SELECT			= "zs";			// dropdown select
ZmId.WIDGET_OVERVIEW_CNTR	= "zovc";		// collection of overviews
ZmId.WIDGET_OVERVIEW		= "zov";		// collection of tree views
ZmId.WIDGET_TREE			= "zt";			// tree view
ZmId.WIDGET_TREE_ITEM_HDR	= "ztih";		// root tree item
ZmId.WIDGET_TREE_ITEM		= "zti";		// tree item (node)
ZmId.WIDGET_TAB				= "ztab";		// tab button
ZmId.WIDGET_AUTOCOMPLETE	= "zac";		// autocomplete list
ZmId.WIDGET_CHECKBOX		= "zcb";		// checkbox
ZmId.WIDGET_COMBOBOX		= "zcombo";		// combo box
ZmId.WIDGET_CHOOSER			= "zchooser";	// folder chooser
ZmId.WIDGET_CALENDAR		= "zcal";		// popup calendar

//
// Preset IDs
//

/*
 * Container IDs defined by the skin.
 * 
 * These must match what's in skin.html. They are used by ZmAppViewMgr to visually
 * match components to the containers in which they should be positioned. 
 */
ZmId.SKIN_APP_BOTTOM_TOOLBAR		= "skin_container_app_bottom_toolbar";
ZmId.SKIN_APP_CHOOSER				= "skin_container_app_chooser";
ZmId.SKIN_APP_MAIN_FULL				= "skin_container_app_main_full";
ZmId.SKIN_APP_MAIN					= "skin_container_app_main";
ZmId.SKIN_APP_MAIN_ROW_FULL			= "skin_tr_main_full";
ZmId.SKIN_APP_MAIN_ROW				= "skin_tr_main";
ZmId.SKIN_APP_TOP_TOOLBAR			= "skin_container_app_top_toolbar";
ZmId.SKIN_APP_NEW_BUTTON			= "skin_container_app_new_button"; 
ZmId.SKIN_LINKS						= "skin_container_links";
ZmId.SKIN_LOGO						= "skin_container_logo";
ZmId.SKIN_QUOTA_INFO				= "skin_container_quota";
ZmId.SKIN_SASH						= "skin_container_tree_app_sash";
ZmId.SKIN_SEARCH_BUILDER			= "skin_container_search_builder";
ZmId.SKIN_SEARCH_BUILDER_TOOLBAR	= "skin_container_search_builder_toolbar";
ZmId.SKIN_SEARCH_BUILDER_TR			= "skin_tr_search_builder";
ZmId.SKIN_SEARCH					= "skin_container_search";
ZmId.SKIN_SEARCH_RESULTS_TOOLBAR	= "skin_container_search_results_toolbar";
ZmId.SKIN_REFRESH					= "skin_container_refresh";
ZmId.SKIN_OFFLINE_STATUS			= "skin_container_offline";
ZmId.SKIN_SHELL						= "skin_outer";
ZmId.SKIN_SPACING_SEARCH			= "skin_spacing_search";
ZmId.SKIN_SPLASH_SCREEN				= "skin_container_splash_screen";
ZmId.SKIN_STATUS					= "skin_container_status";
ZmId.SKIN_STATUS_ROW				= "skin_tr_status";
ZmId.SKIN_TREE_FOOTER				= "skin_container_tree_footer";
ZmId.SKIN_TREE						= "skin_container_tree";
ZmId.SKIN_USER_INFO					= "skin_container_username";
ZmId.SKIN_FOOTER					= "skin_footer";
ZmId.SKIN_AD						= "skin_adsrvc";
ZmId.SKIN_UNITTEST					= "skin_unittest";

//
// Literal IDs
//

/*
 * Top-level components. These are elements that are placed directly into skin containers.
 */
ZmId.SHELL					= "z_shell";			// the main shell
ZmId.LOADING_VIEW			= "z_loading";			// "Loading..." view
ZmId.MAIN_SASH				= "z_sash";				// sash between overview and content
ZmId.BANNER					= "z_banner";			// logo (at upper left by default)
ZmId.SEARCH_TOOLBAR			= "ztb_search";			// search toolbar
ZmId.SEARCHRESULTS_TOOLBAR	= "ztb_searchresults";	// search results toolbar
ZmId.SEARCHRESULTS_PANEL	= "z_filterPanel";		// search results filter panel
ZmId.USER_NAME				= "z_userName";			// account name
ZmId.USER_QUOTA				= "z_userQuota";		// quota
ZmId.PRESENCE				= "z_presence";			// presence
ZmId.NEW_FOLDER_BUTTON		= "zb_newFolder";		// New Folder button on current app toolbar
ZmId.STATUS_VIEW			= "z_status";			// status view (shows toast)
ZmId.TOAST					= "z_toast";			// toast
ZmId.APP_CHOOSER			= "ztb_appChooser";		// app chooser toolbar

//
// Functions for generating IDs
//
// In general, an ID created by one of these functions will consist of several fields joined
// together by a "|" character. The first field indicates the type of element/widget, and will
// be one of the ZmId.WIDGET_* constants. The remaining fields are there to ensure that the ID
// is unique.

/**
 * Generates the ID for a toolbar.
 * 
 * <p>
 * Examples: <code>ztb|CLV ztb|TV|Nav ztb|CV|Inv</code>
 * </p>
 * 
 * @param {String}	context		the toolbar context (ID of owning view)
 * @param {constant}	tbType	the type of toolbar (for example, invite or nav)
 * @return	{String}	the id
 */
ZmId.getToolbarId =
function(context, tbType) {
	return DwtId.makeId(ZmId.WIDGET_TOOLBAR, context, tbType);
};

// special toolbars
ZmId.TB_INVITE	= "Inv";
ZmId.TB_COUNTER	= "Cou";
ZmId.TB_NAV		= "Nav";
ZmId.TB_SHARE	= "Shr";
ZmId.TB_REPLY	= "Rep";
ZmId.TB_SUBSCRIBE = "Sub";

/**
 * Generates the ID for a button. Intended for use with the top toolbar, nav toolbar,
 * and invite toolbar.
 * 
 * <p>
 * Examples: <code>zb|CLV|CHECK_MAIL zb|TV|REPLY zb|COMPOSE|SEND zb|CLV|Nav|PAGE_FORWARD</code>
 * </p>
 * 
 * @param {String}	context	the toolbar context (ID of owning view)
 * @param {constant}	op	the button operation
 * @param {constant}	tbType	the type of toolbar (eg invite or nav)
 * @return	{String}	the id
 */
ZmId.getButtonId =
function(context, op, tbType) {
	return DwtId.makeId(ZmId.WIDGET_BUTTON, context, tbType, op);
};

/**
 * Generates the ID for an action menu.
 * 
 * <p>
 * Examples: <code>zm|CLV zm|Contacts zm|TV|Par</code>
 * </p>
 * 
 * @param {String}	context		the menu context (eg ID of owning view, or app)
 * @param {constant}	menuType		the type of menu (eg participant)
 * @return	{String}	the id
 */
ZmId.getMenuId =
function(context, menuType) {
	return DwtId.makeId(ZmId.WIDGET_MENU, context, menuType);
};

// special menus
ZmId.MENU_PARTICIPANT	= "Par";
ZmId.MENU_DRAFTS		= "Dra";

/**
 * Generates the ID for a menu item in an action menu.
 * 
 * <p>
 * Examples: <code>zmi|CLV|REPLY_ALL zmi|TV|Par|SEARCH</code>
 * </p>
 * 
 * @param {String}	context		the menu context
 * @param {constant}	op			the menu operation
 * @param {constant}	menuType		the type of menu (eg participant)
 * @return	{String}	the id
 */
ZmId.getMenuItemId =
function(context, op, menuType) {
	return DwtId.makeId(ZmId.WIDGET_MENU_ITEM, context, menuType, op);
};

/**
 * Generates the ID for an overview container.
 *
 * @param {String}	overviewContainerId		the overview container ID
 * @return	{String}	the id
 */
ZmId.getOverviewContainerId =
function(overviewContainerId) {
	return DwtId.makeId(ZmId.WIDGET_OVERVIEW_CNTR, overviewContainerId);
};

/**
 * Generates the ID for an overview.
 * 
 * <p>
 * Examples: <code>zov|Mail zov|ZmChooseFolderDialog-ZmListController zov|ZmPickTagDialog</code>
 * </p>
 * 
 * @param {String}	overviewId	the overview ID
 * @return	{String}	the id
 */
ZmId.getOverviewId =
function(overviewId) {
	return DwtId.makeId(ZmId.WIDGET_OVERVIEW, overviewId);
};

/**
 * Generates the ID for a tree within an overview.
 * 
 * <p>
 * Examples: <code>zt|Mail|FOLDER zt|ZmPickTagDialog|TAG</code>
 * </p>
 * 
 * @param {String}	overviewId	the overview ID
 * @param {String}	orgType 		the organizer type (see <code>ZmId.ORG_</code> constants)
 * @return	{String}	the id
 */
ZmId.getTreeId =
function(overviewId, orgType) {
	return DwtId.makeId(ZmId.WIDGET_TREE, overviewId, orgType);
};

/**
 * Generates a tree item ID based on the underlying organizer and the overview ID (since the same
 * organizer may be represented as tree items in more than one overview). Some sample IDs:
 * 
 * <ul>
 * <li><b>zti|Mail|2</b> Inbox</li>
 * <li><b>zti|Mail|172</b>			user-created item in mail overview</li>
 * <li><b>zti|Contacts|7</b>			system Contacts folder</li>
 * <li><b>zti|Calendar|304</b>		user-created item in calendar overview</li>
 * <li><b>ztih|Mail|FOLDER</b>		Folders header in mail overview</li>
 * </ul>
 * 
 * Constants for some system folder tree items have been provided as a convenience.
 * 
 * @param {String}	overviewId	the unique ID for overview
 * @param {ZmOrganizer}	organizerId	the ID of the data object backing tree item
 * @param {constant}	type			the organizer type (for headers only)
 * @return	{String}	the id
 */
ZmId.getTreeItemId =
function(overviewId, organizerId, type) {
	if (!organizerId && !type) { return; }
	if (type) {
		return DwtId.makeId(ZmId.WIDGET_TREE_ITEM_HDR, overviewId, type);
	} else {
		return DwtId.makeId(ZmId.WIDGET_TREE_ITEM, overviewId, organizerId);
	}
};

/**
 * Generates an ID for a view that fills the content area, or for a component of that
 * view. A context should be provided if the view is not a singleton (for example,
 * message view may appear within several double-pane views). The component name
 * is not joined with the "|" character in order to preserve backward compatibility.
 * 
 * <p>
 * Examples: <code>zv|COMPOSE zv|TV zv|TV|MSG zv|TV|MSG_hdrTable</code>
 * </p>
 * 
 * @param {constant}	viewId		the view identifier (see <code>ZmId.VIEW_</code> constants)
 * @param {constant}	component	the component identifier (see <code>ZmId.MV_</code> constants)
 * @param {constant}	context		the ID of owning view
 * @return	{String}	the id
 */
ZmId.getViewId =
function(viewId, component, context) {
	var id = DwtId.makeId(ZmId.WIDGET_VIEW, context, viewId);
	return component ? [id, component].join("") : id;
};

/**
 * Generates an ID for the compose view, or for a component within the compose view. Since
 * only one compose view is created, there is no need for a context to make the ID unique.
 * The component name is not joined with the "|" character for backward compatibility.
 * 
 * <p>
 * Examples: <code>z|ComposeView z|ComposeView_header z|ComposeView_to_row</code>
 * </p>
 * 
 * @param {constant}	component		component identifier (see <code>ZmId.CMP_</code> constants)
 * @return	{String}	the id
 */
ZmId.getComposeViewId =
function(component) {
	var id = DwtId.makeId(ZmId.WIDGET, ZmId.COMPOSE_VIEW);
	return component ? [id, component].join("") : id;
};

/**
 * Generates an ID for a tab (actually the tab button in the tab bar).
 * 
 * <p>
 * Tab contexts and names:
 * 
 * <ul>
 * <li><b>VIEW_PREF</b>			General, Mail, Composing, Signatures, Address Book,
 * 							Accounts, Mail Filters, Calendar, Shortcuts</li>
 * <li><b>VIEW_CONTACT</b>		personal, work, home, other, notes</li>
 * <li><b>VIEW_APPOINTMENT</b>	details, schedule, attendees, locations, equipment</li>
 * <li><b>VIEW_SHORTCUTS</b>		list, {@link ZmId.ORG_FOLDER}, {@link ZmId.ORG_SEARCH}, {@link ZmId.ORG_TAG}</li>
 * </ul>
 * </p>
 * 
 * @param {constant}	context		the owning view identifier (see <code>ZmId.VIEW_</code> constants)
 * @param {String}	tabName		[string]		name of tab
 * @return	{String}	the id
 */
ZmId.getTabId =
function(context, tabName) {
	return DwtId.makeId(ZmId.WIDGET_TAB, context, tabName);
};

/**
 * Generates an ID for a pref page tab.
 *
 * @param	{String}	tabKey		the tab key
 * @return	{String}	the id
 */
ZmId.getPrefPageId = function(tabKey) {
	return "PREF_PAGE_"+tabKey;
};

/*
 * 
 * Gettings IDs for different areas of ZCS
 * 
 */

/*
 * ------------
 * Search Panel
 * ------------
 * 
 * The input box in the search panel has a literal ID. To get the IDs for buttons, menus,
 * and menu items in the search panel, use the functions above.
 * 
 * Buttons:
 * 
 * Pass the context and one of the button constants below:
 * 
 * 		ZmId.getButtonId(ZmId.SEARCH, ZmId.SEARCH_SAVE)
 * 
 * Menus:
 * 
 * There is only one search menu in the panel. Pass the context to get its ID:
 * 
 * 		ZmId.getMenuId(ZmId.SEARCH)
 * 
 * Menu items:
 * 
 * If the search type has a one-to-one mapping with an item type, use the
 * item type constant ZmId.ITEM_* (currently true for contact, appointment, page, and task).
 * Otherwise, pass one of the menu item constants below as the operation:
 * 
 * 		ZmId.getMenuItemId(ZmId.SEARCH, ZmId.ITEM_CONTACT)
 * 		ZmId.getMenuItemId(ZmId.SEARCH, ZmId.SEARCH_SHARED)
 */
 
ZmId.SEARCH_INPUT			= "zi_search";			// text input in search toolbar
ZmId.SEARCH_INPUTFIELD      = ZmId.SEARCH_INPUT + "_inputfield";

// context
ZmId.SEARCH					= "Search";				// element is within search panel

// button, menu item
ZmId.SEARCH_CUSTOM			= "CUSTOM";				// custom search type or button

// button
ZmId.SEARCH_MENU			= "MENU";				// button with dropdown type menu
ZmId.SEARCH_SEARCH			= "SEARCH";				// perform a search
ZmId.SEARCH_SAVE			= "SAVE";				// save a search
ZmId.SEARCH_ADVANCED		= "ADV";				// open/close the search builder
ZmId.SEARCH_LOCAL			= "LOCAL";				// added by the "local" zimlet

// menu item (also see ZmId.ITEM_*)
ZmId.SEARCH_ALL_ACCOUNTS	= "ALL_ACCOUNTS";		// all accounts
ZmId.SEARCH_GAL				= "GAL";				// GAL contacts
ZmId.SEARCH_MAIL			= "MAIL";				// mail items
ZmId.SEARCH_SHARED			= "SHARED";				// include shared items

/*
 * ----------------------
 * Search Results Toolbar
 * ----------------------
 * 
 * This toolbar appears at the top of the search results tab.
 */

ZmId.SEARCHRESULTS_INPUT		= "zi_searchresults";			// text input in search toolbar
ZmId.SEARCHRESULTS_INPUTFIELD	= ZmId.SEARCHRESULTS_INPUT + "_inputfield";

// context
ZmId.SEARCHRESULTS				= "SearchResults";				// element is within search panel

// button
ZmId.SEARCHRESULTS_SEARCH		= "SEARCH";				// perform a search
ZmId.SEARCHRESULTS_SAVE			= "SAVE";				// save a search

/*
 * -----------
 * App toolbar
 * -----------
 * 
 * Also known as the app chooser, the app toolbar contains a button that launches each app.
 * 
 * Buttons:
 * 
 * To get the ID for an app button, pass the app context and an app ID (ZmId.APP_*):
 * 
 * 		ZmId.getButtonId(ZmId.APP, ZmId.APP_MAIL)
 */

// context
ZmId.APP	= "App";

/*
 * ---------
 * Overviews
 * ---------
 * 
 * An overview is a collection of trees. The primary place that the user will see an overview is
 * at the left side of the client. Note that each app has its own overview, since it may want to
 * show a different set of trees. For example, the mail app shows trees for folders, searches, tags,
 * and zimlets by default. Overviews also appear when the user needs to choose something from a tree,
 * for example selecting a folder within a move dialog when moving a message.
 * 
 * A tree is a collection of tree items, each of which may have its own tree items.
 * 
 * The overview IDs for the main overviews that show up at the left are just app names (ZmId.APP_*).
 * The overview IDs elsewhere are more complicated, since they need to be unique for each overview.
 * 
 * Examples: zov|Mail zov|ZmChooseFolderDialog-ZmListController zov|ZmPickTagDialog
 * 
 * Overviews:
 * 
 * 		ZmId.getOverviewId(ZmId.APP_MAIL)
 * 
 * Trees:
 * 
 * 		ZmId.getTreeId(ZmId.APP_MAIL, ZmId.ORG_FOLDER)
 * 
 * Tree items:
 * 
 * 		ZmId.getTreeItemId(ZmId.APP_MAIL, ZmFolder.ID_INBOX)
 * 		ZmId.getTreeItemId(ZmId.APP_MAIL, 2)
 * 		ZmId.TREEITEM_INBOX
 * 
 * TODO: come up with simpler names for other (non-app) overviews
 */

// convenience IDs for system folder tree items
ZmId.TREEITEM_INBOX					= "zti|Mail|2";
ZmId.TREEITEM_JUNK					= "zti|Mail|4";
ZmId.TREEITEM_SENT					= "zti|Mail|5";
ZmId.TREEITEM_DRAFTS				= "zti|Mail|6";
ZmId.TREEITEM_TRASH_MAIL			= "zti|Mail|3";
ZmId.TREEITEM_TRASH_CONTACTS		= "zti|Contacts|3";
ZmId.TREEITEM_CONTACTS				= "zti|Contacts|7";
ZmId.TREEITEM_CALENDAR				= "zti|Calendar|10";
ZmId.TREEITEM_TASKS					= "zti|Tasks|15";
ZmId.TREEITEM_BRIEFCASE				= "zti|Briefcase|16";

/*
 * -----------
 * Top toolbar
 * -----------
 * 
 * To get the ID for the toolbar itself, pass the context (owning view):
 * 
 * 		ZmId.getToolbarId(ZmId.VIEW_TRAD)
 * 
 * Nav toolbar:
 * 
 * 		ZmId.getToolbarId(ZmId.VIEW_TRAD, ZmId.TB_NAV)
 * 
 * Buttons:
 * 
 * 		ZmId.getButtonId(ZmId.VIEW_TRAD, ZmId.OP_CHECK_MAIL)
 * 		ZmId.getButtonId(ZmId.VIEW_TRAD, ZmId.OP_PAGE_FORWARD, ZmId.TB_NAV)
 */

/*
 * -----
 * Views
 * -----
 * 
 * A view is typically a high-level element that occupies the content area. Examples include conversation
 * list view, compose view, and preferences view.
 * 
 * To get the ID for a view, pass the constant for that view:
 * 
 * 		ZmId.getViewId(ZmId.VIEW_CONVLIST)
 */

/*
 * ------------
 * Message view
 * ------------
 * 
 * A message view displays an email message. There are several different instances of message views, which
 * makes it necessary to include a context (owning view) to be able to identify each one of them.
 * 
 * The function to use is:
 * 
 * 		ZmId.getViewId(ZmId.VIEW_MSG, component, context)
 * 
 * Since message views are not singletons, a context is always necessary. Omit the component only when getting
 * an ID for a message view itself.
 * 
 * To get the ID for a message view, pass the constant for the message view as well as the context, which can be
 * ZmId.VIEW_CONVLIST, ZmId.VIEW_CONV, ZmId.VIEW_MSG, or ZmId.VIEW_TRAD:
 * 
 * 		ZmId.getViewId(ZmId.VIEW_MSG, null, ZmId.VIEW_TRAD)
 * 
 * There are also many components within a message view which are useful to retrieve. To get the ID for a
 * message view component, pass the component ID (ZmId.MV_*):
 * 
 * 		ZmId.getViewId(ZmId.VIEW_MSG, ZmId.MV_HDR_TABLE_TOP_ROW, ZmId.VIEW_TRAD)
 * 		ZmId.getViewId(ZmId.VIEW_MSG, ZmId.MV_ATT_LINKS, ZmId.VIEW_TRAD)
 * 
 * 		var bodyId = ZmId.getViewId(ZmId.VIEW_MSG, ZmId.MV_MSG_BODY, ZmId.VIEW_TRAD)
 * 
 * will return the ID for the DIV containing the msg body iframe. To get the ID of the IFRAME element
 * itself, pass that ID as the context for the IFRAME:
 * 
 * 		var iframeId = DwtId.getIframeId(bodyId);
 * 
 * For buttons within msg view, pass the context and operation as usual, and add the identifier for
 * message view (which distinguishes its buttons from, say, those on the VIEW_TRAD toolbar).
 * 
 * 		ZmId.getButtonId(ZmId.VIEW_MSG, ZmId.OP_CLOSE, ZmId.VIEW_CONV)
 * 		ZmId.getButtonId(ZmId.VIEW_MSG, ZmId.OP_EXPAND, ZmId.VIEW_TRAD)
 */

// components that are part of the template
ZmId.MV_HDR_TABLE			= "_hdrTable";			// TABLE that holds header elements
ZmId.MV_HDR_TABLE_TOP_ROW	= "_hdrTableTopRow";	// first TR in header table
ZmId.MV_REPORT_BTN_CELL		= "_reportBtnCell";		// TD that holds Report button (sync failure msg)
ZmId.MV_EXPAND_ROW			= "_expandRow";			// TR that holds expandable headers
ZmId.MV_EXPAND_HDR			= "_expandHeader";		// TD that holds expand button
ZmId.MV_ATT_LINKS			= "_attLinks";			// DIV that holds attachment-related links
ZmId.MV_CONTACT_AREA		= "_contactArea";		// DIV for optional contact actions

// other components
ZmId.MV_HIGHLIGHT_OBJ		= "_highlightObjects";
ZmId.MV_DISPLAY_IMAGES		= "_displayImages";		// DIV with link for showing external images
ZmId.MV_MSG_TRUNC			= "_msgTruncation";		// DIV with link for showing entire msg
ZmId.MV_INFO_BAR			= "_infoBar";			// DIV that is placeholder for optional links above
ZmId.MV_TAG_ROW				= "_tagRow";			// TR for tags
ZmId.MV_TAG_CELL			= "_tagCell";			// TD for tags
ZmId.MV_MSG_BODY			= "_body";				// DIV that contains content iframe
ZmId.MV_MSG_HEADER			= "_header";			// DIV that contains header (conv 2.0 msg capsule view)
ZmId.MV_MSG_FOOTER			= "_footer";			// DIV that contains footer (conv 2.0 msg capsule view)

ZmId.MV_PRIORITY			= "_msgPriority";


/*
 * ------------
 * Compose view
 * ------------
 * 
 * Compose is used to create an email message - a reply, a forward, or a new message.
 * 
 * The function to use is:
 * 
 * 		ZmId.getViewId(ZmId.VIEW_COMPOSE, component)
 * 
 * To get the ID for the compose view:
 * 
 * 		ZmId.getViewId(ZmId.VIEW_COMPOSE)
 * 
 * There are also many components within the compose view which are useful to retrieve. To get the ID for a
 * compose view component, pass the component ID (ZmId.CMP_*):
 * 
 * 		ZmId.getViewId(ZmId.VIEW_COMPOSE, ZmId.CMP_HEADER)
 * 		ZmId.getViewId(ZmId.VIEW_COMPOSE, ZmId.CMP_CC_ROW)
 * 
 * To get the ID of one of the address field buttons, provide the operation:
 * 
 * 		ZmId.getButtonId(ZmId.VIEW_COMPOSE, ZmId.CMP_TO)
 * 
 * To get the ID of the Priority button:
 * 
 * 		ZmId.getButtonId(ZmId.VIEW_COMPOSE, ZmId.CMP_PRIORITY)
 */

// components from the template
ZmId.CMP_HEADER				= "_header";
ZmId.CMP_FROM_SELECT		= "_from_select";
ZmId.CMP_TO_ROW				= "_to_row";
ZmId.CMP_TO_PICKER			= "_to_picker";
ZmId.CMP_TO_INPUT			= "_to_control";
ZmId.CMP_CC_ROW				= "_cc_row";
ZmId.CMP_CC_PICKER			= "_cc_picker";
ZmId.CMP_CC_INPUT			= "_cc_control";
ZmId.CMP_BCC_ROW			= "_bcc_row";
ZmId.CMP_BCC_PICKER			= "_bcc_picker";
ZmId.CMP_BCC_INPUT			= "_bcc_control";
ZmId.CMP_OBO_CHECKBOX		= "_obo_checkbox";
ZmId.CMP_OBO_LABEL			= "_obo_label";
ZmId.CMP_OBO_ROW			= "_obo_row";
ZmId.CMP_OBO_SPAN			= "_obo_span";
ZmId.CMP_BWO_SPAN			= "_bwo_span";
ZmId.CMP_SUBJECT_ROW		= "_subject_row";
ZmId.CMP_SUBJECT_INPUT		= "_subject_control";
ZmId.CMP_IDENTITY_ROW		= "_identity_row";
ZmId.CMP_IDENTITY_SELECT	= "_identity_control";
ZmId.CMP_PRIORITY			= "_priority";
ZmId.CMP_REPLY_ATT_ROW		= "_reply_attachments_link";
ZmId.CMP_ATT_ROW			= "_attachments_row";
ZmId.CMP_ATT_DIV			= "_attachments_div";
ZmId.CMP_ATT_BTN			= "_attachments_btn";
ZmId.CMP_ATT_INP			= "_file_input";
ZmId.CMP_ATT_COMPUTER_INP	= "_file_input_computer";
ZmId.CMP_ATT_INLINE_INP		= "_file_input_inline";
ZmId.CMP_ATT_INCL_ORIG_LINK	= "_show_orig";
ZmId.CMP_DND_TOOLTIP        = "_zdnd_tooltip";

ZmId.CMP_TO_CELL			= "_to_cell";
ZmId.CMP_CC_CELL			= "_cc_cell";
ZmId.CMP_BCC_CELL			= "_bcc_cell";

// compose operations
ZmId.CMP_TO					= "TO";
ZmId.CMP_CC					= "CC";
ZmId.CMP_BCC				= "BCC";

/*
 * 
 * Constants used to generate IDs
 * 
 */

// apps
ZmId.APP_BRIEFCASE		= "Briefcase";
ZmId.APP_CALENDAR		= "Calendar";
ZmId.APP_CONTACTS		= "Contacts";
ZmId.APP_MAIL			= "Mail";
ZmId.APP_PORTAL			= "Portal";
ZmId.APP_PREFERENCES	= "Options";
ZmId.APP_SEARCH			= "Search";
ZmId.APP_SOCIAL			= "Social";
ZmId.APP_TASKS			= "Tasks";
ZmId.APP_VOICE			= "Voice";

// views - often used as context for ID
ZmId.VIEW_ACCOUNT				= "ACCT";
ZmId.VIEW_APPOINTMENT 			= "APPT";
ZmId.VIEW_SIMPLE_ADD_APPOINTMENT= "SAPPT";
ZmId.VIEW_APPOINTMENT_READONLY  = "APPTRO";
ZmId.VIEW_APPT_SCHEDULE			= "APPTS";
ZmId.VIEW_BRIEFCASE			    = "BC";
ZmId.VIEW_BRIEFCASE_DETAIL		= "BCD";
ZmId.VIEW_BRIEFCASE_COLUMN		= "BCC";
ZmId.VIEW_BRIEFCASE_ICON		= "BCI";
ZmId.VIEW_BRIEFCASE_PREVIEW     = "BCP";
ZmId.VIEW_BRIEFCASE_REVISION    = "BRLV";
ZmId.VIEW_BRIEFCASE_DETAIL      = "BDLV";
ZmId.VIEW_CAL					= "CAL";
ZmId.VIEW_CAL_APPT				= "CLA";
ZmId.VIEW_CAL_DAY				= "CLD";
ZmId.VIEW_CAL_LIST				= "CLL";
ZmId.VIEW_CAL_MONTH				= "CLM";
ZmId.VIEW_CAL_WEEK				= "CLW";
ZmId.VIEW_CAL_WORK_WEEK			= "CLWW";
ZmId.VIEW_CAL_FB			    = "CLFB";
ZmId.VIEW_CAL_TRASH             = "CLT";
ZmId.VIEW_SUGGEST_TIME_PANE     = "CSTP";
ZmId.VIEW_SUGGEST_LOCATION_PANE = "CSLP";
ZmId.VIEW_CALL_LIST				= "CLIST";
ZmId.VIEW_COMPOSE				= "COMPOSE";
ZmId.VIEW_CONTACT_SIMPLE 		= "CNS";			// dual panes, list and contact
ZmId.VIEW_CONTACT_SRC			= "CNSRC";			// contact picker
ZmId.VIEW_CONTACT_TGT			= "CNTGT";			// contact picker
ZmId.VIEW_CONTACT				= "CN";
ZmId.VIEW_CONV 					= "CV";				// dual-pane conv view
ZmId.VIEW_CONV2 				= "CV2";			// conv shown in reading pane
ZmId.VIEW_CONVLIST 				= "CLV";			// hybrid conv list view
ZmId.VIEW_FILTER_RULES			= "FRV";
ZmId.VIEW_GROUP					= "GRP";
ZmId.VIEW_LOADING				= "LOADING";		// generic placeholder
ZmId.VIEW_MAIL_CONFIRM			= "MAILCONFIRM";
ZmId.VIEW_MOBILE_DEVICES		= "MD";
ZmId.VIEW_MSG 					= "MSG";
ZmId.VIEW_MSG_CAPSULE			= "MSGC";
ZmId.VIEW_PORTAL                = "PORTAL";
ZmId.VIEW_PREF					= "PREF";
//ZmId.VIEW_QUICK_COMMAND			= "QCV";
ZmId.VIEW_SEARCH_RESULTS		= "SR";
ZmId.VIEW_SHARE_PENDING			= "SVP";
ZmId.VIEW_SHARE_MOUNTED			= "SVM";
ZmId.VIEW_SHARE_GRANTS			= "SVG";
ZmId.VIEW_SHORTCUTS				= "SHORTCUTS";
ZmId.VIEW_TASK					= "TKV";
ZmId.VIEW_TASK_NOT_STARTED		= "TKVN";
ZmId.VIEW_TASK_COMPLETED		= "TKVC";
ZmId.VIEW_TASK_IN_PROGRESS		= "TKVI";
ZmId.VIEW_TASK_WAITING			= "TKVW";
ZmId.VIEW_TASK_DEFERRED 		= "TKVD";
ZmId.VIEW_TASK_ALL				= "TKVA";
ZmId.VIEW_TASK_TODO				= "TKVT";
ZmId.VIEW_TASKEDIT				= "TKE";
ZmId.VIEW_TASKLIST				= "TKL";
ZmId.VIEW_TRAD 					= "TV";
ZmId.VIEW_VOICEMAIL				= "VM";
ZmId.VIEW_ATTACHMENTS           = "AV";

// item types
ZmId.ITEM_APPOINTMENT	= "APPT";
ZmId.ITEM_ATT			= "ATT";
ZmId.ITEM_BRIEFCASE		= "BRIEFCASE_ITEM";
ZmId.ITEM_BRIEFCASE_REV	= "BRIEFCASE_REVISION";
ZmId.ITEM_CALL			= "CALL";
ZmId.ITEM_CONTACT		= "CONTACT";
ZmId.ITEM_CONV			= "CONV";
ZmId.ITEM_DATA_SOURCE	= "DATA_SOURCE";
ZmId.ITEM_DOCUMENT		= "DOCUMENT";
ZmId.ITEM_GAL_CONTACT	= "GAL";
ZmId.ITEM_GROUP			= "GROUP";
ZmId.ITEM_MSG			= "MSG";
ZmId.ITEM_PAGE			= "PAGE";
ZmId.ITEM_RESOURCE		= "RESOURCE";
ZmId.ITEM_TASK			= "TASK";
ZmId.ITEM_VOICEMAIL		= "VOICEMAIL";

// organizer types - generally appear in overview
ZmId.ORG_ADDRBOOK			= "ADDRBOOK";
ZmId.ORG_BRIEFCASE			= "BRIEFCASE";
ZmId.ORG_CALENDAR			= "CALENDAR";
ZmId.ORG_FOLDER				= "FOLDER";
ZmId.ORG_PREF_PAGE			= "PREF_PAGE";
ZmId.ORG_SEARCH				= "SEARCH";				// saved search
ZmId.ORG_TAG				= "TAG";
ZmId.ORG_TASKS				= "TASKS";
ZmId.ORG_ZIMLET				= "ZIMLET";

// fields of an item - generally equates to a column in a list view
ZmId.FLD_ACCOUNT		= "ac";
ZmId.FLD_ATTACHMENT		= "at";
ZmId.FLD_CAPACITY		= "cp";
ZmId.FLD_COMPANY		= "co";
ZmId.FLD_DATE			= "dt";
ZmId.FLD_DEPARTMENT		= "de";
ZmId.FLD_EMAIL			= "em";
ZmId.FLD_EXPAND			= "ex";	// CLV
ZmId.FLD_FILE_TYPE		= "ft";
ZmId.FLD_FLAG			= "fg";
ZmId.FLD_FOLDER			= "fo";
ZmId.FLD_FRAGMENT		= "fm";
ZmId.FLD_FROM			= "fr";
ZmId.FLD_HOME_PHONE		= "hp"; // Contacts
ZmId.FLD_ID				= "id";
ZmId.FLD_INDEX			= "ix";
ZmId.FLD_ITEM_ROW		= "rw";
ZmId.FLD_ITEM_ROW_3PANE	= "r3";
ZmId.FLD_LOCATION		= "lo";
ZmId.FLD_LOCK           = "loid";
ZmId.FLD_SHARES			= "shares";
ZmId.FLD_MSG_PRIORITY   = "mp"; //message prioritization
ZmId.FLD_NAME			= "na";
ZmId.FLD_NOTES			= "no";
ZmId.FLD_PARTICIPANT	= "pa";
ZmId.FLD_PCOMPLETE		= "pc"; // Tasks
ZmId.FLD_PRIORITY		= "pr"; // Tasks
ZmId.FLD_RECURRENCE		= "re";	// Calendar
ZmId.FLD_SELECTION		= "se";
ZmId.FLD_SELECTION_CELL	= "sec";
ZmId.FLD_SIZE			= "sz";
ZmId.FLD_SORTED_BY		= "sb";
ZmId.FLD_STATUS			= "st";
ZmId.FLD_READ			= "rd";
ZmId.FLD_MUTE			= "mt";
ZmId.FLD_SUBJECT		= "su";
ZmId.FLD_TAG			= "tg";
ZmId.FLD_TAG_CELL		= "tc";
ZmId.FLD_TYPE			= "ty";
ZmId.FLD_TO             = "to";
ZmId.FLD_VERSION        = "ver";
ZmId.FLD_WORK_PHONE		= "wp"; // Contacts
ZmId.FLD_CREATED        = "cr";   // Application passcode created
ZmId.FLD_LAST_USED      = "lu";   // Application passcode last used

// operations - things the user can do, usually via a button or menu item
ZmId.OP_ACCEPT_PROPOSAL         = "ACCEPT_PROPOSAL";
ZmId.OP_ADD       		     	= "ADD";
ZmId.OP_ADD_FILTER_RULE			= "ADD_FILTER_RULE";
ZmId.OP_ADD_TO_FILTER_RULE		= "ADD_TO_FILTER_RULE";
//ZmId.OP_ADD_QUICK_COMMAND		= "ADD_QUICK_COMMAND";
ZmId.OP_ADD_SIGNATURE			= "ADD_SIGNATURE";
ZmId.OP_ADD_EXTERNAL_CALENDAR	= "ADD_EXTERNAL_CALENDAR";
ZmId.OP_ATTACHMENT				= "ATTACHMENT";
ZmId.OP_ACTIONS_MENU			= "ACTIONS_MENU";
ZmId.OP_BROWSE					= "BROWSE";
ZmId.OP_BROWSE_FOLDER			= "BROWSE_FOLDER";
ZmId.OP_CALL					= "CALL";
ZmId.OP_CAL_REFRESH				= "CAL_REFRESH";
ZmId.OP_CAL_REPLY				= "CAL_REPLY";
ZmId.OP_CAL_REPLY_ALL			= "CAL_REPLY_ALL";
ZmId.OP_CAL_LIST_VIEW			= "CAL_LIST_VIEW";
ZmId.OP_CAL_VIEW_MENU			= "CAL_VIEW_MENU";
ZmId.OP_CANCEL					= "CANCEL";
ZmId.OP_CHECKIN                 = "CHECKIN";
ZmId.OP_CHECKOUT                = "CHECKOUT";
ZmId.OP_CHECK_ALL				= "CHECK_ALL";
ZmId.OP_CHECK_MAIL				= "CHECK_MAIL";
ZmId.OP_GO_OFFLINE				= "GOOFFLINE";
ZmId.OP_CALL_BACK				= "CALL_BACK";
ZmId.OP_CLEAR_ALL				= "CLEAR_ALL";
ZmId.OP_CLOSE					= "CLOSE";
ZmId.OP_COMPOSE_FORMAT			= "COMPOSE_FORMAT";
ZmId.OP_COMPOSE_OPTIONS			= "COMPOSE_OPTIONS";
ZmId.OP_CONTACT					= "CONTACT";
ZmId.OP_CONTACTGROUP_MENU       = "CONTACTGROUP_MENU";
ZmId.OP_COPY		     		= "COPY";
ZmId.OP_CREATE_APPT     		= "CREATE_APPT";
ZmId.OP_CREATE_TASK     		= "CREATE_TASK";
ZmId.OP_DAY_VIEW				= "DAY_VIEW";
ZmId.OP_DECLINE_PROPOSAL        = "DECLINE_PROPOSAL";
ZmId.OP_DELETE					= "DELETE";
ZmId.OP_DELETE_WITHOUT_SHORTCUT		= "DELETE_WITHOUT_SHORTCUT";
ZmId.OP_DELETE_APPT_INSTANCE	= "DELETE_INSTANCE";
ZmId.OP_DELETE_APPT_SERIES  	= "DELETE_SERIES";
ZmId.OP_DELETE_CONV				= "DELETE_CONV";
ZmId.OP_DELETE_MENU				= "DELETE_MENU";
ZmId.OP_DELETE_MSG				= "DELETE_MSG";
ZmId.OP_DELETE_VERSION          = "DELETE_VERSION";
ZmId.OP_DETACH					= "DETACH";
ZmId.OP_DETACH_WIN				= "DETACH_WIN";
ZmId.OP_DETACH_COMPOSE			= "DETACH_COMPOSE";
ZmId.OP_DISCARD_CHECKOUT        = "DISCARD_CHECKOUT";
ZmId.OP_DOWNLOAD_VOICEMAIL		= "DOWNLOAD_VOICEMAIL";
ZmId.OP_NEW_CALL				= "NEW_CALL";
ZmId.OP_DUPLICATE_APPT  		= "DUPLICATE_APPT";
ZmId.OP_DRAFT					= "DRAFT";
ZmId.OP_EDIT					= "EDIT";
ZmId.OP_EDIT_AS_NEW				= "EDIT_AS_NEW";
ZmId.OP_EDIT_CONTACT			= "EDIT_CONTACT";
ZmId.OP_EDIT_FILE				= "EDIT_FILE";
ZmId.OP_EDIT_FILTER_RULE		= "EDIT_FILTER_RULE";
//ZmId.OP_EDIT_QUICK_COMMAND		= "EDIT_QUICK_COMMAND";
ZmId.OP_EDIT_PROPS				= "EDIT_PROPS";
ZmId.OP_EDIT_REPLY_ACCEPT		= "EDIT_REPLY_ACCEPT";
ZmId.OP_EDIT_REPLY_CANCEL		= "EDIT_REPLY_CANCEL";
ZmId.OP_EDIT_REPLY_DECLINE		= "EDIT_REPLY_DECLINE";
ZmId.OP_EDIT_REPLY_TENTATIVE	= "EDIT_REPLY_TENTATIVE";
ZmId.OP_EMPTY_FOLDER			= "EMPTY_FOLDER";
ZmId.OP_EXPAND					= "EXPAND";
ZmId.OP_EXPAND_ALL				= "EXPAND_ALL";
//ZmId.OP_EXPORT_FOLDER			= "EXPORT_FOLDER";
ZmId.OP_FB_VIEW				    = "FB_VIEW";
ZmId.OP_FLAG					= "FLAG";
ZmId.OP_UNFLAG					= "UNFLAG";
ZmId.OP_FIND_SHARES				= "FIND_SHARES";
ZmId.OP_FORMAT_HTML				= "FORMAT_HTML";
ZmId.OP_FORMAT_HTML_SOURCE		= "FORMAT_HTML_SOURCE";
ZmId.OP_FORMAT_MEDIA_WIKI		= "FORMAT_MEDIA_WIKI";
ZmId.OP_FORMAT_RICH_TEXT		= "FORMAT_RICH_TEXT";
ZmId.OP_FORMAT_TEXT				= "FORMAT_TEXT";
ZmId.OP_FORMAT_TWIKI			= "FORMAT_TWIKI";
ZmId.OP_FORMAT_MORE_OPTIONS		= "FORMAT_MORE_OPTIONS";
ZmId.OP_FORWARD					= "FORWARD";
ZmId.OP_FORWARD_ATT				= "FORWARD_ATT";
ZmId.OP_FORWARD_BY_EMAIL		= "FORWARD_BY_EMAIL";
ZmId.OP_FORWARD_CONV		    = "FORWARD_CONV";
ZmId.OP_FORWARD_INLINE			= "FORWARD_INLINE";
ZmId.OP_FORWARD_MENU			= "FORWARD_MENU";
ZmId.OP_FORWARD_APPT			= "FORWARD_APPT";
ZmId.OP_FORWARD_APPT_INSTANCE	= "FORWARD_APPT_INSTANCE";
ZmId.OP_FORWARD_APPT_SERIES		= "FORWARD_APPT_SERIES";
ZmId.OP_FREE_BUSY_LINK			= "FREE_BUSY_LINK";
ZmId.OP_GROUPBY                 = "GROUPBY";
ZmId.OP_GROUPBY_DATE            = "GROUPBY_DATE";
ZmId.OP_GROUPBY_NONE            = "GROUPBY_NONE";
ZmId.OP_GROUPBY_FROM            = "GROUPBY_FROM";
ZmId.OP_GROUPBY_PRIORITY        = "GROUPBY_PRIORITY";
ZmId.OP_GROUPBY_SIZE            = "GROUPBY_SIZE";
ZmId.OP_GROUPBY_TAG             = "GROUPBY_TAG";
ZmId.OP_GO_TO_URL				= "GO_TO_URL";
ZmId.OP_IMPORT_FILE				= "IMPORT_FILE";
//ZmId.OP_IMPORT_FOLDER			= "IMPORT_FOLDER";
ZmId.OP_INC_ATTACHMENT			= "INC_ATTACHMENT";
ZmId.OP_INC_BODY				= "INC_BODY";
ZmId.OP_INC_NONE				= "INC_NONE";
ZmId.OP_INC_SMART				= "INC_SMART";
ZmId.OP_INCLUDE_HEADERS			= "INCLUDE_HEADERS";
ZmId.OP_INVITE_ATTENDEES		= "INVITE_ATTENDEES";
ZmId.OP_INVITE_REPLY_ACCEPT		= "INVITE_REPLY_ACCEPT";
ZmId.OP_INVITE_REPLY_DECLINE	= "INVITE_REPLY_DECLINE";
ZmId.OP_INVITE_REPLY_MENU		= "INVITE_REPLY_MENU";
ZmId.OP_INVITE_REPLY_TENTATIVE	= "INVITE_REPLY_TENTATIVE";
ZmId.OP_KEEP_READING			= "KEEP_READING";
ZmId.OP_MARK_ALL_READ			= "MARK_ALL_READ";
ZmId.OP_MARK_HEARD				= "MARK_HEARD";
ZmId.OP_MARK_READ				= "MARK_READ";
ZmId.OP_MARK_UNHEARD			= "MARK_UNHEARD";
ZmId.OP_MARK_UNREAD				= "MARK_UNREAD";
ZmId.OP_MARK_AS_COMPLETED		= "MARK_AS_COMPLETED";
ZmId.OP_MOBILE_REMOVE			= "MOBILE_REMOVE";
ZmId.OP_MOBILE_CANCEL_WIPE		= "MOBILE_CANCEL_WIPE";
ZmId.OP_MOBILE_RESUME_SYNC		= "MOBILE_RESUME_SYNC";
ZmId.OP_MOBILE_SUSPEND_SYNC		= "MOBILE_SUSPEND_SYNC";
ZmId.OP_MOBILE_WIPE				= "MOBILE_WIPE";
ZmId.OP_MONTH_VIEW				= "MONTH_VIEW";
ZmId.OP_MOUNT_ADDRBOOK			= "MOUNT_ADDRBOOK";
ZmId.OP_MOUNT_BRIEFCASE			= "MOUNT_BRIEFCASE";
ZmId.OP_MOUNT_CALENDAR			= "MOUNT_CALENDAR";
ZmId.OP_MOUNT_FOLDER			= "MOUNT_FOLDER";
ZmId.OP_MOUNT_TASK_FOLDER		= "MOUNT_TASK_FOLDER";
ZmId.OP_MOVE					= "MOVE";
ZmId.OP_MOVE_MENU				= "MOVE_MENU";
ZmId.OP_MOVE_DOWN_FILTER_RULE	= "MOVE_DOWN_FILTER_RULE";
ZmId.OP_MOVE_TO_BCC				= "MOVE_TO_BCC";
ZmId.OP_MOVE_TO_CC				= "MOVE_TO_CC";
ZmId.OP_MOVE_TO_TO				= "MOVE_TO_TO";
ZmId.OP_MOVE_UP_FILTER_RULE		= "MOVE_UP_FILTER_RULE";
ZmId.OP_MUTE_CONV		        = "MUTE_CONV";
ZmId.OP_NEW_ADDRBOOK			= "NEW_ADDRBOOK";
ZmId.OP_NEW_ALLDAY_APPT			= "NEW_ALLDAY_APPT";
ZmId.OP_NEW_APPT				= "NEW_APPT";
ZmId.OP_NEW_BRIEFCASE			= "NEW_BRIEFCASE";
ZmId.OP_NEW_CALENDAR			= "NEW_CALENDAR";
ZmId.OP_NEW_CONTACT				= "NEW_CONTACT";
ZmId.OP_NEW_DISTRIBUTION_LIST	= "NEW_DISTRIBUTION_LIST";
ZmId.OP_NEW_DOC                 = "NEW_DOC";
ZmId.OP_NEW_FILE				= "NEW_FILE";
ZmId.OP_NEW_FOLDER				= "NEW_FOLDER";
ZmId.OP_NEW_GROUP				= "NEW_GROUP";
ZmId.OP_NEW_MENU				= "NEW_MENU";
ZmId.OP_NEW_MESSAGE				= "NEW_MESSAGE";
ZmId.OP_NEW_MESSAGE_WIN			= "NEW_MESSAGE_WIN";
ZmId.OP_NEW_BRIEFCASE_WIN		= "NEW_BRIEFCASE_WIN";
ZmId.OP_NEW_PAGE				= "NEW_PAGE";
ZmId.OP_NEW_TAG					= "NEW_TAG";
ZmId.OP_NEW_TASK				= "NEW_TASK";
ZmId.OP_NOTIFY                  = "NOTIFY";
ZmId.OP_NEW_TASK_FOLDER			= "NEW_TASK_FOLDER";
ZmId.OP_OPEN_APPT_INSTANCE		= "OPEN_APPT_INSTANCE";
ZmId.OP_OPEN_APPT_SERIES		= "OPEN_APPT_SERIES";
ZmId.OP_OPEN_FILE				= "OPEN_FILE";
ZmId.OP_OPEN_IN_TAB				= "OPEN_IN_TAB";
ZmId.OP_PAGE_BACK				= "PAGE_BACK";
ZmId.OP_PAGE_FORWARD			= "PAGE_FORWARD";
ZmId.OP_PAUSE_TOGGLE			= "PAUSE_TOGGLE";
ZmId.OP_PRINT					= "PRINT";
ZmId.OP_PRINT_ADDRBOOK			= "PRINT_ADDRBOOK";
ZmId.OP_PRINT_CALENDAR			= "PRINT_CALENDAR";
ZmId.OP_PRINT_CONTACT			= "PRINT_CONTACT";
ZmId.OP_PRIORITY_FILTER         = "PRIORITY_FILTER";
ZmId.OP_PRIORITY_HIGH           = "PRIORITY_HIGH";
ZmId.OP_PRIORITY_LOW            = "PRIORITY_LOW";
ZmId.OP_PRIORITY_NORMAL         = "PRIORITY_NORMAL";
ZmId.OP_PROPOSE_NEW_TIME        = "PROPOSE_NEW_TIME";
ZmId.OP_OPTS         			= "OPTIONS";
//ZmId.OP_QUICK_COMMANDS  	    = "QUICK_COMMANDS";
ZmId.OP_RECOVER_DELETED_ITEMS	= "RECOVER_DELETED_ITEMS";
ZmId.OP_REDIRECT				= "REDIRECT";
ZmId.OP_REFRESH					= "REFRESH";
ZmId.OP_REINVITE_ATTENDEES      = "REINVITE_ATTENDEES";
ZmId.OP_REMOVE_FILTER_RULE		= "REMOVE_FILTER_RULE";
//ZmId.OP_REMOVE_QUICK_COMMAND	= "REMOVE_QUICK_COMMAND";
ZmId.OP_RENAME_FILE             = "RENAME_FILE";
ZmId.OP_RENAME_FOLDER			= "RENAME_FOLDER";
ZmId.OP_RENAME_SEARCH			= "RENAME_SEARCH";
ZmId.OP_RENAME_TAG				= "RENAME_TAG";
ZmId.OP_REPLY					= "REPLY";
ZmId.OP_REPLY_ACCEPT			= "REPLY_ACCEPT";
ZmId.OP_REPLY_ACCEPT_IGNORE		= "REPLY_ACCEPT_IGNORE";
ZmId.OP_REPLY_ACCEPT_NOTIFY		= "REPLY_ACCEPT_NOTIFY";
ZmId.OP_REPLY_ALL				= "REPLY_ALL";
ZmId.OP_REPLY_BY_EMAIL			= "REPLY_BY_EMAIL";
ZmId.OP_REPLY_CANCEL			= "REPLY_CANCEL";
ZmId.OP_REPLY_CANCEL			= "REPLY_CANCEL";
ZmId.OP_REPLY_DECLINE			= "REPLY_DECLINE";
ZmId.OP_REPLY_DECLINE_IGNORE	= "REPLY_DECLINE_IGNORE";
ZmId.OP_REPLY_DECLINE_NOTIFY	= "REPLY_DECLINE_NOTIFY";
ZmId.OP_REPLY_MENU				= "REPLY_MENU";
ZmId.OP_REPLY_MODIFY			= "REPLY_MODIFY";
ZmId.OP_REPLY_MODIFY			= "REPLY_MODIFY";
ZmId.OP_REPLY_NEW_TIME			= "REPLY_NEW_TIME";
ZmId.OP_REPLY_NEW_TIME			= "REPLY_NEW_TIME";
ZmId.OP_REPLY_TENTATIVE			= "REPLY_TENTATIVE";
ZmId.OP_REPLY_TENTATIVE_IGNORE	= "REPLY_TENTATIVE_IGNORE";
ZmId.OP_REPLY_TENTATIVE_NOTIFY	= "REPLY_TENTATIVE_NOTIFY";
ZmId.OP_REPORT					= "REPORT";
ZmId.OP_REQUEST_READ_RECEIPT	= "REQUEST_READ_RECEIPT";
ZmId.OP_RESET                   = "RESET";
ZmId.OP_RESTORE_VERSION         = "RESTORE_VERSION";
ZmId.OP_REVERT_PAGE				= "REVERT_PAGE";
ZmId.OP_RUN_FILTER_RULE			= "RUN_FILTER_RULE";
ZmId.OP_SAVE					= "SAVE";
ZmId.OP_SAVE_DRAFT				= "SAVE_DRAFT";
ZmId.OP_SAVE_FILE				= "SAVE_FILE";
ZmId.OP_SEARCH					= "SEARCH";
ZmId.OP_SEARCH_MAIL				= "SEARCH_MAIL";
ZmId.OP_SEARCH_MENU             = "SEARCH_MENU";
ZmId.OP_SEARCH_TO               = "SEARCH_TO";
ZmId.OP_SEND					= "SEND";
ZmId.OP_SEND_FILE				= "SEND_FILE";
ZmId.OP_SEND_FILE_AS_ATT	    = "SEND_FILE_AS_ATT";
ZmId.OP_SEND_FILE_MENU          = "SEND_FILE_MENU";
ZmId.OP_SEND_MENU				= "SEND_MENU";
ZmId.OP_SEND_LATER				= "SEND_LATER";
ZmId.OP_SEND_PAGE				= "SEND_PAGE";
ZmId.OP_SEND_INVITE				= "SEND_INVITE";
ZmId.OP_SEND_FB_HTML			= "SEND_FB_HTML";
ZmId.OP_SEND_FB_ICS			    = "SEND_FB_ICS";
ZmId.OP_SEND_FB_ICS_EVENT	    = "SEND_FB_ICS_EVENT";
ZmId.OP_SHARE					= "SHARE";
ZmId.OP_SHARE_ACCEPT			= "SHARE_ACCEPT";
ZmId.OP_SHARE_ADDRBOOK			= "SHARE_ADDRBOOK";
ZmId.OP_SHARE_BRIEFCASE			= "SHARE_BRIEFCASE";
ZmId.OP_SHARE_CALENDAR			= "SHARE_CALENDAR";
ZmId.OP_SHARE_DECLINE			= "SHARE_DECLINE";
ZmId.OP_SHARE_FOLDER			= "SHARE_FOLDER";
ZmId.OP_SHARE_TASKFOLDER		= "SHARE_TASKFOLDER";
ZmId.OP_SHOW_ALL_ITEM_TYPES		= "SHOW_ALL_ITEM_TYPES";
ZmId.OP_SHOW_BCC				= "SHOW_BCC";
ZmId.OP_SHOW_CONV				= "SHOW_CONV";
ZmId.OP_SHOW_ONLY_MAIL			= "SHOW_ONLY_MAIL";
ZmId.OP_SHOW_ORIG				= "SHOW_ORIG";
ZmId.OP_SORT_ASC                = "SORT_ASC";
ZmId.OP_SORT_DESC               = "SORT_DESC";
ZmId.OP_SPAM					= "SPAM";
ZmId.OP_SPELL_CHECK				= "SPELL_CHECK";
ZmId.OP_SUBSCRIBE_APPROVE		= "SUBSCRIBE_APPROVE";
ZmId.OP_SUBSCRIBE_REJECT		= "SUBSCRIBE_REJECT";
ZmId.OP_SYNC					= "SYNC";
ZmId.OP_SYNC_ALL				= "SYNC_ALL";
ZmId.OP_SYNC_OFFLINE_FOLDER		= "SYNC_OFFLINE_FOLDER";
ZmId.OP_TAG						= "TAG";
ZmId.OP_TAG_COLOR_MENU			= "TAG_COLOR_MENU";
ZmId.OP_TAG_MENU				= "TAG_MENU";
ZmId.OP_PRINT_TASK		    	= "PRINT_TASK";
ZmId.OP_PRINT_TASKFOLDER		= "PRINT_TASKFOLDER";
ZmId.OP_TEXT					= "TEXT";
ZmId.OP_TODAY					= "TODAY";
ZmId.OP_UNDELETE				= "UNDELETE";
ZmId.OP_UNMUTE_CONV		        = "UNMUTE_CONV";
ZmId.OP_USE_PREFIX				= "USE_PREFIX";
ZmId.OP_VERSION_HISTORY         = "VERSION_HISTORY";
ZmId.OP_VIEW					= "VIEW";
ZmId.OP_VIEW_APPOINTMENT		= "VIEW_APPOINTMENT";
ZmId.OP_VIEW_APPT_INSTANCE		= "VIEW_APPT_INSTANCE";
ZmId.OP_VIEW_APPT_SERIES		= "VIEW_APPT_SERIES";
ZmId.OP_VIEW_BY_DATE			= "VIEW_BY_DATE";
ZmId.OP_VIEW_FILE_AS_HTML		= "VIEW_FILE_AS_HTML";
ZmId.OP_VIEW_MENU				= "VIEW_MENU";
ZmId.OP_SORTBY_MENU			    = "SORTBY_MENU";
ZmId.OP_WEEK_VIEW				= "WEEK_VIEW";
ZmId.OP_WORK_WEEK_VIEW			= "WORK_WEEK_VIEW";
ZmId.OP_ZIMLET					= "ZIMLET";

//Group By IDs
ZmId.GROUPBY_DATE               = "GROUPBY_DATE";
ZmId.GROUPBY_FROM               = "GROUPBY_FROM";
ZmId.GROUPBY_NONE               = "GROUPBY_NONE";
ZmId.GROUPBY_PRIORITY           = "GROUPBY_PRIORITY";
ZmId.GROUPBY_SIZE               = "GROUPBY_SIZE";
ZmId.GROUPBY_TAG                = "GROUPBY_TAG";


/*
 * Experimental ID code below. The main idea is to make easier for a third party (such as QA) to find what they're
 * looking for. A secondary goal is to ensure that we always use unique IDs. The systematic approach above is prone
 * to failure in that regard, since the same set of inputs will produce the same ID.
 *
 * The new approach is to introduce a level of indirection between the fields and the ID. IDs will go back to being
 * unique and opaque, based on an incrementing number. A hash will be maintained which maps a collection of fields
 * to the actual ID used in the DOM.
 *
 * The core of the new system is related closely to the old system: a set of params which, taken together, should
 * uniquely identify an element. The possible values for each param are typically constants defined in this class.
 *
 * To make it easy for clients of this ID system to successfully look up IDs, creators of IDs should provide as many
 * parameters as possible. For example, providing both skinComponent and componentType may be redundant, but then
 * the ID can be looked up using either parameter.
 *
 * The parameters and their values:
 *
 * skinComponent
 *
 *      The HTML skin for ZCS defines a number of components and provides containers for them. This param identifies
 *      which skin component contains the element. Note that the IDs for the skin containers themselves (as well as
 *      a few elements within those containers) are defined by the skin (in skin.html), and are not part of this set
 *      of IDs. May often be omitted when looking up an ID.
 *
 *      ZmId.SKIN_*
 *
 * componentType
 *
 *      The general category of view component that maps to the element. It may be a type of widget such as a button
 *      or a menu, something more general like a view, or even a subcomponent like a list view header.
 *
 *      DwtId.WIDGET_*
 *      ZmId.WIDGET_*
 *
 * componentName
 *
 *      The component name identifies the component among components of a similar type. It may be the name of an
 *      operation. For example, a search button would have the type "button" and the name "search". (There may be
 *      more than one search button, so other params may be necessary to uniquely identify each one.)
 *
 *      ZmId.VIEW_*
 *      ZmId.OP_*
 *      ZmId.TB_*
 *      ZmId.SEARCH_*
 *      ZmId.MV_*
 *      ZmId.CMP_*
 *      ZmId.GROUPBY_*
 *
 * app
 *
 *      The name of the application that contains the element. Some elements are global and do not have an associated
 *      application. Elements associated with an app appear within the main content area.
 *
 *      ZmId.APP_*
 *
 * containingView
 *
 *      For an element within the main content area, the identifier of the view that contains it.
 *
 *      ZmId.VIEW_*
 *
 * sessionId
 *
 *      A number identifying a session for a view which can appear in more than one tab at a time. For example, there
 *      may be multiple compose sessions if the user replies to several different messages without sending the replies.
 *
 * itemType
 *
 *      The type of item, such as message, contact, or appointment.
 *
 *      ZmId.ITEM_*
 *
 * itemId
 *
 *      The ID of the item (for example, a mail message) that the element is tied to. For local items, it's a number.
 *      For shared items, it's a compound string comprising an account ID and a local numeric ID.
 *
 * organizerType
 *
 *      The type of organizer, such as folder, tag, or zimlet. Organizers generally appear in the overview.
 *
 *      ZmId.ORG_*
 *
 * organizerId
 *
 *      The ID of the organizer (for example, a mail folder) that the element is tied to. For local organizers, it's a
 *      number. For shared folders, it's a compound string comprising an account ID and a local numeric ID.
 *
 * field
 *
 *      A field identifies a specific sub-part of a component. It might be something that helps make up a widget,
 *      such as the "left icon" in a button, or it might be something ZCS-specific like the "subject" field in a list
 *      view that displays mail messages. The line between componentName and field can be a bit blurry. Generally a
 *      componentName refers to a container of some sort, like a list row or header.
 *
 *      ZmId.FLD_*
 *
 * tagName
 *
 *      The tag name of the HTML element, such as "TABLE" or "TR". May usually be omitted when looking up an ID.
 *
 * sequence
 *
 *      A number used to diffentiate between otherwise identical IDs.
 *
 * parentId
 *
 *      The ID of the parent of this element.
 *
 */

ZmId.BASE = "zcs";
ZmId.SEQ = 1;

ZmId._idList = [];

ZmId._idHash = {};

ZmId._valueToParam = {};

/**
 * Returns a unique ID that can later be looked up. As many params as possible should be provided, in order to
 * make lookup easier. If one or more IDs is found to already have been created with the given set of params,
 * a sequence number is added as a parameter.
 *
 * @param {hash}        params          set of fields describing the ID's element
 * @param {string}      description     (optional) a brief description of the purpose of the ID
 */
ZmId.create = function(params, description) {

	var idParams = AjxUtil.hashCopy(params);
	var ids = ZmId.lookup(params);
	if (ids) {
		idParams.sequence = (typeof ids === "string") ? 1 : ids.length;
	}
	idParams.description = description || "";
	var newId = ZmId.BASE + ZmId.SEQ++;
	idParams.id = newId;
	ZmId._idHash[newId] = idParams;
	ZmId._idList.push(idParams);

	for (var key in params) {
		ZmId._valueToParam[params[key]] = key;
	}

	return newId;
};

/**
 * Returns the DOM ID that matches the given set of params. If more than one ID matches, a list is returned.
 * A partial set of params may be provided. The more params provided, the better the chance of finding just one ID.
 * The best approach is to provide the minimal set of params that will uniquely differentiate the element. If no
 * params are provided, returns all IDs.
 *
 * Optionally, a list of values can be given. An attempt will be made to reverse-engineer the params by figuring
 * out the appropriate key for each value. This method will never be as reliable as providing a hash in the first place.
 *
 * @param {hash|array}  params    set of fields describing the ID(s) being sought
 */
ZmId.lookup = function(params) {

	if (!params) {
		return ZmId._idList;
	}

	if (AjxUtil.isArray(params)) {
		params = ZmId._convertValues(params);
	}

	var ids = [];
	for (var i = 0, len = ZmId._idList.length; i < len; i++) {
		var idParams = ZmId._idList[i];
		var add = true;
		for (var param in params) {
			if (idParams[param] && params[param] !== idParams[param]) {
				add = false;
				continue;
			}
		}
		if (add) {
			ids.push(idParams.id);
		}
	}
	return (ids.length === 0) ? null : (ids.length === 1) ? ids[0] : ids;
};

/**
 * Returns the set of params used to create the given ID.
 *
 * @param id
 */
ZmId.getParams = function(id) {
	return ZmId._idHash[id];
};

/**
 * Displays a list of matching IDs in a popup, with the params used to create them and their descriptions.
 * Intended as a development tool.
 *
 * @param params    set of fields describing the ID(s) being sought
 */
ZmId.showIds = function(params) {

	if (!DBG || DBG.isDisabled()) { return; }

	var ids = ZmId.lookup(params),
		len = ids.length,
		text = "",
		i;

	for (i = 0; i < len; i++) {
		var id = ids[i].id;
		var params = ZmId._idHash[id];
		text += "\n-----\n\n" + id + AjxStringUtil.repeat(" ", 16 - id.length) + params.description + "\n\n";
		var paramNames = AjxUtil.keys(params).sort();
		for (var j = 0; j < paramNames.length; j++) {
			var paramName = paramNames[j];
			if (paramName === 'id' || paramName === 'description') {
				continue;
			}
			var value = params[paramName];
			if (!value) {
				continue;
			}
			value = ZmId._backMap[value] ? "ZmId." + ZmId._backMap[value] : value;
			text += paramName + AjxStringUtil.repeat(" ", 16 - paramName.length) + value + "\n";
		}
	}

	DBG.printRaw(text);
};

ZmId._backMap = AjxUtil.valueHash(ZmId, function(k) {
	return typeof ZmId[k] === 'string';
});

// Create a static hash so we know if a string is a view type (eg "CLV")
ZmId._isViewType = AjxUtil.arrayAsHash(AjxUtil.values(ZmId, function(k) {
	return typeof ZmId[k] === "string" && k.indexOf("VIEW_") === 0;
}));

// Convert a list of values of ID parameters back into a hash by figuring out the matching key for each value.
// View names (such as "CLV") are a bit tricky since they can be either a componentName (for a view widget), or
// a containingView. A small number might be an organizer ID (eg Inbox is 2), or a session ID.
ZmId._convertValues = function(values) {

	var params = {},
		viewValue, numValue;

	for (var i = 0; i < values.length; i++) {
		var value = values[i];
		if (ZmId._isViewType[value]) {
			viewValue = value;
		}
		else if (AjxUtil.isNumber(value) || AjxUtil.isNumeric(value)) {
			var num = parseInt(value);
			if (num < 10) {
				numValue = num;
			}
		}
		else {
			var param = ZmId._valueToParam[value];
			params[param] = value;
		}
	}

	// A view value is a componentName only if the componentType is a view.
	if (viewValue) {
		var viewParam = (params.componentType === ZmId.WIDGET_VIEW) ? "componentName" : "containingView";
		params[viewParam] = viewValue;
	}

	// A single-digit number is probably an organizer ID or a session ID.
	if (numValue) {
		var viewParam = params.organizerType ? "organizerId" : "sessionId";
		params[viewParam] = viewValue;
	}

	return params;
};
}
if (AjxPackage.define("zimbraMail.share.model.events.ZmEvent")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines an event.
 *
 */

/**
 * Creates an empty event of the given type.
 * @class
 * This class represents an event that encapsulates some sort of change to a model (data).
 * The event has a data type (for example, conversation), an event type (for example, delete), a source (the
 * data object generating the event), and a hash of arbitrary information (details).
 * 
 * @param {constant}		type	the source of the event
 */
ZmEvent = function(type) {

	this.type = type;	// source type (conv, msg, contact, folder, etc)
	this.event = null;	// event type (create, modify, etc)
	this.source = null;	// notifying model (often a list)
	this.item = null;	// item that is subject of the notification
	this._details = {};
}

// Listener types
ZmEvent.L_MODIFY = 1;
ZmEvent.L_PICKER = 2;

// Source types (note: there are not separate types for list models)
ZmEvent.S_TAG			= "TAG";
ZmEvent.S_PICKER		= "PICKER";
ZmEvent.S_SEARCH		= "SEARCH";
ZmEvent.S_SETTING		= "SETTING";
ZmEvent.S_SETTINGS		= "SETTINGS";
ZmEvent.S_SHARE         = "SHARE";
ZmEvent.S_MOUNTPOINT	= "MOUNTPOINT";
ZmEvent.S_ZIMLET		= "ZIMLET";
ZmEvent.S_HAB			= "HAB";

// Event types
/**
 * Defines the "create" event type.
 */
ZmEvent.E_CREATE		= "CREATE";
/**
 * Defines the "delete" event type.
 */
ZmEvent.E_DELETE		= "DELETE";
/**
 * Defines the "modify" event type.
 */
ZmEvent.E_MODIFY		= "MODIFY";
/**
 * Defines the "load" event type.
 */
ZmEvent.E_LOAD			= "LOAD";
/**
 * Defines the "remove" event type.
 */
ZmEvent.E_REMOVE		= "REMOVE";
/**
 * Defines the "remove all" event type.
 */
ZmEvent.E_REMOVE_ALL	= "REMOVE ALL";
/**
 * Defines the "move" event type.
 */
ZmEvent.E_MOVE			= "MOVE";
/**
 * Defines the "flags" event type.
 */
ZmEvent.E_FLAGS			= "FLAGS";
/**
 * Defines the "tags" event type.
 */
ZmEvent.E_TAGS			= "TAGS";
/**
 * Defines the "zimlets" event type.
 */
ZmEvent.E_ZIMLETS		= "ZIMLET";
/**
 * Defines the "complete" event type.
 */
ZmEvent.E_COMPLETE		= "COMPLETE";

// Public methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmEvent.prototype.toString = 
function() {
	return "ZmEvent";
};

/**
 * Sets the event type and source.
 *
 * @param {constant}	event	the event type (see <code>ZmEvent.E_</code> constants)
 * @param {Object}	source		the object that generated the event (typically "this")
 */
ZmEvent.prototype.set =
function(event, source) {
	this.event = event;
	this.source = source;
	this.handled = false;
};

/**
 * Adds info to the event details.
 *
 * @param {String}		field		the detail name
 * @param {Object}		value		the detail value
 */
ZmEvent.prototype.setDetail =
function(field, value) {
	this._details[field] = value;
};

/**
 * Gets info from the event details.
 *
 * @param {String}	field		the detail field name
 * @return	{Object}	the details
 */
ZmEvent.prototype.getDetail =
function(field) {
	return this._details[field];
};

/**
 * Gets items by checking for a detail with a name of "items" and returning it.
 * 
 * @return	{Array}		an array of items or empty array if "items" does not exist
 */
ZmEvent.prototype.getItems =
function() {
    var items = this._details["items"];
    return items ? items : [];
};

/**
 * Sets the event details. Any existing details will be lost.
 *
 * @param {Hash}	details		a hash representing event details
 */
ZmEvent.prototype.setDetails =
function(details) {
	this._details = details ? details : {};
};

/**
 * Gets the event details.
 * 
 * @return	{Hash}	the event details
 */
ZmEvent.prototype.getDetails =
function() {
	return this._details;
};
}
if (AjxPackage.define("zimbraMail.share.model.events.ZmAppEvent")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines an application event.
 *
 */

/**
 * Creates an empty application event.
 * @class
 * This class represents an event related to a change of state for an individual
 * application or for ZCS as a whole.
 * 
 * @param {Object}	the application to which this event applies; if <code>null</code>, the event applies to ZCS
 * 
 * @extends		ZmEvent
 */
ZmAppEvent = function(app) {
	ZmEvent.call(this);
};

ZmAppEvent.prototype = new ZmEvent;
ZmAppEvent.prototype.constructor = ZmAppEvent;

/**
 * Event used to notify listeners before startup (i.e. before the first
 * app is activated). This is a bit of a misnomer because this event occurs
 * after the apps are initialized but before the first app is shown. This
 * allows code to be executed after the apps have registered settings
 * but before the app actually acts on those settings.
 *
 * @see ZmAppEvent.POST_STARTUP
 */
ZmAppEvent.PRE_STARTUP	= "PRESTARTUP";

/**
 * Defines the event used to notify listeners post-startup.
 */
ZmAppEvent.POST_STARTUP	= "POSTSTARTUP";
/**
 * Defines the event used to notify listeners pre-launch.
 */
ZmAppEvent.PRE_LAUNCH	= "PRELAUNCH";
/**
 * Defines the event used to notify listeners post-launch.
 */
ZmAppEvent.POST_LAUNCH	= "POSTLAUNCH";
/**
 * Defines the event used to notify listeners post-render.
 */
ZmAppEvent.POST_RENDER	= "POSTRENDER";

ZmAppEvent.ACTIVATE	= "ACTIVATE";

// Triggered after processing of an async response finishes
ZmAppEvent.RESPONSE = "RESPONSE";

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAppEvent.prototype.toString =
function() {
	return "ZmAppEvent";
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmModel")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This class represents a data model which can process change events.
 *
 */

/**
 * Creates the data model.
 * @class
 * This class represents a data model which can process change events.
 * 
 * @author Conrad Damon
 *
 * @param {constant}		type	the event source type {@see ZmEvent}
 */
ZmModel = function(type) {
 	if (arguments.length == 0) return;

	this._evt = new ZmEvent(type);
	this._evtMgr = new AjxEventMgr();
}

ZmModel.prototype.isZmModel = true;
ZmModel.prototype.toString = function() { return "ZmModel"; }

/**
* Adds a change listener.
*
* @param {AjxListener}	listener	the change listener to add
*/
ZmModel.prototype.addChangeListener = 
function(listener) {
	return this._evtMgr.addListener(ZmEvent.L_MODIFY, listener);
}

/**
* Removes the given change listener.
*
* @param {AjxListener}	listener		the change listener to remove
*/
ZmModel.prototype.removeChangeListener = 
function(listener) {
	return this._evtMgr.removeListener(ZmEvent.L_MODIFY, listener);    	
}

/**
* Removes all change listeners.
* 
*/
ZmModel.prototype.removeAllChangeListeners = 
function() {
	return this._evtMgr.removeAll(ZmEvent.L_MODIFY);    	
}

/**
* Notifies listeners of the given change event.
*
* @param {constant}		event		the event type {@see ZmEvent}
* @param {Hash}			details		additional information
* 
* @private
*/
ZmModel.prototype._notify =
function(event, details) {
	if (this._evtMgr.isListenerRegistered(ZmEvent.L_MODIFY)) {
		this._evt.set(event, this);
		this._evt.setDetails(details);
		this._evtMgr.notifyListeners(ZmEvent.L_MODIFY, this._evt);
	}
};

/**
 * @private
 */
ZmModel.notifyEach =
function(list, event, details) {
	if (!(list && list.length)) { return; }
	for (var i = 0; i < list.length; i++) {
		list[i]._notify(event, details);
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmSetting")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains a setting class.
 */

/**
 * Creates a setting.
 * @class
 * This class represents a single setting. A setting's default value never changes; it
 * is available in case the user wishes to restore the current value to the default.
 * Most but not all settings have a corollary on the server side. Settings that don't
 * will depend on the environment or user activity to get their value.
 *
 * @author Conrad Damon
 * 
 * @param {String}	id				a unique ID
 * @param {Hash}	params			a hash of parameters
 * @param {String}	params.name				the name of the pref or attr on the server
 * @param {constant}	params.type			config, pref, or COS (see <code>ZmSetting.T_</code> constants)
 * @param {constant}	params.dataType			string, int, or boolean
 * @param {Object}	params.defaultValue		the default value
 * @param {Boolean}	params.isGlobal			if <code>true</code>, this setting is global across accounts
 * @param {Boolean}	params.isImplicit		if <code>true</code>, this setting is not represented in Preferences
 * 
 * @extends		ZmModel
 */
ZmSetting = function(id, params) {

	if (arguments.length == 0) return;
	ZmModel.call(this, ZmEvent.S_SETTING);
	
	this.id = id;
	this.name = params.name;
	this.type = params.type;
	this.dataType = params.dataType || ZmSetting.D_STRING;
	this.defaultValue = params.defaultValue;
	this.canPreset = params.canPreset;
	if (this.type == ZmSetting.T_METADATA) {
		this.section = params.section;
	}
	if (params.isGlobal) {
		ZmSetting.IS_GLOBAL[id] = true;
	}
	if (params.isImplicit) {
		ZmSetting.IS_IMPLICIT[id] = true;
	}
	
	if (this.dataType == ZmSetting.D_HASH) {
		this.value = {};
		this.defaultValue = {};
	} else if (this.dataType == ZmSetting.D_LIST) {
		this.value = [];
		this.defaultValue = [];
	} else {
		this.value = null;
	}

    this.dontSaveDefault = params.dontSaveDefault;
};

ZmSetting.prototype = new ZmModel;
ZmSetting.prototype.constructor = ZmSetting;

// setting types
/**
 * Defines the "config" type.
 */
ZmSetting.T_CONFIG		= "config";
/**
 * Defines the "COS" type.
 */
ZmSetting.T_COS			= "cos";
/**
 * Defines the "domain" type.
 */
ZmSetting.T_DOMAIN		= "domain";
/**
 * Defines the "meta-data" type.
 */
ZmSetting.T_METADATA	= "meta";
/**
 * Defines the "pref" type.
 */
ZmSetting.T_PREF		= "pref";
/**
 * Defines the "pseudo" type.
 */
ZmSetting.T_PSEUDO		= "pseudo";

// metadata sections
ZmSetting.M_IMPLICIT	= "implicit";
ZmSetting.M_OFFLINE		= "offline";
ZmSetting.M_ZIMLET		= "zimlet";

// setting data types
ZmSetting.D_STRING		= "string"; // default type
ZmSetting.D_INT			= "int";
ZmSetting.D_BOOLEAN		= "boolean";
ZmSetting.D_LDAP_TIME 	= "ldap_time";
ZmSetting.D_HASH 		= "hash";
ZmSetting.D_LIST		= "list";
ZmSetting.D_NONE		= "NONE";	// placeholder setting

// constants used as setting values
// TODO: these should be defined in their respective apps
/**
 * Defines the "all" ACL grantee type.
 * @type String
 */
ZmSetting.ACL_AUTH				= "all";
/**
 * Defines the "group" ACL grantee type.
 * @type String
 */
ZmSetting.ACL_GROUP				= "grp";
/**
 * Defines the "none" ACL grantee type.
 * @type String
 */
ZmSetting.ACL_NONE				= "none";
/**
 * Defines the "public" ACL grantee type.
 * @type String
 */
ZmSetting.ACL_PUBLIC			= "pub";
/**
 * Defines the "domain" ACL grantee type.
 * @type String
 */
ZmSetting.ACL_DOMAIN			= "dom";
/**
 * Defines the "user" ACL grantee type.
 * @type String
 */
ZmSetting.ACL_USER				= "usr";
ZmSetting.CAL_DAY				= "day";
ZmSetting.CAL_LIST				= "list";
ZmSetting.CAL_MONTH				= "month";
ZmSetting.CAL_WEEK				= "week";
ZmSetting.CAL_WORK_WEEK			= "workWeek";
ZmSetting.CAL_VISIBILITY_PRIV	= "private";
ZmSetting.CAL_VISIBILITY_PUB	= "public";
ZmSetting.CLIENT_ADVANCED		= "advanced";				// zimbraPrefClientType
ZmSetting.CLIENT_MODERN			= "modern";
ZmSetting.COMPOSE_FONT_COLOR	= "#000000";	 			// zimbraPrefHtmlEditorDefaultFontColor
ZmSetting.COMPOSE_FONT_FAM 		= "arial,helvetica,sans-serif";		// zimbraPrefHtmlEditorDefaultFontFamily
ZmSetting.COMPOSE_FONT_SIZE 	= AjxMessageFormat.format(ZmMsg.pt,"12"); 			// zimbraPrefHtmlEditorDefaultFontSize
ZmSetting.LTR                   = "LTR";
ZmSetting.RTL                   = "RTL";
ZmSetting.COMPOSE_TEXT 			= "text";					// zimbraPrefComposeFormat
ZmSetting.COMPOSE_HTML 			= "html";
ZmSetting.CV_CARDS				= "cards"; 					// zimbraPrefContactsInitialView
ZmSetting.CV_LIST				= "list";
ZmSetting.DEDUPE_NONE			= "dedupeNone";				// zimbraPrefDedupeMessagesSentToSelf
ZmSetting.DEDUPE_SECOND			= "secondCopyifOnToOrCC";
ZmSetting.DEDUPE_INBOX			= "moveSentMessageToInbox";
ZmSetting.DEDUPE_ALL			= "dedupeAll";
ZmSetting.DELETE_SELECT_NEXT	= "next";					// zimbraPrefMailSelectAfterDelete
ZmSetting.DELETE_SELECT_PREV	= "previous";
ZmSetting.DELETE_SELECT_ADAPT	= "adaptive";
ZmSetting.GROUP_BY_CONV			= "conversation";			// zimbraPrefGroupMailBy
ZmSetting.GROUP_BY_MESSAGE		= "message";
ZmSetting.HTTP_DEFAULT_PORT		= 80;
ZmSetting.HTTPS_DEFAULT_PORT	= 443;
ZmSetting.INC_NONE				= "includeNone";			// zimbraPrefReplyIncludeOriginalText / zimbraPrefForwardIncludeOriginalText
ZmSetting.INC_ATTACH			= "includeAsAttachment";
ZmSetting.INC_BODY				= "includeBody";				// deprecated - same as includeBodyAndHeaders
ZmSetting.INC_BODY_ONLY			= "includeBodyOnly";
ZmSetting.INC_BODY_PRE			= "includeBodyWithPrefix";
ZmSetting.INC_BODY_HDR			= "includeBodyAndHeaders";
ZmSetting.INC_BODY_PRE_HDR		= "includeBodyAndHeadersWithPrefix";
ZmSetting.INC_SMART				= "includeSmart";
ZmSetting.INC_SMART_PRE			= "includeSmartWithPrefix";
ZmSetting.INC_SMART_HDR			= "includeSmartAndHeaders";
ZmSetting.INC_SMART_PRE_HDR		= "includeSmartAndHeadersWithPrefix";
ZmSetting.MARK_READ_NONE		= -1;						// zimbraPrefMarkMsgRead
ZmSetting.MARK_READ_NOW			= 0;						// zimbraPrefMarkMsgRead
ZmSetting.MARK_READ_TIME		= 1;						// zimbraPrefMarkMsgRead
ZmSetting.PRINT_FONT_SIZE 	    = AjxMessageFormat.format(ZmMsg.pt,"12"); 			// zimbraPrefDefaultPrintFontSize
ZmSetting.PROTO_HTTP			= "http:";
ZmSetting.PROTO_HTTPS			= "https:";
ZmSetting.PROTO_MIXED			= "mixed:";
ZmSetting.RIGHT_VIEW_FREE_BUSY	= "viewFreeBusy";
ZmSetting.RIGHT_INVITE			= "invite";
ZmSetting.RP_BOTTOM				= "bottom";					// zimbraPrefReadingPaneLocation / zimbraPrefConvReadingPaneLocation / zimbraPrefTasksReadingPaneLocation / zimbraPrefBriefcaseReadingPaneLocation
ZmSetting.RP_OFF				= "off";
ZmSetting.RP_RIGHT				= "right";
ZmSetting.SIG_INTERNET			= "internet";				// zimbraPrefMailSignatureStyle
ZmSetting.SIG_OUTLOOK			= "outlook";

// values for the 'fetch' param of SearchConvRequest
ZmSetting.CONV_FETCH_NONE                       = "0";
ZmSetting.CONV_FETCH_FIRST_MATCHING             = "1";
ZmSetting.CONV_FETCH_FIRST                      = "!";
ZmSetting.CONV_FETCH_UNREAD                     = "u";
ZmSetting.CONV_FETCH_UNREAD_OR_FIRST_MATCHING   = "u1";
ZmSetting.CONV_FETCH_UNREAD_OR_FIRST            = "u!";
ZmSetting.CONV_FETCH_UNREAD_OR_BOTH_FIRST       = "u1!";
ZmSetting.CONV_FETCH_MATCHES                    = "hits";
ZmSetting.CONV_FETCH_MATCHES_OR_FIRST           = "hits!";
ZmSetting.CONV_FETCH_ALL                        = "all";

// License status (network only)
ZmSetting.LICENSE_GOOD			= "OK";
ZmSetting.LICENSE_NOT_INSTALLED = "NOT_INSTALLED";
ZmSetting.LICENSE_NOT_ACTIVATED = "NOT_ACTIVATED";
ZmSetting.LICENSE_FUTURE		= "IN_FUTURE";
ZmSetting.LICENSE_EXPIRED		= "EXPIRED";
ZmSetting.LICENSE_BAD			= "INVALID";
ZmSetting.LICENSE_GRACE			= "LICENSE_GRACE_PERIOD";
ZmSetting.LICENSE_ACTIV_GRACE	= "ACTIVATION_GRACE_PERIOD";

// warning messages for bad license statuses
ZmSetting.LICENSE_MSG									= {};
ZmSetting.LICENSE_MSG[ZmSetting.LICENSE_NOT_INSTALLED]	= ZmMsg.licenseNotInstalled;
ZmSetting.LICENSE_MSG[ZmSetting.LICENSE_NOT_ACTIVATED]	= ZmMsg.licenseNotActivated;
ZmSetting.LICENSE_MSG[ZmSetting.LICENSE_FUTURE]			= ZmMsg.licenseExpired;
ZmSetting.LICENSE_MSG[ZmSetting.LICENSE_EXPIRED]		= ZmMsg.licenseExpired;
ZmSetting.LICENSE_MSG[ZmSetting.LICENSE_BAD]			= ZmMsg.licenseExpired;

// we need these IDs available when the app classes are parsed
ZmSetting.LOCALE_NAME			= "LOCALE_NAME";
ZmSetting.COMPOSE_INIT_DIRECTION= "COMPOSE_INIT_DIRECTION";
ZmSetting.SHOW_COMPOSE_DIRECTION_BUTTONS = "SHOW_COMPOSE_DIRECTION_BUTTONS";
ZmSetting.FONT_NAME				= "FONT_NAME";
ZmSetting.FONT_SIZE				= "FONT_SIZE";
ZmSetting.SKIN_NAME				= "SKIN_NAME";

ZmSetting.BRIEFCASE_ENABLED		= "BRIEFCASE_ENABLED";
ZmSetting.CALENDAR_ENABLED		= "CALENDAR_ENABLED";
ZmSetting.CONTACTS_ENABLED		= "CONTACTS_ENABLED";
ZmSetting.MAIL_ENABLED			= "MAIL_ENABLED";
ZmSetting.OPTIONS_ENABLED		= "OPTIONS_ENABLED";
ZmSetting.PORTAL_ENABLED		= "PORTAL_ENABLED";
ZmSetting.SEARCH_ENABLED		= "SEARCH_ENABLED";
ZmSetting.SOCIAL_ENABLED		= "SOCIAL_ENABLED";
ZmSetting.TASKS_ENABLED			= "TASKS_ENABLED";
ZmSetting.VOICE_ENABLED			= "VOICE_ENABLED";
ZmSetting.TAGGING_ENABLED		= "TAGGING_ENABLED";

ZmSetting.CALENDAR_UPSELL_ENABLED	= "CALENDAR_UPSELL_ENABLED";
ZmSetting.CONTACTS_UPSELL_ENABLED	= "CONTACTS_UPSELL_ENABLED";
ZmSetting.MAIL_UPSELL_ENABLED		= "MAIL_UPSELL_ENABLED";
ZmSetting.SOCIAL_EXTERNAL_ENABLED   = "SOCIAL_EXTERNAL_ENABLED";
ZmSetting.SOCIAL_EXTERNAL_URL	    = "SOCIAL_EXTERNAL_URL";
ZmSetting.VOICE_UPSELL_ENABLED		= "VOICE_UPSELL_ENABLED";
ZmSetting.SHARING_ENABLED			= "SHARING_ENABLED";

//user selected font
ZmSetting.FONT_CLASSIC	= "classic";
ZmSetting.FONT_MODERN	= "modern";
ZmSetting.FONT_WIDE		= "wide";
ZmSetting.FONT_SYSTEM	= "system";

//user selected font size
ZmSetting.FONT_SIZE_SMALL = "small";
ZmSetting.FONT_SIZE_NORMAL = "normal";
ZmSetting.FONT_SIZE_LARGE = "large";
ZmSetting.FONT_SIZE_LARGER = "larger";


// name for dynamic CSS class created from user font prefs
ZmSetting.USER_FONT_CLASS = "userFontPrefs";

//task filterby setting
ZmSetting.TASK_FILTER_ALL = "";
ZmSetting.TASK_FILTER_TODO = "TODO";
ZmSetting.TASK_FILTER_COMPLETED = "COMPLETED";
ZmSetting.TASK_FILTER_WAITING = "WAITING";
ZmSetting.TASK_FILTER_DEFERRED = "DEFERRED";
ZmSetting.TASK_FILTER_INPROGRESS = "INPROGRESS";
ZmSetting.TASK_FILTER_NOTSTARTED = "NOTSTARTED";

// hash of global settings
ZmSetting.IS_GLOBAL = {};

// hash of implicit settings
ZmSetting.IS_IMPLICIT = {};

// hash of implicit settings that have been changed during the current session
ZmSetting.CHANGED_IMPLICIT = {};

// Send As and Send On Behalf Of settings
ZmSetting.SEND_AS = "sendAs";
ZmSetting.SEND_ON_BEHALF_OF = "sendOnBehalfOf";

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmSetting.prototype.toString =
function() {
	return this.name + ": " + this.value;
};

/**
 * Gets the current value of this setting.
 *
 * @param {String}	key			the optional key for use by hash table data type
 * @param {Boolean}	serialize		if <code>true</code>, serialize non-string value into string
 * @return	{Object}	the value
 */
ZmSetting.prototype.getValue =
function(key, serialize) {

	var value = null;
	if (this.value != null) {
		value = key ? this.value[key] : this.value;
	} else if (this.defaultValue != null) {
		value = key ? this.defaultValue[key] : this.defaultValue;
	} else {
		return null;
	}

    if(this.dontSaveDefault && serialize && !key){
        value = this.getRefinedValue(value);
    }

	return serialize ? ZmSetting.serialize(value, this.dataType) : value;
};

ZmSetting.prototype.getRefinedValue =
function(value){
    if(this.dataType == ZmSetting.D_HASH){
        var refinedValue = {}, dValue = this.defaultValue;
        for(var key in value){
             refinedValue[key] = (dValue[key] != value[key]) ? value[key] : "";
        }
        return refinedValue;
    }
    return value;
};

/**
 * Gets the original value of this setting.
 *
 * @param {String}	key			the optional key for use by hash table data type
 * @param {Boolean}	serialize		if <code>true</code>, serialize non-string value into string
 * @return	{Object}	the value
 */
ZmSetting.prototype.getOrigValue =
function(key, serialize) {

	var origValue = null;
	if (this.origValue != null) {
		origValue = key ? this.origValue[key] : this.origValue;
	} else if (this.defaultValue != null) {
		origValue = key ? this.defaultValue[key] : this.defaultValue;
	} else {
		return null;
	}

	return serialize ? ZmSetting.serialize(origValue, this.dataType) : origValue;
};

/**
 * Gets the default value of this setting.
 *
 * @param {String}	key 			the optional key for use by hash table data type
 * @return	{Object}	the value
 */
ZmSetting.prototype.getDefaultValue =
function(key, serialize) {
	var value = key ? this.defaultValue[key] : this.defaultValue;
	return serialize ? ZmSetting.serialize(value, this.dataType) : value;
};

/**
 * Sets the current value of this setting, performing any necessary data type conversion.
 *
 * @param {Object}	value			the new value for the setting
 * @param {String}	key 			optional key for use by hash table data type
 * @param {Boolean}	setDefault		if <code>true</code>, also set the default value
 * @param {Boolean}	skipNotify		if <code>true</code>, do not notify listeners
 * @param {Boolean}	skipImplicit		if <code>true</code>, do not check for change to implicit pref
 */
ZmSetting.prototype.setValue =
function(value, key, setDefault, skipNotify, skipImplicit) {

	var newValue = value;
	var changed = Boolean(newValue != this.value);
	if (this.dataType == ZmSetting.D_STRING) {
		this.value = newValue;
	} else if (this.dataType == ZmSetting.D_INT) {
		newValue = parseInt(value);
		if (isNaN(newValue)) { // revert to string if NaN
			newValue = value;
		}
		changed = Boolean(newValue != this.value);
		this.value = newValue;
	} else if (this.dataType == ZmSetting.D_BOOLEAN) {
		if (typeof(newValue) == "string") {
			newValue = (newValue.toLowerCase() === "true");
		}
		changed = Boolean(newValue != this.value);
		this.value = newValue;
	} else if (this.dataType == ZmSetting.D_LDAP_TIME) {
		var lastChar = (newValue.toLowerCase) ? lastChar = (newValue.toLowerCase()).charAt(newValue.length-1) : null;
		var num = parseInt(newValue);
		// convert to seconds
		if (lastChar == 'd') {
			newValue = num * 24 * 60 * 60;
		} else if (lastChar == 'h') {
			newValue = num * 60 * 60;
		} else if (lastChar == 'm') {
			newValue = num * 60;
		} else {
			newValue = num;	// default
		}
		changed = Boolean(newValue != this.value);
		this.value = newValue;
	} else if (this.dataType == ZmSetting.D_HASH) {
		changed = true;
        if (key) {
			if (newValue) {
                changed = Boolean(newValue != this.value[key]);
				this.value[key] = newValue;
			} else {
				delete this.value[key];
			}
		} else {
			this.value = newValue;
		}
	} else if (this.dataType == ZmSetting.D_LIST) {
		if (newValue instanceof Array) {
			this.value = newValue;
		} else {
			this.value.push(newValue);
		}
		changed = true;
	}

	if (setDefault) {
		if (key) {
			this.defaultValue[key] = this.value[key];
		} else {
			this.defaultValue = this.value;
		}
	}
	
	if (ZmSetting.IS_IMPLICIT[this.id] && changed && !skipImplicit) {
		if (skipNotify) {
			ZmSetting.CHANGED_IMPLICIT[this.id] = true;
		} else {
			this._notify(ZmEvent.E_MODIFY, key);
			return;
		}
	}

	// Setting an internal pref is equivalent to saving it, so we should notify
	if (!this.name && !skipNotify) {
		this._notify(ZmEvent.E_MODIFY, key);
	}
};

/**
 * Handles modify notification.
 * 
 * @param	{Object}	obj		the object
 */
ZmSetting.prototype.notifyModify = 
function(obj) {
	if (this.id == ZmSetting.QUOTA_USED && obj._name == "mbx" && obj.s != null) {
		this.setValue(obj.s);
		this._notify(ZmEvent.E_MODIFY, {account:obj.account});
	}
};

ZmSetting.prototype.copyValue =
function() {

	if (this.dataType == ZmSetting.D_HASH) {
		return AjxUtil.hashCopy(this.value);
	} else if (this.dataType == ZmSetting.D_LIST) {
		return this.value.concat();
	} else {
		return this.value;
	}
};

ZmSetting.serialize =
function(value, dataType) {

	if (dataType == ZmSetting.D_BOOLEAN) {
		value = value ? "TRUE" : "FALSE";
	} else if (dataType == ZmSetting.D_HASH) {
		var keys = [];
		for (var key in value) {
			keys.push(key);
		}
		keys.sort();
		var pairs = [];
		for (var j = 0; j < keys.length; j++) {
			var key = keys[j];
			pairs.push([key, value[key]].join(":"));
		}
		value = pairs.join(",");
	} else if (dataType == ZmSetting.D_LIST) {
		value = value.join(",");
	}

	return value;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmAccessControlList")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines an access control list and associated classes.
 *
 */

/**
 * Creates an empty access control list (ACL).
 * @class
 * An access control list is a collection of access control entries (ACEs). Each entry contains
 * information about a certain permission applied by the current user to another user or users
 * for a particular type of action. So far, there are two types of rights that are managed in
 * this way:
 * 
 * <ul>
 * <li><b>viewFreeBusy</b> - governs whether other users may view this user's free/busy information</li>
 * <li><b>invite</b> - determines whether an invite from other users will automatically create a tentative appointment on this user's calendar</li>
 * </ul>
 * 
 * Note: that shared organizers ({@link ZmShare}) manage rights (read/write/manage) in their own way.
 * 
 * @author Conrad Damon
 * 
 * @param {Array}	aces		the list of {@link ZmAccessControlEntry} objects
 */
ZmAccessControlList = function(aces) {
	this._aces = {};
}

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAccessControlList.prototype.toString =
function() {
	return "ZmAccessControlList";
};

/**
 * Loads the list.
 * 
 * @param	{AjxCallback}	callback	the function to callback after the loaded
 * 
 * @private
 */
ZmAccessControlList.prototype.load =
function(callback) {
	var jsonObj = {GetRightsRequest:{_jsns:"urn:zimbraAccount"}};
	var respCallback = new AjxCallback(this, this._handleResponseLoad, [callback]);
	appCtxt.getAppController().sendRequest({jsonObj:jsonObj, asyncMode:true, callback:respCallback});
};

/**
 * @private
 */
ZmAccessControlList.prototype._handleResponseLoad =
function(callback, result) {
	var response = result.getResponse();
	var aces = response.GetRightsResponse.ace;
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			this.add(ZmAccessControlEntry.createFromDom(aces[i]));
		}
	}
	if (callback) {
		callback.run();
	}
};

/**
 * Gets the access control entry by right.
 * 
 * @param	{String}	right		the right
 * @return	{ZmAccessControlEntry}	the entry
 */
ZmAccessControlList.prototype.getACLByRight =
function(right) {
	return this._aces[right];
};

/**
 * Gets the grantee type.
 * 
 * @param	{String}	right		the right
 * @return	{constant}	the grantee type (see <code>ZmSetting.ACL_</code> constants)
 * 
 * @see		ZmSetting
 */
ZmAccessControlList.prototype.getGranteeType =
function(right) {
	var aces = this._aces[right];
	var gt = ZmSetting.ACL_PUBLIC;
	
	var gtMap = {};
	if(aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = aces[i];
			DBG.println("<font color=red>ace:</font>" + (ace.negative?"-":"") + ace.granteeType +"," +  ace.grantee );
			var aceGranteeType =  (ace.granteeType == ZmSetting.ACL_USER || ace.granteeType == ZmSetting.ACL_GROUP)  ? ZmSetting.ACL_USER : ace.granteeType;
			gtMap[aceGranteeType] = ace.negative ? -1 : 1;
		}
	}
	
	var allowPublic = (gtMap[ZmSetting.ACL_PUBLIC] == 1);
	var denyPublic  = (gtMap[ZmSetting.ACL_PUBLIC] == -1);
	var allowLocal  = (gtMap[ZmSetting.ACL_AUTH] == 1);
	var denyLocal   = (gtMap[ZmSetting.ACL_AUTH] == -1);
    var allowDomainOnly   = (gtMap[ZmSetting.ACL_DOMAIN] == 1);
	
	var allowUser = (gtMap[ZmSetting.ACL_USER] == 1);
	var allowNone = (denyPublic || denyLocal) && (gtMap[ZmSetting.ACL_USER] == null);
				
	if(allowPublic) {
		return ZmSetting.ACL_PUBLIC;
	}
	
	if(allowLocal) {
		return ZmSetting.ACL_AUTH;
	}
	
	if(denyPublic) {
		if(allowLocal) {
			return ZmSetting.ACL_AUTH;
		}
	}
	
	if(allowUser) {
		return ZmSetting.ACL_USER;
	}

    if(allowDomainOnly) {
		return ZmSetting.ACL_DOMAIN;
	}
	
	if(allowNone) {
		return ZmSetting.ACL_NONE;
	}
	return gt;
};

/**
 * Gets the access control entry by grantee type.
 * 
 * @param	{String}	right	the right
 * @param	{constant}	gt		the grantee type (see <code>ZmSetting.ACL_</code> constants)
 * @return	{Array}	an array of {@link ZmAccessControlEntry} objects
 */
ZmAccessControlList.prototype.getACLByGranteeType =
function(right, gt) {
	var aces = this._aces[right];
	var list = [];
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = aces[i];
			if (ace.granteeType == gt) {
				list.push(ace);
			}
		}
	}
	list.sort();
	return list;
};

/**
 * Gets the grantees.
 * 
 * @param	{String}	right	the right
 * @return	{Array}		an array of grantees
 */
ZmAccessControlList.prototype.getGrantees =
function(right) {
	var aces = this._aces[right];
	var list = [];
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = aces[i];
			if (ace.granteeType == ZmSetting.ACL_USER || ace.granteeType == ZmSetting.ACL_GROUP) {
				list.push(ace.grantee);
			}
		}
	}
	list.sort();
	return list;
};

/**
 * Gets the grantees info.
 * 
 * @param	{String}	right		the right
 * @return	{Array}	an array of grantree info objects (obj.grantee, obj.zid)
 */
ZmAccessControlList.prototype.getGranteesInfo =
function(right) {
	var aces = this._aces[right];
	var list = [];
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = aces[i];
			if (ace.granteeType == ZmSetting.ACL_USER || ace.granteeType == ZmSetting.ACL_GROUP) {
				list.push({grantee: ace.grantee, zid: ace.zid});
			}
		}
	}
	list.sort(ZmAccessControlList.sortByGrantee);
	return list;
};

/**
 * Grants permissions on the access control entries.
 * 
 * @param	{Array}	aces		an array of {@link ZmAccessControlEntry} objects
 * @param	{AjxCallback}	callback	the callback
 * @param	{Boolean}	batchCmd	<code>true</code> to submit as a batch command
 */
ZmAccessControlList.prototype.grant =
function(aces, callback, batchCmd) {
	this._setPerms(aces, false, callback, batchCmd);
};

/**
 * Revokes and denies permissions the access control entries.
 * 
 * @param	{Array}	aces		an array of {@link ZmAccessControlEntry} objects
 * @param	{AjxCallback}	callback	the callback
 * @param	{Boolean}	batchCmd	<code>true</code> to submit as a batch command
 */
ZmAccessControlList.prototype.revoke =
function(aces, callback, batchCmd) {
	this._setPerms(aces, true, callback, batchCmd);
};

/**
 * Sets the permissions.
 * 
 * @param	{Array}	aces		an array of {@link ZmAccessControlEntry} objects
 * @param	{Boolean}	revoke	<code>true</code> to deny; <code>false</code> to grant
 * @param	{AjxCallback}	callback	the callback
 * @param	{Boolean}	batchCmd	<code>true</code> to submit as a batch command
 *
 * @private
 */
ZmAccessControlList.prototype._setPerms =
function(aces, revoke, callback, batchCmd) {
	var reqName = revoke ? "RevokeRightsRequest" : "GrantRightsRequest";
	var soapDoc = AjxSoapDoc.create(reqName, "urn:zimbraAccount");
	for (var i = 0; i < aces.length; i++) {
		var ace = aces[i];
		var aceNode = soapDoc.set("ace");
		aceNode.setAttribute("right", ace.right);
		aceNode.setAttribute("gt", ace.granteeType);
		if(ace.grantee) {
			aceNode.setAttribute("d", ace.grantee);
		}
		if (ace.zid) {
			aceNode.setAttribute("zid", ace.zid);
		}
		if (ace.negative) {
			aceNode.setAttribute("deny", 1);
		}
	}
	var respCallback = new AjxCallback(this, this._handleResponseSetPerms, [revoke, callback]);
	if (batchCmd) {
		batchCmd.addNewRequestParams(soapDoc, respCallback);
	} else {
		appCtxt.getAppController().sendRequest({soapDoc:soapDoc, asyncMode:true, callback:respCallback});
	}
};

/**
 * @private
 */
ZmAccessControlList.prototype._handleResponseSetPerms =
function(revoke, callback, result) {
	var response = result.getResponse();
	var resp = revoke ? response.RevokeRightsResponse : response.GrantRightsResponse;
	var aces = resp && resp.ace;
	var aceList = [];
	if (aces && aces.length) {
		for (var i = 0; i < aces.length; i++) {
			var ace = ZmAccessControlEntry.createFromDom(aces[i]);
			aceList.push(ace);
			if (revoke) {
				this.remove(ace);
			} else {
				this.update(ace);
			}
		}
	}

	if (callback) {
		callback.run(aceList);
	}
};

/**
 * Adds the entry to the ACL.
 * 
 * @param	{ZmAccessControlEntry}	ace	the entry to add
 */
ZmAccessControlList.prototype.add =
function(ace) {
	if (!ace) { return; }
	var right = ace.right;
	if (!this._aces[right]) {
		this._aces[right] = [];
	}
	this._aces[right].push(ace);
};

/**
 * Removes the entry to the ACL.
 * 
 * @param	{ZmAccessControlEntry}	ace	the entry to remove
 */
ZmAccessControlList.prototype.remove =
function(ace) {
	if (!ace) { return; }
	var list = this._aces[ace.right];
	var newList = [];
	if (list && list.length) {
		for (var i = 0; i < list.length; i++) {
			if (list[i].grantee != ace.grantee) {
				newList.push(list[i]);
			}
		}
	}
	this._aces[ace.right] = newList;
};

/**
 * Updates the entry to the ACL.
 * 
 * @param	{ZmAccessControlEntry}	ace	the entry to update
 * @param	{Boolean}	removeEnty	not used
 */
ZmAccessControlList.prototype.update =
function(ace, removeEntry) {
	if (!ace || !ace.right) { return; }
	var found = false;
	
	if(!this._aces[ace.right]) {
		this._aces[ace.right] = [];
	}

	var list = this._aces[ace.right];	
	if (list.length) {
		//search for ace to update
		for (var i = 0; i < list.length; i++) {
			if ((list[i].grantee == ace.grantee) && (list[i].granteeType == ace.granteeType)) {
				this._aces[ace.right][i] = ace;
				found = true;
			}
		}
	}
	if(!found) {
		//adding new entry to ace list
		this._aces[ace.right].push(ace);
	}
};

/**
 * Cleans up the ACL.
 * 
 */
ZmAccessControlList.prototype.cleanup =
function() {
	this._aces = {};
};

/**
 * Sorts the ACL by grantee.
 * 
 * @param	{Hash}	a		grantee "a"
 * @param	{String}	a.grantee	the grantee
 * @param	{Hash}	b		grantee "b"
 * @param	{Hash}	b.grantee		grantee "b"
 * @return	{int}	0 if "a" and "b" are the same; 1 if "a" is before "b"; -1 if "b" is before "a"
 */
ZmAccessControlList.sortByGrantee =
function(a, b) {

    var granteeA = a.grantee || "";
    var granteeB = b.grantee || "";

    if (granteeA.toLowerCase() > granteeB.toLowerCase()) { return 1; }
    if (granteeA.toLowerCase() < granteeB.toLowerCase()) { return -1; }
    
	return 0;
};


/**
 * Creates an access control entry.
 * @class
 * An access control entry encapsulates the permission information pertaining to a user or users
 * regarding a certain right.
 * 
 * @param {Hash}	params		a hash of parameters
 * @param	{String}	params.right		the action governed by this ACE
 * @param	{String}	params.grantee		the account name of user or group permission applies to
 * @param	{String}	params.zid			the ZID of grantee
 * @param	{constant}	params.granteeType	type of grantee (see <code>ZmSetting.ACL_</code> constants)
 * @param	{Boolean}	params.negative		if <code>true</code>, permission is denied by this ACE
 * @see		ZmSetting
 */
ZmAccessControlEntry =
function(params) {
	this.grantee = params.grantee;
	this.zid = params.zid;
	this.granteeType = params.granteeType;
	this.right = params.right;
	this.negative = params.negative;
}

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmAccessControlEntry.prototype.toString =
function() {
	return "ZmAccessControlEntry";
};

/**
 * Creates an entry from the DOM object.
 * 
 * @param	{Hash}	obj		the DOM object
 * @param	{String}	obj.right		the action governed by this ACE
 * @param	{String}	obj.d		the account name of user or group permission applies to
 * @param	{String}	obj.zid			the ZID of grantee
 * @param	{constant}	obj.gt		the type of grantee (see <code>ZmSetting.ACL_</code> constants)
 * @param	{Boolean}	obj.deny		if <code>1</code>, permission is denied by this ACE
 * @return	{ZmAccessControlEntry}	the newly created entry
 */
ZmAccessControlEntry.createFromDom =
function(obj) {
	var params = {};
	params.grantee = obj.d;
	params.granteeType = obj.gt;
	params.zid = obj.zid;
	params.right = obj.right;
	params.negative = (obj.deny == "1");
	
	return new ZmAccessControlEntry(params);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmAutocomplete")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 *
 * This file defines authentication.
 *
 */

/**
 * Creates and initializes support for server-based autocomplete.
 * @class
 * This class manages auto-completion via <code>&lt;AutoCompleteRequest&gt;</code> calls to the server. Currently limited
 * to matching against only one type among people, locations, and equipment.
 *
 * @author Conrad Damon
 */
ZmAutocomplete = function(params) {

	if (arguments.length == 0) {
		return;
	}

	if (appCtxt.get(ZmSetting.CONTACTS_ENABLED)) {
		var listener = this._settingChangeListener.bind(this);
		var settings = [ZmSetting.GAL_AUTOCOMPLETE, ZmSetting.AUTOCOMPLETE_SHARE, ZmSetting.AUTOCOMPLETE_SHARED_ADDR_BOOKS];
		for (var i = 0; i < settings.length; i++) {
			appCtxt.getSettings().getSetting(settings[i]).addChangeListener(listener);
		}
	}
};

// choices for text in the returned match object
ZmAutocomplete.AC_VALUE_FULL = "fullAddress";
ZmAutocomplete.AC_VALUE_EMAIL = "email";
ZmAutocomplete.AC_VALUE_NAME = "name";

// request control
ZmAutocomplete.AC_TIMEOUT = 20;	// autocomplete timeout (in seconds)

// result types
ZmAutocomplete.AC_TYPE_CONTACT = "contact";
ZmAutocomplete.AC_TYPE_GAL = "gal";
ZmAutocomplete.AC_TYPE_TABLE = "rankingTable";

ZmAutocomplete.AC_TYPE_UNKNOWN = "unknown";
ZmAutocomplete.AC_TYPE_LOCATION = "Location";	// same as ZmResource.ATTR_LOCATION
ZmAutocomplete.AC_TYPE_EQUIPMENT = "Equipment";	// same as ZmResource.ATTR_EQUIPMENT

// icons
ZmAutocomplete.AC_ICON = {};
ZmAutocomplete.AC_ICON[ZmAutocomplete.AC_TYPE_CONTACT] = "Contact";
ZmAutocomplete.AC_ICON[ZmAutocomplete.AC_TYPE_GAL] = "GALContact";
ZmAutocomplete.AC_ICON[ZmAutocomplete.AC_TYPE_LOCATION] = "Location";
ZmAutocomplete.AC_ICON[ZmAutocomplete.AC_TYPE_EQUIPMENT] = "Resource";

// cache control
ZmAutocomplete.GAL_RESULTS_TTL = 900000;	// time-to-live for cached GAL autocomplete results (msec)

/**
 * Returns a string representation of the object.
 *
 * @return		{String}		a string representation of the object
 */
ZmAutocomplete.prototype.toString =
		function() {
			return "ZmAutocomplete";
		};

/**
 * Returns a list of matching contacts for a given string. The first name, last
 * name, full name, first/last name, and email addresses are matched against.
 *
 * @param {String}					str				the string to match against
 * @param {closure}					callback		the callback to run with results
 * @param {ZmAutocompleteListView}	aclv			the needed to show wait msg
 * @param {ZmZimbraAccount}			account			the account to fetch cached items from
 * @param {Hash}					options			additional options:
 * @param {constant}				 type			 type of result to match; default is {@link ZmAutocomplete.AC_TYPE_CONTACT}; other valid values are for location or equipment
 * @param {Boolean}					needItem		 if <code>true</code>, return a {@link ZmItem} as part of match result
 * @param {Boolean}					supportForget	allow user to reset ranking for a contact (defaults to true)
 */
ZmAutocomplete.prototype.autocompleteMatch =
		function(str, callback, aclv, options, account, autocompleteType) {

			str = str.toLowerCase().replace(/"/g, '');
			this._curAcStr = str;
			DBG.println("ac", "begin autocomplete for " + str);

			var acType = (options && (options.acType || options.type)) || ZmAutocomplete.AC_TYPE_CONTACT;

			var list = this._checkCache(str, acType, account);
			if (!str || (list !== null)) {
				callback(list);
			}
			else {
				aclv.setWaiting(true, str);
				return this._doSearch(str, aclv, options, acType, callback, account, autocompleteType);
			}
		};

ZmAutocomplete.prototype._doSearch =
		function(str, aclv, options, acType, callback, account, autocompleteType) {

			var params = {query:str, isAutocompleteSearch:true};
			if (acType != ZmAutocomplete.AC_TYPE_CONTACT) {
				params.isGalAutocompleteSearch = true;
				params.isAutocompleteSearch = false;
				params.limit = params.limit * 2;
				var searchType = ((acType === ZmAutocomplete.AC_TYPE_LOCATION) || (acType === ZmAutocomplete.AC_TYPE_EQUIPMENT)) ?  ZmItem.RESOURCE : ZmItem.CONTACT;
				params.types = AjxVector.fromArray([searchType]);
				params.galType = params.galType || ZmSearch.GAL_RESOURCE;
				DBG.println("ac", "AutoCompleteGalRequest: " + str);
			} else {
				DBG.println("ac", "AutoCompleteRequest: " + str);
			}
			params.accountName = account && account.name;

			var search = new ZmSearch(params);
			var searchParams = {
				callback:		this._handleResponseDoAutocomplete.bind(this, str, aclv, options, acType, callback, account),
				errorCallback:	this._handleErrorDoAutocomplete.bind(this, str, aclv),
				timeout:		ZmAutocomplete.AC_TIMEOUT,
				noBusyOverlay:	true
			};
			if (autocompleteType) {
				searchParams.autocompleteType = autocompleteType;
			}
            searchParams.offlineCallback = this._handleOfflineDoAutocomplete.bind(this, str, search, searchParams.callback);
			return search.execute(searchParams);
		};

/**
 * @private
 */
ZmAutocomplete.prototype._handleResponseDoAutocomplete =
		function(str, aclv, options, acType, callback, account, result) {

			DBG.println("ac", "got response for " + str);
			aclv.setWaiting(false);

			var resultList, gotContacts = false, hasGal = false;
			var resp = result.getResponse();
			if (resp && resp.search && resp.search.isGalAutocompleteSearch) {
				var cl = resp.getResults(resp.type);
				resultList = (cl && cl.getArray()) || [];
				gotContacts = hasGal = true;
			} else {
				resultList = resp._respEl.match || [];
			}

			DBG.println("ac", resultList.length + " matches");

			var list = [];
			for (var i = 0; i < resultList.length; i++) {
				var match = new ZmAutocompleteMatch(resultList[i], options, gotContacts, str);
				if (match.acType == acType) {
					if (options.excludeGroups && match.isGroup) continue;
					if (match.type == ZmAutocomplete.AC_TYPE_GAL) {
						hasGal = true;
					}
					list.push(match);
				}
			}
			var complete = !(resp && resp.getAttribute("more"));

			// we assume the results from the server are sorted by ranking
			callback(list);
			this._cacheResults(str, acType, list, hasGal, complete && resp._respEl.canBeCached, null, account);
		};

/**
 * Handle timeout.
 *
 * @private
 */
ZmAutocomplete.prototype._handleErrorDoAutocomplete =
		function(str, aclv, ex) {
			DBG.println("ac", "error on request for " + str + ": " + ex.toString());
			aclv.setWaiting(false);
			appCtxt.setStatusMsg({msg:ZmMsg.autocompleteFailed, level:ZmStatusView.LEVEL_WARNING});

			return true;
		};

/**
 * @private
 */
ZmAutocomplete.prototype._handleOfflineDoAutocomplete =
function(str, search, callback) {
    if (str) {
        var autoCompleteCallback = this._handleOfflineResponseDoAutocomplete.bind(this, search, callback);
        ZmOfflineDB.searchContactsForAutoComplete(str, autoCompleteCallback);
    }
};

ZmAutocomplete.prototype._handleOfflineResponseDoAutocomplete =
function(search, callback, result) {
    var match = [];
    result.forEach(function(contact) {
        var attrs = contact._attrs;
		if (attrs) {
			var obj = {
				id : contact.id,
				l : contact.l
			};
			if (attrs.fullName) {
				var fullName = attrs.fullName;
			}
			else {
				var fullName = [];
				if (attrs.firstName) {
					fullName.push(attrs.firstName);
				}
				if (attrs.middleName) {
					fullName.push(attrs.middleName);
				}
				if (attrs.lastName) {
					fullName.push(attrs.lastName);
				}
				fullName = fullName.join(" ");
			}
			if (attrs.email) {
				obj.email = '"' + fullName + '" <' + attrs.email + '>';
			}
			else if (attrs.type === "group") {
				obj.display = fullName;
				obj.type = ZmAutocomplete.AC_TYPE_CONTACT;
				obj.exp = true;
				obj.isGroup = true;
			}
			match.push(obj);
		}
    });
    if (callback) {
        var zmSearchResult = new ZmSearchResult(search);
        var response = {
            match : match
        };
        zmSearchResult.set(response);
        var zmCsfeResult = new ZmCsfeResult(zmSearchResult);
        callback(zmCsfeResult);
    }
};

/**
 * Sort auto-complete list by ranking scores.
 *
 * @param	{ZmAutocomplete}	a		the auto-complete list
 * @param	{ZmAutocomplete}	b		the auto-complete list
 * @return	{int}	0 if the lists match; 1 if "a" is before "b"; -1 if "b" is before "a"
 */
ZmAutocomplete.acSortCompare =
		function(a, b) {
			var aScore = (a && a.score) || 0;
			var bScore = (b && b.score) || 0;
			return (aScore > bScore) ? 1 : (aScore < bScore) ? -1 : 0;
		};

/**
 * Checks if the given string is a valid email.
 *
 * @param {String}	str		a string
 * @return	{Boolean}	<code>true</code> if a valid email
 */
ZmAutocomplete.prototype.isComplete =
		function(str) {
			return AjxEmailAddress.isValid(str);
		};

/**
 * Asks the server to drop an address from the ranking table.
 *
 * @param {string}	addr		email address
 * @param {closure}	callback	callback to run after response
 */
ZmAutocomplete.prototype.forget =
		function(addr, callback) {

			var jsonObj = {RankingActionRequest:{_jsns:"urn:zimbraMail"}};
			jsonObj.RankingActionRequest.action = {op:"delete", email:addr};
			var respCallback = this._handleResponseForget.bind(this, callback);
			var aCtxt = appCtxt.isChildWindow ? parentAppCtxt : appCtxt;
			aCtxt.getRequestMgr().sendRequest({jsonObj:jsonObj, asyncMode:true, callback:respCallback});
		};

ZmAutocomplete.prototype._handleResponseForget =
		function(callback) {
			appCtxt.clearAutocompleteCache(ZmAutocomplete.AC_TYPE_CONTACT);
			if (appCtxt.isChildWindow) {
				parentAppCtxt.clearAutocompleteCache(ZmAutocomplete.AC_TYPE_CONTACT);
			}
			if (callback) {
				callback();
			}
		};

/**
 * Expands a contact which is a DL and returns a list of its members.
 *
 * @param {ZmContact}	contact		DL contact
 * @param {int}			offset		member to start with (in case we're paging a large DL)
 * @param {closure}		callback	callback to run with results
 */
ZmAutocomplete.prototype.expandDL =
		function(contact, offset, callback) {

			var respCallback = this._handleResponseExpandDL.bind(this, contact, callback);
			contact.getDLMembers(offset, null, respCallback);
		};

ZmAutocomplete.prototype._handleResponseExpandDL =
		function(contact, callback, result) {

			var list = result.list;
			var matches = [];
			if (list && list.length) {
				for (var i = 0, len = list.length; i < len; i++) {
					var addr = list[i];
					var match = {};
					match.type = ZmAutocomplete.AC_TYPE_GAL;
					match.email = addr;
					match.isGroup = result.isDL[addr];
					matches.push(new ZmAutocompleteMatch(match, null, false, contact && contact.str));
				}
			}
			if (callback) {
				callback(matches);
			}
		};

/**
 * @param acType		[constant]			type of result to match
 * @param str			[string]			string to match against
 * @param account		[ZmZimbraAccount]*	account to check cache against
 * @param create		[boolean]			if <code>true</code>, create a cache if none found
 *
 * @private
 */
ZmAutocomplete.prototype._getCache =
		function(acType, str, account, create) {
			var context = AjxEnv.isIE ? window.appCtxt : window.parentAppCtxt || window.appCtxt;
			return context.getAutocompleteCache(acType, str, account, create);
		};

/**
 * @param str			[string]			string to match against
 * @param acType		[constant]			type of result to match
 * @param list			[array]				list of matches
 * @param hasGal		[boolean]*			if true, list includes GAL results
 * @param cacheable		[boolean]*			server indication of cacheability
 * @param baseCache		[hash]*				cache that is superset of this one
 * @param account		[ZmZimbraAccount]*	account to check cache against
 *
 * @private
 */
ZmAutocomplete.prototype._cacheResults =
		function(str, acType, list, hasGal, cacheable, baseCache, account) {

			var cache = this._getCache(acType, str, account, true);
			cache.list = list;
			// we always cache; flag below indicates whether we can do forward matching
			cache.cacheable = (baseCache && baseCache.cacheable) || cacheable;
			if (hasGal) {
				cache.ts = (baseCache && baseCache.ts) || (new Date()).getTime();
			}
		};

/**
 * @private
 *
 * TODO: substring result matching for multiple tokens, eg "tim d"
 */
ZmAutocomplete.prototype._checkCache =
		function(str, acType, account) {

			// check cache for results for this exact string
			var cache = this._getCachedResults(str, acType, null, account);
			var list = cache && cache.list;
			if (list !== null) {
				return list;
			}

			return null;
		};

/**
 * See if we have cached results for the given string. If the cached results have a
 * timestamp, we make sure they haven't expired.
 *
 * @param str				[string]			string to match against
 * @param acType			[constant]			type of result to match
 * @param checkCacheable	[boolean]			if true, make sure results are cacheable
 * @param account			[ZmZimbraAccount]*	account to fetch cached results from
 *
 * @private
 */
ZmAutocomplete.prototype._getCachedResults =
		function(str, acType, checkCacheable, account) {

			var cache = this._getCache(acType, str, account);
			if (cache) {
				if (checkCacheable && (cache.cacheable === false)) {
					return null;
				}
				if (cache.ts) {
					var now = (new Date()).getTime();
					if (now > (cache.ts + ZmAutocomplete.GAL_RESULTS_TTL)) {
						return null;	// expired GAL results
					}
				}
				DBG.println("ac", "cache hit for " + str);
				return cache;
			} else {
				return null;
			}
		};

/**
 * Clears contact autocomplete cache on change to any related setting.
 *
 * @private
 */
ZmAutocomplete.prototype._settingChangeListener =
		function(ev) {
			if (ev.type != ZmEvent.S_SETTING) {
				return;
			}
			var context = AjxEnv.isIE ? window.appCtxt : window.parentAppCtxt || window.appCtxt;
			context.clearAutocompleteCache(ZmAutocomplete.AC_TYPE_CONTACT);
		};


/**
 * Creates an auto-complete match.
 * @class
 * This class represents an auto-complete result, with fields for the caller to look at, and fields to
 * help with further matching.
 *
 * @param {Object}	match		the JSON match object or a {@link ZmContact} object
 * @param {Object}	options		the matching options
 * @param {Boolean}	isContact	if <code>true</code>, provided match is a {@link ZmContact}
 */
ZmAutocompleteMatch = function(match, options, isContact, str) {
	// TODO: figure out how to minimize loading of calendar code
	AjxDispatcher.require(["MailCore", "CalendarCore"]);
	if (!match) {
		return;
	}
	this.type = match.type;
	this.str = str;
	var ac = window.parentAppCtxt || window.appCtxt;
	if (isContact) {
		this.text = this.name = match.getFullName();
		this.email = match.getEmail();
		this.item = match;
		this.type = ZmContact.getAttr(match, ZmResource && ZmResource.F_type || "zimbraCalResType") || ZmAutocomplete.AC_TYPE_GAL;
		this.fullAddress = (new AjxEmailAddress(this.email, null, this.text)).toString(); //bug:60789 formated the email and name to get fullAddress
	} else {
		this.isGroup = Boolean(match.isGroup);
		this.isDL = (this.isGroup && this.type == ZmAutocomplete.AC_TYPE_GAL);
		if (this.isGroup && !this.isDL) {
			// Local contact group; emails need to be looked up by group member ids. 
			var contactGroup = ac.cacheGet(match.id);
			if (contactGroup && contactGroup.isLoaded) {
				this.setContactGroupMembers(match.id);
			}
			else {
				//not a contact group that is in cache.  we'll need to deref it
				this.needDerefGroup = true;
				this.groupId = match.id;
			}
			this.name = match.display;
			this.text = match.display || this.email;
			this.icon = "Group";
		} else {
			// Local contact, GAL contact, or distribution list
			var email = AjxEmailAddress.parse(match.email);
			if (email) {
				this.email = email.getAddress();
				if (this.type == ZmAutocomplete.AC_TYPE_CONTACT) {
					var contactList = AjxDispatcher.run("GetContacts");
					var contact = contactList && contactList.getById(match.id);
					if (contact) {
						var displayName = contact && contact.getFullNameForDisplay(false);
						this.fullAddress = "\"" + displayName + "\" <" + email.getAddress() + ">";
						this.name = displayName;
						this.text = this.fullAddress;
					}
					else {
						//we assume if we don't get contact object, its a shared contact.
						this.fullAddress = email.toString();
						this.name = email.getName();
						this.email = email.getAddress();
						this.text = this.fullAddress;
					}
				} else {
					this.fullAddress = email.toString();
					this.name = email.getName();
					this.text = match.email;
				}
			} else {
				this.email = match.email;
				this.text = match.email;
			}
			if (options && options.needItem && window.ZmContact) {
				this.item = new ZmContact(null);
				this.item.initFromEmail(email || match.email);
			}
			this.icon = this.isDL ? "Group" : ZmAutocomplete.AC_ICON[match.type];
			this.canExpand = this.isDL && match.exp;
			ac.setIsExpandableDL(this.email, this.canExpand);
		}
	}
	this.score = (match.ranking && parseInt(match.ranking)) || 0;
	this.icon = this.icon || ZmAutocomplete.AC_ICON[ZmAutocomplete.AC_TYPE_CONTACT];
	this.acType = (this.type == ZmAutocomplete.AC_TYPE_LOCATION || this.type == ZmAutocomplete.AC_TYPE_EQUIPMENT)
			? this.type : ZmAutocomplete.AC_TYPE_CONTACT;
	if (this.type == ZmAutocomplete.AC_TYPE_LOCATION || this.type == ZmAutocomplete.AC_TYPE_EQUIPMENT) {
		this.icon = ZmAutocomplete.AC_ICON[this.type];
	}
};

/**
 * Sets the email & fullAddress properties of a contact group
 * @param groupId {String} contact group id to lookup from cache
 * @param callback {AjxCallback} callback to be run
 */
ZmAutocompleteMatch.prototype.setContactGroupMembers =
		function(groupId, callback) {
			var ac = window.appCtxt;
			var contactGroup = ac.cacheGet(groupId);
			if (contactGroup) {
				var groups = contactGroup.getGroupMembers();
				var addresses = (groups && groups.good && groups.good.getArray()) || [];
				var emails = [], addrs = [];
				for (var i = 0; i < addresses.length; i++) {
					var addr = addresses[i];
					emails.push(addr.getAddress());
					addrs.push(addr.toString());
				}
				this.email = emails.join(AjxEmailAddress.SEPARATOR);
				this.fullAddress = addrs.join(AjxEmailAddress.SEPARATOR);
			}
			if (callback) {
				callback.run();
			}
		};

/**
 * Returns a string representation of the object.
 *
 * @return		{String}		a string representation of the object
 */
ZmAutocompleteMatch.prototype.toString =
		function() {
			return "ZmAutocompleteMatch";
		};

/**
 * Matches the given string to this auto-complete result.
 *
 * @param {String}	str		the string
 * @return	{Boolean}	<code>true</code> if the given string matches this result
 */
ZmAutocompleteMatch.prototype.matches =
		function(str) {
			if (this.name && !this._nameParsed) {
				var parts = this.name.split(/\s+/, 3);
				var firstName = parts[0];
				this._lastName = parts[parts.length - 1];
				this._firstLast = [firstName, this._lastName].join(" ");
				this._nameParsed = true;
			}

			var fields = [this.email, this.name, this._lastName, this._firstLast];
			for (var i = 0; i < fields.length; i++) {
				var f = fields[i] && fields[i].toLowerCase();
				if (f && (f.indexOf(str) == 0)) {
					return true;
				}
			}
			return false;
		};

/**
 * Creates a search auto-complete.
 * @class
 * This class supports auto-complete for our query language. Each search operator that is supported has an associated handler.
 * A handler is a hash which contains the info needed for auto-complete. A handler can have the following properties:
 *
 * <ul>
 * <li><b>listType</b> - A handler needs a list of objects to autocomplete against. By default, that list is
 *						 identified by the operator. If more than one operator uses the same list, their handlers
 *						 should use this property to identify the list.</li>
 * <li><b>loader</b> - Function that populates the list of objects. Lists used by more than one operator provide
 *						 their loader separately.</li>
 * <li><b>text</b> - Function that returns a string value of data, to autocomplete against and to display in the
 *						 autocomplete list.</li>
 * <li><b>icon</b> - Function that returns an icon to display in the autocomplete list.</li>
 * <li><b>matchText</b> - Function that returns a string to place in the input when the item is selected. Defaults to
 *						 the 'op:' plus the value of the 'text' attribute.</li>
 * <li><b>quoteMatch</b> - If <code>true</code>, the text that goes into matchText will be place in double quotes.</li>
 * </ul>
 *
 */
ZmSearchAutocomplete = function() {

	this._op = {};
	this._list = {};
	this._loadFunc = {};

	var params = {
		loader:		this._loadTags,
		text:		function(o) {
			return o.getName(false, null, true, true);
		},
		icon:		function(o) {
			return o.getIconWithColor();
		},
		matchText:	function(o) {
			return o.createQuery();
		}
	};
	this._registerHandler("tag", params);

	params = {
		listType:	ZmId.ORG_FOLDER,
		text:		function(o) {
			return o.getPath(false, false, null, true, false);
		},
		icon:		function(o) {
			return o.getIconWithColor();
		},
		matchText:	function(o) {
			return o.createQuery();
		}
	};
	this._loadFunc[ZmId.ORG_FOLDER] = this._loadFolders;
	this._registerHandler("in", params);
	params.matchText = function(o) {
		return "under:" + '"' + o.getPath() + '"';
	};
	this._registerHandler("under", params);

	params = { loader:		this._loadFlags };
	this._registerHandler("is", params);

	params = {
		loader:		this._loadObjects,
		icon:		function(o) {
			return ZmSearchAutocomplete.ICON[o];
		}
	};
	this._registerHandler("has", params);

	this._loadFunc[ZmId.ITEM_ATT] = this._loadTypes;
	params = {listType:		ZmId.ITEM_ATT,
		text:			function(o) {
			return o.desc;
		},
		icon:			function(o) {
			return o.image;
		},
		matchText:	function(o) {
			return "type:" + (o.query || o.type);
		},
		quoteMatch:	true
	};
	this._registerHandler("type", params);
	params = {listType:		ZmId.ITEM_ATT,
		text:			function(o) {
			return o.desc;
		},
		icon:			function(o) {
			return o.image;
		},
		matchText:	function(o) {
			return "attachment:" + (o.query || o.type);
		},
		quoteMatch:	true
	};
	this._registerHandler("attachment", params);

	params = {
		loader:		this._loadCommands
	};
	this._registerHandler("set", params);

	var folderTree = appCtxt.getFolderTree();
	if (folderTree) {
		folderTree.addChangeListener(this._folderTreeChangeListener.bind(this));
	}
	var tagTree = appCtxt.getTagTree();
	if (tagTree) {
		tagTree.addChangeListener(this._tagTreeChangeListener.bind(this));
	}
};

ZmSearchAutocomplete.prototype.isZmSearchAutocomplete = true;
ZmSearchAutocomplete.prototype.toString = function() {
	return "ZmSearchAutocomplete";
};

ZmSearchAutocomplete.ICON = {};
ZmSearchAutocomplete.ICON["attachment"] = "Attachment";
ZmSearchAutocomplete.ICON["phone"] = "Telephone";
ZmSearchAutocomplete.ICON["url"] = "URL";

/**
 * @private
 */
ZmSearchAutocomplete.prototype._registerHandler = function(op, params) {

	var loadFunc = params.loader || this._loadFunc[params.listType];
	this._op[op] = {
		loader:     loadFunc.bind(this),
		text:       params.text, icon:params.icon,
		listType:   params.listType || op, matchText:params.matchText || params.text,
		quoteMatch: params.quoteMatch
	};
};

/**
 * Returns a list of matches for a given query operator.
 *
 * @param {String}					str			the string to match against
 * @param {closure}					callback	the callback to run with results
 * @param {ZmAutocompleteListView}	aclv		needed to show wait msg
 * @param {Hash}					options		a hash of additional options
 */
ZmSearchAutocomplete.prototype.autocompleteMatch = function(str, callback, aclv, options) {

	if (ZmSearchAutocomplete._ignoreNextKey) {
		ZmSearchAutocomplete._ignoreNextKey = false;
		return;
	}

	str = str.toLowerCase().replace(/"/g, '');

	var idx = str.lastIndexOf(" ");
	if (idx != -1 && idx <= str.length) {
		str = str.substr(idx + 1);
	}
	var m = str.match(/\b-?\$?([a-z]+):/);
	if (!(m && m.length)) {
		callback();
		return;
	}

	var op = m[1];
	var opHash = this._op[op];
	if (!opHash) {
		callback();
		return;
	}
	var list = this._list[opHash.listType];
	if (list) {
		callback(this._getMatches(op, str));
	} else {
		var respCallback = this._handleResponseLoad.bind(this, op, str, callback);
		this._list[opHash.listType] = [];
		opHash.loader(opHash.listType, respCallback);
	}
};

// TODO - some validation of search ops and args
ZmSearchAutocomplete.prototype.isComplete = function(str, returnStr) {

	var pq = new ZmParsedQuery(str);
	var tokens = pq.getTokens();
	if (!pq.parseFailed && tokens && (tokens.length == 1)) {
		return returnStr ? tokens[0].toString() : true;
	}
	else {
		return false;
	}
};

ZmSearchAutocomplete.prototype.getAddedBubbleClass = function(str) {

	var pq = new ZmParsedQuery(str);
	var tokens = pq.getTokens();
	return (!pq.parseFailed && tokens && (tokens.length == 1) && tokens[0].type);
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._getMatches = function(op, str) {

	var opHash = this._op[op];
	var results = [], app;
	var list = this._list[opHash.listType];
	var rest = str.substr(str.indexOf(":") + 1);
	if (opHash.listType == ZmId.ORG_FOLDER) {
		rest = rest.replace(/^\//, "");	// remove leading slash in folder path
		app = appCtxt.getCurrentAppName();
		if (!ZmApp.ORGANIZER[app]) {
			app = null;
		}
	}
	for (var i = 0, len = list.length; i < len; i++) {
		var o = list[i];
		var text = opHash.text ? opHash.text(o) : o;
		var test = text.toLowerCase();
		if (app && ZmOrganizer.APP[o.type] != app) {
			continue;
		}
		if (!rest || (test.indexOf(rest) == 0)) {
			var matchText = opHash.matchText ? opHash.matchText(o) :
					opHash.quoteMatch ? [op, ":", '"', text, '"'].join("") :
							[op, ":", text].join("");
			matchText = str.replace(op + ":" + rest, matchText);
			results.push({text:			text,
				icon:			opHash.icon ? opHash.icon(o) : null,
				matchText:	matchText,
				exactMatch:	(test.length == rest.length)});
		}
	}

	// no need to show list of one item that is same as what was typed
	if (results.length == 1 && results[0].exactMatch) {
		results = [];
	}

	return results;
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._handleResponseLoad = function(op, str, callback) {
	callback(this._getMatches(op, str));
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._loadTags = function(listType, callback) {

	var list = this._list[listType];
	var tags = appCtxt.getTagTree().asList();
	for (var i = 0, len = tags.length; i < len; i++) {
		var tag = tags[i];
		if (tag.id != ZmOrganizer.ID_ROOT) {
			list.push(tag);
		}
	}
	list.sort(ZmTag.sortCompare);
	if (callback) {
		callback();
	}
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._loadFolders = function(listType, callback) {

	var list = this._list[listType];
	var folderTree = appCtxt.getFolderTree();
	var folders = folderTree ? folderTree.asList({includeRemote:true}) : [];
	for (var i = 0, len = folders.length; i < len; i++) {
		var folder = folders[i];
		if (folder.id !== ZmOrganizer.ID_ROOT && !ZmFolder.HIDE_ID[folder.id] && folder.id !== ZmFolder.ID_DLS) {
			list.push(folder);
		}
	}
	list.sort(ZmFolder.sortComparePath);
	if (callback) {
		callback();
	}
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._loadFlags = function(listType, callback) {

	var flags = AjxUtil.filter(ZmParsedQuery.IS_VALUES, function(flag) {
		return appCtxt.checkPrecondition(ZmParsedQuery.IS_VALUE_PRECONDITION[flag]);
	});
	this._list[listType] = flags.sort();
	if (callback) {
		callback();
	}
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._loadObjects = function(listType, callback) {

	var list = this._list[listType];
	list.push("attachment");
	var idxZimlets = appCtxt.getZimletMgr().getIndexedZimlets();
	if (idxZimlets.length) {
		for (var i = 0; i < idxZimlets.length; i++) {
			list.push(idxZimlets[i].keyword);
		}
	}
	list.sort();
	if (callback) {
		callback();
	}
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._loadTypes = function(listType, callback) {

	AjxDispatcher.require("Extras");
	var attachTypeList = new ZmAttachmentTypeList();
	var respCallback = this._handleResponseLoadTypes.bind(this, attachTypeList, listType, callback);
	attachTypeList.load(respCallback);
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._handleResponseLoadTypes = function(attachTypeList, listType, callback) {

	this._list[listType] = attachTypeList.getAttachments();
	if (callback) {
		callback();
	}
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._loadCommands = function(listType, callback) {

	var list = this._list[listType];
	for (var funcName in ZmClientCmdHandler.prototype) {
		if (funcName.indexOf("execute_") == 0) {
			list.push(funcName.substr(8));
		}
	}
	list.sort();
	if (callback) {
		callback();
	}
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._folderTreeChangeListener = function(ev) {

	var fields = ev.getDetail("fields");
	if (ev.event == ZmEvent.E_DELETE || ev.event == ZmEvent.E_CREATE || ev.event == ZmEvent.E_MOVE ||
			((ev.event == ZmEvent.E_MODIFY) && fields && fields[ZmOrganizer.F_NAME])) {

		var listType = ZmId.ORG_FOLDER;
		if (this._list[listType]) {
			this._list[listType] = [];
			this._loadFolders(listType);
		}
	}
};

/**
 * @private
 */
ZmSearchAutocomplete.prototype._tagTreeChangeListener = function(ev) {

	var fields = ev.getDetail("fields");
	if (ev.event == ZmEvent.E_DELETE || ev.event == ZmEvent.E_CREATE || ev.event == ZmEvent.E_MOVE ||
			((ev.event == ZmEvent.E_MODIFY) && fields && fields[ZmOrganizer.F_NAME])) {

		var listType = "tag";
		if (this._list[listType]) {
			this._list[listType] = [];
			this._loadTags(listType);
		}
	}
};

/**
 * Creates a people search auto-complete.
 * @class
 * This class supports auto-complete for searching the GAL and the user's
 * personal contacts.
 */
ZmPeopleSearchAutocomplete = function() {
	// no need to call ctor
	//	this._acRequests = {};
};

ZmPeopleSearchAutocomplete.prototype = new ZmAutocomplete;
ZmPeopleSearchAutocomplete.prototype.constructor = ZmPeopleSearchAutocomplete;

ZmPeopleSearchAutocomplete.prototype.toString = function() { return "ZmPeopleSearchAutocomplete"; };

ZmPeopleSearchAutocomplete.prototype._doSearch = function(str, aclv, options, acType, callback, account) {

	var params = {
		query: str,
		types: AjxVector.fromArray([ZmItem.CONTACT]),
		sortBy: ZmSearch.NAME_ASC,
		contactSource: ZmId.SEARCH_GAL,
		accountName: account && account.name
	};

	var search = new ZmSearch(params);

	var searchParams = {
		callback:		this._handleResponseDoAutocomplete.bind(this, str, aclv, options, acType, callback, account),
		errorCallback:	this._handleErrorDoAutocomplete.bind(this, str, aclv),
		timeout:		ZmAutocomplete.AC_TIMEOUT,
		noBusyOverlay:	true
	};
	return search.execute(searchParams);
};

/**
 * @private
 */
ZmPeopleSearchAutocomplete.prototype._handleResponseDoAutocomplete = function(str, aclv, options, acType, callback, account, result) {

	// if we get back results for other than the current string, ignore them
	if (str != this._curAcStr) {
		return;
	}

	var resp = result.getResponse();
	var cl = resp.getResults(ZmItem.CONTACT);
	var resultList = (cl && cl.getArray()) || [];
	var list = [];

	for (var i = 0; i < resultList.length; i++) {
		var match = new ZmAutocompleteMatch(resultList[i], options, true);
		list.push(match);
	}
	var complete = !(resp && resp.getAttribute("more"));

	// we assume the results from the server are sorted by ranking
	callback(list);
	this._cacheResults(str, acType, list, true, complete && resp._respEl.canBeCached, null, account);
};
}
if (AjxPackage.define("zimbraMail.core.ZmAppCtxt")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines the application context class.
 *
 */

/**
 * Creates an application context.
 * @class
 * This class is a container for application context information.
 * 
 */
ZmAppCtxt = function() {

	this._trees = {};

	this.accountList = new ZmAccountList();
	// create dummy account for startup
	this.accountList.add(new ZmZimbraAccount(ZmAccountList.DEFAULT_ID, null, false));

	// public properties
	this.inStartup = false;				// true if we are starting app (set in ZmZimbraMail)
	this.currentRequestParams = null;	// params of current SOAP request (set in ZmRequestMgr)
	this.rememberMe = null;
	this.userDomain = "";

	// account-specific
	this.isFamilyMbox = false;
	this.multiAccounts = false;
    this.sendAsEmails = [];
    this.sendOboEmails = [];

	this._evtMgr = new AjxEventMgr();

	this._itemCache			= {};
	this._itemCacheDeferred	= {};
	this._acCache			= {};	// autocomplete
	this._isExpandableDL	= {};	// distribution lists

	this._checkAuthTokenWarning();
};

ZmAppCtxt.ONE_MINUTE  = 60 * 1000;
ZmAppCtxt.MAX_TIMEOUT_VALUE = 2147483647;

ZmAppCtxt._ZIMLETS_EVENT = 'ZIMLETS';
ZmAppCtxt._AUTHTOKEN_EVENT = 'AUTHTOKEN';

//Regex constants
//Bug fix # 79986, #81095. Invalid file names are < > , ? | / \ * :
ZmAppCtxt.INVALID_NAME_CHARS = "[\\|?<>:*\"\\\\\/]";
ZmAppCtxt.INVALID_NAME_CHARS_RE = new RegExp(ZmAppCtxt.INVALID_NAME_CHARS);

/**
 * Returns a string representation of the application context.
 * 
 * @return		{String}		a string representation of the application context
 */
ZmAppCtxt.prototype.toString =
function() {
	return "ZmAppCtxt";
};

ZmAppCtxt.prototype._checkAuthTokenWarning =
function() {
	this._authIntervalId = window.setInterval(this._authTokenWarningTimeout.bind(this), ZmAppCtxt.ONE_MINUTE);
};
ZmAppCtxt.prototype._setAuthTokenWarning =
function(delay) {
    window.setTimeout(this._authTokenWarningTimeout.bind(this), delay);
};

/**
 * Adds a listener to the auth token warning event. This listener is fired once
 * per minute when less than five minutes remain before token expiry.
 *
 * @param	{AjxCallback}	listener		the listener
 * @param	{int}		index		the index to where to add the listener
 * @return	{Boolean}	<code>true</code> if the listener is added; <code>false</code> otherwise
 */
ZmAppCtxt.prototype.addAuthTokenWarningListener =
function(listener, index) {
	return this._evtMgr.addListener(ZmAppCtxt._AUTHTOKEN_EVENT, listener, index);
};

/**
 * Removes a listener for the auth token warning event.
 *
 * @param	{AjxCallback}	listener		the listener
 * @return	{Boolean}	<code>true</code> if the listener is removed; <code>false</code> otherwise
 */
ZmAppCtxt.prototype.removeAuthTokenWarningListener =
function(listener) {
	return this._evtMgr.removeListener(ZmAppCtxt._AUTHTOKEN_EVENT, listener);
};

ZmAppCtxt.prototype._authTokenWarningTimeout =
function () {

	if (!window.authTokenExpires) {
		return; //for cases we the auth token expires is not available. (e.g. some new windows we didn't set it for yet, or for saved rest URLs
	}

	var now = new Date().getTime();
	var millisToLive = window.authTokenExpires - now;
    var minutesToLive = Math.round(millisToLive / ZmAppCtxt.ONE_MINUTE);
    var delay;

	if (minutesToLive > 5 || millisToLive <= 0) {
        // Outside the times to issue warnings
        if (minutesToLive === 6) {
            // Line up the timer to go off at exactly 5 minutes (or as exact as we can make it), which is
            // when we start issuing warnings
            window.clearInterval(this._authIntervalId);
            delay = millisToLive - (5 * ZmAppCtxt.ONE_MINUTE);
            this._setAuthTokenWarning(delay);
        }
		return;
	}

	if (this._evtMgr.isListenerRegistered(ZmAppCtxt._AUTHTOKEN_EVENT)) {
		var event = new ZmEvent(ZmAppCtxt._AUTHTOKEN_EVENT);
		this._evtMgr.notifyListeners(ZmAppCtxt._AUTHTOKEN_EVENT, event);
	}

	var msg;
    var decaSecondsToLive = 0;
    var toastDuration;
    if (minutesToLive > 1) {
        msg = AjxMessageFormat.format(ZmMsg.authTokenExpirationWarning, [minutesToLive, ZmMsg.minutes]);
        toastDuration = ZmAppCtxt.ONE_MINUTE / 4;
    } else {
        // Get the number of 10-second intervals remaining - used once we are within 1 minute
        decaSecondsToLive =  Math.round(millisToLive / 10000);
        toastDuration = 8000;
        if (decaSecondsToLive >= 6) {
            // 1 minute+ to go.  But should be pretty close to 1 minute
            msg = AjxMessageFormat.format(ZmMsg.authTokenExpirationWarning, [1, ZmMsg.minute]);
        } else {
            // Seconds remain
            msg = AjxMessageFormat.format(ZmMsg.authTokenExpirationWarning, [decaSecondsToLive * 10, ZmMsg.seconds]);
        }
    }

	var params = {
		msg:    msg,
		level:  ZmStatusView.LEVEL_WARNING,
		transitions: [{type: "fade-in", duration: 500}, {type: "pause", duration: toastDuration}, {type: "fade-out", duration: 500} ]
	};
	this.setStatusMsg(params);

    if (minutesToLive > 1) {
        var floorMinutesToLive = Math.floor(millisToLive / ZmAppCtxt.ONE_MINUTE);
        if (floorMinutesToLive === minutesToLive) {
            floorMinutesToLive--;
        }
        delay = millisToLive - (floorMinutesToLive * ZmAppCtxt.ONE_MINUTE);
    }  else {
        decaSecondsToLive--;
        delay = millisToLive - (decaSecondsToLive * 10000);
    }
    if (delay > 0) {
        this._setAuthTokenWarning(delay);
    }
};

ZmAppCtxt.prototype.setZimbraMail = function(zimbraMail) {
	this._zimbraMail = zimbraMail;
};

ZmAppCtxt.prototype.getZimbraMail = function() {
	return this._zimbraMail;
};

/**
 * Sets the application controller.
 * 
 * @param	{ZmController}	appController	the controller
 */
ZmAppCtxt.prototype.setAppController =
function(appController) {
	this._appController = appController;
};

/**
 * Gets the application controller.
 * 
 * @return	{ZmController}		the controller
 */
ZmAppCtxt.prototype.getAppController =
function() {
	return this._appController;
};

/**
 * Gets the application chooser.
 * 
 * @return	{ZmAppChooser}		the chooser
 */
ZmAppCtxt.prototype.getAppChooser =
function() {
	return this._appController.getAppChooser();
};

/**
 * Sets the request manager.
 * 
 * @param	{ZmRequestMgr}	requestMgr	the request manager
 */
ZmAppCtxt.prototype.setRequestMgr =
function(requestMgr) {
	this._requestMgr = requestMgr;
};

/**
 * Gets the request manager.
 * 
 * @return	{ZmRequestMgr}		the request manager
 */
ZmAppCtxt.prototype.getRequestMgr =
function() {
	return this._requestMgr;
};

/**
 * Sets the status message to display.
 * 
 * 
 * @param {Hash}	params	a hash of parameters
 * @param	{String}	params.msg 		the status message
 * @param	{constant}	params.level	the status level {@link ZmStatusView}  (may be <code>null</code>)
 * @param	{String}	params.detail 	the details (may be <code>null</code>)
 * @param	{Object}	params.transitions 	the transitions (may be <code>null</code>)
 * @param	{Object}	params.toast	the toast control (may be <code>null</code>)
 * @param   {boolean}   params.force    force any displayed toasts out of the way (dismiss them and run their dismissCallback). Enqueued messages that are not yet displayed will not be displayed
 * @param   {AjxCallback} params.dismissCallback    callback to run when the toast is dismissed (by another message using [force], or explicitly calling ZmStatusView.prototype.dismiss())
 * @param   {AjxCallback} params.finishCallback     callback to run when the toast finishes its transitions by itself (not when dismissed)
 * </ul>
 * 
 */
ZmAppCtxt.prototype.setStatusMsg =
function(params) {
	params = Dwt.getParams(arguments, ZmStatusView.MSG_PARAMS);
	this._appController.setStatusMsg(params);
};

/**
 * Dismisses the displayed status message, if any
 */

ZmAppCtxt.prototype.dismissStatusMsg =
function(all) {
	this._appController.dismissStatusMsg(all);
};

/**
 * Gets the settings for the given account.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{ZmSettings}	the settings
 */
ZmAppCtxt.prototype.getSettings =
function(account) {
	var al = this.accountList;

	var acct = account || al.activeAccount || al.mainAccount
			|| al.getAccount(ZmAccountList.DEFAULT_ID); //Probably doesn't ever happen, and if it does, returns null. Might be some historical artifact - did we ever have account with id "main"? I'm still afraid to remove it without being sure it won't cause regression.

	return acct && acct.settings;
};

/**
 * Sets the settings for the given account.
 * 
 * @param	{ZmSettings}	settings		the settings
 * @param	{ZmZimbraAccount}		account			the account
 */
ZmAppCtxt.prototype.setSettings = 
function(settings, account) {
	var al = this.accountList;
	var id = account
		? account.id
		: al.activeAccount ? al.activeAccount.id : ZmAccountList.DEFAULT_ID;

	var acct = al.getAccount(id);
	if (acct) {
		acct.settings = settings;
	}
};

/**
 * Gets the value of the given setting.
 *
 * @param {constant}	id		the setting id
 * @param {String}	key			the setting key (for settings that are of the hash type)
 * @param {ZmZimbraAccount}	account		the account
 * @return	{Object}		the setting value
 */
ZmAppCtxt.prototype.get = function(id, key, account) {

    //use parentAppCtxt in case of new window
    var context = this.isChildWindow ? parentAppCtxt : this;

	// for offline, global settings always come from the "local" parent account
	var acct = (context.multiAccounts && ZmSetting.IS_GLOBAL[id])
		? context.accountList.mainAccount : account;
	return context.getSettings(acct).get(id, key);
};

/**
 * Sets the value of the given setting.
 *
 * @param {constant}	id					the setting id
 * @param {Object}	value					the setting value
 * @param {String}	key					the setting key (for settings that are of the hash type)
 * @param {Boolean}	setDefault			if <code>true</code>, also replace setting default value
 * @param {Boolean}	skipNotify			if <code>true</code>, do not notify setting listeners
 * @param {ZmZimbraAccount}	account		if set, use this account setting instead of the currently active account
 * @param {Boolean}	skipImplicit		if <code>true</code>, do not check for change to implicit pref
 */
ZmAppCtxt.prototype.set =
function(id, value, key, setDefault, skipNotify, account, skipImplicit) {
	// for offline, global settings always come from "parent" account
	var acct = (this.multiAccounts && ZmSetting.IS_GLOBAL[id])
		? this.accountList.mainAccount : account;
	var setting = this.getSettings(acct).getSetting(id);

	if (setting) {
		setting.setValue(value, key, setDefault, skipNotify, skipImplicit);
	}
};

/**
 * Gets the application.
 * 
 * @param	{String}	appName		the application name
 * @return	{ZmApp}	the application or <code>null</code> if not found
 */
ZmAppCtxt.prototype.getApp =
function(appName) {
	return this._appController.getApp(appName);
};

/**
 * Gets the name of the current application.
 * 
 * @return	{String}		the application name
 */
ZmAppCtxt.prototype.getCurrentAppName =
function() {
	var context = this.isChildWindow ? parentAppCtxt : this;
	return context._appController.getActiveApp();
};
/**
 *
 */
ZmAppCtxt.prototype.getLoggedInUsername =
function() {
	return appCtxt.get(ZmSetting.USERNAME);
};

/**
 * Gets the current application.
 * 
 * @return	{ZmApp}		the current application
 */
ZmAppCtxt.prototype.getCurrentApp =
function() {
	return this.getApp(this.getCurrentAppName());
};

/**
 * Gets the application view manager.
 * 
 * @return	{ZmAppViewMgr}		the view manager
 */
ZmAppCtxt.prototype.getAppViewMgr =
function() {
	return this._appController.getAppViewMgr();
};

/**
 * Gets the client command handler.
 * 
 * @param	{ZmClientCmdHandler}	clientCmdHdlr		not used
 * @return	{ZmClientCmdHandler}		the command handler
 */
ZmAppCtxt.prototype.getClientCmdHandler =
function(clientCmdHdlr) {
	if (!this._clientCmdHandler) {
		AjxDispatcher.require("Extras");
		this._clientCmdHandler = new ZmClientCmdHandler();
	}
	return this._clientCmdHandler;
};

/**
 * Gets the search bar controller.
 * 
 * @return	{ZmSearchController}	the search controller
 */
ZmAppCtxt.prototype.getSearchController =
function() {
	if (!this._searchController) {
		this._searchController = new ZmSearchController(this._shell);
	}
	return this._searchController;
};

/**
 * Gets the overview controller. Creates a new one if not already set, unless dontCreate is true. 
 *
 * @param {boolean} dontCreate (optional) - don't create overviewController if not created already. (see ZmApp for usage with dontCreate == true)
 * 
 * @return	{ZmOverviewController}	the overview controller
 */
ZmAppCtxt.prototype.getOverviewController =
function(dontCreate) {
	if (!this._overviewController) {
		if (dontCreate) {
			return null;
		}
		this._overviewController = new ZmOverviewController(this._shell);
	}
	return this._overviewController;
};

/**
 * Gets the import/export controller.
 * 
 * @return	{ZmImportExportController}	the controller
 */
ZmAppCtxt.prototype.getImportExportController = function() {
	if (!this._importExportController) {
		AjxDispatcher.require("ImportExport");
		this._importExportController = new ZmImportExportController();
	}
	return this._importExportController;
};

/**
 * Gets the message dialog.
 * 
 * @return	{DwtMessageDialog}	the message dialog
 */
ZmAppCtxt.prototype.getMsgDialog =
function() {
	if (!this._msgDialog) {
		this._msgDialog = new DwtMessageDialog({parent:this._shell, id: "ZmMsgDialog"});
	}
	return this._msgDialog;
};

/**
 * Gets the message dialog with a help button.
 *
 * @return	{DwtMessageDialog}	the message dialog
 */
ZmAppCtxt.prototype.getHelpMsgDialog =
	function() {
		if (!this._helpMsgDialog) {
			this._helpMsgDialog = new DwtMessageDialog({parent:this._shell, helpText:ZmMsg.help, id: "ZmHelpMsgDialog"});
		}
		return this._helpMsgDialog;
	};

/**
 * Gets the yes/no message dialog.
 * 
 * @return	{DwtMessageDialog}	the message dialog
 */
ZmAppCtxt.prototype.getYesNoMsgDialog =
function(id) {
	if (!this._yesNoMsgDialog) {
		this._yesNoMsgDialog = new DwtMessageDialog({parent:this._shell, buttons:[DwtDialog.YES_BUTTON, DwtDialog.NO_BUTTON], id: "YesNoMsgDialog"});
	}	
	return this._yesNoMsgDialog;
};

/**
 * Gets the yes/no/cancel message dialog.
 * 
 * @return	{DwtMessageDialog}	the message dialog
 */
ZmAppCtxt.prototype.getYesNoCancelMsgDialog =
function() {
	if (!this._yesNoCancelMsgDialog) {
		this._yesNoCancelMsgDialog = new DwtMessageDialog({parent:this._shell, buttons:[DwtDialog.YES_BUTTON, DwtDialog.NO_BUTTON, DwtDialog.CANCEL_BUTTON], id:"YesNoCancel"});
	}	
	return this._yesNoCancelMsgDialog;
};

/**
 * Gets the ok/cancel message dialog.
 * 
 * @return	{DwtMessageDialog}	the message dialog
 */
ZmAppCtxt.prototype.getOkCancelMsgDialog =
function() {
	if (!this._okCancelMsgDialog) {
		this._okCancelMsgDialog = new DwtMessageDialog({parent:this._shell, buttons:[DwtDialog.OK_BUTTON, DwtDialog.CANCEL_BUTTON], id:"OkCancel"});
	}	
	return this._okCancelMsgDialog;
};

/**
 * Gets the cancel message dialog.
 * 
 * @return	{DwtMessageDialog}	the message dialog
 */
ZmAppCtxt.prototype.getCancelMsgDialog =
function() {
	if (!this._cancelMsgDialog) {
		this._cancelMsgDialog = new DwtMessageDialog({parent:this._shell, buttons:[DwtDialog.CANCEL_BUTTON]});
	}
	return this._cancelMsgDialog;
};

/**
 * Gets the error dialog.
 * 
 * @return	{ZmErrorDialog}	the error dialog
 */
ZmAppCtxt.prototype.getErrorDialog = 
function() {
	if (!this._errorDialog) {
		AjxDispatcher.require("Startup2");
		this._errorDialog = new ZmErrorDialog(this._shell, ZmMsg);
	}
	return this._errorDialog;
};

/**
 * Gets the new tag dialog.
 * 
 * @return	{ZmNewTagDialog}	the new tag dialog
 */
ZmAppCtxt.prototype.getNewTagDialog =
function() {
	if (!this._newTagDialog) {
		this._newTagDialog = new ZmNewTagDialog(this._shell);
	}
	return this._newTagDialog;
};

/**
 * Gets the new contact group dialog.
 *
 * @return	{ZmNewContactGroupDialog}	the new contact group dialog
 */
ZmAppCtxt.prototype.getNewContactGroupDialog =
function() {
	if (!this._newContactGroupDialog) {
		this._newContactGroupDialog = new ZmNewContactGroupDialog(this._shell);
	}
	return this._newContactGroupDialog;
};

/**
 * Gets the rename tag dialog.
 * 
 * @return	{ZmRenameTagDialog}		the rename tag dialog
 */
ZmAppCtxt.prototype.getRenameTagDialog =
function() {
	if (!this._renameTagDialog) {
		AjxDispatcher.require("Extras");
		this._renameTagDialog = new ZmRenameTagDialog(this._shell);
	}
	return this._renameTagDialog;
};

/**
 * Gets the password update dialog.
 *
 * @return	{ZmPasswordUpdateDialog}		the rename tag dialog
 */
ZmAppCtxt.prototype.getPasswordChangeDialog =
function() {
	if (!this._passwordUpdateDialog) {
		AjxDispatcher.require("Extras");
		this._passwordUpdateDialog = new ZmPasswordUpdateDialog(this._shell);
	}
	return this._passwordUpdateDialog;
};

/**
 * Gets the new folder dialog.
 * 
 * @return	{ZmNewFolderDialog}		the new folder dialog
 */
ZmAppCtxt.prototype.getNewFolderDialog =
function() {
	if (!this._newFolderDialog) {
        var title = ZmMsg.createNewFolder;
        var type = ZmOrganizer.FOLDER;
        this._newFolderDialog = new ZmNewOrganizerDialog(this._shell, null, title, type)
	}
	return this._newFolderDialog;
};

/**
 * Gets the new address book dialog.
 * 
 * @return	{ZmNewAddrBookDialog}		the new address book dialog
 */
ZmAppCtxt.prototype.getNewAddrBookDialog = 
function() {
	if (!this._newAddrBookDialog) {
		AjxDispatcher.require("Contacts");
		this._newAddrBookDialog = new ZmNewAddrBookDialog(this._shell);
	}
	return this._newAddrBookDialog;
};

/**
 * Gets the new calendar dialog.
 * 
 * @return	{ZmNewCalendarDialog}		the new calendar dialog
 */
ZmAppCtxt.prototype.getNewCalendarDialog =
function() {
	if (!this._newCalendarDialog) {
		AjxDispatcher.require(["MailCore", "CalendarCore", "Calendar", "CalendarAppt"]);
		this._newCalendarDialog = new ZmNewCalendarDialog(this._shell);
	}
	return this._newCalendarDialog;
};

/**
 * Gets the new task folder dialog.
 * 
 * @return	{ZmNewTaskFolderDialog}		the new task folder dialog
 */
ZmAppCtxt.prototype.getNewTaskFolderDialog =
function() {
	if (!this._newTaskFolderDialog) {
		AjxDispatcher.require(["TasksCore", "Tasks"]);
		this._newTaskFolderDialog = new ZmNewTaskFolderDialog(this._shell);
	}
	return this._newTaskFolderDialog;
};

/**
 * Gets the new suggestion Preferences dialog
 *
 * @return	{ZmTimeSuggestionPrefDialog}
 */
ZmAppCtxt.prototype.getSuggestionPreferenceDialog =
function() {
	if (!this._suggestionPrefDialog) {
		AjxDispatcher.require(["MailCore", "CalendarCore", "Calendar"]);
        this._suggestionPrefDialog = new ZmTimeSuggestionPrefDialog(this._shell);
    }
    return this._suggestionPrefDialog;
};

/**
 * Gets the dialog.
 * 
 * @return	{DwtDialog}		the dialog
 */
ZmAppCtxt.prototype.getDialog =
function(){
	if(!this._dialog){
		this._dialog = new DwtDialog({parent:this._shell});
	}
	return this._dialog;
};

/**
 * Gets the new search dialog.
 * 
 * @return	{ZmNewSearchDialog}		the new search dialog
 */
ZmAppCtxt.prototype.getNewSearchDialog =
function() {
	this._newSearchDialogs = this._newSearchDialogs || {};
	this.searchAppName = this.searchAppName || ZmApp.MAIL;
	if (!this._newSearchDialogs[this.searchAppName]) {
		this._newSearchDialogs[this.searchAppName] = new ZmNewSearchDialog(this._shell);
	}
	this._newSearchDialog = this._newSearchDialogs[this.searchAppName];
	return this._newSearchDialog;
};

/**
 * Gets the rename folder dialog.
 * 
 * @return	{ZmRenameFolderDialog}		the rename folder dialog
 */
ZmAppCtxt.prototype.getRenameFolderDialog =
function() {
	if (!this._renameFolderDialog) {
		AjxDispatcher.require("Extras");
		this._renameFolderDialog = new ZmRenameFolderDialog(this._shell);
	}
	return this._renameFolderDialog;
};

/**
 * Gets the choose folder dialog.
 * 
 * @return	{ZmChooseFolderDialog}		the choose folder dialog
 */
ZmAppCtxt.prototype.getChooseFolderDialog =
function(appName) {
	appCtxt.notifyZimlets("onZmAppCtxt_getChooseFolderDialog1", [this]);

	var app = appName ? this.getApp(appName) : this.getCurrentApp();
	// this.getCurrentAppName() returns "Search" for search apps. Let's re-use dialogs from regular apps.
	appName = app.isZmSearchApp ? this.searchAppName : app.getName();
	this._chooseFolderDialogs = this._chooseFolderDialogs || {};
	if (!this._chooseFolderDialogs[appName]) {
		AjxDispatcher.require("Extras");
		this._chooseFolderDialogs[appName] = new ZmChooseFolderDialog(this._shell);
	}

	appCtxt.notifyZimlets("onZmAppCtxt_getChooseFolderDialog2", [this]);

	this._chooseFolderDialog = this._chooseFolderDialogs[appName];
	return this._chooseFolderDialog;
};

ZmAppCtxt.prototype.getChooseAccountDialog =
function() {
	if (!this._chooseAccountDialog) {
		AjxDispatcher.require("Extras");
		this._chooseAccountDialog = new ZmChooseAccountDialog(this._shell);
	}
	return this._chooseAccountDialog;
};

/**
 * Gets the pick tag dialog.
 * 
 * @return	{ZmPickTagDialog}		the pick tag dialog
 */
ZmAppCtxt.prototype.getPickTagDialog =
function() {
	if (!this._pickTagDialog) {
		AjxDispatcher.require("Extras");
		this._pickTagDialog = new ZmPickTagDialog(this._shell);
	}
	return this._pickTagDialog;
};

/**
 * Gets the folder notify dialog.
 * 
 * @return	{ZmFolderNotifyDialog}		the folder notify dialog
 */
ZmAppCtxt.prototype.getFolderNotifyDialog =
function() {
	if (!this._folderNotifyDialog) {
		this._folderNotifyDialog = new ZmFolderNotifyDialog(this._shell);
	}
	return this._folderNotifyDialog;
};

/**
 * Gets the folder properties dialog.
 * 
 * @return	{ZmFolderPropsDialog}		the folder properties dialog
 */
ZmAppCtxt.prototype.getFolderPropsDialog =
function() {
	if (!this._folderPropsDialog) {
		this._folderPropsDialog = new ZmFolderPropsDialog(this._shell);
	}
	return this._folderPropsDialog;
};

/**
 * Gets the share properties dialog.
 * 
 * @return	{ZmSharePropsDialog}		the share properties dialog
 */
ZmAppCtxt.prototype.getSharePropsDialog =
function() {
	if (!this._sharePropsDialog) {
		AjxDispatcher.require("Share");
		this._sharePropsDialog = new ZmSharePropsDialog(this._shell);
	}
	return this._sharePropsDialog;
};

ZmAppCtxt.prototype.getShareSearchDialog = function() {
	if (!this._shareSearchDialog) {
		AjxDispatcher.require("Share");
		this._shareSearchDialog = new ZmShareSearchDialog({parent:this._shell});
	}
	return this._shareSearchDialog;
};

/**
 * Gets the accept share dialog.
 * 
 * @return	{ZmAcceptShareDialog}		the accept share dialog
 */
ZmAppCtxt.prototype.getAcceptShareDialog =
function() {
	if (!this._acceptShareDialog) {
		AjxDispatcher.require("Share");
		this._acceptShareDialog = new ZmAcceptShareDialog(this._shell);
	}
	return this._acceptShareDialog;
};

/**
 * Gets the decline share dialog.
 * 
 * @return	{ZmDeclineShareDialog}		the decline share dialog
 */
ZmAppCtxt.prototype.getDeclineShareDialog =
function() {
	if (!this._declineShareDialog) {
		AjxDispatcher.require("Share");
		this._declineShareDialog = new ZmDeclineShareDialog(this._shell);
	}
	return this._declineShareDialog;
};

/**
 * Gets the revoke share dialog.
 * 
 * @return	{ZmRevokeShareDialog}		the revoke share dialog
 */
ZmAppCtxt.prototype.getRevokeShareDialog =
function() {
	if (!this._revokeShareDialog) {
		AjxDispatcher.require("Share");
		this._revokeShareDialog = new ZmRevokeShareDialog(this._shell);
	}
	return this._revokeShareDialog;
};

/**
 * Gets the timezone picker dialog.
 * 
 * @return	{ZmTimezonePicker}		the timezone picker dialog
 */
ZmAppCtxt.prototype.getTimezonePickerDialog =
function() {
	if (!this._timezonePickerDialog) {
		AjxDispatcher.require("Share");
		this._timezonePickerDialog = new ZmTimezonePicker(this._shell);
	}
	return this._timezonePickerDialog;
};

/**
 * Gets the filter rule add/edit dialog.
 * 
 * @return	{ZmFilterRuleDialog}		the filter rule add/edit dialog
 */
ZmAppCtxt.prototype.getFilterRuleDialog =
function() {
	if (!this._filterRuleDialog) {
		AjxDispatcher.require(["PreferencesCore", "Preferences"]);
		this._filterRuleDialog = new ZmFilterRuleDialog();
	}
	return this._filterRuleDialog;
};

/**
 * Gets the priority message filter dialog.
 * 
 * @return {ZmPriorityMessageFilterDialog}  the priority message filter dialog
 */
ZmAppCtxt.prototype.getPriorityMessageFilterDialog = 
function() {
	if (!this._priorityMessageFilterDialog) {
		AjxDispatcher.require(["PreferencesCore", "Preferences"]);
		this._priorityMessageFilterDialog = new ZmPriorityMessageFilterDialog();
	}
	return this._priorityMessageFilterDialog;
};


/**
 * Gets the activity stream prompt dialog for running activity stream filters
 * 
 * @return {ZmActivityStreamPromptDialog}
*/
ZmAppCtxt.prototype.getActivityStreamFilterDialog = 
function() {
	if (!this._activityStreamFilterDialog) {
		AjxDispatcher.require(["PreferencesCore", "Preferences"]);
		this._activityStreamFilterDialog = new ZmActivityStreamPromptDialog();
	}
	return this._activityStreamFilterDialog;
};

/**
 * Gets the prompt for moving files from the Activity Stream to the Inbox
 * 
 * @return {ZmActivityToInboxPromptDialog}
 */
ZmAppCtxt.prototype.getActivityToInboxFilterDialog =
function() {
	if (!this._activityToInboxFilterDialog) {
		AjxDispatcher.require(["PreferencesCore", "Preferences"]);
		this._activityToInboxFilterDialog = new ZmActivityToInboxPromptDialog();
	}
	return this._activityToInboxFilterDialog;
};

/**
 * Gets the quickadd dialog for creating a contact
 * 
 * @return {ZmContactQuickAddDialog}
 */
ZmAppCtxt.prototype.getContactQuickAddDialog = 
function() {
	if (!this._contactQuickAddDialog) {
		AjxDispatcher.require(["ContactsCore", "Contacts"]);
		this._contactQuickAddDialog = new ZmContactQuickAddDialog();
	}
	return this._contactQuickAddDialog;
};

/**
 * Gets the confirm dialog.
 * 
 * @return	{DwtConfirmDialog}		the confirmation dialog
 */
ZmAppCtxt.prototype.getConfirmationDialog =
function(id) {
	if (!this._confirmDialog) {
		this._confirmDialog = new DwtConfirmDialog(this._shell, null, "CONFIRM_DIALOG");
	}
	return this._confirmDialog;
};

/**
 * Gets the upload dialog.
 * 
 * @return	{ZmUploadDialog}		the upload dialog
 */
ZmAppCtxt.prototype.getUploadDialog =
function() {
	if (!this._uploadDialog) {
		AjxDispatcher.require(["Extras"]);
		this._uploadDialog = new ZmUploadDialog(this._shell);
	}
	return this._uploadDialog;
};

/**
 * Gets the attach dialog.
 * 
 * @return	{ZmAttachDialog}		the attach dialog
 */
ZmAppCtxt.prototype.getAttachDialog =
function() {
	if (!this._attachDialog) {
		AjxDispatcher.require("Share");
		this._attachDialog = new ZmAttachDialog(this._shell);
		this.runAttachDialogCallbacks();
	}
	return this._attachDialog;
};

ZmAppCtxt.prototype.getDumpsterDialog =
function() {
	if (!this._dumpsterDialog) {
		AjxDispatcher.require("Extras");
		this._dumpsterDialog = new ZmDumpsterDialog(this._shell);
	}
	return this._dumpsterDialog;
};


/**
 * Gets the mail redirect dialog.
 *
 * @return	{ZmMailRedirectDialog}	the new mail redirect dialog
 */
ZmAppCtxt.prototype.getMailRedirectDialog =
function() {
	if (!this._mailRedirectDialog) {
		this._mailRedirectDialog = new ZmMailRedirectDialog(this._shell);
	}
	return this._mailRedirectDialog;
};

/**
 * Gets the mail retention warning dialog.
 *
 * @return	{ZmRetetionWarningDialog}	the new mail retention warning dialog
 */
ZmAppCtxt.prototype.getRetentionWarningDialog =
function() {
	if (!this._retentionWarningDialog) {
		this._retentionWarningDialog = new ZmRetentionWarningDialog(this._shell);
	}
	return this._retentionWarningDialog;
};


/**
 * Runs the attach dialog callbacks.
 *
 * @private
 */
ZmAppCtxt.prototype.runAttachDialogCallbacks =
function() {
	while(this._attachDialogCallback && this._attachDialogCallback.length > 0) {
		var callback = this._attachDialogCallback.shift();
		if(callback && (callback instanceof AjxCallback)) {
			callback.run(this._attachDialog);
		}
	}
};

/**
 * Adds the callback to the attachment dialog callbacks.
 *
 * @param	{AjxCallback}	callback		the callback
 */
ZmAppCtxt.prototype.addAttachmentDialogCallback =
function(callback) {
	if(!this._attachDialogCallback) {
		this._attachDialogCallback = [];
	}
	this._attachDialogCallback.push(callback);
};                                              

/**
 * Gets the upload conflict dialog.
 *
 * @return	{ZmUploadConflictDialog}	the upload conflict dialog
 */
ZmAppCtxt.prototype.getUploadConflictDialog =
function() {
	if (!this._uploadConflictDialog) {
		AjxDispatcher.require(["Extras"]);
		this._uploadConflictDialog = new ZmUploadConflictDialog(this._shell);
	}
	return this._uploadConflictDialog;
};

/**
 * Gets the new briefcase dialog.
 *
 * @return	{ZmNewBriefcaseDialog}	the new briefcase dialog
 */
ZmAppCtxt.prototype.getNewBriefcaseDialog =
function() {
	if (!this._newBriefcaseDialog) {
		AjxDispatcher.require(["BriefcaseCore", "Briefcase"]);
		this._newBriefcaseDialog = new ZmNewBriefcaseDialog(this._shell);
	}
	return this._newBriefcaseDialog;
};

/**
 * Gets the find-and-replace dialog.
 *
 * @return	{ZmFindnReplaceDialog}	the find-and-replace dialog
 */
ZmAppCtxt.prototype.getReplaceDialog =
function() {
	if (!this._replaceDialog) {
		AjxDispatcher.require("Share");
		this._replaceDialog = new ZmFindnReplaceDialog(this._shell);
	}
	return this._replaceDialog;
};

/**
 * Gets the debug log dialog.
 *
 * @return	{ZmDebugLogDialog}		the debug log dialog
 */
ZmAppCtxt.prototype.getDebugLogDialog =
function() {
	if (!this._debugLogDialog) {
		AjxDispatcher.require("Extras");
		this._debugLogDialog = new ZmDebugLogDialog(this._shell);
	}
	return this._debugLogDialog;
};

/**
 * Gets the root tab group.
 *
 * @return	{DwtTabGroup}	the root tab group
 */
ZmAppCtxt.prototype.getRootTabGroup =
function() {
	if (this.isChildWindow) {
		if (!this._childWinTabGrp) {
			this._childWinTabGrp = new DwtTabGroup("CHILD_WINDOW");
		}
	} else {		
		if (!this._rootTabGrp) {
			this._rootTabGrp = new DwtTabGroup("ROOT");
		}
	}
	return this.isChildWindow ? this._childWinTabGrp : this._rootTabGrp;
};

/**
 * Gets the shell.
 *
 * @return	{DwtShell}	the shell
 */
ZmAppCtxt.prototype.getShell =
function() {
	return this._shell;
};

/**
 * Sets the shell.
 *
 * @param	{DwtShell}	the shell
 */
ZmAppCtxt.prototype.setShell =
function(shell) {
	this._shell = shell;
};

/**
 * Gets the active account.
 *
 * @return	{ZmZimbraAccount}	the active account
 */
ZmAppCtxt.prototype.getActiveAccount =
function() {
	return this.isChildWindow
		? parentAppCtxt.accountList.activeAccount
		: this.accountList.activeAccount;
};

/**
 * Gets the active account.
 *
 * @return	{ZmZimbraAccount}	the active account
 */
ZmAppCtxt.prototype.isExternalAccount =
function() {
	return this.get(ZmSetting.IS_EXTERNAL);
};

/*
 * This is a list of Aspell (Ver. 0.61) support locale from the result of the following command:
 *   /opt/zimbra/aspell/bin/aspell dump dicts
 *      (use only the items whose format is "<Primary-tag> *( "_" <Subtag> )")
 * When Aspell is upgraded and more locales are added, please update this list too.
 */
ZmAppCtxt.AVAILABLE_DICTIONARY_LOCALES = ["ar", "ca", "da", "de", "de_AT", "de_CH", "de_DE", "en", "en_CA", "en_GB", "en_US", "es", "fr", "fr_CH", "fr_FR", "hi", "hu", "it", "nl", "pl", "pt_BR", "ru", "sv"];

/**
 * Gets the availability of the spell check feature based on the current locale and user's configuration
 *
 * @return     {Boolean}       <code>true</code> if the spell checker is available.
 */
ZmAppCtxt.prototype.isSpellCheckerAvailable = function () {

	var isSpellCheckServiceEnabled = appCtxt.get(ZmSetting.SPELL_CHECK_ENABLED) && !appCtxt.get(ZmSetting.OFFLINE_ENABLED) && (!AjxEnv.isSafari || AjxEnv.isSafari3up || AjxEnv.isChrome);
	if (!isSpellCheckServiceEnabled) {
		return false;
	}

	if (typeof this._spellCheckAvailable !== 'undefined') {
		return this._spellCheckAvailable;
	}

	this._spellCheckAvailable = false;
	var myLocale = appCtxt.get(ZmSetting.LOCALE_NAME);
	var myDefaultDictionaryName = appCtxt.get(ZmSetting.SPELL_DICTIONARY);

	var myLanguage = myLocale.split('_')[0];

	var dictLocales = ZmAppCtxt.AVAILABLE_DICTIONARY_LOCALES;
	var ln = dictLocales.length;

	for (var i = 0; i < ln; i++) {
		var dictLocale = dictLocales[i];
		if (dictLocale === myLocale ||
		    dictLocale === myLanguage ||
		    dictLocale === myDefaultDictionaryName) {
			this._spellCheckAvailable = true;
			break;
		}
	}

	return this._spellCheckAvailable;
};

/**
 * Gets the identity collection.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{ZmIdentityCollection}	the identity collection
 */
ZmAppCtxt.prototype.getIdentityCollection = function(account) {

    var context = this.isChildWindow ? window && window.opener : window;
	return context && context.AjxDispatcher.run("GetIdentityCollection", account);
};

/**
 * Gets the data source collection.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{ZmModel}	the data source collection
 */
ZmAppCtxt.prototype.getDataSourceCollection = function(account) {

	var context = this.isChildWindow ? window && window.opener : window;
	return context && context.AjxDispatcher.run("GetDataSourceCollection", account);
};

/**
 * Gets the signature collection.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{ZmSignatureCollection}	the signature collection
 */
ZmAppCtxt.prototype.getSignatureCollection = function(account) {

    var context = this.isChildWindow ? window && window.opener : window;
	return context && context.AjxDispatcher.run("GetSignatureCollection", account);
};


ZmAppCtxt.prototype.killMarkReadTimer =
function() {
	if (this.markReadActionId > 0) {
		AjxTimedAction.cancelAction(this.markReadActionId);
		this.markReadActionId = -1;
	}
};
/**
 * Gets the organizer tree.
 * 
 * @param	{ZmOrganizer.FOLDER|ZmOrganizer.TAG|ZmOrganizer.ZIMLET}	type	the type
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{ZmTree}		the tree
 * @see		#getFolderTree
 * @see		#getTagTree
 * @see		#getZimletTree
 */
ZmAppCtxt.prototype.getTree =
function(type, account) {
	if (this.isChildWindow) {
		return parentAppCtxt.getTree(type, account);
	}

	var al = this.accountList;
	var id = account
		? account.id
		: al.activeAccount ? al.activeAccount.id : ZmAccountList.DEFAULT_ID;

	var acct = al.getAccount(id);
	return acct && acct.trees[ZmOrganizer.TREE_TYPE[type]];
};

/**
 * Sets the organizer tree.
 * 
 * @param	{ZmOrganizer.FOLDER|ZmOrganizer.TAG|ZmOrganizer.ZIMLET}	type	the type
 * @param	{ZmTree}	tree		the tree
 * @param	{ZmZimbraAccount}	account		the account
 * @see		#getTree
 */
ZmAppCtxt.prototype.setTree =
function(type, tree, account) {
	var al = this.accountList;
	var id = account
		? account.id
		: al.activeAccount ? al.activeAccount.id : ZmAccountList.DEFAULT_ID;


	var acct = this.accountList.getAccount(id);
	if (acct) {
		acct.trees[type] = tree;
	}
};

/**
 * Gets the folder organizer tree.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{ZmFolderTree}		the tree
 * @see		#getTree
 */
ZmAppCtxt.prototype.getFolderTree =
function(account) {
    return this.getTree(ZmOrganizer.FOLDER, account);
};

/**
 * Gets the tag organizer tree.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{ZmTagTree}		the tree
 * @see		#getTree
 */
ZmAppCtxt.prototype.getTagTree =
function(account) {
	return this.getTree(ZmOrganizer.TAG, account);
};

/**
 * Gets the tag organizer tree's root.
 *
 * @param	{ZmItem}	item		item to look up the account of for and get the account tag list.
 * @return	{ZmTag}		the root of the tree, which is also a list.
 */
ZmAppCtxt.prototype.getAccountTagList =
function(item) {
	var account = (item && appCtxt.multiAccounts) ? item.getAccount() : null;

	return this.getTagTree(account).root;
};


/**
 * Gets the zimlet organizer tree.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{ZmFolderTree}		the tree
 * @see		#getTree
 */
ZmAppCtxt.prototype.getZimletTree =
function(account) {
	return this.getTree(ZmOrganizer.ZIMLET, account);
};

/**
 * Gets the username (which is an email address).
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{String}		the username
 */
ZmAppCtxt.prototype.getUsername =
function(account) { 
	return this.get(ZmSetting.USERNAME, account);
};

/**
 * Gets the user domain.
 * 
 * @param	{ZmZimbraAccount}	account		the account
 * @return	{String}		the user domain
 */
ZmAppCtxt.prototype.getUserDomain =
function(account) {
	if (!this.userDomain) {
        var username = this.getUsername(account);
		if (username) {
			var parts = username.split("@");
			this.userDomain = (parts && parts.length) ? parts[1] : "";
		}
	}
	return this.userDomain;
};

/**
 * Gets the upload frame id.
 * 
 * @return	{String}		the frame id
 */
ZmAppCtxt.prototype.getUploadFrameId =
function() {
	if (!this._uploadManagerIframeId) {
		var iframeId = Dwt.getNextId();
		var html = [ "<iframe name='", iframeId, "' id='", iframeId,
			     "' src='", (AjxEnv.isIE && location.protocol == "https:") ? appContextPath+"/public/blank.html" : "javascript:\"\"",
			     "' style='position: absolute; top: 0; left: 0; visibility: hidden'></iframe>" ];
		var div = document.createElement("div");
		div.innerHTML = html.join("");
		document.body.appendChild(div.firstChild);
		this._uploadManagerIframeId = iframeId;
	}
	return this._uploadManagerIframeId;
};

ZmAppCtxt.prototype.reloadAppCache =
function(force, retryOnError) {
	AjxDebug.println(AjxDebug.OFFLINE, "reloadAppCache :: " + AjxDebug._getTimeStamp());
    if (this.isWebClientOfflineSupported || force) {
		var localOfflineBrowserKey = localStorage.getItem(ZmSetting.WEBCLIENT_OFFLINE_BROWSER_KEY);
		//If application cache status is downloading browser is already downloading the resources mentioned in the manifest file. Resetting the cookie value will result in application cache error event "Manifest changed during update".
		if (localOfflineBrowserKey && AjxEnv.supported.applicationcache && applicationCache.status !== applicationCache.DOWNLOADING) {
			var cookieValue = localOfflineBrowserKey + "_" + new Date().getTime();
			AjxCookie.setCookie(document, "ZM_OFFLINE_KEY", cookieValue, false, "/");
		}
		var manifestURL = appContextPath + "/appcache/images,common,dwt,msgview,login,zm,spellcheck,skin.appcache?";
        var urlParams = [];
        urlParams.push("v=" + window.cacheKillerVersion);
        urlParams.push("debug=" + window.appDevMode);
        urlParams.push("compress=" + !(window.appDevMode === true));
        urlParams.push("templates=only");
        manifestURL = encodeURIComponent(manifestURL + urlParams.join('&'));
        var offlineIframe = document.getElementById("offlineIframe");
        if (!offlineIframe) {
            offlineIframe = document.createElement("iframe");
            offlineIframe.id = "offlineIframe";
            offlineIframe.style.display = "none";
            document.body.appendChild(offlineIframe);
        }
        if (offlineIframe) {
			retryOnError = AjxUtil.isBoolean(retryOnError) ? retryOnError : true;
			offlineIframe.src = "public/Offline.jsp?url=" + manifestURL + "&isFirefox=" + AjxEnv.isFirefox + "&retryOnError=" + retryOnError;
        }
    }
};

/**
 * Gets the upload manager.
 * 
 * @return	{Object}		the upload manager
 */
ZmAppCtxt.prototype.getUploadManager =
function() {
	if (!this._uploadManager) {
		// Create upload manager (for sending attachments)
		this._uploadManager = new AjxPost(this.getUploadFrameId());
	}
	return this._uploadManager;
};

ZmAppCtxt.prototype.getZmUploadManager =
    function() {
        if (!this._zmUploadManager) {
            // Create upload manager (for sending attachments)
            AjxDispatcher.require("Extras");
            this._zmUploadManager = new ZmUploadManager();
        }
        return this._zmUploadManager;
    };

/**
 * Gets the current search.
 * 
 * @return	{ZmSearch}		the current search
 */
ZmAppCtxt.prototype.getCurrentSearch =
function() {
	var app = this.getCurrentApp();
	if (app && app.currentSearch) {
		return app.currentSearch;
	}
	var ctlr = this.getCurrentController();
	return ctlr && ctlr._currentSearch;
};

/**
 * Gets the current view id. If we're showing search results, returns the ID of the
 * view within the search results (rather than the ID of the search results).
 * 
 * @return	{String}		the current view id
 */
ZmAppCtxt.prototype.getCurrentViewId =
function() {
	var viewId = this.getAppViewMgr().getCurrentViewId();
	if (viewId && viewId.indexOf(ZmId.VIEW_SEARCH_RESULTS) === 0) {
		viewId = this.getCurrentController().getCurrentViewId();
	}
	return viewId;
};

/**
 * Gets the current view type. If we're showing search results, returns the type of the
 * view within the search results (rather than the type of the search results).
 * 
 * @return	{String}		the current view type
 */
ZmAppCtxt.prototype.getCurrentViewType =
function() {
	var viewType = this.getAppViewMgr().getCurrentViewType();
	if (viewType && viewType.indexOf(ZmId.VIEW_SEARCH_RESULTS) === 0) {
		viewType = this.getCurrentController().getCurrentViewType();
	}
	return viewType;
};

/**
 * Extracts the view type from a view ID.
 * 
 * @param	{string}	viewId		a view ID
 * @return	{String}	the view type
 */
ZmAppCtxt.prototype.getViewTypeFromId =
function(viewId) {
	var array = viewId && viewId.split(ZmController.SESSION_ID_SEP);
	return array ? array[0] : "";
};

/**
 * Gets the current view.
 * 
 * @return	{DwtComposite}		the current view
 */
ZmAppCtxt.prototype.getCurrentView =
function() {
	return this.getAppViewMgr().getCurrentView();
};

/**
 * Gets the current controller.
 * 
 * @return	{ZmController}		the current controller
 */
ZmAppCtxt.prototype.getCurrentController =
function() {
	var view = this.getCurrentView();
	return (view && view.getController) ? view.getController() : null;
};

/**
 * Sets the current list.
 * 
 * @param	{ZmList}	list		the current list
 */
ZmAppCtxt.prototype.setCurrentList =
function(list) {
	this._list = list;
};

/**
 * Gets the current list.
 * 
 * @return	{ZmList}		the current list
 */
ZmAppCtxt.prototype.getCurrentList =
function() {
	var ctlr = this.getCurrentController();
	return (ctlr && ctlr.getList) ? ctlr.getList() : this._list ? this._list : null;
};

ZmAppCtxt.prototype.getActionController =
function() {
	if (!this._actionController && !this.isChildWindow) {
		this._actionController = new ZmActionController();
	}
	return this._actionController;
};

/**
 * Gets a new window.
 * 
 * @param	{Boolean}	fullView		<code>true</code> to include the full version
 * @param	{int}		width			the width
 * @param	{int}		height			the height
 * @param	{String}	name			window name
 * @param	{Boolean}	isCompoe		compose window or not
 */
ZmAppCtxt.prototype.getNewWindow = 
function(fullVersion, width, height, name, isCompose) {
	// build url
	var url = [];
	var i = 0;
	url[i++] = document.location.protocol;
	url[i++] = "//";
	url[i++] = location.hostname;
	url[i++] = (!location.port || location.port == "80") ? "" : (":" + location.port);
	url[i++] = appContextPath;
	url[i++] = "/public/launchNewWindow.jsp?skin=";
	url[i++] = appCurrentSkin;
	url[i++] = "&localeId=";
	url[i++] = AjxEnv.DEFAULT_LOCALE || "";
	if (fullVersion) {
		url[i++] = "&full=1";
	}
	if (window.appDevMode) {
		url[i++] = "&dev=1";
	}
    if (window.appCoverageMode) {
        url[i++] = "&coverage=1";
    }
	this.__childWindowId = (this.__childWindowId+1) || 0;
	url[i++] = "&childId=" + this.__childWindowId;

    name = name || "_blank";
	width = width || 705;
	height = height || 465;
	var args = ["height=", height, ",width=", width, ",location=no,menubar=no,resizable=yes,scrollbars=no,status=yes,toolbar=no"].join("");
	if (window.appDevMode) {
		args = ["height=", height, ",width=", width, ",location=yes,menubar=yes,resizable=yes,scrollbars=no,status=yes,toolbar=yes"].join("");
	}
	var newWin = window.open(url.join(""), name, args);
	this.handlePopupBlocker(newWin);
	if(newWin) {
		// add this new window to global list so parent can keep track of child windows!
		return this.getAppController().addChildWindow(newWin, this.__childWindowId, isCompose);
	}
};

/**
 * Handle Popup bloker for a given window
 * @param {Object}	win  A Window object
 */
ZmAppCtxt.prototype.handlePopupBlocker =
function(win) {
	if (!win) {
		this.setStatusMsg(ZmMsg.popupBlocker, ZmStatusView.LEVEL_CRITICAL);
	} else if (win && AjxEnv.isChrome) {
		setTimeout(function() { 
					if (win.innerHeight == 0)
						appCtxt.setStatusMsg(ZmMsg.popupBlocker, ZmStatusView.LEVEL_CRITICAL);
				}, 200);
		};
};

/**
 * Caches the given key/value set.
 * 
 * @param	{Object}	key		the key
 * @param	{Object}	value	the value
 * @private
 */
ZmAppCtxt.prototype.cacheSet =
function(key, value) {
	this._itemCache[key] = value;
	delete this._itemCacheDeferred[key];
};

/**
 * Defers caching the given key set.
 * 
 * @param	{Object}	key		the key
 * @param	{String}	appName	the application name
 * @private
 */
ZmAppCtxt.prototype.cacheSetDeferred =
function(key, appName) {
	this._itemCache[key] = this._itemCacheDeferred;
	this._itemCacheDeferred[key] = appName;
};

/**
 * Gets the key from cache.
 * 
 * @param	{Object}	key		the key
 * @return	{Object}	the value
 * @private
 */
ZmAppCtxt.prototype.cacheGet =
function(key) {
	var value = this._itemCache[key];
	if (value === this._itemCacheDeferred) {
		var appName = this._itemCacheDeferred[key];
		this.getApp(appName).createDeferred();
		value = this._itemCache[key];
	}
	return value;
};

/**
 * Removes the key from cache.
 * 
 * @param	{Object}	key		the key
 * @private
 */
ZmAppCtxt.prototype.cacheRemove =
function(key) {
	delete this._itemCache[key];
	delete this._itemCacheDeferred[key];
};

/**
 * Gets the key from cache by id.
 * 
 * @param	{String}	id		the id
 * @return	{Object}	the value
 * @private
 */
ZmAppCtxt.prototype.getById = function(id) {

    // Try parent cache if we're a child window and we don't have it
	return this.cacheGet(id) || (this.isChildWindow && window && window.opener && window.opener.appCtxt && window.opener.appCtxt.getById(id));
};

/**
 * Gets the keyboard manager
 * 
 * @return	{DwtKeyboardMgr}		the keyboard manager
 */
ZmAppCtxt.prototype.getKeyboardMgr =
function() {
	return this._shell.getKeyboardMgr();
};

/**
 * Gets the history manager.
 * 
 * @return	{AjxHistoryMgr}		the history manager
 */
ZmAppCtxt.prototype.getHistoryMgr =
function() {
	if (!this._historyMgr) {
		this._historyMgr = new AjxHistoryMgr();
	}
	return this._historyMgr;
};

/**
 * Checks if the zimlets are present.
 * 
 * @return	{Boolean}		<code>true</code> if zimlets are present
 */
ZmAppCtxt.prototype.zimletsPresent =
function() {
	return this._zimletsPresent;
};

/**
 * Sets if the zimlets are present.
 * 
 * @param	{Boolean}	zimletsPresent		if <code>true</code>, zimlets are present
 */
ZmAppCtxt.prototype.setZimletsPresent =
function(zimletsPresent) {
	this._zimletsPresent = zimletsPresent;
};

/**
 * Gets the zimlet manager
 * 
 * @return	{ZmZimletMgr}	the zimlet manager
 */
ZmAppCtxt.prototype.getZimletMgr =
function() {
	if (!this._zimletMgr) {
		AjxDispatcher.require("Zimlet");
		if (!this._zimletMgr) // Must re-check here, because if this function is called a second time before the "Zimlet" package is loaded, both calls want to set this._zimletMgr, which must NEVER happen (Issue first located in bug #41338)
			this._zimletMgr = new ZmZimletMgr();
	}
	return this._zimletMgr;
};

/**
 * Checks if zimlets are loaded.
 * 
 * @return	{Boolean}		<code>true</code> if zimlets are loaded
 */
ZmAppCtxt.prototype.areZimletsLoaded =
function() {
	return this._zimletsLoaded;
};

/**
 * Adds a listener to the zimlets loaded event.
 * 
 * @param	{AjxCallback}	listener		the listener
 * @param	{int}		index		the index to where to add the listener
 * @return	{Boolean}	<code>true</code> if the listener is added; <code>false</code> otherwise
 */
ZmAppCtxt.prototype.addZimletsLoadedListener =
function(listener, index) {
	if (!this._zimletsLoaded) {
		return this._evtMgr.addListener(ZmAppCtxt._ZIMLETS_EVENT, listener, index);
	}
};

/**
 * Checks is all zimlets are loaded.
 * 
 * @return	{Boolean}	<code>true</code> if all zimlets are loaded
 */
ZmAppCtxt.prototype.allZimletsLoaded =
function() {
	this._zimletsLoaded = true;
	if (this._zimletMgr && !this.isChildWindow && appCtxt.get(ZmSetting.PORTAL_ENABLED)) {
		var portletMgr = this.getApp(ZmApp.PORTAL).getPortletMgr();
		if (portletMgr) {
			portletMgr.allZimletsLoaded();
		}
	}

	if (this._evtMgr.isListenerRegistered(ZmAppCtxt._ZIMLETS_EVENT)) {
		this._evtMgr.notifyListeners(ZmAppCtxt._ZIMLETS_EVENT, new ZmEvent());
		this._evtMgr.removeAll(ZmAppCtxt._ZIMLETS_EVENT);
	}
};

/**
 * Notifies zimlets if they are present and loaded.
 *
 * @param {String}		event		the zimlet event (called as a zimlet function)
 * @param {Array}		args		a list of args to the function
 * @param {Hash}	options			a hash of options
 * @param {Boolean}	options.noChildWindow		if <code>true</code>, skip notify if we are in a child window
 * @param	{Boolean}	options.waitUntilLoaded	if <code>true</code> and zimlets are not yet loaded, add a listener so that notify happens on load
 * @return	{Boolean} Returns <code>true</code> if at least one Zimlet handles the notification
 */
ZmAppCtxt.prototype.notifyZimlets =
function(event, args, options) {
	this.notifySkin(event, args, options); // Also notify skin

	var context = this.isChildWindow ? parentAppCtxt : this;

	if (options && options.noChildWindow && this.isChildWindow) { return false; }

	if (!context.areZimletsLoaded()) {
		if (options && options.waitUntilLoaded) {
			context.addZimletsLoadedListener(new AjxListener(this, this.notifyZimlets, [event, args]));
		}
		return false;
	}

	return this.getZimletMgr().notifyZimlets(event, args);
};

ZmAppCtxt.prototype.notifyZimlet =
function(zimletName, event, args, options) {
	if (options && options.noChildWindow && this.isChildWindow) { return false; }
	return this.getZimletMgr().notifyZimlet(zimletName, event, args);
};

ZmAppCtxt.prototype.notifySkin =
function(event, args, options) {
	var context = this.isChildWindow ? parentAppCtxt : this;
	if (options && options.noChildWindow && this.isChildWindow) { return; }
	try {
		return window.skin && AjxUtil.isFunction(window.skin.handleNotification) && window.skin.handleNotification(event, args);
	} catch (e) {
		if (window.console && window.console.error) {
			console.error(e);
		}
	}
};


/**
 * Gets the calendar manager.
 * 
 * @return	{ZmCalMgr}	the calendar manager
 */
ZmAppCtxt.prototype.getCalManager =
function() {
	if (!this._calMgr) {
        AjxDispatcher.require("Startup2");
		this._calMgr = new ZmCalMgr(this._shell);
	}
	return this._calMgr;
};

ZmAppCtxt.prototype.updateOfflineAppt = function(msgId, field, value, nullData, callback) {
	var calMgr = appCtxt.getCalManager();
	if (calMgr) {
		var calViewController = calMgr && calMgr.getCalViewController();
		if (calViewController) {
			var apptCache = calViewController.getApptCache();
			if (apptCache) {
				apptCache.updateOfflineAppt(msgId, field, value, nullData, callback);
			}
		}
	}
};

/**
 * Gets the task manager.
 *
 * @return	{ZmTaskMgr}	the task manager
 */
ZmAppCtxt.prototype.getTaskManager =
function() {
	if (!this._taskMgr) {
		this._taskMgr = new ZmTaskMgr(this._shell);
	}
	return this._taskMgr;
};


/**
 * Gets the ACL.
 * 
 * @param	{ZmZimbrAccount}	account		the account
 * @param	{AjxCallback}	callback	the callback
 * @return	{ZmAccessControlList}	the ACL
 */
ZmAppCtxt.prototype.getACL =
function(account, callback) {
	var al = this.accountList;
	var id = account
		? account.id
		: al.activeAccount ? al.activeAccount.id : ZmAccountList.DEFAULT_ID;

	var acct = al.getAccount(id);
	return acct && acct.acl;
};

/**
 * Gets the shortcut hint.
 *
 * @param {String}		keyMap		the key map
 * @param {String}		shortcut	the shortcut action
 * @return	{String}	the hint
 */
ZmAppCtxt.prototype.getShortcutHint =
function(keyMap, shortcut) {
	
	var text = null;
	keyMap = keyMap || "global";
	while (!text && keyMap) {
		var scKey = [keyMap, shortcut, "display"].join(".");
		var text = AjxKeys[scKey] || ZmKeys[scKey];
		if (text) {
			var list = text.split(/;\s*/);
			var sc = list[0];	// use first shortcut in list
			if (!sc) { return null; }
			sc = sc.replace(/\b[A-Z]\b/g, function(st) { return st.toLowerCase(); });
			text = [" [", sc.replace(",", ""), "]"].join("");
		} else {
			var key = [keyMap, "INHERIT"].join(".");
			keyMap = AjxKeys[key] || ZmKeys[key];
		}
	}

	return text;
};

/**
 * Gets the shortcuts panel.
 * 
 * @return	{ZmShortcutsPanel}	the shortcuts panel
 */
ZmAppCtxt.prototype.getShortcutsPanel =
function() {
	if (!this._shortcutsPanel) {
		AjxDispatcher.require(["PreferencesCore", "Preferences"]);
		var style = this.isChildWindow ? ZmShortcutList.WINDOW_STYLE : ZmShortcutList.PANEL_STYLE;
		this._shortcutsPanel = new ZmShortcutsPanel(style);
	}
	return this._shortcutsPanel;
};

/**
 * Opens a new change password window
 *
 */
ZmAppCtxt.prototype.getChangePasswordWindow =
function(ev) {
    var url = appCtxt.get(ZmSetting.CHANGE_PASSWORD_URL);

    	if (!url) {
    		var isHttp	= appCtxt.get(ZmSetting.PROTOCOL_MODE) == ZmSetting.PROTO_HTTP;
    		var proto	= isHttp ? ZmSetting.PROTO_HTTP : ZmSetting.PROTO_HTTPS;
    		var port	= appCtxt.get(isHttp ? ZmSetting.HTTP_PORT : ZmSetting.HTTPS_PORT);
    		var path	= appContextPath+"/h/changepass";

    		var publicUrl = appCtxt.get(ZmSetting.PUBLIC_URL);
    		if (publicUrl) {
    			var parts = AjxStringUtil.parseURL(publicUrl);
    			var switchMode = (parts.protocol == "http" && proto == ZmSetting.PROTO_HTTPS);
    			proto = switchMode ? proto : parts.protocol;
    			port = switchMode ? port : parts.port;
    		}
			var qsArgs = {skin: appCurrentSkin};
    		url = AjxUtil.formatUrl({protocol: proto, port: port, path: path, qsReset: true, qsArgs: qsArgs});
    	}

        var args;
        var result = { value: null };
        appCtxt.notifyZimlets("onZmAppCtxt_getChangePasswordWindow", [result]);
        if (result.value) {
            args = result.value;
        } else {
            args = "height=675,width=705,location=no,menubar=no,resizable=yes,scrollbars=no,status=yes,toolbar=no";
        }
        window.open(url,'ChangePasswordWindow', args);
};

/**
 * Gets the skin hint for the given argument(s), which will be used to look
 * successively down the properties chain.
 * 
 * <p>
 * For example, <code>getSkinHint("a", "b")</code> will return the value of <code>skin.hints.a.b</code>.
 * </p>
 * 
 * @return	{String}	the skin hint
 */
ZmAppCtxt.prototype.getSkinHint =
function() {
	if (arguments.length == 0) return "";
	
	var cur = window.skin && window.skin.hints;
	if (!cur) { return ""; }
	for (var i = 0; i < arguments.length; i++) {
		var arg = arguments[i];
		if (!cur[arg]) { return ""; }
		cur = cur[arg];
	}
	return cur;
};

/**
 * Gets the auto completer.
 * 
 * @return	{ZmAutocomplete}	the auto completer
 */
ZmAppCtxt.prototype.getAutocompleter =
function() {
	if (!this._autocompleter) {
		this._autocompleter = new ZmAutocomplete(null);
	}
	return this._autocompleter;
};

/**
 * Checks if my address belongs to the current user (include aliases).
 * 
 * @param {String}		addr			            the address
 * @param {Boolean}		allowLocal		            if <code>true</code>, domain is not required
 * @param {Boolean}		excludeAllowFromAddress		if <code>true</code>, addresses in zimbraAllowFromAddresses are ignored
 * @return	{Boolean}		<code>true</code> if the given address belongs to the current user; <code>false</code> otherwise
 */
ZmAppCtxt.prototype.isMyAddress =
function(addr, allowLocal, excludeAllowFromAddress) {

	if (allowLocal && (addr.indexOf('@') == -1)) {
		addr = [addr, this.getUserDomain()].join("@");
	}
	
	if (addr == this.get(ZmSetting.USERNAME)) {
		return true;
	}

	var allAddresses;
    if(excludeAllowFromAddress){
        allAddresses= appCtxt.get(ZmSetting.MAIL_ALIASES);
    }
    else
    {
        allAddresses= appCtxt.get(ZmSetting.MAIL_ALIASES).concat(appCtxt.get(ZmSetting.ALLOW_FROM_ADDRESSES));
    }

	if (allAddresses && allAddresses.length) {
		for (var i = 0; i < allAddresses.length; i++) {
			if (addr == allAddresses[i])
				return true;
		}
	}

	return false;
};

/**
 * Gets the overview ID, prepending account name if multi-account.
 *
 * @param {Array}		parts		an array of {String} id components
 * @param {ZmAccount}	account		the account
 * @return	{String}	the id
 */
ZmAppCtxt.prototype.getOverviewId =
function(parts, account) {

	var id = (parts instanceof Array) ? parts.join("_") : parts;

	if (appCtxt.multiAccounts && (account !== null)) {
		account = account || appCtxt.getActiveAccount();
		id = [account.name, id].join(":");
	}

	return id;
};

/**
 * Gets the autocomplete cache for the given parameters, optionally creating one.
 *
 * @param	{String}	acType		the item type
 * @param	{String}	str			the autocomplete string
 * @param	{ZmAccount}	account		the account
 * @param	{Boolean}	create		if <code>true</code>, create a cache if none found
 */
ZmAppCtxt.prototype.getAutocompleteCache =
function(acType, str, account, create) {

	var cache = null;
	var acct = account || this.getActiveAccount();
	if (acct) {
		if (this._acCache[acct.id] && this._acCache[acct.id][acType]) {
			cache = this._acCache[acct.id][acType][str];
		}
	}
	if (!cache && create) {
		if (acct && !this._acCache[acct.id]) {
			this._acCache[acct.id] = {};
		}
		if (!this._acCache[acct.id][acType]) {
			this._acCache[acct.id][acType] = {};
		}
		cache = this._acCache[acct.id][acType][str] = {};
	}

	return cache;
};

/**
 * Clears the autocomplete cache.
 *
 * @param	{String}	type		item type
 * @param	{ZmAccount}	account		the account
 */
ZmAppCtxt.prototype.clearAutocompleteCache =
function(type, account) {

	var acct = account || appCtxt.getActiveAccount();
	if (this._acCache[acct.id]) {
		if (type) {
			this._acCache[acct.id][type] = {};
		} else {
			this._acCache[acct.id][ZmAutocomplete.AC_TYPE_CONTACT]		=	{};
			this._acCache[acct.id][ZmAutocomplete.AC_TYPE_LOCATION]		=	{};
			this._acCache[acct.id][ZmAutocomplete.AC_TYPE_EQUIPMENT]	=	{};
		}
	}
};

ZmAppCtxt.prototype.getOutsideMouseEventMgr =
function() {
	return DwtOutsideMouseEventMgr.INSTANCE;
};


/**
* @return Returns language specific charset. Currently supports only Japanese. 
* Returns "Windows-31J" for Japanese or returns "UTF-8" for everything else
*/
ZmAppCtxt.prototype.getCharset =
function() {
	var lang = AjxEnv.isIE ? window.navigator.systemLanguage : window.navigator.language;
	//Currently only differs for Japanese, but can extend for different languages as/if we need it.
	if((AjxEnv.DEFAULT_LOCALE == "ja" || lang == "ja") && AjxEnv.isWindows) {
		return "Windows-31J";
	} else {
		return "UTF-8";
	}
};

/**
 * Returns true if an address is an expandable DL.
 *  
 * @param {string}	addr	email address
 */
ZmAppCtxt.prototype.isExpandableDL =
function(addr) {
	return addr && this._isExpandableDL[addr] && this.get("EXPAND_DL_ENABLED");
};

/**
 * Cache whether an address is an expandable DL.
 * 
 * @param {string}	addr			email address
 * @param {boolean}	isExpandableDL	if true, address is expandable DL
 * 
 * TODO: consider caching AjxEmailAddress objects by addr so we also save display name
 */
ZmAppCtxt.prototype.setIsExpandableDL =
function(addr, isExpandableDL) {
	this._isExpandableDL[addr] = isExpandableDL;
};

ZmAppCtxt.prototype.getToolTipMgr =
function() {
	if (!this._toolTipMgr) {
		this._toolTipMgr = new ZmToolTipMgr();
	}
	return this._toolTipMgr;
};

/**
 * Returns true if Prism and the user is online
 *
 */
ZmAppCtxt.prototype.isZDOnline =
function() {
    var ac = window["appCtxt"].getAppController();
    return !AjxEnv.isPrism || (ac._isPrismOnline && ac._isUserOnline);
};

/**
 * When using pre-auth window.opener.appCtxt may not be accessible.  This function
 * handles appCtxt assignment to avoid a permission denied error
 * @return {Object} ZmAppCtxt
 */
ZmAppCtxt.handleWindowOpener = 
function() {
	try {
		return window && window.opener && window.opener.appCtxt || appCtxt;
	}
	catch (ex) {
		return appCtxt;
	}
};

ZmAppCtxt.prototype.isWebClientOffline =
function() {
    if (this.isWebClientOfflineSupported) {
        return ZmOffline.isServerReachable === false;
    }
    return false;
};

ZmAppCtxt.prototype.initWebOffline =
function() {
    this.isWebClientOfflineSupported = false;
	return;
};

/**
 * Gets the offline settings dialog.
 *
 * @return	{ZmOfflineSettingsDialog}	offline settings dialog
 */
ZmAppCtxt.prototype.getOfflineSettingsDialog =
function() {
    if (!this._offlineSettingsDialog) {
        this._offlineSettingsDialog = new ZmOfflineSettingsDialog();
    }
    return this._offlineSettingsDialog;
};

/**
 * Returns true if the given ID is not local. That's the case if the ID has
 * an account part that is not the active account.
 *
 * @param {String|Number}   id
 * @returns {Boolean}   true if the given ID is not local
 */
ZmAppCtxt.prototype.isRemoteId = function(id) {
	id = String(id);
	var acct = appCtxt.getActiveAccount();
	return (id.indexOf(":") !== -1) && (id.indexOf(acct.id) !== 0);
};

/**
 * Returns the singleton AjxClipboard instance, if it is supported.
 *
 * @returns {AjxClipboard}
 */
ZmAppCtxt.prototype.getClipboard = function() {
	return AjxClipboard.isSupported() ? new AjxClipboard() : null;
};

/**
 * Checks a precondition which may be in one of several forms: a boolean, a settings constant, a function, or a list.
 *
 * @param {Boolean|String|Function|Array}   precondition    something that evaluates to true or false
 * @param {Boolean}                         listAny         (optional) if a list is provided, whether just one (instead of all) must be true
 *
 * @return boolean  false if the precondition evaluates to false or null, otherwise true
 */
ZmAppCtxt.prototype.checkPrecondition = function(precondition, listAny) {

	// Lack of a precondition evaluates to true
	if (precondition === undefined) {
		return true;
	}

	// A precondition of null should not happen
	if (precondition === null) {
		return false;
	}

	// Boolean speaks for itself
	if (AjxUtil.isBoolean(precondition)) {
		return precondition;
	}

	// Client setting: fetch value from settings
	if (AjxUtil.isString(precondition) && ZmSetting.hasOwnProperty(precondition)) {
		return appCtxt.get(precondition);
	}

	// Function: evaluate and return result
	if (AjxUtil.isFunction(precondition)) {
		return precondition();
	}

	// Array can be treated in one of two modes, where all have to be true, or just one does
	if (AjxUtil.isArray(precondition)) {
		for (var i = 0; i < precondition.length; i++) {
			var result = this.checkPrecondition(precondition[i]);
			if (listAny && result) {
				return true;
			}
			if (!listAny && !result) {
				return false;
			}
		}
		return !listAny;
	}

	return true;
};

/**
 * Displays an error message in  dialog.
 *
 * @param {string}  errMsg      error message
 * @param {string}  details     (optional) additional info to show using Details button
 * @param {string}  style       (optional) style constant DwtMessageDialog.*_STYLE
 * @param {string}  title       (optional) title for dialog (other than one for the style)
 * @apram {boolean} noReport    do not add a Report button/function to the dialog (defaults to true)
 */
ZmAppCtxt.prototype.showError = function(params) {

    params = params || {};
    var errMsg = params.errMsg || params;
    var dlg = this.getErrorDialog();
    dlg.reset();
    dlg.setMessage(errMsg, params.details, params.style || DwtMessageDialog.WARNING_STYLE, params.title);
    dlg.popup(null, params.noReport !== false);
};

/**
 * Get shared folders
 *
 * @param {Object} folder     current node
 * @param {string} type       folder type
 * @param {Array}  result     an array of shared folders
 */
ZmAppCtxt.prototype.getSharedFolders =
function(folder, type, result) {
    if (!folder || !folder instanceof ZmFolder || !type || !Array.isArray(result)) {
        return;
    }
    var children = folder.children && folder.children.getArray();
    for (var i = 0; i < children.length; i++) {
        appCtxt.getSharedFolders(children[i], type, result);
    }
    // a shared folder has an owner
    if (folder.owner && folder.type == type && !folder.noSuchFolder) {
        result.push(folder);
    }
};
}
if (AjxPackage.define("zimbraMail.core.ZmOperation")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the operation class.
 * 
 */

/**
 * Creates the operation object.
 * @class
 * This class provides the idea of an "operation", which is a user-initiated action
 * exposed through a button or menu item. Many operations (such as Delete) are shared
 * across applications/controllers. An operation gets defined by specifying its name,
 * tool tip, and image. Then controllers can simply select which operations they'd like
 * to support.
 * <br/>
 * <br/>
 * The two primary clients of this class are {@link ZmButtonToolBar} and {@link ZmActionMenu}. Clients 
 * should support {@link #createOp} and {@link #getOp} methods. See the two aforementioned clients for
 * examples.
 * 
 * @author Conrad Damon
 */
ZmOperation = function() {};

// Special operations
ZmOperation.NONE 					= "NONE";		// no operations or menu items
ZmOperation.SEP 					= "SEP";		// separator
ZmOperation.SPACER 					= "SPACER";		// spacer (toolbar)
ZmOperation.FILLER 					= "FILLER";		// filler (toolbar)

// suffix for disabled image
ZmOperation.DIS = "Dis";

// text and icons displayed for operation
ZmOperation.SETUP = {};

// special-purpose operations
ZmOperation.SETUP[ZmOperation.NONE]		= {};	// means no operations (as opposed to a default set)
ZmOperation.SETUP[ZmOperation.SEP]		= {};	// a thin vertical or horizontal bar
ZmOperation.SETUP[ZmOperation.SPACER]	= {};	// empty space of a given size
ZmOperation.SETUP[ZmOperation.FILLER]	= {};	// expandable space (for right-align in toolbars)

// preconditions for operations - no automatic checking is done, so a client
// of this class has to check them on its own if it wants
ZmOperation.PRECONDITION = {};

// code to run after an operation has been created - typically used to create
// a menu for a button
ZmOperation.CALLBACK = {};

/**
 * Defines the aspects of an operation, and the ID that refers to it.
 * 
 * @param {String}	op		the name of the operation
 * @param {hash}	params:
 *      @param {String}	text		the msg key for button or menu item text
 *      @param {String}	tooltip	the msg key for tooltip text
 *      @param {String}	image		the icon class for the button or menu item
 *      @param {String}	disImage	the disabled version of image; defaults to image + "Dis"
 *      @param {Boolean|String|Function}    precondition (overrides setting if present)
 * @param {constant}	precondition	must evaluate to true for this operation not to be filtered out
 * @param {AjxCallback}	callback	the callback to run after this operation has been created
 */
ZmOperation.registerOp = function(op, params, precondition, callback) {

	ZmOperation[op] = op;
	ZmOperation.SETUP[op] = params || {};
	if (precondition)	{ ZmOperation.PRECONDITION[op]	= precondition; }
	if (callback)	    { ZmOperation.CALLBACK[op]	    = callback; }
};

ZmOperation.unregisterOp = function(op) {
	delete ZmOperation[op];
	delete ZmOperation.SETUP[op];
	delete ZmOperation.PRECONDITION[op];
	delete ZmOperation.CALLBACK[op];
};

ZmOperation.KEY_ID		= "opId";
ZmOperation.MENUITEM_ID	= "menuItemId";

ZmOperation.NEW_ITEM_OPS	= [];
ZmOperation.NEW_ITEM_KEY	= {};
ZmOperation.NEW_ORG_OPS		= [];
ZmOperation.NEW_ORG_KEY		= {};

// Static hash of operation IDs ad descriptors
ZmOperation._operationDesc = {};

/**
 * Initializes and creates standard operations.
 * 
 */
ZmOperation.initialize =
function() {
	ZmOperation.registerOp(ZmId.OP_ACTIONS_MENU, {textKey:"actions", tooltipKey:"", textPrecedence:40});
	
	ZmOperation.registerOp(ZmId.OP_ATTACHMENT, {textKey:"addAttachment", tooltipKey:"attachmentTooltip", image:"Attachment", shortcut:ZmKeyMap.ATTACHMENT, showImageInToolbar: true, ariaLabel: "addAttachment"});
	ZmOperation.registerOp(ZmId.OP_CALL, {image:"Telephone"});
	ZmOperation.registerOp(ZmId.OP_CANCEL, {textKey:"cancel", tooltipKey:"cancelTooltip", image:"Cancel", shortcut:ZmKeyMap.CANCEL});
	ZmOperation.registerOp(ZmId.OP_CHECK_ALL, {textKey:"checkAll", image:"Check"});
	ZmOperation.registerOp(ZmId.OP_CLEAR_ALL, {textKey:"clearAll", image:"Cancel"});
	ZmOperation.registerOp(ZmId.OP_CLOSE, {textKey:"close", tooltipKey:"closeTooltip", image:"Close", shortcut:ZmKeyMap.CANCEL});
	ZmOperation.registerOp(ZmId.OP_COMPOSE_FORMAT, {textKey:"format", tooltipKey:"formatTooltip", image:"SwitchFormat", shortcut:ZmKeyMap.HTML_FORMAT}, ZmSetting.HTML_COMPOSE_ENABLED);
	ZmOperation.registerOp(ZmId.OP_CONTACTGROUP_MENU, {textKey: "AB_CONTACT_GROUP", tooltipKey:"contactGroupTooltip", image:"Group"}, null,
		AjxCallback.simpleClosure(function(parent) {
			ZmOperation.addDeferredMenu(ZmOperation.addContactGroupMenu, parent, true);
	}));
	ZmOperation.registerOp(ZmId.OP_COPY, {textKey:"copy", image:"Copy"});
	ZmOperation.registerOp(ZmId.OP_DELETE, {textKey:"del", tooltipKey:"deleteTooltip", image:"Delete", shortcut:ZmKeyMap.DEL, textPrecedence:60});
	ZmOperation.registerOp(ZmId.OP_DELETE_WITHOUT_SHORTCUT, {textKey:"del", tooltipKey:"deleteTooltip", image:"Delete", textPrecedence:60});
	ZmOperation.registerOp(ZmId.OP_DETACH, {textKey:"detach", tooltipKey:"detach", image:"OpenInNewWindow", showImageInToolbar: true});
    ZmOperation.registerOp(ZmId.OP_DETACH_WIN, {textKey:"detach", image:"OpenInNewWindow"});
	ZmOperation.registerOp(ZmId.OP_EDIT, {textKey:"edit", tooltipKey:"editTooltip", image:"Edit", shortcut:ZmKeyMap.EDIT});
	ZmOperation.registerOp(ZmId.OP_EDIT_AS_NEW, {textKey:"editAsNew", tooltipKey:"editTooltip", image:"Edit", shortcut:ZmKeyMap.EDIT});
	ZmOperation.registerOp(ZmId.OP_EDIT_PROPS, {textKey:"editProperties", tooltipKey:"editPropertiesTooltip", image:"Properties"});
	ZmOperation.registerOp(ZmId.OP_EXPAND, {textKey:"expand", image:"Plus"});
	ZmOperation.registerOp(ZmId.OP_EXPAND_ALL, {textKey:"expandAll", image:"Plus"});
//	ZmOperation.registerOp(ZmId.OP_EXPORT_FOLDER, {textKey:"exportFolder", image:"MailExport"});
	ZmOperation.registerOp(ZmId.OP_EMPTY_FOLDER,{textKey:"emptyFolder",image:"EmptyFolder"});
	ZmOperation.registerOp(ZmId.OP_FORMAT_HTML, {textKey: "formatAsHtml"}, ZmSetting.HTML_COMPOSE_ENABLED);
	ZmOperation.registerOp(ZmId.OP_FORMAT_TEXT, {textKey: "formatAsText"}, ZmSetting.HTML_COMPOSE_ENABLED);
	ZmOperation.registerOp(ZmId.OP_FORMAT_MORE_OPTIONS, {textKey: "moreComposeOptions"});
    ZmOperation.registerOp(ZmId.OP_GROUPBY, {textKey:"groupBy"});
    ZmOperation.registerOp(ZmId.OP_GROUPBY_NONE, {textKey:"groupByNone"});
    ZmOperation.registerOp(ZmId.OP_GROUPBY_DATE, {textKey:"groupByDate"});
    ZmOperation.registerOp(ZmId.OP_GROUPBY_FROM, {textKey:"groupByFrom"});
    ZmOperation.registerOp(ZmId.OP_GROUPBY_PRIORITY, {textKey:"groupByPriority"});
    ZmOperation.registerOp(ZmId.OP_GROUPBY_SIZE, {textKey:"groupBySize"});
    ZmOperation.registerOp(ZmId.OP_GROUPBY_TAG,  {textKey:"groupByTag"});
//	ZmOperation.registerOp(ZmId.OP_IMPORT_FOLDER, {textKey:"importFolder", image:"MailImport"});
	ZmOperation.registerOp(ZmId.OP_MARK_ALL_READ, {textKey:"markAllRead", image:"ReadMessage"});
//	ZmOperation.registerOp(ZmId.OP_MOUNT_FOLDER, {textKey:"mountFolder", image:"Folder"});
	ZmOperation.registerOp(ZmId.OP_MOVE, {textKey:"move", tooltipKey:"moveTooltip", image:"MoveToFolder", textPrecedence:40, showImageInToolbar: true}); //todo - remove
	ZmOperation.registerOp(ZmId.OP_MOVE_MENU, {textKey: "move", tooltipKey:"moveTooltip", image:"MoveToFolder", showImageInToolbar: true, ariaLabel: "move" }, null,
		AjxCallback.simpleClosure(function(parent) {
			ZmOperation.addDeferredMenu(ZmOperation.addMoveMenu, parent, true);
		}));

    ZmOperation.registerOp(ZmId.OP_MUTE_CONV, {textKey:"muteConv", tooltipKey:"muteConvTooltip", image:"Mute", shortcut:ZmKeyMap.MUTE_UNMUTE_CONV});
	ZmOperation.registerOp(ZmId.OP_NEW_FOLDER, {textKey:"newFolder", tooltipKey:"newFolderTooltip", image:"NewFolder", shortcut:ZmKeyMap.NEW_FOLDER}, ZmSetting.USER_FOLDERS_ENABLED);
	ZmOperation.registerOp(ZmId.OP_NEW_MENU, {textKey:"_new", shortcut:ZmKeyMap.NEW, textPrecedence:100}, null,
		AjxCallback.simpleClosure(function(parent) {
			ZmOperation.addDeferredMenu(ZmOperation.addNewMenu, parent);
		}));
	ZmOperation.registerOp(ZmId.OP_NEW_TAG, {textKey:"newTag", tooltipKey:"newTagTooltip", image:"NewTag", shortcut:ZmKeyMap.NEW_TAG}, ZmSetting.TAGGING_ENABLED);
    ZmOperation.registerOp(ZmId.OP_NOTIFY, {textKey: "notify", image:"Feedback"});
	ZmOperation.registerOp(ZmId.OP_OPEN_IN_TAB, {textKey:"openInTab", image:"OpenInNewTab"});
	ZmOperation.registerOp(ZmId.OP_OPTS, {textKey:"options", tooltipKey:"options", image:"ContextMenu"});
	ZmOperation.registerOp(ZmId.OP_PAGE_BACK, {image:"LeftArrow", shortcut:ZmKeyMap.PREV_PAGE, ariaLabel: "goBack"});
	ZmOperation.registerOp(ZmId.OP_PAGE_FORWARD, {image:"RightArrow", shortcut:ZmKeyMap.NEXT_PAGE, ariaLabel: "goForward"});
	ZmOperation.registerOp(ZmId.OP_PRINT, {textKey:"print", tooltipKey:"printTooltip", image:"Print", shortcut:ZmKeyMap.PRINT, textPrecedence:30, showImageInToolbar: true, ariaLabel: "print" }, ZmSetting.PRINT_ENABLED);
    ZmOperation.registerOp(ZmId.OP_PRIORITY_FILTER, {image:"Priority", textKey:"activityStream"}, ZmSetting.PRIORITY_INBOX_ENABLED);
	ZmOperation.registerOp(ZmId.OP_FIND_SHARES, {image:"Group", textKey:"findShares"}, ZmSetting.SHARING_ENABLED);

	//ZmOperation.registerOp(ZmId.OP_QUICK_COMMANDS, {textKey:"quickCommands", image:"QuickCommand"});
	ZmOperation.registerOp(ZmId.OP_RECOVER_DELETED_ITEMS, {textKey:"recoverDeletedItems", image:"Trash", tooltipKey:"recoverDeletedItems"}, ZmSetting.DUMPSTER_ENABLED);
    ZmOperation.registerOp(ZmId.OP_REFRESH, {textKey:"refresh", tooltipKey:"refreshTooltip"});
	ZmOperation.registerOp(ZmId.OP_RENAME_FOLDER, {textKey:"renameFolder", image:"Rename"});
	ZmOperation.registerOp(ZmId.OP_RENAME_SEARCH, {textKey:"renameSearch", image:"Rename"});
	ZmOperation.registerOp(ZmId.OP_RENAME_TAG, {textKey:"renameTag", image:"Rename"}, ZmSetting.TAGGING_ENABLED);
	ZmOperation.registerOp(ZmId.OP_SAVE, {textKey:"save", image:"Save", shortcut:ZmKeyMap.SAVE});
	ZmOperation.registerOp(ZmId.OP_SEARCH, {textKey:"findEmailFromSender", image:"Search"}, ZmSetting.SEARCH_ENABLED);
    ZmOperation.registerOp(ZmId.OP_SEARCH_TO, {textKey:"findEmailToSender", image:"Search"}, ZmSetting.SEARCH_ENABLED);
    ZmOperation.registerOp(ZmId.OP_SEARCH_MENU, {textKey:"findEmails", image:"Search"}, ZmSetting.SEARCH_ENABLED,
		AjxCallback.simpleClosure(function(parent) {
			ZmOperation.addDeferredMenu(ZmOperation.addSearchMenu, parent, true);
		}));
	ZmOperation.registerOp(ZmId.OP_SEND, {textKey:"send", tooltipKey:"sendTooltip", image:"Send", shortcut:ZmKeyMap.SEND});
    ZmOperation.registerOp(ZmId.OP_FREE_BUSY_LINK, {textKey:"freeBusyLink", tooltipKey:"freeBusyLinkTooltip", image:"Send"});
    ZmOperation.registerOp(ZmId.OP_SEND_FB_HTML, {textKey:"sendHTMLLink", tooltipKey:"freeBusyLinkTooltip"});
    ZmOperation.registerOp(ZmId.OP_SEND_FB_ICS, {textKey:"sendICSLink", tooltipKey:"freeBusyLinkTooltip"});
    ZmOperation.registerOp(ZmId.OP_SEND_FB_ICS_EVENT, {textKey:"sendICSEventLink", tooltipKey:"freeBusyLinkTooltip"});
    ZmOperation.registerOp(ZmId.OP_SHARE, {textKey:"share", tooltipKey:"shareTooltip"}, ZmSetting.SHARING_ENABLED);
	ZmOperation.registerOp(ZmId.OP_SHARE_ACCEPT, {textKey:"acceptShare", image:"Check"}, ZmSetting.SHARING_ENABLED);
	ZmOperation.registerOp(ZmId.OP_SHARE_DECLINE, {textKey:"declineShare", image:"Cancel"}, ZmSetting.SHARING_ENABLED);
	ZmOperation.registerOp(ZmId.OP_SUBSCRIBE_APPROVE, {textKey:"dlApprove", image:"Check"});
	ZmOperation.registerOp(ZmId.OP_SUBSCRIBE_REJECT, {textKey:"dlReject", image:"Cancel"});
	ZmOperation.registerOp(ZmId.OP_SHARE_FOLDER, {textKey:"shareFolder", image:"SharedMailFolder"});
	ZmOperation.registerOp(ZmId.OP_SHOW_ALL_ITEM_TYPES, {textKey:"showAllItemTypes", image:"Globe"});
    ZmOperation.registerOp(ZmId.OP_SORT_ASC, {textKey:"sortAscending"});
    ZmOperation.registerOp(ZmId.OP_SORT_DESC, {textKey:"sortDescending"});
	ZmOperation.registerOp(ZmId.OP_SPELL_CHECK, {textKey:"spellCheck", image:"SpellCheck", tooltipKey:"spellCheckTooltip", shortcut:ZmKeyMap.SPELLCHECK, showImageInToolbar: true, ariaLabel: "spellCheck"}, ZmSetting.SPELL_CHECK_ENABLED);
	ZmOperation.registerOp(ZmId.OP_SYNC, {textKey:"reload", image:"Refresh", shortcut:ZmKeyMap.REFRESH});
    ZmOperation.registerOp(ZmId.OP_SYNC_ALL, {textKey:"checkAllFeed", image:"Refresh"});
	ZmOperation.registerOp(ZmId.OP_SYNC_OFFLINE_FOLDER, {textKey:"syncOfflineFolderOff", image:"Refresh"}, ZmSetting.OFFLINE_ENABLED); /* offline only */
	ZmOperation.registerOp(ZmId.OP_TAG, null, ZmSetting.TAGGING_ENABLED);
	ZmOperation.registerOp(ZmId.OP_TAG_COLOR_MENU, {textKey:"tagColor", image:"TagStack"}, ZmSetting.TAGGING_ENABLED,
		AjxCallback.simpleClosure(function(parent) {
			ZmOperation.addDeferredMenu(ZmOperation.addColorMenu, parent);
		}));
	ZmOperation.registerOp(ZmId.OP_TAG_MENU, {textKey: "tag", tooltipKey:"tagTooltip", image:"Tag", showImageInToolbar: true, ariaLabel: "tag" }, ZmSetting.TAGGING_ENABLED,
		AjxCallback.simpleClosure(function(parent) {
			ZmOperation.addDeferredMenu(ZmOperation.addTagMenu, parent, true);
		}));
	// placeholder for toolbar text
	ZmOperation.registerOp(ZmId.OP_TEXT);
	// XXX: need new icon? -
	//      Undelete is stupid. We should either add it for all items types (not just contacts) or just kill it
    ZmOperation.registerOp(ZmId.OP_UNDELETE, {textKey:"undelete", tooltipKey:"undelete", image:"MoveToFolder"});
    ZmOperation.registerOp(ZmId.OP_UNMUTE_CONV, {textKey:"unmuteConv", tooltipKey:"unmuteConvTooltip", image:"Unmute", shortcut:ZmKeyMap.MUTE_UNMUTE_CONV});
	ZmOperation.registerOp(ZmId.OP_VIEW, {textKey:"view", image:"SplitView"});
	ZmOperation.registerOp(ZmId.OP_VIEW_MENU, {tooltipKey:"viewTooltip", textKey:"view", image:"SplitPane", textPrecedence:80, showImageInToolbar: true, showTextInToolbar: true});
	ZmOperation.registerOp(ZmId.OP_ZIMLET, {image:"ZimbraIcon"});

	// invites - needed for both Mail and Calendar
	ZmOperation.registerOp(ZmId.OP_ACCEPT_PROPOSAL, {textKey:"replyAccept", image:"Check"});
	ZmOperation.registerOp(ZmId.OP_DECLINE_PROPOSAL, {textKey:"replyDecline", image:"Cancel"});
	ZmOperation.registerOp(ZmId.OP_CAL_REPLY, {textKey:"reply", tooltipKey:"replyTooltip", image:"Reply", shortcut:ZmKeyMap.REPLY});
	ZmOperation.registerOp(ZmId.OP_CAL_REPLY_ALL, {textKey:"replyAll", tooltipKey:"replyAllTooltip", image:"ReplyAll", shortcut:ZmKeyMap.REPLY_ALL});
	ZmOperation.registerOp(ZmId.OP_REPLY_ACCEPT, {textKey:"replyAccept", image:"Check", showTextInToolbar: true, showImageInToolbar: true});
	ZmOperation.registerOp(ZmId.OP_REPLY_ACCEPT_NOTIFY, {textKey:"notifyOrganizerLabel", image:"Check"});
	ZmOperation.registerOp(ZmId.OP_REPLY_ACCEPT_IGNORE, {textKey:"dontNotifyOrganizerLabel", image:"Check"});
	ZmOperation.registerOp(ZmId.OP_REPLY_CANCEL);
	ZmOperation.registerOp(ZmId.OP_REPLY_DECLINE, {textKey:"replyDecline", image:"Cancel", showTextInToolbar: true, showImageInToolbar: true});
	ZmOperation.registerOp(ZmId.OP_REPLY_DECLINE_NOTIFY, {textKey:"notifyOrganizerLabel", image:"Cancel"});
	ZmOperation.registerOp(ZmId.OP_REPLY_DECLINE_IGNORE, {textKey:"dontNotifyOrganizerLabel", image:"Cancel"});
	ZmOperation.registerOp(ZmId.OP_REPLY_MODIFY);
	ZmOperation.registerOp(ZmId.OP_REPLY_NEW_TIME, {textKey:"replyNewTime", image:"NewTime"});
	ZmOperation.registerOp(ZmId.OP_REPLY_TENTATIVE, {textKey:"replyTentative", image:"QuestionMark", showTextInToolbar: true, showImageInToolbar: true});
	ZmOperation.registerOp(ZmId.OP_REPLY_TENTATIVE_NOTIFY, {textKey:"notifyOrganizerLabel", image:"QuestionMark"});
	ZmOperation.registerOp(ZmId.OP_REPLY_TENTATIVE_IGNORE, {textKey:"dontNotifyOrganizerLabel", image:"QuestionMark"});

	// Compose Options - used by Calendar and Mail
	ZmOperation.registerOp(ZmId.OP_COMPOSE_OPTIONS, {textKey:"options", image:"Preferences"});

	ZmOperation.NEW_ORG_OPS.push(ZmOperation.NEW_FOLDER, ZmOperation.NEW_TAG);
	ZmOperation.NEW_ORG_KEY[ZmOperation.NEW_FOLDER]	= "folder";
	ZmOperation.NEW_ORG_KEY[ZmOperation.NEW_TAG]	= "tag";

	appCtxt.notifyZimlets("onZmOperation_initialize", []);
};

/**
 * Creates operation descriptors for the given operation IDs,
 * then creates the appropriate widget for each operation based on the type of
 * the parent. If it's a toolbar, then buttons are created. If it's a menu, menu items are
 * created.
 * <p>
 * To override or add properties to a particular operation, pass in a hash of properties and
 * values as a value in overrides, with the operation ID as the key.
 * </p>
 *
 * @param {DwtComposite}	parent		the containing widget (toolbar or menu)
 * @param {Array}	operations	a list of operation IDs
 * @param {Hash}	overrides		a hash of overrides by op ID
 *
 * @returns	{Hash}	a hash of operations by ID
 */
ZmOperation.createOperations =
function(parent, operations, overrides) {
	var obj = new ZmOperation();
	return obj._createOperations(parent, operations, overrides);
}

/**
 * Done through an object so that we can have more than one invocation going
 * without sharing memory (eg, creating New submenu).
 * 
 * @private
 */
ZmOperation.prototype._createOperations =
function(parent, operations, overrides) {
	if (operations == ZmOperation.NONE) {
		operations = null;
	}
	overrides = overrides || {};

	var opHash = {};
	if (operations && operations.length) {
		for (var i = 0; i < operations.length; i++) {
			var id = operations[i];
			ZmOperation.defineOperation(id, overrides[id]);
			ZmOperation.addOperation(parent, id, opHash, null, overrides[id] && overrides[id].htmlElId);
		}
	}

	return opHash;
};

/**
* Creates an operation descriptor. The ID of an existing operation can be passed
* in to use as a base, with overridden properties passed in a hash. A new operation
* can be defined by passing its properties in a hash.
*
* @param {String}	baseId		the ID of an existing operation
* @param {Hash}	overrides	the property overrides for the operation
*/
ZmOperation.defineOperation =
function(baseId, overrides) {
	var id = (overrides && overrides.id) || (baseId && baseId.id) || baseId || Dwt.getNextId();
	var textKey = (overrides && overrides.textKey) || ZmOperation.getProp(baseId, "textKey");
	var text = ZmOperation.getProp(baseId, "translatedText") || (textKey && ZmMsg[textKey]);
	var tooltipKey = (overrides && overrides.tooltipKey) || ZmOperation.getProp(baseId, "tooltipKey");
	var tooltip = ZmOperation.getProp(baseId, "translatedTooltip") || (tooltipKey && ZmMsg[tooltipKey]);
	var image = ZmOperation.getProp(baseId, "image");
	var showImageInToolbar = ZmOperation.getProp(baseId, "showImageInToolbar");
	var showTextInToolbar = ZmOperation.getProp(baseId, "showTextInToolbar");
	var disImage = ZmOperation.getProp(baseId, "disImage");
	var enabled = (overrides && (overrides.enabled !== false));
	var style = ZmOperation.getProp(baseId, "style");
	var shortcut = ZmOperation.getProp(baseId, "shortcut");
	var role = ZmOperation.getProp(baseId, "role");
	var ariaLabelkey = ZmOperation.getProp(baseId, "ariaLabel");
	var ariaLabel = ariaLabelkey && ZmMsg[ariaLabelkey]

    var opDesc = {id:id, text:text, image:image, showImageInToolbar:showImageInToolbar, showTextInToolbar:showTextInToolbar, disImage:disImage, enabled:enabled,
				  tooltip:tooltip, style:style, shortcut:shortcut, role: role, ariaLabel: ariaLabel};
	if (overrides) {
		for (var i in overrides) {
			opDesc[i] = overrides[i];
		}
	}

	ZmOperation._operationDesc[id] = opDesc;
	
	return opDesc;
};

/**
 * Gets the value of a given property for a given operation.
 *
 * @param {String}	id		the operation ID
 * @param {String}	prop	the name of an operation property
 * 
 * @return	{Object}	the value
 */
ZmOperation.getProp =
function(id, prop) {
	var value = null;
	var setup = ZmOperation.SETUP[id];
	if (setup) {
		value = setup[prop];
		if (!value && (prop == "disImage") && setup.image) {
			value = setup.image;
		}
	}

	return value;
};

/**
 * Checks if the operation is a separator or spacer.
 * 
 * @param	{String}	id		the id
 * @return	{Boolean}	<code>true</code> if the operation is a spacer
 */
ZmOperation.isSep =
function(id) {
	return (id == ZmOperation.SEP || id == ZmOperation.SPACER || id == ZmOperation.FILLER);
};

/**
 * Adds the operation.
 * 
 * @param {DwtComposite}	parent		the containing widget (toolbar or menu)
 * @param	{String}		id		the id
 * @param	{Hash}		opHash		a hash
 * @param	{String}	[index]		the index
 */
ZmOperation.addOperation =
function(parent, id, opHash, index, htmlElId) {

	var opDesc = ZmOperation._operationDesc[id] || ZmOperation.defineOperation(id);

	if (id == ZmOperation.SEP) {
        if (parent instanceof DwtMenu) {
            parent.createSeparator(index);
        }
        else {
            parent.addSeparator(null, index);
        }
    } else if (id == ZmOperation.SPACER) {	// toolbar only
		parent.addSpacer(null, index);
	} else if (id == ZmOperation.FILLER) {	// toolbar only
		parent.addFiller(null, index);
	} else {
		if (index != null) {
			opDesc.index = index;
		}
		opHash[id] = parent.createOp(id, opDesc, htmlElId);
	}
	var callback = ZmOperation.CALLBACK[id];
	if (callback) {
		callback.run(opHash[id]);
	}
};

/**
 * Adds a deferred menu.
 * 
 * @param	{function}	addMenuFunc		the add menu function
 * @param {DwtComposite}	parent		the containing widget (toolbar or menu)
 */
ZmOperation.addDeferredMenu =
function(addMenuFunc, parent /* ... */) {
    var args = [parent];
    for (var i = 2; i < arguments.length; i++) {
        args.push(arguments[i]);
    }
	var callback = new AjxCallback(null, addMenuFunc, args);
	parent.setMenu(callback);
};

/**
 * Removes the operation.
 * 
 * @param {DwtComposite}	parent		the containing widget (toolbar or menu)
 * @param	{String}	id		the id
 * @param	{Hash}	opHash		a hash
 */
ZmOperation.removeOperation =
function(parent, id, opHash) {
	var op = parent.getOp(id);
	if (op) {
		op.dispose();
		delete opHash[id];
	}
};

/**
 * Replaces the attributes of one operation with those of another, wholly or in part.
 *
 * @param {DwtComposite}	parent		the parent widget
 * @param {String}	oldOp		the ID of operation to replace
 * @param {String}	newOp		the ID of new operation to get replacement attributes from
 * @param {String}	text		the new text (overrides text of newOp)
 * @param {String}	image		the new image (overrides image of newOp)
 * @param {String}	disImage	the new disabled image (overrides that of newOp)
 */
ZmOperation.setOperation =
function(parent, oldOp, newOp, text, image, disImage) {
	var op = parent.getOp(oldOp);
	if (!op) return;

	op.setText(text ? text : ZmMsg[ZmOperation.getProp(newOp, "textKey")]);
	op.setImage(image ? image : ZmOperation.getProp(newOp, "image"));
};

/**
 * Takes a list of operations and removes any who have a corresponding setting that's
 * not set. Also deals with the fact that you don't want a separator or a spacer unless
 * there's stuff on either side of it.
 * 
 * @param	{Array}	list		a list of {ZmOperation} objects
 * @return	{Array}		a list of {ZmOperation} objects
 */
ZmOperation.filterOperations =
function(list) {
	var newList = [];
	if (!(list && list.length)) { return newList; }
	
	// remove disabled operations
	for (var i = 0; i < list.length; i++) {
		var op = list[i];
		if (!op) {
			continue;
		}
		if (appCtxt.checkPrecondition(ZmOperation.PRECONDITION[op])) {
			newList.push(op);
		}
	}
	// reduce multiple consecutive separators to the first one
	var newList1 = [];
	var gotSep = false;
	for (var i = 0; i < newList.length; i++) {
		var op = newList[i];
		if (op == ZmOperation.SEP || op == ZmOperation.SPACER) {
			if (!gotSep) {
				newList1.push(op);
			}
			gotSep = true;
		} else {
			newList1.push(op);
			gotSep = false;
		}
	}
	// remove separator at beginning or end
	if (newList1 && newList1.length) {
		if (newList1[0] == ZmOperation.SEP || newList1[0] == ZmOperation.SPACER) {
			newList1.shift();
		}
		var i = newList1.length - 1;
		if (newList1[i] == ZmOperation.SEP || newList1[i] == ZmOperation.SPACER || newList1[i] == ZmOperation.FILLER) {
			newList1.pop();
		}
	}
	
	return newList1;
};

/**
 * Adds a "New" submenu. Overrides are used because we don't want "New" at the
 * beginning of each label.
 *
 * @param {DwtComposite}		parent		the parent widget
 * @return	{ZmActionMenu}	the menu
 */
ZmOperation.addNewMenu =
function(parent) {
	var list = ZmOperation.NEW_ITEM_OPS;
	list.push(ZmOperation.SEP);
	list = list.concat(ZmOperation.NEW_ORG_OPS);

	var overrides = {};
	for (var i = 0; i < list.length; i++) {
		var op = list[i];
		var htmlElId = parent._htmlElId + "_" + op;
		overrides[op] = {htmlElId: htmlElId};
		var textKey = ZmOperation.NEW_ITEM_KEY[op] || ZmOperation.NEW_ORG_KEY[op];
		if (textKey) {
			overrides[op].textKey = textKey;
		}
	}

	var menu = new ZmActionMenu({parent:parent, menuItems:list, overrides:overrides});
	parent.setMenu(menu);

	return menu;
};

/**
 * Adds a "Search" submenu for searching from/to sender/recipient.
 *
 * @param {DwtComposite}	parent		parent widget (a toolbar or action menu)
 * @return	{ZmActionMenu}	the menu
 */
ZmOperation.addSearchMenu =
function(parent) {
	var list = [ZmOperation.SEARCH, ZmOperation.SEARCH_TO];

	var menu = new ZmActionMenu({parent:parent, menuItems:list});
	parent.setMenu(menu);

	return menu;
};

/**
 * Adds a contact group menu for creating a contacts from the contact list
 * @param {DwtComposite}    parent  parent widget (a toolbar or action menu)
 * @return {ZmActionMenu} the menu
 */
ZmOperation.addContactGroupMenu =
function(parent) {
	var contactGroupMenu = new ZmContactGroupMenu(parent);
	parent.setMenu(contactGroupMenu);
	return contactGroupMenu;
};

/**
 * Adds a "Tag" submenu for tagging items.
 *
 * @param {DwtComposite}	parent		parent widget (a toolbar or action menu)
 * @return	{ZmTagMenu}	the menu
 */
ZmOperation.addTagMenu =
function(parent) {
	var tagMenu = new ZmTagMenu(parent);
	parent.setMenu(tagMenu);
	return tagMenu;
};


/**
* Adds a "Move" submenu for tagging items.
*
* @param {DwtComposite}	parent		parent widget (a toolbar or action menu)
* @return	{ZmTagMenu}	the menu
*/
ZmOperation.addMoveMenu =
function(parent) {
	var moveMenu = new DwtMenu(parent); //this is a dummy menu just so the drop-down would appear
	parent.setMenu(moveMenu);
	return moveMenu;
};

/**
 * Adds a color submenu for choosing tag color.
 *
 * @param {DwtComposite}	parent		parent widget (a toolbar or action menu)
 * @param {boolean} hideNoFill True to hide the no-fill/use-default option.
 * @return	{ZmPopupMenu}	the menu
 */
ZmOperation.addColorMenu =
function(parent, hideNoFill) {
    var menu = new ZmColorMenu({parent:parent,image:"Tag",hideNone:true,hideNoFill:hideNoFill});
    parent.setMenu(menu);
    return menu;
};

/**
 * Gets the tooltip for the operation with the given ID. If the operation has a shortcut associated
 * with it, a shortcut hint is appended to the end of the tooltip.
 *
 * @param {String}	id		the operation ID
 * @param {String}	keyMap	the key map (for resolving shortcut)
 * @param {String}	tooltip	the tooltip override
 * @return	{String}	the tool tip
 */
ZmOperation.getToolTip =
function(id, keyMap, tooltip) {
	var opDesc = ZmOperation._operationDesc[id] || ZmOperation.defineOperation(id);
	tooltip = tooltip || opDesc.tooltip;
	var sc = tooltip && opDesc.shortcut && appCtxt.getShortcutHint(keyMap, opDesc.shortcut);
	return sc ? [tooltip, sc].join("") : tooltip;
};
}
if (AjxPackage.define("zimbraMail.core.ZmMimeTable")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the mime table information utility class.
 * 
 */

/**
 * Creates the mime table class
 * @class
 * This class represents a mime table that contains utility methods for managing mime types.
 * 
 */
ZmMimeTable = function() {
};

// IGNORE means the client will not display these attachment types to the user
ZmMimeTable.APP						= "application";
ZmMimeTable.APP_FLASH				= "application/x-shockwave-flash";
ZmMimeTable.APP_ADOBE_PDF			= "application/pdf";
ZmMimeTable.APP_ADOBE_PS			= "application/postscript";
ZmMimeTable.APP_APPLE_DOUBLE 		= "application/applefile";		// IGNORE
ZmMimeTable.APP_EXE					= "application/exe";
ZmMimeTable.APP_MS_DOWNLOAD			= "application/x-msdownload";
ZmMimeTable.APP_MS_EXCEL			= "application/vnd.ms-excel";
ZmMimeTable.APP_MS_PPT				= "application/vnd.ms-powerpoint";
ZmMimeTable.APP_MS_PROJECT			= "application/vnd.ms-project";
ZmMimeTable.APP_MS_TNEF				= "application/ms-tnef"; 		// IGNORE
ZmMimeTable.APP_MS_TNEF2 			= "application/vnd.ms-tnef"; 	// IGNORE (added per bug 2339)
ZmMimeTable.APP_SIGNATURE           = "application/pkcs7-signature"; // IGNORE (added per bug 69476)
ZmMimeTable.APP_MS_VISIO			= "application/vnd.visio";
ZmMimeTable.APP_MS_WORD				= "application/msword";
ZmMimeTable.APP_OCTET_STREAM		= "application/octet-stream";
ZmMimeTable.APP_OPENXML_DOC			= "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
ZmMimeTable.APP_OPENXML_EXCEL		= "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
ZmMimeTable.APP_OPENXML_PPT			= "application/vnd.openxmlformats-officedocument.presentationml.presentation";
ZmMimeTable.APP_XML     			= "application/xml";
ZmMimeTable.APP_ZIMBRA_DOC			= "application/x-zimbra-doc";
ZmMimeTable.APP_ZIP					= "application/zip";
ZmMimeTable.APP_ZIP2				= "application/x-zip-compressed";
ZmMimeTable.APP_RTF					= "application/rtf";
ZmMimeTable.APP_OPENDOC_ODT			= "application/vnd.oasis.opendocument.text";
ZmMimeTable.APP_OPENDOC_ODP 		= "application/vnd.oasis.opendocument.presentation";
ZmMimeTable.APP_OPENDOC_ODS 		= "application/vnd.oasis.opendocument.spreadsheet";
ZmMimeTable.AUDIO					= "audio";
ZmMimeTable.AUDIO_WAV				= "audio/x-wav";
ZmMimeTable.AUDIO_MP3				= "audio/mpeg";
ZmMimeTable.IMG						= "image";
ZmMimeTable.IMG_GIF					= "image/gif";
ZmMimeTable.IMG_BMP					= "image/bmp";
ZmMimeTable.IMG_JPEG				= "image/jpeg";
ZmMimeTable.IMG_PJPEG				= "image/pjpeg";				// bug 23607
ZmMimeTable.IMG_PNG					= "image/png";
ZmMimeTable.IMG_TIFF				= "image/tiff";
ZmMimeTable.MSG_RFC822				= "message/rfc822";
ZmMimeTable.MSG_READ_RECEIPT		= "message/disposition-notification";
ZmMimeTable.MULTI_ALT				= "multipart/alternative"; 		// IGNORE
ZmMimeTable.MULTI_MIXED				= "multipart/mixed"; 			// IGNORE
ZmMimeTable.MULTI_RELATED			= "multipart/related"; 			// IGNORE
ZmMimeTable.MULTI_APPLE_DBL 		= "multipart/appledouble"; 		// IGNORE
ZmMimeTable.MULTI_DIGEST			= "multipart/digest";			// IGNORE
ZmMimeTable.TEXT					= "text";
ZmMimeTable.TEXT_RTF				= "text/enriched";
ZmMimeTable.TEXT_HTML				= "text/html";
ZmMimeTable.TEXT_CAL				= "text/calendar"; 				// IGNORE
ZmMimeTable.TEXT_JAVA				= "text/x-java";
ZmMimeTable.TEXT_X_VCARD			= "text/x-vcard";
ZmMimeTable.TEXT_VCARD				= "text/vcard";
ZmMimeTable.TEXT_DIRECTORY  	    = "text/directory";
ZmMimeTable.TEXT_PLAIN				= "text/plain";
ZmMimeTable.TEXT_XML				= "text/xml";
ZmMimeTable.TEXT_CSV				= "text/csv";
ZmMimeTable.VIDEO					= "video";
ZmMimeTable.VIDEO_WMV				= "video/x-ms-wmv";
ZmMimeTable.XML_ZIMBRA_SHARE		= "xml/x-zimbra-share";

// Formats for text/plain
ZmMimeTable.FORMAT_FLOWED			= "flowed";

ZmMimeTable._icons = {
	doc: "data:image/png;base64,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",
	exe: "data:image/png;base64,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",
	generic: "data:image/png;base64,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",
	image: "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAwCAYAAAEgBckRAAAAGXRFWHRTb2Z0d2FyZQBBZG9iZSBJbWFnZVJlYWR5ccllPAAABo9JREFUeNpiYMABGEFE1VqL/1glfv78+f/WrVtggb///jEoyMszMIE4b9++Zbh58yaDopISWPDPn78QHc+ePfvPx8/PsGf3bqDgHwYbWzsGFpDEn79/GXZs3w43/9+/vwwAAcSA01XYXASWQHYR2FigHXJAB4DtUFNTY4A4+xfD7du3GP4C7WSBqTx69CjDu3fvGOQVFMASYH9s3rwZLAgCDx88ALrqPwNAAL6oIAdAEIZ58Af7wPyB35evyDPkZFxZl0DUiBCykHRd2w0jGZ0YobL+ggD48+q3Z8sYReQBPN2RXTblvIdWuKT5i7EBU9rir7qEMXs3EHiPvq8AFg2UFpKOUkBG7obr5miCyAivzQNdVAEoJ5ccBmEYiJoKWOdI5QA5FOfqhmWOA8qmgNqQmchVAkhVLHkTWf7MvEhtNDWSxpgetRPafx6ko2FCII+kCY85UXlAHWOMgGjv/fSTFexba4vibd+jHx95RfSfw0C1ihsAG1KLl3mmzBSUEp+M0wna2TlXmHeZoFN0jSuANyshzsXphxJXqfAhMJv4o0LX91wDB34zlrACO0tiaVvf4yEAbdWSgyAMRIfPQeQGXkEIJ/BCrjwWpRvizhOIkYXsCYEEF7XTMAWKRaPYhNA29M1Mee8NKe4sX1tYeRz3J4dKWB28T5y58OehLeMXEDQCRYORnlEPdV2nM4tBb8qy7OtgYRRpzglJAN/2IbKccw5VVU1Yb8v4kudwlQ+yCsRAbH8pG+wTtqvoHh2kjKn9XRiqfa1JEJ8FiGS5BIx6pYxNZbEkkU69GWkYtBb8dz/vlYvbms1Y89R4rAEo4ziONSvosL5jGMCK4jZ4BHUDIaY0tdFtCRiMeY+rRts2qUMZ38uSea6nDoNRplob10D+JGA+RyZJYB4EweEpAPVVz9owDEQVYbfdTDrZoUNDCl46pIPntkvzt/qzvJWSQgJebEjTxWBI8RCv3dJB1anRhz+kKAkp9OCIksC705Pu7gn9SSXvOQttLaXN7g5DQzpR8qyBnrzZ/f8AzpV3ixzHOQqEECKFFq8N9mWOelVVvXie93AoOJvLoBaVguR+2e/3OucBuG3mbOhvyxeeCWoAoZpVO2bYAOj1cGgOABaGoXgm2NICb5kmRdoArPHRbhrHMVtHUYR83zdy3pRo/DftNS2KQqyTJOkEr9brGhgh9XlgDEA7oaz58bgFPp/NUJZl8NxTdtBu3VqKXNcVg74J/r5YgNpm/30sl+hrKwyImAUWFOkOFGiBrFWj+gfJSWNJEQfP0pQdNhw6ONBi1Me2FHHwsiyFcjABy1skFYV2B1Cdn6uVALeS9jWKduwAthkMBsgPglbx1F0e6Nt02rkDp0kLWJ7nzPe1s/OLGjgLwOUhvy1Pk8mOrLvlSjNzThGmAV5NLfdQ8M3mW6oKepjPGON76mJ4EML1EddGms/WkPkVzTej0SOsfwRoz1pWEwai6BXcaFRwYzcVqUgjcdOuXEq/rO2HCe1WrBaDBbsQgimRuggW66OWpnODE2fyqHn4qKUXghJluM+Zc87EbJTnhnzU4HfbPYFDN3bchS+u4bjsFgOhc1yD47PaXlDX0cPG/wD8kKeo0HcbRmUtjmra6Cf72/tkcpfP56/ih85gUMftRGRjANPpFOr1+kGCq1arkEqnHcSfqhC+ApjNZtZ3UjLziZpx6xRmskuzrhE0ShHp62gEQirlqlxsBIxulkwmrRsGRKeodI/HYxN/Fwkrs98+hGkVfXWBQrEbR2s4mAHBA6DW7/eh2+1y74bDoRkg1d7C9jgLBZ3/XyOijcRvl8OJLKjZbJqgTJIkE1ivnTIcFeAdt0oRPgBsF5wFrAIOOX2H7HZTxtFxXdettZ4IrcPnlKxXKpV4/sINLi8DGhCxAtj3SGjZjKNO6eW4QjjSc6/nuZ46GMCEcFW8JWIjcOqXvN4X30arfCwW0Gg0IJvNglguc46/EaceVu0ShN8xNXANwjAiBkBbRe50LPaJ0r+iKJBIJECqVMyMUwXAvwbkr4UgbAtRx19UFWRZ9jw7MOuB9SuXCNwk8NBDjDeZmqZBu9XazdbFapWOFnJIFP4C+FpRfUr3d2l4KOLDtxDvuGH4hNN0V8FdoVAo7B0H5XInK9jCan/OUTErMJ/PIZPJQCwWc93HRbEM56L4Izq0YxXnyek+hJ497pFxVvrjVAlVVduCIFzEyf4OAWDtvh1fLj/xNvSxWDy7hL9g372EhOmMs4Y3AAAAAElFTkSuQmCC",
	audio: "data:image/png;base64,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",
	presentation: "data:image/png;base64,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",
	html: "data:image/png;base64,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",
	video: "data:image/png;base64,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",
	spreadsheet: "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAwCAYAAAEgBckRAAAAGXRFWHRTb2Z0d2FyZQBBZG9iZSBJbWFnZVJlYWR5ccllPAAABOFJREFUeNpiYMABGEGEQqHrf6wSP3/+hEv8/vOH4c/v3wwsIM6xY8cY3r59C1ft4ODIwITN/D9//0KM+vL163+Q9j9//oIF//37ywAQQAw4XYXNRWAJZBdBXPgL6DKoqzZv3oyi2s7eHrur/oJcBTIKph0k8Bfsqv8MAAGEM0hwAbDdZkq6eBX9//8fjFHCFj0Y0YGjkxPDf5iT0L0OipB/f/+BQ/Lvv39ghf/+/wP7gQWmiJAN1jY2YGfB4wiXif+BNMQPYN8wAASgnIx1AARhIKqNnyUDkyz8L274TQRMTERbiIQEVOh86T16x9A7Y89JcTT0Okx/GeQ5pAp8iQ8srD/jMYwxOp31aZqUsmhd+A+cB5fmN1w+YlWRcgyf5WKtLZEWIYZVqcJhZqyORJtrQ6lT0AnpDYO2BjGqd+c0IFeTmLoEANstgGA57jyQMmCgMnjQv5uR5CxNIiA9rZKce7CVAKQCUIkBTv9I+fkfEH/58gWRjqhpMCyz/QcmAAwLsBVNuPITNgDKY6D0CbOEhdphDvYNw3/cFujo6DD8BqZz9KAwNTMDOQvhfRCGlj1gF0OcDc0PDPC8gGHBlStXMILIzc2d4fSpU0T5AJbn/0MtJSqI8NbM2MoIWG3wH62iIpQq4JEHD2NYUKBWM9Bak+H7928HGGE2P3v+fD/EkP8oXoSLQXWi0tCwh5v9H85kBpZDioqKDQABxECXnEyjsugCsLAzZAIavp9GjgeXzjQv7Ia+BSxGcpoMLCyUFUmw5iI6+wHDbgbGV69e7efn53cg13BwvQxq8aKXU0AsJCjISFEQ4TMc5hMmWhkOswAj8NErFlBlAypdQRURISAoKMhgYGhIHR9gj2wG6gURDiuQLKBRPkAvuolKpsREKKrLIWwJcXFGjEhGry4FgBEHrJAYPn38CK+DEfUGap3BzMzCwM3DjRJEWJstyMAD2N4GGX769GmCwcPPL8Cgq6eLEkQE4+Df//8k9Lf+o7QoiLIAOcmRaglNLED3AUoqIje1oBqKMFxaSpKR6Q+wqUILw+FBBLTgIC0M//XrN6JV8fTp0wZGRkZ71PYmlA3rc6G3Q5HCG71NBAIqysqOIBogQHtWs9ogEIRHMFVQNGdFqIHmKO3Vi/SYp2r6Zs0TNBffwGsDlRQES6y7VNkfdTfJJtHSuayMIvPt/Ow3sxrpoJphrOslgXHLpqZDa5Z3IcULTEteEZAmzRKYniRXYV2Tp43/AGSap3OprwppxlrQ09qTbSd697XfvwVB8KzfegePNbw9v6GHD43d8BbAoZIDkKYpfNaEbkjiOMZrnud4SHNMK0cxA0Zv2zY8LJccAPJkFgJAxg/N/KlxVVlKfyvHDBm613oIbh9CsuSTDi2SJ1UTACAkeFMBQBredMiyOdAkqExVcVwXd3GnVBU+xrsNZ6/MhABEF3e49Vyt8N8/djvpkbCMuPM5RFHUCaJSGUJkM36RJB7whBoAxBXBBQoRsCNw5UmM5/I9B5SaiRPdCJPPQgC+7+PBW9eJWRFuRs+GaUIYhsy3fD/bqWP0SO4Mk0tcdo90EVfxPE9YVQ6/fzUMA+5rAKdUld5Q4XS0LzCAoijAcRzQNO1skqXO8O4dBybX2qlElmXvlmU96rMZfwKOyPCy/Ea3odvFInyCvyA/oHKrnTBRFkEAAAAASUVORK5CYII=",
	archive: "data:image/png;base64,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"
};

ZmMimeTable._table = {};

// only add types which are NOT ignored by the client	
ZmMimeTable._table[ZmMimeTable.APP]					= {desc: ZmMsg.application, image: "ExeDoc", imageLarge: "ExeDoc_48", dataUri: ZmMimeTable._icons.exe, query: "application/*"};
ZmMimeTable._table[ZmMimeTable.APP_FLASH]			= {desc: ZmMsg.swf, image: "GenericDoc", imageLarge: "GenericDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.APP_ADOBE_PDF]		= {desc: ZmMsg.adobePdfDocument, image: "Doc", imageLarge: "Doc_48", dataUri: ZmMimeTable._icons.doc};
ZmMimeTable._table[ZmMimeTable.APP_ADOBE_PS]		= {desc: ZmMsg.adobePsDocument, image: "GenericDoc", imageLarge: "GenericDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.APP_EXE]				= {desc: ZmMsg.application, image: "ExeDoc", imageLarge: "ExeDoc_48", dataUri: ZmMimeTable._icons.exe};
ZmMimeTable._table[ZmMimeTable.APP_MS_DOWNLOAD]		= {desc: ZmMsg.msDownload, image: "ExeDoc", imageLarge: "ExeDoc_48", dataUri: ZmMimeTable._icons.exe};
ZmMimeTable._table[ZmMimeTable.APP_MS_EXCEL]		= {desc: ZmMsg.msExcelDocument, image: "MSExcelDoc", imageLarge: "MSExcelDoc_48", dataUri: ZmMimeTable._icons.spreadsheet, query: "excel"};
ZmMimeTable._table[ZmMimeTable.APP_OPENDOC_ODS]		= {desc: ZmMsg.msExcelDocument, image: "MSExcelDoc", imageLarge: "MSExcelDoc_48", dataUri: ZmMimeTable._icons.spreadsheet};
ZmMimeTable._table[ZmMimeTable.APP_MS_PPT]			= {desc: ZmMsg.msPPTDocument, image: "MSPowerpointDoc", imageLarge: "MSPowerpointDoc_48", dataUri: ZmMimeTable._icons.presentation, query: "powerpoint"};
ZmMimeTable._table[ZmMimeTable.APP_OPENDOC_ODP]		= {desc: ZmMsg.msPPTDocument, image: "MSPowerpointDoc", imageLarge: "MSPowerpointDoc_48", dataUri: ZmMimeTable._icons.presentation};
ZmMimeTable._table[ZmMimeTable.APP_MS_PROJECT]		= {desc: ZmMsg.msProjectDocument, image: "MSProjectDoc", imageLarge: "MSProjectDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.APP_MS_VISIO]		= {desc: ZmMsg.msVisioDocument, image: "MSVisioDoc", imageLarge: "MSVisioDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.APP_MS_WORD]			= {desc: ZmMsg.msWordDocument, image: "MSWordDoc", imageLarge: "MSWordDoc_48", dataUri: ZmMimeTable._icons.doc, query: "word"};
ZmMimeTable._table[ZmMimeTable.APP_OPENDOC_ODT]		= {desc: ZmMsg.msWordDocument, image: "MSWordDoc", imageLarge: "MSWordDoc_48", dataUri: ZmMimeTable._icons.doc};
ZmMimeTable._table[ZmMimeTable.APP_RTF]				= {desc: ZmMsg.msWordDocument, image: "MSWordDoc", imageLarge: "MSWordDoc_48", dataUri: ZmMimeTable._icons.doc};
ZmMimeTable._table[ZmMimeTable.APP_OCTET_STREAM]	= {desc: ZmMsg.unknownBinaryType, image: "UnknownDoc", imageLarge: "UnknownDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.APP_OPENXML_DOC]		= {desc: ZmMsg.msWordDocument, image: "MSWordDoc", imageLarge: "MSWordDoc_48", dataUri: ZmMimeTable._icons.doc};
ZmMimeTable._table[ZmMimeTable.APP_OPENXML_EXCEL]	= {desc: ZmMsg.msExcelDocument, image: "MSExcelDoc", imageLarge: "MSExcelDoc_48", dataUri: ZmMimeTable._icons.spreadsheet};
ZmMimeTable._table[ZmMimeTable.APP_OPENXML_PPT]		= {desc: ZmMsg.msPPTDocument, image: "MSPowerpointDoc", imageLarge: "MSPowerpointDoc_48", dataUri: ZmMimeTable._icons.presentation};
ZmMimeTable._table[ZmMimeTable.APP_XML]			    = {desc: ZmMsg.xmlDocument, image: "GenericDoc", imageLarge: "GenericDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.APP_ZIMBRA_DOC]  	= {desc: ZmMsg.zimbraDocument, image: "Doc", imageLarge: "Doc_48", dataUri: ZmMimeTable._icons.doc};
ZmMimeTable._table[ZmMimeTable.APP_ZIP]				= {desc: ZmMsg.zipFile, image: "ZipDoc", imageLarge: "ZipDoc_48", dataUri: ZmMimeTable._icons.archive};
ZmMimeTable._table[ZmMimeTable.APP_ZIP2]			= {desc: ZmMsg.zipFile, image: "ZipDoc", imageLarge: "ZipDoc_48", dataUri: ZmMimeTable._icons.archive};
ZmMimeTable._table[ZmMimeTable.AUDIO]				= {desc: ZmMsg.audio, image: "AudioDoc", imageLarge: "AudioDoc_48", dataUri: ZmMimeTable._icons.audio};
ZmMimeTable._table[ZmMimeTable.AUDIO_WAV]			= {desc: ZmMsg.waveAudio, image: "AudioDoc", imageLarge: "AudioDoc_48", dataUri: ZmMimeTable._icons.audio};
ZmMimeTable._table[ZmMimeTable.AUDIO_MP3]			= {desc: ZmMsg.mp3Audio, image: "AudioDoc", imageLarge: "AudioDoc_48", dataUri: ZmMimeTable._icons.audio};
ZmMimeTable._table[ZmMimeTable.IMG]					= {desc: ZmMsg.image, image: "ImageDoc", imageLarge: "ImageDoc_48", dataUri: ZmMimeTable._icons.image, query: "image/*"};
ZmMimeTable._table[ZmMimeTable.IMG_BMP]				= {desc: ZmMsg.bmpImage, image: "ImageDoc", imageLarge: "ImageDoc_48", dataUri: ZmMimeTable._icons.image, query: "bmp"};
ZmMimeTable._table[ZmMimeTable.IMG_GIF]				= {desc: ZmMsg.gifImage, image: "ImageDoc", imageLarge: "ImageDoc_48", dataUri: ZmMimeTable._icons.image, query: "gif"};
ZmMimeTable._table[ZmMimeTable.IMG_JPEG]			= {desc: ZmMsg.jpegImage, image: "ImageDoc", imageLarge: "ImageDoc_48", dataUri: ZmMimeTable._icons.image, query: "jpeg"};
ZmMimeTable._table[ZmMimeTable.IMG_PNG]				= {desc: ZmMsg.pngImage, image: "ImageDoc", imageLarge: "ImageDoc_48", dataUri: ZmMimeTable._icons.image, query: "png"};
ZmMimeTable._table[ZmMimeTable.IMG_TIFF]			= {desc: ZmMsg.tiffImage, image: "ImageDoc", imageLarge: "ImageDoc_48", dataUri: ZmMimeTable._icons.image};
ZmMimeTable._table[ZmMimeTable.MSG_RFC822]			= {desc: ZmMsg.mailMessage, image: "MessageDoc", imageLarge: "MessageDoc_48", dataUri: ZmMimeTable._icons.doc};
ZmMimeTable._table[ZmMimeTable.TEXT]				= {desc: ZmMsg.textDocuments, image: "Doc", imageLarge: "Doc_48", dataUri: ZmMimeTable._icons.doc};
ZmMimeTable._table[ZmMimeTable.TEXT_RTF]			= {desc: ZmMsg.enrichedText, image: "Doc", imageLarge: "Doc_48", dataUri: ZmMimeTable._icons.doc};
ZmMimeTable._table[ZmMimeTable.TEXT_HTML]			= {desc: ZmMsg.htmlDocument, image: "HtmlDoc", imageLarge: "HtmlDoc_48", dataUri: ZmMimeTable._icons.html};
ZmMimeTable._table[ZmMimeTable.TEXT_JAVA]			= {desc: ZmMsg.javaSource, image: "GenericDoc", imageLarge: "GenericDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.TEXT_PLAIN]			= {desc: ZmMsg.textFile, image: "Doc", imageLarge: "Doc_48", dataUri: ZmMimeTable._icons.doc, query: "text"};
ZmMimeTable._table[ZmMimeTable.TEXT_XML]			= {desc: ZmMsg.xmlDocument, image: "GenericDoc", imageLarge: "GenericDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.TEXT_CSV]			= {desc: ZmMsg.csvDocument, image: "MSExcelDoc", imageLarge: "MSExcelDoc_48", dataUri: ZmMimeTable._icons.spreadsheet};
ZmMimeTable._table[ZmMimeTable.VIDEO]				= {desc: ZmMsg.video, image: "VideoDoc", imageLarge: "VideoDoc_48", dataUri: ZmMimeTable._icons.video};
ZmMimeTable._table[ZmMimeTable.VIDEO_WMV]			= {desc: ZmMsg.msWMV, image: "VideoDoc", imageLarge: "VideoDoc_48", dataUri: ZmMimeTable._icons.video};
ZmMimeTable._table[ZmMimeTable.TEXT_DIRECTORY]		= {desc: ZmMsg.vCard, image: "GenericDoc", imageLarge: "GenericDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.TEXT_VCARD]			= {desc: ZmMsg.vCard, image: "GenericDoc", imageLarge: "GenericDoc_48", dataUri: ZmMimeTable._icons.generic};
ZmMimeTable._table[ZmMimeTable.TEXT_X_VCARD]		= {desc: ZmMsg.vCard, image: "GenericDoc", imageLarge: "GenericDoc_48", dataUri: ZmMimeTable._icons.generic};

ZmMimeTable.getInfo =
function(type, createIfUndefined) {
	var entry = ZmMimeTable._table[type];
	if (!entry && createIfUndefined) {
		entry = ZmMimeTable._table[type] = {desc: type, image: "UnknownDoc", imageLarge: "UnknownDoc_48", dataUri: ZmMimeTable._icons.generic};
	}
	if (entry) {
		if (!entry.type) {
			entry.type = type;
		}
	} else {
		// otherwise, check if main category is in table
		var baseType = type.split("/")[0];
		if (baseType) {
			entry = ZmMimeTable._table[baseType];
		}
	}
	return entry;
};

/**
 * Checks if the type is ignored.
 * 
 * @param	{constant}	type		the type
 * @return	{Boolean}	<code>true</code> if the type is ignored
 */
ZmMimeTable.isIgnored = 
function(type) {
	return (type == ZmMimeTable.MULTI_ALT ||
			type == ZmMimeTable.MULTI_MIXED ||
			type == ZmMimeTable.MULTI_RELATED ||
			type == ZmMimeTable.MULTI_APPLE_DBL ||
			type == ZmMimeTable.APP_MS_TNEF ||
			type == ZmMimeTable.APP_MS_TNEF2 ||
            type == ZmMimeTable.APP_SIGNATURE);
};

/**
 * Checks if the type is renderable.
 * 
 * @param	{constant}	type		the type
 * @return	{Boolean}	<code>true</code> if the type is renderable
 */

ZmMimeTable.isRenderableText =
function(type, body) {
	return ((type === ZmMimeTable.TEXT_HTML && !body) || 
			(type === ZmMimeTable.TEXT_PLAIN && !body));
};

ZmMimeTable.isRenderable =
function(type, textOnly) {
	return (type === ZmMimeTable.TEXT_HTML ||
			type === ZmMimeTable.TEXT_PLAIN ||
			(!textOnly && ZmMimeTable.isRenderableImage(type)) ||
			(!textOnly && type === ZmMimeTable.APP_ADOBE_PDF));
};

ZmMimeTable.isTextType =
function(type){
    return (type.match(/^text\/.*/) &&
            type != ZmMimeTable.TEXT_HTML &&
            type != ZmMimeTable.TEXT_CAL);  
};

/**
 * Checks if the type is a renderable image.
 * 
 * @param	{constant}	type		the type
 * @return	{Boolean}	<code>true</code> if the type is a renderable image
 */
ZmMimeTable.isRenderableImage =
function(type) {
	return (type == ZmMimeTable.IMG_JPEG ||
			type == ZmMimeTable.IMG_GIF ||
			type == ZmMimeTable.IMG_BMP ||
			type == ZmMimeTable.IMG_PNG);
};

/**
 * Checks if the type has an HTML version.
 * 
 * @param	{constant}	type		the type
 * @return	{Boolean}	<code>true</code> if the type has an HTML version
 */
ZmMimeTable.hasHtmlVersion =
function(type) {
	return (!(ZmMimeTable.isIgnored(type) ||
			type.match(/^text\/plain/) ||
			type.match(/^image/) ||
			type.match(/^audio/) ||
			type.match(/^video/)));
};

ZmMimeTable.isMultiMedia =
function(type){
    return (type.match(/^audio/) ||
			type.match(/^video/));
};

ZmMimeTable.isWebDoc =
function(type) {
    return (type == ZmMimeTable.APP_ZIMBRA_DOC);
};

/**
 * Checks if the type is a vCard.
 * 
 * @param	{constant}	type		the type
 * @return	{Boolean}	<code>true</code> if the type is a vCard
 */
ZmMimeTable.isVcard =
function(type) {
    return (type === ZmMimeTable.TEXT_DIRECTORY || 
    		type === ZmMimeTable.TEXT_X_VCARD || 
    		type === ZmMimeTable.TEXT_VCARD);
};
}

if (AjxPackage.define("zimbraMail.share.model.ZmObjectHandler")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file contains the zimlet object handler base class.
 * 
 */

/**
 * Creates the object handler.
 * @class
 * This class defines the default implementation for a zimlet object handler.
 * <br />
 * <br />
 * To write a zimlet, see {@link ZmZimletBase}. 
 * 
 * @param	{string}	typeName	the type name
 * @param	{string}	className	the class name
 */
ZmObjectHandler = function(typeName, className) {
	if (arguments.length > 0) {
		this.init(typeName, className);
	}
}

ZmObjectHandler.prototype.constructor = ZmObjectHandler;
ZmObjectHandler.prototype.isZmObjectHandler = true;

/**
 * This method is called by the Zimlet framework to initialize the object.
 * 
 * @param	{string}	typeName	the type name
 * @param	{string}	className	the class name; if <code>null</code>, "Object" will be used
 */
ZmObjectHandler.prototype.init =
function(typeName, className) {
	this._typeName = typeName;
	this._className = className ? className : "Object";
	this.name = this.toString();
};

/**
 * Returns a string representation of the object.
 * 
 * @return		{string}		a string representation of the object
 */
ZmObjectHandler.prototype.toString = 
function() {
	// If you can find a cleaner way to get the name of 
	// a sub-class without hard coding each instance
	// in a toString() method feel free to change.
	if(!this._toString) {
		var ctor = "" + this.constructor;
		ctor = ctor.substring(0,ctor.indexOf("("));
		this._toString = ctor.substring("function ".length);
	}
	return this._toString;
};

ZmObjectHandler.prototype.getEnabled =	function() {
	return true;
};


/**
 * Gets the type name.
 * 
 * @return	{string}		the type name
 */
ZmObjectHandler.prototype.getTypeName =
function() {
	return this._typeName;
};

/**
 * Gets the class name for a given object.
 * 
 * @param	{Object}		obj			the object
 * @param	{Object}		context		the content
 * @param	{string}		spanId		ID of the SPAN
 *
 * @return	{string}		the class name
 */
ZmObjectHandler.prototype.getClassName =
function(obj, context, spanId) {
	return this._className;
};

/**
 * Gets the hovered class name for the given object.
 * 
 * @param	{Object}		obj			the object
 * @param	{Object}		context		the content
 * @param	{string}		spanId		ID of hovered SPAN
 *
 * @return	{string}		the hovered class name
 */
ZmObjectHandler.prototype.getHoveredClassName =
function(obj, context, spanId) {
	var cname = this.getClassName(obj, context, spanId);
	if (this._cachedClassNameForHovered !== cname) {
		this._cachedClassNameForHovered = cname;
		this._classNameHovered = cname + "-" + DwtCssStyle.HOVER;
	}
	return this._classNameHovered;
};

/**
 * Gets the active class name for a given object.
 * 
 * @param	{Object}		obj		the object
 * @param	{Object}		context		the content
 * @param	{string}		spanId		ID of the SPAN
 *
 * @return	{string}		the active class name
 */
ZmObjectHandler.prototype.getActiveClassName =
function(obj, context, spanId) {
	var cname = this.getClassName(obj, context, spanId);
	if (this._cachedClassNameForActive !== cname) {
		this._cachedClassNameForActive = cname;
		this._classNameActive = cname + "-" + DwtCssStyle.ACTIVE;
	}
	return this._classNameActive;
};

/**
 * @private
 */
ZmObjectHandler.prototype.findObject =
function(content, startIndex, objectMgr) {
	if (startIndex === 0) {
		this._lastMatch = null;
		this._noMatch = false;
	}
	if (this._noMatch) {return null;}
	if (this._lastMatch && this._lastMatch.index >= startIndex) {
		return this._lastMatch;
	}
	this._lastMatch = this.match(content, startIndex, objectMgr);
	this._noMatch = (this._lastMatch === null);
	return this._lastMatch;
};


/**
 * This method is used to match content for a zimlet. Zimlet implementations should
 * override this method. Usage should return a non-null result in the format of
 * <code>String.match</code> if text on the line matched the handler regular expression.
 * 
 * <pre>
 * var result = handler.match(line);
 * result[0] // should be matched string
 * result.index // should be location within line match occurred
 * </pre>
 * 
 * Handlers can also set result.context which will be passed back to
 * them during the various method calls ({@link #getToolTipText}, etc). Handlers should set
 * regex.lastIndex to startIndex and then use <code>regex.exec(content)</code>. Handlers should
 * also use the "g" option when constructing their regex.
 */
ZmObjectHandler.prototype.match =
function(content, startIndex) {
	return null;
};

/**
 * Generates content inside the <code>&lt;span&gt;</code> tag.
 * 
 * @return	{number}	the content index
 * @private
 * */
ZmObjectHandler.prototype._getHtmlContent =
function(html, idx, obj, context, spanId) {
	html[idx++] = AjxStringUtil.htmlEncode(obj, true);
	return idx;
};

/**
 * Generates the <code>&lt;span&gt;</code> tag.
 * 
 * @return	{number}	the content index
 * @private
 */
ZmObjectHandler.prototype.generateSpan = 
function(html, idx, obj, spanId, context, options) {
	html[idx++] = "<span class='";
	html[idx++] = this.getClassName(obj);
	html[idx++] = "' role='link' id='";
	html[idx++] = spanId;
	html[idx++] = "'>";
	idx = this._getHtmlContent(html, idx, obj, context, spanId, options);
	html[idx++] = "</span>";
	return idx;
};

/**
 * Checks if the handler has tool tip text.
 * 
 * @param		{Object}	obj			the object
 * @param		{Object}	context		the context
 * @return		<code>true</code> if the handler has tool tip text; <code>false</code> otherwise
 */
ZmObjectHandler.prototype.hasToolTipText =
function(obj, context) {
	return true;
};

/**
 * Gets the handler tool tip text.
 * 
 * @param		{Object}	obj			the object
 * @param		{Object}	context		the context
 * @return		{string}	the handler has tool tip text
 */
ZmObjectHandler.prototype.getToolTipText =
function(obj, context) {
	return AjxStringUtil.htmlEncode(obj);
};

/**
 * Populates the handler tool tip text.
 * 
 * @param		{Object}	obj			the object
 * @param		{Object}	context		the context
 */
ZmObjectHandler.prototype.populateToolTip =
function(obj, context) {
};

/**
 * Gets the action menu.
 * 
 * @param		{Object}	obj			the object
 * @param		{string}	span		the span element
 * @param		{Object}	context		the context
 * @return		{ZmActionMenu}	the action menu
 * 
 * @private
 */
ZmObjectHandler.prototype.getActionMenu =
function(obj, span, context) {
	return null;
};

/**
 * This method is called by the Zimlet framework when the object is selected.
 * 
 * @param		{Object}	obj			the object
 * @param		{string}	span		the span element
 * @param		{Object}	ev			the event
 * @param		{Object}	context		the context
 * @see		#clicked
 */
ZmObjectHandler.prototype.selected =
function(obj, span, ev, context) {
	return this.clicked(span, obj, context, ev);
};

/**
 * This method is called by the Zimlet framework when the object is clicked.
 * 
 * @param		{Object}	obj			the object
 * @param		{string}	span		the span element
 * @param		{Object}	ev			the event
 * @param		{Object}	context		the context
 */
ZmObjectHandler.prototype.clicked =
function(span, obj, context, ev) {
};

/**
 * This method is called when the object is hovered-over.
 * 
 * @private
 */
ZmObjectHandler.prototype.hoverOver = function(object, context, x, y) {

	var tooltip = this.getToolTipText(object, context) || '',
		content, callback;

	if (typeof(tooltip) === "string") {
		content = tooltip;
	}
	else if (tooltip.isAjxCallback || AjxUtil.isFunction(tooltip)) {
		callback = tooltip;
	}
	else if (typeof(tooltip) === "object") {
		content = tooltip.content;
		callback = tooltip.callback;
	}

	if (!content && callback && tooltip.loading) {
		content = AjxMsg.loading;
	}

	if (content) {
		this._showTooltip(object, context, x, y, content);
	}

	if (callback) {
		var callback1 = new AjxCallback(this, this._showTooltip, [ object, context, x, y ]);
		AjxTimedAction.scheduleAction(new AjxTimedAction(null, function() {
			callback.run(callback1);
		}), 0);
	}
};

ZmObjectHandler.prototype._showTooltip = function(object, context, x, y, content) {
	var shell = DwtShell.getShell(window);
	var tooltip = shell.getToolTip();
	tooltip.setContent(content);
	tooltip.popup(x, y);
	// TODO: call below is odd; not sure if it's used much, appears to be for two-step tooltips (eg a map)
	this.populateToolTip(object, context);
};

/**
 * This method is called when the handler is hovered-out.
 * 
 * @private
 */
ZmObjectHandler.prototype.hoverOut = function(object, context) {
	var shell = DwtShell.getShell(window);
	var tooltip = shell.getToolTip();
	tooltip.popdown();
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmObjectManager")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the object manager.
 * 
 */

/**
 * Creates an object manager.
 * @class
 * This class is used to high-light objects within a given view.
 * 
 * @author Kevin Henrikson
 *
 * @param {DwtComposite}	view			the view this manager is going to high-light for
 * @param {AjxCallback}	selectCallback  the callback triggered when user clicks on high-light object (provide if you want to do something before the clicked on object opens its corresponding view)
 * @param {Boolean}	skipHandlers 	<code>true</code> to avoid adding the standard handlers
 */
ZmObjectManager = function(view, selectCallback, skipHandlers) {

	this._selectCallback = selectCallback;
	this._uuid = Dwt.getNextId();
	this._objectIdPrefix = "OBJ_PREFIX_";
	this._objectHandlers = {};

	// don't include when looking for objects. only used to provide tool tips for images
	if (appCtxt.get(ZmSetting.MAIL_ENABLED) && window["ZmImageAttachmentObjectHandler"]) {
		this._imageAttachmentHandler = new ZmImageAttachmentObjectHandler();
	}

	// create handlers (see registerHandler below)
	if (!skipHandlers) {
        this.initialized = false;
        this._addAutoHandlers();
	} else {
        this.initialized = true;
    }

    this.sortHandlers();
	this.reset();
	this.setView(view);
};

ZmObjectManager._TOOLTIP_DELAY = 275;

// Define common types for quicker object matching.
ZmObjectManager.EMAIL = "email";
ZmObjectManager.URL = "url";
ZmObjectManager.PHONE = "phone";
ZmObjectManager.DATE = "date";
ZmObjectManager.ADDRESS = "address";
ZmObjectManager.TITLE = "title";

// Allows callers to pass in a current date
ZmObjectManager.ATTR_CURRENT_DATE = "currentDate";

ZmObjectManager._autohandlers = [];

/**
 * Registers the handler.
 * 
 * @param	{Object}	obj		the object
 * @param	{constant}	type	the type
 * @param	{constant}	priority	the priority
 */
ZmObjectManager.registerHandler =
function(obj, type, priority) {
	if (typeof obj == "string") {
		obj = eval(obj);
	}
	var c = ZmObjectManager._autohandlers;
	if (!obj.__registered) {
		var id = c.push(obj);
		var i = id - 1;
		if(type) {
			c[i].useType = type;
		}
		if(priority) {
			c[i].usePrio = priority;
		}
		obj.__registered = true;
	}
};

/**
 * @private
 */
ZmObjectManager.unregisterHandler =
function(obj) {
	if (typeof obj == "string") {
		obj = eval(obj);
	}
 	var c = ZmObjectManager._autohandlers, i;
	for (i = c.length; --i >= 0;) {
		if (c[i] === obj) {
			c.splice(i, 1);
			break;
		}
	}
};

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmObjectManager.prototype.toString =
function() {
	return "ZmObjectManager";
};

/**
 * Gets the handlers.
 * 
 * @return	{Array}	an array of {@link ZmObjectHandler} objects
 */
ZmObjectManager.prototype.getHandlers =
function() {
	if (!this.initialized) {
		var zimletMgr = appCtxt.getZimletMgr();
		if (zimletMgr.isLoaded()) {
			this.initialized = true;
			var zimlets = zimletMgr.getContentZimlets();
			for (var i = 0; i < zimlets.length; i++) {
				this.addHandler(zimlets[i], zimlets[i].type, zimlets[i].prio);
			}
		}
	}
	return this._objectHandlers;
};

/**
 * Adds the handler.
 * 
 * @param	{ZmObjectHandler}	h		the handler
 * @param	{constant}			type	the type
 * @param	{constant}		priority	the priority
 */
ZmObjectManager.prototype.addHandler =
function(h, type, priority) {
	type = type || (h.getTypeName() ? h.getTypeName() : "none");
	priority = priority ? priority : -1;
	h._prio = priority;
	//DBG.println(AjxDebug.DBG3, "addHandler " + h + " type: " + type + " prio: " + priority);
	var oh = this.getHandlers();
	if (!oh[type]) {oh[type] = [];}
	oh[type].push(h);
};

/**
 * Removes the handler.
 * 
 * @param	{ZmObjectHandler}	h		the handler
 * @param	{constant}			type	the type
 */
ZmObjectManager.prototype.removeHandler =
function(h, type) {
	type = type || (h.getTypeName() ? h.getTypeName() : "none");
	var oh = this.getHandlers();
	if (oh[type]) {
		for (var i = 0, count = oh[type].length; i < count; i++) {
			if (oh[type][i] == h) {
				oh[type].splice(i, 1);
				break;
			}
		}
	}
};

/**
 * Sorts the handlers.
 * 
 */
ZmObjectManager.prototype.sortHandlers =
function() {
	this._allObjectHandlers = [];
    var objectHandlers = this.getHandlers();
    for (var type in objectHandlers) {
		// Object handlers grouped by Type
		objectHandlers[type].sort(ZmObjectManager.__byPriority);

		// Copy each array to a single array of all Object Handlers
		for (var k = 0; k < objectHandlers[type].length; k++) {
			this._allObjectHandlers.push(objectHandlers[type][k]);
		}
	}
	this._allObjectHandlers.sort(ZmObjectManager.__byPriority);
};

/**
 * @private
 */
ZmObjectManager.prototype._addAutoHandlers =
function() {
	var c = ZmObjectManager._autohandlers, i, obj, prio;
	for (i = 0; i < c.length; ++i) {
		obj = c[i];
		var	handler = obj;
		var type = obj.TYPE;
		if (!(obj.isZmObjectHandler)) {
			handler = new obj();
		}
		if (obj.useType) {
			type = obj.useType;
		}
		if (obj.usePrio) {
			prio = obj.usePrio;
		}
		this.addHandler(handler, type, prio);
	}
};

/**
 * Resets the objects.
 * 
 */
ZmObjectManager.prototype.reset =
function() {
	this._objects = {};
};

/**
 * Sets the view.
 * 
 * @param	{DwtComposite}		view		the view
 */
ZmObjectManager.prototype.setView =
function(view) {
	if (view != null && appCtxt.getZimletMgr().isLoaded()) {
	    view.addListener(DwtEvent.ONMOUSEOVER, new AjxListener(this, this._mouseOverListener));
	    view.addListener(DwtEvent.ONMOUSEOUT, new AjxListener(this, this._mouseOutListener));
	    view.addListener(DwtEvent.ONMOUSEDOWN, new AjxListener(this, this._mouseDownListener));
	    view.addListener(DwtEvent.ONMOUSEUP, new AjxListener(this, this._mouseUpListener));
	    view.addListener(DwtEvent.ONMOUSEMOVE, new AjxListener(this, this._mouseMoveListener));
		view.addListener(DwtEvent.ONCONTEXTMENU, new AjxListener(this, this._rightClickListener));
		this._hoverOverListener = new AjxListener(this, this._handleHoverOver);
	    this._hoverOutListener = new AjxListener(this, this._handleHoverOut);
	}
	this._view = view;
};

ZmObjectManager.prototype.getView = function() {
	return this._view;
};

/**
 * Gets the count of objects.
 * 
 * @return	{int}	the count
 */
ZmObjectManager.prototype.objectsCount =
function() {
	return (appCtxt.zimletsPresent()) ? appCtxt.getZimletMgr().getContentZimlets().length : 0;
};

/**
 * Gets the image attachment handler.
 * 
 * @return	{ZmImageAttachmentObjectHandler}	the handler
 */
ZmObjectManager.prototype.getImageAttachmentHandler =
function() {
	return this._imageAttachmentHandler;
};

/**
 * @private
 */
ZmObjectManager.prototype._getAjxEmailAddress =
function(obj){
    if(appCtxt.isChildWindow && obj.isAjxEmailAddress){ //Making sure child window knows its type AjxEmailAddress
        obj = AjxEmailAddress.copy(obj);
    }
    return obj;
};

/**
 * Finds objects.
 * 
 * @param	{String}	content		the content
 * @param	{Boolean}	htmlEncode	<code>true</code> to HTML encode the content
 * @param	{constant}	type		the type
 * @param	{Boolean}	isTextMsg	<code>true</code> if is text msg
 * @param	{hash}		options		arbitrary options to pass to handler
 *
 * @return	{String}	the object
 */
ZmObjectManager.prototype.findObjects =
function(content, htmlEncode, type, isTextMsg, options) {
	if  (!content) {return "";}
	var html = [];
	var idx = 0;

	var maxIndex = content.length;
	var lastIndex = 0;

    var objectHandlers = this.getHandlers();
    while (true) {
		var lowestResult = null;
		var lowestIndex = maxIndex;
		var lowestHandler = null;

		// if given a type, just go thru the handler defined for that type.
		// otherwise, go thru every handler we have. Regardless, ask each handler
		// to find us a match >= to lastIndex. Handlers that didn't match last
		// time will simply return, handlers that matched last time that we didn't
		// use (because we found a closer match) will simply return that match again.
		//
		// when we are done, we take the handler with the lowest index.
		var i;
		var handlers;
		var chunk;
		var result = null;
		if (type) {
			//DBG.println(AjxDebug.DBG3, "findObjects type [" + type + "]");
			handlers = objectHandlers[type];
			if (handlers) {
				for (i = 0; i < handlers.length; i++) {
					//DBG.println(AjxDebug.DBG3, "findObjects by TYPE (" + handlers[i] + ")");
					result = handlers[i].findObject(content, lastIndex, this);
					// No match keep trying.
					if(!result) {continue;}
					// Got a match let's handle it.
					if (result.index >= lowestIndex) {break;}
					lowestResult = result;
					lowestIndex = result.index;
					lowestHandler = handlers[i];
				}
			}
			// If it's an email address just handle it and return the result.
			if (type == "email" || content instanceof AjxEmailAddress) {
				if (lowestHandler) {
                    content = this._getAjxEmailAddress(content);
					this.generateSpan(lowestHandler, html, idx, content, lowestResult, options);
				} else {
					html[idx++] = AjxStringUtil.htmlEncode(content.toString());
				}
				return html.join("");
			}
		} else {
			for (var j = 0; j < this._allObjectHandlers.length; j++) {
				var handler = this._allObjectHandlers[j];
				//DBG.println(AjxDebug.DBG3, "findObjects trying (" + handler + ")");
				result = handler.findObject(content, lastIndex, this);
				if (result && result.index < lowestIndex) {
					lowestResult = result;
					lowestIndex = result.index;
					lowestHandler = handler;
				}
			}
		}

		if (!lowestResult) {
			// all done
			// do last chunk
			chunk = content.substring(lastIndex, maxIndex);
			if (htmlEncode) {
				html[idx++] = AjxStringUtil.htmlEncode(chunk, !!isTextMsg);
			} else {
				html[idx++] = chunk;
			}
			break;
		}

		//  add anything before the match
		if (lowestIndex > lastIndex) {
			chunk = content.substring(lastIndex, lowestIndex);
			if (htmlEncode) {
				html[idx++] = AjxStringUtil.htmlEncode(chunk, !!isTextMsg);
			} else {
				html[idx++] = chunk;
			}
		}

		// add the match
		if(lowestHandler) {
			idx = this.generateSpan(lowestHandler, html, idx, lowestResult[0], lowestResult.context);
		} else {
			html[idx++] = lowestResult[0];
		}

		// update the index
		lastIndex = lowestResult.index + (lowestResult.matchLength || lowestResult[0].length);
	}

	return html.join("");
};


/**
 * Added this customized method for the sake of ZmMailMsgView performance.
 * 
 * TODO: Integrate this method to findObjectsInNode()
 * 
 * @private
 */
ZmObjectManager.prototype.processObjectsInNode = function(doc, node){

    var objectManager = this;
	doc = doc || node.ownerDocument;
    var tmpdiv = doc.createElement("div");

    var recurse = function(node, handlers) {
		var tmp, i, val, next;
		switch (node.nodeType) {
		    case 1:	// ELEMENT_NODE
			node.normalize();
			tmp = node.tagName.toLowerCase();

			if (next == null) {
				if (/^(img|a)$/.test(tmp)) {
                    var href;
                    try {
                        // IE can throw an "Invalid Argument" error depending on value of href
                        // e.g: http://0:0:0:0:0:0:0:1%0:7070/service/soap/ContactActionRequest:1331608015326:9c4f5868c5b0b4f2
                        href = node.href;
                    }
                    catch(e) {
                        //do nothing
                    }

                    var isMailToLink = tmp === "a" && ZmMailMsgView._MAILTO_RE.test(href),
                        isUrlLink = tmp === "a" && ZmMailMsgView._URL_RE.test(href);

                    if ((isMailToLink || isUrlLink) && node.target){
						// tricky.
						var txt = isMailToLink ? href :RegExp.$1 ;
						tmp = doc.createElement("div");
						tmp.innerHTML = objectManager.findObjects(AjxStringUtil.trim(txt));
						tmp = tmp.firstChild;
						if (tmp.nodeType == 3 /* Node.TEXT_NODE */) {
							// probably no objects were found.  A warning would be OK here
							// since the regexps guarantee that objects _should_ be found.
							return tmp.nextSibling;
						}
						// here, tmp is an object span, but it
						// contains the URL (href) instead of
						// the original link text.
						node.parentNode.insertBefore(tmp, node); // add it to DOM
						tmp.innerHTML = "";
						tmp.appendChild(node); // we have the original link now
						return tmp.nextSibling;	// move on
					}
					handlers = false;
				}
			} else {
				// consider processed
				node = next;
			}

			// bug 28264: the only workaround possible seems to be
			// to remove textIndent styles that have a negative value:
			if (parseFloat(node.style.textIndent) < 0) {
				node.style.textIndent = "";
			}
            for (i = node.firstChild; i; i = recurse(i, handlers)) {}
			return node.nextSibling;

		    case 3:	// TEXT_NODE
		    case 4:	// CDATA_SECTION_NODE (just in case)
			// generate ObjectHandler-s
			if (handlers && /[^\s\xA0]/.test(node.data)) try {
 				var a = null, b = null;

				if (!AjxEnv.isIE) {
					// this block of code is supposed to free the object handlers from
					// dealing with whitespace.  However, IE sometimes crashes here, for
					// reasons that weren't possible to determine--hence we avoid this
					// step for IE.  (bug #5345)
					var results = /^[\s\xA0]+/.exec(node.data);
					if (results) {
						a = node;
						node = node.splitText(results[0].length);
					}
					results = /[\s\xA0]+$/.exec(node.data);
					if (results)
						b = node.splitText(node.data.length - results[0].length);
				}

				tmp = tmpdiv;
				var code = objectManager.findObjects(node.data, true, null, false);
				var disembowel = false;
				if (AjxEnv.isIE) {
					// Bug #6481, #4498: innerHTML in IE massacrates whitespace
					//            unless it sees a <pre> in the code.
					tmp.innerHTML = [ "<pre>", code, "</pre>" ].join("");
					disembowel = true;
				} else {
					tmp.innerHTML = code;
				}

				if (a)
					tmp.insertBefore(a, tmp.firstChild);
				if (b)
					tmp.appendChild(b);

				a = node.parentNode;
				if (disembowel)
					tmp = tmp.firstChild;
				while (tmp.firstChild)
					a.insertBefore(tmp.firstChild, node);
				tmp = node.nextSibling;
				a.removeChild(node);
				return tmp;
			} catch(ex) {};
		}
		return node.nextSibling;
	};

    // Parse through the DOM directly and find objects.
	if (node && node.childNodes && node.childNodes.length) {
		for (var i = 0; i < node.childNodes.length; i++){
			recurse(node.childNodes[i], true);
		}
	}
};

/**
 * @private
 */
ZmObjectManager.prototype.findObjectsInNode =
function(node, re_discard, re_allow, callbacks) {
	var objectManager = this, doc = node.ownerDocument, tmpdiv = doc.createElement("div");

	if (!re_discard)
		re_discard = /^(script|link|object|iframe|applet)$/i;

	// This inner function does the actual work.  BEWARE that it return-s
	// in various places, not only at the end.
	var recurse = function(node, handlers) {
		var tmp, i, val, next;
		switch (node.nodeType) {
		    case 1:	// ELEMENT_NODE
			node.normalize();
			tmp = node.tagName.toLowerCase();
			if (callbacks && callbacks.foreachElement) {
				next = callbacks.foreachElement(node, tmp, re_discard, re_allow);
			}
			if (next == null) {
				if (/^(img|a)$/.test(tmp)) {
					if (tmp == "a" && node.target
					    && (ZmMailMsgView._URL_RE.test(node.href)
						|| ZmMailMsgView._MAILTO_RE.test(node.href)))
					{
						// tricky.
						var txt = RegExp.$1;
						tmp = doc.createElement("div");
						tmp.innerHTML = objectManager.findObjects(AjxStringUtil.trim(RegExp.$1));
						tmp = tmp.firstChild;
						if (tmp.nodeType == 3 /* Node.TEXT_NODE */) {
							// probably no objects were found.  A warning would be OK here
							// since the regexps guarantee that objects _should_ be found.
							return tmp.nextSibling;
						}
						// here, tmp is an object span, but it
						// contains the URL (href) instead of
						// the original link text.
						node.parentNode.insertBefore(tmp, node); // add it to DOM
						tmp.innerHTML = "";
						tmp.appendChild(node); // we have the original link now
						return tmp.nextSibling;	// move on
					}
					handlers = false;
				} else if (re_discard.test(tmp) || (re_allow && !re_allow.test(tmp))) {
					tmp = node.nextSibling;
					node.parentNode.removeChild(node);
					return tmp;
				}
			} else {
				// consider processed
				node = next;
			}

			if (AjxEnv.isIE) {
				// strips expression()-s, bwuahahaha!
				// granted, they get lost on the server-side anyway, but assuming some get through...
				// the line below exterminates them.
				node.style.cssText = node.style.cssText;
			}

			for (i = node.firstChild; i; i = recurse(i, handlers)) {}
			return node.nextSibling;

		    case 3:	// TEXT_NODE
		    case 4:	// CDATA_SECTION_NODE (just in case)
			// generate ObjectHandler-s
			if (handlers && /[^\s\xA0]/.test(node.data)) try {
 				var a = null, b = null;

				if (!AjxEnv.isIE) {
					// this block of code is supposed to free the object handlers from
					// dealing with whitespace.  However, IE sometimes crashes here, for
					// reasons that weren't possible to determine--hence we avoid this
					// step for IE.  (bug #5345)
					var results = /^[\s\xA0]+/.exec(node.data);
					if (results) {
						a = node;
						node = node.splitText(results[0].length);
					}
					results = /[\s\xA0]+$/.exec(node.data);
					if (results)
						b = node.splitText(node.data.length - results[0].length);
				}

				tmp = tmpdiv;
				var code = objectManager.findObjects(node.data, true, null, false);
				var disembowel = false;
				if (AjxEnv.isIE) {
					// Bug #6481, #4498: innerHTML in IE massacrates whitespace
					//            unless it sees a <pre> in the code.
					tmp.innerHTML = [ "<pre>", code, "</pre>" ].join("");
					disembowel = true;
				} else {
					tmp.innerHTML = code;
				}

				if (a)
					tmp.insertBefore(a, tmp.firstChild);
				if (b)
					tmp.appendChild(b);

				a = node.parentNode;
				if (disembowel)
					tmp = tmp.firstChild;
				while (tmp.firstChild)
					a.insertBefore(tmp.firstChild, node);
				tmp = node.nextSibling;
				a.removeChild(node);
				return tmp;
			} catch(ex) {};
		}
		return node.nextSibling;
	};
	var df = doc.createDocumentFragment();
	while (node.firstChild) {
		df.appendChild(node.firstChild); // NODE now out of the displayable DOM
		recurse(df.lastChild, true, this);	 // parse tree and findObjects()
	}
	node.appendChild(df);	// put nodes back in the document
};

/**
 * Sets handler attribute.
 * 
 * @param	{String}	type		the type
 * @param	{String}	name		the attribute name
 * @param	{Object}	value		the value
 */
ZmObjectManager.prototype.setHandlerAttr =
function(type, name, value) {
    var handlers = this.getHandlers()[type];
	if (handlers) {
		for (var i = 0; i < handlers.length; i++) {
			handlers[i][name] = value;
		}
	}
};

/**
 * Generates the span.
 * 
 * @private
 */
ZmObjectManager.prototype.generateSpan =
function(handler, html, idx, obj, context, options) {
	var id = this._objectIdPrefix + Dwt.getNextId();
    if (handler && handler.name) {
        id = id + "_" + handler.name;
    }
	this._objects[id] = {object: obj, handler: handler, id: id, context: context };
	return handler.generateSpan(html, idx, obj, id, context, options);
};

/**
 * @private
 */
ZmObjectManager.prototype._findObjectSpan =
function(e) {
	while (e && (!e.id || e.id.indexOf(this._objectIdPrefix) !== 0)) {
		e = e.parentNode;
	}
	return e;
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseOverListener =
function(ev) {
	var span = this._findObjectSpan(ev.target);
	if (!span) {return false;}
	var object = this._objects[span.id];
	if (!object) {return false;}

	span.className = object.handler.getHoveredClassName(object.object, object.context, span.id);
	if (object.handler.hasToolTipText()) {
		var shell = DwtShell.getShell(window);
		var manager = shell.getHoverMgr();
		if ((!manager.isHovering() || manager.getHoverObject() != object) && !DwtMenu.menuShowing()) {
			manager.reset();
			manager.setHoverOverDelay(ZmObjectManager._TOOLTIP_DELAY);
			manager.setHoverObject(object);
			manager.setHoverOverData(object);
			manager.setHoverOverListener(this._hoverOverListener);
			manager.hoverOver(ev.docX, ev.docY);
			ev.hoverStarted = true;
		}
	}

	ev._returnValue = true;
	ev._dontCallPreventDefault = true;
	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseOutListener =
function(ev) {
	var span = this._findObjectSpan(ev.target);
	var object = span ? this._objects[span.id] : null;

	if (object) {
		span.className = object.handler.getClassName(object.object, object.context, span.id);
		var shell = DwtShell.getShell(window);
		var manager = shell.getHoverMgr();
		manager.setHoverOutDelay(150);
		manager.setHoverOutData(object);
		manager.setHoverOutListener(this._hoverOutListener);
		manager.hoverOut();
	}

	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseMoveListener =
function(ev) {
	ev._returnValue = true;
	ev._dontCallPreventDefault = true;
	ev._stopPropagation = true;
	var span = this._findObjectSpan(ev.target);
	var object = span ? this._objects[span.id] : null;

	if (object) {
		var shell = DwtShell.getShell(window);
		var manager = shell.getHoverMgr();
		if (!manager.isHovering()) {
			// NOTE: mouseOver already init'd hover settings
			manager.hoverOver(ev.docX, ev.docY);
		}
	}

	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._rightClickListener =
function(ev) {
	ev.button = DwtMouseEvent.RIGHT;
	return this._mouseDownListener(ev);
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseDownListener =
function(ev) {

	// "authoritative" means a previous listener doesn't want propagation to get reset
	if (!ev._authoritative) {
		ev._dontCallPreventDefault = true;
		ev._returnValue = true;
		ev._stopPropagation = false;
	}

	var span = this._findObjectSpan(ev.target);
	if (!span) {
		return true;
	}
	var object = this._objects[span.id];
	if (!object) {
		return true;
	}

	ev._stopPropagation = true;

	var shell = DwtShell.getShell(window);
	var manager = shell.getHoverMgr();
	manager.setHoverOutDelay(0);
	manager.setHoverOutData(object);
	manager.setHoverOutListener(this._hoverOutListener);
	manager.hoverOut();

	span.className = object.handler.getActiveClassName(object.object, object.context, span.id);
	if (ev.button == DwtMouseEvent.RIGHT) {
		var menu = object.handler.getActionMenu(object.object, span, object.context, ev);
		if (menu) {
			menu.popup(0, ev.docX, ev.docY);
			// if we have an action menu, don't let the browser show its context menu too
			ev._dontCallPreventDefault = false;
			ev._returnValue = false;
			ev._stopPropagation = true;
			return true;
		}
	} else if (ev.button == DwtMouseEvent.LEFT) {
		if (this._selectCallback) {
			this._selectCallback.run();
		}
		object.handler.selected(object.object, span, ev, object.context);
		return true;
	}
	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._mouseUpListener =
function(ev) {
	ev._returnValue = true;
	ev._dontCallPreventDefault = true;
	ev._stopPropagation = true;
	var span = this._findObjectSpan(ev.target);
	if (!span) {return false;}
	var object = this._objects[span.id];
	if (!object) {return false;}

	span.className = object.handler.getHoveredClassName(object.object, object.context, span.id);
	return false;
};

/**
 * @private
 */
ZmObjectManager.prototype._handleHoverOver =
function(event) {
	if (!(event && event.object)) { return; }

	var span = this._findObjectSpan(event.target);
	var handler = event.object.handler;
	var object = event.object.object;
	var context = event.object.context;
	var id = event.object.id;
	var x = event.x;
	var y = event.y;

	handler.hoverOver(object, context, x, y, span, id);
};

/**
 * @private
 */
ZmObjectManager.prototype._handleHoverOut =
function(event) {
	if (!(event && event.object)) { return; }

	var span = this._findObjectSpan(event.target);
	var handler = event.object.handler;
	var object = event.object.object;
	var context = event.object.context;
	var id = event.object.id;

	handler.hoverOut(object, context, span, id);
};

// Private static functions

/**
 * @private
 */
ZmObjectManager.__byPriority =
function(a, b) {
	return (b._prio < a._prio) - (a._prio < b._prio);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmSettings")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines the settings class.
 */

/**
 * Creates a collection of settings with default values. If no app context is given,
 * then this is a skeletal, non-live version of settings which can provide default
 * settings and parse SOAP settings.
 * @class
 * This class is a collection of various sorts of settings: config values, preferences,
 * and COS features. Each setting has an ID which can be used to retrieve it.
 *
 * @author Conrad Damon
 *
 * @param {Boolean}	noInit	if <code>true</code>, skip initialization
 *
 * @extends		ZmModel
 */
ZmSettings = function(noInit) {

	ZmModel.call(this, ZmEvent.S_SETTINGS);

	this._settings = {};	// settings by ID
	this._nameToId = {};	// map to get from server setting name to setting ID

	this.getInfoResponse = null; // Cached GetInfoResponse for lazy creation of identities, etc.
	this._handleImplicitChange = new AjxListener(this, this._implicitChangeListener);

	if (!noInit) {
        this.initialize();
	}
};

ZmSettings.prototype = new ZmModel;
ZmSettings.prototype.constructor = ZmSettings;

ZmSettings.BASE64_TO_NORMAL_RATIO = 1.34;

/**
 * Creates a new setting and adds it to the settings.
 *
 * @param {String}		id			the unique ID of the setting
 * @param {Hash}	params		a hash of parameters
 * @param {String}	params.name			the name of the pref or attr on the server
 * @param {constant}	params.type		config, pref, or COS
 * @param {constant}	params.dataType	string, int, or boolean (defaults to string)
 * @param {Object}	params.defaultValue	the default value
 */
ZmSettings.prototype.registerSetting =
function(id, params) {
	ZmSetting[id] = id;
	var setting = this._settings[id] = new ZmSetting(id, params);
	if (params.name) {
		this._nameToId[params.name] = id;
	}
	if (params.isImplicit) {
		setting.addChangeListener(this._handleImplicitChange);
	}
	return setting;
};

/**
 * Returns a string representation of the object.
 *
 * @return		{String}		a string representation of the object
 */
ZmSettings.prototype.toString =
function() {
	return "ZmSettings";
};

/**
 * Initializes the settings.
 *
 */
ZmSettings.prototype.initialize =
function() {
	this._initialize();
	this._setDefaults();
	this.userSettingsLoaded = false;

	// set listeners for settings
	var listener = new AjxListener(this, this._changeListener);
	if (!appCtxt.multiAccounts) {
		this.getSetting(ZmSetting.QUOTA_USED).addChangeListener(listener);
	}
	this.getSetting(ZmSetting.POLLING_INTERVAL).addChangeListener(listener);
	this.getSetting(ZmSetting.SKIN_NAME).addChangeListener(listener);
	this.getSetting(ZmSetting.SHOW_SELECTION_CHECKBOX).addChangeListener(listener);
	this.getSetting(ZmSetting.LOCALE_NAME).addChangeListener(listener);
	this.getSetting(ZmSetting.FONT_NAME).addChangeListener(listener);
	this.getSetting(ZmSetting.FONT_SIZE).addChangeListener(listener);
	this.getSetting(ZmSetting.SHORTCUTS).addChangeListener(listener);
	this.getSetting(ZmSetting.CHILD_ACCTS_VISIBLE).addChangeListener(listener);
	this.getSetting(ZmSetting.ATTACHMENTS_BLOCKED).addChangeListener(listener);
	this.getSetting(ZmSetting.DISPLAY_TIME_IN_MAIL_LIST).addChangeListener(listener);


	if (appCtxt.isOffline) {
		this.getSetting(ZmSetting.OFFLINE_IS_MAILTO_HANDLER).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_BACKUP_ACCOUNT_ID).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_BACKUP_INTERVAL).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_BACKUP_PATH).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_BACKUP_KEEP).addChangeListener(listener);
        this.getSetting(ZmSetting.OFFLINE_UPDATE_NOTIFY).addChangeListener(listener);
	}
};

/**
 * Gets the value of the given setting.
 *
 * @param {String}	id		the ID of the setting
 * @param	{String}	key		the key
 * @return	{Object}	the value or <code>null</code> for none
 */
ZmSettings.prototype.get =
function(id, key) {
	return (id && this._settings[id]) ? this._settings[id].getValue(key) : null;
};

/**
 * Sets the value of the given setting.
 *
 * @param {String}	id		the ID of the setting
 * @param {Object}	value			the new value for the setting
 * @param {String}	key 			optional key for use by hash table data type
 * @param {Boolean}	setDefault		if <code>true</code>, also set the default value
 * @param {Boolean}	skipNotify		if <code>true</code>, do not notify listeners
 * @param {Boolean}	skipImplicit		if <code>true</code>, do not check for change to implicit pref
 */
ZmSettings.prototype.set = function(id, value, key, setDefault, skipNotify, skipImplicit) {
	if (id && this._settings[id]) {
		this._settings[id].setValue(value, key, setDefault, skipNotify, skipImplicit);
	}
	else {
		DBG.println(AjxDebug.DBG1, "ZmSettings.set: ID missing (value = " + value);
	}
};

/**
 * Gets the setting.
 *
 * @param {String}	id		the ID of the setting
 * @return	{ZmSetting}	the setting
 */
ZmSettings.prototype.getSetting =
function(id) {
	return this._settings[id];
};

/**
 * Populates settings values.
 *
 * @param {Hash}	list		a hash of preference or attribute values
 */
ZmSettings.prototype.createFromJs =
function(list, setDefault, skipNotify, skipImplicit) {
    // default skipImplicit value is true
    skipImplicit = skipImplicit == null || skipImplicit; 

	for (var i in list) {
		var val = list[i];
		var setting = this._settings[this._nameToId[i]];
		if (setting) {
			if (setting.dataType == ZmSetting.D_HASH) {
				var pairs = val.split(",");
				var value = {};
				for (var j = 0; j < pairs.length; j++) {
					var fields = pairs[j].split(":");
					
					// In case of shared folder there can be more than two element in
					// fields array as shared folder id can be "859095ad-90d5-4db3-919d-630ac30de5a9:465"
					// so consider all the elements as key except last item of the fields array.
					if(setting.name === "zimbraPrefFoldersExpanded" && fields.length > 2) {
						var fieldValue = fields.pop();
						var fieldKey = fields.join(':');
						value[fieldKey] = fieldValue;
					} else {
						value[fields[0]] = fields[1];
					}
				}
				val = value;
			}
			setting.setValue(val, null, setDefault, skipNotify, skipImplicit);
			if (ZmSetting.IS_IMPLICIT[setting.id]) {
				setting.origValue = setting.copyValue();
			}
		} else {
			DBG.println(AjxDebug.DBG3, "*** Unrecognized setting: " + i);
		}
	}
};

/**
 * Gets the setting that is associated with the given server-side setting, if any.
 *
 * @param {String}	name	the server-side setting name (for example, "zimbraFeatureContactsEnabled")
 * @return	{String}	the setting id
 */
ZmSettings.prototype.getSettingByName =
function(name) {
	return this._nameToId[name];
};

/**
 * Checks if the given ID was received from the server. Use this method
 * to determine whether this ID is supported by a ZCS server. Currently used by
 * ZDesktop since it can "talk" to both v5 and v6 ZCS.
 *
 * @param {String}	id	the setting ID
 * @return	{Boolean}	<code>true</code> if the attribute is supported
 */
ZmSettings.prototype.attrExists =
function(id) {
	var name = this.getSetting(id).name;
	return (this.getInfoResponse.prefs._attrs[name] ||
			this.getInfoResponse.attrs._attrs[name]);
};

/**
 * Retrieves the preferences, COS settings, and metadata for the current user.
 * All the data gets stored into the settings collection.
 *
 * @param {AjxCallback}	callback 			the callback to run after response is received
 * @param {AjxCallback}	errorCallback 	the callback to run error is received
 * @param {String}	accountName		the name of account to load settings for
 * @param {Object}	response			the pre-determined JSON response object
 * @param {ZmBatchCommand}	batchCommand		set if part of a batch request
 */
ZmSettings.prototype.loadUserSettings =
function(callback, errorCallback, accountName, response, batchCommand) {
	var args = [callback, accountName];

	var soapDoc = AjxSoapDoc.create("GetInfoRequest", "urn:zimbraAccount");
	soapDoc.setMethodAttribute("rights", "createDistList"); //not sure when this is called, but let's add it anyway. (on login it's called from within launchZCS.JSP calling GetInfoJSONTag.java
	var respCallback = this._handleResponseLoadUserSettings.bind(this, args);
	if (batchCommand) {
		batchCommand.addNewRequestParams(soapDoc, respCallback);
	}
	else {
		var params = {
			soapDoc: (response ? null : soapDoc),
			accountName: accountName,
			asyncMode: true,
			callback: respCallback,
			errorCallback: errorCallback,
			response: response
		};
		appCtxt.getAppController().sendRequest(params);
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseLoadUserSettings =
function(callback, accountName, result) {
    this.setUserSettings(result.getResponse().GetInfoResponse, accountName);
    this.userSettingsLoaded = true;
    if (callback) {
        callback.run(result);
    }
};

/**
 * Sets the user settings.
 *
 * @param {hash}    params
 * @param {object}  params.info             The GetInfoResponse object.
 * @param {string}  [params.accountName]    The name of the account.
 * @param {boolean} [params.setDefault]     Set default value
 * @param {boolean} [params.skipNotify]     Skip change notification
 * @param {boolean} [params.skipImplicit]   Skip implicit changes
 * @param {boolean} [params.preInit]        Only init base settings for startup
 */
ZmSettings.prototype.setUserSettings = function(params) {

    params = Dwt.getParams(arguments, ["info", "accountName", "setDefault", "skipNotify", "skipImplicit", "preInit"]);

    var info = this.getInfoResponse = params.info;

	appCtxt.createDistListAllowed = false;
	appCtxt.createDistListAllowedDomains = [];
	appCtxt.createDistListAllowedDomainsMap = {};
	var rightTargets = info.rights && info.rights.targets;
	if (rightTargets) {
		for (var i = 0; i < rightTargets.length; i++) {
			var target = rightTargets[i];
			if (target.right == "createDistList") {
				if (target.target[0].type == "domain") {
					appCtxt.createDistListAllowed = true;
					appCtxt.createDistListAllowedDomains.push(target.target[0].name);
					appCtxt.createDistListAllowedDomainsMap[target.target[0].name] = true;
					break;
				}
			}

		}
	}

	// For delegated admin who can see prefs but not mail, we still need to register the mail settings so
	// they can be created and set below in createFromJs().
	if (this.get(ZmSetting.ADMIN_DELEGATED)) {
		if (!this.get(ZmSetting.ADMIN_MAIL_ENABLED) && this.get(ZmSetting.ADMIN_PREFERENCES_ENABLED)) {
			this.getMailAppForDelegatedAdmin().enableMailPrefs();
		}
	}

	// Voice feature
    this.set(ZmSetting.VOICE_ENABLED, this._hasVoiceFeature(), null, false, true);

    var accountName = params.accountName;
    var setDefault = params.preInit ? false : params.setDefault;
    var skipNotify = params.preInit ? true : params.skipNotify;
    var skipImplicit = params.preInit ? true : params.skipImplicit;

    var settings = [
        ZmSetting.ADMIN_DELEGATED,          info.adminDelegated,
        ZmSetting.MESSAGE_SIZE_LIMIT,    this._base64toNormalSize(info.attSizeLimit),
        ZmSetting.CHANGE_PASSWORD_URL,      info.changePasswordURL,
        ZmSetting.DOCUMENT_SIZE_LIMIT,      this._base64toNormalSize(info.docSizeLimit),
        ZmSetting.LAST_ACCESS,              info.accessed,
        ZmSetting.LICENSE_STATUS,           info.license && info.license.status,
        ZmSetting.PREVIOUS_SESSION,         info.prevSession,
        ZmSetting.PUBLIC_URL,               info.publicURL,
		ZmSetting.ADMIN_URL,                info.adminURL,
        ZmSetting.QUOTA_USED,               info.used,
        ZmSetting.RECENT_MESSAGES,          info.recent,
        ZmSetting.REST_URL,                 info.rest,
        ZmSetting.USERNAME,                 info.name,
		ZmSetting.EMAIL_VALIDATION_REGEX, 	info.zimbraMailAddressValidationRegex,
		ZmSetting.HAB_ROOT,                 (info.habRoots && info.habRoots.hab ? info.habRoots.hab : false),
		ZmSetting.SPELL_CHECK_ENABLED, 		info.isSpellCheckAvailable,
		ZmSetting.DISABLE_SENSITIVE_ZIMLETS_IN_MIXED_MODE, 	(info.domainSettings && info.domainSettings.zimbraZimletDataSensitiveInMixedModeDisabled ? info.domainSettings.zimbraZimletDataSensitiveInMixedModeDisabled : "FALSE")
    ];
    for (var i = 0; i < settings.length; i += 2) {
        var value = settings[i+1];
        if (value != null) {
            this.set(settings[i], value, null, setDefault, skipNotify, skipImplicit);
        }
    }

    // features and other settings
    if (info.attrs && info.attrs._attrs) {
        this.createFromJs(info.attrs._attrs, setDefault, skipNotify, skipImplicit);
    }

	// By default, everything but mail is enabled for delegated admin. Require an additional setting to allow admin to view mail.
	if (this.get(ZmSetting.ADMIN_DELEGATED)) {
		this.set(ZmSetting.MAIL_ENABLED, this.get(ZmSetting.ADMIN_MAIL_ENABLED), setDefault, skipNotify, skipImplicit);
		var enableMailPrefs = this.get(ZmSetting.MAIL_ENABLED) || (this.get(ZmSetting.ADMIN_DELEGATED) && this.get(ZmSetting.ADMIN_PREFERENCES_ENABLED));
		this.set(ZmSetting.MAIL_PREFERENCES_ENABLED, enableMailPrefs, setDefault, skipNotify, skipImplicit);
		// Disable other areas where mail could be exposed to a prefs-only admin
		if (this.get(ZmSetting.MAIL_PREFERENCES_ENABLED) && !this.get(ZmSetting.MAIL_ENABLED)) {
			var result = { handled: false };
			appCtxt.notifyZimlets("onZmSettings_setUserSettings", [this, setDefault, skipNotify, skipImplicit, result]);
			if (!result.handled) {
				this.set(ZmSetting.MAIL_FORWARDING_ENABLED, false, setDefault, skipNotify, skipImplicit);
				this.set(ZmSetting.FILTERS_MAIL_FORWARDING_ENABLED, false, setDefault, skipNotify, skipImplicit);
				this.set(ZmSetting.NOTIF_FEATURE_ENABLED, false, setDefault, skipNotify, skipImplicit);
			}
		}
	}

	// Server may provide us with SSO-enabled URL for Community integration (integration URL with OAuth signature)
	if (info.communityURL) {
		this.set(ZmSetting.SOCIAL_EXTERNAL_URL, info.communityURL, null, setDefault, skipNotify);
	}

	if (params.preInit) {
	    return;
    }

    // preferences
    if (info.prefs && info.prefs._attrs) {
        this.createFromJs(info.prefs._attrs, setDefault, skipNotify, skipImplicit);
    }

    // accounts
	var setting;
	if (!accountName) {
		// NOTE: only the main account can have children
		appCtxt.accountList.createAccounts(this, info);

		// for offline, find out whether this client supports prism-specific features
		if (appCtxt.isOffline) {
			if (AjxEnv.isPrism && window.platform) {
				this.set(ZmSetting.OFFLINE_SUPPORTS_MAILTO, true, null, setDefault, skipNotify, skipImplicit);
				this.set(ZmSetting.OFFLINE_SUPPORTS_DOCK_UPDATE, true, null, setDefault, skipNotify, skipImplicit);
			}

			// bug #45804 - sharing always enabled for offline
			appCtxt.set(ZmSetting.SHARING_ENABLED, true, null, setDefault, skipNotify);
		}
	}

	// handle settings whose values may depend on other settings
	setting = this._settings[ZmSetting.REPLY_TO_ADDRESS];
	if (setting) {
		setting.defaultValue = this.get(ZmSetting.USERNAME);
	}
	if (this.get(ZmSetting.FORCE_CAL_OFF)) {
		this.set(ZmSetting.CALENDAR_ENABLED, false, null, setDefault, skipNotify, skipImplicit);
	}

	if (!this.get(ZmSetting.OPTIONS_ENABLED)) {
		this.set(ZmSetting.FILTERS_ENABLED, false, null, setDefault, skipNotify, skipImplicit);
	}

	// load zimlets *only* for the main account
	if (!accountName) {
		if (info.zimlets && info.zimlets.zimlet) {
            if (this.get(ZmSetting.ZIMLETS_SYNCHRONOUS)) {
                var action = new AjxTimedAction(this, this._beginLoadZimlets, [info.zimlets.zimlet, info.props.prop, true]);
                AjxTimedAction.scheduleAction(action, 0);
            } else {
                var listener = new AjxListener(this, this._beginLoadZimlets, [info.zimlets.zimlet, info.props.prop, false]);
                appCtxt.getAppController().addListener(ZmAppEvent.POST_STARTUP, listener);
            }
		} else {
			appCtxt.allZimletsLoaded();
		}
	}

	var value = appCtxt.get(ZmSetting.REPLY_INCLUDE_ORIG);
	if (value) {
		var list = ZmMailApp.INC_MAP[value];
		appCtxt.set(ZmSetting.REPLY_INCLUDE_WHAT, list[0], null, setDefault, skipNotify);
		appCtxt.set(ZmSetting.REPLY_USE_PREFIX, list[1], null, setDefault, skipNotify);
		appCtxt.set(ZmSetting.REPLY_INCLUDE_HEADERS, list[2], null, setDefault, skipNotify);
	}

	var value = appCtxt.get(ZmSetting.FORWARD_INCLUDE_ORIG);
	if (value) {
		var list = ZmMailApp.INC_MAP[value];
		appCtxt.set(ZmSetting.FORWARD_INCLUDE_WHAT, list[0], null, setDefault, skipNotify);
		appCtxt.set(ZmSetting.FORWARD_USE_PREFIX, list[1], null, setDefault, skipNotify);
		appCtxt.set(ZmSetting.FORWARD_INCLUDE_HEADERS, list[2], null, setDefault, skipNotify);
	}

    // Populate Sort Order Defaults
    var sortPref =  ZmSettings.DEFAULT_SORT_PREF;
    sortPref[ZmId.VIEW_CONVLIST]			= ZmSearch.DATE_DESC;
    sortPref[ZmId.VIEW_CONV]				= ZmSearch.DATE_DESC;
    sortPref[ZmId.VIEW_TRAD]				= ZmSearch.DATE_DESC;
    sortPref[ZmId.VIEW_CONTACT_SRC]			= ZmSearch.NAME_ASC;
    sortPref[ZmId.VIEW_CONTACT_TGT]			= ZmSearch.NAME_ASC;
    sortPref[ZmId.VIEW_CONTACT_SIMPLE]		= ZmSearch.NAME_ASC;
    sortPref[ZmId.VIEW_CAL]					= ZmSearch.DATE_ASC;
    sortPref[ZmId.VIEW_TASKLIST]			= ZmSearch.DUE_DATE_ASC;
    sortPref[ZmId.VIEW_BRIEFCASE_DETAIL]	= ZmSearch.SUBJ_ASC;

    var sortOrderSetting = this._settings[ZmSetting.SORTING_PREF];
    if (sortOrderSetting) {
        // Populate empty sort pref's with defaultValues
        for (var pref in sortPref){
            if (!sortOrderSetting.getValue(pref)){
                sortOrderSetting.setValue(sortPref[pref], pref, false, true);
            }
        }

        // Explicitly Set defaultValue
        sortOrderSetting.defaultValue = AjxUtil.hashCopy(sortPref);
    }
	
	DwtControl.useBrowserTooltips = this.get(ZmSetting.BROWSER_TOOLTIPS_ENABLED);

	this._updateUserFontPrefsRule();
};


ZmSettings.prototype._base64toNormalSize =
function(base64) {
	if (!base64 || base64 === -1) { //-1 is unlimited
		return base64;
	}
	return base64 / ZmSettings.BASE64_TO_NORMAL_RATIO;
};


/**
 * @private
 */
ZmSettings.prototype._beginLoadZimlets =
function(zimlet, prop, sync) {
	var zimletsCallback = new AjxCallback(this, this._loadZimletPackage, [zimlet, prop, sync]);
	AjxDispatcher.require(["Startup2"], false, zimletsCallback);
};

ZmSettings.prototype._loadZimletPackage =
function(zimlet, prop, sync) {
	var zimletsCallback = new AjxCallback(this, this._loadZimlets, [zimlet, prop, sync]);
	AjxDispatcher.require("Zimlet", false, zimletsCallback);
};

/**
 * @private
 */
ZmSettings.prototype._loadZimlets =
function(allZimlets, props, sync) {

	allZimlets = allZimlets || [];
	this.registerSetting("ZIMLETS",		{type:ZmSetting.T_CONFIG, defaultValue:allZimlets, isGlobal:true});
	this.registerSetting("USER_PROPS",	{type:ZmSetting.T_CONFIG, defaultValue:props});

	var zimlets = this._getCheckedZimlets(allZimlets);

	DBG.println(AjxDebug.DBG1, "Zimlets - Loading " + zimlets.length + " Zimlets");
	var zimletMgr = appCtxt.getZimletMgr();
	zimletMgr.loadZimlets(zimlets, props, null, null, sync);

	if (zimlets && zimlets.length) {
		var activeApp = appCtxt.getCurrentApp();
		if (activeApp) {
			var overview;
			if (appCtxt.multiAccounts) {
				var containerId = activeApp.getOverviewContainer().containerId;
				var zimletLabel = ZmOrganizer.LABEL[ZmOrganizer.ZIMLET];
				var overviewId = [containerId, zimletLabel].join("_");
				overview = appCtxt.getOverviewController().getOverview(overviewId);
			} else {
				overview = activeApp.getOverview();
			}
		}

		// update overview tree
		if (overview) {
			if (overview.appName != ZmApp.PREFERENCES) {
				overview.setTreeView(ZmOrganizer.ZIMLET);
			}

			// HACK: for multi-account, hide the zimlet section if no panel zimlets
			if (appCtxt.multiAccounts && zimletMgr.getPanelZimlets().length == 0) {
				activeApp.getOverviewContainer().removeZimletSection();
			}
		}

		// create global portlets
		if (appCtxt.get(ZmSetting.PORTAL_ENABLED)) {
			var portletMgr = appCtxt.getApp(ZmApp.PORTAL).getPortletMgr();
			var portletIds = portletMgr.createPortlets(true);
		}
	}
};

/**
 * Filters a list of zimlets, returned ones that are checked.
 *
 * @param zimlets			[array]		list of zimlet objects
 *
 * @private
 */
ZmSettings.prototype._getCheckedZimlets =
function(allZimlets) {

	var zimlets = [];
	for (var i = 0; i < allZimlets.length; i++) {
		var zimletObj = allZimlets[i];
		if (zimletObj.zimletContext[0].presence != "disabled") {
			zimlets.push(zimletObj);
		}
	}

	return zimlets;
};

/**
 * Loads the preference data.
 *
 * @param	{AjxCallback}	callback		the callback
 */
ZmSettings.prototype.loadPreferenceData =
function(callback) {
	// force main account (in case multi-account) since locale/skins are global
	var command = new ZmBatchCommand(null, appCtxt.accountList.mainAccount.name);

	var skinDoc = AjxSoapDoc.create("GetAvailableSkinsRequest", "urn:zimbraAccount");
	var skinCallback = new AjxCallback(this, this._handleResponseLoadAvailableSkins);
	command.addNewRequestParams(skinDoc, skinCallback);

	var localeDoc = AjxSoapDoc.create("GetAvailableLocalesRequest", "urn:zimbraAccount");
	var localeCallback = new AjxCallback(this, this._handleResponseGetAllLocales);
	command.addNewRequestParams(localeDoc, localeCallback);

	var csvFormatsDoc = AjxSoapDoc.create("GetAvailableCsvFormatsRequest", "urn:zimbraAccount");
	var csvFormatsCallback = new AjxCallback(this, this._handleResponseGetAvailableCsvFormats);
	command.addNewRequestParams(csvFormatsDoc, csvFormatsCallback);

	command.run(callback);
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseLoadAvailableSkins =
function(result) {
	var resp = result.getResponse().GetAvailableSkinsResponse;
	var skins = resp.skin;
	if (skins && skins.length) {
		var setting = appCtxt.accountList.mainAccount.settings.getSetting(ZmSetting.AVAILABLE_SKINS);
		for (var i = 0; i < skins.length; i++) {
			// always save available skins on the main account (in case multi-account)
			setting.setValue(skins[i].name);
		}
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseGetAllLocales =
function(response) {
	var locales = response._data.GetAvailableLocalesResponse.locale;
	if (locales && locales.length) {
		for (var i = 0, count = locales.length; i < count; i++) {
			var locale = locales[i];
			// bug: 38038
			locale.id = locale.id.replace(/^in/,"id");
			ZmLocale.create(locale.id, locale.name, ZmMsg["localeName_" + locale.id] || locale.localName);
		}
        if (locales.length === 1) {
            //Fix for bug# 80762 - Set the value to always true in case of only one language/locale present
            this.set(ZmSetting.LOCALE_CHANGE_ENABLED, true);
        }
        else {
            this.set(ZmSetting.LOCALE_CHANGE_ENABLED, ZmLocale.hasChoices());
        }
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseGetAvailableCsvFormats =
function(result){
	var formats = result.getResponse().GetAvailableCsvFormatsResponse.csv;
	if (formats && formats.length) {
		var setting = appCtxt.accountList.mainAccount.settings.getSetting(ZmSetting.AVAILABLE_CSVFORMATS);
		for (var i = 0; i < formats.length; i++) {
			setting.setValue(formats[i].name);
		}
	}
};

/**
 * Saves one or more settings.
 *
 * @param {Array}		list			a list of {ZmSetting} objects
 * @param {AjxCallback}	callback		the callback to run after response is received
 * @param {ZmBatchCommand}	batchCommand	the batch command
 * @param {ZmZimbraAccount}	account		the account to save under
 * @param {boolean}			isImplicit	if true, we are saving implicit settings
 */
ZmSettings.prototype.save =
function(list, callback, batchCommand, account, isImplicit) {
	if (!(list && list.length)) { return; }

	var acct = account || appCtxt.getActiveAccount();
	var soapDoc = AjxSoapDoc.create("ModifyPrefsRequest", "urn:zimbraAccount");
	var gotOne = false;
	var metaData = [], done = {}, setting;
	for (var i = 0; i < list.length; i++) {
		setting = list[i];
        if (done[setting.id]) { continue; }
		if (setting.type == ZmSetting.T_METADATA) {
			metaData.push(setting);
			// update the local meta data
			acct.metaData.update(setting.section, setting.name, setting.getValue());
			continue;
		} else if (setting.type != ZmSetting.T_PREF) {
			DBG.println(AjxDebug.DBG1, "*** Attempt to modify non-pref: " + setting.id + " / " + setting.name);
			continue;
		}
		if (!setting.name) {
			DBG.println(AjxDebug.DBG2, "Modify internal pref: " + setting.id);
			continue;
		}
		if (setting.dataType == ZmSetting.D_LIST) {
			// LDAP supports multi-valued attrs, so don't serialize list
			var value = setting.getValue();
			if (value && value.length) {
				for (var j = 0; j < value.length; j++) {
					var node = soapDoc.set("pref", value[j]);
					node.setAttribute("name", setting.name);
				}
			} else {
				var node = soapDoc.set("pref", "");
				node.setAttribute("name", setting.name);
			}
		} else {
			var value = setting.getValue(null, true);
			var node = soapDoc.set("pref", value);
			node.setAttribute("name", setting.name);
		}

        done[setting.id] = true;
		gotOne = true;
	}

    // bug: 50668 if the setting is implicit and global, use main Account
    if(appCtxt.isOffline && ZmSetting.IS_IMPLICIT[setting.id] && ZmSetting.IS_GLOBAL[setting.id]) {
        acct = appCtxt.accountList.mainAccount;
    }

	if (metaData.length > 0) {
		var metaDataCallback = new AjxCallback(this, this._handleResponseSaveMetaData, [metaData]);
		var sections = [ZmSetting.M_IMPLICIT, ZmSetting.M_OFFLINE];
		acct.metaData.save(sections, metaDataCallback);
	}

	if (gotOne) {
		var respCallback;
		if (callback || batchCommand) {
			respCallback = new AjxCallback(this, this._handleResponseSave, [list, callback]);
		}
		if (batchCommand) {
			batchCommand.addNewRequestParams(soapDoc, respCallback);
		} else {
			appCtxt.getAppController().sendRequest({soapDoc:soapDoc, asyncMode:true, callback:respCallback,
			 										accountName:acct.name, noBusyOverlay:isImplicit});
		}
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseSaveMetaData =
function(list, result) {
	for (var i = 0; i < list.length; i++) {
		var setting = list[i];
		if (!ZmSetting.IS_IMPLICIT[setting.id]) {
			setting.origValue = setting.copyValue();
			setting._notify(ZmEvent.E_MODIFY);
		}
	}
};

/**
 * @private
 */
ZmSettings.prototype._handleResponseSave =
function(list, callback, result) {
	var resp = result.getResponse();
	if (resp.ModifyPrefsResponse != null) {
		// notify each changed setting's listeners
		for (var i = 0; i < list.length; i++) {
			var setting = list[i];
			setting.origValue = setting.copyValue();
			if (!ZmSetting.IS_IMPLICIT[setting.id]) {
				setting._notify(ZmEvent.E_MODIFY);
			}
		}
		// notify any listeners on the settings as a whole
		this._notify(ZmEvent.E_MODIFY, {settings:list});
	}

	if (callback) {
		callback.run(result);
	}
};

ZmSettings.DEFAULT_SORT_PREF = {};

/**
 * Set defaults which are determined dynamically (which can't be set in static code).
 *
 * @private
 */
ZmSettings.prototype._setDefaults =
function() {

	var value = AjxUtil.formatUrl({host:location.hostname, path:"/service/soap/", qsReset:true});
	this.set(ZmSetting.CSFE_SERVER_URI, value, null, false, true);

	// CSFE_MSG_FETCHER_URI
	value = AjxUtil.formatUrl({host:location.hostname, path:"/service/home/~/", qsReset:true, qsArgs:{auth:"co"}});
	this.set(ZmSetting.CSFE_MSG_FETCHER_URI, value, null, false, true);

	// CSFE_UPLOAD_URI
	value = AjxUtil.formatUrl({host:location.hostname, path:"/service/upload", qsReset:true, qsArgs:{lbfums:""}});
	this.set(ZmSetting.CSFE_UPLOAD_URI, value, null, false, true);

	// CSFE_ATTACHMENT_UPLOAD_URI
	value = AjxUtil.formatUrl({host:location.hostname, path:"/service/upload", qsReset:true});
	this.set(ZmSetting.CSFE_ATTACHMENT_UPLOAD_URI, value, null, false, true);

	// CSFE EXPORT URI
	value = AjxUtil.formatUrl({host:location.hostname, path:"/service/home/~/", qsReset:true, qsArgs:{auth:"co", id:"{0}", fmt:"csv"}});
	this.set(ZmSetting.CSFE_EXPORT_URI, value, null, false, true);

	var h = location.hostname;
	var isDev = ((h.indexOf(".zimbra.com") != -1) || (window.appDevMode && (/\.local$/.test(h) || (!appCtxt.isOffline && h == "localhost"))));
	this.set(ZmSetting.IS_DEV_SERVER, isDev);
	if (isDev || window.isScriptErrorOn) {
		this.set(ZmSetting.SHOW_SCRIPT_ERRORS, true, null, false, true);
	}

	this.setReportScriptErrorsSettings(AjxException, ZmController.handleScriptError);
};

ZmSettings.prototype.persistImplicitSortPrefs =
function(id){
    return ZmSettings.DEFAULT_SORT_PREF[id];
};

/**
 * sets AjxException static attributes. This is extracted so it can be called from ZmNewwindow as well.
 * this is since the child window gets its own AjxException variable.
 *
 * @param AjxExceptionClassVar
 * @param handler
 */
ZmSettings.prototype.setReportScriptErrorsSettings =
function(AjxExceptionClassVar, handler) {
	// script error reporting
	var rse = AjxExceptionClassVar.reportScriptErrors = this._settings[ZmSetting.SHOW_SCRIPT_ERRORS].getValue();
	if (rse) {
		AjxExceptionClassVar.setScriptErrorHandler(handler);
	}

};

/**
 * Loads the standard settings and their default values.
 *
 * @private
 */
ZmSettings.prototype._initialize =
function() {
	// CONFIG SETTINGS
    this.registerSetting("ADMIN_DELEGATED",                 {type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("AC_TIMER_INTERVAL",				{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_INT, defaultValue:300});
	this.registerSetting("ASYNC_MODE",						{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("BRANCH",							{type:ZmSetting.T_CONFIG, defaultValue:"JUDASPRIEST"});

	// next 3 are replaced during deployment
	this.registerSetting("CLIENT_DATETIME",					{type:ZmSetting.T_CONFIG, defaultValue:"20250812-0119"});
	this.registerSetting("CLIENT_RELEASE",					{type:ZmSetting.T_CONFIG, defaultValue:"20250812011201"});
	this.registerSetting("CLIENT_VERSION",					{type:ZmSetting.T_CONFIG, defaultValue:"10.1.9_GA_0325"});
	this.registerSetting("CONFIG_PATH",						{type:ZmSetting.T_CONFIG, defaultValue:appContextPath + "/js/zimbraMail/config"});
	this.registerSetting("CSFE_EXPORT_URI",					{type:ZmSetting.T_CONFIG});
	this.registerSetting("CSFE_MSG_FETCHER_URI",			{type:ZmSetting.T_CONFIG});
	this.registerSetting("CSFE_SERVER_URI",					{type:ZmSetting.T_CONFIG});
	this.registerSetting("CSFE_UPLOAD_URI",					{type:ZmSetting.T_CONFIG});
	this.registerSetting("CSFE_ATTACHMENT_UPLOAD_URI",		{type:ZmSetting.T_CONFIG});
	this.registerSetting("DEV",								{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("FORCE_CAL_OFF",					{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("HELP_URI",						{name:"zimbraHelpAdvancedURL", type:ZmSetting.T_CONFIG, defaultValue:appContextPath + ZmMsg.helpURI});
	this.registerSetting("HTTP_PORT",						{type:ZmSetting.T_CONFIG, defaultValue:ZmSetting.HTTP_DEFAULT_PORT});
	this.registerSetting("HTTPS_PORT",						{type:ZmSetting.T_CONFIG, defaultValue:ZmSetting.HTTPS_DEFAULT_PORT});
	this.registerSetting("INSTANT_NOTIFY_INTERVAL",			{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_INT, defaultValue:500}); // milliseconds
	this.registerSetting("INSTANT_NOTIFY_TIMEOUT",			{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_INT, defaultValue:300}); // seconds
	this.registerSetting("IS_DEV_SERVER",					{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("LOG_REQUEST",						{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("LOGO_URI",						{type:ZmSetting.T_CONFIG, defaultValue:null});
	this.registerSetting("PROTOCOL_MODE",					{type:ZmSetting.T_CONFIG, defaultValue:ZmSetting.PROTO_HTTP});
	this.registerSetting("SERVER_VERSION",					{type:ZmSetting.T_CONFIG});
	this.registerSetting("SHOW_SCRIPT_ERRORS",				{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("TIMEOUT",							{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_INT, defaultValue:30}); // seconds
	this.registerSetting("USE_XML",							{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SMIME_HELP_URI",						{type:ZmSetting.T_CONFIG, defaultValue:appContextPath + ZmMsg.smimeHelpURI});

	// DOMAIN SETTINGS
	this.registerSetting("CHANGE_PASSWORD_URL",				{type:ZmSetting.T_CONFIG});
	this.registerSetting("PUBLIC_URL",						{type:ZmSetting.T_CONFIG});
	this.registerSetting("ADMIN_URL",						{type:ZmSetting.T_CONFIG});
	this.registerSetting("DISABLE_SENSITIVE_ZIMLETS_IN_MIXED_MODE",		{type:ZmSetting.T_CONFIG});
	this.registerSetting("DISABLE_MODERN_CLIENT",			{name:"zimbraModernWebClientDisabled", type:ZmSetting.T_DOMAIN, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});


	// COS SETTINGS - APPS
	this.registerSetting("BRIEFCASE_ENABLED",				{name:"zimbraFeatureBriefcasesEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("DOCUMENT_EDITOR_ENABLED",			{name:"zimbraFeatureDocumentEditingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("ATTACHMENTS_BLOCKED",				{name:"zimbraAttachmentsBlocked", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CALENDAR_ENABLED",				{name:"zimbraFeatureCalendarEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CALENDAR_UPSELL_ENABLED",			{name:"zimbraFeatureCalendarUpsellEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CALENDAR_UPSELL_URL",				{name:"zimbraFeatureCalendarUpsellURL", type:ZmSetting.T_COS});
	this.registerSetting("CONTACTS_ENABLED",				{name:"zimbraFeatureContactsEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CONTACTS_UPSELL_ENABLED",			{name:"zimbraFeatureContactsUpsellEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CONTACTS_UPSELL_URL",				{name:"zimbraFeatureContactsUpsellURL", type:ZmSetting.T_COS});
	this.registerSetting("IMPORT_ENABLED",					{name:"zimbraFeatureImportFolderEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("EXPORT_ENABLED",					{name:"zimbraFeatureExportFolderEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
    this.registerSetting("MAIL_ENABLED",					{name:"zimbraFeatureMailEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
    this.registerSetting("EXTERNAL_USER_MAIL_ADDRESS",		{name:"zimbraExternalUserMailAddress", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
    this.registerSetting("ADMIN_MAIL_ENABLED",				{name:"zimbraFeatureAdminMailEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
    this.registerSetting("ADMIN_PREFERENCES_ENABLED",		{name:"zimbraFeatureAdminPreferencesEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("MAIL_UPSELL_ENABLED",				{name:"zimbraFeatureMailUpsellEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("MAIL_UPSELL_URL",					{name:"zimbraFeatureMailUpsellURL", type:ZmSetting.T_COS});
	this.registerSetting("OPTIONS_ENABLED",					{name:"zimbraFeatureOptionsEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("PORTAL_ENABLED",					{name:"zimbraFeaturePortalEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SOCIAL_ENABLED",					{name:"zimbraFeatureSocialEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SOCIAL_EXTERNAL_ENABLED",			{name:"zimbraFeatureSocialExternalEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SOCIAL_EXTERNAL_URL",				{name:"zimbraFeatureSocialExternalURL", type:ZmSetting.T_COS});
	this.registerSetting("SOCIAL_NAME",				        {name:"zimbraFeatureSocialName", type:ZmSetting.T_COS, defaultValue:ZmMsg.communityName});
	this.registerSetting("TASKS_ENABLED",					{name:"zimbraFeatureTasksEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("VOICE_ENABLED",					{name:"zimbraFeatureVoiceEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("VOICE_UPSELL_ENABLED",			{name:"zimbraFeatureVoiceUpsellEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("VOICE_UPSELL_URL",				{name:"zimbraFeatureVoiceUpsellURL", type:ZmSetting.T_COS});
	this.registerSetting("DLS_FOLDER_ENABLED",				{name:"zimbraFeatureDistributionListFolderEnabled", type: ZmSetting.T_COS, dataType: ZmSetting.D_BOOLEAN, defaultValue: true});

	// COS SETTINGS
    this.registerSetting("ATTACHMENTS_VIEW_IN_HTML_ONLY",	{name:"zimbraAttachmentsViewInHtmlOnly", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("AVAILABLE_SKINS",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST, isGlobal:true});
	this.registerSetting("AVAILABLE_CSVFORMATS",			{type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST, isGlobal:true});
	this.registerSetting("BLOCK_SEND_FROM_IMAP_POP",		{name:"zimbraBlockEmailSendFromImapPop", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("CHANGE_PASSWORD_ENABLED",			{name:"zimbraFeatureChangePasswordEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("DISPLAY_NAME",					{name:"displayName", type:ZmSetting.T_COS});
	this.registerSetting("DUMPSTER_ENABLED",				{name:"zimbraDumpsterEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("ERROR_REPORT_URL",				{name:"zimbraErrorReportUrl", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("EXPORT_MAX_DAYS",					{name:"zimbraExportMaxDays", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:0});
	this.registerSetting("FILTER_BATCH_SIZE",               {name:"zimbraFilterBatchSize", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue: 10000});
    this.registerSetting("FLAGGING_ENABLED",				{name:"zimbraFeatureFlaggingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("ENABLE_RETENTION_POLICY",			{name:"zimbraFeatureRetentionPolicyEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("FOLDERS_EXPANDED",				{name:"zimbraPrefFoldersExpanded", type:ZmSetting.T_METADATA, dataType: ZmSetting.D_HASH, isImplicit:true, section:ZmSetting.M_IMPLICIT});
	this.registerSetting("FOLDER_TREE_OPEN",				{name:"zimbraPrefFolderTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isImplicit:true});
	this.registerSetting("FOLDER_TREE_SASH_WIDTH",          {name:"zimbraPrefFolderTreeSash", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_INT, isImplicit:true, section:ZmSetting.M_IMPLICIT});
	this.registerSetting("GAL_AUTOCOMPLETE_ENABLED",		{name:"zimbraFeatureGalAutoCompleteEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN,	defaultValue:false});
	this.registerSetting("GAL_ENABLED",						{name:"zimbraFeatureGalEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN,	defaultValue:true});
	this.registerSetting("HAB_ROOT",						{type:ZmSetting.T_CONFIG});
	this.registerSetting("GROUP_CALENDAR_ENABLED",			{name:"zimbraFeatureGroupCalendarEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("HTML_COMPOSE_ENABLED",			{name:"zimbraFeatureHtmlComposeEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("IDLE_SESSION_TIMEOUT",			{name:"zimbraMailIdleSessionTimeout", type:ZmSetting.T_COS, dataType:ZmSetting.D_LDAP_TIME, defaultValue:0});
	this.registerSetting("IMAP_ACCOUNTS_ENABLED",			{name:"zimbraFeatureImapDataSourceEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("IMPORT_ON_LOGIN_ENABLED",			{name:"zimbraDataSourceImportOnLogin", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("INSTANT_NOTIFY",					{name:"zimbraFeatureInstantNotify", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("LOCALE_CHANGE_ENABLED",			{name:"zimbraFeatureLocaleChangeEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("LOCALES",							{type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST});
	this.registerSetting("LOGIN_URL",						{name:"zimbraWebClientLoginURL", type:ZmSetting.T_COS});
	this.registerSetting("LOGOUT_URL",						{name:"zimbraWebClientLogoutURL", type:ZmSetting.T_COS});
	this.registerSetting("MIN_POLLING_INTERVAL",			{name:"zimbraMailMinPollingInterval", type:ZmSetting.T_COS, dataType:ZmSetting.D_LDAP_TIME, defaultValue:120});
	this.registerSetting("MOBILE_SYNC_ENABLED",				{name:"zimbraFeatureMobileSyncEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("MOBILE_POLICY_ENABLED",			{name:"zimbraFeatureMobilePolicyEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("POP_ACCOUNTS_ENABLED",			{name:"zimbraFeaturePop3DataSourceEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("PORTAL_NAME",						{name:"zimbraPortalName", type:ZmSetting.T_COS, defaultValue:"example"});
	this.registerSetting("PRIORITY_INBOX_ENABLED",          {name:"zimbraFeaturePriorityInboxEnabled", type:ZmSetting.T_COS, dataType: ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("PWD_MAX_LENGTH",					{name:"zimbraPasswordMaxLength", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:64});
	this.registerSetting("PWD_MIN_LENGTH",					{name:"zimbraPasswordMinLength", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:6});
	this.registerSetting("QUOTA",							{name:"zimbraMailQuota", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:0});
	this.registerSetting("SAVED_SEARCHES_ENABLED",			{name:"zimbraFeatureSavedSearchesEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SEARCH_TREE_OPEN",				{name:"zimbraPrefSearchTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isImplicit:true});
	this.registerSetting("SHARING_ENABLED",					{name:"zimbraFeatureSharingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHARING_PUBLIC_ENABLED",			{name:"zimbraPublicSharingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHARING_EXTERNAL_ENABLED",		{name:"zimbraExternalSharingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHORTCUT_ALIASES_ENABLED",		{name:"zimbraFeatureShortcutAliasesEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHOW_OFFLINE_LINK",				{name:"zimbraWebClientShowOfflineLink", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SIGNATURES_ENABLED",				{name:"zimbraFeatureSignaturesEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SKIN_CHANGE_ENABLED",				{name:"zimbraFeatureSkinChangeEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
    this.registerSetting("SOCIAL_FILTERS_ENABLED",          {name:"zimbraFeatureSocialFiltersEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST, isImplicit: true, section:ZmSetting.M_IMPLICIT});
	this.registerSetting("SPAM_ENABLED",					{name:"zimbraFeatureAntispamEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("TAG_TREE_OPEN",					{name:"zimbraPrefTagTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isImplicit:true});
	this.registerSetting("TAGGING_ENABLED",					{name:"zimbraFeatureTaggingEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("VIEW_ATTACHMENT_AS_HTML",			{name:"zimbraFeatureViewInHtmlEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("EXPAND_DL_ENABLED",				{name:"zimbraFeatureDistributionListExpandMembersEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("FORCE_CLEAR_COOKIES",				{name:"zimbraForceClearCookies", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("SPELL_DICTIONARY",                                {name:"zimbraPrefSpellDictionary", type:ZmSetting.T_COS, defaultValue:""});
	this.registerSetting("TWO_FACTOR_AUTH_AVAILABLE",	    {name:"zimbraFeatureTwoFactorAuthAvailable", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("TWO_FACTOR_AUTH_REQUIRED",	    {name:"zimbraFeatureTwoFactorAuthRequired", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("TWO_FACTOR_AUTH_METHOD_ALLOWED",  {name:"zimbraTwoFactorAuthMethodAllowed", type:ZmSetting.T_COS, defaultValue:["app"]});
	this.registerSetting("TRUSTED_DEVICES_ENABLED",         {name:"zimbraFeatureTrustedDevicesEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("APP_PASSWORDS_ENABLED",	        {name:"zimbraFeatureAppSpecificPasswordsEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("DISABLE_VERSION_CHANGE_DIALOG",   {name:"zimbraServerVersionChangeNotificationDisabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});

	this.registerSetting("RESET_PASSWORD_STATUS",				{name:"zimbraFeatureResetPasswordStatus", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("PASSWORD_RECOVERY_EMAIL",				{name:"zimbraPrefPasswordRecoveryAddress", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("PASSWORD_RECOVERY_EMAIL_STATUS",			{name:"zimbraPrefPasswordRecoveryAddressStatus", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("RESET_PASSWORD_RECOVERY_CODE_EXPIRY",				{name:"zimbraResetPasswordRecoveryCodeExpiry", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("PASSWORD_RECOVERY_CODE_VALIDITY",				{name:"zimbraRecoveryAccountCodeValidity", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("PASSWORD_RECOVERY_SUSPENSION_TIME",				{name:"zimbraFeatureResetPasswordSuspensionTime", type:ZmSetting.T_COS, dataType:ZmSetting.D_STRING});
	this.registerSetting("SHARED_FOLDER_MOBILE_SYNC_ENABLED",				{name:"zimbraFeatureSharedFolderMobileSyncEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN});

	// user metadata (included with COS since the user can't change them)
	this.registerSetting("LICENSE_STATUS",					{type:ZmSetting.T_COS, defaultValue:ZmSetting.LICENSE_GOOD});
	this.registerSetting("QUOTA_USED",						{type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});    
	this.registerSetting("USERID",							{name:"zimbraId", type:ZmSetting.T_COS});
	this.registerSetting("USERNAME",						{type:ZmSetting.T_COS});
	this.registerSetting("CN",								{name:"cn", type:ZmSetting.T_COS});
	this.registerSetting("LAST_ACCESS",						{type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});
	this.registerSetting("PREVIOUS_SESSION",				{type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});
	this.registerSetting("RECENT_MESSAGES",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});
	this.registerSetting("REST_URL",						{name:"rest" , type:ZmSetting.T_COS});
	this.registerSetting("IS_ADMIN",						{name:"zimbraIsAdminAccount", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue: false});
	this.registerSetting("IS_EXTERNAL",						{name:"zimbraIsExternalVirtualAccount", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue: false});
	this.registerSetting("IS_DELEGATED_ADMIN",				{name:"zimbraIsDelegatedAdminAccount", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue: false});
    this.registerSetting("MESSAGE_SIZE_LIMIT",              {type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});
    this.registerSetting("DOCUMENT_SIZE_LIMIT",             {type:ZmSetting.T_COS, dataType:ZmSetting.D_INT});

	// CLIENT SIDE FEATURE SUPPORT
	this.registerSetting("ATTACHMENT_ENABLED",				{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("ATT_VIEW_ENABLED",				{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("BROWSER_TOOLTIPS_ENABLED",		{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("EVAL_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("FEED_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("HELP_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("HISTORY_SUPPORT_ENABLED",			{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:("onhashchange" in window)});
	this.registerSetting("NOTES_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("PRINT_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SEARCH_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SHORTCUT_LIST_ENABLED",			{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("OFFLINE_ENABLED",					{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:appCtxt.isOffline});
	this.registerSetting("SPELL_CHECK_ENABLED",				{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN});
	this.registerSetting("SPELL_CHECK_ADD_WORD_ENABLED",	{type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:!AjxEnv.isSafari || AjxEnv.isSafari3up || AjxEnv.isChrome});

	//SETTINGS SET AT DOMAIN LEVEL
	this.registerSetting("EMAIL_VALIDATION_REGEX",			{name:"zimbraMailAddressValidationRegex", type:ZmSetting.T_DOMAIN, dataType:ZmSetting.D_LIST});
	this.registerSetting("SUPPORTED_HELPS",					{name:"zimbraWebClientSupportedHelps", type:ZmSetting.T_DOMAIN, dataType:ZmSetting.D_LIST});

	// USER PREFERENCES (mutable)

	// general preferences
	this.registerSetting("ACCOUNTS",						{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("TWO_FACTOR_AUTH_ENABLED",	        {name:"zimbraTwoFactorAuthEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("TWO_FACTOR_AUTH_METHOD_ENABLED",  {name:"zimbraTwoFactorAuthMethodEnabled", type:ZmSetting.T_COS});
	this.registerSetting("TWO_FACTOR_AUTH_PRIMARY_METHOD",  {name:"zimbraPrefPrimaryTwoFactorAuthMethod", type:ZmSetting.T_PREF});
	this.registerSetting("ACCOUNT_TREE_OPEN",				{name:"zimbraPrefAccountTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isImplicit:true});
	this.registerSetting("CHILD_ACCTS_VISIBLE",				{name:"zimbraPrefChildVisibleAccount", type:ZmSetting.T_PREF, dataType:ZmSetting.D_LIST});
	this.registerSetting("CLIENT_TYPE",						{name:"zimbraPrefClientType", type:ZmSetting.T_PREF, defaultValue:ZmSetting.CLIENT_ADVANCED});
	this.registerSetting("COMPOSE_AS_FORMAT",				{name:"zimbraPrefComposeFormat", type:ZmSetting.T_PREF, defaultValue:ZmSetting.COMPOSE_HTML, isGlobal:true});
	this.registerSetting("COMPOSE_INIT_FONT_COLOR",			{name:"zimbraPrefHtmlEditorDefaultFontColor", type:ZmSetting.T_PREF, defaultValue:ZmSetting.COMPOSE_FONT_COLOR, isGlobal:true});
	this.registerSetting("COMPOSE_INIT_FONT_FAMILY",		{name:"zimbraPrefHtmlEditorDefaultFontFamily", type:ZmSetting.T_PREF, defaultValue:ZmSetting.COMPOSE_FONT_FAM, isGlobal:true});
	this.registerSetting("COMPOSE_INIT_FONT_SIZE",			{name:"zimbraPrefHtmlEditorDefaultFontSize", type:ZmSetting.T_PREF, defaultValue:ZmSetting.COMPOSE_FONT_SIZE, isGlobal:true});
    this.registerSetting("COMPOSE_INIT_DIRECTION",			{name:"zimbraPrefComposeDirection", type:ZmSetting.T_PREF, defaultValue:ZmSetting.LTR, isGlobal:true});
    this.registerSetting("SHOW_COMPOSE_DIRECTION_BUTTONS",	{name:"zimbraPrefShowComposeDirection", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("DEFAULT_TIMEZONE",				{name:"zimbraPrefTimeZoneId", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, defaultValue:AjxTimezone.getServerId(AjxTimezone.DEFAULT), isGlobal:true});
    this.registerSetting("WEBCLIENT_OFFLINE_BROWSER_KEY",	{name:"zimbraPrefWebClientOfflineBrowserKey", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, isImplicit:true});
    this.registerSetting("DEFAULT_PRINTFONTSIZE",	    	{name:"zimbraPrefDefaultPrintFontSize", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, defaultValue:ZmSetting.PRINT_FONT_SIZE, isGlobal:true});
	this.registerSetting("GROUPBY_HASH",                    {type: ZmSetting.T_PREF, dataType:ZmSetting.D_HASH});
	this.registerSetting("GROUPBY_LIST",                    {name:"zimbraPrefGroupByList", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_HASH, isImplicit:true, section:ZmSetting.M_IMPLICIT});
    this.registerSetting("FILTERS",							{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("FONT_NAME",						{name:"zimbraPrefFont", type:ZmSetting.T_PREF, defaultValue: ZmSetting.FONT_SYSTEM, isGlobal:true});
	this.registerSetting("FONT_SIZE",						{name:"zimbraPrefFontSize", type:ZmSetting.T_PREF, defaultValue: ZmSetting.FONT_SIZE_NORMAL, isGlobal:true});
	this.registerSetting("IDENTITIES",						{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("INITIALLY_SEARCH_GAL",			{name:"zimbraPrefGalSearchEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("LIST_VIEW_COLUMNS",				{name:"zimbraPrefListViewColumns", type:ZmSetting.T_PREF, dataType:ZmSetting.D_HASH, isImplicit:true});
	this.registerSetting("LOCALE_NAME",						{name:"zimbraPrefLocale", type:ZmSetting.T_PREF, defaultValue:appRequestLocaleId, isGlobal:true});
	this.registerSetting("SHOW_SELECTION_CHECKBOX",			{name:"zimbraPrefShowSelectionCheckbox", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isGlobal:true});
	// PAGE_SIZE: number of items to fetch for virtual paging; also used for number of msgs in one page of a conv
	this.registerSetting("PAGE_SIZE",						{name: "zimbraPrefItemsPerVirtualPage", type:ZmSetting.T_PREF, dataType:ZmSetting.D_INT, defaultValue:50, isGlobal:true});
	this.registerSetting("PASSWORD",						{type:ZmSetting.T_PREF, dataType:ZmSetting.D_NONE});
	this.registerSetting("POLLING_INTERVAL",				{name:"zimbraPrefMailPollingInterval", type:ZmSetting.T_PREF, dataType:ZmSetting.D_LDAP_TIME, defaultValue:300});
	this.registerSetting("POLLING_INTERVAL_ENABLED",		{name:"zimbraFeatureMailPollingIntervalPreferenceEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("SEARCH_INCLUDES_SHARED",			{name:"zimbraPrefIncludeSharedItemsInSearch", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("SEARCH_INCLUDES_SPAM",			{name:"zimbraPrefIncludeSpamInSearch", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("SEARCH_INCLUDES_TRASH",			{name:"zimbraPrefIncludeTrashInSearch", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("SHORT_ADDRESS",					{name:"zimbraPrefShortEmailAddress", type:ZmSetting.T_PREF, dataType: ZmSetting.D_BOOLEAN, defaultValue: true});
	this.registerSetting("SHORTCUTS",						{name:"zimbraPrefShortcuts", type:ZmSetting.T_PREF});
	this.registerSetting("SHOW_SEARCH_STRING",				{name:"zimbraPrefShowSearchString", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("SIGNATURES",						{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("SIGNATURES_MAX",					{name:"zimbraSignatureMaxNumEntries", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:20});
	this.registerSetting("SIGNATURES_MIN",					{name:"zimbraSignatureMinNumEntries", type:ZmSetting.T_COS, dataType:ZmSetting.D_INT, defaultValue:1});
	this.registerSetting("SKIN_NAME",						{name:"zimbraPrefSkin", type:ZmSetting.T_PREF, defaultValue:"skin", isGlobal:true});
	this.registerSetting("SORTING_PREF",					{name:"zimbraPrefSortOrder", type:ZmSetting.T_PREF, dataType:ZmSetting.D_HASH, isImplicit:true, isGlobal:true, dontSaveDefault: true});
	this.registerSetting("USE_KEYBOARD_SHORTCUTS",			{name:"zimbraPrefUseKeyboardShortcuts", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("VIEW_AS_HTML",					{name:"zimbraPrefMessageViewHtmlPreferred", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("VOICE_ACCOUNTS",					{type: ZmSetting.T_PREF, dataType: ZmSetting.D_HASH});
	this.registerSetting("WARN_ON_EXIT",					{name:"zimbraPrefWarnOnExit", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("DISPLAY_TIME_IN_MAIL_LIST",		{name:"zimbraPrefDisplayTimeInMailList", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});

	this._registerOfflineSettings();
	this._registerZimletsSettings();

	// need to do this before loadUserSettings(), and zimlet settings are not tied to an app where it would normally be done
	this.registerSetting("ZIMLET_TREE_OPEN",				{name:"zimbraPrefZimletTreeOpen", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isImplicit:true});
	
	//shared settings
	this.registerSetting("MAIL_ALIASES",					{name:"zimbraMailAlias", type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST});
	this.registerSetting("ALLOW_FROM_ADDRESSES",			{name:"zimbraAllowFromAddress", type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST});

	// Internal pref to control display of mail-related preferences. The only time it will be false is for a delegated admin with zimbraFeatureAdminMailEnabled and
	// zimbraFeatureAdminPreferencesEnabled set to FALSE.
	this.registerSetting("MAIL_PREFERENCES_ENABLED",	    {type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});

	appCtxt.notifyZimlets("onZmSettings_initialize", [this]);

	ZmApp.runAppFunction("registerSettings", this);
};

/**
 * @private
 */
ZmSettings.prototype._registerZimletsSettings =
function() {
	// zimlet-specific
	this.registerSetting("CHECKED_ZIMLETS_ENABLED",			{name:"zimbraFeatureManageZimlets", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, isGlobal:true});
	this.registerSetting("CHECKED_ZIMLETS",					{name:"zimbraPrefZimlets", type:ZmSetting.T_PREF, dataType:ZmSetting.D_LIST, isGlobal:true});
    this.registerSetting("MANDATORY_ZIMLETS",		        {name:"zimbraZimletMandatoryZimlets", type:ZmSetting.T_COS, dataType:ZmSetting.D_LIST});
    this.registerSetting("ZIMLETS_SYNCHRONOUS",		        {name:"zimbraZimletLoadSynchronously", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});

};

/**
 * @private
 */
ZmSettings.prototype._registerOfflineSettings =
function() {
	if (!appCtxt.isOffline) { return; }

	// offline-specific
	this.registerSetting("OFFLINE_ACCOUNT_FLAVOR",			{name:"offlineAccountFlavor", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING});
	this.registerSetting("OFFLINE_COMPOSE_ENABLED",			{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("OFFLINE_DEBUG_TRACE",				{type:ZmSetting.T_CONFIG, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
	this.registerSetting("OFFLINE_IS_MAILTO_HANDLER",		{name:"zimbraPrefMailtoHandlerEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("OFFLINE_REMOTE_SERVER_URI",		{name:"offlineRemoteServerUri", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING});
	this.registerSetting("OFFLINE_REPORT_EMAIL",			{type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, defaultValue:"zdesktop-report@zimbra.com", isGlobal:true});
	this.registerSetting("OFFLINE_SHOW_ALL_MAILBOXES",		{name:"offlineShowAllMailboxes", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, section:ZmSetting.M_OFFLINE, isGlobal:true});
	this.registerSetting("OFFLINE_ALL_MAILBOXES_TREE_OPEN",	{name:"offlineAllMailboxesTreeOpen", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, section:ZmSetting.M_OFFLINE, isGlobal:true, isImplicit:true});
	this.registerSetting("OFFLINE_NOTIFY_NEWMAIL_ON_INBOX",	{name:"offlineNotifyNewMailOnInbox", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_BOOLEAN, defaultValue:true, section:ZmSetting.M_OFFLINE, isGlobal:true});
	this.registerSetting("OFFLINE_SAVED_SEARCHES_TREE_OPEN",{name:"offlineSavedSearchesTreeOpen", type:ZmSetting.T_METADATA, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, section:ZmSetting.M_OFFLINE, isGlobal:true, isImplicit:true});
	this.registerSetting("OFFLINE_SMTP_ENABLED",			{name:"zimbraDataSourceSmtpEnabled", type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:true});
	this.registerSetting("OFFLINE_SUPPORTS_MAILTO",			{type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("OFFLINE_SUPPORTS_DOCK_UPDATE",	{type:ZmSetting.T_PREF, dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
	this.registerSetting("OFFLINE_WEBAPP_URI",				{name:"offlineWebappUri", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING});
    this.registerSetting("OFFLINE_BACKUP_INTERVAL",	        {name:"zimbraPrefOfflineBackupInterval", type:ZmSetting.T_PREF, dataType:ZmSetting.D_INT, defaultValue:0, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_PATH",	            {name:"zimbraPrefOfflineBackupPath", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_KEEP",	            {name:"zimbraPrefOfflineBackupKeep", type:ZmSetting.T_PREF, dataType:ZmSetting.D_INT, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_ACCOUNT_ID",       {name:"zimbraPrefOfflineBackupAccountId", type:ZmSetting.T_PREF, dataType:ZmSetting.D_INT, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_RESTORE",          {name:"zimbraPrefOfflineBackupRestore", dataType:ZmSetting.D_INT, isGlobal:true});
    this.registerSetting("OFFLINE_BACKUP_NOW_BUTTON",       {name:"zimbraPrefOfflineBackupAccount", dataType:ZmSetting.D_BOOLEAN, defaultValue:false, isGlobal:true});
    this.registerSetting("OFFLINE_ZIMLET_SYNC_ACCOUNT_ID",  {name:"zimbraPrefOfflineZimletSyncAccountId", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING, isGlobal:true});
	this.registerSetting("OFFLINE_WEBAPP_URI",				{name:"offlineWebappUri", type:ZmSetting.T_PREF, dataType:ZmSetting.D_STRING});

	// reset the help URI to zimbra.com for offline
	this.registerSetting("HELP_URI",						{type:ZmSetting.T_CONFIG, defaultValue:"https://www.zimbra.com/desktop7/"});
//	// make default false for DUMPSTER_ENABLED. shouldn't be necessary since GetInfoResponse includes zimbraDumpsterEnabled:"FALSE", but can't find why settings is not read correctly
	this.registerSetting("DUMPSTER_ENABLED",				{name:"zimbraDumpsterEnabled", type:ZmSetting.T_COS, dataType:ZmSetting.D_BOOLEAN, defaultValue:false});
    this.registerSetting("OFFLINE_UPDATE_NOTIFY",			{name:"zimbraPrefOfflineUpdateChannel", type: ZmSetting.T_PREF, dataType: ZmSetting.D_STRING, isGlobal:true});
};

/**
 * @private
 */
ZmSettings.prototype._changeListener =
function(ev) {

	if (ev.type !== ZmEvent.S_SETTING) {
		return;
	}

	var id = ev.source.id,
		value = ev.source.getValue();

	if (id === ZmSetting.QUOTA_USED) {
		appCtxt.getAppController().setUserInfo();
	}
	else if (id === ZmSetting.POLLING_INTERVAL) {
		appCtxt.getAppController().setPollInterval();
	}
	else if (id === ZmSetting.SKIN_NAME) {
		this._showConfirmDialog(ZmMsg.skinChangeRestart, this._refreshBrowserCallback.bind(this));
	}
	else if (id === ZmSetting.SHOW_SELECTION_CHECKBOX) {
		this._showConfirmDialog(value ? ZmMsg.checkboxChangeRestartShow : ZmMsg.checkboxChangeRestartHide, this._refreshBrowserCallback.bind(this));
	}
	else if (id === ZmSetting.FONT_NAME || id === ZmSetting.FONT_SIZE) {
		this._showConfirmDialog(ZmMsg.fontChangeRestart, this._refreshBrowserCallback.bind(this));
	}
	else if (id === ZmSetting.LOCALE_NAME) {
		// bug: 29786
		if (appCtxt.isOffline && AjxEnv.isPrism) {
			try {
				netscape.security.PrivilegeManager.enablePrivilege('UniversalXPConnect');
				var prefs = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefBranch);
				if (prefs) {
					var newLocale = appCtxt.get(ZmSetting.LOCALE_NAME).replace("_", "-");
					prefs.setCharPref("general.useragent.locale", newLocale);
					prefs.setCharPref("intl.accept_languages", newLocale);
				}
			} catch (ex) {
				// do nothing for now
			}
		}
		this._showConfirmDialog(ZmMsg.localeChangeRestart, this._refreshBrowserCallback.bind(this));
	}
	else if (id === ZmSetting.CHILD_ACCTS_VISIBLE) {
		this._showConfirmDialog(ZmMsg.accountChangeRestart, this._refreshBrowserCallback.bind(this));
	}
	else if (appCtxt.isOffline && id === ZmSetting.OFFLINE_IS_MAILTO_HANDLER) {
		appCtxt.getAppController().registerMailtoHandler(true, ev.source.getValue());
	}
	else if (appCtxt.isOffline && id === ZmSetting.OFFLINE_UPDATE_NOTIFY) {
		appCtxt.getAppController()._offlineUpdateChannelPref(ev.source.getValue());
	}
	else if (id === ZmSetting.DISPLAY_TIME_IN_MAIL_LIST) {
		this._showConfirmDialog(value ? ZmMsg.timeInMailListChangeRestartShow : ZmMsg.timeInMailListChangeRestartHide, this._refreshBrowserCallback.bind(this));
	}
};

// Shows a confirm dialog that asks the user if they want to reload ZCS to show the change they just made
ZmSettings.prototype._showConfirmDialog = function(text, callback, style) {

	var confirmDialog = appCtxt.getYesNoMsgDialog();
	confirmDialog.reset();
	confirmDialog.registerCallback(DwtDialog.YES_BUTTON, callback);
	confirmDialog.setMessage(text, style || DwtMessageDialog.WARNING_STYLE);
	confirmDialog.popup();
};

ZmSettings.prototype._implicitChangeListener =
function(ev) {
	var id = ev.source.id;
	var sourceType = ev.source.type;

	// Allow to perform implicit settings save (e.g folder expand) if the source type is META even when zimbraFeatureOptionsEnabled is false.
	if (!appCtxt.get(ZmSetting.OPTIONS_ENABLED) && ZmSetting.T_METADATA !== sourceType) {
		return;
	}
	if (ev.type != ZmEvent.S_SETTING) { return; }
	var setting = this.getSetting(id);
	if (id == ZmSetting.FOLDERS_EXPANDED && window.duringExpandAll) {
		if (!window.afterExpandAllCallback) {
			window.afterExpandAllCallback = this.save.bind(this, [setting], null, null, appCtxt.getActiveAccount(), true);
		}
		return;
	}
	if (ZmSetting.IS_IMPLICIT[id] && setting) {
        if (id === ZmSetting.WEBCLIENT_OFFLINE_BROWSER_KEY) {
            var callback = this._offlineSettingsSaveCallback.bind(this, setting);
        }
		else {
			//Once implicit preference is saved, reload the application cache to get the latest changes
			var callback = appCtxt.reloadAppCache.bind(appCtxt, false);
		}
		this.save([setting], callback, null, appCtxt.getActiveAccount(), true);
	}
};

/**
 * @private
 */
ZmSettings.prototype._refreshBrowserCallback =
function(args) {
	appCtxt.getYesNoMsgDialog().popdown();
	window.onbeforeunload = ZmZimbraMail.getConfirmExitMethod();
	var url = AjxUtil.formatUrl({qsArgs : args});
	window.location.replace(url);
};

// Adds/replaces a CSS rule that comprises user font prefs
ZmSettings.prototype._updateUserFontPrefsRule =
function() {
	if (this._userFontPrefsRuleIndex != null) {
		DwtCssStyle.removeRule(document.styleSheets[0], this._userFontPrefsRuleIndex);
	}
	var selector = "." + ZmSetting.USER_FONT_CLASS;
	var declaration = "font-family:" + appCtxt.get(ZmSetting.COMPOSE_INIT_FONT_FAMILY) + ";" +
					  "font-size:" + appCtxt.get(ZmSetting.COMPOSE_INIT_FONT_SIZE) + ";" +
					  "color:" + appCtxt.get(ZmSetting.COMPOSE_INIT_FONT_COLOR) + ";";
	this._userFontPrefsRuleIndex = DwtCssStyle.addRule(document.styleSheets[0], selector, declaration);
};

// Check license to see if voice feature is allowed
// License block format:
//
//  <license status="OK">
//    <attr name="SMIME">FALSE</attr>
//    <attr name="VOICE">TRUE</attr>
//  </license>

ZmSettings.prototype._hasVoiceFeature = function() {

    var info = this.getInfoResponse;
    var license = info && info.license;
    var status = license && license.status;

    if (!license || !license.attr) {
        return false;
    }

    // License not installed or not activated or expired
    if (ZmSetting.LICENSE_MSG[status]) {
        return false;
    }

    // check for VOICE license attribute

    for (var i = 0; license && i < license.attr.length; i++) {
        var attr = license.attr[i];

        if (attr.name == "VOICE") {
            return attr._content == "TRUE";
        }
    }

    return false;
};

/**
 * @private
 */
ZmSettings.prototype._offlineSettingsSaveCallback =
function(setting) {
	var offlineBrowserKey = setting.getValue();
	var localOfflineBrowserKey = localStorage.getItem(ZmSetting.WEBCLIENT_OFFLINE_BROWSER_KEY);
	if (offlineBrowserKey && offlineBrowserKey.indexOf(localOfflineBrowserKey) !== -1) {
		this._showConfirmDialog(ZmMsg.offlineChangeRestart, this._refreshBrowserCallback.bind(this));
    }
    else {
        ZmOffline.deleteOfflineData();
        appCtxt.initWebOffline();// To reset the property isWebClientOfflineSupported
	    //delete outbox folder
	    var outboxFolder = appCtxt.getById(ZmFolder.ID_OUTBOX);
	    if (outboxFolder) {
		    outboxFolder.notifyDelete();
	    }
    }
	//Always reload appcache whenever offline setting is enabled/disabled. Appcache will be updated/emptied depending upon the setting.
	appCtxt.reloadAppCache(true);
};

ZmSettings.prototype.getMailAppForDelegatedAdmin =
function() {
    if (!this.mailApp) {
        this.mailApp = new ZmMailApp();
    }
    return this.mailApp;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmMetaData")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines classes for the storage/retrieval of generic user data.
 */

/**
 * Stores generic data to the server via <code>&lt;Set[Mailbox|Custom]MetadataRequest&gt;</code> and
 * <code>&lt;Get[Mailbox|Custom]MetadataRequest&gt;</code>.
 * @class
 * This class provides a general way to store per-user data using arbitrary
 * key/value pairs. NOTE: the server does not support modifying data so if there
 * are any changes, *all* the data must be re-set per section. Data can be
 * written on the mailbox or on an individual mailbox item. If the constructor
 * receives an itemId, then data will be retrieved/written via [Get|Set]CustomMetadataRequest.
 * <br/>
 * <br/>
 * The section data is mapped into {@link ZmSettings} (based on the key name) to allow
 * for easy access. When creating/setting a *new* key/value, naming conventions
 * should be followed as defined by prefs in {@link ZmSettings}. For example, if adding
 * a new key called "foo", the name for the key should be "zimbraPrefFoo" and
 * should be added to the list of settings in {@link ZmSettings} with type set to
 * ZmSetting.T_METADATA
 *
 * @param {ZmAccount}	account		Optional. The account this meta data belongs to
 * @param {String}	itemId		Optional.  If specified, (custom) meta data will be saved on the item
 *
 * @author Parag Shah
 */
ZmMetaData = function(account, itemId) {

	this._sections = {};
	this._account = account;
	this._itemId = itemId;
};

ZmMetaData.prototype.constructor = ZmMetaData;


// Consts

ZmMetaData.NAMESPACE		= "zwc";


// Public Methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmMetaData.prototype.toString =
function() {
	return "ZmMetaData";
};

/**
 * Saves the given section and corresponding key/value pair to the server.
 *
 * @param {String}	section		the name of the section to save
 * @param {Object}	data			the list of key/value pairs
 * @param {ZmBatchCommand}	batchCommand	if part of a batch command
 * @param {AjxCallback}	callback		the callback to trigger on successful save
 * @param {AjxCallback}	errorCallback	the error callback to trigger on error
 * @param {Boolean} sensitive if <code>true</code>, attempt to use secure conn to protect data
 */
ZmMetaData.prototype.set =
function(section, data, batchCommand, callback, errorCallback, sensitive) {
	var soapDoc;
	if (this._itemId) {
		soapDoc = AjxSoapDoc.create("SetCustomMetadataRequest", "urn:zimbraMail");
		soapDoc.getMethod().setAttribute("id", this._itemId);
	} else {
		soapDoc = AjxSoapDoc.create("SetMailboxMetadataRequest", "urn:zimbraMail");
	}
	var metaNode = soapDoc.set("meta");
	metaNode.setAttribute("section", [ZmMetaData.NAMESPACE, section].join(":"));

	for (var i in data) {
		var d = data[i];

		// serialize if we're dealing with an object
		if (AjxUtil.isObject(d)) {
			d = ZmSetting.serialize(d, ZmSetting.D_HASH);
		}

		// If we want to remove any data from SetMailboxMetadataRequest then don't send it as part of request,
		// server will automatically remove it
		if (d !== null && d !== "") {
			var a = soapDoc.set("a", d, metaNode);
			a.setAttribute("n", i);
		}
	}

	if (batchCommand) {
		batchCommand.addNewRequestParams(soapDoc, callback, errorCallback);
	}
	else {
		var params = {
			soapDoc: soapDoc,
			asyncMode: true,
			callback: callback,
			errorCallback: errorCallback,
			accountName: (this._account ? this._account.name : null),
			sensitive: sensitive
		};

		appCtxt.getAppController().sendRequest(params);
	}
};

/**
 * Fetches the given section name from the server unless its already been
 * fetched (and therefore cached)
 *
 * @param {String}	section		section of meta data to fetch
 * @param {ZmBatchCommand}	batchCommand	if part of a separate batch command
 * @param {AjxCallback}	callback		the callback to trigger once meta data is fetched
 * @param {AjxCallback}	errorCallback	the error callback to trigger on error
 */
ZmMetaData.prototype.get =
function(section, batchCommand, callback, errorCallback) {
	var command = batchCommand || (new ZmBatchCommand());
	var sectionName = [ZmMetaData.NAMESPACE, section].join(":");

	var cachedSection = this._sections[sectionName];

	// if not yet cached, go fetch it
	if (!cachedSection) {
		var soapDoc;
		if (this._itemId) {
			soapDoc = AjxSoapDoc.create("GetCustomMetadataRequest", "urn:zimbraMail");
			soapDoc.getMethod().setAttribute("id", this._itemId);
		} else {
			soapDoc = AjxSoapDoc.create("GetMailboxMetadataRequest", "urn:zimbraMail");
		}
		var metaNode = soapDoc.set("meta");
		metaNode.setAttribute("section", sectionName);

		command.addNewRequestParams(soapDoc);

		if (!batchCommand) {
			command.run(callback, errorCallback);
		}
	}
	else {
		if (callback) {
			callback.run(cachedSection);
		} else {
			return cachedSection;
		}
	}
};

/**
 * Loads meta data from the server
 *
 * @param {Array}		sections	the sections to load
 * @param {AjxCallback}	callback	the callback
 * @param {ZmBatchCommand}		batchCommand if part of batch command
 */
ZmMetaData.prototype.load =
function(sections, callback, batchCommand) {
	if (!sections) { return; }
	if (!(sections instanceof Array)) { sections = [sections]; }

	var command = batchCommand || (new ZmBatchCommand());
	for (var i = 0; i < sections.length; i++) {
		if (sections[i] == ZmSetting.M_OFFLINE && !appCtxt.isOffline) { continue; }
		this.get(sections[i], command);
	}

	if (!batchCommand) {
		if (command.size() > 0) {
			var respCallback = new AjxCallback(this, this._handleLoad, [callback]);
			var offlineCallback = this._handleOfflineLoad.bind(this, respCallback);
			command.run(respCallback, null, offlineCallback);
		}
	} else {
		if (callback) {
			callback.run(this._sections);
		}
	}
};

/**
 * @private
 */
ZmMetaData.prototype._handleLoad =
function(callback, result) {
	this._sections = {};

	var br = result.getResponse().BatchResponse;
	if (br) {
		var metaDataResp = (this._itemId != null) ? br.GetCustomMetadataResponse : br.GetMailboxMetadataResponse;
		if (metaDataResp && metaDataResp.length) {
			if (ZmOffline.isOnlineMode()) {
				localStorage.setItem("MetadataResponse", JSON.stringify(br));
			}
			for (var i = 0; i < metaDataResp.length; i++) {
				var data = metaDataResp[i].meta[0];
				this._sections[data.section] = data._attrs;
			}
		}
	}

	if (callback) {
		callback.run(this._sections);
	}
};

/**
 * @private
 */
ZmMetaData.prototype._handleOfflineLoad =
function(callback) {
	var result = localStorage.getItem("MetadataResponse");
	if (result) {
		var csfeResult = new ZmCsfeResult({BatchResponse : JSON.parse(result)});
		callback.run(csfeResult);
	}
};

/**
 * Modifies the given section with new key/value pairs
 *
 * @param {Array}			section			the section to modify
 * @param {Object}			data			the list of key/value pairs
 * @param {ZmBatchCommand}	batchCommand	Optional. the batch command the request should be a part of
 * @param {AjxCallback}		callback		the callback called on successful modify
 * @param {AjxCallback}		errorCallback	the error callback to trigger on error
 */
ZmMetaData.prototype.modify =
function(section, data, batchCommand, callback, errorCallback) {
	var soapDoc = AjxSoapDoc.create("ModifyMailboxMetadataRequest", "urn:zimbraMail");
	var metaNode = soapDoc.set("meta");
	metaNode.setAttribute("section", [ZmMetaData.NAMESPACE, section].join(":"));

	for (var i in data) {
		var a = soapDoc.set("a", data[i], metaNode);
		a.setAttribute("n", i);
	}

	if (batchCommand) {
		batchCommand.addNewRequestParams(soapDoc, callback, errorCallback);
	}
	else {
		var params = {
			soapDoc: soapDoc,
			asyncMode: true,
			callback: callback,
			errorCallback: errorCallback,
			accountName: (this._account ? this._account.name : null)
		};

		appCtxt.getAppController().sendRequest(params);
	}
};

/**
 * Saves all data within the given section out to the server. If section is not
 * provided, all sections are saved.
 *
 * @param {Array}		sections		the sections to save
 * @param {AjxCallback}	callback		the callback called on successful save
 * @param {ZmBatchCommand}	batchCommand		the batch command the request should be a part of
 */
ZmMetaData.prototype.save =
function(sections, callback, batchCommand) {
	if (!sections) { return; }
	if (!(sections instanceof Array)) { sections = [sections]; }

	var acct = this._account ? this._account.name : null;
	var command = batchCommand || (new ZmBatchCommand(null, acct));

	for (var i = 0; i < sections.length; i++) {
		var s = sections[i];
		var sectionName = [ZmMetaData.NAMESPACE, s].join(":");
		var sectionData = this._sections[sectionName];
		if (sectionData) {
			this.set(s, sectionData, command);
		}
	}

	if (!batchCommand) {
		if (command.size() > 0) {
			command.run(callback);
		}
	} else {
		if (callback) {
			callback.run();
		}
	}
};

/**
 * Updates the local section cache with the given key/value pair.
 *
 * @param {String}	section	the section to update
 * @param {String}	key		the key to update
 * @param {String}	value		the new value
 */
ZmMetaData.prototype.update =
function(section, key, value) {
	var sectionName = [ZmMetaData.NAMESPACE, section].join(":");
	var sectionObj = this._sections[sectionName];
	if (!sectionObj) {
		sectionObj = this._sections[sectionName] = {};
	}
	sectionObj[key] = value;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmKeyMap")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines the key map.
 */

/**
 * Creates a key map for the ZCS application.
 * @class
 * This class maps keys to actions for the ZCS application. There is a global key map
 * with bindings that apply to any key not handled by the current controller; these
 * global bindings apply across applications (mail, contacts, etc). Key bindings that
 * are context-dependent are tied to a particular controller. If that controller has
 * control, then those bindings will be used.
 * <br/>
 * <br/>
 * Bindings are passed in via the <code>ZmKeys</code> object, which is populated from a properties
 * file. The identifiers used in the properties file must match those used here.
 * 
 * @author Ross Dargahi
 * @author Conrad Damon
 * 
 * @extends		DwtKeyMap
 */
ZmKeyMap = function(subclassInit) {
	if (subclassInit) { return };

	ZmKeyMap._setPreconditions();
	DwtKeyMap.call(this);
	this._load(this._map, ZmKeys);
};

ZmKeyMap.prototype = new DwtKeyMap(true);
ZmKeyMap.prototype.constructor = ZmKeyMap;

ZmKeyMap.prototype.isZmKeyMap = true;
ZmKeyMap.prototype.toString = function() { return "ZmKeyMap"; };


// Map names (must match those in the key properties file ZmKeys.properties)
ZmKeyMap.MAP_ADDRESS			= "address";
ZmKeyMap.MAP_BRIEFCASE			= "briefcase";
ZmKeyMap.MAP_CALENDAR			= "calendar";
ZmKeyMap.MAP_CALL				= "call";
ZmKeyMap.MAP_COMPOSE			= "compose";
ZmKeyMap.MAP_CONTACTS			= "contacts";
ZmKeyMap.MAP_CONVERSATION		= "conversation";
ZmKeyMap.MAP_CONVERSATION_LIST	= "conversationList";
ZmKeyMap.MAP_DL_ADDRESS_LIST	= "dlAddressList";
ZmKeyMap.MAP_EDIT_APPOINTMENT	= "editAppointment";
ZmKeyMap.MAP_EDIT_CONTACT		= "editContact";
ZmKeyMap.MAP_EDIT_TASK			= "editTask";
ZmKeyMap.MAP_GLOBAL				= "global";
ZmKeyMap.MAP_MAIL				= "mail";
ZmKeyMap.MAP_MESSAGE			= "message";
ZmKeyMap.MAP_QUICK_REPLY		= "quickReply";
ZmKeyMap.MAP_OPTIONS			= "options";
ZmKeyMap.MAP_TASKS				= "tasks";
ZmKeyMap.MAP_VIEW_APPOINTMENT	= "viewAppointment";
ZmKeyMap.MAP_VOICEMAIL			= "voicemail";

// Action codes
ZmKeyMap.ADDRESS_PICKER			= "AddressPicker";
ZmKeyMap.ADD_EXTERNAL_CALENDAR	= "AddExternalCalendar";
ZmKeyMap.ATTACHMENT				= "Attachment";
ZmKeyMap.CAL_DAY_VIEW			= "DayView";
ZmKeyMap.CAL_FB_VIEW			= "FBView";
ZmKeyMap.CAL_LIST_VIEW			= "CalListView";
ZmKeyMap.CAL_MONTH_VIEW			= "MonthView";
ZmKeyMap.CAL_WEEK_VIEW			= "WeekView";
ZmKeyMap.CAL_WORK_WEEK_VIEW		= "WorkWeekView";
ZmKeyMap.CALL_MANAGER       	= "CallManager";
ZmKeyMap.CANCEL					= "Cancel";
ZmKeyMap.COLLAPSE				= "Collapse";
ZmKeyMap.COLLAPSE_ALL			= "CollapseAll";
ZmKeyMap.DEL					= "Delete";
ZmKeyMap.SHIFT_DEL				= "ShiftDelete";
ZmKeyMap.DOWNLOAD           	= "Download";
ZmKeyMap.EDIT					= "Edit";
ZmKeyMap.EXPAND					= "Expand";
ZmKeyMap.EXPAND_ALL				= "ExpandAll";
ZmKeyMap.FIRST_UNREAD			= "FirstUnread";
ZmKeyMap.FIRST_UNREAD_MSG		= "FirstUnreadMsg";
ZmKeyMap.FLAG					= "Flag";
ZmKeyMap.FOCUS_CONTENT_PANE		= "FocusContentPane";
ZmKeyMap.FOCUS_SEARCH_BOX		= "FocusSearchBox";
ZmKeyMap.FOCUS_TOOLBAR			= "FocusToolbar";
ZmKeyMap.FORWARD				= "Forward";
ZmKeyMap.GET_MAIL				= "GetMail";
ZmKeyMap.GOTO_BRIEFCASE			= "GoToBriefcase";
ZmKeyMap.GOTO_CALENDAR			= "GoToCalendar";
ZmKeyMap.GOTO_CONTACTS			= "GoToContacts";
ZmKeyMap.GOTO_DRAFTS			= "GoToDrafts";
ZmKeyMap.GOTO_JUNK				= "GoToJunk";
ZmKeyMap.GOTO_INBOX				= "GoToInbox";
ZmKeyMap.GOTO_MAIL				= "GoToMail";
ZmKeyMap.GOTO_OPTIONS			= "GoToOptions";
ZmKeyMap.GOTO_SENT				= "GoToSent";
ZmKeyMap.GOTO_TASKS				= "GoToTasks";
ZmKeyMap.GOTO_TRASH				= "GoToTrash";
ZmKeyMap.GOTO_VOICE				= "GoToVoice";
ZmKeyMap.HTML_FORMAT			= "HtmlFormat";
ZmKeyMap.KEEP_READING			= "KeepReading";
ZmKeyMap.LAST_UNREAD			= "LastUnread";
ZmKeyMap.LAST_UNREAD_MSG		= "LastUnreadMsg";
ZmKeyMap.MARK_COMPLETE			= "MarkComplete";
ZmKeyMap.MARK_HEARD				= "MarkHeard";
ZmKeyMap.MARK_READ				= "MarkRead";
ZmKeyMap.MARK_UNCOMPLETE		= "MarkUncomplete";
ZmKeyMap.MARK_UNHEARD			= "MarkUnheard";
ZmKeyMap.MARK_UNREAD			= "MarkUnread";
ZmKeyMap.MOVE					= "Move";
ZmKeyMap.MOVE_TO_INBOX			= "MoveToInbox";
ZmKeyMap.MOVE_TO_JUNK			= "MoveToJunk";
ZmKeyMap.MOVE_TO_TRASH			= "MoveToTrash";
ZmKeyMap.MUTE_UNMUTE_CONV	    = "MuteUnmuteConv";
ZmKeyMap.NEW					= "New";
ZmKeyMap.NEW_APPT				= "NewAppointment";
ZmKeyMap.NEW_BRIEFCASE			= "NewBriefcase";
ZmKeyMap.NEW_CALENDAR			= "NewCalendar";
ZmKeyMap.NEW_CONTACT			= "NewContact";
ZmKeyMap.NEW_DOC    			= "NewDocument";
ZmKeyMap.NEW_FILE				= "NewFile";
ZmKeyMap.NEW_FOLDER				= "NewFolder";
ZmKeyMap.NEW_MESSAGE			= "NewMessage";
ZmKeyMap.NEW_MESSAGE_WIN		= "NewMessageWindow";
ZmKeyMap.NEW_SEARCH				= "NewSearch";
ZmKeyMap.NEW_TAG				= "NewTag";
ZmKeyMap.NEW_TASK				= "NewTask";
ZmKeyMap.NEW_WINDOW				= "NewWindow";
ZmKeyMap.NEXT_APPT				= "NextAppointment";
ZmKeyMap.NEXT_CONV				= "NextConversation";
ZmKeyMap.NEXT_DAY				= "NextDay";
ZmKeyMap.NEXT_MSG				= "NextMessage";
ZmKeyMap.NEXT_PAGE				= "NextPage";
ZmKeyMap.NEXT_UNREAD			= "NextUnread";
ZmKeyMap.NEXT_UNREAD_MSG		= "NextUnreadMsg";
ZmKeyMap.PLAY					= "Play";
ZmKeyMap.PRESENCE_MENU			= "PresenceMenu";
ZmKeyMap.PREV_APPT				= "PreviousAppointment";
ZmKeyMap.PREV_CONV				= "PreviousConversation";
ZmKeyMap.PREV_DAY				= "PreviousDay";
ZmKeyMap.PREV_MSG				= "PreviousMessage";
ZmKeyMap.PREV_PAGE				= "PreviousPage";
ZmKeyMap.PREV_UNREAD			= "PreviousUnread";
ZmKeyMap.PREV_UNREAD_MSG		= "PreviousUnreadMsg";
ZmKeyMap.PRINT					= "Print";
ZmKeyMap.PRINT_ALL				= "PrintAll";
ZmKeyMap.QUICK_ADD				= "QuickAdd";
ZmKeyMap.QUICK_REMINDER 	    = "QuickReminder";
ZmKeyMap.READING_PANE_BOTTOM	= "ReadingPaneAtBottom";
ZmKeyMap.READING_PANE_OFF		= "ReadingPaneOff";
ZmKeyMap.READING_PANE_RIGHT		= "ReadingPaneOnRight";
ZmKeyMap.REFRESH				= "Refresh";
ZmKeyMap.REPLY					= "Reply";
ZmKeyMap.REPLY_ALL				= "ReplyAll";
ZmKeyMap.SAVE					= "Save";
ZmKeyMap.SAVED_SEARCH			= "SavedSearch";
ZmKeyMap.SELECT_ALL				= "SelectAll";
ZmKeyMap.SEND					= "Send";
ZmKeyMap.SHORTCUTS				= "Shortcuts";
ZmKeyMap.SHOW_FRAGMENT			= "ShowFragment";
ZmKeyMap.SPAM					= "Spam";
ZmKeyMap.SPELLCHECK				= "Spellcheck";
ZmKeyMap.TAG					= "Tag";
ZmKeyMap.TODAY					= "Today";
ZmKeyMap.TOGGLE					= "Toggle";
ZmKeyMap.UNTAG					= "Untag";
ZmKeyMap.VIEW_BY_CONV			= "ViewByConversation";
ZmKeyMap.VIEW_BY_MSG			= "ViewByMessage";
ZmKeyMap.VISIT					= "Visit";
ZmKeyMap.VISIT_TAG				= "VisitTag";

// HTML entities (used to display keys)
ZmKeyMap.ENTITY = {};
ZmKeyMap.ENTITY[DwtKeyMap.ARROW_LEFT]	= "&larr;"
ZmKeyMap.ENTITY[DwtKeyMap.ARROW_RIGHT]	= "&rarr;"
ZmKeyMap.ENTITY[DwtKeyMap.ARROW_UP]		= "&uarr;"
ZmKeyMap.ENTITY[DwtKeyMap.ARROW_DOWN]	= "&darr;"
ZmKeyMap.ENTITY['"'] = "&quot;"
ZmKeyMap.ENTITY['&'] = "&amp;"
ZmKeyMap.ENTITY['<'] = "&lt;"
ZmKeyMap.ENTITY['>'] = "&gt;"
ZmKeyMap.ENTITY[DwtKeyMap.COMMA]		= ",";
ZmKeyMap.ENTITY[DwtKeyMap.SEMICOLON]	= ";";
ZmKeyMap.ENTITY[DwtKeyMap.BACKSLASH] 	= "\\";

// preconditions for maps
ZmKeyMap.MAP_PRECONDITION = {};

// preconditions for specific shortcuts
ZmKeyMap.ACTION_PRECONDITION = {};

ZmKeyMap._setPreconditions =
function() {
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_COMPOSE]				= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_MAIL]				= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CONVERSATION_LIST]	= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CONVERSATION]		= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_DL_ADDRESS_LIST]		= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_MESSAGE]				= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CONTACTS]			= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_EDIT_CONTACT]		= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CALENDAR]			= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_EDIT_APPOINTMENT]	= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_OPTIONS]				= ZmSetting.OPTIONS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_BRIEFCASE]			= ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_TASKS]				= ZmSetting.TASKS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_EDIT_TASK]			= ZmSetting.TASKS_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_VOICEMAIL]			= ZmSetting.VOICE_ENABLED;
	ZmKeyMap.MAP_PRECONDITION[ZmKeyMap.MAP_CALL]				= ZmSetting.VOICE_ENABLED;
	
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL] = {};
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.FOCUS_SEARCH_BOX]	= ZmSetting.SEARCH_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_BRIEFCASE]		= ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_CALENDAR]		= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_CONTACTS]		= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_MAIL]			= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_OPTIONS]		= ZmSetting.OPTIONS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_TASKS]			= ZmSetting.TASKS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.GOTO_VOICE]			= ZmSetting.VOICE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_APPT]			= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_BRIEFCASEITEM]	= ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_CALENDAR]		= ZmSetting.CALENDAR_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_CONTACT]			= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_FILE]			= ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_DOC]				= ZmSetting.DOCS_ENABLED && ZmSetting.BRIEFCASE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_FOLDER]			= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_MESSAGE]			= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_MESSAGE_WIN]		= ZmSetting.MAIL_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_TAG]				= ZmSetting.TAGGING_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.NEW_TASK]			= ZmSetting.TASKS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.SAVED_SEARCH]		= ZmSetting.SAVED_SEARCHES_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.TAG]					= ZmSetting.TAGGING_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_GLOBAL][ZmKeyMap.UNTAG]				= ZmSetting.TAGGING_ENABLED;

	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE] = {};
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE][ZmKeyMap.ADDRESS_PICKER]		= ZmSetting.CONTACTS_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE][ZmKeyMap.HTML_FORMAT]		= ZmSetting.HTML_COMPOSE_ENABLED;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE][ZmKeyMap.NEW_WINDOW]			= ZmSetting.NEW_WINDOW_COMPOSE;
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_COMPOSE][ZmKeyMap.SAVE]				= ZmSetting.SAVE_DRAFT_ENABLED;

	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_EDIT_APPOINTMENT] = {};
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_EDIT_APPOINTMENT][ZmKeyMap.HTML_FORMAT]	= ZmSetting.HTML_COMPOSE_ENABLED;

    ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_CALENDAR] = {};
	ZmKeyMap.ACTION_PRECONDITION[ZmKeyMap.MAP_CALENDAR][ZmKeyMap.CAL_FB_VIEW]		= ZmSetting.FREE_BUSY_VIEW_ENABLED;
};

/**
 * Checks if this map is valid. A map may have a precondition,
 * which is either a setting that must be true, or a function that returns
 * true.
 *
 * @param {String}	mapName	the name of map
 * @return	{Boolean}	<code>true</code> if the map is valid
 * 
 * @private
 */
ZmKeyMap.prototype._checkMap = function(mapName) {

	var result = this._checkedMap[mapName] = appCtxt.checkPrecondition(ZmKeyMap.MAP_PRECONDITION[mapName]);
	return result;
};

/**
 * Checks if this action is valid. A map or an action may have a precondition,
 * which is either a setting that must be true, or a function that returns
 * true.
 *
 * @param {String} mapName	the name of map
 * @param {String} action	the action to check
 * @return	{Boolean}	<code>true</code> if the action is valid
 * 
 * @private
 */
ZmKeyMap.prototype._checkAction = function(mapName, action) {

	if (this._checkedMap[mapName] === false || (!this._checkedMap[mapName] && !this._checkMap(mapName))) {
		return false;
	}

	var mapPre = ZmKeyMap.ACTION_PRECONDITION[mapName];
	return appCtxt.checkPrecondition(mapPre && mapPre[action]);
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmTimezone")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the timezone class.
 */

/**
 * Creates a timezone
 * @class
 * This class represents a timezone.
 * 
 */
ZmTimezone = function() {}

// Static methods

/**
 * Gets the default timezone.
 * 
 * @return	{AjxTimezone}	the timezone
 */
ZmTimezone.getDefault =
function() {
	var shell = DwtShell.getShell(window);
	var serverId = appCtxt.get(ZmSetting.DEFAULT_TIMEZONE);
	return (serverId) ? AjxTimezone.getClientId(serverId) : AjxTimezone.DEFAULT;
};

/**
 * Gets the default rule.
 * 
 * @return	{String}	the rule
 */
ZmTimezone.getDefaultRule =
function() {
	return AjxTimezone.getRule(ZmTimezone.getDefault());
};

/**
 * This function mirrors the <code>AjxSoapDoc#set</code> method
 * to add a timezone element at the specific place within the
 * given SOAP document. The added element takes the form of the
 * <code>&lt;tz></code> element as defined for <code>&lt;SearchRequest&gt;</code>.
 *
 * @param {object|AjxSoapDoc}	request			the JSON request object or SOAP document
 * @param {String}	timezoneClientId	the client identifier
 * @param {Node}	parentNode		(optional) the parent node at which to add
 * @param {Boolean}	skipKnownTimezone	(optional) if <code>true</code>, does not add the "tz" element if it's one of the known set
 */
ZmTimezone.set =
function(request, timezoneClientId, parentNode, skipKnownTimezone) {
	var timezone = AjxTimezone.getRule(timezoneClientId);
	if (!timezone) { return; }

	if (timezone.autoDetected || !skipKnownTimezone) {
		if (request instanceof AjxSoapDoc) {
			ZmTimezone._setSoap(request, timezoneClientId, parentNode, timezone);
		} else {
			ZmTimezone._setJson(request, timezoneClientId, timezone);
		}
	}
};

/**
 * @private
 */
ZmTimezone._setSoap =
function(soapDoc, timezoneClientId, parentNode, timezone) {
	var tz = soapDoc.set("tz", null, parentNode);
	var id = AjxTimezone.getServerId(timezoneClientId);
	tz.setAttribute("id", id);
	if (timezone.autoDetected) {
		tz.setAttribute("stdoff", timezone.standard.offset);
		if (timezone.daylight) {
			tz.setAttribute("dayoff", timezone.daylight.offset);
            var enames = [ "standard", "daylight" ];
            var pnames = [ "mon", "mday", "week", "wkday", "hour", "min", "sec" ];
            for (var i = 0; i < enames.length; i++) {
                var ename = enames[i];
                var onset = timezone[ename];
                
                var el = soapDoc.set(ename, null, tz);
                for (var j = 0; j < pnames.length; j++) {
                    var pname = pnames[j];
                    if (pname in onset) {
                        el.setAttribute(pname, onset[pname]);
                    }
                }
            }
        }
	}
};

/**
 * @private
 */
ZmTimezone._setJson =
function(request, timezoneClientId, timezone) {
	var id = AjxTimezone.getServerId(timezoneClientId);
	var tz = request.tz = {id:id};
	if (timezone.autoDetected) {
		tz.stdoff = timezone.standard.offset;
		if (timezone.daylight) {
			tz.dayoff = timezone.daylight.offset;
            var enames = [ "standard", "daylight" ];
            var pnames = [ "mon", "mday", "week", "wkday", "hour", "min", "sec" ];
            for (var i = 0; i < enames.length; i++) {
                var ename = enames[i];
                var onset = timezone[ename];
                tz[ename] = {};
                for (var j = 0; j < pnames.length; j++) {
                    var pname = pnames[j];
                    if (pname in onset) {
                    	tz[ename][pname] = onset[pname];
                    }
                }
            }
        }
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmItem")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines an item.
 */

/**
 * Creates an item.
 * @class
 * An item is a piece of data that may contain user content. Most items are taggable. Currently,
 * the following things are items: conversation, message, attachment, appointment, and contact.
 * <br/>
 * <br/>
 * An item typically appears in the context of a containing list. Its event handling
 * is generally handled by the list so we avoid having the same listeners on each item. If we
 * create a context where an item stands alone outside a list context, then the item will have
 * its own listeners and do its own notification handling.
 *
 * @author Conrad Damon
 * 
 * @param {constant}	type		type of object (conv, msg, etc)
 * @param {int}			id			the unique id
 * @param {ZmList}		list		a list that contains this item
 * @param {Boolean}		noCache		if <code>true</code>, do not cache this item
 * 
 * @extends		ZmModel
 */
ZmItem = function(type, id, list, noCache) {

	if (arguments.length == 0) { return; }
	ZmModel.call(this, type);

	this.type = type;
	this.id = id;
	this.list = list;
	this._list = {};

    // number of views using this item
    this.refCount = 0;

	this.tags = [];
	this.tagHash = {};
	this.folderId = 0;

	// make sure the cached item knows which lists it is in, even if those other lists
	// have separate instances of this item - propagate view IDs from currently cached item
	var curItem = appCtxt.getById(id);
	if (curItem) {
		this._list = AjxUtil.hashCopy(curItem._list);
        if (!list) {
            // No list specified, preserve the previous list
            this.list = curItem.list;
        }
	}
	if (list) {
		this._list[list.id] = true;
	}
	
	if (id && !noCache) {
		appCtxt.cacheSet(id, this);
	}
};

ZmItem.prototype = new ZmModel;
ZmItem.prototype.constructor = ZmItem;

ZmItem.prototype.isZmItem = true;
ZmItem.prototype.toString = function() { return "ZmItem"; };


ZmItem.APP 				= {};	// App responsible for item
ZmItem.MSG_KEY 			= {};	// Type names
ZmItem.ICON 			= {};	// Representative icons
ZmItem.RESULTS_LIST 	= {};	// Function for creating search results list

// fields that can be part of a displayed item
ZmItem.F_ACCOUNT		= ZmId.FLD_ACCOUNT;
ZmItem.F_ATTACHMENT		= ZmId.FLD_ATTACHMENT;
ZmItem.F_CAPACITY		= ZmId.FLD_CAPACITY;
ZmItem.F_COMPANY		= ZmId.FLD_COMPANY;
ZmItem.F_DATE			= ZmId.FLD_DATE;
ZmItem.F_DEPARTMENT		= ZmId.FLD_DEPARTMENT;
ZmItem.F_EMAIL			= ZmId.FLD_EMAIL;
ZmItem.F_EXPAND			= ZmId.FLD_EXPAND;
ZmItem.F_FILE_TYPE		= ZmId.FLD_FILE_TYPE;
ZmItem.F_FLAG			= ZmId.FLD_FLAG;
ZmItem.F_FOLDER			= ZmId.FLD_FOLDER;
ZmItem.F_FRAGMENT		= ZmId.FLD_FRAGMENT;
ZmItem.F_FROM			= ZmId.FLD_FROM;
ZmItem.F_HOME_PHONE		= ZmId.FLD_HOME_PHONE;
ZmItem.F_ID				= ZmId.FLD_ID;
ZmItem.F_INDEX			= ZmId.FLD_INDEX;
ZmItem.F_ITEM_ROW		= ZmId.FLD_ITEM_ROW;
ZmItem.F_ITEM_ROW_3PANE	= ZmId.FLD_ITEM_ROW_3PANE;
ZmItem.F_LOCATION		= ZmId.FLD_LOCATION;
ZmItem.F_NAME			= ZmId.FLD_NAME;
ZmItem.F_NOTES			= ZmId.FLD_NOTES;
ZmItem.F_PARTICIPANT	= ZmId.FLD_PARTICIPANT;
ZmItem.F_PCOMPLETE		= ZmId.FLD_PCOMPLETE;
ZmItem.F_PRIORITY		= ZmId.FLD_PRIORITY;
ZmItem.F_RECURRENCE		= ZmId.FLD_RECURRENCE;
ZmItem.F_SELECTION		= ZmId.FLD_SELECTION;
ZmItem.F_SELECTION_CELL	= ZmId.FLD_SELECTION_CELL;
ZmItem.F_SIZE			= ZmId.FLD_SIZE;
ZmItem.F_SORTED_BY		= ZmId.FLD_SORTED_BY;	// placeholder for 3-pane view
ZmItem.F_STATUS			= ZmId.FLD_STATUS;
ZmItem.F_READ			= ZmId.FLD_READ;
ZmItem.F_MUTE			= ZmId.FLD_MUTE;
ZmItem.F_SUBJECT		= ZmId.FLD_SUBJECT;
ZmItem.F_TAG			= ZmId.FLD_TAG;
ZmItem.F_TAG_CELL		= ZmId.FLD_TAG_CELL;
ZmItem.F_TO             = ZmId.FLD_TO;
ZmItem.F_TYPE			= ZmId.FLD_TYPE;
ZmItem.F_VERSION        = ZmId.FLD_VERSION;
ZmItem.F_WORK_PHONE		= ZmId.FLD_WORK_PHONE;
ZmItem.F_LOCK           = ZmId.FLD_LOCK;
ZmItem.F_SHARES			= ZmId.FLD_SHARES;
ZmItem.F_MSG_PRIORITY   = ZmId.FLD_MSG_PRIORITY;
ZmItem.F_APP_PASSCODE_CREATED = ZmId.FLD_CREATED;
ZmItem.F_APP_PASSCODE_LAST_USED = ZmId.FLD_LAST_USED;

// Action requests for different items
ZmItem.SOAP_CMD = {};

// Item fields (for modify events)
ZmItem.TAGS_FIELD = 1;

// Item flags
ZmItem.FLAG_ATTACH				= "a";
ZmItem.FLAG_FLAGGED				= "f";
ZmItem.FLAG_FORWARDED			= "w";
ZmItem.FLAG_ISDRAFT 			= "d";
ZmItem.FLAG_ISSCHEDULED 		= "c";
ZmItem.FLAG_ISSENT				= "s";
ZmItem.FLAG_READ_RECEIPT_SENT	= "n";
ZmItem.FLAG_REPLIED				= "r";
ZmItem.FLAG_UNREAD				= "u";
ZmItem.FLAG_MUTE				= "(";
ZmItem.FLAG_LOW_PRIORITY		= "?";
ZmItem.FLAG_HIGH_PRIORITY		= "!";
ZmItem.FLAG_PRIORITY            = "+"; //msg prioritization
ZmItem.FLAG_NOTE                = "t"; //specially for notes
ZmItem.FLAG_OFFLINE_CREATED     = "o";

ZmItem.ALL_FLAGS = [
	ZmItem.FLAG_FLAGGED,
	ZmItem.FLAG_ATTACH,
	ZmItem.FLAG_UNREAD,
	ZmItem.FLAG_MUTE,
	ZmItem.FLAG_REPLIED,
	ZmItem.FLAG_FORWARDED,
	ZmItem.FLAG_ISSENT,
	ZmItem.FLAG_READ_RECEIPT_SENT,
	ZmItem.FLAG_ISDRAFT,
	ZmItem.FLAG_ISSCHEDULED,
	ZmItem.FLAG_HIGH_PRIORITY,
	ZmItem.FLAG_LOW_PRIORITY,
	ZmItem.FLAG_PRIORITY,
    ZmItem.FLAG_NOTE,
    ZmItem.FLAG_OFFLINE_CREATED
];

// Map flag to item property
ZmItem.FLAG_PROP = {};
ZmItem.FLAG_PROP[ZmItem.FLAG_ATTACH]			= "hasAttach";
ZmItem.FLAG_PROP[ZmItem.FLAG_FLAGGED]			= "isFlagged";
ZmItem.FLAG_PROP[ZmItem.FLAG_FORWARDED]			= "isForwarded";
ZmItem.FLAG_PROP[ZmItem.FLAG_ISDRAFT] 			= "isDraft";
ZmItem.FLAG_PROP[ZmItem.FLAG_ISSCHEDULED] 		= "isScheduled";
ZmItem.FLAG_PROP[ZmItem.FLAG_ISSENT]			= "isSent";
ZmItem.FLAG_PROP[ZmItem.FLAG_READ_RECEIPT_SENT]	= "readReceiptSent";
ZmItem.FLAG_PROP[ZmItem.FLAG_REPLIED]			= "isReplied";
ZmItem.FLAG_PROP[ZmItem.FLAG_UNREAD]			= "isUnread";
ZmItem.FLAG_PROP[ZmItem.FLAG_MUTE]			    = "isMute";
ZmItem.FLAG_PROP[ZmItem.FLAG_LOW_PRIORITY]		= "isLowPriority";
ZmItem.FLAG_PROP[ZmItem.FLAG_HIGH_PRIORITY]		= "isHighPriority";
ZmItem.FLAG_PROP[ZmItem.FLAG_PRIORITY]          = "isPriority";
ZmItem.FLAG_PROP[ZmItem.FLAG_NOTE]              = "isNote";
ZmItem.FLAG_PROP[ZmItem.FLAG_OFFLINE_CREATED]   = "isOfflineCreated";

// DnD actions this item is allowed

/**
 * Defines the "move" action.
 * 
 * @see		#getDefaultDndAction
 */
ZmItem.DND_ACTION_MOVE = 1 << 0;
/**
 * Defines the "copy" action.
 * 
 * @see		#getDefaultDndAction
 */
ZmItem.DND_ACTION_COPY = 1 << 1;
/**
 * Defines the "move & copy" action.
 * 
 * @see		#getDefaultDndAction
 */
ZmItem.DND_ACTION_BOTH = ZmItem.DND_ACTION_MOVE | ZmItem.DND_ACTION_COPY;

/**
 * Defines the notes separator which is used by items
 * (such as calendar or share invites) that have notes.
 * 
 */
ZmItem.NOTES_SEPARATOR			= "*~*~*~*~*~*~*~*~*~*";

/**
 * Registers an item and stores information about the given item type.
 *
 * @param {constant}	item		the item type
 * @param	{Hash}	params			a hash of parameters
 * @param {constant}	params.app			the app that handles this item type
 * @param {String}		params.nameKey		the message key for item name
 * @param {String}		params.icon			the name of item icon class
 * @param {String}		params.soapCmd		the SOAP command for acting on this item
 * @param {String}		params.itemClass	the name of class that represents this item
 * @param {String}		params.node			the SOAP response node for this item
 * @param {constant}	params.organizer	the associated organizer
 * @param {String}		params.searchType	the associated type in SearchRequest
 * @param {function}	params.resultsList	the function that returns a {@link ZmList} for holding search results of this type
 */
ZmItem.registerItem =
function(item, params) {
	if (params.app)				{ ZmItem.APP[item]					= params.app; }
	if (params.nameKey)			{ ZmItem.MSG_KEY[item]				= params.nameKey; }
	if (params.icon)			{ ZmItem.ICON[item]					= params.icon; }
	if (params.soapCmd)			{ ZmItem.SOAP_CMD[item]				= params.soapCmd; }
	if (params.itemClass)		{ ZmList.ITEM_CLASS[item]			= params.itemClass; }
	if (params.node)			{ ZmList.NODE[item]					= params.node; }
	if (params.organizer)		{ ZmOrganizer.ITEM_ORGANIZER[item]	= params.organizer; }
	if (params.searchType)		{ ZmSearch.TYPE[item]				= params.searchType; }
	if (params.resultsList)		{ ZmItem.RESULTS_LIST[item]			= params.resultsList; }

	if (params.node) {
		ZmList.ITEM_TYPE[params.node] = item;
	}

	if (params.dropTargets) {
		if (!ZmApp.DROP_TARGETS[params.app]) {
			ZmApp.DROP_TARGETS[params.app] = {};
		}
		ZmApp.DROP_TARGETS[params.app][item] = params.dropTargets;
	}
};

/**
* Gets an item id by taking a normalized id (or an item id) and returning the item id.
* 
* @param	{String}	id		the normalized id
* @return	{String}	the item id
*/
ZmItem.getItemId =
function(id) {
	if (!id) {
		return id;
	}
	if (!ZmItem.SHORT_ID_RE) {
		var shell = DwtShell.getShell(window);
		ZmItem.SHORT_ID_RE = new RegExp(appCtxt.get(ZmSetting.USERID) + ':', "gi");
	}
	return id.replace(ZmItem.SHORT_ID_RE, '');
};

// abstract methods
/**
 * Creates an item.
 * 
 * @param	{Hash}	args		the arguments
 */
ZmItem.prototype.create = function(args) {};
/**
 * Modifies an item.
 * 
 * @param	{Hash}	mods		the arguments
 */
ZmItem.prototype.modify = function(mods) {};

/**
 * Gets the item by id.
 *
 * @param {String}	id		an item id
 * @return	{ZmItem}	the item
 */
ZmItem.prototype.getById =
function(id) {
	if (id == this.id) {
		return this;
	}
};

ZmItem.prototype.getAccount =
function() {
	if (!this.account) {
		var account;

		if (this.folderId) {
			var ac = window.parentAppCtxt || window.appCtxt;
			var folder = ac.getById(this.folderId);
			account = folder && folder.getAccount();
		}

		if (!account) {
			var parsed = ZmOrganizer.parseId(this.id);
			account = parsed && parsed.account;
		}
		this.account = account;
	}
	return this.account;
};

/**
 * Clears the item.
 * 
 */
ZmItem.prototype.clear = function() {

    // only clear data if no views are using this item
    if (this.refCount <= 1) {
        this._evtMgr.removeAll(ZmEvent.L_MODIFY);
        if (this.tags.length) {
            for (var i = 0; i < this.tags.length; i++) {
                this.tags[i] = null;
            }
            this.tags = [];
        }
        for (var i in this.tagHash) {
            this.tagHash[i] = null;
        }
        this.tagHash = {};
    }

    this.refCount--;
};

/**
 * Caches the item.
 * 
 * @return	{Boolean}	<code>true</code> if the item is placed into cache; <code>false</code> otherwise
 */
ZmItem.prototype.cache =
function(){
  if (this.id) {
      appCtxt.cacheSet(this.id, this);
      return true;
  }
  return false;  
};

/**
 * Checks if the item has a given tag.
 * 
 * @param {String}		tagName		tag name
 * @return	{Boolean}	<code>true</code> is this item has the given tag.
 */
ZmItem.prototype.hasTag =
function(tagName) {
	return (this.tagHash[tagName] == true);
};

/**
 * is it possible to add a tag to this item?
 * @param tagName
 * @returns {boolean}
 */
ZmItem.prototype.canAddTag =
function(tagName) {
	return !this.hasTag(tagName);
};


/**
* Gets the folder id that contains this item, if available.
* 
* @return	{String}	the folder id or <code>null</code> for none
*/
ZmItem.prototype.getFolderId =
function() {
	return this.folderId;
};

/**
 * @deprecated
 * Use getRestUrl
 * 
 * @private
 * @see		#getRestUrl
 */
ZmItem.prototype.getUrl =
function() {
	return this.getRestUrl();
};

/**
 * Gets the rest url for this item.
 * 
 * @return	{String}	the url
 */
ZmItem.prototype.getRestUrl =
function() {
	// return REST URL as seen by server
	if (this.restUrl) {
		return this.restUrl;
	}

	// if server doesn't tell us what URL to use, do our best to generate
	var organizerType = ZmOrganizer.ITEM_ORGANIZER[this.type];
	var organizer = appCtxt.getById(this.folderId);
	var url = organizer
		? ([organizer.getRestUrl(), "/", AjxStringUtil.urlComponentEncode(this.name)].join(""))
		: null;

	if (url && this.folderId == ZmFolder.ID_FILE_SHARED_WITH_ME) {
		if (this.sfid) {
			url = [url, "?", "id=", this.sfid].join("");
		} else {
			url = [url, "?", "id=", this.id].join("");
		}
	}

	DBG.println(AjxDebug.DBG3, "NO REST URL FROM SERVER. GENERATED URL: " + url);

	return url;
};

/**
* Gets the appropriate tag image info for this item.
* 
* @return	{String}	the tag image info
*/
ZmItem.prototype.getTagImageInfo =
function() {
	return this.getTagImageFromNames(this.getVisibleTags());
};

/**
 * @deprecated
 * */
ZmItem.prototype.getTagImageFromIds =
function(tagIds) {
	var tagImageInfo;

	if (!tagIds || tagIds.length == 0) {
		tagImageInfo = "Blank_16";
	} else if (tagIds.length == 1) {
        tagImageInfo = this.getTagImage(tagIds[0]);
	} else {
		tagImageInfo = "TagStack";
	}

	return tagImageInfo;
};

ZmItem.prototype.getVisibleTags =
function() {
    if(!appCtxt.get(ZmSetting.TAGGING_ENABLED)){
        return [];
    }
    return this.tags;
	//todo - do we need anything from this?
//    var searchAll = appCtxt.getSearchController().searchAllAccounts;
//    if (!searchAll && this.isShared()) {
//        return [];
//    } else {
//        return this.tags;
//    }
};

ZmItem.prototype.getTagImageFromNames =
function(tags) {

	if (!tags || tags.length == 0) {
		return "Blank_16";
	}
	if (tags.length == 1) {
        return this.getTagImage(tags[0]);
	} 

	return "TagStack";
};


ZmItem.prototype.getTagImage =
function(tagName) {
	//todo - I don't think we need the qualified/normalized/whatever id anymore.
//	var tagFullId = (!this.getAccount().isMain)
//		? ([this.getAccount().id, tagName].join(":"))
//		: (ZmOrganizer.getSystemId(tagName));
	var tagList = appCtxt.getAccountTagList(this);

	var tag = tagList.getByNameOrRemote(tagName);
    return tag ? tag.getIconWithColor() : "Blank_16";
};

/**
* Gets the default action to use when dragging this item. This method
* is meant to be overloaded for items that are read-only and can only be copied.
*
* @param {Boolean}		forceCopy		If set, default DnD action is a copy
* @return	{Object}	the action
*/
ZmItem.prototype.getDefaultDndAction =
function(forceCopy) {
	return (this.isReadOnly() || forceCopy)
		? ZmItem.DND_ACTION_COPY
		: ZmItem.DND_ACTION_MOVE;
};

/**
* Checks if this item is read-only. This method should be
* overloaded by the derived object to determine what "read-only" means.
* 
* @return	{Boolean}	the read-only status
*/
ZmItem.prototype.isReadOnly =
function() {
	return false;
};

/**
 * Checks if this item is shared.
 * 
 * @return	{Boolean}	<code>true</code> if this item is shared (remote)
 */
ZmItem.prototype.isShared =
function() {
	if (this._isShared == null) {
		if (this.id === -1) {
			this._isShared = false;
		} else {
			this._isShared = appCtxt.isRemoteId(this.id);
		}
	}
	return this._isShared;
};

// Notification handling

// For delete and modify notifications, we first apply the notification to this item. Then we
// see if the item is a member of any other lists. If so, we have those other copies of this
// item handle the notification as well. Each will notify through the list that created it.

ZmItem.prototype.notifyDelete =
function() {
	this._notifyDelete();
	for (var listId in this._list) {
		var list = appCtxt.getById(listId);
		if (!list || (this.list && listId == this.list.id)) { continue; }
		var ctlr = list.controller;
		if (!ctlr || ctlr.inactive || (ctlr.getList().id != listId)) { continue; }
		var doppleganger = list.getById(this.id);
		if (doppleganger) {
			doppleganger._notifyDelete();
		}
	}
};

ZmItem.prototype._notifyDelete =
function() {
	this.deleteLocal();
	if (this.list) {
		this.list.deleteLocal([this]);
	}
	this._notify(ZmEvent.E_DELETE);
};

ZmItem.prototype.notifyModify =
function(obj, batchMode) {
	this._notifyModify(obj, batchMode);
	for (var listId in this._list) {
		var list = listId ? appCtxt.getById(listId) : null;
		if (!list || (this.list && (listId == this.list.id))) { continue; }
		var ctlr = list.controller;
		if (!ctlr || ctlr.inactive || (ctlr.getList().id != listId)) { continue; }
		var doppleganger = list.getById(this.id);
		if (doppleganger) {
			doppleganger._notifyModify(obj, batchMode);
		}
	}
};

/**
 * Handles a modification notification.
 *
 * @param {Object}	obj			the item with the changed attributes/content
 * @param {boolean}	batchMode	if true, return event type and don't notify
 */
ZmItem.prototype._notifyModify =
function(obj, batchMode) {
	// empty string is meaningful here, it means no tags
	if (obj.tn != null) {
		this._parseTagNames(obj.tn);
		this._notify(ZmEvent.E_TAGS);
	}
	// empty string is meaningful here, it means no flags
	if (obj.f != null) {
		var flags = this._getFlags();
		var origFlags = {};
		for (var i = 0; i < flags.length; i++) {
			origFlags[flags[i]] = this[ZmItem.FLAG_PROP[flags[i]]];
		}
		this._parseFlags(obj.f);
		var changedFlags = [];
		for (var i = 0; i < flags.length; i++) {
			var on = this[ZmItem.FLAG_PROP[flags[i]]];
			if (origFlags[flags[i]] != on) {
				changedFlags.push(flags[i]);
			}
		}
		if (changedFlags.length) {
			this._notify(ZmEvent.E_FLAGS, {flags: changedFlags});
		}
	}
	if (obj.l != null && obj.l != this.folderId) {
		var details = {oldFolderId:this.folderId};
		this.moveLocal(obj.l);
		if (this.list) {
			this.list.moveLocal([this], obj.l);
		}
		if (batchMode) {
			delete obj.l;			// folder has been handled
			return ZmEvent.E_MOVE;
		} else {
			this._notify(ZmEvent.E_MOVE, details);
		}
	}
};

// Local change handling

/**
 * Applies the given flag change to this item by setting a boolean property.
 *
 * @param {constant}	flag	the flag that changed
 * @param {Boolean}	on		<code>true</code> if the flag is now set
 */
ZmItem.prototype.flagLocal =
function(flag, on) {
	this[ZmItem.FLAG_PROP[flag]] = on;
};

/**
 * Sets the given flag change to this item. Both the flags string and the
 * flag properties are affected.
 *
 * @param {constant}	flag	the flag that changed
 * @param {Boolean}	on	<code>true</code> if the flag is now set
 *
 * @return	{String}		the new flags string
 */
ZmItem.prototype.setFlag =
function(flag, on) {
	this.flagLocal(flag, on);
	var flags = this.flags || "";
	if (on && flags.indexOf(flag) == -1) {
		flags = flags + flag;
	} else if (!on && flags.indexOf(flag) != -1) {
		flags = flags.replace(flag, "");
	}
	this.flags = flags;

	return flags;
};

/**
 * Adds or removes the given tag for this item.
 *
 * @param {Object}		tag		tag name
 * @param {Boolean}		doTag		<code>true</code> if tag is being added; <code>false</code> if it is being removed
 * @return	{Boolean}	<code>true</code> to notify
 */
ZmItem.prototype.tagLocal =
function(tag, doTag) {
	var bNotify = false;
	if (doTag) {
		if (!this.tagHash[tag]) {
			bNotify = true;
			this.tags.push(tag);
			this.tagHash[tag] = true;
		}
	} else {
		for (var i = 0; i < this.tags.length; i++) {
			if (this.tags[i] == tag) {
				this.tags.splice(i, 1);
				delete this.tagHash[tag];
				bNotify = true;
				break;
			}
		}
	}
	
	return bNotify;
};

/**
 * Removes all tags.
 * 
 */
ZmItem.prototype.removeAllTagsLocal =
function() {
	this.tags = [];
	for (var i in this.tagHash) {
		delete this.tagHash[i];
	}
};

/**
 * Deletes local, in case an item wants to do something while being deleted.
 */
ZmItem.prototype.deleteLocal = function() {};

/**
 * Moves the item.
 * 
 * @param	{String}	folderId
 * @param	{AjxCallback}	callback		the callback
 * @param	{AjxCallback}	errorCallback	the callback on error
 * @return	{Object}		the result of the move
 */
ZmItem.prototype.move =
function(folderId, callback, errorCallback) {
	return ZmItem.move(this.id, folderId, callback, errorCallback);
};

/**
 * Moves the item.
 * 
 * @return	{Object}		the result of the move
 */
ZmItem.move =
function(itemId, folderId, callback, errorCallback, accountName) {
	var json = {
		ItemActionRequest: {
			_jsns: "urn:zimbraMail",
			action: {
				id:	itemId instanceof Array ? itemId.join() : itemId,
				op:	"move",
				l:	folderId
			}
		}
	};

	var params = {
		jsonObj:		json,
		asyncMode:		Boolean(callback),
		callback:		callback,
		errorCallback:	errorCallback,
		accountName:	accountName
	};
	return appCtxt.getAppController().sendRequest(params);
};

/**
 * Updates the folder for this item.
 *
 * @param {String}		folderId		the new folder ID
 */
ZmItem.prototype.moveLocal =
function(folderId) {
	this.folderId = folderId;
};

/**
 * Takes a comma-separated list of tag IDs and applies the tags to this item.
 * 
 * @private
 */
ZmItem.prototype._parseTags =
function(str) {	
	this.tags = [];
	this.tagHash = {};
	if (str && str.length) {
		var tags = str.split(",");
		for (var i = 0; i < tags.length; i++) {
			var tagId = Number(tags[i]);
			if (tagId >= ZmOrganizer.FIRST_USER_ID[ZmOrganizer.TAG])
				this.tagLocal(tagId, true);
		}
	}
};

/**
 * Takes a comma-separated list of tag names and applies the tags to this item.
 *
 * @private
 */
ZmItem.prototype._parseTagNames =
function(str) {
	this.tags = [];
	this.tagHash = {};
	if (!str || !str.length) {
		return;
	}
	
	// server escapes comma with backslash
	str = str.replace(/\\,/g, "\u001D");
	var tags = str.split(",");
	
	for (var i = 0; i < tags.length; i++) {
		var tagName = tags[i].replace("\u001D", ",");
		this.tagLocal(tagName, true);
	}
};

/**
 * Takes a string of flag chars and applies them to this item.
 * 
 * @private
 */
ZmItem.prototype._parseFlags =
function(str) {
	this.flags = str;
	for (var i = 0; i < ZmItem.ALL_FLAGS.length; i++) {
		var flag = ZmItem.ALL_FLAGS[i];
		var on = (str && (str.indexOf(flag) != -1)) ? true : false;
		this.flagLocal(flag, on);
	}
};

// Listener notification

/**
 * Notify the list as well as this item.
 * 
 * @private
 */
ZmItem.prototype._notify =
function(event, details) {
	this._doNotify(event, details);
};

ZmItem.prototype._setupNotify =
function() {
    this._doNotify();
}

ZmItem.prototype._doNotify =
function(event, details) {
	if (this._evt) {
		this._evt.item = this;
		if (event != null) {
			ZmModel.prototype._notify.call(this, event, details);
		}
	} else {
		var idText = "";
		if (this.type && this.id) {
			idText = ": item = " + this.type + "(" + this.id + ")";
		}
		DBG.println(AjxDebug.DBG1, "ZmItem._doNotify, missing _evt" + idText);
	}
    if (this.list) {
        this.list._evt.item = this;
        this.list._evt.items = [this];
        if (event != null) {
            if (details) {
                details.items = [this];
            } else {
                details = {items: [this]};
            }
            this.list._notify(event, details);
        }
    }
};

/**
 * Returns a list of flags that apply to this type of item.
 * 
 * @private
 */
ZmItem.prototype._getFlags =
function() {
	return [ZmItem.FLAG_FLAGGED, ZmItem.FLAG_ATTACH];
};

/**
 * Rename the item.
 *
 * @param	{String}	newName
 * @param	{AjxCallback}	callback		the callback
 * @param	{AjxCallback}	errorCallback	the callback on error
 * @return	{Object}		the result of the move
 */
ZmItem.prototype.rename =
function(newName, callback, errorCallback) {
	return ZmItem.rename(this.id, newName, callback, errorCallback);
};

/**
 * Rename the item.
 *
 * @return	{Object}		the result of the move
 */
ZmItem.rename =
function(itemId, newName, callback, errorCallback, accountName) {
    var json = {
		ItemActionRequest: {
			_jsns: "urn:zimbraMail",
			action: {
				id:	itemId instanceof Array ? itemId[0] : itemId,
				op:	"rename",
				name:	newName
			}
		}
	};	

	var params = {
		jsonObj:		json,
		asyncMode:		Boolean(callback),
		callback:		callback,
		errorCallback:	errorCallback,
		accountName:	accountName
	};
	return appCtxt.getAppController().sendRequest(params);
};

ZmItem.prototype.getSortedTags =
function() {
	var numTags = this.tags && this.tags.length;
	if (numTags) {
		var tagList = appCtxt.getAccountTagList(this);
		var ta = [];
		for (var i = 0; i < numTags; i++) {
			var tag = tagList.getByNameOrRemote(this.tags[i]);
			//tag could be missing if this was called when deleting a whole tag (not just untagging one message). So this makes sure we don't have a null item.
			if (!tag) {
				continue;
			}
			ta.push(tag);
		}
		ta.sort(ZmTag.sortCompare);
		return ta;
	}
	return null;
};

}
if (AjxPackage.define("zimbraMail.share.model.ZmOrganizer")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines an organizer.
 */

/**
* Creates an empty organizer.
* @class
* This class represents an "organizer", which is something used to classify or contain
* items. So far, that's either a tag or a folder. Tags and folders are represented as
* a tree structure, though tags are flat and have only one level below the root. Folders
* can be nested.
*
* @author Conrad Damon
*
* @param	{Hash}	params		a hash of parameters
* @param {constant}	params.type		the organizer type
* @param {int}		params.id			the numeric ID
* @param {String}	params.name		the name
* @param {ZmOrganizer}	params.parent		the parent organizer
* @param {ZmTree}	params.tree		the tree model that contains this organizer
* @param {constant}	params.color		the color for this organizer
* @param {String}	params.rgb		the color for this organizer, as HTML RGB value
* @param {Boolean}	params.link		<code>true</code> if this organizer is shared
* @param {Boolean}	params.broken		<code>true</code> if this link is broken
* @param {int}	params.numUnread	the number of unread items for this organizer
* @param {int}	params.numTotal	the number of items for this organizer
* @param {Boolean}	params.noTooltip	do not show tooltip 
* @param {int}	params.sizeTotal	the total size of organizer's items
* @param {String}	params.url		the URL for this organizer's feed
* @param {String}	params.owner		the owner for this organizer
* @param {String}	params.oname		the owner's name for this organizer
* @param {String}	params.zid		the Zimbra ID of owner, if remote folder
* @param {String}	params.rid		the remote ID of organizer, if remote folder
* @param {String}	params.restUrl	the REST URL of this organizer.
* @param {String}	params.newOp		the name of operation run by button in overview header
* @param {ZmZimbraAccount}	params.account	the account this organizer belongs to
*/
ZmOrganizer = function(params) {

	if (arguments.length == 0) { return; }

	this.type = params.type;
	var id = this.id = params.id;
	this.nId = ZmOrganizer.normalizeId(id);
	this.name = ZmFolder.MSG_KEY[this.nId] ? ZmMsg[ZmFolder.MSG_KEY[this.nId]] : params.name;
	this.mail = params.mail;
	this._systemName = this.nId < 256 && params.name;
	this.parent = params.parent;
	this.tree = params.tree;
	this.numUnread = params.numUnread || 0;
	this.numTotal = params.numTotal || 0;
	this.noTooltip = params.noTooltip;
	this.sizeTotal = params.sizeTotal || 0;
	this.url = params.url;
	this.owner = params.owner;
	this.oname = params.oname;
	this.link = params.link || (Boolean(params.zid)) || (this.parent && this.parent.link);
	this.isMountpoint = params.link;
	this.zid = params.zid;
	this.rid = params.rid;
	this.restUrl = params.restUrl;
	this.account = params.account;
    this.perm = params.perm;
	this.noSuchFolder = params.broken; // Is this a link to some folder that ain't there.
	this._isAdmin = this._isReadOnly = this._hasPrivateAccess = null;
    this.retentionPolicy = params.retentionPolicy;
	this.webOfflineSyncDays = params.webOfflineSyncDays;

	this.color =
        params.color ||
        (this.parent && this.parent.color) ||
        ZmOrganizer.ORG_COLOR[id] ||
        ZmOrganizer.ORG_COLOR[this.nId] ||
        ZmOrganizer.DEFAULT_COLOR[this.type] ||
        ZmOrganizer.C_NONE
    ;
	this.isColorCustom = params.rgb != null; //set so we know if the user chose a custom color (to distiguish from basic color or none
	this.rgb =
        params.rgb ||
        ZmOrganizer.COLOR_VALUES[this.color] ||
        ZmOrganizer.COLOR_VALUES[ZmOrganizer.ORG_DEFAULT_COLOR]
    ;

	if (appCtxt.isOffline && !this.account && this.id == this.nId) {
		this.account = appCtxt.accountList.mainAccount;
	}

	if (id && params.tree) {
		appCtxt.cacheSet(id, this);
		if (this.link) {
			// also store under ID that items use for parent folder ("l" attribute in node)
			appCtxt.cacheSet([this.zid, this.rid].join(":"), this);
		}
	}

	this.children = new AjxVector();
};

ZmOrganizer.prototype.isZmOrganizer = true;
ZmOrganizer.prototype.toString = function() { return "ZmOrganizer"; };

// global organizer types
ZmOrganizer.TAG					= ZmEvent.S_TAG;
ZmOrganizer.SEARCH				= ZmEvent.S_SEARCH;
ZmOrganizer.SHARE               = ZmEvent.S_SHARE;
ZmOrganizer.MOUNTPOINT			= ZmEvent.S_MOUNTPOINT;
ZmOrganizer.ZIMLET				= ZmEvent.S_ZIMLET;
ZmOrganizer.HAB					= ZmEvent.S_HAB;
ZmOrganizer.SHARING				= ZmEvent.S_SHARING;

// folder IDs defined in com.zimbra.cs.mailbox.Mailbox
// Note: since these are defined as Numbers, and IDs come into our system as Strings,
// we need to use == for comparisons (instead of ===, which will fail)
ZmOrganizer.ID_ROOT				= 1;
ZmOrganizer.ID_INBOX			= 2;
ZmOrganizer.ID_TRASH			= 3;
ZmOrganizer.ID_SPAM				= 4;
ZmOrganizer.ID_ADDRBOOK			= 7;
ZmOrganizer.ID_CALENDAR			= 10;
ZmOrganizer.ID_AUTO_ADDED 		= 13;
ZmOrganizer.ID_CHATS			= 14;
ZmOrganizer.ID_TASKS			= 15;
ZmOrganizer.ID_BRIEFCASE		= 16;
ZmOrganizer.ID_FILE_SHARED_WITH_ME	= 20;
ZmOrganizer.ID_ALL_MAILBOXES	= 249; 
ZmOrganizer.ID_NOTIFICATION_MP	= 250;
ZmOrganizer.ID_SYNC_FAILURES	= 252;		// offline only
ZmOrganizer.ID_OUTBOX    		= 254;		// offline only
ZmOrganizer.ID_ZIMLET			= -1000;	// zimlets need a range.  start from -1000 incrementing up.
ZmOrganizer.ID_ATTACHMENTS		= -17;		// Attachments View
ZmOrganizer.ID_DLS				= -18;

// fields that can be part of a displayed organizer
ZmOrganizer.F_NAME				= "name";
ZmOrganizer.F_UNREAD			= "unread";
ZmOrganizer.F_TOTAL				= "total";
ZmOrganizer.F_SIZE				= "size";
ZmOrganizer.F_COLOR				= "color";
ZmOrganizer.F_RGB				= "rgb";
ZmOrganizer.F_QUERY				= "query";
ZmOrganizer.F_SHARES			= "shares";
ZmOrganizer.F_FLAGS				= "flags";
ZmOrganizer.F_REST_URL			= "rest";
ZmOrganizer.F_PERMS				= "perms";
ZmOrganizer.F_RNAME				= "rname";	// remote name

// server representation of org flags
ZmOrganizer.FLAG_CHECKED			= "#";
ZmOrganizer.FLAG_DISALLOW_SUBFOLDER	= "o";
ZmOrganizer.FLAG_EXCLUDE_FREE_BUSY	= "b";
ZmOrganizer.FLAG_IMAP_SUBSCRIBED	= "*";
ZmOrganizer.FLAG_OFFLINE_GLOBAL		= "g";
ZmOrganizer.FLAG_OFFLINE_SYNCABLE	= "y";
ZmOrganizer.FLAG_OFFLINE_SYNCING	= "~";
ZmOrganizer.ALL_FLAGS = [
	ZmOrganizer.FLAG_CHECKED,
	ZmOrganizer.FLAG_IMAP_SUBSCRIBED,
	ZmOrganizer.FLAG_EXCLUDE_FREE_BUSY,
	ZmOrganizer.FLAG_DISALLOW_SUBFOLDER,
	ZmOrganizer.FLAG_OFFLINE_GLOBAL,
	ZmOrganizer.FLAG_OFFLINE_SYNCABLE,
	ZmOrganizer.FLAG_OFFLINE_SYNCING
];

// org property for each flag
ZmOrganizer.FLAG_PROP = {};
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_CHECKED]				= "isChecked";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_IMAP_SUBSCRIBED]		= "imapSubscribed";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_EXCLUDE_FREE_BUSY]	= "excludeFreeBusy";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_DISALLOW_SUBFOLDER]	= "disallowSubFolder";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_OFFLINE_GLOBAL]		= "isOfflineGlobalSearch";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_OFFLINE_SYNCABLE]	= "isOfflineSyncable";
ZmOrganizer.FLAG_PROP[ZmOrganizer.FLAG_OFFLINE_SYNCING]		= "isOfflineSyncing";

// Following chars invalid in organizer names: " : / [anything less than " "]
ZmOrganizer.VALID_NAME_CHARS = "[^\\x00-\\x1F\\x7F:\\/\\\"]";
ZmOrganizer.VALID_PATH_CHARS = "[^\\x00-\\x1F\\x7F:\\\"]"; // forward slash is OK in path
ZmOrganizer.VALID_NAME_RE = new RegExp('^' + ZmOrganizer.VALID_NAME_CHARS + '+$');

ZmOrganizer.MAX_NAME_LENGTH			= 128;	// max allowed by server
ZmOrganizer.MAX_DISPLAY_NAME_LENGTH	= 30;	// max we will show

// color constants (server stores a number)
ZmOrganizer.C_NONE				= 0;
ZmOrganizer.C_BLUE				= 1;
ZmOrganizer.C_CYAN				= 2;
ZmOrganizer.C_GREEN				= 3;
ZmOrganizer.C_PURPLE			= 4;
ZmOrganizer.C_RED				= 5;
ZmOrganizer.C_YELLOW			= 6;
ZmOrganizer.C_PINK				= 7;
ZmOrganizer.C_GRAY				= 8;
ZmOrganizer.C_ORANGE			= 9;
ZmOrganizer.MAX_COLOR			= ZmOrganizer.C_ORANGE;
ZmOrganizer.ORG_DEFAULT_COLOR 	= ZmOrganizer.C_GRAY;

ZmOrganizer.COLOR_VALUES = [
	null,
	ZmMsg.colorBlue,
	ZmMsg.colorCyan,
	ZmMsg.colorGreen,
	ZmMsg.colorPurple,
	ZmMsg.colorRed,
	ZmMsg.colorYellow,
	ZmMsg.colorPink,
	ZmMsg.colorGray,
	ZmMsg.colorOrange
];

// color names
ZmOrganizer.COLOR_TEXT = {};
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_NONE]		= ZmMsg.none;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_ORANGE]	= ZmMsg.orange;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_BLUE]		= ZmMsg.blue;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_CYAN]		= ZmMsg.cyan;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_GREEN]		= ZmMsg.green;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_PURPLE]	= ZmMsg.purple;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_RED]		= ZmMsg.red;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_YELLOW]	= ZmMsg.yellow;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_PINK]		= ZmMsg.pink;
ZmOrganizer.COLOR_TEXT[ZmOrganizer.C_GRAY]		= ZmMsg.gray;

// list of colors and text for populating a color select menu
ZmOrganizer.COLORS = [];
ZmOrganizer.COLOR_CHOICES = [];
(function() {
	for (var i = 0; i <= ZmOrganizer.MAX_COLOR; i++) {
		var color = ZmOrganizer.COLOR_TEXT[i];
		ZmOrganizer.COLORS.push(color);
		ZmOrganizer.COLOR_CHOICES.push( { value:i, label:color } );
	}
})();


ZmOrganizer.MSG_KEY 		= {};		// keys for org names
ZmOrganizer.ROOT_MSG_KEY	= {};		// key for name of root (used as tree header)
ZmOrganizer.ITEM_ORGANIZER 	= {};		// primary organizer for item types
ZmOrganizer.DEFAULT_FOLDER 	= {};		// default folder for org type
ZmOrganizer.SOAP_CMD 		= {};		// SOAP command for modifying an org
ZmOrganizer.FIRST_USER_ID 	= {};		// lowest valid user ID for an org type
ZmOrganizer.PRECONDITION 	= {};		// setting that this org type depends on
ZmOrganizer.HAS_COLOR 		= {};		// whether an org uses colors
ZmOrganizer.DEFAULT_COLOR 	= {};		// default color for each org type
ZmOrganizer.ORG_COLOR 		= {};		// color overrides by ID
ZmOrganizer.APP 			= {};		// App responsible for organizer
ZmOrganizer.ORG_CLASS 		= {};		// constructor for organizer
ZmOrganizer.ORG_PACKAGE 	= {};		// package required to construct organizer
ZmOrganizer.CREATE_FUNC 	= {};		// function that creates this organizer
ZmOrganizer.LABEL 			= {};		// msg key for text for tree view header item
ZmOrganizer.ITEMS_KEY 		= {};		// msg key for text describing contents
ZmOrganizer.TREE_TYPE 		= {};		// type of server data tree that contains this type of organizer
ZmOrganizer.VIEWS 			= {};		// views by org type
ZmOrganizer.VIEW_HASH		= {};		// view hash by org type
ZmOrganizer.TYPE 			= {};		// types by view (reverse map of above)
ZmOrganizer.FOLDER_KEY 		= {};		// keys for label "[org] folder"
ZmOrganizer.MOUNT_KEY 		= {};		// keys for label "mount [org]"
ZmOrganizer.DEFERRABLE 		= {};		// creation can be deferred to app launch
ZmOrganizer.PATH_IN_NAME	= {};		// if true, provide full path when asked for name
ZmOrganizer.OPEN_SETTING	= {};		// setting that controls whether the tree view is open
ZmOrganizer.NEW_OP			= {};		// name of operation for new button in tree header (optional)
ZmOrganizer.DISPLAY_ORDER	= {};		// sort number to determine order of tree view (optional)
ZmOrganizer.HIDE_EMPTY		= {};		// if true, hide tree header if tree is empty
ZmOrganizer.SHAREABLE 		= {};		// allow share or not

ZmOrganizer.APP2ORGANIZER	= {};		// organizer types, keyed by app name
ZmOrganizer.APP2ORGANIZER_R = {};		// app names, keyed by app organizer type

// allowed permission bits
ZmOrganizer.PERM_READ		= "r";
ZmOrganizer.PERM_WRITE		= "w";
ZmOrganizer.PERM_INSERT		= "i";
ZmOrganizer.PERM_DELETE		= "d";
ZmOrganizer.PERM_ADMIN		= "a";
ZmOrganizer.PERM_WORKFLOW	= "x";
ZmOrganizer.PERM_PRIVATE	= "p";

// Retention Policy Elements - Keep or Purge
ZmOrganizer.RETENTION_KEEP  = "keep";
ZmOrganizer.RETENTION_PURGE = "purge";

// Abstract methods

/**
 * Stores information about the given organizer type.
 * 
 * @param {constant}	org				the organizer type
 * @param {Hash}	params			a hash of parameters
 * @param	{constant}	app				the app that handles this org type
 * @param	{String}	nameKey			the msg key for org name
 * @param	{constant}	precondition		the setting that this org type depends on
 * @param	{int}	defaultFolder		the folder ID of default folder for this org
 * @param	{String}	soapCmd			the SOAP command for acting on this org
 * @param	{int}	firstUserId		the minimum ID for a user instance of this org
 * @param	{String}	orgClass			the name of constructor for this org
 * @param	{String}	orgPackage		the name of smallest package with org class
 * @param	{String}	treeController	the name of associated tree controller
 * @param	{String}	labelKey			the msg key for label in overview
 * @param	{String}	itemsKey			the msg key for text describing contents
 * @param	{Boolean}	hasColor			<code>true</code> if org has color associated with it
 * @param	{constant}	defaultColor		the default color for org in overview
 * @param	{Array}	orgColor			the color override by ID (in pairs)
 * @param	{constant}	treeType			the type of data tree (from server) that contains this org
 * @param	{String}	views				the associated folder views (JSON)
 * @param	{String}	folderKey			the msg key for folder props dialog
 * @param	{String}	mountKey			the msg key for folder mount dialog
 * @param	{String}	createFunc		the name of function for creating this org
 * @param	{String}	compareFunc		the name of function for comparing instances of this org
 * @param	{Boolean}	deferrable		if <code>true</code>, creation can be deferred to app launch
 * @param	{Boolean}	pathInName		if <code>true</code>, provide full path when asked for name
 * @param	{constant}	openSetting		the setting that controls whether the tree view is open
 * @param	{int}	displayOrder		the number that is used when sorting the display of trees. (Lower number means higher display.)
 * @param	{Boolean}	hideEmpty			if <code>true</code>, hide tree header if tree is empty
 */
ZmOrganizer.registerOrg =
function(org, params) {
	if (params.nameKey)			{ ZmOrganizer.MSG_KEY[org]				= params.nameKey; }
	if (params.app)				{
		ZmOrganizer.APP[org] = params.app;
		if (!ZmOrganizer.APP2ORGANIZER[params.app]) {
			ZmOrganizer.APP2ORGANIZER[params.app] = [];
		}
		ZmOrganizer.APP2ORGANIZER[params.app].push(org);
        ZmOrganizer.APP2ORGANIZER_R[org] = params.app;
	}
	if (params.defaultFolder)	{ ZmOrganizer.DEFAULT_FOLDER[org]		= params.defaultFolder; }
	if (params.precondition)	{ ZmOrganizer.PRECONDITION[org]			= params.precondition; }
	if (params.soapCmd)			{ ZmOrganizer.SOAP_CMD[org]				= params.soapCmd; }
	if (params.firstUserId)		{ ZmOrganizer.FIRST_USER_ID[org]		= params.firstUserId; }
	if (params.orgClass)		{ ZmOrganizer.ORG_CLASS[org]			= params.orgClass; }
	if (params.orgPackage)		{ ZmOrganizer.ORG_PACKAGE[org]			= params.orgPackage; }
	if (params.labelKey)		{ ZmOrganizer.LABEL[org]				= params.labelKey; }
	if (params.itemsKey)		{ ZmOrganizer.ITEMS_KEY[org]			= params.itemsKey; }
	if (params.hasColor)		{ ZmOrganizer.HAS_COLOR[org]			= params.hasColor; }
	if (params.views)			{ ZmOrganizer.VIEWS[org]				= params.views; }
	if (params.folderKey)		{ ZmOrganizer.FOLDER_KEY[org]			= params.folderKey; }
	if (params.mountKey)		{ ZmOrganizer.MOUNT_KEY[org]			= params.mountKey; }
	if (params.deferrable)		{ ZmOrganizer.DEFERRABLE[org]			= params.deferrable; }
	if (params.pathInName)		{ ZmOrganizer.PATH_IN_NAME[org]			= params.pathInName; }
	if (params.openSetting)		{ ZmOrganizer.OPEN_SETTING[org]			= params.openSetting; }
	if (params.newOp)			{ ZmOrganizer.NEW_OP[org]				= params.newOp; }
	if (params.displayOrder)	{ ZmOrganizer.DISPLAY_ORDER[org]		= params.displayOrder; }
	if (params.hideEmpty)		{ ZmOrganizer.HIDE_EMPTY[org]			= params.hideEmpty; }
	ZmOrganizer.SHAREABLE[org]	= !params.disableShare; 

	if (!appCtxt.isChildWindow || params.childWindow ) {
		if (params.compareFunc)		{ ZmTreeView.COMPARE_FUNC[org]			= params.compareFunc; }
		if (params.treeController)	{ ZmOverviewController.CONTROLLER[org]	= params.treeController; }
	}

	ZmOrganizer.TREE_TYPE[org] = params.treeType || org; // default to own type
	ZmOrganizer.CREATE_FUNC[org] = params.createFunc || "ZmOrganizer.create";

	if (params.views) {
		ZmOrganizer.VIEW_HASH[org] = AjxUtil.arrayAsHash(ZmOrganizer.VIEWS[org]);
	}

	if (params.hasColor) {
		ZmOrganizer.DEFAULT_COLOR[org] = (params.defaultColor != null)
			? params.defaultColor
			: ZmOrganizer.ORG_DEFAULT_COLOR;
	}

	if (params.orgColor) {
		for (var id in params.orgColor) {
			ZmOrganizer.ORG_COLOR[id] = params.orgColor[id];
		}
	}

	if (params.dropTargets) {
		if (!ZmApp.DROP_TARGETS[params.app]) {
			ZmApp.DROP_TARGETS[params.app] = {};
		}
		ZmApp.DROP_TARGETS[params.app][org] = params.dropTargets;
	}
};

ZmOrganizer.sortCompare = function(organizerA, organizerB) {};

/**
 * nulls value that is the default color for the type.
 * @param value
 */
ZmOrganizer.getColorValue =
function(value, type) {
	// no need to save color if missing or default
	if (value == ZmOrganizer.DEFAULT_COLOR[type]) {
		return null;
	}

	return value;
};

/**
 * Creates an organizer via <code>&lt;CreateFolderRequest&gt;</code>. Attribute pairs can
 * be passed in and will become attributes of the folder node in the request.
 * 
 * @param {Hash}	params	a hash of parameters
 */
ZmOrganizer.create =
function(params) {
	var jsonObj = {CreateFolderRequest:{_jsns:"urn:zimbraMail"}};
	var folder = jsonObj.CreateFolderRequest.folder = {};
	var errorCallback = params.errorCallback || new AjxCallback(null, ZmOrganizer._handleErrorCreate, params);
	var type = params.type;

	// set attributes
	params.view = params.view || ZmOrganizer.VIEWS[type] ? ZmOrganizer.VIEWS[type][0] : null;
	for (var i in params) {
		if (i == "type" || i == "errorCallback" || i == "account") { continue; }

		var value = params[i];
		if (value) {
			folder[i] = value;
		}
	}
	//adding support to asyncMode == false didn't eventually help me, but why not keep it?
	var asyncMode = params.asyncMode === undefined ? true : params.asyncMode; //default is true

	return appCtxt.getAppController().sendRequest({
		jsonObj: jsonObj,
		asyncMode: asyncMode,
		accountName: (params.account && params.account.name),
		callback: params.callback,
		callbackAfterNotifications: params.callbackAfterNotifications, 
		errorCallback: errorCallback
	});
};

/**
 * @private
 */
ZmOrganizer._handleErrorCreate =
function(params, ex) {
	if (!params.url && !params.name) { return false; }
	
	var msg;
	if (params.name && (ex.code == ZmCsfeException.MAIL_ALREADY_EXISTS)) {
		var type = appCtxt.getFolderTree(appCtxt.getActiveAccount()).getFolderTypeByName(params.name);
        msg = AjxMessageFormat.format(ZmMsg.errorAlreadyExists, [params.name, type.toLowerCase()]);
	} else if (params.url) {
		var errorMsg = (ex.code == ZmCsfeException.SVC_RESOURCE_UNREACHABLE) ? ZmMsg.feedUnreachable : ZmMsg.feedInvalid;
		msg = AjxMessageFormat.format(errorMsg, params.url);
	}

	if (msg) {
		ZmOrganizer._showErrorMsg(msg);
		return true;
	}

	return false;
};

/**
 * @private
 */
ZmOrganizer._showErrorMsg =
function(msg) {
	var msgDialog = appCtxt.getMsgDialog();
	msgDialog.reset();
	msgDialog.setMessage(AjxStringUtil.htmlEncode(msg), DwtMessageDialog.CRITICAL_STYLE);
	msgDialog.popup();
};

/**
 * Gets the folder.
 * 
 * @param	{String}	id		the folder id
 * @param	{AjxCallback}	callback	the callback
 * @param	{ZmBatchCommand}	batchCmd	the batch command or <code>null</code> for none
 */
ZmOrganizer.getFolder =
function(id, callback, batchCmd) {
	var jsonObj = {GetFolderRequest:{_jsns:"urn:zimbraMail"}};
	var request = jsonObj.GetFolderRequest;
	request.folder = {l:id};
	var respCallback = new AjxCallback(null, ZmOrganizer._handleResponseGetFolder, [callback]);
	if (batchCmd) {
		batchCmd.addRequestParams(jsonObj, respCallback);
	} else {
		appCtxt.getRequestMgr().sendRequest({jsonObj:jsonObj, asyncMode:true, callback:respCallback});
	}
};

/**
 * @private
 */
ZmOrganizer._handleResponseGetFolder =
function(callback, result) {
	var resp = result.getResponse().GetFolderResponse;
	var folderObj = (resp && resp.folder && resp.folder[0]) ||
					(resp && resp.link && resp.link[0]);
	var folder;
	if (folderObj) {
		folder = appCtxt.getById(folderObj.id);
		if (folder) {
			folder.clearShares();
			folder._setSharesFromJs(folderObj);
		} else {
			var parent = appCtxt.getById(folderObj.l);
			folder = ZmFolderTree.createFromJs(parent, folderObj, appCtxt.getFolderTree(), "folder");
		}
	}
	if (callback) {
		callback.run(folder);
	}
};

/**
 * Gets the folder.
 * 
 * @param	{AjxCallback}	callback	the callback
 * @param	{ZmBatchCommand}	batchCmd	the batch command or <code>null</code> for none
 */
ZmOrganizer.prototype.getFolder =
function(callback, batchCmd) {
	ZmOrganizer.getFolder(this.id, callback, batchCmd);
};


// Static methods

/**
 * Gets the view name by organizer type.
 * 
 * @param	{String}	organizerType		the organizer type
 * @return	{String}	the view
 */
ZmOrganizer.getViewName =
function(organizerType) {
	if (organizerType === ZmItem.BRIEFCASE_ITEM) {
		return ZmItem.BRIEFCASE_ITEM;
	}
	return ZmOrganizer.VIEWS[organizerType][0];
};

/**
 * Checks an organizer (folder or tag) offlineSyncInterval for validity.
 *
 * @param {String}	value		offlineSyncInterval
 * @return	{String}	<code>null</code> if the offlineSyncInterval is valid or an error message if the name is invalid
 */
ZmOrganizer.checkWebOfflineSyncDays =
function(value) {
    if (isNaN(value)) {	return ZmMsg.invalidFolderSyncInterval; }
    var interval = parseInt(value);
	if (interval < 0 ||  interval > 30) {
		return ZmMsg.invalidFolderSyncInterval;
	}
	return null;
};

/**
 * Checks an organizer (folder or tag) name for validity.
 *
 * @param {String}	name		an organizer name
 * @return	{String}	<code>null</code> if the name is valid or an error message if the name is invalid
 */
ZmOrganizer.checkName =
function(name) {
	if (name.length == 0) {	return ZmMsg.nameEmpty; }

	if (name.length > ZmOrganizer.MAX_NAME_LENGTH) {
		return AjxMessageFormat.format(ZmMsg.nameTooLong, ZmOrganizer.MAX_NAME_LENGTH);
	}

	if (!ZmOrganizer.VALID_NAME_RE.test(name)) {
		return AjxMessageFormat.format(ZmMsg.errorInvalidName, AjxStringUtil.htmlEncode(name));
	}

	return null;
};

/**
 * Checks a URL (a folder or calendar feed, for example) for validity.
 *
 * @param {String}	url	a URL
 * @return	{String}	<code>null</code> if valid or an error message
 */
ZmOrganizer.checkUrl =
function(url) {
	// TODO: be friendly and prepend "http://" when it's missing
	if (!url.match(/^[a-zA-Z]+:\/\/.*$/i)) {
		return ZmMsg.errorUrlMissing;
	}

	return null;
};

/**
 * @private
 */
ZmOrganizer.checkSortArgs =
function(orgA, orgB) {
	if (!orgA && !orgB) return 0;
	if (orgA && !orgB) return 1;
	if (!orgA && orgB) return -1;
	return null;
};

/**
 * @private
 */
ZmOrganizer.checkColor =
function(color) {
	return ((color != null) && (color >= 0 && color <= ZmOrganizer.MAX_COLOR))
		? color : ZmOrganizer.ORG_DEFAULT_COLOR;
};

/**
 * Gets the system ID for the given system ID and account. Unless this
 * is a child account, the system ID is returned unchanged. For child
 * accounts, the ID consists of the account ID and the local ID.
 * 
 * @param {int}	id		the ID of a system organizer
 * @param {ZmZimbraAccount}	account	the account
 * @param {Boolean}		force		<code>true</code> to generate the fully qualified ID even if this is the main account
 * @return	{String}	the ID
 */
ZmOrganizer.getSystemId =
function(id, account, force) {
	account = account || appCtxt.getActiveAccount();
	if ((account && !account.isMain) || force) {
		return ((typeof(id) == "string") && (id.indexOf(":") != -1) || !id)
			? id : ([account.id, id].join(":"));
	}
	return id;
};

/**
 * Normalizes the id by stripping the account ID portion from a system ID for a child account, which
 * can then be used to check against known system IDs. Any non-system ID is
 * returned unchanged (if type is provided).
 *
 * @param {String}	id	ID of an organizer
 * @param {constant}	type	the type of organizer
 * @return	{String}	the resulting id
 */
ZmOrganizer.normalizeId =
function(id, type) {
	if (typeof(id) != "string") { return id; }
	var idx = id.indexOf(":");
	var localId = (idx == -1) ? id : id.substr(idx + 1);
	return (type && (localId >= ZmOrganizer.FIRST_USER_ID[type])) ? id : localId;
};

/**
 * Parses an id into an object with fields for account and normalized id
 *
 * @param {String}	id		the ID of an organizer
 * @param {Object}	result	an optional object in which the result is stored
 * @return	{Object}	the resulting ID
 */
ZmOrganizer.parseId =
function(id, result) {
	var ac = window.parentAppCtxt || window.appCtxt;

	result = result || {};
	if (id == null) { return result; }
	var idx = (typeof id == "string") ? id.indexOf(":") : -1;
	if (idx == -1) {
		result.account = ac.accountList.mainAccount;
		result.id = id;
	} else {
		result.acctId = id.substring(0, idx);
		result.account = ac.accountList.getAccount(result.acctId);
		result.id = id.substr(idx + 1);
	}
	return result;
};

// Public methods

/**
* Gets the name of this organizer.
*
* @param {Boolean}	showUnread		<code>true</code> to display the number of unread items (in parens)
* @param {int}	maxLength		the length (in chars) to truncate the name to
* @param {Boolean}	noMarkup		if <code>true</code>, don't return any HTML
* @param {Boolean}	useSystemName	if <code>true</code>, don't use translated version of name
* @return	{String}	the name
*/
ZmOrganizer.prototype.getName = 
function(showUnread, maxLength, noMarkup, useSystemName, useOwnerName, defaultRootType) {
	if (this.nId == ZmFolder.ID_ROOT) {
		var type = defaultRootType || this.type;
		return (ZmOrganizer.LABEL[type])
			? ZmMsg[ZmOrganizer.LABEL[type]] : "";
	}
	var name = (useSystemName && this._systemName) || (useOwnerName && this.oname) || this.name || "";
	if (ZmOrganizer.PATH_IN_NAME[this.type] && this.path) {
		name = [this.path, name].join("/");
	}
	name = (maxLength && name.length > maxLength)
		? name.substring(0, maxLength - 3) + "..." : name;
	return this._markupName(name, showUnread, noMarkup);
};

/**
* Gets the full path as a string.
*
* @param {Boolean}	includeRoot		<code>true</code> to include root name at the beginning of the path
* @param {Boolean}	showUnread		<code>true</code> to display the number of unread items (in parens)
* @param {int}	maxLength		the length (in chars) to truncate the name to
* @param {Boolean}	noMarkup		if <code>true</code>, do not return any HTML
* @param {Boolean}	useSystemName	if <code>true</code>, use untranslated version of system folder names
* @return	{String}	the path
*/
ZmOrganizer.prototype.getPath = 
function(includeRoot, showUnread, maxLength, noMarkup, useSystemName, useOwnerName) {
	var parent = this.parent;
	var path = this.getName(showUnread, maxLength, noMarkup, useSystemName, useOwnerName);
	while (parent && ((parent.nId != ZmOrganizer.ID_ROOT) || includeRoot)) {
		path = parent.getName(showUnread, maxLength, noMarkup, useSystemName, useOwnerName) + ZmFolder.SEP + path;
		parent = parent.parent;
	}

	return path;
};

/**
 * Gets the tooltip. The tooltip shows number of unread items, total messages and the total size.
 *
 * @param {Boolean}	force		if <code>true</code>, don't use cached tooltip
 * @return	{String}	the tooltip
 */
ZmOrganizer.prototype.getToolTip =
function(force) {
	if (this.noTooltip) {
		return null;
	}
    if (!this._tooltip || force) {
		var itemText = this._getItemsText();
		var unreadLabel = this._getUnreadLabel();
		var subs = {name:this.name, itemText:itemText, numTotal:this.numTotal, sizeTotal:this.sizeTotal, numUnread:this.numUnread, unreadLabel:unreadLabel};
		this._tooltip = AjxTemplate.expand("share.App#FolderTooltip", subs);
	}
	return this._tooltip;
};

/**
 * Gets the full path, suitable for use in search expressions.
 *
 * @return	{String}	the path
 */
ZmOrganizer.prototype.getSearchPath =
function(useOwnerName) {
	return (this.nId != ZmOrganizer.ID_ROOT)
		? this.getPath(null, null, null, true, true, useOwnerName) : "/";
};

/**
 * Gets the URL.
 * 
 * @return	{String}	the URL
 * 
 * @deprecated use {@link getRestUrl}
 */
ZmOrganizer.prototype.getUrl =
function() {
	return this.getRestUrl();
};

/**
 * Gets the sync URL.
 * 
 * @return		{String}	the URL
 */
ZmOrganizer.prototype.getSyncUrl =
function() {
	return url;
};

/**
 * Gets the remote ID.
 * 
 * @return	{String}	the ID
 */
ZmOrganizer.prototype.getRemoteId =
function() {
	if (!this._remoteId) {
		this._remoteId = (this.isRemote() && this.zid && this.rid)
			? (this.zid + ":" + this.rid)
			: this.id;
	}
	return this._remoteId;
};


/**
 * Gets the REST URL.
 * 
 * @return	{String}	the URL
 */
ZmOrganizer.prototype.getRestUrl =
function(noRemote) {

	var restUrl = appCtxt.get(ZmSetting.REST_URL);
	if (restUrl && (!this.isRemote() || noRemote || this.id == ZmFolder.ID_FILE_SHARED_WITH_ME)) { //for remote - this does not work. either use this.restUrl (if set, which is for shared folder, but not for sub-folders) or call _generateRestUrl which seems to work for subfodlers of shared as well.
		var path = AjxStringUtil.urlEncode(this.getSearchPath()).replace("#","%23").replace(";", "%3B"); // User may type in a # in a folder name, but that's not ok for our urls
		// return REST URL as seen by the GetInfoResponse
		return ([restUrl, "/", path].join(""));
	}

	// return REST URL as seen by server - this is the remote (isRemote() true) case - shared folder.
	if (this.restUrl) {
		return this.restUrl;
	}

	// if server doesn't tell us what URL to use, do our best to generate
	var url = this._generateRestUrl();
	DBG.println(AjxDebug.DBG3, "NO REST URL FROM SERVER. GENERATED URL: " + url);

	return url;
};

/**
 * Gets the OWNER'S REST URL,used to fetch resturl of shared folders.
 *
 * @return	{String}	the URL
 */
ZmOrganizer.prototype.getOwnerRestUrl =
function(){
	var node = this;
	var restUrl;
	var path = "";

	while (node && !node.restUrl) {
		path = node.name + "/" + path;
		node = node.parent;
	}
	restUrl = node.restUrl;
	path = node.oname + "/" + path;

	path = AjxStringUtil.urlEncode(path).replace("#", "%23").replace(new RegExp("/$"), "");

	// return REST URL as seen by the GetInfoResponse
	return ([restUrl, "/", path].join(""));
};

ZmOrganizer.prototype._generateRestUrl =
function() {
	var loc = document.location;
	var uname = appCtxt.get(ZmSetting.USERNAME);
	var host = loc.host;
	var m = uname.match(/^(.*)@(.*)$/);

	host = (m && m[2]) || host;

	// REVISIT: What about port? For now assume other host uses same port
	if (loc.port && loc.port != 80) {
		host = host + ":" + loc.port;
	}

	var path = AjxStringUtil.urlEncode(this.getSearchPath()).replace("#","%23"); // User may type in a # in a folder name, but that's not ok for our urls
		
	return [
		loc.protocol, "//", host, "/service/user/", uname, "/",	path
	].join("");
};

/**
 * Gets the account.
 * 
 * @return	{ZmZimbraAccount}	the account
 */
ZmOrganizer.prototype.getAccount =
function() {
	if (appCtxt.multiAccounts) {
		if (!this.account) {
			this.account = ZmOrganizer.parseId(this.id).account;
		}
		// bug 46364:
		// still no account?! Must be remote organizer, keep checking parent
		if (!this.account) {
			var parent = this.parent;
			while (parent && !this.account) {
				this.account = parent.getAccount();
				parent = parent.parent;
			}
		}
		return this.account;
	}

	return (this.account || appCtxt.accountList.mainAccount);
};

/**
 * Gets the shares.
 * 
 * @return	{Array}	an array of shares
 */
ZmOrganizer.prototype.getShares =
function() {
	return this.shares;
};

/**
 * Adds the share.
 * 
 * @param	{Object}	share		the share to add
 */
ZmOrganizer.prototype.addShare =
function(share) {
	this.shares = this.shares || [];
	this.shares.push(share);

	var curAcct = appCtxt.getActiveAccount();
	var curZid = curAcct && curAcct.id;
	var shareId = share.grantee && share.grantee.id;
	if (shareId && (shareId == curZid)) {
		this._mainShare = share;
	}
};

/**
 * Clears all shares.
 * 
 */
ZmOrganizer.prototype.clearShares =
function() {
	if (this.shares && this.shares.length) {
		for (var i = 0; i < this.shares.length; i++) {
			this.shares[i] = null;
		}
	}
	this.shares = null;
	this._mainShare = null;
};

/**
 * Gets the share granted to the current user.
 * 
 * @return	{String}	the main share
 */
ZmOrganizer.prototype.getMainShare =
function() {
	if (!this._mainShare) {
		var curAcct = appCtxt.getActiveAccount();
		var curZid = curAcct && curAcct.id;
		if (curZid && this.shares && this.shares.length) {
			for (var i = 0; i < this.shares.length; i++) {
				var share = this.shares[i];
				var id = share && share.grantee && share.grantee.id;
				if (id && id == curZid) {
					this._mainShare = share;
					break;
				}
			}
		}
	}
	return this._mainShare;
};

/**
 * Checks if the organizer supports sharing.
 * 
 * @return	{Boolean}	<code>true</code> if the organizer supports sharing
 */
ZmOrganizer.prototype.supportsSharing =
function() {
	// overload per organizer type
	return true;
};

/**
 * Checks if the organizer supports pulbic access.
 * 
 * @return	{Boolean}	<code>true</code> if the organizer supports public access
 */
ZmOrganizer.prototype.supportsPublicAccess =
function() {
	// overload per organizer type
	return true;
};

/**
 * Checks if the organizer supports private permission.
 * 
 * @return	{Boolean}	<code>true</code> if the organizer supports private permission
 */
ZmOrganizer.prototype.supportsPrivatePermission =
function() {
	// overload per organizer type
	return false;
};

/**
 * Gets the icon.
 * 
 * @return	{String}	the icon
 */
ZmOrganizer.prototype.getIcon = function() {};

/**
 * Gets the color of the organizer
 *
 * @return	{String}	the color
 */
ZmOrganizer.prototype.getColor =
function() {
    return this.rgb || ZmOrganizer.COLOR_VALUES[this.color];
}


/**
 * Gets the icon with color
 * 
 * @return	{String}	the icon
 */
ZmOrganizer.prototype.getIconWithColor =
function() {
	var icon = this.getIcon() || "";
	var color = this.getColor();
	return color ? [icon,color].join(",color=") : icon;
};

/**
 * Gets the icon alter text.
 * 
 * @return	{String}	the icon alter text
 */

ZmOrganizer.prototype.getIconAltText = function() {
	return this.name + " " + ZmMsg.icon;
};

// Actions

/**
 * Renames the organizer.
 * 
 * @param	{String}	name		the name
 * @param	{AjxCallback}	callback		the callback
 * @param	{AjxCallback}	errorCallback		the error callback
 * @param	{ZmBatchCommand}	batchCmd		the batch command
 */
ZmOrganizer.prototype.rename =
function(name, callback, errorCallback, batchCmd) {
	if (name == this.name) { return; }
	var params = {
		action: "rename",
		attrs: {name: name},
		callback: callback,
		errorCallback: errorCallback,
		batchCmd: batchCmd
	};
	this._organizerAction(params);
};

/**
 * Sets the web offline sync interval.
 *
 * @param	{String}	        interval		the web offline sync interval
 * @param	{AjxCallback}	    callback		the callback
 * @param	{AjxCallback}	    errorCallback   the error callback
 * @param   {ZmBatchCommand}    batchCmd        optional batch command
 */
ZmOrganizer.prototype.setOfflineSyncInterval =
function(interval, callback, errorCallback, batchCmd) {
	if (this.webOfflineSyncDays == interval) { return; }

	this._organizerAction({action: "webofflinesyncdays", attrs: {numDays: interval}, callback: callback,
                           errorCallback: errorCallback, batchCmd: batchCmd});
};

/**
 * Sets the color.
 * 
 * @param	{String}	        color		    the color
 * @param	{AjxCallback}	    callback		the callback
 * @param	{AjxCallback}	    errorCallback   the error callback
 * @param   {ZmBatchCommand}    batchCmd        optional batch command
 */
ZmOrganizer.prototype.setColor =
function(color, callback, errorCallback, batchCmd) {
	var color = ZmOrganizer.checkColor(color);
	if (!this.isColorChanged(color)) { return; }

	this._organizerAction({action: "color", attrs: {color: color}, callback: callback,
                           errorCallback: errorCallback, batchCmd: batchCmd});
};

/**
 * Sets the RGB color.
 * 
 * @param	{Object}	        rgb		        the rgb
 * @param	{AjxCallback}	    callback		the callback
 * @param	{AjxCallback}	    errorCallback	the error callback
 * @param   {ZmBatchCommand}    batchCmd        optional batch command
 */
ZmOrganizer.prototype.setRGB = function(rgb, callback, errorCallback, batchCmd) {
	if (!this.isColorChanged(rgb)) { return; }
	this._organizerAction({action: "color", attrs: {rgb: rgb}, callback: callback,
                           errorCallback: errorCallback, batchCmd: batchCmd});
};


ZmOrganizer.prototype.getRetentionPolicy =
function(policyElement) {
    var policy = null;
    if (this.retentionPolicy && this.retentionPolicy[0] && this.retentionPolicy[0][policyElement] &&
        this.retentionPolicy[0][policyElement][0]       && this.retentionPolicy[0][policyElement][0].policy &&
        this.retentionPolicy[0][policyElement][0].policy[0]) {
        policy = this.retentionPolicy[0][policyElement][0].policy[0];
    }
    return policy;
}

ZmOrganizer.prototype.getRetentionPolicyLifetimeMsec =
function(policy) {
    if (policy) {
        // Apply the keep (retention) period
        var lifetime = policy.lifetime;
        var amount = parseInt(lifetime);
        // Intervals taken from DateUtil.java.
        var interval = lifetime.slice(lifetime.length-1);
        var lifetimeMsec = 0;
        switch (interval) {
            case  "d": lifetimeMsec = amount * AjxDateUtil.MSEC_PER_DAY;    break;
            case  "h": lifetimeMsec = amount * AjxDateUtil.MSEC_PER_HOUR;   break;
            case  "m": lifetimeMsec = amount * AjxDateUtil.MSEC_PER_MINUTE; break;
            case  "s": lifetimeMsec = amount * 1000; break;
            case "ms": lifetimeMsec = amount;  break;
            default  : lifetimeMsec = amount * 1000; break;
        }
    }
    return lifetimeMsec;
}

/**
 * Sets the Retention Policy.
 *
 * @param	{Object}	        retentionPolicy     the new retention policy
 * @param	{AjxCallback}	    callback		    the callback
 * @param	{AjxCallback}	    errorCallback	    the error callback
 * @param   {ZmBatchCommand}    batchCmd            optional batch command
 */
ZmOrganizer.prototype.setRetentionPolicy = function(newRetentionPolicy, callback, errorCallback, batchCmd) {
    var keepPolicy  = this.getRetentionPolicy(ZmOrganizer.RETENTION_KEEP);
    var purgePolicy = this.getRetentionPolicy(ZmOrganizer.RETENTION_PURGE);
    if (!this.policiesDiffer(keepPolicy,  newRetentionPolicy.keep) &&
        !this.policiesDiffer(purgePolicy, newRetentionPolicy.purge)) {
        // No updated policy specified or no changes.
        return;
    }

	var cmd = ZmOrganizer.SOAP_CMD[this.type] + "Request";
	var request = {
		_jsns: "urn:zimbraMail",
		action : {
			op: "retentionpolicy",
			id: this.id,
			retentionPolicy: {
				keep: {},
				purge: {}
			}
		}
	};
	var jsonObj = {};
	jsonObj[cmd] = request;

	var retentionNode = request.action.retentionPolicy;

    if (newRetentionPolicy.keep) {
        this._addPolicy(retentionNode.keep, newRetentionPolicy.keep);
    }
    if (newRetentionPolicy.purge) {
        this._addPolicy(retentionNode.purge, newRetentionPolicy.purge);
    }

	if (batchCmd) {
        batchCmd.addRequestParams(jsonObj, callback, errorCallback);
 	}
	else {
		var accountName;
		if (appCtxt.multiAccounts) {
			accountName = (this.account)
				? this.account.name : appCtxt.accountList.mainAccount.name;
		}
		appCtxt.getAppController().sendRequest({
			jsonObj:       jsonObj,
			asyncMode:     true,
			accountName:   accountName,
			callback:      callback,
			errorCallback: errorCallback
		});
	}

};

ZmOrganizer.prototype.policiesDiffer =
function(policyA, policyB) {
    var differ = false;
    if ((policyA && !policyB) || (!policyA && policyB)) {
        differ = true;
    } else if (policyA) {
        // Old and new specified
        if (policyA.type != policyB.type) {
            differ = true;
        } else {
            if (policyA.type == "user") {
                differ = policyA.lifetime != policyB.lifetime;
            } else {
                // System policy
                differ = policyA.id != policyB.id;
            }
        }
    }
    return differ;
}

ZmOrganizer.prototype._addPolicy =
function(node, policy) {
	var policyNode = node.policy = {};
	for (var attr in policy) {
		if (AjxEnv.isIE) {
			policy[attr] += ""; //To string
		}

		policyNode[attr] = policy[attr];
	}
};

/**
 * Returns color number b/w 0-9 for a given color code
 *
 * @param	{String}	color	The color (usually in #43eded format
 * @return {int} Returns 0-9 for a standard color and returns -1 for custom color
 */
ZmOrganizer.getStandardColorNumber =
function(color) {
	if (String(color).match(/^#/)) {
		var len = ZmOrganizer.COLOR_VALUES.length;
		for(var i =0; i < len; i++) {
			var currentVal = ZmOrganizer.COLOR_VALUES[i];
			if(currentVal && currentVal == color) {
				return i;
			}
		}
	} else if(color <= 9 && color >= 0) {
		return color;
	}
	return -1;
};

/**
 * Returns true if the color is changed
 *
 * @param	{String/int}	color	The color (usually in #rgb format or numeric color code
 * @return {Boolean} Returns true if the color is changed
 */
ZmOrganizer.prototype.isColorChanged =
function(color) {
    var isNewColorCustom = ZmOrganizer.getStandardColorNumber(color) === -1,
        isPrevColorCustom = this.isColorCustom;
    if ((isNewColorCustom && !isPrevColorCustom) ||
        (!isNewColorCustom && isPrevColorCustom) ) {
        //Custom changed to standard or standard changed to custom
        return true;
    }
    else if (isNewColorCustom && isPrevColorCustom) {
        //If both are custom colors check the rgb codes
        return color != this.rgb;
    }
    else if (!isNewColorCustom && !isPrevColorCustom){
        //If both are standard check the numeric color codes
        return color != this.color;
    }
    //default fallback
    return false;
};

/**
 * Updates the folder. Although it is possible to use this method to change just about any folder
 * attribute, it should only be used to set multiple attributes at once since it
 * has extra overhead on the server.
 *
 * @param {Hash}	attrs		the attributes
 */
ZmOrganizer.prototype.update =
function(attrs) {
	this._organizerAction({action: "update", attrs: attrs});
};

/**
 * Assigns the organizer a new parent, moving it within its tree.
 *
 * @param {ZmOrganizer}	newParent		the new parent of this organizer
 * @param {boolean}		noUndo			if true, action is not undoable
 */
ZmOrganizer.prototype.move =
function(newParent, noUndo, batchCmd) {

	var newId = (newParent.nId > 0)
		? newParent.id
		: ZmOrganizer.getSystemId(ZmOrganizer.ID_ROOT);

	if ((newId == this.id || newId == this.parent.id) ||
		(this.type == ZmOrganizer.FOLDER && (ZmOrganizer.normalizeId(newId, this.type) == ZmFolder.ID_SPAM)) ||
		(newParent.isChildOf(this)))
	{
		return;
	}
	var params = {};
	params.batchCmd = batchCmd;
	params.actionTextKey = 'actionMoveOrganizer';
	params.orgName = this.getName(false, false, true, false, false, this.type);
	if (newId == ZmOrganizer.ID_TRASH) {
		params.actionArg = ZmMsg.trash;
		params.action = "trash";
		params.noUndo = noUndo;
	}
	else {
		if (newParent.account && newParent.account.isLocal()) {
			newId = [ZmAccount.LOCAL_ACCOUNT_ID, newId].join(":");
		}
		params.actionArg = newParent.getName(false, false, true, false, false, this.type);
		params.action = "move";
		params.attrs = {l: newId};
		params.noUndo = noUndo;
	}
	this._organizerAction(params);
};

/**
 * Deletes an organizer. If it's a folder, the server deletes any contents and/or
 * subfolders. If the organizer is "Trash" or "Spam", the server deletes and re-creates the
 * folder. In that case, we do not bother to remove it from the UI (and we ignore
 * creates on system folders).
 *
 */
ZmOrganizer.prototype._delete =
function(batchCmd) {
	DBG.println(AjxDebug.DBG1, "deleting: " + AjxStringUtil.htmlEncode(this.name) + ", ID: " + this.id);
	var isEmptyOp = ((this.type == ZmOrganizer.FOLDER || this.type == ZmOrganizer.ADDRBOOK || this.type == ZmOrganizer.BRIEFCASE) &&
					 (this.nId == ZmFolder.ID_SPAM || this.nId == ZmFolder.ID_TRASH));
	// make sure we're not deleting a system object (unless we're emptying SPAM or TRASH)
	if (this.isSystem() && !isEmptyOp) return;

	var action = isEmptyOp ? "empty" : "delete";
	this._organizerAction({action: action, batchCmd: batchCmd});
};

/**
 * Empties the organizer.
 *
 * @param	{Boolean}	doRecursive		<code>true</code> to recursively empty the organizer
 * @param	{ZmBatchCommand}	batchCmd	the batch command
 * @param	{Object}	callback
 * @param	{number}	timeout		the timeout(in seconds)
 * @param	{AjxCallback}	errorCallback		the callback to run after timeout
 * @param	{Boolean}	noBusyOverlay		if <code>true</code>, do not show busy overlay
 */
ZmOrganizer.prototype.empty =
function(doRecursive, batchCmd, callback, timeout, errorCallback, noBusyOverlay) {
	doRecursive = doRecursive || false;

	var isEmptyOp = ((this.type == ZmOrganizer.FOLDER || this.type == ZmOrganizer.ADDRBOOK) &&
					 (this.nId == ZmFolder.ID_SPAM ||
					  this.nId == ZmFolder.ID_TRASH ||
					  this.nId == ZmOrganizer.ID_SYNC_FAILURES));

	// make sure we're not emptying a system object (unless it's SPAM/TRASH/SYNCFAILURES)
	if (this.isSystem() && !isEmptyOp) { return; }

	var params = {
		action: "empty",
		batchCmd: batchCmd,
		callback: callback,
		timeout: timeout,
		errorCallback: errorCallback,
		noBusyOverlay: noBusyOverlay
	};
	params.attrs = (this.nId == ZmFolder.ID_TRASH)
		? {recursive:true}
		: {recursive:doRecursive};

	if (this.isRemote()) {
		params.id = this.getRemoteId();
	}

	this._organizerAction(params);
};

/**
 * Marks all items as "read".
 *
 * @param	{ZmBatchCommand}	batchCmd	the batch command
 */
ZmOrganizer.prototype.markAllRead =
function(batchCmd) {
	var id = this.isRemote() ? this.getRemoteId() : null;
	this._organizerAction({action: "read", id: id, attrs: {l: this.id}, batchCmd:batchCmd});
};

/**
 * Synchronizes the organizer.
 *
 */
ZmOrganizer.prototype.sync =
function() {
	this._organizerAction({action: "sync"});
};

// Notification handling

/**
 * Handles delete notification.
 *
 */
ZmOrganizer.prototype.notifyDelete =
function() {
	// select next reasonable organizer if the currently selected organizer is
	// the one being deleted or is a descendent of the one being deleted
	var tc = appCtxt.getOverviewController().getTreeController(this.type);
	var treeView = tc && tc.getTreeView(appCtxt.getCurrentApp().getOverviewId());

	// treeview returns array of organizers for checkbox style trees
	var organizers = treeView && treeView.getSelected();
	if (organizers) {
		if (!(organizers instanceof Array)) organizers = [organizers];
		for (var i = 0; i <  organizers.length; i++) {
			var organizer = organizers[i];
			if (organizer && (organizer == this || organizer.isChildOf(this))) {
				var folderId = this.parent.id;
				if (this.parent.nId == ZmOrganizer.ID_ROOT) {
					folderId = ZmOrganizer.getSystemId(this.getDefaultFolderId());
				}
				var skipNotify = false;
				treeView.setSelected(folderId, skipNotify);
			}
		}
	}

	// perform actual delete
	this.deleteLocal();
	this._notify(ZmEvent.E_DELETE);
};

/**
 * Handles create notification.
 */
ZmOrganizer.prototype.notifyCreate = function() {};

/**
* Handles modifications to fields that organizers have in general. Note that
* the notification object may contain multiple notifications.
*
* @param {Object}	obj		a "modified" notification
* @param {Hash}	details	the event details
*/
ZmOrganizer.prototype.notifyModify =
function(obj, details) {
	var doNotify = false;
	var details = details || {};
	var fields = {};
	if (obj.name != null && (this.name != obj.name || this.id != obj.id)) {
		if (obj.id == this.id) {
			details.oldName = this.name;
			this.name = obj.name;
			fields[ZmOrganizer.F_NAME] = true;
			this.parent.children.sort(eval(ZmTreeView.COMPARE_FUNC[this.type]));
		} else {
			// rename of a remote folder
			details.newName = obj.name;
			fields[ZmOrganizer.F_RNAME] = true;
		}
		doNotify = true;
	}
	if (obj.u != null && this.numUnread != obj.u) {
		this.numUnread = obj.u;
		fields[ZmOrganizer.F_UNREAD] = true;
		doNotify = true;
	}
	if (obj.n != null && this.numTotal != obj.n) {
		this.numTotal = obj.n;
		fields[ZmOrganizer.F_TOTAL] = true;
		doNotify = true;
	}
	if (obj.s != null && this.sizeTotal != obj.s) {
		this.sizeTotal = obj.s;
		fields[ZmOrganizer.F_SIZE] = true;
		doNotify = true;
	}
	if ((obj.rgb != null || obj.color != null) && !obj._isRemote) {
        var color = obj.color || obj.rgb;
		if (this.isColorChanged(color)) {
			this.isColorCustom = obj.rgb != null;
			this.color = obj.color;
            this.rgb = obj.rgb || ZmOrganizer.COLOR_VALUES[color];
			fields[ZmOrganizer.F_COLOR] = true;
            fields[ZmOrganizer.F_RGB] = true;
		}
		doNotify = true;
	}
	if (obj.f != null && !obj._isRemote) {
		var oflags = this._setFlags().split("").sort().join("");
		var nflags = obj.f.split("").sort().join("");
		if (oflags != nflags) {
			this._parseFlags(obj.f);
			fields[ZmOrganizer.F_FLAGS] = true;
			doNotify = true;
		}
	}
	if (obj.rest != null && this.restUrl != obj.rest && !obj._isRemote) {
		this.restUrl = obj.rest;
		fields[ZmOrganizer.F_REST_URL] = true;
		doNotify = true;
	}
	// if shares changed, do wholesale replace
	if (obj.acl) {
		this.clearShares();
		if (obj.acl.grant && obj.acl.grant.length) {
			AjxDispatcher.require("Share");
			for (var i = 0; i < obj.acl.grant.length; i++) {
				share = ZmShare.createFromJs(this, obj.acl.grant[i]);
				this.addShare(share);
			}
		}
		fields[ZmOrganizer.F_SHARES] = true;
		doNotify = true;
	}
	if (obj.perm && obj._isRemote) {
		fields[ZmOrganizer.F_PERMS] = true;
		doNotify = true;

		// clear acl-related flags so they are recalculated
		this._isAdmin = this._isReadOnly = this._hasPrivateAccess = null;
	}
    if (obj.retentionPolicy) {
        // Only displayed in a modal dialog - no need to doNotify
        if (obj.retentionPolicy[0].keep || obj.retentionPolicy[0].purge) {
            this.retentionPolicy = obj.retentionPolicy;
        } else {
            this.retentionPolicy = null;
        }
    }
	if (obj.hasOwnProperty("webOfflineSyncDays")) {
		this.webOfflineSyncDays = obj.webOfflineSyncDays;
	}

	// Send out composite MODIFY change event
	if (doNotify) {
		details.fields = fields;
		this._notify(ZmEvent.E_MODIFY, details);
	}

	if (this.parent && obj.l != null && obj.l != this.parent.id) {
		var newParent = this._getNewParent(obj.l);
		if (newParent) {
			this.reparent(newParent);
			this._notify(ZmEvent.E_MOVE);
			// could be moving search between Folders and Searches - make sure
			// it has the correct tree
			this.tree = newParent.tree;
		}
	}
};

// Local change handling

/**
 * Deletes the organizer (local). Cleans up a deleted organizer:
 *
 * <ul>
 * <li>remove from parent's list of children</li>
 * <li>remove from item cache</li>
 * <li>perform above two steps for each child</li>
 * <li>clear list of children</li>
 * </ul>
 *
 */
ZmOrganizer.prototype.deleteLocal =
function() {
	this.parent.children.remove(this);
	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		var org = a[i];
		if (org) { org.deleteLocal(); }
	}
	this.children.removeAll();
};

/**
 * Checks if the organizer has a child with the given name.
 *
 * @param {String}	name		the name of the organizer to look for
 * @return	{Boolean}	<code>true</code> if the organizer has a child
 */
ZmOrganizer.prototype.hasChild =
function(name) {
	return (this.getChild(name) != null);
};

/**
* Gets the child with the given name
*
* @param {String}	name		the name of the organizer to look for
* @return	{String}	the name of the child or <code>null</code> if no child has the name
*/
ZmOrganizer.prototype.getChild =
function(name) {
	name = name ? name.toLowerCase() : "";
	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (a[i] && a[i].name && (a[i].name.toLowerCase() == name)) {
			return a[i];
		}
	}

	return null;
};

/**
* Gets the child with the given path
*
* @param {String}	path		the path of the organizer to look for
* @return	{String}	the child or <code>null</code> if no child has the path
*/
ZmOrganizer.prototype.getChildByPath =
function(path) {
	// get starting organizer
	var organizer = this;
	if (path.match(/^\//)) {
		while (organizer.nId != ZmOrganizer.ID_ROOT) {
			organizer = organizer.parent;
		}
		path = path.substr(1);
	}

	// if no path, return current organizer
	if (path.length == 0) return organizer;

	// walk descendent axis to find organizer specified by path
	var parts = path.split('/');
	var i = 0;
	while (i < parts.length) {
		var part = parts[i++];
		var child = organizer.getChild(part);
		if (child == null) {
			return null;
		}
		organizer = child;
	}
	return organizer;
};

/**
 * Changes the parent of this organizer. Note that the new parent passed
 * in may be <code>null</code>, which makes this organizer an orphan.
 *
 * @param {ZmOrganizer}	newParent		the new parent
 */
ZmOrganizer.prototype.reparent =
function(newParent) {
	if (this.parent) {
		this.parent.children.remove(this);
	}
	if (newParent) {
		newParent.children.add(this);
	}
	this.parent = newParent;
};

/**
 * Gets the organizer with the given ID, searching recursively through
 * child organizers. The preferred method for getting an organizer by ID
 * is to use <code>appCtxt.getById()</code>.
 *
 * @param {String}	id		the ID to search for
 * @return	{ZmOrganizer}	the organizer or <code>null</code> if not found
 */
ZmOrganizer.prototype.getById =
function(id) {
	if (this.link && id && typeof(id) == "string") {
		var ids = id.split(":");
		if (this.zid == ids[0] && this.rid == ids[1])
			return this;
	}

	if (this.id == id || this.nId == id) {
		return this;
	}

	var organizer;
	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (organizer = a[i].getById(id)) {
			return organizer;
		}
	}
	return null;
};

/**
 * Gets the first organizer found with the given name, starting from the root.
 *
 * @param {String}	name		the name to search for
 * @return	{ZmOrganizer}	the organizer
 */
ZmOrganizer.prototype.getByName =
function(name, skipImap) {
	return this._getByName(name.toLowerCase(), skipImap);
};

/**
 * Gets a list of organizers with the given type.
 *
 * @param {constant}	type			the desired organizer type
 * @return	{Array}	an array of {ZmOrganizer} objects
 */
ZmOrganizer.prototype.getByType =
function(type) {
	var list = [];
	this._getByType(type, list);
	return list;
};

/**
 * @private
 */
ZmOrganizer.prototype._getByType =
function(type, list) {
	if (this.type == type) {
		list.push(this);
	}
	var a = this.children.getArray();
	for (var i = 0; i < a.length; i++) {
		if (a[i]) {
			a[i]._getByType(type, list);
		}
	}
};

/**
 * Gets the organizer with the given path.
 *
 * @param {String}	path			the path to search for
 * @param {Boolean}	useSystemName	if <code>true</code>, use untranslated version of system folder names
 * @return	{ZmOrganizer}	the organizer
 */
ZmOrganizer.prototype.getByPath =
function(path, useSystemName) {
	return this._getByPath(path.toLowerCase(), useSystemName);
};

/**
 * Test the path of this folder and then descendants against the given path, case insensitively.
 *
 * @private
 */
ZmOrganizer.prototype._getByPath =
function(path, useSystemName) {
	if (this.nId == ZmFolder.ID_TAGS) { return null; }

	if (path == this.getPath(false, false, null, true, useSystemName).toLowerCase()) {
		return this;
	}

	var a = this.children.getArray();
	for (var i = 0; i < a.length; i++) {
		var organizer = a[i]._getByPath(path, useSystemName);
		if (organizer) {
			return organizer;
		}
	}
	return null;
};

/**
 * Gets the number of children of this organizer.
 *
 * @return	{int}	the size
 */
ZmOrganizer.prototype.size =
function() {
	return this.children.size();
};

/**
 * Checks if the given organizer is a descendant of this one.
 *
 * @param {ZmOrganizer}	organizer		a possible descendant of ours
 * @return	{Boolean}	<code>if the given organizer is a descendant; <code>false</code> otherwise
 */
ZmOrganizer.prototype.isChildOf =
function (organizer) {
	var parent = this.parent;
	while (parent) {
		if (parent == organizer) {
			return true;
		}
		parent = parent.parent;
	}
	return false;
};

/**
 * Gets the organizer with the given ID (looks in this organizer tree).
 *
 * @param {int}	parentId	the ID of the organizer to find
 * @return	{ZmOrganizer}	the organizer
 *
 * @private
 */
ZmOrganizer.prototype._getNewParent =
function(parentId) {
	return appCtxt.getById(parentId);
};

/**
 * Checks if the organizer with the given ID is under this organizer.
 *
 * @param	{String}	id		the ID
 * @return	{Boolean}	<code>true</code> if the organizer is under this organizer
 */
ZmOrganizer.prototype.isUnder =
function(id) {
	id = id.toString();
	if (this.nId == id || (this.isRemote() && this.rid == id)) { return true; }

	var parent = this.parent;
	while (parent && parent.nId != ZmOrganizer.ID_ROOT) {
		if (parent.nId == id) {
			return true;
		}
		parent = parent.parent;
	}
	return false;
};

/**
 * Checks if this organizer is in "Trash".
 *
 * @return	{Boolean}	<code>true</code> if in "Trash"
 */
ZmOrganizer.prototype.isInTrash =
function() {
	return this.isUnder(ZmOrganizer.ID_TRASH);
};

/**
 * Checks if permissions are allowed.
 *
 * @return	{Boolean}	<code>true</code> if permissions are allowed
 */
ZmOrganizer.prototype.isPermAllowed =
function(perm) {
	if (this.perm) {
		var positivePerms = this.perm.replace(/-./g, "");
		return (positivePerms.indexOf(perm) != -1);
	}
	return false;
};

/**
 * Checks if the organizer is read-only.
 *
 * @return	{Boolean}	<code>true</code> if read-only
 */
ZmOrganizer.prototype.isReadOnly =
function() {
	if (this._isReadOnly == null) {
		var share = this.getMainShare();
		this._isReadOnly = (share != null)
			? (this.isRemote() && !share.isWrite())
			: (this.isRemote() && this.isPermAllowed(ZmOrganizer.PERM_READ) && !this.isPermAllowed(ZmOrganizer.PERM_WRITE));
	}
	return this._isReadOnly;
};

/**
 * Checks if admin.
 *
 * @return	{Boolean}	<code>true</code> if this organizer is admin
 */
ZmOrganizer.prototype.isAdmin =
function() {
	if (this._isAdmin == null) {
		var share = this.getMainShare();
		this._isAdmin = (share != null)
			? (this.isRemote() && share.isAdmin())
			: (this.isRemote() && this.isPermAllowed(ZmOrganizer.PERM_ADMIN));
	}
	return this._isAdmin;
};

/**
 * Checks if the organizer has private access.
 *
 * @return	{Boolean}	<code>true</code> if has private access
 */
ZmOrganizer.prototype.hasPrivateAccess =
function() {
	if (this._hasPrivateAccess == null) {
		var share = this.getMainShare();
		this._hasPrivateAccess = (share != null)
			? (this.isRemote() && share.hasPrivateAccess())
			: (this.isRemote() && this.isPermAllowed(ZmOrganizer.PERM_PRIVATE));
	}
	return this._hasPrivateAccess;
};

/**
 * Checks if the organizer is "remote". That applies to mountpoints (links),
 * the folders they represent, and any subfolders we know about.
 *
 * @return	{Boolean}	<code>true</code> if the organizer is "remote"
 */
ZmOrganizer.prototype.isRemote =
function() {
	if (this._isRemote == null) {
		if (this.zid != null) {
			this._isRemote = true;
		} else {
			if (appCtxt.multiAccounts) {
				var account = this.account;
				var parsed = ZmOrganizer.parseId(this.id);

				if (!account) {
					if (parsed.account && parsed.account.isMain) {
						this._isRemote = false;
						return this._isRemote;
					} else {
						account = appCtxt.getActiveAccount();
					}
				}
				this._isRemote = Boolean(!parsed.account || (parsed.account && (parsed.account != account)));
			} else {
				var id = String(this.id);
				this._isRemote = ((id.indexOf(":") != -1) && (id.indexOf(appCtxt.getActiveAccount().id) != 0));
			}
		}
	}
	return this._isRemote;
};

ZmOrganizer.prototype.isRemoteRoot = function() {
	return this.isRemote() && (this.rid == ZmOrganizer.ID_ROOT);
}

/**
 * Checks if the organizer is a system tag or folder.
 *
 * @return	{Boolean}	<code>true</code> if system tag or folder
 */
ZmOrganizer.prototype.isSystem =
function () {
	return (this.nId < ZmOrganizer.FIRST_USER_ID[this.type]);
};

ZmOrganizer.prototype.isDefault =
function () {
	return this.nId == this.getDefaultFolderId();
};

ZmOrganizer.prototype.getDefaultFolderId =
function() {
	return ZmOrganizer.DEFAULT_FOLDER[this.type];
};

ZmOrganizer.prototype.isTrash =
function () {
	return this.nId == ZmFolder.ID_TRASH;
};


/**
 * Checks if the organizer gets its contents from an external feed.
 *
 * @return	{Boolean}	<code>true</code>  if from external feed
 */
ZmOrganizer.prototype.isFeed =
function () {
	return Boolean(this.url);
};

/** Returns true if organizer has feeds. */
ZmOrganizer.prototype.hasFeeds = function() { return false; };

/**
 * Checks if this folder maps to a datasource. If type is given, returns
 * true if folder maps to a datasource *and* is of the given type.
 *
 * @param	{constant}	type			the type (see {@link ZmAccount.TYPE_POP} or {@link ZmAccount.TYPE_IMAP})
 * @param	{Boolean}	checkParent		if <code>true</code>, walk-up the parent chain
 * @return	{Boolean}	<code>true</code> if this folder maps to a datasource
 */
ZmOrganizer.prototype.isDataSource =
function(type, checkParent) {
	var dss = this.getDataSources(type, checkParent);
	return (dss && dss.length > 0);
};

/**
 * Gets the data sources this folder maps to. If type is given,
 * returns non-null result only if folder maps to datasource(s) *and* is of the
 * given type.
 *
 * @param	{constant}	type			the type (see {@link ZmAccount.TYPE_POP} or {@link ZmAccount.TYPE_IMAP})
 * @param	{Boolean}	checkParent		if <code>true</code>, walk-up the parent chain
 * @return	{Array}	the data sources this folder maps to or <code>null</code> for none
 */
ZmOrganizer.prototype.getDataSources =
function(type, checkParent) {
	if (!appCtxt.get(ZmSetting.MAIL_ENABLED)) { return null; }

	var dsc = appCtxt.getDataSourceCollection();
	var dataSources = dsc.getByFolderId(this.nId, type);

	if (dataSources.length == 0) {
		return (checkParent && this.parent)
			? this.parent.getDataSources(type, checkParent)
			: null;
	}

	return dataSources;
};

/**
 * Gets the owner.
 *
 * @return	{String}	the owner
 */
ZmOrganizer.prototype.getOwner =
function() {
	return this.owner || (this.parent && this.parent.getOwner()) || appCtxt.get(ZmSetting.USERNAME);
};

/**
 * Gets the sort index.
 *
 * @return	{int}	the sort index
 */
ZmOrganizer.getSortIndex =
function(child, sortFunction) {
	if (!(child && child.parent && sortFunction)) { return null; }
	var children = child.parent.children.getArray();
	for (var i = 0; i < children.length; i++) {
		var test = sortFunction(child, children[i]);
		if (test == -1) {
			return i;
		}
	}
	return i;
};

/**
 * Sends a request to the server. Note that it's done asynchronously, but
 * there is no callback given. Hence, an organizer action is the last thing
 * done before returning to the event loop. The result of the action is
 * handled via notifications.
 *
 * @param {String}	action		the operation to perform
 * @param {Hash}	attrs		a hash of additional attributes to set in the request
 * @param {ZmBatchCommand}	batchCmd	the batch command that contains this request
 *
 * @private
 */
ZmOrganizer.prototype._organizerAction =
function(params) {

	var cmd = ZmOrganizer.SOAP_CMD[this.type] + "Request";
	var request = {
		_jsns: "urn:zimbraMail",
		action : {
			op: params.action,
			id: params.id || this.id
		}
	};
	var jsonObj = {};
	jsonObj[cmd] = request;

	for (var attr in params.attrs) {
		if (AjxEnv.isIE) {
			params.attrs[attr] += ""; //To string
		}
		request.action[attr] = params.attrs[attr];
	}
	var actionController = appCtxt.getActionController();
	actionController.dismiss();
	var actionLogItem = (!params.noUndo && actionController && actionController.actionPerformed({op: params.action, id: params.id || this.id, attrs: params.attrs})) || null;
	var respCallback = new AjxCallback(this, this._handleResponseOrganizerAction, [params, actionLogItem]);
	if (params.batchCmd) {
        params.batchCmd.addRequestParams(jsonObj, respCallback, params.errorCallback);
 	}
	else {
		var accountName;
		if (appCtxt.multiAccounts) {
			accountName = (this.account)
				? this.account.name : appCtxt.accountList.mainAccount.name;
		}
		appCtxt.getAppController().sendRequest({
			jsonObj: jsonObj,
			asyncMode: true,
			accountName: accountName,
			callback: respCallback,
			errorCallback: params.errorCallback,
			timeout: params.timeout,
			noBusyOverlay: params.noBusyOverlay
		});
	}
};

/**
 * @private
 */
ZmOrganizer.prototype._handleResponseOrganizerAction =
function(params, actionLogItem, result) {

	if (actionLogItem) {
		actionLogItem.setComplete();
	}
	if (params.callback) {
		params.callback.run(result);
	}
	if (params.actionTextKey) {
		var actionController = appCtxt.getActionController();
		var summary = ZmOrganizer.getActionSummary({
			actionTextKey:  params.actionTextKey,
			numItems:       params.numItems || 1,
			type:           this.type,
			orgName:        params.orgName,
			actionArg:      params.actionArg
		});
		var undoLink = actionLogItem && actionController && actionController.getUndoLink(actionLogItem);
		if (undoLink && actionController) {
			actionController.onPopup();
			appCtxt.setStatusMsg({msg: summary+undoLink, transitions: actionController.getStatusTransitions()});
		} else {
			appCtxt.setStatusMsg(summary);
		}
	}
};

/**
 * Returns a string describing an action, intended for display as toast to tell the
 * user what they just did.
 *
 * @param   {Object}        params          hash of params:
 *          {String}        type            organizer type (ZmOrganizer.*)
 *          {String}        actionTextKey   ZmMsg key for text string describing action
 *          {String}        orgName         name of the organizer that was affected
 *          {String}        actionArg       (optional) additional argument
 *
 * @return {String}     action summary
 */
ZmOrganizer.getActionSummary =
function(params) {

	var type = params.type,
		typeKey = ZmOrganizer.FOLDER_KEY[type],
		typeText = ZmMsg[typeKey],
		capKey = AjxStringUtil.capitalize(typeKey),
		alternateKey = params.actionTextKey + capKey,
		text = ZmMsg[alternateKey] || ZmMsg[params.actionTextKey],
		orgName = AjxStringUtil.htmlEncode(params.orgName),
		arg = AjxStringUtil.htmlEncode(params.actionArg);

	return AjxMessageFormat.format(text, [ typeText, orgName, arg ]);
};

/**
 * Test the name of this organizer and then descendants against the given name, case insensitively.
 * 
 * @private
 */
ZmOrganizer.prototype._getByName =
function(name, skipImap) {
	if (this.name && name == this.name.toLowerCase()) {
		return this;
	}

	var organizer;
	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (organizer = a[i]._getByName(name, skipImap)) {
			if (skipImap && organizer.isDataSource(ZmAccount.TYPE_IMAP, true)) {
				continue;
			}
			return organizer;
		}
	}
	return null;
};

/**
 * Takes a string of flag chars and applies them to this organizer.
 * 
 * @private
 */
ZmOrganizer.prototype._parseFlags =
function(str) {
	for (var i = 0; i < ZmOrganizer.ALL_FLAGS.length; i++) {
		var flag = ZmOrganizer.ALL_FLAGS[i];
		this[ZmOrganizer.FLAG_PROP[flag]] = (Boolean(str && (str.indexOf(flag) != -1)));
	}
};

/**
 * Converts this organizer's flag-related props into a string of flag chars.
 * 
 * @private
 */
ZmOrganizer.prototype._setFlags =
function() {
	var flags = "";
	for (var i = 0; i < ZmOrganizer.ALL_FLAGS.length; i++) {
		var flag = ZmOrganizer.ALL_FLAGS[i];
		var prop = ZmOrganizer.FLAG_PROP[flag];
		if (this[prop]) {
			flags = flags + flag;
		}
	}
	return flags;
};

/**
 * Adds a change listener.
 * 
 * @param	{AjxListener}	the listener
 */
ZmOrganizer.prototype.addChangeListener =
function(listener) {
	this.tree.addChangeListener(listener);
};

/**
 * Removes a change listener.
 * 
 * @param	{AjxListener}	the listener
 */
ZmOrganizer.prototype.removeChangeListener =
function(listener) {
	this.tree.removeChangeListener(listener);
};

/**
 * @private
 */
ZmOrganizer.prototype._setSharesFromJs =
function(obj) {

	// a folder object will have an acl with grants if this user has
	// shared it, or if it has been shared to this user with admin rights
	if (obj.acl && obj.acl.grant && obj.acl.grant.length > 0) {
		AjxDispatcher.require("Share");
		for (var i = 0; i < obj.acl.grant.length; i++) {
			var grant = obj.acl.grant[i];
			this.addShare(ZmShare.createFromJs(this, grant));
		}
	}
};

/**
 * Handle notifications through the tree.
 * 
 * @private
 */
ZmOrganizer.prototype._notify =
function(event, details) {

	if (details) {
		details.organizers = [this];
	} else {
		details = {organizers: [this]};
	}
	this.tree._evt.type = this.type;	// all folder types are in a single tree
	this.tree._notify(event, details);
};

/**
 * Gets a marked-up version of the name.
 *
 * @param {String}	name			the name to mark up
 * @param {Boolean}	showUnread		if <code>true</code>, display the number of unread items (in parens)
 * @param {Boolean}	noMarkup		if <code>true</code>, do not return any HTML
 * 
 * @private
 */
ZmOrganizer.prototype._markupName = 
function(name, showUnread, noMarkup) {
	if (!noMarkup) {
		name = AjxStringUtil.htmlEncode(name, true);
	}
	if (showUnread && this.hasUnreadDescendent()) {
		if (this.numUnread > 0) {
            name = AjxMessageFormat.format(ZmMsg.folderUnread, [name, this.numUnread]);
		}
		if (!noMarkup) {
			name = ["<span style='font-weight:bold'>", name, "</span>"].join("");
		}
	}
	if (this.noSuchFolder && !noMarkup) {
		name = ["<del>", name, "</del>"].join("");
	}
	return name;
};

/**
 * @private
 */
ZmOrganizer.prototype._getItemsText =
function() {
	var result = ZmMsg[ZmOrganizer.ITEMS_KEY[this.type]];
	if (!result || this.isTrash()) {
		result = ZmMsg.items;
	}
	return result;
};

ZmOrganizer.prototype._getUnreadLabel = 
function() {
	return ZmMsg.unread;	
};

/**
 * Returns true if any descendent folders have unread messages.
 *
 * @returns {boolean}   true if any descendent folders have unread messages
 */
ZmOrganizer.prototype.hasUnreadDescendent = function() {

	if (this.numUnread > 0) {
		return true;
	}

	var a = this.children.getArray(),
		sz = this.children.size();

	for (var i = 0; i < sz; i++) {
		if (a[i].hasUnreadDescendent()) {
			return true;
		}
	}

	return false;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmFolder")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines a folder.
 *
 */

/**
 * Creates a folder.
 * @class
 * This class represents a folder, which may contain mail. At some point, folders may be
 * able to contain contacts and/or appointments.
 *
 * @author Conrad Damon
 *
 * @param	{Hash}	params		a hash of parameters
 * @param {int}	params.id		the numeric ID
 * @param {String}	params.name		the name
 * @param {ZmOrganizer}	params.parent	the parent folder
 * @param {ZmTree}	params.tree		the tree model that contains this folder
 * @param {int}	params.numUnread	the number of unread items for this folder
 * @param {int}	params.numTotal		the number of items for this folder
 * @param {int}	params.sizeTotal	the total size of folder's items
 * @param {String}	params.url		the URL for this folder's feed
 * @param {String}	params.owner	the Owner for this organizer
 * @param {String}	params.oname	the Owner's name for this organizer, if remote folder
 * @param {String}	params.zid		the Zimbra ID of owner, if remote folder
 * @param {String}	params.rid		the Remote ID of organizer, if remote folder
 * @param {String}	params.restUrl	the REST URL of this organizer
 * 
 * @extends		ZmOrganizer
 */
ZmFolder = function(params) {
	if (arguments.length == 0) { return; }
	params.type = params.type || ZmOrganizer.FOLDER;
	ZmOrganizer.call(this, params);
};

ZmFolder.prototype = new ZmOrganizer;
ZmFolder.prototype.constructor = ZmFolder;

ZmFolder.prototype.isZmFolder = true;
ZmFolder.prototype.toString = function() { return "ZmFolder"; };


// needed to construct USER_ROOT if mail disabled
ZmOrganizer.ORG_CLASS[ZmId.ORG_FOLDER] = "ZmFolder";

ZmFolder.SEP 									= "/";							// path separator

// system folders (see Mailbox.java in ZimbraServer for positive int consts)
// Note: since these are defined as Numbers, and IDs come into our system as Strings,
// we need to use == for comparisons (instead of ===, which will fail)
ZmFolder.ID_LOAD_FOLDERS						= -3;							// special "Load remaining folders" placeholder
ZmFolder.ID_OTHER								= -2;							// used for tcon value (see below)
ZmFolder.ID_SEP									= -1;							// separator
ZmFolder.ID_ROOT								= ZmOrganizer.ID_ROOT;
ZmFolder.ID_INBOX								= ZmOrganizer.ID_INBOX;
ZmFolder.ID_TRASH								= ZmOrganizer.ID_TRASH;
ZmFolder.ID_SPAM								= ZmOrganizer.ID_SPAM;
ZmFolder.ID_SENT								= 5;
ZmFolder.ID_DRAFTS								= 6;
ZmFolder.ID_CONTACTS							= ZmOrganizer.ID_ADDRBOOK;
ZmFolder.ID_AUTO_ADDED							= ZmOrganizer.ID_AUTO_ADDED;
ZmFolder.ID_TAGS	 							= 8;
ZmFolder.ID_TASKS								= ZmOrganizer.ID_TASKS;
ZmFolder.ID_SYNC_FAILURES						= ZmOrganizer.ID_SYNC_FAILURES;
ZmFolder.ID_OUTBOX	 							= ZmOrganizer.ID_OUTBOX;
ZmFolder.ID_ATTACHMENTS                         = ZmOrganizer.ID_ATTACHMENTS;
ZmFolder.ID_DLS									= ZmOrganizer.ID_DLS;
ZmFolder.ID_FILE_SHARED_WITH_ME					= ZmOrganizer.ID_FILE_SHARED_WITH_ME;

// system folder names
ZmFolder.MSG_KEY = {};
ZmFolder.MSG_KEY[ZmFolder.ID_INBOX]				= "inbox";
ZmFolder.MSG_KEY[ZmFolder.ID_TRASH]				= "trash";
ZmFolder.MSG_KEY[ZmFolder.ID_SPAM]				= "junk";
ZmFolder.MSG_KEY[ZmFolder.ID_SENT]				= "sent";
ZmFolder.MSG_KEY[ZmFolder.ID_DRAFTS]			= "drafts";
ZmFolder.MSG_KEY[ZmFolder.ID_CONTACTS]			= "contacts";
ZmFolder.MSG_KEY[ZmFolder.ID_AUTO_ADDED]		= "emailedContacts";
ZmFolder.MSG_KEY[ZmFolder.ID_TASKS]				= "tasks";
ZmFolder.MSG_KEY[ZmFolder.ID_TAGS]				= "tags";
ZmFolder.MSG_KEY[ZmOrganizer.ID_CALENDAR]		= "calendar";
ZmFolder.MSG_KEY[ZmOrganizer.ID_BRIEFCASE]		= "briefcase";
ZmFolder.MSG_KEY[ZmOrganizer.ID_ALL_MAILBOXES]	= "allMailboxes";
ZmFolder.MSG_KEY[ZmFolder.ID_OUTBOX]			= "outbox";
ZmFolder.MSG_KEY[ZmFolder.ID_SYNC_FAILURES]		= "errorReports";
ZmFolder.MSG_KEY[ZmFolder.ID_ATTACHMENTS]       = "attachments";
ZmFolder.MSG_KEY[ZmFolder.ID_FILE_SHARED_WITH_ME]	= "filesSharedWithMe";

// system folder icons
ZmFolder.ICON = {};
ZmFolder.ICON[ZmFolder.ID_INBOX]				= "Inbox";
ZmFolder.ICON[ZmFolder.ID_TRASH]				= "Trash";
ZmFolder.ICON[ZmFolder.ID_SPAM]					= "SpamFolder";
ZmFolder.ICON[ZmFolder.ID_SENT]					= "SentFolder";
ZmFolder.ICON[ZmFolder.ID_SYNC_FAILURES]		= "SendReceive";
ZmFolder.ICON[ZmFolder.ID_OUTBOX]				= "Outbox";
ZmFolder.ICON[ZmFolder.ID_DRAFTS]				= "DraftFolder";
ZmFolder.ICON[ZmFolder.ID_LOAD_FOLDERS]			= "Plus";
ZmFolder.ICON[ZmFolder.ID_ATTACHMENTS]          = "Attachment";

// name to use within the query language
ZmFolder.QUERY_NAME = {};
ZmFolder.QUERY_NAME[ZmFolder.ID_INBOX]			= "inbox";
ZmFolder.QUERY_NAME[ZmFolder.ID_TRASH]			= "trash";
ZmFolder.QUERY_NAME[ZmFolder.ID_SPAM]			= "junk";
ZmFolder.QUERY_NAME[ZmFolder.ID_SENT]			= "sent";
ZmFolder.QUERY_NAME[ZmFolder.ID_OUTBOX]			= "outbox";
ZmFolder.QUERY_NAME[ZmFolder.ID_DRAFTS]			= "drafts";
ZmFolder.QUERY_NAME[ZmOrganizer.ID_CALENDAR]	= "calendar";
ZmFolder.QUERY_NAME[ZmFolder.ID_CONTACTS]		= "contacts";
ZmFolder.QUERY_NAME[ZmFolder.ID_TASKS]			= "tasks";
ZmFolder.QUERY_NAME[ZmFolder.ID_AUTO_ADDED]		= "Emailed Contacts";
ZmFolder.QUERY_NAME[ZmOrganizer.ID_BRIEFCASE]	= "briefcase";
ZmFolder.QUERY_NAME[ZmFolder.ID_SYNC_FAILURES]	= "Error Reports";
ZmFolder.QUERY_NAME[ZmFolder.ID_FILE_SHARED_WITH_ME]	= "Files shared with me";

ZmFolder.QUERY_ID = AjxUtil.valueHash(ZmFolder.QUERY_NAME);

// order within the overview panel
ZmFolder.SORT_ORDER = {};
ZmFolder.SORT_ORDER[ZmFolder.ID_INBOX]			= 1;
ZmFolder.SORT_ORDER[ZmFolder.ID_SENT]			= 2;
ZmFolder.SORT_ORDER[ZmFolder.ID_DRAFTS]			= 3;
ZmFolder.SORT_ORDER[ZmFolder.ID_SPAM]			= 4;
ZmFolder.SORT_ORDER[ZmFolder.ID_OUTBOX]			= 5;
ZmFolder.SORT_ORDER[ZmFolder.ID_TRASH]			= 6;
ZmFolder.SORT_ORDER[ZmFolder.ID_SYNC_FAILURES]	= 7;
ZmFolder.SORT_ORDER[ZmFolder.ID_SEP]			= 8;
ZmFolder.SORT_ORDER[ZmFolder.ID_ATTACHMENTS]    = 99; // Last

// character codes for "tcon" attribute in conv action request, which controls
// which folders are affected
ZmFolder.TCON_CODE = {};
ZmFolder.TCON_CODE[ZmFolder.ID_TRASH]			= "t";
ZmFolder.TCON_CODE[ZmFolder.ID_SYNC_FAILURES]	= "o";
ZmFolder.TCON_CODE[ZmFolder.ID_SPAM]			= "j";
ZmFolder.TCON_CODE[ZmFolder.ID_SENT]			= "s";
ZmFolder.TCON_CODE[ZmFolder.ID_DRAFTS]			= "d";
ZmFolder.TCON_CODE[ZmFolder.ID_OTHER]			= "o";

// folders that look like mail folders that we don't want to show
ZmFolder.HIDE_ID = {};
ZmFolder.HIDE_ID[ZmOrganizer.ID_CHATS]				= true;
ZmFolder.HIDE_ID[ZmOrganizer.ID_NOTIFICATION_MP]	= true;

// Hide folders migrated from Outlook mailbox
ZmFolder.HIDE_NAME = {};
//ZmFolder.HIDE_NAME["Journal"]		= true;
//ZmFolder.HIDE_NAME["Notes"]		= true;
//ZmFolder.HIDE_NAME["Outbox"]		= true;
//ZmFolder.HIDE_NAME["Tasks"]		= true;

// folders that contain mail from me instead of to me
ZmFolder.OUTBOUND = [ZmFolder.ID_SENT, ZmFolder.ID_OUTBOX, ZmFolder.ID_DRAFTS];

// The extra-special, visible but untouchable outlook folder
ZmFolder.SYNC_ISSUES 							= "Sync Issues";

// map name to ID
ZmFolder.QUERY_ID = {};
(function() {
	for (var i in ZmFolder.QUERY_NAME) {
		ZmFolder.QUERY_ID[ZmFolder.QUERY_NAME[i]] = i;
	}
})();

/**
 * Comparison function for folders. Intended for use on a list of user folders
 * through a call to <code>Array.sort()</code>.
 *
 * @param {ZmFolder}	folderA		a folder
 * @param {ZmFolder}	folderB		a folder
 * @param {Boolean}		nonMail		this is sorting non mail tree.
 * @return	{int} 0 if the folders match
 */
ZmFolder.sortCompare =
function(folderA, folderB, nonMail) {
	var check = ZmOrganizer.checkSortArgs(folderA, folderB);
	if (check != null) { return check; }

	// offline client wants POP folders above all else *unless* we are POP'ing into Inbox
	if (appCtxt.isOffline) {
		if (folderA.isDataSource(ZmAccount.TYPE_POP)) {
			if (folderA.id == ZmFolder.ID_INBOX) return -1;
			if (folderB.isDataSource(ZmAccount.TYPE_POP)) {
				if (folderA.name.toLowerCase() > folderB.name.toLowerCase()) { return 1; }
				if (folderA.name.toLowerCase() < folderB.name.toLowerCase()) { return -1; }
				return 0;
			}
			return -1;
		} else if (folderB.isDataSource(ZmAccount.TYPE_POP)) {
			return 1;
		}
	}

	if (ZmFolder.SORT_ORDER[folderA.nId] && ZmFolder.SORT_ORDER[folderB.nId]) {
		return (ZmFolder.SORT_ORDER[folderA.nId] - ZmFolder.SORT_ORDER[folderB.nId]);
	}

	// links (shared folders or mailboxes) appear after personal folders
	if (folderA.link !== folderB.link) {
		return folderA.link ? 1 : -1;
	}

	if (nonMail) {
		//for nonp-mail apps, trash last of all things
		if (folderA.isTrash()) {
			return 1;
		}
		if (folderB.isTrash()) {
			return -1;
		}
		//system before non-system (except for trash)
		if (folderA.isSystem() && !folderB.isSystem()) {
			return -1;
		}
		if (!folderA.isSystem() && folderB.isSystem()) {
			return 1;
		}
		if (folderA.isSystem() && folderB.isSystem()) {
			//for 2 system folders, the default one is first, and the rest ordered alphabetically (again except for trash that appears after the user folders)
			if (folderA.isDefault()) {
				return -1;
			}
			if (folderB.isDefault()) {
				return 1;
			}
			//the other cases will be sorted by name below. Either 2 system or 2 user folders.
		}
	}
	else {
		if (!ZmFolder.SORT_ORDER[folderA.nId] && ZmFolder.SORT_ORDER[folderB.nId]) { return 1; }
		if (ZmFolder.SORT_ORDER[folderA.nId] && !ZmFolder.SORT_ORDER[folderB.nId]) { return -1; }
	}

	if (folderA.name.toLowerCase() > folderB.name.toLowerCase()) { return 1; }
	if (folderA.name.toLowerCase() < folderB.name.toLowerCase()) { return -1; }
	return 0;
};


ZmFolder.sortCompareNonMail =
function(folderA, folderB) {
	return ZmFolder.sortCompare(folderA, folderB, true);
};

/**
 * Compares the folders by path.
 * 
 * @param {ZmFolder}	folderA		a folder
 * @param {ZmFolder}	folderB		a folder
 * @return	{int} 0 if the folders match
 */
ZmFolder.sortComparePath =
function(folderA, folderB) {

	var pathA = folderA && folderA.getPath(false, false, null, true, true);
	var pathB = folderB && folderB.getPath(false, false, null, true, true);
	var check = ZmOrganizer.checkSortArgs(pathA, pathB);
	if (check != null) { return check; }

	if (ZmFolder.SORT_ORDER[folderA.nId] && ZmFolder.SORT_ORDER[folderB.nId]) {
		return (ZmFolder.SORT_ORDER[folderA.nId] - ZmFolder.SORT_ORDER[folderB.nId]);
	}
	if (!ZmFolder.SORT_ORDER[folderA.nId] && ZmFolder.SORT_ORDER[folderB.nId]) { return 1; }
	if (ZmFolder.SORT_ORDER[folderA.nId] && !ZmFolder.SORT_ORDER[folderB.nId]) { return -1; }
	if (pathA.toLowerCase() > pathB.toLowerCase()) { return 1; }
	if (pathA.toLowerCase() < pathB.toLowerCase()) { return -1; }
	return 0;
};

/**
 * Checks a folder name for validity. Note: that a name, rather than a path, is checked.
 *
 * @param {String}	name		the folder name
 * @param {ZmFolder}	parent		the parent folder
 * @return	{String} an error message if the name is invalid; <code>null</code>if the name is valid. 
 */
ZmFolder.checkName =
function(name, parent) {
	var error = ZmOrganizer.checkName(name);
	if (error) { return error; }

	// make sure path isn't same as a system folder
	parent = parent || appCtxt.getFolderTree().root;
	if (parent && (parent.id == ZmFolder.ID_ROOT)) {
		var lname = name.toLowerCase();
		for (var id in ZmFolder.MSG_KEY) {
			var sysname = ZmMsg[ZmFolder.MSG_KEY[id]];
			if (sysname && (lname == sysname.toLowerCase())) {
				return ZmMsg.folderNameReserved;
			}
		}
		/*if (lname == ZmFolder.SYNC_ISSUES.toLowerCase()) {
			return ZmMsg.folderNameReserved;
		}*/
	}

	return null;
};

/**
 * Gets the "well-known" ID for a given folder name.
 * 
 * @param	{String}	folderName	the folder name
 * @return	{String}	the id or <code>null</code> if not found
 */
ZmFolder.getIdForName =
function(folderName) {
	var name = folderName.toLowerCase();
	for (var i in ZmFolder.MSG_KEY) {
		if (ZmFolder.MSG_KEY[i] == name) {
			return i;
		}
	}
	return null;
};

/**
 * Moves a folder. A user can move a folder to "Trash" even if there is already a folder in "Trash" with the
 * same name. A new name will be generated for this folder and a rename is performed before the move.
 * 
 * @param	{ZmFolder}	newParent		the new parent
 * @param	{boolean}	noUndo			true if the action should not be undoable
 * @param	{String}	actionText		optional custom action text to display as summary
 */
ZmFolder.prototype.move =
function(newParent, noUndo, actionText, batchCmd) {
	var origName = this.name;
	var name = this.name;
	while (newParent.hasChild(name)) {
		name = name + "_";
	}
	if (origName != name) {
		this.rename(name);
	}
	ZmOrganizer.prototype.move.call(this, newParent, noUndo, batchCmd);
};

/**
 * Sends <code>&lt;FolderActionRequest&gt;</code> to turn sync'ing on/off for IMAP folders. Currently,
 * this is only used by Offline/ZDesktop client
 *
 * @param {Boolean}	syncIt		the flag indicating whether to sync this folder
 * @param {AjxCallback}	callback		the callback to call once server request is successful
 * @param {AjxCallback}	errorCallback	the callback to call if server returns error
 */
ZmFolder.prototype.toggleSyncOffline =
function(callback, errorCallback) {
	if (!this.isOfflineSyncable) { return; }

	var op = this.isOfflineSyncing ? "!syncon" : "syncon";
	var soapDoc = AjxSoapDoc.create("FolderActionRequest", "urn:zimbraMail");
	var actionNode = soapDoc.set("action");
	actionNode.setAttribute("op", op);
	actionNode.setAttribute("id", this.id);

	var params = {
		soapDoc: soapDoc,
		asyncMode: true,
		callback: callback,
		errorCallback: errorCallback
	};
	appCtxt.getAppController().sendRequest(params);
};

/**
 * Checks folders recursively for feeds.
 * 
 * @return	{Boolean}	<code>true</code> for feeds
 */
ZmFolder.prototype.hasFeeds =
function() {
	if (this.type != ZmOrganizer.FOLDER) { return false; }

	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (a[i].isFeed()) {
			return true;
		}
		if (a[i].children && a[i].children.size() > 0) {
			return (a[i].hasFeeds && a[i].hasFeeds());
		}
	}
	return false;
};

/**
 * Checks if the folder has search.
 * 
 * @param	{String}	id	not used
 * @return	{Boolean}	<code>true</code> if has search
 */
ZmFolder.prototype.hasSearch =
function(id) {
	if (this.type == ZmOrganizer.SEARCH) { return true; }

	var a = this.children.getArray();
	var sz = this.children.size();
	for (var i = 0; i < sz; i++) {
		if (a[i].hasSearch()) {
			return true;
		}
	}

	return false;
};

/**
 * Checks if the folder supports public access. Override this method if you dont want a folder to be accessed publicly
 * 
 * @return	{Boolean}	always returns <code>true</code>
 */
ZmFolder.prototype.supportsPublicAccess =
function() {
	return true;
};

/**
 * Handles the creation of a folder or search folder. This folder is the parent
 * of the newly created folder. A folder may hold a folder or search folder,
 * and a search folder may hold another search folder.
 *
 * @param {Object}	obj				a JS folder object from the notification
 * @param {String}	elementType		the type of containing JSON element
 * @param {Boolean}	skipNotify		<code>true</code> if notifying client should be ignored
 */
ZmFolder.prototype.notifyCreate =
function(obj, elementType, skipNotify) {
	// ignore creates of system folders
	var nId = ZmOrganizer.normalizeId(obj.id);
	if (this.isSystem() && nId < ZmOrganizer.FIRST_USER_ID[this.type]) { return; }

	var account = ZmOrganizer.parseId(obj.id).account;
	var folder = ZmFolderTree.createFromJs(this, obj, this.tree, elementType, null, account);
	if (folder) {
		var index = ZmOrganizer.getSortIndex(folder, eval(ZmTreeView.COMPARE_FUNC[this.type]));
		this.children.add(folder, index);

		if (!skipNotify) {
			folder._notify(ZmEvent.E_CREATE);
		}
	}
};

/**
 * Provide some extra info in the change event about the former state
 * of the folder. Note that we null out the field after setting up the
 * change event, so the notification isn't also sent when the parent
 * class's method is called.
 *
 * @param {Object}	obj	a "modified" notification
 */
ZmFolder.prototype.notifyModify =
function(obj) {
	var details = {};
	var fields = {};
	var doNotify = false;
	if (obj.name != null && this.name != obj.name && obj.id == this.id) {
		details.oldPath = this.getPath();
		this.name = obj.name;
		fields[ZmOrganizer.F_NAME] = true;
		this.parent.children.sort(eval(ZmTreeView.COMPARE_FUNC[this.type]));
		doNotify = true;
		obj.name = null;
	}
	if (doNotify) {
		details.fields = fields;
		this._notify(ZmEvent.E_MODIFY, details);
	}

	if (obj.l != null && (!this.parent || (obj.l != this.parent.id))) {
		var newParent = this._getNewParent(obj.l);
		if (newParent) {
			details.oldPath = this.getPath();
			this.reparent(newParent);
			this._notify(ZmEvent.E_MOVE, details);
			obj.l = null;
		}
	}

	ZmOrganizer.prototype.notifyModify.apply(this, [obj]);
};

/**
 * Creates a query.
 * 
 * @param	{Boolean}	pathOnly	<code>true</code> if to use the path only
 * @return	{String}	the query
 */
ZmFolder.prototype.createQuery =
function(pathOnly) {
	if (!this.isRemote() && this.isSystem()) {
		var qName = ZmFolder.QUERY_NAME[this.nId] || this.getName(false, null, true, true) || this.name;
		// put quotes around folder names that consist of multiple words or have special characters.
		var quote = /^\w+$/.test(qName) ? "" : "\"";
		return pathOnly
			? qName
			: ("in:" + quote + qName + quote);
	}

	var path = this.isSystem() ? ZmFolder.QUERY_NAME[this.nId] : this.name;
	var f = this.parent;
	while (f && (f.nId != ZmFolder.ID_ROOT) && f.name.length) {
		var name = (f.isSystem() && ZmFolder.QUERY_NAME[f.nId]) || f.name;
		path = name + "/" + path;
		f = f.parent;
	}
	path = '"' + path + '"';
	return pathOnly ? path : ("in:" + path);
};

/**
 * Gets the name.
 * 
 * @param	{Boolean}	 showUnread		<code>true</code> to show unread
 * @param	{int}		maxLength		the max length
 * @param	{Boolean}	noMarkup		<code>true</code> to not include markup
 * @param	{Boolean}	useSystemName	<code>true</code> to use the system name
 * 
 * @return	{String}	the name
 */
ZmFolder.prototype.getName =
function(showUnread, maxLength, noMarkup, useSystemName) {
	if (this.nId == ZmFolder.ID_DRAFTS ||
		this.nId == ZmFolder.ID_OUTBOX ||
		this.rid == ZmFolder.ID_DRAFTS)
	{
		var name = (useSystemName && this._systemName) ? this._systemName : this.name;
		if (showUnread && this.numTotal > 0) {
			name = AjxMessageFormat.format(ZmMsg.folderUnread, [name, this.numTotal]);
			if (!noMarkup) {
				name = ["<span style='font-weight:bold'>", name, "</span>"].join("");
			}
		}
		return name;
	}
	else {
		return ZmOrganizer.prototype.getName.apply(this, arguments);
	}
};

/**
 * Gets the icon.
 * 
 * @return	{String}	the icon
 */
ZmFolder.prototype.getIcon =
function() {
	if (this.nId == ZmOrganizer.ID_ROOT)			{ return null; }
	if (ZmFolder.ICON[this.nId])					{ return ZmFolder.ICON[this.nId]; }
	if (this.isFeed())								{ return "RSS"; }
	if (this.isRemote())							{ return "SharedMailFolder"; }
	if (this.isDataSource(ZmAccount.TYPE_POP))		{ return "POPAccount"; }

	// make a "best-effort" to map imap folders to a well-known icon
	// (parent will be the root imap folder)
	var mappedId = this.getSystemEquivalentFolderId();
	if (mappedId) {
		return ZmFolder.ICON[mappedId] || "Folder";
	}

	return "Folder";
};

ZmFolder.prototype.getSystemEquivalentFolderId =
function() {
	if (this.parent && this.parent.isDataSource(ZmAccount.TYPE_IMAP)) {
		return ZmFolder.getIdForName(this.name);
	}
	return null;
};

ZmFolder.prototype.isSystemEquivalent =
function() {
	return this.getSystemEquivalentFolderId() != null;
};

ZmFolder.prototype.mayContainFolderFromAccount =
function(otherAccount) {
	var thisAccount = this.getAccount();
	if (thisAccount == otherAccount) {
		return true;
	}
	return thisAccount.isLocal(); // can only move to local

};

/**
 * Returns true if the given object(s) may be placed in this folder.
 *
 * If the object is a folder, check that:
 * <ul>
 * <li>We are not the immediate parent of the folder</li>
 * <li>We are not a child of the folder</li>
 * <li>We are not Spam or Drafts</li>
 * <li>We don't already have a child with the folder's name (unless we are in Trash)</li>
 * <li>We are not moving it into a folder of a different type</li>
 * <li>We are not moving a folder into itself</li>
 * </ul>
 *
 * If the object is an item or a list or items, check that:
 * <ul>
 * <li>We are not the Folders container</li>
 * <li>We are not a search folder</li>
 * <li>The items aren't already in this folder</li>
 * <li>A contact can only be moved to Trash</li>
 * <li> A draft can be moved to Trash or Drafts</li>
 * <li>Non-drafts cannot be moved to Drafts</li>
 * </ul>
 *
 * @param {Object}	what		the object(s) to possibly move into this folder (item or organizer)
 * @param {constant}	folderType	the contextual folder type (for tree view root items)
 * @param {boolean}	ignoreExisting  Set to true if checks for item presence in the folder should be skipped (e.g. when recovering deleted items)
 */
ZmFolder.prototype.mayContain =
function(what, folderType, ignoreExisting) {

	if (!what) {
		return true;
	}
	if (this.isFeed() /*|| this.isSyncIssuesFolder()*/) {
		return false;
	}
	// placeholder for showing a large number of folders
	if (this.id == ZmFolder.ID_LOAD_FOLDERS) {
		return false;
	}

	var thisType = folderType || this.type;
	var invalid = false;
	if (what instanceof ZmFolder) {
        invalid = ((what.parent === this && !ignoreExisting) || this.isChildOf(what) || this.nId == ZmFolder.ID_DRAFTS || this.nId == ZmFolder.ID_SPAM ||
				   (!this.isInTrash() && this.hasChild(what.name) && !ignoreExisting) ||
	               (what.type !== thisType && this.nId != ZmFolder.ID_TRASH) ||
				   (what.id === this.id) ||
				   (this.disallowSubFolder) ||
				   (appCtxt.multiAccounts && !this.mayContainFolderFromAccount(what.getAccount())) || // cannot move folders across accounts, unless the target is local
                   (this.isRemote() && !this._remoteMoveOk(what)) ||
				   (what.isRemote() && !this._remoteMoveOk(what)));				// a remote folder can be DnD but not its children
    } else {
		// An item or an array of items is being moved
		var items = AjxUtil.toArray(what);
		var item = items[0];

        // container can only have folders/searches or calendars
		if ((this.nId == ZmOrganizer.ID_ROOT && (what.type !== ZmOrganizer.CALENDAR)) ||
             // nothing can be moved to outbox/sync failures folders
			 this.nId == ZmOrganizer.ID_OUTBOX ||
			 this.nId == ZmOrganizer.ID_SYNC_FAILURES)
		{
			invalid = true;
		} else if (thisType === ZmOrganizer.SEARCH) {
			invalid = true;														// can't drop items into saved searches
		} else if (item && (item.type === ZmItem.CONTACT) && item.isGal) {
			invalid = true;
		} else if (item && (item.type === ZmItem.CONV) && item.list && item.list.search && (item.list.search.folderId === this.id)) {
			invalid = true;														// convs which are a result of a search for this folder
		} else {																// checks that need to be done for each item
			for (var i = 0; i < items.length; i++) {
				var childItem = items[i];
				if (!childItem) {
					invalid = true;
					break;
				}
				if (Dwt.instanceOf(childItem, "ZmBriefcaseFolderItem")) {
                     if (childItem.folder && childItem.folder.isRemote() && !childItem.folder.rid) {
                        invalid = true;
                        break;
                     }
                } else if (item.type === ZmItem.MSG && childItem.isDraft && (this.nId != ZmFolder.ID_TRASH && this.nId != ZmFolder.ID_DRAFTS && this.rid != ZmFolder.ID_DRAFTS)) {
					// can move drafts only into Trash or Drafts
					invalid = true;
					break;
				} else if ((this.nId == ZmFolder.ID_DRAFTS || this.rid == ZmFolder.ID_DRAFTS) && !childItem.isDraft)	{
					// only drafts can be moved into Drafts
					invalid = true;
					break;
				}
			}
			// items in the "Sync Failures" folder cannot be dragged out
			if (appCtxt.isOffline && !invalid) {
				// bug: 41531 - don't allow items to be moved into exchange
				// account when moving across accounts
				var acct = this.getAccount();
				if (acct && item.getAccount() != acct &&
					(acct.type === ZmAccount.TYPE_MSE ||
					 acct.type === ZmAccount.TYPE_EXCHANGE))
				{
					invalid = true;
				}
				else {
					var cs = appCtxt.getCurrentSearch();
					var folder = cs && appCtxt.getById(cs.folderId);
					if (folder && folder.nId == ZmOrganizer.ID_SYNC_FAILURES) {
						invalid = true;
					}
				}
			}

			// bug #42890 - disable moving to shared folders across accounts
			// until server bug is fixed
			if (appCtxt.multiAccounts && this.isRemote() &&
				what.getAccount && this.getAccount().id != what.getAccount().id)
			{
				invalid = true;
			}

			// can't move items to folder they're already in; we're okay if we
			// have one item from another folder
			if (!invalid && !ignoreExisting) {
				if (item && item.folderId) {
					invalid = true;
					for (var i = 0; i < items.length; i++) {
						if (items[i].folderId != this.id) {
							invalid = false;
							break;
						}
					}
				}
			}
		}
		if (!invalid && this.link) {
			invalid = this.isReadOnly();										// cannot drop anything onto a read-only item
		}
	}
	return !invalid;
};

/**
 * Checks if this is the sync issues folder.
 * 
 * @return	{Boolean}	<code>true</code> if the folder is the one dealing with Outlook sync issues
 */

//Bug#68799 Removing special handling of the folder named "Sync Issues"

/*ZmFolder.prototype.isSyncIssuesFolder =
function() {
	return (this.name == ZmFolder.SYNC_ISSUES);
};*/

/**
 * Checks if this folder required hard delete.
 * 
 * @return	{Boolean}	<code>true</code> if deleting items w/in this folder should be hard deleted.
 */
ZmFolder.prototype.isHardDelete =
function() {
	return (this.isInTrash() || this.isInSpam() || (appCtxt.isOffline && this.isUnder(ZmOrganizer.ID_SYNC_FAILURES)));
};

/**
 * Checks if this folder is in spam folder.
 * 
 * @return	{Boolean}	<code>true</code> if in spam
 */
ZmFolder.prototype.isInSpam =
function(){
	return this.isUnder(ZmFolder.ID_SPAM);
};

/**
 *
 * @param {ZmFolder}	folder  the source folder
 * 
 * @return {Boolean}	<code>true/code> if the given remote folder can be moved into this remote folder.
 * The source and the target folder must belong to the same account. The source
 * must have delete permission and the target must have insert permission.
 * 
 * @private
 */
ZmFolder.prototype._remoteMoveOk =
function(folder) {
	if (!this.isRemote() && folder.isMountpoint && folder.rid) { return true; }
	if (!this.link || !folder.link || this.getOwner() !== folder.getOwner()) { return false; }
	if (!this._folderActionOk(this, "isInsert")) {
		return false;
	}
	return this._folderActionOk(folder, "isDelete");
};

ZmFolder.prototype._folderActionOk =
function(folder, func) {
	var share = folder.shares && folder.shares[0];
	if (!share) {
		//if shares is not set, default to readOnly.
		return !folder.isReadOnly();
	}
	return share[func]();
};
/**
 * Returns true if this folder is for outbound mail.
 */
ZmFolder.prototype.isOutbound =
function() {
	for (var i = 0; i < ZmFolder.OUTBOUND.length; i++) {
		if (this.isUnder(ZmFolder.OUTBOUND[i])) {
			return true;
		}
	}
	return false;
};


/**
 * Sets the Global Mark Read flag.  When the user sets this flag, read flags are global for all
 * shared instances of the folder. When not set, each user accessing the shared folder will maintain
 * their own read/unread flag.
 *
 * @param	{Object}	        globalMarkRead		the globalMarkRead boolean flag
 * @param	{AjxCallback}	    callback		    the callback
 * @param	{AjxCallback}	    errorCallback		the error callback
 * @param   {ZmBatchCommand}    batchCmd            optional batch command
 */
ZmOrganizer.prototype.setGlobalMarkRead = function(globalMarkRead, callback, errorCallback, batchCmd) {
	if (this.globalMarkRead == globalMarkRead) { return; }
    // TODO: Bug 59559, awaiting server side implementation (Bug 24567)
    // TODO: - For ZmFolderPropsDialog and ZmSharePropsDialog:
    // TODO:     Make sure that the attrName is indeed globalMarkRead - used in the dialogs
    // TODO:     Make the globalMarkRead labels and controls visible.
    // TODO: - Uncomment this once the server call is ready, make sure action/attrs are correct
	//this._organizerAction({action: "globalMarkRead", attrs: {globalMarkRead: globalMarkRead}, callback: callback,
    //                       errorCallback: errorCallback, batchCmd: batchCmd});
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmSearch")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * The file defines a search class.
 * 
 */

/**
 * Creates a new search with the given properties.
 * @class
 * This class represents a search to be performed on the server. It has properties for
 * the different search parameters that may be used. It can be used for a regular search,
 * or to search within a conversation. The results are returned via a callback.
 *
 * @param {Hash}		params		a hash of parameters
 * @param   {String}	params.query					the query string
 * @param	{String}	params.queryHint				the query string that gets appended to the query but not something the user needs to know about
 * @param	{AjxVector}	params.types					the item types to search for
 * @param	{Boolean}	params.forceTypes				use the types we pass, do not override (in case of mail) to the current user's view pref (MSG vs. CONV).
 * @param	{constant}	params.sortBy					the sort order
 * @param	{int}		params.offset					the starting point within result set
 * @param	{int}		params.limit					the number of results to return
 * @param	{Boolean}	params.getHtml					if <code>true</code>, return HTML part for inlined msg
 * @param	{constant}	params.contactSource			where to search for contacts (GAL or personal)
 * @param	{Boolean}	params.isGalAutocompleteSearch	if <code>true</code>, autocomplete against GAL
 * @param	{constant}	params.galType					the type of GAL autocomplete (account or resource)
 * @param	{constant}	params.autocompleteType			the type of autocomplete (account or resource or all)
 * @param	{int}		params.lastId					the ID of last item displayed (for pagination)
 * @param	{String}	params.lastSortVal				the value of sort field for above item
 * @param	{Boolean}	params.fetch					if <code>true</code>, fetch first hit message
 * @param	{int}		params.searchId					the ID of owning search folder (if any)
 * @param	{Array}		params.conds					the list of search conditions (<code><SearchCalendarResourcesRequest></code>)
 * @param	{Array}		params.attrs					the list of attributes to return (<code><SearchCalendarResourcesRequest></code>)
 * @param	{String}	params.field					the field to search within (instead of default)
 * @param	{Object}	params.soapInfo					the object with method, namespace, response, and additional attribute fields for creating soap doc
 * @param	{Object}	params.response					the canned JSON response (no request will be made)
 * @param	{Array}		params.folders					the list of folders for autocomplete
 * @param	{Array}		params.allowableTaskStatus		the list of task status types to return (assuming one of the values for "types" is "task")
 * @param	{String}	params.accountName				the account name to run this search against
 * @param	{Boolean}	params.idsOnly					if <code>true</code>, response returns item IDs only
 * @param   {Boolean}   params.inDumpster               if <code>true</code>, search in the dumpster
 * @param	{string}	params.origin					indicates what initiated the search
 * @param	{boolean}	params.isEmpty					if true, return empty response without sending a request
 */
ZmSearch = function(params) {

	params = params || {};
	for (var p in params) {
		this[p] = params[p];
	}
	this.galType					= this.galType || ZmSearch.GAL_ACCOUNT;
	this.join						= this.join || ZmSearch.JOIN_AND;

	if (this.query || this.queryHint) {
		// only parse regular searches
		if (!this.isGalSearch && !this.isAutocompleteSearch &&
			!this.isGalAutocompleteSearch && !this.isCalResSearch) {
			
			var pq = this.parsedQuery = new ZmParsedQuery(this.query || this.queryHint);
			this._setProperties();
			var sortTerm = pq.getTerm("sort");
			if (sortTerm) {
				this.sortBy = sortTerm.arg;
			}
		}
	}

	this.isGalSearch = false;
	this.isCalResSearch = false;

	if (ZmSearch._mailEnabled == null) {
		ZmSearch._mailEnabled = appCtxt.get(ZmSetting.MAIL_ENABLED);
		if (ZmSearch._mailEnabled) {
			AjxDispatcher.require("MailCore");
		}
	}

	if (params.checkTypes) {
		var types = AjxUtil.toArray(this.types);
		var enabledTypes = [];
		for (var i = 0; i < types.length; i++) {
			var type = types[i];
			var app = ZmItem.APP[type];
			if (appCtxt.get(ZmApp.SETTING[app])) {
				enabledTypes.push(type);
			}
		}
		this.types = AjxVector.fromArray(enabledTypes);
	}
};

ZmSearch.prototype.isZmSearch = true;
ZmSearch.prototype.toString = function() { return "ZmSearch"; };

// Search types
ZmSearch.TYPE = {};
ZmSearch.TYPE_ANY = "any";

ZmSearch.GAL_ACCOUNT	= "account";
ZmSearch.GAL_RESOURCE	= "resource";
ZmSearch.GAL_ALL		= "";

ZmSearch.JOIN_AND	= 1;
ZmSearch.JOIN_OR	= 2;

ZmSearch.TYPE_MAP = {};

ZmSearch.DEFAULT_LIMIT = DwtListView.DEFAULT_LIMIT;

// Sort By
ZmSearch.DATE_DESC 		= "dateDesc";
ZmSearch.DATE_ASC 		= "dateAsc";
ZmSearch.SUBJ_DESC 		= "subjDesc";
ZmSearch.SUBJ_ASC 		= "subjAsc";
ZmSearch.NAME_DESC 		= "nameDesc";
ZmSearch.NAME_ASC 		= "nameAsc";
ZmSearch.SIZE_DESC 		= "sizeDesc";
ZmSearch.SIZE_ASC 		= "sizeAsc";
ZmSearch.RCPT_ASC       = "rcptAsc";
ZmSearch.RCPT_DESC      = "rcptDesc";
ZmSearch.ATTACH_ASC     = "attachAsc"
ZmSearch.ATTACH_DESC    = "attachDesc"
ZmSearch.FLAG_ASC       = "flagAsc";
ZmSearch.FLAG_DESC      = "flagDesc";
ZmSearch.MUTE_ASC       = "muteAsc";
ZmSearch.MUTE_DESC      = "muteDesc";
ZmSearch.READ_ASC       = "readAsc";
ZmSearch.READ_DESC      = "readDesc";
ZmSearch.PRIORITY_ASC   = "priorityAsc";
ZmSearch.PRIORITY_DESC  = "priorityDesc";
ZmSearch.SCORE_DESC 	= "scoreDesc";
ZmSearch.DURATION_DESC	= "durDesc";
ZmSearch.DURATION_ASC	= "durAsc";
ZmSearch.STATUS_DESC	= "taskStatusDesc";
ZmSearch.STATUS_ASC		= "taskStatusAsc";
ZmSearch.PCOMPLETE_DESC	= "taskPercCompletedDesc";
ZmSearch.PCOMPLETE_ASC	= "taskPercCompletedAsc";
ZmSearch.DUE_DATE_DESC	= "taskDueDesc";
ZmSearch.DUE_DATE_ASC	= "taskDueAsc";



ZmSearch.prototype.execute =
function(params) {
	if (params.batchCmd || this.soapInfo) {
		return this._executeSoap(params);
	} else {
		return this._executeJson(params);
	}
};

/**
 * Creates a SOAP request that represents this search and sends it to the server.
 *
 * @param {Hash}	params		a hash of parameters
 * @param {AjxCallback}	params.callback		the callback to run when response is received
 * @param {AjxCallback}	params.errorCallback	the callback to run if there is an exception
 * @param {ZmBatchCommand}	params.batchCmd		the batch command that contains this request
 * @param {int}	params.timeout		the timeout value (in seconds)
 * @param {Boolean}	params.noBusyOverlay	if <code>true</code>, don't use the busy overlay
 * 
 * @private
 */
ZmSearch.prototype._executeSoap =
function(params) {

	this.isGalSearch = (this.contactSource && (this.contactSource == ZmId.SEARCH_GAL));
	this.isCalResSearch = (!this.contactSource && this.conds != null);
    if (appCtxt.isOffline && this.isCalResSearch) {
        this.isCalResSearch =  appCtxt.isZDOnline();
    }
	if (this.isEmpty) {
		this._handleResponseExecute(params.callback);
		return null;
	}

	var soapDoc;
	if (!this.response) {
		if (this.isGalSearch) {
			// DEPRECATED: Use JSON version
			soapDoc = AjxSoapDoc.create("SearchGalRequest", "urn:zimbraAccount");
			var method = soapDoc.getMethod();
			if (this.galType) {
				method.setAttribute("type", this.galType);
			}
			soapDoc.set("name", this.query);
			var searchFilterEl = soapDoc.set("searchFilter");
			if (this.conds && this.conds.length) {
				var condsEl = soapDoc.set("conds", null, searchFilterEl);
				this._applySoapCond(this.conds, soapDoc, condsEl);
			}
		} else if (this.isAutocompleteSearch) {
			soapDoc = AjxSoapDoc.create("AutoCompleteRequest", "urn:zimbraMail");
			var method = soapDoc.getMethod();
			if (this.limit) {
				method.setAttribute("limit", this.limit);
			}
			soapDoc.set("name", this.query);
		} else if (this.isGalAutocompleteSearch) {
			soapDoc = AjxSoapDoc.create("AutoCompleteGalRequest", "urn:zimbraAccount");
			var method = soapDoc.getMethod();
			method.setAttribute("limit", this._getLimit());
			if (this.galType) {
				method.setAttribute("type", this.galType);
			}
			soapDoc.set("name", this.query);
		} else if (this.isCalResSearch) {
			soapDoc = AjxSoapDoc.create("SearchCalendarResourcesRequest", "urn:zimbraAccount");
			var method = soapDoc.getMethod();
			if (this.attrs) {
				var attrs = [].concat(this.attrs);
				AjxUtil.arrayRemove(attrs, "fullName");
				method.setAttribute("attrs", attrs.join(","));
			}
			var searchFilterEl = soapDoc.set("searchFilter");
			if (this.conds && this.conds.length) {
				var condsEl = soapDoc.set("conds", null, searchFilterEl);
				this._applySoapCond(this.conds, soapDoc, condsEl);
			}
		} else {
			if (this.soapInfo) {
				soapDoc = AjxSoapDoc.create(this.soapInfo.method, this.soapInfo.namespace);
				// Pass along any extra soap data. (Voice searches use this to pass user identification.)
				for (var nodeName in this.soapInfo.additional) {
					var node = soapDoc.set(nodeName);
					var attrs = this.soapInfo.additional[nodeName];
					for (var attr in attrs) {
						node.setAttribute(attr, attrs[attr]);
					}
				}
			} else {
				soapDoc = AjxSoapDoc.create("SearchRequest", "urn:zimbraMail");
			}
			var method = this._getStandardMethod(soapDoc);
			if (this.types) {
				var a = this.types.getArray();
				if (a.length) {
					var typeStr = [];
					for (var i = 0; i < a.length; i++) {
						typeStr.push(ZmSearch.TYPE[a[i]]);
					}
					method.setAttribute("types", typeStr.join(","));
					if (this.types.contains(ZmItem.MSG) || this.types.contains(ZmItem.CONV)) {
						// special handling for showing participants ("To" instead of "From")
						var folder = this.folderId && appCtxt.getById(this.folderId);
						method.setAttribute("recip", (folder && folder.isOutbound()) ? "1" : "0");
					}
					if (this.types.contains(ZmItem.CONV)) {
						// get ID/folder for every msg in each conv result
						method.setAttribute("fullConversation", 1);
					}
					// if we're prefetching the first hit message, also mark it as read
					if (this.fetch) {

						method.setAttribute("fetch", ( this.fetch == "all" ) ? "all" : "1");
						// and set the html flag if we want the html version
						if (this.getHtml) {
							method.setAttribute("html", "1");
						}
					}
					if (this.markRead) {
						method.setAttribute("read", "1");
					}
				}
			}
			if (this.inDumpster) {
				method.setAttribute("inDumpster", "1");
			}
		}
	}

	var soapMethod = this._getStandardMethod(soapDoc);
	soapMethod.setAttribute("needExp", 1);

	var respCallback = this._handleResponseExecute.bind(this, params.callback);

	if (params.batchCmd) {
		params.batchCmd.addRequestParams(soapDoc, respCallback);
	} else {
		return appCtxt.getAppController().sendRequest({soapDoc:soapDoc, asyncMode:true, callback:respCallback,
													   errorCallback:params.errorCallback,
													   timeout:params.timeout, noBusyOverlay:params.noBusyOverlay,
													   response:this.response});
	}
};

/**
 * Creates a JSON request that represents this search and sends it to the server.
 *
 * @param {Hash}	params		a hash of parameters
 * @param {AjxCallback}	params.callback		the callback to run when response is received
 * @param {AjxCallback}	params.errorCallback	the callback to run if there is an exception
 * @param {ZmBatchCommand}	params.batchCmd		the batch command that contains this request
 * @param {int}	params.timeout		the timeout value (in seconds)
 * @param {Boolean}	params.noBusyOverlay	if <code>true</code>, don't use the busy overlay
 * 
 * @private
 */
ZmSearch.prototype._executeJson =
function(params) {

	this.isGalSearch = (this.contactSource && (this.contactSource == ZmId.SEARCH_GAL));
	this.isCalResSearch = (!this.contactSource && this.conds != null);
    if (appCtxt.isOffline && this.isCalResSearch) {
        this.isCalResSearch = appCtxt.isZDOnline();
    }
	if (this.isEmpty) {
		this._handleResponseExecute(params.callback);
		return null;
	}

	var jsonObj, request, soapDoc;
	if (!this.response) {
		if (this.isGalSearch) {
			request = {
				_jsns:"urn:zimbraAccount",
				needIsOwner: "1",
				needIsMember: "directOnly"
			};
			jsonObj = {SearchGalRequest: request};
			if (this.galType) {
				request.type = this.galType;
			}
			request.name = this.query;

			// bug #36188 - add offset/limit for paging support
			request.offset = this.offset = (this.offset || 0);
			request.limit = this._getLimit();

			// bug 15878: see same in ZmSearch.prototype._getStandardMethodJson
			request.locale = { _content: AjxEnv.DEFAULT_LOCALE };

			if (this.lastId) { // add lastSortVal and lastId for cursor-based paging
				request.cursor = {id:this.lastId, sortVal:(this.lastSortVal || "")};
			}
			if (this.sortBy) {
				request.sortBy = this.sortBy;
			}
			if (this.conds && this.conds.length) {
				request.searchFilter = {conds:{}};
				request.searchFilter.conds = ZmSearch.prototype._applyJsonCond(this.conds, request);
			}
		} else if (this.isAutocompleteSearch) {
			jsonObj = {AutoCompleteRequest:{_jsns:"urn:zimbraMail"}};
			request = jsonObj.AutoCompleteRequest;
			if (this.limit) {
				request.limit = this.limit;
			}
			request.name = {_content:this.query};
			if (params.autocompleteType) {
				request.t = params.autocompleteType;
			}
		} else if (this.isGalAutocompleteSearch) {
			jsonObj = {AutoCompleteGalRequest:{_jsns:"urn:zimbraAccount"}};
			request = jsonObj.AutoCompleteGalRequest;
			request.limit = this._getLimit();
			request.name = this.query;
			if (this.galType) {
				request.type = this.galType;
			}
		} else if (this.isCalResSearch) {
			jsonObj = {SearchCalendarResourcesRequest:{_jsns:"urn:zimbraAccount"}};
			request = jsonObj.SearchCalendarResourcesRequest;
			if (this.attrs) {
				var attrs = [].concat(this.attrs);
				request.attrs = attrs.join(",");
			}
            request.offset = this.offset = (this.offset || 0);
            request.limit = this._getLimit();
			if (this.conds && this.conds.length) {
				request.searchFilter = {conds:{}};
				request.searchFilter.conds = ZmSearch.prototype._applyJsonCond(this.conds, request);
			}
		} else {
			if (this.soapInfo) {
				soapDoc = AjxSoapDoc.create(this.soapInfo.method, this.soapInfo.namespace);
				// Pass along any extra soap data. (Voice searches use this to pass user identification.)
				for (var nodeName in this.soapInfo.additional) {
					var node = soapDoc.set(nodeName);
					var attrs = this.soapInfo.additional[nodeName];
					for (var attr in attrs) {
						node.setAttribute(attr, attrs[attr]);
					}
				}
			} else {
				jsonObj = {SearchRequest:{_jsns:"urn:zimbraMail"}};
				request = jsonObj.SearchRequest;
			}
			this._getStandardMethodJson(request);
			if (this.types) {
				var a = this.types.getArray();
				if (a.length) {
					var typeStr = [];
					for (var i = 0; i < a.length; i++) {
						typeStr.push(ZmSearch.TYPE[a[i]]);
					}
					request.types = typeStr.join(",");

					if (this.types.contains(ZmItem.MSG) || this.types.contains(ZmItem.CONV)) {
						// special handling for showing participants ("To" instead of "From")
						var folder = this.folderId && appCtxt.getById(this.folderId);
						request.recip = (folder && folder.isOutbound()) ? "2" : "0";
					}

					if (this.types.contains(ZmItem.CONV)) {
						// get ID/folder for every msg in each conv result
						request.fullConversation = 1;
					}

					// if we're prefetching the first hit message, also mark it as read
					if (this.fetch) {
                        request.fetch = ( this.fetch == "all" ) ? "all" : 1;
						// and set the html flag if we want the html version
						if (this.getHtml) {
							request.html = 1;
						}
					}

					if (this.markRead) {
						request.read = 1;
					}

                    if (this.headers) {
                        for (var hdr in this.headers) {
                            if (!request.header) { request.header = []; }
                            request.header.push({n: this.headers[hdr]});
                        }
                    }

					if (a.length == 1 && a[0] == ZmItem.TASK && this.allowableTaskStatus) {
						request.allowableTaskStatus = this.allowableTaskStatus;
					}
                }
            }
			if (this.inDumpster) {
				request.inDumpster = 1;
			}
        }
    }

	if (request) {
		request.needExp = 1;
	}


	var respCallback = this._handleResponseExecute.bind(this, params.callback);

	if (params.batchCmd) {
		params.batchCmd.addRequestParams(soapDoc, respCallback);
	} else {
		var searchParams = {
			jsonObj:jsonObj,
			soapDoc:soapDoc,
			asyncMode:true,
			callback:respCallback,
			errorCallback:params.errorCallback,
            offlineCallback:params.offlineCallback,
			timeout:params.timeout,
            offlineCache:params.offlineCache,
			noBusyOverlay:params.noBusyOverlay,
			response:this.response,
			accountName:this.accountName,
            offlineRequest:params.offlineRequest
		};
		return appCtxt.getAppController().sendRequest(searchParams);
	}
};

ZmSearch.prototype._applySoapCond =
function(inConds, soapDoc, condsEl, or) {
	if (or || this.join == ZmSearch.JOIN_OR) {
		condsEl.setAttribute("or", 1);
	}
	for (var i = 0; i < inConds.length; i++) {
		var c = inConds[i];
		if (AjxUtil.isArray(c)) {
			var subCondsEl = soapDoc.set("conds", null, condsEl);
			this._applySoapCond(c, soapDoc, subCondsEl, true);
		} else if (c.attr=="fullName" && c.op=="has") {
			var nameEl = soapDoc.set("name", c.value);
		} else {
			var condEl = soapDoc.set("cond", null, condsEl);
			condEl.setAttribute("attr", c.attr);
			condEl.setAttribute("op", c.op);
			condEl.setAttribute("value", c.value);
		}
	}
};

ZmSearch.prototype._applyJsonCond =
function(inConds, request, or) {
	var outConds = {};
	if (or || this.join == ZmSearch.JOIN_OR) {
		outConds.or = 1;
	}

	for (var i = 0; i < inConds.length; i++) {
		var c = inConds[i];
		if (AjxUtil.isArray(c)) {
			if (!outConds.conds)
				outConds.conds = [];
			outConds.conds.push(this._applyJsonCond(c, request, true));
		} else if (c.attr=="fullName" && c.op=="has") {
			request.name = {_content: c.value};
		} else {
			if (!outConds.cond)
				outConds.cond = [];
			outConds.cond.push({attr:c.attr, op:c.op, value:c.value});
		}
	}
	return outConds;
};

/**
 * Converts the response into a {ZmSearchResult} and passes it along.
 * 
 * @private
 */
ZmSearch.prototype._handleResponseExecute =
function(callback, result) {
	
	if (result) {
		var response = result.getResponse();
	
		if      (this.isGalSearch)				{ response = response.SearchGalResponse; }
		else if (this.isCalResSearch)			{ response = response.SearchCalendarResourcesResponse; }
		else if (this.isAutocompleteSearch)		{ response = response.AutoCompleteResponse; }
		else if (this.isGalAutocompleteSearch)	{ response = response.AutoCompleteGalResponse; }
		else if (this.soapInfo)					{ response = response[this.soapInfo.response]; }
		else									{ response = response.SearchResponse; }
	}
	else {
		response = { _jsns: "urn:zimbraMail", more: false };
	}
	var searchResult = new ZmSearchResult(this);
	searchResult.set(response);
	result = result || new ZmCsfeResult();
	result.set(searchResult);

	if (callback) {
		callback.run(result);
	}
};

/**
 * Fetches a conversation from the server.
 *
 * @param {Hash}		params				a hash of parameters:
 * @param {String}		params.cid			the conv ID
 * @param {AjxCallback}	params.callback		the callback to run with result
 * @param {String}		params.fetch		which msg bodies to load (see soap.txt)
 * @param {Boolean}		params.markRead		if <code>true</code>, mark msg read
 * @param {Boolean}		params.noTruncate	if <code>true</code>, do not limit size of msg
 * @param {boolean}		params.needExp		if not <code>false</code>, have server check if addresses are DLs
 */
ZmSearch.prototype.getConv =
function(params) {
	if ((!this.query && !this.queryHint) || !params.cid) { return; }

	var jsonObj = {SearchConvRequest:{_jsns:"urn:zimbraMail"}};
	var request = jsonObj.SearchConvRequest;
	this._getStandardMethodJson(request);
	request.cid = params.cid;
	if (params.fetch) {
		request.fetch = params.fetch;
		if (params.markRead) {
			request.read = 1;			// mark that msg read
		}
		if (this.getHtml) {
			request.html = 1;			// get it as HTML
		}
		if (params.needExp !== false) {
			request.needExp = 1;
		}
	}

	if (!params.noTruncate) {
		request.max = appCtxt.get(ZmSetting.MAX_MESSAGE_SIZE);
	}

	//get both TO and From
	request.recip =  "2";

	var searchParams = {
		jsonObj:		jsonObj,
		asyncMode:		true,
		callback:		this._handleResponseGetConv.bind(this, params.callback),
		accountName:	this.accountName
	};
	appCtxt.getAppController().sendRequest(searchParams);
};

/**
 * @private
 */
ZmSearch.prototype._handleResponseGetConv =
function(callback, result) {
	var response = result.getResponse().SearchConvResponse;
	var searchResult = new ZmSearchResult(this);
	searchResult.set(response, null, true);
	result.set(searchResult);

	if (callback) {
		callback.run(result);
	}
};

/**
 * Clears cursor-related fields from this search so that it will not create a cursor.
 */
ZmSearch.prototype.clearCursor =
function() {
	this.lastId = this.lastSortVal = this.endSortVal = null;
};

/**
 * Gets a title that summarizes this search.
 * 
 * @return	{String}	the title
 */
ZmSearch.prototype.getTitle =
function() {
	var where;
	var pq = this.parsedQuery;
	// if this is a saved search, show its name, otherwise show folder or tag name if it's the only term
	var orgId = this.searchId || ((pq && (pq.getNumTokens() == 1)) ? this.folderId || this.tagId : null);
	if (orgId) {
		var org = appCtxt.getById(ZmOrganizer.getSystemId(orgId));
		if (org) {
			where = org.getName(true, ZmOrganizer.MAX_DISPLAY_NAME_LENGTH, true);
		}
	}
	return where ? ([ZmMsg.zimbraTitle, where].join(": ")) : ([ZmMsg.zimbraTitle, ZmMsg.searchResults].join(": "));
};

/**
 * Checks if this search is multi-account.
 * 
 * @return	{Boolean}	<code>true</code> if multi-account
 */
ZmSearch.prototype.isMultiAccount =
function() {
	if (!this._isMultiAccount) {
		this._isMultiAccount = (this.queryHint && this.queryHint.length > 0 &&
								(this.queryHint.split("inid:").length > 1 ||
								 this.queryHint.split("underid:").length > 1));
	}
	return this._isMultiAccount;
};

/**
 * @private
 */
ZmSearch.prototype._getStandardMethod =
function(soapDoc) {

	var method = soapDoc.getMethod();

	if (this.sortBy) {
		method.setAttribute("sortBy", this.sortBy);
	}

	if (this.types.contains(ZmItem.MSG) || this.types.contains(ZmItem.CONV)) {
		ZmMailMsg.addRequestHeaders(soapDoc);
	}

	// bug 5771: add timezone and locale info
	ZmTimezone.set(soapDoc, AjxTimezone.DEFAULT, null);
	soapDoc.set("locale", appCtxt.get(ZmSetting.LOCALE_NAME), null);

	if (this.lastId != null && this.lastSortVal) {
		// cursor is used for paginated searches
		var cursor = soapDoc.set("cursor");
		cursor.setAttribute("id", this.lastId);
		cursor.setAttribute("sortVal", this.lastSortVal);
		if (this.endSortVal) {
			cursor.setAttribute("endSortVal", this.endSortVal);
		}
	}

	this.offset = this.offset || 0;
	method.setAttribute("offset", this.offset);

	// always set limit
	method.setAttribute("limit", this._getLimit());

	var query = this._getQuery();

	soapDoc.set("query", query);

	// set search field if provided
	if (this.field) {
		method.setAttribute("field", this.field);
	}

	return method;
};

/**
 * @private
 */
ZmSearch.prototype._getStandardMethodJson = 
function(req) {

	if (this.sortBy) {
		req.sortBy = this.sortBy;
	}

	if (this.types.contains(ZmItem.MSG) || this.types.contains(ZmItem.CONV)) {
		ZmMailMsg.addRequestHeaders(req);
	}

	// bug 5771: add timezone and locale info
	ZmTimezone.set(req, AjxTimezone.DEFAULT);
	// bug 15878: We can't use appCtxt.get(ZmSetting.LOCALE) because that
	//            will return the server's default locale if it is not set
	//            set for the user or their COS. But AjxEnv.DEFAULT_LOCALE
	//            is set to the browser's locale setting in the case when
	//            the user's (or their COS) locale is not set.
	req.locale = { _content: AjxEnv.DEFAULT_LOCALE };

	if (this.lastId != null && this.lastSortVal) {
		// cursor is used for paginated searches
		req.cursor = {id:this.lastId, sortVal:this.lastSortVal};
		if (this.endSortVal) {
			req.cursor.endSortVal = this.endSortVal;
		}
	}

	req.offset = this.offset = this.offset || 0;

	// always set limit
	req.limit = this._getLimit();

	if (this.idsOnly) {
		req.resultMode = "IDS";
	}

	req.query = this._getQuery();

	// set search field if provided
	if (this.field) {
		req.field = this.field;
	}
};

/**
 * @private
 */
ZmSearch.prototype._getQuery =
function() {
	// and of course, always set the query and append the query hint if applicable
	// only use query hint if this is not a "simple" search
	if (this.queryHint) {
		var query = this.query ? ["(", this.query, ") "].join("") : "";
		return [query, "(", this.queryHint, ")"].join("");
	}
	return this.query;
};

/**
 * @private
 */
ZmSearch.prototype._getLimit =
function() {

	if (this.limit) { return this.limit; }

	var limit;
	if (this.isGalAutocompleteSearch) {
		limit = appCtxt.get(ZmSetting.AUTOCOMPLETE_LIMIT);
	} else {
		var type = this.types && this.types.get(0);
		var app = appCtxt.getApp(ZmItem.APP[type]) || appCtxt.getCurrentApp();
		if (app && app.getLimit) {
			limit = app.getLimit(this.offset);
		} else {
			limit = appCtxt.get(ZmSetting.PAGE_SIZE) || ZmSearch.DEFAULT_LIMIT;
		}
	}

	this.limit = limit;
	return limit;
};

/**
 * Tests the given item against a matching function generated from the query.
 * 
 * @param {ZmItem}	item		an item
 * @return	true if the item matches, false if it doesn't, and null if a matching function could not be generated
 */
ZmSearch.prototype.matches =
function(item) {

	if (!this.parsedQuery) {
		return null;
	}

	// if search is constrained to a folder, we can return false if item is not in that folder
	if (this.folderId && !this.parsedQuery.hasOrTerm) {
		if (item.type === ZmItem.CONV) {
			if (item.folders && !item.folders[this.folderId]) {
				return false;
			}
		}
		else if (item.folderId && item.folderId !== this.folderId) {
			return false;
		}
	}

	var matchFunc = this.parsedQuery.getMatchFunction();
	return matchFunc ? matchFunc(item) : null;
};

/**
 * Returns true if the query has a folder-related term with the given value.
 * 
 * @param 	{string}	path		a folder path (optional)
 */
ZmSearch.prototype.hasFolderTerm =
function(path) {
	return this.parsedQuery && this.parsedQuery.hasTerm(["in", "under"], path);
};

/**
 * Replaces the old folder path with the new folder path in the query string, if found.
 * 
 * @param	{string}	oldPath		the old folder path
 * @param	{string}	newPath		the new folder path
 * 
 * @return	{boolean}	true if replacement was performed
 */
ZmSearch.prototype.replaceFolderTerm =
function(oldPath, newPath) {
	if (!this.parsedQuery) {
		return this.query;
	}
	var newQuery = this.parsedQuery.replaceTerm(["in", "under"], oldPath, newPath);
	if (newQuery) {
		this.query = newQuery;
	}
	return Boolean(newQuery);
};

/**
 * Returns true if the query has a tag term with the given value.
 * 
 * @param 	{string}	tagName		a tag name (optional)
 */
ZmSearch.prototype.hasTagTerm =
function(tagName) {
	return this.parsedQuery && this.parsedQuery.hasTerm("tag", tagName);
};

/**
 * Replaces the old tag name with the new tag name in the query string, if found.
 * 
 * @param	{string}	oldName		the old tag name
 * @param	{string}	newName		the new tag name
 * 
 * @return	{boolean}	true if replacement was performed
 */
ZmSearch.prototype.replaceTagTerm =
function(oldName, newName) {
	if (!this.parsedQuery) {
		return this.query;
	}
	var newQuery = this.parsedQuery.replaceTerm("tag", oldName, newName);
	if (newQuery) {
		this.query = newQuery;
	}
	return Boolean(newQuery);
};

/**
 * Returns true if the query has a term related to unread status.
 */
ZmSearch.prototype.hasUnreadTerm =
function() {
	return (this.parsedQuery && (this.parsedQuery.hasTerm("is", "read") ||
								 this.parsedQuery.hasTerm("is", "unread")));
};

/**
 * Returns true if the query has the term "is:anywhere".
 */
ZmSearch.prototype.isAnywhere =
function() {
	return (this.parsedQuery && this.parsedQuery.hasTerm("is", "anywhere"));
};

/**
 * Returns true if the query has a "content" term.
 */
ZmSearch.prototype.hasContentTerm =
function() {
	return (this.parsedQuery && this.parsedQuery.hasTerm("content"));
};

/**
 * Returns true if the query has just one term, and it's a folder or tag term.
 */
ZmSearch.prototype.isSimple =
function() {
	var pq = this.parsedQuery;
	if (pq && (pq.getNumTokens() == 1)) {
		return pq.hasTerm(["in", "inid", "tag"]);
	}
	return false;
};

ZmSearch.prototype.getTokens =
function() {
	return this.parsedQuery && this.parsedQuery.getTokens();
};

ZmSearch.prototype._setProperties =
function() {
	var props = this.parsedQuery && this.parsedQuery.getProperties();
	for (var key in props) {
		this[key] = props[key];
	}
};





/**
 * This class is a parsed representation of a query string. It parses the string into tokens.
 * A token is a paren, a conditional operator, or a search term (which has an operator and an
 * argument). The query string is assumed to be valid.
 * 
 * Compound terms such as "in:(inbox or sent)" will be exploded into multiple terms.
 * 
 * @param	{string}	query		a query string
 * 
 * TODO: handle "field[lastName]" and "#lastName"
 */
ZmParsedQuery = function(query) {

	this.hasOrTerm = false;
	this._tokens = this._parse(AjxStringUtil.trim(query, true));

	// preconditions for flags
	if (!ZmParsedQuery.IS_VALUE_PRECONDITION) {
		ZmParsedQuery.IS_VALUE_PRECONDITION = {};
		ZmParsedQuery.IS_VALUE_PRECONDITION['flagged']      = ZmSetting.FLAGGING_ENABLED;
		ZmParsedQuery.IS_VALUE_PRECONDITION['unflagged']    = ZmSetting.FLAGGING_ENABLED;
	}
};

ZmParsedQuery.prototype.isZmParsedQuery = true;
ZmParsedQuery.prototype.toString = function() { return "ZmParsedQuery"; };

ZmParsedQuery.TERM	= "TERM";	// search operator such as "in"
ZmParsedQuery.COND	= "COND";	// AND OR NOT
ZmParsedQuery.GROUP	= "GROUP";	// ( or )

ZmParsedQuery.OP_CONTENT	= "content";

ZmParsedQuery.OP_LIST = [
	"content", "subject", "msgid", "envto", "envfrom", "contact", "to", "from", "cc", "tofrom", 
	"tocc", "fromcc", "tofromcc", "in", "under", "inid", "underid", "has", "filename", "type", 
	"attachment", "is", "date", "mdate", "day", "week", "month", "year", "after", "before", 
	"size", "bigger", "larger", "smaller", "tag", "priority", "message", "my", "modseq", "conv", 
	"conv-count", "conv-minm", "conv-maxm", "conv-start", "conv-end", "appt-start", "appt-end", "author", "title", "keywords", 
	"company", "metadata", "item", "sort"
];
ZmParsedQuery.IS_OP		= AjxUtil.arrayAsHash(ZmParsedQuery.OP_LIST);

// valid arguments for the search term "is:"
ZmParsedQuery.IS_VALUES = [	"unread", "read", "flagged", "unflagged",
							"draft", "sent", "received", "replied", "unreplied", "forwarded", "unforwarded",
							"invite",
							"solo",
							"tome", "fromme", "ccme", "tofromme", "toccme", "fromccme", "tofromccme",
							"local", "remote", "anywhere" ];

// ops that can appear more than once in a query
ZmParsedQuery.MULTIPLE = {};
ZmParsedQuery.MULTIPLE["to"]			= true;
ZmParsedQuery.MULTIPLE["is"]			= true;
ZmParsedQuery.MULTIPLE["has"]			= true;
ZmParsedQuery.MULTIPLE["tag"]			= true;
ZmParsedQuery.MULTIPLE["appt-start"]	= true;
ZmParsedQuery.MULTIPLE["appt-end"]		= true;
ZmParsedQuery.MULTIPLE["type"]			= true;

ZmParsedQuery.isMultiple =
function(term) {
	return Boolean(term && ZmParsedQuery.MULTIPLE[term.op]);
};

// ops that are mutually exclusive
ZmParsedQuery.EXCLUDE = {};
ZmParsedQuery.EXCLUDE["before"]	= ["date"];
ZmParsedQuery.EXCLUDE["after"]	= ["date"];

// values that are mutually exclusive - list value implies full multi-way exclusivity
ZmParsedQuery.EXCLUDE["is"]					= {};
ZmParsedQuery.EXCLUDE["is"]["read"]			= ["unread"];
ZmParsedQuery.EXCLUDE["is"]["flagged"]		= ["unflagged"];
ZmParsedQuery.EXCLUDE["is"]["sent"]			= ["received"];
ZmParsedQuery.EXCLUDE["is"]["replied"]		= ["unreplied"];
ZmParsedQuery.EXCLUDE["is"]["forwarded"]	= ["unforwarded"];
ZmParsedQuery.EXCLUDE["is"]["local"]		= ["remote", "anywhere"];
ZmParsedQuery.EXCLUDE["is"]["tome"]			= ["tofromme", "toccme", "tofromccme"];
ZmParsedQuery.EXCLUDE["is"]["fromme"]		= ["tofromme", "fromccme", "tofromccme"];
ZmParsedQuery.EXCLUDE["is"]["ccme"]			= ["toccme", "fromccme", "tofromccme"];

ZmParsedQuery._createExcludeMap =
function(excludes) {

	var excludeMap = {};
	for (var key in excludes) {
		var value = excludes[key];
		if (AjxUtil.isArray1(value)) {
			value.push(key);
			ZmParsedQuery._permuteExcludeMap(excludeMap, value);
		}
		else {
			for (var key1 in value) {
				var value1 = excludes[key][key1];
				value1.push(key1);
				ZmParsedQuery._permuteExcludeMap(excludeMap, AjxUtil.map(value1,
						function(val) {
							return new ZmSearchToken(key, val).toString();
						}));
			}
		}
	}
	return excludeMap;
};

// makes each possible pair in the list exclusive
ZmParsedQuery._permuteExcludeMap =
function(excludeMap, list) {
	if (list.length < 2) { return; }
	for (var i = 0; i < list.length - 1; i++) {
		var a = list[i];
		for (var j = i + 1; j < list.length; j++) {
			var b = list[j];
			excludeMap[a] = excludeMap[a] || {};
			excludeMap[b] = excludeMap[b] || {};
			excludeMap[a][b] = true;
			excludeMap[b][a] = true;
		}
	}
};

/**
 * Returns true if the given search terms should not appear in the same query.
 * 
 * @param {ZmSearchToken}	termA	search term
 * @param {ZmSearchToken}	termB	search term
 */
ZmParsedQuery.areExclusive =
function(termA, termB) {
	if (!termA || !termB) { return false; }
	var map = ZmParsedQuery.EXCLUDE_MAP;
	if (!map) {
		map = ZmParsedQuery.EXCLUDE_MAP = ZmParsedQuery._createExcludeMap(ZmParsedQuery.EXCLUDE);
	}
	var opA = termA.op, opB = termB.op;
	var strA = termA.toString(), strB = termB.toString();
	return Boolean((map[opA] && map[opA][opB]) || (map[opB] && map[opB][opA]) ||
				   (map[strA] && map[strA][strB]) || (map[strB] && map[strB][strA]));
};

// conditional ops
ZmParsedQuery.COND_AND		= "and"
ZmParsedQuery.COND_OR		= "or";
ZmParsedQuery.COND_NOT		= "not";
ZmParsedQuery.GROUP_OPEN	= "(";
ZmParsedQuery.GROUP_CLOSE	= ")";

// JS version of conditional
ZmParsedQuery.COND_OP = {};
ZmParsedQuery.COND_OP[ZmParsedQuery.COND_AND]	= " && ";
ZmParsedQuery.COND_OP[ZmParsedQuery.COND_OR]	= " || ";
ZmParsedQuery.COND_OP[ZmParsedQuery.COND_NOT]	= " !";

// word separators
ZmParsedQuery.EOW_LIST	= [" ", ":", ZmParsedQuery.GROUP_OPEN, ZmParsedQuery.GROUP_CLOSE];
ZmParsedQuery.IS_EOW	= AjxUtil.arrayAsHash(ZmParsedQuery.EOW_LIST);

// map is:xxx to item properties
ZmParsedQuery.FLAG = {};
ZmParsedQuery.FLAG["unread"]		= "item.isUnread";
ZmParsedQuery.FLAG["read"]			= "!item.isUnread";
ZmParsedQuery.FLAG["flagged"]		= "item.isFlagged";
ZmParsedQuery.FLAG["unflagged"]		= "!item.isFlagged";
ZmParsedQuery.FLAG["forwarded"]		= "item.isForwarded";
ZmParsedQuery.FLAG["unforwarded"]	= "!item.isForwarded";
ZmParsedQuery.FLAG["sent"]			= "item.isSent";
ZmParsedQuery.FLAG["draft"]			= "item.isDraft";
ZmParsedQuery.FLAG["replied"]		= "item.isReplied";
ZmParsedQuery.FLAG["unreplied"]		= "!item.isReplied";

ZmParsedQuery.prototype._parse =
function(query) {

	function getQuotedStr(str, pos, q) {
		var q = q || str.charAt(pos);
		pos++;
		var done = false, ch, quoted = "";
		while (pos < str.length && !done) {
			ch = str.charAt(pos);
			if (ch == q) {
				done = true;
			} else {
				quoted += ch;
				pos++;
			}
		}

		return done ? {str:quoted, pos:pos + 1} : null;
	}
	
	function skipSpace(str, pos) {
		while (pos < str.length && str.charAt(pos) == " ") {
			pos++;
		}
		return pos;
	}
	
	function fail(reason, query) {
		DBG.println(AjxDebug.DBG1, "ZmParsedQuery failure: " + reason + "; query: [" + query + "]");
		this.parseFailed = reason;
		return tokens;		
	}

	var len = query.length;
	var tokens = [], ch, lastCh, op, word = "", isEow = false, endOk = true, compound = 0, numParens = 0;
	var pos = skipSpace(query, 0);
	while (pos < len) {
		lastCh = (ch != " ") ? ch : lastCh;
		ch = query.charAt(pos);
		isEow = ZmParsedQuery.IS_EOW[ch];

		if (ch == ":") {
			if (ZmParsedQuery.IS_OP[word]) {
				op = word;
			} else {
				return fail("unrecognized op '" + word + "'", query);
			}
			word = "";
			pos = skipSpace(query, pos + 1);
			continue;
		}

		if (isEow) {
			var lcWord = word.toLowerCase();
			var isCondOp = !!ZmParsedQuery.COND_OP[lcWord];
			if (op && word && !(isCondOp && compound > 0)) {
				tokens.push(new ZmSearchToken(op, lcWord));
				if (compound == 0) {
					op = "";
				}
				word = "";
				endOk = true;
			} else if (!op || (op && compound > 0)) {
				if (isCondOp) {
					tokens.push(new ZmSearchToken(lcWord));
					endOk = false;
					if (lcWord == ZmParsedQuery.COND_OR) {
						this.hasOrTerm = true;
					}
				} else if (word) {
					tokens.push(new ZmSearchToken(ZmParsedQuery.OP_CONTENT, word));
				}
				word = "";
			}
		}

		if (ch == '"') {
			var results = getQuotedStr(query, pos);
			if (results) {
				word = results.str;
				pos = results.pos;
			} else {
				return fail("improper use of quotes", query);
			}
		} else if (ch == ZmParsedQuery.GROUP_OPEN) {
			var done = false;
			if (compound > 0) {
				compound++;
			}
			else if (lastCh == ":") {
				compound = 1;
				// see if parens are being used as secondary quoting mechanism by looking for and/or
				var inside = query.substr(pos, query.indexOf(ZmParsedQuery.GROUP_CLOSE, pos + 1));
				inside = inside && inside.toLowerCase();
				if (inside && (inside.indexOf(" " + ZmParsedQuery.COND_OR + " ") == -1) &&
							  (inside.indexOf(" " + ZmParsedQuery.COND_AND + " ") == -1)) {
					var results = getQuotedStr(query, pos, ZmParsedQuery.GROUP_CLOSE);
					if (results) {
						word = results.str;
						pos = results.pos;
						compound = 0;
					} else {
						return fail("improper use of paren-based quoting", query);
					}
					done = true;
				}
			}
			if (!done) {
				tokens.push(new ZmSearchToken(ch));
				numParens++;
			}
			pos = skipSpace(query, pos + 1);
		} else if (ch == ZmParsedQuery.GROUP_CLOSE) {
			if (compound > 0) {
				compound--;
			}
			if (compound == 0) {
				op = "";
			}
			tokens.push(new ZmSearchToken(ch));
			pos = skipSpace(query, pos + 1);
		} else if (ch == "-" && !word && !op) {
			tokens.push(new ZmSearchToken(ZmParsedQuery.COND_NOT));
			pos = skipSpace(query, pos + 1);
			endOk = false;
		} else {
			if (ch != " ") {
				word += ch;
			}
			pos++;
		}
	}

	// check for term at end
	if ((pos >= query.length) && op && word) {
		tokens.push(new ZmSearchToken(op, word));
		endOk = true;
	} else if (!op && word) {
		tokens.push(new ZmSearchToken(word));
	}
	
	// remove unnecessary enclosing parens from when a single compound term is expanded, for example when
	// "subject:(foo bar)" is expanded into "(subject:foo subject:bar)"
	if (tokens.length >= 3 && numParens == 1 && tokens[0].op == ZmParsedQuery.GROUP_OPEN &&
			tokens[tokens.length - 1].op == ZmParsedQuery.GROUP_CLOSE) {
		tokens.shift();
		tokens.pop();
	}

	if (!endOk) {
		return fail("unexpected end of query", query);
	}
	
	return tokens;
};

ZmParsedQuery.prototype.getTokens =
function() {
	return this._tokens;
};

ZmParsedQuery.prototype.getNumTokens =
function() {
	return this._tokens ? this._tokens.length : 0;
};

ZmParsedQuery.prototype.getProperties =
function() {
	
	var props = {};
	for (var i = 0, len = this._tokens.length; i < len; i++) {
		var t = this._tokens[i];
		if (t.type == ZmParsedQuery.TERM) {
			var prev = i > 0 ? this._tokens[i-1] : null;
			if (!((prev && prev.op == ZmParsedQuery.COND_NOT) || this.hasOrTerm)) {
				if ((t.op == "in" || t.op == "inid") ) {
					this.folderId = props.folderId = (t.op == "in") ? this._getFolderId(t.arg) : t.arg;
				} else if (t.op == "tag") {
					// TODO: make sure there's only one tag term?
					this.tagId = props.tagId = this._getTagId(t.arg, true);
				}
			}
		}
	}
	return props;
};

/**
 * Returns a function based on the parsed query. The function is passed an item (msg or conv) and returns
 * true if the item matches the search.
 * 
 * @return {Function}	the match function
 */
ZmParsedQuery.prototype.getMatchFunction =
function() {
	
	if (this._matchFunction) {
		return this._matchFunction;
	}
	if (this.parseFailed || this.hasTerm(ZmParsedQuery.OP_CONTENT)) {
		return null;
	}
	
	var folderId, tagId;
	var func = ["return Boolean("];
	for (var i = 0, len = this._tokens.length; i < len; i++) {
		var t = this._tokens[i];
		if (t.type === ZmParsedQuery.TERM) {
			if (t.op === "in" || t.op === "inid") {
				folderId = (t.op === "in") ? this._getFolderId(t.arg) : t.arg;
				if (folderId) {
					func.push("((item.type === ZmItem.CONV) ? item.folders && item.folders['" + folderId +"'] : item.folderId === '" + folderId + "')");
				}
			}
			else if (t.op === "tag") {
				tagId = this._getTagId(t.arg, true);
				if (tagId) {
					func.push("item.hasTag('" + t.arg + "')");
				}
			}
			else if (t.op === "is") {
				var test = ZmParsedQuery.FLAG[t.arg];
				if (test) {
					func.push(test);
				}
			}
			else if (t.op === 'has' && t.arg === 'attachment') {
				func.push("item.hasAttach");
			}
			else {
				// search had a term we don't know how to match
				return null;
			}
			var next = this._tokens[i + 1];
			if (next && (next.type == ZmParsedQuery.TERM || next == ZmParsedQuery.COND_OP[ZmParsedQuery.COND_NOT] || next == ZmParsedQuery.GROUP_CLOSE)) {
				func.push(ZmParsedQuery.COND_OP[ZmParsedQuery.COND_AND]);
			}
		}
		else if (t.type === ZmParsedQuery.COND) {
			func.push(ZmParsedQuery.COND_OP[t.op]);
		}
		else if (t.type === ZmParsedQuery.GROUP) {
			func.push(t.op);
		}
	}
	func.push(")");

	// the way multi-account searches are done, we set the queryHint *only* so
	// set the folderId if it exists for simple multi-account searches
	// TODO: multi-acct part seems wrong; search with many folders joined by OR would incorrectly set folderId to last folder
	var isMultiAccountSearch = (appCtxt.multiAccounts && this.isMultiAccount() && !this.query && this.queryHint);
	if (!this.hasOrTerm || isMultiAccountSearch) {
		this.folderId = folderId;
		this.tagId = tagId;
	}
	
	try {
		this._matchFunction = new Function("item", func.join(""));
	} catch(ex) {}
	
	return this._matchFunction;
};

/**
 * Returns a query string that should be logically equivalent to the original query.
 */
ZmParsedQuery.prototype.createQuery =
function() {
	var terms = [];
	for (var i = 0, len = this._tokens.length; i < len; i++) {
		terms.push(this._tokens[i].toString());
	}
	return terms.join(" ");
};

// Returns the fully-qualified ID for the given folder path.
ZmParsedQuery.prototype._getFolderId =
function(path) {
	// first check if it's a system folder (name in query string may not match actual name)
	var folderId = ZmFolder.QUERY_ID[path];

	var accountName = this.accountName;
	if (!accountName) {
		var active = appCtxt.getActiveAccount();
		accountName = active ? active.name : appCtxt.accountList.mainAccount;
	}

	// now check all folders by name
	if (!folderId) {
		var account = accountName && appCtxt.accountList.getAccountByName(accountName);
		var folders = appCtxt.getFolderTree(account);
		var folder = folders ? folders.getByPath(path, true) : null;
		if (folder) {
			folderId = folder.id;
		}
	}

	if (accountName) {
		folderId = ZmOrganizer.getSystemId(folderId, appCtxt.accountList.getAccountByName(accountName));
	}

	return folderId;
};

// Returns the ID for the given tag name.
ZmParsedQuery.prototype._getTagId =
function(name, normalized) {
	var tagTree = appCtxt.getTagTree();
	if (tagTree) {
		var tag = tagTree.getByName(name.toLowerCase());
		if (tag) {
			return normalized ? tag.nId : tag.id;
		}
	}
	return null;
};

/**
 * Gets the given term with the given argument. Case-insensitive. Returns the first term found.
 * 
 * @param	{array}		opList		list of ops 
 * @param	{string}	value		argument value (optional)
 * 
 * @return	{object}	a token object, or null
 */
ZmParsedQuery.prototype.getTerm =
function(opList, value) {
	var opHash = AjxUtil.arrayAsHash(opList);
	var lcValue = value && value.toLowerCase();
	for (var i = 0, len = this._tokens.length; i < len; i++) {
		var t = this._tokens[i];
		var lcArg = t.arg && t.arg.toLowerCase();
		if (t.type == ZmParsedQuery.TERM && opHash[t.op] && (!value || lcArg == lcValue)) {
			return t;
		}
	}
	return null;
};

/**
 * Returns true if the query contains the given term with the given argument. Case-insensitive.
 * 
 * @param	{array}		opList		list of ops 
 * @param	{string}	value		argument value (optional)
 * 
 * @return	{boolean}	true if the query contains the given term with the given argument
 */
ZmParsedQuery.prototype.hasTerm =
function(opList, value) {
	return Boolean(this.getTerm(opList, value));
};

/**
 * Replaces the argument within the query for the given ops, if found. Case-insensitive. Replaces
 * only the first match.
 * 
 * @param	{array}		opList		list of ops 
 * @param	{string}	oldValue	the old argument
 * @param	{string}	newValue	the new argument
 * 
 * @return	{string}	a new query string (if the old argument was found and replaced), or the empty string
 */
ZmParsedQuery.prototype.replaceTerm =
function(opList, oldValue, newValue) {
	var lcValue = oldValue && oldValue.toLowerCase();
	var opHash = AjxUtil.arrayAsHash(opList);
	if (oldValue && newValue) {
		for (var i = 0, len = this._tokens.length; i < len; i++) {
			var t = this._tokens[i];
			var lcArg = t.arg && t.arg.toLowerCase();
			if (t.type == ZmParsedQuery.TERM && opHash[t.op] && (lcArg == lcValue)) {
				t.arg = newValue;
				return this.createQuery();
			}
		}
	}
	return "";
};

/**
 * This class represents one unit of a search query. That may be a search term ("is:unread"),
 * and conditional operator (AND, OR, NOT), or a grouping operator (left or right paren).
 * 
 * @param {string}	op		operator
 * @param {string}	arg		argument part of search term
 */
ZmSearchToken = function(op, arg) {
	
	if (op && arguments.length == 1) {
		var parts = op.split(":");
		op = parts[0];
		arg = parts[1];
	}
	
	this.op = op;
	this.arg = arg;
	if (ZmParsedQuery.IS_OP[op] && arg) {
		this.type = ZmParsedQuery.TERM;
	}
	else if (op && ZmParsedQuery.COND_OP[op.toLowerCase()]) {
		this.type = ZmParsedQuery.COND;
		this.op = op.toLowerCase();
	}
	else if (op == ZmParsedQuery.GROUP_OPEN || op == ZmParsedQuery.GROUP_CLOSE) {
		this.type = ZmParsedQuery.GROUP;
	} else if (op) {
		this.type = ZmParsedQuery.TERM;
		this.op = ZmParsedQuery.OP_CONTENT;
		this.arg = op;
	}
};

ZmSearchToken.prototype.isZmSearchToken = true;

/**
 * Returns the string version of this token.
 * 
 * @param {boolean}		force		if true, return "and" instead of an empty string ("and" is implied)
 */
ZmSearchToken.prototype.toString =
function(force) {
	if (this.type == ZmParsedQuery.TERM) {
		var arg = this.arg;
		if (this.op == ZmParsedQuery.OP_CONTENT) {
			return /\W/.test(arg) ? '"' + arg.replace(/"/g, '\\"') + '"' : arg;
		}
		else {
			// quote arg if it has any spaces and is not already quoted
			arg = (arg && (arg.indexOf('"') !== 0) && arg.indexOf(" ") != -1) ? '"' + arg + '"' : arg;
			return [this.op, arg].join(":");
		}
	}
	else {
		return (!force && this.op == ZmParsedQuery.COND_AND) ? "" : this.op;
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmSearchResult")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the search result class.
 */

/**
 * Creates the search result
 * @class
 * This class represents a search result.
 * 
 * @param	{ZmSearch}	search		the search
 */
ZmSearchResult = function(search) {
	if (!search) { return; }
	this._results = {};
	this.search = search;
	this.type = search.searchFor;
};

ZmSearchResult.prototype.isZmSearchResult = true;
ZmSearchResult.prototype.toString = function() { return "ZmSearchResult"; };

/**
 * Gets the results.
 * 
 * @param	{constant}	type	the type
 * @return	{Array}	an array of results
 */
ZmSearchResult.prototype.getResults =
function(type) {

	type = type || this.type;
	if (!this._results) {
		// probably got an exception - return an empty list
		return this._getResultsList(type);
	} else if (this.search.idsOnly) {
		return this._results;
	} else {
		// if we don't have results for the requested type, the search was probably for the wrong type
		return this._results[type] ? this._results[type] : type && this._getResultsList(type);
	}
};

/**
 * Gets the attribute.
 * 
 * @param	{String}	name		the attribute name
 * @return	{Object}	the attribute
 */
ZmSearchResult.prototype.getAttribute = 
function(name) {
	return this._respEl ? this._respEl[name] : null;
};

/**
 * Sets the response.
 * 
 * @private
 */
ZmSearchResult.prototype.set =
function(respEl) {

	if (!this.search) { return; }

	this._respEl = respEl;

	// <match> objects are returned for autocomplete search, not items; let caller handle them
	if (this.search.isAutocompleteSearch) { return; }

	var foundType = {};
	var numTypes = 0;
	var currentType, defaultType;
	var isGalSearch = this.search.isGalSearch;
	
	var _st = new Date();
	var count = 0;
	if (isGalSearch || this.search.isCalResSearch) {
		// process JS eval result for SearchGalRequest
		currentType = defaultType = isGalSearch ? ZmItem.CONTACT : ZmItem.RESOURCE;
		var data = isGalSearch ? respEl.cn : respEl.calresource;
		if (data) {
			if (!this._results[currentType]) {
				// create list as needed - may invoke package load
				this._results[currentType] =  this._getResultsList(currentType);
			}
			for (var j = 0; j < data.length; j++) {
				this._results[currentType].addFromDom(data[j]);
			}

			// manually sort gal results since server won't do it for us :(
			if (isGalSearch) {
				this._results[currentType].getArray().sort(ZmSearchResult._sortGalResults)
			}
			count = data.length;
		}
	} else if (this.search.idsOnly) {
		this._results = respEl.hit || [];
		return;
	} else {
		// process JS eval result for SearchResponse
		var types = this.search.types.getArray();
		defaultType = types[0];

		// bug fix #44232 - resolve default type if none provided
		if (!defaultType) {
			var allTypes = AjxUtil.values(ZmList.NODE);
			for (var i = 0; i < allTypes.length; i++) {
				var t = allTypes[i];
				if (respEl[t]) {
					defaultType = ZmList.ITEM_TYPE[t];
					if (types && types.length == 0) {
						types = [defaultType];
					}
					break;
				}
			}
		}

		if (!defaultType) {
			var curApp = appCtxt.getCurrentAppName();
			var types = ZmApp.SEARCH_TYPES[curApp];
			defaultType = types && types.length && types[0];
		}

		for (var i = 0; i < types.length; i++) {
			var type = types[i];
			var data = respEl[ZmList.NODE[type]];

			// A Resource is a Contact. Futz with the types to deal with this.
			if (!data && (type == ZmItem.RESOURCE)) {
				data = respEl[ZmList.NODE[ZmItem.CONTACT]];
			}

			// do a bunch of sanity checks
			if (data && data.length) {
				count += data.length;
				if (!this._results[type]) {
					// create list as needed - may invoke package load
					this._results[type] = this._getResultsList(type);
				}
				for (var j = 0; j < data.length; j++) {
					var item = data[j];
					item._type = type;
					this._results[type].addFromDom(item);
				}

				if (!foundType[type]) {
					foundType[type] = true;
					numTypes++;
					currentType = type;
				}
			}
		}
	}
	if (!count && defaultType) {
		this._results[defaultType] = this._getResultsList(defaultType);
	}
	if ((isGalSearch || this.search.isGalAutocompleteSearch) && this._results[ZmItem.CONTACT]) {
		this._results[ZmItem.CONTACT].setIsGal(true);
	}
	if (this.search.isGalAutocompleteSearch) {
		this.isTokenized = (this._respEl.tokenizeKey != null);
	}
	
	var _en = new Date();
	DBG.println(AjxDebug.DBG1, "TOTAL PARSE TIME for " + count + " NODES: " + (_en.getTime() - _st.getTime()));

	currentType = currentType || defaultType;
	if (numTypes <= 1) {
		this.type = currentType;
	}

	return this.type;
};

/**
 * @private
 */
ZmSearchResult._sortGalResults =
function(a, b) {
	var af = a.getFileAs && a.getFileAs().toLowerCase();
	var bf = b.getFileAs && b.getFileAs().toLowerCase();
	return af < bf ? -1 : (af > bf ? 1 : 0);
};

ZmSearchResult.prototype._getResultsList =
function(type) {

	if (type && typeof(ZmItem.RESULTS_LIST[type]) === "function") {
		return ZmItem.RESULTS_LIST[type](this.search);
	} else {
		DBG.println(
			AjxDebug.DBG1,
			AjxMessageFormat.format(
				"!type || ZmItem.RESULTS_LIST[type] !== function. Active app: {0}, type: {1}, searchFor: {2}.",
				[appCtxt.getCurrentAppName(), type, this.search.searchFor]
			)
		);
		return new ZmList(type, this.search);
	}

};

}
if (AjxPackage.define("zimbraMail.share.model.ZmTree")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file contains the tree class
 */

/**
 * Creates the tree
 * @class
 * This class represents a tree.
 * 
 * @param	{constant}	type		the type
 * @extends	ZmModel
 */
ZmTree = function(type) {

	if (arguments.length == 0) { return; }
	ZmModel.call(this, type);

	this.type = type;
	this.root = null;
};

ZmTree.prototype = new ZmModel;
ZmTree.prototype.constructor = ZmTree;

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmTree.prototype.toString = 
function() {
	return "ZmTree";
};

/**
 * Gets this tree as a string.
 * 
 * @return	{String}	the tree
 */
ZmTree.prototype.asString = 
function() {
	return this.root ? this._asString(this.root, "") : "";
};

/**
 * Gets the item by id.
 * 
 * @param	{String}	id		the id
 * @return	{Object}	the item
 */
ZmTree.prototype.getById =
function(id) {
	return this.root ? this.root.getById(id) : null;
};

/**
 * Gets the item by name.
 * 
 * @param	{String}	name		the name
 * @return	{Object}	the item
 */
ZmTree.prototype.getByName =
function(name) {
	return this.root ? this.root.getByName(name) : null;
};

/**
 * Gets the item type by name.
 *
 * @param	{String}	name		the name
 * @return	{String}    type of folder
 */
//Bug:47848: new method that returns type of the item given its name
ZmTree.prototype.getFolderTypeByName =
function(name){

    //As folder names are case-insensitive
    var formattedName = name.toLowerCase();

    //Iterate through folders of loaded apps
    var folderList = appCtxt.getTree(ZmOrganizer.FOLDER).asList();
    var type;
    var i;
    for(i=0 ; i < folderList.length ; i ++){
        var currentName = folderList[i].name;
        if(formattedName == currentName.toLowerCase()){
            return folderList[i].type;
        }
    }

    // check for _deferredFolders in the apps that have not been loaded
    var apps = ZmApp.APPS;

    for(i=0 ; i<apps.length; i++){
       var currentApp = appCtxt.getApp(apps[i]);
       var deferredFolders = currentApp && currentApp._deferredFolders;
       if(!deferredFolders){
           continue;
       }
       var j;
       for(j=0 ; j < deferredFolders.length ; j++){
           var currentFolder = deferredFolders[j];
           var currentName = currentFolder.obj && currentFolder.obj.name;
            if(formattedName == currentName.toLowerCase()){
                return currentFolder.type;
            }
       }

    }
    // if still not found return type as "Folder"
    type = ZmOrganizer.FOLDER;
    return type;
}

/**
 * Gets the item by type.
 * 
 * @param	{String}	name		the type name
 * @return	{Object}	the item
 */
ZmTree.prototype.getByType =
function(name) {
	return this.root ? this.root.getByType(name) : null;
};

/**
 * Gets the size of the tree.
 * 
 * @return	{int}	the size
 */
ZmTree.prototype.size =
function() {
	return this.root ? this.root.size() : 0;
};

/**
 * Resets the tree.
 */
ZmTree.prototype.reset =
function() {
	this.root = null;
};

/**
 * Gets the tree as a list.
 * 
 * @return	{Array}	an array
 */
ZmTree.prototype.asList =
function(options) {
	var list = [];
	return this.root ? this._addToList(this.root, list, options) : list;
};

/**
 * Gets the unread hash.
 * 
 * @param	{Hash}	unread		the unread hash
 * @return	{Hash} the unread tree as a hash
 */
ZmTree.prototype.getUnreadHash =
function(unread) {
	if (!unread) {
		unread = {};
	}
	return this.root ? this._getUnreadHash(this.root, unread) : unread;
};

/**
 * @private
 */
ZmTree.prototype._addToList =
function(organizer, list, options) {
	var incRemote = options && options.includeRemote;
	var remoteOnly = options && options.remoteOnly;
	var isRemote = organizer.isRemote();
	if ((!isRemote && !remoteOnly) || (isRemote && (remoteOnly || incRemote))) {
		list.push(organizer);
	}
	var children = organizer.children.getArray();
    for (var i = 0; i < children.length; i++) {
        this._addToList(children[i], list, options);
    }
	return list;
};

/**
 * @private
 */
ZmTree.prototype._asString =
function(organizer, str) {
	if (organizer.id) {
		str = str + organizer.id;
	}
	var children = organizer.children.clone().getArray();
	if (children.length) {
		children.sort(function(a,b){return a.id - b.id;});
		str = str + "[";
		for (var i = 0; i < children.length; i++) {
			if (children[i].id == ZmFolder.ID_TAGS) { // Tags "folder" added when view is set
				continue;
			}
			if (i > 0) {
				str = str + ",";
			}
			str = this._asString(children[i], str);
		}
		str = str + "]";
	}
	return str;
};

/**
 * @private
 */
ZmTree.prototype._getUnreadHash =
function(organizer, unread) {
	unread[organizer.id] = organizer.numUnread;
	var children = organizer.children.getArray();
	for (var i = 0; i < children.length; i++) {
		this._getUnreadHash(children[i], unread);
	}

	return unread;
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmFolderTree")) {
	/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * 
 * This file defines a folder tree.
 *
 */

/**
 * Creates an empty folder tree.
 * @class
 * This class represents a tree of folders. It may be typed, in which case
 * the folders are all of that type, or untyped.
 * 
 * @author Conrad Damon
 * 
 * @param {constant}	type		the organizer type
 * 
 * @extends	ZmTree
 */
ZmFolderTree = function(type) {
	ZmTree.call(this, type);
};

ZmFolderTree.prototype = new ZmTree;
ZmFolderTree.prototype.constructor = ZmFolderTree;


// Consts
ZmFolderTree.IS_PARSED = {};


// Public Methods

/**
 * Returns a string representation of the object.
 * 
 * @return		{String}		a string representation of the object
 */
ZmFolderTree.prototype.toString =
function() {
	return "ZmFolderTree";
};

/**
 * Loads the folder or the zimlet tree.
 * 
 * @param	{Object}		rootObj		the root object
 * @param	{String}		elementType		the element type
 * @param	{ZmZimbraAccount}		account		the account
 */
ZmFolderTree.prototype.loadFromJs =
function(rootObj, elementType, account) {
	this.root = (elementType == "zimlet")
		? ZmZimlet.createFromJs(null, rootObj, this)
		: ZmFolderTree.createFromJs(null, rootObj, this, elementType, null, account);
};

/**
 * Generic function for creating a folder. Handles any organizer type that comes
 * in the folder list.
 * 
 * @param {ZmFolder}	parent		the parent folder
 * @param {Object}	obj			the JSON with folder data
 * @param {ZmFolderTree}	tree			the containing tree
 * @param {String}	elementType		the type of containing JSON element
 * @param {Array}	path			the list of path elements
 * @param {ZmZimbraAccount}	account		the account this folder belongs to
 */
ZmFolderTree.createFromJs =
function(parent, obj, tree, elementType, path, account) {
	if (!(obj && obj.id)) { return; }
	if (obj && (obj.owner || obj.id == ZmFolder.ID_FILE_SHARED_WITH_ME) && (!appCtxt.isExternalAccount() && !appCtxt.get(ZmSetting.SHARING_ENABLED))) { return; }

	var folder;
	if (elementType == "search") {
		var types;
		var idParts = obj.id.split(":");
		// Suppress display of searches for the shared mailbox (See Bug 96090) - it will have an id
		// of the form 'uuid:id'.  Local searches will just have 'id'
		if (!idParts || (idParts.length <= 1)) {
			if (obj.types) {
				var t = obj.types.split(",");
				types = [];
				var mailEnabled = appCtxt.get(ZmSetting.MAIL_ENABLED);
				for (var i = 0; i < t.length; i++) {
					var type = ZmSearch.TYPE_MAP[t[i]];
					if (!type || (!mailEnabled && (type == ZmItem.CONV || type == ZmItem.MSG))) {
						continue;
					}
					types.push(type);
				}
				if (types.length == 0) {
					return null;
				}
			}
			DBG.println(AjxDebug.DBG2, "Creating SEARCH with id " + obj.id + " and name " + obj.name);
			var params = {
				id: obj.id,
				name: obj.name,
				parent: parent,
				tree: tree,
				numUnread: obj.u,
				query: obj.query,
				types: types,
				sortBy: obj.sortBy,
				account: account,
				color: obj.color,
				rgb: obj.rgb
			};
			folder = new ZmSearchFolder(params);
			ZmFolderTree._fillInFolder(folder, obj, path);
			ZmFolderTree._traverse(folder, obj, tree, (path || []), elementType, account);
		}
	} else {
		var type = obj.view
			? (ZmOrganizer.TYPE[obj.view])
			: (parent ? parent.type : ZmOrganizer.FOLDER);

		if (!type) {
			DBG.println(AjxDebug.DBG1, "No known type for view " + obj.view);
			return;
		}
		// let's avoid deferring folders for offline since multi-account folder deferring is hairy
		var hasGrants = (obj.acl && obj.acl.grant && obj.acl.grant.length > 0);
		if (appCtxt.inStartup && ZmOrganizer.DEFERRABLE[type] && !appCtxt.isOffline) {
			var app = appCtxt.getApp(ZmOrganizer.APP[type]);
			var defParams = {
				type:			type,
				parent:			parent,
				obj:			obj,
				tree:			tree,
				path:			path,
				elementType:	elementType,
				account:		account
			};
			app.addDeferredFolder(defParams);
		} else {
			var pkg = ZmOrganizer.ORG_PACKAGE[type];
			if (pkg) {
				AjxDispatcher.require(pkg);
			}
			folder = ZmFolderTree.createFolder(type, parent, obj, tree, path, elementType, account);
            if (appCtxt.isExternalAccount() && folder.isSystem() && folder.id != ZmOrganizer.ID_ROOT) { return; }
			ZmFolderTree._traverse(folder, obj, tree, (path || []), elementType, account);
		}
	}

	return folder;
};

ZmFolderTree.createAllDeferredFolders =
function() {
	var ac = appCtxt.getAppController();
	for (var appId in ZmApp.ORGANIZER) {
		var app = ac.getApp(appId);
		app.createDeferred();
	}
};

/**
 * @private
 */
ZmFolderTree._traverse =
function(folder, obj, tree, path, elementType, account) {

	var isRoot = (folder.nId == ZmOrganizer.ID_ROOT);

	if (elementType === "hab") {
		obj.folder = obj.habGroup
	}

	if (obj.folder && obj.folder.length) {
		if (!isRoot) {
			path.push(obj.name);
		}
		for (var i = 0; i < obj.folder.length; i++) {
			var folderObj = obj.folder[i];
			var childFolder = ZmFolderTree.createFromJs(folder, folderObj, tree, (elementType || "folder"), path, account);
			if (folder && childFolder) {
				folder.children.add(childFolder);
			}
		}
		if (!isRoot) {
			path.pop();
		}
	}
	
	if (obj.search && obj.search.length) {
		if (!isRoot) {
			path.push(obj.name);
		}
		for (var i = 0; i < obj.search.length; i++) {
			var searchObj = obj.search[i];
			var childSearch = ZmFolderTree.createFromJs(folder, searchObj, tree, "search", path, account);
			if (childSearch) {
				folder.children.add(childSearch);
			}
		}
		if (!isRoot) {
			path.pop();
		}
	}

	if (obj.link && obj.link.length) {
		for (var i = 0; i < obj.link.length; i++) {
			var link = obj.link[i];
			var childFolder = ZmFolderTree.createFromJs(folder, link, tree, "link", path, account);
			if (childFolder) {
				folder.children.add(childFolder);
			}
		}
	}
};

/**
 * Creates the folder.
 * 
 * @param {String}	type		the folder type
 * @param {ZmFolder}	parent		the parent folder
 * @param {Object}	obj			the JSON with folder data
 * @param {ZmFolderTree}	tree			the containing tree
 * @param {Array}	path			the list of path elements
 * @param {String}	elementType		the type of containing JSON element
 * @param {ZmZimbraAccount}	account		the account this folder belongs to
 */
ZmFolderTree.createFolder =
function(type, parent, obj, tree, path, elementType, account) {
	var orgClass = eval(ZmOrganizer.ORG_CLASS[type]);
	if (!orgClass) { return null; }

	DBG.println(AjxDebug.DBG2, "Creating " + type + " with id " + obj.id + " and name " + obj.name);

	var params = {
		id: 		obj.id,
		name: 		obj.name,
		parent: 	parent,
		tree: 		tree,
		color: 		obj.color,
		rgb:		obj.rgb,
		owner: 		obj.owner,
		oname: 		obj.oname,
		zid: 		obj.zid,
		rid: 		obj.rid,
		restUrl: 	obj.rest,
		url: 		obj.url,
		numUnread: 	obj.u,
		numTotal: 	obj.n,
		sizeTotal: 	obj.s,
		perm: 		obj.perm,
		link: 		elementType == "link",
		broken: 	obj.broken,
		account:	account,
        webOfflineSyncDays : obj.webOfflineSyncDays,
        retentionPolicy: obj.retentionPolicy
	};

	if (elementType == "hab") {
		params.name = obj._attrs ? obj._attrs.displayName : obj.name;
		params.mail = obj._attrs ? obj._attrs.mail : undefined;
	}

	var folder = new orgClass(params);
	ZmFolderTree._fillInFolder(folder, obj, path);
	ZmFolderTree.IS_PARSED[type] = true;

	return folder;
};

/**
 * @private
 */
ZmFolderTree._fillInFolder =
function(folder, obj, path) {
	if (path && path.length) {
		folder.path = path.join("/");
	}

	if (obj.f && folder._parseFlags) {
		folder._parseFlags(obj.f);
	}

	folder._setSharesFromJs(obj);
};

/**
 * Gets the folder by type.
 * 
 * @param	{String}	type	the type
 * @return	{ZmFolder}	the folder or <code>null</code> if not found
 */
ZmFolderTree.prototype.getByType =
function(type) {
	return this.root ? this.root.getByType(type) : null;
};

/**
 * Gets the folder by path.
 * 
 * @param	{String}	path	the path
 * @param	{Boolean}	useSystemName		<code>true</code> to use the system name
 * @return	{ZmFolder}	the folder or <code>null</code> if not found
 */
ZmFolderTree.prototype.getByPath =
function(path, useSystemName) {
	return this.root ? this.root.getByPath(path, useSystemName) : null;
};

/**
 * Handles a missing link by marking its organizer as not there, redrawing it in
 * any tree views, and asking to delete it.
 *
 * @param {int}	organizerType		the type of organizer (constants defined in {@link ZmOrganizer})
 * @param {String}	zid			the zid of the missing folder
 * @param {String}	rid			the rid of the missing folder
 * @return	{Boolean}	<code>true</code> if the error is handled
 */
ZmFolderTree.prototype.handleNoSuchFolderError =
function(organizerType, zid, rid) {
	var items = this.getByType(organizerType);

	var treeView;
	var handled = false;
	if (items) {
		for (var i = 0; i < items.length; i++) {
			if ((items[i].zid == zid) && (items[i].rid == rid)) {
				// Mark that the item is not there any more.
				items[i].noSuchFolder = true;

				// Change its appearance in the tree.
				if (!treeView) {
					var overviewId = appCtxt.getAppController().getOverviewId();
					treeView = appCtxt.getOverviewController().getTreeView(overviewId, organizerType);
				}
				var node = treeView.getTreeItemById(items[i].id);
				node.setText(items[i].getName(true));

				// Ask if it should be deleted now.
				this.handleDeleteNoSuchFolder(items[i]);
				handled = true;
			}
		}
	}
	return handled;
};

/**
 * Handles no such folder. The user will be notified that a linked organizer generated a "no such folder",
 * error, giving the user a chance to delete the folder.
 *
 * @param {ZmOrganizer}	organizer	the organizer
 */
ZmFolderTree.prototype.handleDeleteNoSuchFolder =
function(organizer) {
	var ds = appCtxt.getYesNoMsgDialog();
	ds.reset();
	ds.registerCallback(DwtDialog.YES_BUTTON, this._deleteOrganizerYesCallback, this, [organizer, ds]);
	ds.registerCallback(DwtDialog.NO_BUTTON, appCtxt.getAppController()._clearDialog, this, ds);
	var msg = AjxMessageFormat.format(ZmMsg.confirmDeleteMissingFolder, AjxStringUtil.htmlEncode(organizer.getName(false, 0, true)));
	ds.setMessage(msg, DwtMessageDialog.WARNING_STYLE);
	ds.popup();
};

/**
 * Handles the "Yes" button in the delete organizer dialog.
 * 
 * @param	{ZmOrganizer}	organizer		the organizer
 * @param	{ZmDialog}		dialog		the dialog
 */
ZmFolderTree.prototype._deleteOrganizerYesCallback =
function(organizer, dialog) {
	organizer._delete();
	appCtxt.getAppController()._clearDialog(dialog);
};

/**
 * Issues a <code>&lt;BatchRequest&gt;</code> of <code>&lt;GetFolderRequest&gt;</code>s for existing
 * mountpoints that do not have permissions set.
 *
 * @param	{Hash}	params	a hash of parameters
 * @param {int}	params.type			the {@link ZmItem} type constant
 * @param {AjxCallback}	params.callback			the callback to trigger after fetching permissions
 * @param {Boolean}	params.skipNotify		<code>true</code> to skip notify after fetching permissions
 * @param {Array}	params.folderIds			the list of folder Id's to fetch permissions for
 * @param {Boolean}	params.noBusyOverlay		<code>true</code> to not block the UI while fetching permissions
 * @param {String}	params.accountName		the account to issue request under
 */
ZmFolderTree.prototype.getPermissions =
function(params) {
	var needPerms = params.folderIds || this._getItemsWithoutPerms(params.type);

	// build batch request to get all permissions at once
	if (needPerms.length > 0) {
		var soapDoc = AjxSoapDoc.create("BatchRequest", "urn:zimbra");
		soapDoc.setMethodAttribute("onerror", "continue");

		var doc = soapDoc.getDoc();
		for (var j = 0; j < needPerms.length; j++) {
			var folderRequest = soapDoc.set("GetFolderRequest", null, null, "urn:zimbraMail");
			var folderNode = doc.createElement("folder");
			folderNode.setAttribute("l", needPerms[j]);
			folderRequest.appendChild(folderNode);
		}

		var respCallback = new AjxCallback(this, this._handleResponseGetShares, [params.callback, params.skipNotify]);
		appCtxt.getRequestMgr().sendRequest({
			soapDoc: soapDoc, 
			asyncMode: true,
			callback: respCallback,
			noBusyOverlay: params.noBusyOverlay,
			accountName: params.accountName
		});
	} else {
		if (params.callback) {
			params.callback.run();
		}
	}
};

/**
 * @private
 */
ZmFolderTree.prototype._getItemsWithoutPerms =
function(type) {
	var needPerms = [];
	var orgs = type ? [type] : [ZmOrganizer.FOLDER, ZmOrganizer.CALENDAR, ZmOrganizer.TASKS, ZmOrganizer.BRIEFCASE, ZmOrganizer.ADDRBOOK];

	for (var j = 0; j < orgs.length; j++) {
		var org = orgs[j];
		if (!ZmFolderTree.IS_PARSED[org]) { continue; }

		var items = this.getByType(org);

		for (var i = 0; i < items.length; i++) {
			if (items[i].link && items[i].shares == null) {
				needPerms.push(items[i].id);
			}
		}
	}

	return needPerms;
};

/**
 * @private
 */
ZmFolderTree.prototype._handleResponseGetShares =
function(callback, skipNotify, result) {
	var batchResp = result.getResponse().BatchResponse;
	this._handleErrorGetShares(batchResp);

	var resp = batchResp.GetFolderResponse;
	if (resp) {
		for (var i = 0; i < resp.length; i++) {
			var link = resp[i].link ? resp[i].link[0] : null;
			if (link) {
				var mtpt = appCtxt.getById(link.id);
				if (mtpt) {
					// update the mtpt perms with the updated link perms
					mtpt.perm = link.perm;
                    if (link.n) mtpt.numTotal=link.n;
                    if (link.u) mtpt.numUnread=link.u;
					mtpt._setSharesFromJs(link);
				}

				if (link.folder && link.folder.length > 0) {
					var parent = appCtxt.getById(link.id);
					if (parent) {
						// TODO: only goes one level deep - should we recurse?
						for (var j = 0; j < link.folder.length; j++) {
							if (appCtxt.getById(link.folder[j].id)) { continue; }
							parent.notifyCreate(link.folder[j], "link", skipNotify);
						}
					}
				}
			}
		}
	}

	if (callback) {
		callback.run();
	}
};

/**
 * Handles errors that come back from the GetShares batch request.
 *
 * @param {Array}	organizerTypes	the types of organizer (constants defined in {@link ZmOrganizer})
 * @param {Object}	batchResp			the response
 *
 */
ZmFolderTree.prototype._handleErrorGetShares =
function(batchResp) {
	var faults = batchResp.Fault;
	if (faults) {
		var rids = [];
		var zids = [];
		for (var i = 0, length = faults.length; i < length; i++) {
			var ex = ZmCsfeCommand.faultToEx(faults[i]);
			if (ex.code == ZmCsfeException.MAIL_NO_SUCH_FOLDER) {
				var itemId = ex.data.itemId[0];
				var index = itemId.lastIndexOf(':');
				zids.push(itemId.substring(0, index));
				rids.push(itemId.substring(index + 1, itemId.length));
			}
		}
		if (zids.length) {
			this._markNoSuchFolder(zids, rids);
		}
	}
};

/**
 * Handles missing links by marking the organizers as not there
 *
 * @param {Array}	zids		the zids of the missing folders
 * @param {Array}	rids		the rids of the missing folders. rids and zids must have the same length
 *
 */
ZmFolderTree.prototype._markNoSuchFolder =
function(zids, rids) {
	var treeData = appCtxt.getFolderTree();
	var items = treeData && treeData.root
		? treeData.root.children.getArray()
		: null;

	for (var i = 0; i < items.length; i++) {
		for (var j = 0; j < rids.length; j++) {
			if ((items[i].zid == zids[j]) && (items[i].rid == rids[j])) {
				items[i].noSuchFolder = true;
			}
		}
	}
};

/**
 * @private
 */
ZmFolderTree.prototype._sortFolder =
function(folder) {
	var children = folder.children;
	if (children && children.length) {
		children.sort(ZmFolder.sortCompare);
		for (var i = 0; i < children.length; i++)
			this._sortFolder(children[i]);
	}
};
}
if (AjxPackage.define("zimbraMail.share.model.ZmList")) {
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */

/**
 * @overview
 * This file defines a list of items.
 */

/**
 * Creates an empty list of items of the given type.
 * @class
 * This class represents a list of items ({@link ZmItem} objects). Any SOAP method that can be
 * applied to a list of item IDs is represented here, so that we can perform an action
 * on multiple items with just one CSFE call. For the sake of convenience, a hash 
 * matching item IDs to items is maintained. Items are assumed to have an 'id'
 * property.
 * <br/>
 * <br/>
 * The calls are made asynchronously. We are assuming that any action taken will result
 * in a notification, so the action methods generally do not have an async callback 
 * chain and thus are leaf nodes. An exception is moving conversations. We don't
 * know enough from the ensuing notifications (which only indicate that messages have
 * moved), we need to update the UI based on the response.
 *
 * @author Conrad Damon
 * 
 * @param {constant}	type		the item type
 * @param {ZmSearch}	search	the search that generated this list
 * 
 * @extends	ZmModel
 */
ZmList = function(type, search) {

	if (arguments.length == 0) return;
	ZmModel.call(this, type);

	this.type = type;
	this.search = search;
	
	this._vector = new AjxVector();
	this._hasMore = false;
	this._idHash = {};

	var tagList = appCtxt.getTagTree();
	if (tagList) {
		this._tagChangeListener = new AjxListener(this, this._tagTreeChangeListener);
		tagList.addChangeListener(this._tagChangeListener);
	}
	
	this.id = "LIST" + ZmList.NEXT++;
	appCtxt.cacheSet(this.id, this);
};

ZmList.prototype = new ZmModel;
ZmList.prototype.constructor = ZmList;

ZmList.prototype.isZmList = true;
ZmList.prototype.toString = function() { return "ZmList"; };


ZmList.NEXT = 1;

// for item creation
ZmList.ITEM_CLASS = {};

// node names for item types
ZmList.NODE = {};

// item types based on node name (reverse map of above)
ZmList.ITEM_TYPE = {};

ZmList.CHUNK_SIZE	= 100;	// how many items to act on at a time via a server request
ZmList.CHUNK_PAUSE	= 500;	// how long to pause to allow UI to catch up


/**
 * Gets the item.
 * 
 * @param	{int}	index		the index
 * @return	{ZmItem}	the index
 */
ZmList.prototype.get =
function(index) {
	return this._vector.get(index);
};

/**
 * Adds an item to the list.
 *
 * @param {ZmItem}	item	the item to add
 * @param {int}	index	the index at which to add the item (defaults to end of list)
 */
ZmList.prototype.add = 
function(item, index) {
	this._vector.add(item, index);
	if (item.id) {
		this._idHash[item.id] = item;
	}
};

/**
 * Removes an item from the list.
 *
 * @param {ZmItem}	item	the item to remove
 */
ZmList.prototype.remove = 
function(item) {
	this._vector.remove(item);
	if (item.id) {
		delete this._idHash[item.id];
	}
};

/**
 * Creates an item from the given arguments. A subclass may override
 * <code>sortIndex()</code> to add it to a particular point in the list. By default, it
 * will be added at the end.
 *
 * <p>
 * The item will invoke a SOAP call, which generates a create notification from the
 * server. That will be handled by notifyCreate(), which will call _notify()
 * so that views can be updated.
 * </p>
 *
 * @param {Hash}	args	a hash of arugments to pass along to the item constructor
 * @return	{ZmItem}	the newly created item
 */
ZmList.prototype.create =
function(args) {
	var item;
	var obj = eval(ZmList.ITEM_CLASS[this.type]);
	if (obj) {
		item = new obj(this);
		item.create(args);
	}

	return item;
};

/**
 * Returns the number of items in the list.
 * 
 * @return	{int}	the number of items
 */
ZmList.prototype.size = 
function() {
	return this._vector.size();
};

/**
 * Returns the index of the given item in the list.
 * 
 * @param	{ZmItem}	item		the item
 * @return	{int}	the index
 */
ZmList.prototype.indexOf = 
function(item) {
	return this._vector.indexOf(item);
};

/**
 * Gets if there are more items for this search.
 * 
 * @return	{Boolean}	<code>true</code> if there are more items
 */
ZmList.prototype.hasMore = 
function() {
	return this._hasMore;
};

/**
 * Sets the "more" flag for this list.
 *
 * @param {Boolean}	bHasMore	<code>true</code> if there are more items
 */
ZmList.prototype.setHasMore = 
function(bHasMore) {
	this._hasMore = bHasMore;
};

/**
 * Returns the list as an array.
 * 
 * @return	{Array}	an array of {ZmItem} objects
 */
ZmList.prototype.getArray =
function() {
	return this._vector.getArray();
};

/**
 * Returns the list as a vector.
 * 
 * @return	{AjxVector}	a vector of {ZmItem} objects
 */
ZmList.prototype.getVector =
function() {
	return this._vector;
};

/**
 * Gets the item with the given id.
 *
 * @param {String}	id		an item id
 * 
 * @return	{ZmItem}	the item
 */
ZmList.prototype.getById =
function(id) {
	return this._idHash[id];
};

/**
 * Clears the list, including the id hash.
 * 
 */
ZmList.prototype.clear =
function() {
	// First, let each item run its clear() method
	var a = this.getArray();
	for (var i = 0; i < a.length; i++) {
		a[i].clear && a[i].clear();
	}

	this._evtMgr.removeAll(ZmEvent.L_MODIFY);
	this._vector.removeAll();
	for (var id in this._idHash) {
		this._idHash[id] = null;
	}
	this._idHash = {};
};

/**
 * Populates the list with elements created from the response to a SOAP command. Each
 * node in the response should represent an item of the list's type. Items are added
 * in the order they are received; no sorting is done.
 *
 * @param {Object}	respNode	an XML node whose children are item nodes
 */
ZmList.prototype.set = 
function(respNode) {
	this.clear();
	var nodes = respNode.childNodes;
	var args = {list:this};
	for (var i = 0; i < nodes.length; i++) {
		var node = nodes[i];
		if (node.nodeName == ZmList.NODE[this.type]) {
			/// TODO: take this out, let view decide whether to show items in Trash
			if (parseInt(node.getAttribute("l")) == ZmFolder.ID_TRASH && (this.type != ZmItem.CONTACT))	{ continue; }
			var obj = eval(ZmList.ITEM_CLASS[this.type]);
			if (obj) {
				this.add(obj.createFromDom(node, args));
			}
		}
	}
};

/**
 * Adds an item to the list from the given XML node.
 *
 * @param {Object}	node	an XML node
 * @param {Hash}	args	an optional list of arguments to pass to the item contructor
 */
ZmList.prototype.addFromDom = 
function(node, args) {
	args = args || {};
	args.list = this;
	var obj = eval(ZmList.ITEM_CLASS[this.type]);
	if (obj) {
		this.add(obj.createFromDom(node, args));
	}
};

/**
 * Gets a vector containing a subset of items of this list.
 *
 * @param {int}		offset		the starting index
 * @param {int}		limit		the size of sublist
 * @return	{AjxVector}	the vector
 */
ZmList.prototype.getSubList = 
function(offset, limit) {
	var subVector = null;
	var end = (offset + limit > this.size()) ? this.size() : offset + limit;
	var subList = this.getArray();
	if (offset < end) {
		subVector = AjxVector.fromArray(subList.slice(offset, end));
	}
	return subVector;
};

/**
 * Caches the list.
 * 
 * @param	{int}	offset	the index
 * @param	{AjxVector}	newList		the new list
 */
ZmList.prototype.cache = 
function(offset, newList) {
	this.getVector().merge(offset, newList);
	// reparent each item within new list, and add it to ID hash
	var list = newList.getArray();
	for (var i = 0; i < list.length; i++) {
		var item = list[i];
		item.list = this;
		if (item.id) {
			this._idHash[item.id] = item;
		}
	}
};

// Actions

/**
 * Sets and unsets a flag for each of a list of items.
 *
 * @param 	{Hash}				params					a hash of parameters
 * @param	{Array}     		params.items			a list of items to set/unset a flag for
 * @param	{String}			params.op				the name of the flag operation ("flag" or "read")
 * @param	{Boolean|String}	params.value			whether to set the flag, or for "update" the flags string
 * @param	{AjxCallback}		params.callback			the callback to run after each sub-request
 * @param	{closure}			params.finalCallback	the callback to run after all items have been processed
 * @param	{int}				params.count			the starting count for number of items processed
 * @param   {String}    		params.actionTextKey   	pattern for generating action summarykey to action summary message
 */
ZmList.prototype.flagItems =
function(p